##############################################################################
# 
# Zope Public License (ZPL) Version 1.0
# -------------------------------------
# 
# Copyright (c) Digital Creations.  All rights reserved.
# 
# This license has been certified as Open Source(tm).
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
# 
# 1. Redistributions in source code must retain the above copyright
#    notice, this list of conditions, and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions, and the following disclaimer in
#    the documentation and/or other materials provided with the
#    distribution.
# 
# 3. Digital Creations requests that attribution be given to Zope
#    in any manner possible. Zope includes a "Powered by Zope"
#    button that is installed by default. While it is not a license
#    violation to remove this button, it is requested that the
#    attribution remain. A significant investment has been put
#    into Zope, and this effort will continue if the Zope community
#    continues to grow. This is one way to assure that growth.
# 
# 4. All advertising materials and documentation mentioning
#    features derived from or use of this software must display
#    the following acknowledgement:
# 
#      "This product includes software developed by Digital Creations
#      for use in the Z Object Publishing Environment
#      (http://www.zope.org/)."
# 
#    In the event that the product being advertised includes an
#    intact Zope distribution (with copyright and license included)
#    then this clause is waived.
# 
# 5. Names associated with Zope or Digital Creations must not be used to
#    endorse or promote products derived from this software without
#    prior written permission from Digital Creations.
# 
# 6. Modified redistributions of any form whatsoever must retain
#    the following acknowledgment:
# 
#      "This product includes software developed by Digital Creations
#      for use in the Z Object Publishing Environment
#      (http://www.zope.org/)."
# 
#    Intact (re-)distributions of any official Zope release do not
#    require an external acknowledgement.
# 
# 7. Modifications are encouraged but must be packaged separately as
#    patches to official Zope releases.  Distributions that do not
#    clearly separate the patches from the original work must be clearly
#    labeled as unofficial distributions.  Modifications which do not
#    carry the name Zope may be packaged in any form, as long as they
#    conform to all of the clauses above.
# 
# 
# Disclaimer
# 
#   THIS SOFTWARE IS PROVIDED BY DIGITAL CREATIONS ``AS IS'' AND ANY
#   EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
#   PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL DIGITAL CREATIONS OR ITS
#   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
#   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
#   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
#   USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
#   ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
#   OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
#   OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#   SUCH DAMAGE.
# 
# 
# This software consists of contributions made by Digital Creations and
# many individuals on behalf of Digital Creations.  Specific
# attributions are listed in the accompanying credits file.
# 
##############################################################################

import unittest
import ZODB # for Persistent
import string

# FIXME: could test with Core DOM instead, do we want to?
from Products.ParsedXML import ParsedXML, PrettyPrinter
from Products.ParsedXML.StrIO import StringIO

def printElement(element, encoding = None, html = 0, contentType = None):
    output = StringIO()
    PrettyPrinter.PrintVisitor(element, output, encoding, html, contentType)()
    return output.getvalue()

def checkOutput(wanted, got, message="Bad output"):
    assert wanted == got, \
           ("%s.  Wanted:\n%s[[EOF]]\nGot:\n%s[[EOF]]\n" 
            % (message, wanted, got))


class PrintTestBase(unittest.TestCase):
    implementation = ParsedXML.theDOMImplementation

    def parse(self, xml):
        return ParsedXML.ParsedXML('foo', xml)


class PrintTestCase(PrintTestBase):

    def checkAttrOrder(self):
        inStr = '<?xml version="1.0" ?>\n<doc a1="v1" a2="v2" a3="v3"/>\n'
        doc = self.parse(inStr)
        output = printElement(doc)

        checkOutput(inStr, output, "attribute order not preserved")

    def checkDefaultAttrSkipped(self):
        inStr = '<!DOCTYPE doc [<!ELEMENT doc EMPTY>' \
                '<!ATTLIST doc a1 CDATA "v1">]><doc></doc>'
        outStr = '<?xml version="1.0" ?>\n<doc/>\n'
        doc = self.parse(inStr)
        output = printElement(doc)

        checkOutput(outStr, output, "default attribute printed")

    def checkAttrEntRefExpansion(self):
        "entity references must be expanded in attributes"
        doc = self.implementation.createDocument(None, 'root', None)
        attr = doc.createAttribute("attrName")
        # this is a text string; &test; is not an entity reference,
        # so its & should be converted to the proper ref on printing,
        # as should the & and < and >.
        attr.value = "&test; &<>"
        # this reference should be expanded to the empty string
        attr.appendChild(doc.createEntityReference('foo'))
        doc.documentElement.setAttributeNode(attr)

        outStr = '<root attrName="&amp;test; &amp;&lt;&gt;"/>'
        output = printElement(doc.documentElement)
        checkOutput(outStr, output, "improper attr entity expansion")

    def checkTextEntRefExpansion(self):
        "only some entity references should be expanded in text"
        doc = self.implementation.createDocument(None, 'root', None)
        # &< must be converted to the proper reference; > should not.
        text = doc.createTextNode("&<>]]>")
        outStr = '&amp;&lt;>]]&gt;'
        output = printElement(text)
        checkOutput(outStr, output, "improper text entity expansion")

    #TODO: check for expansion of entity refs in other contexts;
    #currently we're expanding aggressively, but it's not a priority
    #because the parser gets to play around with refs too

class HTMLPrintTestCase(PrintTestBase):

    def checkMinimize(self):
        inStr = ('<html><p><hr/><hr noshade="1"/></p><p/><p></p>' +
                 '<p align="center" /></html>')
        outStr = ('<html><p><hr /><hr noshade="1" /></p><p></p><p></p>' +
                  '<p align="center"></p></html>\n')
        doc = self.parse(inStr)
        output = printElement(doc, encoding = None, html = 1)

        checkOutput(outStr, output, "improper HTML minimization")

    def checkCapitalize(self):
        inStr = ('<html><HR nOsHaDe="1" /></html>\n')
        doc = self.parse(inStr)
        output = printElement(doc, encoding = None, html = 1,
                              contentType = 'html')
        checkOutput(string.upper(inStr), output,
                    "improper HTML contenttype HTML capitalization")
        output = printElement(doc, encoding = None, html = 1,
                              contentType = 'xml')
        checkOutput(string.lower(inStr), output,
                    "improper XML contenttype HTML capitalization")
        

class Lvl2PrintTestCase(PrintTestBase):

    def checkNamespacePrint(self):
        outStr = '<?xml version="1.0" ?>\n' \
                    '<bar xmlns:foo="uri:foo">\n' \
                    '<foo:baz/></bar>\n'
        doc = self.parse(outStr)
        output = printElement(doc)
        checkOutput(outStr, output)

    def checkNamespaceAttrOrder(self):
        inStr = ('<?xml version="1.0" ?>\n'
                  '<doc>'
                  '  <fooE xmlns="defaultURL" xmlns:oneN="oneURL" '
                          'xmlns:twoN="twoURL" xmlns:threeN="threeURL"/>'
                  '</doc>\n')
        doc = self.parse(inStr)
        output = printElement(doc)
        checkOutput(inStr, output, "attribute order not preserved")


    def checkHierarchicalElementNamespacePrint(self):
        # print new ns, don't print ns printed by ancestor
        outStr = ('<?xml version="1.0" ?>\n'
                  '<fooN:fooE xmlns:fooN="fooURL">'
                  '  <barN:barE xmlns:barN="barURL">'
                  '  <fooN:fooE>'
                  '    <fooN:fooE/>'
                  '  </fooN:fooE>'
                  '  </barN:barE>'
                  '</fooN:fooE>\n')
        doc = self.parse(outStr)
        output = printElement(doc)
        checkOutput(outStr, output)

    def checkDefaultNamespacePrint(self):
        outStr = ('<?xml version="1.0" ?>\n'
                  '<fooE xmlns="defaultURL"><fooE xmlns="barURL"/>'
                  '</fooE>\n')
        doc = self.parse(outStr)
        output = printElement(doc)
        checkOutput(outStr, output)

    def checkDefaultAndPrefixNamespacePrint(self):
        # try and tickle a namespace printing bug
        outStr = ('<?xml version="1.0" ?>\n'
                  '<doc>'
                  '  <fooE xmlns="defaultURL" xmlns:fooN="fooURL" '
                          'xmlns:barN="barUrl" xmlns:bazN="bazUrl">'
                  '  <barE bazN:bazE="baz"/>'
                  '  <bazN:bazE/>'
                  '  <quxN xmlns:quxE="quxUrl"/>'
                  '  <quxN xmlns:quxE="quxUrl" bazN:bazE="baz"/>'
                  '  </fooE>'
                  '</doc>\n')
        doc = self.parse(outStr)
        output = printElement(doc)
        checkOutput(outStr, output)


def test_suite():
    """Return a test suite for the Zope testing framework."""

    suite = unittest.TestSuite()
    suite.addTest(unittest.makeSuite(PrintTestCase, 'check'))
    suite.addTest(unittest.makeSuite(HTMLPrintTestCase, 'check'))
    suite.addTest(unittest.makeSuite(Lvl2PrintTestCase, 'check'))
    return suite

def main():
    unittest.TextTestRunner().run(test_suite())

if __name__ == '__main__':
    main()
