###############################################################################################
# _xmllib.py
#
# $Id: _xmllib.py,v 1.9 2004/02/24 19:57:21 dnordmann Exp $
# $Name:  $
# $Author: dnordmann $
# $Revision: 1.9 $
#
# Implementation of XML-Library (see below).
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
###############################################################################################

# Imports.
from __future__ import nested_scopes
from App.Common import package_home
from Shared.DC.xml import pyexpat
import Globals
import copy
import os
import tempfile
# Product Imports.
from _objattrs import *
import _blobfields
import _globals
import _fileutil


"""
###############################################################################################
#
#  Datatypes
#
###############################################################################################
"""

# ---------------------------------------------------------------------------------------------
#  _xmllib.getXmlType:
# ---------------------------------------------------------------------------------------------
def getXmlType(v):
  t = ''
  if type(v) is type(float(0.0)):
    t = ' type="float"'
  if type(v) is type(int(0)):
    t = ' type="int"'
  if type(v) is type({}):
    t = ' type="dictionary"'
  if type(v) is type([]):
    t = ' type="list"'
  if type(v) is type(()):
    t = ' type="datetime"'
  if hasattr(v,'__class__') and getattr(v,'__class__') is _blobfields.MyImage:
    t = ' type="image"'
  if hasattr(v,'__class__') and getattr(v,'__class__') is _blobfields.MyFile:
    t = ' type="file"'
  return t


# ---------------------------------------------------------------------------------------------
#  _xmllib.getXmlTypeSaveValue:
# ---------------------------------------------------------------------------------------------
def getXmlTypeSaveValue(v, attrs):
  # Strip.
  if type(v) is type(''): 
    while len(v) > 0 and v[0] <= ' ':
      v = v[1:]
    while len(v) > 0 and v[-1] <= ' ':
      v = v[:-1]
  # Type.
  t = '?'
  if 'type' in attrs and attrs.index('type')%2 == 0:
    t = attrs[attrs.index('type')+1]
  if t == 'float':
    try:
      v = float(v)
    except:
      _globals.writeException(self,"[_xmllib.getXmlTypeSaveValue]: Conversion to '%s' failed for '%s'!"%(t,str(v)))
  elif t == 'int':
    try:
      v = int(v)
    except:
      _globals.writeException(self,"[_xmllib.getXmlTypeSaveValue]: Conversion to '%s' failed for '%s'!"%(t,str(v)))
  elif t == 'datetime':
    new = _globals.parseLangFmtDate(v)
    if new is not None:
      v = new
  # Return value.
  return v


"""
###############################################################################################
#
#  XML-Encoding
#
###############################################################################################
"""

# ---------------------------------------------------------------------------------------------
#  _xmllib.xml_header:
# ---------------------------------------------------------------------------------------------
def xml_header(encoding='utf-8'):
  return '<?xml version="1.0" encoding="%s"?>'%encoding


# ---------------------------------------------------------------------------------------------
#  _xmllib.xml_quote:
# ---------------------------------------------------------------------------------------------
def xml_quote(s, encoding='utf-8'):
  return unicode(str(s),'latin-1').encode(encoding)


# ---------------------------------------------------------------------------------------------
#  _xmllib.inv_xml_quote:
# ---------------------------------------------------------------------------------------------
def inv_xml_quote(s, encoding='utf-8'):
  return unicode(str(s),encoding).encode('latin-1')



"""
###############################################################################################
#
#  Import
#
###############################################################################################
"""

# ---------------------------------------------------------------------------------------------
#  _xmllib.xmlInitObjProperty:
#
#  Only for internal use: init specified property with value.
# ---------------------------------------------------------------------------------------------
def xmlInitObjProperty(self, key, value, lang=None):
  
  #-- DEFINITION
  obj_attr = self.getObjAttr(key)
  
  #-- ATTR
  attr = self.getObjAttrName(obj_attr,lang)
  
  #-- DATATYPE
  datatype = obj_attr['datatype_key']
  
  if value is not None:
    #-- Date-Fields
    if datatype in _globals.DT_DATETIMES:
      if type(value) is type('') and len(value) > 0:
        value = self.parseLangFmtDate(value)
    #-- Integer-Fields
    elif datatype in _globals.DT_INTS:
      if type(value) is type('') and len(value) > 0:
        value = int(value)
    #-- Float-Fields
    elif datatype == _globals.DT_FLOAT:
      if type(value) is type('') and len(value) > 0:
        value = float(value)
    #-- String-Fields
    elif datatype in _globals.DT_STRINGS:
      value = str(value)
  
  #-- INIT
  for ob in self.getObjVersions():
    setattr(ob,attr,value)


# ---------------------------------------------------------------------------------------------
#  _xmllib.xmlOnCharacterData:
# ---------------------------------------------------------------------------------------------
def xmlOnCharacterData(self, sData, bInCData):

  #-- TAG-STACK
  if self.dTagStack.size() > 0:
    tag = self.dTagStack.pop()
    tag['cdata'] += sData
    self.dTagStack.push(tag)

  #-- Return
  return 1  # accept any character data


# ---------------------------------------------------------------------------------------------
#  _xmllib.xmlOnUnknownStartTag:
# ---------------------------------------------------------------------------------------------
def xmlOnUnknownStartTag(self, sTagName, dTagAttrs):
        
  #-- TAG-STACK
  tag = {'name':sTagName,'attrs':dTagAttrs,'cdata':''}
  tag['dValueStack'] = self.dValueStack.size()
  self.dTagStack.push(tag)
  
  #-- VALUE-STACK
  if sTagName == 'lang':
    if self.dValueStack.size()==0:
      self.dValueStack.push({})
  elif sTagName == 'dictionary':
    self.dValueStack.push({})
  elif sTagName == 'list':
    self.dValueStack.push([])
    
  #-- Return
  return 1  # accept any unknown tag


# ---------------------------------------------------------------------------------------------
#  _xmllib.xmlOnUnknownEndTag:
# ---------------------------------------------------------------------------------------------
def xmlOnUnknownEndTag(self, sTagName):
  
  #-- TAG-STACK
  tag = self.dTagStack.pop()
  name = tag['name']
  if name==sTagName:
    
    attrs = tag['attrs']
    cdata = inv_xml_quote(tag['cdata'])
    
    #-- ITEM (DICTIONARY|LIST) --
    #----------------------------
    if sTagName in ['list','dictionary']:
      pass
    elif sTagName in ['item']:
      item = cdata
      if tag['dValueStack'] < self.dValueStack.size():
        item = self.dValueStack.pop()
      else:
        item = cdata
      item = getXmlTypeSaveValue(item,attrs)
      value = self.dValueStack.pop()
      if type(value) is type({}):
        key = attrs[attrs.index('key')+1]
        value[key] = item
      if type(value) is type([]):
        value.append(item)
      self.dValueStack.push(value)
    
    #-- DATA (IMAGE|FILE) --
    #-----------------------
    elif sTagName=='data':
      value = {}
      for i in range(len(attrs)/2):
        key = attrs[i*2]
        item = attrs[i*2+1]
        value[key] = item
      self.dValueStack.push(value)
    
    #-- LANGUAGE --
    #--------------
    elif sTagName=='lang':
      lang = self.getPrimaryLanguage()
      if 'id' in attrs:
        lang = attrs[attrs.index('id')+1]
      if self.dValueStack.size()==1:
        item = cdata
      else:
        item = self.dValueStack.pop()
      values = self.dValueStack.pop()
      values[lang] = item
      self.dValueStack.push(values)
      ##### Object State ####
      req = {}
      req['lang'] = lang
      req['AUTHENTICATED_USER'] = 'xml'
      self.setObjStateNew(req,reset=0)
    
    #-- OBJECT-ATTRIBUTES --
    #-----------------------
    elif sTagName in self.getObjAttrs().keys():
    
      obj_attr = self.getObjAttr(sTagName)
      
      #-- DATATYPE
      datatype = obj_attr['datatype_key']
      
      #-- Multi-Language Attributes.
      if obj_attr['multilang']:
        item = self.dValueStack.pop()
        if item is not None:
          if not type(item) is type({}):
            item = {self.getPrimaryLanguage():item}
          for s_lang in item.keys():
            value = item[s_lang]
            # Data
            if datatype in _globals.DT_BLOBS:
              if type(value) is type({}) and len(value.keys()) > 0:
                ob = _blobfields.createBlobField(self,datatype)
                for key in value.keys():
                  setattr(ob,key,value[key])
                xmlInitObjProperty(self,sTagName,ob,s_lang)
            # Others
            else:
              xmlInitObjProperty(self,sTagName,value,s_lang)
      
      #-- Complex Attributes (Dictionary|List).
      elif self.dValueStack.size() > 0:
        value = self.dValueStack.pop()
        # Data
        if datatype in _globals.DT_BLOBS:
          if type(value) is type({}) and len(value.keys()) > 0:
            ob = _blobfields.createBlobField(self,datatype)
            for key in value.keys():
              setattr(ob,key,value[key])
            xmlInitObjProperty(self,sTagName,ob)
        # Others
        else:
          #-- Convert multilingual to monolingual attributes.
          if obj_attr['multilang']==0 and \
             type(value) is type({}) and \
             len(value.keys())==1 and \
             value.keys()[0]==self.getPrimaryLanguage():
            value = value[value.keys()[0]]
          xmlInitObjProperty(self,sTagName,value)
        if self.dValueStack.size() > 0:
          raise "Items on self.dValueStack=%s"%self.dValueStack
      
      #-- Simple Attributes (String, Integer, etc.)
      else:
        value = cdata
        #-- OPTIONS
        if obj_attr.has_key('options'):
          options = obj_attr['options']
          if type(options) is type([]):
            try:
              i = options.index(int(value))
              if i%2==1: value = options[i-1]
            except:
              try: 
                i = options.index(str(value))
                if i%2==1: value = options[i-1]
              except: 
                pass
        xmlInitObjProperty(self,sTagName,value)
      
      # Clear value stack.
      self.dValueStack.clear()
    
    #-- OTHERS --
    #------------
    else:
      value = self.dTagStack.pop()
      if value is None: value = {'cdata':''}
      cdata = value.get('cdata','')
      cdata += '<' + tag['name'] 
      for i in range(len(tag['attrs'])/2):
        attr_name = tag['attrs'][i*2]
        attr_value = tag['attrs'][i*2+1]
        cdata += ' ' + attr_name + '="' + attr_value + '"'
      cdata += '>' + tag['cdata'] 
      cdata += '</' + tag['name'] + '>'
      value['cdata'] = cdata
      self.dTagStack.push(value)
    
    return 1 # accept matching end tag
  else:
    return 0  # don't accept any unknown tag


"""
###############################################################################################
#
#  Export
#
###############################################################################################
"""

# ---------------------------------------------------------------------------------------------
#  _xmllib.toCdata
# ---------------------------------------------------------------------------------------------
def toCdata(self, s, xhtml=0, encoding='utf-8'):
  rtn = ''
  
  # Return Text (HTML) in CDATA as XHTML.
  from _filtermanager import processCommand
  processId = 'tidy'
  if xhtml == 0 \
     and self.getConfProperty('ZMS.export.xml.tidy',0) \
     and processId in self.getProcessIds():

    # Create temporary folder.
    folder = tempfile.mktemp()
    os.mkdir(folder)

    # Save <HTML> to file.
    filename = _fileutil.getOSPath('%s/xhtml.html'%folder)
    _fileutil.exportObj(s, filename)

    # Call <HTML>Tidy
    processOb = self.getProcess(processId)
    command = processOb.get('command')
    if command.find('{trans}') >= 0:
      trans = _fileutil.getOSPath(package_home(globals())+'/conf/xsl/tidy.html2xhtml.conf')
      command = command.replace('{trans}',trans)
    filename = processCommand(self, filename, command)

    # Read <XHTML> from file.
    f = open(htmfilename,'rb')
    rtn = xml_quote(f.read()).strip()
    f.close()
    
    # Read Error-Log from file.
    f = open(logfilename,'rb')
    log = xml_quote(f.read()).strip()
    f.close()

    # Remove temporary files.
    _fileutil.remove(folder,deep=1)

    # Process <XHTML>.
    prefix = '<p>'
    if s[:len(prefix)] != prefix and rtn[:len(prefix)] == prefix:
      rtn = rtn[len(prefix):]
      suffix = '</p>'
      if s[-len(suffix):] != suffix and rtn[-len(suffix):] == suffix:
        rtn = rtn[:-len(suffix)]
    f.close()

    # Process Error-Log.
    if log.find('0 errors') < 0:
      rtn += '<!-- ' + log + '-->'

  # Return Text.
  elif xhtml == -1 \
     or type(s) is type(float(0.0)) \
     or type(s) is type(int(0)):
    rtn = s

  # Return Text in CDATA.
  elif s is not None:
    rtn = '<![CDATA[%s]]>'%xml_quote(s, encoding)

  # Return.
  return rtn


# ---------------------------------------------------------------------------------------------
#  _xmllib.toXml:
# ---------------------------------------------------------------------------------------------
def toXml(self, value, xhtml=0, encoding='utf-8'):
  xml = []
  
  if value is not None:
    
    # Image
    if hasattr(value,'__class__') and getattr(value,'__class__') is _blobfields.MyImage:
      xml.append(value.toXml())
    
    # File
    elif hasattr(value,'__class__') and getattr(value,'__class__') is _blobfields.MyFile:
      xml.append(value.toXml())
    
    # Dictionaries
    elif type(value) is type({}):
      keys = value.keys()
      keys.sort()
      xml.append('<dictionary>\n')
      xml.extend(map(lambda key: '<item key="%s"%s>%s</item>\n'%(key,getXmlType(value[key]),toXml(self,value[key],xhtml,encoding)),keys))
      xml.append('</dictionary>\n')
    
    # Lists
    elif type(value) is type([]):
      xml.append('<list>\n')
      xml.extend(map(lambda item: '<item%s>%s</item>\n'%(getXmlType(item),toXml(self,item,xhtml,encoding)),value))
      xml.append('</list>\n')
      
    # Tuples (DateTime)
    elif type(value) is type(()):
      try:
        s_value = self.getLangFmtDate(value,self.getPrimaryLanguage(),'DATETIME_FMT')
        if len(s_value) > 0:
          xml.append(toCdata(self,s_value,-1,encoding))
      except:
        pass
    
    # Numbers
    elif type(value) is type(0) or type(value) is type(0.0):
      s_value = str(value)
      xml.append(toCdata(self,s_value,-1,encoding))
  
    # Others
    else:
      s_value = str(value)
      if len(s_value) > 0:
        xml.append(toCdata(self,s_value,xhtml,encoding))
  
  # Return xml.
  return ''.join(xml)


# ---------------------------------------------------------------------------------------------
#  _xmllib.getAttrToXml:
# ---------------------------------------------------------------------------------------------
def getAttrToXml(self, obj_attr, REQUEST):
  xml = ''
  
  #-- DATATYPE
  datatype = obj_attr['datatype_key']
  
  #-- VALUE
  obj_vers = self.getObjVersion(REQUEST)
  value = self._getObjAttrValue(obj_attr,obj_vers,REQUEST.get('lang',self.getPrimaryLanguage()))
  
  if value is not None:
    
    # Retrieve value from options.
    if obj_attr.has_key('options'):
      options = obj_attr['options']
      try:
        i = options.index(int(value))
        if i%2==0: value = options[i+1]
      except:
        try:
          i = options.index(str(value))
          if i%2==0: value = options[i+1]
        except:
          pass
    
    # Objects.
    if datatype in _globals.DT_BLOBS:
      xml += value.toXml(self)
    
    # XML.
    elif datatype == _globals.DT_XML or \
         datatype == _globals.DT_URL or \
	 datatype == _globals.DT_BOOLEAN or \
	 datatype in _globals.DT_NUMBERS:
      xml += toXml(self,value,-1)

    # Others.
    else:
      xml += toXml(self,value)
    
  # Return xml.
  return xml


# ---------------------------------------------------------------------------------------------
#  _xmllib.getObjPropertyToXml:
# ---------------------------------------------------------------------------------------------
def getObjPropertyToXml(self, obj_attr, REQUEST):
  xml = ''
  # Multi-Language Attributes.
  if obj_attr['multilang']:
    lang = REQUEST.get('lang')
    langIds = self.getLangIds()
    for langId in langIds:
      REQUEST.set('lang',langId)
      s_attr_xml = getAttrToXml(self,obj_attr,REQUEST)
      if len(s_attr_xml) > 0:
        xml += '\n<lang id="%s"'%langId
        xml += '>%s</lang>'%s_attr_xml
    REQUEST.set('lang',lang)
  # Simple Attributes.
  else:
    xml += getAttrToXml(self,obj_attr,REQUEST)
  # Return xml.
  return xml


# ---------------------------------------------------------------------------------------------
#  _xmllib.getObjToXml:
# ---------------------------------------------------------------------------------------------
def getObjToXml(self, REQUEST):
  xml = []
  # Start tag.
  xml.append('<%s'%self.xmlGetTagName())
  id = self.id 
  prefix = _globals.id_prefix(id)
  if id == prefix:
    xml.append(' id_fix="%s"'%id)
  else:
    xml.append(' id="%s"'%id)
  if self.getParentNode() is not None and self.getParentNode().meta_type == 'ZMSCustom':
    xml.append('\n id_prefix="%s"'%_globals.id_prefix(self.id))
  if self.meta_type == 'ZMSCustom':
    xml.append('\n meta_id="%s"'%self.meta_id)
  xml.append('>')
  # Attributes.
  keys = self.getObjAttrs().keys()
  if self.meta_type=='ZMSCustom' and self.getType()=='ZMSRecordSet':
    keys = ['sort_id','active',self.getMetaobjAttrIds(self.meta_id)[0]]
  for key in keys:
    obj_attr = self.getObjAttr(key)
    if obj_attr['xml']:
      ob_prop = getObjPropertyToXml(self,obj_attr,REQUEST)
      if len(ob_prop) > 0:
        xml.append('\n<%s>%s</%s>'%(key,ob_prop,key))
  # Process children.
  xml.extend(map(lambda ob: getObjToXml(ob,REQUEST),self.getChildNodes()))
  # End tag.
  xml.append('</%s>\n'%self.xmlGetTagName())
  # Return xml.
  return "".join(xml)



"""
###############################################################################################
# class ParseError(Exception):
#
# General exception class to indicate parsing errors.
###############################################################################################
"""
class ParseError(Exception): pass



"""
###############################################################################################
# class XmlAttrBuilder:
# 
# Parser for complex Python-Attributes (dictionaries, lists).
###############################################################################################
"""
class XmlAttrBuilder:
    "class XmlAttrBuilder"

    ######## class variables ########
    iBufferSize=1028 * 32   # buffer size for XML file parsing

    ###########################################################################################
    # XmlAttrBuilder.__init__(self):
    #
    # Constructor.
    ###########################################################################################
    def __init__(self):
      """ XmlAttrBuilder.__init__ """
      pass


    ###########################################################################################
    # XmlAttrBuilder.parse(self, input):
    #
    # Parse a given XML document.
    #
    # IN:  input = XML document as string
    #            = XML document as file object
    # OUT: value or None, if nothing was parsed
    ###########################################################################################
    def parse(self, input):
        """ XmlAttrBuilder.parse """
        
        # prepare builder
        self.dValueStack = _globals.MyStack()
        self.dTagStack   = _globals.MyStack()
        
        # create parser object
        p = pyexpat.ParserCreate()
        
        # connect parser object with handler methods
        p.StartElementHandler = self.OnStartElement
        p.EndElementHandler = self.OnEndElement
        p.CharacterDataHandler = self.OnCharacterData
        p.StartCdataSectionHandler = self.OnStartCData
        p.EndCdataSectionHandler = self.OnEndCData
        p.ProcessingInstructionHandler = self.OnProcessingInstruction
        p.CommentHandler = self.OnComment
        p.StartNamespaceDeclHandler = self.OnStartNamespaceDecl
        p.EndNamespaceDeclHandler = self.OnEndNamespaceDecl
        
        #### parsing ####
        #++ if _globals.debug: print "#### parsing ####"
        if type(input)=='string':
          # input is a string!
          rv = p.Parse(input, 1)
        else:
          # input is a file object!
          while 1:
            if Globals.DatabaseVersion == '3':
              get_transaction().commit(1)
            
            v=input.read(self.iBufferSize)
            if v=="":
              rv = 1
              break
            
            rv = p.Parse(v, 0)
            if not rv:
              break 
              
        # raise parser exception
        if not rv:
          raise ParseError('%s at line %s' % (pyexpat.ErrorString(p.ErrorCode), p.ErrorLineNumber))
        
        return self.dValueStack.pop()


    ###########################################################################################
    # XmlAttrBuilder.OnStartElement(self, name, attrs):
    #
    # Handler of XML-Parser: 
    # Called at the start of a XML element (resp. on occurence of a XML start tag).
    # Usually, the occurence of a XML tag induces the instanciation of a new node object. Therefore,
    # XmlAttrBuilder contains a mapping table ("dGlobalAttrs"), that maps XML tags to python classes. The
    # newly created node object is then made current. If no matching class is found for a XML tag,
    # the event handler "xmlOnUnknownStart()" is called on the current object.
    #
    # IN: name  = element name (=tag name)
    #     attrs = dictionary of element attributes
    ###########################################################################################
    def OnStartElement(self, sTagName, dTagAttrs):
      """ XmlAttrBuilder.OnStartElement """
      _globals.writeLog("OnStartElement(" + str(sTagName) + "," + str(dTagAttrs) + ") ")
      
      #-- TAG-STACK
      tag = {'name':sTagName,'attrs':dTagAttrs,'cdata':''}
      tag['dValueStack'] = self.dValueStack.size()
      self.dTagStack.push(tag)
      
      #-- VALUE-STACK
      if sTagName == 'data':
        self.dValueStack.push(None)
      elif sTagName == 'dictionary':
        self.dValueStack.push({})
      elif sTagName == 'list':
        self.dValueStack.push([])


    ###########################################################################################
    # XmlAttrBuilder.OnEndElement(self, name):
    #
    # Handler of XML-Parser: 
    # Called at the end of a XML element (resp. on occurence of a XML end tag).
    #
    # IN: name  = element name (=tag name)
    ###########################################################################################
    def OnEndElement(self, sTagName):
      """ XmlAttrBuilder.OnEndElement """
      _globals.writeLog("OnEndElement(" + str(sTagName) + ")")
      
      #-- TAG-STACK
      tag = self.dTagStack.pop()
      name = tag['name']
      attrs = tag['attrs']
      cdata = inv_xml_quote(tag['cdata'])
      
      if name != sTagName:
        raise ParseError("Unmatching end tag (" + str(sTagName) + ")")
      
      #-- DATA
      if sTagName in ['data']:
        filename = attrs[attrs.index('filename')+1]
        content_type = attrs[attrs.index('content_type')+1]
        if content_type.find('text/') == 0:
          data = cdata
        else:
          data = _blobfields.hex2bin(cdata)
        file = {'data':data,'filename':filename,'content_type':content_type}
        objtype = attrs[attrs.index('type')+1]
        item = None
        if objtype == 'image':
          item = _blobfields.createBlobField(None,_globals.DT_IMAGE,file)
        elif objtype == 'file':
          item = _blobfields.createBlobField(None,_globals.DT_FILE,file)
        for i in range(len(attrs)/2):
          key = attrs[i*2]
          value = attrs[i*2+1]
          setattr(item,key,value)
        self.dValueStack.pop()
        self.dValueStack.push(item)
      
      #-- ITEM
      elif sTagName in ['item']:
        if tag['dValueStack'] < self.dValueStack.size():
          item = self.dValueStack.pop()
        else:
          item = cdata
        item = getXmlTypeSaveValue(item,attrs)
        value = self.dValueStack.pop()
        if type(value) is type({}):
          key = attrs[attrs.index('key')+1]
          value[key] = item
        if type(value) is type([]):
          value.append(item)
        self.dValueStack.push(value)


    ###########################################################################################
    # XmlAttrBuilder.OnCharacterData(self, data):
    #
    # Handler of XML-Parser:
    # Called after plain character data was parsed. Forwards the character data to the current
    # node. The class attribute "bInCData" determines, wether the character data is nested in a
    # CDATA block.
    #
    # IN: data = character data string
    ###########################################################################################
    def OnCharacterData(self, sData):
      """ XmlAttrBuilder.OnCharacterData """
      _globals.writeLog("OnCharacterData(" + str(sData) + ")")
      
      #-- TAG-STACK
      if self.dTagStack.size() > 0:
        tag = self.dTagStack.pop()
        tag['cdata'] += sData
        self.dTagStack.push(tag)


    ###########################################################################################
    # XmlAttrBuilder.OnStartCData(self):
    #
    # Handler of XML-Parser:
    # Called at the start of a CDATA block (resp. on occurence of the "CDATA[" tag).
    ###########################################################################################
    def OnStartCData(self):
      """ XmlAttrBuilder.OnStartCData """
      self.bInCData = 1 


    ###########################################################################################
    # XmlAttrBuilder.OnEndCData(self):
    #
    # Handler of XML-Parser:
    # Called at the end of a CDATA block (resp. on occurence of the "]" tag).
    ###########################################################################################
    def OnEndCData(self):
      """ XmlAttrBuilder.OnEndCData """
      self.bInCData = 0


    ###########################################################################################
    # XmlAttrBuilder.OnProcessingInstruction(self, target, data):
    #
    # Handler of XML-Parser:
    # Called on occurence of a processing instruction.
    #
    # IN: target = target (processing instruction)
    #     data   = dictionary of data
    ###########################################################################################
    def OnProcessingInstruction(self, target, data):
      """ XmlAttrBuilder.OnProcessingInstruction """
      pass  # ignored


    ###########################################################################################
    # XmlAttrBuilder.OnComment(self, data):
    #
    # Handler of XML-Parser:
    # Called on occurence of a comment.
    #
    # IN: data = comment string
    ###########################################################################################
    def OnComment(self, data):
      """ XmlAttrBuilder.OnComment """
      pass  # ignored


    ###########################################################################################
    # XmlAttrBuilder.OnStartNamespaceDecl(self, prefix, uri):
    #
    # Handler of XML-Parser:
    # Called at the start of a namespace declaration.
    #
    # IN: prefix = prefix of namespace
    #     uri    = namespace identifier
    ###########################################################################################
    def OnStartNamespaceDecl(self, prefix, uri):
      """ XmlAttrBuilder.OnStartNamespaceDecl """
      pass  # ignored


    ###########################################################################################
    # XmlAttrBuilder.OnEndNamespaceDecl(self, prefix):
    #
    # Handler of XML-Parser:
    # Called at the end of a namespace declaration.
    #
    # IN: prefix = prefix of namespace
    ###########################################################################################
    def OnEndNamespaceDecl(self, prefix):
      """ XmlAttrBuilder.OnEndNamespaceDecl """
      pass  # ignored



"""
###############################################################################################
# class XmlBuilder:
# 
# Parser for custom xml.
###############################################################################################
"""
class XmlBuilder:
    "class XmlBuilder"

    ######## class variables ########
    iBufferSize=1028 * 32   # buffer size for XML file parsing

    ###########################################################################################
    # XmlBuilder.__init__(self):
    #
    # Constructor.
    ###########################################################################################
    def __init__(self):
      """ XmlBuilder.__init__ """
      pass


    ###########################################################################################
    # XmlBuilder.parse(self, input):
    #
    # Parse a given XML document.
    #
    # IN:  input = XML document as string
    #            = XML document as file object
    # OUT: value or None, if nothing was parsed
    ###########################################################################################
    def parse(self, input):
        """ XmlBuilder.parse """
        
        # prepare builder
        self.dTagStack = _globals.MyStack()
        self.dTagStack.push({'tags':[]})
        
        # create parser object
        p = pyexpat.ParserCreate()
        
        # connect parser object with handler methods
        p.StartElementHandler = self.OnStartElement
        p.EndElementHandler = self.OnEndElement
        p.CharacterDataHandler = self.OnCharacterData
        p.StartCdataSectionHandler = self.OnStartCData
        p.EndCdataSectionHandler = self.OnEndCData
        p.ProcessingInstructionHandler = self.OnProcessingInstruction
        p.CommentHandler = self.OnComment
        p.StartNamespaceDeclHandler = self.OnStartNamespaceDecl
        p.EndNamespaceDeclHandler = self.OnEndNamespaceDecl
        
        #### parsing ####
        #++ if _globals.debug: print "#### parsing ####"
        if type(input)=='string':
          # input is a string!
          rv = p.Parse(input, 1)
        else:
          # input is a file object!
          while 1:
            if Globals.DatabaseVersion == '3':
              get_transaction().commit(1)
            
            v=input.read(self.iBufferSize)
            if v=="":
              rv = 1
              break
            
            rv = p.Parse(v, 0)
            if not rv:
              break 
        
        # raise parser exception
        if not rv:
          raise ParseError('%s at line %s'%(pyexpat.ErrorString(p.ErrorCode), p.ErrorLineNumber))
        
        return self.dTagStack.pop()['tags']


    ###########################################################################################
    # XmlBuilder.OnStartElement(self, name, attrs):
    #
    # Handler of XML-Parser: 
    # Called at the start of a XML element (resp. on occurence of a XML start tag).
    # Usually, the occurence of a XML tag induces the instanciation of a new node object. Therefore,
    # XmlBuilder contains a mapping table ("dGlobalAttrs"), that maps XML tags to python classes. The
    # newly created node object is then made current. If no matching class is found for a XML tag,
    # the event handler "xmlOnUnknownStart()" is called on the current object.
    #
    # IN: name  = element name (=tag name)
    #     attrs = dictionary of element attributes
    ###########################################################################################
    def OnStartElement(self, sTagName, dTagAttrs):
      """ XmlBuilder.OnStartElement """
      #++ print "OnStartElement(" + str(sTagName) + "," + str(dTagAttrs) + ") "
      
      tag = {'name':sTagName,'attrs':dTagAttrs,'cdata':'','tags':[]}
      self.dTagStack.push(tag)


    ###########################################################################################
    # XmlBuilder.OnEndElement(self, name):
    #
    # Handler of XML-Parser: 
    # Called at the end of a XML element (resp. on occurence of a XML end tag).
    #
    # IN: name  = element name (=tag name)
    ###########################################################################################
    def OnEndElement(self, sTagName):
      """ XmlBuilder.OnEndElement """
      #++ print "OnEndElement(" + str(sTagName) + ")"
      
      lTag = self.dTagStack.pop()
      lTagName = lTag['name']
      lAttrs = lTag['attrs']
      lCdata = inv_xml_quote(lTag['cdata'])
      lTags = lTag['tags']
      
      if lTagName != sTagName:
        raise ParseError("Unmatching end tag (" + sTagName + ")")
      
      lTag = {}
      lTag['level'] = self.dTagStack.size()
      lTag['name'] = lTagName
      if len(lAttrs)>0: 
        lTag['attrs'] = {}
        for i in range(len(lAttrs)/2):
          key = lAttrs[i*2]
          value = lAttrs[i*2+1]
          lTag['attrs'][key] = value
      lCdata = lCdata.strip()
      if len(lCdata) > 0: 
        lTag['cdata'] = lCdata
      if len(lTags) > 0: 
        lTag['tags'] = lTags
      parent = self.dTagStack.pop()
      parent['tags'].append(lTagName)
      parent['tags'].append(lTag)
      self.dTagStack.push(parent)


    ###########################################################################################
    # XmlBuilder.OnCharacterData(self, data):
    #
    # Handler of XML-Parser:
    # Called after plain character data was parsed. Forwards the character data to the current 
    # node. The class attribute "bInCData" determines, wether the character data is nested in a 
    # CDATA block.
    #
    # IN: data = character data string
    ###########################################################################################
    def OnCharacterData(self, sData):
      """ XmlBuilder.OnCharacterData """
      #++ print "OnCharacterData(" + str(sData) + ")"
      
      tag = self.dTagStack.pop()
      tag['cdata'] = tag['cdata'] + sData
      self.dTagStack.push(tag)


    ###########################################################################################
    # XmlBuilder.OnStartCData(self):
    #
    # Handler of XML-Parser:
    # Called at the start of a CDATA block (resp. on occurence of the "CDATA[" tag).
    ###########################################################################################
    def OnStartCData(self):
      """ XmlBuilder.OnStartCData """
      self.bInCData=1


    ###########################################################################################
    # XmlBuilder.OnEndCData(self):
    #
    # Handler of XML-Parser:
    # Called at the end of a CDATA block (resp. on occurence of the "]" tag).
    ###########################################################################################
    def OnEndCData(self):
      """ XmlBuilder.OnEndCData """
      self.bInCData=0


    ###########################################################################################
    # XmlBuilder.OnProcessingInstruction(self, target, data):
    #
    # Handler of XML-Parser:
    # Called on occurence of a processing instruction.
    #
    # IN: target = target (processing instruction)
    #     data   = dictionary of data
    ###########################################################################################
    def OnProcessingInstruction(self, target, data):
      """ XmlBuilder.OnProcessingInstruction """
      pass  # ignored


    ###########################################################################################
    # XmlBuilder.OnComment(self, data):
    #
    # Handler of XML-Parser:
    # Called on occurence of a comment.
    #
    # IN: data = comment string
    ###########################################################################################
    def OnComment(self, data):
      """ XmlBuilder.OnComment """
      pass  # ignored


    ###########################################################################################
    # XmlBuilder.OnStartNamespaceDecl(self, prefix, uri):
    #
    # Handler of XML-Parser:
    # Called at the start of a namespace declaration.
    #
    # IN: prefix = prefix of namespace
    #     uri    = namespace identifier
    ###########################################################################################
    def OnStartNamespaceDecl(self, prefix, uri):
      """ XmlBuilder.OnStartNamespaceDecl """
      pass  # ignored


    ###########################################################################################
    # XmlBuilder.OnEndNamespaceDecl(self, prefix):
    #
    # Handler of XML-Parser:
    # Called at the end of a namespace declaration.
    #
    # IN: prefix = prefix of namespace
    ###########################################################################################
    def OnEndNamespaceDecl(self, prefix):
      """ XmlBuilder.OnEndNamespaceDecl """
      pass  # ignored

###############################################################################################
