// vs_editline.cc
//
//  Copyright 2000 Daniel Burrows

#include "vs_editline.h"
#include "config/keybindings.h"
#include "config/colors.h"
#include "vscreen.h"

#include <sigc++/object_slot.h>

using namespace std;

keybindings *vs_editline::bindings=NULL;

vs_editline::vs_editline(string _prompt, string _text,
			 history_list *_history)
  :vscreen_widget(),prompt(_prompt), text(_text), curloc(_text.size()),
   startloc(0), desired_size(-1), history(_history),
   history_loc(0), using_history(false)
{
  set_bg(get_color("EditLine"));

  // This ensures that the cursor is set to the right location when the
  // widget is displayed or resized:
  do_layout.connect(SigC::slot(*this, &vs_editline::normalize_cursor));
}

vs_editline::vs_editline(int maxlength, const string &_prompt,
			 const string &_text, history_list *_history)
  :vscreen_widget(),prompt(_prompt), text(_text), curloc(0),
   startloc(0), desired_size(maxlength), history(_history), history_loc(0),
   using_history(false)
{
  set_bg(get_color("EditLine"));
  do_layout.connect(SigC::slot(*this, &vs_editline::normalize_cursor));
}

void vs_editline::normalize_cursor()
{
  // Dodge the signedness bullet, hopefully
  if(get_width()<0)
    return;

  string::size_type w=(string::size_type) get_width();

  if(prompt.size()+text.size()+1<w)
    startloc=0;
  else if(w>2)
    {
      // The extra 2 is to avoid "I'm typing and I can't see what comes next"
      // problems.
      if(prompt.size()+curloc-startloc>=w-2)
	startloc=prompt.size()+curloc-w+2;

      if(prompt.size()+curloc>=2 && prompt.size()+curloc<startloc+2)
	startloc=prompt.size()+curloc-2;
    }
  else
    {
      if(prompt.size()+curloc-startloc>=w)
	startloc=prompt.size()+curloc-w+1;

      if(prompt.size()+curloc<startloc)
	startloc=prompt.size()+curloc;
    }

  vscreen_updatecursor();
}

bool vs_editline::get_cursorvisible()
{
  return true;
}

point vs_editline::get_cursorloc()
{
  if(getmaxx()>0)
    return point(curloc+prompt.size()-startloc, 0);
  else
    return point(0,0);
}

bool vs_editline::focus_me()
{
  return true;
}

bool vs_editline::handle_char(chtype ch)
{
  if(bindings->key_matches(ch, "DelBack"))
    {
      if(curloc>0)
	{
	  text.erase(--curloc, 1);
	  normalize_cursor();
	  text_changed(string(text));
	  vscreen_update();
	}
      else
	{
	  beep();
	  //	  refresh();
	}
      return true;
    }
  else if(bindings->key_matches(ch, "DelForward"))
    {
      if(curloc<text.size())
	{
	  text.erase(curloc, 1);
	  normalize_cursor();
	  text_changed(string(text));
	  vscreen_update();
	}
      else
	{
	  beep();
	  //	  refresh();
	}
      return true;
    }
  else if(bindings->key_matches(ch, "Confirm"))
    {
      // I create a new string here because otherwise modifications to
      // "text" are seen by the widgets! (grr, sigc++)
      entered(string(text));
      return true;
    }
  else if(bindings->key_matches(ch, "Left"))
    {
      if(curloc>0)
	{
	  curloc--;
	  normalize_cursor();
	  vscreen_update();
	}
      else
	{
	  beep();
	  //	  refresh();
	}
      return true;
    }
  else if(bindings->key_matches(ch, "Right"))
    {
      if(curloc<text.size())
	{
	  curloc++;
	  normalize_cursor();
	  vscreen_update();
	}
      else
	{
	  beep();
	  //	  refresh();
	}
      return true;
    }
  else if(bindings->key_matches(ch, "Begin"))
    {
      curloc=0;
      startloc=0;
      normalize_cursor();
      vscreen_update();
      return true;
    }
  else if(bindings->key_matches(ch, "End"))
    {
      curloc=text.size();
      normalize_cursor();
      vscreen_update();
      return true;
    }
  else if(bindings->key_matches(ch, "DelEOL"))
    {
      text.erase(curloc);
      normalize_cursor();
      text_changed(string(text));
      vscreen_update();
      return true;
    }
  else if(bindings->key_matches(ch, "DelBOL"))
    {
      text.erase(0, curloc);
      curloc=0;
      normalize_cursor();
      text_changed(string(text));
      vscreen_update();
      return true;
    }
  else if(history && bindings->key_matches(ch, "HistoryPrev"))
    {
      if(history->size()==0)
	return true;

      if(!using_history)
	{
	  using_history=true;
	  history_loc=history->size()-1;
	  pre_history_text=text;
	}
      else if(history_loc>0)
	--history_loc;
      else
	// Break out
	return true;

      text=(*history)[history_loc];
      curloc=text.size();
      startloc=0;
      normalize_cursor();
      text_changed(string(text));
      vscreen_update();

      return true;
    }
  else if(history && bindings->key_matches(ch, "HistoryNext"))
    {
      if(history->size()==0 || !using_history)
	return true;

      if(history_loc>=history->size()-1)
	{
	  using_history=false;
	  history_loc=0;
	  text=pre_history_text;
	  pre_history_text="";
	  curloc=text.size();
	  startloc=0;
	  normalize_cursor();
	  text_changed(string(text));
	  vscreen_update();

	  // FIXME: store the pre-history edit and restore that.
	  return true;
	}

      if(history_loc>=0)
	{
	  ++history_loc;
	  text=(*history)[history_loc];
	  curloc=text.size();
	  startloc=0;
	  normalize_cursor();
	  text_changed(string(text));
	  vscreen_update();

	  return true;
	}
      else
	return true;
    }
  else if(ch>255)
    return vscreen_widget::handle_char(ch);
  else if(ch=='\t') // HACK
    return false;
  else
    {
      text.insert(curloc++, 1, (char) ch);
      normalize_cursor();
      text_changed(string(text));
      vscreen_update();
      return true;
    }
}

void vs_editline::paint()
{
  int width=getmaxx();

  string todisp(prompt+text, startloc, width);

  mvaddstr(0, 0, todisp.c_str());
}

void vs_editline::dispatch_mouse(short id, int x, int y, int z, mmask_t bstate)
{
  string::size_type mouseloc=x+startloc;

  if(mouseloc>=prompt.size())
    {
      mouseloc-=prompt.size();

      if(mouseloc<=text.size())
	curloc=mouseloc;
      else if(mouseloc>text.size())
	curloc=text.size();
    }
  else
    return; // Break out

  vscreen_update();
}

void vs_editline::add_to_history(std::string s,
				 history_list *lst)
{
  assert(lst);

  if(lst->empty() || lst->back()!=s)
    lst->push_back(s);
}

void vs_editline::add_to_history(std::string s)
{
  if(history)
    add_to_history(s, history);
}

void vs_editline::reset_history()
{
  pre_history_text="";
  using_history=false;
  history_loc=0;
}

void vs_editline::set_text(string _text)
{
  text=_text;
  if(curloc>text.size())
    curloc=text.size();
  text_changed(string(text));
  vscreen_update();
}

void vs_editline::init_bindings()
{
  bindings=new keybindings(&global_bindings);

  bindings->set("Left", KEY_LEFT);
  bindings->set("Right", KEY_RIGHT);
  // Override these for the case where left and right have multiple bindings
  // in the global keymap
}
