/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.IO;
using System.Threading;

using TagLib;

namespace Cowbell.Base
{
	public class Song : IComparable
	{
		/* public events */
		public event SongChangedHandler Changed;

		/* public properties */
		public string Filename {
			get { return filename; }
		}

		/**
		 * Returns filesize in megabytes
		 */
		public double Size {
			get {
				double size = 0;
				try {
					System.IO.FileInfo info = new System.IO.FileInfo (filename);
					size = (double)info.Length / 1048576;
				} catch { }

				return size;
			}
		}
		
		public string Title {
			get { return title; }
			set {
				if (title == value) {
					return;
				}

				title = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public string Artist {
			get { return artist; }
			set {
				if (artist == value) {
					return;
				}

				artist = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public string Album {
			get { return album; }
			set {
				if (album == value) {
					return;
				}

				album = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public string Genre {
			get { return genre; }
			set {
				if (genre == value) {
					return;
				}

				genre = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public uint TrackNumber {
			get { return tracknumber; }
			set {
				if (tracknumber == value) {
					return;
				}

				tracknumber = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public uint Year {
			get { return year; }
			set {
				if (year == value) {
					return;
				}

				year = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public string Comment {
			get { return comment; }
			set {
				if (comment == value) {
					return;
				}

				comment = value;
				
				if (Changed != null) {
					Changed (this);
				}
			}
		}

		public bool Exists {
			get {
				if (filename != String.Empty) {
					return System.IO.File.Exists (filename);
				}
				return false;
			}
		}
		
		public string ArtistAlbumStamp {
			get {
				if (Artist != null && Album != null
				    && Artist != String.Empty && Album != String.Empty) {
					return Artist.ToLower () + "+" + Album.ToLower ();
				}
				return "0";
			}
		}

		/*
		 * TagLib Audio Properties
		 */ 
		public TimeSpan Length {
			get { return new TimeSpan (0, 0, file.AudioProperties.Length); }
		}

		public int Bitrate {
			get { return file.AudioProperties.Bitrate; }
		}
		
		public int SampleRate {
			get { return file.AudioProperties.SampleRate; }
		}

		public int Channels {
			get { return file.AudioProperties.Channels; }
		}

		/* public methods */
		public Song ()
		{
			filename = "";
			file = null;

			title = String.Empty;
			artist = String.Empty;
			album = String.Empty;
			year = 0;
			tracknumber = 0;
			genre = String.Empty;
			comment = String.Empty;
		}

		public Song (string name) : this ()
		{
			filename = name;
			file = new TagLib.File (filename);

			PopulateFromFile ();
			SanitizeTags ();
		}

		public void Revert ()
		{
			if (Exists) {
				PopulateFromFile ();
				SanitizeTags ();
			}
		}

		public void Save ()
		{
			file.Tag.Title = title;
			file.Tag.Artist = artist;
			file.Tag.Album = album;
			file.Tag.Year = year;
			file.Tag.Track = tracknumber;
			file.Tag.Genre = genre;
			file.Tag.Comment = comment;

			file.Save ();
		}

		public void Rename (string newname)
		{
			if (System.IO.File.Exists (newname) 
			    && !System.IO.File.Exists (filename)) {
				return;
			}

			System.IO.File.Move (filename, newname);

			filename = newname;
			file = new TagLib.File (filename);
		}

		public void PathToTitle (string path)
		{
			string title;
			string[] array;
			char[] delim = {'-'};

			title = Path.GetFileNameWithoutExtension (path);
			title = title.Replace ('_', ' ');
			
			// Clobber everything after the dash in a filename
			array = title.Split (delim, 2);

			try {
				title = array[1].Trim ();
			} catch {
				title = Path.GetFileNameWithoutExtension (path);
			}

			Title = Utils.ToTitleCase (title);
		}

		public override string ToString ()
		{
			string ret = String.Empty;
			if (Exists) {
				ret += "    Filename: " + Filename + "\n";
				ret += "        Size: " + Size + " mb\n";
			}

			ret += "       Title: " + Title + "\n";
			ret += "      Artist: " + Artist + "\n";
			ret += "       Album: " + Album + "\n";
			ret += "       Genre: " + Genre + "\n";
			ret += " TrackNumber: " + TrackNumber + "\n";
			ret += "        Year: " + Year + "\n";
			ret += "     Comment: " + Comment + "\n";

			if (Exists) {
				ret += "      Length: " + Length + " m\n";
				ret += "     Bitrate: " + Bitrate + " kbps\n";
				ret += " Sample Rate: " + SampleRate + " kHz\n";
				ret += "    Channels: " + Channels + "\n";
			}

			return ret;
		}	

		public int CompareTo (object o)
		{
			Song s = (Song)o;
			int ret = 0;

			if (TrackNumber < s.TrackNumber) {
				ret = -1;
			} else if (TrackNumber > s.TrackNumber) {
				ret = 1;
			}

			return ret;
		}

		/* private fields */
		TagLib.File file;
		string filename, title, artist, album, genre, comment;
		uint tracknumber, year;

		/* private methods */
		private void PopulateFromFile ()
		{
			Title = file.Tag.Title;
			Artist = file.Tag.Artist;
			Album = file.Tag.Album;
			Genre = file.Tag.Genre;
			TrackNumber = file.Tag.Track;
			Year = file.Tag.Year;
			Comment = file.Tag.Comment;
		}

		private void SanitizeTags ()
		{
			/* Split Artist and Title if specified in the Title field */
			if (Title != null) {
				int index = Title.IndexOf (" / "); 
				if (index == -1) {
					index = Title.IndexOf (" - "); 
				}

				if (index > 0) {
					Artist = Title.Substring (0, index);
					Title = Title.Substring (index + 2);
				}
			}

			/* Strip out leading and trailing whitespace */
			Title = Title.Trim ();
			Artist = Artist.Trim ();
			Album = Album.Trim ();
			Genre = Genre.Trim ();
			Comment = Comment.Trim ();
		}
	}
}
