#
#

# Copyright (C) 2009, 2010 Google Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.


"""Ganeti configuration daemon

Ganeti-confd is a daemon to query master candidates for configuration values.
It uses UDP+HMAC for authentication with a global cluster key.

"""

# pylint: disable=C0103
# C0103: Invalid name ganeti-confd

import os
import sys
import logging
import time

try:
  # pylint: disable=E0611
  from pyinotify import pyinotify
except ImportError:
  import pyinotify

from optparse import OptionParser

from ganeti import asyncnotifier
from ganeti import confd
from ganeti.confd import server as confd_server
from ganeti import constants
from ganeti import errors
from ganeti import daemon
from ganeti import netutils


class ConfdAsyncUDPServer(daemon.AsyncUDPSocket):
  """The confd udp server, suitable for use with asyncore.

  """
  def __init__(self, bind_address, port, processor):
    """Constructor for ConfdAsyncUDPServer

    @type bind_address: string
    @param bind_address: socket bind address
    @type port: int
    @param port: udp port
    @type processor: L{confd.server.ConfdProcessor}
    @param processor: ConfdProcessor to use to handle queries

    """
    family = netutils.IPAddress.GetAddressFamily(bind_address)
    daemon.AsyncUDPSocket.__init__(self, family)
    self.bind_address = bind_address
    self.port = port
    self.processor = processor
    self.bind((bind_address, port))
    logging.debug("listening on ('%s':%d)", bind_address, port)

  # this method is overriding a daemon.AsyncUDPSocket method
  def handle_datagram(self, payload_in, ip, port):
    try:
      query = confd.UnpackMagic(payload_in)
    except errors.ConfdMagicError, err:
      logging.debug(err)
      return

    answer = self.processor.ExecQuery(query, ip, port)
    if answer is not None:
      try:
        self.enqueue_send(ip, port, confd.PackMagic(answer))
      except errors.UdpDataSizeError:
        logging.error("Reply too big to fit in an udp packet.")


class ConfdConfigurationReloader(object):
  """Logic to control when to reload the ganeti configuration

  This class is able to alter between inotify and polling, to rate-limit the
  number of reloads. When using inotify it also supports a fallback timed
  check, to verify that the reload hasn't failed.

  """
  def __init__(self, processor, mainloop):
    """Constructor for ConfdConfigurationReloader

    @type processor: L{confd.server.ConfdProcessor}
    @param processor: ganeti-confd ConfdProcessor
    @type mainloop: L{daemon.Mainloop}
    @param mainloop: ganeti-confd mainloop

    """
    self.processor = processor
    self.mainloop = mainloop

    self.polling = True
    self.last_notification = 0

    # Asyncronous inotify handler for config changes
    cfg_file = constants.CLUSTER_CONF_FILE
    self.wm = pyinotify.WatchManager()
    self.inotify_handler = asyncnotifier.SingleFileEventHandler(self.wm,
                                                                self.OnInotify,
                                                                cfg_file)
    notifier_class = asyncnotifier.ErrorLoggingAsyncNotifier
    self.notifier = notifier_class(self.wm, self.inotify_handler)

    self.timer_handle = None
    self._EnableTimer()

  def OnInotify(self, notifier_enabled):
    """Receive an inotify notification.

    @type notifier_enabled: boolean
    @param notifier_enabled: whether the notifier is still enabled

    """
    current_time = time.time()
    time_delta = current_time - self.last_notification
    self.last_notification = current_time

    if time_delta < constants.CONFD_CONFIG_RELOAD_RATELIMIT:
      logging.debug("Moving from inotify mode to polling mode")
      self.polling = True
      if notifier_enabled:
        self.inotify_handler.disable()

    if not self.polling and not notifier_enabled:
      try:
        self.inotify_handler.enable()
      except errors.InotifyError:
        self.polling = True

    try:
      reloaded = self.processor.reader.Reload()
      if reloaded:
        logging.info("Reloaded ganeti config")
      else:
        logging.debug("Skipped double config reload")
    except errors.ConfigurationError:
      self.DisableConfd()
      self.inotify_handler.disable()
      return

    # Reset the timer. If we're polling it will go to the polling rate, if
    # we're not it will delay it again to its base safe timeout.
    self._ResetTimer()

  def _DisableTimer(self):
    if self.timer_handle is not None:
      self.mainloop.scheduler.cancel(self.timer_handle)
      self.timer_handle = None

  def _EnableTimer(self):
    if self.polling:
      timeout = constants.CONFD_CONFIG_RELOAD_RATELIMIT
    else:
      timeout = constants.CONFD_CONFIG_RELOAD_TIMEOUT

    if self.timer_handle is None:
      self.timer_handle = self.mainloop.scheduler.enter(
        timeout, 1, self.OnTimer, [])

  def _ResetTimer(self):
    self._DisableTimer()
    self._EnableTimer()

  def OnTimer(self):
    """Function called when the timer fires

    """
    self.timer_handle = None
    reloaded = False
    was_disabled = False
    try:
      if self.processor.reader is None:
        was_disabled = True
        self.EnableConfd()
        reloaded = True
      else:
        reloaded = self.processor.reader.Reload()
    except errors.ConfigurationError:
      self.DisableConfd(silent=was_disabled)
      return

    if self.polling and reloaded:
      logging.info("Reloaded ganeti config")
    elif reloaded:
      # We have reloaded the config files, but received no inotify event.  If
      # an event is pending though, we just happen to have timed out before
      # receiving it, so this is not a problem, and we shouldn't alert
      if not self.notifier.check_events() and not was_disabled:
        logging.warning("Config file reload at timeout (inotify failure)")
    elif self.polling:
      # We're polling, but we haven't reloaded the config:
      # Going back to inotify mode
      logging.debug("Moving from polling mode to inotify mode")
      self.polling = False
      try:
        self.inotify_handler.enable()
      except errors.InotifyError:
        self.polling = True
    else:
      logging.debug("Performed configuration check")

    self._EnableTimer()

  def DisableConfd(self, silent=False):
    """Puts confd in non-serving mode

    """
    if not silent:
      logging.warning("Confd is being disabled")
    self.processor.Disable()
    self.polling = False
    self._ResetTimer()

  def EnableConfd(self):
    self.processor.Enable()
    logging.warning("Confd is being enabled")
    self.polling = True
    self._ResetTimer()


def CheckConfd(_, args):
  """Initial checks whether to run exit with a failure.

  """
  if args: # confd doesn't take any arguments
    print >> sys.stderr, ("Usage: %s [-f] [-d] [-b ADDRESS]" % sys.argv[0])
    sys.exit(constants.EXIT_FAILURE)

  # TODO: collapse HMAC daemons handling in daemons GenericMain, when we'll
  # have more than one.
  if not os.path.isfile(constants.CONFD_HMAC_KEY):
    print >> sys.stderr, "Need HMAC key %s to run" % constants.CONFD_HMAC_KEY
    sys.exit(constants.EXIT_FAILURE)

  # TODO: once we have a cluster param specifying the address family
  # preference, we need to check if the requested options.bind_address does not
  # conflict with that. If so, we might warn or EXIT_FAILURE.


def PrepConfd(options, _):
  """Prep confd function, executed with PID file held

  """
  # TODO: clarify how the server and reloader variables work (they are
  # not used)

  # pylint: disable=W0612
  mainloop = daemon.Mainloop()

  # Asyncronous confd UDP server
  processor = confd_server.ConfdProcessor()
  try:
    processor.Enable()
  except errors.ConfigurationError:
    # If enabling the processor has failed, we can still go on, but confd will
    # be disabled
    logging.warning("Confd is starting in disabled mode")

  server = ConfdAsyncUDPServer(options.bind_address, options.port, processor)

  # Configuration reloader
  reloader = ConfdConfigurationReloader(processor, mainloop)

  return mainloop


def ExecConfd(options, args, prep_data): # pylint: disable=W0613
  """Main confd function, executed with PID file held

  """
  mainloop = prep_data
  mainloop.Run()


def Main():
  """Main function for the confd daemon.

  """
  parser = OptionParser(description="Ganeti configuration daemon",
                        usage="%prog [-f] [-d] [-b ADDRESS]",
                        version="%%prog (ganeti) %s" %
                        constants.RELEASE_VERSION)

  daemon.GenericMain(constants.CONFD, parser, CheckConfd, PrepConfd, ExecConfd)
