/*
 * system_environment.cpp - namespace systemEnvironment, providing interface for transparent use of operating-system-dependent functions
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 * Copyright (c) 2005 Patrick William <italc@patrick-william.de>
 * Copyright (c) 2005 Jan Seiffert
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qfileinfo.h>

#include "system_environment.h"
#include "italc_rfb_ext.h"
#include "client.h"



namespace systemEnvironment
{

#ifdef COMPILE_LINUX

// environment variable name to look for a given ip-addr (should not be neccessary, because we're getting all ifs from system)
#define ITALC_MASTER_IP_ENV "ITALC_MASTER_IP"


#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#include <net/if.h>
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif


inline QString getStdOut( const QString & _cmd )
{
	FILE * f = popen( _cmd, "r" );
	const int BUFSIZE = 128;
	char buf[BUFSIZE];
	QString out;
	while( fgets( buf, BUFSIZE, f ) != NULL )
	{
		out += buf;
	}
	pclose( f );

	return( out );
}




#define stringify2(x) #x
#define stringify(x) stringify2(x)
#define debug_pr(x) __FILE__ ", " stringify(__LINE__) ": " x


QMap<QString, QString> localIPAddresses( void )
{
	QMap<QString, QString> ip_addr_map;

	// get socket-fd
	int fd = socket(PF_INET, SOCK_DGRAM, 0);
	if( fd == -1 )
	{
		perror( debug_pr( "getting a socket" ) );
		return( ip_addr_map );
	}

	// get list if interface-names
	struct if_nameindex * our_inter = if_nameindex();
	if( !our_inter )
	{
		perror( debug_pr( "getting interface list" ) );
		close( fd );
		return( ip_addr_map );
	}

	struct if_nameindex * w_ptr_if = our_inter;

	// loop through all interfaces
	while( w_ptr_if->if_name )
	{
		struct ifreq ifr;
		// copy iface-name in struct
		strcpy( ifr.ifr_name, w_ptr_if->if_name );
		// set address-family
		ifr.ifr_addr.sa_family = AF_INET;
	
		// this ioctl assigns an interface-name an address (can be a hostname or an IP)
		// defined in: Linux: /usr/include/linux/sockios.h - Solaris: /usr/include/sys/sockios.h
		int ioctl_ret = ioctl( fd, SIOCGIFADDR, &ifr );

		// if ioctl successful, add IP-address to our map
		if( 0 <= ioctl_ret )
		{
			ip_addr_map[w_ptr_if->if_name] = inet_ntoa( ( ( struct sockaddr_in* ) &ifr.ifr_addr )->sin_addr );
		}
		else
		{
			perror( debug_pr( "getting interface addr" ) );
		}
		w_ptr_if++;
	}

	if_freenameindex( our_inter );
	close( fd );

	// nothing found yet?? (should never occur, but who knows...)
	if( ip_addr_map.count() == 0 )
	{
	
#ifndef HOST_NAME_MAX
// according to Single Unix Specification v2
# define HOST_NAME_MAX 255
#endif
#ifndef INET6_ADDRSTRLEN
// according to man inet_pton on Solaris(1999)
# define INET6_ADDRSTRLEN 46
#endif
		// ask the environment for a given ip-addr
		char *tmp_hostname = getenv( ITALC_MASTER_IP_ENV );
		if( tmp_hostname == NULL )
		{
			// no given ip-addr in environment; try with hostname instead
			tmp_hostname = (char *) alloca( HOST_NAME_MAX );
			if( !gethostname( tmp_hostname, HOST_NAME_MAX ) )
			{
					// *always* terminate the string with NUL at
					// the last array-elem for security-reasons,
					// if implementation doesn't do it for us
				tmp_hostname[ HOST_NAME_MAX - 1 ] = '\0';
			}
			else
			{
				// nothing of the above succeeded
				perror( "gethostname (now using 127.0.0.1)" );
				tmp_hostname = "127.0.0.1";
			}
		}

		// try to resolve
		struct hostent *host = gethostbyname( tmp_hostname );
		if( host )
		{
			// resolving succeeded
			char hostip[INET6_ADDRSTRLEN];

			// make a string out of the numerical ip-addr
			if( inet_ntop( host->h_addrtype, host->h_addr, hostip, sizeof(hostip) ) )
			{
				ip_addr_map["eth0"] = hostip;
			}
			else
			{
				//Ok, did not work
				perror( "inet_ntop hostname (now using hostname directly)" );
				ip_addr_map["eth0"] = tmp_hostname;
			}
		}
		else
		{
			// resolving failed
			// gethostbyname() uses h_errno instead of errno; so we can't
			// use perror() and do the output-handling by ourselfs
			char *error_txt = NULL;

			switch( h_errno )
			{
				case HOST_NOT_FOUND:
					error_txt = "HOST_NOT_FOUND";
					break;
				// case NO_DATA: // is duplicate
				case NO_ADDRESS:
					error_txt = "NO_ADDRESS / NO_DATA";
					break;
				case NO_RECOVERY:
					error_txt = "NO_RECOVERY";
					break;
				case TRY_AGAIN:
					error_txt = "TRY_AGAIN";
					break;
				default:
					error_txt = "unknown";
					break;
			}
			fprintf( stderr, "gethostbyname failed (now using %s instead of addr): %s\n", tmp_hostname, error_txt );
			ip_addr_map["eth0"] = tmp_hostname;
		}
	}

/*	// ok, last try...
	if( ip_addr_map.count() == 0 )
	{
		ip_addr_map.insertItem( getStdOut( "hostname -i" ).remove( '\n' ) );
	}*/

	return( ip_addr_map );
}




QString localUser( void )
{
	return( ( getenv( "USER" ) != NULL )? getenv( "USER" ) : "" );
}




void execBg( const QString & _cmds )
{
	system( "echo `" + _cmds + "` > /dev/null &" );
}




void execInTerminal( const QString & _cmds )
{
	//system( "konsole -e " + _cmds + " &" );
	system( "xterm -e " + _cmds + " &" );
}




void startClientViewer( const QString & _ip, const QString & _user )
{
	system( "icv " + _ip + " " + _user + " &" );
}



#ifdef USE_PYTHON_C_API
#include <python2.3/Python.h>
#endif


void executePythonCode( const QString & _pc )
{
#ifdef USE_PYTHON_C_API
	Py_Initialize();
	PyRun_SimpleString( _pc.ascii() );
	Py_Finalize();
#else
	system( "python <<< \""+_pc+"\"" );
#endif
}




void distributeFile( const QString & _file, client * _client )
{
	system( "cp -f "+_file+" ~/PUBLIC/" );
	QString destfile = "~"+localUser()+"/PUBLIC/"+QFileInfo( _file ).fileName();

	_client->execCmdsIRFB( "mkdir -p ~" + _client->user() + "/material ; cp -f "+destfile+" ~" + _client->user() + "/material/" );
}




void collectFiles( const QString & _files, const QString & _user )
{
	QString dirname = client::tr( "collected-files" );
	QString destdir = "~/" + dirname + "/"+_user+"/";
	// let bash do the work for us :-)
	system( "mkdir -p "+destdir );
	system( "cp -f ~"+_user+"/PUBLIC/"+_files+" "+destdir );
	system( "for i in "+destdir+"* ; do mv $i ~/" + dirname + "/`echo $i | rev | cut -d \"/\" -f1 | cut -d \".\" -f2- | rev`_"+
		_user+".`echo $i | rev | cut -d \".\" -f1 | rev` ; done" );
	system( "rm -rf "+destdir );
}



namespace demoServer
{

void start()
{
#ifdef X11VNC_07
	system( "ivs -q -noauth -allowremote -allow none -viewonly -cursor X -rfbport "DEMO_PORT" &" );
#else
	system( "ivs -q -noauth -viewonly -X -rfbport "DEMO_PORT" &" );
#endif
}



static QString allowed_clients;

void allowClient( const QString & _ip )
{
#ifdef X11VNC_07
	if( !allowed_clients.contains( _ip ) )
	{
		allowed_clients += _ip + ",";
	}
	system( "ivs -remote allow:" + allowed_clients );
#endif
}





void denyClient( const QString & _ip )
{
#ifdef X11VNC_07
	allowed_clients.remove( _ip + "," );

	if( allowed_clients.length() )
	{
		system( "ivs -remote allow:" + allowed_clients );
	}
	else
	{
		system( "ivs -remote allow:none" );
	}
#endif
}



void stop()
{
#ifdef X11VNC_07
	system( "ivs -remote stop" );
#else
	system( "killall ivs" );
#endif
}


} // end of namespace demoServer



bool hostAvailable( const QString & _host_ip )
{
	QString opts = " -c1 "
#if PING_KNOWS_OPTION_W
			"-w1 "
#endif
#if PING_KNOWS_OPTION_BIG_W
			"-W1 "
#endif
		;
	return( system( "ping " + _host_ip + opts + "2>&1 > /dev/null" ) == 0 );
}




QString realUserName( const QString & _user )
{
	QString out = getStdOut( "finger "+_user+" | grep Name | cut -f 3 -d \":\" | sed -e \"s/+/\\ /g\" | cut -c2-" );
	return( out.remove( out.find( '\n' ), 1 ) );
}


#else

#error There is no systemEnvironment-implementation for your platform!

#endif


} // end of namespace systemEnvironment
