//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>
#include <assert.h>

#include "silcprotocol.h"
#include "silcaccount.h"
#include "silcbuddycontact.h"
#include "silcchannelcontact.h"
#include "silcbuddycontactinfowidget.h"
#include "silcfiletransfer.h"
#include "silcbuddyattributes.h"

#include <kopetemetacontact.h>
#include <kopetechatsession.h>
#include <kopeteuiglobal.h>

#include <kmimemagic.h>
#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>

// sorry for this hack, unfortunately we need it for
// the macros of recent libsilc to work ...
typedef unsigned char SilcUInt8;


SilcBuddyContact::SilcBuddyContact(SilcAccount *account,
				   const QString &nick, 
                                   const QString &fingerprint,
				   Kopete::MetaContact *meta,
				   const QString &icon)
  : SilcContact(account, QString("@%1").arg(fingerprint), meta, icon),
    _channels(), _clientEntries(), _fpTrusted(false), _watched(false), 
    _allowRichText(false)
{
  actionIsOp = NULL;
  actionKick = NULL;

  attributes = new SilcBuddyAttributes(this);

  setNickName(nick);
  setFileCapable(true);

  QObject::connect
    (this, SIGNAL(onlineStatusChanged(Kopete::Contact *,
                                      const Kopete::OnlineStatus &,
                                      const Kopete::OnlineStatus &)),
     this, SLOT(slotOnlineStatusChanged(Kopete::Contact *,
                                        const Kopete::OnlineStatus &,
                                        const Kopete::OnlineStatus &)));

  QObject::connect
    (this, SIGNAL(propertyChanged(Kopete::Contact *, const QString &,
				  const QVariant &, const QVariant &)),
     this, SLOT(slotPropertyChanged(Kopete::Contact *, const QString &,
				    const QVariant &, const QVariant &)));
}
 
SilcBuddyContact::~SilcBuddyContact() {
  this->watchme(false);
  delete attributes;
}

void
SilcBuddyContact::slotSendMessage(Kopete::Message &msg,
				  Kopete::ChatSession *session)
{
  if(session != manager()) return;

  SilcAccount *account = static_cast<SilcAccount *>(this->account());

  if(! account->conn()) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry, 
       i18n("Unable to send this message now. The protocol is currently "
	    "offline and does not support offline sending."),
       i18n( "User is Not Reachable"));
    return;
  }

  if(onlineStatus() == SilcProtocol::protocol()->statusDetached) {
    KMessageBox::queuedMessageBox
      (Kopete::UI::Global::mainWidget(), KMessageBox::Sorry,
       i18n("This message cannot be sent right now, since the remote client "
	    "is currently detached and the protocol does not support "
	    "offline sending."),
       i18n("User is Not Reachable"));
    return;
  }

  // get plain text message ...
  SilcTK::SilcMessageFlags flags = SILC_MESSAGE_FLAG_UTF8;
  unsigned char *buf = NULL;
  SilcTK::SilcUInt32 buflen = 0;
  QCString plaintext;

  if(account->signPrivateMessages())
    flags |= SILC_MESSAGE_FLAG_SIGNED;

  if(allowRichText()) {
    SilcTK::SilcMime mime = getMessageAsMime(msg);
    buf = SilcTK::silc_mime_encode(mime, &buflen);
    SilcTK::silc_mime_free(mime);

    flags |= SILC_MESSAGE_FLAG_DATA;
  }
  else {
    plaintext = msg.plainBody().utf8();
    buf = (unsigned char *) (const char *) plaintext;
    buflen = plaintext.length();

    // use of rich text is forbidden, reset message to plain
    // (so our channel log doesn't show any markup as well)
    msg.setBody(msg.plainBody());
  }

  prettyPrintMessage(msg, flags);

  assert(clientEntry());

  // pass message to libsilc ...
  SilcTK::silc_client_send_private_message
    (account->client(), account->conn(), clientEntry(), flags,
     account->sha1hash, buf, buflen);     

  // append message locally ...
  session->appendMessage(msg);
  session->messageSucceeded();
}



void 
SilcBuddyContact::silc_private_message(SilcTK::SilcClient client,
				       SilcTK::SilcClientConnection,
				       SilcTK::SilcClientEntry sender,
				       SilcTK::SilcMessagePayload payload,
				       SilcTK::SilcMessageFlags flags,
				       const unsigned char *message,
				       SilcTK::SilcUInt32 message_len)
{
  SilcAccount *account = static_cast<SilcAccount *>(client->application);
  SilcBuddyContact *buddy = (SilcBuddyContact *) sender->context;
  
  if(! buddy)
    buddy = account->contactManager()->createBuddy
      (sender->nickname, NULL, sender);

  if(! buddy) {
    std::cerr << "unable to allocate new buddy instance" << std::endl;
    return;
  }

  // If the messages is digitally signed, verify it, if possible.
  SignatureStatus sigstat = Unknown;
  if (flags & SILC_MESSAGE_FLAG_SIGNED)
    sigstat = buddy->verifySignature(payload);

  // make sure there's a ChatSession, then add ourself ...
  Kopete::ChatSession *session = buddy->manager(Kopete::Contact::CanCreate);
  session->addContact(buddy, Kopete::OnlineStatus::Online);

  // convert Utf8 stuff depending on whether SILC_MESSAGE_FLAG_UTF8 is set
  QString text;
  if(flags & SILC_MESSAGE_FLAG_UTF8)
    text = QString::fromUtf8((const char *) message, message_len);
  else if(flags & SILC_MESSAGE_FLAG_DATA);
  else
    text = QString::fromLatin1((const char *) message, message_len);

  Kopete::Message msg;
  if(flags & SILC_MESSAGE_FLAG_NOTICE)
    msg = Kopete::Message(buddy, account->myself(), QString("%1 -*- %2")
			  .arg(buddy->nickName()).arg(text),
			  Kopete::Message::Internal,
			  Kopete::Message::PlainText, QString::null,
			  Kopete::Message::TypeAction);

  else if(flags & SILC_MESSAGE_FLAG_DATA) {
    /* SilcMimeMessage */
    QStringList *filenames;
    SilcTK::SilcMime tmp = SilcTK::silc_mime_decode(NULL, message, message_len);
    /* Assemble mime partials */
    SilcTK::SilcMime mime = buddy->mime_asm(tmp);
    if (!mime) return;

    QString type = SilcTK::silc_mime_get_field(mime, "Content-Type");
    if(type.isEmpty()) goto mimeout;

    if (type.left(21).compare("multipart/alternative") == 0) {
      msg = Kopete::Message(buddy, account->myself(), 
			    QString::null,
			    Kopete::Message::Inbound,
			    Kopete::Message::PlainText, QString::null,
			    Kopete::Message::TypeNormal);
      buddy->mimeAlternateToMsg(msg, mime, buddy->allowRichText());
      session->appendMessage(msg);
    }
    else {
      filenames = buddy->saveMime(mime);
      for(QStringList::Iterator it = filenames->begin(); 
	  it != filenames->end(); ++it ) {
        msg = Kopete::Message(buddy, account->myself(),
                              buddy->mimeDisplayMessage(*it),
                              Kopete::Message::Inbound,
                              Kopete::Message::RichText, QString::null,
                              Kopete::Message::TypeNormal);

        prettyPrintMessage(msg, flags, sigstat);
        session->appendMessage(msg);
      }
      delete filenames;
    }

mimeout:
    SilcTK::silc_mime_free(mime);
    return;
  }
  else
    msg = Kopete::Message(buddy, account->myself(), text, 
			  Kopete::Message::Inbound,
			  Kopete::Message::PlainText, QString::null,
			  (flags & SILC_MESSAGE_FLAG_ACTION)
			  ? Kopete::Message::TypeAction
			  : Kopete::Message::TypeNormal);

  prettyPrintMessage(msg, flags, sigstat);
  session->appendMessage(msg);

  // make sure there's a view associated with this ChatSession ...
  buddy->view();

  // force online status online, in case buddy was marked offline - we just got
  // a message, therefore he or she needs to be online :-)
  //
  // FIXME, this shouldn't be necessary actually, implement some slotConnected
  if(buddy->onlineStatus() == SilcProtocol::protocol()->statusOffline)
    buddy->setOnlineStatus(SilcProtocol::protocol()->statusOnline);
}


void 
SilcBuddyContact::serialize(QMap<QString, QString> &serializedData,
			    QMap<QString, QString>&)
{
  serializedData["fpTrusted"] = fpTrusted() ? "yes" : "no";
  serializedData["allowRichText"] = allowRichText() ? "yes" : "no";
}

/**
 * note: if we trust some fingerprint we gotta have it's key
 */

void 
SilcBuddyContact::setFpTrusted(bool trust)
{
  if(trust) {
    if( !havePublicKey() ) {
      account()->sendSilcCommand( QString("GETKEY %1").arg(nickName()) );
    }
  }
  _fpTrusted = trust;
}

QString
SilcBuddyContact::convFingerprint(const char *fp)
{
  QString qfp = QString::null;
  //asserting 20 byte of valid data
  for(int i = 0; i<20 ; fp ++) {
    qfp += ((*fp >> 4) & 0xF) > 9 
      ? ((*fp >> 4) & 0xF) + 'A' - 10
      : ((*fp >> 4) & 0xF) + '0';
    qfp += (*fp & 0xF) > 9 ? (*fp & 0xF) + 'A' - 10 : (*fp & 0xF) + '0';
    if((++i % 2) == 0 && i!=20 ) qfp += ':'; //&& fp[1]
    if(i == 10) qfp += ':';
  }

  return qfp;
}


void 
SilcBuddyContact::setClientEntry(SilcTK::SilcClientEntry e)
{
  int count;
  for (count = 0; count < _clientEntries.count(); count++)
    SilcTK::silc_client_unref_client(account()->client(), 
                                     account()->conn(),
                                     _clientEntries[count]);
  _clientEntries.clear();

  if(! e) return;
  this->addClientEntry(e);
}


void
SilcBuddyContact::addClientEntry(SilcTK::SilcClientEntry e)
{
  assert(e);

  e = SilcTK::silc_client_ref_client(account()->client(), 
                                     account()->conn(), e);

  _clientEntries.push_back(e);
  e->context = this;
}

void
SilcBuddyContact::removeClientEntry(SilcTK::SilcClientEntry e)
{
  assert(e);

  _clientEntries.remove(e);
  e->context = NULL;

  SilcTK::silc_client_unref_client(account()->client(),
                                   account()->conn(), e);

  if(_clientEntries.empty()) {
    // last client entry has been dropped, mark buddy as offline ...
    this->setOnlineStatus(SilcProtocol::protocol()->statusOffline);
  }
}


const SilcTK::SilcClientEntry
SilcBuddyContact::clientEntry(SilcChannelContact *ch) const
{
  SilcTK::SilcChannelEntry che = ch->channelEntry();
  if(! che) return NULL;

  for(unsigned int i = 0; i < this->clientEntriesCount(); i ++) {
    SilcTK::SilcClientEntry ce = this->clientEntry(i);
    SilcTK::SilcChannelUser cu = SilcTK::silc_client_on_channel(che, ce);
    if(cu) return ce;
  }
  
  return NULL;
}

SilcTK::SilcClientEntry 
SilcBuddyContact::clientEntry(SilcChannelContact *ch)
{
  SilcTK::SilcChannelEntry che = ch->channelEntry();
  if(! che) return NULL;

  for(unsigned int i = 0; i < this->clientEntriesCount(); i ++) {
    SilcTK::SilcClientEntry ce = this->clientEntry(i);
    SilcTK::SilcChannelUser cu = SilcTK::silc_client_on_channel(che, ce);
    if(cu) return ce;
  }

  return NULL;
}


void
SilcBuddyContact::slotIsOp(void)
{
  if(! activeManager) return;

  Kopete::ContactPtrList members = activeManager->members();
  SilcChannelContact *ch = static_cast<SilcChannelContact *>(members.first());

  ch->setOp(this, actionIsOp->isChecked());
}


void
SilcBuddyContact::slotKick(void)
{

  if(! activeManager) return;

  Kopete::ContactPtrList members = activeManager->members();
  SilcChannelContact *ch = static_cast<SilcChannelContact *>(members.first());

  ch->kick(this);
}

QPtrList<KAction> *
SilcBuddyContact::customContextMenuActions(Kopete::ChatSession *manager)
{
  // store a ptr. to the active ChatSession, we need this, to remember
  // for which channel we need to supply or drop operator rights, etc. ...
  activeManager = manager; 

  QPtrList<KAction> *actions = new QPtrList<KAction>();

  if(! actionIsOp)
    actionIsOp = new KToggleAction(i18n("&Operator"), KShortcut::null(), 
				   this, SLOT(slotIsOp()), this);
  
  if (! actionKick)
      actionKick = new KAction(i18n("&Kick from Channel"),
                                    KShortcut::null(), this,
                                    SLOT(slotKick()), this);

  SilcBuddyContact *me = static_cast<SilcBuddyContact *>(account()->myself());
  SilcChannelContact *channel = NULL;
  if(manager) {
    Kopete::ContactPtrList members = manager->members();

    if(! strcmp(members.first()->className(), "SilcChannelContact"))
      channel = static_cast<SilcChannelContact *>(members.first());
  }

  // add operator entry ...
  actionIsOp->setEnabled(channel && channel->isOp(me));
  actionIsOp->setChecked(channel && channel->isOp(this));
  actions->append(actionIsOp);

  actionKick->setEnabled(channel && channel->isOp(me) 
                        && me != this);
  actions->append(actionKick);
  
  return actions;
}


void
SilcBuddyContact::slotOnlineStatusChanged(Kopete::Contact *,
                                          const Kopete::OnlineStatus &status,
                                          const Kopete::OnlineStatus & /*old*/)
{
  if(status == SilcProtocol::protocol()->statusOffline
     || status.status() == Kopete::OnlineStatus::Unknown) {
    // assume we're offline, thus left all channels ...
    setClientEntry(NULL);
  }
}


void 
SilcBuddyContact::sendFile(const KURL &sourceURL,
			   const QString & /* fileName */, uint /* fileSize */)
{
  QString filePath;

  if(! sourceURL.isValid())
    filePath = KFileDialog::getOpenFileName(QString::null, "*", 0L,
					    i18n("Kopete File Transfer"));
  else
    filePath = sourceURL.path(-1);

  QFile file(filePath);
  if(! file.exists())
    return;

  if(account()->useSilcMime()) {
    if (file.size() < (23 << (5 + (2 * 3)))) 
      sendFileAsMime(filePath);
    else {
        int answer = KMessageBox::questionYesNo
          (Kopete::UI::Global::mainWidget(),
           QString(i18n("You are trying to send a big file via Silc"
                        "MIME message. Do you want to send it via Silc"
                        "Filetransfer?")),
           i18n("Sending MIME message"));
        if (answer == KMessageBox::Yes) 
          new SilcFileTransfer(account(), this, filePath);
        else
          sendFileAsMime(filePath);
    }
  }
  else 
    new SilcFileTransfer(account(), this, filePath);
}


void 
SilcBuddyContact::slotUserInfo(void)
{
  new SilcBuddyContactInfoWidget(this);;
}


/**
 * @short verify the validity of the signature on the provided message
 * @todo we should have a locally stored public key cache
 */
SilcContact::SignatureStatus 
SilcBuddyContact::verifySignature(SilcTK::SilcMessagePayload message)
{
  // get public key from the signature payload and compare it with the
  // one stored in the client entry
  const unsigned char *pk_data;
  SilcTK::SilcUInt32 pk_datalen;
  SilcTK::SilcPublicKey pk = 
     silc_message_signed_get_public_key(message, &pk_data, &pk_datalen);

  SilcContact::SignatureStatus retval;
  QString fp;
  if(pk) {
    // extract the fingerprint from the signatures public key
    char *fpKey = SilcTK::silc_hash_fingerprint(NULL, pk_data, pk_datalen);
    fp = QString(fpKey).replace(QChar(' '), QChar(':'));
    free(fpKey);

    // compare the stored fingerprint with the received key's one
    if(fp.compare(fingerprint()) == 0)
      retval = fpTrusted() ? Trusted : Valid;
    else
      retval = Unknown;
  }
  
  else {
    fp = fingerprint();
    retval = fpTrusted() ? Trusted : Valid;
  }

  QString fn = publicKeyPath(fp);
  if(QFile::exists(fn)) {
    if(pk) SilcTK::silc_pkcs_public_key_free(pk);
    if (! SilcTK::silc_pkcs_load_public_key(fn.latin1(), &pk) &&
	! SilcTK::silc_pkcs_load_public_key(fn.latin1(), &pk))
      return Unknown;
  }
  
  // if we don't have a public key, we simply cannot verify ...
  if(! pk) return Unknown;
  
  // check whether the signature is valid
  if(SilcTK::silc_message_signed_verify
     (message, pk, account()->sha1hash) != SILC_AUTH_OK)
    retval = Failed;

  SilcTK::silc_pkcs_public_key_free(pk);
  return retval;
}

/**
 * returns the path to look for the clients public key.
 *
 * The returned filename is of the form 
 @verbatim
  $kopete_appdata/kopete_silc_clientkeys/clientkey_XXXX_XXXX_XXXX_XXXX_XXXX__XXXX_XXXX_XXXX_XXXX_XXXX.pub
 @endverbatim
 *
 * @param fp provide a fingerprint
 */

const QString 
SilcBuddyContact::publicKeyPath(const QString fp) 
{
  QString fn = locateLocal("appdata",
			   QString("kopete_silc_clientkeys/clientkey_%1.pub")
			   .arg(QString(fp).replace(":","_")));
  return fn;
}

/**
 * Behaves like the above function but receives fingerprint from this buddy.
 */

const QString 
SilcBuddyContact::publicKeyPath(void) const {
  return SilcBuddyContact::publicKeyPath(this->fingerprint());
}

/**
 * performs a WHOIS request for a buddy.
 *
 * If possible a known public key is used. 
 *
 * @returns true if has be called with public key and false otherwise
 */

bool 
SilcBuddyContact::whoami(void) 
{
  QString fp = publicKeyPath();
  if(QFile::exists(fp)) {
    account()->sendSilcCommand( QString("WHOIS -pubkey %1 -details")
				.arg(fp.latin1()) ); 
    return true;
  } else {
    account()->sendSilcCommand( QString("WHOIS %1 -details")
				.arg(nickName()) ); 
    return false;
  }
}

/**
 * @brief either add or remote the buddy from the watch list
 *
 * @return whether public keys are used for watching
 */
 /*
 * This seems to be wrong, pubkey watch of offline buddies and libsilc
 * 1.0.3 works 
 *
 *   We rely on always watching using the nickname, i.e. never use the 
 *   public key for watching purposes. This is simply because we cannot
 *   watch offline buddies using public key approach ...
 *
 *   ... and first testing whether the buddy is offline before setting the
 *   suitable watch is painful.
 *
 *   @return whether public keys are used for watching, i.e. always false
 */
bool
SilcBuddyContact::watchme(bool watch)
{ 
  QString fp = this->publicKeyPath();
  this->_watched = watch;

  if(! watch) {
    if(QFile::exists(fp)) {
      account()->sendSilcCommand(QString("WATCH -pubkey -%1").arg(fp.latin1()),
                                 (SilcTK::SilcClientCommandReply) 
                                 watchme_callback, 
                                 this);
      return true;
    } else {
      // @fixme check whether there is another buddy with this
      // nickname, which we might disable watching for, ...
      account()->sendSilcCommand(QString("WATCH -del %1").arg(nickName()),
                                 (SilcTK::SilcClientCommandReply)
                                 watchme_callback, 
                                 this);
      return false;
    }
  } else {
    if(QFile::exists(fp)) {
     account()->sendSilcCommand(QString("WATCH -pubkey +%1").arg(fp.latin1()),
                                (SilcTK::SilcClientCommandReply)
                                watchme_callback, 
                                this);
     return true;
    } else {
      account()->sendSilcCommand(QString("WATCH -add %1").arg(nickName()),
                                 (SilcTK::SilcClientCommandReply)
                                 watchme_callback, 
                                 this);
      return false;
    }
  }
}

/*
 * @brief callback for buddy->watchme to retry the watch, if it fails
 */

bool
SilcBuddyContact::watchme_callback(SilcTK::SilcClient,
                 SilcTK::SilcClientConnection,
                 SilcTK::SilcCommand,
                 SilcTK::SilcStatus status,
                 SilcTK::SilcStatus error,
                 void *context,
                 va_list)
{
  SilcBuddyContact *buddy = (SilcBuddyContact *) context;
  if(status != SILC_STATUS_OK && ( error == SILC_STATUS_ERR_NO_SUCH_CLIENT_ID
                                   || error == SILC_STATUS_ERR_TIMEDOUT))
    buddy->watchme(buddy->watched());
  return false;
}


/**
* Updates the local cache of WHOIS information which may be displayerd
* via the user information interface.
*
*/

void
SilcBuddyContact::updateWhois(QString username, QString realname) {
  this->_username=username;
  this->_realname=realname;
  this->_update=QDateTime::currentDateTime();
  emit signalWhois(this->nickName(),username,realname);
}


/**
 * returns true if a public key matching a given fingerprint is found locally.
 */
bool 
SilcBuddyContact::havePublicKey(QString fp) 
{
  return QFile::exists(SilcBuddyContact::publicKeyPath(fp));
}

/**
 * behaves essentially as the above function but receives the fingerprint from
 * this buddy.
 */

bool
SilcBuddyContact::havePublicKey(void) {
  return SilcBuddyContact::havePublicKey(this->fingerprint());
}


/**
 * make sure to update /watch status in case of nickname changes 
 */
void 
SilcBuddyContact::slotPropertyChanged(Kopete::Contact *contact,
				      const QString &key,
				      const QVariant &oldValue,
				      const QVariant &)
{
  if(contact != this) return;
  if(key.compare("nickName")) return;
  
  if(! _watched) return;

  // @fixme check whether there is another buddy with this nickname, which
  // we might disable watching for, ...
  if(! oldValue.toString().isNull())
    account()->sendSilcCommand(QString("WATCH -del %1")
			       .arg(oldValue.toString()));
  watchme(true);
}

void 
SilcBuddyContact::sendFileAsMime(const QString &fileName) 
{
  int chunks = 0;
  SilcTK::SilcBuffer buffer;
  QFile file(fileName);
  Kopete::ChatSession *session = manager(Kopete::Contact::CanCreate);

  /* Sending Chunks */
  SilcTK::SilcDList parts = getFileAsMime(fileName);
  SilcTK::silc_dlist_start(parts);
  while ((buffer = (SilcTK::SilcBuffer)SilcTK::silc_dlist_get(parts)) != SILC_LIST_END) { 
    chunks++;
    SilcTK::silc_client_send_private_message
      (account()->client(), account()->conn(), clientEntry(), 
       SILC_MESSAGE_FLAG_DATA, account()->sha1hash, 
       (unsigned char*)buffer->data, (buffer->end - buffer->data));
  }
  SilcTK::silc_mime_partial_free(parts);

  
  Kopete::Message msg = 
    Kopete::Message(account()->myself(), this, 
		    account()->myself()->mimeDisplayMessage(fileName, chunks),
                    Kopete::Message::Outbound,
                    Kopete::Message::RichText, QString::null,
                    Kopete::Message::TypeNormal);
  

  session->appendMessage(msg);
  session->messageSucceeded();
}

void 
SilcBuddyContact::mimeAlternateToMsg( Kopete::Message &msg, 
                                      SilcTK::SilcMime mime,
                                      bool allowRichText) const
{
  SilcTK::SilcDList parts = SilcTK::silc_mime_get_multiparts(mime, NULL);
  SilcTK::SilcMime part;
  QString type, text, html;

  SilcTK::silc_dlist_start(parts);
  while ((part = (SilcTK::SilcMime)SilcTK::silc_dlist_get(parts)) != SILC_LIST_END) {
    type = SilcTK::silc_mime_get_field(part, "Content-Type");
    if(type.left(10).compare("text/plain") == 0) {
      if (type.contains("utf-8"))
        text = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(part, NULL));
      else
        text = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(part, NULL));
    }
    else if(type.left(9).compare("text/html") == 0) {
      if (type.contains("utf-8"))
        html = QString::fromUtf8((char *)SilcTK::silc_mime_get_data(part, NULL));
      else
        html = QString::fromLatin1((char *)SilcTK::silc_mime_get_data(part, NULL));
    }
  }
  if ( ! allowRichText || html.isEmpty())
    /* Use text */
    msg.setBody(text, Kopete::Message::PlainText);  
  else 
    msg.setBody(html, Kopete::Message::RichText);
}

SilcBuddyContactData::SilcBuddyContactData(SilcAccount * account, 
                                                QString nickName,
                                                QString finger,
                                                Kopete::MetaContact *meta)
 :nickname(nickName),finger(finger),meta(meta),account(account)
{
  return; 
}

#include "silcbuddycontact.moc"
