# arch-tag: 8f2927c7-53a3-4a84-9055-1f99466c8a8d
# Copyright (C) 2005 Canonical Ltd.
#               Author: David Allouche <david@canonical.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Utilities for handling compatibility with multiple versions."""

__all__ = ['status', 'BazaarCommandVersion']


class _Enum(object):
    def __init__(self, name):
        self.__name = name

    def __repr__(self):
        class_name = self.__class__.__name__
        return '%s.status.%s' % (__name__, self.__name)

    def __str__(self):
        return self.__name

    def __eq__(self, rhs):
        return self is rhs

    def __ne__(self, rhs):
        return self is not rhs


class status(object):

    def __init__(self):
        raise AssertionError('Tried to instanciate enum class')

    # baz is a supported release
    OK = _Enum('OK')

    # baz a future release, pybaz must be upgraded
    BAZ_TOO_NEW = _Enum('BAZ_TOO_NEW')

    # baz is an unsupported obsolete release, baz should be upgraded
    UNSUPPORTED = _Enum('UNSUPPORTED')

    # baz and pybaz are prereleases for the same version, if a problem arises,
    # both must be upgraded to the latest release before reporting an issue.
    COTM = _Enum('COTM')


def _get_status(pybaz, baz):
    bazminor = baz.release[:2]
    pybazminor = pybaz.release[:2]
    if bazminor == pybazminor:
        if pybaz.cotm and not baz.cotm:
            return status.BAZ_TOO_NEW
        if baz.cotm and pybaz.cotm:
            return status.COTM
        if baz.cotm and not pybaz.cotm:
            return status.UNSUPPORTED
        return status.OK
    if bazminor < pybazminor:
        if baz.cotm:
            return status.UNSUPPORTED
        return status.OK
    if bazminor > pybazminor:
        return status.BAZ_TOO_NEW



class BazaarCommandVersion(object):

    """Parsing and comparison of Bazaar version strings."""

    def __init__(self, version_line):
        """Parse a Bazaar version string.

        :param version_line: line of the form
            ``'baz Bazaar version MAJOR.MINOR[.POINT][[-]~COTM]'``
        :type version_line: str
        """
        self._version_line = version_line
        vstr, vtuple, cotm = self._parse(version_line)
        self._string = vstr
        self._release_tuple = vtuple
        self._cotm = cotm

    def __repr__(self):
        class_name = self.__class__.__name__
        return '%s.%s(%r)' % (__name__, class_name, self._version_line)

    def _parse(line):
        """Actual version string parsing method.

        Extracted as a static method for unit testing.
        """
        words = line.split(' ')
        assert words[:3] == ['baz', 'Bazaar', 'version']
        full_version_str = words[3]
        version_cotm = full_version_str.split('~')
        if len(version_cotm) == 1:
            version_str = version_cotm[0]
            cotm = None
        elif len(version_cotm) == 2:
            version_str, cotm = version_cotm
            cotm = long(cotm)
        else:
            raise AssertionError('Version string seems to have multiple'
                                 ' COTM qualifiers: ' + full_version_str)
        version_str = version_str.rstrip('-')
        version = map(int, version_str.split('.'))
        if len(version) == 2:
            version.append(0)
        else:
            assert len(version) == 3
        return full_version_str, tuple(version), cotm

    _parse = staticmethod(_parse)

    def _get_string(self):
        return self._string

    string = property(_get_string, doc="""
    Unparsed version string, without the "baz Bazaar version" prefix.
    Use this attribute in user-visible messages.
    """)

    def _get_release(self):
        return self._release_tuple

    release = property(_get_release, doc="""
    Tuple of integers representing the Bazaar release version.""")

    def _get_cotm(self):
        return self._cotm

    cotm = property(_get_cotm, doc="""
    Long integer identifying the Crack Of The Minute build.
    None if the version is a release.
    """)

    _end_of_times = (10 ** (4 + 2 + 2 + 2 + 2)) - 1

    def _comparable(self):
        a = list(self.release)
        if self.cotm is None:
            a.append(0)
        else:
            a.append(self.cotm - self._end_of_times)
        return a

    def __cmp__(self, rhs):
        if not isinstance(rhs, BazaarCommandVersion):
            msg = 'Tried to compare BazaarCommandVersion with %r' % rhs
            raise TypeError(msg)
        a = self._comparable()
        b = rhs._comparable()
        return cmp(a, b)
