/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#ifdef HAVE_GETOPT_H
#define _GNU_SOURCE
#include <getopt.h>
#endif

#include <gtk/gtk.h>

#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
#define SN_API_NOT_YET_FROZEN 1
#include <libsn/sn.h>
#endif

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "xfmedia-xine.h"

#include "main.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-settings.h>
#include "xfmedia-internal.h"
#include "remote.h"
#include <xfmedia/xfmedia-remote-client.h>
#include "mediamarks.h"
#include "playlist-files.h"
/*#include "equaliser.h"*/

static void mainwin_destroy_cb(GtkWidget *w, gpointer user_data);
static void vwin_destroy_cb(GtkWidget *w, gpointer user_data);

static XfmediaMainwin *g_mwin = NULL;
static pid_t main_pid = 0;
static gboolean quitting = FALSE;
static guint mwin_dest_id = 0, vwin_dest_id = 0;

void
xfmedia_quit(XfmediaMainwin *mwin, XfmediaQuitMode quit_mode)
{
    GtkWidget *main_window, *video_window;
    
    if(!mwin)
        exit(0);
    
    xfmedia_mediamarks_cleanup();
    xfmedia_remote_cleanup(mwin);
    xfmedia_plugins_unload();
    
    if(xfmedia_settings_get_bool("/xfmedia/playlist/save_on_exit_always")
            || !mwin->oneshot
            || xfmedia_playlist_is_dirty(mwin->plist))
    {
        xfmedia_settings_save_playlist(mwin);
    }
    xfmedia_settings_cleanup();
    
    /* save pointers */
    main_window = mwin->window;
    video_window = mwin->video_window;
    
    xfmedia_mainwin_destroy(mwin);
    
    switch(quit_mode) {
        case XFMEDIA_QUIT_MODE_EXT:
            g_signal_handler_disconnect(G_OBJECT(main_window), mwin_dest_id);
            g_signal_handler_disconnect(G_OBJECT(video_window), vwin_dest_id);
            gtk_widget_destroy(video_window);
            gtk_widget_destroy(main_window);
            break;
        
        case XFMEDIA_QUIT_MODE_MAINWIN_DESTROY:
            g_signal_handler_disconnect(G_OBJECT(video_window), vwin_dest_id);
            gtk_widget_destroy(video_window);
            break;
        
        case XFMEDIA_QUIT_MODE_VWIN_DESTROY:
            g_signal_handler_disconnect(G_OBJECT(main_window), mwin_dest_id);
            gtk_widget_destroy(main_window);
            break;
    }
    
    if(gtk_main_level())
        gtk_main_quit();
    
    exit(0);
}

static void
mainwin_destroy_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("entering");
    
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_MAINWIN_DESTROY);
}

static void
vwin_destroy_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    DBG("entering");
    
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_VWIN_DESTROY);
}

static void
sighandler_cb(int sig)
{
    if(quitting)
        return;
    
    quitting = TRUE;
    
    if(main_pid == getpid())
        xfmedia_quit(g_mwin, XFMEDIA_QUIT_MODE_EXT);
}

static void
session_save_yourself_cb(ExoXsessionClient *client, gpointer user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
}

static void
do_startup_notify(GtkWidget *window)
{
#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    SnDisplay *sdpy;
    SnLauncheeContext *ctx;
    
    sdpy = sn_display_new(GDK_DISPLAY(),
            (SnDisplayErrorTrapPush)gdk_error_trap_push,
            (SnDisplayErrorTrapPop)gdk_error_trap_pop);
    ctx = sn_launchee_context_new_from_environment(sdpy,
            gdk_screen_get_number(gtk_widget_get_screen(window)));
    if(ctx) {
        if(!GTK_WIDGET_REALIZED(window))
            gtk_widget_realize(window);
        sn_launchee_context_setup_window(ctx, GDK_WINDOW_XWINDOW(window->window));
        sn_launchee_context_complete(ctx);
        sn_launchee_context_unref(ctx);
    }
    sn_display_unref(sdpy);
#endif
}

static gboolean
initial_autoplay(gpointer data)
{
    XfmediaMainwin *mwin = data;
    
    gdk_threads_enter();
    xfmedia_mainwin_play_file_at_index(mwin, 0);
    gdk_threads_leave();
    
    return FALSE;
}

static void
xfmedia_print_usage(const gchar *argv0)
{
	gchar *cmd;
	
	if((cmd=g_strrstr(argv0, "/")))
		cmd++;
	else
		cmd = (gchar *)argv0;
	
	g_print(_("Xfmedia %s.  Released under the terms of the GNU GPL.  See the COPYING\nfile for details.\n\n"), VERSION);
	g_print(_("Usage: %s [options] [files]\n\n"), cmd);
	g_print(_("    -h, --help           Print this help message and exit.\n"));
	g_print(_("    -V, --version        Print version information and exit.\n"));
	g_print(_("    -v, --verbose        Instruct xine to print useful status information.\n"));
	g_print(_("    --debug              Instruct xine to print detailed debug information.\n"));
	g_print(_("    -f, --fullscreen     Start xfmedia's video window in fullscreen mode.\n"));
	g_print(_("    -p, --autoplay       Start playing when Xfmedia starts.  (This is the\n"
	          "                         default if xfmedia is given filenames on the command\n"
	          "                         line.)\n"));
    g_print(_("    -c, --clear-playlist Start xfmedia with an empty playlist.\n"));
	g_print(_("    --audio-out=DRIVER   Specify the audio output driver to use.\n"));
	g_print(_("    --video-out=DRIVER   Specify the video output driver to use.\n"));
    g_print(_("    --vwin-geometry=WxH[+X+Y]\n"
              "                         Specify a fixed size and/or position for xfmedia's\n"
              "                         video window.\n"));
}

enum
{
    OPT_DISPLAY = 1000,
    OPT_SMCLIENTID,
    OPT_DEBUG,
    OPT_AUDIO_OUT,
    OPT_VIDEO_OUT,
    OPT_VWIN_GEOMETRY,
};

static const gchar *short_opts = "Vhvfpc";
static const struct option long_opts[] =
{
    /* stuff we just want out of the way */
    { "display", required_argument, NULL, OPT_DISPLAY },
    { "sm-client-id", required_argument, NULL, OPT_SMCLIENTID },
    /* stuff we care about */
    { "version", no_argument, NULL, 'V' },
    { "help", no_argument, NULL, 'h' },
    { "verbose", no_argument, NULL, 'v' },
    { "debug", no_argument, NULL, OPT_DEBUG },
    { "fullscreen", no_argument, NULL, 'f' },
    { "autoplay", no_argument, NULL, 'p' },
    { "clear-playlist", no_argument, NULL, 'c' },
    { "audio-out", required_argument, NULL, OPT_AUDIO_OUT },
    { "video-out", required_argument, NULL, OPT_VIDEO_OUT },
    { "vwin-geometry", required_argument, NULL, OPT_VWIN_GEOMETRY },
    { NULL, 0, NULL, 0 },
};

enum
{
    VWIN_GEOM_WIDTH  = 0x0001,
    VWIN_GEOM_HEIGHT = 0x0002,
    VWIN_GEOM_X      = 0x0004,
    VWIN_GEOM_Y      = 0x0008,
};

typedef struct
{
    gboolean is_remote_cmd;
    gint session_id;
    gboolean is_enqueue;
    
    gint verbose_level;
    gboolean use_fullscreen;
    gboolean use_autoplay;
    gboolean dont_load_playlist;
    
    gint vwin_geom_flags;
    GdkRectangle vwin_geom;
} OptionData;

static gint
xfmedia_parse_options(int argc, char **argv, OptionData *odata)
{
    gint c = 0, option_index = 0;
    
    opterr = 0;
    for(;;) {
        c = getopt_long(argc, argv, short_opts, long_opts, &option_index);
        if(c == -1)
            break;
        
        switch(c) {
            case OPT_DISPLAY:
            case OPT_SMCLIENTID:
                break;
            
            case 'V':
                g_print(_("Xfmedia version %s, Copyright (c) Brian Tarricone, <bjt23@cornell.edu>\n"), VERSION);
                g_print(_("Released under the terms of the GNU General Public License.\n"));
                g_print(_("Compiled against xine-lib %s, using xine-lib %s.\n"),
                        XINE_VERSION, xine_get_version_string());
                g_print(_("Compiled against Xfce %d.%d.%d, using Xfce %s.\n"),
                        LIBXFCE4UTIL_MAJOR_VERSION, LIBXFCE4UTIL_MINOR_VERSION,
                        LIBXFCE4UTIL_MICRO_VERSION, xfce_version_string());
                exit(0);
                break;
            
            case 'h':
                xfmedia_print_usage(argv[0]);
                exit(0);
                break;
            
            case 'v':
                odata->verbose_level = XINE_VERBOSITY_LOG;
                break;
            
            case OPT_DEBUG:
                odata->verbose_level = XINE_VERBOSITY_DEBUG;
                break;
            
            case 'f':
                odata->use_fullscreen = TRUE;
                break;
            
            case 'p':
                odata->use_autoplay = TRUE;
                break;
            
            case 'c':
                odata->dont_load_playlist = TRUE;
                break;
            
            case OPT_AUDIO_OUT:
                if(optarg && *optarg)
                    xfmedia_settings_set_string("/xfmedia/plugins/libxine/audio_out", optarg);
                else {
                    g_printerr(_("The option --audio-out requires an argument.\n"));
                    exit(1);
                }
                break;
            
            case OPT_VIDEO_OUT:
                if(optarg && *optarg)
                    xfmedia_settings_set_string("/xfmedia/plugins/libxine/video_out", optarg);
                else {
                    g_printerr(_("The option --video-out requires an argument.\n"));
                    exit(1);
                }
                break;
            
            case OPT_VWIN_GEOMETRY:
                if(optarg && *optarg) {
                    guint w, h, x, y;
                    gint args = sscanf(optarg, "%ux%u+%u+%u", &w, &h, &x, &y);
                    
                    odata->vwin_geom_flags = 0;
                    
                    if(args >= 2) {
                        odata->vwin_geom.width = w;
                        odata->vwin_geom.height = h;
                        odata->vwin_geom_flags |= VWIN_GEOM_WIDTH|VWIN_GEOM_HEIGHT;
                    }
                    
                    if(args == 4) {
                        odata->vwin_geom.x = x;
                        odata->vwin_geom.y = y;
                        odata->vwin_geom_flags |= VWIN_GEOM_X|VWIN_GEOM_Y;
                    }
                }
                break;
            
            default:
                break;
        }
    }
    
    return optind;
}

int
main(int argc, char **argv)
{
    GtkWidget *xfx, *video_window;
    XfmediaMainwin *mwin;
    gint i, fd;
    gboolean defer = FALSE, do_autoplay = FALSE;
    unsigned int seed;
    gint x, y, ppos, next_arg;
    ExoXsessionClient *session_client;
    OptionData odata;
    gchar *session_argv[1];
    GdkPixbuf *winicon;
    const gchar *video_port_id = NULL;
    
    main_pid = getpid();
    
    signal(SIGINT, sighandler_cb);
    signal(SIGHUP, sighandler_cb);
    signal(SIGQUIT, sighandler_cb);
    signal(SIGUSR1, SIG_IGN);
    signal(SIGUSR1, SIG_IGN);
    signal(SIGPIPE, SIG_IGN);
    
    xfce_textdomain(GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");
    gtk_set_locale();
    
    memset(&odata, 0, sizeof(odata));
    odata.verbose_level = XINE_VERBOSITY_NONE;
    odata.session_id = -1;
    xfmedia_settings_init();
    next_arg = xfmedia_parse_options(argc, argv, &odata);
    
    /* seed PRNG */
    fd = open("/dev/random", O_RDONLY);
    if(fd >= 0) {
        read(fd, &seed, sizeof(unsigned int));
        close(fd);
    } else {
        fd = open("/dev/urandom", O_RDONLY);
        if(fd >= 0) {
            read(fd, &seed, sizeof(unsigned int));
            close(fd);
        } else
            seed = time(NULL) ^ (main_pid + (main_pid << 15));
    }
#if HAVE_SRANDOM
    srandom(seed);
#else
    srand(seed);
#endif
    
    g_thread_init(NULL);
    gdk_threads_init();
    gtk_init(&argc, &argv);
    
    DBG("calling XInitThreads()");
    if(!XInitThreads()) {
        g_critical("XfmediaXine: XInitThreads() failed; thread-safe X11 libs unavailable!\n");
        video_port_id = "none";
    } else
        video_port_id = xfmedia_settings_get_string("/xfmedia/plugins/libxine/video_out");
    
    winicon = gdk_pixbuf_from_pixdata(&xfmedia_icon, TRUE, NULL);
    if(winicon) {
        gtk_window_set_default_icon(winicon);
        g_object_unref(G_OBJECT(winicon));
    }
    
    xfmedia_init_icon_theme();
    xfmedia_stock_icons_init();
    xfmedia_mediamarks_init();
    
    gdk_threads_enter();
    
    video_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(video_window), _("Xfmedia Video"));
    gtk_container_set_border_width(GTK_CONTAINER(video_window), 0);
    gtk_widget_realize(video_window);
    if(odata.vwin_geom_flags & (VWIN_GEOM_X|VWIN_GEOM_Y)) {
        gtk_window_move(GTK_WINDOW(video_window),
                        odata.vwin_geom.x,
                        odata.vwin_geom.y);
    }
    if(odata.vwin_geom_flags & (VWIN_GEOM_WIDTH|VWIN_GEOM_HEIGHT)) {
        gtk_widget_set_size_request(video_window,
                                    odata.vwin_geom.width,
                                    odata.vwin_geom.height);
        gtk_window_resize(GTK_WINDOW(video_window),
                          odata.vwin_geom.width,
                          odata.vwin_geom.height);
        g_object_set_data(G_OBJECT(video_window),
                          "vwin-size-fixed",
                          GINT_TO_POINTER(0xfeedface));
    }
    
    xfx = xfmedia_xine_new(video_port_id,
            xfmedia_settings_get_string("/xfmedia/plugins/libxine/audio_out"));
    gtk_widget_show(xfx);
    gtk_container_add(GTK_CONTAINER(video_window), xfx);
    DBG("about to realize...");
    gtk_widget_realize(xfx);
    DBG("    realize done");
    
    xfmedia_xine_engine_set_param(XFMEDIA_XINE(xfx),
            XINE_ENGINE_PARAM_VERBOSITY, odata.verbose_level);
    
    mwin = g_mwin = xfmedia_mainwin_new(XFMEDIA_XINE(xfx), GTK_WINDOW(video_window));
    mwin_dest_id = g_signal_connect(G_OBJECT(mwin->window), "destroy",
            G_CALLBACK(mainwin_destroy_cb), mwin);
    vwin_dest_id = g_signal_connect(G_OBJECT(video_window), "destroy",
            G_CALLBACK(vwin_destroy_cb), mwin);
    if(argc-next_arg > 11)
        defer = TRUE;
    mwin->oneshot = FALSE;
    for(i = next_arg; i < argc; i++) {
        if(xfmedia_playlists_is_playlist_file(argv[i]))
            xfmedia_playlists_load(mwin->plist, argv[i]);
        else {
            gchar *title_utf8 = xfmedia_filename_to_name(argv[i]);
            xfmedia_playlist_append_entry(mwin->plist, title_utf8, -1, argv[i], FALSE);
            g_free(title_utf8);
            mwin->oneshot = TRUE;
        }
        do_autoplay = TRUE;
    }
    
    if(next_arg >= argc && !odata.dont_load_playlist)
        xfmedia_settings_load_playlist(mwin);
    
    do_startup_notify(mwin->window);
    
    x = xfmedia_settings_get_int("/xfmedia/general/pos_x");
    y = xfmedia_settings_get_int("/xfmedia/general/pos_y");
    if(x != -1 && y != -1)
        gtk_window_move(GTK_WINDOW(mwin->window), x, y);
    gtk_widget_show(mwin->window);
    gtk_widget_set_size_request(mwin->window, -1, -1);
    if(xfmedia_settings_get_bool("/xfmedia/general/dock_autohide"))
        xfmedia_mainwin_set_autohide(mwin, TRUE);
    
    session_argv[0] = argv[0];
    session_client = exo_xsession_client_new_with_group(mwin->window->window);
    exo_xsession_client_set_restart_command(session_client, session_argv, 1);
    g_signal_connect(G_OBJECT(session_client), "save-yourself",
            G_CALLBACK(session_save_yourself_cb), mwin);
    
    gtk_widget_realize(GTK_WIDGET(mwin->plist));
    ppos = xfmedia_settings_get_int("/xfmedia/playlist/playlist_position");
    if(ppos > -1 && ppos < xfmedia_playlist_get_n_entries(mwin->plist)) {
        xfmedia_playlist_scroll_to_index(mwin->plist, ppos, 0.5);
        mwin->cur_playing = xfmedia_playlist_entry_ref_new_for_index(mwin->plist, ppos);
        if(mwin->cur_playing)
            xfmedia_playlist_set_bold_entry(mwin->plist, mwin->cur_playing);
    }
    
    if(xfmedia_settings_get_bool("/xfmedia/general/show_tray_icon"))
        xfmedia_mainwin_activate_systray(mwin);
    
    /*xfmedia_equaliser_init(mwin);*/
    
    mwin->session_id = xfmedia_remote_init(mwin);
    xfmedia_plugins_load(mwin);
    
    if(odata.use_fullscreen)
        xfmedia_xine_set_fullscreen(mwin->xfx, TRUE);
    if(do_autoplay || odata.use_autoplay)
        g_idle_add(initial_autoplay, mwin);
    
    gtk_main();
    
    gdk_threads_leave();
    
    /* xfmedia_quit() handles the cleanup */
    
    return 0;
}
