/*
 * Copyright (c) VMware, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * on the rights to use, copy, modify, merge, publish, distribute, sub
 * license, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS AND/OR THEIR SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/**
 * @file
 * Test texture unit state with respect to the different number of
 * texture coord units, image units, combined units, etc.
 */
#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_core_version = 31;
	config.supports_gl_compat_version = 20;

	config.window_visual = PIGLIT_GL_VISUAL_RGB | PIGLIT_GL_VISUAL_DOUBLE;
	config.khr_no_error_support = PIGLIT_NO_ERRORS;

PIGLIT_GL_TEST_CONFIG_END

#define MAX_UNITS 256

/** random number for checking state */
static GLfloat Random[MAX_UNITS][4];

static GLint MaxTextureCoordUnits;
static GLint MaxTextureVertexUnits;
static GLint MaxTextureImageUnits;
static GLint MaxTextureCombinedUnits;


static void
generate_random_numbers(void)
{
   int i, j;
   for (i = 0; i < ARRAY_SIZE(Random); i++) {
      for (j = 0; j < 4; j++) {
         /* values in [0, 1] */
         Random[i][j] = (rand() % 1000) * .001;
      }
   }
}


static bool
equal4v(const GLfloat v1[4], const GLfloat v2[4])
{
   return (v1[0] == v2[0] &&
           v1[1] == v2[1] &&
           v1[2] == v2[2] &&
           v1[3] == v2[3]);
}


static void
report4v(const GLfloat exp[4], const GLfloat act[4])
{
   printf("Expected (%g, %g, %g, %g) but found (%g, %g, %g, %g)\n",
          exp[0], exp[1], exp[2], exp[3],
          act[0], act[1], act[2], act[3]);
}


static bool
test_texture_params(void)
{
   GLuint tex[MAX_UNITS];
   int i;
   int maxUnit;

   piglit_reset_gl_error();

   glCreateTextures(GL_TEXTURE_2D, MaxTextureCombinedUnits, tex);

   /* set per-unit state */
   for (i = 0; i < MaxTextureCombinedUnits; i++) {
      glBindTextureUnit(i, tex[i]);
      glTextureParameterfv(tex[i], GL_TEXTURE_BORDER_COLOR, Random[i]);
   }

   /* check per-unit state */
   for (i = 0; i < MaxTextureCombinedUnits; i++) {
      GLfloat v[4];
      glBindTextureUnit(i, tex[i]);
      /* any per-unit state will do: */
      glGetTextureParameterfv(tex[i], GL_TEXTURE_BORDER_COLOR, v);
      if (!equal4v(v, Random[i])) {
         printf("Setting per-unit param state failed for unit %d\n", i);
         report4v(Random[i], v);
         return false;
      }
   }

   /* there should be no errors at this point */
   if (!piglit_check_gl_error(GL_NO_ERROR)) {
      return false;
   }

   maxUnit = MAX2(MaxTextureCombinedUnits, MaxTextureCoordUnits);

   /* This should generate an error.  The GL_ARB_direct_state_access extension
    * isn't explicit about which error should be generated, but the typical
    * error for a out-of-range int/uint is GL_INVALID_VALUE.  That's what
    * NVIDIA's driver does.
    */
   if (!piglit_khr_no_error) {
      glBindTextureUnit(maxUnit, tex[0]);
      if (!piglit_check_gl_error(GL_INVALID_VALUE)) {
         return false;
      }
   }

   return true;
}


static void
report_info(void)
{
   printf("GL_RENDERER = %s\n", (char *) glGetString(GL_RENDERER));
   printf("GL_MAX_TEXTURE_COORDS = %d\n", MaxTextureCoordUnits);
   printf("GL_MAX_TEXTURE_IMAGE_UNITS = %d\n", MaxTextureImageUnits);
   printf("GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS = %d\n", MaxTextureVertexUnits);
   printf("GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS = %d\n", MaxTextureCombinedUnits);
}


enum piglit_result
piglit_display(void)
{
	bool pass = true;

	pass = test_texture_params() && pass;

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}


void
piglit_init(int argc, char *argv[])
{
   piglit_require_extension("GL_ARB_direct_state_access");

   glGetIntegerv(GL_MAX_TEXTURE_COORDS, &MaxTextureCoordUnits);
   glGetIntegerv(GL_MAX_TEXTURE_IMAGE_UNITS, &MaxTextureImageUnits);
   glGetIntegerv(GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS, &MaxTextureVertexUnits);
   glGetIntegerv(GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS, &MaxTextureCombinedUnits);

   report_info();

   if (MaxTextureCombinedUnits > MAX_UNITS) {
      /* Need to increase the MAX_UNITS limit */
      piglit_report_result(PIGLIT_WARN);
   }

   generate_random_numbers();
}
