/* ScummVM Tools
 *
 * ScummVM Tools is the legal property of its developers, whose
 * names are too numerous to list here. Please refer to the
 * COPYRIGHT file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/* Scumm Script Disassembler (version 6-8 scripts) */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "descumm.h"

#include "common/util.h"

/*
switch/case statements have a pattern that look as follows (they were probably
generated by some Scumm script compiler):

push SOMETHING		- the value we are switching on
dup					- duplicate it
push caseA			- push the first case value
eq					- compare
jump if false		- if not equal, jump to next case
  kill				- we entered this case - kill the switch value from the stack
  ...
  jump AFTER		- leave this switch
dup
push caseB
eq
jump if false
  kill
  ...
  jump AFTER
[optional: default case here ?!?]
AFTER:


Unfortunatly, in reality it is somewhat more complicated. E.g. in script 55,
in some cases the "jump AFTER" instead jumps into the middle of some other case.
Maybe they did have some sort of code optimized after all?

Anyway, the fixed key pattern here seems to be for each case:
  dup - push - eq - jumpFalse - kill
And of course a push before the first of these. Note that there doesn't have
to be a jump before each case: after all, "fall through" is possible with C(++)
switch/case statements, too, so it's possible they used that in Scumm, too.

*/

class StackEnt;

const char *getVarName(uint var);
StackEnt *pop();


static int dupindex = 0;



enum StackEntType {
	seInt = 1,
	seVar = 2,
	seArray = 3,
	seBinary = 4,
	seUnary = 5,
	seComplex = 6,
	seStackList = 7,
	seDup = 8,
	seNeg = 9
};

enum {
	isZero,
	isEqual,
	isNotEqual,
	isGreater,
	isLess,
	isLessEqual,
	isGreaterEqual,
	operAdd,
	operSub,
	operMul,
	operDiv,
	operLand,
	operLor,
	operBand,
	operBor,
	operMod
};

static const char *oper_list[] = {
	"!",
	"==",
	"!=",
	">",
	"<",
	"<=",
	">=",
	"+",
	"-",
	"*",
	"/",
	"&&",
	"||",
	"&",
	"|",
	"%"
};



class StackEnt {
public:
	StackEntType type;

public:
	virtual ~StackEnt() {}
	virtual char *asText(char *where, bool wantparens = true) const = 0;
	virtual StackEnt* dup(char *output);

	virtual int getIntVal() const { error("getIntVal call on StackEnt type %d", type); }
};

class IntStackEnt : public StackEnt {
	int _val;
public:
	IntStackEnt(int val) : _val(val) { type = seInt; }
	virtual char *asText(char *where, bool wantparens) const {
		where += sprintf(where, "%d", _val);
		return where;
	}
	virtual StackEnt* dup(char *output) {
		return new IntStackEnt(_val);
	}
	virtual int getIntVal() const { return _val; }
};

class VarStackEnt : public StackEnt {
	int _var;
public:
	VarStackEnt(int var) : _var(var) { type = seVar; }
	virtual char *asText(char *where, bool wantparens = true) const {
		int var;
		const char *s;
		if (g_options.scriptVersion == 8) {
			if (!(_var & 0xF0000000)) {
				var = _var & 0xFFFFFFF;
				if ((s = getVarName(var)) != NULL)
					where = strecpy(where, s);
				else
					where += sprintf(where, "var%d", _var & 0xFFFFFFF);
			} else if (_var & 0x80000000) {
				where += sprintf(where, "bitvar%d", _var & 0x7FFFFFFF);
			} else if (_var & 0x40000000) {
				where += sprintf(where, "localvar%d", _var & 0xFFFFFFF);
			} else {
				where += sprintf(where, "?var?%d", _var);
			}
		} else {
			if (!(_var & 0xF000)) {
				var = _var & 0xFFF;
				if ((s = getVarName(var)) != NULL)
					where = strecpy(where, s);
				else
					where += sprintf(where, "var%d", _var & 0xFFF);
			} else if (_var & 0x8000) {
				if (g_options.heVersion >= 80) {
					where += sprintf(where, "roomvar%d", _var & 0xFFF);
				} else {
					where += sprintf(where, "bitvar%d", _var & 0x7FFF);
				}
			} else if (_var & 0x4000) {
				where += sprintf(where, "localvar%d", _var & 0xFFF);
			} else {
				where += sprintf(where, "?var?%d", _var);
			}
		}
		return where;
	}
};

class ArrayStackEnt : public StackEnt {
	int _idx;
	StackEnt *_dim1;
	StackEnt *_dim2;
public:
	ArrayStackEnt(int idx, StackEnt *dim2, StackEnt *dim1) : _idx(idx), _dim1(dim1), _dim2(dim2) { type = seArray; }
	virtual char *asText(char *where, bool wantparens) const {
		const char *s;

		if(g_options.scriptVersion == 8 && !(_idx & 0xF0000000) &&
		   (s = getVarName(_idx & 0xFFFFFFF)) != NULL)
			where += sprintf(where, "%s[",s);
		else if(g_options.scriptVersion < 8 && !(_idx & 0xF000) &&
			(s = getVarName(_idx & 0xFFF)) != NULL)
			where += sprintf(where, "%s[",s);
		else if(g_options.scriptVersion < 8 && g_options.heVersion >= 80 && (_idx & 0x8000))
			where += sprintf(where, "roomarray%d[", _idx & 0xfff);
		else if(g_options.scriptVersion < 8 && (_idx & 0x4000))
			where += sprintf(where, "localarray%d[", _idx & 0xfff);
		else
			where += sprintf(where, "array%d[", _idx);

		if (_dim2) {
			where = _dim2->asText(where);
			where = strecpy(where, "][");
		}

		where = _dim1->asText(where);
		where = strecpy(where, "]");

		return where;
	}
};

class UnaryOpStackEnt : public StackEnt {
	int _op;
	StackEnt *_valA;
public:
	UnaryOpStackEnt(int op, StackEnt *valA) : _op(op), _valA(valA) { type = seUnary; }
	virtual char *asText(char *where, bool wantparens) const {
		where += sprintf(where, "%s", oper_list[_op]);
		where = _valA->asText(where);
		return where;
	}
};

class BinaryOpStackEnt : public StackEnt {
	int _op;
	StackEnt *_valA;
	StackEnt *_valB;
public:
	BinaryOpStackEnt(int op, StackEnt *valA, StackEnt *valB) : _op(op), _valA(valA), _valB(valB) { type = seBinary; }
	virtual char *asText(char *where, bool wantparens) const {
		if (wantparens)
			*where++ = '(';
		where = _valA->asText(where);
		where += sprintf(where, " %s ", oper_list[_op]);
		where = _valB->asText(where);
		if (wantparens)
			*where++ = ')';
		*where = 0;
		return where;
	}
};

class ComplexStackEnt : public StackEnt {
	char *_str;
public:
	ComplexStackEnt(const char *s) { _str = strdup(s); type = seComplex; }
	~ComplexStackEnt() { free(_str); }
	virtual char *asText(char *where, bool wantparens) const {
		where = strecpy(where, _str);
		return where;
	}
};

class ListStackEnt : public StackEnt {
public:
	int _size;
	StackEnt **_list;
public:
	ListStackEnt(StackEnt *senum) {
		type = seStackList;

		_size = senum->getIntVal();
		_list = new StackEnt* [_size];

		for (int i = 0; i < _size; ++i) {
			_list[i] = pop();
		}
	}
	~ListStackEnt() { delete [] _list; }
	virtual char *asText(char *where, bool wantparens) const {
		*where++ = '[';
		for (int i = _size - 1; i >= 0; --i) {
			where = _list[i]->asText(where);
			if (i)
				*where++ = ',';
		}
		*where++ = ']';
		*where = 0;
		return where;
	}
};

class DupStackEnt : public StackEnt {
public:
	int _idx;
public:
	DupStackEnt(int idx) : _idx(idx) { type = seDup; }
	virtual char *asText(char *where, bool wantparens) const {
		where += sprintf(where, "dup[%d]", _idx);
		return where;
	}
};

class NegStackEnt : public StackEnt {
	StackEnt *_op;
public:
	NegStackEnt(StackEnt *op) : _op(op) { type = seNeg; }
	virtual char *asText(char *where, bool wantparens) const {
		*where++ = '!';
		where = _op->asText(where);
		return where;
	}
};
#define MAX_STACK_SIZE	256
static StackEnt *stack[MAX_STACK_SIZE];
static int num_stack = 0;

const char *var_names72[] = {
	/* 0 */
	"VAR_KEYPRESS",
	"VAR_DEBUGMODE",
	"VAR_TIMER_NEXT",
	"VAR_OVERRIDE",
	/* 4 */
	"VAR_WALKTO_OBJ",
	"VAR_RANDOM_NR",
	NULL,
	NULL,
	/* 8 */
	"VAR_GAME_LOADED",
	"VAR_EGO",
	"VAR_NUM_ACTOR",
	NULL,
	/* 12 */
	NULL,
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_MOUSE_X",
	/* 16 */
	"VAR_MOUSE_Y",
	"VAR_LEFTBTN_HOLD",
	"VAR_RIGHTBTN_HOLD",
	"VAR_CURSORSTATE",
	/* 20 */
	"VAR_USERPUT",
	"VAR_ROOM",
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	/* 24 */
	"VAR_CAMERA_POS_X",
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_ROOM_RESOURCE",
	/* 28 */
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	/* 32 */
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	/* 36 */
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	"VAR_SAVELOAD_ENTER_SCRIPT",
	"VAR_SAVELOAD_EXIT_SCRIPT",
	/* 40 */
	"VAR_SAVELOAD_SCRIPT",
	"VAR_OPTIONS_SCRIPT",
	"VAR_RESTART_KEY",
	"VAR_PAUSE_KEY",
	/* 44 */
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_TALKSTOP_KEY",
	"VAR_HAVE_MSG",
	"VAR_NOSUBTITLES",
	/* 48 */
	"VAR_CHARINC",
	"VAR_TALK_ACTOR",
	"VAR_LAST_SOUND",
	"VAR_TALK_CHANNEL",
	/* 52 */
	"VAR_SOUND_CHANNEL",
	NULL,
	"VAR_SOUND_ENABLED",
	NULL,
	/* 56 */
	"VAR_NUM_SOUND_CHANNELS",
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	"VAR_NEW_ROOM",
	/* 60 */
	"VAR_TMR_1",
	"VAR_TMR_2",
	"VAR_TMR_3",
	"VAR_TIMEDATE_HOUR",
	/* 64 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_YEAR",
	/* 68 */
	"VAR_NUM_ROOMS",
	"VAR_NUM_SCRIPTS",
	"VAR_NUM_SOUNDS",
	"VAR_NUM_COSTUMES",
	/* 72 */
	"VAR_NUM_IMAGES",
	"VAR_NUM_CHARSETS",
	"VAR_NUM_GLOBAL_OBJS",
	"VAR_MOUSE_STATE",
	/* 76 */
	"VAR_POLYGONS_ONLY",
	NULL,
	"VAR_PLATFORM",
	"VAR_PLATFORM_VERSION",
	/* 80 */
	"VAR_CURRENT_CHARSET",
	"VAR_HIDE_ALL_TEXT",
	"VAR_NO_SPEECH",
	"VAR_MOUSE_OVERRIDE",
	/* 84 */
	"VAR_SOUNDCODE_TMR",
	"VAR_DEBUG_ENABLED",
	"VAR_KEY_STATE",
	NULL,
	/* 88 */
	"VAR_NUM_SOUND_CHANNELS",
	"VAR_COLOR_DEPTH",
	NULL,
	"VAR_PRINTING_ENABED",
	/* 92 */
	"VAR_FADE_COLOR",
	"VAR_BLACK_COLOR",
	"VAR_WHITE_COLOR",
	"VAR_REDRAW_ALL_ACTORS",
	/* 96 */
	"VAR_TIMER_1",
	"VAR_TIMER_2",
	"VAR_NETWORK_START_SCRIPT",
	"VAR_NETWORK_TIMEOUT",
	/* 100 */
	"VAR_NETWORK_ENABLED",
	"VAR_NETWORK_ARRAY_SCRIPT",
	NULL,
	"VAR_SCRIPT_CYCLE",
	/* 104 */
	"VAR_NUM_SCRIPT_CYCLES",
	"VAR_NUM_SPRITE_GROUPS",
	"VAR_NUM_SPRITES",
	"VAR_U32_VERSION",
	/* 108 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 112 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 116 */
	"VAR_U32_ARRAY_UNK",
	"VAR_WIZ_TCOLOR",
	"VAR_SMACKER_ENABLED",
	"VAR_ERROR_CODE",
	/* 120 */
	"VAR_RESERVED_SOUND_CHANNELS",
	NULL,
	"VAR_JOYSTICK_UNK1",
	"VAR_JOYSTICK_UNK2",
	/* 124 */
	"VAR_JOYSTICK_DETECTED",
	"VAR_SKIP_RESET_TALK_ACTOR",
	"VAR_UPDATE_SCREEN_TYPE",
	"VAR_MAIN_SCRIPT",
	/* 128 */
	"VAR_JOYSTICK_DISABLED",
	"VAR_BASE_NAME",
	"VAR_NUM_PALETTES",
	"VAR_NUM_WINDOWS",
	/* 132 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 136 */
	NULL,
	NULL,
	NULL,
	"VAR_SET_SPRITE_DIFF",
	/* 140 */
	"VAR_BINK_ENABLED",
	NULL,
	NULL,
	NULL,
};

const char *var_names6[] = {
	/* 0 */
	NULL,
	"VAR_EGO",
	"VAR_CAMERA_POS_X",
	"VAR_HAVE_MSG",
	/* 4 */
	"VAR_ROOM",
	"VAR_OVERRIDE",
	"VAR_MACHINE_SPEED",
	NULL,
	/* 8 */
	"VAR_NUM_ACTOR",
	"VAR_V6_SOUNDMODE",
	"VAR_CURRENTDRIVE",
	"VAR_TMR_1",
	/* 12 */
	"VAR_TMR_2",
	"VAR_TMR_3",
	NULL,
	NULL,
	/* 16 */
	NULL,
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_TIMER_NEXT",
	/* 20 */
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_ROOM_RESOURCE",
	"VAR_LAST_SOUND",
	/* 24 */
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_TALK_ACTOR",
	"VAR_CAMERA_FAST_X",
	"VAR_SCROLL_SCRIPT",
	/* 28 */
	"VAR_ENTRY_SCRIPT",
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	/* 32 */
	"VAR_VERB_SCRIPT",
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	/* 36 */
	"VAR_CUTSCENE_END_SCRIPT",
	"VAR_CHARINC",
	"VAR_WALKTO_OBJ",
	"VAR_DEBUGMODE",
	/* 40 */
	"VAR_HEAPSPACE",
	"VAR_ROOM_WIDTH",
	"VAR_RESTART_KEY",
	"VAR_PAUSE_KEY",
	/* 44 */
	"VAR_MOUSE_X",
	"VAR_MOUSE_Y",
	"VAR_TIMER",
	"VAR_TMR_4",
	/* 48 */
	NULL,
	"VAR_VIDEOMODE",
	"VAR_MAINMENU_KEY",
	"VAR_FIXEDDISK",
	/* 52 */
	"VAR_CURSORSTATE",
	"VAR_USERPUT",
	"VAR_ROOM_HEIGHT",
	NULL,
	/* 56 */
	"VAR_SOUNDRESULT",
	"VAR_TALKSTOP_KEY",
	NULL,
	"VAR_FADE_DELAY",
	/* 60 */
	"VAR_NOSUBTITLES",
	"VAR_SAVELOAD_SCRIPT",
	"VAR_SAVELOAD_SCRIPT2",
	NULL,
	/* 64 */
	"VAR_SOUNDPARAM",
	"VAR_SOUNDPARAM2",
	"VAR_SOUNDPARAM3",
	"VAR_INPUTMODE",
	/* 68 */
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	"VAR_ROOM_FLAG",
	"VAR_GAME_LOADED",
	/* 72 */
	"VAR_NEW_ROOM",
	NULL,
	"VAR_LEFTBTN_DOWN",
	"VAR_RIGHTBTN_DOWN",
	/* 76 */
	"VAR_V6_EMSSPACE",
	NULL,
	NULL,
	NULL,
	/* 80 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	"VAR_GAME_DISK_MSG",
	"VAR_OPEN_FAILED_MSG",
	/* 92 */
	"VAR_READ_ERROR_MSG",
	"VAR_PAUSE_MSG",
	"VAR_RESTART_MSG",
	"VAR_QUIT_MSG",
	/* 96 */
	"VAR_SAVE_BTN",
	"VAR_LOAD_BTN",
	"VAR_PLAY_BTN",
	"VAR_CANCEL_BTN",
	/* 100 */
	"VAR_QUIT_BTN",
	"VAR_OK_BTN",
	"VAR_SAVE_DISK_MSG",
	"VAR_ENTER_NAME_MSG",
	/* 104 */
	"VAR_NOT_SAVED_MSG",
	"VAR_NOT_LOADED_MSG",
	"VAR_SAVE_MSG",
	"VAR_LOAD_MSG",
	/* 108 */
	"VAR_SAVE_MENU_TITLE",
	"VAR_LOAD_MENU_TITLE",
	"VAR_GUI_COLORS",
	"VAR_DEBUG_PASSWORD",
	/* 112 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 116 */
	NULL,
	"VAR_MAIN_MENU_TITLE",
	"VAR_RANDOM_NR",
	"VAR_TIMEDATE_YEAR",
	/* 120 */
	NULL,
	"VAR_GAME_VERSION",
	NULL,
	"VAR_CHARSET_MASK",
	/* 124 */
	NULL,
	"VAR_TIMEDATE_HOUR",
	"VAR_TIMEDATE_MINUTE",
	NULL,
	/* 128 */
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_MONTH",
	NULL,
	NULL,
};

const char *var_names7[] = {
	/* 0 */
	NULL,
	"VAR_MOUSE_X",
	"VAR_MOUSE_Y",
	"VAR_VIRT_MOUSE_X",
	/* 4 */
	"VAR_VIRT_MOUSE_Y",
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	"VAR_CAMERA_POS_X",
	/* 8 */
	"VAR_CAMERA_POS_Y",
	"VAR_OVERRIDE",
	"VAR_ROOM",
	"VAR_ROOM_RESOURCE",
	/* 12 */
	"VAR_TALK_ACTOR",
	"VAR_HAVE_MSG",
	"VAR_TIMER",
	"VAR_TMR_4",
	/* 16 */
	"VAR_TIMEDATE_YEAR",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_HOUR",
	/* 20 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_SECOND",
	"VAR_LEFTBTN_DOWN",
	"VAR_RIGHTBTN_DOWN",
	/* 24 */
	"VAR_LEFTBTN_HOLD",
	"VAR_RIGHTBTN_HOLD",
	"VAR_MEMORY_PERFORMANCE",
	"VAR_VIDEO_PERFORMANCE",
	/* 28 */
	NULL,
	"VAR_GAME_LOADED",
	NULL,
	NULL,
	/* 32 */
	"VAR_V6_EMSSPACE",
	"VAR_VOICE_MODE",
	"VAR_RANDOM_NR",
	"VAR_NEW_ROOM",
	/* 36 */
	"VAR_WALKTO_OBJ",
	"VAR_NUM_GLOBAL_OBJS",
	"VAR_CAMERA_DEST_X",
	"VAR_CAMERA_DEST_>",
	/* 40 */
	"VAR_CAMERA_FOLLOWED_ACTOR",
	NULL,
	NULL,
	NULL,
	/* 44 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 48 */
	NULL,
	NULL,
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	/* 52 */
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	/* 56 */
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	/* 60 */
	"VAR_SAVELOAD_SCRIPT",
	"VAR_SAVELOAD_SCRIPT2",
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_RESTART_KEY",
	/* 64 */
	"VAR_PAUSE_KEY",
	"VAR_MAINMENU_KEY",
	"VAR_VERSION_KEY",
	"VAR_TALKSTOP_KEY",
	/* 68 */
	NULL,
	NULL,
	"VAR_SAVE_BTN",
	"VAR_LOAD_BTN",
	/* 72 */
	"VAR_PLAY_BTN",
	"VAR_CANCEL_BTN",
	"VAR_QUIT_BTN",
	"VAR_OK_BTN",
	/* 76 */
	"VAR_SAVE_MENU_TITLE",
	"VAR_LOAD_MENU_TITLE",
	"VAR_ENTER_NAME_MSG",
	"VAR_SAVE_MSG",
	/* 80 */
	"VAR_LOAD_MSG",
	"VAR_NOT_SAVED_MSG",
	"VAR_NOT_LOADED_MSG",
	"VAR_OPEN_FAILED_MSG",
	/* 84 */
	"VAR_READ_ERROR_MSG",
	"VAR_PAUSE_MSG",
	"VAR_RESTART_MSG",
	"VAR_QUIT_MSG",
	/* 88 */
	NULL,
	"VAR_DEBUG_PASSWORD",
	NULL,
	NULL,
	/* 92 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 96 */
	NULL,
	"VAR_TIMER_NEXT",
	"VAR_TMR_1",
	"VAR_TMR_2",
	/* 100 */
	"VAR_TMR_3",
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_CAMERA_MIN_Y",
	/* 104 */
	"VAR_CAMERA_MAX_Y",
	"VAR_CAMERA_THRESHOLD_X",
	"VAR_CAMERA_THRESHOLD_Y",
	"VAR_CAMERA_SPEED_X",
	/* 108 */
	"VAR_CAMERA_SPEED_Y",
	"VAR_CAMERA_ACCEL_X",
	"VAR_CAMERA_ACCEL_Y",
	"VAR_EGO",
	/* 112 */
	"VAR_CURSORSTATE",
	"VAR_USERPUT",
	"VAR_DEFAULT_TALK_DELAY",
	"VAR_CHARINC",
	/* 116 */
	"VAR_DEBUGMODE",
	"VAR_FADE_DELAY",
	NULL,
	"VAR_CHARSET_MASK",
	/* 120 */
	NULL,
	NULL,
	NULL,
	"VAR_VIDEONAME",
	/* 124 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 128 */
	NULL,
	NULL,
	"VAR_STRING2DRAW",
	"VAR_CUSTOMSCALETABLE",
	/* 132 */
	NULL,
	"VAR_BLAST_ABOVE_TEXT",
	NULL,
	"VAR_MUSIC_BUNDLE_LOADED",
	/* 136 */
	"VAR_VOICE_BUNDLE_LOADED",
	NULL,
};

const char *var_names8[] = {
	/* 0 */
	NULL,
	"VAR_ROOM_WIDTH",
	"VAR_ROOM_HEIGHT",
	"VAR_MOUSE_X",
	/* 4 */
	"VAR_MOUSE_Y",
	"VAR_VIRT_MOUSE_X",
	"VAR_VIRT_MOUSE_Y",
	"VAR_CURSORSTATE",
	/* 8 */
	"VAR_USERPUT",
	"VAR_CAMERA_POS_X",
	"VAR_CAMERA_POS_Y",
	"VAR_CAMERA_DEST_X",
	/* 12 */
	"VAR_CAMERA_DEST_Y",
	"VAR_CAMERA_FOLLOWED_ACTOR",
	"VAR_TALK_ACTOR",
	"VAR_HAVE_MSG",
	/* 16 */
	"VAR_LEFTBTN_DOWN",
	"VAR_RIGHTBTN_DOWN",
	"VAR_LEFTBTN_HOLD",
	"VAR_RIGHTBTN_HOLD",
	/* 20 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 24 */
	"VAR_TIMEDATE_YEAR",
	"VAR_TIMEDATE_MONTH",
	"VAR_TIMEDATE_DAY",
	"VAR_TIMEDATE_HOUR",
	/* 28 */
	"VAR_TIMEDATE_MINUTE",
	"VAR_TIMEDATE_SECOND",
	"VAR_OVERRIDE",
	"VAR_ROOM",
	/* 32 */
	"VAR_NEW_ROOM",
	"VAR_WALKTO_OBJ",
	"VAR_TIMER",
	NULL,
	/* 36 */
	NULL,
	NULL,
	NULL,
	"VAR_VOICE_MODE",
	/* 40 */
	"VAR_GAME_LOADED",
	"VAR_LANGUAGE",
	"VAR_CURRENTDISK",
	NULL,
	/* 44 */
	NULL,
	"VAR_MUSIC_BUNDLE_LOADED",
	"VAR_VOICE_BUNDLE_LOADED",
	NULL,
	/* 48 */
	NULL,
	NULL,
	"VAR_SCROLL_SCRIPT",
	"VAR_ENTRY_SCRIPT",
	/* 52 */
	"VAR_ENTRY_SCRIPT2",
	"VAR_EXIT_SCRIPT",
	"VAR_EXIT_SCRIPT2",
	"VAR_VERB_SCRIPT",
	/* 56 */
	"VAR_SENTENCE_SCRIPT",
	"VAR_INVENTORY_SCRIPT",
	"VAR_CUTSCENE_START_SCRIPT",
	"VAR_CUTSCENE_END_SCRIPT",
	/* 60 */
	NULL,
	NULL,
	"VAR_CUTSCENEEXIT_KEY",
	"VAR_RESTART_KEY",
	/* 64 */
	"VAR_PAUSE_KEY",
	"VAR_MAINMENU_KEY",
	"VAR_VERSION_KEY",
	"VAR_TALKSTOP_KEY",
	/* 68 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 72 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 76 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 80 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 84 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 88 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 92 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 96 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 100 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 104 */
	NULL,
	NULL,
	NULL,
	NULL,
	/* 108 */
	NULL,
	NULL,
	NULL,
	"VAR_CUSTOMSCALETABLE",
	/* 112 */
	"VAR_TIMER_NEXT",
	"VAR_TMR_1",
	"VAR_TMR_2",
	"VAR_TMR_3",
	/* 116 */
	"VAR_CAMERA_MIN_X",
	"VAR_CAMERA_MAX_X",
	"VAR_CAMERA_MIN_Y",
	"VAR_CAMERA_MAX_Y",
	/* 120 */
	"VAR_CAMERA_SPEED_X",
	"VAR_CAMERA_SPEED_Y",
	"VAR_CAMERA_ACCEL_X",
	"VAR_CAMERA_ACCEL_Y",
	/* 124 */
	"VAR_CAMERA_THRESHOLD_X",
	"VAR_CAMERA_THRESHOLD_Y",
	"VAR_EGO",
	NULL,
	/* 128 */
	"VAR_DEFAULT_TALK_DELAY",
	"VAR_CHARINC",
	"VAR_DEBUGMODE",
	NULL,
	/* 132 */
	"VAR_KEYPRESS",
	"VAR_BLAST_ABOVE_TEXT",
	"VAR_SYNC",
	NULL,
};

const char *getVarName(uint var) {
	if (g_options.heVersion >= 72) {
		if (var >= sizeof(var_names72) / sizeof(var_names72[0]))
			return NULL;
		return var_names72[var];
	} else if (g_options.scriptVersion == 8) {
		if (var >= sizeof(var_names8) / sizeof(var_names8[0]))
			return NULL;
		return var_names8[var];
	} else if (g_options.scriptVersion == 7) {
		if (var >= sizeof(var_names7) / sizeof(var_names7[0]))
			return NULL;
		return var_names7[var];
	} else {
		if (var >= sizeof(var_names6) / sizeof(var_names6[0]))
			return NULL;
		return var_names6[var];
	}
}

StackEnt *se_neg(StackEnt *se) {
	return new NegStackEnt(se);
}

StackEnt *se_int(int i) {
	return new IntStackEnt(i);
}

StackEnt *se_var(int i) {
	return new VarStackEnt(i);
}

StackEnt *se_array(int i, StackEnt *dim2, StackEnt *dim1) {
	return new ArrayStackEnt(i, dim2, dim1);
}

StackEnt *se_oper(StackEnt *a, int op) {
	return new UnaryOpStackEnt(op, a);
}

StackEnt *se_oper(StackEnt *a, int op, StackEnt *b) {
	return new BinaryOpStackEnt(op, a, b);
}

StackEnt *se_complex(const char *s) {
	return new ComplexStackEnt(s);
}

char *se_astext(StackEnt *se, char *where, bool wantparens = true) {
	return se->asText(where, wantparens);
}

StackEnt *se_get_list() {
	return new ListStackEnt(pop());
}

char *get_var6(char *buf) {
	VarStackEnt tmp(get_word());
	return tmp.asText(buf);
}

void invalidop(const char *cmd, int op) {
	if (cmd)
		error("Unknown opcode %s:0x%x (stack count %d)", cmd, op, num_stack);
	else
		error("Unknown opcode 0x%x (stack count %d)", op, num_stack);
}

void push(StackEnt *se) {
	assert(se);
	assert(num_stack < MAX_STACK_SIZE);
	stack[num_stack++] = se;
}

StackEnt *pop() {
	if (num_stack == 0) {
		printf("ERROR: No items on stack to pop!\n");

		if (!g_options.haltOnError)
			return se_complex("**** INVALID DATA ****");
		exit(1);
	}
	return stack[--num_stack];
}


void kill(char *output, StackEnt *se) {
	if (se->type != seDup) {
		char *e = strecpy(output, "pop(");
		e = se_astext(se, e);
		strcpy(e, ")");
		delete se;
	} else {
		// FIXME: Evil hack: We re-push DUPs, instead of killing
		// them. We do this to support switch-case constructs
		// (see comment at the start of this file) w/o applying a full
		// flow analysis (which would normally be required).
		push(se);
	}
}

void doAssign(char *output, StackEnt *dst, StackEnt *src) {
	if (src->type == seDup && dst->type == seDup) {
		((DupStackEnt *)dst)->_idx = ((DupStackEnt *)src)->_idx;
		return;
	}
	char *e = se_astext(dst, output);
	e = strecpy(e, " = ");
	se_astext(src, e);
}

StackEnt* StackEnt::dup(char *output) {
	StackEnt *dse = new DupStackEnt(++dupindex);
	doAssign(output, dse, this);
	return dse;
}

void doAdd(char *output, StackEnt *se, int val) {
	char *e = se_astext(se, output);
	if (val == 1) {
		sprintf(e, "++");
	} else if (val == -1) {
		sprintf(e, "--");
	} else {
		/* SCUMM doesn't support this */
		sprintf(e, " += %d", val);
	}
}

StackEnt *dup(char *output, StackEnt *se) {
	return se->dup(output);
}

void writeArray(char *output, int i, StackEnt *dim2, StackEnt *dim1, StackEnt *value) {
	StackEnt *array = se_array(i, dim2, dim1);
	doAssign(output, array, value);
	delete array;
}

void writeVar(char *output, int i, StackEnt *value) {
	StackEnt *se = se_var(i);
	doAssign(output, se, value);
	delete se;
}

void addArray(char *output, int i, StackEnt *dim1, int val) {
	StackEnt *array = se_array(i, NULL, dim1);
	doAdd(output, array, val);
	delete array;
}

void addVar(char *output, int i, int val) {
	StackEnt *se = se_var(i);
	doAdd(output, se, val);
	delete se;
}


StackEnt *se_get_string() {
	char buf[1024];
	get_string(buf); // ignore returned value
	return se_complex(buf);
}

int _stringLength = 1;
byte _stringBuffer[4096];

void getScriptString() {
	byte chr;

	while ((chr = get_byte()) != 0) {
		_stringBuffer[_stringLength] = chr;
		_stringLength++;

		if (_stringLength >= 4096)
			error("String stack overflow");
	}

	_stringBuffer[_stringLength] = 0;
	_stringLength++;
}

StackEnt *se_get_string_he() {
	char buf[1024];
	char *e = buf;

	byte string[1024];
	byte chr;
	int len = 1;
	StackEnt *value;

	value = pop();

	*e++ = '"';
	if (value->type == seInt && value->getIntVal() == -1) {
		if (_stringLength == 1) {
			*e++ = '"';
			*e++ = 0;
			return se_complex(buf);
		}

		_stringLength -= 2;
		 while ((chr = _stringBuffer[_stringLength]) != 0) {
			string[len++] = chr;
			_stringLength--;
		}

		string[len] = 0;
		_stringLength++;

		// Reverse string
		while (--len)
			*e++ = string[len];
	} else {
		e += sprintf(e, ":");
		e = value->asText(e);
		e += sprintf(e, ":");
	}
	*e++ = '"';

	*e++ = 0;
	return se_complex(buf);
}

void ext(char *output, const char *fmt) {
	bool wantresult;
	byte cmd, extcmd;
	const char *extstr = NULL;
	const char *prep = NULL;
	StackEnt *args[20];
	int numArgs = 0;
	char *e = (char *)output;

	/* return the result? */
	wantresult = false;
	if (*fmt == 'r') {
		wantresult = true;
		fmt++;
	}

	while ((cmd = *fmt++) != '|') {
		if (cmd == 'x' && !extstr) {
			/* Sub-op: next byte specifies which one */
			extstr = fmt;
			while (*fmt++)
				;
			e += sprintf(e, "%s.", extstr);

			/* extended thing */
			extcmd = get_byte();

			/* locate our extended item */
			while ((cmd = *fmt++) != extcmd) {
				/* scan until we find , or \0 */
				while ((cmd = *fmt++) != ',') {
					if (cmd == 0) {
						invalidop(extstr, extcmd);
					}
				}
			}
			/* found a command, continue at the beginning */
			continue;
		}
		if (cmd == 'u' && !extstr) {
			/* Sub-op: next byte specifies which one */
			extstr = fmt;
			while (*fmt++)
				;
			e += sprintf(e, "%s.", extstr);

			/* extended thing */
			extcmd = pop()->getIntVal();

			/* locate our extended item */
			while ((cmd = *fmt++) != extcmd) {
				/* scan until we find , or \0 */
				while ((cmd = *fmt++) != ',') {
					if (cmd == 0) {
						invalidop(extstr, extcmd);
					}
				}
			}
			/* found a command, continue at the beginning */
			continue;
		}
		if (cmd == 'y' && !extstr) {
			/* Sub-op: parameters are in a list, first element of the list specified the command */
			ListStackEnt *se;
			extstr = fmt;
			while (*fmt++)
				;
			e += sprintf(e, "%s.", extstr);

			se = new ListStackEnt(pop());
			args[numArgs++] = se;

			/* extended thing */
			se->_size--;
			extcmd = (byte) se->_list[se->_size]->getIntVal();

			/* locate our extended item */
			while ((cmd = *fmt++) != extcmd) {
				/* scan until we find , or \0 */
				while ((cmd = *fmt++) != ',') {
					if (cmd == 0) {
						/* End reached and command was not found: re-add the extcmd to
						   the list and output the whole thing as "unknown".
						 */
						se->_size++;
						fmt = "Unknown";
						goto output_command;
					}
				}
			}

			/* found a command, continue at the beginning */
			continue;
		}

		if (cmd == 'p') {
			args[numArgs++] = pop();
		} else if (cmd == 'z') {	// = popRoomAndObj()
			args[numArgs++] = pop();
			if (g_options.scriptVersion < 7 && g_options.heVersion == 0)
				args[numArgs++] = pop();
		} else if (cmd == 'h') {
			if (g_options.heVersion >= 72)
				args[numArgs++] = se_get_string_he();
		} else if (cmd == 's') {
			args[numArgs++] = se_get_string();
		} else if (cmd == 'w') {
			args[numArgs++] = se_int(get_word());
		} else if (cmd == 'i') {
			args[numArgs++] = se_int(get_byte());
		} else if (cmd == 'l') {
			args[numArgs++] = se_get_list();
		} else if (cmd == 'j') {
			args[numArgs++] = se_int(get_word());
		} else if (cmd == 'v') {
			args[numArgs++] = se_var(get_word());
		} else {
			error("Character '%c' unknown in argument string '%s', \n", fmt, cmd);
		}
	}

output_command:

	/* create a string from the arguments */
	if (prep)
		e = strecpy(e, prep);
	while (*fmt != 0 && *fmt != ',')
		*e++ = *fmt++;
	*e++ = '(';
	while (--numArgs >= 0) {
		e = se_astext(args[numArgs], e);
		if (numArgs)
			*e++ = ',';
	}
	*e++ = ')';
	*e = 0;

	if (wantresult) {
		push(se_complex((char *)output));
		output[0] = 0;
		return;
	}
}

void jump(char *output) {
	int offset = get_word();
	int cur = get_curoffs();
	int to = cur + offset;

	if (offset == 1) {
		// Sometimes, jumps with offset 1 occur. I used to suppress those.
		// But it turns out that's not quite correct in some cases. With this
		// code, it can sometimes happens that you get an empty 'else' branch;
		// but in many other cases, an otherwise hidden instruction is revealed,
		// or an instruction is placed into an else branch instead of being
		// (incorrectly) placed inside the body of the 'if' itself.
		sprintf(output, "/* jump %x; */", to);
	} else if (!g_options.dontOutputElse && maybeAddElse(cur, to)) {
		pendingElse = true;
		pendingElseTo = to;
		pendingElseOffs = cur;
		pendingElseOpcode = g_jump_opcode;
		pendingElseIndent = g_blockStack.size();
	} else {
		if (!g_blockStack.empty() && !g_options.dontOutputWhile) {
			Block p = g_blockStack.top();
			if (p.isWhile && cur == (int)p.to)
				return;		// A 'while' ends here.
			if (!g_options.dontOutputBreaks && maybeAddBreak(cur, to)) {
				sprintf(output, "break");
				return;
		  }
		}
		sprintf(output, "jump %x", to);
	}
}

void jumpif(char *output, StackEnt *se, bool negate) {
	int offset = get_word();
	int cur = get_curoffs();
	int to = cur + offset;
	char *e = output;

	if (!g_options.dontOutputElseif && pendingElse) {
		if (maybeAddElseIf(cur, pendingElseTo, to)) {
			pendingElse = false;
			haveElse = true;
			e = strecpy(e, "} else if (");
			e = se_astext(se, e, false);
			sprintf(e, g_options.alwaysShowOffs ? ") /*%.4X*/ {" : ") {", to);
			return;
		}
	}

	if (!g_options.dontOutputIfs && maybeAddIf(cur, to)) {
		if (!g_options.dontOutputWhile && g_blockStack.top().isWhile)
			e = strecpy(e, negate ? "until (" : "while (");
		else
			e = strecpy(e, negate ? "unless (" : "if (");
		e = se_astext(se, e, false);
		sprintf(e, g_options.alwaysShowOffs ? ") /*%.4X*/ {" : ") {", to);
		return;
	}

	e = strecpy(e, negate ? "if (" : "unless (");
	e = se_astext(se, e);
	sprintf(e, ") jump %x", to);
}

void dup_n() {
	push(se_int(get_word()));
	ListStackEnt *list = (ListStackEnt *)se_get_list();

	for (int i = 0; i < 2; i++)
		for (int j = 0; j < list->_size; j++)
			push(list->_list[j]);
}


#define PRINT_V100HE(name)           \
	do {                          \
		ext(output, "x" name "\0"         \
				"\x6pp|XY,"      \
				"\xC|center,"    \
				"\x12p|right,"    \
				"\x14p|color,"    \
				"\x15l|colors,"	  \
				"\x23lps|debug,"  \
				"\x2E|left,"      \
				"\x33|mumble,"    \
				"\x38|overhead,"  \
				"\x4Ep|getText,"  \
				"\x4Fs|msg,"      \
				"\x5B|begin,"     \
				"\x5C|end"        \
				);                \
	} while(0)

void next_line_HE_V100(char *output) {
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x0:
		ext(output, "x" "actorOps\0"
				"\x3ppp|actorSet:3:??,"
				"\x4p|setAnimSpeed,"
				"\x6pp|putActor,"
				"\x8|resetDrawToBackBuf,"
				"\x9|drawToBackBuf,"
				"\xEp|charset,"
				"\x12pppp|setClipRect,"
				"\x16l|setUserConditions,"
				"\x19p|setCostume,"
				"\x1B|init,"
				"\x20p|setHEFlag,"
				"\x34h|setName,"
				"\x35|initLittle,"
				"\x39pp|setPalette,"
				"\x3Bp|layer,"
				"\x3Fp|setPaletteNum,"
				"\x41p|setScale,"
				"\x46p|setShadowMode,"
				"\x4App|setWalkSpeed,"
				"\x4Ehp|setTalkieSlot,"
				"\x53pp|setAnimVar,"
				"\x57p|setAlwayZClip,"
				"\x59|setNeverZClip,"
				"\x80pppp|setActorClipRect,"
				"\x81p|setCurActor,"
				"\x82l|setSound,"
				"\x83p|setWidth,"
				"\x84|setDefAnim,"
				"\x85p|setElevation,"
				"\x86|setFollowBoxes,"
				"\x87|setIgnoreBoxes,"
				"\x88|setIgnoreTurnsOff,"
				"\x89|setIgnoreTurnsOn,"
				"\x8Ap|setInitFrame,"
				"\x8Bp|setStandFrame,"
				"\x8Cpp|setTalkFrame,"
				"\x8Dp|setTalkColor,"
				"\x8Ep|setTalkCondition,"
				"\x8Fpp|setTalkPos,"
				"\x90p|setWalkFrame");
		break;
	case 0x1:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 6 + isEqual, se_a));
		break;
	case 0x2:
		ext(output, "pp|faceActor");
		break;

	case 0x03:
		ext(output, "x" "sortArray\0"
				"\x86pppppv|sort,");
		break;
	case 0x04:
		switch (get_byte()) {
		case 35:
			se_get_list();
			pop();
			se_a = se_get_string_he();
			writeArray(output, get_word(), NULL, se_a, se_a);
			break;
		case 77:
			se_a = se_get_string_he();
			writeArray(output, get_word(), NULL, se_a, se_a);
			break;
		case 128:
			se_b = se_get_list();
			se_a = pop();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 129:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 130:
			// TODO: Add support for writeListIntoArray
			ext(output, "lppppv|writeListIntoArray");
			break;
		case 131:
			// TODO: Add support for copyArray
			ext(output, "ppppvppppv|copyArray");
			break;
		case 132:
			// TODO: Add support for math operations
			ext(output, "pppppppppppppvvv|mathOpsInArray");
			break;
		case 133:
			// TODO: Add support for writeRangeIntoArray
			ext(output, "ppppppv|writeRangeIntoArray");
			break;
		}
		break;
	case 0x05:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBand, se_a));
		break;
	case 0x06:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBor, se_a));
		break;
	case 0x07:
		ext(output, "|breakHere");
		break;
	case 0x08:
		ext(output, "p|delayFrames");
		break;
	case 0x09:
		ext(output, "rpp|shl");
		break;
	case 0x0A:
		ext(output, "rpp|shr");
		break;
	case 0x0B:
		ext(output, "rpp|xor");
		break;
	case 0x0C:
		ext(output, "p|setCameraAt");
		break;
	case 0x0D:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x0E:
		ext(output, "p|loadRoom");
		break;
	case 0x0F:
		ext(output, "p|panCameraTo");
		break;
	case 0x10:
		ext(output, "ppppp|captureWizImage");
		break;
	case 0x11:
		ext(output, "lpi|jumpToScript");
		break;
	case 0x12:
		ext(output, "lp|setClass");
		break;
	case 0x13:
		ext(output, "p|closeFile");
		break;
	case 0x14:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x16:
		ext(output, "h|createDirectory");
		break;
	case 0x17:
		ext(output, "x" "createSound\0"
				"\x0p|setId,"
				"\x35|reset,"
				"\x5C|dummy,"
				"\x80p|create");
		break;
	case 0x18:
		ext(output, "l|beginCutscene");
		break;
	case 0x19:
	case 0x53:
		kill(output, pop());
		break;
	case 0x1A:
		ext(output, "hp|traceStatus");
		break;
	case 0x1B:
		addVar(output, get_word(), -1);
		break;
	case 0x1C:
		addArray(output, get_word(), pop(), -1);
		break;
	case 0x1D:
		ext(output, "h|deleteFile");
		break;
	case 0x1E:
		ext(output, "x" "dim2dimArray\0"
				"\x29ppv|bit,"
				"\x2Appv|int,"
				"\x2Bppv|dword,"
				"\x2Cppv|nibble,"
				"\x2Dppv|byte,"
				"\x4Dppv|string");
		break;
	case 0x1F:
		ext(output, "x" "dimArray\0"
				"\x29pv|bit,"
				"\x2Apv|int,"
				"\x2Bpv|dword,"
				"\x2Cpv|nibble,"
				"\x2Dpv|byte,"
				"\x4Dpv|string,"
				"\x87v|nukeArray");
		break;
	case 0x20:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 9 + isEqual, se_a));
		break;
	case 0x21:
		ext(output, "pp|animateActor");
		break;
	case 0x22:
		ext(output, "pppp|doSentence");
		break;
	case 0x23:
		ext(output, "ppppp|drawBox");
		break;
	case 0x24:
		ext(output, "pppp|drawWizImage");
		break;
	case 0x25:
		ext(output, "pp|drawWizPolygon");
		break;
	case 0x26:
		ext(output, "x" "drawLine\0"
				"\x1pppppp|pixel,"
				"\x14pppppp|wizImage,"
				"\x28pppppp|actor");
		break;
	case 0x27:
		ext(output, "x" "drawObject\0"
				"\x6ppp|setPosition,"
				"\x7pppp|setup,"
				"\x28pp|setState");
		break;
	case 0x28:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0x29:
		dup_n();
		break;
	case 0x2A:
		ext(output, "|endCutscene");
		break;
	case 0x2B:
		ext(output, "|stopObjectCodeA");
		break;
	case 0x2C:
		ext(output, "|stopObjectCodeB");
		break;
	case 0x2D:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 0 + isEqual, se_a));
		break;
	case 0x2E:
		ext(output, "x" "floodFill\0"
				"\x0p|reset,"
				"\x6pp|setXY,"
				"\x12pppp|setBoxRect,"
				"\x14p|setFlags,"
				"\x36|dummy,"
				"\x5C|floodFill");
		break;
	case 0x2F:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x30:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 5 + isEqual, se_a));
		break;
	case 0x31:
		ext(output, "|getDateTime");
		break;
	case 0x32:
		ext(output, "x" "setSpriteGroupInfo\0"
				"\x0p|setId,"
				"\x6pp|setPosition,"
				"\x12pppp|setBounds,"
				"\x26pp|misc,"
				"\x28p|setImage,"
				"\x31pp|move,"
				"\x34h|stringUnk,"
				"\x35|reset,"
				"\x36pp|dummy,"
				"\x3Bp|setPriority,"
				"\x3Cpp|setXYScale?,"
				"\x59|resetBounds");
		break;
	case 0x33:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 2 + isEqual, se_a));
		break;
	case 0x34:
		ext(output, "x" "resourceRoutines\0"
				"\xEp|setCharset,"
				"\x19p|setCostume,"
				"\x22p|setFlObject,"
				"\x28p|setImage,"
				"\x2F|loadResource,"
				"\x3Ep|setRoom,"
				"\x42p|setScript,"
				"\x48p|setSound,"
				"\x80|clearHeap,"
				"\x81|dummy,"
				"\x84|lockResource,"
				"\x85|nukeResource,"
				"\x86|setOffHeapOn,"
				"\x87|setOffHeapOff,"
				"\x88|queueResource,"
				"\x89|unlock");
		break;
	case 0x35:
		jumpif(output, pop(), true);
		break;
	case 0x36:
		jumpif(output, pop(), false);
		break;
	case 0x37:
		ext(output, "x" "wizImageOps\0"
				"\x0p|setImage,"
				"\x6pp|setPosition,"
				"\x7p|setSourceImage,"
				"\xBppppp|setCaptureRect,"
				"\x12pppp|setClipRect,"
				"\x15p|remapWizImagePal,"
				"\x1D|processMode1,"
				"\x24pppp|setRect,"
				"\x27p|setResDefImageHeight,"
				"\x2Fh|readFile,"
				"\x35|createWizImage,"
				"\x36pp|setThickLine,"
				"\x37ppppp|drawWizImage,"
				"\x39p|setPalette,"
				"\x3Appp|capturePolygon,"
				"\x40hp|writeFile,"
				"\x41p|setScale,"
				"\x43p|setFlags,"
				"\x44p|setupPolygon,"
				"\x46p|setShadow,"
				"\x49p|setImageState,"
				"\x54p|setResDefImageWidth,"
				"\x5C|processWizImage,"
				"\x80pppph|createFont,"
				"\x81|endFont,"
				"\x82pph|renderFontString,"
				"\x83|startFont,"
				"\x84pp|setPosition,"
				"\x85pppppppp|ellipse,"
				"\x86ppp|fillWizFlood,"
				"\x87p|setDstResNum,"
				"\x88ppppp|fillWizLine,"
				"\x89ppp|fillWizPixel,"
				"\x8Appppp|fillWizRect");
		break;
	case 0x38:
		ext(output, "rlp|isAnyOf2");
		break;
	case 0x39:
		addVar(output, get_word(), 1);
		break;
	case 0x3A:
		addArray(output, get_word(), pop(), 1);
		break;
	case 0x3B:
		jump(output);
		break;
	case 0x3C:
		ext(output, "y" "kernelSetFunctions\0"
				"\x1|virtScreenLoad,"
				"\x14|queueAuxBlock,"
				"\x15|pauseDrawObjects,"
				"\x16|resumeDrawObjects,"
				"\x17|clearCharsetMask,"
				"\x18|pauseActors,"
				"\x19|resumActors,"
				"\x1E|actorBottomClipOverride,"
				"\x2A|setWizImageClip,"
				"\x2B|setWizImageClipOff");
		break;
	case 0x3D:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 0xA + isEqual, se_a));
		break;
	case 0x3E:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 4 + isEqual, se_a));
		break;
	case 0x3F:
		ext(output, "p|localizeArrayToScript");
		break;
	case 0x40:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0x41:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0x42:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 0xB + isEqual, se_a));
		break;
	case 0x43:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 3 + isEqual, se_a));
		break;
	case 0x44:
		ext(output, "rpp|mod");
		break;
	case 0x45:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 8 + isEqual, se_a));
		break;
	case 0x46:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 1 + isEqual, se_a));
		break;
	case 0x47:
		ext(output, "x" "dim2dim2Array\0"
				"\x29pppppv|bit,"
				"\x2Apppppv|int,"
				"\x2Bpppppv|dword,"
				"\x2Cpppppv|nibble,"
				"\x2Dpppppv|byte,"
				"\x4Dpppppv|string");
		break;
	case 0x49:
		ext(output, "x" "redim2dimArray\0"
				"\x2Appppv|int,"
				"\x2Bppppv|dword,"
				"\x2Dppppv|byte");
		break;
	case 0x4A:
		push(se_oper(pop(), isZero));
		break;
	case 0x4C:
		ext(output, "|beginOverride");
		break;
	case 0x4D:
		ext(output, "|endOverride");
		break;
	case 0x4E:
		ext(output, "|resetCutScene");
		break;
	case 0x4F:
		ext(output, "pp|setOwner");
		break;
	case 0x50:
		ext(output, "x" "paletteOps\0"
				"\x0p|setPaletteNum,"
				"\x14ppppp|setPaletteColor,"
				"\x19p|setPaletteFromCostume,"
				"\x28pp|setPaletteFromImage,"
				"\x35|restorePalette,"
				"\x39p|copyPalette,"
				"\x3Fpp|setPaletteFromRoom,"
				"\x51ppp|copyPaletteColor,"
				"\x5C|resetPaletteNum");
		break;
	case 0x51:
		ext(output, "z|pickupObject");
		break;
	case 0x52:
		ext(output, "x" "polygonOps\0"
				"\x1Cpp|polygonErase,"
				"\x44ppppppppp|polygonStore,"
				"\x45ppppppppp|polygonStore");
		break;
	case 0x54:
		PRINT_V100HE("printDebug");
		break;
	case 0x55:
		ext(output, "p|printWizImage");
		break;
	case 0x56:
		PRINT_V100HE("printLine");
		break;
	case 0x57:
		PRINT_V100HE("printSystem");
		break;
	case 0x58:
		PRINT_V100HE("printCursor");
		break;
	case 0x59:
		ext(output, "lppi|jumpToScriptUnk");
		break;
	case 0x5A:
		ext(output, "lppi|startScriptUnk");
		break;
	case 0x5B:
		ext(output, "lp|pseudoRoom");
		break;
	case 0x5C:
		push(se_int(get_byte()));
		break;
	case 0x5D:
		push(se_int(get_dword()));
		break;
	case 0x5E:
		getScriptString();
		break;
	case 0x5F:
		push(se_int(get_word()));
		break;
	case 0x60:
		push(se_var(get_word()));
		break;
	case 0x61:
		ext(output, "zp|putActorAtObject");
		break;
	case 0x62:
		ext(output, "pppp|putActorInXY");
		break;
	case 0x64:
		ext(output, "x" "redimArray\0"
				"\x2Appv|int,"
				"\x2Bppv|dword,"
				"\x2Dppv|byte");
		break;
	case 0x65:
		ext(output, "hh|renameFile");
		break;
	case 0x66:
		ext(output, "|stopObjectCode");
		break;
	case 0x67:
		ext(output, "p|localizeArrayToRoom");
		break;
	case 0x68:
		ext(output, "x" "roomOps\0"
				"\x3Fpppp|setPalColor,"
				"\x81pp|swapObjects,"
				"\x82pp|copyPalColor,"
				"\x83p|screenEffect,"
				"\x84ppp|darkenPalette,"
				"\x85ppppp|darkenPalette,"
				"\x86p|setPalette,"
				"\x87pp|setRoomPalette,"
				"\x88pp|saveLoadRoom,"
				"\x89hp|saveOrLoad,"
				"\x8App|setScreen,"
				"\x8Bpp|roomScroll");
		break;
	case 0x69:
		// This is *almost* identical to the other print opcodes, only the 'begine' subop differs
		ext(output, "x" "printActor\0"
				"\x6pp|XY,"
				"\xC|center,"
				"\x12p|right,"
				"\x14p|color,"
				"\x15l|colors,"
				"\x2E|left,"
				"\x33|mumble,"
				"\x38|overhead,"
				"\x4Ep|getText,"
				"\x4Fs|msg,"
				"\x5Bp|begin,"
				"\x5C|end");
		break;
	case 0x6A:
		PRINT_V100HE("printEgo");
		break;
	case 0x6B:
		ext(output, "ps|talkActor");
		break;
	case 0x6C:
		ext(output, "s|talkEgo");
		break;
	case 0x6E:
		ext(output, "ppp|seekFilePos");
		break;
	case 0x6F:
		ext(output, "pl|setBoxFlags");
		break;
	case 0x71:
		ext(output, "p|setBotSet");
		break;
	case 0x72:
		ext(output, "x" "setSystemMessage\0"
				"\x50h|titleMsg,"
				"\x83h|versionMsg");
		break;
	case 0x73:
		ext(output, "wpp|shuffle");
		break;
	case 0x74:
		ext(output, "p|delay");
		break;
	case 0x75:
		ext(output, "p|delayMinutes");
		break;
	case 0x76:
		ext(output, "p|delaySeconds");
		break;
	case 0x77:
		ext(output, "x" "soundOps\0"
				"\x6p|setOffset,"
				"\x2Fhp|loadSoundFromFile,"
				"\x37|setQuickStart,"
				"\x53ppp|setSoundVar,"
				"\x5C|start,"
				"\x80|setAppend,"
				"\x81p|setChannel,"
				"\x82p|setFrequency,"
				"\x83|setLoop,"
				"\x84p|setMusicId,"
				"\x85p|setPan,"
				"\x86p|setSoundId,"
				"\x87|setSoftSound,"
				"\x88p|setVolume");
		break;
	case 0x79:
		ext(output, "x" "setSpriteInfo\0"
				"\x0pp|setRange,"
				"\x2p|setAngle,"
				"\x3p|setAutoAnimFlag,"
				"\x4p|setAnimSpeed,"
				"\x6pp|setPosition,"
				"\x7p|setSourceImage,"
				"\x10l|setClass,"
				"\x20p|setEraseType,"
				"\x26p|setGroup,"
				"\x28p|setImage,"
				"\x30p|setMaskImage,"
				"\x31pp|move,"
				"\x34|stringUnk,"
				"\x35|resetSprite,"
				"\x36pp|setGeneralProperty,"
				"\x39p|setPalette,"
				"\x3Bp|setPriority,"
				"\x3Cpp|setFlags,"
				"\x3D|resetTables,"
				"\x41p|setScale,"
				"\x46p|setAutoShadow,"
				"\x49p|setImageState,"
				"\x4App|setDist,"
				"\x4Bp|setDistX,"
				"\x4Cp|setDistY,"
				"\x52p|setUpdateType,"
				"\x53pp|setUserValue,"
				"\x58p|setField84,"
				"\x59|clearField84");
		break;
	case 0x7A:
		ext(output, "pppp|stampObject");
		break;
	case 0x7B:
		ext(output, "lppi|startObject");
		break;
	case 0x7C:
		ext(output, "lpi|startScript");
		break;
	case 0x7D:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x7E:
		ext(output, "pp|setState");
		break;
	case 0x7F:
		ext(output, "p|stopObjectScript");
		break;
	case 0x80:
		ext(output, "p|stopScript");
		break;
	case 0x81:
		ext(output, "|stopSentence");
		break;
	case 0x82:
		ext(output, "p|stopSound");
		break;
	case 0x83:
		ext(output, "|stopTalking");
		break;
	case 0x84:
		writeVar(output, get_word(), pop());
		break;
	case 0x85:
		se_a = pop();
		writeArray(output, get_word(), NULL, pop(), se_a);
		break;
	case 0x86:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_word(), pop(), se_b, se_a);
		break;
	case 0x87:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, 7 + isEqual, se_a));
		break;
	case 0x88:
		ext(output, "x" "systemOps\0"
				 "\x3D|restart,"
				 "\x80|clearDrawQueue,"
				 "\x84|confirmShutDown,"
				 "\x85|shutDown,"
				 "\x86|startGame,"
				 "\x87|startExec,"
				 "\x88|copyVirtBuf");
		break;
	case 0x89:
		ext(output, "x" "windowOps\0"
				"\x0p|case0,"
				"\x6pp|case6,"
				"\x11p|case17,"
				"\x27p|case39,"
				"\x28p|case40,"
				"\x31pp|case49,"
				"\x35|case53,"
				"\x42p|case66,"
				"\x43p|case67,"
				"\x47p|case71,"
				"\x50h|case80,"
				"\x54p|case84,"
				"\x5C|case92");
		break;
	case 0x8A:
		ext(output, "pi|setTimer");
		break;
	case 0x8B:
		ext(output, "x" "cursorCommand\0"
				"\xEp|initCharset,"
				"\xFl|charsetColors,"
				"\x80z|setCursorImg,"
				"\x81z|setCursorImg,"
				"\x82zp|setCursorImgWithPal,"
				"\x86|cursorOn,"
				"\x87|cursorOff,"
				"\x88|softCursorOn,"
				"\x89|softCursorOff,"
				"\x8B|userPutOn,"
				"\x8C|userPutOff,"
				"\x8D|softUserputOn,"
				"\x8E|softUserputOff");

		break;
	case 0x8C:
		ext(output, "x" "videoOps\0"
				"\x0p|setUnk2,"
				"\x13|setStatus,"
				"\x28p|setWizResNumX,"
				"\x2Fh|setFilename,"
				"\x43p|setFlags,"
				"\x5C|playOrStopVideo,");
		break;
	case 0x8D:
		ext(output, "x" "wait\0"
				"\x80pj|waitForActor,"
				"\x81|waitForCamera,"
				"\x82|waitForMessage,"
				"\x83|waitForSentence");
		break;
	case 0x8E:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0x8F:
		ext(output, "ppp|walkActorTo");
		break;
	case 0x90:
		ext(output, "x" "writeFile\0"
				"\x5ppi|writeArrayToFile,"
				"\x2App|writeWord,"
				"\x2Bpp|writeDWord,"
				"\x2Dpp|writeByte");
		break;
	case 0x91:
		ext(output, "x" "writeINI\0"
				"\x2Bph|number,"
				"\x4Dhh|string");
		break;
	case 0x92:
		ext(output, "x" "writeConfigFile\0"
				"\x2Bphhh|number,"
				"\x4Dhhhh|string");
		break;
	case 0x93:
		ext(output, "rp|abs");
		break;
	case 0x94:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0x95:
		ext(output, "rp|getActorCostume");
		break;
	case 0x96:
		ext(output, "rp|getActorElevation");
		break;
	case 0x97:
		ext(output, "rp|getObjectDir");
		break;
	case 0x98:
		ext(output, "rp|getActorMoving");
		break;
	case 0x99:
		ext(output, "rppp|getActorData");
		break;
	case 0x9A:
		ext(output, "rp|getActorRoom");
		break;
	case 0x9B:
		ext(output, "rp|getActorScaleX");
		break;
	case 0x9C:
		ext(output, "rpp|getAnimateVariable");
	case 0x9D:
		ext(output, "rp|getActorWidth");
		break;
	case 0x9E:
		ext(output, "rp|objectX");
		break;
	case 0x9F:
		ext(output, "rp|objectY");
		break;
	case 0xA0:
		ext(output, "rpp|atan2");
		break;
	case 0xA1:
		ext(output, "rpppp|getSegmentAngle");
		break;
	case 0xA2:
		ext(output, "rp|getActorAnimProgress");
		break;
	case 0xA3:
		ext(output, "rx" "getDistanceBetweenPoints\0"
				"\x17pppp|case17,"
				"\x18pppppp|case18");
		break;
	case 0xA4:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0xA6:
		ext(output, "rppp|cond");
		break;
	case 0xA7:
		ext(output, "rp|cos");
		break;
	case 0xA8:
		// Backyard Baseball 2003 / Basketball / Football 2002
		if (g_options.heVersion == 101) {
			ext(output, "rh|debugInput");
		} else {
		// All other games
			switch (get_byte()) {
			case 0:
				ext(output, "h|debugInput: case 0");
				break;
			case 26:
				ext(output, "p|debugInput: case 26");
				break;
			case 27:
				ext(output, "h|debugInput: case 27");
				break;
			case 80:
				ext(output, "h|debugInput: case 80");
				break;
			case 92:
				ext(output, "r|debugInput: case 92");
				break;
			}
		}
		break;
	case 0xA9:
		ext(output, "rh|getFileSize");
		break;
	case 0xAA:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xAB:
		ext(output, "rp|findAllObjects");
		break;
	case 0xAC:
		ext(output, "rlp|findAllObjectsWithClassOf");
		break;

	case 0xAE:
		ext(output, "rpp|findInventory");
		break;
	case 0xAF:
		ext(output, "rpp|findObject");
		break;
	case 0xB0:
		ext(output, "rlpp|findObjectWithClassOf");
		break;
	case 0xB1:
			ext(output, "rpp|polygonHit");
		break;
	case 0xB2:
		ext(output, "rwwpppppppp|getLinesIntersectionPoint");
		break;
	case 0xB3:
		ext(output, "rx" "fontUnk\0"
				"\x0|case0,"
				"\x3Cpp|case60");
		break;
	case 0xB4:
		ext(output, "r|getNumFreeArrays");
		break;
	case 0xB5:
		ext(output, "rx" "getArrayDimSize\0"
				"\x1v|dim1size,"
				"\x2v|dim2size,"
				"\x3v|dim1size,"
				"\x4v|dim1start,"
				"\x5v|dim1end,"
				"\x6v|dim2start,"
				"\x7v|dim2end");
		break;
	case 0xB6:
		ext(output, "rx" "isResourceLoaded\0"
				"\x19p|costume,"
				"\x28p|image,"
				"\x3Ep|room,"
				"\x42p|script,"
				"\x48p|sound");
		break;
	case 0xB7:
		ext(output, "rx" "getResourceSize\0"
				"\x19p|costume,"
				"\x28p|image,"
				"\x3Ep|roomImage,"
				"\x42p|script,"
				"\x48p|sound");
		break;
	case 0xB8:
		ext(output, "rx" "getSpriteGroupInfo\0"
				"\x5p|getSpriteArray,"
				"\x28p|getDstResNum,"
				"\x36pp|dummy,"
				"\x3Bp|getPriority,"
				"\x3Cpp|getXYScale?,"
				"\x55p|getPositionX,"
				"\x56p|getPositionY");
		break;
	case 0xB9:
		ext(output, "rx" "getHeap\0"
				"\x82|freeSpace,"
				"\x83|largestBlockSize");
		break;
	case 0xBA:
		ext(output, "rx" "getWizData\0"
				"\x14pppp|pixelColor,"
				"\x1Ap|imageCount,"
				"\x21pppp|isPixelNonTransparentnumber,"
				"\x27pp|height,"
				"\x36ppp|block,"
				"\x54pp|width,"
				"\x55pp|imageSpotX,"
				"\x56pp|imageSpotY,"
				"\x83php|case131,"
				"\x84pppppp|histogram");
		break;
	case 0xBB:
		ext(output, "rpp|isActorInBox");
		break;
	case 0xBC:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xBD:
		ext(output, "rp|getInventoryCount");
		break;
	case 0xBE:
		ext(output, "ry" "kernelGetFunctions\0"
				"\x1|virtScreenSave"
				);
		break;
	case 0xBF:
		ext(output, "rpp|max");
		break;
	case 0xC0:
		ext(output, "rpp|min");
		break;
	case 0xC1:
		ext(output, "rp|getObjectX");
		break;
	case 0xC2:
		ext(output, "rp|getObjectY");
		break;
	case 0xC3:
		ext(output, "rp|isRoomScriptRunning");
		break;
	case 0xC4:
		ext(output, "rx" "getObjectData\0"
				"\x20|getWidth,"
				"\x21|getHeight,"
				"\x24|getImageCount,"
				"\x26|getX,"
				"\x27|getY,"
				"\x34|getState,"
				"\x39p|setId,"
				"\x8Bp|dummy");
		break;
	case 0xC5:
		ext(output, "rph|openFile");
		break;
	case 0xC6:
		ext(output, "rllp|getPolygonOverlap");
		break;
	case 0xC7:
		ext(output, "rp|getOwner");
		break;
	case 0xC8:
		ext(output, "rx" "getPaletteData\0"
				"\xDpp|get16BitColorComponent,"
				"\x14pp|getColor,"
				"\x21pppppp|getSimilarColor,"
				"\x35ppp|get16BitColor,"
				"\x49ppp|getColorCompontent");
		break;
	case 0xC9:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCA:
		ext(output, "rplp|pickOneOfDefault");
		break;
	case 0xCB:
		ext(output, "rlw|pickVarRandom");
		break;
	case 0xCC:
		ext(output, "rx" "getPixel\0"
				"\x8pp|foreground,"
				"\x9pp|background");
		break;
	case 0xCD:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xCE:
		ext(output, "rppp|getDistObjPt");
		break;
	case 0xCF:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xD0:
		ext(output, "rp|getRandomNumber");
		break;
	case 0xD1:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0xD3:
		ext(output, "rx" "readFile\0"
				"\x5ppi|readArrayFromFile,"
				"\x2Ap|readWord,"
				"\x2Bp|readDWord,"
				"\x2Dp|readByte");
		break;
	case 0xD4:
		ext(output, "rx" "readINI\0"
				"\x2Bh|number,"
				"\x4Dh|string");
		break;
	case 0xD5:
		ext(output, "rx" "readConfigFile\0"
				"\x2Bhhh|number,"
				"\x4Dhhh|string");
		break;
	case 0xD6:
		ext(output, "rp|isScriptRunning");
		break;
	case 0xD7:
		ext(output, "rp|sin");
		break;
	case 0xD8:
		ext(output, "rp|getSoundPosition");
		break;
	case 0xD9:
		ext(output, "rp|isSoundRunning");
		break;
	case 0xDA:
		ext(output, "rpp|getSoundVar");
		break;
	case 0xDB:
			ext(output, "rx" "getSpriteInfo\0"
					"\x3p|getFlagAutoAnim,"
					"\x4p|getAnimSpeed,"
					"\x7p|getSourceImage,"
					"\x10lp|getClass,"
					"\x1Ap|getImageStateCount,"
					"\x1Ep|getDisplayX,"
					"\x1Fp|getDisplayY,"
					"\x20p|getEraseType,"
					"\x21lpppp|findSprite,"
					"\x26p|getGroup,"
					"\x27p|getImageY,"
					"\x28p|getImage,"
					"\x30p|getMaskImage,"
					"\x36pp|getGeneralProperty,"
					"\x39p|getPalette,"
					"\x3Bp|getPriority,"
					"\x3Cpp|getFlags,"
					"\x41p|getScale,"
					"\x46p|getShadow,"
					"\x49p|getImageState,"
					"\x4Bp|getDistX,"
					"\x4Cp|getDistY,"
					"\x52p|getUpdateType,"
					"\x53pp|getUserValue,"
					"\x54p|getImageX,"
					"\x55p|getPosX,"
					"\x56p|getPosY");
		break;
	case 0xDC:
		ext(output, "rp|sqrt");
		break;
	case 0xDD:
		// TODO: this loads another script which does something like
		// stack altering and then finishes (usually with opcode 0xBD).
		// When stack is changed, further disassembly is wrong.
		// This is widely used in HE games.
		// As there are cases when called script does not alter the
		// stack, it's not correct to use "rlpp|..." here
		ext(output, "lpp|startObjectQuick");
		break;
	case 0xDE:
		ext(output, "lp|startScriptQuick2");
		break;
	case 0xDF:
		ext(output, "rp|getState");
		break;
	case 0xE0:
		ext(output, "rpp|compareString");
		break;
	case 0xE1:
		ext(output, "rp|copyString");
		break;
	case 0xE2:
		ext(output, "rppp|appendString");
		break;
	case 0xE3:
		ext(output, "rpp|concatString");
		break;
	case 0xE4:
		ext(output, "rp|getStringLen");
		break;
	case 0xE5:
		ext(output, "rppp|getStringLenForWidth");
		break;
	case 0xE6:
		ext(output, "rp|stringToInt");
		break;
	case 0xE7:
		ext(output, "rpppp|getCharIndexInString");
		break;
	case 0xE8:
		ext(output, "rppp|getStringWidth");
		break;
	case 0xE9:
		ext(output, "rp|readFilePos");
		break;
	case 0xEA:
		ext(output, "ri|getTimer");
		break;
	case 0xEB:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xEC:
		ext(output, "rx" "getVideoData\0"
				"\x1Ap|frameCount,"
				"\x27p|height,"
				"\x28p|imageNum,"
				"\x36pp|statistics,"
				"\x49p|curFrame,"
				"\x54p|width");
		break;
	default:
		invalidop(NULL, code);
		break;
	}
}

#define PRINT_V7HE(name)           \
	do {                          \
		ext(output, "x" name "\0"         \
				"\x41pp|XY,"      \
				"\x42p|color,"    \
				"\x43p|right,"    \
				"\x45|center,"    \
				"\x47|left,"      \
				"\x48|overhead,"  \
				"\x4A|mumble,"    \
				"\x4Bs|msg,"      \
				"\xF9l|colors,"	  \
				"\xC2lps|debug,"  \
				"\xE1p|getText,"  \
				"\xFE|begin,"     \
				"\xFF|end"        \
				);                \
	} while(0)

void next_line_HE_V72(char *output) {
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x0:
		push(se_int(get_byte()));
		break;
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_int(get_dword()));
		break;
	case 0x3:
		push(se_var(get_word()));
		break;
	case 0x4:
		getScriptString();
		break;
	case 0x7:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0xA: // HE90+
		dup_n();
		break;
	case 0xB:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0xC:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0xD:
		push(se_oper(pop(), isZero));
		break;
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
	case 0x17:
	case 0x18:
	case 0x19:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0xE) + isEqual, se_a));
		break;
	case 0x1A:
	case 0xA7:
		kill(output, pop());
		break;
	case 0x1B:
		ext(output, "rlp|isAnyOf2");
		break;
	case 0x1C: // HE90+
		ext(output, "x" "wizImageOps\0"
				"\x20p|setResDefImageWidth,"
				"\x21p|setResDefImageHeight,"
				"\x30|processMode1,"
				"\x31h|readFile,"
				"\x32hp|writeFile,"
				"\x33ppppp|setCaptureRect,"
				"\x34p|setImageState,"
				"\x35p|setAngle,"
				"\x36p|setFlags,"
				"\x38ppppp|drawWizImage,"
				"\x39p|setImage,"
				"\x3Ep|setSourceImage,"
				"\x41pp|setPosition,"
				"\x42pp|remapPalette,"
				"\x43pppp|setClipRect,"
				"\x56p|setPalette,"
				"\x5Cp|setScale,"
				"\x62p|setShadow,"
				"\x83ppp|capturePolygon,"
				"\x85ppppp|fillWizRect,"
				"\x86ppppp|fillWizLine,"
				"\x87ppp|fillWizPixel,"
				"\x88ppp|fillWizFlood,"
				"\x89p|setDstResNum,"
				"\x8Bpp|setThickLine,"
				"\x8D|startFont,"
				"\x8Epppph|createFont,"
				"\x8Fpph|renderFontString,"
				"\x9App|setPosition,"
				"\xBDpppppppp|ellipse,"
				"\xC4|endFont,"
				"\xD9|createWizImage,"
				"\xF6p|setupPolygon,"
				"\xF9pp|remapPalette,"
				"\xFF|processWizImage");
		break;
	case 0x1D: // HE90+
		ext(output, "rpp|min");
		break;
	case 0x1E: // HE90+
		ext(output, "rpp|max");
		break;
	case 0x1F: // HE90+
		ext(output, "rp|sin");
		break;
	case 0x20: // HE90+
		ext(output, "rp|cos");
		break;
	case 0x21: // HE90+
		ext(output, "rp|sqrt");
		break;
	case 0x22: // HE90+
		ext(output, "rpp|atan2");
		break;
	case 0x23: // HE90+
		ext(output, "rpppp|getSegmentAngle");
		break;
	case 0x24: // HE90+
		ext(output, "rx" "getDistanceBetweenPoints\0"
				"\x1Cpppp|case28,"
				"\x1Dpppppp|case29");
		break;
	case 0x25: // HE90+
		if (g_options.heVersion >= 99) {
			ext(output, "rx" "getSpriteInfo\0"
					"\x1Ep|getPosX,"
					"\x1Fp|getPosY,"
					"\x20p|getImageX,"
					"\x21p|getImageY,"
					"\x22p|getDistX,"
					"\x23p|getDistY,"
					"\x24p|getImageStateCount,"
					"\x25p|getGroup,"
					"\x26p|getDisplayX,"
					"\x27p|getDisplayY,"
					"\x2App|getFlags,"
					"\x2Bp|getPriority,"
					"\x2Dlpppp|findSprite,"
					"\x34p|getImageState,"
					"\x3Ep|getSourceImage,"
					"\x3Fp|getImage,"
					"\x44p|getEraseType,"
					"\x52p|getFlagAutoAnim,"
					"\x56p|getPalette,"
					"\x5Cp|getScale,"
					"\x61p|getAnimSpeed,"
					"\x62p|getShadow,"
					"\x7Cp|getUpdateType,"
					"\x7Dlp|getClass,"
					"\x8Bpp|getGeneralProperty,"
					"\x8Cp|getMaskImage,"
					"\xC6pp|getUserValue");
		} else if (g_options.heVersion >= 98) {
			ext(output, "rx" "getSpriteInfo\0"
					"\x1Ep|getPosX,"
					"\x1Fp|getPosY,"
					"\x20p|getImageX,"
					"\x21p|getImageY,"
					"\x22p|getDistX,"
					"\x23p|getDistY,"
					"\x24p|getImageStateCount,"
					"\x25p|getGroup,"
					"\x26p|getDisplayX,"
					"\x27p|getDisplayY,"
					"\x2App|getFlags,"
					"\x2Bp|getPriority,"
					"\x2Dpppp|findSprite,"
					"\x34p|getImageState,"
					"\x3Ep|getSourceImage,"
					"\x3Fp|getImage,"
					"\x44p|getEraseType,"
					"\x52p|getFlagAutoAnim,"
					"\x56p|getPalette,"
					"\x5Cp|getScale,"
					"\x61p|getAnimSpeed,"
					"\x62p|getShadow,"
					"\x7Cp|getUpdateType,"
					"\x7Dlp|getClass,"
					"\x8Bpp|getGeneralProperty,"
					"\x8Cp|getMaskImage,"
					"\xC6pp|getUserValue");
		} else {
			ext(output, "rx" "getSpriteInfo\0"
					"\x1Ep|getPosX,"
					"\x1Fp|getPosY,"
					"\x20p|getImageX,"
					"\x21p|getImageY,"
					"\x22p|getDistX,"
					"\x23p|getDistY,"
					"\x24p|getImageStateCount,"
					"\x25p|getGroup,"
					"\x26p|getDisplayX,"
					"\x27p|getDisplayY,"
					"\x2App|getFlags,"
					"\x2Bp|getPriority,"
					"\x2Dppp|findSprite,"
					"\x34p|getImageState,"
					"\x3Ep|getSourceImage,"
					"\x3Fp|getImage,"
					"\x44p|getEraseType,"
					"\x52p|getFlagAutoAnim,"
					"\x56p|getPalette,"
					"\x5Cp|getScale,"
					"\x61p|getAnimSpeed,"
					"\x62p|getShadow,"
					"\x7Cp|getUpdateType,"
					"\x7Dlp|getClass,"
					"\x8Bpp|getGeneralProperty,"
					"\x8Cp|getMaskImage,"
					"\xC6pp|getUserValue");
		}
		break;
	case 0x26: // HE90+
		if (g_options.heVersion >= 99) {
			ext(output, "x" "setSpriteInfo\0"
					"\x22p|setDistX,"
					"\x23p|setDistY,"
					"\x25p|setGroup,"
					"\x2App|setFlags,"
					"\x2Bp|setPriority,"
					"\x2Cpp|move,"
					"\x34p|setImageState,"
					"\x35p|setAngle,"
					"\x39pp|setRange,"
					"\x3Ep|setSourceImage,"
					"\x3Fp|setImage,"
					"\x41pp|setPosition,"
					"\x44p|setEraseType,"
					"\x4Dpp|setDist,"
					"\x52p|setAutoAnimFlag,"
					"\x56p|setPalette,"
					"\x5Cp|setScale,"
					"\x61p|setAnimSpeed,"
					"\x62p|setAutoShadow,"
					"\x7Cp|setUpdateType,"
					"\x7Dl|setClass,"
					"\x8Bpp|setGeneralProperty,"
					"\x8Cp|setMaskImage,"
					"\x9E|resetTables,"
					"\xC6pp|setUserValue,"
					"\xD9|resetSprite");
		} else {
			ext(output, "x" "setSpriteInfo\0"
					"\x22p|setDistX,"
					"\x23p|setDistY,"
					"\x25p|setGroup,"
					"\x2App|setFlags,"
					"\x2Bp|setPriority,"
					"\x2Cpp|move,"
					"\x34p|setImageState,"
					"\x35p|setAngle,"
					"\x39p|setRange,"
					"\x3Ep|setSourceImage,"
					"\x3Fp|setImage,"
					"\x41pp|setPosition,"
					"\x44p|setEraseType,"
					"\x4Dpp|setDist,"
					"\x52p|setAutoAnimFlag,"
					"\x56p|setPalette,"
					"\x5Cp|setScale,"
					"\x61p|setAnimSpeed,"
					"\x62p|setAutoShadow,"
					"\x7Cp|setUpdateType,"
					"\x7Dl|setClass,"
					"\x8Bpp|setGeneralProperty,"
					"\x8Cp|setMaskImage,"
					"\x9E|resetTables,"
					"\xC6pp|setUserValue,"
					"\xD9|resetSprite");
		}
		break;
	case 0x27: // HE90+
		ext(output, "rx" "getSpriteGroupInfo\0"
				"\x8p|getSpriteArray,"
				"\x1Ep|getPositionX,"
				"\x1Fp|getPositionY,"
				"\x2App|getXYScale?,"
				"\x2Bp|getPriority,"
				"\x3Fp|getDstResNum,"
				"\x8Bpp|dummy");
		break;
	case 0x28: // HE90+
		ext(output, "x" "setSpriteGroupInfo\0"
				"\x25pp|misc,"
				"\x2App|setXYScale?,"
				"\x2Bp|setPriority,"
				"\x2Cpp|move,"
				"\x39p|setId,"
				"\x3Fp|setImage,"
				"\x41pp|setPosition,"
				"\x43pppp|setBounds,"
				"\x5D|resetBounds,"
				"\xD9|reset");
		break;
	case 0x29: // HE90+
		ext(output, "rx" "getWizData\0"
				"\x1Epp|imageSpotX,"
				"\x1Fpp|imageSpotY,"
				"\x20pp|width,"
				"\x21pp|height,"
				"\x24p|imageCount,"
				"\x2Dpppp|isPixelNonTransparentnumber,"
				"\x42pppp|pixelColor,"
				"\x82pppppp|histogram,"
				"\x8Dphp|case141");
		break;
	case 0x2A: // HE90+
		ext(output, "rppp|getActorData");
		break;
	case 0x2B: // HE90+
		ext(output, "lppi|startScriptUnk");
		break;
	case 0x2C: // HE90+
		ext(output, "lppi|jumpToScriptUnk");
		break;
	case 0x2D: // HE90+
		ext(output, "x" "videoOps\0"
				"\x31h|setFilename,"
				"\x36p|setFlags,"
				"\x39p|setUnk2,"
				"\x3Fp|setWizResNumX,"
				"\xA5|setStatus,"
				"\xFF|playOrStopVideo,");
		break;
	case 0x2E: // HE95+
		ext(output, "rx" "getVideoData\0"
				"\x20p|width,"
				"\x21p|height,"
				"\x24p|frameCount,"
				"\x34p|curFrame,"
				"\x3Fp|imageNum,"
				"\x8Bpp|statistics");
		break;
	case 0x2F: // HE90+
		ext(output, "x" "floodFill\0"
				"\x36p|dummy,"
				"\x39|reset,"
				"\x41pp|setXY,"
				"\x42p|setFlags,"
				"\x43pppp|setBoxRect,"
				"\xFF|floodFill");
		break;
	case 0x30: // HE90+
		ext(output, "rpp|mod");
		break;
	case 0x31: // HE90+
		ext(output, "rpp|shl");
		break;
	case 0x32: // HE90+
		ext(output, "rpp|shr");
		break;
	case 0x33: // HE90+
		ext(output, "rpp|xor");
		break;
	case 0x34:
		ext(output, "rlp|findAllObjectsWithClassOf");
		break;
	case 0x35: // HE90+
		ext(output, "rllp|getPolygonOverlap");
		break;
	case 0x36: // HE90+
		ext(output, "rppp|cond");
		break;
	case 0x37: // HE90+
		ext(output, "x" "dim2dim2Array\0"
				"\x2pppppv|bit,"
				"\x3pppppv|nibble,"
				"\x4pppppv|byte,"
				"\x5pppppv|int,"
				"\x6pppppv|dword,"
				"\x7pppppv|string");
		break;
	case 0x38: // HE90+
		ext(output, "x" "redim2dimArray\0"
				"\x4ppppv|byte,"
				"\x5ppppv|int,"
				"\x6ppppv|dword");
		break;
	case 0x39: // HE90+
		ext(output, "rwwpppppppp|getLinesIntersectionPoint");
		break;
	case 0x3A: // HE90+
		ext(output, "x" "sortArray\0"
				"\x81pppppv|sort,");
		break;
	case 0x43:
		writeVar(output, get_word(), pop());
		break;
	case 0x44: // HE90+
		ext(output, "rx" "getObjectData\0"
				"\x20|getWidth,"
				"\x21|getHeight,"
				"\x24|getImageCount,"
				"\x26|getX,"
				"\x27|getY,"
				"\x34|getState,"
				"\x39p|setId,"
				"\x8Bp|dummy");
		break;
	case 0x45: // HE80+
		ext(output, "x" "createSound\0"
				"\x1Bp|create,"
				"\xD9|reset,"
				"\xE8p|setId,"
				"\xFF|dummy");
		break;
	case 0x46: // HE80+
		ext(output, "rh|getFileSize");
		break;
	case 0x47:
		se_a = pop();
		writeArray(output, get_word(), NULL, pop(), se_a);
		break;
	case 0x48: // HE80+
		ext(output, "rp|stringToInt");
		break;
	case 0x49: // HE80+
		ext(output, "rpp|getSoundVar");
		break;
	case 0x4A: // HE80+
		ext(output, "p|localizeArrayToRoom");
		break;
	case 0x4B:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_word(), pop(), se_b, se_a);
		break;
	case 0x4D: // HE80+
		ext(output, "rx" "readConfigFile\0"
				"\x6hhh|number,"
				"\x7hhh|string");
		break;
	case 0x4E: // HE80+
		ext(output, "x" "writeConfigFile\0"
				"\x6phhh|number,"
				"\x7hhhh|string");
		break;
	case 0x4F:
		addVar(output, get_word(), 1);
		break;
	case 0x50:
		ext(output, "|resetCutScene");
		break;
	case 0x51:
		ext(output, "rx" "getHeap\0"
				"\xb|freeSpace,"
				"\xc|largestBlockSize");
		break;
	case 0x52:
		ext(output, "rlpp|findObjectWithClassOf");
		break;
	case 0x53:
		addArray(output, get_word(), pop(), 1);
		break;
	case 0x54:
		ext(output, "rp|objectX");
		break;
	case 0x55:
		ext(output, "rp|objectY");
		break;
	case 0x56:
		ext(output, "ppppp|captureWizImage");
		break;
	case 0x57:
		addVar(output, get_word(), -1);
		break;
	case 0x58:
		ext(output, "ri|getTimer");
		break;
	case 0x59:
		ext(output, "pi|setTimer");
		break;
	case 0x5A:
		ext(output, "rp|getSoundPosition");
		break;
	case 0x5B:
		addArray(output, get_word(), pop(), -1);
		break;
	case 0x5C:
		jumpif(output, pop(), true);
		break;
	case 0x5D:
		jumpif(output, pop(), false);
		break;
	case 0x5E:
		ext(output, "lpi|startScript");
		break;
	case 0x5F:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x60:
		ext(output, "lppi|startObject");
		break;
	case 0x61:
		ext(output, "x" "drawObject\0"
				"\x3Epppp|setup,"
				"\x3Fpp|setState,"
				"\x41ppp|setPosition,");
		break;
	case 0x62:
		ext(output, "p|printWizImage");
		break;
	case 0x63:
		ext(output, "rx" "getArrayDimSize\0"
				"\x1v|dim1size,"
				"\x2v|dim2size,"
				"\x3v|dim1size,"
				"\x4v|dim1start,"
				"\x5v|dim1end,"
				"\x6v|dim2start,"
				"\x7v|dim2end,");
		break;
	case 0x64:
		ext(output, "r|getNumFreeArrays");
		break;
	case 0x65:
		ext(output, "|stopObjectCodeA");
		break;
	case 0x66:
		ext(output, "|stopObjectCodeB");
		break;
	case 0x67:
		ext(output, "|endCutscene");
		break;
	case 0x68:
		ext(output, "l|beginCutscene");
		break;
	case 0x69:
		if (g_options.heVersion >= 80) {
			ext(output, "x" "windowOps\0"
					"\x2Cpp|case44,"
					"\x39p|case57,"
					"\x3Ap|case58,"
					"\x3Bp|case59,"
					"\x3Fp|case63,"
					"\x41pp|case65,"
					"\xD9|case217,"
					"\xF3h|case243,"
					"\xFF|case255");
		} else {
			ext(output, "|stopMusic");
		}
		break;
	case 0x6A:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x6B:
		ext(output, "x" "cursorCommand\0"
				"\x13z|setCursorImg,"
				"\x14z|setCursorImg,"
				"\x3Czp|setCursorImgWithPal,"
				"\x90|cursorOn,"
				"\x91|cursorOff,"
				"\x92|userPutOn,"
				"\x93|userPutOff,"
				"\x94|softCursorOn,"
				"\x95|softCursorOff,"
				"\x96|softUserputOn,"
				"\x97|softUserputOff,"
				"\x99z|setCursorImg,"
				"\x9App|setCursorHotspot,"
				"\x9Cp|initCharset,"
				"\x9Dl|charsetColors,"
				"\xD6p|makeCursorColorTransparent");
		break;
	case 0x6C:
		ext(output, "|breakHere");
		break;
	case 0x6D:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0x6E:
		ext(output, "lp|setClass");
		break;
	case 0x6F:
		ext(output, "rp|getState");
		break;
	case 0x70:
		ext(output, "pp|setState");
		break;
	case 0x71:
		ext(output, "pp|setOwner");
		break;
	case 0x72:
		ext(output, "rp|getOwner");
		break;
	case 0x73:
		jump(output);
		break;
	case 0x74:
		ext(output, "x" "soundOps\0"
				"\x9|setSoftSound,"
				"\x17ppp|setSoundVar,"
				"\x19pp|setVolume,"
				"\x38|setQuickStart,"
				"\xA4|setAppend,"
				"\xDE|dummy,"
				"\xE0p|setFrequency,"
				"\xE6p|setChannel,"
				"\xE7p|setOffset,"
				"\xE8p|setSoundId,"
				"\xF5|setLoop,"
				"\xFF|start");
		break;
	case 0x75:
		ext(output, "p|stopSound");
		break;
	case 0x76:
		ext(output, "p|startMusic");
		break;
	case 0x77:
		ext(output, "p|stopObjectScript");
		break;
	case 0x78:
		ext(output, "p|panCameraTo");
		break;
	case 0x79:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x7A:
		ext(output, "p|setCameraAt");
		break;
	case 0x7B:
		ext(output, "p|loadRoom");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0x7E:
		ext(output, "ppp|walkActorTo");
		break;
	case 0x7F:
		ext(output, "pppp|putActorInXY");
		break;
	case 0x80:
		ext(output, "zp|putActorAtObject");
		break;
	case 0x81:
		ext(output, "pp|faceActor");
		break;
	case 0x82:
		ext(output, "pp|animateActor");
		break;
	case 0x83:
		ext(output, "pppp|doSentence");
		break;
	case 0x84:
		ext(output, "z|pickupObject");
		break;
	case 0x85:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x87:
		ext(output, "rp|getRandomNumber");
		break;
	case 0x88:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0x8A:
		ext(output, "rp|getActorMoving");
		break;
	case 0x8B:
		ext(output, "rp|isScriptRunning");
		break;
	case 0x8C:
		ext(output, "rp|getActorRoom");
		break;
	case 0x8D:
		ext(output, "rp|getObjectX");
		break;
	case 0x8E:
		ext(output, "rp|getObjectY");
		break;
	case 0x8F:
		ext(output, "rp|getObjectDir");
		break;
	case 0x90:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0x91:
		ext(output, "rp|getActorCostume");
		break;
	case 0x92:
		ext(output, "rpp|findInventory");
		break;
	case 0x93:
		ext(output, "rp|getInventoryCount");
		break;
	case 0x94:
		if (g_options.heVersion >= 90) {
			ext(output, "rx" "getPaletteData\0"
					"\x2Dpppppp|getSimilarColor,"
					"\x34ppp|getColorCompontent,"
					"\x42pp|getColor,"
					"\x84pp|get16BitColorComponent,"
					"\xD9ppp|get16BitColor");
		} else {
			ext(output, "rpp|getVerbFromXY");
		}
		break;
	case 0x95:
		ext(output, "|beginOverride");
		break;
	case 0x96:
		ext(output, "|endOverride");
		break;
	case 0x97:
		ext(output, "ps|setObjectName");
		break;
	case 0x98:
		ext(output, "rp|isSoundRunning");
		break;
	case 0x99:
		ext(output, "pl|setBoxFlags");
		break;
	case 0x9B:
		ext(output, "x" "resourceRoutines\0"
				"\x64p|loadScript,"
				"\x65p|loadSound,"
				"\x66p|loadCostume,"
				"\x67p|loadRoom,"
				"\x68p|nukeScript,"
				"\x69p|nukeSound,"
				"\x6Ap|nukeCostume,"
				"\x6Bp|nukeRoom,"
				"\x6Cp|lockScript,"
				"\x6Dp|lockSound,"
				"\x6Ep|lockCostume,"
				"\x6Fp|lockRoom,"
				"\x70p|unlockScript,"
				"\x71p|unlockSound,"
				"\x72p|unlockCostume,"
				"\x73p|unlockRoom,"
				"\x74|clearHeap,"
				"\x75p|loadCharset,"
				"\x76p|nukeCharset,"
				"\x77z|loadFlObject,"
				"\x78p|queueloadScript,"
				"\x79p|queueloadSound,"
				"\x7Ap|queueloadCostume,"
				"\x7Bp|queueloadRoomImage,"
				"\x9fp|unlockImage,"
				"\xc0p|nukeImage,"
				"\xc9p|loadImage,"
				"\xcap|lockImage,"
				"\xcbp|queueloadImage,"
				"\xe9p|lockFlObject,"
				"\xebp|unlockFlObject,"
				"\xef|dummy");
		break;
	case 0x9C:
		ext(output, "x" "roomOps\0"
				"\xACpp|roomScroll,"
				"\xAEpp|setScreen,"
				"\xAFpppp|setPalColor,"
				"\xB0|shakeOn,"
				"\xB1|shakeOff,"
				"\xB3ppp|darkenPalette,"
				"\xB4pp|saveLoadRoom,"
				"\xB5p|screenEffect,"
				"\xB6ppppp|darkenPalette,"
				"\xB7ppppp|setupShadowPalette,"
				"\xBApppp|palManipulate,"
				"\xBBpp|colorCycleDelay,"
				"\xD5p|setPalette,"
				"\xDCpp|copyPalColor,"
				"\xDDhp|saveOrLoad,"
				"\xEApp|swapObjects,"
				"\xECpp|setRoomPalette");
		break;
	case 0x9D:
		ext(output, "x" "actorOps\0"
				"\x15l|setUserConditions,"
				"\x18p|setTalkCondition,"
				"\x2Bp|layer,"
				"\xC5p|setCurActor,"
				"\x40pppp|setClipRect,"
				"\x41pp|putActor,"
				"\x43pppp|setActorClipRect,"
				"\x44p|setHEFlag,"
				"\x4Cp|setCostume,"
				"\x4Dpp|setWalkSpeed,"
				"\x4El|setSound,"
				"\x4Fp|setWalkFrame,"
				"\x50pp|setTalkFrame,"
				"\x51p|setStandFrame,"
				"\x52ppp|actorSet:82:??,"
				"\x53|init,"
				"\x54p|setElevation,"
				"\x55|setDefAnim,"
				"\x56pp|setPalette,"
				"\x57p|setTalkColor,"
				"\x58h|setName,"
				"\x59p|setInitFrame,"
				"\x5Bp|setWidth,"
				"\x5Cp|setScale,"
				"\x5D|setNeverZClip,"
				"\x5Ep|setAlwayZClip,"
				"\x5F|setIgnoreBoxes,"
				"\x60|setFollowBoxes,"
				"\x61p|setAnimSpeed,"
				"\x62p|setShadowMode,"
				"\x63pp|setTalkPos,"
				"\x8Bpp|case139,"
				"\x9Cp|charset,"
				"\xAFp|setPaletteNum,"
				"\xC6pp|setAnimVar,"
				"\xD7|setIgnoreTurnsOn,"
				"\xD8|setIgnoreTurnsOff,"
				"\xD9|initLittle,"
				"\xDA|drawToBackBuf,"
				"\xE1hp|setTalkieSlot");
		break;
	case 0x9E:
		if (g_options.heVersion >= 90) {
			ext(output, "x" "paletteOps\0"
					"\x39p|setPaletteNum,"
					"\x3Fpp|setPaletteFromImage,"
					"\x42ppppp|setPaletteColor,"
					"\x46ppp|copyPaletteColor,"
					"\x4Cp|setPaletteFromCostume,"
					"\x56p|copyPalette,"
					"\xAFpp|setPaletteFromRoom,"
					"\xD9|restorePalette,"
					"\xFF|resetPaletteNum");
		} else {
			ext(output, "x" "verbOps\0"
					"\xC4p|setCurVerb,"
					"\x7Cp|loadImg,"
					"\x7Dh|loadString,"
					"\x7Ep|setColor,"
					"\x7Fp|setHiColor,"
					"\x80pp|setXY,"
					"\x81|setOn,"
					"\x82|setOff,"
					"\x83p|kill,"
					"\x84|init,"
					"\x85p|setDimColor,"
					"\x86|setDimmed,"
					"\x87p|setKey,"
					"\x88|setCenter,"
					"\x89p|setToString,"
					"\x8Bpp|setToObject,"
					"\x8Cp|setBkColor,"
					"\xFF|redraw");
		}
		break;
	case 0x9F:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xA0:
		ext(output, "rpp|findObject");
		break;
	case 0xA1:
		ext(output, "lp|pseudoRoom");
		break;
	case 0xA2:
		ext(output, "rp|getActorElevation");
		break;
	case 0xA3:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xA4:
		switch (get_byte()) {
		case 7:
			se_a = se_get_string_he();
			writeArray(output, get_word(), NULL, se_a, se_a);
			break;
		case 126:
			// TODO: Add support for writeListIntoArray
			ext(output, "lppppv|writeListIntoArray");
			break;
		case 127:
			// TODO: Add support for copyArray
			ext(output, "ppppvppppv|copyArray");
			break;
		case 128:
			// TODO: Add support for writeRangeIntoArray
			ext(output, "ppppppv|writeRangeIntoArray");
			break;
		case 138:
			// TODO: Add support for math operations
			ext(output, "pppppppppppppvvv|mathOpsInArray");
			break;
		case 194:
			se_get_list();
			pop();
			se_a = se_get_string_he();
			writeArray(output, get_word(), NULL, se_a, se_a);
			break;
		case 208:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 212:
			se_b = se_get_list();
			se_a = pop();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		}
		break;
	case 0xA5:
		if (g_options.heVersion >= 99) {
			ext(output, "rx" "fontUnk\0"
					"\x2App|case42,"
					"\x39|case57");
		} else if (g_options.heVersion >= 80) {
			invalidop(NULL, code);
		} else {
			ext(output, "x" "saveRestoreVerbs\0"
					"\x8Dppp|saveVerbs,"
					"\x8Eppp|restoreVerbs,"
					"\x8Fppp|deleteVerbs");
		}
		break;
	case 0xA6:
		ext(output, "ppppp|drawBox");
		break;
	case 0xA8:
		ext(output, "rp|getActorWidth");
		break;
	case 0xA9:
		ext(output, "x" "wait\0"
				"\xA8pj|waitForActor,"
				"\xA9|waitForMessage,"
				"\xAA|waitForCamera,"
				"\xAB|waitForSentence");
		break;
	case 0xAA:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xAB:
		if (g_options.heVersion >= 90) {
			ext(output, "rp|getActorAnimProgress");
		} else {
			ext(output, "rp|getActorAnimCounter1");
		}
		break;
	case 0xAC: // HE80+
		ext(output, "pp|drawWizPolygon");
		break;
	case 0xAD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xAE:
		ext(output, "x" "systemOps\0"
				 "\x16|clearDrawQueue,"
				 "\x1A|copyVirtBuf,"
				 "\x9E|restart,"
				 "\xA0|confirmShutDown,"
				 "\xF4|shutDown,"
				 "\xFB|startExec,"
				 "\xFC|startGame");
		break;
	case 0xAF:
		ext(output, "rpp|isActorInBox");
		break;
	case 0xB0:
		ext(output, "p|delay");
		break;
	case 0xB1:
		ext(output, "p|delaySeconds");
		break;
	case 0xB2:
		ext(output, "p|delayMinutes");
		break;
	case 0xB3:
		ext(output, "|stopSentence");
		break;
	case 0xB4:
		PRINT_V7HE("printLine");
		break;
	case 0xB5:
		PRINT_V7HE("printCursor");
		break;
	case 0xB6:
		PRINT_V7HE("printDebug");
		break;
	case 0xB7:
		PRINT_V7HE("printSystem");
		break;
	case 0xB8:
		// This is *almost* identical to the other print opcodes, only the 'begine' subop differs
		ext(output, "x" "printActor\0"
				"\x41pp|XY,"
				"\x42p|color,"
				"\x43p|right,"
				"\x45|center,"
				"\x47|left,"
				"\x48|overhead,"
				"\x4A|mumble,"
				"\x4Bs|msg,"
				"\xE1p|getText,"
				"\xF9l|colors,"
				"\xFEp|begin,"
				"\xFF|end");
		break;
	case 0xB9:
		PRINT_V7HE("printEgo");
		break;
	case 0xBA:
		ext(output, "ps|talkActor");
		break;
	case 0xBB:
		ext(output, "s|talkEgo");
		break;
	case 0xBC:
		ext(output, "x" "dimArray\0"
				"\x2pv|bit,"
				"\x3pv|nibble,"
				"\x4pv|byte,"
				"\x5pv|int,"
				"\x6pv|dword,"
				"\x7pv|string,"
				"\xCCv|nukeArray");
		break;
	case 0xBD:
		ext(output, "|stopObjectCode");
		break;
	case 0xBE:
		// TODO: this loads another script which does something like
		// stack altering and then finishes (usually with opcode 0xBD).
		// When stack is changed, further disassembly is wrong.
		// This is widely used in HE games.
		// As there are cases when called script does not alter the
		// stack, it's not correct to use "rlpp|..." here
		ext(output, "lpp|startObjectQuick");
		break;
	case 0xBF:
		ext(output, "lp|startScriptQuick2");
		break;
	case 0xC0:
		ext(output, "x" "dim2dimArray\0"
				"\x2ppv|bit,"
				"\x3ppv|nibble,"
				"\x4ppv|byte,"
				"\x5ppv|int,"
				"\x6ppv|dword,"
				"\x7ppv|string");
		break;
	case 0xC1:
		ext(output, "hp|traceStatus");
		break;
	case 0xC4:
		ext(output, "rp|abs");
		break;
	case 0xC5:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xC6:
		ext(output, "rppp|getDistObjPt");
		break;
	case 0xC7:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xC8:
		ext(output, "ry" "kernelGetFunctions\0"
				"\x1|virtScreenSave"
				);
		break;
	case 0xC9:
		ext(output, "y" "kernelSetFunctions\0"
				"\x1|virtScreenLoad,"
				"\x14|queueAuxBlock,"
				"\x15|pauseDrawObjects,"
				"\x16|resumeDrawObjects,"
				"\x17|clearCharsetMask,"
				"\x18|pauseActors,"
				"\x19|resumActors,"
				"\x1E|actorBottomClipOverride,"
				"\x2A|setWizImageClip,"
				"\x2B|setWizImageClipOff,"
				);
		break;
	case 0xCA:
		ext(output, "p|delayFrames");
		break;
	case 0xCB:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCC:
		ext(output, "rplp|pickOneOfDefault");
		break;
	case 0xCD:
		ext(output, "pppp|stampObject");
		break;
	case 0xCE:
		ext(output, "pppp|drawWizImage");
		break;
	case 0xCF:
		ext(output, "rh|debugInput");
		break;
	case 0xD0:
		ext(output, "|getDateTime");
		break;
	case 0xD1:
		ext(output, "|stopTalking");
		break;
	case 0xD2:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xD4:
		ext(output, "wpp|shuffle");
		break;
	case 0xD5:
		ext(output, "lpi|jumpToScript");
		break;
	case 0xD6:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBand, se_a));
		break;
	case 0xD7:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBor, se_a));
		break;
	case 0xD8:
		ext(output, "rp|isRoomScriptRunning");
		break;
	case 0xD9:
		ext(output, "p|closeFile");
		break;
	case 0xDA:
		ext(output, "rph|openFile");
		break;
	case 0xDB:
		ext(output, "rx" "readFile\0"
				"\x4p|readByte,"
				"\x5p|readWord,"
				"\x6p|readDWord,"
				"\x8ppi|readArrayFromFile");
		break;
	case 0xDC:
		ext(output, "x" "writeFile\0"
				"\x4pp|writeByte,"
				"\x5pp|writeWord,"
				"\x6pp|writeDWord,"
				"\x8ppi|writeArrayToFile");
		break;
	case 0xDD:
		ext(output, "rp|findAllObjects");
		break;
	case 0xDE:
		ext(output, "h|deleteFile");
		break;
	case 0xDF:
		ext(output, "hh|renameFile");
		break;
	case 0xE0:
		ext(output, "x" "drawLine\0"
				"\x37pppppp|pixel,"
				"\x3Fpppppp|actor,"
				"\x42pppppp|wizImage");
		break;
	case 0xE1:
		ext(output, "rx" "getPixel\0"
				"\xDApp|background,"
				"\xDBpp|foreground");
		break;
	case 0xE2:
		ext(output, "p|localizeArrayToScript");
		break;
	case 0xE3:
		ext(output, "rlw|pickVarRandom");
		break;
	case 0xE4:
		ext(output, "p|setBotSet");
		break;
	case 0xE9:
		ext(output, "ppp|seekFilePos");
		break;
	case 0xEA:
		ext(output, "x" "redimArray\0"
				"\x4ppv|byte,"
				"\x5ppv|int,"
				"\x6ppv|dword");
		break;
	case 0xEB:
		ext(output, "rp|readFilePos");
		break;
	case 0xEC:
		ext(output, "rp|copyString");
		break;
	case 0xED:
		ext(output, "rppp|getStringWidth");
		break;
	case 0xEE:
		ext(output, "rp|getStringLen");
		break;
	case 0xEF:
		ext(output, "rppp|appendString");
		break;
	case 0xF0:
		ext(output, "rpp|concatString");
		break;
	case 0xF1:
		ext(output, "rpp|compareString");
		break;
	case 0xF2:
		ext(output, "rx" "isResourceLoaded\0"
				"\x12p|image,"
				"\xE2p|room,"
				"\xE3p|costume,"
				"\xE4p|sound,"
				"\xE5p|script");
		break;
	case 0xF3:
		ext(output, "rx" "readINI\0"
				"\x06h|number,"
				"\x07h|string");
		break;
	case 0xF4:
		ext(output, "x" "writeINI\0"
				"\x06ph|number,"
				"\x07hh|string");
		break;
	case 0xF5:
		ext(output, "rppp|getStringLenForWidth");
		break;
	case 0xF6:
		ext(output, "rpppp|getCharIndexInString");
		break;
	case 0xF8:
		if (g_options.heVersion >= 73) {
			ext(output, "rx" "getResourceSize\0"
					"\xDp|sound,"
					"\xEp|roomImage,"
					"\xFp|image,"
					"\x10p|costume,"
					"\x11p|script");
		} else {
			ext(output, "rp|getSoundResourceSize");
		}
		break;
	case 0xF9:
		ext(output, "h|createDirectory");
		break;
	case 0xFA:
		ext(output, "x" "setSystemMessage\0"
				"\xF0h|case240,"
				"\xF1h|versionMsg,"
				"\xF2h|case242,"
				"\xF3h|titleMsg");
		break;
	case 0xFB:
		ext(output, "x" "polygonOps\0"
				"\xF6ppppppppp|polygonStore,"
				"\xF7pp|polygonErase,"
				"\xF8ppppppppp|polygonStore,"
				);
		break;
	case 0xFC:
			ext(output, "rpp|polygonHit");
		break;

	default:
		invalidop(NULL, code);
		break;
	}
}

#define PRINT_V8(name)            \
	do {                          \
		ext(output,               \
				"x" name "\0"     \
				"\xC8|baseop,"    \
				"\xC9|end,"       \
				"\xCApp|XY,"      \
				"\xCBp|color,"    \
				"\xCC|center,"    \
				"\xCDp|charset,"  \
				"\xCE|left,"      \
				"\xCF|overhead,"  \
				"\xD0|mumble,"    \
				"\xD1s|msg,"      \
				"\xD2|wrap"       \
				);                \
	} while(0)


void next_line_V8(char *output) {
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_var(get_word()));
		break;
	case 0x3:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0x4:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0x5:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0x6:
		kill(output, pop());
		break;
	case 0x7:
		push(se_oper(pop(), isZero));
		break;
	case 0x8:
	case 0x9:
	case 0xA:
	case 0xB:
	case 0xC:
	case 0xD:
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0x8) + isEqual, se_a));
		break;

	case 0x64:
		jumpif(output, pop(), true);
		break;
	case 0x65:
		jumpif(output, pop(), false);
		break;
	case 0x66:
		jump(output);
		break;
	case 0x67:
		ext(output, "|breakHere");
		break;
	case 0x68:
		ext(output, "p|delayFrames");
		break;
	case 0x69:
		ext(output, "x" "wait\0"
				"\x1Epj|waitForActor,"
				"\x1F|waitForMessage,"
				"\x20|waitForCamera,"
				"\x21|waitForSentence,"
				"\x22pj|waitUntilActorDrawn,"
				"\x23pj|waitUntilActorTurned,"
				);
		break;
	case 0x6A:
		ext(output, "p|delay");
		break;
	case 0x6B:
		ext(output, "p|delaySeconds");
		break;
	case 0x6C:
		ext(output, "p|delayMinutes");
		break;
	case 0x6D:
		writeVar(output, get_word(), pop());
		break;
	case 0x6E:
		addVar(output, get_word(), +1);
		break;
	case 0x6F:
		addVar(output, get_word(), -1);
		break;
	case 0x70:
		// FIXME - is this correct?!? Also, make the display nicer...
		ext(output, "x" "dimArray\0"
				"\x0Apw|dim-scummvar,"
				"\x0Bpw|dim-string,"
				"\xCAw|undim"
				);
		break;
	case 0x71:
		se_a = pop();
		writeArray(output, get_word(), NULL, pop(), se_a);
		break;

	case 0x74:
		// FIXME - is this correct?!? Also, make the display nicer...
		ext(output, "x" "dim2dimArray\0"
				"\x0Appw|dim2-scummvar,"
				"\x0Bppw|dim2-string,"
				"\xCAw|undim2"
				);
		break;
	case 0x75:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_word(), pop(), se_b, se_a);
		break;
	case 0x76:
		switch (get_byte()) {
		case 0x14:{
			int array = get_word();
			writeArray(output, array, NULL, pop(), se_get_string());
			}
			break;
		case 0x15:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 0x16:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), pop(), se_a, se_b);
			break;
		}
		break;

	case 0x79:
		ext(output, "lpp|startScript");
		break;
	case 0x7A:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x7B:
		ext(output, "|stopObjectCode");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "lpp|jumpToScript");
		break;
	case 0x7E:
		ext(output, "p|return");
		break;
	case 0x7F:
		ext(output, "lppp|startObject");
		break;

	case 0x81:
		ext(output, "l|beginCutscene");
		break;
	case 0x82:
		ext(output, "|endCutscene");
		break;
	case 0x83:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x84:
		ext(output, "|beginOverride");
		break;
	case 0x85:
		ext(output, "|endOverride");
		break;
	case 0x86:
		ext(output, "|stopSentence");
		break;
	case 0x87:
		ext(output, "p|debug");
		break;
	case 0x89:
		ext(output, "lp|setClass");
		break;
	case 0x8A:
		ext(output, "pp|setState");
		break;
	case 0x8B:
		ext(output, "pp|setOwner");
		break;
	case 0x8C:
		ext(output, "pp|panCameraTo");
		break;
	case 0x8D:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x8E:
		ext(output, "pp|setCameraAt");
		break;
	case 0x8F:
		ext(output,
				"x" "printActor\0"
				"\xC8p|baseop,"
				"\xC9|end,"
				"\xCApp|XY,"
				"\xCBp|color,"
				"\xCC|center,"
				"\xCDp|charset,"
				"\xCE|left,"
				"\xCF|overhead,"
				"\xD0|mumble,"
				"\xD1s|msg,"
				"\xD2|wrap"
				);
		break;
	case 0x90:
		PRINT_V8("printEgo");
		break;
	case 0x91:
		ext(output, "ps|talkActor");
		break;
	case 0x92:
		ext(output, "s|talkEgo");
		break;
	case 0x93:
		PRINT_V8("printLine");
		break;
	case 0x94:
		PRINT_V8("printCursor");
		break;
	case 0x95:
		PRINT_V8("printDebug");
		break;
	case 0x96:
		PRINT_V8("printSystem");
		break;
	case 0x97:
		PRINT_V8("blastText");
		break;
	case 0x98:
		ext(output, "pppp|drawObject");
		break;

	case 0x9C:
		ext(output, "x" "cursorCommand\0"
				"\xDC|cursorOn,"
				"\xDD|cursorOff,"
				"\xDE|userPutOn,"
				"\xDF|userPutOff,"
				"\xE0|softCursorOn,"
				"\xE1|softCursorOff,"
				"\xE2|softUserputOn,"
				"\xE3|softUserputOff,"
				"\xE4pp|setCursorImg,"
				"\xE5pp|setCursorHotspot,"
				"\xE6p|makeCursorColorTransparent,"
				"\xE7p|initCharset,"
				"\xE8l|charsetColors,"
				"\xE9pp|setCursorPosition");
		break;
	case 0x9D:
		ext(output, "p|loadRoom");
		break;
	case 0x9E:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x9F:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0xA0:
		ext(output, "ppp|walkActorTo");
		break;
	case 0xA1:
		ext(output, "pppp|putActorAtXY");
		break;
	case 0xA2:
		ext(output, "zp|putActorAtObject");
		break;
	case 0xA3:
		ext(output, "pp|faceActor");
		break;
	case 0xA4:
		ext(output, "pp|animateActor");
		break;
	case 0xA5:
		ext(output, "ppp|doSentence");
		break;
	case 0xA6:
		ext(output, "z|pickupObject");
		break;
	case 0xA7:
		ext(output, "pl|setBoxFlags");
		break;
	case 0xA8:
		ext(output, "|createBoxMatrix");
		break;

	case 0xAA:
		ext(output, "x" "resourceRoutines\0"
				"\x3Cp|loadCharset,"
				"\x3Dp|loadCostume,"
				"\x3Ep|loadObject,"
				"\x3Fp|loadRoom,"
				"\x40p|loadScript,"
				"\x41p|loadSound,"
				"\x42p|lockCostume,"
				"\x43p|lockRoom,"
				"\x44p|lockScript,"
				"\x45p|lockSound,"
				"\x46p|unlockCostume,"
				"\x47p|unlockRoom,"
				"\x48p|unlockScript,"
				"\x49p|unlockSound,"
				"\x4Ap|nukeCostume,"
				"\x4Bp|nukeRoom,"
				"\x4Cp|nukeScript,"
				"\x4Dp|nukeSound"
				);
		break;
	case 0xAB:
		ext(output, "x" "roomOps\0"
				"\x52pppp|setRoomPalette,"
				"\x55ppp|setRoomIntensity,"
				"\x57p|fade,"
				"\x58ppppp|setRoomRBGIntensity,"
				"\x59pppp|transformRoom,"
				"\x5App|colorCycleDelay,"
				"\x5Bpp|copyPalette,"
				"\x5Cp|newPalette,"
				"\x5D|saveGame,"
				"\x5Ep|LoadGame,"
				"\x5Fppppp|setRoomSaturation"
				);
		break;
	case 0xAC:
		// Note: these are guesses and may partially be wrong
		ext(output, "x" "actorOps\0"
				"\x64p|setActorCostume,"
				"\x65pp|setActorWalkSpeed,"
				"\x67|setActorDefAnim,"
				"\x68p|setActorInitFrame,"
				"\x69pp|setActorTalkFrame,"
				"\x6Ap|setActorWalkFrame,"
				"\x6Bp|setActorStandFrame,"
				"\x6C|setActorAnimSpeed,"
				"\x6D|setActorDefault,"	// = initActorLittle ?
				"\x6Ep|setActorElevation,"
				"\x6Fpp|setActorPalette,"
				"\x70p|setActorTalkColor,"
				"\x71s|setActorName,"
				"\x72p|setActorWidth,"
				"\x73p|setActorScale,"
				"\x74|setActorNeverZClip,"
				"\x75p|setActorAlwayZClip?,"
				"\x76|setActorIgnoreBoxes,"
				"\x77|setActorFollowBoxes,"
				"\x78p|setShadowMode,"
				"\x79pp|setActorTalkPos,"
				"\x7Ap|setCurActor,"
				"\x7Bpp|setActorAnimVar,"
				"\x7C|setActorIgnoreTurnsOn,"
				"\x7D|setActorIgnoreTurnsOff,"
				"\x7E|newActor,"
				"\x7Fp|setActorLayer,"
				"\x80|setActorStanding,"
				"\x81p|setActorDirection,"
				"\x82p|actorTurnToDirection,"
				"\x83p|setActorWalkScript,"
				"\x84p|setTalkScript,"
				"\x85|freezeActor,"
				"\x86|unfreezeActor,"
				"\x87p|setActorVolume,"
				"\x88p|setActorFrequency,"
				"\x89p|setActorPan"
				);
		break;
	case 0xAD:
		ext(output, "x" "cameraOps\0"
				"\x32|freezeCamera,"
				"\x33|unfreezeCamera"
				);
		break;
	case 0xAE:
		ext(output, "x" "verbOps\0"
				"\x96p|verbInit,"
				"\x97|verbNew,"
				"\x98|verbDelete,"
				"\x99s|verbLoadString,"
				"\x9App|verbSetXY,"
				"\x9B|verbOn,"
				"\x9C|verbOff,"
				"\x9Dp|verbSetColor,"
				"\x9Ep|verbSetHiColor,"

				"\xA0p|verbSetDimColor,"
				"\xA1|verbSetDim,"
				"\xA2p|verbSetKey,"
				"\xA3p|verbLoadImg,"
				"\xA4p|verbSetToString,"
				"\xA5|verbSetCenter,"
				"\xA6p|verbSetCharset,"
				"\xA7p|verbSetLineSpacing"
				);
		break;
	case 0xAF:
		ext(output, "p|startSound");
		break;

	case 0xB1:
		ext(output, "p|stopSound");
		break;
	case 0xB2:
		ext(output, "l|soundKludge");
		break;
	case 0xB3:
		ext(output, "x" "systemOps\0"
				"\x28|restart,"
				"\x29|quit");
		break;
	case 0xB4:
		ext(output, "x" "saveRestoreVerbs\0"
				"\xB4ppp|saveVerbs,"
				"\xB5ppp|restoreVerbs,"
				"\xB6ppp|deleteVerbs");
		break;
	case 0xB5:
		ext(output, "ps|setObjectName");
		break;
	case 0xB6:
		ext(output, "|getDateTime");
		break;
	case 0xB7:
		ext(output, "ppppp|drawBox");
		break;

	case 0xB9:
		ext(output, "s|startVideo");
		break;
	case 0xBA:
		ext(output, "y" "kernelSetFunctions\0"
				"\xB|lockObject,"
				"\xC|unlockObject,"
				"\xD|remapCostume,"
				"\xE|remapCostumeInsert,"
				"\xF|setVideoFrameRate,"

				"\x14|setBoxScale,"
				"\x15|setScaleSlot,"
				"\x16|setBannerColors,"
				"\x17|setActorChoreLimbFrame,"
				"\x18|clearTextQueue,"
				"\x19|saveGameWrite,"
				"\x1A|saveGameRead,"
				"\x1B|saveGameReadName,"
				"\x1C|saveGameStampScreenshot,"
				"\x1D|setKeyScript,"
				"\x1E|killAllScriptsExceptCurrent,"
				"\x1F|stopAllVideo,"
				"\x20|writeRegistryValue,"
				"\x21|paletteSetIntensity,"
				"\x22|queryQuit,"

				"\x6C|buildPaletteShadow,"
				"\x6D|setPaletteShadow,"

				"\x76|blastShadowObject,"
				"\x77|superBlastObject"
				);
		break;

	case 0xC8:
		ext(output, "rlp|startScriptQuick2");
		break;
	case 0xC9:
		ext(output, "lppp|startObjectQuick");
		break;
	case 0xCA:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCB:
		ext(output, "rplp|pickOneOfDefault");
		break;

	case 0xCD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xCE:
		ext(output, "rp|getRandomNumber");
		break;
	case 0xCF:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0xD0:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0xD1:
		ext(output, "rp|getState");
		break;
	case 0xD2:
		ext(output, "rp|getOwner");
		break;
	case 0xD3:
		ext(output, "rp|isScriptRunning");
		break;

	case 0xD5:
		ext(output, "rp|isSoundRunning");
		break;
	case 0xD6:
		ext(output, "rp|abs");
		break;

	case 0xD8:
		ext(output, "ry" "kernelGetFunctions\0"
				"\x73|getWalkBoxAt,"
				"\x74|isPointInBox,"
				"\xCE|getRGBSlot,"
				"\xD3|getKeyState,"
				"\xD7|getBox,"
				"\xD8|findBlastObject,"
				"\xD9|actorHit,"
				"\xDA|lipSyncWidth,"
				"\xDB|lipSyncHeight,"
				"\xDC|actorTalkAnimation,"
				"\xDD|getMasterSFXVol,"
				"\xDE|getMasterVoiceVol,"
				"\xDF|getMasterMusicVol,"
				"\xE0|readRegistryValue,"
				"\xE1|imGetMusicPosition,"
				"\xE2|musicLipSyncWidth,"
				"\xE3|musicLipSyncHeight"
				);
		break;
	case 0xD9:
		ext(output, "rpp|isActorInBox");
		break;
	case 0xDA:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xDB:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xDC:
		ext(output, "rpp|findObject");
		break;
	case 0xDD:
		ext(output, "rpp|getVerbFromXY");
		break;

	case 0xDF:
		ext(output, "rpp|findInventory");
		break;
	case 0xE0:
		ext(output, "rp|getInventoryCount");
		break;
	case 0xE1:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xE2:
		ext(output, "rp|getActorRoom");
		break;
	case 0xE3:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0xE4:
		ext(output, "rp|getActorMoving");
		break;
	case 0xE5:
		ext(output, "rp|getActorCostume");
		break;
	case 0xE6:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xE7:
		ext(output, "rp|getActorLayer");
		break;
	case 0xE8:
		ext(output, "rp|getActorElevation");
		break;
	case 0xE9:
		ext(output, "rp|getActorWidth");
		break;
	case 0xEA:
		ext(output, "rp|getObjectDir");
		break;
	case 0xEB:
		ext(output, "rp|getObjectX");
		break;
	case 0xEC:
		ext(output, "rp|getObjectY");
		break;
	case 0xED:
		ext(output, "rp|getActorChore");
		break;
	case 0xEE:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xEF:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xF0:
		ext(output, "rp|getObjectImageX");
		break;
	case 0xF1:
		ext(output, "rp|getObjectImageY");
		break;
	case 0xF2:
		ext(output, "rp|getObjectImageWidth");
		break;
	case 0xF3:
		ext(output, "rp|getObjectImageHeight");
		break;
	case 0xF4:
		ext(output, "rp|getVerbX");
		break;
	case 0xF5:
		ext(output, "rp|getVerbY");
		break;
	case 0xF6:
		ext(output, "rps|stringWidth");
		break;
	case 0xF7:
		ext(output, "rp|getActorZPlane");
		break;

	default:
		invalidop(NULL, code);
		break;
	}
}

#define PRINT_V67(name)           \
	do {                          \
		ext(output, "x" name "\0"         \
				"\x41pp|XY,"      \
				"\x42p|color,"    \
				"\x43p|right,"    \
				"\x45|center,"    \
				"\x47|left,"      \
				"\x48|overhead,"  \
				"\x4A|mumble,"    \
				"\x4Bs|msg,"      \
				"\xFE|begin,"     \
				"\xFF|end"        \
				);                \
	} while(0)

void next_line_V67(char *output) {
	byte code = get_byte();
	StackEnt *se_a, *se_b;

	switch (code) {
	case 0x0:
		push(se_int(get_byte()));
		break;
	case 0x1:
		push(se_int(get_word()));
		break;
	case 0x2:
		push(se_var(get_byte()));
		break;
	case 0x3:
		push(se_var(get_word()));
		break;
	case 0x6:
		push(se_array(get_byte(), NULL, pop()));
		break;
	case 0x7:
		push(se_array(get_word(), NULL, pop()));
		break;
	case 0xA:
		se_a = pop();
		push(se_array(get_byte(), pop(), se_a));
		break;
	case 0xB:
		se_a = pop();
		push(se_array(get_word(), pop(), se_a));
		break;
	case 0xC:
		se_a = dup(output, pop());
		push(se_a);
		push(se_a);
		break;
	case 0xD:
		push(se_oper(pop(), isZero));
		break;
	case 0xE:
	case 0xF:
	case 0x10:
	case 0x11:
	case 0x12:
	case 0x13:
	case 0x14:
	case 0x15:
	case 0x16:
	case 0x17:
	case 0x18:
	case 0x19:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, (code - 0xE) + isEqual, se_a));
		break;
	case 0x1A:
	case 0xA7:
		kill(output, pop());
		break;
	case 0x42:
		writeVar(output, get_byte(), pop());
		break;
	case 0x43:
		writeVar(output, get_word(), pop());
		break;
	case 0x46:
		se_a = pop();
		writeArray(output, get_byte(), NULL, pop(), se_a);
		break;
	case 0x47:
		se_a = pop();
		writeArray(output, get_word(), NULL, pop(), se_a);
		break;
	case 0x4A:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_byte(), pop(), se_b, se_a);
		break;
	case 0x4B:
		se_a = pop();
		se_b = pop();
		writeArray(output, get_word(), pop(), se_b, se_a);
		break;
	case 0x4E:
		addVar(output, get_byte(), 1);
		break;
	case 0x4F:
		addVar(output, get_word(), 1);
		break;
	case 0x52:
		addArray(output, get_byte(), pop(), 1);
		break;
	case 0x53:
		addArray(output, get_word(), pop(), 1);
		break;
	case 0x56:
		addVar(output, get_byte(), -1);
		break;
	case 0x57:
		addVar(output, get_word(), -1);
		break;
	case 0x5A:
		addArray(output, get_byte(), pop(), -1);
		break;
	case 0x5B:
		addArray(output, get_word(), pop(), -1);
		break;
	case 0x5C:
		jumpif(output, pop(), true);
		break;
	case 0x5D:
		jumpif(output, pop(), false);
		break;
	case 0x5E:
		ext(output, "lpp|startScript");
		break;
	case 0x5F:
		ext(output, "lp|startScriptQuick");
		break;
	case 0x60:
		ext(output, "lppp|startObject");
		break;
	case 0x61:
		ext(output, "pp|drawObject");
		break;
	case 0x62:
		ext(output, "ppp|drawObjectAt");
		break;
	case 0x63:
		if (g_options.heVersion)
			invalidop(NULL, code);
		else
			ext(output, "ppppp|drawBlastObject");
		break;
	case 0x64:
		if (g_options.heVersion)
			invalidop(NULL, code);
		else
			ext(output, "pppp|setBlastObjectWindow");
		break;
	case 0x65:
		ext(output, "|stopObjectCodeA");
		break;
	case 0x66:
		ext(output, "|stopObjectCodeB");
		break;
	case 0x67:
		ext(output, "|endCutscene");
		break;
	case 0x68:
		ext(output, "l|beginCutscene");
		break;
	case 0x69:
		ext(output, "|stopMusic");
		break;
	case 0x6A:
		ext(output, "p|freezeUnfreeze");
		break;
	case 0x6B:
		ext(output, "x" "cursorCommand\0"
				"\x90|cursorOn,"
				"\x91|cursorOff,"
				"\x92|userPutOn,"
				"\x93|userPutOff,"
				"\x94|softCursorOn,"
				"\x95|softCursorOff,"
				"\x96|softUserputOn,"
				"\x97|softUserputOff,"
				"\x99z|setCursorImg,"
				"\x9App|setCursorHotspot,"
				"\x9Cp|initCharset,"
				"\x9Dl|charsetColors,"
				"\xD6p|makeCursorColorTransparent");
		break;
	case 0x6C:
		ext(output, "|breakHere");
		break;
	case 0x6D:
		ext(output, "rlp|ifClassOfIs");
		break;
	case 0x6E:
		ext(output, "lp|setClass");
		break;
	case 0x6F:
		ext(output, "rp|getState");
		break;
	case 0x70:
		ext(output, "pp|setState");
		break;
	case 0x71:
		ext(output, "pp|setOwner");
		break;
	case 0x72:
		ext(output, "rp|getOwner");
		break;
	case 0x73:
		jump(output);
		break;
	case 0x74:
		if (g_options.heVersion >= 70) {
			ext(output, "x" "startSound\0"
					"\x9|setSoundFlag4,"
					"\x17ppp|setSoundVar,"
					"\x19pp|startWithFlag8,"
					"\x38|setQuickStartFlag,"
					"\xA4|setForceQueueFlag,"
					"\xDE|dummy,"
					"\xE0p|setFrequency,"
					"\xE6p|setChannel,"
					"\xE7p|setOffset,"
					"\xE8p|setId,"
					"\xF5|setLoop,"
					"\xFF|start");
			break;
		} else if (g_options.heVersion) {
			ext(output, "pp|startSound");
		} else {
			ext(output, "p|startSound");
		}
		break;
	case 0x75:
		ext(output, "p|stopSound");
		break;
	case 0x76:
		ext(output, "p|startMusic");
		break;
	case 0x77:
		ext(output, "p|stopObjectScript");
		break;
	case 0x78:
		if (g_options.scriptVersion < 7)
			ext(output, "p|panCameraTo");
		else
			ext(output, "pp|panCameraTo");
		break;
	case 0x79:
		ext(output, "p|actorFollowCamera");
		break;
	case 0x7A:
		if (g_options.scriptVersion < 7)
			ext(output, "p|setCameraAt");
		else
			ext(output, "pp|setCameraAt");
		break;
	case 0x7B:
		ext(output, "p|loadRoom");
		break;
	case 0x7C:
		ext(output, "p|stopScript");
		break;
	case 0x7D:
		ext(output, "ppp|walkActorToObj");
		break;
	case 0x7E:
		ext(output, "ppp|walkActorTo");
		break;
	case 0x7F:
		ext(output, "pppp|putActorInXY");
		break;
	case 0x80:
		ext(output, "zp|putActorAtObject");
		break;
	case 0x81:
		ext(output, "pp|faceActor");
		break;
	case 0x82:
		ext(output, "pp|animateActor");
		break;
	case 0x83:
		ext(output, "pppp|doSentence");
		break;
	case 0x84:
		ext(output, "z|pickupObject");
		break;
	case 0x85:
		ext(output, "ppz|loadRoomWithEgo");
		break;
	case 0x87:
		ext(output, "rp|getRandomNumber");
		break;
	case 0x88:
		ext(output, "rpp|getRandomNumberRange");
		break;
	case 0x8A:
		ext(output, "rp|getActorMoving");
		break;
	case 0x8B:
		ext(output, "rp|isScriptRunning");
		break;
	case 0x8C:
		ext(output, "rp|getActorRoom");
		break;
	case 0x8D:
		ext(output, "rp|getObjectX");
		break;
	case 0x8E:
		ext(output, "rp|getObjectY");
		break;
	case 0x8F:
		ext(output, "rp|getObjectDir");
		break;
	case 0x90:
		ext(output, "rp|getActorWalkBox");
		break;
	case 0x91:
		ext(output, "rp|getActorCostume");
		break;
	case 0x92:
		ext(output, "rpp|findInventory");
		break;
	case 0x93:
		ext(output, "rp|getInventoryCount");
		break;
	case 0x94:
		ext(output, "rpp|getVerbFromXY");
		break;
	case 0x95:
		ext(output, "|beginOverride");
		break;
	case 0x96:
		ext(output, "|endOverride");
		break;
	case 0x97:
		ext(output, "ps|setObjectName");
		break;
	case 0x98:
		ext(output, "rp|isSoundRunning");
		break;
	case 0x99:
		ext(output, "pl|setBoxFlags");
		break;
	case 0x9A:
		if (g_options.heVersion)
			invalidop(NULL, code);
		else
			ext(output, "|createBoxMatrix");
		break;
	case 0x9B:
		if (g_options.heVersion)
			ext(output, "x" "resourceRoutines\0"
					"\x64p|loadScript,"
					"\x65p|loadSound,"
					"\x66p|loadCostume,"
					"\x67p|loadRoom,"
					"\x68p|nukeScript,"
					"\x69p|nukeSound,"
					"\x6Ap|nukeCostume,"
					"\x6Bp|nukeRoom,"
					"\x6Cp|lockScript,"
					"\x6Dp|lockSound,"
					"\x6Ep|lockCostume,"
					"\x6Fp|lockRoom,"
					"\x70p|unlockScript,"
					"\x71p|unlockSound,"
					"\x72p|unlockCostume,"
					"\x73p|unlockRoom,"
					"\x75p|loadCharset,"
					"\x76p|nukeCharset,"
					"\x77z|loadFlObject,"
					"\x78p|queueloadScript,"
					"\x79p|queueloadSound,"
					"\x7Ap|queueloadCostume,"
					"\x7Bp|queueloadRoomImage,"
					"\x9fp|unlockImage,"
					"\xc0p|nukeImage,"
					"\xc9p|loadImage,"
					"\xcap|lockImage,"
					"\xcbp|queueloadImage,"
					"\xe9p|lockFlObject,"
					"\xebp|unlockFlObject,"
					"\xef|dummy");
		else
			ext(output, "x" "resourceRoutines\0"
					"\x64p|loadScript,"
					"\x65p|loadSound,"
					"\x66p|loadCostume,"
					"\x67p|loadRoom,"
					"\x68p|nukeScript,"
					"\x69p|nukeSound,"
					"\x6Ap|nukeCostume,"
					"\x6Bp|nukeRoom,"
					"\x6Cp|lockScript,"
					"\x6Dp|lockSound,"
					"\x6Ep|lockCostume,"
					"\x6Fp|lockRoom,"
					"\x70p|unlockScript,"
					"\x71p|unlockSound,"
					"\x72p|unlockCostume,"
					"\x73p|unlockRoom,"
					"\x75p|loadCharset,"
					"\x76p|nukeCharset,"
					"\x77z|loadFlObject");
		break;
	case 0x9C:
		if (g_options.heVersion)
			ext(output, "x" "roomOps\0"
					"\xACpp|roomScroll,"
					"\xAEpp|setScreen,"
					"\xAFpppp|setPalColor,"
					"\xB0|shakeOn,"
					"\xB1|shakeOff,"
					"\xB3ppp|darkenPalette,"
					"\xB4pp|saveLoadRoom,"
					"\xB5p|screenEffect,"
					"\xB6ppppp|darkenPalette,"
					"\xB7ppppp|setupShadowPalette,"
					"\xBApppp|palManipulate,"
					"\xBBpp|colorCycleDelay,"
					"\xD5p|setPalette,"
					"\xDCpp|copyPalColor,"
					"\xDDsp|saveLoad,"
					"\xEApp|swapObjects,"
					"\xECpp|setRoomPalette");
		else
			ext(output, "x" "roomOps\0"
					"\xACpp|roomScroll,"
					"\xAEpp|setScreen,"
					"\xAFpppp|setPalColor,"
					"\xB0|shakeOn,"
					"\xB1|shakeOff,"
					"\xB3ppp|darkenPalette,"
					"\xB4pp|saveLoadRoom,"
					"\xB5p|screenEffect,"
					"\xB6ppppp|darkenPalette,"
					"\xB7ppppp|setupShadowPalette,"
					"\xBApppp|palManipulate,"
					"\xBBpp|colorCycleDelay,"
					"\xD5p|setPalette,"
					"\xDCpp|copyPalColor");
		break;
	case 0x9D:
		if (g_options.heVersion)
			ext(output, "x" "actorOps\0"
					"\xC5p|setCurActor,"
					"\x1Epppp|setClipRect,"
					"\x4Cp|setCostume,"
					"\x4Dpp|setWalkSpeed,"
					"\x4El|setSound,"
					"\x4Fp|setWalkFrame,"
					"\x50pp|setTalkFrame,"
					"\x51p|setStandFrame,"
					"\x52ppp|actorSet:82:??,"
					"\x53|init,"
					"\x54p|setElevation,"
					"\x55|setDefAnim,"
					"\x56pp|setPalette,"
					"\x57p|setTalkColor,"
					"\x58s|setName,"
					"\x59p|setInitFrame,"
					"\x5Bp|setWidth,"
					"\x5Cp|setScale,"
					"\x5D|setNeverZClip,"
					"\x5Ep|setAlwayZClip,"
					"\x5F|setIgnoreBoxes,"
					"\x60|setFollowBoxes,"
					"\x61p|setAnimSpeed,"
					"\x62p|setShadowMode,"
					"\x63pp|setTalkPos,"
					"\xC6pp|setAnimVar,"
					"\xD7|setIgnoreTurnsOn,"
					"\xD8|setIgnoreTurnsOff,"
					"\xD9|initLittle,"
					"\xDA|drawToBackBuf,"
					"\xE1sp|setTalkieSlot");
		else
			ext(output, "x" "actorOps\0"
					"\xC5p|setCurActor,"
					"\x4Cp|setCostume,"
					"\x4Dpp|setWalkSpeed,"
					"\x4El|setSound,"
					"\x4Fp|setWalkFrame,"
					"\x50pp|setTalkFrame,"
					"\x51p|setStandFrame,"
					"\x52ppp|actorSet:82:??,"
					"\x53|init,"
					"\x54p|setElevation,"
					"\x55|setDefAnim,"
					"\x56pp|setPalette,"
					"\x57p|setTalkColor,"
					"\x58s|setName,"
					"\x59p|setInitFrame,"
					"\x5Bp|setWidth,"
					"\x5Cp|setScale,"
					"\x5D|setNeverZClip,"
					"\x5Ep|setAlwayZClip,"
					"\x5F|setIgnoreBoxes,"
					"\x60|setFollowBoxes,"
					"\x61p|setAnimSpeed,"
					"\x62p|setShadowMode,"
					"\x63pp|setTalkPos,"
					"\xC6pp|setAnimVar,"
					"\xD7|setIgnoreTurnsOn,"
					"\xD8|setIgnoreTurnsOff,"
					"\xD9|initLittle,"
					"\xE1p|setAlwayZClip?,"
					"\xE3p|setLayer,"
					"\xE4p|setWalkScript,"
					"\xE5|setStanding,"
					"\xE6p|setDirection,"
					"\xE7p|turnToDirection,"
					"\xE9|freeze,"
					"\xEA|unfreeze,"
					"\xEBp|setTalkScript");
		break;
	case 0x9E:
		if (g_options.heVersion)
			ext(output, "x" "verbOps\0"
					"\xC4p|setCurVerb,"
					"\x7Cp|loadImg,"
					"\x7Ds|loadString,"
					"\x7Ep|setColor,"
					"\x7Fp|setHiColor,"
					"\x80pp|setXY,"
					"\x81|setOn,"
					"\x82|setOff,"
					"\x83p|kill,"
					"\x84|init,"
					"\x85p|setDimColor,"
					"\x86|setDimmed,"
					"\x87p|setKey,"
					"\x88|setCenter,"
					"\x89p|setToString,"
					"\x8Bpp|setToObject,"
					"\x8Cp|setBkColor,"
					"\xFF|redraw");
		else
			ext(output, "x" "verbOps\0"
					"\xC4p|setCurVerb,"
					"\x7Cp|loadImg,"
					"\x7Ds|loadString,"
					"\x7Ep|setColor,"
					"\x7Fp|setHiColor,"
					"\x80pp|setXY,"
					"\x81|setOn,"
					"\x82|setOff,"
					"\x83|kill,"
					"\x84|init,"
					"\x85p|setDimColor,"
					"\x86|setDimmed,"
					"\x87p|setKey,"
					"\x88|setCenter,"
					"\x89p|setToString,"
					"\x8Bpp|setToObject,"
					"\x8Cp|setBkColor,"
					"\xFF|redraw");
		break;
	case 0x9F:
		ext(output, "rpp|getActorFromXY");
		break;
	case 0xA0:
		ext(output, "rpp|findObject");
		break;
	case 0xA1:
		ext(output, "lp|pseudoRoom");
		break;
	case 0xA2:
		ext(output, "rp|getActorElevation");
		break;
	case 0xA3:
		ext(output, "rpp|getVerbEntrypoint");
		break;
	case 0xA4:
		switch (get_byte()) {
		case 205:{
			int array = get_word();
			writeArray(output, array, NULL, pop(), se_get_string());
			}
			break;
		case 208:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), NULL, se_a, se_b);
			break;
		case 212:
			se_a = pop();
			se_b = se_get_list();
			writeArray(output, get_word(), pop(), se_a, se_b);
			break;
		}
		break;
	case 0xA5:
		ext(output, "x" "saveRestoreVerbs\0"
				"\x8Dppp|saveVerbs,"
				"\x8Eppp|restoreVerbs,"
				"\x8Fppp|deleteVerbs");
		break;
	case 0xA6:
		ext(output, "ppppp|drawBox");
		break;
	case 0xA8:
		ext(output, "rp|getActorWidth");
		break;
	case 0xA9:
		ext(output, "x" "wait\0"
				"\xA8pj|waitForActor,"
				"\xA9|waitForMessage,"
				"\xAA|waitForCamera,"
				"\xAB|waitForSentence,"
				"\xE2pj|waitUntilActorDrawn,"
				"\xE8pj|waitUntilActorTurned,"
				);
		break;
	case 0xAA:
		ext(output, "rp|getActorScaleX");
		break;
	case 0xAB:
		ext(output, "rp|getActorAnimCounter1");
		break;
	case 0xAC:
		ext(output, "l|soundKludge");
		break;
	case 0xAD:
		ext(output, "rlp|isAnyOf");
		break;
	case 0xAE:
		if (g_options.heVersion)
			ext(output, "x" "systemOps\0"
					 "\x9E|restart,"
					 "\xA0|confirmShutDown,"
					 "\xF4|shutDown,"
					 "\xFB|startExec,"
					 "\xFC|startGame");
		else
			ext(output, "x" "systemOps\0"
					"\x9E|restartGame,"
					"\x9F|pauseGame,"
					"\xA0|shutDown");
		break;
	case 0xAF:
		ext(output, "rpp|isActorInBox");
		break;
	case 0xB0:
		ext(output, "p|delay");
		break;
	case 0xB1:
		ext(output, "p|delaySeconds");
		break;
	case 0xB2:
		ext(output, "p|delayMinutes");
		break;
	case 0xB3:
		ext(output, "|stopSentence");
		break;
	case 0xB4:
		if (g_options.heVersion)
			PRINT_V7HE("printLine");
		else
			PRINT_V67("printLine");
		break;
	case 0xB5:
		if (g_options.heVersion)
			PRINT_V7HE("printCursor");
		else
			PRINT_V67("printCursor");
		break;
	case 0xB6:
		if (g_options.heVersion)
			PRINT_V7HE("printDebug");
		else
			PRINT_V67("printDebug");
		break;
	case 0xB7:
		if (g_options.heVersion)
			PRINT_V7HE("printSystem");
		else
			PRINT_V67("printSystem");
		break;
	case 0xB8:
		// This is *almost* identical to the other print opcodes, only the 'begin' subop differs
		if (g_options.heVersion) {
			ext(output, "x" "printActor\0"
					"\x41pp|XY,"
					"\x42p|color,"
					"\x43p|right,"
					"\x45|center,"
					"\x47|left,"
					"\x48|overhead,"
					"\x4A|mumble,"
					"\x4Bs|msg,"
					"\xF9l|colors,"
					"\xC2lps|debug,"
					"\xE1p|getText,"
					"\xFEp|begin,"
					"\xFF|end");
		} else {
			ext(output, "x" "printActor\0"
					"\x41pp|XY,"
					"\x42p|color,"
					"\x43p|right,"
					"\x45|center,"
					"\x47|left,"
					"\x48|overhead,"
					"\x4A|mumble,"
					"\x4Bs|msg,"
					"\xFEp|begin,"
					"\xFF|end");
		}
		break;
	case 0xB9:
		if (g_options.heVersion)
			PRINT_V7HE("printEgo");
		else
			PRINT_V67("printEgo");
		break;
	case 0xBA:
		ext(output, "ps|talkActor");
		break;
	case 0xBB:
		ext(output, "s|talkEgo");
		break;
	case 0xBC:
		ext(output, "x" "dimArray\0"
				"\xC7pv|int,"
				"\xC8pv|bit,"
				"\xC9pv|nibble,"
				"\xCApv|byte,"
				"\xCBpv|string,"
				"\xCCv|nukeArray");
		break;
	case 0xBD:
		if (g_options.heVersion)
			ext(output, "|stopObjectCode");
		else
			invalidop(NULL, code);
		break;
	case 0xBE:
		// TODO: this loads another script which does something like
		// stack altering and then finishes (usually with opcode 0xBD).
		// When stack is changed, further disassembly is wrong.
		// This is widely used in HE games.
		// As there are cases when called script does not alter the
		// stack, it's not correct to use "rlpp|..." here
		ext(output, "lpp|startObjectQuick");
		break;
	case 0xBF:
		ext(output, "lp|startScriptQuick2");
		break;
	case 0xC0:
		ext(output, "x" "dim2dimArray\0"
				"\xC7ppv|int,"
				"\xC8ppv|bit,"
				"\xC9ppv|nibble,"
				"\xCAppv|byte,"
				"\xCBppv|string");
		break;
	case 0xC1:
		ext(output, "ps|trace");
		break;
	case 0xC4:
		ext(output, "rp|abs");
		break;
	case 0xC5:
		ext(output, "rpp|getDistObjObj");
		break;
	case 0xC6:
		ext(output, "rppp|getDistObjPt");
		break;
	case 0xC7:
		ext(output, "rpppp|getDistPtPt");
		break;
	case 0xC8:
		if (g_options.heVersion)
				ext(output, "ry" "kernelGetFunctions\0"
					"\x1|virtScreenSave"
					);
		else if (g_options.scriptVersion == 7)
				ext(output, "ry" "kernelGetFunctions\0"
					"\x73|getWalkBoxAt,"
					"\x74|isPointInBox,"
					"\xCE|getRGBSlot,"
					"\xCF|getObjectXPos,"
					"\xD0|getObjectYPos,"
					"\xD1|getObjectWidth,"
					"\xD2|getObjectHeight,"
					"\xD3|getKeyState,"
					"\xD4|getActorFrame,"
					"\xD5|getVerbXPos,"
					"\xD6|getVerbYPos,"
					"\xD7|getBoxFlags"
					);
		else
				ext(output, "ry" "kernelGetFunctions\0"
					"\x71|getPixel"
					);
		break;
	case 0xC9:
		if (g_options.heVersion)
				ext(output, "y" "kernelSetFunctions\0"
					"\x1|virtScreenLoad"
					);
		else if (g_options.scriptVersion == 7)
			ext(output, "y" "kernelSetFunctions\0"
					"\x4|grabCursor,"
					"\x6|startVideo,"
					"\xC|setCursorImg,"
					"\xD|remapCostume,"
					"\xE|remapCostumeInsert,"
					"\xF|setVideoFrameRate,"

					"\x10|enqueueTextCentered,"
					"\x11|enqueueTextNormal,"
					"\x12|setMouseXY,"
					"\x14|setRadioChatter,"

					"\x6B|setActorScale,"
					"\x6C|buildPaletteShadow,"
					"\x6D|setPaletteShadow,"

					"\x72|unk114,"
					"\x75|freezeScripts,"
					"\x76|blastShadowObject,"
					"\x77|superBlastObject,"
					"\x7C|setSaveSound,"
					"\xD7|setSubtitles,"
					);
		else
			ext(output, "y" "kernelSetFunctions\0"
					"\x3|dummy,"
					"\x4|grabCursor,"
					"\x5|fadeOut,"
					"\x6|redrawScreen,"
					"\x8|startManiac,"
					"\x9|killAllScriptsExceptCurrent,"
					"\x68|nukeFlObjects,"
					"\x6B|setActorScale,"
					"\x6C|setupShadowPalette,"
					"\x6D|setupShadowPalette,"
					"\x6E|clearCharsetMask,"
					"\x6F|setActorShadowMode,"
					"\x70|shiftShadowPalette,"
					"\x72|noirMode,"
					"\x75|freezeScripts,"
					"\x77|superBlastObject,"
					"\x78|swapPalColors,"
					"\x7A|setSoundResult,"
					"\x7B|copyPalColor,"
					"\x7C|setSaveSound,"
					);
		break;
	case 0xCA:
		ext(output, "p|delayFrames");
		break;
	case 0xCB:
		ext(output, "rlp|pickOneOf");
		break;
	case 0xCC:
		ext(output, "rplp|pickOneOfDefault");
		break;
	case 0xCD:
		ext(output, "pppp|stampObject");
		break;
	case 0xD0:
		ext(output, "|getDateTime");
		break;
	case 0xD1:
		ext(output, "|stopTalking");
		break;
	case 0xD2:
		ext(output, "rpp|getAnimateVariable");
		break;
	case 0xD4:
		ext(output, "vpp|shuffle");
		break;
	case 0xD5:
		ext(output, "lpp|jumpToScript");
		break;
	case 0xD6:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBand, se_a));
		break;
	case 0xD7:
		se_a = pop();
		se_b = pop();
		push(se_oper(se_b, operBor, se_a));
		break;
	case 0xD8:
		ext(output, "rp|isRoomScriptRunning");
		break;
	case 0xD9:
		if (g_options.heVersion)
			ext(output, "p|closeFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDA:
		if (g_options.heVersion)
			ext(output, "rsp|openFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDB:
		if (g_options.heVersion)
			ext(output, "rpp|readFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDC:
		if (g_options.heVersion)

			ext(output, "ppp|writeFile");
		else
			invalidop(NULL, code);
		break;

	case 0xDD:
		ext(output, "rp|findAllObjects");
		break;
	case 0xDE:
		if (g_options.heVersion)
			ext(output, "s|deleteFile");
		else
			invalidop(NULL, code);
		break;
	case 0xDF:
		if (g_options.heVersion)
			ext(output, "ss|renameFile");
		else
			invalidop(NULL, code);
		break;
	case 0xE0:
		if (g_options.heVersion)
			ext(output, "x" "soundOps\0"
				"\xDEp|setMusicVolume,"
				"\xDF|dummy,"
				"\xE0p|setSoundFrequency");
		else
			invalidop(NULL, code);
		break;
	case 0xE1:
		ext(output, "rpp|getPixel");
		break;
	case 0xE2:
		if (g_options.heVersion)
			ext(output, "p|localizeArrayToScript");
		else
			invalidop(NULL, code);
		break;
	case 0xE3:
		ext(output, "rlw|pickVarRandom");
		break;
	case 0xE4:
		ext(output, "p|setBotSet");
		break;
	case 0xE9:
		if (g_options.heVersion)
			ext(output, "ppp|seekFilePos");
		else
			invalidop(NULL, code);
		break;
	case 0xEA:
		if (g_options.heVersion)
			ext(output, "x" "redimArray\0"
					"\xC7ppv|int,"
					"\xCAppv|byte");
		else
			invalidop(NULL, code);
		break;
	case 0xEB:
		if (g_options.heVersion)
			ext(output, "rp|readFilePos");
		else
			invalidop(NULL, code);
		break;
	case 0xEC:
		if (g_options.heVersion)
			invalidop(NULL, code);
		else
			ext(output, "rp|getActorLayer");
		break;
	case 0xED:
		if (g_options.heVersion)
			ext(output, "rppp|getStringWidth");
		else
			ext(output, "rp|getObjectNewDir");
		break;
	case 0xEE:
		if (g_options.heVersion)
			ext(output, "rp|getStringLen");
		else
			invalidop(NULL, code);
		break;
	case 0xEF:
		if (g_options.heVersion)
			ext(output, "rppp|appendString");
		else
			invalidop(NULL, code);
		break;
	case 0xF1:
		if (g_options.heVersion)
			ext(output, "rpp|compareString");
		else
			invalidop(NULL, code);
		break;
	case 0xF2:
		if (g_options.heVersion) {
			ext(output, "rx" "isResourceLoaded\0"
					"\x12p|image,"
					"\xE2p|room,"
					"\xE3p|costume,"
					"\xE4p|sound,"
					"\xE5p|script");
		} else {
			invalidop(NULL, code);
		}
		break;
	case 0xF3:
		if (g_options.heVersion) {
			ext(output, "ru" "readINI\0"
					"\x01s|number,"
					"\x02s|string");
		} else {
			invalidop(NULL, code);
		}
		break;
	case 0xF4:
		if (g_options.heVersion) {
			ext(output, "u" "writeINI\0"
					"\x01ps|number,"
					"\x02pss|string");
		} else {
			invalidop(NULL, code);
		}
		break;
	case 0xF5:
		if (g_options.heVersion)
			ext(output, "rppp|getStringLenForWidth");
		else
			invalidop(NULL, code);
		break;
	case 0xF6:
		if (g_options.heVersion)
			ext(output, "rpppp|getCharIndexInString");
		else
			invalidop(NULL, code);
		break;
	case 0xF7:
		if (g_options.heVersion)
			ext(output, "rpp|findBox");
		else
			invalidop(NULL, code);
		break;
	case 0xF9:
		if (g_options.heVersion)
			ext(output, "s|createDirectory");
		else
			invalidop(NULL, code);
		break;
	case 0xFA:
		if (g_options.heVersion) {
			ext(output, "x" "setSystemMessage\0"
					"\xF0s|unk1,"
					"\xF1s|versionMsg,"
					"\xF2s|unk3,"
					"\xF3s|titleMsg");
		} else
			invalidop(NULL, code);
		break;
	case 0xFB:
		if (g_options.heVersion)
			ext(output, "x" "polygonOps\0"
					"\xF6ppppppppp|polygonStore,"
					"\xF7pp|polygonErase,"
					"\xF8ppppppppp|polygonStore,"
					);
		else
			invalidop(NULL, code);
		break;
	case 0xFC:
		if (g_options.heVersion)
			ext(output, "rpp|polygonHit");
		else
			invalidop(NULL, code);
		break;
	default:
		invalidop(NULL, code);
		break;
	}
}
