/*
 ----------------------------------------------------------------------------
 "THE BEER-WARE LICENSE" (Revision 42):
 <daniel.kratzert@ac.uni-freiburg.de> wrote this file. As long as you retain
 this notice you can do whatever you want with this stuff. If we meet some day,
 and you think this stuff is worth it, you can buy me a beer in return.
 Daniel Kratzert
 ----------------------------------------------------------------------------
*/


#include <QtGui>
#include <QDir>
#include "dsrgui.h"
#include "window.h"
#include "dsrglwindow.h"
#include "dsreditwindow.h"
#include "itsme.h"

#ifdef Q_WS_WIN32
   QString mysystem = "win";
#else
   QString mysystem = "unix";
#endif

/*
Explanations

- This GUI is for DSR https://www.xs3.uni-freiburg.de/research/dsr
- DSR stores the information for pre-defined molecular fragments and their
  restraints in a text database in $DSR_DB_DIR.
- This GUI uses that information to provide a convenient interface for DSR.
- DSR is controlled by a special command in the SHELXL res file starting with REM DSR ...
  and by commandline options.
- A special not documented command line option is -ah. It tells DSR to print the
  information about a given fragment name in a special format for this GUI.
  It is a list of html-like tags <tag> data </tag> with ;; separated values containing for
  example the unit cell information and coordinates of the fragment. The first line
  contains the version number of DSR.
- The GUI first runs dsr with -lc to get a list of all fragment names in the database.
  A mouse click on a fragment in the list invokes "dsr -ah name" to get the details of
  this special fragment (struct DSRMol). This details are also used to edit the fragment
  (edit fragment button).
- The text widget below the search field displays some status output from DSR,
  especially the text output from DSR during the fragment transfer/fit to the target structure.

- merging: go into shelxle-kratzert and svn merge -r 165:HEAD /Users/daniel/Downloads/shelxle-trunk
       "-r (latest own commit):HEAD"    svn merge -r 165:HEAD d:\downloads\shelxle-svn\trunk
*/


/*
 * TODO:
*/


DSRGui::DSRGui(Molecule *mole, QString shelxlPath, Window *parent): QWidget(parent) {
  m_shelxle = parent;
  m_molecule = mole;
  m_shelxlPath = shelxlPath;
  setWindowFlags(Qt::Window);
  setWindowTitle(QString(tr("DSR GUI")));
  resFileName = m_shelxle->dirName;
  if (mysystem == QString("win")){
    dsrpath = getDSRDir()+"/dsr.bat";
    dsr_db_path = getDSRdbDir();
  } else if (mysystem == QString("unix")){
    dsrpath = QDir::cleanPath(getDSRDir()+"/dsr");
    dsr_db_path = QDir::cleanPath(getDSRdbDir());
  }
  dsrVersion.clear();
  replace = false;
  norefine = false; // this has to stay here
  runext = false;
  invert = false;
  rigid = false;
  fragmentsList = new QVector<QStringList>;
  fragmentNameTag.clear();
  header = new DSRMol;
  mainVLayout = new QVBoxLayout(this);
  editLayout = new QHBoxLayout;
  chooserLayout = new QGridLayout;
  searchLayout = new QHBoxLayout;
  partLayout = new QHBoxLayout;
  occLayout = new QHBoxLayout;
  resclassLayout = new QHBoxLayout;
  optionsLayout1 = new QVBoxLayout;
  optionsLayout2 = new QVBoxLayout;
  optionsLayout3 = new QVBoxLayout;
  groupBox1 = new QGroupBox();
  groupBox2 = new QGroupBox();
  residueOptionsBox = new QGroupBox(tr("Use a Residue"));
  buttonLayout = new QVBoxLayout;
  optionboxes = new QHBoxLayout;
  fitDSRButton = new QPushButton(tr("Fit Fragment!"));
  fitDSRButton->setStyleSheet("QPushButton {"
                              "font-weight: bold; "
                              "color: #209920}");
  exportFragButton = new QPushButton(tr("Export Fragment"));
  editButton = new QPushButton(tr("Edit Fragment"));
  editButton->setStyleSheet("QPushButton {"
                            "font-weight: bold; "
                            "color: rgb(8, 82, 182)}");
  runExtBox = new QCheckBox(tr("External Restraints"));
  invertFragBox = new QCheckBox(tr("Invert Coordinates"));
  dfixBox = new QCheckBox(tr("Calculate DFIX"));
  rigidBox = new QCheckBox(tr("Rigid Group (no restraints)"));
  replaceModeBox = new QCheckBox(tr("Replace Target"));
  searchLabel = new QLabel(tr("Search:"));
  SearchInp = new QLineEdit;//(tr("Search a fragment here"));
  partLabel = new QLabel(tr("PART:"));
  occLabel = new QLabel(tr("FVAR+Occupancy:"));
  classLabel = new QLabel(tr("Residue Class:"));
  usermanualLabel = new QLabel(tr("<a href=\"http://www.xs3-data.uni-freiburg.de/data/DSR-manual.pdf\"> DSR User Manual </a>"));
  usermanualLabel->setOpenExternalLinks(true);
  resnumbertext = new QString(tr("A residue number will be\nchosen automatically."));
  resiTextLabel = new QLabel();
  resiTextLabel->setText(*resnumbertext);
  outtext = new QTextEdit;
  info = new QTextBrowser;
  info->hide();
  info->setMinimumWidth(235);
  info->setMaximumWidth(235);
  fragmentTableView = new QTableView;
  occEdit = new QLineEdit;
  partspinner = new QSpinBox;
  resiclassEdit = new QLineEdit;
  // layout for the interactions
  optionboxes->addWidget(groupBox1);
  optionboxes->addWidget(groupBox2);
  optionboxes->addWidget(residueOptionsBox);
  optionboxes->addLayout(buttonLayout);
  buttonLayout->setSizeConstraint(QLayout::SetMaximumSize);
  // The search field:
  searchLayout->addWidget(searchLabel);
  searchLayout->addWidget(SearchInp);
  SearchInp->setFocus();  // searching should be the default
  SearchInp->setMinimumWidth(getCharWidth(9));
  fragmentNameTag.clear();
  // The OpenGL widget with the molecule:
  mygl = new DSRGlWindow(this, m_molecule, *header, fragmentNameTag);
  m_molecule->loadSettings();
  mygl->showFit = new QCheckBox(tr("show fitted target overlay"));
  mygl->showFit->setChecked(true);
  mygl->showFitLabel = new QCheckBox(tr("labels on target overlay"));
  mygl->showFitLabel->setChecked(true);
  // Table of fragments and 3D window:
  chooserLayout->addWidget(fragmentTableView,  0, 0, 1, 1);
  chooserLayout->addWidget(mygl,               0, 1, 1, 2);
  chooserLayout->addWidget(mygl->showFit,      2, 1, 1, 1);
  chooserLayout->addWidget(mygl->showFitLabel, 2, 2, 1, 1);
  chooserLayout->addLayout(searchLayout,       2, 0, 1, 1);
  chooserLayout->setColumnStretch(0, 4);
  chooserLayout->setColumnStretch(1, 3);
  chooserLayout->setRowStretch(2, 0);
  // Three main layours:
  mainVLayout->addLayout(chooserLayout, 3);
  mainVLayout->addLayout(editLayout, 2);
  mainVLayout->addLayout(optionboxes, 0);
  prepareOuttext();
  editLayout->addWidget(outtext);
  editLayout->addWidget(info);
  outtext->setMinimumHeight(170);
  // Layouts for optionbox1
  partLayout->addStretch();
  partLayout->addWidget(partLabel);
  partLayout->addWidget(partspinner);
  partspinner->setRange(-99, 99);
  partspinner->setValue(0);
  occLayout->addWidget(occLabel);
  occLayout->addWidget(occEdit);
  occLabel->setAlignment(Qt::AlignRight);
  occEdit->setMaximumWidth(getCharWidth(8));
  occEdit->setMinimumWidth(getCharWidth(6));
  // box1
  optionsLayout1->addLayout(partLayout);
  optionsLayout1->addLayout(occLayout);
  optionsLayout1->addWidget(replaceModeBox);
  replaceModeBox->setLayoutDirection(Qt::RightToLeft);
  optionsLayout1->addStretch();
  groupBox1->setLayout(optionsLayout1);
  // box2
  optionsLayout2->addWidget(invertFragBox);
  optionsLayout2->addWidget(runExtBox);
  optionsLayout2->addWidget(dfixBox);
  optionsLayout2->addWidget(rigidBox);
  optionsLayout2->addStretch();
  groupBox2->setLayout(optionsLayout2);
  // box3
  resclassLayout->addWidget(classLabel);
  resclassLayout->addWidget(resiclassEdit);
  classLabel->setAlignment(Qt::AlignRight);
  resiclassEdit->setMaximumWidth(getCharWidth(8));
  resiclassEdit->setMinimumWidth(getCharWidth(6));
  optionsLayout3->addLayout(resclassLayout);
  optionsLayout3->addWidget(resiTextLabel);
  optionsLayout3->addWidget(usermanualLabel);
  optionsLayout3->addStretch();
  residueOptionsBox->setLayout(optionsLayout3);
  residueOptionsBox->setCheckable(true);
  // buttons:
  buttonLayout->addWidget(fitDSRButton);
  buttonLayout->addWidget(exportFragButton);
  exportFragButton->setEnabled(false);
  buttonLayout->addSpacing(10);
  buttonLayout->addWidget(editButton);
  // tooltips:
  setToolTips();
  QPixmap pix = QPixmap(250, 50);
  pix.fill(); // need to fill in order to see the text.
  QPainter painter(&pix);
  QRectF rectangle(0, 0, 250-1, 49);
  painter.setFont(QFont("Sans-Serif", 12));
  painter.drawRect(rectangle);
  painter.drawText(QPoint(12, 30), "Loading fragment list...");
  QSplashScreen *splash = new QSplashScreen(pix);
  splash->show();
  splash->showMessage("");
  target_atoms = getSelectedAtomsList(); // must be before signals
  // request version.txt to warn for updates:
  getVersionFromServer();
  // Signal slot connections:
  connect_signals_and_slots();
  //this->setMinimumSize(815, 615);
  //this->setMaximumSize(950, 999);
  this->move(220, 20);
  checkForDSRexecutable(splash);
  this->show();
  splash->finish(this);
  // call last, because keyboard focus is reated to tab order,
  // which is based on the order the widgets are created:
  SearchInp->setFocus();
  // Has to be here, otherwise fragmentTableView is not initialized:
  QItemSelectionModel *sm = fragmentTableView->selectionModel();
  if (!(sm == 0x0)){  // prevents error about missing connection if DSR is not present
    connect(sm, SIGNAL(currentRowChanged(QModelIndex,QModelIndex)),
        this, SLOT(setFragName(QModelIndex)), Qt::UniqueConnection);
  }
  outtext->append(tr("\n* Please pick a fragment first."));
  outtext->append(tr("* Then select at least three atoms from "
                    "the main structure and the fragment respectively. (STRG+click)"));
  outtext->append(tr("* Do not forget to apply a part number and free "
                     "variable in case of disorder."));
  outtext->append(tr("* The fragment will be placed after FVAR in the SHELX file."));
  if (!dsrVersion.isEmpty()) {
    outtext->append(QString("\nFound DSR version %1.").arg(dsrVersion));
  }
}


DSRGui::~DSRGui() {
  delete mygl;
}


void DSRGui::connect_signals_and_slots() {
  //! Handles most of the signal slot connections
    connect(this, SIGNAL(fragmentSelected(void)),
            this, SLOT(activateFitButton(void)));
    connect(this, SIGNAL(fragmentSelected()),
            this, SLOT(resetSelection()));
    connect(fitDSRButton, SIGNAL (clicked(bool)),
            this, SLOT (fitDSR()));
    connect(fitDSRButton, SIGNAL (clicked(bool)),
            this->info, SLOT(hide()));
    connect(runExtBox, SIGNAL (clicked(bool)),
            this, SLOT (fitDSRExtern(bool)));
    connect(dfixBox, SIGNAL (clicked(bool)),
            this, SLOT (DFIX(bool)));
    connect(invertFragBox, SIGNAL (clicked(bool)),
            this, SLOT (invertFrag(bool)));
    connect(rigidBox, SIGNAL (clicked(bool)),
            this, SLOT (rigid_group(bool)));
    connect(replaceModeBox, SIGNAL(clicked(bool)),
            this, SLOT(replaceOrNot(bool)));
    connect(SearchInp, SIGNAL(textChanged(QString)),
            this, SLOT(searchFragment(QString)));
    connect(occEdit, SIGNAL(textChanged(QString)),
            this, SLOT(setFvarOcc(QString)));
    connect(resiclassEdit, SIGNAL(textChanged(QString)),
            this, SLOT(setResiClass(QString)));
    connect(residueOptionsBox, SIGNAL(toggled(bool)),
            this, SLOT(combineOptionstext()));
    connect(partspinner, SIGNAL(valueChanged(int)),
            this, SLOT(setPART(int)));
    connect(exportFragButton, SIGNAL(clicked(bool)),
            this, SLOT(setExportDirDialog(void)));
    connect(fragmentTableView, SIGNAL(clicked(QModelIndex)),
            this, SLOT(setFragName(QModelIndex)));
    connect(this, SIGNAL(optionTextChanged(void)),
            this, SLOT(combineOptionstext(void)));
    connect(editButton, SIGNAL(clicked(bool)),
            this, SLOT(runEditWindow()));
    connect(this, SIGNAL(exportDirChanged(QString)),
            this, SLOT(exportFrag(QString)));
    // sigslot for uppdating the 3D view
    connect(this, SIGNAL(currentFragmentChanged(DSRMol)),
            mygl, SLOT(display_fragment(DSRMol)));
    connect(m_shelxle->chgl, SIGNAL(selectionChanged()),
            mygl, SLOT(updateGL()));
    connect(m_shelxle->chgl, SIGNAL(selectionChanged()),
            this, SLOT(updateTarget()));
    connect(m_shelxle->chgl, SIGNAL(selectionChanged()),
            this, SLOT(update()));
    connect(mygl->showFitLabel, SIGNAL(stateChanged(int)),
            mygl, SLOT(updateGL())); 
    connect(mygl->showFit, SIGNAL(stateChanged(int)),
            mygl, SLOT(updateGL()));
    connect(m_shelxle->chgl, SIGNAL(selectionChanged()),
            mygl, SLOT(makeInfo()));
    connect(mygl, SIGNAL(sourceStringChanged()),
            this, SLOT(combineOptionstext()));
    connect(mygl, SIGNAL(updateInfo(QString)),
            this, SLOT(setInfo(QString)));
    connect(net_manager, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(isDSRUpToDate(QNetworkReply*)));
}

QSize DSRGui::minimumSizeHint() const
{
  return QSize(800, 800);
}

QSize DSRGui::sizeHint() const
{
  return QSize(800, 930);
}

void DSRGui::getVersionFromServer() {
  //! Writes the proposed version number of DSR into reply
    net_manager = new QNetworkAccessManager(this);
    QNetworkRequest request;
    request.setUrl(QUrl("http://www.xs3-data.uni-freiburg.de/data/version.txt"));
    request.setRawHeader("User-Agent", "DSRGui");
    reply = net_manager->get(request);
  }


void DSRGui::isDSRUpToDate(QNetworkReply* reply){
  /*
   *! Displays a warning if the current DSR version is too old.
  */
  QString latestRev_str=reply->readAll();
  bool ok;
  bool ok2;
  QMessageBox info;
  info.addButton(QMessageBox::Close);
  int dsrv = dsrVersion.toInt(&ok2, 10);
  QPushButton *updateButton = info.addButton(tr("Update now"), QMessageBox::ActionRole);
  if (dsrv < 193) {  // This is the first version with self-update mechanism
    updateButton->hide();
  }
  int latestrev_int=latestRev_str.toInt(&ok,10);
  if ((ok&&ok2)&&(latestrev_int)>(dsrv)){
    info.setText(QString(
                   "<h3>You should probably update DSR!</h3>"
                   "This is revision: <b>%1</b><br> "
                   "The latest version is: <b>%2</b> <br><br>"
                   "New versions can be downloaded here: <br>"
                   "<a href=\"https://www.xs3.uni-freiburg.de/research/dsr\">"
                   "https://www.xs3.uni-freiburg.de/research/dsr</a><br><br>"
                   "Please contact the author Daniel Kratzert"
                   " <a href=\"mailto:dkratzert@gmx.de\">dkratzert@gmx.de</a> "
                   "if you find any bugs.<br> Thank you!")
                    .arg(dsrVersion.toInt())
                    .arg(latestrev_int));
    info.exec();
    if (info.clickedButton() == updateButton) {
      updateDSR();
    }
  }
  reply->close();
  reply->deleteLater();
  disconnect(net_manager, SIGNAL(finished(QNetworkReply*)), 0, 0);
}


void DSRGui::updateTarget(){
  //! updates the target atoms list
  target_atoms = getSelectedAtomsList();
  combineOptionstext();
}

void DSRGui::writeFavorites(QString name) {
  //! Stores favorites in dsrgui.ini
  QSettings dsr_settings( QSettings::IniFormat, QSettings::UserScope ,PROGRAM_NAME,"dsrgui" );
  dsr_settings.beginGroup("LastFragment");
  dsr_settings.setValue("last", name);
  dsr_settings.endGroup();
}

QString DSRGui::loadFavoriteFragment() {
  //! Loads Favorites from dsrgui.ini
  QSettings dsr_settings( QSettings::IniFormat, QSettings::UserScope ,PROGRAM_NAME,"dsrgui" );
  dsr_settings.beginGroup("LastFragment");
  QString last = dsr_settings.value("last").toString();
  dsr_settings.endGroup();
  return last;
}

QStringList DSRGui::which(QString programName) {
  //! Implements a which like method
  //! It returns all paths where programName is found in
  //! the system path variable
  QStringList foundInPath;
  QStringList execlist;
  QStringList pathList;
  pathList.clear();
  foundInPath.clear();
  execlist.clear();
  if (mysystem == "win") {
    pathList = QString(qgetenv("PATH").constData()).split(";");
  } else {
    pathList = QString(qgetenv("PATH").constData()).split(":");
  }
  execlist << ".bat" << ".exe" << "";
  foreach (QString exec, execlist) {
    foreach (QString path, pathList) {
      QString fullpath = path+"/"+programName+exec;
      QFileInfo fi;
      fi = fullpath;
      if (QFile::exists(fullpath) && (fi.isExecutable())){
        foundInPath.append(QDir::cleanPath(fullpath));
      }
    }
  }
  return foundInPath;
}

void DSRGui::checkForDSRexecutable(QSplashScreen *splash){
  //! list fragments only if the path of dsr installation is found:
  if (QFile::exists(dsrpath)) {
    listDSRdbFragments("None");
  } else {
    outtext->clear();
    outtext->append(QString(tr("Unable to find DSR executable."
                "\nIs DSR_DIR environment variable set correctly?")));
    outtext->append(QString(tr("Please find the most recent version of DSR at "
                               "<a href=\"https://www.xs3.uni-freiburg.de/research/dsr\">"
                               "https://www.xs3.uni-freiburg.de/research/dsr</a>")));
    splash->finish(this);
  }
}

void DSRGui::setToolTips() {
  partspinner->setToolTip(tr("The PART of a fragment controls the binding of atoms.\n"
                             "For example two fragments in PART 1 and PART 2 do not\n"
                             "bind each other but to PART 0. Negative PARTs only bind themselves."));
  replaceModeBox->setToolTip(tr("Toggle replacement of target atoms and all atoms of "
                             "PART 0 \nin 1.3A distance around the fitted fragment atoms."));
  occEdit->setToolTip(tr("Free variable and occupancy of the fragment "
                         "\ncombined according SHELXL syntax.\n"
                         "For example, 21 means second free variable with occupation of 1."));
  invertFragBox->setToolTip(tr("Invert the fragment coordinates during fit."));
  runExtBox->setToolTip(tr("Write restraints to external file."));
  dfixBox->setToolTip(tr("Calculate DFIX/DANG/FLAT restraints "
                         "\naccording to fragment geometry. \n"
                         "Database restraints will be ignored."));
  rigidBox->setToolTip(tr("Keep the fragment as rigid (AFIX 9) group. \n"
                          "Apply no restraints."));
  residueOptionsBox->setToolTip(tr("Enables residues. Usually, you can leave the default.\n"
                           "It will always take the next free residue number."));
  fitDSRButton->setToolTip("Run DSR to fit the fragment into the structure.");
  exportFragButton->setToolTip("Export the current fragment to a .res file.");
  editButton->setToolTip("Edit the current fragment or create a new one.");
}

QString DSRGui::textWrap(QString inText, int length = 75) {
  //! returns a SHELXL compatible wrapped string (string =\n string) in case of over 77 characters
  //! line length
  QString firstLine = "";
  QString secondline = "";
  if (inText.length() > length) {
    foreach (QString chunk, inText.split(" ")) {
      if (firstLine.length()+(chunk+QString(" ")).length() < length) {
        firstLine.append(chunk+QString(" "));
      } else {
        secondline.append(chunk+QString(" "));
      }
    }
    return firstLine+QString("=\n")+QString("   ")+secondline;
  } else {
    // nothing to wrap
    return inText;
  }
}

void DSRGui::resetSelection() {
  //! reset the selection of source atoms
  mygl->source_atoms.clear();
  combineOptionstext();
  info->hide();
}

void DSRGui::setInfo(QString s){
  //! shows an info table with bond distances
  if (mygl->selFragAt.size()<1) {
    info->hide();
  }
  else {
    info->show();
    info->setText(s);
  }
}

void DSRGui::runEditWindow(void){
  //! Starts the edit window in order to edit a fragment
  editwindow = new DSREditWindow(m_molecule, header, dsr_db_path,
                                 fragmentNameTag, *fragmentsList, this);
  editwindow->move(60, 50);
  editwindow->show();
  editwindow->setFocus();
  editwindow->setMinimumWidth(800);
  outtext->clear();
  connect(editwindow, SIGNAL(updated(QString)),
          this, SLOT(listDSRdbFragments(QString)));
}

void DSRGui::closeEvent(QCloseEvent *event)
//! close event is emmited during DSRGui closing to reset its pointer
{
  emit closed();
  (void)event; // prevent compiler warning
}

int DSRGui::findFVARlines(QStringList *reslist) {
  //! finds the line number of last FVAR or the first atom
  //! I restrict the use of this plugin to stuctures with a valid
  //! FVAR, because a missing FVAR makes it all too error prone.
  int fvarline;
  fvarline = reslist->lastIndexOf(QRegExp("^FVAR.*", Qt::CaseInsensitive));
  if (fvarline > 0) {
    return fvarline;
  } else {
    return 0;
  }
}

QVector<int> DSRGui::findDSRLines(QStringList *reslist) {
  //! Find line with "rem DSR ..." in res file and return line number
  //! if it exists.
  QVector<int> lineNums;
  int Num = 0;
  foreach (QString line, *reslist) {
    if (line.contains(QRegExp("^rem\\s{1,6}DSR\\s{1,6}.*", Qt::CaseInsensitive))) {
      lineNums.append(Num);
    }
    Num++;
  }
  return lineNums;
}

int DSRGui::decideDSRInsertLine(QStringList *reslist) {
  //! decides where to instert the DSR command
  int fvarline = findFVARlines(reslist);
  if (fvarline > 0) {
    return fvarline;
  } else {
    return 0;
  }
}

QStringList DSRGui::readResfile() {
  //! read the entire res file into a stringlist
  QFileInfo fi(resFileName);
  QFile file(fi.absoluteFilePath());
  QStringList stringList;
  if (file.open(QFile::ReadOnly | QFile::Text)) {
    QTextStream textStream(&file);
    while (true) {
      QString line = textStream.readLine();
      if (line.isNull()) {
          break;
      } else {
        stringList.append(line);
      }
    }
  }
  file.close();
  return stringList;
}

QString DSRGui::findFreeResiNumber() {
  //! returns the next free residue number in the structure
  QSet<int> resiset;
  QList<int> resilist;
  int resnum = 0;
  for (int i=0; i < m_molecule->asymm.size(); i++) {
    MyAtom atom;
    atom = m_molecule->asymm.at(i);
    if (atom.resiNr >= 0) {
      resiset.insert(atom.resiNr);
    }
  }
  resilist = resiset.toList();
  qSort(resilist);
  int count = 0;
  foreach (int num, resilist) {
    if (num != count) {
      // a gap in residue numbers is found, use this number:
      resnum = count;
      break;
    }
    count++;
  }
  // no gap found, use next number:
  resnum = count;
  // In this case we have no residue in the file:
  if (resilist.isEmpty()) {
    resnum = 1;
  }
  return QString("%1").arg(resnum);
}


bool DSRGui::getFragmentHeader(QString frag) {
  //! defines Name, residue, comment and unit cell of the fragment
  //! in DSRMol
  /*!
    <tag>
     benze
    </tag>
    <comment>
     Benzene2, Benzol, C6H6
    </comment>
    <source>
     CCDC UGEDEQ
    </source>
    <cell>
     1;;1;;1;;90;;90;;90
    </cell>
    <residue>
     ERT
    </residue>
    <dbtype>
     dsr_user_db
    </dbtype>
    <restr>
     RIGU C1 > C6;;SADI C1 C2 C3 C4
     </restr>
    <atoms>
      C1 6 1.78099 7.14907 12.00423;;C2 6 2.20089 8.30676 11.13758;;C3 6 1.26895 9.02168 10.39032;;C4 6 1.64225 10.07768 9.58845;;C5 6 2.98081 10.44432 9.51725;;C6 6 3.92045 9.74974 10.25408
    </atoms>
   */
  header->atoms.clear();
  header->cell.clear();
  header->comment.clear();
  header->dbtype.clear();
  header->residue.clear();
  header->restr.clear();
  header->tag.clear();
  QString *rawheader = new QString;
  QStringList *headerlist = new QStringList;
  QProcess dsr;
  dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.start(dsrpath, QStringList() << "-ah" << frag);
  dsr.closeWriteChannel();
  if (!dsr.waitForFinished(5000))
  {
    outtext->clear();
    outtext->append(tr("Unable to run DSR."));
    editButton->setDisabled(true); // editing the empty header would crash ShelXle
    return false;
  } else {
    rawheader->append(dsr.readAll());
  }
  // in this case no fragment list returned. Hence, we have an error.
  if (!rawheader->contains(";;")) {
    outtext->clear();
    outtext->append(*rawheader);
    //editButton->setDisabled(true); // editing the empty header would crash ShelXle
    return false;
  }
  headerlist->append(rawheader->split(QRegExp("\n|\r\n|\r")));
  QStringList line;
  for (int i=0; i<headerlist->size(); i++){
    if (headerlist->at(i).size() == 0){
      continue;
    }
    line.clear();
    if (headerlist->at(i).trimmed().startsWith("<tag>")) {
      line = headerlist->at(i+1).trimmed().split(" ");
      header->tag = line.join("");
    }
    if (headerlist->at(i).trimmed().startsWith("<comment>")) {
      line = headerlist->at(i+1).trimmed().split(" ");
      header->comment = line.join(" ");
    }
    if (headerlist->at(i).trimmed().startsWith("<source>")) {
      line = headerlist->at(i+1).trimmed().split(" ");
      header->source = line.join(" ");
    }
    if (headerlist->at(i).trimmed().startsWith("<cell>")) {
      line = headerlist->at(i+1).simplified().split(";;");
      foreach (QString item, line) {
        header->cell.append(item.toDouble());
      }
    }
    if (headerlist->at(i).trimmed().startsWith("<residue>")) {
      line = headerlist->at(i+1).trimmed().split(" ");
      header->residue = line.join("");
      resiclass = header->residue;
      resiclassEdit->setText(resiclass);
    }
    if (headerlist->at(i).trimmed().startsWith("<dbtype>")) {
      line = headerlist->at(i+1).trimmed().split(" ");
      header->dbtype = line.join("");
    }
    if (headerlist->at(i).trimmed().startsWith("<restr>")) {
      line = headerlist->at(i+1).simplified().split(";;");
      foreach (QString item, line) {
        header->restr.append(item);
      }
    }
    if (headerlist->at(i).trimmed().contains("<atoms>")) {
      line = headerlist->at(i+1).simplified().split(";;");
      foreach (QString item, line) {
        header->atoms.append(item.split(" "));
      }
    }
    if (headerlist->at(i).trimmed().startsWith(tr("Suspicious deviation in atom"))) {
      outtext->append(headerlist->at(i-1));
      outtext->append(headerlist->at(i));
      outtext->append(headerlist->at(i+1));
      outtext->append(tr("Please check this database entry!"));
      return true;
    }
  }
  editButton->setEnabled(true);
  return true;
}


void DSRGui::prepareOuttext() {
//! prepares the outtext field in order to display infos/warnings
  outtext->setReadOnly(true);
  QFont font("Courier");
  font.setStyleHint(QFont::TypeWriter);
  outtext->setFont(font);
  outtext->setMinimumWidth(getCharWidth(70));
  outtext->setMinimumHeight(55);
}

int DSRGui::getCharWidth(int numchars) {
  //! returns the width in pixel of numchars times the # character
  QString buchstaben;
  buchstaben.clear();
  for (int i=1; i<=numchars; i++) {
    buchstaben += "#";
  }
  return QFontMetrics(this->font()).width(buchstaben);
}


void DSRGui::activateFitButton(void){
    fitDSRButton->setEnabled(true);
    exportFragButton->setEnabled(true);
}


void DSRGui::combineOptionstext(void)
//! combines all options to a single DSR command line
//! This method gets invoked if optionTextChanged signal is emmitted
{
  outtext->clear();
  //target_atoms = getSelectedAtomsList();
  QString putreplace = QString("PUT ");
  if (replace) {
      putreplace = QString("REPLACE ");
  }
  if (target_atoms.split(" ").length() != 3) {
    target_atoms = "<font color=red> Please select 3 target atoms/q-peaks! </font>";
  }
  if (mygl->source_atoms.split(" ").length() != 3){
    mygl->source_atoms = "<font color=red> Please select 3 fragment atoms! </font>";
  }
  QString resistr = "RESI ";
  if (!residueOptionsBox->isChecked()) {
    resistr = "";
    resiTextLabel->setText("");
  } else {
    resiTextLabel->setText(*resnumbertext);
    if (resiclassEdit->text() == resiclass) {
      resistr = "RESI "+resiclass;
    } else {
      resistr = "RESI "+resiclassEdit->text();
    }
  }
  QString outstring = QString(QString("REM DSR ")+putreplace+fragmentNameTag+" "+
                          QString(" WITH ")+mygl->source_atoms+" "+QString("ON ")+
                          target_atoms+" "+part+" "+fvarocc+" "+dfix+" "+resistr);
  combiDSRline = outstring.simplified().toUpper();
  outtext->append(combiDSRline);
}


void DSRGui::setResiClass(QString rclass)
//! defines the residue class
{
  if (rclass[0].isLetter()){
    if (rclass.length() > 4) {
      resiclassEdit->setText(QString(rclass.mid(0, 4)));
    }
    emit optionTextChanged();
  } else {
    outtext->append(QString(tr("Please start residue "
                            "class with a letter.")));
    resiclassEdit->clear();
  }
}

void DSRGui::setFragName(QModelIndex name)
//! set the fragment name variable
{
  outtext->clear();
  fragmentNameTag = name.sibling(name.row(), 0).data().toString();
  writeFavorites(fragmentNameTag);
  outtext->clear();
  if (!getFragmentHeader(fragmentNameTag)) {
    // In this case, the most important thing is missing
    return;
  }
  if (header->comment.isEmpty()){
    outtext->append("You should give this fragment a name.");
  }
  if (header->atoms.isEmpty()){
    return;
  }
  if (header->tag.isEmpty()){
    return;
  }
  if (header->restr.isEmpty()){
    return;
  }
  if (header->cell.isEmpty()){
    return;
  }
  emit optionTextChanged();
  emit fragmentSelected();
  emit currentFragmentChanged(*header);
}


void DSRGui::DFIX(bool checked)
//! toggles the dfix option
{
  outtext->clear();
  if (checked)
  {
    this->dfix = QString("DFIX");
  } else
  {
    this->dfix.clear();
  }
  emit optionTextChanged();
}

void DSRGui::setFvarOcc(QString focc)
//! defines the PART option
{
  outtext->clear();
  bool ok;
  focc.toFloat(&ok);
  if (ok)
  {
    fvarocc = QString("OCC ")+QString::number(focc.toFloat());
  } else if (!ok and focc.length() >= 1) {
    outtext->append(QString("Please provide a "
                              "real number for FVAR/OCC"));
  } else if (!ok and focc.length() == 0) {
    fvarocc.clear();
  }
  emit optionTextChanged();
}

void DSRGui::setPART(int partnum)
//! defines the PART option
{
  outtext->clear();
  if (partnum == 0)
  {
    part.clear();
  }
  else
  {
    QString s = QString::number(partnum);
    part = QString("PART ")+s ;
  }
  emit optionTextChanged();
}

void DSRGui::fitDSRExtern(bool checked)
//! enable write restraints to external file
{
  if (checked) {
    this->runext = true;
  } else {
    this->runext = false;
  }
}

void DSRGui::invertFrag(bool checked)
//! Inverts the fragment coordinates.
//! They are also inverted in the 3D view.
{
  // Invert coordinates in DSR:
  if (checked){
    this->invert = true;
  } else {
    this->invert = false;
  }
  // Invert the OpenGL coordinates:
  float coord;
  int index;
  index = 0;
  foreach(QStringList line, header->atoms) {
    for (int n = 2; n<5; n++) {
      coord = line[n].toFloat();
      header->atoms[index][n] = QString("%1").arg(-coord);
    }
    index = index+1;
  }
  mygl->display_fragment(*header);
}

void DSRGui::rigid_group(bool checked)
//! toggle rigid group refinenement
{
  if (checked) {
    this->rigid = true;
  } else {
    this->rigid = false;
  }
}

bool DSRGui::exportFrag(QString dirname) {
  //! export the current fragment to a res file+png
  //! change to the current dir here, because dsr exports in current dir:
  QDir::setCurrent(dirname);
  outtext->clear();
  if (fragmentNameTag.isEmpty()) {
      outtext->append("No fragment chosen. Doing nothing!");
      return false;
  }
  QProcess dsr;
  dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.start(dsrpath, QStringList() << QString("-e") << fragmentNameTag);
  dsr.closeWriteChannel();
  outtext->clear();
  if (!dsr.waitForFinished()) {
    outtext->append("Unable to start DSR.");
    outtext->append(dsr.readAll());
    return false;
  } else {
    outtext->append(dsr.readAll());
  }
  return true;
}

void DSRGui::refineOrNot(bool checked)
//! enable or disable refinement after transfer
{
  if (checked) {
    this->norefine = true;
  } else {
    this->norefine = false;
  }
}

void DSRGui::replaceOrNot(bool checked)
//! enable or disable replace mode
{
  outtext->clear();
  if (checked) {
    this->replace = true;
  } else {
    this->replace = false;
  }
  emit optionTextChanged();
}

QString DSRGui::getSelectedAtomsList() {
  //! returns the currently selected atoms of the structure
  //! loaded in ShelXle
  QStringList atoms;
  atoms.clear();
  for (int i=0; i<m_molecule->selectedatoms.size(); i++) {
    atoms.append(m_molecule->selectedatoms.at(i).Label);
  }
  return atoms.join(" ");
}

bool DSRGui::fitDSR() {
  /*!
   runs DSR from command line

   bool QDir::setCurrent(const QString & path)
   QString QFileInfo::completeBaseName() const : c:\programme\dsr\p21c.res -> p21c.res
   QString QFileInfo::absolutePath() const : c:\programme\dsr\p21c.res -> c:\programme\dsr\
   QString QFileInfo::absoluteFilePath() const : c:\programme\dsr\p21c.res -> c:\programme\dsr\p21c.res
   QFileInfo fi("c:/temp/foo"); => fi.absoluteFilePath()
  */
  if (m_molecule->selectedatoms.size() != 3) {
    emit optionTextChanged();
    return false;
  }
  if (mygl->source_atoms.split(" ").length() != 3) {
    emit optionTextChanged();
    return false;
  }
  QString option;
  QString target_atoms_tmp;
  QString source_atoms_tmp = mygl->source_atoms;
  if (m_molecule->selectedatoms.size() < 3) {
    outtext->clear();
    outtext->append("<font color=red> Please select at least three "
                    "atoms or Q-peaks as target in your structure! </font>");
    return false;
  }
  target_atoms = getSelectedAtomsList();
  target_atoms_tmp = target_atoms;
  emit optionTextChanged();
  bool save_sucess = m_shelxle->fileSave();
  if (!save_sucess) {
    outtext->append("Could not save the instruction file!");
    return false;
  }
  target_atoms = target_atoms_tmp;
  QStringList *reslist = new QStringList;
  reslist->append(readResfile());
  mygl->source_atoms = source_atoms_tmp;
  int DSRposition = decideDSRInsertLine(reslist);
  if (DSRposition == 0){
    outtext->append("Could not fit fragment. No FVAR command found. \nPlease refine at least one cycle.");
    return false;
  }
  QVector<int> previousLines = findDSRLines(reslist);
  QString wrappedDSRLine = textWrap(combiDSRline, 75);
  m_shelxle->insertDSRLine(wrappedDSRLine, DSRposition, previousLines);
  dsrResulttext.clear();
  if (rigid) {
    option = "-g ";
  } else {
    option = "";
  }
  if (norefine){
    option += " -n ";
  }
  if (!runext and !invert) // the standard run without extra options
  {
    option += " -r ";
  }
  else if (runext and !invert) {
    option += " -re ";
  }
  else if (!runext and invert)
  {
    option += " -t -r ";
  }
  else if (runext and invert) {
    option += " -t -re ";
  } else {
    qDebug() << "Unhandeled option case in DSRGui occoured!!";
    return false;
  }
  if (!m_shelxlPath.isEmpty() && (dsrVersion.toInt() > 182)) {
    option = " -shx " + m_shelxlPath + " " + option;
  }
  QFileInfo resfip(resFileName);
  QFileInfo check_res(resfip.completeBaseName()+".res");
  if ( resfip.completeSuffix() == "ins" && check_res.size() == 0) {
    outtext->clear();
    outtext->append("Warning! Something went wrong with your .res file.\n"
                    "You are currently working on the .ins file. Please restore "
                    "the .res file before you continue.");
    return false;
  }
  QProcess dsr;
  dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.closeWriteChannel();
  dsr.setWorkingDirectory(resfip.absolutePath());
  dsr.start(dsrpath, option.split(" ", QString::SkipEmptyParts)
                                      << resfip.completeBaseName());
  if (!dsr.waitForFinished())
  {
    dsrResulttext.append("Unable to start DSR.");
    dsrResulttext.append(dsrpath);
    dsrResulttext.append(dsr.readAll());
    return false;
  } else {
    dsrResulttext.append(dsr.readAll());
  }
  // Display the results:
  outtext->clear();
  outtext->append(combiDSRline);
  outtext->append(dsrResulttext);
  m_shelxle->loadAFile();
  return true;
}

void DSRGui::updateDSR() {
  //!
  //! Updates DSR to the current version on the web server
  //!
  QProcess dsr;
  dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.closeWriteChannel();
  QString option = " -u";
  dsr.start(dsrpath, option.split(" ", QString::SkipEmptyParts));
  if (!dsr.waitForFinished())
  {
    dsrResulttext.append("Unable to start DSR.");
    dsrResulttext.append(dsrpath);
    dsrResulttext.append(dsr.readAll());
  } else {
    dsrResulttext.append(dsr.readAll());
  }
  // Display the results:
  outtext->clear();
  outtext->append(dsrResulttext);
}

void DSRGui::searchFragment(QString searchName) {
  //! Searches for fragments in the database
  QString *result = new QString;
  if ((searchName.length() < 3) && (searchName.length() > 0)){
    return;
  }
  // If search length is zero, restore full list:
  if (searchName.length() == 0){
    if (!listDSRdbFragments("None")) {
      outtext->append("Unable to find DSR fragment database.");
      return;
    }
    return;
  }
  QProcess dsr;
  //dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.start(dsrpath, QStringList() << QString("-x")+searchName);
  dsr.closeWriteChannel();
  outtext->clear();
  if (!dsr.waitForFinished()) {
    outtext->append("Unable to start DSR.");
    outtext->append(dsr.readAll());
    outtext->append(dsr.errorString());
  } else {
    result->append(dsr.readAll());
  }
  QStringList list = result->split(QRegExp("\n|\r\n|\r"));
  displayFragmentsList(list, "None");
}

bool DSRGui::isDSRVersionCorrect(QString version) {
  //! checks if the version of DSR found in DSR_DIR
  //! is compatible with this GUI
  bool ok;
  ok = false;
  if (version.trimmed().toInt() >= 182) {
    ok = true;
  }
  return ok;
}

bool DSRGui::listDSRdbFragments(QString fav="None") {
  //! list fragments in DSR database
  //! The -lc parameter of DSR returns a semicolon (;;) separated list
  //! tag;;fullname;;line number;;db
  if (QString("None") == fav) {
    fav = loadFavoriteFragment().toAscii().toUpper();
  }
  QString *fraglist = new QString;
  fragmentsList->clear();
  QProcess dsr;
  fragmentNameTag.clear();
  //dsr.setProcessChannelMode(QProcess::MergedChannels);
  dsr.start(dsrpath, QStringList() << "-lc");
  dsr.closeWriteChannel();
  if (!dsr.waitForFinished()) {
    outtext->append(QString(tr("Unable to start DSR.")));
    outtext->append(QString(tr("Please get the most recent version from https://www.xs3.uni-freiburg.de/research/dsr")));
    outtext->append(dsr.readAll());
    outtext->append(dsr.errorString());
  } else {
    fraglist->append(dsr.readAll());
  }
  QStringList list = fraglist->split(QRegExp("\n|\r\n|\r"));
  if (list.last().isEmpty())
    list.removeLast();
  bool versionOk;
  versionOk = false;
  foreach (QString line, list) {
    if (line.contains("Duplicate database entry")) {
      outtext->clear();
      outtext->append(*fraglist);
      return false;
    }
  }
  // in this case no fragment list returned. Hence, we have an error.
  if (!fraglist->contains(";;")) {
    outtext->clear();
    outtext->append(*fraglist);
    outtext->append(dsr.errorString());
    return false;
  }
  foreach (QString line, list) {
    if (line.isEmpty()){
      list.removeFirst();
    }
    if ( (line.split(":").size() >= 1)
         && line.contains("DSR version:")) {
      dsrVersion = line.split(":").at(1).trimmed();
    }
  }
  if (list[0].trimmed().contains("DSR version:")) {
    versionOk = isDSRVersionCorrect(dsrVersion);
    list.removeFirst();
  } else {
    outtext->clear();
    outtext->append(*fraglist);
    outtext->append("Something went wrong with DSR. Please report this to Daniel Kratzert.");
    return false;
  }
  if (!versionOk) {
    outtext->clear();
    outtext->append(QString(tr("Detected an old version of DSR.")));
    outtext->append(QString(tr("Please get the most recent version from https://www.xs3.uni-freiburg.de/research/dsr")));
    return false;
  }
  displayFragmentsList(list, fav);
  QStringList line;
  for (int i = 0; i < list.size(); ++i) {
    line.clear();
    if (!list[i].contains(";;")){
      continue;
    }
    line = list[i].split(";;");
    if ( line.size() < 3 ) {
      continue;
    }
    fragmentsList->append(line); // the global list of all fragments
  }
  return true;
}


void DSRGui::displayFragmentsList(QStringList list, QString fav="None") {
  //! displays the list of fragments in a table view
  //! The first column (the tag) is hidden, only the name is displayed.
  QStandardItem *fraglabel = new QStandardItem(QString("Fragment Name"));
  fraglabel->setTextAlignment(Qt::AlignLeft);
  QStandardItemModel *FragListmodel = new QStandardItemModel(
                                list.size()-1, 2, this); //x Rows and 2 Columns
  FragListmodel->setHorizontalHeaderItem(0, new QStandardItem(QString("tag")));
  FragListmodel->setHorizontalHeaderItem(1, fraglabel);
  QStringList line;
  for (int i = 0; i < list.size(); ++i) {
    line.clear();
    if (!list[i].contains(";;")){
      continue;
    }
    line = list[i].split(";;");
    if ( line.size() < 3 ) {
      continue;
    }
    QString column1 = line[0];
    QString column2 = line[1];
    QStandardItem *nameItem;
    // Add a bold *user* to all fragments from the user db
    if (line[3].contains("dsr_user_db")) {
      column2.append(" *user*");
      nameItem = new QStandardItem(QString(column2));
      QFont f = nameItem->font();
      f.setBold(true);
      nameItem->setFont(f);
    } else {
      nameItem = new QStandardItem(QString(column2));
    }
    FragListmodel->setItem(i, 0, new QStandardItem(QString(column1)));
    FragListmodel->setItem(i, 1, nameItem);
    // load the last fragment:
    if (fav == line.at(0).toAscii().toUpper()) {
      if (FragListmodel->hasIndex(i, 0)) {
        setFragName(FragListmodel->index(i, 0));
      }
    }
  }
  fragmentTableView->setModel(FragListmodel);
  fragmentTableView->verticalHeader()->hide();
  fragmentTableView->hideColumn(0);
  fragmentTableView->setColumnWidth(1, 600);
  fragmentTableView->setGridStyle(Qt::PenStyle(Qt::NoPen));
  fragmentTableView->setEditTriggers(QAbstractItemView::NoEditTriggers);
}

QString DSRGui::getDSRDir(){
  //! returns the value of the DSR_DIR variable
  //! in case of error, it returns an empty string
  QString dsrdir = qgetenv("DSR_DIR").data();
  if (!QFile::exists(dsrdir)){
    if (!which("dsr").isEmpty()) {
      QFileInfo fi(which("dsr").at(0));
      dsrdir = fi.absolutePath();
    }
    if (QFile::exists("/Applications/DSR")) {
        dsrdir = "/Applications/DSR";
    }
    QString dsrdir2 = qgetenv("DSRDIR").data();
    if (QFile::exists(dsrdir2)) {
      qDebug() << "You are probably using an older version of DSR."
                  "\n Please use version 1.7.7 or above.";
      // return empty string, because dsr.bat will
      // hinder dsr from starting properly:
      return QString("");
    }
  }
  return QDir::fromNativeSeparators(dsrdir);
}

QString DSRGui::getDSRdbDir(){
  //! returns the directory with the userdb -> the home directory
  QString dbdir = QDir::homePath();
  return QDir::fromNativeSeparators(dbdir);;
}


void DSRGui::setExportDirDialog() {
  //! File dialog to define the directory for the res file
  //! exported by DSR
  export_dir = "";
  export_dir = QFileDialog::getExistingDirectory(this,
      tr("Export Fragment to ..."), tr("Directory"));
  if (QFile::exists(export_dir)){
    emit exportDirChanged(export_dir);
  }
}


