;;; s7 test suite
;;;
;;; sources include 
;;;   clisp test suite
;;;   sbcl test suite
;;;   Paul Dietz's CL test suite (gcl/ansi-tests/*)
;;;   R Kelsey, W Clinger, and J Rees r5rs.html (and r6rs.html)
;;;   A Jaffer's r4rstest.scm (the inspiration for this...)
;;;   guile test suite
;;;   gauche test suite
;;;   sacla test suite
;;;   Kent Dybvig's "The Scheme Programming Language"
;;;   Brad Lucier and Peter Bex
;;;   GSL tests
;;;   Abramowitz and Stegun, "Handbook of Mathematical Functions"
;;;   Weisstein, "Encyclopedia of Mathematics"
;;;   the arprec package of David Bailey et al
;;;   Maxima, William Schelter et al
;;;   H Cohen, "A Course in Computational Algebraic Number Theory"
;;;   N Higham, "Accuracy and Stability of Numerical Algorithms"
;;;   various mailing lists and websites (see individual cases below)

(define full-test #f) ; this includes some time-consuming stuff
(define with-bignums (provided? 'gmp))  ; scheme number has any number of bits
					; we assume s7_double is double, and s7_int is long long int
                                        ;   a few of the bignum tests assume the default bignum-precision is 128

(define with-complex (provided? 'complex-numbers))
(define with-windows (provided? 'windows))
(if (not (defined? 's7test-exits)) (define s7test-exits #t))


;;; ---------------- pure-s7 ----------------
(define pure-s7 (provided? 'pure-s7))
(when pure-s7
  (define (make-polar mag ang)
    (if (and (real? mag) (real? ang))
	(complex (* mag (cos ang)) (* mag (sin ang)))
	(error 'wrong-type-arg "make-polar args should be real")))
  (define make-rectangular complex)

  (define (char-ci=? . chars) (apply char=? (map char-upcase chars)))
  (define (char-ci<=? . chars) (apply char<=? (map char-upcase chars)))
  (define (char-ci>=? . chars) (apply char>=? (map char-upcase chars)))
  (define (char-ci<? . chars) (apply char<? (map char-upcase chars)))
  (define (char-ci>? . chars) (apply char>? (map char-upcase chars)))
  
  (define (string-ci=? . strs) (apply string=? (map string-upcase strs)))
  (define (string-ci<=? . strs) (apply string<=? (map string-upcase strs)))
  (define (string-ci>=? . strs) (apply string>=? (map string-upcase strs)))
  (define (string-ci<? . strs) (apply string<? (map string-upcase strs)))
  (define (string-ci>? . strs) (apply string>? (map string-upcase strs)))

  (define (list->string lst) (apply string lst))
  (define (list->vector lst) (apply vector lst))

  (define (let->list e)
    (if (let? e)
	(reverse! (map values e))
	(error 'wrong-type-arg "let->list argument should be an environment: ~A" str)))

  (define* (string->list str (start 0) end)
    (if (and (string? str)
	     (integer? start)
	     (not (negative? start))
	     (or (not end)
		 (and (integer? end)
		      (>= end start))))
	(map values (substring str start (or end (length str))))
	(error 'wrong-type-arg "string->list argument should be a string: ~A" str)))

  (define (string-copy str)
    (if (string? str)
	(copy str)
	(error 'wrong-type-arg "string-copy argument should be a string: ~A" str)))
  
  (define (string-length str)
    (if (string? str)
	(length str)
	(error 'wrong-type-arg "string-length argument should be a string: ~A" str)))
  
  (define (string-fill! str chr . args)
    (if (string? str)
	(apply fill! str chr args)
	(error 'wrong-type-arg "string-fill! argument should be a string: ~A" str)))
  
  (define* (vector->list vect (start 0) end)
    (if (and (vector? vect)
	     (integer? start)
	     (not (negative? start))
	     (or (not end)
		 (and (integer? end)
		      (>= end start))))
	(if start
	    (let ((stop (or end (length vect))))
	      (if (= start stop)
		  ()
		  (map values (make-shared-vector vect (list (- stop start)) start))))
	    (map values vect))
	(error 'wrong-type-arg "vector->list argument should be a vector: ~A" vect)))

  (define (vector-length vect)
    (if (vector? vect)
	(length vect)
	(error 'wrong-type-arg "vector-length argument should be a vector: ~A" vect)))
  
  (define (vector-fill! vect val . args)
    (if (vector? vect)
	(apply fill! vect val args)
	(error 'wrong-type-arg "vector-fill! argument should be a vector: ~A" str)))
  
  (define (vector-append . args)
    (if (null? args)
	#()
	(if (vector? (car args))
	    (apply append args)
	    (error 'wrong-type-arg "vector-append arguments should be vectors: ~A" args))))
  
  (define* (char-ready? p)
    (and p (not (input-port? p))
	 (error 'wrong-type-arg "char-ready? arg should be an input port")))

  (define (set-current-output-port port) (error 'undefined-function "set-current-output-port is not in pure-s7"))
  (define (set-current-input-port port) (error 'undefined-function "set-current-input-port is not in pure-s7"))

  (define (exact? n) 
    (if (number? n)
	(rational? n)
	(error 'wrong-type-arg "exact? argument should be a number: ~A" n)))

  (define (inexact? x) 
    (if (number? x)
	(not (rational? x))
	(error 'wrong-type-arg "inexact? argument should be a number: ~A" x)))

  (define (inexact->exact x)
    (if (not (number? x))
	(error 'wrong-type-arg "inexact->exact argument should be a number: ~A" x)
	(if (rational? x)
	    x
	    (rationalize x))))

  (define (exact->inexact x)
    (if (number? x)
	(* x 1.0)
	(error 'wrong-type-arg "exact->inexact argument should be a number: ~A" x)))

  (define (integer-length i)
    (if (integer? i)
	(if (zero? i)
	    0
	    (+ (ceiling (log (abs i) 2))
	       (if (and (positive? i)
			(zero? (logand i (- i 1))))
		   1 0)))
	(error 'wrong-type-arg "integer-length argument should be an integer: ~A" x)))

  (set! *#readers* (list (cons #\i (lambda (str) (* 1.0 (string->number (substring str 1)))))
			 (cons #\e (lambda (str) (floor (string->number (substring str 1)))))))
  ;; one problem (of many): (string->number "#e0a" 16) -- no simple way to tell the #e reader that we want base 16
  ;;   similarly #x#if -- so in pure-s7, the reader complains

  (define-macro (defmacro name args . body) `(define-macro ,(cons name args) ,@body))
  (define-macro (defmacro* name args . body) `(define-macro* ,(cons name args) ,@body))

  (define-macro (call-with-values producer consumer) `(,consumer (,producer)))

  (define-macro (multiple-value-bind vars expression . body)   ; named "receive" in srfi-8 which strikes me as perverse
    (if (or (symbol? vars) (negative? (length vars)))
	`((lambda ,vars ,@body) ,expression)
	`((lambda* (,@vars . ,(gensym)) ,@body) ,expression)))

  (define-macro (multiple-value-set! vars expr . body)
    (let ((local-vars (map (lambda (n) (gensym)) vars)))
      `((lambda* (,@local-vars . ,(gensym))
	  ,@(map (lambda (n ln) `(set! ,n ,ln)) vars local-vars)
	  ,@body)
	,expr)))

  (define-macro (cond-expand . clauses)
    (letrec ((traverse (lambda (tree)
			 (if (pair? tree)
			     (cons (traverse (car tree))
				   (if (null? (cdr tree)) () (traverse (cdr tree))))
			     (if (memq tree '(and or not else))
				 tree
				 (and (symbol? tree) (provided? tree)))))))
      `(cond ,@(map (lambda (clause)
		      (cons (traverse (car clause))
			    (if (null? (cdr clause)) '(#f) (cdr clause))))
		    clauses))))
  )
;;; ---------------- end pure-s7 ----------------

(define tmp-output-file "tmp1.r5rs")
(define tmp-data-file "test.dat")
(define bold-text (format #f "~C[1m" #\escape))
(define unbold-text (format #f "~C[22m" #\escape))
(set! (hook-functions *unbound-variable-hook*) ())
(set! (hook-functions *missing-close-paren-hook*) ())
(define s7test-output #f) ; if a string, it's treated as a logfile
;(set! (*s7* 'gc-stats) #t)
;(set! (*s7* 'undefined-identifier-warnings) #t)

(define old-stdin *stdin*)
(define old-stdout *stdout*)
(define old-stderr *stderr*)
(define *max-arity* #x20000000)

(define (-s7-stack-top-) (*s7* 'stack-top))
(define -s7-symbol-table-locked? (dilambda 
				  (lambda ()
				    (*s7* 'symbol-table-locked?))
				  (lambda (val)
				    (set! (*s7* 'symbol-table-locked?) val))))

(if (provided? 'profiling)
    (load "profile.scm"))

;;; --------------------------------------------------------------------------------

(if (and (defined? 'current-time) ; in Snd
	 (defined? 'mus-rand-seed))
    (set! (mus-rand-seed) (current-time)))

(define (format-logged . args)
  ;(if (not (eq? (current-output-port) old-stdout)) (apply format (cons old-stdout (cdr args))))
  (let ((str (apply format args)))
    ;(if (eq? (car args) #t) (flush-output-port (current-output-port)))
    (if (string? s7test-output)
	(let ((p (open-output-file s7test-output "a")))
	  (display str p)
	  (flush-output-port p)
	  (close-output-port p)))
    str))

(define (ok? otst ola oexp)
  (let ((result (catch #t ola
		       (lambda args
			 (if (not (eq? oexp 'error)) 
			     (begin (display args) (newline)))
			 'error))))
    (if (not (equal? result oexp))
	(format-logged #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) otst result oexp))))

(if (not (defined? 'test))
    (define-macro (test tst expected) ;(display tst *stderr*) (newline *stderr*)
      ;; `(ok? ',tst (lambda () (eval-string (format #f "~S" ',tst))) ,expected))
      ;; `(ok? ',tst (lambda () (eval ',tst)) ,expected))
      ;; `(ok? ',tst (lambda () ,tst) ,expected))
      ;; `(ok? ',tst (lambda () (eval-string (object->string ,tst :readable))) ,expected))
      ;; `(ok? ',tst (let () (define (_s7_) ,tst)) ,expected))
      ;; `(ok? ',tst (lambda () (let ((_s7_ #f)) (set! _s7_ ,tst))) ,expected))
      ;; `(ok? ',tst (lambda () (let ((_s7_ ,tst)) _s7_)) ,expected))
      ;; `(ok? ',tst (catch #t (lambda () (lambda* ((_a_ ,tst)) _a_)) (lambda any (lambda () 'error))) ,expected))
      ;; `(ok? ',tst (lambda () (do ((_a_ ,tst)) (#t _a_))) ,expected))
      ;; `(ok? ',tst (lambda () (call-with-exit (lambda (_a_) (_a_ ,tst)))) ,expected))
      ;; `(ok? ',tst (lambda () (values ,tst)) ,expected))
      ;; `(ok? ',tst (lambda () (define (_s7_ _a_) _a_) (_s7_ ,tst)) ,expected))
      ;; `(ok? ',tst (lambda () (define* (_s7_ (_a_ #f)) (or _a_)) (_s7_ ,tst)) ,expected))
      ;; `(ok? ',tst (lambda () (caadr (catch 'receive (lambda () (throw 'receive ,tst)) (lambda any any)))) ,expected))
      ;; `(ok? ',tst (lambda () (stacktrace (- (random 100) 50) (- (random 100) 50) (- (random 100) 50) (- (random 100) 50) (> (random 100) 50)) ,tst) ,expected))
      (list-values 'ok? (list-values quote tst) (list-values lambda () tst) expected))
#|
      `(let ((_result_ #f))
	 (define (stest) (set! _result_ ,tst))
	 (catch #t stest
		(lambda args
		  (set! _result_ 'error)))
	 (if (not (equal? _result_ ,expected))
	     (format-logged #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) ',tst _result_ ,expected))))
|#
    )

(define (tok? otst ola)
  (let* ((data #f)
	 (result (catch #t ola 
			(lambda args 
			 (set! data args) 
			 'error))))
    (if (or (not result)
	    (eq? result 'error))
	(format-logged #t "~A: ~A got ~S ~A~%~%" (port-line-number) otst result (or data "")))))

(define-macro (test-t tst) ;(display tst *stderr*) (newline *stderr*)
  `(tok? ',tst (lambda () ,tst)))

(define-macro (test-e tst op arg) ;(display tst *stderr*) (newline *stderr*)
  `(let ((result (catch #t (lambda () ,tst) 
			(lambda args 
			 'error))))
     (if (not (eq? result 'error))
	 (format-logged #t "~A: (~A ~S) got ~S but expected 'error~%~%" (port-line-number) ,op ,arg result))))


(define (op-error op result expected)
  
  (if (and (real? result)
	   (real? expected))
      (/ (abs (- result expected)) (max 1.0 (abs expected)))
      (case op
	((acosh)
	 (/ (magnitude (- (cosh result) (cosh expected)))
	    (max 0.001 (magnitude (cosh expected)))))
	((asin)
	 (/ (min (magnitude (- (sin result) (sin expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (* 10 (magnitude (sin expected))))))
	((acos)
	 (/ (min (magnitude (- (cos result) (cos expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (cos expected)))))
	((asinh)
	 (/ (magnitude (- (sinh result) (sinh expected)))
	    (max 0.001 (magnitude (sinh expected)))))
	((atanh)
	 (/ (min (magnitude (- (tanh result) (tanh expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tanh expected)))))
	((atan)
	 (/ (min (magnitude (- (tan result) (tan expected)))
		 (magnitude (- result expected)))
	    (max 0.001 (magnitude (tan expected)))))
	((cosh)
	 (/ (min (magnitude (- result expected))
		 (magnitude (+ result expected)))
	    (max 0.001 (magnitude expected))))
	(else (/ (magnitude (- result expected)) (max 0.001 (magnitude expected)))))))


;;; relative error (/ (abs (- x res) (abs x)))

(define error-12 1e-12)
(define error-6  1e-6)

(define (number-ok? tst result expected)
  (if (and (not (eq? result expected))
	   (or (not (or (number? expected) 
			(eq? result expected)))
	       (and (number? expected)
		    (nan? expected)
		    (not (nan? result)))
	       (and (number? result)
		    (nan? result)
		    (number? expected)
		    (not (nan? expected)))
	       (and (number? expected)
		    (or (not (number? result))
			(nan? result)))
	       (and (rational? expected)
		    (rational? result)
		    (not (= result expected)))
	       (and (or (rational? expected)
			(rational? result))
		    (real? expected)
		    (real? result)
		    (> (/ (abs (- result expected)) (max 1.0 (abs expected))) error-12))
	       (and (pair? tst)
		    (> (op-error (car tst) result expected) error-6))))
      (format-logged #t "~A: ~A got ~A~Abut expected ~A~%~%" 
		     (port-line-number) tst result 
		     (if (and (rational? result) (not (rational? expected)))
			 (format #f " (~A) " (* 1.0 result))
			 " ")
		     expected)))

(define (nok? otst ola oexp)
  (let ((result (catch #t ola 
		       (lambda args 
			 'error))))
    (number-ok? otst result oexp)))

(if (not (defined? 'num-test))
    (define-macro (num-test tst expected) ;(display tst *stderr*) (newline *stderr*)
      ;; `(nok? ',tst  (lambda () ,tst) ,expected))
      ;; `(nok? ',tst (let () (define (_s7_) ,tst)) ,expected))
      (list-values 'nok? (list-values quote tst) (list-values lambda () tst) expected)))

(define-macro (num-test-1 proc val tst expected)
  `(let ((result (catch #t (lambda () ,tst) 
			(lambda args 
			 'error))))
     (number-ok? (list ,proc ,val) result ,expected)))

(define-macro (num-test-2 proc val1 val2 tst expected)
  `(let ((result (catch #t (lambda () ,tst) 
			(lambda args 
			 'error))))
     (number-ok? (list ,proc ,val1 ,val2) result ,expected)))

(define (reinvert n op1 op2 arg)
  (let ((body (op2 (op1 arg))))
    (do ((i 1 (+ i 1)))
	((= i n) body)
      (set! body (op2 (op1 body))))))
    
(define (recompose n op arg)
  (define (recompose-1 n)
    (if (= n 1)
	(op arg)
	(op (recompose-1 (- n 1)))))
  (recompose-1 n))


(if (symbol-access 'val) (set! (symbol-access 'val) #f)) ; might get here from snd-test

(define _ht_ (make-hash-table))



;;; --------------------------------------------------------------------------------
;;; before starting, make a test c-object

(define with-block (not (provided? 'windows)))

(if with-block
    (begin
      (call-with-output-file "s7test-block.c"
	(lambda (p)
	  (format p "
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include \"s7.h\"
static s7_scheme *s7;

/* c-object tests */
typedef struct {
  size_t size;
  double *data;
} g_block;    

static int g_block_type = 0;
static s7_pointer g_block_methods;

static s7_pointer g_make_block(s7_scheme *sc, s7_pointer args)
{
  #define g_make_block_help \"(make-block size) returns a new block of the given size\"
  g_block *g;
  s7_pointer new_g;
  s7_int size;
  if (!s7_is_integer(s7_car(args)))
    return(s7_wrong_type_arg_error(sc, \"make-block\", 1, s7_car(args), \"an integer\"));
  size = s7_integer(s7_car(args));
  if ((size < 0) ||
      (size > 1073741824))
     return(s7_out_of_range_error(sc, \"make-block\", 1, s7_car(args), \"should be something reasonable\"));
  g = (g_block *)calloc(1, sizeof(g_block));
  g->size = (size_t)size;
  g->data = (double *)calloc(g->size, sizeof(double));
  new_g = s7_make_object(sc, g_block_type, (void *)g);
  s7_object_set_let(new_g, g_block_methods);
  s7_openlet(sc, new_g);
  return(new_g);
}

static s7_pointer g_to_block(s7_scheme *sc, s7_pointer args)
{
  #define g_block_help \"(block ...) returns a block object with the arguments as its contents.\"
  s7_pointer p, b;
  size_t i, len;
  g_block *gb;
  len = s7_list_length(sc, args);
  b = g_make_block(sc, s7_cons(sc, s7_make_integer(sc, len), s7_nil(sc)));
  gb = (g_block *)s7_object_value(b);
  for (i = 0, p = args; i < len; i++, p = s7_cdr(p))
    gb->data[i] = s7_number_to_real(sc, s7_car(p));
  return(b);
}

static char *g_block_display(s7_scheme *sc, void *value)
{
  g_block *b = (g_block *)value;
  int i, len;
  char *buf;
  char flt[64];
  len = b->size;
  if (len > s7_print_length(sc)) len = s7_print_length(sc);
  buf = (char *)calloc((len + 1) * 64, sizeof(char));
  snprintf(buf, (len + 1) * 64, \"(block\");
  for (i = 0; i < len; i++)
    {
      snprintf(flt, 64, \" %.3f\", b->data[i]);
      strcat(buf, flt);
    }
  if (b->size > s7_print_length(sc))
    strcat(buf, \" ...)\");
  else strcat(buf, \")\");
  return(buf);
}

static char *g_block_display_readably(s7_scheme *sc, void *value)
{
  char *buf;
  s7_int plen;
  g_block *b = (g_block *)value;
  plen = s7_set_print_length(sc, b->size + 1);
  buf = g_block_display(sc, value);
  s7_set_print_length(sc, plen);
  return(buf);
}

static void g_block_free(void *value)
{
  g_block *g = (g_block *)value;
  free(g->data);
  free(g);
}

static bool g_block_is_equal(void *val1, void *val2)
{
  int i, len;
  g_block *b1 = (g_block *)val1;
  g_block *b2 = (g_block *)val2;
  if (val1 == val2) return(true);
  len = b1->size;
  if (len != b2->size) return(false);
  for (i = 0; i < len; i++)
    if (b1->data[i] != b2->data[i])
      return(false);
  return(true);	 
}

static void g_block_mark(void *val)
{
  /* nothing to mark */
}

static s7_pointer g_is_block(s7_scheme *sc, s7_pointer args)
{
  #define g_is_block_help \"(block? obj) returns #t if obj is a block.\"
  return(s7_make_boolean(sc, s7_object_type(s7_car(args)) == g_block_type));
}

static s7_pointer g_block_ref(s7_scheme *sc, s7_pointer obj, s7_pointer args)
{
  g_block *g = (g_block *)s7_object_value(obj);
  size_t index;
  s7_pointer ind;
  if (s7_is_null(sc, args)) /* this is for an (obj) test */
    return(s7_make_integer(sc, 32));
  ind = s7_car(args);
  if (!s7_is_integer(ind))
    {
      if (s7_is_symbol(ind))
         return(s7_symbol_local_value(sc, ind, g_block_methods));
      return(s7_wrong_type_arg_error(sc, \"block-ref\", 1, ind, \"an integer\"));
    }
  index = (size_t)s7_integer(ind);
  if (index < g->size)
    return(s7_make_real(sc, g->data[index]));
  return(s7_out_of_range_error(sc, \"block-ref\", 2, ind, \"should be less than block length\"));
}

static s7_pointer g_block_set(s7_scheme *sc, s7_pointer obj, s7_pointer args)
{
  g_block *g = (g_block *)s7_object_value(obj);
  s7_int index;
  if (!s7_is_integer(s7_car(args)))
    return(s7_wrong_type_arg_error(sc, \"block-set!\", 1, s7_car(args), \"an integer\"));
  index = s7_integer(s7_car(args));
  if ((index >= 0) && (index < g->size))
    {
      g->data[index] = s7_number_to_real(sc, s7_cadr(args));
      return(s7_cadr(args));
    }
  return(s7_out_of_range_error(sc, \"block-set\", 2, s7_car(args), \"should be less than block length\"));
}

static s7_double c_block_ref(s7_scheme *sc, s7_pointer **p)
{
  s7_int ind;
  s7_if_t xf;
  g_block *g;
  g = (g_block *)(**p); (*p)++;
  xf = (s7_if_t)(**p); (*p)++;
  ind = xf(sc, p);
  return(g->data[ind]);
}

static s7_double c_block_set(s7_scheme *sc, s7_pointer **p)
{
  s7_int ind;
  s7_double x;
  s7_rf_t rf;
  s7_if_t xf;
  g_block *g;
  g = (g_block *)(**p); (*p)++;
  xf = (s7_if_t)(**p); (*p)++;
  ind = xf(sc, p);
  rf = (s7_rf_t)(**p); (*p)++;
  x = rf(sc, p);
  g->data[ind] = x;
  return(x);
}

static s7_rf_t block_rf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1, gs;
  a1 = s7_car(expr);
  if ((s7_is_symbol(a1)) &&
      (s7_object_type(gs = s7_symbol_value(sc, a1)) == g_block_type))
    {
      s7_xf_store(sc, (s7_pointer)s7_object_value(gs));
      if (s7_arg_to_if(sc, s7_cadr(expr))) return(c_block_ref);
    }
  return(NULL);
}

static s7_rf_t block_set_rf(s7_scheme *sc, s7_pointer expr)
{
  s7_pointer a1, gs;
  a1 = s7_cadr(expr);
  if ((!s7_is_pair(a1)) || (!s7_is_symbol(s7_car(a1))) || (!s7_is_null(sc, s7_cddr(a1)))) return(NULL);
  gs = s7_symbol_value(sc, s7_car(a1));
  if (s7_object_type(gs) == g_block_type)
    {
      s7_xf_store(sc, (s7_pointer)s7_object_value(gs));
      if (!s7_arg_to_if(sc, s7_cadr(a1))) return(NULL);
      if (!s7_arg_to_rf(sc, s7_caddr(expr))) return(NULL);
      return(c_block_set);
    }
  return(NULL);
}

static s7_pointer block_direct_ref(s7_scheme *sc, s7_pointer obj, s7_int index)
{
  g_block *g;
  g = (g_block *)s7_object_value(obj);
  return(s7_make_real(sc, g->data[index]));
}
 
static s7_pointer block_direct_set(s7_scheme *sc, s7_pointer obj, s7_int index, s7_pointer val)
{
  g_block *g;
  g = (g_block *)s7_object_value(obj);
  g->data[index] = s7_number_to_real(sc, val);
  return(val);
}

static s7_pointer g_block_length(s7_scheme *sc, s7_pointer obj)
{
  g_block *g = (g_block *)s7_object_value(obj);
  return(s7_make_integer(sc, g->size));
}

static int get_start_and_end(s7_scheme *sc, s7_pointer args, int *start, int end)
{
  if (s7_is_pair(s7_cdr(args)))
    {
      s7_pointer p;
      p = s7_cadr(args);
      if (s7_is_integer(p))
        {
          int nstart;
          nstart = s7_integer(p);
          if ((nstart < 0) || (nstart >= end))
            {s7_out_of_range_error(sc, \"subblock\", 2, p, \"should be less than block length\"); return(0);}
          *start = nstart;
        }
      if (s7_is_pair(s7_cddr(args)))
        {
          p = s7_caddr(args);
          if (s7_is_integer(p))
            {
              int nend;
              nend = s7_integer(p);
              if (nend <= *start)
                {s7_out_of_range_error(sc, \"subblock\", 3, p, \"should be greater than the start point\"); return(0);}
              if (nend < end) end = nend;
            }
        }
    }
  return(end - *start);
}

static s7_pointer g_block_copy(s7_scheme *sc, s7_pointer args)
{
  s7_pointer new_g, obj;
  g_block *g, *g1;
  size_t len;
  int start = 0;
  obj = s7_car(args);
  g = (g_block *)s7_object_value(obj);
  len = g->size;
  if (s7_is_pair(s7_cdr(args)))
    {
      new_g = s7_cadr(args);
      if (s7_object_type(new_g) != g_block_type) /* fall back on the float-vector code using a wrapper */
        {
          int gc_loc;
          s7_pointer v;
	  v = s7_make_float_vector_wrapper(sc, len, g->data, 1, NULL, false);
          gc_loc = s7_gc_protect(sc, v);
          new_g = s7_copy(sc, s7_append(sc, s7_list(sc, 1, v), s7_cdr(args)));
          s7_gc_unprotect_at(sc, gc_loc);
          return(new_g);
        }
      if (s7_is_pair(s7_cddr(args)))
        len = get_start_and_end(sc, s7_cdr(args), &start, len);
    }
  else new_g = g_make_block(sc, s7_cons(sc, s7_make_integer(sc, len), s7_nil(sc)));
  g1 = (g_block *)s7_object_value(new_g);
  if (g1->size < len) len = g1->size;
  memcpy((void *)(g1->data), (void *)(g->data + start), len * sizeof(double));
  return(new_g);
}

static s7_pointer g_block_append(s7_scheme *sc, s7_pointer args)
{
  #define g_block_append_help \"(append block...) returns a new block containing the argument blocks concatenated.\"
  int i, len = 0;
  s7_pointer p, new_g;
  g_block *g;
  for (i = 0, p = args; s7_is_pair(p); p = s7_cdr(p), i++)
    {
      g_block *g1;
      if (s7_object_type(s7_car(p)) != g_block_type)
        return(s7_wrong_type_arg_error(sc, \"block-append\", i, s7_car(p), \"a block\"));
      g1 = (g_block *)s7_object_value(s7_car(p));
      len += g1->size;
    }
  new_g = g_make_block(sc, s7_cons(sc, s7_make_integer(sc, len), s7_nil(sc)));
  g = (g_block *)s7_object_value(new_g);
  for (i = 0, p = args; s7_is_pair(p); p = s7_cdr(p))
    {
      g_block *g1;
      g1 = (g_block *)s7_object_value(s7_car(p));
      memcpy((void *)(g->data + i), (void *)(g1->data), g1->size * sizeof(double));
      i += g1->size;
    }
  return(new_g);
}

static s7_pointer g_block_reverse(s7_scheme *sc, s7_pointer args)
{
  size_t i, j;
  g_block *g, *g1; 
  s7_pointer new_g;
  g = (g_block *)s7_object_value(s7_car(args));
  new_g = g_make_block(sc, s7_cons(sc, s7_make_integer(sc, g->size), s7_nil(sc)));
  g1 = (g_block *)s7_object_value(new_g);
  for (i = 0, j = g->size - 1; i < g->size; i++, j--)
    g1->data[i] = g->data[j];
  return(new_g);
}

static s7_pointer g_block_reverse_in_place(s7_scheme *sc, s7_pointer args)
{
  #define g_block_reverse_in_place_help \"(block-reverse! block) returns block with its data reversed.\"
  size_t i, j;
  g_block *g; 
  s7_pointer obj;
  obj = s7_car(args);
  if (s7_object_type(obj) != g_block_type)
    return(s7_wrong_type_arg_error(sc, \"block-reverse!\", 0, obj, \"a block\"));
  g = (g_block *)s7_object_value(obj);
  if (g->size < 2) return(obj);
  for (i = 0, j = g->size - 1; i < j; i++, j--)
    {
      double temp;
      temp = g->data[i];
      g->data[i] = g->data[j];
      g->data[j] = temp;
    }
  return(obj);
}

static s7_pointer g_block_fill(s7_scheme *sc, s7_pointer args)
{
  s7_pointer obj, val;
  size_t i, len;
  int start = 0;
  double fill_val;
  g_block *g;
  obj = s7_car(args);
  val = s7_cadr(args);
  g = (g_block *)s7_object_value(obj);
  fill_val = s7_number_to_real(sc, val);
  len = g->size;
  if (s7_is_pair(s7_cddr(args)))
    len = get_start_and_end(sc, s7_cdr(args), &start, len);
  if (fill_val == 0.0)
    memset((void *)(g->data + start), 0, len * sizeof(double));
  else
    {
      for (i = 0; i < len; i++)
        g->data[i + start] = fill_val;
    }
  return(obj);
}

static s7_pointer g_blocks(s7_scheme *sc, s7_pointer args)
{
  return(s7_copy(sc, s7_list(sc, 1, args)));
}

static s7_pointer g_subblock(s7_scheme *sc, s7_pointer args)
{
  #define g_subblock_help \"(subblock block (start 0) end) returns a portion of the block.\"
  s7_pointer p, new_g, obj;
  int start = 0, new_len, i;
  g_block *g, *g1;
  obj = s7_car(args);
  if (s7_object_type(obj) != g_block_type)
    return(s7_wrong_type_arg_error(sc, \"subblock\", 1, obj, \"a block\"));
  g = (g_block *)s7_object_value(obj);
  new_len = get_start_and_end(sc, args, &start, g->size);
  new_g = g_make_block(sc, s7_cons(sc, s7_make_integer(sc, new_len), s7_nil(sc)));
  g1 = (g_block *)s7_object_value(new_g);
  memcpy((void *)(g1->data), (void *)(g->data + start), new_len * sizeof(double));
  return(new_g);
}


/* function port tests */
static unsigned char *fout = NULL;
static unsigned int fout_size = 0, fout_loc = 0;
static void foutput(s7_scheme *sc, unsigned char c, s7_pointer port)
{
  if (fout_size == fout_loc)
    {
      if (fout_size == 0)
        {
          fout_size = 128;
          fout = (unsigned char *)malloc(fout_size * sizeof(unsigned char));
        }
      else
        {
          fout_size += 128;
          fout = (unsigned char *)realloc(fout, fout_size * sizeof(unsigned char));
        }
    }
  fout[fout_loc++] = c;
}

static s7_pointer fout_open(s7_scheme *sc, s7_pointer args)
{
  return(s7_open_output_function(sc, foutput));
}

static s7_pointer fout_get_output(s7_scheme *sc, s7_pointer args)
{
  foutput(sc, 0, s7_car(args)); /* make sure it's null-terminated */
  return(s7_make_string_with_length(sc, (const char *)fout, fout_loc - 1));
}

static s7_pointer fout_close(s7_scheme *sc, s7_pointer args)
{
  fout_loc = 0;
  return(s7_car(args));
}

static const char *fin = NULL;
static unsigned int fin_size = 0, fin_loc = 0;
static s7_pointer finput(s7_scheme *sc, s7_read_t peek, s7_pointer port)
{
  switch (peek)
    {
      case S7_READ_BYTE:
        return(s7_make_integer(sc, (int)fin[fin_loc++]));
      case S7_READ_CHAR:
        return(s7_make_character(sc, fin[fin_loc++]));
      case S7_PEEK_CHAR:
        return(s7_make_character(sc, fin[fin_loc]));
      case S7_READ_LINE:
        {
          unsigned int i;
          s7_pointer result;
          for (i = fin_loc; (i < fin_size) && (fin[i] != '\\n'); i++);
          result = s7_make_string_with_length(sc, (char *)(fin + fin_loc), i - fin_loc);
          fin_loc = i + 1;
          return(result);
        }
      case S7_IS_CHAR_READY:
        return(s7_make_boolean(sc, fin_loc < fin_size));
      case S7_READ:
        return(s7_error(sc, s7_make_symbol(sc, \"read-error\"), s7_make_string(sc, \"can't read yet!\")));
    }
}

static s7_pointer fin_open(s7_scheme *sc, s7_pointer args)
{
  /* arg = string to read */
  s7_pointer str;
  fin_loc = 0;
  str = s7_car(args);
  fin = s7_string(str); /* assume caller will GC protect the string */
  fin_size = s7_string_length(str);
  return(s7_open_input_function(sc, finput));
}

/* dilambda test */
static s7_pointer g_dilambda_test(s7_scheme *sc, s7_pointer args) {return(s7_f(sc));}
static s7_pointer g_dilambda_set_test(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

/* hash-table tests */
static s7_pointer g_hloc(s7_scheme *sc, s7_pointer args) {return(s7_make_integer(sc, 0));}
static s7_pointer g_heq(s7_scheme *sc, s7_pointer args) {return(s7_make_boolean(sc, s7_is_eq(s7_car(args), s7_cadr(args))));}

/* optimizer tests */
static s7_pointer g_cf10(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf11(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cs11(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

static s7_pointer g_cf20(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf21(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf22(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}

static s7_pointer g_cf30(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf31(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf32(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}
static s7_pointer g_cf33(s7_scheme *sc, s7_pointer args) {return(s7_caddr(args));}

static s7_pointer g_cf41(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}
static s7_pointer g_cf42(s7_scheme *sc, s7_pointer args) {return(s7_cadr(args));}
static s7_pointer g_cf43(s7_scheme *sc, s7_pointer args) {return(s7_caddr(args));}
static s7_pointer g_cf44(s7_scheme *sc, s7_pointer args) {return(s7_cadddr(args));}
static s7_pointer g_rs11(s7_scheme *sc, s7_pointer args) {return(s7_make_integer(sc, s7_integer(s7_car(args)) + 1));}

static s7_pointer g_cf51(s7_scheme *sc, s7_pointer args) {return(s7_car(args));}

static s7_pointer sload(s7_scheme *sc, s7_pointer args) 
{
  if (s7_is_string(s7_car(args)))
    {
      if (s7_is_pair(s7_cdr(args)))
	 {
	   if (s7_is_let(s7_cadr(args)))
             return(s7_load_with_environment(sc, s7_string(s7_car(args)), s7_cadr(args)));
           return(s7_wrong_type_arg_error(sc, \"load\", 2, s7_cadr(args), \"an environment\"));
         }
      return(s7_load(sc, s7_string(s7_car(args))));
    }
  return(s7_wrong_type_arg_error(sc, \"load\", 1, s7_car(args), \"file name\"));
}
static s7_pointer scall(s7_scheme *sc, s7_pointer args) {return(s7_call(sc, s7_car(args), s7_cadr(args)));}
static s7_pointer sread(s7_scheme *sc, s7_pointer args) 
{
  if (s7_is_pair(args))
    return(s7_read(sc, s7_car(args)));
  return(s7_read(sc, s7_current_input_port(sc)));
}
static s7_pointer swind(s7_scheme *sc, s7_pointer args) {return(s7_dynamic_wind(sc, s7_car(args), s7_cadr(args), s7_caddr(args)));}
static s7_pointer seval(s7_scheme *sc, s7_pointer args) 
{
  if (s7_is_pair(s7_cdr(args)))
    return(s7_eval(sc, s7_car(args), s7_cadr(args)));
  return(s7_eval(sc, s7_car(args), s7_curlet(sc)));
}
static s7_pointer sevalstr(s7_scheme *sc, s7_pointer args) 
{
  if (s7_is_string(s7_car(args)))
    {
      if (s7_is_pair(s7_cdr(args)))
	 {
	   if (s7_is_let(s7_cadr(args)))
             return(s7_eval_c_string_with_environment(sc, s7_string(s7_car(args)), s7_cadr(args)));
           return(s7_wrong_type_arg_error(sc, \"eval-string\", 2, s7_cadr(args), \"an environment\"));
	 }
      return(s7_eval_c_string_with_environment(sc, s7_string(s7_car(args)), s7_curlet(sc)));
    }
  return(s7_wrong_type_arg_error(sc, \"eval-string\", 1, s7_car(args), \"string of code\"));
}

void block_init(s7_scheme *sc);
void block_init(s7_scheme *sc)
{
  s7_pointer cur_env;
  cur_env = s7_outlet(sc, s7_curlet(sc));
  g_block_type = s7_new_type_x(sc, \"#<block>\", 
			       g_block_display, g_block_free, 
			       g_block_is_equal, g_block_mark,
			       g_block_ref, g_block_set, g_block_length, 
			       g_block_copy, g_block_reverse, g_block_fill);
  s7_set_object_print_readably(g_block_type, g_block_display_readably);
  s7_define_function(sc, \"make-block\", g_make_block, 1, 0, false, g_make_block_help);
  s7_define_function(sc, \"block\", g_to_block, 0, 0, true, g_block_help);
  s7_define_function(sc, \"subblock\", g_subblock, 1, 0, true, g_subblock_help);
  s7_define_function(sc, \"block-append\", g_block_append, 0, 0, true, g_block_append_help);
  s7_define_function(sc, \"block-reverse!\", g_block_reverse_in_place, 1, 0, true, g_block_reverse_in_place_help);
  s7_define_function(sc, \"block?\", g_is_block, 1, 0, false, g_is_block_help);
  s7_define_function_star(sc, \"blocks\", g_blocks, \"(frequency 4) (scaler 1)\", \"test for function*\");
  g_block_methods = s7_eval_c_string(sc, \"(openlet (inlet 'float-vector? (lambda (p) #t) \
						           'object->let (lambda (p e) (varlet e :empty (zero? (length p)))) \
                                                           'subsequence subblock \
						           'append block-append \
						           'reverse! block-reverse!))\");
  s7_gc_protect(sc, g_block_methods);
  s7_object_type_set_xf(g_block_type, NULL, NULL, block_rf, block_set_rf);
  s7_object_type_set_direct(g_block_type, block_direct_ref, block_direct_set);

  s7_define_safe_function(sc, \"function-open-output\", fout_open, 0, 0, false, \"\");
  s7_define_safe_function(sc, \"function-get-output\", fout_get_output, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"function-close-output\", fout_close, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"function-open-input\", fin_open, 1, 0, false, \"\");

  s7_define_function_with_setter(sc, \"dilambda-test\", g_dilambda_test, g_dilambda_set_test, 0, 0, \"dilambda-test info\");

  s7_define_safe_function(sc, \"hash_heq\", g_heq, 2, 0, false, \"hash-table test\");
  s7_define_safe_function(sc, \"hash_hloc\", g_hloc, 1, 0, false, \"hash-table test\");

  s7_define_safe_function(sc, \"cf10\", g_cf10, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"cf11\", g_cf11, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"cs11\", g_cs11, 1, 0, false, \"\");
  s7_define_safe_function(sc, \"rs11\", g_rs11, 1, 0, false, \"\");

  s7_define_safe_function(sc, \"cf20\", g_cf20, 2, 0, false, \"\");
  s7_define_safe_function(sc, \"cf21\", g_cf21, 2, 0, false, \"\");
  s7_define_safe_function(sc, \"cf22\", g_cf22, 2, 0, false, \"\");

  s7_define_safe_function(sc, \"cf30\", g_cf30, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf31\", g_cf31, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf32\", g_cf32, 3, 0, false, \"\");
  s7_define_safe_function(sc, \"cf33\", g_cf33, 3, 0, false, \"\");

  s7_define_safe_function(sc, \"cf41\", g_cf41, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf42\", g_cf42, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf43\", g_cf43, 4, 0, false, \"\");
  s7_define_safe_function(sc, \"cf44\", g_cf44, 4, 0, false, \"\");

  s7_define_safe_function(sc, \"cf51\", g_cf51, 5, 0, false, \"\");

  s7_define_function(sc, \"sload\",  sload,        1, 1, false, \"test s7_load\");
  s7_define_function(sc, \"scall\",  scall,        2, 0, false, \"test s7_call\");
  s7_define_function(sc, \"sread\",  sread,        0, 1, false, \"test s7_read\");
  s7_define_function(sc, \"swind\",  swind,        3, 0, false, \"test s7_dynamic_wind\");
  s7_define_function(sc, \"seval\",  seval,        1, 1, false, \"test s7_eval\");
  s7_define_function(sc, \"sevalstr\",  sevalstr,  1, 1, false, \"test s7_eval_c_string\");
}
")))

(cond ((provided? 'osx)
       (system "gcc -c s7test-block.c -O2")
       (system "gcc s7test-block.o -o s7test-block.so -dynamic -bundle -undefined suppress -flat_namespace"))

      ((or (provided? 'freebsd)
	   (provided? 'netbsd))
       (system "cc -fPIC -c s7test-block.c -O2")
       (system "cc s7test-block.o -shared -o s7test-block.so -lm -lc"))

      ((provided? 'openbsd)
       (system "gcc -fPIC -ftrampolines -c s7test-block.c -O2")
       (system "gcc s7test-block.o -shared -o s7test-block.so -lm -lc"))

      ((provided? 'solaris)
       (system "gcc -fPIC -c s7test-block.c")
       (system "gcc s7test-block.o -shared -o s7test-block.so -G -ldl -lm"))

      (else 
       (system "gcc -fPIC -c s7test-block.c -O2")
       (system "gcc s7test-block.o -shared -o s7test-block.so -ldl -lm -Wl,-export-dynamic")))

(let ((new-env (sublet (curlet) (cons 'init_func 'block_init)))) ; load calls init_func
  (load "s7test-block.so" new-env))

(define _c_obj_ (make-block 16))
)                               ; with-block

(define _c_obj_ (c-pointer 0))) ; not with-block

(define _null_ (c-pointer 0))

(when (and (provided? 'linux)
	   (not (provided? 'gmp)))
  (system "gcc -o ffitest ffitest.c -g3 -Wall s7.o -lm -I. -ldl")
  (system "ffitest"))


(when with-block
;(define eval seval)           ; finished ok
;(define dynamic-wind swind)   ; finished ok
;(define read sread)           ; ok until (unsupported) let-as-port 80480, also read-error in sevalstr does not unwind C stack
;(define load sload)           ; ok until (unsupported) cload
;(define-macro (test tst expected) `(ok? ',tst (lambda () (define (_s7_ _a_) ,tst) (scall _s7_ (list 0))) ,expected)) ; finished ok!
;(define eval-string sevalstr) ; ok except various local exiters, eval-string as method, multiple statements etc, gloms up C stack
(test (sevalstr "(+ 1 2)") 3)
(test (sevalstr "(+ 1 #())") 'error)
(test (let ((x 1) (y 2)) (sevalstr "(begin (set! x 3) (set! y 4))") (list x y)) '(3 4))
(test (+ 1 (values (sevalstr "(catch #t (lambda () asdf) (lambda args 2))") (sevalstr "(catch #t (lambda () asdf) (lambda args 3))"))) 6)
(test (seval '(+ 1 #())) 'error)
)

#|
(let ()
  (if (null? (hook-functions *error-hook*))
      (set! (hook-functions *error-hook*)
	    (list (lambda (hook)
		    (apply format *stderr* (hook 'data))
		    (newline *stderr*)
		    (throw 'all-done)))))
  (catch 'all-done
    (lambda ()
      (+ 1 #()))
    (lambda args
      (format *stderr* "in catch~%"))))

(define (ok1? otst ola oexp)
  (let ((result (catch 'all-done ola
		       (lambda args
			 (if (not (eq? oexp 'error)) 
			     (begin (display args) (newline)))
			 'error))))
    (if (not (equal? result oexp))
	(format-logged #t "~A: ~A got ~S but expected ~S~%~%" (port-line-number) otst result oexp))))

(define-macro (test tst expected)
  `(ok1? ',tst (lambda () 
		 (if (null? (hook-functions *error-hook*))
		     (set! (hook-functions *error-hook*)
			   (list (lambda (hook)
				   (apply format *stderr* (hook 'data))
				   (newline *stderr*)
				   (set! (hook 'result) 'error)
				   (throw 'all-done)))))
		 ,tst) 
	 ,expected))
|#


;;; --------------------------------------------------------------------------------
;;; eq?

(test (eq? 'a 3) #f)
(test (eq? #t 't) #f)
(test (eq? "abs" 'abc) #f)
(test (eq? "hi" '(hi)) #f)
(test (eq? "hi" "hi") #f)
(test (eq? "()" ()) #f)
(test (eq? '(1) '(1)) #f)
(test (eq? '(#f) '(#f)) #f)
(test (eq? #\a #\b) #f)
(test (eq? #t #t) #t)
(test (eq? #f #f) #t)
(test (eq? #f #t) #f)
(test (eq? (null? ()) #t) #t)
(test (eq? (null? '(a)) #f) #t)
(test (eq? (cdr '(a)) ()) #t)
(test (eq? 'a 'a) #t)
(test (eq? 'a 'b) #f)
(test (eq? 'a (string->symbol "a")) #t)
(test (eq? (symbol "a") (string->symbol "a")) #t)
(test (eq? :a :a) #t)
(test (eq? ':a 'a) #f)
(test (eq? ':a ':a) #t)
(test (eq? :a a:) #f)
(test (eq? ':a 'a:) #f)
(test (eq? 'a: 'a:) #t)
(test (eq? ':a: 'a:) #f)
(test (eq? 'a (symbol "a")) #t)
(test (eq? :: '::) #t)
(test (eq? ':a (symbol->keyword (symbol "a"))) #t) ; but not a:
(test (eq? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eq? x x)) #t)
(test (let ((x (cons 'a 'b))) (eq? x x)) #t)
(test (eq? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eq? "abc" "cba") #f)
(test (let ((x "hi")) (eq? x x)) #t)
(test (eq? (string #\h #\i) (string #\h #\i)) #f)
(test (eq? #(a) #(b)) #f)
(test (let ((x (vector 'a))) (eq? x x)) #t)
(test (eq? (vector 'a) (vector 'a)) #f)
(test (eq? car car) #t)
(test (eq? car cdr) #f)
(test (let ((x (lambda () 1))) (eq? x x)) #t)
(test (let ((x (lambda () 1))) (let ((y x)) (eq? x y))) #t)
(test (let ((x (lambda () 1))) (let ((y (lambda () 1))) (eq? x y))) #f)
(test (eq? 'abc 'abc) #t)
(test (eq? eq? eq?) #t)
(test (eq? (if #f 1) 1) #f)
(test (eq? () '(#||#)) #t)
(test (eq? () '(#|@%$&|#)) #t)
(test (eq? '#||#hi 'hi) #t) ; ??
(test (eq? '; a comment
         hi 'hi) #t) ; similar:
    (test (cadr '#| a comment |#(+ 1 2)) 1)
    (test `(+ 1 ,@#||#(list 2 3)) '(+ 1 2 3))
    (test `(+ 1 ,#||#(+ 3 4)) '(+ 1 7))
    ;; but not splitting the ",@" or splitting a number:
    (test (+ 1 2.0+#||#3i) 'error)
    (test `(+ 1 ,#||#@(list 2 3)) 'error)
(test (eq? #||# (#|%%|# append #|^|#) #|?|# (#|+|# list #|<>|#) #||#) #t)
(test (eq? '() ;a comment
	   '()) #t)
(test (eq? 3/4 3) #f)
(test (eq? '() '()) #t)
(test (eq? '() '(  )) #t)
(test (eq? '()'()) #t)
(test (eq? '()(list)) #t)
(test (eq? () (list)) #t)
(test (eq? (begin) (append)) #t)
(test (let ((lst (list 1 2 3))) (eq? lst (apply list lst))) #f) ; changed 26-Sep-11

;(test (eq? 1/0 1/0) #f)
;(test (let ((+nan.0 1/0)) (eq? +nan.0 +nan.0)) #f)
;; these are "unspecified" so any boolean value is ok

(test (eq? ''2 '2) #f)
(test (eq? '2 '2) #t) ; unspecified??
(test (eq? '2 2) #t)
(test (eq? ''2 ''2) #f)
(test (eq? ''#\a '#\a) #f)
(test (eq? '#\a #\a) #t) ; was #f 
(test (eq? 'car car) #f)
(test (eq? '()()) #t)
(test (eq? ''() '()) #f)
(test (eq? '   () '
()) #t)
(test (eq? '#f #f) #t)
(test (eq? '#f '#f) #t)
(test (eq? #f '  #f) #t)
(test (eq? '()'()) #t) ; no space
(test (#||# eq? #||# #f #||# #f #||#) #t)
(test (eq? (current-input-port) (current-input-port)) #t)
(test (let ((f (lambda () (quote (1 . "H"))))) (eq? (f) (f))) #t)
(test (let ((f (lambda () (cons 1 (string #\H))))) (eq? (f) (f))) #f)
(test (eq? *stdin* *stdin*) #t)
(test (eq? *stdout* *stderr*) #f)
(test (eq? *stdin* *stderr*) #f)
(test (eq? else else) #t)
(test (eq? :else else) #f)
(test (eq? :else 'else) #f)
(test (eq? :if if) #f)
(test (eq? 'if 'if) #t)
(test (eq? :if :if) #t)

(test (eq? (string) (string)) #f)
(test (eq? (string) "") #f)
(test (eq? (vector) (vector)) #f)
(test (eq? (vector) #()) #f)
(test (eq? (list) (list)) #t)
(test (eq? (list) ()) #t)
(test (eq? (hash-table) (hash-table)) #f)
(test (eq? (curlet) (curlet)) #t)
(test (eq? (rootlet) (rootlet)) #t)
(test (eq? (funclet abs) (funclet abs)) #t) ; or any other built-in...
(test (eq? letrec* letrec*) #t)

(test (eq? (current-input-port) (current-input-port)) #t)
(test (eq? (current-error-port) (current-error-port)) #t)
(test (eq? (current-output-port) (current-output-port)) #t)
(test (eq? (current-input-port) (current-output-port)) #f)

(test (eq? (string #\a) (string #\a)) #f)
(test (eq? "a" "a") #f)
(test (eq? #(1) #(1)) #f)
(test (let ((a "hello") (b "hello")) (eq? a b)) #f)
(test (let ((a "foo")) (eq? a (copy a))) #f)
(test (let ((p (c-pointer 0))) (eq? p (copy p))) #f)
(test (let ((p (c-pointer 0))) (let ((p1 p)) (eq? p p1))) #t)

(begin #| ; |# (display ""))
(newline)

(test (;
       eq? ';!
       (;)()#
	);((")";
       ;"#|)#""
       '#|";"|#(#|;|#); ;#
	 ;\;"#"#f 
	       )#t)

(test (+ #| this is a comment |# 2 #| and this is another |# 3) 5)
(test (eq? #| a comment |# #f #f) #t)
(test (eq? #| a comment |##f #f) #t)  ; ??
(test (eq? #| a comment | ##f|##f #f) #t) ; ??
(test (eq? #||##||##|a comment| ##f|##f #f) #t)

(test (+ ;#|
            3 ;|#
            4)
      7)
(test (+ #| ; |# 3
		 4)
      7)

(test (eq? (if #f #t) (if #f 3)) #t)

(test (eq?) 'error)           ; "this comment is missing a double-quote
(test (eq? #t) 'error)        #| "this comment is missing a double-quote |#
(test (eq? #t #t #t) 'error)  #| and this has redundant starts #| #| |#
(test (eq? #f . 1) 'error)
(test (eq #f #f) 'error)

(let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector) (vector 1) (list 1) 'f 't #\t)))
  (let ((len (length things)))
    (do ((i 0 (+ i 1)))
	((= i (- len 1)))
    (do ((j (+ i 1) (+ j 1)))
	((= j len))
      (if (eq? (vector-ref things i) (vector-ref things j))
	  (format-logged #t ";(eq? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j)))))))

;;; these are defined at user-level in s7 -- why are other schemes so coy about them?
(test (eq? (if #f #f) #<unspecified>) #t)
(test (eq? (symbol->value '_?__undefined__?_) #<undefined>) #t)
(test (eq? #<eof> #<eof>) #t)
(test (eq? #<undefined> #<undefined>) #t)
(test (eq? #<unspecified> #<unspecified>) #t)
(test (eq? #<eof> #<undefined>) #f)
(test (eq? #<eof> ()) #f)

(test (let () (define-macro (hi a) `(+ 1 ,a)) (eq? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eq? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eq? x x)) #t)

(test (eq? quasiquote quasiquote) #t)
(test (eq? `quasiquote 'quasiquote) #t)
(test (eq? 'if (keyword->symbol :if)) #t)
(test (eq? 'if (string->symbol "if")) #t)
(test (eq? (copy lambda) (copy 'lambda)) #f)
(test (eq? if 'if) #f)
(test (eq? if `if) #f)
(test (eq? if (keyword->symbol :if)) #f)
(test (eq? if (string->symbol "if")) #f)
(test (eq? lambda and) #f)
(test (eq? let let*) #f)
(test (eq? quote quote) #t)
(test (eq? '"hi" '"hi") #f) ; guile also
;(test (eq? '"" "") #f)
;(test (eq? '"" '"") #f)
;(test (eq? "" "") #f)
(test (eq? #() #()) #f)
(test (eq? '#() #()) #f)
(test (eq? '#() '#()) #f)
(test (let ((v #())) (eq? v #())) #f)
(test (let ((v #())) (eq? v #())) #f)
(test (let ((v #())) (eq? v v)) #t)
(test (call/cc (lambda (return) (return (eq? return return)))) #t)
(test (let ((r #f)) (call/cc (lambda (return) (set! r return) #f)) (eq? r r)) #t)
(test (eq? _unbound_variable_ #f) 'error)

(when with-block
  (let ((b (make-block 4))) 
    (test (eq? b b) #t)
    (test (equal? b b) #t)
    (test (block? b) #t)
    (test (block? #()) #f)
    (test (block? #f) #f)
    (set! (b 0) 32)
    (test (b 0) 32.0)
    (let () (define (hi b i) (b i)) (hi b 0) (test (hi b 0) 32.0))
    (let () (define (hi b) (b 0)) (hi b) (test (hi b) 32.0))
    (let () (define (hi b) (b)) (hi b) (test (hi b) 32))
    (test b (block 32.0 0.0 0.0 0.0))
    (test (object->string b) "(block 32.000 0.000 0.000 0.000)")
    (let ((b1 (make-block 4)))
      (test (eq? b b1) #f)))
  (test (blocks) (list 4 1))
  (test (blocks :frequency 2) (list 2 1))
  (test (blocks :scaler 3 :frequency 2) (list 2 3))
  (test (blocks :scaler 3 :phase 1) 'error)
  (test (map blocks '(1 2 3)) '((1 1) (2 1) (3 1)))
  (test (map blocks '( 1 2 3) '(4 5 6)) '((1 4) (2 5) (3 6)))
  (test (procedure-documentation blocks) "test for function*")
  (test (apply blocks '(:frequency 5 :scaler 4)) '(5 4))
  (test (let () (define (b1) (blocks 100)) (b1)) '(100 1))
  (test (procedure? blocks) #t)

  (define (fv32)
    (let ((b (block 1 2 3 4))
	  (f (make-float-vector 4)))
      (do ((i 0 (+ i 1)))
	  ((= i 4) f)
	(set! (f i) (+ (b i) 1.0)))))
  (test (fv32) (float-vector 2.0 3.0 4.0 5.0))

  (define (fv33)
    (let ((b (block 1 2 3 4))
	  (f (make-block 4)))
      (do ((i 0 (+ i 1)))
	  ((= i 4) f)
	(set! (f i) (+ (b i) 1.0)))))
  (test (fv33) (block 2.0 3.0 4.0 5.0))
  )

(test (c-pointer? 0) #f)
(test (c-pointer? _null_) #t)
(if with-block
    (test (integer? (c-object? _c_obj_)) #t)
    (test (c-pointer? _c_obj_) #t))

(for-each
 (lambda (arg)
   (test (c-pointer? arg) #f)
   (test (c-object? arg) #f))
 (list "hi" () (integer->char 65) #f #t 0+i '(1 2) _ht_ 'a-symbol (cons 1 2) (make-vector 3) abs 
       #<eof> '(1 2 3) #\newline (lambda (a) (+ a 1)) #<unspecified> #<undefined>))

(test (c-pointer?) 'error)
(test (c-object?) 'error)
(test (c-pointer? _c_obj_ 2) 'error)
(test (c-object? _c_obj_ 2) 'error)
(when with-bignums
  (test (c-pointer? (c-pointer (bignum "12341234"))) #t)
  (test (c-pointer (bignum "1.4")) 'error))

(when with-block
  (test (pair? (*s7* 'c-types)) #t))


;;; a ridiculous optimizer typo...
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a a))) #f)
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a . a))) #t)
(test (let ((sym 'a)) (define (hi a) (eq? (cdr a) sym)) (hi '(a . b))) #f)

(for-each
 (lambda (arg)
   (let ((x arg)
	 (y arg))
     (if (not (eq? x x))
	 (format-logged #t ";(eq? x x) of ~A -> #f?~%" x))
     (if (not (eq? x arg))
	 (format-logged #t ";(eq? x arg) of ~A ~A -> #f?~%" x arg))
     (if (not (eq? x y))
	 (format-logged #t ";(eq? x y) of ~A ~A -> #f?~%" x y))))
 ;; actually I hear that #f is ok here for numbers
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3/4 #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))
;; this used to include 3.14 and 1+i but that means the (eq? x x) case differs from the (eq? 3.14 3.14) case

(define comment-start (port-line-number))
#|
:'(1(1))
(1 (1))
:'(1#(1))
(1# (1))
|#
(if (not (= (- (port-line-number) comment-start) 7)) (format *stderr* ";block comment newline counter: ~D ~D~%" comment-start (port-line-number)))

;;; this comes from G Sussman
(let ()
  (define (counter count)
    (lambda ()
      (set! count (+ 1 count))
      count))

  (define c1 (counter 0))
  (define c2 (counter 0))

  (test (eq? c1 c2) #f)
  (test (eq? c1 c1) #t)
  (test (eq? c2 c2) #t)

  (test (let ((p (lambda (x) x))) (eqv? p p)) #t)
  (for-each
   (lambda (arg)
     (if (not ((lambda (p) (eq? p p)) arg))
	 (format-logged #t "~A not eq? to itself?~%" arg)))
   (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined> '(1 2 . 3)
	 (let ((lst (list 1 2)))
	   (set! (cdr (cdr lst)) lst)
	   lst)
	 (vector) (string) (list)
	 (let ((x 3))
	   (lambda (y) (+ x y))))))

;;; this for r7rs
(test (eq? #t #true) #t)
(test (eq? #f #false) #t)
(test (eq? () (map values ())) #t)

(let () (define (f2) f2) (test (eq? f2 (f2)) #t))
(letrec ((f2 (lambda () f2))) (test (eq? f2 (f2)) #t))




;;; --------------------------------------------------------------------------------
;;; eqv?

(test (eqv? 'a 3) #f)
(test (eqv? #t 't) #f)
(test (eqv? "abs" 'abc) #f)
(test (eqv? "hi" '(hi)) #f)
(test (eqv? "()" ()) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '(#f) '(#f)) #f)
(test (eqv? #\a #\b) #f)
(test (eqv? #\a #\a) #t)
(test (eqv? (integer->char 255) (string-ref (string #\x (integer->char 255) #\x) 1)) #t)
(test (eqv? (integer->char #xf0) (integer->char #x70)) #f)
(test (eqv? #\space #\space) #t)
(test (let ((x (string-ref "hi" 0))) (eqv? x x)) #t)
(test (eqv? #t #t) #t)
(test (eqv? #f #f) #t)
(test (eqv? #f #t) #f)
(test (eqv? (null? ()) #t) #t)
(test (eqv? (null? '(a)) #f) #t)
(test (eqv? (cdr '(a)) '()) #t)
(test (eqv? 'a 'a) #t)
(test (eqv? 'a 'b) #f)
(test (eqv? 'a (string->symbol "a")) #t)
(test (eqv? '(a) '(b)) #f)
(test (let ((x '(a . b))) (eqv? x x)) #t)
(test (let ((x (cons 'a 'b))) (eqv? x x)) #t)
(test (eqv? (cons 'a 'b) (cons 'a 'b)) #f)
(test (eqv? "abc" "cba") #f)
(test (let ((x "hi")) (eqv? x x)) #t)
(test (eqv? (string #\h #\i) (string #\h #\i)) #f)
(test (eqv? #(a) #(b)) #f)
(test (let ((x (vector 'a))) (eqv? x x)) #t)
(test (eqv? (vector 'a) (vector 'a)) #f)
(test (eqv? car car) #t)
(test (eqv? car cdr) #f)
(test (let ((x (lambda () 1))) (eqv? x x)) #t)
(test (eqv? (lambda () 1) (lambda () 1)) #f)
(test (let () (define (make-adder x) (lambda (y) (+ x y))) (eqv? (make-adder 1) (make-adder 1))) #f)
(test (eqv? 9/2 9/2) #t)
(test (eqv? quote quote) #t)
(test (eqv? () ()) #t)
(test (eqv? () '()) #t)
;(test (eqv? "" "") #f)
(test (eqv? "hi" "hi") #f) ; unspecified 
(test (eqv? #() #()) #f)   ; unspecified, but in s7 (eqv? () ()) is #t

(let ((c1 (let ((x 32))
	    (lambda () x)))
      (c2 (let ((x 123))
	    (lambda () x))))
  (test (eqv? c1 c2) #f)
  (test (eqv? c1 c1) #t))

(test (eqv? most-positive-fixnum most-positive-fixnum) #t)
(test (eqv? most-positive-fixnum most-negative-fixnum) #f)
(test (eqv? 9223372036854775807 9223372036854775806) #f)
(test (eqv? 9223372036854775807 -9223372036854775808) #f)
(test (eqv? -9223372036854775808 -9223372036854775808) #t)
(test (eqv? 123456789/2 123456789/2) #t)
(test (eqv? 123456789/2 123456787/2) #f)
(test (eqv? -123456789/2 -123456789/2) #t)
(test (eqv? 2/123456789 2/123456789) #t)
(test (eqv? -2/123456789 -2/123456789) #t)
(test (eqv? 2147483647/2147483646 2147483647/2147483646) #t)
(test (eqv? 3/4 12/16) #t)
(test (eqv? 1/1 1) #t)
(test (eqv? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (eqv? x x)) #t)
(test (let ((x 1+i)) (eqv? x x)) #t)
(test (let* ((x 3.141) (y x)) (eqv? x y)) #t)
(test (let* ((x 1+i) (y x)) (eqv? x y)) #t)
(test (let* ((x 3/4) (y x)) (eqv? x y)) #t)
(test (eqv? 1.0 1.0) #t)
(test (eqv? 0.6 0.6) #t)
(test (eqv? 0.6 0.60) #t)
(test (eqv? 1+i 1+i) #t)
(test (eqv? -3.14 -3.14) #t)
(test (eqv? 1e2 1e2) #t)
(test (eqv? #i3/5 #i3/5) #t)
(test (eqv? #e0.6 #e0.6) #t)
(test (eqv? 1 1.0) #f)
(test (eqv? 1/2 0.5) #f)
(test (eqv? 1 1/1) #t)
(test (eqv? 0.5 5e-1) #t)
(test (eqv? 1/0 1/0) #f)
(test (let ((+nan.0 1/0)) (eqv? +nan.0 +nan.0)) #f)

(test (eqv? (cons 'a 'b) (cons 'a 'c)) #f)
(test (eqv? eqv? eqv?) #t)
(test (eqv? #(1) #(1)) #f)
(test (eqv? '(1) '(1)) #f)
(test (eqv? '() '()) #t)
(test (eqv? '() (list)) #t)
(test (eqv? '(()) '(())) #f)
(test (eqv? (list 'abs 'cons) '(abs cons)) #f)

(let ((things (vector #t #f #\space '() "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector) (vector 1) (list 1) 'f 't #\t)))
  (let ((len (length things)))
    (do ((i 0 (+ i 1)))
	((= i (- len 1)))
      (do ((j (+ i 1) (+ j 1)))
	  ((= j len))
	(if (eqv? (vector-ref things i) (vector-ref things j))
	    (format-logged #t ";(eqv? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j)))))))

(test (eqv?) 'error)
(test (eqv? #t) 'error)
(test (eqv? #t #t #t) 'error)
(test (eqv #f #f) 'error)

(test (eqv? ''2 '2) #f)
(test (eqv? '2 '2) #t)
(test (eqv? '2 2) #t)
(test (eqv? ''2 ''2) #f)
(test (eqv? ''#\a '#\a) #f)
(test (eqv? '#\a #\a) #t)
(test (eqv? 'car car) #f)
(test (eqv? ''() '()) #f)
(test (eqv? '#f #f) #t)
(test (eqv? '#f '#f) #t)
(test (eqv? #<eof> #<eof>) #t)
(test (eqv? #<undefined> #<undefined>) #t)
(test (eqv? #<unspecified> #<unspecified>) #t)
(test (eqv? (if #f #f) #<unspecified>) #t)
(test (eqv? #<eof> #<undefined>) #f)
(test (eqv? #<eof> '()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (eqv? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (eqv? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (eqv? x x)) #t)
(test (eqv? else else) #t)
(test (let ((p (lambda (x) x))) (eqv? p p)) #t)

(when with-bignums
  (test (eqv? (bignum "1+i") (bignum "1+i")) #t)
  (test (eqv? (bignum "1+i") 1+i) #t)
  (test (eqv? 1+i (bignum "1+i")) #t)
  (test (eqv? (bignum "2.0") (bignum "2.0")) #t)
  (test (eqv? (bignum "2.0") (bignum "1.0")) #f))

;; from M Weaver:
(test (list (eqv? +0.0 -0.0)
	    (eqv? (complex +0.0  1.0)
		  (complex -0.0  1.0))
	    (eqv? (complex  1.0 +0.0)
		  (complex  1.0 -0.0)))
      '(#t #t #t))
(test (list (eq? +0.0 -0.0)
        (eq? (complex  +0.0  1.0)
              (complex -0.0  1.0))
        (eq? (complex   1.0 +0.0)
              (complex  1.0 -0.0)))
      '(#t #f #f))
(test (list (eq? +0 -0)
        (eq? (complex  +0  1)
              (complex -0  1))
        (eq? (complex   1 +0)
              (complex  1 -0)))
      '(#t #f #t))




;;; --------------------------------------------------------------------------------
;;; equal?

(test (equal? 'a 3) #f)
(test (equal? #t 't) #f)
(test (equal? "abs" 'abc) #f)
(test (equal? "hi" '(hi)) #f)
(test (equal? "()" '()) #f)
(test (equal? '(1) '(1)) #t)
(test (equal? '(#f) '(#f)) #t)
(test (equal? '(()) '(() . ())) #t)
(test (equal? #\a #\b) #f)
(test (equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (equal? x x)) #t)
(test (equal? #t #t) #t)
(test (equal? #f #f) #t)
(test (equal? #f #t) #f)
(test (equal? (null? '()) #t) #t)
(test (equal? (null? '(a)) #f) #t)
(test (equal? (cdr '(a)) '()) #t)
(test (equal? 'a 'a) #t)
(test (equal? 'a 'b) #f)
(test (equal? 'a (string->symbol "a")) #t)
(test (equal? '(a) '(b)) #f)
(test (equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (equal? x x)) #t)
(test (equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (equal?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (equal? "abc" "cba") #f)
(test (equal? "abc" "abc") #t)
(test (let ((x "hi")) (equal? x x)) #t)
(test (equal? (string #\h #\i) (string #\h #\i)) #t)
(test (equal? #(a) #(b)) #f)
(test (equal? #(a) #(a)) #t)
(test (let ((x (vector 'a))) (equal? x x)) #t)
(test (equal? (vector 'a) (vector 'a)) #t)
(test (equal? #(1 2) (vector 1 2)) #t)
(test (equal? #(1.0 2/3) (vector 1.0 2/3)) #t)
(test (equal? #(1 2) (vector 1 2.0)) #f) ; 2 not equal 2.0!
(test (equal? '(1 . 2) (cons 1 2)) #t)
(test (equal? '(1 #||# . #||# 2) (cons 1 2)) #t)
(test (- '#||#1) -1) ; hmm
(test (equal? #(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (equal? #((1 . 2)) (vector (cons 1 2))) #t)
(test (equal? #(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (equal? #(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (equal? #(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (equal? #(()) (vector '())) #t)
(test (equal? #("hi" "ho") (vector "hi" '"ho")) #t)
(test (equal? `#(1) #(1)) #t)
(test (equal? ``#(1) #(1)) #t)
(test (equal? '`#(1) #(1)) #t)
(test (equal? ''#(1) #(1)) #f)
(test (equal? ''#(1) '#(1)) #f)
(test (equal? '(1) '        (   1    )) #t)
(test (equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (equal? (list 1 2) '(1 2.0)) #f)
(test (equal? #(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (equal? '((())) (list (list (list)))) #t)
(test (equal? '((())) (cons (cons () ()) ())) #t)
(test (equal? car car) #t)
(test (equal? car cdr) #f)
(test (let ((x (lambda () 1))) (equal? x x)) #t)
(test (equal? (lambda () 1) (lambda () 1)) #f)
(test (equal? 9/2 9/2) #t)
(test (equal? #((())) #((()))) #t)
(test (equal? "123""123") #t);no space
(test (equal? """") #t)#|nospace|#
(test (equal? #()#()) #t)
(test (equal? #()()) #f)
(test (equal? ()"") #f)
(test (equal? "hi""hi") #t)
(test (equal? #<eof> #<eof>) #t)
(test (equal? #<undefined> #<undefined>) #t)
(test (equal? #<unspecified> #<unspecified>) #t)
(test (equal? (if #f #f) #<unspecified>) #t)
(test (equal? #<eof> #<undefined>) #f)
(test (equal? (values) #<eof>) #f)
(test (equal? (values) (values)) #t)
(test (equal? #<eof> #<unspecified>) #f)
(test (equal? (values) #<unspecified>) #t)
(test (equal? #<unspecified> (values)) #t)
(test (equal? #<eof> ()) #f)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (equal? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (equal? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (equal? x x)) #t)
(test (equal? ``"" '"") #t)
(test (let ((pws (dilambda (lambda () 1) (lambda (x) x)))) (equal? pws pws)) #t)
(test (equal? if :if) #f)
(test (equal? (list 'abs 'cons) '(abs cons)) #t)
(test (equal? '(1) '(list 1)) #f)

(test (equal? (values) #<unspecified>) #t)
(test (equal? (list (values)) (list #<unspecified>)) #t)

(test (equal? most-positive-fixnum most-positive-fixnum) #t)
(test (equal? most-positive-fixnum most-negative-fixnum) #f)
(test (equal? pi pi) #t)
(test (equal? 9223372036854775807 9223372036854775806) #f)
(test (equal? 9223372036854775807 -9223372036854775808) #f)
(test (equal? -9223372036854775808 -9223372036854775808) #t)
(test (equal? 123456789/2 123456789/2) #t)
(test (equal? 123456789/2 123456787/2) #f)
(test (equal? -123456789/2 -123456789/2) #t)
(test (equal? 2/123456789 2/123456789) #t)
(test (equal? -2/123456789 -2/123456789) #t)
(test (equal? 2147483647/2147483646 2147483647/2147483646) #t)
(test (equal? 3/4 12/16) #t)
(test (equal? 1/1 1) #t)
(test (equal? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (equal? x x)) #t)
(test (let ((x 1+i)) (equal? x x)) #t)
(test (let* ((x 3.141) (y x)) (equal? x y)) #t)
(test (let* ((x 1+i) (y x)) (equal? x y)) #t)
(test (let* ((x 3/4) (y x)) (equal? x y)) #t)
(test (equal? '(+ '1) '(+ 1)) #f) ; !?

(test (equal? '(1/0) '(1/0)) #f)
(test (equal? '1/0 '1/0) #f) 
(test (let ((+nan.0 1/0)) (equal? '(+nan.0) '(+nan.0))) #t)
(test (let ((+nan.0 1/0)) (equal? (list +nan.0) (list +nan.0))) #f)
;;; in the first case we're looking at the symbol, not its value
(test (let ((+nan.0 1/0)) (equal? (vector +nan.0) (vector +nan.0))) #f)
(test (let ((+nan.0 1/0)) (equal? #(1/0) #(1/0))) #f)

(test (let ((x 3.141)) (equal? x x)) #t)
(test (equal? 3 3) #t)
(test (equal? 3 3.0) #f)
(test (equal? 3.0 3.0) #t)
(test (equal? 3-4i 3-4i) #t)
(test (equal? (string #\c) "c") #t)
(test (equal? equal? equal?) #t)
(test (equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (equal? '() '()) #t)
(test (equal? '() (list)) #t)
(test (equal? (cdr '   ''0) '((quote 0))) #t)
(test (equal? "\n" "\n") #t)
(test (equal? #f ((lambda () #f))) #t)
(test (equal? (+) 0) #t)
(test (equal? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (equal? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (equal? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (equal? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (equal? (recompose 32 (lambda (a) (cons 1 a)) ()) (recompose 32 (lambda (a) (cons 1 a)) ())) #t)
(test (equal? (recompose 32 (lambda (a) (list 1 a)) ()) (recompose 32 (lambda (a) (list 1 a)) ())) #t)

(test (equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (equal? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector 1) (list 1) 'f 't #\t)))
  (let ((len (length things)))
    (do ((i 0 (+ i 1)))
	((= i (- len 1)))
      (do ((j (+ i 1) (+ j 1)))
	  ((= j len))
	(if (equal? (vector-ref things i) (vector-ref things j))
	    (format-logged #t ";(equal? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j)))))))

(test (equal?) 'error)
(test (equal? #t) 'error)
(test (equal? #t #t #t) 'error)
(test (equal #t #t) 'error)

(when with-block
  (let ((b (make-block 4))) 
    (test (equal? b b) #t)
    (let ((b1 (make-block 4)))
      (test (equal? b b1) #t)
      (set! (b 1) 1.0)
      (test (equal? b b1) #f))))
(test (let ((p (c-pointer 0))) (equal? p (copy p))) #t)

(test (call-with-exit (lambda (return) (return (equal? return return)))) #t)
(test (call-with-exit (lambda (return) (call-with-exit (lambda (quit) (return (equal? return quit)))))) #f)
(test (call/cc (lambda (return) (return (equal? return return)))) #t)
(test (let hiho ((i 0)) (equal? hiho hiho)) #t)
(test (let hiho ((i 0)) (let hoho ((i 0)) (equal? hiho hoho))) #f)
(test (equal? + *) #f)
(test (equal? lambda lambda) #t)
(test (equal? lambda lambda*) #f)
(test (equal? let let) #t)
(test (equal? let letrec) #f)
(test (equal? define define) #t)
(test (equal? + ((lambda (a) a) +)) #t)
(test (let ((x "hi")) (define (hi) x) (equal? (hi) (hi))) #t)

;; so (eq? 3/4 3/4) is #f, (eqv? 3/4 3/4) is #t,
;;    (eqv? #(1) #(1)) is #f, (equal? #(1) #(1)) is #t
;;    (equal? 3 3.0) is #f, (= 3 3.0) is #t
;; in s7 
;;    (eq? 0.0 0.0) is #t,
;;    (eq? 2.0 2.0) is #f
(test (equal? .0 0.) #t)
(test (equal? 
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal? 
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (equal? (make-string 3) (make-string 3)) #t)
(test (equal? (make-list 3) (make-list 3)) #t)
(test (equal? (make-vector 3) (make-vector 3)) #t)
(unless with-bignums (test (equal? (random-state 100) (random-state 100)) #t))

(test (equal? (current-input-port) (current-input-port)) #t)
(test (equal? (current-input-port) (current-output-port)) #f)
(test (equal? *stdin* *stderr*) #f)
(test (let ((l1 (list 'a 'b)) 
	    (l2 (list 'a 'b 'a 'b))) 
	(set! (cdr (cdr l1)) l1) 
	(set! (cdr (cdr (cdr (cdr l2)))) l2)
	(equal? l1 l2))
      #t)
(test (let ((l1 (list 'a 'b)) 
	    (l2 (list 'a 'b 'a))) 
	(set! (cdr (cdr l1)) l1) 
	(set! (cdr (cdr (cdr l2))) l2)
	(equal? l1 l2))
      #f)
(test (let ((v1 (vector 1 2 3))
	    (v2 (vector 1 2 3)))
	(set! (v1 1) v1)
	(set! (v2 1) v2)
	(equal? v1 v2))
      #t)
(test (let ((v1 (vector 1 2 3))
	    (v2 (vector 1 2 4)))
	(set! (v1 1) v1)
	(set! (v2 1) v2)
	(equal? v1 v2))
      #f)

(when with-bignums
  (test (equal? (/ (* 5 most-positive-fixnum) (* 3 most-negative-fixnum)) -46116860184273879035/27670116110564327424) #t))



;;; --------------------------------------------------------------------------------
;;; morally-equal?

(test (morally-equal? 'a 3) #f)
(test (morally-equal? #t 't) #f)
(test (morally-equal? "abs" 'abc) #f)
(test (morally-equal? "hi" '(hi)) #f)
(test (morally-equal? "()" '()) #f)
(test (morally-equal? '(1) '(1)) #t)
(test (morally-equal? '(#f) '(#f)) #t)
(test (morally-equal? '(()) '(() . ())) #t)
(test (morally-equal? #\a #\b) #f)
(test (morally-equal? #\a #\a) #t)
(test (let ((x (string-ref "hi" 0))) (morally-equal? x x)) #t)
(test (morally-equal? #t #t) #t)
(test (morally-equal? #f #f) #t)
(test (morally-equal? #f #t) #f)
(test (morally-equal? (null? '()) #t) #t)
(test (morally-equal? (null? '(a)) #f) #t)
(test (morally-equal? (cdr '(a)) '()) #t)
(test (morally-equal? 'a 'a) #t)
(test (morally-equal? 'a 'b) #f)
(test (morally-equal? 'a (string->symbol "a")) #t)
(test (morally-equal? '(a) '(b)) #f)
(test (morally-equal? '(a) '(a)) #t)
(test (let ((x '(a . b))) (morally-equal? x x)) #t)
(test (let ((x (cons 'a 'b))) (morally-equal? x x)) #t)
(test (morally-equal? (cons 'a 'b) (cons 'a 'b)) #t)
(test (morally-equal?(cons 'a 'b)(cons 'a 'b)) #t) ; no space
(test (morally-equal? "abc" "cba") #f)
(test (morally-equal? "abc" "abc") #t)
(test (let ((x "hi")) (morally-equal? x x)) #t)
(test (morally-equal? (string #\h #\i) (string #\h #\i)) #t)
(test (morally-equal? #(a) #(b)) #f)
(test (morally-equal? #(a) #(a)) #t)
(test (let ((x (vector 'a))) (morally-equal? x x)) #t)
(test (morally-equal? (vector 'a) (vector 'a)) #t)
(test (morally-equal? #(1 2) (vector 1 2)) #t)
(test (morally-equal? #(1.0 2/3) (vector 1.0 2/3)) #t)
(test (morally-equal? #(1 2) (vector 1 2.0)) #t)
(test (morally-equal? '(1 . 2) (cons 1 2)) #t)
(test (morally-equal? '(1 #||# . #||# 2) (cons 1 2)) #t)
(test (- '#||#1) -1) ; hmm
(test (morally-equal? #(1 "hi" #\a) (vector 1 "hi" #\a)) #t)
(test (morally-equal? #((1 . 2)) (vector (cons 1 2))) #t)
(test (morally-equal? #(1 "hi" #\a (1 . 2)) (vector 1 "hi" #\a (cons 1 2))) #t)
(test (morally-equal? #(#f hi (1 2) 1 "hi" #\a (1 . 2)) (vector #f 'hi (list 1 2) 1 "hi" #\a (cons 1 2))) #t)
(test (morally-equal? #(#(1) #(1)) (vector (vector 1) (vector 1))) #t)
(test (morally-equal? #(()) (vector '())) #t)
(test (morally-equal? #("hi" "ho") (vector "hi" '"ho")) #t)
(test (morally-equal? `#(1) #(1)) #t)
(test (morally-equal? ``#(1) #(1)) #t)
(test (morally-equal? '`#(1) #(1)) #t)
(test (morally-equal? ''#(1) #(1)) #f)
(test (morally-equal? ''#(1) '#(1)) #f)
(test (morally-equal? (list 1 "hi" #\a) '(1 "hi" #\a)) #t)
(test (morally-equal? (list 1.0 2/3) '(1.0 2/3)) #t)
(test (morally-equal? (list 1 2) '(1 2.0)) #t)
(test (morally-equal? #(1.0+1.0i) (vector 1.0+1.0i)) #t)
(test (morally-equal? (list 1.0+1.0i) '(1.0+1.0i)) #t)
(test (morally-equal? '((())) (list (list (list)))) #t)
(test (morally-equal? car car) #t)
(test (morally-equal? car cdr) #f)
(test (let ((x (lambda () 1))) (morally-equal? x x)) #t)
(test (morally-equal? (lambda () 1) (lambda () 1)) #t)
(test (morally-equal? 9/2 9/2) #t)
(test (morally-equal? #((())) #((()))) #t)
(test (morally-equal? "123""123") #t);no space
(test (morally-equal? """") #t)#|nospace|#
(test (morally-equal? #()#()) #t)
(test (morally-equal? #()()) #f)
(test (morally-equal? ()"") #f)
(test (morally-equal? "hi""hi") #t)
(test (morally-equal? #<eof> #<eof>) #t)
(test (morally-equal? #<undefined> #<undefined>) #t)
(test (morally-equal? #<unspecified> #<unspecified>) #t)
(test (morally-equal? (if #f #f) #<unspecified>) #t)
(test (morally-equal? #<eof> #<undefined>) #f)
(test (morally-equal? #<eof> '()) #f)
(test (morally-equal? (values) #<eof>) #f)
(test (morally-equal? #<eof> (values)) #f)
(test (morally-equal? (values) (values)) #t)
(test (morally-equal? #<eof> #<unspecified>) #f)
(test (morally-equal? (values) #<unspecified>) #t)
(test (morally-equal? #<unspecified> (values)) #t)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (morally-equal? hi hi)) #t)
(test (let () (define (hi a) (+ 1 a)) (morally-equal? hi hi)) #t)
(test (let ((x (lambda* (hi (a 1)) (+ 1 a)))) (morally-equal? x x)) #t)
(test (morally-equal? ``"" '"") #t)
(test (let ((pws (dilambda (lambda () 1) (lambda (x) x)))) (morally-equal? pws pws)) #t)
(test (morally-equal? if :if) #f)
(test (morally-equal? (list 'abs 'cons) '(abs cons)) #t)
(test (morally-equal? (make-int-vector 2 0) (vector 0 0)) #t)
(test (morally-equal? (make-int-vector 2 0) (make-vector 2 0)) #t)
(test (morally-equal? (make-int-vector 2 0) (make-int-vector 2 0)) #t)
(test (morally-equal? (make-int-vector 2 0) (make-float-vector 2)) #t)
(test (morally-equal? (vector 0.0 0) (make-float-vector 2 0.0)) #t)
(test (morally-equal? (make-int-vector 2 0) (vector 0 1.0)) #f)
(test (morally-equal? (make-float-vector 1 -nan.0) (make-float-vector 1 -nan.0)) #t)

(test (morally-equal? (make-iterator "") (make-iterator "")) #t)
(test (morally-equal? (make-iterator "1") (make-iterator "1" (cons 1 1))) #t)
(test (morally-equal? (make-iterator "1" (cons 3 4)) (make-iterator "1" (cons 1 1))) #t)
(test (morally-equal? (make-iterator #()) (make-iterator #())) #t)

(let ((str "123"))
  (let ((i1 (make-iterator str))
	(i2 (make-iterator str)))
    (test (equal? i1 i2) #t)
    (test (morally-equal? i1 i2) #t)
    (iterate i1)
    (test (equal? i1 i2) #f)
    (test (morally-equal? i1 i2) #f)
    (iterate i2)
    (test (equal? i1 i2) #t)
    (test (morally-equal? i1 i2) #t)))

(let ((i1 (make-iterator "123"))
      (i2 (make-iterator "123")))
  (test (morally-equal? i1 i2) #t)
  (iterate i1)
  (test (morally-equal? i1 i2) #f)
  (iterate i2)
  (test (morally-equal? i1 i2) #t))

(let ((i1 (make-iterator (vector 1 2 3)))
      (i2 (make-iterator (int-vector 1 2 3))))
  (test (morally-equal? i1 i2) #t)
  (iterate i1)
  (test (morally-equal? i1 i2) #f)
  (iterate i2)
  (test (morally-equal? i1 i2) #t))
  
(let ((i1 (make-iterator (vector 1 2 3)))
      (i2 (make-iterator (vector 1 2 3))))
  (test (equal? i1 i2) #t)
  (test (morally-equal? i1 i2) #t)
  (iterate i1)
  (test (equal? i1 i2) #f)
  (test (morally-equal? i1 i2) #f)
  (iterate i2)
  (test (equal? i1 i2) #t)
  (test (morally-equal? i1 i2) #t))
  
(let ((str (hash-table* 'a 1 'b 2)))
  (let ((i1 (make-iterator str))
	(i2 (make-iterator str)))
    (test (equal? i1 i2) #t)
    (test (morally-equal? i1 i2) #t)
    (iterate i1)
    (test (equal? i1 i2) #f)
    (test (morally-equal? i1 i2) #f)
    (iterate i2)
    (test (equal? i1 i2) #t)
    (test (morally-equal? i1 i2) #t)))


;;; opt bug
(test (morally-equal? ''(1) (list 1)) #f)
(test (morally-equal? ''(1+i) '(1+i)) #f)
(test (morally-equal? '(1) (list 1)) #t)
(test (morally-equal? '(1) ''(1)) #f)
(test (morally-equal? (list 1) ''(1)) #f)
(test (morally-equal? (list 1) '(1)) #t)
(test (morally-equal? ''(1) ''(1)) #t)
(test (morally-equal? '''(1) ''(1)) #f)

(let ()
  (define-macro (mac a) `(+ 1 ,a))
  (define-macro (mac1 a) `(+ 1 ,a))
  (define-macro (mac2 a) `(+ 2 ,a))
  (define-macro (mac3 a b) `(+ ,b ,a))
  (test (morally-equal? mac mac1) #t)
  (test (morally-equal? mac mac2) #f)
  (test (morally-equal? mac1 mac3) #f)
  (test (morally-equal? mac3 mac3) #t)
  (let ()
    (define-macro (mac4 a) `(+ 1 ,a))
    (test (morally-equal? mac mac4) #t)) ; was #f
  (define-bacro (mac5 a) `(+ 1 ,a))
  (test (morally-equal? mac mac5) #f)
  (define-bacro (mac6 a) `(+ 1 ,a))
  (test (morally-equal? mac5 mac6) #t))

(test (morally-equal? most-positive-fixnum most-positive-fixnum) #t)
(test (morally-equal? most-positive-fixnum most-negative-fixnum) #f)
(test (morally-equal? pi pi) #t)
(test (morally-equal? 9223372036854775807 9223372036854775806) #f)
(test (morally-equal? 9223372036854775807 -9223372036854775808) #f)
(test (morally-equal? -9223372036854775808 -9223372036854775808) #t)
(test (morally-equal? 123456789/2 123456789/2) #t)
(test (morally-equal? 123456789/2 123456787/2) #f)
(test (morally-equal? -123456789/2 -123456789/2) #t)
(test (morally-equal? 2/123456789 2/123456789) #t)
(test (morally-equal? -2/123456789 -2/123456789) #t)
(test (morally-equal? 2147483647/2147483646 2147483647/2147483646) #t)
(test (morally-equal? 3/4 12/16) #t)
(test (morally-equal? 1/1 1) #t)
(test (morally-equal? 312689/99532 833719/265381) #f)
(test (let ((x 3.141)) (morally-equal? x x)) #t)
(test (let ((x 1+i)) (morally-equal? x x)) #t)
(test (let* ((x 3.141) (y x)) (morally-equal? x y)) #t)
(test (let* ((x 1+i) (y x)) (morally-equal? x y)) #t)
(test (let* ((x 3/4) (y x)) (morally-equal? x y)) #t)
(test (morally-equal? .1 1/10) #t)
(test (morally-equal? pi '(1 2)) #f)

(test (let ((x 3.141)) (morally-equal? x x)) #t)
(test (morally-equal? 3 3) #t)
(test (morally-equal? 3 3.0) #t)
(test (morally-equal? 3.0 3.0) #t)
(test (morally-equal? 3-4i 3-4i) #t)
(test (morally-equal? 1/0 0/0) #t)
(test (morally-equal? 1/0 (- 1/0)) #t) ; but they print as nan.0 and -nan.0 (this is C based I think), and equal? here is #f
(test (morally-equal? (real-part (log 0)) (- (real-part (log 0)))) #f)
(test (morally-equal? (log 0) (log 0)) #t)
(test (morally-equal? 0/0+i 0/0+i) #t)
(test (morally-equal? 0/0+i 0/0-i) #f)

(test (morally-equal? (list 3) (list 3.0)) #t)
(test (morally-equal? (list 3.0) (list 3.0)) #t)
(test (morally-equal? (list 3-4i) (list 3-4i)) #t)
(test (morally-equal? (list 1/0) (list 0/0)) #t)
(test (morally-equal? (list (log 0)) (list (log 0))) #t)
(test (morally-equal? (list 0/0+i) (list 0/0+i)) #t)

(test (morally-equal? (vector 3) (vector 3.0)) #t)
(test (morally-equal? (vector 3.0) (vector 3.0)) #t)
(test (morally-equal? (vector 3-4i) (vector 3-4i)) #t)
(test (morally-equal? (vector 1/0) (vector 0/0)) #t)
(test (morally-equal? (vector (log 0)) (vector (log 0))) #t)
(test (morally-equal? (vector 0/0+i) (vector 0/0+i)) #t)

(test (morally-equal? (string #\c) "c") #t)
(test (morally-equal? morally-equal? morally-equal?) #t)
(test (morally-equal? (cons 1 (cons 2 3)) '(1 2 . 3)) #t)
(test (morally-equal? '() '()) #t)
(test (morally-equal? '() (list)) #t)
(test (morally-equal? (cdr '   ''0) '((quote 0))) #t)
(test (morally-equal? "\n" "\n") #t)
(test (morally-equal? #f ((lambda () #f))) #t)
(test (morally-equal? (+) 0) #t)
(test (morally-equal? (recompose 32 list '(1)) (recompose 32 list (list 1))) #t)
(test (morally-equal? (recompose 100 list '(1)) (recompose 100 list (list 1))) #t)
(test (morally-equal? (recompose 32 vector 1) (recompose 32 vector 1)) #t)
(test (morally-equal? (reinvert 32 list vector 1) (reinvert 32 list vector 1)) #t)
(test (morally-equal? (recompose 32 (lambda (a) (cons 1 a)) ()) (recompose 32 (lambda (a) (cons 1 a)) ())) #t)
(test (morally-equal? (recompose 32 (lambda (a) (list 1 a)) ()) (recompose 32 (lambda (a) (list 1 a)) ())) #t)

(test (morally-equal? "asd""asd") #t) ; is this the norm?
(let ((streq (lambda (a b) (morally-equal? a b)))) (test (streq "asd""asd") #t))

(let ((things (vector #t #f #\space () "" 0 1 3/4 1+i 1.5 '(1 .2) #() (vector 1) (list 1) 'f 't #\t)))
  (let ((len (length things)))
    (do ((i 0 (+ i 1)))
	((= i (- len 1)))
      (do ((j (+ i 1) (+ j 1)))
	  ((= j len))
	(if (morally-equal? (vector-ref things i) (vector-ref things j))
	    (format-logged #t ";(morally-equal? ~A ~A) -> #t?~%" (vector-ref things i) (vector-ref things j)))))))

(test (morally-equal?) 'error)
(test (morally-equal? #t) 'error)
(test (morally-equal? #t #t #t) 'error)
(test (equal #t #t) 'error)

(test (call-with-exit (lambda (return) (return (morally-equal? return return)))) #t)
(test (call-with-exit (lambda (return) (call-with-exit (lambda (quit) (return (morally-equal? return quit)))))) #f)
(test (call/cc (lambda (return) (return (morally-equal? return return)))) #t)
(test (let hiho ((i 0)) (morally-equal? hiho hiho)) #t)
(test (let hiho ((i 0)) (let hoho ((i 0)) (morally-equal? hiho hoho))) #f)
(test (morally-equal? + *) #f)
(test (morally-equal? lambda lambda) #t)
(test (morally-equal? lambda lambda*) #f)
(test (morally-equal? let let) #t)
(test (morally-equal? let letrec) #f)
(test (morally-equal? define define) #t)
(test (morally-equal? + ((lambda (a) a) +)) #t)
(test (let ((x "hi")) (define (hi) x) (morally-equal? (hi) (hi))) #t)

(test (morally-equal? 
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) (list) (cons 1 2) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (morally-equal? 
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>)
       (vector "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs quasiquote 3 3/4 1.0+1.0i #\f (if #f #f) #<eof> #<undefined>))
      #t)
(test (morally-equal? (make-string 3) (make-string 3)) #t)
(test (morally-equal? (make-list 3) (make-list 3)) #t)
(test (morally-equal? (make-vector 3) (make-vector 3)) #t)
(test (morally-equal? (make-float-vector 3 1.0) (vector 1 1 1)) #t)
(test (morally-equal? (int-vector 1) (int-vector 2)) #f)
(test (morally-equal? (int-vector 1) (int-vector 1)) #t)
(test (morally-equal? (float-vector 0.0) (float-vector nan.0)) #f)
(test (morally-equal? (float-vector nan.0) (float-vector nan.0)) #t)
(let-temporarily (((*s7* 'morally-equal-float-epsilon) 0.0))
  (test (morally-equal? (float-vector 0.0) (float-vector nan.0)) #f)
  (test (morally-equal? (float-vector nan.0) (float-vector nan.0)) #t)
  (test (morally-equal? (float-vector 0.0) (float-vector 0.0)) #t)
  (test (morally-equal? (float-vector 0.0) (float-vector 1e-15)) #f)
  (set! (*s7* 'morally-equal-float-epsilon) 0.01)
  (test (morally-equal? (float-vector 0.0) (float-vector 1e-15)) #t)
  (test (morally-equal? (float-vector 0.0) (float-vector 0.005)) #t)
  (test (morally-equal? (float-vector 0.0) (float-vector 0.02)) #f))

(unless with-bignums (test (morally-equal? (random-state 100) (random-state 100)) #t))

(test (morally-equal? (current-input-port) (current-input-port)) #t)
(test (morally-equal? (current-input-port) (current-output-port)) #f)
(test (morally-equal? *stdin* *stderr*) #f)

(test (morally-equal? 
       (let () 
	 (define-macro* (a_func (an_arg (lambda () #t))) 
	   `,an_arg) 
	 (a_func)) 
       (let () 
	 (define-macro (a_func an_arg) 
	   `,an_arg) 
	 (a_func (lambda () #t))))
      #t) ; was #f

(test (morally-equal? (- 4/3 1 -63.0) 190/3) #t)
(test (morally-equal? 190/3 (- 4/3 1 -63.0)) #t)

(unless with-bignums
  (test (morally-equal? (+ 5e-16 nan.0) nan.0) #t)
  (test (morally-equal? (+ 0+5e-16i nan.0) nan.0) #t)
  (test (morally-equal? (+ 1/0 0+5e-16i) 1/0) #t)
  (test (morally-equal? 1/0 (+ 1/0 0+5e-16i)) #t)
  (test (morally-equal? 0 (+ 0 5e-16)) #t)
  (test (morally-equal? 0 (- 0 1/1428571428571429)) #t)
  (test (morally-equal? 0 (+ 0 0+5e-16i)) #t)
  (test (morally-equal? 0 (+ 0 0-1/1428571428571429i)) #t)
  (test (morally-equal? 0 (+ 0 1e-11)) #f)
  (test (morally-equal? 0 0) #t)
  (test (morally-equal? 0 1/1000) #f)
  (test (morally-equal? 0 0.0) #t)
  (test (morally-equal? 0 1e-16) #t)
  (test (morally-equal? 0 0+i) #f)
  (test (morally-equal? 0 1e-16+i) #f)
  (test (morally-equal? 0 0+1e-16i) #t)
  (test (morally-equal? 0 1e-300) #t)
  (test (morally-equal? 0 0+1e-300i) #t)
  (test (morally-equal? 0 1/0) #f)
  (test (morally-equal? 0 (- 0/0)) #f)
  (test (morally-equal? 0 (log 0)) #f)
  (test (morally-equal? 1 (+ 1 5e-16)) #t)
  (test (morally-equal? 1 (- 1 1/1428571428571429)) #t)
  (test (morally-equal? 1 (+ 1 0+5e-16i)) #t)
  (test (morally-equal? 1 (+ 1 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1 (+ 1 1e-11)) #f)
  (test (morally-equal? 1 1) #t)
  (test (morally-equal? 1 1.0) #t)
  (test (morally-equal? 1 1e-16) #f)
  (test (morally-equal? 1 1e4) #f)
  (test (morally-equal? 1 0+i) #f)
  (test (morally-equal? 1 1e-16+i) #f)
  (test (morally-equal? 1 (complex 1 1/0)) #f)
  (test (morally-equal? 1 (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? 1 (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? 1000 (+ 1000 5e-16)) #t)
  (test (morally-equal? 1000 (- 1000 1/1428571428571429)) #t)
  (test (morally-equal? 1000 (+ 1000 0+5e-16i)) #t)
  (test (morally-equal? 1000 (+ 1000 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1000 (+ 1000 1e-11)) #f)
  (test (morally-equal? 1000 1000) #t)
  (test (morally-equal? 1000 1/1000) #f)
  (test (morally-equal? 1000 1e4) #f)
  (test (morally-equal? 1/1000 (+ 1/1000 5e-16)) #t)
  (test (morally-equal? 1/1000 (- 1/1000 1/1428571428571429)) #t)
  (test (morally-equal? 1/1000 (+ 1/1000 0+5e-16i)) #t)
  (test (morally-equal? 1/1000 (+ 1/1000 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1/1000 (+ 1/1000 1e-11)) #f)
  (test (morally-equal? 1/1000 0) #f)
  (test (morally-equal? 1/1000 1/1000) #t)
  (test (morally-equal? 1/1000 0.0) #f)
  (test (morally-equal? 1/1000 1e-16) #f)
  (test (morally-equal? 1/1000 1e-16+i) #f)
  (test (morally-equal? 1/1000 0+1e-16i) #f)
  (test (morally-equal? 1/1000 1e-300) #f)
  (test (morally-equal? 1/1000 0+1e-300i) #f)
  (test (morally-equal? 1/1000 1/0) #f)
  (test (morally-equal? 0.0 (+ 0.0 5e-16)) #t)
  (test (morally-equal? 0.0 (- 0.0 1/1428571428571429)) #t)
  (test (morally-equal? 0.0 (+ 0.0 0+5e-16i)) #t)
  (test (morally-equal? 0.0 (+ 0.0 0-1/1428571428571429i)) #t)
  (test (morally-equal? 0.0 (+ 0.0 1e-11)) #f)
  (test (morally-equal? 0.0 0) #t)
  (test (morally-equal? 0.0 1/1000) #f)
  (test (morally-equal? 0.0 0.0) #t)
  (test (morally-equal? 0.0 1e-16) #t)
  (test (morally-equal? 0.0 0+i) #f)
  (test (morally-equal? 0.0 1+i) #f)
  (test (morally-equal? 0.0 1e-16+i) #f)
  (test (morally-equal? 0.0 0+1e-16i) #t)
  (test (morally-equal? 0.0 1e-300) #t)
  (test (morally-equal? 0.0 0+1e-300i) #t)
  (test (morally-equal? 0.0 1/0) #f)
  (test (morally-equal? 0.0 (real-part (log 0))) #f)
  (test (morally-equal? 0.0 (- (real-part (log 0)))) #f)
  (test (morally-equal? 0.0 (- 0/0)) #f)
  (test (morally-equal? 0.0 (log 0)) #f)
  (test (morally-equal? 1.0 (+ 1.0 5e-16)) #t)
  (test (morally-equal? 1.0 (- 1.0 1/1428571428571429)) #t)
  (test (morally-equal? 1.0 (+ 1.0 0+5e-16i)) #t)
  (test (morally-equal? 1.0 (+ 1.0 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1.0 (+ 1.0 1e-11)) #f)
  (test (morally-equal? 1.0 1) #t)
  (test (morally-equal? 1.0 1.0) #t)
  (test (morally-equal? 1.0 1e-16+i) #f)
  (test (morally-equal? 1.0 0+1e-16i) #f)
  (test (morally-equal? 1.0 1e-300) #f)
  (test (morally-equal? 1.0 0+1e-300i) #f)
  (test (morally-equal? 1.0 1/0) #f)
  (test (morally-equal? 1.0 (- 0/0)) #f)
  (test (morally-equal? 1.0 (complex 1/0 1)) #f)
  (test (morally-equal? 1.0 (complex 1 1/0)) #f)
  (test (morally-equal? 1.0 (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? 1e-16 (+ 1e-16 5e-16)) #t)
  (test (morally-equal? 1e-16 (- 1e-16 1/1428571428571429)) #t)
  (test (morally-equal? 1e-16 (+ 1e-16 0+5e-16i)) #t)
  (test (morally-equal? 1e-16 (+ 1e-16 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1e-16 (+ 1e-16 1e-11)) #f)
  (test (morally-equal? 1e-16 0) #t)
  (test (morally-equal? 1e-16 1/1000) #f)
  (test (morally-equal? 1e-16 0.0) #t)
  (test (morally-equal? 1e-16 1e-16) #t)
  (test (morally-equal? 1e-16 1e-16+i) #f)
  (test (morally-equal? 1e-16 0+1e-16i) #t)
  (test (morally-equal? 1e-16 1e-300) #t)
  (test (morally-equal? 1e-16 0+1e-300i) #t)
  (test (morally-equal? 1e-16 1/0) #f)
  (test (morally-equal? 1e4 (+ 1e4 5e-16)) #t)
  (test (morally-equal? 1e4 (- 1e4 1/1428571428571429)) #t)
  (test (morally-equal? 1e4 (+ 1e4 0+5e-16i)) #t)
  (test (morally-equal? 1e4 (+ 1e4 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1e4 (+ 1e4 1e-11)) #f)
  (test (morally-equal? 1e4 1000) #f)
  (test (morally-equal? 1e4 1/1000) #f)
  (test (morally-equal? 1e4 1e-16) #f)
  (test (morally-equal? 1e4 1e4) #t)
  (test (morally-equal? 1e4 1e-16+i) #f)
  (test (morally-equal? 1e4 0+1e-16i) #f)
  (test (morally-equal? 1e4 1e-300) #f)
  (test (morally-equal? 1e4 0+1e-300i) #f)
  (test (morally-equal? 1e4 1/0) #f)
  (test (morally-equal? 0+i (+ 0+i 5e-16)) #t)
  (test (morally-equal? 0+i (- 0+i 1/1428571428571429)) #t)
  (test (morally-equal? 0+i (+ 0+i 0+5e-16i)) #t)
  (test (morally-equal? 0+i (+ 0+i 0-1/1428571428571429i)) #t)
  (test (morally-equal? 0+i (+ 0+i 1e-11)) #f)
  (test (morally-equal? 0+i 0) #f)
  (test (morally-equal? 0+i 1/1000) #f)
  (test (morally-equal? 0+i 0.0) #f)
  (test (morally-equal? 0+i 1e-16) #f)
  (test (morally-equal? 0+i 0+i) #t)
  (test (morally-equal? 0+i 1+i) #f)
  (test (morally-equal? 0+i 1e-16+i) #t)
  (test (morally-equal? 0+i 0+1e-16i) #f)
  (test (morally-equal? 0+i 1e-300) #f)
  (test (morally-equal? 0+i 0+1e-300i) #f)
  (test (morally-equal? 0+i 1/0) #f)
  (test (morally-equal? 0+i (real-part (log 0))) #f)
  (test (morally-equal? 0+i (- (real-part (log 0)))) #f)
  (test (morally-equal? 0+i (- 0/0)) #f)
  (test (morally-equal? 0+i (log 0)) #f)
  (test (morally-equal? 0+i (complex 1/0 1)) #f)
  (test (morally-equal? 0+i (complex 1 1/0)) #f)
  (test (morally-equal? 0+i (complex 1/0 1/0)) #f)
  (test (morally-equal? 0+i (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? 1+i (+ 1+i 5e-16)) #t)
  (test (morally-equal? 1+i (- 1+i 1/1428571428571429)) #t)
  (test (morally-equal? 1+i (+ 1+i 0+5e-16i)) #t)
  (test (morally-equal? 1+i (+ 1+i 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1+i (+ 1+i 1e-11)) #f)
  (test (morally-equal? 1+i 0+i) #f)
  (test (morally-equal? 1+i 1+i) #t)
  (test (morally-equal? 1+i 1e-16+i) #f)
  (test (morally-equal? 1+i 0+1e-16i) #f)
  (test (morally-equal? 1+i 1e-300) #f)
  (test (morally-equal? 1+i 0+1e-300i) #f)
  (test (morally-equal? 1e-16+i (+ 1e-16+i 5e-16)) #t)
  (test (morally-equal? 1e-16+i (- 1e-16+i 1/1428571428571429)) #t)
  (test (morally-equal? 1e-16+i (+ 1e-16+i 0+5e-16i)) #t)
  (test (morally-equal? 1e-16+i (+ 1e-16+i 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1e-16+i (+ 1e-16+i 1e-11)) #f)
  (test (morally-equal? 1e-16+i 0) #f)
  (test (morally-equal? 1e-16+i 1e-16) #f)
  (test (morally-equal? 1e-16+i 1e4) #f)
  (test (morally-equal? 1e-16+i 0+i) #t)
  (test (morally-equal? 1e-16+i 1+i) #f)
  (test (morally-equal? 1e-16+i 1e-16+i) #t)
  (test (morally-equal? 1e-16+i 0+1e-16i) #f)
  (test (morally-equal? 1e-16+i 1e-300) #f)
  (test (morally-equal? 1e-16+i 0+1e-300i) #f)
  (test (morally-equal? 1e-16+i 1/0) #f)
  (test (morally-equal? 1e-16+i (real-part (log 0))) #f)
  (test (morally-equal? 1e-16+i (- (real-part (log 0)))) #f)
  (test (morally-equal? 1e-16+i (- 0/0)) #f)
  (test (morally-equal? 1e-16+i (log 0)) #f)
  (test (morally-equal? 1e-16+i (complex 1/0 1)) #f)
  (test (morally-equal? 1e-16+i (complex 1 1/0)) #f)
  (test (morally-equal? 1e-16+i (complex 1/0 1/0)) #f)
  (test (morally-equal? 1e-16+i (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? 0+1e-16i (+ 0+1e-16i 5e-16)) #t)
  (test (morally-equal? 0+1e-16i (- 0+1e-16i 1/1428571428571429)) #t)
  (test (morally-equal? 0+1e-16i (+ 0+1e-16i 0+5e-16i)) #t)
  (test (morally-equal? 0+1e-16i (+ 0+1e-16i 0-1/1428571428571429i)) #t)
  (test (morally-equal? 0+1e-16i (+ 0+1e-16i 1e-11)) #f)
  (test (morally-equal? 0+1e-16i 0) #t)
  (test (morally-equal? 0+1e-16i 1/1000) #f)
  (test (morally-equal? 0+1e-16i 0.0) #t)
  (test (morally-equal? 0+1e-16i 1e-16) #t)
  (test (morally-equal? 0+1e-16i 0+i) #f)
  (test (morally-equal? 0+1e-16i 1+i) #f)
  (test (morally-equal? 0+1e-16i 1e-16+i) #f)
  (test (morally-equal? 0+1e-16i 0+1e-16i) #t)
  (test (morally-equal? 0+1e-16i 1e-300) #t)
  (test (morally-equal? 0+1e-16i 0+1e-300i) #t)
  (test (morally-equal? 0+1e-16i 1/0) #f)
  (test (morally-equal? 0+1e-16i (real-part (log 0))) #f)
  (test (morally-equal? 0+1e-16i (- (real-part (log 0)))) #f)
  (test (morally-equal? 0+1e-16i (- 0/0)) #f)
  (test (morally-equal? 0+1e-16i (log 0)) #f)
  (test (morally-equal? 1e-300 (+ 1e-300 5e-16)) #t)
  (test (morally-equal? 1e-300 (- 1e-300 1/1428571428571429)) #t)
  (test (morally-equal? 1e-300 (+ 1e-300 0+5e-16i)) #t)
  (test (morally-equal? 1e-300 (+ 1e-300 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1e-300 (+ 1e-300 1e-11)) #f)
  (test (morally-equal? 1e-300 0) #t)
  (test (morally-equal? 1e-300 1/1000) #f)
  (test (morally-equal? 1e-300 0.0) #t)
  (test (morally-equal? 1e-300 1e-16) #t)
  (test (morally-equal? 1e-300 1e-16+i) #f)
  (test (morally-equal? 1e-300 0+1e-16i) #t)
  (test (morally-equal? 1e-300 1e-300) #t)
  (test (morally-equal? 1e-300 0+1e-300i) #t)
  (test (morally-equal? 1e-300 1/0) #f)
  (test (morally-equal? 1e-300 (- 0/0)) #f)
  (test (morally-equal? 1e-300 (log 0)) #f)
  (test (morally-equal? 0+1e-300i (+ 0+1e-300i 5e-16)) #t)
  (test (morally-equal? 0+1e-300i (- 0+1e-300i 1/1428571428571429)) #t)
  (test (morally-equal? 0+1e-300i (+ 0+1e-300i 0+5e-16i)) #t)
  (test (morally-equal? 0+1e-300i (+ 0+1e-300i 0-1/1428571428571429i)) #t)
  (test (morally-equal? 0+1e-300i (+ 0+1e-300i 1e-11)) #f)
  (test (morally-equal? 0+1e-300i 0) #t)
  (test (morally-equal? 0+1e-300i 1000) #f)
  (test (morally-equal? 0+1e-300i 1/1000) #f)
  (test (morally-equal? 0+1e-300i 0.0) #t)
  (test (morally-equal? 0+1e-300i 1e-16) #t)
  (test (morally-equal? 0+1e-300i 0+i) #f)
  (test (morally-equal? 0+1e-300i 1e-16+i) #f)
  (test (morally-equal? 0+1e-300i 0+1e-16i) #t)
  (test (morally-equal? 0+1e-300i 1e-300) #t)
  (test (morally-equal? 0+1e-300i 0+1e-300i) #t)
  (test (morally-equal? 0+1e-300i 1/0) #f)
  (test (morally-equal? 0+1e-300i (- 0/0)) #f)
  (test (morally-equal? 1/0 (+ 1/0 5e-16)) #t)
  (test (morally-equal? 1/0 (- 1/0 1/1428571428571429)) #t)
  (test (morally-equal? 1/0 (+ 1/0 0+5e-16i)) #t)
  (test (morally-equal? 1/0 (+ 1/0 0-1/1428571428571429i)) #t)
  (test (morally-equal? 1/0 0) #f)
  (test (morally-equal? 1/0 1/0) #t)
  (test (morally-equal? 1/0 (real-part (log 0))) #f)
  (test (morally-equal? 1/0 (- (real-part (log 0)))) #f)
  (test (morally-equal? 1/0 (- 0/0)) #t)
  (test (morally-equal? 1/0 (log 0)) #f)
  (test (morally-equal? 1/0 (complex 1/0 1)) #f)
  (test (morally-equal? 1/0 (complex 1 1/0)) #f)
  (test (morally-equal? 1/0 (complex 1/0 1/0)) #f)
  (test (morally-equal? 1/0 (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? 1/0 (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? 1/0 (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? 1/0 (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? 1/0 (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (real-part (log 0)) (+ (real-part (log 0)) 5e-16)) #t)
  (test (morally-equal? (real-part (log 0)) (- (real-part (log 0)) 1/1428571428571429)) #t)
  (test (morally-equal? (real-part (log 0)) (+ (real-part (log 0)) 0+5e-16i)) #t)
  (test (morally-equal? (real-part (log 0)) (+ (real-part (log 0)) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (real-part (log 0)) 0) #f)
  (test (morally-equal? (real-part (log 0)) 1e-16+i) #f)
  (test (morally-equal? (real-part (log 0)) 0+1e-16i) #f)
  (test (morally-equal? (real-part (log 0)) 1e-300) #f)
  (test (morally-equal? (real-part (log 0)) 0+1e-300i) #f)
  (test (morally-equal? (real-part (log 0)) 1/0) #f)
  (test (morally-equal? (real-part (log 0)) (real-part (log 0))) #t)
  (test (morally-equal? (real-part (log 0)) (- (real-part (log 0)))) #f)
  (test (morally-equal? (real-part (log 0)) (- 0/0)) #f)
  (test (morally-equal? (real-part (log 0)) (log 0)) #f)
  (test (morally-equal? (real-part (log 0)) (complex 1/0 1)) #f)
  (test (morally-equal? (real-part (log 0)) (complex 1 1/0)) #f)
  (test (morally-equal? (real-part (log 0)) (complex 1/0 1/0)) #f)
  (test (morally-equal? (real-part (log 0)) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (real-part (log 0)) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (real-part (log 0)) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (real-part (log 0)) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (real-part (log 0)) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (- (real-part (log 0))) (+ (- (real-part (log 0))) 5e-16)) #t)
  (test (morally-equal? (- (real-part (log 0))) (- (- (real-part (log 0))) 1/1428571428571429)) #t)
  (test (morally-equal? (- (real-part (log 0))) (+ (- (real-part (log 0))) 0+5e-16i)) #t)
  (test (morally-equal? (- (real-part (log 0))) (+ (- (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (- (real-part (log 0))) 1e-16+i) #f)
  (test (morally-equal? (- (real-part (log 0))) 0+1e-16i) #f)
  (test (morally-equal? (- (real-part (log 0))) 1e-300) #f)
  (test (morally-equal? (- (real-part (log 0))) 0+1e-300i) #f)
  (test (morally-equal? (- (real-part (log 0))) 1/0) #f)
  (test (morally-equal? (- (real-part (log 0))) (real-part (log 0))) #f)
  (test (morally-equal? (- (real-part (log 0))) (- (real-part (log 0)))) #t)
  (test (morally-equal? (- (real-part (log 0))) (- 0/0)) #f)
  (test (morally-equal? (- (real-part (log 0))) (log 0)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex 1/0 1)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex 1 1/0)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (- (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (- 0/0) (+ (- 0/0) 5e-16)) #t)
  (test (morally-equal? (- 0/0) (- (- 0/0) 1/1428571428571429)) #t)
  (test (morally-equal? (- 0/0) (+ (- 0/0) 0+5e-16i)) #t)
  (test (morally-equal? (- 0/0) (+ (- 0/0) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (- 0/0) 0) #f)
  (test (morally-equal? (- 0/0) 1e-300) #f)
  (test (morally-equal? (- 0/0) 0+1e-300i) #f)
  (test (morally-equal? (- 0/0) 1/0) #t)
  (test (morally-equal? (- 0/0) (real-part (log 0))) #f)
  (test (morally-equal? (- 0/0) (- (real-part (log 0)))) #f)
  (test (morally-equal? (- 0/0) (- 0/0)) #t)
  (test (morally-equal? (- 0/0) (log 0)) #f)
  (test (morally-equal? (- 0/0) (complex 1/0 1)) #f)
  (test (morally-equal? (- 0/0) (complex 1 1/0)) #f)
  (test (morally-equal? (- 0/0) (complex 1/0 1/0)) #f)
  (test (morally-equal? (- 0/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (- 0/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (- 0/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (- 0/0) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (- 0/0) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (log 0) (+ (log 0) 5e-16)) #t)
  (test (morally-equal? (log 0) (- (log 0) 1/1428571428571429)) #t)
  (test (morally-equal? (log 0) (+ (log 0) 0+5e-16i)) #t)
  (test (morally-equal? (log 0) (+ (log 0) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (log 0) 0) #f)
  (test (morally-equal? (log 0) 1/0) #f)
  (test (morally-equal? (log 0) (real-part (log 0))) #f)
  (test (morally-equal? (log 0) (- (real-part (log 0)))) #f)
  (test (morally-equal? (log 0) (- 0/0)) #f)
  (test (morally-equal? (log 0) (log 0)) #t)
  (test (morally-equal? (log 0) (complex 1/0 1)) #f)
  (test (morally-equal? (log 0) (complex 1 1/0)) #f)
  (test (morally-equal? (log 0) (complex 1/0 1/0)) #f)
  (test (morally-equal? (log 0) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (log 0) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (log 0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (log 0) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (log 0) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1) (+ (complex 1/0 1) 5e-16)) #t)
  (test (morally-equal? (complex 1/0 1) (- (complex 1/0 1) 1/1428571428571429)) #t)
  (test (morally-equal? (complex 1/0 1) (+ (complex 1/0 1) 0+5e-16i)) #t)
  (test (morally-equal? (complex 1/0 1) (+ (complex 1/0 1) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex 1/0 1) 0) #f)
  (test (morally-equal? (complex 1/0 1) 1) #f)
  (test (morally-equal? (complex 1/0 1) 1e-16+i) #f)
  (test (morally-equal? (complex 1/0 1) 0+1e-16i) #f)
  (test (morally-equal? (complex 1/0 1) 1e-300) #f)
  (test (morally-equal? (complex 1/0 1) 0+1e-300i) #f)
  (test (morally-equal? (complex 1/0 1) 1/0) #f)
  (test (morally-equal? (complex 1/0 1) (real-part (log 0))) #f)
  (test (morally-equal? (complex 1/0 1) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1) (- 0/0)) #f)
  (test (morally-equal? (complex 1/0 1) (log 0)) #f)
  (test (morally-equal? (complex 1/0 1) (complex 1/0 1)) #t)
  (test (morally-equal? (complex 1/0 1) (complex 1 1/0)) #f)
  (test (morally-equal? (complex 1/0 1) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex 1/0 1) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex 1/0 1) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex 1/0 1) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 1/0) (+ (complex 1 1/0) 5e-16)) #t)
  (test (morally-equal? (complex 1 1/0) (- (complex 1 1/0) 1/1428571428571429)) #t)
  (test (morally-equal? (complex 1 1/0) (+ (complex 1 1/0) 0+5e-16i)) #t)
  (test (morally-equal? (complex 1 1/0) (+ (complex 1 1/0) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex 1 1/0) 0) #f)
  (test (morally-equal? (complex 1 1/0) 1) #f)
  (test (morally-equal? (complex 1 1/0) 1e-300) #f)
  (test (morally-equal? (complex 1 1/0) 0+1e-300i) #f)
  (test (morally-equal? (complex 1 1/0) 1/0) #f)
  (test (morally-equal? (complex 1 1/0) (real-part (log 0))) #f)
  (test (morally-equal? (complex 1 1/0) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 1/0) (- 0/0)) #f)
  (test (morally-equal? (complex 1 1/0) (log 0)) #f)
  (test (morally-equal? (complex 1 1/0) (complex 1/0 1)) #f)
  (test (morally-equal? (complex 1 1/0) (complex 1 1/0)) #t)
  (test (morally-equal? (complex 1 1/0) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex 1 1/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex 1 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex 1 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1/0) (+ (complex 1/0 1/0) 5e-16)) #t)
  (test (morally-equal? (complex 1/0 1/0) (- (complex 1/0 1/0) 1/1428571428571429)) #t)
  (test (morally-equal? (complex 1/0 1/0) (+ (complex 1/0 1/0) 0+5e-16i)) #t)
  (test (morally-equal? (complex 1/0 1/0) (+ (complex 1/0 1/0) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex 1/0 1/0) 0) #f)
  (test (morally-equal? (complex 1/0 1/0) 1/0) #f)
  (test (morally-equal? (complex 1/0 1/0) (real-part (log 0))) #f)
  (test (morally-equal? (complex 1/0 1/0) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1/0) (- 0/0)) #f)
  (test (morally-equal? (complex 1/0 1/0) (log 0)) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex 1/0 1)) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex 1 1/0)) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex 1/0 1/0)) #t)
  (test (morally-equal? (complex 1/0 1/0) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex 1/0 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 5e-16)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (- (complex (real-part (log 0)) 1/0) 1/1428571428571429)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 0+5e-16i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (+ (complex (real-part (log 0)) 1/0) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1000) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1/1000) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 0.0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1.0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1e-16) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1e4) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) 1/0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (real-part (log 0))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (- 0/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (log 0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex 1/0 1)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex 1 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) 1/0)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1/0) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 5e-16)) #t)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (- (complex 1/0 (real-part (log 0))) 1/1428571428571429)) #t)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 0+5e-16i)) #t)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (+ (complex 1/0 (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (real-part (log 0))) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (- 0/0)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (log 0)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex 1/0 1)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex 1 1/0)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #t)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex 1/0 (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 5e-16)) #t)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (- (complex (real-part (log 0)) (real-part (log 0))) 1/1428571428571429)) #t)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 0+5e-16i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (+ (complex (real-part (log 0)) (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) 0) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) 1/0) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (real-part (log 0))) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (- 0/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (log 0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 1)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex 1 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #t)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex (real-part (log 0)) (real-part (log 0))) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 5e-16)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1) (- (complex (real-part (log 0)) 1) 1/1428571428571429)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 0+5e-16i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1) (+ (complex (real-part (log 0)) 1) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1) 0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) 1) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) 0+1e-300i) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) 1/0) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (real-part (log 0))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (- 0/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (log 0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex 1/0 1)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex 1 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex (real-part (log 0)) 1)) #t)
  (test (morally-equal? (complex (real-part (log 0)) 1) (complex 1 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 5e-16)) #t)
  (test (morally-equal? (complex 1 (real-part (log 0))) (- (complex 1 (real-part (log 0))) 1/1428571428571429)) #t)
  (test (morally-equal? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 0+5e-16i)) #t)
  (test (morally-equal? (complex 1 (real-part (log 0))) (+ (complex 1 (real-part (log 0))) 0-1/1428571428571429i)) #t)
  (test (morally-equal? (complex 1 (real-part (log 0))) (real-part (log 0))) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (- (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (- 0/0)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (log 0)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex 1/0 1)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex 1 1/0)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex 1/0 1/0)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) 1/0)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex 1/0 (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) (real-part (log 0)))) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex (real-part (log 0)) 1)) #f)
  (test (morally-equal? (complex 1 (real-part (log 0))) (complex 1 (real-part (log 0)))) #t)) ; end with-bignums


;;; ----------------
;;; try a bunch of combinations

(define-expansion (format-logged-with-line port str . args)
  `(format-logged ,port ,str ,(port-line-number) ,@args))

(let ((lst1 ())
      (lst2 ()))
  (if (not (eq? lst1 lst2)) (format-logged-with-line #t ";~A: nils are not eq?~%"))
  (if (not (eqv? lst1 lst2)) (format-logged-with-line #t ";~A: nils are not eqv?~%"))
  (if (not (equal? lst1 lst2)) (format-logged-with-line #t ";~A: nils are not equal?~%"))

  (let ((v1 (make-vector 100 #f))
	(v2 (make-vector 100 #f)))
    (if (not (equal? v1 v2)) (format-logged-with-line #t ";~A: base vectors are not equal?~%"))

    (let ((h1 (make-hash-table))
	  (h2 (make-hash-table)))
      (if (not (equal? h1 h2)) (format-logged-with-line #t ";~A: base hash-tables are not equal?~%"))

      (let ((e1 (sublet (curlet)))
	    (e2 (sublet (curlet))))
	(if (not (equal? e1 e2)) (format-logged-with-line #t ";~A: base environments are not equal?~%"))

	(let ((ctr 0))
	  (for-each
	   (lambda (arg1 arg2)
	     ;; make sure the args are eq? to themselves
	     ;; if equal? and equal to copy place in lst1, place copy in lst2, check that they are still equal
	     ;;     similarly for vector, hash-table, envs
	   (let ((a1 arg1)
		 (a2 arg2))
	     (if (not (eq? a1 arg1)) 
		 (format-logged-with-line #t ";~A: ~A is not eq? to itself? ~A~%" arg1 a1))
	     (if (and (eq? a1 a2) (not (eqv? a1 a2)))
		 (format-logged-with-line #t ";~A: ~A is eq? but not eqv? ~A~%" a1 a2))

	     (if (equal? a1 a2)
		 (begin
		   (if (and (eq? a1 a2) (not (eqv? a1 a2))) 
		       (format-logged-with-line #t ";~A: ~A is eq? and equal? but not eqv?? ~A~%" a1 a2))
		   (if (not (morally-equal? a1 a2))
		       (format-logged-with-line #t ";~A: ~A is equal? but not morally-equal? ~A~%" a1 a2))
		   (set! lst1 (cons a1 lst1))
		   (set! lst2 (cons a2 lst2))
		   (set! (v1 ctr) a1)
		   (set! (v2 ctr) a2)
		   (let* ((sym1 (string->symbol (string-append "symbol-" (number->string ctr))))
			  (sym2 (copy sym1)))
		     (set! (h1 sym1) a1)
		     (set! (h2 sym2) a2)
		     (varlet e1 (cons sym1 a1))
		     (varlet e2 (cons sym2 a2))

		     (if (not (equal? lst1 lst2))
			 (begin
			   (format-logged-with-line #t ";~A: add ~A to lists, now not equal?~%" a1)
			   (set! lst1 (cdr lst1))
			   (set! lst2 (cdr lst2))))
		     (if (not (equal? v1 v2))
			 (begin
			   (format-logged-with-line #t ";~A: add ~A to vectors, now not equal?~%" a1)
			   (set! (v1 ctr) #f)
			   (set! (v2 ctr) #f)))
		     (if (not (equal? h1 h2))
			 (begin
			   (format-logged-with-line #t ";~A: add ~A to hash-tables, now not equal?~%" a1)
			   (set! (h1 sym1) #f)
			   (set! (h2 sym2) #f)))
		     (if (not (equal? e1 e2))
			 (begin
			   (format-logged-with-line #t ";~A: add ~A to environments, now not equal?~% ~A~% ~A~%" a1 e1 e2)
			   (eval `(set! ,sym1 #f) e1)
			   (eval `(set! ,sym2 #f) e2)))
		     ))
		 (begin
		   (if (eq? a1 arg1) (format-logged-with-line #t ";~A: ~A is eq? but not equal? ~A~%" a1 a2))
		   (if (eqv? a1 arg1) (format-logged-with-line #t ";~A: ~A is eqv? but not equal? ~A~%" a1 a2))
		   (format-logged-with-line #t ";~A: ~A is not equal to ~A~%" a1 a2)))

	     (set! ctr (+ ctr 1))))

	 (list "hi" ""
	       (integer->char 65) #\space #\newline #\null
	       1 3/4 
	       ;; 1.0 1+i pi (real-part (log 0)) 1e18
	       most-negative-fixnum most-positive-fixnum 
	       'a-symbol 
	       (make-vector 3 #f) #() #2d((1 2) (3 4))
	       abs quasiquote macroexpand (log 0) 
	       (hash-table '(a . 1) '(b . 2)) (hash-table)
	       (sublet (curlet) '(a . 1)) (rootlet)
	       #f #t :hi 
	       #<eof> #<undefined> #<unspecified>
	       (cons 1 2) () '(1) (list (cons 1 2)) '(1 2 . 3) 
	       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
	       )
	 (list (string #\h #\i) (string)
	       #\A #\space #\newline (integer->char 0)
	       (- 2 1) (/ 3 4) 
	       ;; 1.0 1+i pi (real-part (log 0)) 1e18
	       -9223372036854775808 9223372036854775807 
	       (string->symbol "a-symbol")
	       (vector #f #f #f) (vector)  #2d((1 2) (3 4))
	       abs quasiquote macroexpand (log 0) 
	       (let ((h (make-hash-table 31))) (set! (h 'a) 1) (set! (h 'b) 2) h) (make-hash-table 123)
	       (sublet (curlet) '(a . 1)) (rootlet)
	       #f #t :hi 
	       #<eof> #<undefined> (if #f #f)
	       '(1 . 2) (list) (list 1) (list (cons 1 2)) '(1 2 . 3) 
	       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
	       ))
	  
	  (set! (v1 ctr) lst1)
	  (set! (v2 ctr) lst2)
	  (set! ctr (+ ctr 1))
	  (if (not (equal? v1 v2))
	      (format-logged-with-line #t ";~A: add lists to vectors, now vectors not equal?~%")
	      (begin
		(set! lst1 (cons v1 lst1))
		(set! lst2 (cons v2 lst2)) 
		(if (not (equal? lst1 lst2))
		    (begin
		      (format-logged-with-line #t ";~A: add vectors to lists, now lists not equal?~%")
		      (set! (h1 'lst1) lst1)
		      (set! (h2 'lst2) lst2)
		      (if (not (equal? h1 h2))
			  (format-logged-with-line #t ";~A: add lists to hash-tables, not hash-tables not equal?~%")
			  (begin
			    (set! (v1 ctr) v1)
			    (set! (v2 ctr) v2)
			    (set! ctr (+ ctr 1))
			    (if (not (equal? v1 v2))
				(format-logged-with-line #t ";~A: add vectors to themselves, now vectors not equal?~%"))
			    (if (not (equal? lst1 lst2))
				(format-logged-with-line #t ";~A: add vectors to themselves, now lists not equal?~%"))
			    (set! (h1 'h1) h1)
			    (set! (h2 'h2) h2)
			    (if (not (equal? h1 h2))
				(format-logged-with-line #t ";~A: add hash-tables to themselves, not hash-tables not equal?~%"))
			    )))))))))))

(define old-readers *#readers*)
(set! *#readers* (cons (cons #\u (lambda (str) (string->number (substring str 1)))) ()))
(test (eval (with-input-from-string "(+ 10 #u12)" read)) 22)
(test (eval (with-input-from-string "(+ 10 #u87)" read)) 97)

(do ((i (char->integer #\") (+ i 1)))
    ((= i 127))
  (when (not (member (integer->char i) '(#\( #\: #\|)))
    (set! *#readers* (cons (cons (integer->char i) (lambda (str) (string->number (substring str 1)))) ()))
    (let ((val (eval (with-input-from-string (string-append "(+ 10 #" (string (integer->char i)) "12)") read))))
      (if (not (equal? val 22)) (format *stderr* "~D (~C): ~A~%" i (integer->char i) val)))))

(set! *#readers* 
  (list (cons #\[
	      (lambda (str)
		(let ((h (make-hash-table)))
		  (do ((c (read) (read)))
		      ((eq? c ']#) h)
		    (set! (h (car c)) (cdr c))))))))

(eval-string "(let ((table #[(a . 1) (b . #[(c . 3)]#)]#))
  (test (hash-table? table) #t)
  (test (table 'a) 1)
  (test (hash-table? (table 'b)) #t)
  (test ((table 'b) 'c) 3))")

(set! *#readers* old-readers)

(when with-block
  (let ((b (make-block 4))) 
    (test (morally-equal? b b) #t)
    (let ((b1 (make-block 4)))
      (test (morally-equal? b b1) #t)
      (set! (b 1) 1.0)
      (test (morally-equal? b b1) #f))))
(test (let ((p (c-pointer 0))) (morally-equal? p (copy p))) #t)



;;; --------------------------------------------------------------------------------
;;; boolean?

(test (boolean? #f) #t)
(test (boolean? #t) #t)
(test (boolean? 0) #f)
(test (boolean? 1) #f)
(test (boolean? "") #f)
(test (boolean? #\0) #f)
(test (boolean? ()) #f)
(test (boolean? #()) #f)
(test (boolean? 't) #f)
(test (boolean? (list)) #f)
(test ( boolean? #t) #t)
(test (boolean? boolean?) #f)
(test (boolean? or) #f)
(test (   ; a comment 
       boolean?  ;;; and another
       #t
       )
      #t)

(for-each
 (lambda (arg)
   (if (boolean? arg)
       (format-logged #t ";(boolean? ~A) -> #t?~%" arg)))
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))

(test (recompose 12 boolean? #f) #t)

(test (boolean?) 'error)
(test (boolean? #f #t) 'error)
(test (boolean #f) 'error)
(test (boolean? (lambda (x) #f)) #f)
(test (boolean? and) #f)
(test (boolean? if) #f)
(test (boolean? (values)) #f)
;(test (boolean? else) #f) ; this could also be an error -> unbound variable, like (symbol? else)




;;; --------------------------------------------------------------------------------
;;; not

(test (not #f) #t)
(test (not #t) #f)
(test (not (not #t)) #t)
(test (not 0) #f)
(test (not 1) #f)
(test (not ()) #f)
(test (not 't) #f)
(test (not (list)) #f)
(test (not (list 3)) #f)
(test (not 'nil) #f)
(test (not not) #f)
(test (not "") #f)
(test (not lambda) #f)
(test (not quote) #f)

(for-each
 (lambda (arg)
   (if (not arg)
       (format-logged #t ";(not ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi #<eof> #<undefined> (if #f #f)))

(test (recompose 12 not #f) #f)

(test (not) 'error)
(test (not #f #t) 'error)
(test (not and) #f)
(test (not case) #f)

(let () ; check some optimizer branches
  (define (f1 sym) (not (symbol? sym))) (test (f1 'hi) #f) (test (f1 "hi") #t)  
  (define (f2 sym) (not (integer? sym))) (test (f2 2) #f) (test (f2 'hi) #t)
  (define (f3 sym) (not (char? sym))) (test (f3 2) #t) (test (f3 #\a) #f)
  (define (f4 sym) (not (list? sym))) (test (f4 2) #t) (test (f4 '(1 2 3)) #f)
  (define (f5 sym) (not (boolean? sym))) (test (f5 2) #t) (test (f5 #f) #f)
  (define (f6 sym) (not (eof-object? sym))) (test (f6 2) #t) (test (f6 #<eof>) #f)
  (define (f7 sym) (not (pair? (car sym)))) (test (f7 '(hi)) #t) (test (f7 '((1))) #f)
  (define (f8 sym) (not (eq? sym 'q))) (test (f8 'a) #t) (test (f8 'q) #f)
  (define (f9 sym) (pair? (cadr sym))) (test (f9 '(1 2 3)) #f) (test (f9 '(1 (2 3) 4)) #t)
  (define (f10 lst val) (eq? (car lst) val)) (test (f10 '(#f) #f) #t) (test (f10 '(a) 32) #f)
  (define (f11 lst) (eq? (caar lst) 'q)) (test (f11 '((a))) #f) (test (f11 '((q))) #t)
  (define (f12 lst) (= (length lst) 2)) (test (f12 '(1 2)) #t) (test (f12 '(1 2 3)) #f)
  (define (f13 lst) (< (length lst) 2)) (test (f13 '(1 2)) #f) (test (f13 '(1)) #t)
  (define (f14 lst) (negative? (length lst))) (test (f14 '(1 2)) #f) (test (f14 '(1 . 3)) #t)
  (define (f15 lst) (memq (car lst) '(a b c))) (test (f15 '(a)) '(a b c)) (test (f15 '(d)) #f)
  (define (f16 a b) (if a (begin (+ b a) (format #f "~A" a) (+ a a)))) (test (f16 1 2) 2)
  (define (f17 a) (aritable? a 1)) (test (f17 abs) #t)
  (define (f18) (set! (-s7-symbol-table-locked?) #f)) (f18) (test (f18) #f)
  (define (f18a) (set! (-s7-symbol-table-locked?) #f)) (test (f18a) #f) (test (let () (f18a)) #f)
  (define (f19) (set! (-s7-symbol-table-locked?) #f) 1) (f19) (test (f19) 1)
  (define (f19a) (set! (-s7-symbol-table-locked?) #f) 1) (test (f19a) 1) (test (let () (f19a)) 1)
  (define (f20) (set! (-s7-symbol-table-locked?) #f) (+ 1 2)) (f20) (test (f20) 3)
  (define (f20a) (set! (-s7-symbol-table-locked?) #f) (+ 1 2)) (test (f20a) 3) (test (let () (f20a)) 3)
  (define (f21) (set! (-s7-symbol-table-locked?) #f) (+ 1 2) 4) (f21) (test (f21) 4)
  (define (f21a) (set! (-s7-symbol-table-locked?) #f) (+ 1 2) 4) (test (f21a) 4) (test (let () (f21a)) 4)
  (define (f22) (begin (display ":") (display (object->string 2)) (display ":"))) (test (with-output-to-string (lambda () (f22))) ":2:")
  (define (f23 a b) (list a b))
  (define (f24 x y) (f23 (car x) (car y)))
  (define (f25 x y) (f23 (cdr x) (cdr y)))
  (test (f24 '(1 2) '(3 4)) '(1 3)) (test (f25 '(1 2) '(3 4)) '((2) (4)))
  (define (f24a s1 s2 s3) (+ (* s1 s2) (* (- 1.0 s1) s3))) (test (f24a 2.0 3.0 4.0) 2.0)
  (let () (define (a b) (define c 1) (+ b c)) (define (tst) (a 2)) (tst) (test (tst) 3))
  (define (f25) 
    (let ((x 0.0) (y 1.0)) 
      (call-with-exit 
       (lambda (return) 
	 (do ((i y (+ i 1))) ((= i 6)) 
	   (do ((i i (+ i 1))) ((>= i 7)) 
	     (set! x (+ x i)) 
	     (if (> x 123.0) (return x)))))) 
      x))
  (test (f25) 85.0)
  )
(let ()
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (ho 1) (ho 2))) (hi)) 7)
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (ho 1) (values 3 4))) (hi)) 10)
  (test (let () (define (ho a) (+ a 2)) (define (hi) (+ (values 3 4) (ho 1))) (hi)) 10)
  (test (let () (define (hi) (+ (values 1 2) (values 3 4))) (hi)) 10)
  (test (let () (define (ho a) (values a 1)) (define (hi) (- (ho 2))) (hi)) 1)
  (test (let () (define (ho1) (s7-version)) (define (ho2) (ho1)) (string? (ho2))) #t)
  (test (let () (define (hi) (vector 0)) (define (ho) (hi)) (ho)) #(0)))
(let ()
  (define (make-it . names) (apply vector names))
  (define (hi) (make-it pi pi pi pi))
  (test (hi) (vector pi pi pi pi)))
(test (let () (define (hi a b c d) (+ a (* (- b c) d))) (define (ho) (hi 1 2 3 4)) (ho)) -3)
(test (let () (define (hi a b c d) (+ a (* d (- b c)))) (define (ho) (hi 1 2 3 4)) (ho)) -3)
(test (let () (define (hi) (let ((x (values 1 2))) (if x (list x)))) (define (ho) (hi)) (catch #t (lambda () (ho)) (lambda args #f)) (ho)) 'error)
(test (let () (define (hi a b) (- (+ a (abs b)))) (define (ho) (hi 1 -2)) (ho)) -3)

(let () (define (e1) (((lambda () list)) 'a 'b 'c)) (define (e2) (e1)) (e2) (test (e2) '(a b c)))
(let () (define (c1 s i) (case (string-ref s i) ((#\a) 1) (else 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a) 1) ((#\i) 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a #\h) 1) ((#\i #\o) 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (c1 s i) (case (string-ref s i) ((#\a #\h) 1) (else 2))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) 2))
(let () (define (d1) (do ((lst () (cons i lst)) (i 0 (+ i 1))) ((> i 6) (reverse lst)))) (define (d2) (d1)) (d2) (test (d2) '(0 1 2 3 4 5 6)))
(let () (define (d3) ((define (hi a) (+ a 1)) 2)) (define (d4) (d3)) (d4) (test (d4) 3))
(let () (define (fif) (if (< 2 3) (quote . -1))) (catch #t fif (lambda args 'error)) (test (catch #t fif (lambda args 'error)) 'error))
;(let () (define (fcond) (cond ((< 2 3) ((lambda (x) x 1 . 5) 2)))) (catch #t fcond (lambda args 'error)) (test (fcond) 'error))
;(let () (define (fcond1) (cond ((< 2 3) ((lambda* (x) x . 5) 2)))) (catch #t fcond1 (lambda args 'error)) (test (fcond1) 'error))
; those aren't what they appear to be: the catch does the stray dot check/error, then a call simply does what it can
(let () (define (incsaa k i) (let ((sum 1)) (set! sum (+ sum (expt k i) (expt (- k) i))) sum)) (define (f1) (incsaa 3 2)) (test (f1) 19))
(let () (define (unks v1 i) (let ((x 0)) (set! x (v1 i)) x)) (define (f1) (unks (vector 1 2 3) 2)) (test (f1) 3))


;;; --------------------------------------------------------------------------------
;;; symbol?

(test (symbol? 't) #t)
(test (symbol? "t") #f)
(test (symbol? '(t)) #f)
(test (symbol? #t) #f)
(test (symbol? 4) #f)
(test (symbol? 'foo) #t)
(test (symbol? (car '(a b))) #t)
(test (symbol? 'nil) #t)
(test (symbol? ()) #f)
(test (symbol? #()) #f)
(test (symbol? #f) #f)
(test (symbol? 'car) #t)
(test (symbol? car) #f)
(test (symbol? '#f) #f)
(test (symbol? #()) #f)
(test (symbol? :hi) #t)
(test (symbol? hi:) #t)
(test (symbol? :hi:) #t)
(test (symbol? ::) #t)
(test (symbol? ':) #t)
(test (symbol? '|) #t)
(test (symbol? '|') #t)
(test (symbol? '@) #t)
;(test (symbol? '#:) #t) ; confusable given guile-style keywords
(test (symbol? #b1) #f)
(test (symbol? 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789) #t) ;M Gran
(test (symbol? (vector-ref #(1 a 34) 1)) #t)
(test (if (symbol? '1+) (symbol? '0e) #t) #t)
(test (symbol? 'begin) #t)
(test (symbol? 'if) #t)
(test (symbol? (keyword->symbol :if)) #t)
(test (symbol? (string->symbol "if")) #t)
(test (symbol? if) #f)
(test (symbol? quote) #f)
(test (symbol? '(AB\c () xyz)) #f)

(for-each
 (lambda (arg)
   (if (symbol? arg)
       (format-logged #t ";(symbol? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) #<eof> #<undefined>))

(test (symbol?) 'error)
(test (symbol? 'hi 'ho) 'error)
(test (symbol? 'hi 3) 'error)
(test (symbol? 3 3) 'error)
(test (symbol? 3 'hi) 'error)
(test (symbol 'hi) 'error) ; symbol takes a string

;;; "Returns #t if obj is a symbol, otherwise returns #f" (r5|6rs.html)
(test (symbol? begin) #f) ; ?? this is an error in Guile, it was #t in s7
(test (symbol? expt) #f)
(test (symbol? if) #f)
(test (symbol? and) #f)
(test (symbol? lambda) #f)
(test (symbol? 'let) #t)
(test (symbol? call/cc) #f)
(test (symbol? '1.2.3) #t)
(test (symbol? '1.2) #f)
(test (symbol? ''1.2) #f)
(test (symbol? '"hi") #f)

(test (let ((sym000000000000000000000 3))
	(let ((sym000000000000000000001 4))
	  (+ sym000000000000000000000 sym000000000000000000001)))
      7)
(test (let ((11-1 10)
	    (2012-4-19 21)
	    (1+the-road 18)
	    (-1+2 1)
	    (1e. 2)
	    (0+i' 3)
	    (0.. 4))
	(+ 11-1 2012-4-19 1+the-road -1+2 1e. 0+i' 0..))
      59)

(test (let ((name "hiho"))
	(string-set! name 2 #\null)
	(symbol? (string->symbol name)))
      #t)




;;; --------------------------------------------------------------------------------
;;; procedure?

(test (procedure? car) #t)
(test (procedure? procedure?) #t)
(test (procedure? 'car) #f)
(test (procedure? (lambda (x) x)) #t)
(test (procedure? '(lambda (x) x)) #f)
(test (call/cc procedure?) #t) ; ??
(test (let ((a (lambda (x) x)))	(procedure? a)) #t)
(test (letrec ((a (lambda () (procedure? a)))) (a)) #t)
(test (let ((a 1)) (let ((a (lambda () (procedure? a)))) (a))) #f)
(test (let () (define (hi) 1) (procedure? hi)) #t)
(test (let () (define-macro (hi a) `(+ ,a 1)) (procedure? hi)) #f)
(test (procedure? begin) #f)
(test (procedure? lambda) #f)
(test (procedure? (lambda* ((a 1)) a)) #t)
(test (procedure? and) #f)
(test (procedure? 'let) #f)
(test (procedure? (dilambda (lambda () 1) (lambda (x) x))) #t)
(if with-bignums (test (procedure? (bignum "1e100")) #f))
(test (procedure? quasiquote) #f)
(let () (define-macro (hi a) `(+ ,a 1)) (test (procedure? hi) #f))
(test (procedure? (random-state 1234)) #f)
(test (procedure? pi) #f)
(test (procedure? cond) #f)
(test (procedure? do) #f)
(test (procedure? set!) #f)

(for-each
 (lambda (arg)
   (if (procedure? arg)
       (format-logged #t ";(procedure? ~A) -> #t?~%" arg)))
 (list "hi" _ht_ _null_ _c_obj_ :hi (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f #() (if #f #f)))

(test (procedure?) 'error)
(test (procedure? abs car) 'error)
(test (procedure abs) 'error)

;; these are questionable -- an applicable object is a procedure
(test (procedure? "hi") #f)
(test (procedure? '(1 2)) #f)
(test (procedure? #(1 2)) #f)





;;; --------------------------------------------------------------------------------
;;; CHARACTERS
;;; --------------------------------------------------------------------------------

(test (eqv? '#\  #\space) #t)
(test (eqv? #\newline '#\newline) #t)

;;; --------------------------------------------------------------------------------
;;; char?

(test (char? #\a) #t)
(test (char? #\() #t)
(test (char? #\space) #t)
(test (char? '#\newline) #t)
(test (char? #\1) #t)
(test (char? #\$) #t)
(test (char? #\.) #t)
(test (char? #\\) #t)
(test (char? #\)) #t)
(test (char? #\%) #t)
(test (char? '#\space) #t)
(test (char? '#\ ) #t)
(test (char? '#\newline) #t)
(test (char? '#\a) #t)
(test (char? '#\8) #t)
(test (char? #\-) #t)
(test (char? #\n) #t)
(test (char? #\() #t)
(test (char? #e1) #f)
(test (char? #\#) #t)
(test (char? #\x) #t)
(test (char? #\o) #t)
(test (char? #\b) #t)
(test (char? #b101) #f)
(test (char? #o73) #f)
(test (char? #x73) #f)
(test (char? 'a) #f)
(test (char? 97) #f)
(test (char? "a") #f)
(test (char? (string-ref "hi" 0)) #t)
(test (char? (string-ref (make-string 1) 0)) #t)
(test (char? #\") #t)
(test (char? #\') #t)
(test (char? #\`) #t)
(test (char? #\@) #t)
(test (char? #<eof>) #f)
(test (char? '1e311) #f)

(for-each
 (lambda (arg)
   (if (char? arg)
       (format-logged #t ";(char? ~A) -> #t?~%" arg)))
 (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) :hi (lambda (a) (+ a 1))))

(test (char? begin) #f)

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (not (char? (integer->char i)))
      (format-logged #t ";(char? (integer->char ~A)) -> #f?~%" i)))

(test (char?) 'error)
(test (char? #\a #\b) 'error)
(test (char #\a) 'error)

(test (char? #\x65) #t)
(test (char? #\x000000000065) #t)
(test (char? #\x0) #t)
(test (char=? #\x000 #\null) #t)
(test (char=? #\x08 #\x8) #t)
(test (char=? #\x0e #\xe) #t) ; Guile thinks both of these names are bogus
(test (char=? #\x00e #\xe) #t)
(test (char=? #\x0000e #\xe) #t)
(test (char=? #\x00000000e #\xe) #t) ; hmmm -- surely this is a bug
(test (char? #\xff) #t)
;; any larger number is a reader error

(test (eval-string "(char? #\xbdca2cbec)") 'error) ; this can overflow internally!
(test (eval-string "(char? #\\xbdca2cbec)") 'error)
(test (eval-string "(char? #\\100)") 'error)
(test (eval-string "(char? #\\x-65)") 'error)
(test (eval-string "(char? #\\x6.5)") 'error)
(test (eval-string "(char? #\\x6/5)") 'error)
(test (eval-string "(char? #\\x6/3)") 'error)
(test (eval-string "(char? #\\x6+i)") 'error)
(test (eval-string "(char? #\\x6asd)") 'error)
(test (eval-string "(char? #\\x6#)") 'error)
(test (eval-string "(char? #\\x#b0)") 'error)
(test (eval-string "(char? #\\x#b0") 'error)
(test (eval-string "(char? #\\x#e0.0") 'error)
(test (eval-string "(char? #\\x-0") 'error)
(test (eval-string "(char? #\\x#e0e100") 'error)
(test (eval-string "(char? #\\x1.4)") 'error)
(test (eval-string "(char? #\\x#b0)") 'error)
(test (eval-string "(char? #\\x#e0.0)") 'error)
(test (eval-string "(char? #\\x-0)") 'error)
(test (eval-string "(char? #\\x#e0e100)") 'error)
(test (eval-string "(char? #\\x1.4)") 'error)

(test (char=? #\x6a #\j) #t)

(test (char? #\return) #t)
(test (char? #\null) #t)
(test (char? #\nul) #t)
(test (char? #\linefeed) #t)
(test (char? #\tab) #t)
(test (char? #\space) #t)
(test (char=? #\null #\nul) #t)
(test (char=? #\newline #\linefeed) #t)
(test (char=? #\return #\xd) #t)
(test (char=? #\nul #\x0) #t)
;(test (char? #\ÿ) #t) ; this seems to involve unwanted translations in emacs?
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 255)) ")")) #t)
(test (eval-string (string-append "(char? " (format #f "#\\~C" (integer->char 127)) ")")) #t)
(test (apply char? (list (integer->char 255))) #t)

(test (char? #\escape) #t)
(test (char? #\alarm) #t)
(test (char? #\backspace) #t)
(test (char? #\delete) #t)
(test (char=? #\delete #\backspace) #f)

(num-test (let ((str (make-string 258 #\space)))
	    (do ((i 1 (+ i 1)))
		((= i 256))
	      (string-set! str i (integer->char i)))
	    (string-set! str 257 (integer->char 0))
	    (string-length str))
	  258)


(let ((a-to-z (list #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\x #\y #\z))
      (cap-a-to-z (list #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\X #\Y #\Z))
      (mixed-a-to-z (list #\a #\B #\c #\D #\e #\F #\g #\H #\I #\j #\K #\L #\m #\n #\O #\p #\Q #\R #\s #\t #\U #\v #\X #\y #\Z))
      (digits (list #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)))
  
;;; --------------------------------------------------------------------------------
;;; char-upper-case?

  (test (char-upper-case? #\a) #f)
  (test (char-upper-case? #\A) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-upper-case? arg))
	 (format-logged #t ";(char-upper-case? ~A) -> #f?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-upper-case? arg)
	 (format-logged #t ";(char-upper-case? ~A) -> #t?~%" arg)))
   a-to-z)
  
  (test (char-upper-case? (integer->char 192)) #t) ; 192..208 for unicode
  ;; non-alpha chars are "unspecified" here
  
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? 1) 'error)
  (test (char-upper-case?) 'error)
  (test (char-upper-case? #\a #\b) 'error)
  (test (char-upper-case #\a) 'error)


  
;;; --------------------------------------------------------------------------------
;;; char-lower-case?

  (test (char-lower-case? #\A) #f)
  (test (char-lower-case? #\a) #t)
  
  (for-each
   (lambda (arg)
     (if (not (char-lower-case? arg))
	 (format-logged #t ";(char-lower-case? ~A) -> #f?~%" arg)))
   a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-lower-case? arg)
	 (format-logged #t ";(char-lower-case? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? 1) 'error)
  (test (char-lower-case?) 'error)
  (test (char-lower-case? #\a #\b) 'error)
  (test (char-lower-case #\a) 'error)

;;  (test (char-lower-case? #\xb5) #t)  ; what is this?  in Snd it's #t, in ex1 it's #f -- is this a locale choice?
  (test (char-lower-case? #\xb6) #f)

  (for-each
   (lambda (c)
     (test (and (not (char-upper-case? c)) 
		(not (char-lower-case? c))) #t))
   (map integer->char (list 0 1 2 3 32 33 34 170 182 247)))


  
;;; --------------------------------------------------------------------------------
;;; char-upcase

  (test (char-upcase #\A) #\A)
  (test (char-upcase #\a) #\A)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\$) #\$)
  (test (char-upcase #\.) #\.)
  (test (char-upcase #\\) #\\)
  (test (char-upcase #\5) #\5)
  (test (char-upcase #\)) #\))
  (test (char-upcase #\%) #\%)
  (test (char-upcase #\0) #\0)
  (test (char-upcase #\_) #\_)
  (test (char-upcase #\?) #\?)
  (test (char-upcase #\space) #\space)
  (test (char-upcase #\newline) #\newline)
  (test (char-upcase #\null) #\null)
  (test (char-upper-case? (char-upcase #\?)) #f) ; !
  (test (char-lower-case? (char-downcase #\?)) #f)
  (test (char-upper-case? (char-upcase #\_)) #f)
  (test (or (char-upper-case? #\?) (char-lower-case? #\?)) #f)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-upcase arg1) arg2))
	 (format-logged #t ";(char-upcase ~A) != ~A?~%" arg1 arg2)))
   a-to-z
   cap-a-to-z)
  
  (do ((i 1 (+ i 1)))
      ((= i 256))
    (if (and (not (char=? (integer->char i) (char-upcase (integer->char i))))
	     (not (char-alphabetic? (integer->char i))))
	(format-logged #t ";(char-upcase ~A) -> ~A but not alphabetic?~%" (integer->char i) (char-upcase (integer->char i)))))

  (test (recompose 12 char-upcase #\a) #\A)
  (test (reinvert 12 char-upcase char-downcase #\a) #\a)

  (test (char-upcase) 'error)
  (test (char-upcase #\a #\b) 'error)
  (test (char-upcase #<eof>) 'error)
  (test (char-upcase #f) 'error)
  (test (char-upcase (list)) 'error)


  
;;; --------------------------------------------------------------------------------
;;; char-downcase

  (test (char-downcase #\A) #\a)
  (test (char-downcase #\a) #\a)
  (test (char-downcase #\?) #\?)
  (test (char-downcase #\$) #\$)
  (test (char-downcase #\.) #\.)
  (test (char-downcase #\_) #\_)
  (test (char-downcase #\\) #\\)
  (test (char-downcase #\5) #\5)
  (test (char-downcase #\)) #\))
  (test (char-downcase #\%) #\%)
  (test (char-downcase #\0) #\0)
  (test (char-downcase #\space) #\space)
  
  (for-each
   (lambda (arg1 arg2)
     (if (not (char=? (char-downcase arg1) arg2))
	 (format-logged #t ";(char-downcase ~A) != ~A?~%" arg1 arg2)))
   cap-a-to-z
   a-to-z)

  (test (recompose 12 char-downcase #\A) #\a)

  (test (char-downcase) 'error)
  (test (char-downcase #\a #\b) 'error)  


;;; --------------------------------------------------------------------------------
;;; char-numeric?  

  (test (char-numeric? #\a) #f)
  (test (char-numeric? #\5) #t)
  (test (char-numeric? #\A) #f)
  (test (char-numeric? #\z) #f)
  (test (char-numeric? #\Z) #f)
  (test (char-numeric? #\0) #t)
  (test (char-numeric? #\9) #t)
  (test (char-numeric? #\space) #f)
  (test (char-numeric? #\;) #f)
  (test (char-numeric? #\.) #f)
  (test (char-numeric? #\-) #f)
  (test (char-numeric? (integer->char 200)) #f)
  (test (char-numeric? (integer->char 128)) #f)
  (test (char-numeric? (integer->char 216)) #f) ; 0 slash
  (test (char-numeric? (integer->char 189)) #f) ; 1/2
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format-logged #t ";(char-numeric? ~A) -> #t?~%" arg)))
   cap-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-numeric? arg)
	 (format-logged #t ";(char-numeric? ~A) -> #t?~%" arg)))
   a-to-z)

  (test (char-numeric?) 'error)
  (test (char-numeric? #\a #\b) 'error)  

  
;;; --------------------------------------------------------------------------------
;;; char-whitespace?

  (test (char-whitespace? #\a) #f)
  (test (char-whitespace? #\A) #f)
  (test (char-whitespace? #\z) #f)
  (test (char-whitespace? #\Z) #f)
  (test (char-whitespace? #\0) #f)
  (test (char-whitespace? #\9) #f)
  (test (char-whitespace? #\space) #t)
  (test (char-whitespace? #\tab) #t)
  (test (char-whitespace? #\newline) #t)
  (test (char-whitespace? #\return) #t)
  (test (char-whitespace? #\linefeed) #t)
  (test (char-whitespace? #\null) #f)
  (test (char-whitespace? #\;) #f)
  (test (char-whitespace? #\xb) #t)
  (test (char-whitespace? #\x0b) #t)
  (test (char-whitespace? #\xc) #t)
  (test (char-whitespace? #\xd) #t) ; #\return
  (test (char-whitespace? #\xe) #f) 

  ;; unicode whitespace apparently:
  (test (char-whitespace? (integer->char 9)) #t)
  (test (char-whitespace? (integer->char 10)) #t)
  (test (char-whitespace? (integer->char 11)) #t)
  (test (char-whitespace? (integer->char 12)) #t)
  (test (char-whitespace? (integer->char 13)) #t)
  (test (char-whitespace? (integer->char 32)) #t)
  (test (char-whitespace? (integer->char 133)) #t)
  (test (char-whitespace? (integer->char 160)) #t)

  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format-logged #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   mixed-a-to-z)
  
  (for-each
   (lambda (arg)
     (if (char-whitespace? arg)
	 (format-logged #t ";(char-whitespace? ~A) -> #t?~%" arg)))
   digits)

  (test (char-whitespace?) 'error)
  (test (char-whitespace? #\a #\b) 'error)   
 
  
;;; --------------------------------------------------------------------------------
;;; char-alphabetic?

  (test (char-alphabetic? #\a) #t)
  (test (char-alphabetic? #\$) #f)
  (test (char-alphabetic? #\A) #t)
  (test (char-alphabetic? #\z) #t)
  (test (char-alphabetic? #\Z) #t)
  (test (char-alphabetic? #\0) #f)
  (test (char-alphabetic? #\9) #f)
  (test (char-alphabetic? #\space) #f)
  (test (char-alphabetic? #\;) #f)
  (test (char-alphabetic? #\.) #f)
  (test (char-alphabetic? #\-) #f)
  (test (char-alphabetic? #\_) #f)
  (test (char-alphabetic? #\^) #f)
  (test (char-alphabetic? #\[) #f)

  ;(test (char-alphabetic? (integer->char 200)) #t) ; ??
  (test (char-alphabetic? (integer->char 127)) #f)  ; backspace
  
  (for-each
   (lambda (arg)
     (if (char-alphabetic? arg)
	 (format-logged #t ";(char-alphabetic? ~A) -> #t?~%" arg)))
   digits)
  
  (for-each
   (lambda (arg)
     (if (not (char-alphabetic? arg))
	 (format-logged #t ";(char-alphabetic? ~A) -> #f?~%" arg)))
   mixed-a-to-z)

  (test (char-alphabetic?) 'error)
  (test (char-alphabetic? #\a #\b) 'error)  

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char-upper-case? char-lower-case? char-upcase char-downcase char-numeric? char-whitespace? char-alphabetic?))


  
  (test 
   (let ((unhappy ()))
     (do ((i 0 (+ i 1))) 
	 ((= i 256)) 
       (let* ((ch (integer->char i))
	      (chu (char-upcase ch))
	      (chd (char-downcase ch)))

	 (if (and (not (char=? ch chu))
		  (not (char-upper-case? chu)))
	     (format-logged #t ";(char-upper-case? (char-upcase ~C)) is #f~%" ch))

	 (if (and (not (char=? ch chd))
		  (not (char-lower-case? chd)))
	     (format-logged #t ";(char-lower-case? (char-downcase ~C)) is #f~%" ch))

	 (if (or (and (not (char=? ch chu))
		      (not (char=? ch (char-downcase chu))))
		 (and (not (char=? ch chd))
		      (not (char=? ch (char-upcase chd))))
		 (and (not (char=? ch chd))
		      (not (char=? ch chu)))
		 (not (char-ci=? chu chd))
		 (not (char-ci=? ch chu))
		 (and (char-alphabetic? ch)
		      (or (not (char-alphabetic? chd))
			  (not (char-alphabetic? chu))))
		 (and (char-numeric? ch)
		      (or (not (char-numeric? chd))
			  (not (char-numeric? chu))))
		 (and (char-whitespace? ch)
		      (or (not (char-whitespace? chd))
			  (not (char-whitespace? chu))))
		 (and (char-alphabetic? ch)
		      (char-whitespace? ch))
		 (and (char-numeric? ch)
		      (char-whitespace? ch))
		 (and (char-alphabetic? ch)
		      (char-numeric? ch)))
	     ;; there are characters that are alphabetic but the result of char-upcase is not an upper-case character
	     ;; 223 for example, or 186 for lower case
	     (set! unhappy (cons (format #f "~C: ~C ~C (~D)~%" ch chu chd i) unhappy)))))
     unhappy)
   ())
  
  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op #\a arg) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))

  (for-each
   (lambda (op)
     (for-each
      (lambda (arg)
	(test (op arg #\a) 'error))
      (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	    3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?))


;;; --------------------------------------------------------------------------------
;;; char=?

  (test (char=? #\d #\d) #t)
  (test (char=? #\A #\a) #f)
  (test (char=? #\d #\x) #f)
  (test (char=? #\d #\D) #f)
  (test (char=? #\a #\a) #t)
  (test (char=? #\A #\B) #f)
  (test (char=? #\a #\b) #f)
  (test (char=? #\9 #\0) #f)
  (test (char=? #\A #\A) #t)
  (test (char=? #\  #\space) #t)
  (let ((i (char->integer #\space)))
    (test (char=? (integer->char i) #\space) #t))
  (test (char=? (integer->char (char->integer #\")) #\") #t)
  (test (char=? #\x65 #\e) #t)
  
  (test (char=? #\d #\d #\d #\d) #t)
  (test (char=? #\d #\d #\x #\d) #f)
  (test (char=? #\d #\y #\x #\c) #f)
  (test (apply char=? cap-a-to-z) #f)
  (test (apply char=? mixed-a-to-z) #f)
  (test (apply char=? digits) #f)
  (test (char=? #\d #\c #\d) #f)

  (test (char=? #\a) 'error)
  (test (char=?) 'error)
  (test (char=? #\a 0) 'error)
  (test (char=? #\a #\b 0) 'error)
  

;;; --------------------------------------------------------------------------------
;;; char<?  

  (test (char<? #\z #\0) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\d #\d) #f)
  (test (char<? #\d #\x) #t)
  (test (char<? #\A #\B) #t)
  (test (char<? #\a #\b) #t)
  (test (char<? #\9 #\0) #f)
  (test (char<? #\A #\A) #f)
  (test (char<? #\space #\space) #f)
  
  (test (char<? #\a #\e #\y #\z) #t)
  (test (char<? #\a #\e #\e #\y) #f)
  (test (apply char<? a-to-z) #t)
  (test (apply char<? cap-a-to-z) #t)
  (test (apply char<? mixed-a-to-z) #f)
  (test (apply char<? digits) #t)
  (test (apply char<? (reverse a-to-z)) #f)
  (test (apply char<? (reverse cap-a-to-z)) #f)
  (test (apply char<? (reverse mixed-a-to-z)) #f)
  (test (apply char<? (reverse digits)) #f)
  (test (char<? #\b #\c #\a) #f)
  (test (char<? #\B #\B #\A) #f)
  (test (char<? #\b #\c #\e) #t)
  (test (char<? (integer->char #xf0) (integer->char #x70)) #f)

  (test (char<?) 'error)
  (test (char<? #\b #\a "hi") 'error)
  (test (char<? #\b #\a 0) 'error)
  (test (char<? (integer->char 0) (integer->char 255)) #t)
  
  

;;; --------------------------------------------------------------------------------
;;; char<=?

  (test (char<=? #\d #\x) #t)
  (test (char<=? #\d #\d) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\A #\B) #t)
  (test (char<=? #\a #\b) #t)
  (test (char<=? #\9 #\0) #f)
  (test (char<=? #\A #\A) #t)
  (test (char<=? #\space #\space) #t)
  
  (test (char<=? #\a #\e #\y #\z) #t)
  (test (char<=? #\a #\e #\e #\y) #t)
  (test (char<=? #\e #\e #\d #\y) #f)
  (test (apply char<=? a-to-z) #t)
  (test (apply char<=? cap-a-to-z) #t)
  (test (apply char<=? mixed-a-to-z) #f)
  (test (apply char<=? digits) #t)
  (test (apply char<=? (reverse a-to-z)) #f)
  (test (apply char<=? (reverse cap-a-to-z)) #f)
  (test (apply char<=? (reverse mixed-a-to-z)) #f)
  (test (apply char<=? (reverse digits)) #f)
  (test (char<=? #\b #\c #\a) #f)
  (test (char<=? #\B #\B #\A) #f)
  (test (char<=? #\b #\c #\e) #t)
  
  (test (char<=? #\b #\a "hi") 'error)
  (test (char<=? #\b #\a 0) 'error)
  (test (char<=?) 'error)


  
;;; --------------------------------------------------------------------------------
;;; char>?

  (test (char>? #\e #\d) #t)
  (test (char>? #\z #\a) #t)
  (test (char>? #\A #\B) #f)
  (test (char>? #\a #\b) #f)
  (test (char>? #\9 #\0) #t)
  (test (char>? #\A #\A) #f)
  (test (char>? #\space #\space) #f)
  
  (test (char>? #\d #\c #\b #\a) #t)
  (test (char>? #\d #\d #\c #\a) #f)
  (test (char>? #\e #\d #\b #\c #\a) #f)
  (test (apply char>? a-to-z) #f)
  (test (apply char>? cap-a-to-z) #f)
  (test (apply char>? mixed-a-to-z) #f)
  (test (apply char>? digits) #f)
  (test (apply char>? (reverse a-to-z)) #t)
  (test (apply char>? (reverse cap-a-to-z)) #t)
  (test (apply char>? (reverse mixed-a-to-z)) #f)
  (test (apply char>? (reverse digits)) #t)
  (test (char>? #\d #\c #\a) #t)
  (test (char>? #\d #\c #\c) #f)
  (test (char>? #\B #\B #\C) #f)
  (test (char>? #\b #\c #\e) #f)
  (test (char>? (integer->char #xf0) (integer->char #x70)) #t)

  (test (char>? #\a #\b "hi") 'error)
  (test (char>? #\a #\b 0) 'error)
  (test (char>?) 'error)

  
  
;;; --------------------------------------------------------------------------------
;;; char>=?

  (test (char>=? #\e #\d) #t)
  (test (char>=? #\A #\B) #f)
  (test (char>=? #\a #\b) #f)
  (test (char>=? #\9 #\0) #t)
  (test (char>=? #\A #\A) #t)
  (test (char>=? #\space #\space) #t)
  
  (test (char>=? #\d #\c #\b #\a) #t)
  (test (char>=? #\d #\d #\c #\a) #t)
  (test (char>=? #\e #\d #\b #\c #\a) #f)
  (test (apply char>=? a-to-z) #f)
  (test (apply char>=? cap-a-to-z) #f)
  (test (apply char>=? mixed-a-to-z) #f)
  (test (apply char>=? digits) #f)
  (test (apply char>=? (reverse a-to-z)) #t)
  (test (apply char>=? (reverse cap-a-to-z)) #t)
  (test (apply char>=? (reverse mixed-a-to-z)) #f)
  (test (apply char>=? (reverse digits)) #t)
  (test (char>=? #\d #\c #\a) #t)
  (test (char>=? #\d #\c #\c) #t)
  (test (char>=? #\B #\B #\C) #f)
  (test (char>=? #\b #\c #\e) #f)

  (test (char>=? #\a #\b "hi") 'error)
  (test (char>=? #\a #\b 0) 'error)
  (test (char>=?) 'error)

  
  
;;; --------------------------------------------------------------------------------
;;; char-ci=?

  (test (char-ci=? #\A #\B) #f)
  (test (char-ci=? #\a #\B) #f)
  (test (char-ci=? #\A #\b) #f)
  (test (char-ci=? #\a #\b) #f)
  (test (char-ci=? #\9 #\0) #f)
  (test (char-ci=? #\A #\A) #t)
  (test (char-ci=? #\A #\a) #t)
  (test (char-ci=? #\a #\A) #t)
  (test (char-ci=? #\space #\space) #t)
  
  (test (char-ci=? #\d #\D #\d #\d) #t)
  (test (char-ci=? #\d #\d #\X #\d) #f)
  (test (char-ci=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci=? cap-a-to-z) #f)
  (test (apply char-ci=? mixed-a-to-z) #f)
  (test (apply char-ci=? digits) #f)
  (test (char-ci=? #\d #\c #\d) #f)

  (test (char-ci=?) 'error)
  (test (char-ci=? #\a #\b 0) 'error)
  

  
;;; --------------------------------------------------------------------------------
;;; char-ci<?

  (test (char-ci<? #\A #\B) #t)
  (test (char-ci<? #\a #\B) #t)
  (test (char-ci<? #\A #\b) #t)
  (test (char-ci<? #\a #\b) #t)
  (test (char-ci<? #\9 #\0) #f)
  (test (char-ci<? #\0 #\9) #t)
  (test (char-ci<? #\A #\A) #f)
  (test (char-ci<? #\A #\a) #f)
  (test (char-ci<? #\Y #\_) #t)
  (test (char-ci<? #\\ #\J) #f)
  (test (char-ci<? #\_ #\e) #f)
  (test (char-ci<? #\t #\_) #t)
  (test (char-ci<? #\a #\]) #t)
  (test (char-ci<? #\z #\^) #t)
  
  (test (char-ci<? #\b #\a "hi") 'error)
  (test (char-ci<? #\b #\a 0) 'error)
  (test (char-ci>? (integer->char #xf0) (integer->char #x70)) #t)

#|
    ;; this tries them all:
    (do ((i 0 (+ i 1)))
	((= i 256))
      (do ((k 0 (+ k 1)))
	  ((= k 256))
	(let ((c1 (integer->char i))
	      (c2 (integer->char k)))
	  (for-each
	   (lambda (op1 op2)
	     (if (not (eq? (op1 c1 c2) (op2 (string c1) (string c2))))
		 (format-logged #t ";(~A|~A ~A ~A) -> ~A|~A~%" op1 op2 c1 c2 (op1 c1 c2) (op2 (string c1) (string c2)))))
	   (list char=? char<? char<=? char>? char>=? char-ci=? char-ci<? char-ci<=? char-ci>? char-ci>=?)
	   (list string=? string<? string<=? string>? string>=? string-ci=? string-ci<? string-ci<=? string-ci>? string-ci>=?)))))
|#
  
  (test (char-ci<? #\d #\D #\d #\d) #f)
  (test (char-ci<? #\d #\d #\X #\d) #f)
  (test (char-ci<? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<? cap-a-to-z) #t)
  (test (apply char-ci<? mixed-a-to-z) #t)
  (test (apply char-ci<? digits) #t)
  (test (char-ci<? #\d #\c #\d) #f)
  (test (char-ci<? #\b #\c #\a) #f)
  (test (char-ci<? #\b #\C #\e) #t)
  (test (char-ci<? #\3 #\? #\Z #\[) #t)
  
  (test (char-ci>? #\a #\b "hi") 'error)
  (test (char-ci>? #\a #\b 0) 'error)


  
;;; --------------------------------------------------------------------------------
;;; char-ci>?

  (test (char-ci>? #\A #\B) #f)
  (test (char-ci>? #\a #\B) #f)
  (test (char-ci>? #\A #\b) #f)
  (test (char-ci>? #\a #\b) #f)
  (test (char-ci>? #\9 #\0) #t)
  (test (char-ci>? #\A #\A) #f)
  (test (char-ci>? #\A #\a) #f)
  (test (char-ci>? #\^ #\a) #t)
  (test (char-ci>? #\_ #\e) #t)
  (test (char-ci>? #\[ #\S) #t)
  (test (char-ci>? #\\ #\l) #t)
  (test (char-ci>? #\t #\_) #f)
  (test (char-ci>? #\a #\]) #f)
  (test (char-ci>? #\z #\^) #f)
  (test (char-ci>? #\] #\X) #t)
  
  (test (char-ci>? #\d #\D #\d #\d) #f)
  (test (char-ci>? #\d #\d #\X #\d) #f)
  (test (char-ci>? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>? cap-a-to-z) #f)
  (test (apply char-ci>? mixed-a-to-z) #f)
  (test (apply char-ci>? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>? digits) #f)
  (test (char-ci>? #\d #\c #\d) #f)
  (test (char-ci>? #\b #\c #\a) #f)
  (test (char-ci>? #\d #\C #\a) #t)
  
  
;;; --------------------------------------------------------------------------------
;;; char-ci<=?

  (test (char-ci<=? #\A #\B) #t)
  (test (char-ci<=? #\a #\B) #t)
  (test (char-ci<=? #\A #\b) #t)
  (test (char-ci<=? #\a #\b) #t)
  (test (char-ci<=? #\9 #\0) #f)
  (test (char-ci<=? #\A #\A) #t)
  (test (char-ci<=? #\A #\a) #t)
  (test (char-ci<=? #\` #\H) #f)
  (test (char-ci<=? #\[ #\m) #f)
  (test (char-ci<=? #\j #\`) #t)
  (test (char-ci<=? #\\ #\E) #f)
  (test (char-ci<=? #\t #\_) #t)
  (test (char-ci<=? #\a #\]) #t)
  (test (char-ci<=? #\z #\^) #t)
  
  (test (char-ci<=? #\d #\D #\d #\d) #t)
  (test (char-ci<=? #\d #\d #\X #\d) #f)
  (test (char-ci<=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci<=? cap-a-to-z) #t)
  (test (apply char-ci<=? mixed-a-to-z) #t)
  (test (apply char-ci<=? digits) #t)
  (test (char-ci<=? #\d #\c #\d) #f)
  (test (char-ci<=? #\b #\c #\a) #f)
  (test (char-ci<=? #\b #\c #\C) #t)
  (test (char-ci<=? #\b #\C #\e) #t)
  
  (test (char-ci<=? #\b #\a "hi") 'error)
  (test (char-ci<=? #\b #\a 0) 'error)


  
;;; --------------------------------------------------------------------------------
;;; char-ci>=?

  (test (char-ci>=? #\A #\B) #f)
  (test (char-ci>=? #\a #\B) #f)
  (test (char-ci>=? #\A #\b) #f)
  (test (char-ci>=? #\a #\b) #f)
  (test (char-ci>=? #\9 #\0) #t)
  (test (char-ci>=? #\A #\A) #t)
  (test (char-ci>=? #\A #\a) #t)
  (test (char-ci>=? #\Y #\_) #f)
  (test (char-ci>=? #\` #\S) #t)
  (test (char-ci>=? #\[ #\Y) #t)
  (test (char-ci>=? #\t #\_) #f)
  (test (char-ci>=? #\a #\]) #f)
  (test (char-ci>=? #\z #\^) #f)
  
  (test (char-ci>=? #\d #\D #\d #\d) #t)
  (test (char-ci>=? #\d #\d #\X #\d) #f)
  (test (char-ci>=? #\d #\Y #\x #\c) #f)
  (test (apply char-ci>=? cap-a-to-z) #f)
  (test (apply char-ci>=? mixed-a-to-z) #f)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? (reverse mixed-a-to-z)) #t)
  (test (apply char-ci>=? digits) #f)
  (test (char-ci>=? #\d #\c #\d) #f)
  (test (char-ci>=? #\b #\c #\a) #f)
  (test (char-ci>=? #\d #\D #\a) #t)
  (test (char-ci>=? #\\ #\J #\+) #t)

  (test (char-ci>=? #\a #\b "hi") 'error)
  (test (char-ci>=? #\a #\b 0) 'error)

  ) ; end let with a-to-z



;;; --------------------------------------------------------------------------------
;;; integer->char
;;; char->integer

(test (integer->char (char->integer #\.)) #\.)
(test (integer->char (char->integer #\A)) #\A)
(test (integer->char (char->integer #\a)) #\a)
(test (integer->char (char->integer #\space)) #\space)
(test (char->integer (integer->char #xf0)) #xf0)

(do ((i 0 (+ i 1)))
    ((= i 256)) 
  (if (not (= (char->integer (integer->char i)) i)) 
      (format-logged #t ";char->integer ~D ~A != ~A~%" i (integer->char i) (char->integer (integer->char i)))))

(test (reinvert 12 integer->char char->integer 60) 60)

(test (char->integer 33) 'error)
(test (char->integer) 'error)
(test (integer->char) 'error)
(test (integer->char (expt 2 31)) 'error)
(test (integer->char (expt 2 32)) 'error)
(test (integer->char 12 14) 'error)
(test (char->integer #\a #\b) 'error)
;(test (char->integer #\ÿ) 255) ; emacs confusion?
(test (eval-string (string-append "(char->integer " (format #f "#\\~C" (integer->char 255)) ")")) 255)

(for-each
 (lambda (arg)
   (test (char->integer arg) 'error))
 (list -1 1 0 123456789 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (integer->char arg) 'error))
 (list -1 257 123456789 -123456789 #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi most-positive-fixnum 1/0 (if #f #f) (lambda (a) (+ a 1))))

(test (#\a) 'error)
(test (#\newline 1) 'error)



;;; --------------------------------------------------------------------------------
;;; STRINGS
;;; --------------------------------------------------------------------------------

;;; --------------------------------------------------------------------------------
;;; string?

(test (string? "abc") #t)
(test (string? ':+*/-) #f)
(test (string? "das ist einer der teststrings") #t)
(test (string? '(das ist natuerlich falsch)) #f)
(test (string? "aaaaaa") #t)
(test (string? #\a) #f)
(test (string? "\"\\\"") #t)
(test (string? lambda) #f)
(test (string? format) #f)

(for-each
 (lambda (arg)
   (test (string? arg) #f))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string?) 'error)
(test (string? "hi" "ho") 'error)
(test (string? #\null) #f)



;;; --------------------------------------------------------------------------------
;;; string=?

(test (string=? "foo" "foo") #t)
(test (string=? "foo" "FOO") #f)
(test (string=? "foo" "bar") #f)
(test (string=? "FOO" "FOO") #t)
(test (string=? "A" "B") #f)
(test (string=? "a" "b") #f)
(test (string=? "9" "0") #f)
(test (string=? "A" "A") #t)
(test (string=? "" "") #t)
(test (string=? (string #\newline) (string #\newline)) #t)

(test (string=? "A" "B" "a") #f)
(test (string=? "A" "A" "a") #f)
(test (string=? "A" "A" "A") #t)
(test (string=? "foo" "foo" "foo") #t)
(test (string=? "foo" "foo" "") #f)
(test (string=? "foo" "foo" "fOo") #f)

(test (string=? "foo" "FOO" 1.0) 'error)

(test (let ((str (string #\" #\1 #\\ #\2 #\")))	(string=? str "\"1\\2\"")) #t)
(test (let ((str (string #\\ #\\ #\\)))	(string=? str "\\\\\\")) #t)
(test (let ((str (string #\")))	(string=? str "\"")) #t)
(test (let ((str (string #\\ #\"))) (string=? str "\\\"")) #t)
(test (let ((str (string #\space #\? #\)))) (string=? str " ?)")) #t)
(test (let ((str (string #\# #\\ #\t))) (string=? str "#\\t")) #t)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #x70) #\x)) #f)
(test (string=? (string #\x (integer->char #xf0) #\x) (string #\x (integer->char #xf0) #\x)) #t)

(test (string=? (string) "") #t)
(test (string=? (string) (make-string 0)) #t)
(test (string=? (string-copy (string)) (make-string 0)) #t)
(test (string=? "" (make-string 0)) #t)
(test (string=? "" (string-append)) #t)
(test (string=? (string #\space #\newline) " \n") #t)

(test (string=? "......" "...\
...") #t)
(test (string=? "\n" (string #\newline)) #t)
(test (string=? "\
\
\
\
" "") #t)
(test (string=? "" (string #\null)) #f)
(test (string=? (string #\null #\null) (string #\null)) #f)
(test (string=? "" "asd") #f)
(test (string=? "asd" "") #f)
(test (string=? "xx" (make-string 2 #\x) (string #\x #\x) (list->string (list #\x #\x)) (substring "axxb" 1 3) (string-append "x" "x")) #t)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2)) #f)
(test (string=? (make-string 3 #\space) (let ((s (make-string 4 #\space))) (set! (s 3) #\null) s)) #f)
(test "\x3012" "012")

(for-each
 (lambda (arg)
   (test (string=? "hi" arg) 'error)
   (test (string=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


;; this strikes me as highly dubious
(test (call-with-input-string "1\n2" (lambda (p) (read p))) 1)
(test (call-with-input-string "1\\ \n2" (lambda (p) (read p))) (symbol "1\\"))

(test (call-with-input-string "1\
2" (lambda (p) (read p))) 12)

;; do we guarantee that read takes place in the current environment?
(test (let ((xyzzy 32)) (call-with-input-string "xy\
zzy" (lambda (p) (read p)))) 'xyzzy)

(test (let ((xyzzy 32)) (call-with-input-string "xy\
zzy" (lambda (p) (eval (read p))))) 32)

(test (let ((xyzzy 32)) (call-with-input-string "(set! xyzzy;\
 this is presumably a comment
 321)" (lambda (p) (eval (read p)))) xyzzy) 321)

(test (let ((xyzzy 32)) (call-with-input-string "(set! xyzzy;\
 this is presumably a comment;\
 and more commentary
 321)" (lambda (p) (eval (read p)))) xyzzy) 321)



;;; --------------------------------------------------------------------------------
;;; string<?

(test (string<? "aaaa" "aaab") #t)
(test (string<? "aaaa" "aaaaa") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "a" "abcdefgh") #t)
(test (string<? "abc" "abcdefgh") #t)
(test (string<? "cabc" "abcdefgh") #f)
(test (string<? "abcdefgh" "abcdefgh") #f)
(test (string<? "xyzabc" "abcdefgh") #f)
(test (string<? "abc" "xyzabcdefgh") #t)
(test (string<? "abcdefgh" "") #f)
(test (string<? "abcdefgh" "a") #f)
(test (string<? "abcdefgh" "abc") #f)
(test (string<? "abcdefgh" "cabc") #t)
(test (string<? "abcdefgh" "xyzabc") #t)
(test (string<? "xyzabcdefgh" "abc") #f)
(test (string<? "abcdef" "bcdefgh") #t)
(test (string<? "" "abcdefgh") #t)
(test (string<? "" "") #f)
(test (string<? "A" "B") #t)
(test (string<? "a" "b") #t)
(test (string<? "9" "0") #f)
(test (string<? "A" "A") #f)

(test (string<? "A" "B" "A") #f)
(test (string<? "A" "A" "B") #f)
(test (string<? "A" "A" "A") #f)
(test (string<? "B" "B" "C") #f)
(test (string<? "foo" "foo" "foo") #f)
(test (string<? "foo" "foo" "") #f)
(test (string<? "foo" "foo" "fOo") #f)

(test (string<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<? s1 s2)) #t)
(test (not (string<? "foo\x0a" "foo\x0a")) #t)
(test (string<? "foo\x0a" "foo\x0b") #t)

(test (string<? (string (integer->char #xf0)) (string (integer->char #x70))) #f) 

(for-each
 (lambda (arg)
   (test (string<? "hi" arg) 'error)
   (test (string<? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string>?

(test (string>? "aaab" "aaaa") #t)
(test (string>? "aaaaa" "aaaa") #t)
(test (string>? "" "abcdefgh") #f)
(test (string>? "a" "abcdefgh") #f)
(test (string>? "abc" "abcdefgh") #f)
(test (string>? "cabc" "abcdefgh") #t)
(test (string>? "abcdefgh" "abcdefgh") #f)
(test (string>? "xyzabc" "abcdefgh") #t)
(test (string>? "abc" "xyzabcdefgh") #f)
(test (string>? "abcdefgh" "") #t)
(test (string>? "abcdefgh" "a") #t)
(test (string>? "abcdefgh" "abc") #t)
(test (string>? "abcdefgh" "cabc") #f)
(test (string>? "abcdefgh" "xyzabc") #f)
(test (string>? "xyzabcdefgh" "abc") #t)
(test (string>? "abcde" "bc") #f)
(test (string>? "bcdef" "abcde") #t)
(test (string>? "bcdef" "abcdef") #t)
(test (string>? "" "") #f)
(test (string>? "A" "B") #f)
(test (string>? "a" "b") #f)
(test (string>? "9" "0") #t)
(test (string>? "A" "A") #f)

(test (string>? "A" "B" "a") #f)
(test (string>? "C" "B" "A") #t)
(test (string>? "A" "A" "A") #f)
(test (string>? "B" "B" "A") #f)
(test (string>? "foo" "foo" "foo") #f)
(test (string>? "foo" "foo" "") #f)
(test (string>? "foo" "foo" "fOo") #f)

(test (string>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>? s1 s2)) #f)

(test (string>? (string (integer->char #xf0)) (string (integer->char #x70))) #t) ; ??

(for-each
 (lambda (arg)
   (test (string>? "hi" arg) 'error)
   (test (string>? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string<=?

(test (string<=? "aaa" "aaaa") #t)
(test (string<=? "aaaaa" "aaaa") #f)
(test (string<=? "a" "abcdefgh") #t)
(test (string<=? "abc" "abcdefgh") #t)
(test (string<=? "aaabce" "aaabcdefgh") #f)
(test (string<=? "cabc" "abcdefgh") #f)
(test (string<=? "abcdefgh" "abcdefgh") #t)
(test (string<=? "xyzabc" "abcdefgh") #f)
(test (string<=? "abc" "xyzabcdefgh") #t)
(test (string<=? "abcdefgh" "") #f)
(test (string<=? "abcdefgh" "a") #f)
(test (string<=? "abcdefgh" "abc") #f)
(test (string<=? "abcdefgh" "cabc") #t)
(test (string<=? "abcdefgh" "xyzabc") #t)
(test (string<=? "xyzabcdefgh" "abc") #f)
(test (string<=? "abcdef" "bcdefgh") #t)
(test (string<=? "" "") #t)
(test (string<=? "A" "B") #t)
(test (string<=? "a" "b") #t)
(test (string<=? "9" "0") #f)
(test (string<=? "A" "A") #t)

(test (string<=? "A" "B" "C") #t)
(test (string<=? "C" "B" "A") #f)
(test (string<=? "A" "B" "B") #t)
(test (string<=? "A" "A" "A") #t)
(test (string<=? "B" "B" "A") #f)
(test (string<=? "foo" "foo" "foo") #t)
(test (string<=? "foo" "foo" "") #f)
(test (string<=? "foo" "foo" "fooo") #t)

(test (string<=? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #f)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string<=? "hi" arg) 'error)
   (test (string<=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string>=?

(test (string>=? "aaaaa" "aaaa") #t)
(test (string>=? "aaaa" "aaaa") #t)
(test (string>=? "aaa" "aaaa") #f)
(test (string>=? "" "abcdefgh") #f)
(test (string>=? "a" "abcdefgh") #f)
(test (string>=? "abc" "abcdefgh") #f)
(test (string>=? "cabc" "abcdefgh") #t)
(test (string>=? "abcdefgh" "abcdefgh") #t)
(test (string>=? "xyzabc" "abcdefgh") #t)
(test (string>=? "abc" "xyzabcdefgh") #f)
(test (string>=? "abcdefgh" "") #t)
(test (string>=? "abcdefgh" "a") #t)
(test (string>=? "abcdefgh" "abc") #t)
(test (string>=? "abcdefgh" "cabc") #f)
(test (string>=? "abcdefgh" "xyzabc") #f)
(test (string>=? "xyzabcdefgh" "abc") #t)
(test (string>=? "bcdef" "abcdef") #t)
(test (string>=? "A" "B") #f)
(test (string>=? "a" "b") #f)
(test (string>=? "9" "0") #t)
(test (string>=? "A" "A") #t)
(test (string>=? "" "") #t)

(test (string>=? "A" "B" "C") #f)
(test (string>=? "C" "B" "A") #t)
(test (string>=? "C" "B" "B") #t)
(test (string>=? "A" "B" "B") #f)
(test (string>=? "A" "A" "A") #t)
(test (string>=? "B" "B" "A") #t)
(test (string>=? "B" "B" "C") #f)
(test (string>=? "foo" "foo" "foo") #t)
(test (string>=? "foo" "foo" "") #t)
(test (string>=? "foo" "foo" "fo") #t)

(test (string>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "123")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)
(test (let ((s1 "123") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string>=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string>=? "hi" arg) 'error)
   (test (string>=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci=?

(test (string-ci=? "A" "B") #f)
(test (string-ci=? "a" "B") #f)
(test (string-ci=? "A" "b") #f)
(test (string-ci=? "a" "b") #f)
(test (string-ci=? "9" "0") #f)
(test (string-ci=? "A" "A") #t)
(test (string-ci=? "A" "a") #t)
(test (string-ci=? "" "") #t)
(test (string-ci=? "aaaa" "AAAA") #t)
(test (string-ci=? "aaaa" "Aaaa") #t)

(test (string-ci=? "A" "B" "a") #f)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "A" "A" "a") #t)
(test (string-ci=? "foo" "foo" "foo") #t)
(test (string-ci=? "foo" "foo" "") #f)
(test (string-ci=? "foo" "Foo" "fOo") #t)

(test (string-ci=? "foo" "GOO" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci=? "hi" arg) 'error)
   (test (string-ci=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


(when full-test
  (let ((size 15)
	(tries 10000))
    (let ((str1 (make-string size))
	  (str2 (make-string size)))
      (do ((i 0 (+ i 1)))
	  ((= i tries))
	(do ((k 0 (+ k 1)))
	    ((= k size))
	  (set! (str1 k) (integer->char (random 128)))
	  (if (> (random 10) 4)
	      (set! (str2 k) (char-upcase (str1 k)))
	      (set! (str2 k) (char-downcase (str1 k)))))
	(if (not (string-ci=? str1 str2))
	    (format-logged #t "not =: ~S ~S~%" str1 str2))
	(if (and (string-ci<? str1 str2)
		 (string-ci>=? str1 str2))
	    (format-logged #t "< : ~S ~S~%" str1 str2))
	(if (and (string-ci>? str1 str2)
		 (string-ci<=? str1 str2))
	    (format-logged #t "> : ~S ~S~%" str1 str2))))))



;;; --------------------------------------------------------------------------------
;;; string-ci<?

(test (string-ci<? "a" "Aa") #t)
(test (string-ci<? "A" "B") #t)
(test (string-ci<? "a" "B") #t)
(test (string-ci<? "A" "b") #t)
(test (string-ci<? "a" "b") #t)
(test (string-ci<? "9" "0") #f)
(test (string-ci<? "0" "9") #t)
(test (string-ci<? "A" "A") #f)
(test (string-ci<? "A" "a") #f)
(test (string-ci<? "" "") #f)

(test (string-ci<? "t" "_") #t)
(test (string-ci<? "a" "]") #t)
(test (string-ci<? "z" "^") #t)
(test (string-ci<? "]4.jVKo\\\\^:\\A9Z4" "MImKA[mNv1`") #f)

(test (string-ci<? "A" "B" "A") #f)
(test (string-ci<? "A" "A" "B") #f)
(test (string-ci<? "A" "A" "A") #f)
(test (string-ci<? "B" "B" "C") #f)
(test (string-ci<? "B" "b" "C") #f)
(test (string-ci<? "foo" "foo" "foo") #f)
(test (string-ci<? "foo" "foo" "") #f)
(test (string-ci<? "foo" "foo" "fOo") #f)
(test (string-ci<? "34ZsfQD<obff33FBPFl" "7o" "9l7OM" "FC?M63=" "rLM5*J") #t)
(test (string-ci<? "NX7" "-;h>P" "DMhk3Bg") #f)
(test (string-ci<? "+\\mZl" "bE7\\e(HaW5CDXbPi@U_" "B_") #t)

(if (char-ci<? (integer->char #xf0) (integer->char #x70))
    (test (string-ci<? (string (integer->char #xf0)) (string (integer->char #x70))) #t))

(test (string-ci<? "foo" "fo" 1.0) 'error)
(test (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "1234") (s2 "1234")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "1234") (s2 "124")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "123") (s2 "12")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<? "hi" arg) 'error)
   (test (string-ci<? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci>?

(test (string-ci>? "Aaa" "AA") #t)
(test (string-ci>? "A" "B") #f)
(test (string-ci>? "a" "B") #f)
(test (string-ci>? "A" "b") #f)
(test (string-ci>? "a" "b") #f)
(test (string-ci>? "9" "0") #t)
(test (string-ci>? "A" "A") #f)
(test (string-ci>? "A" "a") #f)
(test (string-ci>? "" "") #f)
(test (string-ci>? "Z" "DjNTl0") #t)
(test (string-ci>? "2399dt7BVN[,A" "^KHboHV") #f)

(test (string-ci>? "t" "_") #f)
(test (string-ci>? "a" "]") #f)
(test (string-ci>? "z" "^") #f)
(test (string-ci>? "R*95oG.k;?" "`2?J6LBbLG^alB[fMD") #f)
(test (string-ci>? "]" "X") #t)

(test (string-ci>? "A" "B" "a") #f)
(test (string-ci>? "C" "b" "A") #t)
(test (string-ci>? "a" "A" "A") #f)
(test (string-ci>? "B" "B" "A") #f)
(test (string-ci>? "foo" "foo" "foo") #f)
(test (string-ci>? "foo" "foo" "") #f)
(test (string-ci>? "foo" "foo" "fOo") #f)
(test (string-ci>? "ZNiuEa@/V" "KGbKliYMY" "9=69q3ica" ":]") #f)
(test (string-ci>? "^" "aN@di;iEO" "7*9q6uPmX9)PaY,6J" "15vH") #t)

(test (string-ci>? "foo" "fooo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>? "hi" arg) 'error)
   (test (string-ci>? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci<=?

(test (string-ci<=? "A" "B") #t)
(test (string-ci<=? "a" "B") #t)
(test (string-ci<=? "A" "b") #t)
(test (string-ci<=? "a" "b") #t)
(test (string-ci<=? "9" "0") #f)
(test (string-ci<=? "A" "A") #t)
(test (string-ci<=? "A" "a") #t)
(test (string-ci<=? "" "") #t)
(test (string-ci<=? ":LPC`" ",O0>affA?(") #f)

(test (string-ci<=? "t" "_") #t)
(test (string-ci<=? "a" "]") #t)
(test (string-ci<=? "z" "^") #t)
(test (string-ci<=? "G888E>beF)*mwCNnagP" "`2uTd?h") #t)

(test (string-ci<=? "A" "b" "C") #t)
(test (string-ci<=? "c" "B" "A") #f)
(test (string-ci<=? "A" "B" "B") #t)
(test (string-ci<=? "a" "A" "A") #t)
(test (string-ci<=? "B" "b" "A") #f)
(test (string-ci<=? "foo" "foo" "foo") #t)
(test (string-ci<=? "foo" "foo" "") #f)
(test (string-ci<=? "FOO" "fOo" "fooo") #t)
(test (string-ci<=? "78mdL82*" "EFaCrIdm@_D+" "eMu\\@dSSY") #t)
(test (string-ci<=? "`5pNuFc3PM<rNs" "e\\Su_raVNk6HD" "vXnuN7?S0?S(w+M?p") #f)

(test (string-ci<=? "fOo" "fo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #f)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci<=? s1 s2)) #t)

(for-each
 (lambda (arg)
   (test (string-ci<=? "hi" arg) 'error)
   (test (string-ci<=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ci>=?

(test (string-ci>=? "A" "B") #f)
(test (string-ci>=? "a" "B") #f)
(test (string-ci>=? "A" "b") #f)
(test (string-ci>=? "a" "b") #f)
(test (string-ci>=? "9" "0") #t)
(test (string-ci>=? "A" "A") #t)
(test (string-ci>=? "A" "a") #t)
(test (string-ci>=? "" "") #t)
(test (string-ci>=? "5d7?[o[:hop=ktv;9)" "p^r9;TAXO=^") #f)

(test (string-ci>=? "t" "_") #f)
(test (string-ci>=? "a" "]") #f)
(test (string-ci>=? "z" "^") #f)
(test (string-ci>=? "jBS" "`<+s[[:`l") #f)

(test (string-ci>=? "A" "b" "C") #f)
(test (string-ci>=? "C" "B" "A") #t)
(test (string-ci>=? "C" "B" "b") #t)
(test (string-ci>=? "a" "B" "B") #f)
(test (string-ci>=? "A" "A" "A") #t)
(test (string-ci>=? "B" "B" "A") #t)
(test (string-ci>=? "B" "b" "C") #f)
(test (string-ci>=? "foo" "foo" "foo") #t)
(test (string-ci>=? "foo" "foo" "") #t)
(test (string-ci>=? "foo" "foo" "fo") #t)
(test (string-ci>=? "tF?8`Sa" "NIkMd7" "f`" "1td-Z?teE" "-ik1SK)hh)Nq].>") #t)
(test (string-ci>=? "Z6a8P" "^/VpmWwt):?o[a9\\_N" "8[^h)<KX?[utsc") #f)

(test (string-ci>=? "fo" "foo" 1.0) 'error)
(test (let ((s1 "abcd") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abcd") (s2 "ABCE")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)
(test (let ((s1 "abcd") (s2 "ABC")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #t)
(test (let ((s1 "abc") (s2 "ABCD")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string-ci>=? s1 s2)) #f)

(for-each
 (lambda (arg)
   (test (string-ci>=? "hi" arg) 'error)
   (test (string-ci>=? arg "hi") 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-length

(test (string-length "abc") 3)
(test (string-length "") 0)
(test (string-length (string)) 0)
(test (string-length "\"\\\"") 3)
(test (string-length (string #\newline)) 1)
(test (string-length "hi there") 8)
(test (string-length "\"") 1)
(test (string-length "\\") 1)
(test (string-length "\n") 1)
(test (string-length (make-string 100 #\a)) 100)
(test (string-length "1\\2") 3)
(test (string-length "1\\") 2)
(test (string-length "hi\\") 3)
(test (string-length "\\\\\\\"") 4)
(test (string-length "A ; comment") 11)
(test (string-length "#| comment |#") 13)
(test (string-length "'123") 4)
(test (string-length '"'123") 4)
(test (let ((str (string #\# #\\ #\t))) (string-length str)) 3)

(test (string-length "#\\(") 3)
(test (string-length ")()") 3)
(test (string-length "(()") 3)
(test (string-length "(string #\\( #\\+ #\\space #\\1 #\\space #\\3 #\\))") 44)
(test (string-length) 'error)
(test (string-length "hi" "ho") 'error)
(test (string-length (string #\null)) 1) ; ??
(test (string-length (string #\null #\null)) 2) ; ??
(test (string-length (string #\null #\newline)) 2) ; ??
(test (string-length ``"hi") 2) ; ?? and in s7 ,"hi" is "hi" as with numbers

(test (string-length ";~S ~S") 6)
(test (string-length "\n;~S ~S") 7)
(test (string-length "\n\t") 2)
(test (string-length "#\newline") 8)
(test (string-length "#\tab") 4)
(test (string-length "a\x00b") 3)

(test (string-length "123\
456") 6)
(test (string-length"123\n
456") 8)
(test (string-length"123\n\
456") 7)

(for-each
 (lambda (arg)
   (test (string-length arg) 'error))
 (list #\a () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string

(for-each
 (lambda (arg)
   (test (string #\a arg) 'error)
   (test (string #\a #\null arg) 'error)
   (test (string arg) 'error))
 (list () (list 1) '(1 . 2) "a" #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string) "")
(test (string #\a #\b #\c) "abc")
(test (string #\a) "a")
(test (map string '(#\a #\b)) '("a" "b"))
(test (map string '(#\a #\b) '(#\c #\d)) '("ac" "bd"))
(test (map string '(#\a #\b #\c) '(#\d #\e #\f) '(#\g #\h #\i)) '("adg" "beh" "cfi"))
(test (map string "abc" "def" "ghi") '("adg" "beh" "cfi"))
(test (string #\" #\# #\") "\"#\"")
(test (string #\\ #\\ #\# #\\ #\# #\#) "\\\\#\\##")
(test (string #\' #\' #\` #\") '"''`\"")
;;; some schemes accept \' and other such sequences in a string, but the spec only mentions \\ and \"
(test (string ()) 'error)
(test (string "j" #\a) 'error)




;;; --------------------------------------------------------------------------------
;;; make-string

(test (make-string 0) "")
(test (make-string 3 #\a) "aaa")
(test (make-string 0 #\a) "")
(test (make-string 3 #\space) "   ")
(test (let ((hi (make-string 3 #\newline))) (string-length hi)) 3)

(test (make-string -1) 'error)
(test (make-string -0) "")
(test (make-string 2 #\a #\b) 'error)
(test (make-string) 'error)
(test (make-string most-positive-fixnum) 'error)
(test (make-string most-negative-fixnum) 'error)
(let () (define (hi size) (make-string size (integer->char (+ 1 (random 255))))) (string? (hi 3)))

(for-each
 (lambda (arg)
   (test (make-string 3 arg) 'error))
 (list "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg #\a) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (make-string arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; string-ref

(test (string-ref "abcdef-dg1ndh" 0) #\a)
(test (string-ref "abcdef-dg1ndh" 1) #\b)
(test (string-ref "abcdef-dg1ndh" 6) #\-)
(test (string-ref "\"\\\"" 1) #\\)
(test (string-ref "\"\\\"" 2) #\")

(test (let ((str (make-string 3 #\x))) (set! (string-ref str 1) #\a) str) "xax")

(test (string-ref "abcdef-dg1ndh" 20) 'error)
(test (string-ref "abcdef-dg1ndh") 'error)
(test (string-ref "abcdef-dg1ndh" -3) 'error)
(test (string-ref) 'error)
(test (string-ref 2) 'error)
(test (string-ref "\"\\\"" 3) 'error)
(test (string-ref "" 0) 'error)  
(test (string-ref "" 1) 'error)
(test (string-ref "hiho" (expt 2 32)) 'error)
(test (char=? (string-ref (string #\null) 0) #\null) #t)
(test (char=? (string-ref (string #\1 #\null #\2) 1) #\null) #t)
(test (char=? ("1\x002" 1) #\null) #t)
(test (char=? (string-ref (string #\newline) 0) #\newline) #t)
(test (char=? (string-ref (string #\space) 0) #\space) #t)

(for-each
 (lambda (arg)
   (test (string-ref arg 0) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-ref "hiho" arg) 'error))
 (list #\a -1 123 4 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test ("hi" 1) #\i)
(test (("hi" 1) 0) 'error)
(test ("hi" 1 2) 'error)
(test ("" 0) 'error)
(test (set! ("" 0) #\a) 'error)
(test (set! ("hi" 1 2) #\a) 'error)
(test (set! ("hi" 1) #\a #\b) 'error)
(test ("hi") 'error)
(test ("") 'error)
(test ((let () "hi")) 'error)
(test ((let () "hi") 0) #\h)

(test ("abs" most-negative-fixnum) 'error)
(test (string-ref "abs" most-negative-fixnum) 'error)
(test ("abs" (+ 1 most-negative-fixnum)) 'error)
(test ("abs" most-positive-fixnum) 'error)




;;; --------------------------------------------------------------------------------
;;; string-copy

(test (let ((hi (string-copy "hi"))) (string-set! hi 0 #\H) hi) "Hi")
(test (let ((hi (string-copy "hi"))) (string-set! hi 1 #\H) hi) "hH")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 0 #\a) hi) "a\\\"")
(test (let ((hi (string-copy "\"\\\""))) (string-set! hi 1 #\a) hi) "\"a\"")
(test (let ((hi (string #\a #\newline #\b))) (string-set! hi 1 #\c) hi) "acb")
(test (string-copy "ab") "ab")
(test (string-copy "") "")
(test (string-copy "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-copy hi))) #f)
(test (let ((hi (string-copy (make-string 8 (integer->char 0))))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (string-copy (string-copy (string-copy "a"))) "a")
(test (string-copy (string-copy (string-copy ""))) "")
(test (string-copy "a\x00b") "a\x00b") ; prints normally as "a" however
(test (string-copy (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-copy) 'error)
(test (string-copy "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (string-copy arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (length (string-copy (string #\null))) 1)


;;; --------------------------------------------------------------------------------
;;; string-set!

(let ((str (make-string 10 #\x)))
  (string-set! str 3 (integer->char 0))
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\x) #t)
  (string-set! str 4 #\a)
  (test (string=? str "xxx") #f)
  (test (char=? (string-ref str 4) #\a) #t)
  (string-set! str 3 #\x)
  (test (string=? str "xxxxaxxxxx") #t))

(test (string-set! "hiho" 1 #\c) #\c)
(test (set! ("hi" 1 2) #\i) 'error)
(test (set! ("hi" 1) "ho") 'error)
(test (set! ("hi") #\i) 'error)
(test (let ((x "hi") (y 'x)) (string-set! y 0 #\x) x) 'error)
(test (let ((str "ABS")) (set! (str 0) #\a)) #\a)
(test (let ((str "ABS")) (string-set! str 0 #\a)) #\a)
(test (let ((str "ABS")) (set! (string-ref str 0) #\a)) #\a)

(test (let ((hi (make-string 3 #\a)))
	(string-set! hi 1 (let ((ho (make-string 4 #\x)))
			    (string-set! ho 1 #\b)
			    (string-ref ho 0)))
	hi)
      "axa")

(test (string-set! "hiho" (expt 2 32) #\a) 'error)

(test (let ((hi (string-copy "hi"))) (string-set! hi 2 #\H) hi) 'error)
(test (let ((hi (string-copy "hi"))) (string-set! hi -1 #\H) hi) 'error)
(test (let ((g (lambda () "***"))) (string-set! (g) 0 #\?)) #\?)
(test (string-set! "" 0 #\a) 'error)
(test (string-set! "" 1 #\a) 'error)
(test (string-set! (string) 0 #\a) 'error)
(test (string-set! (symbol->string 'lambda) 0 #\a) #\a)
(test (let ((ho (make-string 0 #\x))) (string-set! ho 0 #\a) ho) 'error)
(test (let ((str "hi")) (string-set! (let () str) 1 #\a) str) "ha") ; (also in Guile)
(test (let ((x 2) (str "hi")) (string-set! (let () (set! x 3) str) 1 #\a) (list x str)) '(3 "ha"))
(test (let ((str "hi")) (set! ((let () str) 1) #\b) str) "hb")
(test (let ((str "hi")) (string-set! (let () (string-set! (let () str) 0 #\x) str) 1 #\x) str) "xx")
(test (let ((str "hi")) (string-set! (let () (set! str "hiho") str) 3 #\x) str) "hihx") ; ! (this works in Guile also)

(for-each
 (lambda (arg)
   (test (string-set! arg 0 #\a) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" arg #\a) 'error))
 (list #\a -1 123 4 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-set! "hiho" 0 arg) 'error))
 (list 1 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (equal? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (string=? (let ((str "hiho")) (string-set! str 2 #\null) str) "hi") #f)
(test (let* ((s1 "hi") (s2 s1)) (string-set! s2 1 #\x) s1) "hx")
(test (let* ((s1 "hi") (s2 (copy s1))) (string-set! s2 1 #\x) s1) "hi")

(test (eq? (car (catch #t (lambda () (set! ("hi") #\a)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! ("hi" 0 0) #\a)) (lambda args args))) 'wrong-number-of-args) #t) ; (vector-set! 1 ...)
(test (eq? (car (catch #t (lambda () (set! (("hi" 0) 0) #\a)) (lambda args args))) 'syntax-error) #t) ; (set! (1 ...))

(test (let ((s "012345")) (set! (apply s 2) #\a) s) 'error)
(test (string-set! #u8(0 1 0) 0 -9223372036854775808) 'error)



;;; --------------------------------------------------------------------------------
;;; string-fill!

(test (string-fill! "hiho" #\c) #\c)
(test (string-fill! "" #\a) #\a)
(test (string-fill! "hiho" #\a) #\a)
(test (let ((g (lambda () "***"))) (string-fill! (g) #\?)) #\?)
(test (string-fill!) 'error)
(test (string-fill! "hiho" #\a #\b) 'error)

(test (let ((hi (string-copy "hi"))) (string-fill! hi #\s) hi) "ss")
(test (let ((hi (string-copy ""))) (string-fill! hi #\x) hi) "")
(test (let ((str (make-string 0))) (string-fill! str #\a) str) "")
(test (let ((hi (make-string 8 (integer->char 0)))) (string-fill! hi #\a) hi) "aaaaaaaa") ; is this result widely accepted?
(test (recompose 12 string-copy "xax") "xax")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! a #\a) a) hi)) "aaa")
(test (let ((hi (make-string 3 #\x))) (recompose 12 (lambda (a) (string-fill! hi a)) #\a) hi) "aaa")
(test (let ((str (string #\null #\null))) (fill! str #\x) str) "xx")

(for-each
 (lambda (arg)
   (test (let ((hiho "hiho")) (string-fill! hiho arg) hiho) 'error))
 (list 1 "hi" () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string-fill! arg #\a) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((str "1234567890")) (string-fill! str #\a 0) str) "aaaaaaaaaa")
(test (let ((str "1234567890")) (string-fill! str #\a 0 10) str) "aaaaaaaaaa")
(test (let ((str "1234567890")) (string-fill! str #\a 0 0) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 4 4) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 10 10) str) "1234567890")
(test (let ((str "1234567890")) (string-fill! str #\a 0 4) str) "aaaa567890")
(test (let ((str "1234567890")) (string-fill! str #\a 3 4) str) "123a567890")
(test (let ((str "1234567890")) (string-fill! str #\a 1 9) str) "1aaaaaaaa0")
(test (let ((str "1234567890")) (string-fill! str #\a 8) str) "12345678aa")
(test (let ((str "1234567890")) (string-fill! str #\a 1 9 0) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 1 0) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 11) str) 'error)
(test (let ((str "1234567890")) (string-fill! str #\a 9 11) str) 'error)
(test (string-fill! "" 0 "hi") 'error)
(test (string-fill! "" 0 -1 3) 'error)
(test (string-fill! "" 0 1) 'error)
(test (string-fill! "" 0 0 4/3) 'error)



;;; --------------------------------------------------------------------------------
;;; string-upcase
;;; string-downcase

(test (string-downcase "") "")
(test (string-downcase "a") "a")
(test (string-downcase "A") "a")
(test (string-downcase "AbC") "abc")
(test (string-downcase "\"\\\"") "\"\\\"")
(test (let ((hi "abc")) (eq? hi (string-downcase hi))) #f)
(test (string-downcase (string-upcase (string-downcase "a"))) "a")
(test (string-downcase "a\x00b") "a\x00b") 
(test (string-downcase (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-downcase) 'error)
(test (string-downcase "hi" "ho") 'error)

(test (string-upcase "") "")
(test (string-upcase "a") "A")
(test (string-upcase "A") "A")
(test (string-upcase "AbC") "ABC")
(test (string-upcase "\"\\\"") "\"\\\"")
(test (let ((hi "ABC")) (eq? hi (string-upcase hi))) #f)
(test (string-upcase (string-downcase (string-upcase "a"))) "A")
(test (string-upcase "a\x00b") "A\x00B") 
(test (string-upcase (string #\1 #\null #\2)) (string #\1 #\null #\2))
(test (string-upcase) 'error)
(test (string-upcase "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (test (string-downcase arg) 'error)
   (test (string-upcase arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

;;; for r7rs, these need to be unicode-aware


;;; --------------------------------------------------------------------------------
;;; substring

(test (substring "ab" 0 0) "")
(test (substring "ab" 1 1) "")
(test (substring "ab" 2 2) "")
(test (substring "ab" 0 1) "a")
(test (substring "ab" 1 2) "b")
(test (substring "ab" 0 2) "ab")
(test (substring "hi there" 3 6) "the")
(test (substring "hi there" 0 (string-length "hi there")) "hi there")
(test (substring "" 0 0) "")
(let ((str "012345"))
  (let ((str1 (substring str 2 4)))
    (string-set! str1 1 #\x)
    (test (string=? str "012345") #t)
    (let ((str2 (substring str1 1)))
      (set! (str2 0) #\z)
      (test (string=? str "012345") #t)
      (test (string=? str1 "2x") #t)
      (test (string=? str2 "z") #t))))
(test (substring (substring "hiho" 0 2) 1) "i")
(test (substring (substring "hiho" 0 2) 2) "")
(test (substring (substring "hiho" 0 2) 0 1) "h")
(test (substring "hi\nho" 3 5) "ho")
(test (substring (substring "hi\nho" 1 4) 2) "h")
(test (substring (substring "hi\nho" 3 5) 1 2) "o")
(test (substring "hi\"ho" 3 5) "ho")
(test (substring (substring "hi\"ho" 1 4) 2) "h")
(test (substring (substring "hi\"ho" 3 5) 1 2) "o")
(test (let* ((s1 "0123456789") (s2 (substring s1 1 3))) (string-set! s2 1 #\x) s1) "0123456789")
(test (substring (substring "" 0 0) 0 0) "")
(test (substring (format #f "") 0 0) "")
(test (string=? (substring (substring (substring "01234567" 1) 1) 1) "34567") #t)
(let ()
  (define (hi) (string=? (substring (substring (substring "01234567" 1) 1) 1) "34567"))
  (define (ho) (hi)) (ho)
  (test (ho) #t))

(test (substring "012" 3) "")
(test (substring "012" 10) 'error)
(test (substring "012" most-positive-fixnum) 'error)
(test (substring "012" -1) 'error)
(test (substring "012" 3 3) "")
(test (substring "012" 3 4) 'error)
(test (substring "012" 3 2) 'error)
(test (substring "012" 3 -2) 'error)
(test (substring "012" 3 0) 'error)
(test (substring "012" 0) "012")
(test (substring "012" 2) "2")
(test (substring "" 0) "")

(test (recompose 12 (lambda (a) (substring a 0 3)) "12345") "123")
(test (reinvert 12 (lambda (a) (substring a 0 3)) (lambda (a) (string-append a "45")) "12345") "12345")

(test (substring "ab" 0 3) 'error)
(test (substring "ab" 3 3) 'error)
(test (substring "ab" 2 3) 'error)
(test (substring "" 0 1) 'error)
(test (substring "" -1 0) 'error)
(test (substring "abc" -1 0) 'error)
(test (substring "hiho" (expt 2 32) (+ 2 (expt 2 32))) 'error)
(test (substring) 'error)
(test (substring "hiho" 0 1 2) 'error)
(test (substring "1234" -1 -1) 'error)
(test (substring "1234" 1 0) 'error)
(test (substring "" most-positive-fixnum 1) 'error)

(let ((str "0123456789"))
  (string-set! str 5 #\null)
  (test (substring str 6) "6789")
  (test (substring str 5 5) "")
  (test (substring str 4 5) "4")
  (test (substring str 5 6) "\x00")
  (test (substring str 5 7) "\x006")
  (test (substring str 4 7) "4\x006"))

(for-each
 (lambda (arg)
   (test (substring "hiho" arg 0) 'error))
 (list "hi" #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring "0123" arg) 'error)
   (test (substring "hiho" 1 arg) 'error))
 (list "hi" #\a -1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (substring arg 1 2) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(define (substring? pattern target) ; taken from net somewhere (umich?) with changes for s7 (which now has string-position, so this is unneeded)
  (define (build-shift-vector pattern)
    (let* ((pat-len (length pattern))
	   (shift-vec (make-vector 256 (+ pat-len 1)))
	   (max-pat-index (- pat-len 1)))
      (let loop ((index 0))
	(set! (shift-vec (char->integer (pattern index))) (- pat-len index))
	(if (< index max-pat-index)
	    (loop (+ index 1))
	    shift-vec))))
  (if (or (not (string? pattern))
	  (not (string? target)))
      (error 'wrong-type-arg "substring? args should be strings: ~S ~S" pattern target)
      (let ((pat-len (length pattern)))
	(if (zero? pat-len)
	    0
	    (let ((shift-vec (build-shift-vector pattern)))
	      (let* ((tar-len (length target))
		     (max-tar-index (- tar-len 1))
		     (max-pat-index (- pat-len 1)))
		(let outer ((start-index 0))
		  (and (<= (+ pat-len start-index) tar-len)
		       (let inner ((p-ind 0) (t-ind start-index))
			 (cond
			  ((> p-ind max-pat-index) #f)           ; nothing left to check
			  ((char=? (pattern p-ind) (target t-ind))
			   (if (= p-ind max-pat-index)
			       start-index                       ; success -- return start index of match
			       (inner (+ p-ind 1) (+ t-ind 1)))) ; keep checking
			  ((> (+ pat-len start-index) max-tar-index) #f) ; fail
			  (else (outer (+ start-index (shift-vec (char->integer (target (+ start-index pat-len)))))))))))))))))

(test (substring? "hiho" "test hiho test") 5)
(test (substring? "hiho" "test hihptest") #f)
(test (substring? "hiho" "test hih") #f)
(test (substring? "hiho" "") #f)
(test (substring? "hiho" "hiho") 0)
(test (substring? "" "hiho") 0)
(test (substring? "abc" 'abc) 'error)



;;; --------------------------------------------------------------------------------
;;; string-append

(test (string-append "hi" "ho") "hiho")
(test (string-append "hi") "hi")
(test (string-append "hi" "") "hi")
(test (string-append "hi" "" "ho") "hiho")
(test (string-append "" "hi") "hi")
(test (string-append) "")
(test (string-append "a" (string-append (string-append "b" "c") "d") "e") "abcde")
(test (string-append "a" "b" "c" "d" "e") "abcde")
(test (string-append (string-append) (string-append (string-append))) "")
(test (let ((hi "hi")) (let ((ho (string-append hi))) (eq? hi ho))) #f)
(test (let ((hi "hi")) (let ((ho (string-append hi))) (string-set! ho 0 #\a) hi)) "hi")
(test (let ((hi "hi")) (set! hi (string-append hi hi hi hi)) hi) "hihihihi")
(test (string-append ()) 'error)
(test (string=? (string-append "012" (string #\null) "456") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #t)
(test (string=? (string-append "012" (string #\null) "356") 
		(let ((str "0123456")) (string-set! str 3 #\null) str))
      #f)
(test (string-append """hi""ho""") "hiho")
(test (let* ((s1 "hi") (s2 (string-append s1 s1))) (string-set! s2 1 #\x) s1) "hi")
(test (let* ((s1 "hi") (s2 (string-append s1))) (string-set! s2 1 #\x) s1) "hi")
(test (length (string-append (string #\x #\y (integer->char 127) #\z) (string #\a (integer->char 0) #\b #\c))) 8)

(test (length (string-append "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc" "abc")) 915)
(test (length (string-append (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c) (string #\a #\b #\c))) 915)


(num-test (letrec ((hi (lambda (str n)
			 (if (= n 0)
			     str
			     (hi (string-append str "a") (- n 1))))))
	    (string-length (hi "" 100)))
	  100)

(test (let* ((str "hiho")
	     (str1 "ha")
	     (str2 (string-append str1 str)))
	(string-set! str2 1 #\x)
	(string-set! str2 4 #\x)
	(and (string=? str "hiho")
	     (string=? str1 "ha")
	     (string=? str2 "hxhixo")))
      #t)
(test (let* ((str (string-copy "hiho"))
	     (str1 (string-copy "ha"))
	     (str2 (string-append str1 str)))
	(string-set! str1 1 #\x)
	(string-set! str 2 #\x)
	(and (string=? str "hixo")
	     (string=? str1 "hx")
	     (string=? str2 "hahiho")))
      #t)

(let ((s1 (string #\x #\null #\y))
      (s2 (string #\z #\null)))
  (test (string=? (string-append s1 s2) (string #\x #\null #\y #\z #\null)) #t)
  (test (string=? (string-append s2 s1) (string #\z #\null #\x #\null #\y)) #t))

(test (recompose 12 string-append "x") "x")
(test (recompose 12 (lambda (a) (string-append a "x")) "a") "axxxxxxxxxxxx")
(test (recompose 12 (lambda (a) (string-append "x" a)) "a") "xxxxxxxxxxxxa")

(test (length (string-append "\\?" "hi")) 4)
(test (string-append "hi" 1) 'error)
(test (eval-string "(string-append \"\\?\")") 'error) ; guile mailing list
(test (eval-string "(string-append \"\\?\" \"hi\")") 'error) ; guile mailing list
(for-each
 (lambda (arg)
   (test (string-append "hiho" arg) 'error)
   (test (string-append arg "hi") 'error)
   (test (string-append "a" "b" arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))


(test (let ((str (make-string 4 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c)
	   (string-set! str ctr c)
	   (set! ctr (+ ctr 1)))
	 "1234")
	str)
      "1234")

(test (let ((str (make-string 8 #\x))
	    (ctr 0))
	(for-each
	 (lambda (c1 c2)
	   (string-set! str ctr c1)
	   (string-set! str (+ ctr 1) c2)
	   (set! ctr (+ ctr 2)))
	 "1234"
	 "hiho")
	str)
      "1h2i3h4o")

#|
(let ((size 1024))
  (let ((str (make-string size)))
    (do ((i 0 (+ i 1)))
	((= i size))
      (set! (str i) (integer->char (+ 1 (modulo i 255)))))
    (let ((str1 (string-copy str)))
      (test (string? str1) #t)
      (test (string-length str1) 1024)
      (test (string-ref str1 556) (string-ref str 556))
      (test (string=? str str1) #t)
      (test (string<=? str str1) #t)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #t)
      (test (string-ci<=? str str1) #t)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #f)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #f)
      (test (substring str 123 321) (substring str1 123 321))

      (string-set! str1 1000 #\space)
      (test (string=? str str1) #f)
      (test (string<=? str str1) #f)
      (test (string>=? str str1) #t)
      (test (string-ci=? str str1) #f)
      (test (string-ci<=? str str1) #f)
      (test (string-ci>=? str str1) #t)
      (test (string<? str str1) #f)
      (test (string>? str str1) #t)
      (test (string-ci<? str str1) #f)
      (test (string-ci>? str str1) #t)

      (test (string-length (string-append str str1)) 2048)
      ))))
|#



;;; --------------------------------------------------------------------------------
;;; string->list
;;; list->string

(test (string->list "abc") (list #\a #\b #\c))
(test (string->list "") ())
(test (string->list (make-string 0)) ())
(test (string->list (string #\null)) '(#\null))
(test (string->list (string)) ())
(test (string->list (substring "hi" 0 0)) ())
(test (string->list (list->string (list #\a #\b #\c))) (list #\a #\b #\c))
(test (string->list (list->string ())) ())
(test (list->string (string->list "abc")) "abc")
(test (list->string (string->list "hi there")) "hi there")
(test (list->string (string->list "&*#%^@%$)~@")) "&*#%^@%$)~@")
(test (list->string (string->list "")) "")
(test (let* ((str "abc")
	     (lst (string->list str)))
	(and (string=? str "abc")
	     (equal? lst (list #\a #\b #\c))))
      #t)
(test (list->string ()) "")

(test (list->string (list #\a #\b #\c)) "abc")
(test (list->string (list)) "")

(test (list->string (list #\" #\# #\")) "\"#\"")
(test (list->string (list #\\ #\\ #\# #\\ #\# #\#)) "\\\\#\\##")
(test (list->string (list #\' #\' #\` #\")) '"''`\"")

(test (reinvert 12 string->list list->string "12345") "12345")

(test (string->list) 'error)
(test (list->string) 'error)
(test (string->list "hi" "ho") 'error)
(test (list->string () '(1 2)) 'error)
(test (string->list " hi ") '(#\space #\h #\i #\space))
(test (string->list (string (integer->char #xf0) (integer->char #x70))) (list (integer->char #xf0) (integer->char #x70)))

(for-each
 (lambda (arg)
   (test (string->list arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->string x)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->string lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply string lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->string arg) 'error))
 (list "hi" #\a 1 ''foo '(1 . 2) (cons #\a #\b) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((str (list->string '(#\x #\space #\null #\x))))
  (test (length str) 4)
  (test (str 1) #\space)
  (test (str 2) #\null)
  (test (str 3) #\x)
  (test (object->string str) "\"x \\x00x\"")
  (let ((lst (string->list str)))
    (test lst '(#\x #\space #\null #\x))))

(let ((strlen 8))
  (let ((str (make-string strlen)))
    (do ((i 0 (+ i 1)))
	((= i 10))
      (do ((k 0 (+ k 1)))
	  ((= k strlen))
	(set! (str k) (integer->char (random 256))))
      (let ((lst (string->list str)))
	(let ((newstr (list->string lst)))
	  (let ((lstlen (length lst))
		(newstrlen (length newstr)))
	    (if (or (not (= lstlen strlen newstrlen))
		    (not (string=? newstr str)))
		(format-logged #t ";string->list->string: ~S -> ~A -> ~S~%" str lst newstr))))))))

(when full-test
  (let ()
    (define (all-strs len file)
      (let* ((funny-chars (list #\` #\# #\, #\@ #\' #\" #\. #\( #\) #\\))
	     (num-chars (length funny-chars)))
	(let ((ctrs (make-vector len 0)))
	  
	  (do ((i 0 (+ i 1)))
	      ((= i (expt num-chars len)))
	    (let ((carry #t))
	      (do ((k 0 (+ k 1)))
		  ((or (= k len)
		       (not carry)))
		(vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
		(if (= (vector-ref ctrs k) num-chars)
		    (vector-set! ctrs k 0)
		    (set! carry #f)))
	      
	      (let ((strlst ()))
		(do ((k 0 (+ k 1)))
		    ((= k len))
		  (let ((c (list-ref funny-chars (vector-ref ctrs k))))
		    (set! strlst (cons c strlst))))
		
		(let ((str (list->string strlst)))
		  (format file "(test (and (string=? ~S (string ~{#\\~C~^ ~})) (equal? '~A (string->list ~S))) #t)~%" str strlst strlst str))))))))
    
    (call-with-output-file "strtst.scm"
      (lambda (p)
	(do ((len 3 (+ len 1)))
	    ((= len 5))
	  (all-strs len p))))
    
    (load "strtst.scm")))


(test (and (string=? "\"" (string #\")) (equal? '(#\") (string->list "\""))) #t)
(test (and (string=? "#\\" (string #\# #\\)) (equal? '(#\# #\\) (string->list "#\\"))) #t)
(test (and (string=? "#(" (string #\# #\()) (equal? '(#\# #\() (string->list "#("))) #t)
(test (and (string=? "\"@" (string #\" #\@)) (equal? '(#\" #\@) (string->list "\"@"))) #t)
(test (and (string=? "\";" (string #\" #\;)) (equal? '(#\" #\;) (string->list "\";"))) #t)
(test (and (string=? ")(" (string #\) #\()) (equal? '(#\) #\() (string->list ")("))) #t)
(test (and (string=? "`)#" (string #\` #\) #\#)) (equal? '(#\` #\) #\#) (string->list "`)#"))) #t)
(test (and (string=? "##\\" (string #\# #\# #\\)) (equal? '(#\# #\# #\\) (string->list "##\\"))) #t)
(test (and (string=? "#\"(" (string #\# #\" #\()) (equal? '(#\# #\" #\() (string->list "#\"("))) #t)
(test (and (string=? "#.@" (string #\# #\. #\@)) (equal? '(#\# #\. #\@) (string->list "#.@"))) #t)
(test (and (string=? ",`@" (string #\, #\` #\@)) (equal? '(#\, #\` #\@) (string->list ",`@"))) #t)
(test (and (string=? "',@" (string #\' #\, #\@)) (equal? '(#\' #\, #\@) (string->list "',@"))) #t)
(test (and (string=? "\"#@" (string #\" #\# #\@)) (equal? '(#\" #\# #\@) (string->list "\"#@"))) #t)
(test (and (string=? "\")\"" (string #\" #\) #\")) (equal? '(#\" #\) #\") (string->list "\")\""))) #t)
(test (and (string=? ")#(" (string #\) #\# #\()) (equal? '(#\) #\# #\() (string->list ")#("))) #t)
(test (and (string=? "`(,@" (string #\` #\( #\, #\@)) (equal? '(#\` #\( #\, #\@) (string->list "`(,@"))) #t)
(test (and (string=? "`)#\"" (string #\` #\) #\# #\")) (equal? '(#\` #\) #\# #\") (string->list "`)#\""))) #t)
(test (and (string=? "#\"'#" (string #\# #\" #\' #\#)) (equal? '(#\# #\" #\' #\#) (string->list "#\"'#"))) #t)
(test (and (string=? "#(@\\" (string #\# #\( #\@ #\\)) (equal? '(#\# #\( #\@ #\\) (string->list "#(@\\"))) #t)
(test (and (string=? "#(\\\\" (string #\# #\( #\\ #\\)) (equal? '(#\# #\( #\\ #\\) (string->list "#(\\\\"))) #t)
(test (and (string=? ",,.@" (string #\, #\, #\. #\@)) (equal? '(#\, #\, #\. #\@) (string->list ",,.@"))) #t)
(test (and (string=? ",@`\"" (string #\, #\@ #\` #\")) (equal? '(#\, #\@ #\` #\") (string->list ",@`\""))) #t)
(test (and (string=? "\"'\")" (string #\" #\' #\" #\))) (equal? '(#\" #\' #\" #\)) (string->list "\"'\")"))) #t)
(test (and (string=? "\")#\"" (string #\" #\) #\# #\")) (equal? '(#\" #\) #\# #\") (string->list "\")#\""))) #t)
(test (and (string=? "(\\`)" (string #\( #\\ #\` #\))) (equal? '(#\( #\\ #\` #\)) (string->list "(\\`)"))) #t)
(test (and (string=? "))\"'" (string #\) #\) #\" #\')) (equal? '(#\) #\) #\" #\') (string->list "))\"'"))) #t)
(test (and (string=? "\\,\\\"" (string #\\ #\, #\\ #\")) (equal? '(#\\ #\, #\\ #\") (string->list "\\,\\\""))) #t)
(test (and (string=? "\\\"`\"" (string #\\ #\" #\` #\")) (equal? '(#\\ #\" #\` #\") (string->list "\\\"`\""))) #t)
(test (and (string=? "\\\\#\"" (string #\\ #\\ #\# #\")) (equal? '(#\\ #\\ #\# #\") (string->list "\\\\#\""))) #t)

(test (string->list "" 0 10) 'error)
(test (string->list "1" 0 2) 'error)
(test (string->list "" 0 0) ())
(test (string->list "1" 1) ())
(test (string->list "1" 0) '(#\1))
(test (string->list "" #\null) 'error)
(test (string->list "" 0 #\null) 'error)
(test (string->list "" -1) 'error)
(test (string->list "1" -1) 'error)
(test (string->list "1" 0 -1) 'error)
(test (string->list "1" -2 -1) 'error)
(test (string->list "1" most-negative-fixnum) 'error)
(test (string->list "1" 2) 'error)

(for-each
 (lambda (arg)
   (test (string->list "012345" arg) 'error)
   (test (string->list "012345" 1 arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (string->list "12345" 0) '(#\1 #\2 #\3 #\4 #\5))
(test (string->list "12345" 0 5) '(#\1 #\2 #\3 #\4 #\5))
(test (string->list "12345" 5 5) ())
(test (string->list "12345" 4 5) '(#\5))
(test (string->list "12345" 2 4) '(#\3 #\4))
(test (string->list "12345" 2 1) 'error)
(test (string->list "12345" 2 3 4) 'error)
(test (string->list (make-string 3 #\null) 2 3) '(#\null))



;;; --------------------------------------------------------------------------------
;;; char-position
;;; string-position

(test (char-position) 'error)
(test (char-position #\a) 'error)
(test (char-position #\a "abc" #\0) 'error)
(test (char-position #\a "abc" 0 1) 'error)
(test (string-position) 'error)
(test (string-position #\a) 'error)
(test (string-position "a" "abc" #\0) 'error)
(test (string-position "a" "abc" 0 1) 'error)

(for-each
 (lambda (arg) 
   (test (string-position arg "abc") 'error)
   (test (char-position arg "abc") 'error)
   (test (string-position "a" arg) 'error)
   (test (char-position #\a arg) 'error)
   (test (string-position "a" "abc" arg) 'error)
   (test (char-position #\a "abc" arg) 'error))
 (list () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 -1 most-negative-fixnum 1.0+1.0i :hi (if #f #f) (lambda (a) (+ a 1))))
(test (char-position #\a "abc" most-positive-fixnum) #f)
(test (char-position "a" "abc" most-positive-fixnum) #f)
(test (string-position "a" "abc" most-positive-fixnum) #f)

(test (char-position #\b "abc") 1)
(test (char-position #\b "abc" 0) 1)
(test (char-position #\b "abc" 1) 1)
(test (char-position "b" "abc") 1)
(test (char-position "b" "abc" 1) 1)
(test (char-position "b" "abc") 1)
(test (string-position "b" "abc") 1)
(test (string-position "b" "abc" 1) 1)
(test (string-position "b" "abc" 2) #f)
(test (string-position "b" "abc" 3) #f)
(test (char-position "b" "abc" 2) #f)
(test (char-position "b" "abc" 3) #f)
(test (char-position #\b "abc" 2) #f)
(test (char-position #\b "abc" 3) #f)
(test (char-position "ab" "abcd") 0)
(test (char-position "ab" "ffbcd") 2)
(test (char-position "ab" "ffacd") 2)
(test (string-position "ab" "ffacd") #f)
(test (string-position "ab" "ffabd") 2)
(test (string-position "ab" "ffabab" 2) 2)
(test (string-position "ab" "ffabab" 3) 4)
(test (string-position "ab" "ffabab" 4) 4)
(test (string-position "ab" "ffabab" 5) #f)
(test (string-position "abc" "ab") #f)
(test (string-position "abc" "") #f)
(test (string-position "" "") #f)
(test (char-position "\"" "a") #f)
(test (char-position "\"" "a\"b") 1)
(test (char-position #\" "a\"b") 1)
(test (string-position "\"hiho\"" "hiho") #f)
(test (string-position "\"hiho\"" "\"\"hiho\"") 1)

(test (string-position "" "a") #f) ; this is a deliberate choice in s7.c
(test (char-position "" "a") #f) 
(test (char-position #\null "a") 1)  ; ??
(test (char-position #\null "") #f)  ; ??
(test (string-position (string #\null) "a") 0) ; ??
(test (string-position (string #\null) "") #f) ; ??
(test (char-position #\null (string #\null)) 0) ; ??
(test (char-position #\null (string #\a #\null #\n)) 1)
(test (char-position "" (string #\a #\null #\n)) #f)
;(test (char-position #\n (string #\a #\null #\n)) 2)   ; ?? returns #f due to assumption of C-style strings
;(test (char-position "n" (string #\a #\null #\n)) 1)   ; oops!
;(test (string-position "n" (string #\a #\null #\n)) 2) ; oops!
(test (char-position "" (string #\a #\n)) #f)
(test (char-position #(1) "asdasd" 63) 'error)

;; if "" as string-pos first, -> #f so same for char-pos, even if string contains a null

(let ()
  ;; actually more of a string-append/temp substring test
  (define (fixit str)
    (let ((pos (char-position #\& str)))
      (if (not pos)
	  str
	  (string-append (substring str 0 pos)
			 (let ((epos (char-position #\; str pos)))
			   (let ((substr (substring str (+ pos 1) epos)))
			     (let ((replacement (cond ((string=? substr "gt") ">")
						      ((string=? substr "lt") "<")
						      ((string=? substr "mdash") "-")
						      (else (format-logged #t "unknown: ~A~%" substr)))))
			       (string-append replacement
					      (fixit (substring str (+ epos 1)))))))))))
  (test (fixit "(let ((f (hz-&gt;radians 100)) (g (hz-&gt;radians 200))) (&lt; f g))")
	"(let ((f (hz->radians 100)) (g (hz->radians 200))) (< f g))"))

;;; opt bug
(test (apply char-position '(#\a #u8() #f)) 'error)
(test (let () (define (f1) (do ((i 0 (+ i 1))) ((= i 1) (char-position #\a #u8() #f)) (char-position #\a #u8() #f))) (f1)) 'error)




;;; --------------------------------------------------------------------------------
;;; symbol->string
;;; string->symbol
;;; symbol

(test (symbol->string 'hi) "hi")
(test (string->symbol (symbol->string 'hi)) 'hi)
(test (eq? (string->symbol "hi") 'hi) #t)
(test (eq? (string->symbol "hi") (string->symbol "hi")) #t)

(test (string->symbol "hi") 'hi)

(test (let ((str (symbol->string 'hi)))
	(catch #t (lambda () (string-set! str 1 #\x)) (lambda args 'error)) ; can be disallowed
	(symbol->string 'hi))
      "hi")

(test (symbol->string 'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
(test (string->symbol "sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789")
      'sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 32))
	(+ sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1))
      33)

(test (symbol->string (string->symbol "hi there")) "hi there")
(test (symbol->string (string->symbol "Hi There")) "Hi There")
(test (symbol->string (string->symbol "HI THERE")) "HI THERE")
(test (symbol->string (string->symbol "")) 'error) ; this fluctuates
(test (symbol? (string->symbol "(weird name for a symbol!)")) #t)
(test (symbol->string (string->symbol "()")) "()")
(test (symbol->string (string->symbol (string #\"))) "\"")
(test (symbol->string 'quote) "quote")
(test (symbol->string if) 'error)
(test (symbol->string quote) 'error)

(test (symbol? (string->symbol "0")) #t)
(test (symbol? (symbol "0")) #t)
(test (symbol? (symbol ".")) #t) ; hmmm
(test (let () (define |.| 1) (+ |.| 2)) 3)
(test (string->symbol "0e") '0e)
(test (string->symbol "1+") '1+)
(test (symbol? (string->symbol "1+i")) #t)
(test (string->symbol ":0") ':0)
(test (symbol? (string->symbol " hi") ) #t)
(test (symbol? (string->symbol "hi ")) #t)

(test (reinvert 12 string->symbol symbol->string "hiho") "hiho")

(test (symbol->string) 'error)
(test (string->symbol) 'error)
(test (symbol->string 'hi 'ho) 'error)
(test (string->symbol "hi" "ho") 'error)

(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 8) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 128) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 200) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 255) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 20) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 2) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 7) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 17) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 170) #\x))) #t)
(test (symbol? (string->symbol (string #\x (integer->char 0) #\x))) #t)       ; but the symbol's name here is "x"
;(test (eq? (string->symbol (string #\x (integer->char 0) #\x)) 'x) #t)        ;   hmmm...
(test (symbol? (string->symbol (string #\x #\y (integer->char 127) #\z))) #t) ; xy(backspace)z

(test (symbol? (string->symbol (string #\; #\" #\)))) #t)
(test (let (((symbol ";")) 3) (symbol ";")) 'error)
(test (symbol "") 'error)

(for-each
 (lambda (arg)
   (test (symbol->string arg) 'error))
 (list #\a 1 "hi" () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (string->symbol arg) 'error)
   (test (symbol arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (symbol? (string->symbol (string arg))) #t)
   (test (symbol? (symbol (string arg))) #t))
 (list #\; #\, #\. #\) #\( #\" #\' #\` #\x33 #\xff #\x7f #\# #\]))

(test (symbol) 'error)
(test (symbol "hi" "ho") 'hiho)

(let ()
  (define-macro (string-case selector . clauses)
    `(case (symbol ,selector)
       ,@(map (lambda (clause)
		(if (pair? (car clause))
		    `(,(map symbol (car clause)) ,@(cdr clause))
		    clause))
	      clauses)))

  (test (let ((str "hi"))
	  (string-case str
            (("hi" "ho") 1 2 3)
	    (("hiho") 2)
	    (else 4)))
	3))

(let ()
  (apply define (list (symbol "(#)") 3))
  (test (eval (symbol "(#)")) 3))

(let ()
  (define (immutable obj) (string->symbol (object->string obj :readable)))
  (define (symbol->object sym) (eval-string (symbol->string sym)))
  (test (symbol->object (immutable (list 1 2 3))) (list 1 2 3))
  (test (symbol->object (immutable "hi")) "hi"))

#|
(let ((str "(let ((X 3)) X)"))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (catch #t
	   (lambda ()
	     (if (symbol? (string->symbol (string (integer->char i))))
		 (catch #t
			(lambda ()
			  (set! (str 7) (integer->char i))
			  (set! (str 13) (integer->char i))
			  (let ((val (eval-string str)))
			    #t)) ;(format-logged #t "ok: ~S -> ~S~%" str val)))
			(lambda args
			  (format-logged #t "bad but symbol: ~S~%" str))))) ; 11 12 # ' , . 
	   (lambda args
	     (format-logged #t "bad: ~C~%" (integer->char i))))))  ; # ( ) ' " . ` nul 9 10 13 space 0..9 ;

(let ((str "(let ((XY 3)) XY)"))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (do ((k 0 (+ k 1)))
	((= k 256))
      (catch #t
	     (lambda ()
	       (if (symbol? (string->symbol (string (integer->char i))))
		   (catch #t
			  (lambda ()
			    (set! (str 7) (integer->char i))
			    (set! (str 8) (integer->char k))
			    (set! (str 14) (integer->char i))
			    (set! (str 15) (integer->char k))
			    (let ((val (eval-string str)))
			      #t)) ;(format-logged #t "ok: ~S -> ~S~%" str val)))
			  (lambda args
			    (format-logged #t "bad but symbol: ~S~%" str))))) ; 11 12 # ' , . 
	     (lambda args
	       (format-logged #t "bad: ~C~%" (integer->char i)))))))  ; # ( ) ' " . ` nul 9 10 13 space 0..9 ;
|#




;;; --------------------------------------------------------------------------------
;;; symbol->value
;;; symbol->dynamic-value

(let ((sym 0))
  (test (symbol->value 'sym) 0)
  (test (symbol->dynamic-value 'sym) 0)
  (for-each
   (lambda (arg)
     (set! sym arg)
     (test (symbol->value 'sym) arg)
     (test (symbol->dynamic-value 'sym) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
	 3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (symbol->value arg) 'error)
   (test (symbol->value 'abs arg) 'error)
   (test (symbol->dynamic-value arg) 'error)
   (test (symbol->dynamic-value 'abs arg) 'error))
 (list #\a 1 () (list 1) "hi" '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))
  
(test (symbol->value) 'error)
(test (symbol->value 'hi 'ho) 'error)
(test (symbol->dynamic-value) 'error)
(test (symbol->dynamic-value 'hi 'ho) 'error)

(test (symbol->value 'abs (unlet)) abs)
(test (symbol->value 'abs (rootlet)) abs)
(test (symbol->value 'lambda) lambda)
(test (symbol->value 'do) do)
(test (symbol->value do) 'error)
(test (symbol->value 'macroexpand) macroexpand)
(test (symbol->value 'quasiquote) quasiquote)
(test (symbol->value 'else) else)
(test (symbol->value :hi) :hi)
(test (symbol->value hi:) hi:)

(test (symbol->dynamic-value 'lambda) lambda)
(test (symbol->dynamic-value 'do) do)
(test (symbol->dynamic-value do) 'error)
(test (symbol->dynamic-value 'macroexpand) macroexpand)
(test (symbol->dynamic-value 'quasiquote) quasiquote)
(test (symbol->dynamic-value 'else) else)
(test (symbol->dynamic-value :hi) :hi)
(test (symbol->dynamic-value hi:) hi:)

(test (symbol->value '#<eof>) 'error) ; because it's not a symbol:
(test (symbol? '#<eof>) #f)
(test (let ((a1 32)) (let () (symbol->value 'a1 (curlet)))) 32)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'a1 (curlet)))) 0)
(test (let ((a1 32)) (let ((a1 0)) (symbol->value 'b1 (curlet)))) #<undefined>)
(test (symbol->value 'abs ()) 'error)
(test (let ((a1 (let ((b1 32)) (lambda () b1)))) (symbol->value 'b1 (funclet a1))) 32)
(test (let ((x #f)) (set! x (let ((a1 (let ((b1 32)) (lambda () b1)))) a1)) (symbol->value 'b1 (funclet x))) 32)
(test (symbol->value 'if) if)
(test (symbol->value if) 'error)
(test ((define (hi a) (+ a 1)) 2) 3)
(test ((define-macro (hi a) `(+ ,a 1)) 2) 3)
(test (let ((mac (define-macro (hi a) `(+ ,a 1)))) (mac 3)) 4)

(test (eq? #_abs (symbol->value 'abs 'unlet)) #t)
(test (eq? #_lambda (symbol->value 'lambda 'unlet)) #t)

(let ()
  (define *ds* 0)
  (define (get-ds) (list *ds* (symbol->dynamic-value '*ds*)))
  (test (get-ds) '(0 0))
  (let ((*ds* 32))
    (test (values (get-ds)) '(0 32)))
  (let ((*ds* 3))
    (define (gds) (list *ds* (symbol->dynamic-value '*ds*)))
    (test (list (get-ds) (gds)) '((0 3) (3 3)))
    (let ((*ds* 123)) 
      (test (list (get-ds) (gds)) '((0 123) (3 123)))))
  (let ((*ds* 3))
    (define (gds) (list *ds* (symbol->dynamic-value '*ds*)))
    (let ((*ds* 123)) 
      (set! *ds* 321)
      (test (list (get-ds) (gds)) '((0 321) (3 321))))))

(test (symbol->dynamic-value 'asdasfasdasfg) #<undefined>)

(let ((x 32))
  (define (gx) (symbol->dynamic-value 'x))
  (let ((x 12))
    (test (values (gx)) 12)))

(let ((x "hi")
      (y 0)
      (z '(1 2 3)))
  (define (gx) (+ (symbol->dynamic-value 'x) (symbol->dynamic-value 'z)))
  (let ((x 32) 
	(z (+ 123 (car z))))
    (test (values (gx)) 156)))

(let ((x 32))
  (define (gx) (symbol->dynamic-value 'x))
  (let ((x 100))
    (let ((x 12))
      (test (values (gx)) 12))))

(let ((x 32))
  (define (gx) ; return both bindings of 'x
    (list x (symbol->value 'x) (symbol->dynamic-value 'x)))
  (let ((x 100))
    (let ((x 12))
      (test (values (gx)) '(32 32 12)))))

(let ((bindings ()))
  ;; taken from the MIT_Scheme documentation (changing fluid-let to let)

  (define (write-line v) 
    (set! bindings (cons v bindings)))

  (define (complicated-dynamic-binding)
    (let ((variable 1)
	  (inside-continuation #f))
      (write-line variable)
      (call-with-current-continuation
       (lambda (outside-continuation)
	 (let ((variable 2))
	   (write-line variable)
	   (set! variable 3)
	   (call-with-current-continuation
	    (lambda (k)
	      (set! inside-continuation k)
	      (outside-continuation #t)))
	   (write-line variable)
	   (set! inside-continuation #f))))
      (write-line variable)
      (if inside-continuation
	  (begin
	    (set! variable 4)
	    (inside-continuation #f)))))

  (complicated-dynamic-binding)
  (test (reverse bindings) '(1 2 1 3 4)))




;;; --------------------------------------------------------------------------------
;;; BYTE-VECTORS
;;; --------------------------------------------------------------------------------

(let ((bv #u8(1 0 3)))
  (test bv #u8(1 0 3))
  (test (object->string bv) "#u8(1 0 3)")
  (test (equal? bv #u8(1 0 3)) #t)
  (test (eq? bv bv) #t)
  (test (eqv? bv bv) #t)
  (test (equal? (byte-vector 1 0 3) #u8(1 0 3)) #t)
  (test (byte-vector? bv) #t)
  (test (equal? (make-byte-vector 3) #u8(0 0 0)) #t)
  (test (string-ref #u8(64 65 66) 1) 65)
  (test (let ((nbv (copy bv))) (equal? nbv bv)) #t)
  (test (let ((rbv (reverse bv))) (equal? rbv #u8(3 0 1))) #t)
  (test (length bv) 3)
  )

(test (eval-string "#u8(-1)") 'error)
(test (eval-string "#u8(1.0)") 'error)
(test (eval-string "#u8(3/2)") 'error)
(test (eval-string "#u8(1+i)") 'error)
(test (eval-string "#u8((32))") 'error)
(test (eval-string "#u8(#\\a)") 'error)
(test (eval-string "#u8(256)") 'error)
(test (eval-string "#u8(1/0)") 'error)
(test (eval-string "#u8(9223372036854775807)") 'error)
(test (eval-string "#u8(-9223372036854775808)") 'error)
(test #u8(#b11 #x8) #u8(3 8))
(test (eval-string "#u8(1 2 . 3)") 'error)

(test #u8(255) (byte-vector 255))
(test (object->string #u8()) "#u8()")
(test (object->string #u8(255)) "#u8(255)")
(test (object->string #u8(255 255)) "#u8(255 255)")
(test (object->string #u8(128)) "#u8(128)")
(test (object->string #u8(128 128)) "#u8(128 128)")

(test (length #u8(0)) 1)
(test (length #u8(0 0)) 2)
(test (length #u8()) 0)
(test (length (byte-vector)) 0)
(test (byte-vector? #u8()) #t)
(test (equal? (let ((bv #u8(1 0 3))) (set! (bv 2) 64) bv) #u8(1 0 64)) #t)
(test (let ((bv #u8(1 0 3))) (map values bv)) '(1 0 3))
(test (let ((bv #u8(1 0 3)) (lst ())) (for-each (lambda (x) (set! lst (cons x lst))) bv) lst) '(3 0 1))
(test (let ((bv #u8(1 2 3))) (bv 1)) 2)
(test (let ((bv #u8(1 2 3))) (reverse bv)) #u8(3 2 1))
(test (let ((bv #u8(1 2 3))) (object->string (reverse bv))) "#u8(3 2 1)")
(test (let ((bv #u8(1 2 3))) (copy bv)) #u8(1 2 3))
(test (#u8(1 2 3) 2) 3)
(test (let ((v #u8(0 1 2))) (let ((v1 (reverse! v))) (eq? v v1))) #t)
(test (let ((v #u8(0 1 2))) (reverse! v)) #u8(2 1 0))
;; should (vector? #u8(1 2)) be #t?
(test (format #f "~{~A ~}" (byte-vector 255 0)) "255 0 ")

;;; string->byte-vector
(test (byte-vector? (string->byte-vector (string #\0))) #t)
(test (byte-vector? (string->byte-vector "")) #t)
(test (byte-vector? (string->byte-vector "1230")) #t)
(test (byte-vector? (string->byte-vector (string->byte-vector (string #\0)))) #t)
(test (byte-vector? (string->byte-vector (string))) #t)
(test (byte-vector? (string->byte-vector #u8(1 2))) #t)
(test (byte-vector? (string->byte-vector #u8())) #t)
(test (byte-vector? (string->byte-vector #(1 2))) 'error)
(test (byte-vector? (string-append #u8(1 2) #u8(3 4))) #t)
(for-each
 (lambda (arg)
   (test (string->byte-vector arg) 'error)
   (test (byte-vector? arg) #f))
 (list #\a () (list 1)  '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))
;;; an experiment:
(test (string->byte-vector #x010203) #u8(3 2 1 0 0 0 0 0))

;;; make-byte-vector
(test (equal? (make-byte-vector 0) #u8()) #t)
(test (equal? (make-byte-vector 0 32) #u8()) #t)
(test (equal? (make-byte-vector 1 32) #u8(32)) #t)
(test (make-byte-vector 0 -32) 'error)
(test (make-byte-vector 1 -32) 'error)
(test (make-byte-vector 1 256) 'error)
(for-each
 (lambda (arg)
   (test (make-byte-vector arg) 'error)
   (test (make-byte-vector 1 arg) 'error))
 (list #\a () (list 1)  '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))


;;; byte-vector
(test (byte-vector) #u8())
(test (byte-vector 32) (make-byte-vector 1 32))
(test (byte-vector 0 256) 'error)
(test (byte-vector -1) 'error)
(for-each
 (lambda (arg)
   (test (byte-vector arg) 'error))
 (list #\a () (list 1)  '(1 . 2) #f (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (if #f #f) #<eof> (lambda (a) (+ a 1))))

(test (map append #u8(0 1 2)) '(0 1 2))
(test (format #f "~{#x~X~| ~}" #u8(49 50 51)) "#x31 #x32 #x33")
(test (format #f "~{~D~| ~}" (string->byte-vector "abcd")) "97 98 99 100")
(test (let ((lst ())) (for-each (lambda (c) (set! lst (cons c lst))) #u8(90 91 92)) (reverse lst)) '(90 91 92))
(test (integer? (#u8(1 2 3) 0)) #t)
(test (integer? ((string->byte-vector "abc") 1)) #t)

(test ((vector (byte-vector 1)) 0 0) 1) ; i.e. not a character

(let ((bv (byte-vector 0 1 2 3)))
  (fill! bv 4)
  (test bv #u8(4 4 4 4))
  (fill! bv 1 1 3)
  (test bv #u8(4 1 1 4))
  (let ((bv1 (copy bv)))
    (test bv1 #u8(4 1 1 4))
    (fill! bv 1)
    (copy bv bv1)
    (test bv1 #u8(1 1 1 1))
    (fill! bv 255)
    (copy bv bv1 1 3)
    (test bv1 #u8(255 255 1 1)))) ; copy and fill do not interpret their indices in the same way (one is source, the other destination)




;;; --------------------------------------------------------------------------------
;;; LISTS
;;; --------------------------------------------------------------------------------


;;; --------------------------------------------------------------------------------
;;; cons

(test (cons 'a ()) '(a))
(test (cons '(a) '(b c d)) '((a) b c d))
(test (cons "a" '(b c)) '("a" b c))
(test (cons 'a 3) '(a . 3))
(test (cons '(a b) 'c) '((a b) . c))
(test (cons () ()) '(()))
(test (cons () 1) '(() . 1))
(test (cons 1 2) '(1 . 2))
(test (cons 1 ()) '(1))
(test (cons () 2) '(() . 2))
(test (cons 1 (cons 2 (cons 3 (cons 4 ())))) '(1 2 3 4))
(test (cons 'a 'b) '(a . b))
(test (cons 'a (cons 'b (cons 'c ()))) '(a b c))
(test (cons 'a (list 'b 'c 'd)) '(a b c d))
(test (cons 'a (cons 'b (cons 'c 'd))) '(a b c . d))
(test '(a b c d e) '(a . (b . (c . (d . (e . ()))))))
(test (cons (cons 1 2) (cons 3 4)) '((1 . 2) 3 . 4))
(test (list (cons 1 2) (cons 3 4)) '((1 . 2) (3 . 4)))
(test (cons (cons 1 (cons 2 3)) 4) '((1 . (2 . 3)) . 4))
(test (cons (cons 1 (cons 2 ())) (cons 1 2)) '((1 2) . (1 . 2)))
(test (let ((lst (list 1 2))) (list (apply cons lst) lst)) '((1 . 2) (1 2)))
(test (let ((lst (list 1 2))) (list lst (apply cons lst))) '((1 2) (1 . 2)))
(test (cdadr (let ((lst (list 1 2))) (list (apply cons lst) lst))) '(2))
(test (cons '+ '=) '(+ . =))
(test (cons .(cadddr 10)) (cons cadddr 10))
(test (cons 1 ()) '(
                      1
		       ))



;;; --------------------------------------------------------------------------------
;;; car

(test (car (list 1 2 3)) 1)
(test (car (cons 1 2)) 1)
(test (car (list 1)) 1)
(test (car '(1 2 3)) 1)
(test (car '(1)) 1)
(test (car '(1 . 2)) 1)
(test (car '((1 2) 3)) '(1 2))
(test (car '(((1 . 2) . 3) 4)) '((1 . 2) . 3))
(test (car (list (list) (list 1 2))) ())
(test (car '(a b c)) 'a)
(test (car '((a) b c d)) '(a))
(test (car (reverse (list 1 2 3 4))) 4)
(test (car (list 'a 'b 'c 'd 'e 'f 'g)) 'a)
(test (car '(a b c d e f g)) 'a)
(test (car '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((1 2 3) 4) 5) (6 7)))
(test (car '(a)) 'a)
(test (car '(1 ^ 2)) 1)
(test (car '(1 .. 2)) 1)
(test (car ''foo) 'quote)
(test (car '(1 2 . 3)) 1)
(test (car (cons 1 ())) 1)
(test (car (if #f #f)) 'error)
(test (car ()) 'error)
(test (car #(1 2)) 'error)
(test (car #(1 2)) 'error)

(for-each
 (lambda (arg)
   (if (not (equal? (car (cons arg ())) arg))
       (format-logged #t ";(car '(~A)) returned ~A?~%" arg (car (cons arg ()))))
   (test (car arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (reinvert 12 car (lambda (a) (cons a ())) '(1)) '(1))



;;; --------------------------------------------------------------------------------
;;; cdr

(test (cdr (list 1 2 3)) '(2 3))
(test (cdr (cons 1 2)) 2)
(test (cdr (list 1)) ())
(test (cdr '(1 2 3)) '(2 3))
(test (cdr '(1)) ())
(test (cdr '(1 . 2)) 2)
(test (cdr '((1 2) 3)) '(3))
(test (cdr '(((1 . 2) . 3) 4)) '(4))
(test (cdr (list (list) (list 1 2))) '((1 2)))
(test (cdr '(a b c)) '(b c))
(test (cdr '((a) b c d)) '(b c d))
(test (equal? (cdr (reverse (list 1 2 3 4))) 4) #f)
(test (equal? (cdr (list 'a 'b 'c 'd 'e 'f 'g)) 'a) #f)
(test (cdr '((((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f) g)) '(g))
(test (cdr '(a)) ())
(test (cdr '(a b c d e f g)) '(b c d e f g))
(test (cdr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((((u v w) x) y) ((q w e) r) (a b c) e f g))
(test (cdr ''foo) '(foo))
(test (cdr (cons (cons 1 2) (cons 3 4))) '(3 . 4))
(test (cdr '(1 2 . 3)) '(2 . 3))
(test (cdr (if #f #f)) 'error)
(test (cdr ()) 'error)

(for-each
 (lambda (arg)
   (if (not (equal? (cdr (cons () arg)) arg))
       (format-logged #t ";(cdr '(() ~A) -> ~A?~%" arg (cdr (cons () arg))))
   (test (cdr arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let* ((a (list 1 2 3))
       (b a))
  (set! (car a) (cadr a)) 
  (set! (cdr a) (cddr a))
  (test a (list 2 3))
  (test b a))

(define (cons-r a b n) (if (= 0 n) (cons a b) (cons (cons-r (+ a 1) (+ b 1) (- n 1)) (cons-r (- a 1) (- b 1) (- n 1)))))
(define (list-r a b n) (if (= 0 n) (list a b) (list (list-r (+ a 1) (+ b 1) (- n 1)) (list-r (- a 1) (- b 1) (- n 1)))))

(define lists (list (list 1 2 3)
		    (cons 1 2)
		    (list 1)
		    (list)
		    (list (list 1 2) (list 3 4))
		    (list (list 1 2) 3)
		    '(1 . 2)
		    '(a b c)
		    '((a) b (c))
		    '((1 2) (3 4))
		    '((1 2 3) (4 5 6) (7 8 9))
		    '(((1) (2) (3)) ((4) (5) (6)) ((7) (8) (9)))
		    '((((1 123) (2 124) (3 125) (4 126)) ((5) (6) (7) (8)) ((9) (10) (11) (12)) ((13) (14) (15) (16)))
		      (((21 127) (22 128) (23 129) (24 130)) ((25) (26) (27) (28)) ((29) (30) (31) (32)) ((33) (34) (35) (36)))
		      (((41 131) (42 132) (43 133) (44 134)) ((45) (46) (47) (48)) ((49) (50) (51) (52)) ((53) (54) (55) (56)))
		      (((61 135) (62 136) (63 137) (64 138)) ((65) (66) (67) (68)) ((69) (70) (71) (72)) ((73) (74) (75) (76)))
		      321)
		    (cons 1 (cons 2 (cons 3 4)))
		    (cons (cons 2 (cons 3 4)) 5)
		    (cons () 1)
		    (cons 1 ())
		    (cons () ())
		    (list 1 2 (cons 3 4) 5 (list (list 6) 7))
		    (cons-r 0 0 4)
		    (cons-r 0 0 5)
		    (cons-r 0 0 10)
		    (list-r 0 0 3)
		    (list-r 0 0 7)
		    (list-r 0 0 11)
		    ''a
		    ))


;;; --------------------------------------------------------------------------------
;;; cxr

(define (caar-1 x) (car (car x)))
(define (cadr-1 x) (car (cdr x)))
(define (cdar-1 x) (cdr (car x)))
(define (cddr-1 x) (cdr (cdr x)))
(define (caaar-1 x) (car (car (car x))))
(define (caadr-1 x) (car (car (cdr x))))
(define (cadar-1 x) (car (cdr (car x))))
(define (caddr-1 x) (car (cdr (cdr x))))
(define (cdaar-1 x) (cdr (car (car x))))
(define (cdadr-1 x) (cdr (car (cdr x))))
(define (cddar-1 x) (cdr (cdr (car x))))
(define (cdddr-1 x) (cdr (cdr (cdr x))))
(define (caaaar-1 x) (car (car (car (car x)))))
(define (caaadr-1 x) (car (car (car (cdr x)))))
(define (caadar-1 x) (car (car (cdr (car x)))))
(define (caaddr-1 x) (car (car (cdr (cdr x)))))
(define (cadaar-1 x) (car (cdr (car (car x)))))
(define (cadadr-1 x) (car (cdr (car (cdr x)))))
(define (caddar-1 x) (car (cdr (cdr (car x)))))
(define (cadddr-1 x) (car (cdr (cdr (cdr x)))))
(define (cdaaar-1 x) (cdr (car (car (car x)))))
(define (cdaadr-1 x) (cdr (car (car (cdr x)))))
(define (cdadar-1 x) (cdr (car (cdr (car x)))))
(define (cdaddr-1 x) (cdr (car (cdr (cdr x)))))
(define (cddaar-1 x) (cdr (cdr (car (car x)))))
(define (cddadr-1 x) (cdr (cdr (car (cdr x)))))
(define (cdddar-1 x) (cdr (cdr (cdr (car x)))))
(define (cddddr-1 x) (cdr (cdr (cdr (cdr x)))))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format-logged #t ";(~A ~S) -> ~S, (~A-1): ~S?~%" name lst val1 name val2))))
    lists))
 (list 'caar 'cadr 'cdar 'cddr 'caaar 'caadr 'cadar 'cdaar 'caddr 'cdddr 'cdadr 'cddar 
       'caaaar 'caaadr 'caadar 'cadaar 'caaddr 'cadddr 'cadadr 'caddar 'cdaaar 
       'cdaadr 'cdadar 'cddaar 'cdaddr 'cddddr 'cddadr 'cdddar)
 
 (list caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar 
       cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar)
 
 (list caar-1 cadr-1 cdar-1 cddr-1 caaar-1 caadr-1 cadar-1 cdaar-1 caddr-1 cdddr-1 cdadr-1 cddar-1 
       caaaar-1 caaadr-1 caadar-1 cadaar-1 caaddr-1 cadddr-1 cadadr-1 caddar-1 cdaaar-1 
       cdaadr-1 cdadar-1 cddaar-1 cdaddr-1 cddddr-1 cddadr-1 cdddar-1))



(test (equal? (cadr (list 'a 'b 'c 'd 'e 'f 'g)) 'b) #t)
(test (equal? (cddr (list 'a 'b 'c 'd 'e 'f 'g)) '(c d e f g)) #t)
(test (equal? (caddr (list 'a 'b 'c 'd 'e 'f 'g)) 'c) #t)
(test (equal? (cdddr (list 'a 'b 'c 'd 'e 'f 'g)) '(d e f g)) #t)
(test (equal? (cadddr (list 'a 'b 'c 'd 'e 'f 'g)) 'd) #t)
(test (equal? (cddddr (list 'a 'b 'c 'd 'e 'f 'g)) '(e f g)) #t)
(test (equal? (caadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '((u v w) x)) #t)
(test (equal? (cadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(6 7)) #t)
(test (equal? (cdaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(5)) #t)
(test (equal? (cdadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(y)) #t)
(test (equal? (cddar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) ()) #t)
(test (equal? (caaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(1 2 3)) #t)
(test (equal? (caadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (caaddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(q w e)) #t)
(test (equal? (cadaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 5) #t)
(test (equal? (cadadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 'y) #t)
(test (equal? (caddar (list (list (list (list (list 1 2 3) 4) 5) 1 6 (list 5 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) 6) #t)
(test (equal? (cadddr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(a b c)) #t)
(test (equal? (cdaaar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(4)) #t)
(test (equal? (cdaadr (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(x)) #t)
(test (equal? (cdadar (list (list (list (list (list 1 2 3) 4) 5) (list 6 7)) (list (list (list 'u 'v 'w) 'x) 'y) (list (list 'q 'w 'e) 'r) (list 'a 'b 'c) 'e 'f 'g)) '(7)) #t)

(test (caar '((a) b c d e f g)) 'a)
(test (cadr '(a b c d e f g)) 'b)
(test (cdar '((a b) c d e f g)) '(b))
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caaar '(((a)) b c d e f g)) 'a)
(test (caadr '(a (b) c d e f g)) 'b)
(test (cadar '((a b) c d e f g)) 'b)
(test (caddr '(a b c d e f g)) 'c)
(test (cdaar '(((a b)) c d e f g)) '(b))
(test (cdadr '(a (b c) d e f g)) '(c))
(test (cddar '((a b c) d e f g)) '(c))
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (caaaar '((((a))) b c d e f g)) 'a)
(test (caaadr '(a ((b)) c d e f g)) 'b)
(test (caadar '((a (b)) c d e f g)) 'b)
(test (caaddr '(a b (c) d e f g)) 'c)
(test (cadaar '(((a b)) c d e f g)) 'b)
(test (cadadr '(a (b c) d e f g)) 'c)
(test (caddar '((a b c) d e f g)) 'c)
(test (cadddr '(a b c d e f g)) 'd)
(test (cdaaar '((((a b))) c d e f g)) '(b))
(test (cdaadr '(a ((b c)) d e f g)) '(c))
(test (cdadar '((a (b c)) d e f g)) '(c))
(test (cdaddr '(a b (c d) e f g)) '(d))
(test (cddaar '(((a b c)) d e f g)) '(c))
(test (cddadr '(a (b c d) e f g)) '(d))
(test (cdddar '((a b c d) e f g)) '(d))
(test (cddddr '(a b c d e f g)) '(e f g))
(test (cadr '(1 2 . 3)) 2)
(test (cddr '(1 2 . 3)) 3)
(test (cadadr '''1) 1)
(test (cdadr '''1) '(1))

;; sacla
(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(test (caar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((1 2 3) 4) 5))
(test (cadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((u v w) x) y))
(test (cdar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((6 7)))
(test (cddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(((q w e) r) (a b c) e f g))
(test (caaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((1 2 3) 4))
(test (caadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((u v w) x))
(test (cadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(6 7))
(test (caddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((q w e) r))
(test (cdaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(5))
(test (cdadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(y))
(test (cddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cdddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '((a b c) e f g))
(test (caaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(1 2 3))
(test (caaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(u v w))
(test (caadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 6)
(test (caaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(q w e))
(test (cadaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 5)
(test (cadadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'y)
(test (cadddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(a b c))
(test (cdaaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(4))
(test (cdaadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(x))
(test (cdadar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(7))
(test (cdaddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(r))
(test (cddaar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cddadr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) ())
(test (cddddr '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) '(e f g))

(test (cadr '(a b c d e f g)) 'b)
(test (cddr '(a b c d e f g)) '(c d e f g))
(test (caddr '(a b c d e f g)) 'c)
(test (cdddr '(a b c d e f g)) '(d e f g))
(test (cadddr '(a b c d e f g)) 'd)
(test (cddddr '(a b c d e f g)) '(e f g))

(test (caar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((a . b) c . d))
(test (caar '(((a . b) c . d) (e . f) g . h)) '(a . b))
(test (caar '((a . b) c . d)) 'a)
(test (cadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((i . j) k . l))
(test (cadr '(((a . b) c . d) (e . f) g . h)) '(e . f))
(test (cadr '((a . b) c . d)) 'c)
(test (cdar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((e . f) g . h))
(test (cdar '(((a . b) c . d) (e . f) g . h)) '(c . d))
(test (cdar '((a . b) c . d)) 'b)
(test (cddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '((m . n) o . p))
(test (cddr '(((a . b) c . d) (e . f) g . h)) '(g . h))
(test (cddr '((a . b) c . d)) 'd)
(test (caaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(a . b))
(test (caaar '(((a . b) c . d) (e . f) g . h)) 'a)
(test (caadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(i . j))
(test (caadr '(((a . b) c . d) (e . f) g . h)) 'e)
(test (cddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(g . h))
(test (cddar '(((a . b) c . d) (e . f) g . h)) 'd)
(test (cdddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) '(o . p))
(test (cdddr '(((a . b) c . d) (e . f) g . h)) 'h)
(test (caaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'a)
(test (caaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'i)
(test (caddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'g)
(test (cadddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'o)
(test (cdaaar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'b)
(test (cdaadr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'j)
(test (cdddar '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'h)
(test (cddddr '((((a . b) c . d) (e . f) g . h) ((i . j) k . l) (m . n) o . p)) 'p)

(test (cadr ''foo) 'foo)

(test (caar '((a) b c)) 'a)
(test (cadr '(a b c)) 'b)
(test (cdar '((a . aa) b c)) 'aa)
(test (cddr '(a b . c)) 'c)
(test (caaar '(((a)) b c)) 'a)
(test (caadr '(a (b) c)) 'b)
(test (cadar '((a aa) b c)) 'aa)
(test (caddr '(a b c)) 'c)
(test (cdaar '(((a . aa)) b c)) 'aa)
(test (cdadr '(a (b . bb) c)) 'bb)
(test (cddar '((a aa . aaa) b c)) 'aaa)
(test (cdddr '(a b c . d)) 'd)
(test (caaaar '((((a))) b c)) 'a)
(test (caaadr '(a ((b)) c)) 'b)
(test (caadar '((a (aa)) b c)) 'aa)
(test (caaddr '(a b (c))) 'c)
(test (cadaar '(((a aa)) b c)) 'aa)
(test (cadadr '(a (b bb) c)) 'bb)
(test (caddar '((a aa aaa) b c)) 'aaa)
(test (cadddr '(a b c d)) 'd)
(test (cdaaar '((((a . aa))) b c)) 'aa)
(test (cdaadr '(a ((b . bb)) c)) 'bb)
(test (cdadar '((a (aa . aaa)) b c)) 'aaa)
(test (cdaddr '(a b (c . cc))) 'cc)
(test (cddaar '(((a aa . aaa)) b c)) 'aaa)
(test (cddadr '(a (b bb . bbb) c)) 'bbb)
(test (cdddar '((a aa aaa . aaaa) b c)) 'aaaa)
(test (cddddr '(a b c d . e)) 'e)

(let ((lst '((((A . B) C . D) (E . F) G . H) ((I . J) K . L) (M . N) O . P))) ; from comp.lang.lisp I think
  (test (car lst) '(((A . B) C . D) (E . F) G . H))
  (test (cdr lst) '(((I . J) K . L) (M . N) O . P))
  (test (caar lst) '((A . B) C . D))
  (test (cadr lst) '((I . J) K . L))
  (test (cdar lst) '((E . F) G . H))
  (test (cddr lst) '((M . N) O . P))
  (test (caaar lst) '(A . B))
  (test (caadr lst) '(I . J))
  (test (cadar lst) '(E . F))
  (test (caddr lst) '(M . N))
  (test (cdaar lst) '(C . D))
  (test (cdadr lst) '(K . L))
  (test (cddar lst) '(G . H))
  (test (cdddr lst) '(O . P))
  (test (caaaar lst) 'A)
  (test (caaadr lst) 'I)
  (test (caadar lst) 'E)
  (test (caaddr lst) 'M)
  (test (cadaar lst) 'C)
  (test (cadadr lst) 'K)
  (test (caddar lst) 'G)
  (test (cadddr lst) 'O)
  (test (cdaaar lst) 'B)
  (test (cdaadr lst) 'J)
  (test (cdadar lst) 'F)
  (test (cdaddr lst) 'N)
  (test (cddaar lst) 'D)
  (test (cddadr lst) 'L)
  (test (cdddar lst) 'H)
  (test (cddddr lst) 'P))

(test (recompose 10 cdr '(1 2 3 4 5 6 7 8 9 10 11 12)) '(11 12))
(test (recompose 10 car '(((((((((((1 2 3)))))))))))) '(1 2 3))

(test (cons 1 . 2) 'error)
(test (eval-string "(1 . 2 . 3)") 'error)
(test (car (list)) 'error)
(test (car ()) 'error)
(test (cdr (list)) 'error)
(test (cdr ()) 'error)
(test (caddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (cdddar '(((((1 2 3) 4) 5) (6 7)) (((u v w) x) y) ((q w e) r) (a b c) e f g)) 'error)
(test (caar '(a b c d e f g)) 'error)
(test (cdar '(a b c d e f g)) 'error)
(test (caaar '(a b c d e f g)) 'error)
(test (caadr '(a b c d e f g)) 'error)
(test (cadar '(a b c d e f g)) 'error)
(test (cdaar '(a b c d e f g)) 'error)
(test (cdadr '(a b c d e f g)) 'error)
(test (cddar '(a b c d e f g)) 'error)
(test (caaaar '(a b c d e f g)) 'error)
(test (caaadr '(a b c d e f g)) 'error)
(test (caadar '(a b c d e f g)) 'error)
(test (caaddr '(a b c d e f g)) 'error)
(test (cadaar '(a b c d e f g)) 'error)
(test (cadadr '(a b c d e f g)) 'error)
(test (caddar '(a b c d e f g)) 'error)
(test (cdaaar '(a b c d e f g)) 'error)
(test (cdaadr '(a b c d e f g)) 'error)
(test (cdadar '(a b c d e f g)) 'error)
(test (cdaddr '(a b c d e f g)) 'error)
(test (cddaar '(a b c d e f g)) 'error)
(test (cddadr '(a b c d e f g)) 'error)
(test (cdddar '(a b c d e f g)) 'error)
(test (caar 'a) 'error)
(test (caar '(a)) 'error)
(test (cadr 'a) 'error)
(test (cadr '(a . b)) 'error)
(test (cdar 'a) 'error)
(test (cdar '(a . b)) 'error)
(test (cddr 'a) 'error)
(test (cddr '(a . b)) 'error)
(test (caaar 'a) 'error)
(test (caaar '(a)) 'error)
(test (caaar '((a))) 'error)
(test (caadr 'a) 'error)
(test (caadr '(a . b)) 'error)
(test (caadr '(a b)) 'error)
(test (cadar 'a) 'error)
(test (cadar '(a . b)) 'error)
(test (cadar '((a . c) . b)) 'error)
(test (caddr 'a) 'error)
(test (caddr '(a . b)) 'error)
(test (caddr '(a c . b)) 'error)
(test (cdaar 'a) 'error)
(test (cdaar '(a)) 'error)
(test (cdaar '((a . b))) 'error)
(test (cdadr 'a) 'error)
(test (cdadr '(a . b)) 'error)
(test (cdadr '(a b . c)) 'error)
(test (cddar 'a) 'error)
(test (cddar '(a . b)) 'error)
(test (cddar '((a . b) . b)) 'error)
(test (cdddr 'a) 'error)
(test (cdddr '(a . b)) 'error)
(test (cdddr '(a c . b)) 'error)
(test (caaaar 'a) 'error)
(test (caaaar '(a)) 'error)
(test (caaaar '((a))) 'error)
(test (caaaar '(((a)))) 'error)
(test (caaadr 'a) 'error)
(test (caaadr '(a . b)) 'error)
(test (caaadr '(a b)) 'error)
(test (caaadr '(a (b))) 'error)
(test (caadar 'a) 'error)
(test (caadar '(a . b)) 'error)
(test (caadar '((a . c) . b)) 'error)
(test (caadar '((a c) . b)) 'error)
(test (caaddr 'a) 'error)
(test (caaddr '(a . b)) 'error)
(test (caaddr '(a c . b)) 'error)
(test (caaddr '(a c b)) 'error)
(test (cadaar 'a) 'error)
(test (cadaar '(a)) 'error)
(test (cadaar '((a . b))) 'error)
(test (cadaar '((a b))) 'error)
(test (cadadr 'a) 'error)
(test (cadadr '(a . b)) 'error)
(test (cadadr '(a b . c)) 'error)
(test (cadadr '(a (b . e) . c)) 'error)
(test (caddar 'a) 'error)
(test (caddar '(a . b)) 'error)
(test (caddar '((a . b) . b)) 'error)
(test (caddar '((a b . c) . b)) 'error)
(test (cadddr 'a) 'error)
(test (cadddr '(a . b)) 'error)
(test (cadddr '(a c . b)) 'error)
(test (cadddr '(a c e . b)) 'error)
(test (cdaaar 'a) 'error)
(test (cdaaar '(a)) 'error)
(test (cdaaar '((a))) 'error)
(test (cdaaar '(((a . b)))) 'error)
(test (cdaadr 'a) 'error)
(test (cdaadr '(a . b)) 'error)
(test (cdaadr '(a b)) 'error)
(test (cdaadr '(a (b . c))) 'error)
(test (cdadar 'a) 'error)
(test (cdadar '(a . b)) 'error)
(test (cdadar '((a . c) . b)) 'error)
(test (cdadar '((a c . d) . b)) 'error)
(test (cdaddr 'a) 'error)
(test (cdaddr '(a . b)) 'error)
(test (cdaddr '(a c . b)) 'error)
(test (cdaddr '(a c b . d)) 'error)
(test (cddaar 'a) 'error)
(test (cddaar '(a)) 'error)
(test (cddaar '((a . b))) 'error)
(test (cddaar '((a b))) 'error)
(test (cddadr 'a) 'error)
(test (cddadr '(a . b)) 'error)
(test (cddadr '(a b . c)) 'error)
(test (cddadr '(a (b . e) . c)) 'error)
(test (cdddar 'a) 'error)
(test (cdddar '(a . b)) 'error)
(test (cdddar '((a . b) . b)) 'error)
(test (cdddar '((a b . c) . b)) 'error)
(test (cddddr 'a) 'error)
(test (cddddr '(a . b)) 'error)
(test (cddddr '(a c . b)) 'error)
(test (cddddr '(a c e . b)) 'error)

(test (caar '((1))) 1)
(test (cadr '(1 2)) 2)
(test (cdar '((1 . 2))) 2)
(test (cddr '(1 2 . 3)) 3)
(test (caaar '(((1)))) 1)
(test (caadr '(1 (2))) 2)
(test (cadar '((1 2))) 2)
(test (cdaar '(((1 . 2)))) 2)
(test (caddr '(1 2 3)) 3)
(test (cdddr '(1 2 3 . 4)) 4)
(test (cdadr '(1 (2 . 3))) 3)
(test (cddar '((1 2 . 3))) 3)
(test (caaaar '((((1))))) 1)
(test (caaadr '(1 ((2)))) 2)
(test (caadar '((1 (2)))) 2)
(test (cadaar '(((1 2)))) 2)
(test (caaddr '(1 2 (3))) 3)
(test (cadddr '(1 2 3 4)) 4)
(test (cadadr '(1 (2 3))) 3)
(test (caddar '((1 2 3))) 3)
(test (cdaaar '((((1 . 2))))) 2)
(test (cdaadr '(1 ((2 . 3)))) 3)
(test (cdadar '((1 (2 . 3)))) 3)
(test (cddaar '(((1 2 . 3)))) 3)
(test (cdaddr '(1 2 (3 . 4))) 4)
(test (cddddr '(1 2 3 4 . 5)) 5)
(test (cddadr '(1 (2 3 . 4))) 4)
(test (cdddar '((1 2 3 . 4))) 4)

(let () (define (f1 x) (eq? (car x) 'y)) (let ((z 1)) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cdr x) 'y)) (let ((z 1)) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (caar x) 'y)) (let ((z (list 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cadr x) 'y)) (let ((z (cons 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cdar x) 'y)) (let ((z (list 1 2))) (test (f1 z) 'error)))
(let () (define (f1 x) (eq? (cddr x) 'y)) (let ((z (cons 1 2))) (test (f1 z) 'error)))



;;; --------------------------------------------------------------------------------
;;; length

(test (length (list 'a 'b 'c 'd 'e 'f)) 6)
(test (length (list 'a 'b 'c 'd)) 4)
(test (length (list 'a (list 'b 'c) 'd)) 3)
(test (length ()) 0)
(test (length '(this-that)) 1)
(test (length '(this - that)) 3)
(test (length '(a b)) 2)
(test (length '(a b c)) 3)
(test (length '(a (b) (c d e))) 3)
(test (length (list 1 (cons 1 2))) 2)
(test (length (list 1 (cons 1 ()))) 2)

(for-each
 (lambda (arg)
   (test (length arg) #f))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (length 'x) #f)
(test (length (cons 1 2)) -1)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (infinite? (length x)) #t))
(test (length '(1 2 . 3)) -2)
(test (length) 'error)
(test (length '(1 2 3) #(1 2 3)) 'error)
(test (integer? (length (funclet cons))) #t)

(test (length '((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) 1)



;;; --------------------------------------------------------------------------------
;;; reverse

(test (reverse '(a b c d)) '(d c b a))
(test (reverse '(a b c))  '(c b a))
(test (reverse '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse ()) ())
(test (reverse (list 1 2 3)) '(3 2 1))
(test (reverse (list 1)) '(1))
(test (reverse (list)) (list))
(test (reverse '(1 2 3)) (list 3 2 1))
(test (reverse '(1)) '(1))
(test (reverse '((1 2) 3)) '(3 (1 2)))
(test (reverse '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse (list (list) (list 1 2))) '((1 2) ()))
(test (reverse '((a) b c d)) '(d c b (a)))
(test (reverse (reverse (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse ''foo) '(foo quote))
(test (let ((x (list 1 2 3 4)))
	(let ((y (reverse x)))
	  (and (equal? x (list 1 2 3 4))
	       (equal? y (list 4 3 2 1)))))
      #t)
(test (letrec ((hi (lambda (lst n)
		     (if (= n 0)
			 lst
			 (hi (reverse lst) (- n 1))))))
	(hi (list 1 2 3) 100))
      (list 1 2 3))
(test (let ((var (list 1 2 3))) (reverse (cdr var)) var) (list 1 2 3))
(test (let ((var '(1 2 3))) (reverse (cdr var)) var) '(1 2 3))
(test (let ((var (list 1 (list 2 3)))) (reverse (cdr var)) var) (list 1 (list 2 3)))
(test (let ((var '(1 (2 3)))) (reverse (cdr var)) var) '(1 (2 3)))
(test (let ((var (list (list 1 2) (list 3 4 5)))) (reverse (car var)) var) '((1 2) (3 4 5)))
(test (let ((x '(1 2 3))) (list (reverse x) x)) '((3 2 1) (1 2 3)))
(test (reverse '(1 2)) '(2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse '(1 2 3 4)) '(4 3 2 1))

(when with-block 
  (test (block? (reverse _c_obj_)) #t)
  (let ((b (block 1 2 3 4)))
    (let ((b1 (reverse b)))
      (test b1 (block 4 3 2 1))
      (test b (block 1 2 3 4)))))

(for-each
 (lambda (lst)
   (if (proper-list? lst)
       (if (not (equal? lst (reverse (reverse lst))))
	   (format-logged #t ";(reverse (reverse ~A)) -> ~A?~%" lst (reverse (reverse lst))))))
 lists)

(for-each
 (lambda (lst)
   (if (proper-list? lst)
       (if (not (equal? lst (reverse (reverse (reverse (reverse lst))))))
	   (format-logged #t ";(reverse...(4x) ~A) -> ~A?~%" lst (reverse (reverse (reverse (reverse lst))))))))
 lists)

(test (let ((x (list 1 2 3))) (list (recompose 32 reverse x) x)) '((1 2 3) (1 2 3)))
(test (let ((x (list 1 2 3))) (list (recompose 31 reverse x) x)) '((3 2 1) (1 2 3)))

(test (reverse (cons 1 2)) '(2 . 1))
(test (reverse '(1 . 2)) '(2 . 1))
(test (reverse '(1 2 . 3)) '(3 2 1))
(test (reverse) 'error)
(test (reverse '(1 2 3) '(3 2 1)) 'error)
(test (reverse (make-shared-vector (make-int-vector '(2 3) 0) '(6))) (make-int-vector 6 0))
(test (reverse (make-float-vector 6 0.0)) (make-float-vector 6 0.0))

(for-each
 (lambda (arg)
   (test (reverse arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (reverse "hi") "ih")
(test (reverse "") "")
(test (reverse "123") "321")
(test (reverse "1234") "4321")
(test (reverse "12") "21")
(test (reverse "a\x00b") "b\x00a")
(test (reverse #()) #())
(test (reverse #(1)) #(1))
(test (reverse #(1 2)) #(2 1))
(test (reverse #(1 2 3)) #(3 2 1))
(test (reverse #(1 2 3 4)) #(4 3 2 1))
(test (reverse #2D((1 2) (3 4))) #2D((4 3) (2 1)))
(test (reverse (string #\a #\null #\b)) "b\x00a")
(test (reverse abs) 'error)
(test (vector->list (reverse (let ((v (make-int-vector 3))) (set! (v 1) 1) (set! (v 2) 2) v))) '(2 1 0))
(test (reverse (int-vector)) #())
(test (reverse (int-vector 1)) (int-vector 1))
(test (reverse (int-vector 1 2)) (int-vector 2 1))
(test (reverse (int-vector 1 2 3)) (int-vector 3 2 1))
(test (reverse (int-vector 1 2 3 4)) (int-vector 4 3 2 1))
(test (reverse (float-vector)) #())
(test (reverse (float-vector 1)) (float-vector 1))
(test (reverse (float-vector 1 2)) (float-vector 2 1))
(test (reverse (float-vector 1 2 3)) (float-vector 3 2 1))
(test (reverse (float-vector 1 2 3 4)) (float-vector 4 3 2 1))
(test (let ((v #(1 2 3))) (reverse v) v) #(1 2 3))
(test (reverse #u8(1 2 3)) #u8(3 2 1))
(test (reverse #u8(1 2)) #u8(2 1))
(test (reverse #u8(1 2 3 4)) #u8(4 3 2 1))

(when with-block
  (let ((b (block 1.0 2.0 3.0)))
    (set! (b 1) 32.0)
    (test (b 1) 32.0)
    (let ((b1 (reverse b)))
      (test b1 (block 3.0 32.0 1.0)))))




;;; --------------------------------------------------------------------------------
;;; reverse!

(test (reverse! '(1 . 2)) 'error)
(test (reverse! (cons 1 2)) 'error)
(test (reverse! (cons 1 (cons 2 3))) 'error)
(test (reverse!) 'error)
(test (reverse! '(1 2 3) '(3 2 1)) 'error)

(test (reverse! '(a b c d)) '(d c b a))
(test (reverse! '(a b c))  '(c b a))
(test (reverse! '(a (b c) d (e (f))))  '((e (f)) d (b c) a))
(test (reverse! ()) ())
(test (reverse! (list 1 2 3)) '(3 2 1))
(test (reverse! (list 1)) '(1))
(test (reverse! (list)) (list))
(test (reverse! '(1 2 3)) (list 3 2 1))
(test (reverse! '(1)) '(1))
(test (reverse! '((1 2) 3)) '(3 (1 2)))
(test (reverse! '(((1 . 2) . 3) 4)) '(4 ((1 . 2) . 3)))
(test (reverse! (list (list) (list 1 2))) '((1 2) ()))
(test (reverse! '((a) b c d)) '(d c b (a)))
(test (reverse! (reverse! (list 1 2 3 4))) (list 1 2 3 4))
(test (reverse! (reverse! (sort! (list 1 2 3 4) >))) (sort! (list 1 2 3 4) >))
(test (reverse! ''foo) '(foo quote))
(test (reverse (reverse! (list 1 2 3))) (list 1 2 3))
(test (reverse (reverse! (reverse! (reverse (list 1 2 3))))) (list 1 2 3))

(test (let ((x (list 1 2 3))) (recompose 31 reverse! x)) '(3 2 1))
(test (reverse! '(1 2 . 3)) 'error)

(let* ((lst1 (list 1 2 3))
       (lst2 (apply list '(4 5 6)))
       (lst3 (sort! (reverse! (append lst1 lst2)) <)))
  (test lst3 '(1 2 3 4 5 6))
  (define (lt . args)
    args)
  (set! lst3 (sort! (apply reverse! (lt lst3)) >))
  (test lst3 '(6 5 4 3 2 1)))

(for-each
 (lambda (arg)
   (test (reverse! arg) 'error))
 (list (integer->char 65) #f 'a-symbol abs _ht_ _null_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) (lambda (a) (+ a 1))))

(test (let ((str "1234")) (reverse! str) str) "4321")
(test (let ((str "123")) (reverse! str) str) "321")
(test (let ((str "")) (reverse! str) str) "")
(test (let ((v #(1 2 3))) (reverse! v) v) #(3 2 1))
(test (let ((v #(1 2 3 4))) (reverse! v) v) #(4 3 2 1))
(test (let ((v #())) (reverse! v) v) #())
(test (let ((v (float-vector 1.0 2.0 3.0))) (reverse! v) v) (float-vector 3.0 2.0 1.0))
(test (let ((v (float-vector 1.0 2.0 3.0 4.0))) (reverse! v) v) (float-vector 4.0 3.0 2.0 1.0))
(test (let ((v (float-vector))) (reverse! v) v) #())
(test (let ((v (int-vector 1 2 3))) (reverse! v) v) (int-vector 3 2 1))
(test (let ((v (int-vector 1 2 3 4))) (reverse! v) v) (int-vector 4 3 2 1))
(test (let ((v (int-vector))) (reverse! v) v) #())
(when with-block 
  (test (block? (reverse! _c_obj_)) #t)
  (let ((b (block 1 2 3 4)))
    (reverse! b)
    (test b (block 4 3 2 1))))
(test (let ((v (make-int-vector 3 1))) (set! (v 1) 2) (set! (v 2) 3) (reverse! v) v) (let ((v (make-int-vector 3 3))) (set! (v 1) 2) (set! (v 2) 1) v))

(when full-test
  (let ()
    ;; some sequence tests
    
    (define (fv-tst len)
      (let ((fv (make-float-vector len)))
	(if (not (= (length fv) len))
	    (format *stderr* "float-vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0.0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000.0) 500.0)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "float-vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "float-vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (morally-equal? fv fv-ran1))
		(format *stderr* "float-vector cond closure not equal~%")))
	  
	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (morally-equal? fv-copy fv))
		     (morally-equal? fv fv-orig))
		(format *stderr* "float-vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (morally-equal? fv-copy fv))
		(format *stderr* "float-vector reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply float-vector (make-list len 1.0))))
	      (if (or (not (= (length fv1) len))
		      (not (= (fv1 (- len 1)) 1.0)))
		  (format *stderr* "float-vector apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))
    
    (define (iv-tst len)
      (let ((fv (make-int-vector len 0)))
	(if (not (= (length fv) len))
	    (format *stderr* "int-vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000000) 500000)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "int-vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "int-vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (morally-equal? fv fv-ran1))
		(format *stderr* "int-vector cond closure not equal~%")))
	  
	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (morally-equal? fv-copy fv))
		     (morally-equal? fv fv-orig))
		(format *stderr* "int-vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (morally-equal? fv-copy fv))
		(format *stderr* "int-vector reverse! twice: ~A ~A~%" fv fv-copy))
	    ))))
    
    (define (v-tst len)
      (let ((fv (make-vector len)))
	(if (not (= (length fv) len))
	    (format *stderr* "vector length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (- (random 1000000) 500000)))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (> (fv (- i 1)) (fv i))
		   (format *stderr* "vector: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "vector closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((< a b) #t) (#t #f))))
	    (if (not (morally-equal? fv fv-ran1))
		(format *stderr* "vector cond closure not equal~%")))
	  
	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (morally-equal? fv-copy fv))
		     (morally-equal? fv fv-orig))
		(format *stderr* "vector reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (morally-equal? fv-copy fv))
		(format *stderr* "vector reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply vector (make-list len 1))))
	      (if (or (not (= (length fv1) len))
		      (not (= (fv1 (- len 1)) 1)))
		  (format *stderr* "vector apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))
    
    (define (s-tst len)
      (let ((fv (make-string len)))
	(if (not (= (length fv) len))
	    (format *stderr* "string length ~A: ~A~%" fv (length fv)))
	(fill! fv #\a)
	(let ((fv-orig (copy fv)))
	  (do ((i 0 (+ i 1)))
	      ((= i len))
	    (set! (fv i) (integer->char (+ 20 (random 100)))))
	  (let ((fv-ran (copy fv))
		(fv-ran1 (copy fv)))
	    (sort! fv char<?)
	    (call-with-exit
	     (lambda (quit)
	       (do ((i 1 (+ i 1)))
		   ((= i len))
		 (when (char>? (fv (- i 1)) (fv i))
		   (format *stderr* "string: ~A > ~A at ~D~%" (fv (- i 1)) (fv i) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (char<? a b)))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "string closure not equal~%"))
	    (sort! fv-ran1 (lambda (a b) (cond ((char<? a b) #t) (#t #f))))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "string cond closure not equal~%")))
	  
	  (let ((fv-copy (copy fv)))
	    (reverse! fv)
	    (if (and (not (morally-equal? fv-copy fv))
		     (morally-equal? fv fv-orig))
		(format *stderr* "string reverse!: ~A ~A~%" fv fv-orig))
	    (reverse! fv)
	    (if (not (morally-equal? fv-copy fv))
		(format *stderr* "string reverse! twice: ~A ~A~%" fv fv-copy))
	    (let ((fv1 (apply string (make-list len #\a))))
	      (if (or (not (= (length fv1) len))
		      (not (char=? (fv1 (- len 1)) #\a)))
		  (format *stderr* "string apply: ~A ~A~%" len (fv (- len 1)))))
	    ))))
    
    (define (p-tst len)
      (let ((fv (make-list len)))
	(if (not (= (length fv) len))
	    (format *stderr* "list length ~A: ~A~%" fv (length fv)))
	(fill! fv 0)
	(let ((fv-orig (copy fv)))
	  (do ((p fv (cdr p)))
	      ((null? p))
	    (set-car! p (- (random 100000) 50000)))
	  (let ((fv-ran (copy fv)))
	    (sort! fv <)
	    (call-with-exit
	     (lambda (quit)
	       (do ((p0 fv (cdr p0))
		    (p1 (cdr fv) (cdr p1))
		    (i 1 (+ i 1)))
		   ((null? p1))
		 (when (> (car p0) (car p1))
		   (format *stderr* "list: ~A > ~A at ~D~%" (car p0) (car p1) i)
		   (quit)))))
	    (sort! fv-ran (lambda (a b) (< a b)))
	    (if (not (morally-equal? fv fv-ran))
		(format *stderr* "pair closure not equal~%")))
	  
	  (let ((fv-copy (copy fv)))
	    (set! fv (reverse! fv))
	    (if (and (not (morally-equal? fv-copy fv))
		     (morally-equal? fv fv-orig))
		(format *stderr* "list reverse!: ~A ~A~%" fv fv-orig))
	    (set! fv (reverse! fv))
	    (if (not (morally-equal? fv-copy fv))
		(format *stderr* "list reverse! twice: ~A ~A~%" fv fv-copy))
	    ))))
    
    (for-each
     (lambda (b p)
       (do ((k 0 (+ k 1)))
	   ((= k 1000))
	 (fv-tst b)
	 (iv-tst b)
	 (v-tst b)
	 (s-tst b)
	 (p-tst b))
       (do ((i 0 (+ i 1)))
	   ((= i p))
	 (format *stderr* "~D fv " (expt b i))
	 (fv-tst (expt b i))
	 (format *stderr* "iv ")
	 (iv-tst (expt b i))
	 (format *stderr* "v ")
	 (v-tst (expt b i))
	 (format *stderr* "s ")
	 (s-tst (expt b i))
	 (format *stderr* "p ")
	 (p-tst (expt b i))
	 (newline *stderr*)
	 ))
     (list 2 3 4 7 10)
     (list 12 4 3 6 6))
    ))

(test (let ((v (vector 1 2 3 4))) (let ((sv (make-shared-vector v 3))) (reverse! sv) v)) #(3 2 1 4))
(test (let ((v (vector 1 2 3 4))) (let ((sv (make-shared-vector v 3 1))) (reverse! sv) v)) #(1 4 3 2))
(test (let ((v (vector 1 2 3 4))) (let ((sv (make-shared-vector v 3 1))) (fill! sv 5) v)) #(1 5 5 5))
(test (let ((v (vector 1 2 3 4))) (let ((sv (make-shared-vector v 3 1))) (reverse sv) v)) #(1 2 3 4))
(test (let ((v (vector 1 2 3 4))) (let ((sv (make-shared-vector v 3 1))) (sort! sv >) v)) #(1 4 3 2))
(test (let ((v (make-int-vector '(3 3) 1))) (let ((sv (v 1))) (fill! sv 2) v)) (make-shared-vector (int-vector 1 1 1 2 2 2 1 1 1) '(3 3)))

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (fill! sv 2)
	  v))
      (make-shared-vector (int-vector 0 1 2 2 2 2 6 7 8) '(3 3)))

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (sort! sv >)
	  v))
      (make-shared-vector (int-vector 0 1 2 5 4 3 6 7 8) '(3 3)))

(test (let ((v (make-int-vector '(3 3) 1)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (do ((j 0 (+ j 1)))
	      ((= j 3))
	    (set! (v i j) (+ j (* i 3)))))
	(let ((sv (v 1)))
	  (reverse! sv)
	  v)) 
      (make-shared-vector (int-vector 0 1 2 5 4 3 6 7 8) '(3 3)))




;;; --------------------------------------------------------------------------------
;;; pair?

(test (pair? 'a) #f)
(test (pair? '()) #f)
(test (pair? ()) #f)
(test (pair? '(a b c)) #t)
(test (pair? (cons 1 2)) #t)
(test (pair? ''()) #t)
(test (pair? #f) #f)
(test (pair? (make-vector 6)) #f)
(test (pair? #t) #f)
(test (pair? '(a . b)) #t)
(test (pair? #(a b))  #f)
(test (pair? (list 1 2)) #t)
(test (pair? (list)) #f)
(test (pair? ''foo) #t)
(test (pair? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (pair? '(this-that)) #t)
(test (pair? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (pair? x) #t))
(test (pair? (list 1 (cons 1 2))) #t)
(test (pair? (list 1 (cons 1 ()))) #t)
(test (pair? (cons 1 ())) #t)
(test (pair? (cons () ())) #t)
(test (pair? (cons () 1)) #t)
(test (pair? (list (list))) #t)
(test (pair? '(())) #t)
(test (pair? (cons 1 (cons 2 3))) #t)
(test (pair?) 'error)
(test (pair? `'1) #t)
(test (pair? begin) #f)
(test (pair? 'begin) #f)
(test (pair? ''begin) #t)
(test (pair? list) #f)

(for-each
 (lambda (arg)
   (if (pair? arg)
       (format-logged #t ";(pair? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; pair-line-number

(test (pair-line-number) 'error)
(test (pair-line-number () ()) 'error)
(for-each
 (lambda (arg)
   (test (pair-line-number arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

;;; pair-filename

(test (pair-filename) 'error)
(test (pair-filename () ()) 'error)
(for-each
 (lambda (arg)
   (test (pair-filename arg) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))




;;; --------------------------------------------------------------------------------
;;; list?

(test (list? 'a) #f)
(test (list? ()) #t)
(test (list? '(a b c)) #t)
(test (list? (cons 1 2)) #t)
(test (list? ''()) #t)
(test (list? #f) #f)
(test (list? (make-vector 6)) #f)
(test (list? #t) #f)
(test (list? '(a . b)) #t)
(test (list? #(a b))  #f)
(test (list? (list 1 2)) #t)
(test (list? (list)) #t)
(test (list? ''foo) #t)
(test (list? ''2) #t)
(test (list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (list? '(this-that)) #t)
(test (list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (proper-list? x) #f)
  (test (list? x) #t))
(test (list? (list 1 (cons 1 2))) #t)
(test (list? (list 1 (cons 1 ()))) #t)
(test (list? (cons 1 ())) #t)
(test (list? (cons () ())) #t)
(test (list? (cons () 1)) #t)
(test (list? (list (list))) #t)
(test (list? '(())) #t)
(test (list? '(1 2 . 3)) #t)
(test (list? (cons 1 (cons 2 3))) #t)
(test (list? '(1 . ())) #t)

(test (list? '(1 2) ()) 'error)
(test (list?) 'error)
(for-each
 (lambda (arg)
   (if (list? arg)
       (format-logged #t ";(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; proper-list?

(test (proper-list? 'a) #f)
(test (proper-list? ()) #t)
(test (proper-list? '(a b c)) #t)
(test (proper-list? (cons 1 2)) #f)
(test (proper-list? ''()) #t)
(test (proper-list? #f) #f)
(test (proper-list? (make-vector 6)) #f)
(test (proper-list? #t) #f)
(test (proper-list? '(a . b)) #f)
(test (proper-list? #(a b))  #f)
(test (proper-list? (list 1 2)) #t)
(test (proper-list? (list)) #t)
(test (proper-list? ''foo) #t)
(test (proper-list? ''2) #t)
(test (proper-list? (list 'a 'b 'c 'd 'e 'f)) #t)
(test (proper-list? '(this-that)) #t)
(test (proper-list? '(this - that)) #t)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (proper-list? x) #f))
(test (proper-list? (list 1 (cons 1 2))) #t)
(test (proper-list? (list 1 (cons 1 ()))) #t)
(test (proper-list? (cons 1 ())) #t)
(test (proper-list? (cons () ())) #t)
(test (proper-list? (cons () 1)) #f)
(test (proper-list? (list (list))) #t)
(test (proper-list? '(())) #t)
(test (proper-list? '(1 2 . 3)) #f)
(test (proper-list? (cons 1 (cons 2 3))) #f)
(test (proper-list? '(1 . ())) #t)

(test (proper-list? '(1 2) ()) 'error)
(test (proper-list?) 'error)
(for-each
 (lambda (arg)
   (if (proper-list? arg)
       (format-logged #t ";(list? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))


;;; --------------------------------------------------------------------------------
;;; null?

(test (null? 'a) '#f)
(test (null? ()) #t)
(test (null? ()) #t)
(test (null? '(a b c)) #f)
(test (null? (cons 1 2)) #f)
(test (null? ''()) #f)
(test (null? #f) #f)
(test (null? (make-vector 6)) #f)
(test (null? #t) #f)
(test (null? '(a . b)) #f)
(test (null? #(a b))  #f)
(test (null? (list 1 2)) #f)
(test (null? (list)) #t)
(test (null? ''foo) #f)
(test (null? (list 'a 'b 'c 'd 'e 'f)) #f)
(test (null? '(this-that)) #f)
(test (null? '(this - that)) #f)
(let ((x (list 1 2)))
  (set-cdr! x x)
  (test (null? x) #f))
(test (null? (list 1 (cons 1 2))) #f)
(test (null? (list 1 (cons 1 ()))) #f)
(test (null? (cons 1 ())) #f)
(test (null? (cons () ())) #f)
(test (null? (cons () 1)) #f)
(test (null? (list (list))) #f)
(test (null? '(())) #f)
(test (null? #()) #f)
(test (null? (make-vector '(2 0 3))) #f)
(test (null? "") #f)
(test (null? lambda) #f)
(test (null? cons) #f)
(test (null? (begin)) #t)
(test (null? (cdr (list 1))) #t)
(test (null? (cdr (cons () '(())))) #f)

(test (null? () ()) 'error)
(test (null?) 'error)

(for-each
 (lambda (arg)
   (if (null? arg)
       (format-logged #t ";(null? ~A) -> #t?~%" arg)))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) :hi #<eof> #<undefined> (values) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; set-car!

(test (let ((x (cons 1 2))) (set-car! x 3) x) (cons 3 2))
(test (let ((x (list 1 2))) (set-car! x 3) x) (list 3 2))
(test (let ((x (list (list 1 2) 3))) (set-car! x 22) x) (list 22 3))
(test (let ((x (cons 1 2))) (set-car! x ()) x) (cons () 2))
(test (let ((x (list 1 (list 2 3 4)))) (set-car! x (list 5 (list 6))) x) (list (list 5 (list 6)) (list 2 3 4)))
(test (let ((x '(((1) 2) (3)))) (set-car! x '((2) 1)) x) '(((2) 1) (3)))
(test (let ((x ''foo)) (set-car! x "hi") x) (list "hi" 'foo))
(test (let ((x '((1 . 2) . 3))) (set-car! x 4) x) '(4 . 3))
(test (let ((x '(1 . 2))) (set-car! x (cdr x)) x) '(2 . 2))
(test (let ((x '(1 . 2))) (set-car! x x) (proper-list? x)) #f)
(test (let ((x (list 1))) (set-car! x ()) x) '(()))
(test (let ((x '(((1 2) . 3) 4))) (set-car! x 1) x) '(1 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! (cdr lst) 4) lst) (cons 1 (cons 4 3)))
(test (let ((lst (cons 1 (cons 2 3)))) (set-car! lst 4) lst) (cons 4 (cons 2 3)))
(test (let ((x (list 1 2))) (set! (car x) 0) x) (list 0 2))
(test (let ((x (cons 1 2))) (set! (cdr x) 0) x) (cons 1 0))
(test (let ((x (list 1 2))) (set-car! x (list 3 4)) x) '((3 4) 2))
(test (let ((x (cons 1 2))) (set-car! x (list 3 4)) x) '((3 4) . 2))
(test (let ((x (cons (list 1 2) 3))) (set-car! (car x) (list 3 4)) x) '(((3 4) 2) . 3))
(test (let ((lst (list 1 2 3))) (set! (car lst) 32) lst) '(32 2 3))

(test (set-car! '() 32) 'error)
(test (set-car! () 32) 'error)
(test (set-car! (list) 32) 'error)
(test (set-car! 'x 32) 'error)
(test (set-car! #f 32) 'error)
(test (set-car!) 'error)
(test (set-car! '(1 2) 1 2) 'error)
(test (let ((lst (list 1 2))) (set-car! lst (values 2 3)) lst) 'error)
(test (let ((lst '(1 2))) (set-car! lst 32)) 32)
(test (let ((lst '(1 2))) (set! (car lst) 32)) 32)

(test (let ((c (cons 1 2))) (set-car! c #\a) (car c)) #\a)
(test (let ((c (cons 1 2))) (set-car! c #()) (car c)) #())
(test (let ((c (cons 1 2))) (set-car! c #f) (car c)) #f)
(test (let ((c (cons 1 2))) (set-car! c _ht_) (car c)) _ht_)



;;; --------------------------------------------------------------------------------
;;; set-cdr!

(test (let ((x (cons 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list 1 2))) (set-cdr! x 3) x) (cons 1 3))
(test (let ((x (list (list 1 2) 3))) (set-cdr! x 22) x) '((1 2) . 22))
(test (let ((x (cons 1 2))) (set-cdr! x '()) x) (list 1))
(test (let ((x (list 1 (list 2 3 4)))) (set-cdr! x (list 5 (list 6))) x) '(1 5 (6)))
(test (let ((x '(((1) 2) (3)))) (set-cdr! x '((2) 1)) x) '(((1) 2) (2) 1))
(test (let ((x ''foo)) (set-cdr! x "hi") x) (cons 'quote "hi"))
(test (let ((x '((1 . 2) . 3))) (set-cdr! x 4) x) '((1 . 2) . 4))
(test (let ((x '(1 . 2))) (set-cdr! x (cdr x)) x) '(1 . 2))
(test (let ((x '(1 . 2))) (set-cdr! x x) (proper-list? x)) #f)
(test (let ((x (list 1))) (set-cdr! x '()) x) (list 1))
(test (let ((x '(1 . (2 . (3 (4 5)))))) (set-cdr! x 4) x) '(1 . 4))
(test (let ((lst (cons 1 (cons 2 3)))) (set-cdr! (cdr lst) 4) lst) (cons 1 (cons 2 4)))
(test (let ((x (cons (list 1 2) 3))) (set-cdr! (car x) (list 3 4)) x) '((1 3 4) . 3))
(test (let ((x (list 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5))
(test (let ((x (cons 1 2))) (set-cdr! x (list 4 5)) x) '(1 4 5)) ;!
(test (let ((x (cons 1 2))) (set-cdr! x (cons 4 5)) x) '(1 4 . 5))
(test (let ((lst (list 1 2 3))) (set! (cdr lst) 32) lst) (cons 1 32))

(test (set-cdr! '() 32) 'error)
(test (set-cdr! () 32) 'error)
(test (set-cdr! (list) 32) 'error)
(test (set-cdr! 'x 32) 'error)
(test (set-cdr! #f 32) 'error)
(test (set-cdr!) 'error)
(test (set-cdr! '(1 2) 1 2) 'error)
(test (let ((lst '(1 2))) (set-cdr! lst 32)) 32)
(test (let ((lst '(1 2))) (set! (cdr lst) 32)) 32)

(test (let ((c (cons 1 2))) (set-cdr! c #\a) (cdr c)) #\a)
(test (let ((c (cons 1 2))) (set-cdr! c #()) (cdr c)) #())
(test (let ((c (cons 1 2))) (set-cdr! c #f) (cdr c)) #f)
(test (let ((c (cons 1 2))) (set-cdr! c _ht_) (cdr c)) _ht_)
(test (let ((c (cons 1 2))) (set-cdr! c (list 3)) c) '(1 3))




;;; --------------------------------------------------------------------------------
;;; list-ref

(test (list-ref (list 1 2) 1) 2)
(test (list-ref '(a b c d) 2) 'c)
(test (list-ref (cons 1 2) 0) 1) ; !!
(test (list-ref ''foo 0) 'quote)
(test (list-ref '((1 2) (3 4)) 1) '(3 4))
(test (list-ref (list-ref (list (list 1 2) (list 3 4)) 1) 1) 4)
(test (let ((x (list 1 2 3))) (list-ref x (list-ref x 1))) 3)
(test (list-ref '(1 2 . 3) 1) 2)
(test (list-ref '(1 2 . 3) 2) 'error) ; hmm...
(test ('(1 2 . 3) 0) 1)
(test ('(1 . 2) 0) 1)

(test (let ((lst (list 1 2))) (set! (list-ref lst 1) 0) lst) (list 1 0))
(test (((lambda () list)) 'a 'b 'c) '(a b c))
(test (apply ((lambda () list)) (list 'a 'b 'c) (list 'c 'd 'e)) '((a b c) c d e))
(test (((lambda () (values list))) 1 2 3) '(1 2 3))
(test (apply list 'a 'b '(c)) '(a b c))

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format-logged #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-ref:0 'list-ref:1 'list-ref:2 'list-ref:3)
 (list car cadr caddr cadddr)
 (list (lambda (l) (list-ref l 0)) (lambda (l) (list-ref l 1)) (lambda (l) (list-ref l 2)) (lambda (l) (list-ref l 3))))

(for-each
 (lambda (arg)
   (test (list-ref (list 1 arg) 1) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 0)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 1)) 1)
(test (let ((x '(1 . 2))) (set-cdr! x x) (list-ref x 100)) 1)

(test (list-ref '((1 2 3) (4 5 6)) 1) '(4 5 6))
(test (list-ref '((1 2 3) (4 5 6)) 1 2) 6)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test ('((1 2 3) (4 5 6)) 1) '(4 5 6))
(test ('((1 2 3) (4 5 6)) 1 2) 6)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1) '((7 8 9) (10 11 12)))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0) '(7 8 9))
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2) 9)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 3) 'error)
(test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) 1 0 2 0) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (L 1)) '(4 5 6))
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1)) '((7 8 9) (10 11 12)))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) ((L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L 1) 2)) 6)
(test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L 1) 2) 3)) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L 1) 0)) '(7 8 9))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L 1) 0) 2)) 9)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (((L 1) 0) 2) 3)) 'error)

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3))
  (test (list-ref '((1 2 3) (4 5 6)) one) '(4 5 6))
  (test (list-ref '((1 2 3) (4 5 6)) 1 two) 6)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test (list-ref '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test ('((1 2 3) (4 5 6)) one) '(4 5 6))
  (test ('((1 2 3) (4 5 6)) 1 two) 6)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one) '((7 8 9) (10 11 12)))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero) '(7 8 9))
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero two) 9)
  (test ('(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) one zero three) 'error)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (L one)) '(4 5 6))
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (L 1 2 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one)) '((7 8 9) (10 11 12)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) ((L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((L one) 0 two)) 9)
  
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref (L one) two)) 6)
  (test (let ((L '((1 2 3) (4 5 6)))) (list-ref ((L one) two) 3)) 'error)
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-ref (L one) zero)) '(7 8 9))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((list-ref (L one) zero) two)) 9))


(test (list-ref () 0) 'error)
(test (list-ref (list 1 2) 2) 'error)
(test (list-ref (list 1 2) -1) 'error)
(test (list-ref (list 1 2) 1.3) 'error)
(test (list-ref (list 1 2) 1/3) 'error)
(test (list-ref (list 1 2) 1+2.0i) 'error)
(test (list-ref (cons 1 2) 1) 'error)
(test (list-ref (cons 1 2) 2) 'error)
(test (list-ref (list 1 2 3) (expt 2 32)) 'error)
(test (list-ref '(1 2 3) 1 2) 'error)
(test (list-ref) 'error)
(test (list-ref '(1 2)) 'error)
(test ('(0)) 'error)
(test ((0)) 'error)
(test (list-ref '((1 2) (3 4)) 1 1) 4)
(test ('(1 2 3) 1) 2)
(test ((list 1 2 3) 2) 3)
(test ((list)) 'error)
(test ((list 1) 0 0) 'error)
(test ((list 1 (list 2 3)) 1 1) 3)
(test ((append '(3) () '(1 2)) 0) 3)
(test ((append '(3) () 1) 0) 3)
(test ((append '(3) () 1) 1) 'error)
;; this works with 0 because:
(test ((cons 1 2) 0) 1)
(test (list-ref (cons 1 2) 0) 1)
(test (((list (list 1 2 3)) 0) 0) 1)
(test (((list (list 1 2 3)) 0) 1) 2)
(test (((list (list 1 2 3)) 0 1)) 'error) ; see below
(test (let ((lst (list (list 1 2 3)))) (lst 0 1)) 2) 
(test ((list (list 1 2 3)) 0 1) 2)
(test (list-ref (list (list 1 2)) 0 ()) 'error)
(test (((list +) 0) 1 2 3) 6)


(let ((lst (list 1 2)))
  (for-each
   (lambda (arg)
     (test (list-ref (list 1 2) arg) 'error)
     (test ((list 1 2) arg) 'error)
     (test (lst arg) 'error))
   (list "hi" (integer->char 65) #f '(1 2) () 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))




;;; --------------------------------------------------------------------------------
;;; list-set!

(test (let ((x (list 1))) (list-set! x 0 2) x) (list 2))
(test (let ((x (cons 1 2))) (list-set! x 0 3) x) '(3 . 2))
(test (let ((x (cons 1 2))) (list-set! x 1 3) x) 'error)
(test (let ((x '((1) 2))) (list-set! x 0 1) x) '(1 2))
(test (let ((x '(1 2))) (list-set! x 1 (list 3 4)) x) '(1 (3 4)))
(test (let ((x ''foo)) (list-set! x 0 "hi") x ) '("hi" foo))
(test (let ((x (list 1 2))) (list-set! x 0 x) (list? x)) #t)
(test (let ((x (list 1 2))) (list-set! x 1 x) (list? x)) #t)
(test (let ((x 2) (lst '(1 2))) (list-set! (let () (set! x 3) lst) 1 23) (list x lst)) '(3 (1 23)))
(test (apply list-set! '((1 2) (3 2)) 1 '(1 2)) 2)

(test (list-set! '(1 2 3) 1 4) 4)
(test (set-car! '(1 2) 4) 4)
(test (set-cdr! '(1 2) 4) 4)
(test (fill! (list 1 2) 4) 4)
(test (fill! () 1) 1)
(test (list-set! '(1 2 . 3) 1 23) 23)
(test (list-set! '(1 2 . 3) 2 23) 'error)
(test (set! ('(1 2 . 3) 1) 23) 23)
(test (let ((lst '(1 2 3))) (list-set! lst 0 32)) 32)
(test (let ((lst '(1 2 3))) (set! (lst 0) 32)) 32)
(test (let ((lst '(1 2 3))) (set! (list-ref lst 0) 32)) 32)

(for-each
 (lambda (arg)
   (test (let ((x (list 1 2))) (list-set! x 0 arg) (list-ref x 0)) arg))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 0 2 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (list-set! L 1 3 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 0 2 1 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L 1 4 2 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1) 32) L) '((1 2 3) 32))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1) 32) L) '(((1 2 3) (4 5 6)) 32))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 0) 32) L) '((1 2 3) (32 5 6)))
(test (let ((L '((1 2 3) (4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L '((1 2 3) (4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1) 0) 32) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L '(((1 2 3))))) (set! ((L 0) 0 1) 32) L) '(((1 32 3))))
(test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L 1 0) 2) 32) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L 0 0 1) 32) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0) 0 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L 0 0 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0) 0) 1 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L 0 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L 0) 0) 1) 2) 32) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
(test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) 1) 32) L)) '(1 32 3))

(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (list-set! L one zero thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (list-set! L one zero two thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one) thirty-two) L) '((1 2 3) 32))
  (test (let ((L '((1 2 3) (4 5 6)))) (set! (L one zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one) thirty-two) L) '(((1 2 3) (4 5 6)) 32))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (L one zero two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  
  (test (let ((L '((1 2 3) (4 5 6)))) (set! ((L one) zero) thirty-two) L) '((1 2 3) (32 5 6)))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one) zero) thirty-two) L) '(((1 2 3) (4 5 6)) (32 (10 11 12))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((L one) zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '(((1 2 3))))) (set! ((L zero) zero one) thirty-two) L) '(((1 32 3))))
  (test (let ((L '(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! ((L one zero) two) thirty-two) L) '(((1 2 3) (4 5 6)) ((7 8 32) (10 11 12))))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (L zero zero one) thirty-two) L) '((((1 2 3) 32) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero) zero one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((L zero 0 one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero) zero) one two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! (((L zero zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '((((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))) (13 14 15)))) (set! ((((L zero) zero) one) two) thirty-two) L) '((((1 2 3) (4 5 32)) ((7 8 9) (10 11 12))) (13 14 15)))
  (test (let ((L '(1 2 3))) (let ((L1 (list L))) (set! ((car L1) one) thirty-two) L)) '(1 32 3)))
  
(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) (list x y)) '((32) (2)))
  
(test (list-set! () 0 1) 'error)
(test (list-set! () -1 1) 'error)
(test (list-set! '(1) 1 2) 'error)
(test (list-set! '(1 2 3) -1 2) 'error)
(test (list-set! '(1) 1.5 2) 'error)
(test (list-set! '(1) 3/2 2) 'error)
(test (list-set! '(1) 1+3i 2) 'error)
(test (list-set! '(1 2 3) 1 2 3) 'error)
(test (list-set! (list 1 2 3) (expt 2 32)  0) 'error)
(test (list-set! (list 1 2) () 1) 'error)

(for-each
 (lambda (arg)
   (test (list-set! (list 1 2) arg arg) 'error)
   (test (list-set! arg 1 2) 'error)
   (test (list-set! (list 1 2) arg 1) 'error))
 (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; list

(test (let ((tree1 (list 1 (list 1 2) (list (list 1 2 3)) (list (list (list 1 2 3 4)))))) tree1) '(1 (1 2) ((1 2 3)) (((1 2 3 4)))))
(test (let ((tree2 (list "one" (list "one" "two") (list (list "one" "two" "three"))))) tree2) '("one" ("one" "two") (("one" "two" "three"))))
(test (let ((tree1 (list 1 (list 1 2) (list 1 2 3) (list 1 2 3 4)))) tree1) '(1 (1 2) (1 2 3) (1 2 3 4)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) tree2) '(1 (1 2)))
(test (let ((tree1 (list 1 (list 1 2))) (tree2 (list 1 (list 1 2)))) (eqv? tree1 tree2)) #f)
(test (let ((tree1 (list ''a (list ''b ''c))) (tree2 (list ''a (list ''b ''c)))) tree2) '('a ('b 'c)))
(test (let ((lst (list 1 (list 2 3)))) lst) '(1 (2 3)))
(test (let* ((lst (list 1 (list 2 3))) (slst lst)) slst) '(1 (2 3)))
(test (list 1) '(1))
(test (let ((a 1)) (list a 2)) '(1 2))
(test (let ((a 1)) (list 'a '2)) '(a 2))
(test (let ((a 1)) (list 'a 2)) '(a 2))
(test (list) ())
(test (let ((a (list 1 2))) a) '(1 2))
(test (let ((a (list 1 2))) (list 3 4 'a (car (cons 'b 'c)) (+ 6 -2))) '(3 4 a b 4))
(test (list) ())
(test (length (list quote do map call/cc lambda define if begin set! let let* cond and or for-each)) 15)
(test (list 1(list 2)) '(1(2)))
(test (list 1 2 . 3) 'error)
;(test (list 1 2 , 3) 'error) ; ,3 -> 3 in the reader now
(test (list 1 2 ,@ 3) 'error)




;;; --------------------------------------------------------------------------------
;;; list-tail

(test (list-tail '(1 2 3) 0) '(1 2 3))
(test (list-tail '(1 2 3) 2) '(3))
(test (list-tail '(1 2 3) 3) ())
(test (list-tail '(1 2 3 . 4) 2) '(3 . 4))
(test (list-tail '(1 2 3 . 4) 3) 4)
(test (let ((x (list 1 2 3))) (eq? (list-tail x 2) (cddr x))) #t)
(test (list-tail () 0) ())
(test (list-tail () 1) 'error)
(test (list-tail '(1 2 3) 4) 'error)
(test (list-tail () -1) 'error)
(test (list-tail (list 1 2) 2) ())
(test (list-tail (cons 1 2) 0) '(1 . 2))
(test (list-tail (cons 1 2) 1) 2)
(test (list-tail (cons 1 2) 2) 'error)
(test (list-tail (cons 1 2) -1) 'error)
(test (list-tail ''foo 1) '(foo))
(test (list-tail '((1 2) (3 4)) 1) '((3 4)))
(test (list-tail (list-tail '(1 2 3) 1) 1) '(3))
(test (list-tail (list-tail (list-tail '(1 2 3 4) 1) 1) 1) '(4))
(test (list-tail '(1 2) (list-tail '(0 . 1) 1)) '(2))

(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 0) x))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 1) (cdr x)))
(let ((x '(1 . 2))) (set-cdr! x x) (test (list-tail x 100) x))

(let ((x (list 1 2 3)))
  (let ((y (list-tail x 1)))
    (set! (y 1) 32)
    (test (equal? y '(2 32)) #t)
    (test (equal? x '(1 2 32)) #t))) ; list-tail is not like substring

(for-each
 (lambda (name op1 op2)
   (for-each
    (lambda (lst)
      (let ((val1 (catch #t (lambda () (op1 lst)) (lambda args 'error)))
	    (val2 (catch #t (lambda () (op2 lst)) (lambda args 'error))))
	(if (not (equal? val1 val2))
	    (format-logged #t ";(~A ~A) -> ~A ~A?~%" name lst val1 val2))))
    lists))
 (list 'list-tail:0 'list-tail:1 'list-tail:2 'list-tail:3 'list-tail:4)
 (list (lambda (l) l) cdr cddr cdddr cddddr)
 (list (lambda (l) (list-tail l 0)) (lambda (l) (list-tail l 1)) (lambda (l) (list-tail l 2)) (lambda (l) (list-tail l 3)) (lambda (l) (list-tail l 4))))

(test (list-tail (list 1 2) 3) 'error)
(test (list-tail (list 1 2) -1) 'error)
(test (list-tail (list 1 2) 1.3) 'error)
(test (list-tail (list 1 2) 1/3) 'error)
(test (list-tail (list 1 2) 1+2.0i) 'error)
(test (list-tail '(1 2 . 3)) 'error)
(test (list-tail '(1 2 . 3) 1) '(2 . 3))
(test (list-tail '(1 2 . 3) 0) '(1 2 . 3))
(test (list-tail (list 1 2 3) (+ 1 (expt 2 32))) 'error)
(test (list-tail) 'error)
(test (list-tail '(1)) 'error)
(test (list-tail '(1) 1 2) 'error)
(test (set! (list-tail (list 1 2 3)) '(32)) 'error) ; should this work?

(for-each
 (lambda (arg)
   (test (list-tail (list 1 2) arg) 'error)
   (test (list-tail arg 0) 'error))
 (list "hi" -1 3 most-negative-fixnum most-positive-fixnum 
       (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f #t (if #f #f) #<eof> #() #(1 2 3) (lambda (a) (+ a 1))))


;;; --------------------------------------------------------------------------------
;;; make-list 

(test (make-list 0) ())
(test (make-list 0 123) ())
(test (make-list 1) '(#f))
(test (make-list 1 123) '(123))
(test (make-list 1 ()) '(()))
(test (make-list 2) '(#f #f))
(test (make-list 2 1) '(1 1))
(test (make-list 2/1 1) '(1 1))
(test (make-list 2 (make-list 1 1)) '((1) (1)))
(test (make-list -1) 'error)
(test (make-list -0) ())
(test (make-list most-negative-fixnum) 'error)
(test (make-list most-positive-fixnum) 'error)
(test (make-list 0 #\a) ())
(test (make-list 1 #\a) '(#\a))

(for-each
 (lambda (arg)
   (test (make-list arg) 'error))
 (list #\a #(1 2 3) 3.14 3/4 1.0+1.0i 0.0 1.0 () #t 'hi #(()) (list 1 2 3) '(1 . 2) "hi" (- (real-part (log 0.0)))))

(for-each
 (lambda (arg)
   (test ((make-list 1 arg) 0) arg))
 (list #\a #(1 2 3) 3.14 3/4 1.0+1.0i () #f 'hi #(()) (list 1 2 3) '(1 . 2) "hi"))

(test (make-list) 'error)
(test (make-list 1 2 3) 'error)
(test (let ((lst (make-list 2 (make-list 1 0)))) (eq? (lst 0) (lst 1))) #t)




;;; --------------------------------------------------------------------------------
;;; assq

(let ((e '((a 1) (b 2) (c 3))))
  (test (assq 'a e) '(a 1))
  (test (assq 'b e) '(b 2))
  (test (assq 'd e) #f))
(test (assq (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assq #t e) (list #t 1))
    (test (assq #f e) (list #f 2))
    (test (assq 'a e) (list 'a 3))
    (test (assq xcons e) (list xcons 4))
    (test (assq xvect e) (list xvect 5))
    (test (assq xlambda e) (list xlambda 6))
    (test (assq xstr e) (list xstr 7))
    (test (assq car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3))))
  (test (assq 1+i e) #f)
  (test (assq 3.0 e) #f)
  (test (assq 5/3 e) #f))

(test (assq 'x (cdr (assq 'a '((b . 32) (a . ((a . 12) (b . 32) (x . 1))) (c . 1))))) '(x . 1))

(test (assq #f '(#f 2 . 3)) #f)
(test (assq #f '((#f 2) . 3)) '(#f 2))
(test (assq () '((() 1) (#f 2))) '(() 1))
(test (assq () '((1) (#f 2))) #f)
(test (assq #() '((#f 1) (() 2) (#() 3))) #f)  ; (eq? #() #()) -> #f

(test (assq 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assq 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assq 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assq 'b (list '(a . 1) '(b . 2) () '(c . 3) #f)) '(b . 2))
(test (assq 'asdf (list '(a . 1) '(b . 2) () '(c . 3) #f)) #f)
(test (assq "" (list '("a" . 1) '("" . 2) '(#() . 3))) #f) ; since (eq? "" "") is #f
(test (assq 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assq 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

;; check the even/odd cases
(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assq 'a odd) '(a . 2))
  (test (assq 'a even) '(a . 3))
  (test (assq 3/4 odd) #f)
  (test (assq 3/4 even) #f)
  (test (assq 3.0 odd) #f)
  (test (assq 3.0 even) #f)
  (test (assq #(1) odd) #f)
  (test (assq #(1) even) #f))



;;; --------------------------------------------------------------------------------
;;; assv

(test (assv 1 '(1 2 . 3)) #f)
(test (assv 1 '((1 2) . 3)) '(1 2))

(let ((e '((a 1) (b 2) (c 3))))
  (test (assv 'a e) '(a 1))
  (test (assv 'b e) '(b 2))
  (test (assv 'd e) #f))
(test (assv (list 'a) '(((a)) ((b)) ((c))))  #f)

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assv #t e) (list #t 1))
    (test (assv #f e) (list #f 2))
    (test (assv 'a e) (list 'a 3))
    (test (assv xcons e) (list xcons 4))
    (test (assv xvect e) (list xvect 5))
    (test (assv xlambda e) (list xlambda 6))
    (test (assv xstr e) (list xstr 7))
    (test (assv car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assv 1+i e) '(1+i 1))
  (test (assv 3.0 e) '(3.0 2))
  (test (assv 5/3 e) '(5/3 3))
  (test (assv #\a e) '(#\a 4))
  (test (assv "hiho" e) #f))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assv '(a) e) #f)
  (test (assv #(a) e) #f)
  (test (assv (string #\c) e) #f))

(let ((lst '((2 . a) (3 . b))))
  (set-cdr! (assv 3 lst) 'c)
  (test lst '((2 . a) (3 . c))))

(test (assv () '((() 1) (#f 2))) '(() 1))
(test (assv () '((1) (#f 2))) #f)
(test (assv #() '((#f 1) (() 2) (#() 3))) #f)  ; (eqv? #() #()) -> #f ??

(test (assv 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assv 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assv 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assv 'asdf '((a . 1) (b . 2) () (c . 3) . 4)) #f)
(test (assv 'd '((a . 1) (b . 2) () (c . 3) (d . 5))) '(d . 5))
(test (assv 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assv 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assv 'a odd) '(a . 2))
  (test (assv 'a even) '(a . 3))
  (test (assv 3 odd) '(3 . 1))
  (test (assv 3 even) '(3 . 2))
  (test (assv 3/4 odd) '(3/4 . 5))
  (test (assv 3/4 even) '(3/4 . 6))
  (test (assv 3.0 odd) '(3.0 . 3))
  (test (assv 3.0 even) '(3.0 . 4))
  (test (assv #(1) odd) #f)
  (test (assv #(1) even) #f))

(test (assv 1/0 '((1/0 . 1) (1.0 . 3))) #f)
(test (pair? (assv (real-part (log 0)) (list (cons 1/0 1) (cons (real-part (log 0)) 2) (cons -1 3)))) #t)
(test (pair? (assv (- (real-part (log 0))) (list (cons 1/0 1) (cons (real-part (log 0)) 2) (cons -1 3)))) #f)



;;; --------------------------------------------------------------------------------
;;; assoc

(let ((e '((a 1) (b 2) (c 3))))
  (test (assoc 'a e) '(a 1))
  (test (assoc 'b e) '(b 2))
  (test (assoc 'd e) #f))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))

(let ((xcons (cons 1 2))
      (xvect (vector 1 2))
      (xlambda (lambda () 1))
      (xstr "abs"))
  (let ((e (list (list #t 1) (list #f 2) (list 'a 3) (list xcons 4) (list xvect 5) (list xlambda 6) (list xstr 7) (list car 8))))
    (test (assoc #t e) (list #t 1))
    (test (assoc #f e) (list #f 2))
    (test (assoc 'a e) (list 'a 3))
    (test (assoc xcons e) (list xcons 4))
    (test (assoc xvect e) (list xvect 5))
    (test (assoc xlambda e) (list xlambda 6))
    (test (assoc xstr e) (list xstr 7))
    (test (assoc car e) (list car 8))))

(let ((e '((1+i 1) (3.0 2) (5/3 3) (#\a 4) ("hiho" 5))))
  (test (assoc 1+i e) '(1+i 1))
  (test (assoc 3.0 e) '(3.0 2))
  (test (assoc 5/3 e) '(5/3 3))
  (test (assoc #\a e) '(#\a 4))
  (test (assoc "hiho" e) '("hiho" 5)))

(let ((e '(((a) 1) (#(a) 2) ("c" 3))))
  (test (assoc '(a) e) '((a) 1))
  (test (assoc #(a) e) '(#(a) 2))
  (test (assoc (string #\c) e) '("c" 3)))

(test (assoc 'a '((b c) (a u) (a i))) '(a u))
(test (assoc 'a '((b c) ((a) u) (a i))) '(a i))
(test (assoc (list 'a) '(((a)) ((b)) ((c))))  '((a)))
(test (assoc 5 '((2 3) (5 7) (11 13))) '(5 7))
(test (assoc 'key ()) #f)
(test (assoc 'key '(() ())) 'error)
(test (assoc () ()) #f)
(test (assoc 1 '((1 (2)) (((3) 4)))) '(1 (2)))
(test (assoc #f () 1/9223372036854775807) 'error)

(test (assoc () 1) 'error)
(test (assoc (cons 1 2) 1) 'error)
(test (assoc (let ((x (cons 1 2))) (set-cdr! x x)) 1) 'error)
(test (assoc '((1 2) .3) 1) 'error)
(test (assoc ''foo quote) 'error)
(test (assoc 3 '((a . 3)) abs =) 'error)
(test (assoc 1 '(1 2 . 3)) 'error)
(test (assoc 1 '((1 2) . 3)) '(1 2))
(test (assoc 1 '((1) (1 3) (1 . 2))) '(1))
(test (assoc 1 '((1 2 . 3) (1 . 2))) '(1 2 . 3))
(test (assoc '(((1 2))) '((1 2) ((1 2) 3) (((1 2) 3) 4) ((((1 2) 3) 4) 5))) #f)
(test (assoc '(((1 2))) '((1 2) ((1 2)) (((1 2))) ((((1 2)))))) '((((1 2)))))
(test (assoc 'a '((a . 1) (a . 2) (a . 3))) '(a . 1)) ; is this specified?
(test (assoc 'a '((b . 1) (a . 2) (a . 3))) '(a . 2))

(test (assoc () '((() 1) (#f 2))) '(() 1))
(test (assoc () '((1) (#f 2))) #f)
(test (assoc #() '((#f 1) (() 2) (#() 3))) '(#() 3))
(test (assoc #<unspecified> (list (cons (apply values ()) #f))) '(#<unspecified> . #f))

(for-each
 (lambda (arg)
   (test (assoc arg (list (list 1 2) (list arg 3))) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol #() abs 3/4 #\f #t (if #f #f)))

(test (assoc 'b '((a . 1) (b . 2) () (c . 3) #f)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) #f)) '(c . 3))
(test (assoc 'b '((a . 1) (b . 2) () (c . 3) . 4)) '(b . 2))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) . 4)) '(c . 3))
(test (assoc 'c '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) '(c . 3))
(test (assoc 'asdf '((a . 1) (b . 2) () (c . 3) (c . 4) . 4)) #f)
(test (assoc "" (list '("a" . 1) '("" . 2) '(#() . 3))) '("" . 2))
(test (assoc assoc (list (cons abs 1) (cons assoc 2) (cons + 3))) (cons assoc 2))

(let ((odd '((3 . 1) (a . 2) (3.0 . 3) (b . 4) (3/4 . 5) (c . 6) (#(1) . 7) (d . 8)))
      (even '((e . 1) (3 . 2) (a . 3) (3.0 . 4) (b . 5) (3/4 . 6) (c . 7) (#(1) . 8) (d . 9))))
  (test (assoc 'a odd) '(a . 2))
  (test (assoc 'a even) '(a . 3))
  (test (assoc 3 odd) '(3 . 1))
  (test (assoc 3 even) '(3 . 2))
  (test (assoc 3/4 odd) '(3/4 . 5))
  (test (assoc 3/4 even) '(3/4 . 6))
  (test (assoc 3.0 odd =) '(3 . 1)) 
  (test (assoc 3.0 odd) '(3.0 . 3)) 
  (test (assoc 3.0 even) '(3.0 . 4))
  (test (assoc #(1) odd) '(#(1) . 7))
  (test (assoc #(1) even) '(#(1) . 8)))

(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (31 . c) (4 . d)) =) #f)
(test (assoc 3 () =) #f)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) =) '(3 . c))
(test (assoc #\a '((#\A . 1) (#\b . 2)) char=?) #f)
(test (assoc #\a '((#\A . 1) (#\b . 2)) char-ci=?) '(#\A . 1))
(test (assoc #\a '((#\A . 1) (#\b . 2)) (lambda (a b) (char-ci=? a b))) '(#\A . 1))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) #(1)) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) abs) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) quasiquote) 'error)
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b c) (= a b))) 'error)
(test (assoc 3.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda* (a b c) (= a b))) '(3 . c))
(test (assoc 3 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a) (= a 1))) 'error)
(test (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (dilambda = =)) '(4 . d))
(test (catch #t (lambda () (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (error 'assoc a)))) (lambda args (car args))) 'assoc)
(test (call-with-exit (lambda (go) (assoc 4.0 '((1 . a) (2 . b) (3 . c) (4 . d)) (lambda (a b) (go 'assoc))))) 'assoc)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0))) #f)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (= a b))) 'error)
(test (assoc 3 '((#\a . 3) (#() . 2) (3.0 . 1) ("3" . 0)) (lambda (a b) (and (number? b) (= a b)))) '(3.0 . 1)) ; is this order specified?
(test (let ((lst (list (cons 1 2) (cons 3 4) (cons 5 6)))) (set! (cdr (cdr lst)) lst) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst)) #f)
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 3 lst =)) '(3 . 4))
(test (let ((lst '((1 . 2) (3 . 4) . 5))) (assoc 5 lst =)) #f)
(test (assoc 3 '((1 . 2) . 3)) #f)
(test (assoc 1 '((1 . 2) . 3)) '(1 . 2))
(test (assoc 3 '((1 . 2) . 3) =) #f)
(test (assoc 1 '((1 . 2) . 3) =) '(1 . 2))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (and (assoc 2 lst =) lst)) '((1 . 2) (2 . 3) (3 . 4)))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 2 lst =)) '(2 . 3))
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr lst)) lst) (assoc 4 lst =)) #f)
(test (let ((lst (list (cons 1 2) (cons 2 3) (cons 3 4)))) (set! (cdr (cdr (cdr lst))) lst) (assoc 3 lst =)) '(3 . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . 4))) '((1 2) . 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 4)))) '((1 2) 3 4))
(test (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4)))) '((1 2) 3 . 4))
(test (cdr (assoc '(1 2) '((a . 3) ((1 2) . (3 . 4))))) (cons 3 4))

(test (assoc #t (list 1 2) #()) 'error)
(test (assoc #t (list 1 2) (integer->char 127)) 'error)
(test (assoc #t (list 1 2) (lambda (x y) (+ x 1))) 'error) ; (+ #t 1)
(test (assoc #t (list 1 2) abs) 'error)
(test (assoc #t (list 1 2) (lambda args args)) 'error)
(test (assoc 1 '((3 . 2) 3) =) 'error)
(test (assoc 1 '((1 . 2) 3) =) '(1 . 2)) ; this is like other trailing error unchecked cases -- should we check?

(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b c) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b c . d) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a b . c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda a (apply eq? a))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda (a . b) (eq? a (car b)))) '(a 1))

(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a) (eq? a b))) 'error)
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b c . d) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a b . c) (eq? a b))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* a (apply eq? a))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a . b) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a :rest b) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (a :rest b :rest c) (eq? a (car b)))) '(a 1))
(test (assoc 'a '((c 3) (a 1) (b 2)) (lambda* (:rest a) (apply eq? a))) '(a 1))
(test (assoc 'a '((a 1) (b 2) (c 3)) (define-macro (_m_ a b) `(eq? ',a ',b))) '(a 1))
(test (assoc 'c '((a 1) (b 2) (c 3)) (define-macro (_m_ a b) `(eq? ',a ',b))) '(c 3))

(let ()
  (define (atest a b)
    (eq? a b))
  (atest 1 1)
  (let ((lst (list (cons 'a 1) (cons 'b 2))))
    (test (assoc 'b lst atest) '(b . 2))))

(for-each
  (lambda (arg lst)
    (test (assoc arg lst eq?) (assq arg lst))
    (test (assoc arg lst eqv?) (assv arg lst))
    (test (assoc arg lst equal?) (assoc arg lst)))
  (list 'a #f (list 'a) 'a 1 3/4 #(1) "hi")
  (list '((a . 1) (b . 2) (c . 3)) '((1 . 1) ("hi" . 2) (#t . 4) (#f . 5) (2 . 3)) 
	'((b . 1) ((a) . 2) (c . 3)) '((d . 1) (a . 2) (b . 4) . (c . 3)) 
	'((1 . 1) (3/4 . 2) (23 . 3)) '((a . 1) (1 . 2) (#(1) . 4) (23 . 3)) 
	'((1 . 1) ("hi" . 2) (23 . 3))))



;;; --------------------------------------------------------------------------------
;;; memq

(test (memq 'a '(a b c)) '(a b c))
(test (memq 'a (list 'a 'b 'c)) '(a b c))
(test (memq 'b '(a b c)) '(b c))
(test (memq 'a '(b c d)) #f)
(test (memq (list 'a) '(b (a) c))  #f)
(test (memq 'a '(b a c a d a)) '(a c a d a))
(let ((v (vector 'a))) (test (memq v (list 'a 1.2 v "hi")) (list v "hi")))
(test (memq #f '(1 a #t "hi" #f 2)) '(#f 2))
(test (memq eq? (list 2 eqv? 1 eq?)) (list eq?))
(test (memq eq? (list 2 eqv? 2)) #f)
(test (memq 6 (memq 5 (memq 4 (memq 3 (memq 2 (memq 1 '(1 2 3 4 5 6))))))) '(6))
(test (memq 1/2 (list (/ 2.0) .5 1/2)) #f)
(test (memq 'a (cons 'a 'b)) '(a . b))
(test (memq) 'error)
(test (memq 'a) 'error)
(test (memq 'a 'b) 'error)
(test (memq 'a '(a b . c)) '(a b . c))
(test (memq 'b '(a b . c)) '(b . c))
(test (memq 'c '(a b . c)) #f) ; or should it be 'c?
(test (memq () '(1 () 3)) '(() 3))
(test (memq () '(1 2)) #f)
(test (memq 'a '(c d a b c)) '(a b c))
(test (memq 'a '(c d f b c)) #f)
(test (memq 'a ()) #f)
(test (memq 'a '(c d a b . c)) '(a b . c))
(test (memq 'a '(c d f b . c)) #f)
(test (memq #f '(1 "hi" #t)) #f)
(test (memq () ()) #f)
(test (memq () (list)) #f)
(test (memq () (list ())) '(()))
(test (let ((x (cons 1 2))) (memq x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (pair? (let ((x (lambda () 1))) (memq x (list 1 2 x 3)))) #t)
(test (memq memq (list abs + memq car)) (list memq car))
(test (memq 'a '(a a a)) '(a a a)) ;?
(test (memq 'a '(b a a)) '(a a))
(test (memq "hi" '(1 "hi" 2)) #f)
(test (let ((str "hi")) (memq str (list 1 str 2))) '("hi" 2))
(test (memq #\a '(1 #f #\a 2)) '(#\a 2))

(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (memq x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (memq x lst)) #f)

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memq 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memq 3/4 odd) #f)
  (test (memq 3/4 even) #f)
  (test (memq 3.0 odd) #f)
  (test (memq 3.0 even) #f)
  (test (memq #(1) odd) #f)
  (test (memq #(1) even) #f))

;;; but (memq pi (list 1 pi 2)) -> '(3.1415926535898 2)

(test (memq (values #\a '(#\A 97 a))) #f)
(test (memq (values #\a '(#\A 97 #\a))) '(#\a))
(test (memq #\a (values #\a '(#\A 97 #\a))) 'error)
(test (memq #\a (values '(#\A 97 #\a))) '(#\a))
(test (memq #\a '(1 2) (values)) 'error) ; hmmm
(test ((values memq (values #\a '(#\A 97 #\a)))) '(#\a))




;;; --------------------------------------------------------------------------------
;;; memv

(test (memv 101 '(100 101 102)) '(101 102))
(test (memv 101 (list 100 101 102)) '(101 102))
(test (memv 3.4 '(1.2 2.3 3.4 4.5)) '(3.4 4.5))
(test (memv 3.4 '(1.3 2.5 3.7 4.9)) #f)
(test (memv 1/2 (list (/ 2.0) .5 1/2)) '(1/2))
(test (memv 1.0 '(1 2 3)) #f)
(test (memv 1/0 '(1/0 1.0 3)) #f)
(test (pair? (memv (real-part (log 0)) (list 1/0 (real-part (log 0)) -1))) #t)
(test (pair? (memv (- (real-part (log 0))) (list 1/0 (real-part (log 0)) -1))) #f)

(let ((ls (list 'a 'b 'c)))
  (set-car! (memv 'b ls) 'z)
  (test ls '(a z c)))
(test (memv 1 (cons 1 2)) '(1 . 2))
(test (memv 'a (list 'a 'b . 'c)) 'error)
(test (memv 'a '(a b . c)) '(a b . c))
(test (memv 'asdf '(a b . c)) #f)
(test (memv) 'error)
(test (memv 'a) 'error)
(test (memv 'a 'b) 'error)
(test (memv 'c '(a b c)) '(c))
(test (memv 'c '(a b . c)) #f)
(test (memv ''a '('a b c)) #f)
(test (let ((x (cons 1 2))) (memv x (list (cons 1 2) (cons 3 4)))) #f)
(test (let ((x (cons 1 2))) (memv x (list x (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (memv 'a '(a a a)) '(a a a)) ;?
(test (memv 'a '(b a a)) '(a a))
(test (memv "hi" '(1 "hi" 2)) #f)
(test (memv #\a '(1 #f #\a 2)) '(#\a 2))
(test (memv cons (list car cdr cons +)) (list cons +))
(test (memv (apply values ()) (list #<unspecified>)) (list #<unspecified>))

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (memv 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (memv 3/4 odd) '(3/4 c #(1) d))
  (test (memv 3/4 even) '(3/4 c #(1) d))
  (test (memv 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (memv 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (memv #(1) odd) #f)
  (test (memv #(1) even) #f))
(test (memv #(1) '(1 #(1) 2)) #f)
(test (memv () '(1 () 2)) '(() 2))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (memv x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (memv x lst)) #f)



;;; --------------------------------------------------------------------------------
;;; member

(test (member (list 'a) '(b (a) c)) '((a) c))
(test (member "b" '("a" "c" "b")) '("b"))
(test (member 1 '(3 2 1 4)) '(1 4))
(test (member 1 (list 3 2 1 4)) '(1 4))
(test (member car (list abs car modulo)) (list car modulo))
(test (member do (list quote map do)) (list do))
(test (member 5/2 (list 1/3 2/4 5/2)) '(5/2))
(test (member 'a '(a b c d)) '(a b c d))
(test (member 'b '(a b c d)) '(b c d))
(test (member 'c '(a b c d)) '(c d))
(test (member 'd '(a b c d)) '(d))
(test (member 'e '(a b c d)) #f)
(test (member 1 (cons 1 2)) '(1 . 2))
(test (member 1 '(1 2 . 3)) '(1 2 . 3))
(test (member 2 '(1 2 . 3)) '(2 . 3))
(test (member 3 '(1 2 . 3)) #f)
(test (member 4 '(1 2 . 3)) #f)
(test (member 1/2 (list (/ 2.0) .5 1/2)) '(1/2))
(test (member) 'error)
(test (member 'a) 'error)
(test (member 'a 'b) 'error)
(test (member () '(1 2 3)) #f)
(test (member () '(1 2 ())) '(()))
(test (member #() '(1 () 2 #() 3)) '(#() 3))
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (1 2)))) #f)
(test (member #2d((1 2) (3 4)) '(1 #() #2d((1 2) (3 4)))) '(#2d((1 2) (3 4))))
(test (let ((x (cons 1 2))) (member x (list (cons 1 2) (cons 3 4)))) '((1 . 2) (3 . 4)))
(test (let ((x (list 1 2))) (member x (list (cons 1 2) (list 1 2)))) '((1 2)))
(test (member ''a '('a b c)) '('a b c))
(test (member 'a '(a a a)) '(a a a)) ;?
(test (member 'a '(b a a)) '(a a))
(test (member (member 3 '(1 2 3 4)) '((1 2) (2 3) (3 4) (4 5))) '((3 4) (4 5)))
(test (member "hi" '(1 "hi" 2)) '("hi" 2))
(test (member #\a '(1 #f #\a 2)) '(#\a 2))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" x (vector 1 2)))) (member x lst)) '(#(1 2 3) #(1 2)))
(test (let* ((x (vector 1 2 3)) (lst (list 1 "hi" (vector 1 2 3)))) (member x lst)) '(#(1 2 3)))

(for-each
 (lambda (arg)
   (test (member arg (list 1 2 arg 3)) (list arg 3)))
 (list "hi" (integer->char 65) #f 'a-symbol abs 3/4 #\f #t (if #f #f) '(1 2 (3 (4))) most-positive-fixnum))

(test (member 3 . (1 '(2 3))) 'error)
(test (member 3 '(1 2 3) = =) 'error)
(test (member 3 . ('(1 2 3))) '(3))
(test (member 3 . ('(1 2 3 . 4))) '(3 . 4))
(test (member . (3 '(1 2 3))) '(3))
(test (member '(1 2) '(1 2)) #f)
(test (member '(1 2) '((1 2))) '((1 2)))
(test (member . '(quote . ((quote)))) #f)
(test (member . '(quote . ((quote) .()))) #f)
(test (member '(((1))) '((((1).()).()).())) '((((1)))))
(test (member '((1)) '(1 (1) ((1)) (((1))))) '(((1)) (((1)))))
(test (member member (list abs car memq member +)) (list member +))
(test (member () () "") 'error)

(let ((odd '(3 a 3.0 b 3/4 c #(1) d))
      (even '(e 3 a 3.0 b 3/4 c #(1) d)))
  (test (member 'a odd) '(a 3.0 b 3/4 c #(1) d))
  (test (member 'a even) '(a 3.0 b 3/4 c #(1) d))
  (test (member 3/4 odd) '(3/4 c #(1) d))
  (test (member 3/4 even) '(3/4 c #(1) d))
  (test (member 3.0 odd) '(3.0 b 3/4 c #(1) d))
  (test (member 3.0 even) '(3.0 b 3/4 c #(1) d))
  (test (member #(1) odd) '(#(1) d))
  (test (member #(1) even) '(#(1) d)))

(test (member 3 '(1 2 3 4) =) '(3 4))
(test (member 3 () =) #f)
(test (member 3 '(1 2 4 5) =) #f)
(test (member 4.0 '(1 2 4 5) =) '(4 5))
(test (member #\a '(#\b #\A #\c) char=?) #f)
(test (member #\a '(#\b #\A #\c) char-ci=?) '(#\A #\c))
(test (member #\a '(#\b #\A #\c) (lambda (a b) (char-ci=? a b))) '(#\A #\c))
(test (char=? (car (member #\a '(#\b #\a))) #\a) #t)
(test (char=? (car (member #\a '(#\b #\a) (lambda (a b) (char=? a b)))) #\a) #t)
(test (member 3 '(1 2 3 4) <) '(4))
(test (member 3 '((1 2) (3 4)) member) '((3 4)))
(test (member 3 '(((1 . 2) (4 . 5)) ((3 . 4))) assoc) '(((3 . 4))))
(test (member '(#f #f #t) '(0 1 2) list-ref) '(2))
(test (let ((v (vector 1 2 3))) (member v (list 0 v) vector-fill!)) '(0 #(0 0 0)))

(test (member 3 '(1 2 3) abs) 'error)
(test (member 3 '(1 2 3) quasiquote) 'error)
(test (member 3 '(1 2 3) (lambda (a b c) (= a b))) 'error)
(test (member 3 '(1 2 3) (lambda* (a b c) (= a b))) '(3))
(test (member 3 '(1 2 3 4) (dilambda = =)) '(3 4))
(test (catch #t (lambda () (member 3 '(1 2 3) (lambda (a b) (error 'member a)))) (lambda args (car args))) 'member)
(test (call-with-exit (lambda (go) (member 3 '(1 2 3) (lambda (a b) (go 'member))))) 'member)
(test (member 'a '(a a a) eq?) '(a a a))
(test (member 'a '(b a a) eqv?) '(a a))
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i)) #f)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (= (real-part a) b))) 'error)
(test (member 3.0 '(1 #\a (3 . 3) abs #() 3+i) (lambda (a b) (and (number? b) (= (real-part b) a)))) '(3+i))
;; is it guaranteed that in the comparison function the value is first and the list member 2nd?
(test (member 4 '((1 2 3) (4 5 6) (7 8 9)) member) '((4 5 6) (7 8 9)))
(test (member 4 '(1 2 3) member) 'error)
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (member a (map (lambda (c) (+ c 1)) b)))) '((3 5) 7))
(test (member 4 '((1 2) (3 5) 7) (lambda (a b) (assoc a (map (lambda (c) (cons (+ c 1) c)) b)))) '((3 5) 7))
(test (let ((f #f)) (member 'a '(a b c) (lambda (a b) (if (eq? b 'a) (set! f (lambda () b))) (eq? a 123))) (f)) 'a)
(test (let ((i 0) (f (make-vector 3))) (member 'a '(a b c) (lambda (a b) (vector-set! f i b) (set! i (+ i 1)) (eq? a 123))) f) #(a b c))
(test (member 1 '(0 1 2) (lambda (a b . c) (= a b))) '(1 2))
(test (member 1 '(0 1 2) (lambda* (a b c) (= a b))) '(1 2))
(test (member 1 '(0 1 2) (lambda (a) (= a b))) 'error)
(test (member 1 '(0 1 2) (lambda a (= (car a) (cadr a)))) '(1 2))

(test (member 'a '(c 3 a 1 b 2) (lambda (a) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda (a b c) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b c . d) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda (a b . c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda a (apply eq? a))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda (a . b) (eq? a (car b)))) '(a 1 b 2))

(test (member 'a '(c 3 a 1 b 2) (lambda* (a) (eq? a b))) 'error)
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b c . d) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a b . c) (eq? a b))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* a (apply eq? a))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a . b) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a :rest b) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (a :rest b :rest c) (eq? a (car b)))) '(a 1 b 2))
(test (member 'a '(c 3 a 1 b 2) (lambda* (:rest a) (apply eq? a))) '(a 1 b 2))
(test (member 'a '(a b c) (define-macro (_m_ a b) `(eq? ',a ',b))) '(a b c))
(test (member 'c '(a b c) (define-macro (_m_ a b) `(eq? ',a ',b))) '(c))

(test (member 4 '(1 2 3 4 . 5)) '(4 . 5))
(test (member 4 '(1 2 3 4 . 5) =) '(4 . 5))
(test (member 4 '(1 2 3 . 4)) #f)
(test (member 4 '(1 2 3 . 4) =) #f)
(test (let ((lst (list 1 2 3))) (and (member 2 lst =) lst)) '(1 2 3))
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst))) #t)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 2 lst =))) #t)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst)) #f)
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr lst)) lst) (member 4 lst =)) #f)
(test (pair? (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (member 3 lst =))) #t)
(test (pair? (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 3 lst =))) #t)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (member 5 lst =)) #f)
(test (let ((lst (list 1 2 3 4))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (member 4 lst =)) #f)
(test (let ((lst '(1 2 3 5 6 9 10))) (member 3 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(9 10))
(test (let ((lst '(1 2 3 5 6 9 10))) (member 2 lst (let ((last (car lst))) (lambda (a b) (let ((result (= (- b last) a))) (set! last b) result))))) '(5 6 9 10))
(test (member 1 () =) #f)
(test (member 1 #(1) =) 'error)
(test (member 3 '(5 4 3 2 1) >) '(2 1))
(test (member 3 '(5 4 3 2 1) >=) '(3 2 1))
(test (member '(1 2) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2) 1 . 2))
(test (member '(1 2 . 3) '((1) (1 . 2) (1 2 . 3) (1 2 3) (1 2) 1 . 2)) '((1 2 . 3) (1 2 3) (1 2) 1 . 2))

(let ()
  (define (sfind obj lst)
    (member obj lst (lambda (a b)
		      (catch #t
			(lambda ()
			  (and (equal? a b)
			       (member obj lst (lambda (a b)
						 (catch #t
						   (lambda ()
						     (error 'oops))
						   (lambda args
						     (equal? a b)))))))
			(lambda args
			  'oops)))))
  (test (sfind 'a '(b c a d)) '(a d)))

(let ()
  (define-macro (do-list lst . body) 
    `(member #t ,(cadr lst) (lambda (a b) 
			      (let ((,(car lst) b)) 
				,@body 
				#f))))
  (let ((sum 0))
    (do-list (x '(1 2 3)) (set! sum (+ sum x)))
    (test (= sum 6) #t)))

(let ()
  (define (tree-member a lst) 
    (member a lst (lambda (c d) 
		    (if (pair? d) 
			(tree-member c d) 
			(equal? c d)))))
  (test (tree-member 1 '(2 3 (4 1) 5)) '((4 1) 5))
  (test (tree-member -1 '(2 3 (4 1) 5)) #f)
  (test (tree-member 1 '(2 3 ((4 (1) 5)))) '(((4 (1) 5)))))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 2 lst) (member 2 lst equal?)))

(let ((lst (list 1 2 3)))
  (set! (cdr (cdr (cdr lst))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) lst)
  (test (member 4 lst) (member 4 lst equal?)))

(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdr (cdr (cdr lst)))) (cdr lst))
  (test (member 4 lst) (member 4 lst equal?)))

(for-each
  (lambda (arg lst)
    (test (member arg lst eq?) (memq arg lst))
    (test (member arg lst eqv?) (memv arg lst))
    (test (member arg lst equal?) (member arg lst)))
  (list 'a #f (list 'a) 'a 1 3/4 #(1) "hi")
  (list '(a b c) '(1 "hi" #t #f 2) '(b (a) c) '(d a b . c) '(1 3/4 23) '(1 3/4 23) '(a 1 #(1) 23) '(1 "hi" 23)))

(for-each
 (lambda (op)
   (test (op) 'error)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format-logged #t ";(~A ~A) returned ~A?~%" op arg result))
	(test (op arg () arg) 'error)
	(test (op arg) 'error)))
    (list () "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list cons car cdr set-car! set-cdr! caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       assq assv memq memv list-ref list-tail))

(for-each
 (lambda (op)
   (test (op '(1) '(2)) 'error))
 (list reverse car cdr caar cadr cdar cddr caaar caadr cadar cdaar caddr cdddr cdadr cddar 
       caaaar caaadr caadar cadaar caaddr cadddr cadadr caddar cdaaar cdaadr cdadar cddaar cdaddr cddddr cddadr cdddar
       list-ref list-tail list-set!))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (let ((result (catch #t (lambda () (op #f arg)) (lambda args 'error))))
	(if (not (eq? result 'error))
	    (format-logged #t ";(~A #f ~A) returned ~A?~%" op arg result))))
    (list "hi" (integer->char 65) #f 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list assq assv assoc memq memv member))




;;; --------------------------------------------------------------------------------
;;; append

(test (append '(a b c) ()) '(a b c))
(test (append () '(a b c)) '(a b c))
(test (append '(a b) '(c d)) '(a b c d))
(test (append '(a b) 'c) '(a b . c))
(test (equal? (append (list 'a 'b 'c) (list 'd 'e 'f) () '(g)) '(a b c d e f g)) #t)
(test (append (list 'a 'b 'c) (list 'd 'e 'f) () (list 'g)) '(a b c d e f g))
(test (append (list 'a 'b 'c) 'd) '(a b c . d))
(test (append () ()) ())
(test (append () (list 'a 'b 'c)) '(a b c))
(test (append) ())
(test (append () 1) 1)
(test (append 'a) 'a)
(test (append '(x) '(y))  '(x y))
(test (append '(a) '(b c d)) '(a b c d))
(test (append '(a (b)) '((c)))  '(a (b) (c)))
(test (append '(a b) '(c . d))  '(a b c . d))
(test (append () 'a)  'a)
(test (append '(a b) (append (append '(c)) '(e) 'f)) '(a b c e . f))
(test (append ''foo 'foo) '(quote foo . foo))
(test (append () (cons 1 2)) '(1 . 2))
(test (append () () ()) ())
(test (append (cons 1 2)) '(1 . 2))
(test (append (list 1) 2) '(1 . 2))

(test (append #f) #f)
(test (append () #f) #f)
(test (append '(1 2) #f) '(1 2 . #f))
(test (append () () #f) #f)
(test (append () '(1 2) #f) '(1 2 . #f))
(test (append '(1 2) () #f) '(1 2 . #f))
(test (append '(1 2) '(3 4) #f) '(1 2 3 4 . #f))
(test (append () () () #f) #f)
(test (append '(1 2) '(3 4) '(5 6) #f) '(1 2 3 4 5 6 . #f))
(test (append () () #()) #())
(test (append () ((lambda () #f))) #f)

(test (append (begin) do) do)
(test (append if) if)
(test (append quote) quote)
(test (append 0) 0) ; is this correct?
(test (append () 0) 0)
(test (append () () 0) 0)
(test (let* ((x '(1 2 3)) (y (append x ()))) (eq? x y)) #f) ; check that append returns a new list
(test (let* ((x '(1 2 3)) (y (append x ()))) (equal? x y)) #t)
(test (let* ((x (list 1 2 3)) (y (append x (list)))) (eq? x y)) #f) 
(test (append '(1) 2) '(1 . 2))
(let ((x (list 1 2 3)))
  (let ((y (append x ())))
    (set-car! x 0)
    (test (= (car y) 1) #t)))
(let ((x (list 1 2 3)))
  (let ((y (append x ())))
    (set-cdr! x 0)
    (test (and (= (car y) 1)
	       (= (cadr y) 2)
	       (= (caddr y) 3))
	  #t)))

(test (let ((xx (list 1 2))) (recompose 12 (lambda (x) (append (list (car x)) (cdr x))) xx)) '(1 2))

(test (append 'a 'b) 'error)
(test (append 'a ()) 'error)
(test (append (cons 1 2) ()) 'error)
(test (append '(1) 2 '(3)) 'error)
(test (append '(1) 2 3) 'error)
(test (let ((lst (list 1 2 3))) (append lst lst)) '(1 2 3 1 2 3))
(test (append ''1 ''1) '(quote 1 quote 1))
(test (append '(1 2 . 3) '(4)) 'error)
(test (append '(1 2 . 3)) '(1 2 . 3))
(test (append '(4) '(1 2 . 3)) '(4 1 2 . 3))
(test (append () 12 . ()) 12)
(test (append '(1) 12) '(1 . 12))
(test (append '(1) 12 . ()) '(1 . 12))
(test (append () () '(1) 12) '(1 . 12))
(test (append '(1) '(2) '(3) 12) '(1 2 3 . 12))
(test (append '(((1))) '(((2)))) '(((1)) ((2))))
(test (append () . (2)) 2)
(test (append . (2)) 2)
(test (append ''() ()) ''())
(test (let ((i 1)) (logior 123 (append i))) 123) ; !

(for-each
 (lambda (arg)
   (test (append arg) arg)
   (test (append () arg) arg)
   (test (append () () () arg) arg))
 (list "hi" #\a #f 'a-symbol _ht_ _null_ (make-vector 3) abs 1 3.14 3/4 1.0+1.0i #t #<unspecified> #<eof> () #() (list 1 2) (cons 1 2) #(0) (lambda (a) (+ a 1))))
(test (append not) not)

(test (let ((l0 (list 0))
	    (l1 (list 0)))
	(let ((m0 (append '(2) l0))
	      (m1 (append '(2) l1 '())))
	  (and (equal? l0 l1)
	       (equal? m0 m1)
	       (let ()
		 (list-set! m0 1 3)
		 (list-set! m1 1 3)
		 (list l0 l1)))))
      '((3) (0)))

;;; generic append
(test (append "asdasd" '("asd")) 'error)
(test (append "asdasd" #("asd")) 'error)
(test (append (string->byte-vector "asdasd") '("asd")) 'error)
(test (append (string->byte-vector "asdasd") #("asd")) 'error)

(test (let ((h1 (hash-table* 'a 1 'b 2)) (h2 (hash-table* 'c 3))) (append h1 h2)) (hash-table '(c . 3) '(a . 1) '(b . 2)))
(test (let ((i1 (inlet 'a 1)) (i2 (inlet 'b 2 'c 3))) (append i1 i2)) (inlet 'a 1 'c 3 'b 2))
(test (let ((s1 "abc") (s2 "def")) (append s1 s2)) "abcdef")
(test (let ((v1 #(0 1)) (v2 #(2 3))) (append v1 v2)) #(0 1 2 3))
(test (let ((p1 '(1 2)) (p2 '(3 4))) (append p1 p2)) '(1 2 3 4))
(test (vector? (append #())) #t)
(test (float-vector? (append (float-vector))) #t)
(test (int-vector? (append (int-vector))) #t)
(test (append "12" '(1 . 2) "3") 'error)
(for-each
 (lambda (arg)
   (test (append arg) arg)
   (test (append () arg) arg))
 (list "" #u8() () #() (int-vector) (float-vector) (inlet) (hash-table)
       "123" #u8(101 102) 
       '(1 2 3) '((e . 5) (f . 6)) 
       #(1 2 3) #((g . 8) (h . 9)) (int-vector 1 2 3) (float-vector 1 2 3)
       (inlet 'a 1 'b 2)
       (hash-table* 'c 3 'd 4)))
(test (append #u8() (int-vector 1 2 3)) #u8(1 2 3))
(test (append #u8() "123") #u8(49 50 51))
(test (append "" "123") "123")
(test (append #() (hash-table)) #())
(test (append #() #u8(101 102)) #(101 102))
(test (append (float-vector) #u8(101 102)) (float-vector 101.0 102.0))
(test (append (int-vector) #u8(101 102)) (int-vector 101 102))
(test (append (hash-table) '((e . 5) (f . 6))) (hash-table '(e . 5) '(f . 6)))
(test (append (inlet) #((g . 8) (h . 9))) (inlet 'g 8 'h 9))
(test (append '(1 2 3) #u8()) '(1 2 3 . #u8()))
(test (append '(1 2 3) #u8() #(4 5)) '(1 2 3 4 5))
(test (append '(1 2 3) #u8(101 102)) '(1 2 3 . #u8(101 102)))
(test (append '(1 2 3) #() (inlet 'a 1 'b 2)) '(1 2 3 (b . 2) (a . 1)))
(test (let ((lst (append '((e . 5) (f . 6)) "" (hash-table '(c . 3) '(d . 4)))))
	(or (equal? lst '((e . 5) (f . 6) (c . 3) (d . 4)))
	    (equal? lst '((e . 5) (f . 6) (d . 4) (c . 3))))) #t)
(test (append (list 1) "hi") '(1 . "hi"))
(test (append #(1 2 3) "123") #(1 2 3 #\1 #\2 #\3))
(test (append (int-vector 1 2 3) #(1 2 3)) (int-vector 1 2 3 1 2 3))
(test (append (int-vector 1 2 3) "123") (int-vector 1 2 3 49 50 51))
(test (append (float-vector 1.0 2.0 3.0) (int-vector 1 2 3)) (float-vector 1.0 2.0 3.0 1.0 2.0 3.0))
(test (append (int-vector 1 2 3) (float-vector 1.0 2.0 3.0)) (int-vector 1 2 3 1 2 3))
(test (append (inlet 'a 1 'b 2) '((e . 5) (f . 6))) (inlet 'b 2 'a 1 'e 5 'f 6))
(test (append (inlet 'a 1 'b 2) (hash-table '(c . 3) '(d . 4))) (inlet 'b 2 'a 1 'c 3 'd 4))
(test (append "" #() #u8(101 102)) "ef")
(test (append "" #u8(101 102) (hash-table)) "ef")
(test (append #u8() #() #u8(101 102)) #u8(101 102))
(test (append #u8() (inlet) "") #u8())
(test (append #u8() #u8(101 102) "123") #u8(101 102 49 50 51))
(test (append () "" (int-vector 1 2 3)) (int-vector 1 2 3))
(test (let ((v (append #() #u8() (hash-table '(c . 3) '(d . 4)))))
	(or (equal? v #((c . 3) (d . 4)))
	    (equal? v #((d . 4) (c . 3))))) #t)
(test (append #() #(1 2 3) (inlet)) #(1 2 3))
(test (append #() (float-vector 1.0 2.0 3.0) ()) #(1.0 2.0 3.0))
(test (append (float-vector) "" "123") (float-vector 49.0 50.0 51.0))
(test (append (float-vector) (int-vector 1 2 3) #u8(101 102)) (float-vector 1.0 2.0 3.0 101.0 102.0))
(test (append (inlet) #() #((g . 8) (h . 9))) (inlet 'g 8 'h 9))
(test (append (inlet) '((e . 5) (f . 6)) (hash-table '(c . 3) '(d . 4))) (inlet 'e 5 'f 6 'c 3 'd 4))
(test (append (hash-table) "" (inlet 'a 1 'b 2)) (hash-table '(b . 2) '(a . 1)))
(test (append (hash-table) '((e . 5) (f . 6)) (inlet 'a 1 'b 2)) (hash-table '(b . 2) '(e . 5) '(f . 6) '(a . 1)))
(test (append (hash-table) #((g . 8) (h . 9)) '((e . 5) (f . 6))) (hash-table '(e . 5) '(g . 8) '(f . 6) '(h . 9)))
(test (append "123" #u8(101 102) (hash-table)) "123ef")
(test (append #u8(101 102) #u8() #u8(101 102)) #u8(101 102 101 102))
(test (append #u8(101 102) "123" (int-vector 1 2 3)) #u8(101 102 49 50 51 1 2 3))
(test (append #u8(101 102) '(1 2 3) "") #u8(101 102 1 2 3))
(test (append '(1 2 3) #u8(101 102) #(1 2 3)) '(1 2 3 101 102 1 2 3))
(test (let ((lst (append '(1 2 3) (hash-table '(c . 3) '(d . 4)) "")))
	(or (equal? lst '(1 2 3 (c . 3) (d . 4)))
	    (equal? lst '(1 2 3 (d . 4) (c . 3))))) #t)
(test (append (int-vector 1 2 3) #u8(101 102) (float-vector 1.0 2.0 3.0)) (int-vector 1 2 3 101 102 1 2 3))
(test (append (int-vector 1 2 3) '(1 2 3) #u8(101 102)) (int-vector 1 2 3 1 2 3 101 102))
(test (append (hash-table '(c . 3) '(d . 4)) (hash-table '(c . 3) '(d . 4)) '((e . 5) (f . 6))) (hash-table '(e . 5) '(f . 6) '(c . 3) '(d . 4)))
(when with-block
  (test (append (block 1 2) (block 3 4)) (block 1 2 3 4)))

(test (random-state? (cdr (append '(1) (random-state 123)))) #t)
(test (append '(1) (random-state 123) ()) 'error)
(test (random-state? (append () (random-state 123))) #t)

(when full-test
  (let ((seqs (list "" #u8() () #() (int-vector) (float-vector) (inlet) (hash-table)
		    "123" #u8(101 102) 
		    '(1 2 3) '((e . 5) (f . 6)) 
		    #(1 2 3) #((g . 8) (h . 9)) (int-vector 1 2 3) (float-vector 1 2 3)
		    (inlet 'a 1 'b 2)
		    (hash-table* 'c 3 'd 4)
		    1 #f '(1 . 2) (let ((lst (list 1))) (set-cdr! lst lst)))))
    (define (test-append)
      (for-each
       (lambda (s1)
	 (catch #t 
	   (lambda () 
	     (append s1)
	     (for-each
	      (lambda (s2)
		(catch #t 
		  (lambda () 
		    (append s1 s2)
		    (for-each
		     (lambda (s3)
		       (catch #t 
			 (lambda () 
			   (append s1 s2 s3)
			   (for-each
			    (lambda (s4)
			      (catch #t 
				(lambda () 
				  (append s1 s2 s3 s4)
				  (for-each
				   (lambda (s5)
				     (catch #t (lambda () 
						 (append s1 s2 s3 s4 s5)) 
					    (lambda args 'error)))
				   seqs))
				(lambda args 'error)))
			    seqs))
			 (lambda args 'error)))
		     seqs))
		  (lambda args 'error)))
	      seqs))
	   (lambda args 'error)))
       seqs))
    (test-append)))



;;; --------------------------------------------------------------------------------
;;; VECTORS
;;; --------------------------------------------------------------------------------


;;; --------------------------------------------------------------------------------
;;; vector?

(test (vector? (make-vector 6)) #t)
(test (vector? (make-vector 6 #\a)) #t)
(test (vector? (make-vector 0)) #t)
;; (test (vector? #*1011) #f)
(test (vector? #(0 (2 2 2 2) "Anna")) #t)
(test (vector? #()) #t)
(test (vector? #("hi")) #t)
(test (vector? (vector 1)) #t)
(test (let ((v (vector 1 2 3))) (vector? v)) #t)

(for-each
 (lambda (arg)
   (test (vector? arg) #f))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (vector?) 'error)
(test (vector? #() #(1)) 'error)
(test (vector? begin) #f)
(test (vector? vector?) #f)

;;; make a shared ref -- we'll check it later after enough has happened that an intervening GC is likely

(define check-shared-vector-after-gc #f)
(let ((avect (make-vector '(6 6) 32)))
  (do ((i 0 (+ i 1)))
      ((= i 6))
    (do ((j 0 (+ j 1)))
	((= j 6))
      (set! (avect i j) (cons i j))))
  (set! check-shared-vector-after-gc (avect 3)))

(if (not with-bignums)
    (test (vector? (make-float-vector 3 pi)) #t))
(test (vector? (make-vector 3 pi)) #t)
(test (vector? (make-shared-vector (make-int-vector '(2 3)) '(3 2))) #t)





;;; --------------------------------------------------------------------------------
;;; make-vector

(test (let ((v (make-vector 3 #f))) (and (vector? v) (= (vector-length v) 3) (eq? (vector-ref v 1) #f))) #t)
(test (let ((v (make-vector 1 1))) (and (vector? v) (= (vector-length v) 1) (vector-ref v 0))) 1)
(test (let ((v (make-vector 0 1))) (and (vector? v) (= (vector-length v) 0))) #t)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) #(0 1 2 3 4))
(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) #(0 1 4 9 16))
(test (make-vector 2 'hi) #(hi hi))
(test (make-vector 0) #())
(test (make-vector -0) #())
(test (make-vector 0 'hi) #())
(test (make-vector 3 (make-vector 1 'hi)) #(#(hi) #(hi) #(hi)))
(test (make-vector 3 #(hi)) #(#(hi) #(hi) #(hi)))
(test (make-vector 9/3 (list)) #(() () ()))
(test (make-vector 3/1 (make-vector 1 (make-vector 1 'hi))) #(#(#(hi)) #(#(hi)) #(#(hi))))
(test (make-float-vector 0 0.0) #())
(test (make-vector 0 0.0) #())

(test (let ((v (make-vector 3 0))) (set! (vector-ref v 1) 32) v) #(0 32 0))
(test (let ((v (make-int-vector 3))) (set! (vector-ref v 1) 0) v) (make-int-vector 3 0))

(for-each
 (lambda (arg)
   (test (vector-ref (make-vector 1 arg) 0) arg))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (make-vector) 'error)
(test (make-vector 1 #f #t) 'error)
(test (make-vector 1 2 3) 'error)
(test (make-vector most-positive-fixnum) 'error)
(test (make-vector most-negative-fixnum) 'error)
(test (make-vector '(2 -2)) 'error)
(test (make-vector (list 2 -2 -3)) 'error)
(test (make-vector (cons 2 3)) 'error)
(test (make-vector '(2 3 . 4)) 'error)
(test (make-vector '(2 (3))) 'error)
(test (make-vector most-negative-fixnum) 'error)
(test (make-vector 3 0 #f 1) 'error)

(for-each
 (lambda (arg)
   (test (make-vector arg) 'error)
   (test (make-vector (list 2 arg)) 'error))
 (list #\a () -1 #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(test (eval-string "#2147483649D()") 'error)
(test (eval-string "#-9223372036854775808D()") 'error)
(test (eval-string "#922D()") 'error)


;;; make-shared-vector
(test (let ((v1 #2d((1 2 3) (4 5 6)))) (let ((v2 (make-shared-vector v1 '(6)))) v2)) #(1 2 3 4 5 6))
(test (let ((v1 #(1 2 3 4 5 6))) (let ((v2 (make-shared-vector v1 '(3 2)))) v2)) #2D((1 2) (3 4) (5 6)))
(test (make-shared-vector #2d() '(0)) #())
(test (make-shared-vector '(1) '(1)) 'error)
(test (make-shared-vector #(1) '(2)) 'error)
(test (make-shared-vector #(1) '(1 2)) 'error)
(test (make-shared-vector #(1 2 3 4) ()) 'error)
(test (make-shared-vector #(1 2 3 4) most-positive-fixnum) 'error)
(test (make-shared-vector #(1 2 3 4) most-negative-fixnum) 'error)
(test (make-shared-vector #(1 2 3 4) -1) 'error)
(test (make-shared-vector #(1 2 3 4) 5) 'error)
(test (make-shared-vector #(1 2 3 4) 0) #())
(test (make-shared-vector #(1 2 3 4) '(2)) #(1 2))
(test (make-shared-vector #(1 2 3 4) '(2 1)) #2D((1) (2)))
(test (make-shared-vector #(1 2 3 4) '(0)) #())
(for-each
 (lambda (arg)
   (test (make-shared-vector arg) 'error)
   (test (make-shared-vector #(1 2 3) arg) 'error))
 (list #\a () -1 #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))

(let ((v #2d((1 2) (3 4))))
  (test (make-shared-vector v '((1 2) (3 4))) 'error)
  (test (make-shared-vector v ()) 'error)
  (test (make-shared-vector v '(1.4)) 'error)
  (test (make-shared-vector v '(14 15)) 'error)
  (test (make-shared-vector v (list most-positive-fixnum)) 'error)
  (test (make-shared-vector v '(-1 0)) 'error)
  (test (make-shared-vector v '(1) most-positive-fixnum) 'error))

(let ((v (float-vector 0.0 1.0 2.0)))
  (let ((v1 (make-shared-vector v (list 1 3))))
    (test (float-vector? v1) #t)
    (test (morally-equal? (v 0) (v1 0 0)) #t)))

(let ((v (int-vector 0 1 2)))
  (let ((v1 (make-shared-vector v (list 1 3))))
    (test (int-vector? v1) #t)
    (test (morally-equal? (v 0) (v1 0 0)) #t)))

(let ((v (make-int-vector 3)))
  (set! (v 1) 1)
  (set! (v 2) 2)
  (let ((v1 (make-shared-vector v (list 1 3))))
    (test (float-vector? v1) #f)
    (test (int-vector? v1) #t)
    (test (integer? (v1 0 2)) #t)
    (test (= (v 2) (v1 0 2)) #t)))

(let ((v (vector 0 1 2 3 4 5 6 7 8)))
  (test (make-shared-vector v (list 3 2) 1) #2D((1 2) (3 4) (5 6)))
  (test (make-shared-vector v (list 3 2) 2) #2D((2 3) (4 5) (6 7)))
  (test (make-shared-vector v (list 3) 2) #(2 3 4))
  (test (make-shared-vector v (list 3) 0) (make-shared-vector v (list 3)))
  (test (make-shared-vector v (list 3) -1) 'error)
  (test (make-shared-vector v (list 3) 10) 'error)
  (test (make-shared-vector v (list 3) 3.2) 'error)
  (test (make-shared-vector v (list 3) "0") 'error)
  )

(test (make-shared-vector (make-shared-vector (float-vector 1.0 2.0 3.0 4.0) '(2 2)) '(0)) #())
(test (make-shared-vector (make-shared-vector (float-vector 1.0 2.0 3.0 4.0) '(2 2)) '(1)) (float-vector 1.0))
(test ((make-shared-vector (make-shared-vector (float-vector 1.0 2.0 3.0 4.0) '(2 2)) '(4 1)) 2 0) 3.0)


(when with-bignums
  (let ((v (float-vector (bignum "1.0") (bignum "2.0"))))
    (test (float-vector? v) #t)
    (test (v 0) 1.0)))

(test (vector? (float-vector)) #t)
(test (vector? (int-vector)) #t)
(when with-block (test (float-vector? _c_obj_) #t))
(test (float-vector? 1 2) 'error)
(test (float-vector?) 'error)
(test (int-vector? 1 2) 'error)
(test (int-vector?) 'error)
(for-each
 (lambda (arg)
   (if (float-vector? arg) (format *stderr* ";~A is a float-vector?~%" arg))
   (test (float-vector arg) 'error)
   (if (int-vector? arg) (format *stderr* ";~A is an int-vector?~%" arg))
   (test (int-vector arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _null_ quasiquote macroexpand #() #t (vector 1 2 3) (lambda (a) (+ a 1))))
 

;;; make-float-vector
(test (float-vector? (make-float-vector 3)) #t)
(test (float-vector? (make-float-vector 3 pi)) #t)
(test ((make-float-vector 3) 1) 0.0)
(test (float-vector? (float-vector)) #t)
(test (float-vector? (make-float-vector 0)) #t)
(test (float-vector? (int-vector)) #f)
(test (equal? (float-vector) (int-vector)) #t)
(test (equal? (vector) (int-vector)) #t)

(test (equal? (make-float-vector 3 1.0) (float-vector 1.0 1.0 1.0)) #t)
(test ((make-float-vector '(2 3) 2.0) 1 2) 2.0)
(test (nan? ((make-float-vector 3 1/0) 0)) #t)
(for-each
 (lambda (arg)
   (test (make-float-vector arg) 'error)
   (test (make-float-vector 3 arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (equal? (vector) (float-vector)) #t)
(test (float-vector? (make-float-vector 3 0)) #t)
(test (float-vector? (make-float-vector 3 1/2)) #t)


;;; make-int-vector
(test (int-vector? (make-int-vector 3)) #t)
(test (int-vector? (make-int-vector 3 2)) #t)
(test ((make-int-vector 3) 1) 0)
(test (int-vector? (int-vector)) #t)
(test (int-vector? (make-int-vector 0)) #t)
(test (int-vector? (float-vector)) #f)
(test (int-vector? (vector)) #f)

(test (equal? (make-int-vector 3 1) (int-vector 1 1 1)) #t)
(test ((make-int-vector '(2 3) 2) 1 2) 2)
(for-each
 (lambda (arg)
   (test (make-int-vector arg) 'error)
   (test (make-int-vector 3 arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i 1/2 pi #t (vector 1 2 3) (lambda (a) (+ a 1))))
(test (equal? (vector) (int-vector)) #t)

(test (catch #t
	(lambda () (make-float-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-float-vector argument 1, 3.0, is a real but should be an integer or a list of integers"))

(test (catch #t
	(lambda () (make-int-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-int-vector argument 1, 3.0, is a real but should be an integer or a list of integers"))

(test (catch #t
	(lambda () (make-vector 3.0))
	(lambda args
	  (let ((type (car args))
		(errmsg (apply format #f (cadr args))))
	    (list type errmsg))))
      '(wrong-type-arg "make-vector argument 1, 3.0, is a real but should be an integer or a list of integers"))


;;; float-vector-ref
;;; float-vector-set!

(test (float-vector-ref (float-vector 1.0 2.0) 1) 2.0)
(for-each
 (lambda (arg)
   (test (float-vector-ref arg 0) 'error)
   (test (float-vector-ref (float-vector 1.0) arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(let ((v (make-float-vector (list 2 3) 1.0))
      (v1 (make-float-vector 3)))
  (set! (v 1 1) 2.0)
  (test (v 1 1) 2.0)
  (test (v 0 1) 1.0)
  (test (float-vector-ref v 1 1) 2.0)
  (test (float-vector-ref v 0) (float-vector 1.0 1.0 1.0))
  (test (float-vector-set! v 0 0 3.0) 3.0)
  (test (float-vector-ref v 0 0) 3.0)
  (test (float-vector-ref v1 3) 'error)
  (test (float-vector-ref v 1 3) 'error)
  (test (float-vector-ref v 2 2) 'error)
  (test (float-vector-ref v1 most-positive-fixnum) 'error)
  (test (float-vector-ref v1 most-negative-fixnum) 'error)
  (test (float-vector-set! v1 3 0.0) 'error)
  (test (float-vector-set! v 1 3 0.0) 'error)
  (test (float-vector-set! v 2 2 0.0) 'error)
  (test (float-vector-set! v1 most-positive-fixnum 0.0) 'error)
  (test (float-vector-set! v1 most-negative-fixnum 0.0) 'error)
  (test (float-vector-set! v1 0 0+i) 'error)
  (for-each
   (lambda (arg)
     (test (float-vector-ref v 0 arg) 'error)
     (test (float-vector-set! arg 0 1.0) 'error)
     (test (float-vector-set! v1 arg) 'error)
     (test (float-vector-set! v1 0 arg) 'error)
     (test (float-vector-set! v 0 arg 1.0) 'error))
   (list #\a () #f "hi" 1+i 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand #t (vector 1 2 3) (lambda (a) (+ a 1))))
  (test (float-vector-ref v) 'error)
  (test (float-vector-set! v) 'error)
  (test (float-vector-ref v1 0 1) 'error)
  (test (float-vector-ref v 0 1 0) 'error)
  (test (float-vector-ref v1 -1) 'error)
  (float-vector-set! v1 0 2/5)
  (test (float-vector-ref v1 0) 0.4)
  (test (float-vector-set! v1 1 4) 4)
  (test (float-vector-ref v1 1) 4.0)
  (test (float-vector-ref v 3 0) 'error)
  (test (float-vector-ref v 1 3) 'error)
  (test (fill! v 0.0) 0.0))
(test (float-vector-ref (float-vector) 0) 'error)
(let ((v (float-vector 1 2 3)))
  (set! (float-vector-ref v 1) 32)
  (test v (float-vector 1 32 3))
  (set! (v 0) 64)
  (test v (float-vector 64 32 3))
  (test (float-vector-set! v 2 (float-vector-set! v 1 0.0)) 0.0)
  (test v (float-vector 64 0 0)))
(let ((v0 (make-float-vector '(3 0)))
      (v1 (make-float-vector '(0 3)))
      (v2 (make-float-vector '(2 3)))
      (v3 (make-float-vector '(1 3)))
      (v4 (make-float-vector '(3 1))))
  (test (float-vector? v0) #t)
  (test (float-vector-ref v0 0 0) 'error)
  (test (vector? v0) #t)
  (test (vector-ref v0 0 0) 'error)
  (test (v0 0 0) 'error)
  (test (float-vector? v1) #t)
  (test (float-vector-ref v1 0 0) 'error)
  (test (vector? v1) #t)
  (test (vector-ref v1 0 0) 'error)
  (test (v1 0 0) 'error)
  (test (equal? v0 v1) #f)
  (test (float-vector? (float-vector-ref v2 1)) #t)
  (test (float-vector-set! v2 1 32.0) 'error)
  (test (float-vector-set! (float-vector-ref v2 1) 1 32.0) 32.0)
  (test (float-vector-ref v2 1 1) 32.0)
  (test (float-vector-ref v3 0) (float-vector 0 0 0))
  (test (float-vector-ref v4 0) (float-vector 0))
  )
(let ()
  (define (hi) (let ((v2 (make-float-vector '(2 3)))) (float-vector-set! v2 1 12.0) v2))
  (test (hi) 'error))

(let () ; regression test for optimizer safe_c_opcq_opcq bug 
  (define (fx n x y)
    (make-float-vector (if x (+ n 1) n)
		       (if y 0 (/ pi 2))))
  (test (morally-equal? (fx 3 #f #f) (make-float-vector 3 (/ pi 2))) #t)
  (test (morally-equal? (fx 3 #f #t) (make-float-vector 3)) #t)
  (test (morally-equal? (fx 3 #t #f) (make-float-vector 4 (/ pi 2))) #t)
  (test (morally-equal? (fx 3 #t #t) (make-float-vector 4)) #t)

  (define (fx1 n x y)
    (make-float-vector (if x (+ n 1) (- n 1))
		       (if y (* pi 2) (/ pi 2))))
  (test (morally-equal? (fx1 3 #f #f) (make-float-vector 2 (/ pi 2))) #t)
  (test (morally-equal? (fx1 3 #f #t) (make-float-vector 2 (* pi 2))) #t)
  (test (morally-equal? (fx1 3 #t #f) (make-float-vector 4 (/ pi 2))) #t)
  (test (morally-equal? (fx1 3 #t #t) (make-float-vector 4 (* pi 2))) #t)

  (define (fx2 n x y)
    (make-float-vector (if x (+ n 1) n)
		       (if y (* pi 2) 0.0)))
  (test (morally-equal? (fx2 3 #f #f) (make-float-vector 3)) #t)
  (test (morally-equal? (fx2 3 #f #t) (make-float-vector 3 (* pi 2))) #t)
  (test (morally-equal? (fx2 3 #t #f) (make-float-vector 4)) #t)
  (test (morally-equal? (fx2 3 #t #t) (make-float-vector 4 (* pi 2))) #t)

  (define (fx3 n y) ; same for safe_c_opssq_opcq
    (make-float-vector (+ n n)
		       (if y 0.0 (/ pi 2))))
  (test (morally-equal? (fx3 3 #f) (make-float-vector 6 (/ pi 2))) #t)
  (test (morally-equal? (fx3 3 #t) (make-float-vector 6)) #t)
  )



;;; int-vector-ref
;;; int-vector-set!

(test (int-vector-ref (int-vector 1 2) 1) 2)
(for-each
 (lambda (arg)
   (test (int-vector-ref arg 0) 'error)
   (test (int-vector-ref (int-vector 1) arg) 'error))
 (list #\a () #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))
(let ((v (make-int-vector (list 2 3) 1))
      (v1 (make-int-vector 3)))
  (set! (v 1 1) 2)
  (test (v 1 1) 2)
  (test (v 0 1) 1)
  (test (int-vector-ref v 1 1) 2)
  (test (int-vector-ref v 0) (int-vector 1 1 1))
  (test (int-vector-set! v 0 0 3) 3)
  (test (int-vector-ref v 0 0) 3)
  (test (int-vector-ref v1 3) 'error)
  (test (int-vector-ref v 1 3) 'error)
  (test (int-vector-ref v 2 2) 'error)
  (test (int-vector-ref v1 most-positive-fixnum) 'error)
  (test (int-vector-ref v1 most-negative-fixnum) 'error)
  (test (int-vector-set! v1 3 0) 'error)
  (test (int-vector-set! v 1 3 0) 'error)
  (test (int-vector-set! v 2 2 0) 'error)
  (test (int-vector-set! v1 most-positive-fixnum 0) 'error)
  (test (int-vector-set! v1 most-negative-fixnum 0) 'error)
  (test (int-vector-set! v1 0 0+i) 'error)
  (for-each
   (lambda (arg)
     (test (int-vector-ref v 0 arg) 'error)
     (test (int-vector-set! arg 0 1) 'error)
     (test (int-vector-set! v1 arg) 'error)
     (test (int-vector-set! v1 0 arg) 'error)
     (test (int-vector-set! v 0 arg 1) 'error))
   (list #\a () #f "hi" 1+i 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand #t (vector 1 2 3) (lambda (a) (+ a 1))))
  (test (int-vector-ref v) 'error)
  (test (int-vector-set! v) 'error)
  (test (int-vector-ref v1 0 1) 'error)
  (test (int-vector-ref v 0 1 0) 'error)
  (test (int-vector-ref v1 -1) 'error)
  (int-vector-set! v1 0 2)
  (test (int-vector-ref v1 0) 2)
  (test (int-vector-set! v1 1 4) 4)
  (test (int-vector-ref v1 1) 4)
  (test (int-vector-ref v 3 0) 'error)
  (test (int-vector-ref v 1 3) 'error)
  (test (fill! v 0) 0))
(test (int-vector-ref (int-vector) 0) 'error)
(let ((v (int-vector 1 2 3)))
  (set! (int-vector-ref v 1) 32)
  (test v (int-vector 1 32 3))
  (set! (v 0) 64)
  (test v (int-vector 64 32 3))
  (test (int-vector-set! v 2 (int-vector-set! v 1 0)) 0)
  (test v (int-vector 64 0 0)))
(let ((v0 (make-int-vector '(3 0)))
      (v1 (make-int-vector '(0 3)))
      (v2 (make-int-vector '(2 3)))
      (v3 (make-int-vector '(1 3)))
      (v4 (make-int-vector '(3 1))))
  (test (int-vector? v0) #t)
  (test (int-vector-ref v0 0 0) 'error)
  (test (vector? v0) #t)
  (test (vector-ref v0 0 0) 'error)
  (test (v0 0 0) 'error)
  (test (int-vector? v1) #t)
  (test (int-vector-ref v1 0 0) 'error)
  (test (vector? v1) #t)
  (test (vector-ref v1 0 0) 'error)
  (test (v1 0 0) 'error)
  (test (equal? v0 v1) #f)
  (test (int-vector? (int-vector-ref v2 1)) #t)
  (test (int-vector-set! v2 1 32) 'error)
  (test (int-vector-set! (int-vector-ref v2 1) 1 32) 32)
  (test (int-vector-ref v2 1 1) 32)
  (test (int-vector-ref v3 0) (int-vector 0 0 0))
  (test (int-vector-ref v4 0) (int-vector 0))
  )
(let ()
  (define (hi) (let ((v2 (make-int-vector '(2 3)))) (int-vector-set! v2 1 12) v2))
  (test (hi) 'error))

(let ()
  (define (f1)
    (let ((x (float-vector 0.0)))
      (set! (x 0) (complex 1 2))))
  (test (let ((x (float-vector 0.0))) (set! (x 0) 1+i)) 'error)
  (test (f1) 'error)

  (define (f2)
    (let ((x (int-vector 0)))
      (int-vector-set! x 0 (complex 1 2))))
  (test (let ((x (int-vector 0))) (set! (x 0) 0+i)) 'error)
  (test (f2) 'error)

  (define (f3)
    (let ((x (float-vector 0.0)))
      (float-vector-set! x 0 (complex 1 2))))
  (test (f3) 'error)

  (define (f4)
    (let ((x (int-vector 0)))
      (set! (x 0) (complex 1 2))))
  (test (f4) 'error))


;;; --------------------------------------------------------------------------------
;;; vector

(test (vector 1 2 3) #(1 2 3))
(test (vector 1 '(2) 3) #(1 (2) 3))
(test (vector) #())
(test (vector (vector (vector))) #(#(#())))
(test (vector (vector) (vector) (vector)) #(#() #() #()))
(test (vector (list)) #(()))
(test #(1 #\a "hi" hi) (vector 1 #\a "hi" 'hi))
(test (let ((v (make-vector 4 "hi")))
	(vector-set! v 0 1)
	(vector-set! v 1 #\a)
	(vector-set! v 3 'hi)
	v)
      #(1 #\a "hi" hi))
(let ((x 34))
  (test (vector x 'x) #(34 x)))

(for-each
 (lambda (arg)
   (test (vector-ref (vector arg) 0) arg))
 (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
       3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1))))




;;; --------------------------------------------------------------------------------
;;; vector->list
;;; list->vector

(test (vector->list #(0)) (list 0))
(test (vector->list (vector)) ())
(test (vector->list #(a b c)) '(a b c))
(test (vector->list #(#(0) #(1))) '(#(0) #(1)))
(test (vector? (list-ref (let ((v (vector 1 2))) (vector-set! v 1 v) (vector->list v)) 1)) #t)

(test (list->vector ()) #())
(test (list->vector '(a b c)) #(a b c))
(test (list->vector (list (list 1 2) (list 3 4))) #((1 2) (3 4)))
(test (list->vector ''foo) #(quote foo))
(test (list->vector (list)) #())
(test (list->vector (list 1)) #(1))
(test (list->vector (list (list))) #(()))
(test (list->vector (list 1 #\a "hi" 'hi)) #(1 #\a "hi" hi))
(test (list->vector ''1) #(quote 1))
(test (list->vector '''1) #(quote '1))

(for-each
 (lambda (arg)
   (if (proper-list? arg)
       (test (vector->list (list->vector arg)) arg)))
 lists)
(set! lists ())

(test (list->vector (vector->list (vector))) #())
(test (list->vector (vector->list (vector 1))) #(1))
(test (vector->list (list->vector (list))) ())
(test (vector->list (list->vector (list 1))) '(1))

(test (reinvert 12 vector->list list->vector #(1 2 3)) #(1 2 3))

(test (vector->list) 'error)
(test (list->vector) 'error)
(test (vector->list #(1) #(2)) 'error)
(test (list->vector '(1) '(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector->list arg) 'error))
 (list #\a 1 () (list 1) '(1 . 2) #f 'a-symbol "hi" abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (let ((x (cons #\a #\b))) (set-cdr! x x) (list->vector x)) 'error)
(test (list->vector (cons 1 2)) 'error)
(test (list->vector '(1 2 . 3)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (list->vector lst)) 'error)
(test (let ((lst (list #\a #\b))) (set! (cdr (cdr lst)) lst) (apply vector lst)) 'error)

(for-each
 (lambda (arg)
   (test (list->vector arg) 'error))
 (list "hi" #\a 1 '(1 . 2) (cons #\a #\b) 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (vector->list #(1 2 3 4) 0) '(1 2 3 4))
(test (vector->list #(1 2 3 4) 2) '(3 4))
(test (vector->list #(1 2 3 4) 0 4) '(1 2 3 4))
(test (vector->list #(1 2 3 4) 4 4) ())
(test (vector->list #(1 2 3 4) 1 2) '(2))

(test (vector->list #(1 2 3 4) -1 4) 'error)
(test (vector->list #(1 2 3 4) 1 0) 'error)
(test (vector->list #(1 2 3 4) 5) 'error)
(test (vector->list #(1 2 3 4) 1 5) 'error)
(test (vector->list #(1 2 3 4) 1 2 3) 'error)

(test (vector->list #() 0 10) 'error)
(test (vector->list #(1) 0 2) 'error)
(test (vector->list #() 0 0) ())
(test (vector->list #(1) 1) ())
(test (vector->list #(1) 0) '(1))
(test (vector->list #() #\null) 'error)
(test (vector->list #() 0 #\null) 'error)
(test (vector->list #() -1) 'error)
(test (vector->list #(1) -1) 'error)
(test (vector->list #(1) 0 -1) 'error)
(test (vector->list #(1) -2 -1) 'error)
(test (vector->list #(1) most-negative-fixnum) 'error)
(test (vector->list #(1) 2) 'error)

(test (vector->list (make-int-vector 3)) '(0 0 0))
(test (vector->list (make-float-vector 3)) '(0.0 0.0 0.0))

(for-each
 (lambda (arg)
   (test (vector->list #(0 1 2 3 4 5) arg) 'error)
   (test (vector->list #(0 1 2 3 4 5) 1 arg) 'error))
 (list #\a "hi" () (list 1) '(1 . 2) 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))



;;; --------------------------------------------------------------------------------
;;; vector-length

(test (vector-length (vector)) 0)
(test (vector-length (vector 1)) 1)
(test (vector-length (make-vector 128)) 128)
(test (vector-length #(a b c d e f)) 6)
(test (vector-length #()) 0)
(test (vector-length (vector #\a (list 1 2) (vector 1 2))) 3)
(test (vector-length #(#(#(hi)) #(#(hi)) #(#(hi)))) 3)
(test (vector-length (vector 1 2 3 4)) 4)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) v)) 2)
(test (vector-length (let ((v (vector 1 2))) (vector-set! v 1 v) (vector-ref v 1))) 2)
(test (vector-length (make-int-vector 3 0)) 3)
(if (not with-bignums) (test (vector-length (make-float-vector 3 pi)) 3))
(if (not with-bignums) (test (vector-length (make-float-vector '(2 3) pi)) 6))

(test (vector-length) 'error)
(test (vector-length #(1) #(2)) 'error)

(for-each
 (lambda (arg)
   (test (vector-length arg) 'error))
 (list "hi" #\a 1 () '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))




;;; --------------------------------------------------------------------------------
;;; vector-ref

(test (vector-ref #(1 1 2 3 5 8 13 21) 5) 8)
(test (vector-ref #(1 1 2 3 5 8 13 21) (let ((i (round (* 2 (acos -1))))) (if (inexact? i) (inexact->exact i)  i))) 13)
(test (let ((v (make-vector 1 0))) (vector-ref v 0)) 0)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 1)) (list 2))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref v 2)) #(#\a #\a #\a))
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 1)) #\a)
(test (vector-ref #(a b c) 1) 'b)
(test (vector-ref #(()) 0) ())
(test (vector-ref #(#()) 0) #())
(test (vector-ref (vector-ref (vector-ref #(1 (2) #(3 (4) #(5))) 2) 2) 0) 5)
(test (let ((v (vector 1 2))) (vector-set! v 1 v) (eq? (vector-ref v 1) v)) #t)
(test (let ((v (make-int-vector 3))) (vector-ref v 1)) 0)
(test (let ((v (make-vector 3 0))) (vector-ref v 1)) 0)
(test (let ((v (make-float-vector 3 1.0))) (vector-ref v 1)) 1.0)
(test (let ((v (make-int-vector 6 0))) (vector-set! v 3 32) (let ((v1 (make-shared-vector v '(2 3)))) (vector-ref v1 1 0))) 32)

(test (vector-ref) 'error)
(test (vector-ref #(1)) 'error)
(test (vector-ref #(1) 0 0) 'error)
(test (vector-ref () 0) 'error)

(test (let ((v (make-vector 1 0))) (vector-ref v 1)) 'error)
(test (let ((v (make-vector 1 0))) (vector-ref v -1)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 2) 3)) 'error)
(test (let ((v (vector 1 (list 2) (make-vector 3 #\a)))) (vector-ref (vector-ref v 3) 0)) 'error)
(test (vector-ref (vector) 0) 'error)
(test (vector-ref #() 0) 'error)
(test (vector-ref #() -1) 'error)
(test (vector-ref #() 1) 'error)
(test (vector-ref #(1 2 3) (floor .1)) 1)
(test (vector-ref #(1 2 3) (floor 0+0i)) 1)
(test (vector-ref #10D((((((((((0 1)))))))))) 0 0 0 0 0 0 0 0 0 1) 1)

(test (#(1 2) 1) 2)
(test (#(1 2) 1 2) 'error)
(test ((#("hi" "ho") 0) 1) #\i)
(test (((vector (list 1 2) (cons 3 4)) 0) 1) 2)
(test ((#(#(1 2) #(3 4)) 0) 1) 2)
(test ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) 0) 1)
(test ((((list (list (list 1 2) 0) 0) 0) 0) ((((vector (vector (vector 1 2) 0) 0) 0) 0) 0)) 2)
(test (#(1 2) -1) 'error)
(test (#()) 'error)
(test (#(1)) 'error)
(test (#2d((1 2) (3 4))) 'error)
(test (apply (make-vector '(1 2))) 'error)

(test (eval-string "#2/3d(1 2)") 'error)
(test (eval-string "#2.1d(1 2)") 'error)
(test (eval-string "#(1 2 . 3)") 'error)
(test (#(#(#(#t))) 0 0 0) #t)
(test (let ((v (make-vector 3 0 #t))) (v 0 0)) 'error)
(test (let ((v (make-int-vector '(2 2)))) (v 0 0 0)) 'error)
(test (let ((v (make-float-vector 3))) (vector-ref v 0 0)) 'error)
(test (let ((v (make-vector '(2 2) 0.0 #t))) (vector-ref v 0 0 0)) 'error)
(test (let ((v (make-vector 3 0))) (v 0 0)) 'error)
(test (let ((v (make-vector '(2 2) 0))) (v 0 0 0)) 'error)

(let ((v #(1 2 3)))
  (for-each
   (lambda (arg)
     ; (format *stderr* "~A~%" arg)
     (test (vector-ref arg 0) 'error)
     (test (v arg) 'error)
     (test (v arg 0) 'error)
     (test (vector-ref v arg) 'error)
     (test (vector-ref v arg 0) 'error)
     (test (vector-ref #2d((1 2) (3 4)) 0 arg) 'error))
   (list "hi" () #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table))))


(test (vector-ref #(#(1 2 3) #(4 5 6)) 1) #(4 5 6))
(test (vector-ref #(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) #(#(7 8 9) #(10 11 12)))
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) #(7 8 9))
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (#(#(1 2 3) #(4 5 6)) 1) #(4 5 6))
(test (#(#(1 2 3) #(4 5 6)) 1 2) 6)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1) #(#(7 8 9) #(10 11 12)))
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0) #(7 8 9))
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2) 9)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 3) 'error)
(test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) 1 0 2 0) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1)) #(4 5 6))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (L 1 2 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1)) #(#(7 8 9) #(10 11 12)))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L 1 0 2 3)) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) ((L 1) 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (((L 1) 2) 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L 1) 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L 1) 0 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((((L 1) 0) 2) 3)) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref (L 1) 2)) 6)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref ((L 1) 2) 3)) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L 1) 0)) #(7 8 9))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L 1) 0) 2)) 9)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (((L 1) 0) 2) 3)) 'error)


(let ((zero 0)
      (one 1)
      (two 2)
      (three 3)
      (thirty-two 32))
  (test (vector-ref #(#(1 2 3) #(4 5 6)) one) #(4 5 6))
  (test (vector-ref #(#(1 2 3) #(4 5 6)) one two) 6)
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) #(#(7 8 9) #(10 11 12)))
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) #(7 8 9))
  (test (vector-ref #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test (#(#(1 2 3) #(4 5 6)) one) #(4 5 6))
  (test (#(#(1 2 3) #(4 5 6)) one two) 6)
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one) #(#(7 8 9) #(10 11 12)))
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero) #(7 8 9))
  (test (#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) one zero two) 9)
  
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (L one)) #(4 5 6))
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (L one two)) 6)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one)) #(#(7 8 9) #(10 11 12)))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (L one zero two)) 9)
  
  (test (let ((L #(#(1 2 3) #(4 5 6)))) ((L one) two)) 6)
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (((L one) two) 3)) 'error)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (((L one) zero) two)) 9)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one zero) two)) 9)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((L one) zero two)) 9)
  
  (test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-ref (L one) two)) 6)
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-ref (L one) zero)) #(7 8 9))
  (test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) ((vector-ref (L one) zero) two)) 9))

(test ((#(#(:hi) #\a (3)) (#("hi" 2) 1)) (#2d((#() ()) (0 #(0))) 1 ('(cons 0) 1))) 3)
(test (#(1 2 3) (#(1 2 3) 1)) 3)
(test ((#(#(1 2)) (#(1 0) 1)) (#(3 2 1 0) 2)) 2)
(test (apply min (#(1 #\a (3)) (#(1 2) 1))) 3) ; i.e vector ref here 2 levels -- (#(1 2) 1) is 2 and (#(1 #\a (3)) 2) is (3) 

;;; vector-ref optimizer checks
(define global_vector (vector 1 2 3))
(let ()
  (define (hi i) (vector-ref global_vector i))
  (test (hi 1) 2))
(let ()
  (define (hi i) (vector-ref global_vector (vector-ref global_vector i)))
  (test (hi 0) 2))

(test (let ((v #(0 1 2 3 4 5))) (define (f1) (v 4/3)) (f1)) 'error)
(test (let ((v "012345")) (define (f1) (v 4/3)) (f1)) 'error)
(test (let ((v (list 0 1 2 3 4 5))) (define (f1) (v 4/3)) (f1)) 'error)

(define-constant -a-global-vector- (vector 1 2 3))
(let ()
  (define (fg a) (vector-ref -a-global-vector- a))
  (test (fg 0) 1))

(let ()
  (define (f1)    
    (let ((v (vector #f)) (X #2D((1 2) (3 4))))    
      (do ((i 0 (+ i 1))) ((= i 1) v)    
	(vector-set! v 0 (vector-ref X 1)))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)     
    (let ((v (vector #f)) (X #2D((1 2) (3 4))))     
      (do ((i 0 (+ i 1))) ((= i 1) v)     
	(vector-set! v 0 (vector-ref X 2)))))
  (test (f1) 'error))

(let ()
  (define (f1)    
    (let ((I 0) (v (vector #f)) (X #2D((1 2) (3 4))))    
      (do ((i 0 (+ i 1))) ((= i 1) v)    
	(vector-set! v 0 (vector-ref X (+ I 1))))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)    
    (let ((I 1) (v (vector #f)) (X #2D((1 2) (3 4))))    
      (do ((i 0 (+ i 1))) ((= i 1) v)    
	(vector-set! v 0 (vector-ref X (+ I 1))))))
  (test (f1) 'error))

(set! global_vector #2D((1 2) (3 4)))
(let ()
  (define (f1)    
    (let ((I 1) (v (vector #f)))
      (do ((i 0 (+ i 1))) ((= i 1) v)    
	(vector-set! v 0 (vector-ref global_vector I)))))
  (test (f1) #(#(3 4))))

(let ()
  (define (f1)    
    (let ((I 2) (v (vector #f)))
      (do ((i 0 (+ i 1))) ((= i 1) v)    
	(vector-set! v 0 (vector-ref global_vector I)))))
  (test (f1) 'error))



;;; --------------------------------------------------------------------------------
;;; vector-set!

(test (let ((vec (vector 0 '(2 2 2 2) "Anna"))) (vector-set! vec 1 '("Sue" "Sue")) vec) #(0 ("Sue" "Sue") "Anna"))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 32) v) #(1 32 3))
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-set! v 1 arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))
(test (let ((v (vector 1 2 3))) (vector-set! v 1 0) v) #(1 0 3))
(test (let ((v (vector #f))) (vector-set! v 0 (vector)) v) #(#()))
(test (let ((v (vector 1 (list 2) (vector 1 2 3)))) (vector-set! (vector-ref v 2) 0 21) v) #(1 (2) #(21 2 3)))
(test (let ((v (make-int-vector 3))) (vector-set! v 1 32) (vector->list v)) '(0 32 0))
(test (let ((v (make-int-vector 3 0))) (set! (v 1) 32) (vector->list v)) '(0 32 0))

(test (vector-set! (vector 1 2) 0 4) 4)
(test (vector-set!) 'error)
(test (vector-set! #(1)) 'error)
(test (vector-set! #(1) 0) 'error)
(test (vector-set! #(1) 0 0 1) 'error)
(test (vector-set! #(1) 0 0 1 2 3) 'error)
(test (vector-set! #(1) #(0) 1) 'error)
(test (vector-set! #(1 2) 0 2) 2)
(test (let ((x 2) (v (vector 1 2))) (vector-set! (let () (set! x 3) v) 1 23) (list x v)) '(3 #(1 23)))
(test (let ((v #(1 2))) (vector-set! v 0 32)) 32)
(test (let ((v #(1 2))) (set! (v 0) 32)) 32)
(test (let ((v #(1 2))) (set! (vector-ref v 0) 32)) 32)
(test (let ((v (make-vector '(2 3) 0))) (set! (v (values 0 1)) 23) v) #2D((0 23 0) (0 0 0)))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" () #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t (lambda (a) (+ a 1)) (make-hash-table)))

(let ((v (vector 1 2 3)))
  (for-each
   (lambda (arg)
     (test (vector-set! v arg 0) 'error))
   (list "hi" () #() #\a -1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #t (make-vector 3) (lambda (a) (+ a 1)))))

(for-each
 (lambda (arg)
   (test (vector-set! arg 0 0) 'error))
 (list "hi" () #\a 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((v #(#(0 1) #(2 3))))
  (vector-set! (vector-ref v 1) 1 4)
  (test (v 1 1) 4)
  (set! ((vector-ref v 1) 1) 5)
  (test (v 1 1) 5)
  (set! ((v 1) 1) 6)
  (test (v 1 1) 6)
  (vector-set! (v 1) 1 7)
  (test (v 1 1) 7)
  (set! (v 1 1) 8)
  (test (v 1 1) 8))

(let ((v (vector)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-set! v 1 0) 'error)
  (test (vector-set! v -1 0) 'error))
(test (vector-set! #() 0 123) 'error)
(test (vector-set! #(1 2 3) 0 123) 123)
(test (let ((v #(1 2 3))) (set! (v 0) '(+ 1 2)) v) #((+ 1 2) 2 3))
(test (let ((v #(1 2 3))) (set! (v '(+ 1 1)) 2) v) 'error)
(test (let ((v #(1 2 3))) (set! (v (+ 1 1)) 2) v) #(1 2 2))

(test (let ((g (lambda () #(1 2 3)))) (vector-set! (g) 0 #\?) (g)) #(#\? 2 3))
(test (let ((g (lambda () '(1 . 2)))) (set-car! (g) 123) (g)) '(123 . 2))
(test (let ((g (lambda () '(1 2)))) (list-set! (g) 0 123) (g)) '(123 2))
(test (let ((g (lambda () (symbol->string 'hi)))) (string-set! (g) 1 #\a) (symbol->string 'hi)) "hi")

(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 32) L) #(#(1 2 3) 32))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 0 2 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (vector-set! L 1 3 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 32) L) #(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 0 2 1 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (vector-set! L 1 4 2 32) L) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1) 32) L) #(#(1 2 3) 32))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 0) 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 0 2) 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (L 1 3) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1) 32) L) #(#(#(1 2 3) #(4 5 6)) 32))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0) 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 0 2 1) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (L 1 4 2) 32) L) 'error)

(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! ((L 1) 0) 32) L) #(#(1 2 3) #(32 5 6)))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! (((L 1) 0) 2) 32) L) 'error)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (set! ((L 1) 3) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1) 0) 32) L) #(#(#(1 2 3) #(4 5 6)) #(32 #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 0) 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((((L 1) 0) 2) 1) 32) L) 'error)
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! (((L 1) 4) 2) 32) L) 'error)
(test (let ((L #(#(#(1 2 3))))) (set! ((L 0) 0 1) 32) L) #(#(#(1 32 3))))
(test (let ((L #(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))))) (set! ((L 1 0) 2) 32) L) #(#(#(1 2 3) #(4 5 6)) #(#(7 8 32) #(10 11 12))))

(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (L 0 0 1) 32) 
	L) 
      #(#(#(#(1 2 3) 32) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0) 0 1 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0) 1 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((L 0 0 1) 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0) 0) 1 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! (((L 0 0) 1) 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))
(test (let ((L #(#(#(#(1 2 3) #(4 5 6)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))) 
	(set! ((((L 0) 0) 1) 2) 32) 
	L) 
      #(#(#(#(1 2 3) #(4 5 32)) #(#(7 8 9) #(10 11 12))) #(13 14 15)))


(test (eq? (car (catch #t (lambda () (set! (#(1)) 2)) (lambda args args))) 'wrong-number-of-args) #t)
(test (eq? (car (catch #t (lambda () (set! (#(1) 0 0) 2)) (lambda args args))) 'syntax-error) #t) 
(test (eq? (car (catch #t (lambda () (set! ((#(1) 0) 0) 2)) (lambda args args))) 'syntax-error) #t) ; (set! (1 ...))
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L) 1) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L)) 32) L) (lambda args args))) 'wrong-number-of-args)) #t)
(test (let ((L #(#(1 2 3) #(4 5 6)))) (eq? (car (catch #t (lambda () (set! ((L 1) 2)) L) (lambda args args))) 'syntax-error)) #t)

(let ((v #(1 2 3))) (define (vr v a) (vector-ref v (+ a 1))) (test (vr v 1) 3))
(let () (define (fillv) (let ((v (make-vector 10))) (do ((i 0 (+ i 1))) ((= i 10) v) (vector-set! v i i)))) (test (fillv) #(0 1 2 3 4 5 6 7 8 9)))
(let () (define (vv) (let ((v #(0 1)) (i 0) (x 2)) (vector-set! v i (+ (vector-ref v i) x)))) (test (vv) 2))
(let () (define (hi) (let ((v1 #(0 1)) (i 0) (j 1)) (vector-set! v1 i (vector-ref v1 j)))) (hi) 1)



;;; --------------------------------------------------------------------------------
;;; vector-fill!

(test (fill! (vector 1 2) 4) 4)

(test (let ((v (vector 1 2 3))) (vector-fill! v 0) v) #(0 0 0))
(test (let ((v (vector))) (vector-fill! v #f) v) #())
(let ((v (make-vector 8 #f)))
  (for-each
   (lambda (arg)
     (vector-fill! v arg)
     (test (vector-ref v 1) arg))
   (list #\a 1 () (list 1) '(1 . 2) #f "hi" 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
	 3.14 3/4 1.0+1.0i #t (vector 1 2 3) (lambda (a) (+ a 1)))))

(test (let ((str "hi") (v (make-vector 3))) (vector-fill! v str) (string-set! (vector-ref v 0) 1 #\a) str) "ha")
(test (let ((lst (list 1 2)) (v (make-vector 3))) (vector-fill! v lst) (list-set! (vector-ref v 0) 1 #\a) lst) '(1 #\a))

(test (let ((v (vector 1 2 3))) (vector-set! v -1 0)) 'error)
(test (let ((v (vector 1 2 3))) (vector-set! v 3 0)) 'error)
(test (vector-fill! #(1 2) 2) 2)
(test (vector-fill! #() 0) 0)
(test (vector-fill! (vector) 0) 0)
(test (let ((v (vector 1))) (vector-fill! v 32) (v 0)) 32)
(test (let ((v (make-vector 11 0))) (vector-fill! v 32) (v 10)) 32)
(test (let ((v (make-vector 16 0))) (vector-fill! v 32) (v 15)) 32)
(test (let ((v (make-vector 3 0))) (vector-fill! v 32) (v 1)) 32)
(test (let ((v (make-vector 3 0))) (fill! v 32) (v 1)) 32)
(test (let ((v #2d((1 2 3) (4 5 6)))) (vector-fill! (v 1) 12) v) #2D((1 2 3) (12 12 12)))

(for-each
 (lambda (arg)
   (test (vector-fill! arg 0) 'error))
 (list "hi" #\a () 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(when with-bignums
  (let ((v (make-vector 2 0.0)))
    (vector-fill! v 1180591620717411303424)
    (num-test (v 1) (expt 2 70))
    (vector-fill! v 3/1180591620717411303424)
    (num-test (v 0) 3/1180591620717411303424)
    (vector-fill! v 1180591620717411303424.0)
    (num-test (v 1) 1180591620717411303424.0)
    (vector-fill! v (complex (expt 2 70) 1.0))
    (num-test (v 0) (complex (expt 2 70) 1.0))
    (set! v (float-vector 1.0))
    (vector-fill! v (bignum "2.0"))
    (num-test (v 0) 2.0)
    (vector-fill! v pi)
    (num-test (v 0) pi)
    (set! v (float-vector 0.0 0.0 0.0))
    (vector-fill! v (bignum "2.0") 1 2)
    (num-test (v 0) 0.0)
    (num-test (v 1) 2.0)
    (set! v (make-int-vector 1))
    (vector-fill! v (bignum "2"))
    (num-test (v 0) 2)
    (set! v (make-int-vector 3 0))
    (vector-fill! v (bignum "2") 1 2)
    (num-test (v 0) 0)
    (num-test (v 1) 2)))

(let ((v (make-vector 3)))
  (vector-fill! v v)
  (test (v 0) v)
  (set! (v 1) 32)
  (test ((v 0) 1) 32))

(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0) v) #(21 21 21 21 21))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0 5) v) #(21 21 21 21 21))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 0 3) v) #(21 21 21 4 5))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 2 3) v) #(1 2 21 4 5))
(test (let ((v (vector 1 2 3 4 5))) (vector-fill! v 21 3 3) v) #(1 2 3 4 5))

(if (not with-bignums) (test (let ((v (make-float-vector 3 pi))) (vector-fill! v 0.0) (vector->list v)) '(0.0 0.0 0.0)))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v "2.5")) 'error)
(test (let ((v (make-float-vector 3 pi))) (vector-fill! v #\a)) 'error)
(test (let ((v (make-float-vector 3))) (vector-fill! v 1+i) v) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3/4) (vector->list v)) '(0.75 0.75 0.75))
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3) (vector->list v)) '(3.0 3.0 3.0))
(test (let ((v (make-int-vector 3))) (vector-fill! v 1+i) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3/4) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3.0) v) 'error)

(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1) (vector->list v)) '(1 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 1) (vector->list v)) '(1 1 1))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 2) (vector->list v)) '(1 2 1))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 3) (vector->list v)) '(1 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 0 3) (vector->list v)) '(2 2 2))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 1 4) (vector->list v)) 'error)
(test (let ((v (make-int-vector 3 1))) (vector-fill! v 2 -1) (vector->list v)) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 1.0 1) (vector->list v)) '(0.0 1.0 1.0))
(test (let ((v (make-int-vector 3 1))) (vector-fill! v "2.5" 1)) 'error)
(test (let ((v (make-float-vector 3 pi))) (vector-fill! v #\a 0 1)) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 1+i 1) v) 'error)
(test (let ((v (make-float-vector 3 0.0))) (vector-fill! v 3/4 1) (vector->list v)) '(0.0 0.75 0.75))
(test (let ((v (make-float-vector 3))) (vector-fill! v 3 2) (vector->list v)) '(0.0 0.0 3.0))
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 1+i 2) v) 'error)
(test (let ((v (make-int-vector 3 0))) (vector-fill! v 3/4 0 1) v) 'error)
(test (let ((v (make-int-vector 3))) (vector-fill! v 3.0 2) v) 'error)
(test (vector-fill! #() 0 "hi") 'error)
(test (vector-fill! #() 0 -1 3) 'error)
(test (vector-fill! #() 0 1) 'error)
(test (vector-fill! #() 0 0 4/3) 'error)





;;; --------------------------------------------------------------------------------
;;; vector-append

(test (vector-append #() #2d()) #())
(test (vector-append) #())
(test (vector-append #()) #())
(test (vector-append #(1 2)) #(1 2))
(test (vector-append #(1) #(2 3) #() #(4)) #(1 2 3 4))
(test (vector-append #(1) #2d((2 3) (4 5)) #3d()) #(1 2 3 4 5))
(test (vector-append #2d((1 2) (3 4)) #3d(((5 6) (7 8)) ((9 10) (11 12)))) #(1 2 3 4 5 6 7 8 9 10 11 12))

(test (vector-append (vector 1 2) (make-int-vector 1 3) #(4)) #(1 2 3 4))
(test (vector-append (vector 1 2) (make-float-vector 1) #(4)) #(1 2 0.0 4))
(test (vector->list (vector-append (make-int-vector 1 3) (make-int-vector 2 1))) '(3 1 1))
(test (vector->list (vector-append (make-float-vector 1 0.0) (make-float-vector 2 1.0))) '(0.0 1.0 1.0))

(for-each
 (lambda (arg)
   (test (vector-append arg) 'error)
   (test (vector-append #(1 2) arg) 'error))
 (list "hi" #\a () 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (equal? (make-vector 3 1) (make-int-vector 3 1)) #f)

(let ((iv (make-int-vector 3 1))
      (fv (make-float-vector 3 2.0))
      (vv (make-vector 3 #f)))
  (test (equal? (vector-append iv iv iv) (make-int-vector 9 1)) #t)
  (test (vector-append iv iv vv) 'error)
  (test (vector-append iv fv iv) (int-vector 1 1 1 2 2 2 1 1 1))
  (test (vector-append iv fv fv) (int-vector 1 1 1 2 2 2 2 2 2))
  (test (vector-append iv fv vv) 'error)
  (test (vector-append iv vv iv) 'error)
  (test (vector-append fv iv iv) (float-vector 2.0 2.0 2.0 1 1 1 1 1 1))       ; #(2.0 2.0 2.0 1 1 1 1 1 1))
  (test (vector-append fv iv fv) (float-vector 2.0 2.0 2.0 1 1 1 2.0 2.0 2.0)) ; #(2.0 2.0 2.0 1 1 1 2.0 2.0 2.0))
  (test (vector-append fv fv iv) (float-vector 2.0 2.0 2.0 2.0 2.0 2.0 1 1 1)) ; #(2.0 2.0 2.0 2.0 2.0 2.0 1 1 1))
  (test (vector-append fv fv fv) (make-float-vector 9 2.0))
  (test (vector-append fv fv vv) 'error)
  (test (vector-append vv iv iv) #(#f #f #f 1 1 1 1 1 1))
  (test (vector-append vv iv fv) #(#f #f #f 1 1 1 2.0 2.0 2.0))
  (test (vector-append vv iv vv) #(#f #f #f 1 1 1 #f #f #f))
  (test (vector-append vv fv iv) #(#f #f #f 2.0 2.0 2.0 1 1 1))
  (test (vector-append vv fv fv) #(#f #f #f 2.0 2.0 2.0 2.0 2.0 2.0))
  (test (vector-append vv fv vv) #(#f #f #f 2.0 2.0 2.0 #f #f #f))
  (test (vector-append vv vv iv) #(#f #f #f #f #f #f 1 1 1))
  (test (vector-append vv vv fv) #(#f #f #f #f #f #f 2.0 2.0 2.0))
  (test (vector-append vv vv vv) #(#f #f #f #f #f #f #f #f #f)))

(test (equal? (vector-append (float-vector 1 2 3) #()) (float-vector 1 2 3)) #t)
(test (equal? (vector-append (float-vector) #(1 2 3) #() (make-int-vector 0 0)) (float-vector 1 2 3)) #t)
(test (equal? (float-vector) (vector-append (float-vector))) #t)
(test (equal? (vector-append #() (float-vector) (make-int-vector 3 1) (vector)) (make-vector 3 1)) #t)
(test (equal? (vector-append (int-vector 1 2 3) #()) (int-vector 1 2 3)) #t)
(test (equal? (vector-append (int-vector) #(1 2 3) #() (make-int-vector 0 0)) (int-vector 1 2 3)) #t)
(test (equal? (int-vector) (vector-append (int-vector))) #t)
(test (equal? (vector-append #() (int-vector) (make-int-vector 3 1) (vector)) (make-vector 3 1)) #t)

(when full-test
  (define (test-append size)
    (let ((strs ())
	  (vecs ())
	  (fvecs ())
	  (ivecs ())
	  (ifvecs ())
	  (allvecs ())
	  (bvecs ())
	  (lsts ()))
      (do ((i 0 (+ i 1)))
	  ((= i size))
	(set! strs (cons (make-string size (integer->char (+ 1 (random 255)))) strs))
	(set! bvecs (cons (string->byte-vector (make-string size (integer->char (random 256)))) bvecs))
	(set! vecs (cons (make-vector size i) vecs))
	(set! ivecs (cons (make-int-vector size i) ivecs))
	(set! fvecs (cons (make-float-vector size (* i 1.0)) fvecs))
	(set! ifvecs (cons (make-vector size (if (even? i) (* i 1.0) i)) ifvecs))
	(set! allvecs (cons (make-vector size (if (even? i) (* i 1.0) i)) allvecs))
	(set! lsts (cons (make-list size i) lsts)))
      (let ((lst (apply append lsts))
	    (vec (apply vector-append vecs))
	    (fvec (apply vector-append fvecs))
	    (ivec (apply vector-append ivecs))
	    (ifvec (apply vector-append ifvecs))
	    (allvec (apply vector-append allvecs))
	    (str (apply string-append strs))
	    (bvec (string->byte-vector (apply string-append bvecs))))
	(test (vector? vec) #t)
	(test (length vec) (* size size))
	(test (float-vector? fvec) #t)
	(test (length fvec) (* size size))
	(test (int-vector? ivec) #t)
	(test (length ivec) (* size size))
	(test (vector? allvec) #t)
	(test (length allvec) (* size size))
	(test (vector? ifvec) #t)
	(test (length ifvec) (* size size))
	(test (pair? lst) #t)
	(test (length lst) (* size size))
	(test (string? str) #t)
	(test (length str) (* size size))
	(test (byte-vector? bvec) #t)
	(test (length bvec) (* size size))
	)))
  
  (do ((i 1 (* i 10)))
      ((> i 1000))
    (test-append i)))


  
;;; --------------------------------------------------------------------------------
;;; miscellaneous vectors

(test (let ((sum 0)) (for-each (lambda (n) (set! sum (+ sum n))) (vector 1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n (- m)))) (vector 1 2 3) (vector 4 5 6)) sum) -9)
(test (let () (for-each (lambda (n) (error 'wrong-type-arg "oops")) (vector)) #f) #f)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n (- m) (* 2 p)))) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) sum) 33)
(test (let ((sum 0)) (for-each (lambda (n) (for-each (lambda (m) (set! sum (+ sum (* m n)))) (vector 1 2 3))) (vector 4 5 6)) sum) 90)
(test (call/cc (lambda (return) (for-each (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (for-each (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(for-each
 (lambda (data)
   (let ((v data)
	 (c #f)
	 (y 0))
     
     (do ((i 0 (+ i 1)))
	 ((= i 10))
       (set! (v i) i))
     
     (let ((tag 
	    (call/cc
	     (lambda (exit)
	       
	       (for-each
		(lambda (x)
		  
		  (call/cc
		   (lambda (return)
		     (set! c return)))
		  
		  (if (and (even? (inexact->exact x))
			   (> x y) 
			   (< x 10)) 
		      (begin 
			(set! (v (inexact->exact y)) 100)
			(set! y x) 
			(exit x))
		      (set! y x)))
		v)))))
       
       (if (and (number? tag) (< tag 10))
	   (c)))
     
     (let ((correct (vector 0 100 2 100 4 100 6 100 8 9)))
       (do ((i 0 (+ i 1)))
	   ((= i (length v)))
	 (if (not (= (correct i) (inexact->exact (v i))))
	     (format-logged #t ";for-each call/cc data: ~A~%" v))))))
 
 (list (make-vector 10)
       (make-list 10)))


(test (map (lambda (n) (+ 1 n)) (vector 1 2 3)) '(2 3 4))
(test (map (lambda (n m) (- n m)) (vector 1 2 3) (vector 4 5 6)) '(-3 -3 -3))
(test (map (lambda (n m p) (+ n m p)) (vector 1 2 3) (vector 4 5 6) (vector 6 7 8)) '(11 14 17))
(test (map (lambda (n) (map (lambda (m) (* m n)) (vector 1 2 3))) (vector 4 5 6)) '((4 8 12) (5 10 15) (6 12 18)))
(test (call/cc (lambda (return) (map (lambda (n) (return "oops")) (vector 1 2 3)))) "oops")
(test (call/cc (lambda (return) (map (lambda (n) (if (even? n) (return n))) (vector 1 3 8 7 9 10)))) 8)

(test (map (lambda (x) x) (make-int-vector 3 0)) '(0 0 0))
(test (map (lambda (x) x) (let ((v (make-int-vector 3 0))) (set! (v 1) 1) (set! (v 2) 2) v)) '(0 1 2))
(test (map (lambda (x) x) (make-float-vector 3 0.0)) '(0.0 0.0 0.0))
(test (let ((lst ())) (for-each (lambda (n) (set! lst (cons n lst))) (let ((v (make-int-vector 3 0))) (set! (v 1) 1) v)) lst) '(0 1 0))

(test (vector? (symbol-table)) #t)

(let ((v (make-vector 3 (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #t)
  (test (eqv? (v 0) (v 1)) #t))

(let ((v (vector (vector 1 2) (vector 1 2) (vector 1 2))))
  (test (equal? (v 0) (v 1)) #t)
  (test (eq? (v 0) (v 1)) #f)
  (test (eqv? (v 0) (v 1)) #f))

(let ((v (vector (vector (vector (vector 1 2) 3) 4) 5)))
  (test (v 0) #(#(#(1 2) 3) 4))
  (test (v 1) 5)
  (test (((v 0) 0) 1) 3)
  (test ((((v 0) 0) 0) 1) 2))

(test (make-vector 1 (make-vector 1 (make-vector 1 0))) #(#(#(0))))
(test (vector->list (let ((v (make-int-vector 3 0))) (set! (v 0) 32) (set! (v 1) -1) (set! (v 2) 2) (sort! v <))) '(-1 2 32))

(let ((v1 (make-vector 3 1)))
  (num-test (v1 1) 1)
  (set! (v1 1) 2)
  (num-test (v1 1) 2)
  (let ((i0 0)
	(i2 2))
    (num-test (v1 i0) 1)
    (num-test (vector-ref v1 i2) 1)
    (set! (v1 i0) 0)
    (num-test (v1 0) 0)
    (set! (v1 i0) i2)
    (num-test (v1 i0) i2))
  (test (vector-dimensions v1) '(3))
  (set! v1 (make-vector '(3 2)))
  (test (vector-dimensions v1) '(3 2))
  (vector-set! v1 1 1 0)
  (num-test (vector-ref v1 1 1) 0)
  (let ((i0 1)
	(i1 1)
	(i2 32))
    (set! (v1 i0 i1) i2)
    (num-test (vector-ref v1 1 1) 32)
    (num-test (v1 i0 i1) i2)
    (vector-set! v1 0 1 3)
    (num-test (v1 0 1) 3)
    (num-test (v1 1 1) 32))
  (set! v1 (make-vector '(2 4 3) 1))
  (test (vector-dimensions v1) '(2 4 3))      
  (num-test (vector-ref v1 1 1 1) 1)
  (vector-set! v1 0 0 0 32)
  (num-test (v1 0 0 0) 32)
  (set! (v1 0 1 1) 3)
  (num-test (v1 0 1 1) 3))

(for-each
 (lambda (arg)
   (test (vector-dimensions arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol '(1 . 2) '(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))
(test (vector-dimensions) 'error)
(test (vector-dimensions #() #()) 'error)
(test (vector-dimensions #()) '(0))
(test (vector-dimensions (vector)) '(0))
(test (vector-dimensions (vector 0)) '(1))
(test (vector-dimensions (vector-ref #2d((1 2 3) (3 4 5)) 0)) '(3))
(test (vector-dimensions (vector-ref #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0)) '(2 3))
(test (vector-dimensions (vector-ref #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) 0 1)) '(3))
(test (set! (vector-dimensions #(1 2)) 1) 'error)
(test (let ((v #(1 2 3))) (set! (car (vector-dimensions v)) 0) v) #(1 2 3))
(test (vector-dimensions (make-int-vector '(2 3) 0)) '(2 3))

(let-temporarily (((*s7* 'print-length) 32))
  (let ((vect1 #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
	(vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
	(vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
	(vect4 #3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))))
	(vect1t (make-int-vector '(2 2 3) 0)))
    (let ((v (make-shared-vector vect1t '(12))))
      (set! (v 0) 1) (set! (v 1) 2) (set! (v 2) 3) (set! (v 3) 3) (set! (v 4) 4) (set! (v 5) 5) 
      (set! (v 6) 5) (set! (v 7) 6) (set! (v 8) 1) (set! (v 9) 7) (set! (v 10) 8) (set! (v 11) 2))
    (do ((i 1 (+ i 1)))
	((= i 15))
      (set! (*s7* 'print-length) i)
      (let ((str (object->string vect1)))
	(test str (case i
		    ((1) "#3D(((1 ...)...)...)")
		    ((2) "#3D(((1 2 ...)...)...)")
		    ((3) "#3D(((1 2 3)...)...)")
		    ((4) "#3D(((1 2 3) (3 ...))...)")
		    ((5) "#3D(((1 2 3) (3 4 ...))...)")
		    ((6) "#3D(((1 2 3) (3 4 5))...)")
		    ((7) "#3D(((1 2 3) (3 4 5)) ((5 ...)...))")
		    ((8) "#3D(((1 2 3) (3 4 5)) ((5 6 ...)...))")
		    ((9) "#3D(((1 2 3) (3 4 5)) ((5 6 1)...))")
		    ((10) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 ...)))")
		    ((11) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...)))")
		    ((12) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((13) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))")
		    ((14) "#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2)))"))))

      (let ((str (object->string vect1t)))
	(test str (case i
		    ((1) "(make-shared-vector (int-vector 1 ...) '(2 2 3))")
		    ((2) "(make-shared-vector (int-vector 1 2 ...) '(2 2 3))")
		    ((3) "(make-shared-vector (int-vector 1 2 3 ...) '(2 2 3))")
		    ((4) "(make-shared-vector (int-vector 1 2 3 3 ...) '(2 2 3))")
		    ((5) "(make-shared-vector (int-vector 1 2 3 3 4 ...) '(2 2 3))")
		    ((6) "(make-shared-vector (int-vector 1 2 3 3 4 5 ...) '(2 2 3))")
		    ((7) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 ...) '(2 2 3))")
		    ((8) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 ...) '(2 2 3))")
		    ((9) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 ...) '(2 2 3))")
		    ((10) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 7 ...) '(2 2 3))")
		    ((11) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 7 8 ...) '(2 2 3))")
		    ((12) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 7 8 2) '(2 2 3))")
		    ((13) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 7 8 2) '(2 2 3))")
		    ((14) "(make-shared-vector (int-vector 1 2 3 3 4 5 5 6 1 7 8 2) '(2 2 3))"))))


      (let ((str (object->string vect4)))
	(test str (case i
		    ((1) "#3D(((1 ...)...)...)")
		    ((2) "#3D(((1 2)...)...)")
		    ((3) "#3D(((1 2) (3 ...)...)...)")
		    ((4) "#3D(((1 2) (3 4)...)...)")
		    ((5) "#3D(((1 2) (3 4) (5 ...))...)")
		    ((6) "#3D(((1 2) (3 4) (5 6))...)")
		    ((7) "#3D(((1 2) (3 4) (5 6)) ((7 ...)...))")
		    ((8) "#3D(((1 2) (3 4) (5 6)) ((7 8)...))")
		    ((9) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...))")
		    ((10) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10)...))")
		    ((11) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...)))")
		    ((12) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((13) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))")
		    ((14) "#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))"))))

      (let ((str (object->string vect2)))
	(test str (case i
		    ((1) "#2D((1 ...)...)")
		    ((2) "#2D((1 2 ...)...)")
		    ((3) "#2D((1 2 3 ...)...)")
		    ((4) "#2D((1 2 3 4 ...)...)")
		    ((5) "#2D((1 2 3 4 5 ...)...)")
		    ((6) "#2D((1 2 3 4 5 6)...)")
		    ((7) "#2D((1 2 3 4 5 6) (7 ...))")
		    ((8) "#2D((1 2 3 4 5 6) (7 8 ...))")
		    ((9) "#2D((1 2 3 4 5 6) (7 8 9 ...))")
		    ((10) "#2D((1 2 3 4 5 6) (7 8 9 10 ...))")
		    ((11) "#2D((1 2 3 4 5 6) (7 8 9 10 11 ...))")
		    ((12) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((13) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))")
		    ((14) "#2D((1 2 3 4 5 6) (7 8 9 10 11 12))"))))

      (let ((str (object->string vect3)))
	(test str (case i
		    ((1) "#(1 ...)")
		    ((2) "#(1 2 ...)")
		    ((3) "#(1 2 3 ...)")
		    ((4) "#(1 2 3 4 ...)")
		    ((5) "#(1 2 3 4 5 ...)")
		    ((6) "#(1 2 3 4 5 6 ...)")
		    ((7) "#(1 2 3 4 5 6 7 ...)")
		    ((8) "#(1 2 3 4 5 6 7 8 ...)")
		    ((9) "#(1 2 3 4 5 6 7 8 9 ...)")
		    ((10) "#(1 2 3 4 5 6 7 8 9 10 ...)")
		    ((11) "#(1 2 3 4 5 6 7 8 9 10 11 ...)")
		    ((12) "#(1 2 3 4 5 6 7 8 9 10 11 12 ...)")
		    ((13) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)")
		    ((14) "#(1 2 3 4 5 6 7 8 9 10 11 12 13 14)")))))

    (let ((vect5 (make-vector '(2 3))))
      (set! (vect5 0 0) vect1)
      (set! (vect5 0 1) vect2)
      (set! (vect5 0 2) vect3)
      (set! (vect5 1 0) vect4)
      (set! (vect5 1 1) (vector 1 2 3))
      (set! (vect5 1 2) #2d())

      (do ((i 1 (+ i 1)))
	  ((= i 15))
	(set! (*s7* 'print-length) i)
	(let ((str (object->string vect5)))
	  (test str (case i

		      ((1) "#2D((#3D(((1 ...)...)...) ...)...)")
		      ((2) "#2D((#3D(((1 2 ...)...)...) #2D((1 2 ...)...) ...)...)")
		      ((3) "#2D((#3D(((1 2 3)...)...) #2D((1 2 3 ...)...) #(1 2 3 ...))...)")
		      ((4) "#2D((#3D(((1 2 3) (3 ...))...) #2D((1 2 3 4 ...)...) #(1 2 3 4 ...)) (#3D(((1 2) (3 4)...)...) ...))")
		      ((5) "#2D((#3D(((1 2 3) (3 4 ...))...) #2D((1 2 3 4 5 ...)...) #(1 2 3 4 5 ...)) (#3D(((1 2) (3 4) (5 ...))...) #(1 2 3) ...))")
		      ((6) "#2D((#3D(((1 2 3) (3 4 5))...) #2D((1 2 3 4 5 6)...) #(1 2 3 4 5 6 ...)) (#3D(((1 2) (3 4) (5 6))...) #(1 2 3) #2D()))")
		      ((7) "#2D((#3D(((1 2 3) (3 4 5)) ((5 ...)...)) #2D((1 2 3 4 5 6) (7 ...)) #(1 2 3 4 5 6 7 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 ...)...)) #(1 2 3) #2D()))")
		      ((8) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 ...)...)) #2D((1 2 3 4 5 6) (7 8 ...)) #(1 2 3 4 5 6 7 8 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8)...)) #(1 2 3) #2D()))")
		      ((9) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1)...)) #2D((1 2 3 4 5 6) (7 8 9 ...)) #(1 2 3 4 5 6 7 8 9 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 ...)...)) #(1 2 3) #2D()))")
		      ((10) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 ...))) #2D((1 2 3 4 5 6) (7 8 9 10 ...)) #(1 2 3 4 5 6 7 8 9 10 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10)...)) #(1 2 3) #2D()))")
		      ((11) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 ...))) #2D((1 2 3 4 5 6) (7 8 9 10 11 ...)) #(1 2 3 4 5 6 7 8 9 10 11 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 ...))) #(1 2 3) #2D()))")
		      ((12) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))")
		      ((13) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 ...)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))")
		      ((14) "#2D((#3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))) #2D((1 2 3 4 5 6) (7 8 9 10 11 12)) #(1 2 3 4 5 6 7 8 9 10 11 12 13 14)) (#3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12))) #(1 2 3) #2D()))"))))))))
  
(test (object->string (make-int-vector 3 0)) "(int-vector 0 0 0)")

(let ((v (make-vector '(2 2))))
  (set! (v 0 0) 1)
  (set! (v 0 1) 2)
  (set! (v 1 0) 3)
  (set! (v 1 1) 4)
  (set! (v 0 1) #2d((1 2) (3 4)))
  (test (object->string v) "#2D((1 #2D((1 2) (3 4))) (3 4))"))

(let ((v #2d((1 2) (3 4)))) 
  (set! (v 0 1) #2d((1 2) (3 4))) 
  (test (object->string v) "#2D((1 #2D((1 2) (3 4))) (3 4))"))

(let ((v (make-vector '(2 3))))
  (do ((i 0 (+ i 1)))
      ((= i 2))
    (do ((j 0 (+ j 1)))
	((= j 3))
      (set! (v i j) (list i j))))
  (test (v 0 0) '(0 0))
  (test ((v 1 2) 0) 1)
  (test (v 1 2 0) 1)
  (test (v 1 2 0 0) 'error)
  (test (object->string v) "#2D(((0 0) (0 1) (0 2)) ((1 0) (1 1) (1 2)))"))

(test (object->string (make-float-vector 3 1.0)) "(float-vector 1.0 1.0 1.0)")
(test (object->string (make-float-vector 3 -1.5)) "(float-vector -1.5 -1.5 -1.5)")
(test (object->string (make-int-vector 3 1)) "(int-vector 1 1 1)")
(test (object->string (make-int-vector 3 -1)) "(int-vector -1 -1 -1)")
(test (object->string (make-int-vector 0 0)) "#()")
(test (object->string (make-float-vector '(3 2 0) 0.0)) "#()")

(test (let ((v1 (make-vector '(3 2) 1))
	    (v2 (make-vector '(3 2) 2))
	    (sum 0))
	(for-each (lambda (n m) (set! sum (+ sum n m))) v1 v2)
	sum)
      18)
(test (vector->list (make-vector '(2 3) 1)) '(1 1 1 1 1 1))
(test (vector->list #2d((1 2) (3 4))) '(1 2 3 4))
(test (list->vector '((1 2) (3 4))) #((1 2) (3 4)))
(test (vector->list (make-vector (list 2 0))) ())
(test (vector-dimensions #2d((1 2 3))) '(1 3))

(test (#2d((1 2 3) (4 5 6)) 0 0) 1)
(test (#2d((1 2 3) (4 5 6)) 0 1) 2)
(test (#2d((1 2 3) (4 5 6)) 1 1) 5)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 0 0 0) 1)
(test (#3D(((1 2) (3 4)) ((5 6) (7 8))) 1 1 0) 7)
(test (#4d((((1) (2)) ((3) (4)) ((5) (6)))) 0 0 0 0) 1)
(test (vector? #2d((1 2) (3 4))) #t)
(test ((#2d((1 #2d((2 3) (4 5))) (6 7)) 0 1) 1 0) 4)
(test ((((((((((#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0) 0) 0) 0) 0) 0) 0) 0) 0) 0) 1)
(test (#10D((((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))))) (((((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1))))))) (((((((1) (1))) (((1) (1))))) (((((1) (1))) (((1) (1)))))))))) 0 0 0 0 0 0 0 0 0 0) 1)
(let ((v (make-vector (make-list 100 1) 0)))
  (test (equal? v #100D((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((0))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) #t)
  (test (apply v (make-list 100 0)) 0)
  (test (v 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 0))

;; eval-string here else these are read errors
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5) (7 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) (() (7 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 8 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) (5 (7 8 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 . 8)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) (7 8 . 9)))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6) ()))") 'error)
(test (eval-string "#3D(((1 2) (3 4)) ((5 6)))") 'error)

(test (vector-dimensions #3D(((1 2) (3 4)) ((5 6) (7 8)))) '(2 2 2))
(test (vector-dimensions #2d((1 2 3) (4 5 6))) '(2 3))
(test (vector-dimensions #4d((((1) (2)) ((3) (4)) ((5) (6))))) '(1 3 2 1))

(test (vector-length #3D(((1 2) (3 4)) ((5 6) (7 8)))) 8)
(test (length #2d((1 2 3) (4 5 6))) 6)

(test (#2d((1 (2) 3) (4 () 6)) 0 1) '(2))
(test (#2d((1 (2) 3) (4 () 6)) 1 1) ())
(test (#2d((1 (2) 3) (4 6 ())) 1 2) ())
(test (#2d((() (2) ()) (4 5 6)) 0 2) ())

(test (equal? (make-vector 0) (make-vector '(0))) #t)
(test (equal? #() (make-vector '(0))) #t)

(test (equal? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #t)
(test (eq? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (eqv? #2d((1 2) (3 4)) #2D((1 2) (3 4))) #f)
(test (make-vector (1 . 2) "hi") 'error)
(test (make-vector (cons 1 2) "hi") 'error)
(test (equal? (make-vector 0) (vector)) #t)
(test (equal? #() (vector)) #t)
(test (equal? (make-int-vector 0 0) (make-int-vector 0 0)) #t)
(test (equal? #() (make-int-vector 0 0)) #t)
(test (equal? (make-vector '(2 0)) (make-int-vector '(2 0) 0)) #t)
(test (equal? (make-vector '(2 0)) (make-int-vector '(0 2) 0)) #f)

(let ((v (make-vector '(2 3) 0)))
  (num-test (vector-length v) 6)
  (test (vector-dimensions v) '(2 3))
  (num-test (v 0 0) 0)
  (num-test (v 1 2) 0)
  (test (v 2 2) 'error)
  (test (v 2 -1) 'error)
  (test (v 2 0) 'error)
  (set! (v 0 1) 1)
  (num-test (v 0 1) 1)
  (num-test (v 1 0) 0)
  (set! (v 1 2) 2)
  (num-test (v 1 2) 2)
  (test (set! (v 2 2) 32) 'error)
  (test (set! (v 1 -1) 0) 'error)
  (test (set! (v 2 0) 0) 'error)
  (num-test (vector-ref v 0 1) 1)
  (num-test (vector-ref v 1 2) 2)
  (test (vector-ref v 2 2) 'error)
  (test (vector-ref v 1 -1) 'error)
  (vector-set! v 1 1 64)
  (num-test (vector-ref v 1 1) 64)
  (num-test (vector-ref v 0 0) 0)
  (test (vector-ref v 1 2 3) 'error)
  (test (vector-set! v 1 2 3 4) 'error)
  (test (v 1 1 1) 'error)
  (test (set! (v 1 1 1) 1) 'error))


(let ((v1 (make-vector '(3 2) 0))
      (v2 (make-vector '(2 3) 0))
      (v3 (make-vector '(2 3 4) 0))
      (v4 (make-vector 6 0))
      (v5 (make-vector '(2 3) 0)))
  (test (equal? v1 v2) #f)
  (test (equal? v1 v3) #f)
  (test (equal? v1 v4) #f)
  (test (equal? v2 v2) #t)
  (test (equal? v3 v2) #f)
  (test (equal? v4 v2) #f)
  (test (equal? v5 v2) #t)
  (test (equal? v4 v3) #f)
  (test (vector-dimensions v3) '(2 3 4))
  (test (vector-dimensions v4) '(6))
  (num-test (v3 1 2 3) 0)
  (set! (v3 1 2 3) 32)
  (num-test (v3 1 2 3) 32)
  (num-test (vector-length v3) 24)
  (num-test (vector-ref v3 1 2 3) 32)
  (vector-set! v3 1 2 3 -32)
  (num-test (v3 1 2 3) -32)
  (test (v3 1 2) #(0 0 0 -32))
  (test (set! (v3 1 2) 3) 'error)
  (test (vector-ref v3 1 2) #(0 0 0 -32))
  (test (vector-set! v3 1 2 32) 'error))

(test (let ((v #2d((1 2) (3 4)))) (vector-fill! v #t) v) #2D((#t #t) (#t #t)))

(test (eval-string "#2d((1 2) #2d((3 4) 5 6))") 'error)
(test (string=? (object->string #2d((1 2) (3 #2d((3 4) (5 6))))) "#2D((1 2) (3 #2D((3 4) (5 6))))") #t)
(test (string=? (object->string #3d(((#2d((1 2) (3 4)) #(1)) (#3d(((1))) 6)))) "#3D(((#2D((1 2) (3 4)) #(1)) (#3D(((1))) 6)))") #t)

(test (make-vector '(2 -2)) 'error)
(test (make-vector '(2 1/2)) 'error)
(test (make-vector '(2 1.2)) 'error)
(test (make-vector '(2 2+i)) 'error)
(test (make-vector '(2 "hi")) 'error)

(let ((v (make-vector '(1 1 1) 32)))
  (test (vector? v) #t)
  (test (equal? v #()) #f)
  (test (vector->list v) '(32))
  (test (vector-ref v 0) '#2D((32)))
  (test (vector-set! v 0 0) 'error)
  (test (vector-ref v 0 0) #(32))
  (test (vector-set! v 0 0 0) 'error)
  (test (vector-ref v 0 0 0) 32)
  (test (let () (vector-set! v 0 0 0 31) (vector-ref v 0 0 0)) 31)
  (test (vector-length v) 1)
  (test (vector-dimensions v) '(1 1 1))
  (test (object->string v) "#3D(((31)))")
  )

(test (vector? #3D(((32)))) #t)
(test (equal? #3D(((32))) #()) #f)
(test (vector->list #3D(((32)))) '(32))
(test (#3D(((32))) 0) '#2D((32)))
(test (set! (#3D(((32))) 0) 0) 'error)
(test (#3D(((32))) 0 0) #(32))
(test (set! (#3D(((32))) 0 0) 0) 'error)
(test (#3D(((32))) 0 0 0) 32)
(test (vector-length #3D(((32)))) 1)
(test (vector-dimensions #3D(((32)))) '(1 1 1))
(test (object->string #3D(((32)))) "#3D(((32)))")


(let ((v1 (make-vector '(1 0))))
  (test (vector? v1) #t)
  (test (equal? v1 #()) #f)
  (test (vector->list v1) ())
  (test (vector-ref v1 0) 'error)
  (test (vector-set! v1 0 0) 'error)
  (test (vector-ref v1 0 0) 'error)
  (test (vector-set! v1 0 0 0) 'error)
  (test (vector-length v1) 0)
  (test (vector-dimensions v1) '(1 0))
  (test (object->string v1) "#2D()")
  )

(let ((v2 (make-vector '(10 3 0))))
  (test (vector? v2) #t)
  (test (equal? v2 #()) #f)
  (test (vector->list v2) ())
  (test (vector-ref v2) 'error)
  (test (vector-set! v2 0) 'error)
  (test (vector-ref v2 0) 'error)
  (test (vector-set! v2 0 0) 'error)
  (test (vector-ref v2 0 0) 'error)
  (test (vector-set! v2 0 0 0) 'error)
  (test (vector-ref v2 1 2 0) 'error)
  (test (vector-set! v2 1 2 0 0) 'error)
  (test (vector-length v2) 0)
  (test (vector-dimensions v2) '(10 3 0))
  (test (object->string v2) "#3D()")
  )

(let ((v3 (make-vector '(10 0 3))))
  (test (vector? v3) #t)
  (test (equal? v3 #()) #f)
  (test (vector->list v3) ())
  (test (vector-ref v3) 'error)
  (test (vector-set! v3 0) 'error)
  (test (vector-ref v3 0) 'error)
  (test (vector-set! v3 0 0) 'error)
  (test (vector-ref v3 0 0) 'error)
  (test (vector-set! v3 0 0 0) 'error)
  (test (vector-ref v3 1 0 2) 'error)
  (test (vector-set! v3 1 0 2 0) 'error)
  (test (vector-length v3) 0)
  (test (vector-dimensions v3) '(10 0 3))
  (test (object->string v3) "#3D()")
  )

(test (((#(("hi") ("ho")) 0) 0) 1) #\i)
(test (string-ref (list-ref (vector-ref #(("hi") ("ho")) 0) 0) 1) #\i)

(test (equal? #2D() (make-vector '(0 0))) #t)
(test (equal? #2D() (make-vector '(1 0))) #f)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (2 2))) #t)
(test (equal? (make-vector '(2 2) 2) #2D((2 2) (1 2))) #f)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 2 3) 0)) #t)
(test (equal? (make-vector '(1 2 3) 0) (make-vector '(1 3 2) 0)) #f)
(test (make-vector '1 2 3) 'error)

(test (set! (vector) 1) 'error)
(test (set! (make-vector 1) 1) 'error)
(test (equal? (make-vector 10 ()) (make-hash-table 10)) #f)
(test (equal? #() (copy #())) #t)
(test (equal? #2d() (copy #2d())) #t)
(test (fill! #() 1) 1)
(test (fill! #2d() 1) 1)

(test (equal? #2d((1 2) (3 4)) (copy #2d((1 2) (3 4)))) #t)
(test (equal? #3d() #3d(((())))) #f)
(test (equal? #3d() #3d()) #t)
(test (equal? #1d() #1d()) #t)
(test (equal? #3d() #2d()) #f)
(test (equal? #3d() (copy #3d())) #t)
(test (equal? #2d((1) (2)) #2d((1) (3))) #f)
(test (equal? #2d((1) (2)) (copy #2d((1) (2)))) #t)
(test (equal? (make-vector '(3 0 1)) (make-vector '(3 0 2))) #f)
(test (eval-string "#0d()") 'error)

(let ((v #2d((1 2 3) (4 5 6))))
  (let ((v1 (v 0))
	(v2 (v 1)))
    (if (not (equal? v1 #(1 2 3)))
	(format-logged #t ";(v 0) subvector: ~A~%" v1))
    (if (not (equal? v2 #(4 5 6)))
	(format-logged #t ";(v 1) subvector: ~A~%" v2))
    (let ((v3 (copy v1)))
      (if (not (equal? v3 #(1 2 3)))
	  (format-logged #t ";(v 0) copied subvector: ~A~%" v3))
      (if (not (= (length v3) 3))
	  (format-logged #t ";(v 0) copied length: ~A~%" (length v3)))
      (if (not (equal? v3 (copy (v 0))))
	  (format-logged #t ";(v 0) copied subvectors: ~A ~A~%" v3 (copy (v 0)))))))

(let ((v1 (make-vector '(3 2 1) #f))
      (v2 (make-vector '(3 2 1) #f)))
  (test (equal? v1 v2) #t)
  (set! (v2 0 0 0) 1)
  (test (equal? v1 v2) #f))
(test (equal? (make-vector '(3 2 1) #f) (make-vector '(1 2 3) #f)) #f)

(test (map (lambda (n) n) #2d((1 2) (3 4))) '(1 2 3 4))
(test (let ((vals ())) (for-each (lambda (n) (set! vals (cons n vals))) #2d((1 2) (3 4))) vals) '(4 3 2 1))
(test (map (lambda (x y) (+ x y)) #2d((1 2) (3 4)) #1d(4 3 2 1)) '(5 5 5 5))
(test (let ((vals ())) (for-each (lambda (x y) (set! vals (cons (+ x y) vals))) #2d((1 2) (3 4)) #1d(4 3 2 1)) vals) '(5 5 5 5))

(let ((v #2D((#(1 2) #(3 4)) (#2d((5 6) (7 8)) #2D((9 10 11) (12 13 14))))))
  (test (v 0 0) #(1 2))
  (test (v 0 1) #(3 4))
  (test (v 1 0) #2d((5 6) (7 8)))
  (test (v 1 1) #2D((9 10 11) (12 13 14)))
  (test ((v 1 0) 0 1) 6)
  (test ((v 0 1) 1) 4)
  (test ((v 1 1) 1 2) 14))

(let ((v #2D((#((1) #(2)) #(#(3) (4))) (#2d(((5) #(6)) (#(7) #(8))) #2D((#2d((9 10) (11 12)) (13)) (14 15))))))
  (test (v 0 0) #((1) #(2)))
  (test (v 0 1) #(#(3) (4)))
  (test (v 1 0) #2D(((5) #(6)) (#(7) #(8))))
  (test (v 1 1) #2D((#2D((9 10) (11 12)) (13)) (14 15)))
  (test ((v 1 0) 0 1) #(6))
  (test (((v 1 0) 0 1) 0) 6)
  (test ((v 0 1) 1) '(4))
  (test (((v 1 1) 0 0) 1 0) 11))


(test (let ((V #2D((1 2 3) (4 5 6)))) (V 0)) #(1 2 3))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 1)) #(4 5 6))
(test (let ((V #2D((1 2 3) (4 5 6)))) (V 2)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (set! (V 1) 0)) 'error)
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 1) 32) V)) '#2D((1 32 3) (4 5 6)))
(test (let ((V #2D((1 2 3) (4 5 6)))) (let ((V1 (V 0))) (set! (V1 3) 32) V)) 'error)

(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1)) '#2D((7 8 9) (10 11 12)))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 1 1)) #(10 11 12))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 0 1)) #(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (V 2 1)) 'error)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) ((V 0) 1)) #(4 5 6))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (set! (((V 0) 1) 1) 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 0 1 1 32) V) '#3D(((1 2 3) (4 32 6)) ((7 8 9) (10 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-set! V 1 1 0 32) V) '#3D(((1 2 3) (4 5 6)) ((7 8 9) (32 11 12))))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 1))) 6)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 1))) '(2 3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-length (V 0 1))) 3)
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12))))) (vector-dimensions (V 0 1))) '(3))
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (one 1) (zero 0)) 
	(let ((V1 (V one zero))
	      (sum 0))
	  (for-each (lambda (n) (set! sum (+ sum n))) V1)
	  sum))
      24) ; 7 8 9
(test (let ((V '#3D(((1 2 3) (4 5 6)) ((7 8 9) (10 11 12)))) (two 2) (one 1) (zero 0)) 
	(let ((V10 (V one zero))
	      (V00 (V zero zero))
	      (V01 (V zero one))
	      (V11 (V one one))
	      (sum 0))
	  (for-each (lambda (n0 n1 n2 n3) (set! sum (+ sum n0 n1 n2 n3))) V00 V01 V10 V11)
	  sum))
      78)

(let-temporarily (((*s7* 'print-length) 32))
  (test (object->string (make-vector '(8 8) 0)) "#2D((0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0) (0 0 0 0 0 0 0 0)...)")
  (test (object->string (make-vector 64 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector 32 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0)")
  (test (object->string (make-vector 33 0)) "#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ...)")
  (test (object->string (make-vector '(8 4) 0)) "#2D((0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0) (0 0 0 0))"))

(let-temporarily (((*s7* 'print-length) 1024))
  (test (object->string (make-vector '(2 1 2 1 2 1 2 1 2 1 2 1 2 1) 0)) "#14D((((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))) (((((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))) (((((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))) (((((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))) (((((((0) (0))) (((0) (0))))) (((((0) (0))) (((0) (0))))))))))))))")

  (test (object->string (make-vector '(16 1 1 1 1 1 1 1 1 1 1 1 1 1) 0)) "#14D((((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))) (((((((((((((0))))))))))))))")

;;; now see if our shared vector has survived...
  (test (and (vector? check-shared-vector-after-gc)
	     (= (length check-shared-vector-after-gc) 6)
	     (do ((i 0 (+ i 1))
		  (happy #t))
		 ((= i 6) happy)
	       (if (or (not (pair? (check-shared-vector-after-gc i)))
		       (not (equal? (check-shared-vector-after-gc i) (cons 3 i))))
		   (set! happy #f))))
	#t)
  (set! check-shared-vector-after-gc #f))



;;; -------- circular structures --------

;;; here's an oddity:

(let ((l1 (make-list 1 #f))
      (l2 (make-list 3 #f)))
  (set-cdr! l1 l1)
  (set-cdr! (list-tail l2 2) l2)
  (test (equal? l1 l2) #t))  ; but (eq? l1 (cdr l1)): #t, and (eq? l2 (cdr l2)): #f

(let ((l1 (make-list 1 #f))
      (l2 (make-list 3 #f)))
  (set-car! l1 #t)
  (set-car! l2 #t)
  (set-cdr! l1 l1)
  (set-cdr! (list-tail l2 2) l2)
  (test (equal? l1 l2) #f))

;;; Guile agrees on the first, but hangs on the second
;;; CL says the first is false, but hangs on the second
;;; r7rs agrees with s7 here, to my dismay.

;;; other cases:
(let ((l1 (list #f #f))
      (l2 (list #f #f)))
  (set-cdr! l1 l1)
  (set-cdr! (cdr l2) l2)
  (test (equal? l1 l2) #t))

(let ((l1 (list #f #f #f))
      (l2 (list #f #f #f)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) l2)
  (test (equal? l1 l2) #f)) ; r7rs says #t I think

(let ((l1 (list #f #f #f))
      (l2 (list #f #f #f)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) (cdr l2))
  (test (equal? l1 l2) #t))

;;; Gauche says #t #f #t #t #t, as does chibi
;;; Guile-2.0 hangs on all, as does Chicken

(let ((l1 (list #t #f #f))
      (l2 (list #t #f #t)))
  (set-cdr! (cdr l1) l1)
  (set-cdr! (cddr l2) (cdr l2))
  (test (equal? l1 l2) #t))

(let ((l1 (list #t #f #f))
      (l2 (list #t #f #f #t)))
  (set-cdr! (cddr l1) l1)
  (set-cdr! (cdddr l2) (cdr l2))
  (test (equal? l1 l2) #t))


;;; cyclic-sequences

(define* (make-circular-list n init)
  (let ((l (make-list n init)))
    (set-cdr! (list-tail l (- n 1)) l)))

(define (cyclic? obj) (not (null? (cyclic-sequences obj))))

(for-each
 (lambda (arg)
   (test (cyclic? arg) #f))
  (list "hi" "" #\null #\a () #() 1 '(1 . 2) (cons #\a #\b) #f 'a-symbol abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #t :hi (if #f #f) (lambda (a) (+ a 1))
       (let ((x '(1 2))) (list x x))
       (let ((x #(1 2))) (vector x x))
       (let ((x "a")) (list (vector x) x))
       (let ((x (hash-table '(a . 1)))) (vector x (list x x) (inlet 'b x)))
       (let ((x '(1))) (let ((y (list x))) (list x (list y))))))

(test (cyclic-sequences) 'error)
(test (cyclic-sequences (list 1 2) (list 3 4)) 'error)

(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences y))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences (vector y)))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let ((y (make-circular-list 3))) (let ((x (cyclic-sequences (list y (vector y))))) (list (length x) (eq? (car x) y)))) '(1 #t))
(test (let* ((y (list 1)) (x (vector y))) (set! (y 0) x) (eq? x (car (cyclic-sequences x)))) #t)
(test (let ((x (hash-table (cons 'a (make-circular-list 1))))) (eq? (car (cyclic-sequences x)) (x 'a))) #t)
(test (let ((x (list (make-circular-list 1) (make-circular-list 2)))) (length (cyclic-sequences x))) 2)
(test (let ((l1 '(1))) (let ((l2 (cons l1 l1))) (cyclic-sequences l2))) ())
(test (let ((l1 '(1))) (let ((l2 (list l1 l1))) (cyclic-sequences l2))) ())
(test (let ((y '(1))) 
	(let ((x (list (make-circular-list 1) y y)))
	  (set-cdr! (cddr x) (cdr x))
	  (let ((z (cyclic-sequences x)))
	    (list (length z) (and (memq (cdr x) z) #t))))) ; "and" here just to make the result easier to check
      '(2 #t))
(test (let ((z (vector 1 2)))
	(let ((y (list 1 z 2)))
	  (let ((x (hash-table (cons 'x y))))
	    (set! (z 1) x)
	    (length (cyclic-sequences z)))))
      1)
(test (let ((x '(1 2)))
	(let ((y (list x x)))
	  (let ((z (vector x y)))
	    (null? (cyclic-sequences z)))))
      #t)
(test (let ((v (vector 1 2 3 4)))
	(let ((lst (list 1 2)))
	  (set-cdr! (cdr lst) lst)
	  (set! (v 0) v)
	  (set! (v 3) lst) 
	  (length (cyclic-sequences v))))
      2)

(test (infinite? (length (make-circular-list 3))) #t)
(test (object->string (make-circular-list 3)) "#1=(#f #f #f . #1#)")

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error)
   (test (cyclic? lst) #t)
   (test (eq? (car (cyclic-sequences lst)) lst) #t))

(let ((l1 (list 1)))
  (test (object->string (list l1 1 l1)) "((1) 1 (1))") ; was "(#1=(1) 1 #1#)"
  (test (cyclic? (list l1 1 l1)) #f))

(let ((lst (list 1 2)))
   (set! (cdr (cdr lst)) (cdr lst))
   (test (object->string lst) "(1 . #1=(2 . #1#))")
   (test (object->string lst :readable) "(let (({1} #f)) (let (({lst} (make-list 2))) (let (({x} {lst})) (set-car! {x} 1) (set! {x} (cdr {x})) (set-car! {x} 2) (set-cdr! {x} (set! {1} (let (({lst} (make-list 1))) (set! {1} {lst}) (let (({x} {lst})) (set-car! {x} 2) (set-cdr! {x} {1}) ) {lst}))) ) {lst}))"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (append '(1) lst)) "(1 . #1=(1 2 3 . #1#))"))
(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (append lst ()) 'error)) 

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (sort! lst <) 'error))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (list lst)) "(#1=(1 2 3 . #1#))"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (make-list 4 lst)) "(#1=(1 2 3 . #1#) #1# #1# #1#)"))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (object->string (vector lst lst)) "#(#1=(1 2 3 . #1#) #1#)"))

(let ((lst `(+ 1 2 3)))
   (set! (cdr (cdddr lst)) (cddr lst)) 
   (test (object->string lst) "(+ 1 . #1=(2 3 . #1#))"))


(let ((x (list 1 2)))
  (test (equal? x x) #t)
  (test (equal? x (cdr x)) #f)
  (test (equal? x ()) #f))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5))))))
  (test (equal? x y) #t))
(let ((x (list 1 (list 2 3) (list (list 4 (list 5)))))
      (y (list 1 (list 2 3) (list (list 4 (list 5) 6)))))
  (test (equal? x y) #f))

(test (length ()) 0)
(test (length (cons 1 2)) -1)
(test (length '(1 2 3)) 3)

(let ((lst1 (list 1 2))) 
  (test (length lst1) 2)
  (list-set! lst1 0 lst1)
  (test (length lst1) 2) ; its car is a circular list, but it isn't
  (test (eq? ((cyclic-sequences lst1) 0) lst1) #t)
  (test (list->string lst1) 'error)
  (let ((lst2 (list 1 2)))
    (set-car! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (morally-equal? lst1 lst2) #t)
    (test (eq? lst1 lst2) #f)
    (test (eqv? lst1 lst2) #f)
    (test (pair? lst1) #t)
    (test (null? lst1) #f)
    (test (car lst2) lst2)
    (test (car lst1) lst1)
    (test (let ()
	    (fill! lst1 32)
	    lst1)
	  '(32 32))))

(let ((lst1 (list 1))) 
  (test (length lst1) 1)
  (set-cdr! lst1 lst1)
  (test (infinite? (length lst1)) #t)
  (test (eq? (cdr ((cyclic-sequences lst1) 0)) lst1) #t)
  (test (null? lst1) #f)
  (test (pair? lst1) #t)
  (let ((lst2 (cons 1 ())))
    (set-cdr! lst2 lst2)
    (test (equal? lst1 lst2) #t)
    (test (morally-equal? lst1 lst2) #t)
    (set-car! lst2 0)
    (test (equal? lst1 lst2) #f)
    (test (morally-equal? lst1 lst2) #f)
    (test (infinite? (length lst2)) #t)))

(let ((lst1 (list 1))) 
  (set-cdr! lst1 lst1)
  (test (list-tail lst1 0) lst1)
  (test (list-tail lst1 3) lst1)
  (test (list-tail lst1 10) lst1))

(let ((lst1 (let ((lst (list 'a))) 
	      (set-cdr! lst lst)
	      lst)))
  (test (apply lambda lst1 (list 1)) 'error)) ; lambda parameter 'a is used twice in the lambda argument list !

(let ((lst1 (list 1))
      (lst2 (list 1)))
  (set-car! lst1 lst2)
  (set-car! lst2 lst1)
  (test (equal? lst1 lst2) #t)
  (test (morally-equal? lst1 lst2) #t)
  (test (length lst1) 1)
  (let ((lst3 (list 1)))
    (test (equal? lst1 lst3) #f)
    (test (morally-equal? lst1 lst3) #f)
    (set-cdr! lst3 lst3)
    (test (equal? lst1 lst3) #f)
    (test (morally-equal? lst1 lst3) #f)))

(let ((lst1 (list 'a 'b 'c)))
  (set! (cdr (cddr lst1)) lst1)
  (test (infinite? (length lst1)) #t)
  (test (memq 'd lst1) #f)
  (test (memq 'a lst1) lst1)
  (test (memq 'b lst1) (cdr lst1)))

(let ((lst1 (list 1 2 3)))
  (list-set! lst1 1 lst1)
  (test (object->string lst1) "#1=(1 #1# 3)"))

(let ((lst1 (let ((lst (list 1))) 
	      (set-cdr! lst lst)
	      lst)))
  (test (list-ref lst1 9223372036854775807) 'error)
  (test (list-set! lst1 9223372036854775807 2) 'error)
  (test (list-tail lst1 9223372036854775807) 'error)
  (test (make-vector lst1 9223372036854775807) 'error)
  (let-temporarily (((*s7* 'safety) 1))
    (test (not (member (map (lambda (x) x) lst1) (list () '(1)))) #f) ; geez -- just want to allow two possible ok results
    (test (not (member (map (lambda (x y) x) lst1 lst1) (list () '(1)))) #f)
    (test (for-each (lambda (x) x) lst1) #<unspecified>) ; was 'error
    (test (for-each (lambda (x y) x) lst1 lst1) #<unspecified>) ; was 'error
    (test (not (member (map (lambda (x y) (+ x y)) lst1 '(1 2 3)) (list () '(2)))) #f)))

(let ((lst1 (list 1 -1)))
  (set-cdr! (cdr lst1) lst1)
  (let ((vals (map * '(1 2 3 4) lst1)))
    (test vals '(1)))) ; was '(1 -2 3 -4) -- as in other cases above, map/for-each stop when a cycle is encountered

(test (let ((lst '(a b c)))
	(set! (cdr (cddr lst)) lst)
	(map cons lst '(0 1 2 3 4 5)))
      '((a . 0) (b . 1) (c . 2)))

(test (object->string (let ((l1 (list 0 1))) (set! (l1 1) l1) (copy l1))) "(0 #1=(0 #1#))")

;;; this changed 11-Mar-15
;;;(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "(1 . #1=(1 . #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) (copy lst))) "#1=(1 . #1#)") 
(test (object->string (let ((lst (list 1 2))) (set! (cdr lst) lst) lst))        "#1=(1 . #1#)")

(test (object->string (let ((l1 (list 1 2))) (copy (list l1 4 l1)))) "((1 2) 4 (1 2))") ; was "(#1=(1 2) 4 #1#)"
;;;(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 2 3 . #1=(2 3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) (copy lst))) "(1 . #1=(2 3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cdr lst)) lst))        "(1 . #1=(2 3 . #1#))")

(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cddr lst)) (copy lst))) "(1 2 . #1=(3 . #1#))")
(test (object->string (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) (cddr lst)) lst))        "(1 2 . #1=(3 . #1#))")

;;;(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) (copy lst))) "(1 2 3 4 . #1=(3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) (copy lst))) "(1 2 . #1=(3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cddr lst)) lst))        "(1 2 . #1=(3 4 . #1#))")

;;;(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) (copy lst))) "(1 2 3 4 . #1=(2 3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) (copy lst))) "(1 . #1=(2 3 4 . #1#))")
(test (object->string (let ((lst (list 1 2 3 4))) (set! (cdr (cdddr lst)) (cdr lst)) lst))        "(1 . #1=(2 3 4 . #1#))")

(test (object->string (vector (let ((lst (list 1))) (set-cdr! lst lst)))) "#(#1=(1 . #1#))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (set! (car lst) (vector lst)) lst)) "#1=(#(#1#) 2 . #1#)")

;; these are ugly!
(test (object->string (vector (let ((lst (list 1))) (set-cdr! lst lst))) :readable)
      "(let (({1} #f)) (vector (set! {1} (let (({lst} (make-list 1))) (set! {1} {lst}) (let (({x} {lst})) (set-car! {x} 1) (set-cdr! {x} {1}) ) {lst}))))")
(test (object->string (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (set! (car lst) (vector lst)) lst) :readable)
      "(let (({1} #f)) (set! {1} (let (({lst} (make-list 2))) (set! {1} {lst}) (let (({x} {lst})) (set-car! {x} (vector {1})) (set! {x} (cdr {x})) (set-car! {x} 2) (set-cdr! {x} {1}) ) {lst})))")
(test (let ((v (vector 1 2))) (set! (v 0) v) (object->string v :readable)) 
      "(let (({1} #f)) (set! {1} (let (({v} (make-vector 2))) (set! {1} {v}) (set! ({v} 0) {1}) (set! ({v} 1) 2) {v})))")
(test (let ((v (make-vector '(2 2) 0))) (set! (v 1 1) v) (object->string v :readable))
      "(let (({1} #f)) (set! {1} (let (({v} (make-vector '(2 2 )))) (set! {1} {v}) (set! ({v} 0 0) 0) (set! ({v} 0 1) 0) (set! ({v} 1 0) 0) (set! ({v} 1 1) {1}) {v})))")

(test (reverse '(1 2 (3 4))) '((3 4) 2 1))
(test (reverse '(1 2 3)) '(3 2 1))
(test (reverse ()) ())
(test (let ((lst (list 1 2 3))) (set! (lst 2) lst) (object->string (reverse lst))) "(#1=(1 2 #1#) 2 1)")
(test (let ((l1 (cons 1 ()))) (set-cdr! l1 l1) (object->string (reverse l1))) "(#1=(1 . #1#) 1 1 1)")


(test (equal? (vector 0) (vector 0)) #t)
(test (equal? (vector 0 #\a "hi" (list 1 2 3)) (vector 0 #\a "hi" (list 1 2 3))) #t)
(test (let ((v (vector 0))) (equal? (vector v) (vector v))) #t)

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (vector? v1) #t)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (vector-length v1) 1)
    (test (equal? v1 v2) #t)
    (test (equal? (vector-ref v1 0) v1) #t)
    (test (equal? (vector->list v1) (list v1)) #t)
    (vector-fill! v1 0)
    (test (equal? v1 (vector 0)) #t)
    (let ((v3 (copy v2)))
      (test (equal? v2 v3) #t)
      (vector-set! v3 0 0)
      (test (equal? v3 (vector 0)) #t))
    ))

(let ((v1 (make-vector 1 0))
      (v2 (vector 0)))
  (set! (v1 0) v2)
  (set! (v2 0) v1)
  (test (equal? v1 v2) #t)) 

(test (vector? (let ((v (vector 0))) (set! (v 0) v) (v 0 0 0 0))) #t) ; ?

(let* ((l1 (list 1 2))
       (v1 (vector 1 2))
       (l2 (list 1 l1 2))
       (v2 (vector l1 v1 l2)))
  (vector-set! v1 0 v2)
  (list-set! l1 1 l2)
  (test (equal? v1 v2) #f))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (let ((v2 (vector 0)))
    (vector-set! v2 0 v2)
    (test (equal? v1 v2) #t)))

(let ((v1 (make-vector 1 0)))
  (set! (v1 0) v1)
  (test (eq? ((cyclic-sequences v1) 0) v1) #t)
  (test (object->string v1) "#1=#(#1#)"))

(let ((l1 (cons 0 ()))) 
  (set-cdr! l1 l1) 
  (test (list->vector l1) 'error))

(let ((lst (list "nothing" "can" "go" "wrong")))
  (let ((slst (cddr lst))
	(result ()))
    (set! (cdr (cdddr lst)) slst)
    (test (do ((i 0 (+ i 1))
	       (l lst (cdr l)))
	      ((or (null? l) (= i 12))
	       (reverse result))
	    (set! result (cons (car l) result)))
	  '("nothing" "can" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong" "go" "wrong"))))

#|
;;; here is a circular function
(let ()
  (define (cfunc)
    (begin
      (display "cfunc! ")
      #f))

  (let ((clst (procedure-source cfunc)))
    (set! (cdr (cdr (car (cdr (cdr clst)))))
	  (cdr (car (cdr (cdr clst))))))

  (cfunc))
|#

(test (let ((l (list 1 2))) 
	(list-set! l 0 l) 
	(string=? (object->string l) "#1=(#1# 2)")) 
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(string=? (object->string lst) "#1=(#1# . #1#)"))
      #t)
(test (let ((lst (list 1)))
	(set! (car lst) lst)
	(set! (cdr lst) lst)
	(equal? (car lst) (cdr lst)))
      #t)
(test (let ((lst (cons 1 2))) 
	(set-cdr! lst lst)
	(string=? (object->string lst) "#1=(1 . #1#)"))
      #t)
(test (let ((lst (cons 1 2))) 
	(set-car! lst lst)
	(string=? (object->string lst) "#1=(#1# . 2)"))
      #t)
(test (let ((lst (cons (cons 1 2) 3))) 
	(set-car! (car lst) lst)
	(string=? (object->string lst) "#1=((#1# . 2) . 3)"))
      #t)
(test (let ((v (vector 1 2))) 
	(vector-set! v 0 v) 
	(string=? (object->string v) "#1=#(#1# 2)")) 
      #t)
(test (let* ((l1 (list 1 2)) (l2 (list l1))) 
	(list-set! l1 0 l1) 
	(string=? (object->string l2) "(#1=(#1# 2))")) 
      #t)

(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) lst) (object->string lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr lst)) (object->string lst)) "(1 . #1=(2 3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (cdr (cdr (cdr lst))) (cdr (cdr lst))) (object->string lst)) "(1 2 . #1=(3 . #1#))")
(test (let ((lst (list 1 2 3))) (set! (car lst) (cdr lst)) (object->string lst)) "((2 3) 2 3)") ; was "(#1=(2 3) . #1#)"
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) (cdr lst)) (object->string lst)) "(1 . #1=(#1# 3))")
(test (let ((lst (list 1 2 3))) (set! (car (cdr lst)) lst) (object->string lst)) "#1=(1 #1# 3)")
(test (let ((l1 (list 1))) (let ((l2 (list l1 l1))) (object->string l2))) "((1) (1))") ; was "(#1=(1) #1#)"

(test (let* ((v1 (vector 1 2)) (v2 (vector v1))) 
	(vector-set! v1 1 v1) 
	(string=? (object->string v2) "#(#1=#(1 #1#))")) 
      #t)
(test (let ((v1 (make-vector 3 1))) 
	(vector-set! v1 0 (cons 3 v1)) 
	(string=? (object->string v1) "#1=#((3 . #1#) 1 1)")) 
      #t)
(test (let ((h1 (make-hash-table 11))
	    (old-print-length (*s7* 'print-length)))
	(set! (*s7* 'print-length) 32)
	(hash-table-set! h1 "hi" h1)
	(let ((result (object->string h1)))
	  (set! (*s7* 'print-length) old-print-length)
	  (let ((val (string=? result "#1=(hash-table '(\"hi\" . #1#))")))
	    (if (not val)
		(format-logged #t ";hash display:~%  ~A~%" (object->string h1)))
	    val)))
      #t)

(test (let* ((l1 (list 1 2))
	     (v1 (vector 1 2))
	     (l2 (list 1 l1 2))
	     (v2 (vector l1 v1 l2)))
	(vector-set! v1 0 v2)
	(list-set! l1 1 l2)
	(string=? (object->string v2) "#2=#(#1=(1 #3=(1 #1# 2)) #(#2# 2) #3#)"))
      #t)

(test (let ((l1 (list 1 2))
	    (l2 (list 1 2)))
	(set! (car l1) l2)
	(set! (car l2) l1)
	(object->string (list l1 l2)))
      "(#1=(#2=(#1# 2) 2) #2#)")

(test (let* ((l1 (list 1 2)) 
	     (l2 (list 3 4)) 
	     (l3 (list 5 l1 6 l2 7)))
	(set! (cdr (cdr l1)) l1) 
	(set! (cdr (cdr l2)) l2)
	(string=? (object->string l3) "(5 #1=(1 2 . #1#) 6 #2=(3 4 . #2#) 7)"))
      #t)
(test (let* ((lst1 (list 1 2))
	     (lst2 (list (list (list 1 (list (list (list 2 (list (list (list 3 (list (list (list 4 lst1 5))))))))))))))
	(set! (cdr (cdr lst1)) lst1)
	(string=? (object->string lst2) "(((1 (((2 (((3 (((4 #1=(1 2 . #1#) 5))))))))))))"))
      #t)


(test (equal? '(a) (list 'a)) #t)
(test (equal? '(a b . c) '(a b . c)) #t)
(test (equal? '(a b (c . d)) '(a b (c . d))) #t)
(test (equal? (list "hi" "hi" "hi") '("hi" "hi" "hi")) #t)
(let ((l1 (list "hi" "hi" "hi"))
      (l2 (list "hi" "hi" "hi")))
  (fill! l1 "ho")
  (test (equal? l1 l2) #f)
  (fill! l2 (car l1))
  (test (equal? l1 l2) #t))
(let ((lst (list 1 2 3 4))) 
  (fill! lst "hi") 
  (test (equal? lst '("hi" "hi" "hi" "hi")) #t))
(let ((vect (vector 1 2 3 4)))
  (fill! vect "hi")
  (test (equal? vect #("hi" "hi" "hi" "hi")) #t))
(let ((lst (list 1 2 (list 3 4) (list (list 5) 6))))
  (test (equal? lst '(1 2 (3 4) ((5) 6))) #t)
  (fill! lst #f)
  (test (equal? lst '(#f #f #f #f)) #t))
(let ((lst (list 1 2 3 4)))
  (set! (cdr (cdddr lst)) lst)
  (test (equal? lst lst) #t)
  (test (eq? lst lst) #t)
  (test (eqv? lst lst) #t)
  (fill! lst #f)
  (test (object->string lst) "#1=(#f #f #f #f . #1#)")
  (let ((l1 (copy lst)))
    (test (equal? lst l1) #t)
    (test (eq? lst l1) #f)
    (test (eqv? lst l1) #f)))


(let ((lst '(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((str (apply string lst)))
    (let ((lstr (list->string lst)))
      (let ((strl (string->list str)))
	(test (eq? str str) #t)
	(test (eq? str lstr) #f)
	(test (eqv? str str) #t)
	(test (eqv? str lstr) #f)
	(test (equal? str lstr) #t)
	(test (equal? str str) #t)
	(test (eq? lst strl) #f)	
	(test (eqv? lst strl) #f)	
	(test (equal? lst strl) #t)
	(let ((l2 (copy lst))
	      (s2 (copy str)))
	  (test (eq? l2 lst) #f)
	  (test (eq? s2 str) #f)
	  (test (eqv? l2 lst) #f)
	  (test (eqv? s2 str) #f)
	  (test (equal? l2 lst) #t)
	  (test (equal? s2 str) #t))))))


(let ((vect #(#\( #\) #\* #\+ #\, #\- #\. #\/ #\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9 #\: #\; #\< #\= #\> #\? #\@ #\A #\B #\C #\D #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X #\Y #\Z #\[ #\\ #\] #\^ #\_ #\` #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t #\u #\v #\w #\x #\y #\z #\{ #\| #\} #\~)))
  (let ((lst (vector->list vect)))
    (let ((vect1 (list->vector lst)))
	(test (eq? lst lst) #t)
	(test (eq? lst vect) #f)
	(test (eqv? lst lst) #t)
	(test (eqv? lst vect) #f)
	(test (equal? vect1 vect) #t)
	(test (equal? lst lst) #t)
	(test (eq? vect vect1) #f)	
	(test (eqv? vect vect1) #f)	
	(test (equal? vect vect1) #t)
	(let ((l2 (copy vect))
	      (s2 (copy lst)))
	  (test (eq? l2 vect) #f)
	  (test (eq? s2 lst) #f)
	  (test (eqv? l2 vect) #f)
	  (test (eqv? s2 lst) #f)
	  (test (equal? l2 vect) #t)
	  (test (equal? s2 lst) #t)))))

(let* ((vals (list "hi" #\A 1 'a #(1) abs _ht_ _null_ _c_obj_ quasiquote macroexpand (log 0) 
		   3.14 3/4 1.0+1.0i #\f '(1 . 2)))
       (vlen (length vals)))
  (do ((i 0 (+ i 1)))
      ((= i 20))
    (let* ((size (max 1 (random 20)))
	   (vect (make-vector size ())))
      (do ((n 0 (+ n 1)))
	  ((= n size))
	(let ((choice (random 4))
	      (len (random 4)))
	  (if (= choice 0)
	      (let ((v (make-vector len)))
		(do ((k 0 (+ k 1)))
		    ((= k len))
		  (vector-set! v k (list-ref vals (random vlen))))
		(vector-set! vect n v))
	      (if (= choice 1)
		  (let ((lst (make-list len #f)))
		    (do ((k 0 (+ k 1)))
			((= k len))
		      (list-set! lst k (list-ref vals (random vlen))))
		    (vector-set! vect n lst))
		  (vector-set! vect n (list-ref vals (random vlen)))))))
      (test (eq? vect vect) #t)
      (test (eqv? vect vect) #t)
      (test (equal? vect vect) #t)
      (let ((lst1 (vector->list vect)))
	(let ((lst2 (copy lst1)))
	  (test (eq? lst1 lst2) #f)
	  (test (eqv? lst1 lst2) #f)
	  (test (equal? lst1 lst2) #t))))))

(let* ((lst1 (list 1 2 3))
       (vec1 (vector 1 2 lst1)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 lst2)))
    (list-set! lst2 2 vec2)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (vector-set! vec1 1 vec1)
    (test (equal? lst1 lst2) #f)
    (test (equal? vec1 vec2) #f)
    ))
  
(let* ((base (list #f))
       (lst1 (list 1 2 3))
       (vec1 (vector 1 2 base)))
  (list-set! lst1 2 vec1)
  (let* ((lst2 (list 1 2 3))
	 (vec2 (vector 1 2 base)))
    (list-set! lst2 2 vec2)
    (set! (car lst1) lst1)
    (set! (car lst2) lst2)
    (set! (cdr (cddr lst1)) base)
    (set! (cdr (cddr lst2)) base)
    (test (length (cyclic-sequences lst2)) 1)
    (test (equal? lst1 lst2) #t)
    (test (equal? vec1 vec2) #t)
    (test (object->string lst1) "#1=(#1# 2 #(1 2 (#f)) #f)"))) ; was "#1=(#1# 2 #(1 2 #2=(#f)) . #2#)"

(let ((base (list 0 #f)))
  (let ((lst1 (list 1 base 2))
	(lst2 (list 1 base 2)))
    (set! (cdr (cdr base)) base)
    (test (equal? lst1 lst2) #t)))

(let ((base1 (list 0 #f))
      (base2 (list 0 #f)))
  (let ((lst1 (list 1 base1 2))
	(lst2 (list 1 base2 2)))
    (set! (cdr (cdr base1)) lst2)
    (set! (cdr (cdr base2)) lst1)
    (test (equal? lst1 lst2) #t)
    (test (object->string lst1) "#1=(1 (0 #f 1 (0 #f . #1#) 2) 2)")))

(let ()
  (define-macro (c?r path)

    (define (X-marks-the-spot accessor tree)
      (if (pair? tree)
	  (or (X-marks-the-spot (cons 'car accessor) (car tree))
	      (X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	  (if (eq? tree 'X) accessor #f)))
    
    (let ((body 'lst))
      (for-each
       (lambda (f)
	 (set! body (list f body)))
       (reverse (X-marks-the-spot () path)))
      
      `(dilambda
	(lambda (lst) 
	  ,body)
	(lambda (lst val)
	  (set! ,body val)))))
  
  (define (copy-tree lis)
    (if (pair? lis)
	(cons (copy-tree (car lis))
	      (copy-tree (cdr lis)))
	lis))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (X))))))))))))
    (set! (cxr l1) 3)
    (set! (cxr l2) 4)
    (test (equal? l1 l2) #f)
    (test (equal? l1 l3) #f)
    (set! (cxr l2) 3)
    (test (cxr l2) 3)
    (test (cxr l1) 3)
    (test (cxr l3) 8)
    (test (equal? l1 l2) #t)
    (test (equal? l2 l3) #f))

  (let* ((l1 '(0 (1 (2 (3 (4 (5 (6 (7 (8))))))))))
	 (l2 (list 0 (list 1 (list 2 (list 3 (list 4 (list 5 (list 6 (list 7 (list 8))))))))))
	 (l3 (copy-tree l1))
	 (cxr (c?r (0 (1 (2 (3 (4 (5 (6 (7 (8 . X))))))))))))
    (set! (cxr l1) l1)
    (set! (cxr l2) l2)
    (test (equal? l1 l2) #t)
    (test (equal? l1 l3) #f)
    (test (object->string l2) "#1=(0 (1 (2 (3 (4 (5 (6 (7 (8 . #1#)))))))))"))

  (let* ((l1 '(0 ((((((1))))))))
	 (l2 (copy-tree l1))
	 (cxr (c?r (0 ((((((1 . X))))))))))
    (set! (cxr l1) l2)
    (set! (cxr l2) l1)
    (test (equal? l1 l2) #t))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (cxr (c?r (0 1 (2 3 . X) 4 5))))
    (set! (cxr l1) (cdr l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 . #1#) 4 5))"))

  (let* ((l1 '(0 1 (2 3) 4 5))
	 (l2 '(6 (7 8 9) 10))
	 (cxr1 (c?r (0 1 (2 3 . X) 4 5)))
	 (cxr2 (c?r (6 . X)))
	 (cxr3 (c?r (6 (7 8 9) 10 . X)))
	 (cxr4 (c?r (0 . X))))
    (set! (cxr1 l1) (cxr2 l2))
    (set! (cxr3 l2) (cxr4 l1))
    (test (object->string l1) "(0 . #1=(1 (2 3 (7 8 9) 10 . #1#) 4 5))")
    (test (cadr l1) 1)
    (test (cadddr l1) 4)
    )

  (let ((l1 '((a . 2) (b . 3) (c . 4)))
	(cxr (c?r ((a . 2) (b . 3) (c . 4) . X))))
    (set! (cxr l1) (cdr l1))
    (test (assq 'a l1) '(a . 2))
    (test (assv 'b l1) '(b . 3))
    (test (assoc 'c l1) '(c . 4))
    (test (object->string l1) "((a . 2) . #1=((b . 3) (c . 4) . #1#))")
    (test (assq 'asdf l1) #f)
    (test (assv 'asdf l1) #f)
    (test (assoc 'asdf l1) #f)
    )

  (let ((l1 '(a b c d e))
	(cxr (c?r (a b c d e . X))))
    (set! (cxr l1) (cddr l1))
    (test (memq 'b l1) (cdr l1))
    (test (memv 'c l1) (cddr l1))
    (test (member 'd l1) (cdddr l1))
    (test (object->string l1) "(a b . #1=(c d e . #1#))")
    (test (memq 'asdf l1) #f)
    (test (memv 'asdf l1) #f)
    (test (member 'asdf l1) #f)
    (test (pair? (member 'd l1)) #t) ; #1=(d e c . #1#)
    )

  (let ((ctr 0)
	(x 0))
    (let ((lst `(call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0)))))
      (let ((acc1 (c?r (call-with-exit (lambda (return) . X))))
	    (acc2 (c?r (call-with-exit (lambda (return) (set! x (+ x 1)) (if (> x 10) (return x) 0) . X)))))
	(set! (acc2 lst) (acc1 lst))
	(test (eval lst) 11))))
  )

(let ()
  ;; anonymous recursion...
  (define (fc?r path)
    (define (X-marks-the-spot accessor tree)
      (if (pair? tree)
	  (or (X-marks-the-spot (cons 'car accessor) (car tree))
	      (X-marks-the-spot (cons 'cdr accessor) (cdr tree)))
	  (if (eq? tree 'X) accessor #f)))
    (let ((body 'lst))
      (for-each
       (lambda (f)
	 (set! body (list f body)))
       (reverse (X-marks-the-spot () path)))
      (let ((getter (apply lambda '(lst) body ()))
	    (setter (apply lambda '(lst val) `(set! ,body val) ())))
	(dilambda getter setter))))
  
  (let ((body '(if (not (pair? (cdr lst))) lst (begin (set! lst (cdr lst)) X)))) ; X is where we jump back to the start
    (let ((recurse (fc?r body)))
      (set! (recurse body) body)
      (test ((apply lambda '(lst) body ()) '(1 2 3)) '(3)))))
  

  
(let ((v #2d((1 2) (3 4))))
  (set! (v 1 0) v)
  (test (object->string v) "#1=#2D((1 2) (#1# 4))")
  (test (length v) 4)
  (test ((((v 1 0) 1 0) 1 0) 0 0) 1))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (lst 100) 2)
  (test ((cdddr (cdddr (cdddr lst))) 100) 2)
  (set! (lst 100) 32)
  (test (object->string lst) "#1=(1 32 3 . #1#)"))

(let* ((l1 (list 1 2))
       (l2 (list l1 l1)))
  (set! (l1 0) 32)
  (test (equal? l2 '((32 2) (32 2))) #t))

(let ((q (list 1 2 3 4)))
  (set! (cdr (cdddr q)) q) 
  (test (car q) 1)
  (set! (car q) 5)
  (set! q (cdr q))
  (test (car q) 2)
  (test (object->string q) "#1=(2 3 4 5 . #1#)"))

(let ()
  (define make-node vector)
  (define prev (dilambda (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (dilambda (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (dilambda (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=#(#7=#(#6=#(#5=#(#4=#(#3=#(#2=#(#8=#(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
#|
    ;; in CL:
    (let* ((head (vector nil 0 nil))
	   (cur head))
      (do ((i 1 (+ i 1)))
	  ((= i 8))
	(let ((node (vector nil i nil)))
	  (setf (aref node 0) cur)
	  (setf (aref cur 2) node)
	  (setf cur node)))
      (setf (aref head 0) cur)
      (setf (aref cur 2) head)
      (format t "~A~%" head)) -> "#1=#(#2=#(#3=#(#4=#(#5=#(#6=#(#7=#(#8=#(#1# 1 #7#) 2 #6#) 3 #5#) 4 #4#) 5 #3#) 6 #2#) 7 #1#) 0 #8#)"
|#
    (let ((ahead (do ((cur head (next cur))
		      (dat () (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat () (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t)))))

(let ()
  (define make-node list)
  (define prev (dilambda (lambda (node) (node 0)) (lambda (node val) (set! (node 0) val))))
  (define next (dilambda (lambda (node) (node 2)) (lambda (node val) (set! (node 2) val))))
  (define data (dilambda (lambda (node) (node 1)) (lambda (node val) (set! (node 1) val))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 8))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#7=(#6=(#5=(#4=(#3=(#2=(#8=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #1#) 0 #8#)")
    (let ((ahead (do ((cur head (next cur))
		      (dat () (cons (data cur) dat)))
		     ((member (data cur) dat)
		      (reverse dat)))))
      (let ((behind (do ((cur (prev head) (prev cur))
			 (dat () (cons (data cur) dat)))
			((member (data cur) dat)
			 dat))))
	(test (equal? ahead behind) #t))))
  (let* ((head (make-node () 0 ()))
	 (cur head))
    (do ((i 1 (+ i 1)))
	((= i 32))
      (let ((next-node (make-node cur i ())))
	(set! (next cur) next-node)
	(set! cur (next cur))))
    (set! (next cur) head)
    (set! (prev head) cur)
    (test (object->string head) "#1=(#31=(#30=(#29=(#28=(#27=(#26=(#25=(#24=(#23=(#22=(#21=(#20=(#19=(#18=(#17=(#16=(#15=(#14=(#13=(#12=(#11=(#10=(#9=(#8=(#7=(#6=(#5=(#4=(#3=(#2=(#32=(#1# 1 #2#) 2 #3#) 3 #4#) 4 #5#) 5 #6#) 6 #7#) 7 #8#) 8 #9#) 9 #10#) 10 #11#) 11 #12#) 12 #13#) 13 #14#) 14 #15#) 15 #16#) 16 #17#) 17 #18#) 18 #19#) 19 #20#) 20 #21#) 21 #22#) 22 #23#) 23 #24#) 24 #25#) 25 #26#) 26 #27#) 27 #28#) 28 #29#) 29 #30#) 30 #31#) 31 #1#) 0 #32#)")))

(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (append lst lst ())) 'error)
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (object->string (append (list lst) (list lst) ()))) "(#1=(1 2 3 . #1#) #1#)")

(let ((ht (make-hash-table 3)))
  (set! (ht "hi") ht)
  (test (object->string ht) "#1=(hash-table '(\"hi\" . #1#))")
  (test (equal? (ht "hi") ht) #t))

(let ((l1 '(0)) (l2 '(0))) 
  (set! (car l1) l1) (set! (cdr l1) l1) (set! (car l2) l2) (set! (cdr l2) l2)
  (test (object->string l1) "#1=(#1# . #1#)")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) l2)
  (test (object->string l1) "#1=(#1# . #2=(#2# . #2#))")
  (test (equal? l1 l2) #t)
  (set! (cdr l1) ())
  (test (equal? l1 l2) #f))

(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (list 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)) 
	    (result ()))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! result (cons (+ a b) result)))
		  (list 4 5 6)
		  lst)
	result)
      '(9 7 5))
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6)
	     lst)
	'(5 7 9)))
(test (let ((lst (list 1 2 3)))
	(set! (cdr (cddr lst)) lst)
	(map (lambda (a b)
	       (+ a b))
	     (vector 4 5 6 7 8 9 10)
	     lst))
      '(5 7 9)) ; this now quits when it sees the cycle
      ;'(5 7 9 8 10 12 11))
(test (map (lambda (a) a) '(0 1 2 . 3)) '(0 1 2))
(test (let ((ctr 0)) (for-each (lambda (a) (set! ctr (+ ctr a))) '(1 2 . 3)) ctr) 3)
(let ((lst (list 1 2 3)))
  (set! (cdr (cddr lst)) lst)
  (test (map (lambda (a b)
	       (+ a b))
	     ()
	     lst)
	()))
(test (let ((lst (list 1 2 3))
	    (ctr 0))
	(set! (cdr (cddr lst)) lst)
	(for-each (lambda (a b)
		    (set! ctr (+ ctr (+ a b))))
		  lst ())
	ctr)
      0)

(test (let ((lst (list 1))) (set! (cdr lst) (car lst)) (object->string lst)) "(1 . 1)")
(test (let ((lst (list 1))) (set! (car lst) (cdr lst)) (object->string lst)) "(())")

(let ((ctr 0) (lst `(let ((x 3)) (set! ctr (+ ctr 1)) (set! (cdr (cddr lst)) `((+ x ctr))) (+ x 1))))
  (test (eval lst) 4)
  (test (eval lst) 5)
  (test (eval lst) 6))
  

(let ()
  (define fact         ; Reini Urban, http://autocad.xarch.at/lisp/self-mod.lsp.txt
    (let ((old ())
	  (result ()))
      
      (define (last lst)
	(list-tail lst (- (length lst) 1)))
      
      (define (butlast lis)
	(let ((len (length lis)))
	  (if (<= len 1) ()
	      (let ((result ()))
		(do ((i 0 (+ i 1))
		     (lst lis (cdr lst)))
		    ((= i (- len 1)) (reverse result))
		  (set! result (cons (car lst) result)))))))
      
      (lambda (n)
	(cond ((zero? n) 1)
	      (#t 
	       (set! old (procedure-source fact))
	       (set! fact (apply lambda '(n)
				 `((cond 
				    ,@(butlast (cdr (car (cdr (cdr old)))))
				    ((= n ,n) ,(let ()
						 (set! result (* n (fact (- n 1))))
						 result))
				    ,@(last (cdr (car (cdr (cdr old)))))))))
	       result)))))

  (test (fact 3) 6)
  (test (fact 5) 120)
  (test (fact 2) 2))

(test (let ((f #f))
	(set! f (lambda () 
		  (let* ((code (procedure-source f))
			 (pos (- (length code) 1)))
		    (set! (code pos) (+ (code pos) 1)))
		  1))
	(f) (f) (f))
      4)

(let* ((x (list 1 2 3)) ; from Lambda the Ultimate I think -- I lost the reference
       (y (list 4 5))	
       (z (cons (car x) (cdr y)))
       (w (append y z))
       (v (cons (cdr x) (cdr y))))
  (set-car! x 6)
  (set-car! y 7)
  (set-cdr! (cdr x) (list 8))
  (test (object->string (list x y z w v)) "((6 2 8) (7 5) (1 5) (4 5 1 5) ((2 8) 5))"))
;; was "((6 . #3=(2 8)) (7 . #1=(5)) #2=(1 . #1#) (4 5 . #2#) (#3# . #1#))"

;; circular eval
(test (let ((e (list (list '+ 1)))) (set-cdr! (car e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1 2)))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1 2) 3))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((e (list (list '+ 1) 3 4))) (set-cdr! (cdar e) e) (eval e)) 'error)
(test (let ((x '(1 2 3)))
	(set! (x 0) (cons x 2))
	(eval (list-values 'let () (list-values 'define '(f1) (list-values 'list-set! x 0 (list-values 'cons x 2))) '(catch #t f1 (lambda a 'error)))))
      'error)
(test (let ((x '(car (list 1 2 3)))) 
	(set! (x 0) x) 
	(eval (list-values 'let () (list-values 'define '(f1) x) '(catch #t f1 (lambda a 'error)))))
      'error)


#|
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format-logged #t "~{~A~^ ~}~%" args)) '(1 2 3))
  (define (pinner cur nvals len)
    (if (= len 1)
	(apply func (cons (car nvals) cur))
	(do ((i 0 (+ i 1)))                       ; I suppose a named let would be more Schemish
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (let ((cur1 (cons (car nvals) cur)))  ; add (car nvals) to our arg list
	      (set! (cdr start) (cdr nvals))      ; splice out that element and 
	      (pinner cur1 (cdr start) (- len 1)) ;   pass a smaller circle on down
	      (set! (cdr start) nvals))))))       ; restore original circle
  (let ((len (length vals)))
    (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
    (pinner () vals len)
    (set-cdr! (list-tail vals (- len 1)) ())))   ; restore its original shape
|#

#|
;; a slightly faster version (avoids consing and some recursion)
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format-logged #t "~A~%" args)) '(1 2 3))
  (let ((cur (make-list (length vals))))

    (define (pinner nvals len)
      (if (= len 2)
	  (begin
	    (set! (cur 0) (car nvals))
	    (set! (cur 1) (cadr nvals))
	    (apply func cur)
	    (set! (cur 1) (car nvals))
	    (set! (cur 0) (cadr nvals))
	    (apply func cur))
		
	(do ((i 0 (+ i 1)))                       ; I suppose a named let would be more Schemish
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (set! (cur (- len 1)) (car nvals)) 
	    (set! (cdr start) (cdr nvals))        ; splice out that element and 
	    (pinner (cdr start) (- len 1))        ;   pass a smaller circle on down
	    (set! (cdr start) nvals)))))          ; restore original circle

  (let ((len (length vals)))
    (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
    (pinner vals len)
    (set-cdr! (list-tail vals (- len 1)) ()))))  ; restore its original shape
|#

;; and continuing down that line...
(define (for-each-permutation func vals)          ; for-each-combination -- use for-each-subset below
  "(for-each-permutation func vals) applies func to every permutation of vals"
  ;;   (for-each-permutation (lambda args (format-logged #t "~A~%" args)) '(1 2 3))
  (let ((cur (make-list (length vals))))

    (define (pinner nvals len)
      (if (= len 3)
	  (let ((a0 (car nvals))
		(a1 (cadr nvals))
		(a2 (caddr nvals))
		(c1 (cdr cur))
		(c2 (cddr cur)))
	    (set-car! cur a2)
	    (set-car! c1 a0)
	    (set-car! c2 a1)
	    (apply func cur)
	    (set-car! cur a0)
	    (set-car! c1 a2)
	    ;(set-car! c2 a1)
	    (apply func cur)
	    ;(set-car! cur a0)
	    (set-car! c1 a1)
	    (set-car! c2 a2)
	    (apply func cur)
	    (set-car! cur a1)
	    (set-car! c1 a0)
	    ;(set-car! c2 a2)
	    (apply func cur)
	    ;(set-car! cur a1)
	    (set-car! c1 a2)
	    (set-car! c2 a0)
	    (apply func cur)
	    (set-car! cur a2)
	    (set-car! c1 a1)
	    ;(set-car! c2 a0)
	    (apply func cur)
	    )
		
	(do ((i 0 (+ i 1)))                       
	    ((= i len))
	  (let ((start nvals))
	    (set! nvals (cdr nvals))
	    (list-set! cur (- len 1) (car nvals)) 
	    (set! (cdr start) (cdr nvals))        ; splice out that element and 
	    (pinner (cdr start) (- len 1))        ;   pass a smaller circle on down
	    (set! (cdr start) nvals)))))          ; restore original circle

  (let ((len (length vals)))
    (if (< len 2)
	(apply func vals)
	(if (= len 2)
	    (let ((c1 (cdr cur)))
	      (set-car! cur (car vals))
	      (set-car! c1 (cadr vals))
	      (apply func cur)
	      (set-car! c1 (car vals))
	      (set-car! cur (cadr vals))
	      (apply func cur))
	    (begin
	      (set-cdr! (list-tail vals (- len 1)) vals)    ; make vals into a circle
	      (pinner vals len)
	      (set-cdr! (list-tail vals (- len 1)) ())))))))  ; restore its original shape


(when full-test
  (let()
    (define ops '(+ *))
    (define args '(1 pi 1+i 2/3 x y))
    
    (define (listify lst)
      ((if (memq (car lst) ops) list 
	   (if (null? (cdr lst)) append values))
       (if (null? (cdr lst))
	   (car lst)
	   (values (car lst) (listify (cdr lst))))))
    
    (call-with-output-file "t923.scm"
      (lambda (p)
	(let ((fctr 0))
	  (for-each-permutation
	   (lambda lst
	     (let ((expr (list (listify lst))))
	       (format p "(define (f~D x y) ~{~^~S ~})~%" fctr expr)
	       (format p "(let ((e1 (f~D 3 4)))~%" fctr)
	       (format p "  (let ((e2 (let ((x 3) (y 4)) ~{~^~S ~})))~%" expr)
	       (format p "    (let ((e3 (let ((x 3) (y 4)) (f~D x y))))~%" fctr)
	       (format p "      (if (not (= e1 e2 e3))~%          (format *stderr* \"~{~^~S ~}: ~~A ~~A ~~A~~%\" e1 e2 e3)))))~%~%" expr))
	     (set! fctr (+ fctr 1)))
	   (append ops args)))))
    
    (load "t923.scm")))

  ;; t224 also applies this to +/*

(let ((perms '((3 1 2) (1 3 2) (1 2 3) (2 1 3) (2 3 1) (3 2 1)))
      (pos ()))
  (for-each-permutation
   (lambda args
     (call-with-exit
      (lambda (ok)
	(let ((ctr 0))
	  (for-each
	   (lambda (a)
	     (if (equal? a args)
		 (begin
		   (set! pos (cons ctr pos))
		   (ok)))
	     (set! ctr (+ ctr 1)))
	   perms)))))
   '(1 2 3))
  (test pos '(5 4 3 2 1 0)))

(test (let ((v1 (make-vector 16 0)) 
	    (v2 (make-vector 16 0))) 
	(set! (v2 12) v2) 
	(set! (v1 12) v1) 
	(equal? v1 v2))        ; hmmm -- not sure this is correct
      #t)
(test (let ((lst1 (list 1)) 
	    (lst2 (list 1))) 
	(set-cdr! lst1 lst1) 
	(set-cdr! lst2 lst2) 
	(equal? lst1 lst2))
      #t)


(test (let ((hi 3))
	(let ((e (curlet)))
	  (set! hi (curlet)) 
	  (object->string e)))
      "#1=(inlet 'hi (inlet 'e #1#))")
(let ((e (inlet 'a 0 'b 1)))
  (let ((e1 (inlet 'a e)))
    (set! (e 'b) e1)
    (test (equal? e (copy e)) #t)
    (test (object->string e) "#1=(inlet 'a 0 'b (inlet 'a #1#))")))

;; eval circles -- there are many more of these that will cause stack overflow 
(test (let ((x '(1 2 3))) (set! (x 0) (cons x 2)) (eval `(let () (define (f1) (list-set! ,x 0 (cons ,x 2))) (catch #t f1 (lambda a 'error))))) 'error)
(test (let ((x '(car (list 1 2 3)))) (set! (x 0) x) (eval `(let () (define (f1) ,x) (catch #t f1 (lambda a 'error))))) 'error)


(test (apply + (cons 1 2)) 'error)
(test (let ((L (list 0))) (set-cdr! L L) (apply + L)) 'error)
(test (let ((L (list 0))) (set-cdr! L L) (format #f "(~S~{~^ ~S~})~%" '+ L)) 'error)
(test (apply + (list (let ((L (list 0 1))) (set-cdr! L L) L))) 'error)
(test (apply + (let ((L (list 0 1))) (set-cdr! L L) L)) 'error)
(test (length (let ((E (inlet 'value 0))) (varlet E 'self E))) 2)
;(test (apply case 2 (list (let ((L (list (list 0 1)))) (set-cdr! L L) L))) 'error)
;(test (apply cond (list (let ((L (list 0 1))) (set-cdr! L L) L))) 'error)
;(test (apply quote (let ((L (list 0 1))) (set-car! L L) L)) 'error)
;(test (apply letrec (hash-table) (let ((L (list 0 1))) (set-car! L L) L)) 'error)
;I now think the caller should check for these, not s7



;;; --------------------------------------------------------------------------------
;;; HOOKS
;;; make-hook
;;; hook-functions
;;; --------------------------------------------------------------------------------

(let-temporarily (((hook-functions *error-hook*) ())
		  ((hook-functions *load-hook*) ())
		  ((hook-functions *unbound-variable-hook*) ())
		  ((hook-functions *missing-close-paren-hook*) ()))
  (for-each
   (lambda (arg)
     (test (set! *unbound-variable-hook* arg) 'error)
     (test (set! *missing-close-paren-hook* arg) 'error)
     (test (set! *load-hook* arg) 'error)
     
     (test (set! (hook-functions *unbound-variable-hook*) arg) 'error)
     (test (set! (hook-functions *missing-close-paren-hook*) arg) 'error)
     (test (set! (hook-functions *error-hook*) arg) 'error)
     (test (set! (hook-functions *load-hook*) arg) 'error)
     
     (test (set! (hook-functions *unbound-variable-hook*) (list arg)) 'error)
     (test (set! (hook-functions *missing-close-paren-hook*) (list arg)) 'error)
     (test (set! (hook-functions *error-hook*) (list arg)) 'error)
     (test (set! (hook-functions *load-hook*) (list arg)) 'error))
   (list -1 #\a #(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> #(1 2 3) #(()) "hi" '(1 . 2) '(1 2 3))))

(let ((hook-val #f))
  (let-temporarily (((hook-functions *unbound-variable-hook*)
		     (list (lambda (hook) 
			     (set! hook-val (hook 'variable)) 
			     (set! (hook 'result) 123)))))
    (let ((val (catch #t
		 (lambda ()
		   (+ 1 one-two-three))
		 (lambda args
		   (apply format *stderr* (cadr args))
		   'error))))
      (test val 124))
    (test (equal? one-two-three 123) #t)
    (test (equal? hook-val 'one-two-three) #t)))

(let-temporarily (((hook-functions *unbound-variable-hook*) (list (lambda (hook) 
								    (set! (hook 'result) 32)))))
  (let ((val (+ 1 _an_undefined_variable_i_hope_)))
    (test val 33))
  (let ((val (* _an_undefined_variable_i_hope_ _an_undefined_variable_i_hope_)))
    (test val 1024)))

(let ((x #f))
  (let-temporarily (((hook-functions *unbound-variable-hook*) 
		     (list (lambda (hook)
			     (set! x 0)
			     (set! (hook 'result) #<undefined>))
			   (lambda (hook) 
			     (set! (hook 'result) 32))
			   (lambda (hook)
			     (if (not (number? (hook 'result)))
				 (format *stderr* "oops -- *unbound-variable-hook* func called incorrectly~%"))))))
    (let ((val (+ 1 _an_undefined_variable_i_hope_)))
      (test val 33))
    (test x 0)
    (test (+ 1 _an_undefined_variable_i_hope_) 33)))


(define (-a-rootlet-entry- x) (- x (abs x)))
(define -a-rootlet-entry-value- #f)
(set! (hook-functions *rootlet-redefinition-hook*) 
      (list (lambda (hook) 
	      (set! -a-rootlet-entry-value- (hook 'value)))))
(define (-a-rootlet-entry- x) (+ x (abs x)))
(if (not (and (procedure? -a-rootlet-entry-value-)
	      (equal? (procedure-source -a-rootlet-entry-value-) '(lambda (x) (+ x (abs x))))))
    (format *stderr* "rootlet redef: ~W~%" -a-rootlet-entry-value-))
(set! (hook-functions *rootlet-redefinition-hook*) ())


;;; optimizer bug involving unbound variable
(let ()
  (define (opt1)
    (let ((val (let () 
		 (define (hi x y) (let ((m (memq x y)) (loc (and m (- x (length m))))) loc))
		 (hi 'a '(a b c)))))
      (format-logged #t "~A: opt1 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt1
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" m))))
	       (format *stderr* "opt 1type: ~A, info: ~A~%" type info))
	   'error)))
(let ()
  (define (opt2)
    (let ((val (let () 
		 (define (hi x y) (let* ((n (memq x y)) (loc (and m (- x (length m)))) (m (+ n 1))) loc)) 
		 (hi 'a '(a b c)))))
      (format-logged #t "~A: opt2 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt2
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" m))))
	       (format *stderr* "opt2 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt3)
    (let ((val (let () 
		 (define (hi x y) (do ((m (memq x y) 0) (loc (and m (- x (length m))) 0)) (loc #t))) 
		 (hi 'a '(a b c)))))
      (format-logged #t "~A: opt3 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt3
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" m))))
	       (format *stderr* "opt3 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt4)
    (let () 
      (define (hi x y) (letrec ((m (memq x y)) (loc (and m (length m)))) loc)) 
      (hi 'a '(a b c))))
  (catch #t opt4
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt5)
    (let ((val (let () 
		 (define (hi x y) (letrec* ((n (memq x y)) (loc (and m (- x (length m)))) (m (+ n 1))) loc))
		 (hi 'a '(a b c)))))
      (format-logged #t "~A: opt5 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt5
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt6)
    (let ((val (let () 
		 (define (hi x) (let ((m (memq n x)) (loc (and m (- x (length m))))) (define n 1) loc)) 
		 (hi '(a b c)))))
      (format-logged #t "~A: opt6 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt6
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" n))))
	       (format *stderr* "opt6 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt7)
    (let ((val (let () 
		 (define* (f1 (a (+ m 1)) (m (+ a 1))) (+ a m))
		 (f1))))
      (format-logged #t "~A: opt7 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt7
	 (lambda (type info)
	   'error)))

(let ()
  (define (opt8)
    (let ((val (let () 
		 (let ((x 1)) 
		   (set! x (+ m 1)) 
		   (define m 2) 
		   x))))
      (format-logged #t "~A: opt8 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt8
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" m))))
	       (format *stderr* "opt8 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt9)
    (let ((val (let () 
		 (let ((x 1)) (set! x (and m (length m))) (define m 2) x))))
      (format-logged #t "~A: opt9 got ~S but expected 'error~%~%" (port-line-number) val)))
  (catch #t opt9
	 (lambda (type info)
	   (if (or (not (eq? type 'syntax-error))
		   (not (equal? info '("~A: unbound variable" m))))
	       (format *stderr* "opt9 type: ~A, info: ~A~%" type info))
	   'error)))

(let ()
  (define (opt10)
    (let () 
      (define* (f1 (a (and m (length m))) (m 1)) (+ a m))
      (f1))) ; but not unbound var error!  isn't this a bug?
  (catch #t opt10
	 (lambda (type info)
	   'error)))


(let ((val #f))
  (let-temporarily (((hook-functions *load-hook*) (list (lambda (hook) 
							  (if (or val
								  (defined? 'load-hook-test))
							      (format-logged #t ";*load-hook*: ~A ~A?~%" val load-hook-test))
							  (set! val (hook 'name))))))
    (with-output-to-file "load-hook-test.scm"
      (lambda ()
	(format-logged #t "(define (load-hook-test val) (+ val 1))")))
    (load "load-hook-test.scm")
    (if (or (not (string? val))
	    (not (string=? val "load-hook-test.scm")))
	(format-logged #t ";*load-hook-test* file: ~S~%" val))
    (if (not (defined? 'load-hook-test))
	(format-logged #t ";load-hook-test function not defined?~%")
	(if (not (= (load-hook-test 1) 2))
	    (format-logged #t ";load-hook-test: ~A~%" (load-hook-test 1))))))

(let-temporarily (((hook-functions *error-hook*) ()))
  (test (hook-functions *error-hook*) ())
  (set! (hook-functions *error-hook*) (list (lambda (hook) #f)))
  (test (list? (hook-functions *error-hook*)) #t))

(let-temporarily (((hook-functions *missing-close-paren-hook*) (list (lambda (h) (set! (h 'result) 'incomplete-expr)))))
  (test (catch #t (lambda () (eval-string "(+ 1 2")) (lambda args (car args))) 'incomplete-expr)
  (test (catch #t (lambda () (eval-string "(")) (lambda args (car args))) 'incomplete-expr)
  (test (catch #t (lambda () (eval-string "(abs ")) (lambda args (car args))) 'incomplete-expr))

(let ((h (make-hook 'x)))
  (test (procedure? h) #t)
  (test (eq? h h) #t) 
  (test (eqv? h h) #t)
  (test (equal? h h) #t)
  (test (morally-equal? h h) #t)
  (let ((h1 (copy h)))
    (test (eq? h h1) #f) ; fluctutates...
    (test (morally-equal? h h1) #t))
  (test (hook-functions h) ())
  (test (h) #<unspecified>)
  (test (h 1) #<unspecified>)
  (test (h 1 2) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) (hook 'x)))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h 1) 1)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())
    (let ((f2 (lambda* args (set! ((car args) 'result) ((car args) 'x)))))
      (set! (hook-functions h) (list f2))
      (test (hook-functions h) (list f2))
      (test (h 1) 1)))
  (for-each
   (lambda (arg)
     (test (set! (hook-functions h) arg) 'error))
   (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>)))

(let ((h (make-hook)))
  (test (procedure? h) #t)
  (test (procedure-documentation h) "")
  (test (hook-functions h) ())
  (test (h) #<unspecified>)
  (test (h 1) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) 123))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h) 123)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())
    (let ((f2 (lambda* args (set! ((car args) 'result) 321))))
      (set! (hook-functions h) (list f2))
      (test (hook-functions h) (list f2))
      (test (h) 321))))

(let ((h (make-hook '(a 32) 'b)))
  (test (procedure? h) #t)
  (test (hook-functions h) ())
  (test (h) #<unspecified>)
  (test (h 1) #<unspecified>)
  (test (h 1 2) #<unspecified>)
  (test (h 1 2 3) 'error)
  (let ((f1 (lambda (hook) (set! (hook 'result) (+ (hook 'a) (or (hook 'b) 0))))))
    (set! (hook-functions h) (list f1))
    (test (member f1 (hook-functions h)) (list f1))
    (test (hook-functions h) (list f1))
    (test (h) 32)
    (test (h 1) 1)
    (test (h 1 2) 3)
    (set! (hook-functions h) ())
    (test (hook-functions h) ())))

(let ()
  (for-each
   (lambda (arg)
     (test (make-hook arg) 'error))
   (list "hi" #f 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>)))

(let ((h (make-hook)))
  (let ((f1 (lambda (hook) (if (number? (hook 'result)) (set! (hook 'result) (+ (hook 'result) 1)) (set! (hook 'result) 0)))))
    (test (h) #<unspecified>)
    (set! (hook-functions h) (list f1))
    (test (h) 0)
    (set! (hook-functions h) (list f1 f1 f1))
    (test (h) 2)))

(if (not (defined? 'hook-push))
    (define (hook-push hook func)
      (set! (hook-functions hook) (cons func (hook-functions hook)))))

(let ((h (make-hook)))
  (hook-push h (lambda (hook) (set! (hook 'result) 32)))
  (test (dynamic-wind h h h) 32)
  (test (catch h h h) 32)
  )

(let ((h (make-hook 'x)))
  (hook-push h (lambda (hook) (set! (hook 'result) (hook 'x))))
  (test (continuation? (call/cc h)) #t)
  (set! (hook-functions h) (list (lambda (hook) (set! (hook 'result) (+ 1 (hook 'x))))))
  (test (map h '(1 2 3)) '(2 3 4))
  )

(let ()
  (define-macro (hook . body)
    `(let ((h (make-hook)))
       (set! (hook-functions h) 
	     (list (lambda (h) 
		     (set! (h 'result) (begin ,@body)))))
       h)) 
  (let ((x 0))
    (define hi (hook (set! x 32) (+ 2 3 1)))
    (test (hi) 6)
    (test x 32)))

(let ()
  (define-macro (hooked-catch hook . body)
    `(catch #t 
       (lambda () 
	 ,@body) 
       (lambda args 
	 (let ((val (apply ,hook args)))
	   (if (eq? val #<unspecified>) ; hook did not do anything
	       (apply error args)       ; so re-raise the error
	       val)))))

 (let ((a-hook (make-hook 'error-type :rest 'error-info)))
   (set! (hook-functions a-hook)
         (list (lambda (hook) 
		 ;(format-logged #t "hooked-catch: ~A~%" (apply format #t (car (hook 'error-info))))
		 (set! (hook 'result) 32))))
   (test (hooked-catch a-hook (abs "hi")) 32)
   
   (set! (hook-functions a-hook) ())

   (test (catch #t
	   (lambda ()
	     (hooked-catch a-hook (abs "hi")))
	   (lambda args
	     123))
	 123)
   ))

(let ()
  (define *breaklet* #f)
  (define *step-hook* (make-hook 'code 'e))
  
  (define-macro* (trace/break code . break-points)
    (define (caller tree)
      (if (pair? tree)
	  (cons 
	   (if (pair? (car tree))
	       (if (and (symbol? (caar tree))
			(procedure? (symbol->value (caar tree))))
		   (if (member (car tree) break-points)
		       `(__break__ ,(caller (car tree)))
		       `(__call__ ,(caller (car tree))))
		   (caller (car tree)))
	       (car tree))
	   (caller (cdr tree)))
	  tree))
    `(call-with-exit (lambda (__top__) ,(caller code))))
  
  (define (go . args)
    (and (let? *breaklet*)
	 (apply (*breaklet* 'go) args)))
  
  (define (clear-break)
    (set! *breaklet* #f))
  
  (define-macro (__call__ code)
    `(*step-hook* ',code (curlet)))
  
  (define-macro (__break__ code)
    `(begin
       (call/cc
	(lambda (go)
	  (set! *breaklet* (curlet))
	  (__top__ (format #f "break at: ~A~%" ',code))))
       ,code))
  
  (set! (hook-functions *step-hook*) 
	(list (lambda (hook)
		(set! (hook 'result) (eval (hook 'code) (hook 'e))))
	      (lambda (hook)
		(define (uncaller tree)
		  (if (pair? tree)
		      (cons 
		       (if (and (pair? (car tree))
				(memq (caar tree) '(__call__ __break__)))
			   (uncaller (cadar tree))
			   (uncaller (car tree)))
		       (uncaller (cdr tree)))
		      tree))
		(format (current-output-port) ": ~A -> ~A~40T~A~%" 
			(uncaller (hook 'code)) 
			(hook 'result)
			(if (and (not (eq? (hook 'e) (rootlet)))
				 (not (defined? '__top__ (hook 'e))))
			    (map values (hook 'e)) 
			    "")))))
  
  (let ((str (with-output-to-string
	       (lambda ()
		 (trace/break (let ((a (+ 3 1)) (b 2)) (if (> (* 2 a) b) 2 3)))))))
    (test (or (string=? str ": (+ 3 1) -> 4                         
: (* 2 a) -> 8                         ((a . 4) (b . 2))
: (> (* 2 a) b) -> #t                  ((a . 4) (b . 2))
") 
	      (string=? str ": (+ 3 1) -> 4                         
: (* 2 a) -> 8                         ((b . 2) (a . 4))
: (> (* 2 a) b) -> #t                  ((b . 2) (a . 4))
")) #t)))





;;; --------------------------------------------------------------------------------
;;; HASH-TABLES
;;; --------------------------------------------------------------------------------

(let ((ht (make-hash-table)))
  (test (hash-table? ht) #t)
  (test (equal? ht ht) #t)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (test (let () (hash-table-set! ht 123 "hiho") (hash-table-ref ht 123)) "hiho")
  (test (let () (hash-table-set! ht 3.14 "hi") (hash-table-ref ht 3.14)) "hi")
  (test (let () (hash-table-set! ht pi "hiho") (hash-table-ref ht pi)) "hiho")
  (test (hash-table-ref ht "123") #f)
  (let ((ht1 (copy ht)))
    (test (hash-table? ht1) #t)
    (test (iterator? ht1) #f)
    (test (iterator? (make-iterator ht1)) #t)
    (test (= (length ht) (length ht1)) #t)
    (test (equal? ht ht1) #t)
    (test (eq? ht ht) #t)
    (test (eqv? ht ht) #t)
    (set! (ht 'key) 32)
    (set! (ht1 'key) 123)
    (test (and (= (ht 'key) 32) (= (ht1 'key) 123)) #t)
    (set! (ht "key") 321)
    (test (ht "key") 321)
    (test (ht 'key) 32)
    (set! (ht 123) 43)
    (set! (ht "123") 45)
    (test (ht 123) 43)
    (test (ht "123") 45)
    (test (hash-table-set! ht "1" 1) 1)
    (test (set! (ht "2") 1) 1)
    (test (set! (hash-table-ref ht "3") 1) 1)
    (test (hash-table-ref ht "3") 1))
  (test (let () (set! (hash-table-ref ht 'key) 32) (hash-table-ref ht 'key)) 32)

  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (iterator? arg) #f)
   (test (hash-table-set! arg 'key 32) 'error))
 (list "hi" () -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table 31))
      (ht2 (make-hash-table 31)))
  (if (not (equal? ht1 ht2))
      (format-logged #t ";ht1 and ht2 are empty, but not equal??~%"))

      ;; these first tests take advantage of s7's hashing function
  (hash-table-set! ht1 'abc 1)
  (hash-table-set! ht1 'abcabc 2)
  (hash-table-set! ht1 'abcabcabc 3)
  (hash-table-set! ht2 'abcabcabc 3)  
  (hash-table-set! ht2 'abcabc 2) 
  (hash-table-set! ht2 'abc 1)
  (if (not (equal? ht1 ht2))
      (format-logged #t ";ht1 and ht2 have the same key value pairs, but are not equal??~%"))

  (test (make-hash-table 1 (call-with-exit (lambda (goto) goto))) 'error)
  
  (set! ht2 (make-hash-table 31))
  (hash-table-set! ht2 'abc 1)
  (hash-table-set! ht2 'abcabc 2) 
  (hash-table-set! ht2 'abcabcabc 3)  
  (if (not (equal? ht1 ht2))
      (format-logged #t ";ht1 and ht2 have the same key value pairs in the same order, but are not equal??~%"))
  
  (hash-table-set! ht2 'abc "1")
  (if (equal? ht1 ht2) 
      (format-logged #t ";ht1 and ht2 are equal but values are not~%"))
  (hash-table-set! ht2 'abc 1)
  (if (not (equal? ht1 ht2))
      (format-logged #t ";after reset ht1 and ht2 have the same key value pairs in the same order, but are not equal??~%"))
  (hash-table-set! ht2 1 'abc)
  (if (equal? ht1 ht2)
      (format-logged #t ";ht1 and ht2 are equal but entries are not~%"))
  (hash-table-set! ht1 1 'abc)
  (if (not (equal? ht1 ht2))
      (format-logged #t ";after add ht1 and ht2 have the same key value pairs, but are not equal??~%"))

      ;; these should force chaining in any case
  (set! ht1 (make-hash-table 31))
  (set! ht2 (make-hash-table 60))
  (do ((i 0 (+ i 1)))
      ((= i 100))
    (hash-table-set! ht1 i (* i 2))
    (hash-table-set! ht2 i (* i 2)))
  (if (not (equal? ht1 ht2))
      (format-logged #t ";ht1 and ht2 have the same (integer) key value pairs in the same order, but are not equal??~%"))

  (let ((h1 (hash-table* "a" 1))
          (h2 (hash-table* 'a 1)))
     (set! (h2 "a") 1)
     (set! (h2 'a) #f)
     test (equal? h1 h2) #t)

  (let ((ht (make-hash-table)))
    (set! (ht (expt 2 40)) 40)
    (set! (ht (expt 2 50)) 50)
    (set! (ht (- (expt 2 60))) -60) ; these all hash into 0 unfortunately -- maybe fold halves?
    (test (ht (expt 2 40)) 40)
    (test (ht (expt 2 50)) 50)
    (test (ht (expt 2 60)) #f)
    (test (ht (- (expt 2 60))) -60)
    (test (ht (expt 2 41)) #f))
  
  (set! ht2 (make-hash-table 31))
  (do ((i 99 (- i 1)))
      ((< i 0))
    (hash-table-set! ht2 i (* i 2)))
  (test (hash-table-entries ht2) 100)
  (if (not (equal? ht1 ht2))
      (format-logged #t ";ht1 and ht2 have the same (integer) key value pairs, but are not equal??~%"))
  
  (fill! ht1 ())
  (test (hash-table-entries ht1) 100)
  (test (ht1 32) ()))

(let ((h (make-hash-table)))
  (test (hash-table-entries h) 0)
  (set! (h 'a) 1)
  (test (hash-table-entries h) 1)
  (set! (h 'a) #f)
  (test (hash-table-entries h) 0))

(let ((ht (make-hash-table))
      (l1 '(x y z))
      (l2 '(y x z)))
  (set! (hash-table-ref ht 'x) 123)
  (define (hi)
    (hash-table-ref ht (cadr l1))) ; 123
  (test (hi) #f))

(test (make-hash-table most-positive-fixnum) 'error)
;(test (make-hash-table (+ 1 (expt 2 31))) 'error)  ; out-of-memory error except in clang
(test (make-hash-table most-negative-fixnum) 'error)
(test (make-hash-table 21 eq? 12) 'error)
(test (make-hash-table 21 12) 'error)
(test (make-hash-table eq? eq?) 'error)
(test (make-hash-table eq? eq? 12) 'error)
(test (make-hash-table ()) 'error)
(test (make-hash-table 3 ()) 'error)
(test (make-hash-table eq? ()) 'error)
(test (make-hash-table 0) 'error)
(test (make-hash-table -4) 'error)
(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error))
 (list "hi" #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht (hash-table* :a 1/0)))
  (test (nan? (ht :a)) #t)
  (set! (ht 1/0) :a)
  (test (ht 1/0) :a)) ; is this just by chance?

(let ((ht (make-hash-table))) 
  (define (f1) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i #t)))
  (f1)
  (test (hash-table-entries ht) 100)
  (set! ht (make-hash-table))
  (define (f2) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i 0)))
  (f2)
  (test (hash-table-entries ht) 100)
  (set! ht (make-hash-table))
  (define (f3) (do ((i 0 (+ i 1))) ((= i 100)) (hash-table-set! ht i i)))
  (f3)
  (test (hash-table-entries ht) 100))

(let ((ht (make-hash-table))) 
  (define (f1) (do ((i 0 (+ i 1))) ((= i 1000)) (hash-table-set! ht i #t)))
  (f1)
  (test (hash-table-entries ht) 1000))

(let ((hi (make-hash-table 7)))
  (test (object->string hi) "(hash-table)")
  (set! (hi 1) "1")
  (test (object->string hi) "(hash-table '(1 . \"1\"))")
  (set! (hi -1) "-1")
  (test (or (string=? (object->string hi) "(hash-table '(-1 . \"-1\") '(1 . \"1\"))")
	    (string=? (object->string hi) "(hash-table '(1 . \"1\") '(-1 . \"-1\"))"))
	#t)
  (set! (hi 9) "9")
  (test (or (string=? (object->string hi) "(hash-table '(9 . \"9\") '(-1 . \"-1\") '(1 . \"1\"))")
	    (string=? (object->string hi) "(hash-table '(9 . \"9\") '(1 . \"1\") '(-1 . \"-1\"))"))
	#t)
  (set! (hi -9) "-9")
  (test (or (string=? (object->string hi) "(hash-table '(-9 . \"-9\") '(9 . \"9\") '(-1 . \"-1\") '(1 . \"1\"))")
	    (string=? (object->string hi) "(hash-table '(9 . \"9\") '(1 . \"1\") '(-9 . \"-9\") '(-1 . \"-1\"))"))
	#t)
  (test (hi 1) "1")
  (test (hi -1) "-1")
  (test (hi -9) "-9")
  (set! (hi 2) "2")
  (test (or (string=? (object->string hi) "(hash-table '(-9 . \"-9\") '(9 . \"9\") '(-1 . \"-1\") '(1 . \"1\") '(2 . \"2\"))")
	    (string=? (object->string hi) "(hash-table '(9 . \"9\") '(1 . \"1\") '(2 . \"2\") '(-9 . \"-9\") '(-1 . \"-1\"))"))
	#t)
  (let-temporarily (((*s7* 'print-length) 3))
    (test (or (string=? (object->string hi) "(hash-table '(-9 . \"-9\") '(9 . \"9\") '(-1 . \"-1\") ...)")
	      (string=? (object->string hi) "(hash-table '(9 . \"9\") '(1 . \"1\") '(2 . \"2\") ...)"))
	  #t)
    (set! (*s7* 'print-length) 0)
    (test (object->string hi) "(hash-table ...)")
    (test (object->string (hash-table)) "(hash-table)")))

(let ((ht (make-hash-table 277)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 277) #t)
  (test (hash-table-entries ht) 0)
  (test (let () (hash-table-set! ht 'key 3.14) (hash-table-ref ht 'key)) 3.14)
  (test (let () (hash-table-set! ht "ky" 3.14) (hash-table-ref ht "ky")) 3.14)
  (for-each
   (lambda (arg)
     (test (let () (hash-table-set! ht 'key arg) (hash-table-ref ht 'key)) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2))))

(for-each
 (lambda (arg)
   (test (hash-table? arg) #f))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t #f () #(()) (list 1 2 3) '(1 . 2)))

(test (hash-table? (make-vector 3 ())) #f)
(test (let ((ht (make-hash-table))) (set! (ht 'a) 123) (map values ht)) '((a . 123)))

(let ((ht (make-hash-table)))	
  (test (hash-table-ref ht 'not-a-key) #f)
  (test (hash-table-ref ht "not-a-key") #f)
  (hash-table-set! ht 'key 3/4)
  (hash-table-set! ht "key" "hi")
  (test (hash-table-ref ht "key") "hi")
  (test (hash-table-ref ht 'key) 3/4)
  
  (hash-table-set! ht 'asd 'hiho)
  (test (hash-table-ref ht 'asd) 'hiho)
  (hash-table-set! ht 'asd 1234)
  (test (hash-table-ref ht 'asd) 1234))

(let ((ht (make-hash-table)))
  (define (ht-add h)
    (+ (h 1) (h 2)))
  (hash-table-set! ht 1 2)
  (hash-table-set! ht 2 3)
  (test (ht-add ht) 5))

(let ((let1 (inlet 'a 1))
      (let2 (inlet 'a 1))
      (let3 (inlet 'a 2))
      (let4 (inlet 'b 1))
      (let5 (inlet 'a 2 'a 1)))
  (test (equal? let1 let2) #t)
  (test (equal? let1 let3) #f)
  (test (equal? let1 let5) #t)
  (let ((hash1 (hash-table* let1 32)))
    (test (integer? (hash1 let1)) #t)
    (test (integer? (hash1 let2)) #t)
    (test (integer? (hash1 let3)) #f)
    (test (integer? (hash1 let4)) #f)
    (test (integer? (hash1 let5)) #t)))

(test ((hash-table* 1.5 #t #f #t) #f) #t) ; this is checking hash_float if debugging
(test ((hash-table* 1.5 #t 1 #t) 1) #t)

(let ((let1 (inlet 'a 1 'b 2))
      (let2 (inlet 'b 2 'a 1))
      (let3 (inlet 'a 1 'b 1)))
  (test (equal? let1 let2) #t)
  (let ((hash1 (hash-table* let1 32)))
    (test (integer? (hash1 let1)) #t)
    (test (integer? (hash1 let2)) #t)
    (test (integer? (hash1 let3)) #f)))

(let ((hash1 (hash-table* 'a 1 'b 2))
      (hash2 (hash-table* 'b 2 'a 1)))
  (test (equal? hash1 hash2) #t)
  (let ((hash3 (hash-table* hash1 32)))
    (test (integer? (hash3 hash1)) #t)
    (test (integer? (hash3 hash2)) #t)))

(let ((hash1 (hash-table* 'b 2 'a 1)))
  (let ((hash2 (make-hash-table (* (length hash1) 2))))
    (set! (hash2 'a) 1)
    (set! (hash2 'b) 2)
    (test (equal? hash1 hash2) #t)
    (let ((hash3 (make-hash-table (* 2 (length hash2)))))
      (set! (hash3 hash1) 32)
      (test (integer? (hash3 hash1)) #t)
      (test (integer? (hash3 hash2)) #t))))

(for-each
 (lambda (arg)
   (test (hash-table-ref arg 'key) 'error))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((ht1 (make-hash-table 653))
      (ht2 (make-hash-table 277)))
  (test (equal? ht1 ht2) #t) ; equal? because both are empty
  (hash-table-set! ht1 'key 'hiho)
  (hash-table-set! ht2 (hash-table-ref ht1 'key) 3.14)
  (test (>= (length ht1) 653) #t)
  (test (hash-table-ref ht2 'hiho) 3.14)
  (test (hash-table-ref ht2 (hash-table-ref ht1 'key)) 3.14))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1) 'hi)
   (let ((ht2 (make-hash-table)))
      (set! (ht2 1) ht1)
      (test ((ht2 1) 1) 'hi)))

(let ((ht1 (make-hash-table)))
   (set! (ht1 1/0) "NaN!")
   (let ((nan 1/0))
      (test (ht1 nan) "NaN!")
      (set! (ht1 nan) 0)
      (test (ht1 nan) 0)
      (if (not with-windows)
	  (test (object->string ht1) "(hash-table '(nan.0 . 0))"))))

(unless with-bignums
  (let ((ht1 (make-hash-table)))
    (set! (ht1 1) "1")
    (set! (ht1 1.0) "1.0")
    (test (ht1 1) "1")
    (set! (ht1 1/0) "nan")
    (test (ht1 0/0) "nan")
    (set! (ht1 (/ (log 0) (log 0))) "nan-nani")
    (test (ht1 (/ (log 0) (log 0))) "nan-nani")
    (test (ht1 (- 0/0)) "nan")
    (test (ht1 (real-part (/ (log 0) (log 0)))) "nan")
    (test (ht1 (complex 0/0 1/0)) "nan-nani")
    (set! (ht1 (real-part (log 0))) "-inf")
    (test (ht1 (real-part (log 0))) "-inf")
    (set! (ht1 (- (real-part (log 0)))) "inf")
    (test (ht1 (- (real-part (log 0)))) "inf")
    (set! (ht1 (log 0)) "log(0)")
    (test (ht1 (log 0)) "log(0)")
    (set! (ht1 (complex 80143857/25510582 1)) "pi+i")
    (test (ht1 (complex pi (- 1.0 1e-16))) "pi+i")))

(let ((ht (make-hash-table)))
  (set! (ht (string #\a #\null #\b)) 1)
  (test (ht (string #\a #\null #\b)) 1)
  (test (ht (string #\a)) #f)
  (set! (ht (string #\a #\null #\b)) 12)
  (test (ht (string #\a #\null #\b)) 12)
  (fill! ht #f)
  (test (hash-table-entries ht) 0)
  (set! (ht #u8(3 0 21)) 1)
  (test (ht #u8(3 0 21)) 1))

(let ((ht (hash-table* 'a #t)))
  (test (hash-table-entries ht) 1)
  (do ((i 0 (+ i 1))) ((= i 10)) (set! (ht 'a) #f) (set! (ht 'a) #t))
  (test (hash-table-entries ht) 1))

(when with-bignums
  (let ((ht (make-hash-table)))
    (set! (ht pi) 1)
    (test (ht pi) 1)
    (set! (ht (bignum "1")) 32)
    (test (ht (bignum "1")) 32)
    (set! (ht (/ (bignum "11") (bignum "3"))) 12)
    (test (ht (/ (bignum "11") (bignum "3"))) 12)
    (set! (ht (bignum "1+i")) -1)
    (test (ht (bignum "1+i")) -1)
    (set! (ht 3) 2)
    (test (ht 3) 2)
    (set! (ht 3.0) 3)
    (test (ht 3.0) 3)))
    
(test (hash-table?) 'error)
(test (hash-table? 1 2) 'error)

(test (make-hash-table most-positive-fixnum) 'error)
(test (make-hash-table most-negative-fixnum) 'error)
(test (make-hash-table 10 1) 'error)

(let ((ht (make-hash-table)))
  (test (hash-table? ht ht) 'error)
  (test (hash-table-ref ht #\a #\b) 'error)
  (test (hash-table-ref ht) 'error)
  (test (hash-table-ref) 'error)
  (test (hash-table-set!) 'error)
  (test (hash-table-set! ht) 'error)
  (test (hash-table-set! ht #\a) 'error)
  (test (hash-table-set! ht #\a #\b #\c) 'error)
  (set! (ht 'key) 32)
  (test (fill! ht 123) 123)
  (test (ht 'key) 123)
  (set! (ht 'key) 32)
  (test (ht 'key) 32)
  (set! (ht :key) 123)
  (test (ht 'key) 32)
  (test (ht :key) 123)
  (fill! ht ())
  (test (ht 'key) ()))

(let ((ht (make-hash-table)))
  (test (hash-table-set! ht #\a 'key) 'key)
  (for-each
   (lambda (arg)
     (test (hash-table-set! ht arg 3.14) 3.14))
   (list #\a #(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (for-each
   (lambda (arg)
     (test (hash-table-ref ht arg) 3.14))
   (list #\a #(1 2 3) 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  (test (length ht 123) 'error))

(for-each
 (lambda (arg)
   (test (make-hash-table arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ()
 (define ht (make-hash-table))
 (set! (ht 123) "123")
 (set! (ht 456) "456")
 (define hti (make-iterator ht))
 (test (iterator? hti) #t)
 (test (object->string hti) "#<iterator: hash-table>")
 (test (equal? hti hti) #t)
 (test (eq? hti hti) #t)
 (test (eqv? hti hti) #t)
 (test (morally-equal? hti hti) #t)

 (let ((hti2 hti))
   (test (equal? hti2 hti) #t)
   (test (morally-equal? hti2 hti) #t)
   (set! hti2 (copy hti))
   (test (equal? hti2 hti) #t)
   (test (morally-equal? hti2 hti) #t)
   (test (let ((val (hti2))) (or (equal? val '(123 . "123")) (equal? val '(456 . "456")))) #t) ; order depends on table size
   (test (equal? hti2 hti) #f)
   (test (morally-equal? hti2 hti) #f)
   )

 (let ((vals (list (hti) (hti))))
   (if (not (equal? (sort! vals (lambda (a b) (< (car a) (car b)))) '((123 . "123") (456 . "456"))))
       (format-logged #t ";iterator: ~A~%" vals))
   (let ((val (hti)))
     (if (not (eof-object? val))
	 (format-logged #t ";iterator at end: ~A~%" val)))
   (let ((val (hti)))
     (if (not (eof-object? val))
	 (format-logged #t ";iterator at end (2): ~A~%" val)))))

(test (make-iterator) 'error)
(test (make-iterator (make-hash-table) 1) 'error)
(test (iterator?) 'error)
(test (iterator? 1 2) 'error)

(let ()
  (define (get-iter)
    (let ((ht (hash-table '(a . 1) '(b . 2))))
      (test (hash-table-entries ht) 2)
      (make-iterator ht)))
  (let ((hti (get-iter)))
    (gc)
    (let ((a (hti)))
      (let ((b (hti)))
	(let ((c (hti)))
	  (test (let ((lst (list a b c)))
		  (or (equal? lst '((a . 1) (b . 2) #<eof>))
		      (equal? lst '((b . 2) (a . 1) #<eof>))))
		#t))))))

(let ((ht1 (make-hash-table))
      (ht2 (make-hash-table)))
  (test (equal? ht1 ht2) #t)
  (test (equal? ht1 (make-vector (length ht1) ())) #f)
  (hash-table-set! ht1 'key 'hiho)
  (test (equal? ht1 ht2) #f)
  (hash-table-set! ht2 'key 'hiho)
  (test (equal? ht1 ht2) #t)

  (hash-table-set! ht1 'a ())
  (test (ht1 'a) ())
  )

(let ((ht (make-hash-table 1)))
  (test (>= (length ht) 1) #t)
  (set! (ht 1) 32)
  (test (>= (length ht) 1) #t))

(let ((ht (hash-table '("hi" . 32) '("ho" . 1))))
  (test (hash-table-entries ht) 2)
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table* "hi" 32 "ho" 1)))
  (test (hash-table-entries ht) 2)
  (test (ht "hi") 32)
  (test (ht "ho") 1))

(let ((ht (hash-table)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 1) #t)
  (test (ht 1) #f))

(let ((ht (hash-table*)))
  (test (hash-table? ht) #t)
  (test (>= (length ht) 1) #t)
  (test (ht 1) #f))

(for-each
 (lambda (arg)
   (test (hash-table arg) 'error)
   (test (hash-table* arg) 'error)
   (test ((hash-table* 'a arg) 'a) arg)
   (test ((hash-table* arg 'a) arg) 'a))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(let ((ht (make-hash-table))
      (lst (list 1)))
  (set-cdr! lst lst)
  (set! (ht lst) lst)
  (let ((lst1 (list 1 2)))
    (set-cdr! (cdr lst1) lst1)
    (set! (ht lst1) lst1)
    (test (ht lst) lst)
    (test (ht lst1) lst1)
    (test (or (string=? (object->string ht) "(hash-table '(#1=(1 2 . #1#) . #1#) '(#2=(1 . #2#) . #2#))")
	      (string=? (object->string ht) "(hash-table '(#1=(1 . #1#) . #1#) '(#2=(1 2 . #2#) . #2#))"))
	  #t)))

(test (set! (hash-table) 1) 'error)
(test (set! (hash-table*) 1) 'error)
(test (set! (make-hash-table) 1) 'error)

;; no null hash-tables?

(let ((ht (make-hash-table)))
  (test (map (lambda (x) x) ht) ())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 0)
  (test (map (lambda (x y) (cons x y)) (list 1 2 3) ht) ())
  ;(test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) #(1 2 3) ht) ctr) 0) ; this is now an error 15-Jan-15
  (test (map (lambda (x y) (cons x y)) ht "123") ())
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht ()) ctr) 'error) ; 2 args

  (let ((rt (reverse ht)))
    (test (map (lambda (x) x) rt) ())
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 0))

  (set! (ht 1) 32)
  ;; these need to be independent of entry order
  
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 1)
  (test (map (lambda (x y) (cons x y)) () ht) ())
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht "") ctr) 0)
  (test (sort! (map (lambda (x y) (max (cdr x) y)) ht (list 1 2 3)) <) '(32))
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (max (cdr x) y))) ht #(1 2 3)) ctr) 32)

  (let ((rt (reverse ht)))
    (test (equal? (rt 32) 1) #t)
    (test (equal? (rt 1) #f) #t)
    (test (ht (rt 32)) 32)
    (test (sort! (map (lambda (x) (cdr x)) rt) <) '(1))
    (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) rt) ctr) 1)
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 32) 123)
    (test (rt 32) 123)
    (test (ht 32) #f)
    (test (ht 1) 32))

  (set! (ht 2) 1)
  (test (ht (ht 2)) 32)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 2)
  (set! (ht 3) 123)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(1 32 123))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)
  (test (let ((ctr 0)) (for-each (lambda (x y) (set! ctr (+ ctr 1))) ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1)) ctr) 1)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12" ht '(1 2)) ctr) 2)
  (test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr 1))) "12345" ht '(1 2 3 4 5 6)) ctr) 3)
  
  (test (sort! (map (lambda (x y) (max x (cdr y))) (list -1 -2 -3 -4) ht) <) '(1 32 123))
  (test (let ((sum 0)) (for-each (lambda (x y) (set! sum (+ sum x (cdr y)))) #(10 20 30) ht) sum) 216)

  
  (let ((rt (reverse ht)))
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht))
  
  (set! (ht (list 1 2 3)) "hi")
  (test (ht '(1 2 3)) "hi")
  (test (ht 2) 1)
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 4)
  (set! (ht "hi") 2)
  (test (ht "hi") 2)
  (test (ht (ht (ht "hi"))) 32)

  (let ((rt (reverse ht)))
    (test (rt "hi") '(1 2 3))
    (test (rt 2) "hi")
    (for-each (lambda (x) (test (ht (rt (cdr x))) (cdr x)) (test (rt (ht (car x))) (car x))) ht)
    (set! (rt 2) "ho")
    (test (rt 2) "ho")
    (test (ht '(1 2 3)) "hi")
    (set! (rt 123) 321)
    (test (rt 123) 321)
    (test (ht 3) 123))

  (fill! ht 0)
  (set! (ht "hi") 1)
  (set! (ht "hoi") 2)
  (test (sort! (map (lambda (x) (cdr x)) ht) <) '(0 0 0 0 1 2))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 6)
  
  (let ((rt (reverse ht)))
    (test (rt 2) "hoi")
    (set! (rt 2) "ha")
    (test (ht "hoi") 2))

  (set! (ht #\a) #\b)
  (test (ht #\a) #\b)
  (test (ht "hi") 1)

  (set! ht (hash-table))
  (set! (ht #(1)) #(2))
  (test (ht #(1)) #(2))
  (set! (ht '(1)) '(3))
  (set! (ht "1") "4")
  ;(set! (ht ht) "5")
  ;(test (ht ht) "5")
  (test (ht '(1)) '(3))
  (test (let ((ctr 0)) (for-each (lambda (x) (set! ctr (+ ctr 1))) ht) ctr) 3)  
    
  (let ((rt (reverse ht)))
    ;(test (rt "5") ht)
    (test (rt "4") "1")
    (for-each (lambda (x) 
		(test (ht (rt (cdr x))) (cdr x))
		(test (rt (ht (car x))) (car x)))
	      ht))
  )  

(let ((ht (make-hash-table)))
  (let ((str (string (integer->char 255)))
	(u8 #u8(254 0))
	(rl 1e18)
	(int most-negative-fixnum)
	(rat (/ 1 most-negative-fixnum)))
    (set! (ht str) 1)
    (set! (ht u8) 2)
    (set! (ht rl) 3)
    (set! (ht int) 4)
    (set! (ht rat) 5)
    (test (ht str) 1) 
    (test (ht u8) 2)
    (test (ht rl) 3)
    (test (ht int) 4)
    (test (ht rat) 5)))

(let ((ht1 (make-hash-table 32))
      (ht2 (make-hash-table 1024)))
  (do ((i 0 (+ i 1)))
      ((= i 256))
    (let ((str (number->string i)))
      (set! (ht1 str) i)
      (set! (ht2 i) str)))
  (let ((cases 0))
    (for-each
     (lambda (a b)
       (if (not (equal? (string->number (car a)) (cdr a)))
	   (format-logged #t ";hash-table for-each (str . i): ~A?~%" a))
       (if (not (equal? (number->string (car b)) (cdr b)))
	   (format-logged #t ";hash-table for-each (i . str): ~A?~%" b))
       (set! cases (+ cases 1)))
     ht1 ht2)
    (if (not (= cases 256))
	(format-logged #t ";hash-table for-each cases: ~A~%" cases)))
  (let ((iter1 (make-iterator ht1))
	(iter2 (make-iterator ht2)))
    (test (equal? iter1 iter2) #f)
    (test (morally-equal? iter1 iter2) #f)
    (test (iterator? iter2) #t)
    (let ((cases 0))
      (do ((a (iter1) (iter1))
	   (b (iter2) (iter2)))
	  ((or (eof-object? a)
	       (eof-object? b)))
	(if (not (equal? (string->number (car a)) (cdr a)))
	    (format-logged #t ";hash-table iter1 (str . i): ~A?~%" a))
	(if (not (equal? (number->string (car b)) (cdr b)))
	    (format-logged #t ";hash-table iter2 (i . str): ~A?~%" b))
	(set! cases (+ cases 1)))
      (if (not (= cases 256))
	  (format-logged #t ";hash-table iter1/2 cases: ~A~%" cases)))))

(let ((ht (make-hash-table 31)))
  (let ((ht1 (make-hash-table 31)))
    (set! (ht1 'a1) 'b1)
    (set! (ht 'a0) ht1)
    (test ((ht 'a0) 'a1) 'b1)
    (test (hash-table-ref ht 'a0 'a1) 'b1)
    (test (ht 'a0 'a1) 'b1)))

(let ((ht (make-hash-table 31))
      (e (curlet)))
  (define (a-func a) (+ a 1))
  (define-macro (a-macro a) `(+ 1 , a))
  (define (any-func a) (let ((x a)) (lambda () x)))

  (set! (ht abs) 1)
  (set! (ht begin) 2)
  (set! (ht quasiquote) 3)
  (set! (ht a-func) 4)
  (set! (ht a-macro) 5)
  (set! (ht (any-func 6)) 6)
  (set! (ht e) 7)
  (test (ht e) 7)
  (set! (ht (rootlet)) 8)
  (test (ht abs) 1)
  (test (ht round) #f)
  (test (ht quasiquote) 3)
  (test (ht begin) 2)
  (test (ht lambda) #f)
  (test (ht a-func) 4)
  (test (ht a-macro) 5)
  (test (ht (any-func 6)) #f)
  (test (ht (rootlet)) 8)
  (call-with-exit
   (lambda (return)
     (set! (ht return) 9)
     (test (ht return) 9)))
  ;(set! (ht ht) 10)
  ;(test (ht ht) 10)
  )
  

(test (let ((h1 (hash-table '(a . 1) '(b . 2))) (h2 (make-hash-table 31))) (set! (h2 'a) 1) (set! (h2 'b) 2.0) (morally-equal? h1 h2)) #t)
(test (let ((h1 (hash-table '(a . 1) '(b . 2))) (h2 (make-hash-table 31))) (set! (h2 'a) 1.0) (set! (h2 'b) 2) (morally-equal? (list h1) (list h2))) #t)

(test (let ((h1 (hash-table* 'a 1 'b 2)) (h2 (make-hash-table 31))) (set! (h2 'a) 1) (set! (h2 'b) 2.0) (morally-equal? h1 h2)) #t)
(test (let ((h1 (hash-table* 'a 1 'b 2)) (h2 (make-hash-table 31))) (set! (h2 'a) 1.0) (set! (h2 'b) 2) (morally-equal? (list h1) (list h2))) #t)

;(test (let ((ht (make-hash-table))) (hash-table-set! ht ht 1) (ht ht)) #f)
; this is #f now because the old ht is not equal to the new one (different number of entries)
;(test (let ((ht (make-hash-table))) (hash-table-set! ht ht ht) (equal? (ht ht) ht)) #t)

(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (object->string ht)) "#1=(hash-table '(a . #1#))")
(test (let ((h1 (make-hash-table))) (hash-table-set! h1 "hi" h1) (object->string h1)) "#1=(hash-table '(\"hi\" . #1#))")
(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (morally-equal? ht (copy ht))) #t)
(test (let ((ht (make-hash-table))) (hash-table-set! ht 'a ht) (equal? ht (copy ht))) #t)

(test (hash-table 'a 1 'b) 'error)

;; there's no real need for multidim hashes:

(let ((ht (make-hash-table)))
   (set! (ht (cons 'a 1)) 'b)
   (set! (ht (cons 'a 2)) 'c)
   (set! (ht (cons 'b 1)) 'd)
   (test (ht '(a . 1)) 'b)
   (test (ht '(b . 1)) 'd)
   (set! (ht '(a . 2)) 32)
   (test (ht '(a . 2)) 32)
   (let ((lst1 (list 1))
	 (lst2 (list 1)))
     (set-car! lst1 lst2)
     (set-car! lst2 lst1)
     (set! (ht lst1) 32)
     (set! (ht lst2) 3)
     (test (equal? lst1 lst2) #t)
     (test (ht lst1) 3)
     (test (ht lst2) 3)))

(let ((ht (make-hash-table)))
  (set! (ht 1.0) 'a)
  (set! (ht 2.0) 'b)
  (set! (ht 3.0) 'c)
  (test (ht 2.0) 'b)
  (set! (ht 2.0) 'd)
  (test (ht 2.0) 'd)
  (test (ht 0.0) #f)
  (test (ht 1.0) 'a))

(let ((ht (make-hash-table)))
  (test (ht) 'error)
  (test (ht 0 1) 'error))

(let ((h (hash-table* 'a (hash-table* 'b 2 'c 3)
		      'b (hash-table* 'b 3 'c 4))))
  (test (h 'a 'b) 2)
  (test (h 'b 'b) 3)
  (test (h 'a 'c) 3))

(let ()
  (define-macro (memoize f)
    `(define ,f (let ((ht (make-hash-table))
		      (old-f ,f))
		  (lambda args
		    (or (ht args)
			(let ((new-val (apply old-f args)))
			  (set! (ht args) new-val)
			  new-val))))))

  (define (our-abs num) (abs num))
  (memoize our-abs)
  (num-test (our-abs -1) 1)
  (with-let (funclet our-abs)
    (test (ht '(-1)) 1)))

(let ()
  (define-macro (define-memoized name&arg . body)
    (let ((arg (cadr name&arg))
	  (memo (gensym "memo")))
      `(define ,(car name&arg)
	 (let ((,memo (make-hash-table)))
	   (lambda (,arg)
	     (or (,memo ,arg)
		 (set! (,memo ,arg) (begin ,@body))))))))
  
  (define-memoized (f1 abc) (+ abc 2))
  (test (f1 3) 5)
  (test (f1 3) 5)
  (test (f1 2) 4)
  (let ((ht (call-with-exit
	     (lambda (return)
	       (for-each (lambda (x)
			   (if (hash-table? (cdr x))
			       (return (cdr x))))
			 (outlet (funclet f1)))
	       #f))))
    (if (not (hash-table? ht))
	(format-logged #t ";can't find memo? ~A~%" (let->list (outlet (funclet f1))))
	(test (length (map (lambda (x) x) ht)) 2))))

(let ()
  (define-macro (define-memoized name&args . body)
    (let ((args (cdr name&args))
	  (memo (gensym "memo")))
      `(define ,(car name&args)
	 (let ((,memo (make-hash-table)))
	   (lambda ,args
	     (or (,memo (list ,@args))
		 (set! (,memo (list ,@args)) (begin ,@body))))))))

  (define (ack m n)
    (cond ((= m 0) (+ n 1))
	  ((= n 0) (ack (- m 1) 1))
	  (else (ack (- m 1) (ack m (- n 1))))))

  (define-memoized (ack1 m n)
    (cond ((= m 0) (+ n 1))
	  ((= n 0) (ack1 (- m 1) 1))
	  (else (ack1 (- m 1) 
		      (ack1 m (- n 1))))))

  (test (ack 2 3) (ack1 2 3)))

 
(let ((ht (make-hash-table)))
  (test (eq? (car (catch #t (lambda () (set! (ht) 2)) (lambda args args))) 'wrong-number-of-args) #t)
  (test (eq? (car (catch #t (lambda () (set! (ht 0 0) 2)) (lambda args args))) 'syntax-error) #t) ; attempt to apply boolean #f to (0)
  (test (eq? (car (catch #t (lambda () (set! ((ht 0) 0) 2)) (lambda args args))) 'syntax-error) #t))

(let ()
  (define (merge-hash-tables . tables)
    (apply hash-table (apply append (map (lambda (table) (map values table)) tables))))
  (let ((ht (merge-hash-tables (hash-table '(a . 1) '(b . 2)) (hash-table '(c . 3)))))
    (test (ht 'c) 3))
  (test ((append (hash-table '(a . 1) '(b . 2)) (hash-table '(c . 3))) 'c) 3))

;;; test the eq-func business

(let ((ht (make-hash-table 8 eq?)))
  (test (hash-table-ref ht 'a) #f)
  (hash-table-set! ht 'a 1)
  (hash-table-set! ht 'c 'd)
  (test (hash-table-ref ht 'a) 1)
  (hash-table-set! ht "hi" 3)
  (test (hash-table-ref ht "hi") #f)
  (set! (ht '(a . 1)) "ho")
  (test (ht '(a . 1)) #f)
  (let ((ht1 (copy ht)))
    (test (ht1 'a) 1)
    (test (ht1 "hi") #f)
    (set! (ht1 #\a) #\b)
    (test (ht1 #\a) #\b)
    (test (ht #\a) #f)
    (let ((ht2 (reverse ht1)))
      (test (ht1 #\a) #\b)
      (test (ht2 #\b) #\a)
      (test (ht2 'd) 'c)))
  (do ((i 0 (+ i 1)))
      ((= i 32))
    (set! (ht (string->symbol (string-append "g" (number->string i)))) i))
  (test (ht 'a) 1)
  (test (ht 'g3) 3)
  (set! (ht ht) 123)
  (test (ht ht) 123))

(let ((ht (make-hash-table 31 string=?)))
  (test (length ht) 32)
  (set! (ht "hi") 'a)
  (test (ht "hi") 'a)
  (test (ht "Hi") #f)
  (set! (ht 32) 'b)
  (test (ht 32) #f)
  )

(let ((ht (make-hash-table 31 char=?)))
  (test (length ht) 32)
  (set! (ht #\a) 'a)
  (test (ht #\a) 'a)
  (test (ht #\A) #f)
  (set! (ht 32) 'b)
  (test (ht 32) #f)
  )

(unless pure-s7
  (let ((ht (make-hash-table 31 string-ci=?)))
    (test (length ht) 32)
    (set! (ht "hi") 'a)
    (test (ht "hi") 'a)
    (test (ht "Hi") 'a)
    (test (ht "HI") 'a)
    (set! (ht 32) 'b)
    (test (ht 32) #f)
    )
  
  (let ((ht (make-hash-table 31 char-ci=?)))
    (test (length ht) 32)
    (set! (ht #\a) 'a)
    (test (ht #\a) 'a)
    (test (ht #\A) 'a)
    (set! (ht 32) 'b)
    (test (ht 32) #f)
    ))

(let ((ht (make-hash-table 31 =)))
  (test (length ht) 32)
  (set! (ht 1) 'a)
  (test (ht 1.0) 'a)
  (test (ht 1+i) #f)
  (set! (ht 32) 'b)
  (test (ht 32) 'b)
  (set! (ht 1/2) 'c)
  (test (ht 0.5) 'c)
  )

(let ((ht (make-hash-table 31 eqv?)))
  (test (length ht) 32)
  (set! (ht 1) 'a)
  (test (ht 1.0) #f)
  (set! (ht 2.0) 'b)
  (test (ht 2.0) 'b)
  (set! (ht 32) 'b)
  (test (ht 32) 'b)
  (set! (ht #\a) 1)
  (test (ht #\a) 1)
  (set! (ht ()) 2)
  (test (ht ()) 2)
  (set! (ht abs) 3)
  (test (ht abs) 3)
  )

(let ((ht (make-hash-table 8 (cons string=? (lambda (a) (string-length a))))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons (lambda (a b) (string=? a b)) string-length))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons (lambda (a b) (string=? a b)) (lambda (a) (string-length a))))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((ht (make-hash-table 8 (cons string=? string-length))))
  (set! (ht "a") 'a)
  (test (ht "a") 'a)
  (set! (ht "abc") 'abc)
  (test (ht "abc") 'abc))

(let ((h (make-hash-table 8 equal?)))
  (set! (h (make-int-vector 3 0)) 3)
  (test (h (make-int-vector 3 0)) 3)
  (test (h (make-vector 3 0)) #f)
  (test (h (make-float-vector 3 0)) #f)
  (let ((x 1.0)
	(y (+ 1.0 (* 0.5 (*s7* 'morally-equal-float-epsilon))))
	(z (+ 1.0 (* 2 (*s7* 'morally-equal-float-epsilon)))))
    (set! (h x) 12)
    (test (h x) 12)
    ;(test (h y) #f)
    ;(test (h z) #f)
    ;default vs explicit equal? here -- sigh
    ))

(let ((h (make-hash-table 8 morally-equal?)))
  (set! (h (make-int-vector 3 0)) 3)
  (test (h (make-int-vector 3 0)) 3)
  (test (h (make-vector 3 0)) 3)
  (test (h (make-float-vector 3 0)) 3)
  (let ((x 1.0)
	(y (+ 1.0 (* 0.5 (*s7* 'morally-equal-float-epsilon))))
	(z (+ 1.0 (* 2 (*s7* 'morally-equal-float-epsilon)))))
    (set! (h x) 12)
    (test (h x) 12)
    (test (h y) 12)
    (test (h z) #f)
    (set! (h 1/10) 3)
    (test (h 0.1) 3)
    (set! (h #(1 2.0)) 4)
    (test (h (vector 1 2)) 4)
    (set! (h 1.0) 5)
    (test (h 1) 5)
    (set! (h (list 3)) 6)
    (test (h (list 3.0)) 6)
    ))

(when with-block
  (let ((ht (make-hash-table 31 (cons hash_heq hash_hloc))))
    (test (length ht) 32)
    (set! (ht 'a) 'b)
    (test (ht 'a) 'b) 
    (test (ht 1) #f)
    (let ((ht1 (reverse ht)))
      (test (ht1 'b) 'a))
    ))

(let ((ht (make-hash-table 31 morally-equal?))
      (ht1 (make-hash-table 31)))
  (test (length ht) 32)
  (test (equal? ht ht1) #t) 
  (set! (ht 3) 1)
  (test (ht 3) 1)
  (set! (ht1 3) 1)
  (test (equal? ht ht1) #f)
  (set! (ht 3.14) 'a)
  (test (ht 3.14) 'a)
  (set! (ht "hi") 123)
  (test (ht "hi") 123)
  (set! (ht 1/0) #<eof>)
  (test (ht 1/0) #<eof>))

(let ((ht (make-hash-table 31 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
  (test (hash-table-ref ht 'a) #f)
  (hash-table-set! ht 'a 1)
  (hash-table-set! ht 'c 'd)
  (test (hash-table-ref ht 'a) 1)
  (hash-table-set! ht "hi" 3)
  (test (hash-table-ref ht "hi") #f)
  (set! (ht '(a . 1)) "ho")
  (test (ht '(a . 1)) #f)
  (let ((ht1 (copy ht)))
    (test (ht1 'a) 1)
    (test (ht1 "hi") #f)
    (set! (ht1 #\a) #\b)
    (test (ht1 #\a) #\b)
    (test (ht #\a) #f)))

(when (provided? 'snd)
  (let ((ht (make-hash-table 31 (cons equal? mus-type))))
    (let ((g1 (make-oscil 100))
	  (g2 (make-oscil 100)))
      (set! (ht g1) 32)
      (test (ht g1) 32)
      (test (ht g2) 32)
      (test (equal? g1 g2) #t))))


#|
(define constants (vector 1)) ; etc -- see tauto.scm
(define ops (list eq? eqv? equal? morally-equal? = char=? string=? char-ci=? string-ci=?
		  (cons string=? (lambda (a) (string-length a)))
		  (cons (lambda (a b) (string=? a b)) string-length)))
(for-each
 (lambda (op)
   (for-each
    (lambda (val)
      (let ((h (make-hash-table 8 op)))
	(catch #t
	  (lambda ()
	    (set! (h val) #t)
	    (if (not (eq? (h val) (op val val)))
		(format *stderr* "~A ~A: ~A ~A~%" op val (h val) (op val val))))
	  (lambda any #f))))
    constants))
 ops)
|#

(let ()
  (let ((ht (make-hash-table 31 (cons (lambda (a b) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (let ((ht (make-hash-table 31 (cons (lambda* (a b) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (let ((ht (make-hash-table 31 (cons (lambda* (a (b 0)) (eq? a b)) (lambda (a) 0)))))
    (hash-table-set! ht 'a 1)
    (test (ht 'a) 1))
  (test (let ((ht (make-hash-table 31 (list (eq? a b)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons abs +))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons eq? float-vector-ref))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (dilambda (lambda (a) (eq? a b)) (lambda (a) 0)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (lambda a (eq? car a) (cadr s)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (cons (lambda (a b c) (eq? a b)) (lambda (a) 0)))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 (define-macro (_m_ . args) #f))))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 abs)))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error)
  (test (let ((ht (make-hash-table 31 list-set!)))
	  (hash-table-set! ht 'a 1)
	  (ht 'a))
	'error))

(let ()
  (define (test-hash size)
    (let ((c #t))
      (let ((int-hash (make-hash-table (max size 511) (cons (lambda (a b) (= a b)) (lambda (a) a)))))
	(do ((i 0 (+ i 1))) 
	    ((= i size)) 
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))	
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(0) ~D -> ~D~%" i x)))))
      (let ((int-hash (make-hash-table (max size 511) (cons (lambda (a b) (and c (= a b))) (lambda (a) a)))))
	(do ((i 0 (+ i 1))) 
	    ((= i size)) 
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))	
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(1) ~D -> ~D~%" i x)))))
      (let ((int-hash (make-hash-table (max size 511) (let ((c #f)) (cons (lambda (a b) (and (not c) (= a b))) (lambda (a) a))))))
	(do ((i 0 (+ i 1))) 
	    ((= i size)) 
	  (hash-table-set! int-hash i i))
	(do ((i 0 (+ i 1)))	
	    ((= i size))
	  (let ((x (hash-table-ref int-hash i)))
	    (if (not (= x i)) (format *stderr* ";test-hash(2) ~D -> ~D~%" i x)))))
      ))

  (test-hash 10))

#|
;; another problem
(let ((ht (make-hash-table))
      (lst (list 1)))
  (set! (ht lst) 32)
  (let ((v1 (ht '(1)))
	(v2 (ht '(2))))
    (set-car! lst 2)
    (let ((v3 (ht '(1)))
	  (v4 (ht '(2))))
      (list v1 v2 v3 v4)))) ; 32 #f #f #f

(let ((ht (make-hash-table))
      (lst (list 1)))
  (set! (ht (copy lst)) 32)
  (let ((v1 (ht '(1)))
	(v2 (ht '(2))))
    (set-car! lst 2)
    (let ((v3 (ht '(1)))
	  (v4 (ht '(2))))
      (list v1 v2 v3 v4)))) ; 32 #f 32 #f

;; so copy the key if it might be changed 
|#



;;; --------------------------------------------------------------------------------
;;; some implicit index tests

(test (#(#(1 2) #(3 4)) 1 1) 4)
(test (#("12" "34") 0 1) #\2)
(test (#((1 2) (3 4)) 1 0) 3)
(test (#((1 (2 3))) 0 1 0) 2)
(test ((vector (hash-table '(a . 1) '(b . 2))) 0 'a) 1)
(test ((list (lambda (x) x)) 0 "hi") "hi")
(test (let ((lst '("12" "34"))) (lst 0 1)) #\2)
(test (let ((lst (list #(1 2) #(3 4)))) (lst 0 1)) 2)
(test (#2d(("hi" "ho") ("ha" "hu")) 1 1 0) #\h)
(test ((list (lambda (a) (+ a 1)) (lambda (b) (* b 2))) 1 2) 4)
(test ((lambda (arg) arg) "hi" 0) 'error)

(let ((L1 (list 1 2 3))
      (V1 (vector 1 2 3))
      (M1 #2d((1 2 3) (4 5 6) (7 8 9)))
      (S1 "123")
      (H1 (hash-table '(1 . 1) '(2 . 2) '(3 . 3))))
  (let ((L2 (list L1 V1 M1 S1 H1))
	(V2 (vector L1 V1 M1 S1 H1))
	(H2 (hash-table (cons 0 L1) (cons 1 V1) (cons 2 M1) (cons 3 S1) (cons 4 H1)))
	(M2 (let ((v (make-vector '(3 3))))
	      (set! (v 0 0) L1)
	      (set! (v 0 1) V1)
	      (set! (v 0 2) M1)
	      (set! (v 1 0) S1)
	      (set! (v 1 1) H1)
	      (set! (v 1 2) L1)
	      (set! (v 2 0) S1)
	      (set! (v 2 1) H1)
	      (set! (v 2 2) L1)
	      v)))
#|
    ;; this code generates the tests below
    (for-each
     (lambda (arg)
       (let* ((val (symbol->value arg))
	      (len (min 5 (length val))))
	 (do ((i 0 (+ i 1)))
	     ((= i len))
	   (format *stderr* "(test (~S ~S) ~S)~%" arg i
		   (catch #t (lambda () (val i)) (lambda args 'error)))
	   (let ((len1 (catch #t (lambda () (min 5 (length (val i)))) (lambda args 0))))
	     (if (> len1 0)
		 (do ((k 0 (+ k 1)))
		     ((= k len1))
		   (format *stderr* "(test (~S ~S ~S) ~S)~%" arg i k
			   (catch #t (lambda () (val i k)) (lambda args 'error)))
		   (let ((len2 (catch #t (lambda () (min 5 (length (val i k)))) (lambda args 0))))
		     (if (> len2 0)
			 (do ((m 0 (+ m 1)))
			     ((= m len2))
			   (format *stderr* "(test (~S ~S ~S ~S) ~S)~%" arg i k m
				   (catch #t (lambda () (val i k m)) (lambda args 'error)))
			   (let ((len3 (catch #t (lambda () (min 5 (length (val i k m)))) (lambda args 0))))
			     (if (> len3 0)
				 (do ((n 0 (+ n 1)))
				     ((= n len3))
				   (format *stderr* "(test (~S ~S ~S ~S ~S) ~S)~%" arg i k m n
					   (catch #t (lambda () (val i k m n)) (lambda args 'error)))))))))))))))
     (list 'L2 'V2 'M2 'H2))
|#

    (test (L2 0) '(1 2 3))
    (test (L2 0 0) 1)
    (test (L2 0 1) 2)
    (test (L2 0 2) 3)
    (test (L2 1) #(1 2 3))
    (test (L2 1 0) 1)
    (test (L2 1 1) 2)
    (test (L2 1 2) 3)
    (test (L2 2) #2D((1 2 3) (4 5 6) (7 8 9)))
    (test (L2 2 0) #(1 2 3))
    (test (L2 2 0 0) 1)
    (test (L2 2 0 1) 2)
    (test (L2 2 0 2) 3)
    (test (L2 2 1) #(4 5 6))
    (test (L2 2 1 0) 4)
    (test (L2 2 1 1) 5)
    (test (L2 2 1 2) 6)
    (test (L2 2 2) #(7 8 9))
    (test (L2 2 2 0) 7)
    (test (L2 2 2 1) 8)
    (test (L2 2 2 2) 9)
    (test (L2 2 3) 'error)
    (test (L2 2 4) 'error)
    (test (L2 3) "123")
    (test (L2 3 0) #\1)
    (test (L2 3 1) #\2)
    (test (L2 3 2) #\3)
    (test (L2 4) H1)
    (test (L2 4 0) #f)
    (test (L2 4 1) 1)
    (test (L2 4 2) 2)
    (test (L2 4 3) 3)
    (test (L2 4 4) #f)
    (test (V2 0) '(1 2 3))
    (test (V2 0 0) 1)
    (test (V2 0 1) 2)
    (test (V2 0 2) 3)
    (test (V2 1) #(1 2 3))
    (test (V2 1 0) 1)
    (test (V2 1 1) 2)
    (test (V2 1 2) 3)
    (test (V2 2) #2D((1 2 3) (4 5 6) (7 8 9)))
    (test (V2 2 0) #(1 2 3))
    (test (V2 2 0 0) 1)
    (test (V2 2 0 1) 2)
    (test (V2 2 0 2) 3)
    (test (V2 2 1) #(4 5 6))
    (test (V2 2 1 0) 4)
    (test (V2 2 1 1) 5)
    (test (V2 2 1 2) 6)
    (test (V2 2 2) #(7 8 9))
    (test (V2 2 2 0) 7)
    (test (V2 2 2 1) 8)
    (test (V2 2 2 2) 9)
    (test (V2 2 3) 'error)
    (test (V2 2 4) 'error)
    (test (V2 3) "123")
    (test (V2 3 0) #\1)
    (test (V2 3 1) #\2)
    (test (V2 3 2) #\3)
    (test (V2 4) H1)
    (test (V2 4 0) #f)
    (test (V2 4 1) 1)
    (test (V2 4 2) 2)
    (test (V2 4 3) 3)
    (test (V2 4 4) #f)
    (test (M2 0) #((1 2 3) #(1 2 3) #2D((1 2 3) (4 5 6) (7 8 9))))
    (test (M2 0 0) '(1 2 3))
    (test (M2 0 0 0) 1)
    (test (M2 0 0 1) 2)
    (test (M2 0 0 2) 3)
    (test (M2 0 1) #(1 2 3))
    (test (M2 0 1 0) 1)
    (test (M2 0 1 1) 2)
    (test (M2 0 1 2) 3)
    (test (M2 0 2) #2D((1 2 3) (4 5 6) (7 8 9)))
    (test (M2 0 2 0) #(1 2 3))
    (test (M2 0 2 0 0) 1)
    (test (M2 0 2 0 1) 2)
    (test (M2 0 2 0 2) 3)
    (test (M2 0 2 1) #(4 5 6))
    (test (M2 0 2 1 0) 4)
    (test (M2 0 2 1 1) 5)
    (test (M2 0 2 1 2) 6)
    (test (M2 0 2 2) #(7 8 9))
    (test (M2 0 2 2 0) 7)
    (test (M2 0 2 2 1) 8)
    (test (M2 0 2 2 2) 9)
    (test (M2 0 2 3) 'error)
    (test (M2 0 2 4) 'error)
    (test (M2 1) (vector "123" H1 '(1 2 3)))
    (test (M2 1 0) "123")
    (test (M2 1 0 0) #\1)
    (test (M2 1 0 1) #\2)
    (test (M2 1 0 2) #\3)
    (test (M2 1 1) H1)
    (test (M2 1 1 0) #f)
    (test (M2 1 1 1) 1)
    (test (M2 1 1 2) 2)
    (test (M2 1 1 3) 3)
    (test (M2 1 1 4) #f)
    (test (M2 1 2) '(1 2 3))
    (test (M2 1 2 0) 1)
    (test (M2 1 2 1) 2)
    (test (M2 1 2 2) 3)
    (test (M2 2) (vector "123" H1 '(1 2 3)))
    (test (M2 2 0) "123")
    (test (M2 2 0 0) #\1)
    (test (M2 2 0 1) #\2)
    (test (M2 2 0 2) #\3)
    (test (M2 2 1) H1)
    (test (M2 2 1 0) #f)
    (test (M2 2 1 1) 1)
    (test (M2 2 1 2) 2)
    (test (M2 2 1 3) 3)
    (test (M2 2 1 4) #f)
    (test (M2 2 2) '(1 2 3))
    (test (M2 2 2 0) 1)
    (test (M2 2 2 1) 2)
    (test (M2 2 2 2) 3)
    (test (M2 3) 'error)
    (test (M2 4) 'error)
    (test (H2 0) '(1 2 3))
    (test (H2 0 0) 1)
    (test (H2 0 1) 2)
    (test (H2 0 2) 3)
    (test (H2 1) #(1 2 3))
    (test (H2 1 0) 1)
    (test (H2 1 1) 2)
    (test (H2 1 2) 3)
    (test (H2 2) #2D((1 2 3) (4 5 6) (7 8 9)))
    (test (H2 2 0) #(1 2 3))
    (test (H2 2 0 0) 1)
    (test (H2 2 0 1) 2)
    (test (H2 2 0 2) 3)
    (test (H2 2 1) #(4 5 6))
    (test (H2 2 1 0) 4)
    (test (H2 2 1 1) 5)
    (test (H2 2 1 2) 6)
    (test (H2 2 2) #(7 8 9))
    (test (H2 2 2 0) 7)
    (test (H2 2 2 1) 8)
    (test (H2 2 2 2) 9)
    (test (H2 2 3) 'error)
    (test (H2 2 4) 'error)
    (test (H2 3) "123")
    (test (H2 3 0) #\1)
    (test (H2 3 1) #\2)
    (test (H2 3 2) #\3)
    (test (H2 4) H1)
    (test (H2 4 0) #f)
    (test (H2 4 1) 1)
    (test (H2 4 2) 2)
    (test (H2 4 3) 3)
    (test (H2 4 4) #f)
     ))

(let* ((L1 (cons 1 2))
       (L2 (list L1 3)))
  (test (L1 0) 1)
  (test (L1 1) 'error)
  (test (L1 2) 'error)
  (test (L2 0 0) 1)
  (test (L2 0 1) 'error)
  (test ((cons "123" 0) 0 1) #\2))

(let ((L1 (list "123" "456" "789")))
  (set-cdr! (cdr L1) L1)
  (test (L1 0 1) #\2)
  (test (L1 1 1) #\5)
  (test (L1 2 1) #\2)
  (test (L1 12 0) #\1))

(let ((L1 (list "123" "456" "789")))
  (set-car! (cdr L1) L1) 
  (test (L1 1 1 1 1 1 0 0) #\1))

(test ((list (list) "") 1 0) 'error)
(test ((list (list) "") 0 0) 'error)
(test (#(1 2) 0 0) 'error)
(test (#(1 #()) 1 0) 'error)

(test ('(((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((((12))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))))) 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0) 12)

;;; implicit index as expression (_A cases)

(let ((L1 (list 1 2 3))
      (V1 (vector 1 2 3))
      (S1 "123")
      (H1 (hash-table '(1 . 1) '(2 . 2) '(3 . 3)))
      (E1 (inlet :a 1 :b 2)))
  (define (f1 i s L V S H E)
    (vector (L (+ i 1)) (V (+ i 1)) (S (+ i 1)) (H (+ i 1)) (E (string->symbol s))))
  (test (f1 0 "a" L1 V1 S1 H1 E1) (vector 2 2 #\2 1 1))
  (test (f1 1 "b" L1 V1 S1 H1 E1) (vector 3 3 #\3 2 2))
  (define (f2 i s L V S H E)
    (vector (L (abs i)) (V (abs i)) (S (abs i)) (H (abs i)) (E (vector-ref s 0))))
  (test (f2 -2 #(b a) L1 V1 S1 H1 E1) (vector 3 3 #\3 2 2)))

(when with-block
  (define (f3 B i) (B (+ i 1)))
  (define (f4 B i) (B (abs i)))
  (let ((b (make-block 4))) 
    (set! (b 0) 1.0)
    (set! (b 1) 2.0)
    (test (f3 b -1) 1.0)
    (test (f4 b -1) 2.0)))

(let ((v1 #(0 1 2 3 4 5 6 7))
      (v2 #2D((0 1 2 3) (4 5 6 7)))
      (e1 (inlet :a 1))
      (p1 (list 0 1 2 3))
      (s1 "0123")
      )
  (define (call-1 func arg1 arg2) (func arg1 arg2)) 
  (define (call-2 func arg1 arg2) (func arg1 arg2)) 
  (define (call-3 func arg1 arg2) (func arg1 arg2)) 
  (define (call-4 func arg1 arg2) (func arg1 arg2)) 
  (define (call-5 func arg1 arg2) (func arg1 arg2))
  (define (call-6 func) (func 'a))
  (define (call-7 func) (func 'a))
  (define (call-8 func arg) (func (* 2 (+ arg 1))))
  (define (call-9 func arg) (func (* 2 (+ arg 1))))
  (define (call-10 func arg) (func arg))
  (define (call-11 func arg) (func arg))
  (define (call-12 func) (func 0))
  (define (call-13 func) (func 0))
  (define (call-14 func arg) (func arg 2))
  (define (call-15 func arg) (func 2 arg))

  (define (f+ x y) (+ x y))
  (define (f- x y) (- x y))
  (define* (f++ (x 0) y) (+ x y))
  (define* (f-- x (y 0)) (- x y))
  (define (fabs x) (abs x))
  (define-macro (m+ x y) `(+ ,x ,y))

  (test (call-1 + 5 2) 7)
  (test (call-1 f- 5 2) 3)
  (test (call-2 f+ 5 2) 7)
  (test (call-2 - 5 2) 3)
  (test (call-3 v2 0 3) 3)
  (test (call-3 list 0 3) (list 0 3))
  (test (call-4 f++ 5 2) 7)
  (test (call-4 f-- 5 2) 3)
  (test (call-5 m+ 5 2) 7)
  (test (call-5 - 5 2) 3)
  (test (call-6 e1) 1)
  (test (call-6 symbol?) #t)
  (test (call-7 symbol?) #t)
  (test (call-7 list) (list 'a))
  (test (call-8 abs -3) 4)
  (test (call-8 f-- 10) 22)
  (test (call-9 fabs -3) 4)
  (test (call-9 list -3) (list -4))
  (test (call-10 e1 'a) 1)
  (test (call-10 list 'a) (list 'a))
  (test (call-11 symbol? 'a) #t)
  (test (call-11 e1 'a) 1)
  (test (call-12 p1) 0)
  (test (call-12 s1) #\0)
  (test (call-13 v1) 0)
  (test (call-13 (lambda (x) (+ x 1))) 1)
  (test (call-14 * 3) 6)
  (test (call-14 (lambda (x y) (- x y)) 3) 1)
  (test (call-15 (lambda (x y) (- x y)) 3) -1)
  (test (call-15 - 3) -1)
  )
	    
;; multi-index get/set
(let ((v (vector (hash-table* 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (vector (inlet 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (vector (list 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (vector (string #\1 #\2)))) (test (v 0 1) #\2) (set! (v 0 1) #\5) (test (v 0 1) #\5))
(let ((v (vector (vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (vector (byte-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(when with-block (let ((v (vector (block 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0)))
(let ((v (vector (float-vector 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0))
(let ((v (vector (int-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (hash-table* 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (list (inlet 'a 1 'b 2)))) (test (v 0 'a) 1) (set! (v 0 'a) 5) (test (v 0 'a) 5))
(let ((v (list (list 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (string #\1 #\2)))) (test (v 0 1) #\2) (set! (v 0 1) #\5) (test (v 0 1) #\5))
(let ((v (list (vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (list (byte-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(when with-block (let ((v (list (block 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0)))
(let ((v (list (float-vector 1 2)))) (test (v 0 1) 2.0) (set! (v 0 1) 5) (test (v 0 1) 5.0))
(let ((v (list (int-vector 1 2)))) (test (v 0 1) 2) (set! (v 0 1) 5) (test (v 0 1) 5))
(let ((v (hash-table* 'a (hash-table* 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (hash-table* 'a (inlet 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (hash-table* 'a (list 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (hash-table* 'a (string #\1 #\2)))) (test (v 'a 1) #\2) (set! (v 'a 1) #\5) (test (v 'a 1) #\5))
(let ((v (hash-table* 'a (vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (hash-table* 'a (byte-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(when with-block (let ((v (hash-table* 'a (block 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0)))
(let ((v (hash-table* 'a (float-vector 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0))
(let ((v (hash-table* 'a (int-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (hash-table* 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (inlet 'a (inlet 'a 1 'b 2)))) (test (v 'a 'a) 1) (set! (v 'a 'a) 5) (test (v 'a 'a) 5))
(let ((v (inlet 'a (list 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (string #\1 #\2)))) (test (v 'a 1) #\2) (set! (v 'a 1) #\5) (test (v 'a 1) #\5))
(let ((v (inlet 'a (vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(let ((v (inlet 'a (byte-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))
(when with-block (let ((v (inlet 'a (block 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0)))
(let ((v (inlet 'a (float-vector 1 2)))) (test (v 'a 1) 2.0) (set! (v 'a 1) 5) (test (v 'a 1) 5.0))
(let ((v (inlet 'a (int-vector 1 2)))) (test (v 'a 1) 2) (set! (v 'a 1) 5) (test (v 'a 1) 5))

(let ((ind 0) (sym 'a) (v (vector (hash-table* 'a 1 'b 2)))) (test (v ind sym) 1) (set! (v (+ ind ind) sym) (+ ind 5)) (test (v 0 'a) 5))
(let ((v (vector (hash-table* 'a "123" 'b 2)))) (test (v 0 'a 1) #\2) (set! (v 0 'a 1) #\5) (test (v 0 'a) "153"))

(let ((iv (make-vector '(2 2))))
  (set! (iv 1 0) 2)
  (set! (iv 1 1) 4)
  (let ((v (vector iv))) 
    (test (v 0 1 0) 2) 
    (set! (v 0 1 0) 5) 
    (test (v 0 1) #(5 4))))

(let ((ov (make-vector '(2 2)))
      (iv (make-vector '(2 2))))
  (set! (ov 1 0) iv)
  (set! (iv 0 1) 3)
  (test (ov 1 0 0 1) 3)
  (set! (ov 1 0 0 1) 5)
  (test (ov 1 0 0 1) 5))



;;; --------------------------------------------------------------------------------
;;; PORTS
;;; --------------------------------------------------------------------------------

(define start-input-port (current-input-port))
(define start-output-port (current-output-port))

(test (input-port? (current-input-port)) #t)
(test (input-port? *stdin*) #t)
(test (input-port? (current-output-port)) #f)
(test (input-port? *stdout*) #f)
(test (input-port? (current-error-port)) #f)
(test (input-port? *stderr*) #f)

(for-each
 (lambda (arg)
   (if (input-port? arg)
       (format-logged #t ";(input-port? ~A) -> #t?~%" arg)))
 (list "hi" #f (integer->char 65) 1 (list 1 2) '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f :hi #<eof> #<undefined> #<unspecified>))

(test (call-with-input-file "s7test.scm" input-port?) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format-logged #t "call-with-input-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (let ((this-file (open-input-file "s7test.scm"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(if (not (eq? start-input-port (current-input-port)))
    (format-logged #t "open-input-file clobbered current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (call-with-input-string "(+ 1 2)" input-port?) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((res (input-port? this-file))) (close-input-port this-file) res)) #t)
(test (let ((this-file (open-input-string "(+ 1 2)"))) (let ((len (length this-file))) (close-input-port this-file) len)) 7)

;;; (test (let ((str "1234567890")) (let ((p (open-input-string str))) (string-set! str 0 #\a) (let ((c (read-char p))) (close-input-port p) c))) #\1)
;;; is that result demanded by the scheme spec? perhaps make str immutable if so?

;;; read
;;; write
(test (+ 100 (call-with-input-string "123" (lambda (p) (values (read p) 1)))) 224)


(test (call-with-input-string
       "1234567890"
       (lambda (p)
	 (call-with-input-string
	  "0987654321"
	  (lambda (q)
            (+ (read p) (read q))))))
      2222222211)

(test (call-with-input-string
       "12345 67890"
       (lambda (p)
	 (call-with-input-string
	  "09876 54321"
	  (lambda (q)
            (- (+ (read p) (read q)) (read p) (read q))))))
      -99990)

(call-with-output-file "empty-file" (lambda (p) #f))
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-char p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-byte p)))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (eof-object? (read-line p)))) #t)
(test (load "empty-file") #<unspecified>)
(test (call-with-input-file "empty-file" (lambda (p) (port-closed? p))) #f)
(test (eof-object? (call-with-input-string "" (lambda (p) (read p)))) #t)
(test (eof-object? #<eof>) #t)
(test (let () (define (hi a) (eof-object? a)) (hi #<eof>)) #t)

(let ()
  (define (io-func) (lambda (p) (eof-object? (read-line p))))
  (test (call-with-input-file (let () "empty-file") (io-func)) #t))

(let ((p1 #f))
  (call-with-output-file "empty-file" (lambda (p) (set! p1 p) (write-char #\a p)))
  (test (port-closed? p1) #t))
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (read-char p) #\a) (eof-object? (read-char p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (symbol->string (read p)) "a") (eof-object? (read p))))) #t) ; Guile also returns a symbol here
(test (call-with-input-file "empty-file" (lambda (p) (and (char=? (integer->char (read-byte p)) #\a) (eof-object? (read-byte p))))) #t)
(test (call-with-input-file "empty-file" (lambda (p) (and (string=? (read-line p) "a") (eof-object? (read-line p))))) #t)

(test (call-with-input-string "(lambda (a) (+ a 1))" (lambda (p) (let ((f (eval (read p)))) (f 123)))) 124)
(test (call-with-input-string "(let ((x 21)) (+ x 1))" (lambda (p) (eval (read p)))) 22)
(test (call-with-input-string "(1 2 3) (4 5 6)" (lambda (p) (list (read p) (read p)))) '((1 2 3) (4 5 6)))

(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(lambda (a) (+ a 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (let ((f (eval (read p)))) (f 123)))))
      124)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(let ((x 21)) (+ x 1)) p)))
	(call-with-input-file "empty-file" (lambda (p) (eval (read p)))))
      22)
(test (let ()
	(call-with-output-file "empty-file" (lambda (p) (write '(1 2 3) p) (write '(4 5 6) p)))
	(call-with-input-file "empty-file" (lambda (p) (list (read p) (read p)))))
      '((1 2 3) (4 5 6)))

(call-with-output-file "empty-file" (lambda (p) (for-each (lambda (c) (write-char c p)) "#b11")))
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (char=? (read-char p) #\#) 
						(char=? (read-char p) #\b) 
						(char=? (read-char p) #\1) 
						(char=? (read-char p) #\1) 
						(eof-object? (read-char p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read p) 3) 
						(eof-object? (read p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (= (read-byte p) (char->integer #\#))
						(= (read-byte p) (char->integer #\b))
						(= (read-byte p) (char->integer #\1))
						(= (read-byte p) (char->integer #\1))
						(eof-object? (read-byte p))))) 
      #t)
(test (call-with-input-file "empty-file" (lambda (p) 
					   (and (string=? (read-line p) "#b11") 
						(eof-object? (read-line p))))) 
      #t)
(test (load "empty-file") 3)
(let ((p1 (dilambda (lambda (p) (and (= (read p) 3) (eof-object? (read p)))) (lambda (p) #f))))
  (test (call-with-input-file "empty-file" p1) #t))


;;; load
(for-each
 (lambda (arg)
   (test (load arg) 'error)
   (test (load "empty-file" arg) 'error))
 (list () (list 1) '(1 . 2) #f #\a 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
(test (load) 'error)
(test (load "empty-file" (curlet) 1) 'error)
(test (load "not a file") 'error)
(test (load "") 'error)
(test (load "/home/bil/cl") 'error)
(test (call-with-input-string "(display (+ 1 2))" load) 'error)

(call-with-output-file "empty-file" (lambda (p) (write '(+ 1 2 3) p)))
(let ((x 4))
  (test (+ x (load "empty-file")) 10))

(call-with-output-file "empty-file" (lambda (p) (write '(list 1 2 3) p)))
(let ((x 4))
  (test (cons x (load "empty-file")) '(4 1 2 3)))

(call-with-output-file "empty-file" (lambda (p) (write '(values 1 2 3) p)))
(let ((x 4))
  (test (+ x (load "empty-file")) 10))
(test (+ 4 (eval (call-with-input-file "empty-file" (lambda (p) (read p))))) 10)

(call-with-output-file "empty-file" (lambda (p) (write '(+ x 1) p)))
(let ((x 2))
  (test (load "empty-file" (curlet)) 3))

(call-with-output-file "empty-file" (lambda (p) (write '(set! x 1) p)))
(let ((x 2))
  (load "empty-file" (curlet))
  (test x 1))

(call-with-output-file "empty-file" (lambda (p) (write '(define (hi a) (values a 2)) p) (write '(hi x) p)))
(let ((x 4))
  (test (+ x (load "empty-file" (curlet))) 10))

(let ((x 1)
      (e #f))
  (set! e (curlet))
  (let ((x 4))
    (test (+ x (load "empty-file" e)) 7)))

(let ()
  (let ()
    (call-with-output-file "empty-file" (lambda (p) (write '(define (load_hi a) (+ a 1)) p)))
    (load "empty-file" (curlet))
    (test (load_hi 2) 3))
  (test (defined? 'load_hi) #f))

(let ()
  (apply load '("empty-file"))
  (test (load_hi 2) 3))

(call-with-output-file "empty-file" (lambda (p) (display "\"empty-file\"" p)))
(test (load (load "empty-file")) "empty-file")

;;; autoload
(test (autoload) 'error)
(test (autoload 'abs) 'error)
(test (autoload :abs "dsp.scm") 'error)
(for-each
 (lambda (arg)
   (test (autoload arg "dsp.scm") 'error)
   (test (autoload 'hi arg) 'error))
 (list #f () (integer->char 65) 1 (list 1 2) _ht_ _null_ _c_obj_ '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))
(test (autoload 'abs "dsp.scm" 123) 'error)
(test (autoload "" "dsp.scm") 'error)

(autoload 'auto_test_var "empty-file")
(test (defined? 'auto_test_var) #f)
(call-with-output-file "empty-file" (lambda (p) (format p "(define auto_test_var 123)~%")))
(load "empty-file")
(test (+ 1 auto_test_var) 124)

(autoload 'auto_test_var_2 (lambda (e) (varlet e (cons 'auto_test_var_2 1))))
(test (let () (+ 1 auto_test_var_2)) 2)

(autoload 'auto_test_var_3 (lambda (e) (varlet e (cons 'auto_test_var_3 1))))
(autoload 'auto_test_var_4 (lambda (e) (varlet e (cons 'auto_test_var_4 (+ auto_test_var_3 1)))))
(test (let () (+ auto_test_var_4 1)) 3)
(test (autoload 'auto_test_var_1 (lambda () #f)) 'error)
(test (autoload 'auto_test_var_1 (lambda (a b) #f)) 'error)

(let ((str3 #f))
  ;; IO tests mainly

  (set! str3 "0123456789")
  (set! str3 (string-append str3 str3 str3 str3 str3 str3 str3 str3 str3 str3))
  (set! str3 (string-append str3 str3 str3 str3 str3 str3 str3 str3 str3 str3))
  (set! str3 (string-append str3 str3 str3))

  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-string)~%")
      (format p "  \"")
      (display str3 p)
      (format p "\\\n") ; this becomes \<newline> in the midst of a string which we ignore
      (display str3 p)
      (format p "\"")
      (format p ")~%")))
  
  (load "test.scm")
  (let ((str (big-string)))
    (test (length str) 6000))
  
  
  (let ((big-string (eval (call-with-input-string
			   (call-with-output-string
			    (lambda (p)
			      (format p "(lambda ()~%")
			      (format p "  \"")
			      (display str3 p)
			      (format p "\\\n") ; this becomes \<newline> in the midst of a string which we ignore
			      (display str3 p)
			      (format p "\"")
			      (format p ")~%")))
			   read))))
    (let ((str (big-string)))
      (test (length str) 6000)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-string)~%")
      (format p "  \"")
      (display str3 p)
      (format p "\\\"") 
      (display str3 p)
      (format p "\"")
      (format p ")~%")))
  
  (load "test.scm")
  (let ((str (big-string)))
    (test (length str) 6001))
  
  
  (let ((big-string (eval (call-with-input-string
			   (call-with-output-string
			    (lambda (p)
			      (format p "(lambda ()~%")
			      (format p "  \"")
			      (display str3 p)
			      (format p "\\\"") 
			      (display str3 p)
			      (format p "\"")
			      (format p ")~%")))
			   read))))
    (let ((str (big-string)))
      (test (length str) 6001)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "")))
  (load "test.scm") ; #<unspecified>
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p ";")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(let ((c (integer->char (random 128))))
	  (if (char<? c #\space)
	      (display #\space p)
	      (display c p))))
      (format p "~%32~%")))
  (test (load "test.scm") 32)
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-list)~%  (list ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "~D " i))
      (format p "))~%")))
  
  (load "test.scm")
  (let ((lst (big-list)))
    (test (length lst) 2000))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-list)~%  ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "(cons ~D " i))
      (format p "()")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p ")"))
      (format p ")~%")))
  
  (load "test.scm")
  (let ((lst (big-list)))
    (test (length lst) 2000))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (a-char)~%  #\\a)~%")))
  
  (load "test.scm")
  (test (a-char) #\a)
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-char)~%  (string ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "#\\~C " (integer->char (+ a (modulo i 26)))))
	(format p "))~%"))))
  
  (load "test.scm")
  (let ((chars (big-char)))
    (test (length chars) 2000))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-xchar)~%  (string ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "#\\x~X " (+ a (modulo i 26))))
	(format p "))~%"))))
  
  (load "test.scm")
  (let ((chars (big-xchar)))
    (test (length chars) 2000))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (ychar) #\\~C)" (integer->char 255))))
  (load "test.scm") 
  (test (ychar) (integer->char 255))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "~D" i))
      (format p "~%")
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "~D" i))))
  
  (call-with-input-file "test.scm"
    (lambda (p)
      (let ((s1 (read-line p))
	    (s2 (read-line p)))
	(test (and (string=? s1 s2)
		   (= (length s1) 2890))
	      #t))))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-int)~%")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "123)~%")))
  
  (load "test.scm")
  (test (big-int) 123)
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-rat)~%")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "123/")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(format p "0"))
      (format p "2)~%")))
  
  (load "test.scm")
  (test (big-rat) 123/2)
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-hash)~%  (hash-table ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "'(~D . ~D) " i (+ i 1)))
      (format p "))~%")))
  
  (load "test.scm")
  (let ((ht (big-hash)))
    (let ((entries 0))
      (for-each
       (lambda (htv)
	 (set! entries (+ entries 1))
	 (if (not (= (+ (car htv) 1) (cdr htv)))
	     (format *stderr* ";hashed: ~A~%" htv)))
       ht)
      (test entries 2000)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-hash)~%  (apply hash-table (list ")
      (do ((i 0 (+ i 1)))
	  ((= i 2000))
	(format p "(cons ~D ~D) " i (+ i 1)))
      (format p ")))~%")))
  
  (load "test.scm")
  (let ((ht (big-hash)))
    (let ((entries 0))
      (for-each
       (lambda (htv)
	 (set! entries (+ entries 1))
	 (if (not (= (+ (car htv) 1) (cdr htv)))
	     (format *stderr* ";hashed: ~A~%" htv)))
       ht)
      (test entries 2000)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (let ((a (char->integer #\a)))
	(format p "(define (big-env)~%  (inlet ")
	(do ((i 0 (+ i 1)))
	    ((= i 2000))
	  (format p "'(~A . ~D) " 
		  (string (integer->char (+ a (modulo i 26)))
			  (integer->char (+ a (modulo (floor (/ i 26)) 26)))
			  (integer->char (+ a (modulo (floor (/ i (* 26 26))) 26))))
		  i))
	(format p "))~%"))))
  
  (load "test.scm")
  (let ((E (big-env))
	(a (char->integer #\a)))
    (do ((i 0 (+ i 1)))
	((= i 2000))
      (let ((sym (string->symbol
		  (string (integer->char (+ a (modulo i 26)))
			  (integer->char (+ a (modulo (floor (/ i 26)) 26)))
			  (integer->char (+ a (modulo (floor (/ i (* 26 26))) 26)))))))
	(let ((val (E sym)))
	  (if (not (equal? val i))
	      (format *stderr* ";env: ~A -> ~A, not ~D~%" sym val i))))))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "")))
  
  (let ((val (call-with-input-file "test.scm"
	       (lambda (p)
		 (read p)))))
    (if (not (eof-object? val))
	(format *stderr* ";read empty file: ~A~%" val)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p " ;")
      (do ((i 0 (+ i 1)))
	  ((= i 3000))
	(let ((c (integer->char (random 128))))
	  (if (char<? c #\space)
	      (display #\space p)
	      (display c p))))
      (format p "~%")))
  
  (let ((val (call-with-input-file "test.scm"
	       (lambda (p)
		 (read p)))))
    (if (not (eof-object? val))
	(format *stderr* ";read comment file: ~A~%" val)))
  
  
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "\"~3001TT\"~%")))
  
  (let ((str (call-with-input-file "test.scm"
	       (lambda (p)
		 (read p)))))
    (test (length str) 3000)))

#|
(let ((c #f)
      (i 0)
      (e #f))
  (set! e (curlet))
  (call-with-output-file "empty-file" (lambda (p) (write '(call/cc (lambda (c1) (set! c c1) (set! i (+ i 1)))) p)))
  (load "empty-file" e)
  (test (c) 'error)) ; ;read-error ("our input port got clobbered!")
|#


(test (reverse *stdin*) 'error)
(test (fill! (current-output-port)) 'error)
(test (length *stderr*) #f)

(test (output-port? (current-input-port)) #f)
(test (output-port? *stdin*) #f)
(test (output-port? (current-output-port)) #t)
(test (output-port? *stdout*) #t)
(test (output-port? (current-error-port)) #t)
(test (output-port? *stderr*) #t)

;(write-char #\space (current-output-port))
;(write " " (current-output-port))
(newline (current-output-port))


(for-each
 (lambda (arg)
   (if (output-port? arg)
       (format-logged #t ";(output-port? ~A) -> #t?~%" arg)))
 (list "hi" #f () 'hi (integer->char 65) 1 (list 1 2) _ht_ _null_ _c_obj_ '#t '3 (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(for-each
 (lambda (arg)
   (test (read-line () arg) 'error)
   (test (read-line arg) 'error))
 (list "hi" (integer->char 65) 1 #f _ht_ _null_ _c_obj_ (list) (cons 1 2) (list 1 2) (make-vector 3) 3.14 3/4 1.0+1.0i #\f))

(test (call-with-output-file tmp-output-file output-port?) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format-logged #t "call-with-output-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((this-file (open-output-file tmp-output-file))) (let ((res (output-port? this-file))) (close-output-port this-file) res)) #t)
(if (not (eq? start-output-port (current-output-port)))
    (format-logged #t "open-output-file clobbered current-output-port? ~A from ~A~%" start-output-port (current-output-port)))

(test (let ((val #f)) (call-with-output-string (lambda (p) (set! val (output-port? p)))) val) #t)
(test (let ((res #f)) (let ((this-file (open-output-string))) (set! res (output-port? this-file)) (close-output-port this-file) res)) #t)

(for-each
 (lambda (arg)
   (if (eof-object? arg)
       (format-logged #t ";(eof-object? ~A) -> #t?~%" arg)))
 (list "hi" () '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (let ((val (catch #t
		     (lambda () (port-closed? arg))
		     (lambda args 'error))))
     (if (not (eq? val 'error))
	 (format-logged #t ";(port-closed? ~A) -> ~S?~%" arg val))))
 (list "hi" '(1 2) -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #f #t (if #f #f) #<undefined> #<eof> (lambda (a) (+ a 1))))

(test (port-closed?) 'error)
(test (port-closed? (current-input-port) (current-output-port)) 'error)

(call-with-output-file tmp-output-file (lambda (p) (display "3.14" p)))
(test (call-with-input-file tmp-output-file (lambda (p) (read p) (let ((val (read p))) (eof-object? val)))) #t)

(test (call-with-input-file tmp-output-file (lambda (p) (read-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (peek-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (peek-char p) (read-char p))) #\3)
(test (call-with-input-file tmp-output-file (lambda (p) (list->string (list (read-char p) (read-char p) (read-char p) (read-char p))))) "3.14")
(test (call-with-input-file tmp-output-file (lambda (p) (list->string (list (read-char p) (peek-char p) (read-char p) (read-char p) (peek-char p) (read-char p))))) "3..144")

(for-each
 (lambda (arg)
   (call-with-output-file tmp-output-file (lambda (p) (write arg p)))
   (if (not (morally-equal? (call-with-input-file tmp-output-file (lambda (p) (read p))) arg))
       (format *stderr* "~A different after write~%" arg)))
 (list "hi" -1 #\a 1 'a-symbol (make-vector 3 0) 3.14 3/4 .6 1.0+1.0i #f #t (list 1 2 3) (cons 1 2)
       '(1 2 . 3) () '((1 2) (3 . 4)) '(()) (list (list 'a "hi") #\b 3/4) ''a
       (string #\a #\null #\b) "" "\"hi\""
       (integer->char 128) (integer->char 127) (integer->char 255) #\space #\null #\newline #\tab
       #() #2d((1 2) (3 4)) #3d()
       :hi #<eof> #<undefined> #<unspecified>
       most-negative-fixnum
       (if with-bignums 1239223372036854775808 123)
       (if with-bignums 144580536300674537151081081515762353325831/229154728370723013560448485454219755525522 11/10)
       (if with-bignums 221529797579218180403518826416685087012.0 1000.1)
       (if with-bignums 1239223372036854775808+1239223372036854775808i 1000.1-1234i)

       ))

(for-each
 (lambda (arg)
   (call-with-output-file tmp-output-file (lambda (p) (write arg p)))
   (test (call-with-input-file tmp-output-file (lambda (p) (eval (read p)))) arg)) ; so read -> symbol?
 (list *stdout* *stdin* *stderr*
       abs + quasiquote
  
;       (hash-table '(a . 1) '(b . 2)) (hash-table)
;       0/0 (real-part (log 0))
;;; for these we need nan? and infinite? since equal? might be #f
;       (lambda (a) (+ a 1))
; pws?
;       (current-output-port)
;       (random-state 1234)
;       (symbol ":\"")
; (let () (define-macro (hi1 a) `(+ ,a 1)) hi1)
;;; and how could a continuation work in general?        
       ))

;;; (call-with-input-file tmp-output-file (lambda (p) (read p))) got (symbol ":\"") but expected (symbol ":\"")


;;; r4rstest
(let* ((write-test-obj '(#t #f a () 9739 -3 . #((test) "te \" \" st" "" test #() b c)))
       (load-test-obj (list 'define 'foo (list 'quote write-test-obj))))
  
  (define (check-test-file name)
    (let ((val (call-with-input-file
		   name
		 (lambda (test-file)
		   (test (read test-file) load-test-obj)
		   (test (eof-object? (peek-char test-file)) #t)
		   (test (eof-object? (read-char test-file)) #t)
		   (input-port? test-file)))))
      (if (not (eq? val #t))
	  (format-logged #t "input-port? in call-with-input-file? returned ~A from ~A~%" val name))))
  
  (test (call-with-output-file
	    tmp-output-file
	  (lambda (test-file)
	    (write-char #\; test-file)
	    (display #\; test-file)
	    (display ";" test-file)
	    (write write-test-obj test-file)
	    (newline test-file)
	    (write load-test-obj test-file)
	    (output-port? test-file))) #t)
  (check-test-file tmp-output-file)
  
  (let ((test-file (open-output-file "tmp2.r5rs")))
    (test (port-closed? test-file) #f)
    (write-char #\; test-file)
    (display #\; test-file)
    (display ";" test-file)
    (write write-test-obj test-file)
    (newline test-file)
    (write load-test-obj test-file)
    (test (output-port? test-file) #t)
    (close-output-port test-file)
    (check-test-file "tmp2.r5rs")))


(call-with-output-file tmp-output-file (lambda (p) (display "3.14" p)))
(test (with-input-from-file tmp-output-file read) 3.14)
(if (not (eq? start-input-port (current-input-port)))
    (format-logged #t "with-input-from-file did not restore current-input-port? ~A from ~A~%" start-input-port (current-input-port)))

(test (with-input-from-file tmp-output-file (lambda () (eq? (current-input-port) start-input-port))) #f)
(test (char->integer ((with-input-from-string (string (integer->char 255))(lambda () (read-string 1))) 0)) 255)

(test (with-output-to-file tmp-output-file (lambda () (eq? (current-output-port) start-output-port))) #f)
(if (not (eq? start-output-port (current-output-port)))
    (format-logged #t "with-output-to-file did not restore current-output-port? ~A from ~A~%" start-output-port (current-output-port)))


(let ((newly-found-sonnet-probably-by-shakespeare 
       "This is the story, a sad tale but true \
        Of a programmer who had far too little to do.\
        One day as he sat in his hut swilling stew, \
        He cried \"CLM takes forever, it's stuck in a slough!,\
        Its C code is slow, too slow by a few.\
        Why, with just a small effort, say one line or two,\
        It could outpace a no-op, you could scarcely say 'boo'\"!\
        So he sat in his kitchen and worked like a dog.\
        He typed and he typed 'til his mind was a fog. \
        Now 6000 lines later, what wonders we see!  \
        CLM is much faster, and faster still it will be!\
        In fact, for most cases, C beats the DSP!  \
        But bummed is our coder; he grumbles at night.  \
        That DSP code took him a year to write.  \
        He was paid many dollars, and spent them with glee,\
        But his employer might mutter, this result were he to see."))
  
  (call-with-output-file tmp-output-file
    (lambda (p)
      (write newly-found-sonnet-probably-by-shakespeare p)))
  
  (let ((sonnet (with-input-from-file tmp-output-file
		  (lambda ()
		    (read)))))
    (if (or (not (string? sonnet))
	    (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	(format-logged #t "write/read long string returned: ~A~%" sonnet)))
  
  (let ((file (open-output-file tmp-output-file)))
    (let ((len (string-length newly-found-sonnet-probably-by-shakespeare)))
      (write-char #\" file)
      (do ((i 0 (+ i 1)))
	  ((= i len))
	(let ((chr (string-ref newly-found-sonnet-probably-by-shakespeare i)))
	  (if (char=? chr #\")
	      (write-char #\\ file))
	  (write-char chr file)))
      (write-char #\" file)
      (close-output-port file)))
  
  (let ((file (open-input-file tmp-output-file)))
    (let ((sonnet (read file)))
      (close-input-port file)
      (if (or (not (string? sonnet))
	      (not (string=? sonnet newly-found-sonnet-probably-by-shakespeare)))
	  (format-logged #t "write-char/read long string returned: ~A~%" sonnet)))))

(let ((file (open-output-file tmp-output-file)))
  (for-each
   (lambda (arg)
     (write arg file)
     (write-char #\space file))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-output-port file))

(let ((file (open-input-file tmp-output-file)))
  (for-each
   (lambda (arg)
     (let ((val (read file)))
       (if (not (equal? val arg))
	   (format-logged #t "read/write ~A returned ~A~%" arg val))))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))
  (close-input-port file))

(with-output-to-file tmp-output-file
  (lambda ()
    (write lists)))

(let ((val (with-input-from-file tmp-output-file
	     (lambda ()
	       (read)))))
  (if (not (equal? val lists))
      (format-logged #t "read/write lists returned ~A~%" val)))

(if (not (string=? "" (with-output-to-string (lambda () (display "")))))
    (format-logged #t "with-output-to-string null string?"))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string "hiho123"
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str "hiho123"))
      (format-logged #t "with string ports 0: ~S?~%" str)))

(let ((p1 (open-input-string "123"))
      (p2 (open-input-string "123")))
  (test (morally-equal? p1 p2) #t)
  (read-char p1)
  (test (morally-equal? p1 p2) #f)
  (read-char p2)
  (test (morally-equal? p1 p2) #t)
  (close-input-port p1)
  (close-input-port p2))

(let ((p1 (open-input-string "1234"))
      (p2 (open-input-string "123")))
  (test (morally-equal? p1 p2) #f)
  (read-char p1)
  (test (morally-equal? p1 p2) #f)
  (close-input-port p1)
  (close-input-port p2))

(let ((p1 (open-output-string))
      (p2 (open-output-string)))
  (test (morally-equal? p1 p2) #t)
  (write-char #\a p1)
  (test (morally-equal? p1 p2) #f)
  (close-output-port p1)
  (close-output-port p2))

(let ()
  (define* (f1 (b 123)) (display b))
  (test (with-output-to-string f1) "123")
  (define (f2) (display "123"))
  (test (with-output-to-string f2) "123")
  (define (f3 . args) (display 123))
  (test (with-output-to-string f3) "123")
  (define-macro (m1) `(write 123))
  (test (with-output-to-string m1) "123")
  (define-macro (m2) (write 123))
  (test (with-output-to-string m2) "123")
  (define (f4 a b) (display 123))
  (test (with-output-to-string f4) 'error)
  (test (with-output-to-string s7-version) "")) ; the output is a string -- not written to stdout or whatever

(let ()
  (define* (f1 a (b 123)) (display b a))
  (test (call-with-output-string f1) "123")
  (define (f2 a) (display "123" a))
  (test (call-with-output-string f2) "123")
  (define (f3 . args) (display 123 (car args)))
  (test (call-with-output-string f3) "123")
  (define-macro (m1 p) `(write 123 ,p))
  (test (call-with-output-string m1) "123")
  (define-macro* (m2 (p #f)) (write 123 p))
  (test (call-with-output-string m2) "123")
  (define (f4 a b) (display 123 a))
  (test (call-with-output-string f4) 'error)
  (test (call-with-output-string s7-version) 'error))

(let ()
  (define* (f1 (a #f)) (read))
  (test (with-input-from-string "(+ 1 2 3)" f1) '(+ 1 2 3))
  (define* (f2 . args) (read))
  (test (with-input-from-string "(+ 1 2 3)" f2) '(+ 1 2 3))
  (define f3 read)
  (test (with-input-from-string "(+ 1 2 3)" f3) '(+ 1 2 3))
  (define (f4) (read))
  (test (with-input-from-string "(+ 1 2 3)" f4) '(+ 1 2 3))
  (define-macro (m1) `(read))
  (test (with-input-from-string "(+ 1 2 3)" m1) '(+ 1 2 3))
  (define-macro (m2) (read))
  (test (with-input-from-string "(+ 1 2 3)" m2) 6)
  (define (f5 a) (read a))
  (test (with-input-from-string "(+ 1 2 3)" f5) 'error)
  (test (with-input-from-string "(+ 1 2 3)" s7-version) (s7-version)))

(let ()
  (define* (f1 (a #f)) (read a))
  (test (call-with-input-string "(+ 1 2 3)" f1) '(+ 1 2 3))
  (define* (f2 . args) (read (car args)))
  (test (call-with-input-string "(+ 1 2 3)" f2) '(+ 1 2 3))
  (define f3 read)
  (test (call-with-input-string "(+ 1 2 3)" f3) '(+ 1 2 3))
  (define-macro (m1 p) `(read ,p))
  (test (call-with-input-string "(+ 1 2 3)" m1) '(+ 1 2 3))
  (define-macro* (m2 (p #f)) (read p))
  (test (call-with-input-string "(+ 1 2 3)" m2) 6)
  (define (f4) (read))
  (test (call-with-input-string "(+ 1 2 3)" f4) 'error)
  (test (call-with-input-string "(+ 1 2 3)" s7-version) 'error))


(let ()
  (with-output-to-file tmp-output-file
    (lambda ()
      (display "(+ 1 2 3)")))
  (define* (f1 (a #f)) (read))
  (test (with-input-from-file tmp-output-file f1) '(+ 1 2 3))
  (define* (f2 . args) (read))
  (test (with-input-from-file tmp-output-file f2) '(+ 1 2 3))
  (define f3 read)
  (test (with-input-from-file tmp-output-file f3) '(+ 1 2 3))
  (define (f4) (read))
  (test (with-input-from-file tmp-output-file f4) '(+ 1 2 3))
  (define-macro (m1) `(read))
  (test (with-input-from-file tmp-output-file m1) '(+ 1 2 3))
  (define-macro (m2) (read))
  (test (with-input-from-file tmp-output-file m2) 6)
  (define (f5 a) (read a))
  (test (with-input-from-file tmp-output-file f5) 'error)
  (test (with-input-from-file tmp-output-file s7-version) (s7-version)))

(let ()
  (define (eval-from-string-1 str)
    (define-macro (m) (read))
    (with-input-from-string str m))
  (test (eval-from-string-1 "(+ 1 2 3)") 6)
  (define (eval-from-string str)
    (with-input-from-string str (define-macro (m) (read))))
  (test (eval-from-string "(+ 1 2 3)") 6))

(let ()
  (define* (f1 (a #f)) (read a))
  (test (call-with-input-file tmp-output-file f1) '(+ 1 2 3))
  (define* (f2 . args) (read (car args)))
  (test (call-with-input-file tmp-output-file f2) '(+ 1 2 3))
  (define f3 read)
  (test (call-with-input-file tmp-output-file f3) '(+ 1 2 3))
  (define-macro (m1 p) `(read ,p))
  (test (call-with-input-file tmp-output-file m1) '(+ 1 2 3))
  (define-macro* (m2 (p #f)) (read p))
  (test (call-with-input-file tmp-output-file m2) 6)
  (define (f4) (read))
  (test (call-with-input-file tmp-output-file f4) 'error)
  (test (call-with-input-file tmp-output-file s7-version) 'error))

(let ((ofile tmp-output-file))
  (define (get-file-contents)
    (with-input-from-file ofile read-line))

  (define* (f1 (b 123)) (display b))
  (test (let () (with-output-to-file ofile f1) (get-file-contents)) "123")
  (define (f2) (display "123"))
  (test (let () (with-output-to-file ofile f2) (get-file-contents)) "123")
  (define (f3 . args) (display 123))
  (test (let () (with-output-to-file ofile f3) (get-file-contents)) "123")
  (define-macro (m1) `(write 123))
  (test (let () (with-output-to-file ofile m1) (get-file-contents)) "123")
  (define-macro (m2) (write 123))
  (test (let () (with-output-to-file ofile m2) (get-file-contents)) "123")
  (define (f4 a b) (display 123))
  (test (let () (with-output-to-file ofile f4) (get-file-contents)) 'error)
  (test (let () (with-output-to-file ofile s7-version) (get-file-contents)) #<eof>)

  (define* (f11 a (b 123)) (display b a))
  (test (let () (call-with-output-file ofile f11) (get-file-contents)) "123")
  (define (f21 a) (display "123" a))
  (test (let () (call-with-output-file ofile f21) (get-file-contents)) "123")
  (define (f31 . args) (display 123 (car args)))
  (test (let () (call-with-output-file ofile f31) (get-file-contents)) "123")
  (define-macro (m3 p) `(write 123 ,p))
  (test (let () (call-with-output-file ofile m3) (get-file-contents)) "123")
  (define-bacro* (m2 (p 123)) `(write 123 ,p))
  (test (let () (call-with-output-file ofile m2) (get-file-contents)) "123")
  (define (f41 a b) (display 123 a))
  (test (let () (call-with-output-file ofile f41) (get-file-contents)) 'error)
  (test (let () (call-with-output-file ofile s7-version) (get-file-contents)) 'error))

(if (not (eof-object? (with-input-from-string "" (lambda () (read-char)))))
    (format-logged #t ";input from null string not #<eof>?~%")
    (let ((EOF (with-input-from-string "" (lambda () (read-char)))))
      (if (not (eq? (with-input-from-string "" (lambda () (read-char)))
		    (with-input-from-string "" (lambda () (read-char)))))
	  (format-logged #t "#<eof> is not eq? to itself?~%"))
      (if (char? EOF)
	  (do ((c 0 (+ c 1)))
	      ((= c 256))
	    (if (char=? EOF (integer->char c))
		(format-logged #t "#<eof> is char=? to ~C~%" (integer->char c)))))))

(test (+ 100 (call-with-output-file "tmp.r5rs" (lambda (p) (write "1" p) (values 1 2)))) 103)
(test (+ 100 (with-output-to-file "tmp.r5rs" (lambda () (write "2") (values 1 2)))) 103)

(if (not pure-s7)
    (let ((str (with-output-to-string
		 (lambda ()
		   (with-input-from-string "hiho123"
		     (lambda ()
		       (do ((c (read-char) (read-char)))
			   ((or (not (char-ready?))
				(eof-object? c)))
			 (display c))))))))
      (if (not (string=? str "hiho123"))
	  (format-logged #t "with string ports 1: ~S?~%" str))))

(let ((str (with-output-to-string
	     (lambda ()
	       (with-input-from-string ""
		 (lambda ()
		   (do ((c (read-char) (read-char)))
		       ((eof-object? c))
		     (display c))))))))
  (if (not (string=? str ""))
      (format-logged #t "with string ports and null string: ~S?~%" str)))

(let ((str (with-output-to-string ; this is from the guile-user mailing list, I think -- don't know who wrote it
	     (lambda ()
	       (with-input-from-string "A2B5E3426FG0ZYW3210PQ89R."
		 (lambda ()
		   (call/cc
		    (lambda (hlt)
		      (define (nextchar)
			(let ((c (read-char)))
			  (if (and (char? c) 
				   (char=? c #\space))
			      (nextchar) 
			      c)))
		      
		      (define inx
			(lambda()
			  (let in1 ()
			    (let ((c (nextchar)))
			      (if (char-numeric? c)
				  (let ((r (nextchar)))
				    (let out*n ((n (- (char->integer c) (char->integer #\0))))
				      (out r)
				      (if (not (zero? n))
					  (out*n (- n 1)))))
				  (out c))
			      (in1)))))
		      
		      (define (move-char c)
			(write-char c)
			(if (char=? c #\.)
			    (begin (hlt))))
		      
		      (define outx
			(lambda()
			  (let out1 ()
			    (let h1 ((n 16))
			      (move-char (in))
			      (move-char (in))
			      (move-char (in))
			      (if (= n 1)
				  (begin (out1))
				  (begin (write-char #\space) (h1 (- n 1))) )))))
		      
		      (define (in)
			(call/cc (lambda(return)
				   (set! outx return)
				   (inx))))
		      
		      (define (out c)
			(call/cc (lambda(return) 
				   (set! inx return)
				   (outx c))))
		      (outx)))))))))
  (if (not (string=? str "ABB BEE EEE E44 446 66F GZY W22 220 0PQ 999 999 999 R."))
      (format-logged #t "call/cc with-input-from-string str: ~A~%" str)))

(let ((badfile tmp-output-file))
  (let ((p (open-output-file badfile)))
    (close-output-port p))
  (load badfile))

(for-each
 (lambda (str)
   ;;(test (eval-string str) 'error)
   ;; eval-string is confused somehow
   (test (with-input-from-string str read) 'error))
 (list "\"\\x" "\"\\x0" "`(+ ," "`(+ ,@" "#2d(" "#\\"))

(let ((loadit tmp-output-file))
  (let ((p1 (open-output-file loadit)))
    (display "(define s7test-var 314) (define (s7test-func) 314) (define-macro (s7test-mac a) `(+ ,a 2))" p1)
    (newline p1)
    (close-output-port p1)
    (load loadit)
    (test (= s7test-var 314) #t)
    (test (s7test-func) 314)
    (test (s7test-mac 1) 3)
    (let ((p2 (open-output-file loadit))) ; hopefully this starts a new file
      (display "(define s7test-var 3) (define (s7test-func) 3) (define-macro (s7test-mac a) `(+ ,a 1))" p2)
      (newline p2)
      (close-output-port p2)
      (load loadit)
      (test (= s7test-var 3) #t)
      (test (s7test-func) 3)
      (test (s7test-mac 1) 2)
      (test (morally-equal? p1 p2) #t))))

(test (+ 100 (with-input-from-string "123" (lambda () (values (read) 1)))) 224)

(for-each
 (lambda (op)
   (for-each
    (lambda (arg) ;(format-logged #t ";(~A ~A)~%" op arg)
      (test (op arg) 'error))
    (list (integer->char 65) 1 0 -1 (list 1) (cons 1 2) 'a-symbol (make-vector 3) abs lambda with-let
	  _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	  3.14 3/4 1.0+1.0i #\f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list char-ready? set-current-output-port set-current-input-port set-current-error-port
       close-input-port close-output-port open-input-file open-output-file
       read-char peek-char read 
       (lambda (arg) (write-char #\a arg))
       (lambda (arg) (write "hi" arg))
       (lambda (arg) (display "hi" arg))
       call-with-input-file with-input-from-file call-with-output-file with-output-to-file))

(with-output-to-file tmp-output-file
  (lambda ()
    (display "this is a test")
    (newline)))
    
(test (call-with-input-file tmp-output-file (lambda (p) (integer->char (read-byte p)))) #\t)
(test (with-input-from-string "123" (lambda () (read-byte))) 49)
;(test (with-input-from-string "1/0" read) 'error) ; this is a reader error in CL
;;; this test causes trouble when s7test is called from snd-test -- I can't see why

(let ((bytes (vector #o000 #o000 #o000 #o034 #o000 #o001 #o215 #o030 #o000 #o000 #o000 #o022 #o000 
		     #o000 #o126 #o042 #o000 #o000 #o000 #o001 #o000 #o000 #o000 #o000 #o000 #o001)))
  (with-output-to-file tmp-output-file
    (lambda ()
      (for-each
       (lambda (b)
	 (write-byte b))
       bytes)))
  
  (let ((ctr 0))
    (call-with-input-file tmp-output-file
      (lambda (p)	
	(if (not (string=? (port-filename p) tmp-output-file)) (display (port-filename p)))	
	(let loop ((val (read-byte p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format-logged #t "read-byte done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) val))
		    (format-logged #t "read-byte bytes[~D]: ~A ~A~%" ctr (bytes ctr) val))
		(set! ctr (+ 1 ctr))
		(loop (read-byte p))))))))
  
  (let ((ctr 0))
    (call-with-input-file tmp-output-file
      (lambda (p)
	(let loop ((val (read-char p)))
	  (if (eof-object? val)
	      (if (not (= ctr 26))
		  (format-logged #t "read-char done at ~A~%" ctr))
	      (begin
		(if (not (= (bytes ctr) (char->integer val)))
		    (format-logged #t "read-char bytes[~D]: ~A ~A~%" ctr (bytes ctr) (char->integer val)))
		(set! ctr (+ 1 ctr))
		(loop (read-char p))))))))
  )

(with-output-to-file tmp-output-file
  (lambda ()
    (if (not (string=? (port-filename (current-output-port)) tmp-output-file)) (display (port-filename (current-output-port))))
    (display "(+ 1 2) 32")
    (newline)
    (display "#\\a  -1")))

(with-input-from-file tmp-output-file
  (lambda ()
    (if (not (string=? (port-filename (current-input-port)) tmp-output-file)) (display (port-filename (current-input-port))))
    (let ((val (read)))
      (if (not (equal? val (list '+ 1 2)))
	  (format-logged #t ";file read +: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val 32))
	  (format-logged #t "file read 32: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val #\a))
	  (format-logged #t "file read a: ~A~%" val)))
    (let ((val (read)))
      (if (not (equal? val -1))
	  (format-logged #t "file read -1: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format-logged #t "file read #<eof>: ~A~%" val)))
    (let ((val (read)))
      (if (not (eof-object? val))
	  (format-logged #t "file read #<eof> again: ~A~%" val)))))

(let ()
  (call-with-input-string "012"
    (lambda (p)
      (do ((i 0 (+ i 1)))
	  ((= i 4))
	(let ((c (peek-char p)))
	  (let ((r (read-char p)))
	    (if (not (equal? c r))
		(format-logged #t ";peek-char: ~A ~A~%" c r))))))))
	  
(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format-logged #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format-logged #t ";c/e-> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";c/e -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(call-with-output-file tmp-output-file (lambda (p) (display "0123456789" p)))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (call-with-input-file tmp-output-file
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format-logged #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (go)))))
  (if (not (input-port? port))
      (format-logged #t ";c/e -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";c/e -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-string "0123456789"
             (lambda (p)
	       (set! port p)
	       (if (not (char=? (peek-char p) #\0))
		   (format-logged #t ";peek-char input-string 1: ~A~%" (peek-char p)))
	       (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format-logged #t ";c/e -> dw -> c/is -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";c/e -> dw -> c/is -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (call-with-exit
   (lambda (go)
     (dynamic-wind
	 (lambda () #f)
	 (lambda ()
	   (call-with-input-file tmp-output-file
            (lambda (p)
	      (set! port p)
	      (if (not (char=? (peek-char p) #\0))
		  (format-logged #t ";peek-char input-file: ~A~%" (peek-char p)))
	      (go))))
	 (lambda ()
	   (close-input-port port)))))
  (if (not (input-port? port))
      (format-logged #t ";c/e -> dw -> c/if -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";c/e -> dw -> c/if -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-string "0123456789"
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format-logged #t ";peek-char input-string: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format-logged #t ";catch -> c/is -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";catch -> c/is -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(let ((port #f))
  (catch #t
    (lambda ()
     (call-with-input-file tmp-output-file
       (lambda (p)
	 (set! port p)
	 (if (not (char=? (peek-char p) #\0))
	     (format-logged #t ";peek-char input-file: ~A~%" (peek-char p)))
	 (error 'oops))))
    (lambda args #f))
  (if (not (input-port? port))
      (format-logged #t ";catch -> c/if -> error -> port? ~A~%" port)
      (if (not (port-closed? port))
	  (begin
	    (format-logged #t ";catch -> c/if -> error -> closed? ~A~%" port)
	    (close-input-port port)))))

(test (with-output-to-string (lambda () (write (string (integer->char 4) (integer->char 8) (integer->char 20) (integer->char 30))))) "\"\\x04\\x08\\x14\\x1e\"")
(test (string-length "\x04\x08\x14\x1e") 4)
(test (char->integer (string-ref "\x0" 0)) 0)
(test (char->integer (string-ref "\x0e" 0)) 14)
(test (char->integer (string-ref "\x1e" 0)) 30)
(test (char->integer (string-ref "\xff" 0)) 255)
(test (string=?
        "\"\\x01\\x02\\x03\\x04\\x05\\x06\\x07\\x08\\x09x\\x0b\\x0c\\x0d\\x0e\\x0f\\x10\\x11\\x12\\x13\\x14\\x15\\x16\\x17\\x18\\x19\\x1a\\x1b\\x1c\\x1d\\x1e\\x1f !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f\\x80\\x81\\x82\\x83\\x84\\x85\\x86\\x87\\x88\\x89\\x8a\\x8b\\x8c\\x8d\\x8e\\x8f\\x90\\x91\\x92\\x93\\x94\\x95\\x96\\x97\\x98\\x99\\x9a\\x9b\\x9c\\x9d\\x9e\\x9f\\xa0¡¢£¤¥¦§¨©ª«¬\\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ\""             "\"\\x01\\x02\\x03\\x04\\x05\\x06\\x07\\x08\\x09x\\x0b\\x0c\\x0d\\x0e\\x0f\\x10\\x11\\x12\\x13\\x14\\x15\\x16\\x17\\x18\\x19\\x1a\\x1b\\x1c\\x1d\\x1e\\x1f !\\\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\\x7f\\x80\\x81\\x82\\x83\\x84\\x85\\x86\\x87\\x88\\x89\\x8a\\x8b\\x8c\\x8d\\x8e\\x8f\\x90\\x91\\x92\\x93\\x94\\x95\\x96\\x97\\x98\\x99\\x9a\\x9b\\x9c\\x9d\\x9e\\x9f\\xa0¡¢£¤¥¦§¨©ª«¬\\xad®¯°±²³´µ¶·¸¹º»¼½¾¿ÀÁÂÃÄÅÆÇÈÉÊËÌÍÎÏÐÑÒÓÔÕÖ×ØÙÚÛÜÝÞßàáâãäåæçèéêëìíîïðñòóôõö÷øùúûüýþÿ\"") 
      #t)
(test (string=? "\x61\x42\x63" "aBc") #t)


(when (provided? 'system-extras)
  ;; directory?
  (test (directory? tmp-output-file) #f)
  (test (directory? ".") #t)
  (test (directory?) 'error)
  (test (directory? "." 0) 'error)
  (for-each
   (lambda (arg)
     (test (directory? arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  
  ;; file-exists?
  (test (file-exists? tmp-output-file) #t)
  (test (file-exists? "not-a-file-I-hope") #f)
  (test (file-exists?) 'error)
  (test (file-exists? tmp-output-file 0) 'error)
  (for-each
   (lambda (arg)
     (test (file-exists? arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  
  ;; delete-file
  (test (delete-file tmp-output-file) 0)
  (test (file-exists? tmp-output-file) #f)
  (test (delete-file "not-a-file-I-hope") -1)
  (test (delete-file) 'error)
  (test (delete-file tmp-output-file 0) 'error)
  (for-each
   (lambda (arg)
     (test (delete-file arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  
  ;; getenv
  (test (pair? (member (getenv "HOME") '("/usr/home/bil" "/Users/bil" "/home/bil") string=?)) #t)
  (test (getenv "NO-ENV") "")
  (test (getenv) 'error)
  (test (getenv "HOME" #t) 'error)
  (for-each
   (lambda (arg)
     (test (getenv arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  
  ;; directory->list
  (test (directory->list) 'error)
  (test (directory->list "." 0) 'error)
  (for-each
   (lambda (arg)
     (test (directory->list arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
  
  ;; system
  (test (system "test -f s7test.scm") 0)
  (test (system) 'error)
  (test (let ((str (system "man fgrep" #t)))
	  (and (string? str)
	       (> (length str) 10000))) ; osx: 14479, linux: 40761
	#t)
  (for-each
   (lambda (arg)
     (test (system arg) 'error))
   (list -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	 3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
        
(let ()
  (define (args2)
    (with-output-to-file tmp-data-file
      (lambda ()
	(write-byte 1)
	(write-byte 2)
	(write-byte 1)
	(write-byte 2)))
    
    (let ((v (with-input-from-file tmp-data-file
	       (lambda ()
		 (vector (+ (read-byte) (ash (read-byte) 8))
			 (+ 1 (ash 2 8))
			 (+ (ash (read-byte) 8) (read-byte))
			 (+ (ash 1 8) 2))))))
      (if (not (equal? v #(513 513 258 258)))
	  (format *stderr* ";2 arg order check: ~A~%" v))))
  
  
  (args2)
  
  (define (args3)
    (with-output-to-file tmp-data-file
      (lambda ()
	(do ((i 0 (+ i 1)))
	    ((= i 8))
	  (write-byte 1)
	  (write-byte 2)
	  (write-byte 3))))
    
    (let ((v (with-input-from-file tmp-data-file
	       (lambda ()
		 (vector (+ (read-byte) (ash (read-byte) 8) (ash (read-byte) 16))
			 (+ 1 (ash 2 8) (ash 3 16))
			 (+ (read-byte) (read-byte) (ash (read-byte) 8))
			 (+ 1 2 (ash 3 8))
			 (+ (read-byte) (ash (read-byte) 8) (read-byte))
			 (+ 1 (ash 2 8) 3)
			 (+ (ash (read-byte) 8) (read-byte) (read-byte))
			 (+ (ash 1 8) 2 3)
			 (+ (ash (read-byte) 8) (ash (read-byte) 16) (read-byte))
			 (+ (ash 1 8) (ash 2 16) 3)
			 (+ (ash (read-byte) 8) (read-byte) (ash (read-byte) 16))
			 (+ (ash 1 8) 2 (ash 3 16)))))))
      (if (not (equal? v #(197121 197121 771 771 516 516 261 261 131331 131331 196866 196866)))
	  (format *stderr* ";3 arg order check: ~A~%" v))))
  
  (args3))

(if (not pure-s7)
    (for-each
     (lambda (arg)
       (test (char-ready? arg) 'error))
     (list "hi" -1 #\a 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
	   3.14 3/4 1.0+1.0i #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))


;;; -------- format --------
;;; format

(test (format #f "hiho") "hiho")
(test (format #f "") "")
(test (format #f "" 1) 'error)
(test (format #f "a") "a")
;(test (format #f "a\x00b") "a")

(test (format #f "~~") "~") ; guile returns this, but clisp thinks it's an error
(test (format #f "~~~~") "~~")
(test (format #f "a~~") "a~")
(test (format #f "~~a") "~a")
(test (format #f "~A" "") "")
(test (format #f "~{~^~A~}" ()) "")
(test (format #f "~{~^~{~^~A~}~}" '(())) "")
(test (format #f "~P" 1) "")
(test (format #f "~P" #\a) 'error)
(test (format #f "~0T") "")
(test (format #f "") "")
(test (format #f "~*~*" 1 2) "")
(test (format #f "~20,'~D" 3) "~~~~~~~~~~~~~~~~~~~3")
(test (format #f "~0D" 123) "123")
(test (format #f "~{~S~}" ()) "")
(test (format #f "~-1D" 123) 'error)
(test (format #f "~+1D" 123) 'error)
(test (format #f "~1.D" 123) 'error)
(test (format #f "~1+iD" 123) 'error)
(test (format #f "~1/2D" 123) 'error)
(test (format #f "~1/1D" 123) 'error)
(test (format #f "~20,'-1D" 123) 'error)

(for-each 
 (lambda (arg)
   (test (format arg "~D" 1) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (directive)
   (for-each 
    (lambda (arg)
      (test (format #f directive arg) 'error)
      (test (format #f directive) 'error))
    (list "hi" #\a 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand
	  #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list "~D" "~F" "~G" "~X" "~B" "~O" "~E" "~P"))

(test (format #f "~,1" 123) 'error)
;format "~,1" 123: numeric argument, but no directive!
;    (format #f "~,1" 123)

(test (format #f "~,123456789123456789123456789d" 1) 'error)
;format "~,123456789123456789123456789d" 1: numeric argument too large
;    (format-logged #t "~,123456789123456789123456789d" 1)
(test (format #f "~969424987x" 12) 'error)

(test (format #f "~D" 1 2) 'error)
;format: "~D" 1 2
;           ^: too many arguments
;    (format #f "~D" 1 2)

(test (format #f "~D~" 1) 'error)
;format: "~D~" 1
;           ^: control string ends in tilde
;    (format #f "~D~" 1)
(test (format #f "~") 'error)
(test (format #f " ~") 'error)
(test (format #f "~~~") 'error)
(test (format #f " ~~~") 'error)

(test (format #f "~@D" 1) 'error)
;format "~@D" 1: unknown '@' directive
;    (format #f "~@D" 1)

(test (format #f "~@p" #\a) 'error)
;format "~@p" #\a: '@P' directive argument is not an integer
;    (format #f "~@p" #\a)

(test (format #f "~P" 1+i) 'error)
;format "~P" 1+1i: 'P' directive argument is not a real number
;    (format #f "~P" 1+1i)

(test (format #f "~P" (real-part (log 0))) "s")

(test (format #f "~@p" 0+i) 'error)
;format "~@p" 0+1i: '@P' directive argument is not a real number
;    (format #f "~@p" 0+1i)

(test (format #f "~{~}") 'error)
;format "~{~}": missing argument
;    (format #f "~{~}")

(test (format #f "~{~a" '(1 2 3)) 'error)
;format "~{~a" (1 2 3): '{' directive, but no matching '}'
;    (format #f "~{~a" '(1 2 3))

(test (format #f "~{~a~}" '(1 . 2)) 'error)
;format "~{~a~}" (1 . 2): '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~a~}" '(1 . 2))

(test (let ((lst (cons 1 2))) (set-cdr! lst lst) (format #f "~{~A~}" lst)) 'error)
;format "~{~A~}" #1=(1 . #1#): '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~A~}" lst)

(test (format #f "~{~a~}" 'asdf) 'error)
;format "~{~a~}" asdf: '{' directive argument should be a proper list or something we can turn into a list
;    (format #f "~{~a~}" 'asdf)

(test (format #f "~{~a~}" ()) "")
(test (format #f "~{asd~}" '(1 2 3)) 'error)
;format: "~{asd~}" (1 2 3)
;             ^: '{...}' doesn't consume any arguments!
;    (format #f "~{asd~}" '(1 2 3))

(test (format #f "~}" '(1 2 3)) 'error)
;format "~}" (1 2 3): unmatched '}'
;    (format #f "~}" '(1 2 3))

(test (format #f "~C") 'error)
;format "~C": ~C: missing argument
;    (format #f "~C")

(test (format #f "~A ~C" #\a) 'error)
;format: "~A ~C" #\a
;            ^: ~C: missing argument
;    (format #f "~A ~C" #\a)

(test (format #f "~C" 1) 'error)
;format "~C" 1: 'C' directive requires a character argument
;    (format #f "~C" 1)

(test (format #f "~C" #<eof>) 'error)
;format "~C" #<eof>: 'C' directive requires a character argument
;    (format #f "~C" #<eof>)

(test (format #f "~1,9223372036854775807f" 1) 'error)
;format "~1,9223372036854775807f" 1: numeric argument too large
;    (format #f "~1,9223372036854775807f" 1)

(test (format #f "~1,2A" 1) 'error)
;format "~1,2A" 1: extra numeric argument
;    (format #f "~1,2A" 1)

(test (format #f "~F" #\a) 'error)
;format "~F" #\a: ~F: numeric argument required
;    (format #f "~F" #\a)

(test (format #f "~1,") 'error)
;format "~1,": format directive does not take a numeric argument
;    (format #f "~1,")

(test (format #f "~-1,") 'error)
;format "~-1,": unimplemented format directive
;    (format #f "~-1,")

(test (format #f "~L" 1) 'error)
;format "~L" 1: unimplemented format directive
;    (format #f "~L" 1)

(test (format #f "~*") 'error)
;format "~*": can't skip argument!

(test (format #f "~*~A") 'error)
(test (format #f "~*~*" 1) 'error)
(test (format #f "~N") 'error)
(test (format #f "~N" 2) 'error)
(test (format #f "~N." 2) 'error)
(test (format #f "~NT" 2.1) 'error)
(test (format #f "~NT" #\a) 'error)
(test (format #f "~N," 1) 'error)
(test (format #f "~N,N" 1 2) 'error)
(test (format #f "~N,N." 1 2) 'error)
(test (format #f "~,N" 1) 'error)
(test (format #f "~,N." 1) 'error)
(test (format #f "~ND" 123456789) 'error)
(test (format #f "~ND" -1) 'error)

(for-each
 (lambda (c)
   (test (apply format #f (string-append "~" (string c)) '(a)) 'error))
 (list #\H #\I #\J #\K #\L #\M #\Q #\R #\U #\V #\Y #\Z 
       #\[ #\\ #\] #\_ #\` #\$ #\# #\! #\" #\' #\( #\) #\+ #\, #\- #\. #\/ #\< #\= #\> #\?
       #\h #\i #\j #\k #\l #\m #\q #\r #\u #\v #\y #\z))

(test (format #f "~A" 1 2) 'error)
;format: "~A" 1 2
;           ^: too many arguments
;    (format #f "~A" 1 2)

(test (format #f "hiho~%ha") (string-append "hiho" (string #\newline) "ha"))
(test (format #f "~%") (string #\newline))
(test (format #f "~%ha") (string-append (string #\newline) "ha"))
(test (format #f "hiho~%") (string-append "hiho" (string #\newline)))

(test (eq? #\tab ((format #f "\t") 0)) #t)
(test (eq? #\newline ((format #f "\n") 0)) #t)
(test (eq? #\\ ((format #f "\\") 0)) #t)
(test (eq? #\" ((format #f "\"") 0)) #t)

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~A" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~A\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  #(1 2 3)   3.14   3/4  1.5+1.5i ()  #(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "hiho" "-1" "a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~A ho" 1) "hi 1 ho")
(test (format #f "hi ~a ho" 1) "hi 1 ho")
(test (format #f "~a~A~a" 1 2 3) "123")
(test (format #f "~a~~~a" 1 3) "1~3")
(test (format #f "~a~%~a" 1 3) (string-append "1" (string #\newline) "3"))

(for-each
 (lambda (arg res)
   (let ((val (catch #t (lambda () (format #f "~S" arg)) (lambda args 'error))))
     (if (or (not (string? val))
	     (not (string=? val res)))
	 (begin (display "(format #f \"~S\" ") (display arg) 
		(display " returned \"") (display val) 
		(display "\" but expected \"") (display res) (display "\"") 
		(newline)))))
 (list "hiho"  -1  #\a  1   #f   #t  #(1 2 3)   3.14   3/4  1.5+1.5i ()  #(())  (list 1 2 3) '(1 . 2) 'hi)
 (list "\"hiho\"" "-1" "#\\a" "1" "#f" "#t" "#(1 2 3)" "3.14" "3/4" "1.5+1.5i"   "()" "#(())" "(1 2 3)"    "(1 . 2)" "hi"))

(test (format #f "hi ~S ho" 1) "hi 1 ho")
(test (format #f "hi ~S ho" "abc") "hi \"abc\" ho")
(test (format #f "~s~a" #\a #\b) "#\\ab")
(test (format #f "~C~c~C" #\a #\b #\c) "abc")
;(test (format #f "1 2~C 3 4" #\null) "1 2") ; ?? everyone does something different here
;; s7 used to return "1 2 3 4" because it treated ~C as a string (empty in this case)
(test  (format #f "1 2~C 3 4" #\null) "1 2\x00 3 4") ; this is also what Guile returns
(test (format #f "~nc" 3 #\a) "aaa")
(test (format #f "~nc" 0 #\a) "")
(test (format #f "~0c" #\a) "")
(test (format #f "~01c" #\a) "a")
(test (format #f "~002c" #\a) "aa")
(test (format #f "~nc" -1 #\a) 'error)
(test (format #f "~nc" most-positive-fixnum #\a) 'error)
(test (format #f "~nc" 1.0 #\a) 'error)
(test (format #f "~n~nc" 1 2 #\a) 'error)
(test (format #f "~na" 1 #\a) 'error)

(test (format #f "~{~A~}" '(1 2 3)) "123")
(test (format #f "asb~{~A ~}asb" '(1 2 3 4)) "asb1 2 3 4 asb")
(test (format #f "asb~{~A ~A.~}asb" '(1 2 3 4)) "asb1 2.3 4.asb")
(test (format #f ".~{~A~}." ()) "..")

(test (format #f "~{~A ~A ~}" '(1 "hi" 2 "ho")) "1 hi 2 ho ")
(test (format #f "~{.~{+~A+~}.~}" (list (list 1 2 3) (list 4 5 6))) ".+1++2++3+..+4++5++6+.")
(test (format #f "~{~s ~}" '(fred jerry jill)) "fred jerry jill ")
(test (format #f "~{~s~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{~s~^~^ ~}" '(fred jerry jill)) "fred jerry jill")
(test (format #f "~{.~{~A~}+~{~A~}~}" '((1 2) (3 4 5) (6 7 8) (9))) ".12+345.678+9")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)))) ".+-1-2+-3-4-5")
(test (format #f "~{.~{+~{-~A~}~}~}" '(((1 2) (3 4 5)) ((6) (7 8 9)))) ".+-1-2+-3-4-5.+-6+-7-8-9")

(test (format #f "~A ~* ~A" 1 2 3) "1  3")
(test (format #f "~*" 1) "")
(test (format #f "~{~* ~}" '(1 2 3)) "   ")
(test (format #f "~A" catch) "catch")
(test (format #f "this is a ~
             sentence") "this is a sentence")
(test (format #f "~{~C~}" "hi") "hi")
(test (format #f "~{~C~}" #(#\h #\i)) "hi")

(test (format #f "~{.~{~C+~}~}" '((#\h #\i) (#\h #\o))) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" '("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #("hi" "ho")) ".h+i+.h+o+")
(test (format #f "~{.~{~C+~}~}" #(#(#\h #\i) #(#\h #\o))) ".h+i+.h+o+")

; (format #f "~{.~{~C~+~}~}" #2d((#\h #\i) (#\h #\o))) error?? -- this is documented...
(test (format #f "~{~A~}" #2D((1 2) (3 4))) "1234") ; this seems inconsistent with:
(test (format #f "~{~A~}" '((1 2) (3 4))) "(1 2)(3 4)")
(test (format #f "~{~A ~}" #2d((1 2) (3 4))) "1 2 3 4 ")
(test (format #f "1~\
a2" 3) "132")
(test (format #f "1~
                 ~a2" 3) "132")

(test (format #f "~{~{~C~^ ~}~^...~}" (list "hiho" "test")) "h i h o...t e s t")

;; ~nT handling is a mess -- what are the defaults?  which is column 1? do we space up to or up to and including?

(test (format #f "~A:~8T~A" 100 'a)   "100:   a")
(test (format #f "~A:~nT~A" 100 8 'a)   "100:   a")
(test (format #f "~A:~8T~A" 0 'a)     "0:     a")
(test (format #f "~A:~8T~A" 10000 'a) "10000: a")
(test (format #f "~8T~A" 'a)      "       a")
(test (format #f "1212:~8T~A" 'a) "1212:  a")
(test (format #f "~D:~8T~A" 100 'a)   "100:   a")
(test (format #f "~D:~8T~A" 0 'a)     "0:     a")
(test (format #f "~D:~8T~A" 10000 'a) "10000: a")
(test (format #f "~a~10,7Tb" 1)     "1               b")
(test (format #f "~a~10,7Tb" 10000) "10000           b")
(test (format #f "~a~10,12Tb" 1)     "1                    b")
(test (format #f "~a~10,12Tb" 10000) "10000                b")
(test (format #f "~a~n,nTb" 10000 10 12) "10000                b")
(test (format #f "~n,'xT" 8) "xxxxxxx")
(test (format #f "~n,' T" 8) "       ")

(test (length (format #f "~{~A~}~40T." '(1 2 3))) 40)
(test (length (format #f "~{~A ~}~40T." '(1 2 3))) 40)
(test (length (format #f "~{~,3F ~}~40T." '(1.0 2.0 3.0))) 40)
(test (length (format #f "~S~40T." pi)) (if with-bignums 44 40))

(test (format #f "asdh~20Thiho") "asdh               hiho")
(test (format #f "asdh~2Thiho") "asdhhiho")
(test (format #f "a~Tb") "ab")
(test (format #f "0123456~4,8Tb") "0123456    b")
(test (format #f "0123456~0,8Tb") "0123456b")
(test (format #f "0123456~10,8Tb") "0123456          b")
(test (format #f "0123456~1,0Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~1,Tb") "0123456b")
(test (format #f "0123456~,Tb") "0123456b")
(test (format #f "0123456~7,10Tb") "0123456         b")
(test (format #f "0123456~8,10tb") "0123456          b")
(test (format #f "0123456~3,12tb") "0123456       b")
(test (format #f "~40TX") "                                       X")
(test (format #f "X~,8TX~,8TX") "X      X       X")
(test (format #f "X~8,TX~8,TX") "X      XX")
(test (format #f "X~8,10TX~8,10TX") "X                X         X")
(test (format #f "X~8,0TX~8,0TX") "X      XX")
(test (format #f "X~0,8TX~0,8TX") "X      X       X")
(test (format #f "X~1,8TX~1,8TX") "X       X       X")
(test (format #f "X~,8TX~,8TX") "X      X       X") ; ??
(test (format #f "X~TX~TX") "XXX") ; clisp and sbcl say "X X X" here and similar differences elsewhere -- is it colnum or colinc as default if no comma?
(test (format #f "X~2TX~4TX") "XX X")
(test (format #f "X~0,0TX~0,0TX") "XXX")
(test (format #f "X~0,TX~0,TX") "XXX")
(test (format #f "X~,0TX~,0TX") "XXX")

(test (format #f "~0D" 123) "123")
(test (format #f "~0F" 123.123) "123.123000")
(test (format #f "~,0D" 123) "123")
(test (format #f "~,0F" 123.123) "123.0")
(test (format #f "~,D" 123) "123")
(test (format #f "~,F" 123.123) "123.123000")
(test (format #f "~0,D" 123) "123")
(test (format #f "~0,F" 123.123) "123.123000")
(test (format #f "~0,0D" 123) "123")
(test (format #f "~n,nD" 0 0 123) "123")
(test (format #f "~0,0F" 123.123) "123.0")
(test (format #f "~0,0,D" 123) 'error)
(test (format #f "~n,n,D" 0 0 123) 'error)
(test (format #f "~0,0,F" 123.123) 'error)

(test (format #f "~,3F" 1+i) "1.000+1.000i")
(test (format #f "~,nF" 3 1+i) "1.000+1.000i")
(test (format #f "~,3G" 1+i) "1+1i")
(test (format #f "~,3E" 1+i) "1.000e+00+1.000e+00i")
(test (format #f "~,3F" 1-i) "1.000-1.000i")
(test (format #f "~,3G" 1-i) "1-1i")
(test (format #f "~,3E" 1-i) "1.000e+00-1.000e+00i")

;; not sure about these:
(test (format #f "~X" 1-i) "1.0-1.0i")
(test (format #f "~,3D" 1-i) "1.000e+00-1.000e+00i")
(test (format #f "~A" 1-i) "1-1i")

(test (format #f "~W" 3) "3")
(test (format #f "~W" 3/4) "3/4")
(test (format #f "~W" 3.4) "3.4")
(if with-bignums
    (test (format #f "~W" pi) "3.141592653589793238462643383279502884195E0")
    (test (format #f "~W" pi) "3.141592653589793"))
(test (format #f "~W" 3+4i) "3+4i")
(test (format #f "~W" 3-4i) "3-4i")
(unless with-bignums
  (let ((name (if pure-s7 'complex 'complex))
	(func (if pure-s7 complex complex)))
    (test (format #f "~W" (func 1/0 0)) "nan.0")
    (test (format #f "~W" (func 1/0 1)) (format #f "(~S nan.0 1)" name))
    (test (format #f "~W" (func inf.0 1/0)) (format #f "(~S inf.0 nan.0)" name))
    (test (format #f "~W" (log 0)) (format #f "(~S -inf.0 3.141592653589793)" name))))

;; see also object->string with :readable

(test (format #f "~000000000000000000000000000000000000000000003F" 123.123456789) "123.123457")
(test (format #f "~922337203685477580F" 123.123) 'error)   ; numeric argument too large
(test (format #f "~,922337203685477580F" 123.123) 'error)  
(test (format #f "~1,922337203685477580F" 123.123) 'error) 
(test (format #f "~1 ,2F" 123.456789) 'error)
(test (format #f "~1, 2F" 123.456789) 'error)
(test (format #f "~1, F" 123.456789) 'error)

(when with-bignums
    (begin
      (test (format #f "~o" 1e19) "1.053071060221172E21")
      (test (format #f "~o" -1e19) "-1.053071060221172E21")
      (test (format #f "~x" 1e19) "8.ac7230489e8@15")
      (test (format #f "~b" 1e19) "1.00010101100011100100011000001001000100111101E63")
      (test (format #f "~o" 1e308) "1.071474702753621177617256074117252375235444E341")
      (test (format #f "~o" -1e308) "-1.071474702753621177617256074117252375235444E341")
      (test (format #f "~x" 1e308) "8.e679c2f5e44ff8f570f09eaa7ea7648@255")
      (test (format #f "~x" 9.22e18) "7.ff405267d1a@15")
      (test (format #f "~b" 1e308) "1.0001110011001111001110000101111010111100100010011111111100011110101011100001111000010011110101010100111111010100111011001001E1023")
      (test (format #f "~,791o" 1e308) "1.071474702753621177617256074117252375235444E341")
      (test (format #f "~1200,2g" 1e308) "                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   9.999999999999999999999999999999999999982E307")
      (test (format #f "~o" 1e19-1e20i) "1.053071060221172E21-1.2657072742654304E22i")
      (test (format #f "~x" 1e308+1e300i) "8.e679c2f5e44ff8f570f09eaa7ea7648@255+1.7e43c8800759ba59c08e14c7cd7aad86@249i"))
    (begin
      (test (format #f "~o" 1e19) "1.053071e21")
      (test (format #f "~o" -1e19) "-1.053071e21")
      (test (format #f "~x" 1e19) "8.ac723e15")
      (test (format #f "~b" 1e19) "1.000101e63")
      (test (format #f "~o" 1e308) "1.071474e341")
      (test (format #f "~o" -1e308) "-1.071474e341")
      (test (format #f "~x" 1e308) "8.e679c2e255")
      (test (or (string=? (format #f "~x" 9.22e18) "7ff405267d1a0000.0")
		(string=? (format #f "~x" 9.22e18) "7.ff4052e15"))
	    #t) ; this depends on a cutoff point in s7.c, L8850, number_to_string_with_radix
      (test (format #f "~b" 1e308) "1.000111e1023")
      (test (format #f "~,791o" 1e308) "1.0714747027536212e341")
      (test (format #f "~1200,2g" 1e308) "                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          1e+308")
      (test (format #f "~o" 1e19-1e20i) "1.053071e21-1.265707e22i")
      (test (format #f "~x" 1e308+1e300i) "8.e679c2e255+1.7e43c8e249i")))

(test (= (length (substring (format #f "~%~10T.") 1)) (length (format #f "~10T."))) #t)
(test (= (length (substring (format #f "~%-~10T.~%") 1)) (length (format #f "-~10T.~%"))) #t)
(test (string=? (format #f "~%|0 1 2|~21T|5  8  3  2|~%~
                              |1 2 3| |0 1 2 3|~21T|8 14  8  6|~%~
                              |2 3 0| |1 2 3 0| = ~21T|3  8 13  6|~%~
                              |3 0 1| |2 3 0 1|~21T|2  6  6 10|~%")
		"
|0 1 2|             |5  8  3  2|
|1 2 3| |0 1 2 3|   |8 14  8  6|
|2 3 0| |1 2 3 0| = |3  8 13  6|
|3 0 1| |2 3 0 1|   |2  6  6 10|
") #t)


(if (not with-windows) (test (format #f "~S" '(+ 1/0 1/0)) "(+ nan.0 nan.0)")) ; !?
(if (not with-windows) (test (format #f "~S" '(+ '1/0 1/0)) "(+ 'nan.0 nan.0)")) ; !? 
(test (format #f "~S" '(+ 1/0 1.0/0.0)) (format #f "~S" (list '+ '1/0 '1.0/0.0)))
(test (format #f "~S" (quote (+ '1 1))) "(+ '1 1)")


(test (format #f "~12,''D" 1) "'''''''''''1")
(test (let ((str "~12,'xD")) (set! (str 5) #\space) (format #f str 1)) "           1")
(test (format #f "~12,' D" 1) "           1")
(test (format #f "~12,'\\D" 1) "\\\\\\\\\\\\\\\\\\\\\\1")
(test (format #f "~12,'\"D" 1) "\"\"\"\"\"\"\"\"\"\"\"1")
(test (format #f "~12,'~D" 1) "~~~~~~~~~~~1")
(test (format #f "~12,',d" 1) ",,,,,,,,,,,1")
(test (format #f "~12,',,d" 1) 'error)
(test (format #f "~12,,d" 1) 'error)
(test (format #f "~n,,d" 12 1) 'error)

(test (string=? (format #f "~%~&" ) (string #\newline)) #t)
(test (string=? (format #f "~%a~&" ) (string #\newline #\a #\newline)) #t)
(test (string=? (format #f "~%~%") (string #\newline #\newline)) #t)
(test (string=? (format #f "~10T~%~&~10T.") (format #f "~10T~&~&~10T.")) #t)
(test (string=? (format #f "~10T~&~10T.") (format #f "~10T~%~&~&~&~&~10T.")) #t)
(test (length (format #f "~%~&~%")) 2)
(test (length (format #f "~%~&~&~&~&~%")) 2)
(test (length (format #f "~&~%")) 1)

(test (format #f "~2,1F" 0.5) "0.5")
(test (format #f "~:2T") 'error)
(test (format #f "~2,1,3F" 0.5) 'error)
(test (format #f "~<~W~>" 'foo) 'error)
(test (format #f "~{12") 'error)
(test (format #f "~{}") 'error)
(test (format #f "~{}" '(1 2)) 'error)
(test (format #f "{~}" '(1 2)) 'error)
(test (format #f "~{~{~}}" '(1 2)) 'error)
(test (format #f "~}" ) 'error)
;(test (format #f "#|~|#|") 'error) ; ~| is ~^+ now
(test (format #f "~1.5F" 1.5) 'error)
(test (format #f "~1+iF" 1.5) 'error)
(test (format #f "~1,1iF" 1.5) 'error)
(test (format #f "~0" 1) 'error)
(test (format #f "~1") 'error)
(test (format #f "~^" 1) 'error)
(test (format #f "~.0F" 1.0) 'error)
(test (format #f "~1.0F" 1.0) 'error)
(test (format #f "~-1F" 1.0) 'error)
(test (format #f "~^") "")
(test (format #f "~A ~A~|this is not printed" 1 2) "1 2")
(test (format #f "~^~A~^~A~^this is not printed" 1 2) "12")
(test (format #f "~|") "")
(test (format #f "~D~" 9) 'error)
(test (format #f "~&" 9) 'error)
(test (format #f "~D~100T~D" 1 1) "1                                                                                                  1")
(test (format #f ".~P." 1) "..")
(test (format #f ".~P." 1.0) "..")
(test (format #f ".~P." 1.2) ".s.")
(test (format #f ".~P." 2/3) ".s.")
(test (format #f ".~P." 2) ".s.")
(test (format #f ".~p." 1) "..")
(test (format #f ".~p." 1.0) "..")
(test (format #f ".~p." 1.2) ".s.")
(test (format #f ".~p." 2) ".s.")
(test (format #f ".~@P." 1) ".y.")
(test (format #f ".~@P." 1.0) ".y.")
(test (format #f ".~@P." 1.2) ".ies.")
(test (format #f ".~@P." 2) ".ies.")
(test (format #f ".~@p." 1) ".y.")
(test (format #f ".~@p." 1.0) ".y.")
(test (format #f ".~@p." 1.2) ".ies.")
(test (format #f ".~@p." 2) ".ies.")
(test (format #f ".~P." 1.0+i) 'error)
(test (format #f ".~P." 1/0) ".s.")
(test (format #f "~P" 1) "") ; Clisp does this
(if (not with-windows) (test (format #f ".~P." (real-part (log 0))) ".s."))

(test (format #f (string #\~ #\a) 1) "1")
(test (format #f (format #f "~~a") 1) "1")
(test (format #f (format #f "~~a") (format #f "~D" 1)) "1")
(test (format #f "~A" (quasiquote quote)) "quote")

(test (format #f "~f" (/ 1 3)) "1/3") ; hmmm -- should it call exact->inexact?
(test (format #f "~f" 1) "1")
(test (format #f "~F" most-positive-fixnum) "9223372036854775807")

(unless with-bignums
  (test (format #f "~,20F" 1e-20) "0.00000000000000000001")
  (test (format #f "~,40F" 1e-40) "0.0000000000000000000000000000000000000001"))
;; if with bignums, these needs more bits

;;; the usual troubles here with big floats:
;;; (format #f "~F" 922337203685477580.9) -> "922337203685477632.000000"
;;; (format #f "~F" 9223372036854775.9) -> "9223372036854776.000000"
;;; (format #f "~F" 1e25) -> "10000000000000000905969664.000000"
;;; or small:
;;; (format #f "~,30F" 1e-1) -> "0.100000000000000005551115123126"

(when with-bignums
  (test (format #f "~A" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601")
  (test (format #f "~D" -7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601) "-7043009959286724629649270926654940933664689003233793014518979272497911394287216967075767325693021717277238746020477538876750544587281879084559996466844417586093291189295867052594478662802691926547232838591510540917276694295393715934079679531035912244103731582711556740654671309980075069010778644542022/670550434139267031632063192770201289106737062379324644110801846820471752716238484923370056920388400273070254958650831435834503195629325418985020030706879602898158806736813101434594805676212779217311897830937606064579213895527844045511878668289820732425014254579493444623868748969110751636786165152601"))

(test (format #f "~@F" 1.23) 'error)
(test (format #f "~{testing ~D ~C ~}" (list 0 #\( 1 #\) 2 #\* 3 #\+ 4 #\, 5 #\- 6 #\. 7 #\/ 8 #\0 9 #\1 10 #\2 11 #\3 12 #\4 13 #\5 14 #\6 15 #\7 16 #\8 17 #\9 18 #\: 19 #\; 20 #\< 21 #\= 22 #\> 23 #\? 24 #\@ 25 #\A 26 #\B 27 #\C 28 #\D 29 #\E 30 #\F 31 #\G 32 #\H 33 #\I 34 #\J 35 #\K 36 #\L 37 #\M 38 #\N 39 #\O 40 #\P 41 #\Q 42 #\R 43 #\S 44 #\T 45 #\U 46 #\V 47 #\W 48 #\X 49 #\Y 50 #\( 51 #\) 52 #\* 53 #\+ 54 #\, 55 #\- 56 #\. 57 #\/ 58 #\0 59 #\1 60 #\2 61 #\3 62 #\4 63 #\5 64 #\6 65 #\7 66 #\8 67 #\9 68 #\: 69 #\; 70 #\< 71 #\= 72 #\> 73 #\? 74 #\@ 75 #\A 76 #\B 77 #\C 78 #\D 79 #\E 80 #\F 81 #\G 82 #\H 83 #\I 84 #\J 85 #\K 86 #\L 87 #\M 88 #\N 89 #\O 90 #\P 91 #\Q 92 #\R 93 #\S 94 #\T 95 #\U 96 #\V 97 #\W 98 #\X 99 #\Y))
      "testing 0 ( testing 1 ) testing 2 * testing 3 + testing 4 , testing 5 - testing 6 . testing 7 / testing 8 0 testing 9 1 testing 10 2 testing 11 3 testing 12 4 testing 13 5 testing 14 6 testing 15 7 testing 16 8 testing 17 9 testing 18 : testing 19 ; testing 20 < testing 21 = testing 22 > testing 23 ? testing 24 @ testing 25 A testing 26 B testing 27 C testing 28 D testing 29 E testing 30 F testing 31 G testing 32 H testing 33 I testing 34 J testing 35 K testing 36 L testing 37 M testing 38 N testing 39 O testing 40 P testing 41 Q testing 42 R testing 43 S testing 44 T testing 45 U testing 46 V testing 47 W testing 48 X testing 49 Y testing 50 ( testing 51 ) testing 52 * testing 53 + testing 54 , testing 55 - testing 56 . testing 57 / testing 58 0 testing 59 1 testing 60 2 testing 61 3 testing 62 4 testing 63 5 testing 64 6 testing 65 7 testing 66 8 testing 67 9 testing 68 : testing 69 ; testing 70 < testing 71 = testing 72 > testing 73 ? testing 74 @ testing 75 A testing 76 B testing 77 C testing 78 D testing 79 E testing 80 F testing 81 G testing 82 H testing 83 I testing 84 J testing 85 K testing 86 L testing 87 M testing 88 N testing 89 O testing 90 P testing 91 Q testing 92 R testing 93 S testing 94 T testing 95 U testing 96 V testing 97 W testing 98 X testing 99 Y ")


(let ((vect1 #3D(((1 2 3) (3 4 5)) ((5 6 1) (7 8 2))))
      (vect2 #2d((1 2 3 4 5 6) (7 8 9 10 11 12)))
      (vect3 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14))
      (vect4 #3D(((1 2) (3 4) (5 6)) ((7 8) (9 10) (11 12)))))
  (do ((i 0 (+ i 2)))
      ((>= i 10))
    (set! (*s7* 'print-length) i)
    (test (object->string vect1) (format #f "~A" vect1))
    (test (object->string vect2) (format #f "~A" vect2))
    (test (object->string vect3) (format #f "~A" vect3))
    (test (object->string vect4) (format #f "~A" vect4))))

(let-temporarily (((*s7* 'print-length) 0))
  (test (format #f "~A" #()) "#()"))

(let-temporarily (((*s7* 'print-length) 3))
  (let ((lst (list 1)))
    (set-car! lst lst)
    (let ((v (vector 1 1 1 1 1 1 1 1 1 lst)))
      (let ((str (format #f "~A" v)))
	(test (string=? str "#(1 1 1 ...)") #t)))))

(test (format #f "~D" 123) "123")
(test (format #f "~X" 123) "7b")
(test (format #f "~B" 123) "1111011")
(test (format #f "~O" 123) "173")

(test (format #f "~10D" 123) "       123")
(test (format #f "~nD" 10 123) "       123")
(test (format #f "~10X" 123) "        7b")
(test (format #f "~10B" 123) "   1111011")
(test (format #f "~10O" 123) "       173")

(test (format #f "~D" -123) "-123")
(test (format #f "~X" -123) "-7b")
(test (format #f "~B" -123) "-1111011")
(test (format #f "~O" -123) "-173")

(test (format #f "~10D" -123) "      -123")
(test (format #f "~10X" -123) "       -7b")
(test (format #f "~10B" -123) "  -1111011")
(test (format #f "~10O" -123) "      -173")

(test (format #f "~d" 123) "123")
(test (format #f "~x" 123) "7b")
(test (format #f "~b" 123) "1111011")
(test (format #f "~o" 123) "173")

(test (format #f "~10d" 123) "       123")
(test (format #f "~10x" 123) "        7b")
(test (format #f "~10b" 123) "   1111011")
(test (format #f "~10o" 123) "       173")

(test (format #f "~d" -123) "-123")
(test (format #f "~x" -123) "-7b")
(test (format #f "~b" -123) "-1111011")
(test (format #f "~o" -123) "-173")

(test (format #f "~10d" -123) "      -123")
(test (format #f "~10x" -123) "       -7b")
(test (format #f "~10b" -123) "  -1111011")
(test (format #f "~10o" -123) "      -173")

(test (format #f "~D" most-positive-fixnum) "9223372036854775807")
(test (format #f "~D" (+ 1 most-negative-fixnum)) "-9223372036854775807")
      
(test (format #f "~X" most-positive-fixnum) "7fffffffffffffff")
(test (format #f "~X" (+ 1 most-negative-fixnum)) "-7fffffffffffffff")
      
(test (format #f "~O" most-positive-fixnum) "777777777777777777777")
(test (format #f "~O" (+ 1 most-negative-fixnum)) "-777777777777777777777")
      
(test (format #f "~B" most-positive-fixnum) "111111111111111111111111111111111111111111111111111111111111111")
(test (format #f "~B" (+ 1 most-negative-fixnum)) "-111111111111111111111111111111111111111111111111111111111111111")
      
(num-test (inexact->exact most-positive-fixnum) most-positive-fixnum)

(test (format #f "~0D" 123) "123")
(test (format #f "~0X" 123) "7b")
(test (format #f "~0B" 123) "1111011")
(test (format #f "~0O" 123) "173")

(test (format #f "" 1) 'error)
(test (format #f "hiho" 1) 'error)
(test (format #f "a~%" 1) 'error) ; some just ignore extra args

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format arg "hiho")) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format ") (display arg) (display " \"hiho\")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #(1 2 3) 3.14 3/4 1.0+1.0i 'hi :hi #<eof> abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f ") (display arg) (display ")")
		(display " returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list -1 #\a 1 #f #t #(1 2 3) 3.14 3/4 1.0+1.0i () 'hi abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(test (format #f "hi ~A ho" 1 2) 'error)
(test (format #f "hi ~A ho") 'error)
(test (format #f "hi ~S ho") 'error)
(test (format #f "hi ~S ho" 1 2) 'error)
(test (format #f "~C" 1) 'error)
(test (format #f "123 ~R 321" 1) 'error)
(test (format #f "123 ~,3R 321" 1) 'error)
(test (format #f "~,2,3,4D" 123) 'error)

(test (format #f "hi ~Z ho") 'error)
(test (format #f "hi ~+ ho") 'error)
(test (format #f "hi ~# ho") 'error)
(test (format #f "hi ~, ho") 'error)

(test (format #f "hi ~} ho") 'error)
(test (format #f "hi {ho~}") 'error)

(test (format #f "asb~{~A asd" '(1 2 3)) 'error)
(test (format #f "~{~A~}" 1 2 3) 'error)
(test (format #f "asb~{~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ . ~}asd" '(1 2 3)) 'error)
(test (format #f "asb~{ hiho~~~}asd" '(1 2 3)) 'error)

(test (format #f "~12C" #\a) "aaaaaaaaaaaa")
(test (format #f ".~0C." #\a) "..")
(test (format #f "~10C" #\space) "          ")

(test (format #f "~12P" #\a) 'error)
(test (format #f "~12*" #\a) 'error)
(test (format #f "~12%" #\a) 'error)
(test (format #f "~12^" #\a) 'error)
(test (format #f "~12{" #\a) 'error)
(test (format #f "~12,2A" #\a) 'error)

(test (format #f "~12,A" #\a) 'error) ; s7 misses padding errors such as (format #f "~12,' A" #\a)

(for-each
 (lambda (arg)
   (test (format #f "~F" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~D" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~P" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~X" arg) 'error))
 (list "hi" #\a 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f "~C" arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(for-each
 (lambda (arg)
   (test (format #f arg 123) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (format #f "~{~A ~A ~}" '(1 "hi" 2)) 'error)
(for-each
 (lambda (arg)
   (let ((result (catch #t (lambda () (format #f "~F" arg)) (lambda args 'error))))
     (if (not (eq? result 'error))
	 (begin (display "(format #f \"~F\" ") (display arg)
		(display ") returned ") (display result) 
		(display " but expected 'error")
		(newline)))))
 (list #\a #(1 2 3) "hi" () 'hi abs (lambda () 1) #(()) (list 1 2 3) '(1 . 2)))

(test (format #f "~D") 'error)
(test (format () "hi") 'error)
(test (format #f "~F" "hi") 'error)
(test (format #f "~D" #\x) 'error)
(test (format #f "~C" (list 1 2 3)) 'error)
(test (format #f "~1/4F" 1.4) 'error)
(test (format #f "~1.4F" 1.4) 'error)
(test (format #f "~F" (real-part (log 0.0))) "-inf.0")
(test (let ((val (format #f "~F" (/ (real-part (log 0.0)) (real-part (log 0.0)))))) (or (string=? val "nan.0") (string=? val "-nan.0"))) #t)
(test (format #f "~1/4T~A" 1) 'error)
(test (format #f "~T") "")
(test (format #f "~@P~S" 1 '(1)) "y(1)")
(test (format #f ".~A~*" 1 '(1)) ".1")
(test (format #f "~*~*~T" 1 '(1)) "")

(test (format #f "~A" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~S" 'AB\c) "(symbol \"AB\\\\c\")")
(test (format #f "~A" '(AB\c () xyz)) "((symbol \"AB\\\\c\") () xyz)")
(test (format #f "~,2f" 1234567.1234) "1234567.12")
(test (format #f "~5D" 3) "    3")
(test (format #f "~5,'0D" 3) "00003")
(test (format #f "++~{-=~s=-~}++" (quote (1 2 3))) "++-=1=--=2=--=3=-++")

(test (format) 'error)
(for-each
 (lambda (arg)
   (test (format arg) 'error))
 (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1))))
(test (format "hi") "hi") ; !?
(test (format "~A ~D" 1/3 2) "1/3 2")
(test (format "") "")

;; from slib/formatst.scm
(test (string=? (format #f "abc") "abc") #t)
(test (string=? (format #f "~a" 10) "10") #t)
(test (string=? (format #f "~a" -1.2) "-1.2") #t)
(test (string=? (format #f "~a" 'a) "a") #t)
(test (string=? (format #f "~a" #t) "#t") #t)
(test (string=? (format #f "~a" #f) "#f") #t)
(test (string=? (format #f "~a" "abc") "abc") #t)
(test (string=? (format #f "~a" #(1 2 3)) "#(1 2 3)") #t)
(test (string=? (format #f "~a" ()) "()") #t)
(test (string=? (format #f "~a" '(a)) "(a)") #t)
(test (string=? (format #f "~a" '(a b)) "(a b)") #t)
(test (string=? (format #f "~a" '(a (b c) d)) "(a (b c) d)") #t)
(test (string=? (format #f "~a" '(a . b)) "(a . b)") #t)
(test (string=? (format #f "~a ~a" 10 20) "10 20") #t)
(test (string=? (format #f "~a abc ~a def" 10 20) "10 abc 20 def") #t)
(test (string=? (format #f "~d" 100) "100") #t)
(test (string=? (format #f "~x" 100) "64") #t)
(test (string=? (format #f "~o" 100) "144") #t)
(test (string=? (format #f "~b" 100) "1100100") #t)
(test (string=? (format #f "~10d" 100) "       100") #t)
(test (string=? (format #f "~10,'*d" 100) "*******100") #t)
(test (string=? (format #f "~c" #\a) "a") #t)
(test (string=? (format #f "~c" #\space) " ") #t)
(test (string=? (format #f "~C" #\x91) "\x91") #t)
(test (string=? (format #f "~C" #\x9) "\x09") #t)
(test (string=? (format #f "~C" #\~) "~") #t)
(test (string=? (format #f "~A" #\x91) "\x91") #t)
(test (string=? (format #f "~S" #\x91) "#\\x91") #t)
(test (string=? (format #f "~A" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~S" (string->symbol "hi")) "hi") #t)
(test (string=? (format #f "~A" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~S" (string->symbol ";\\\";")) "(symbol \";\\\\\\\";\")") #t)
(test (string=? (format #f "~A" (string->symbol (string #\, #\. #\# #\; #\" #\\ #\' #\`))) "(symbol \",.#;\\\"\\\\'`\")") #t)

(test (string=? (format #f "~~~~") "~~") #t)
(test (string=? (format #f "~s" "abc") "\"abc\"") #t)
(test (string=? (format #f "~s" "abc \\ abc") "\"abc \\\\ abc\"") #t)
(test (string=? (format #f "~a" "abc \\ abc") "abc \\ abc") #t)
(test (string=? (format #f "~s" "abc \" abc") "\"abc \\\" abc\"") #t)
(test (string=? (format #f "~a" "abc \" abc") "abc \" abc") #t)
(test (string=? (format #f "~s" #\space) "#\\space") #t)
(test (string=? (format #f "~s" #\newline) "#\\newline") #t)
(test (string=? (format #f "~s" #\a) "#\\a") #t)
(test (string=? (format #f "~a" '(a "b" c)) "(a \"b\" c)") #t)
(test (string=? (format #f "abc~
         123") "abc123") #t)
(test (string=? (format #f "abc~
123") "abc123") #t)
(test (string=? (format #f "abc~
") "abc") #t)
(test (string=? (format #f "~{ ~a ~}" '(a b c)) " a  b  c ") #t)
(test (string=? (format #f "~{ ~a ~}" ()) "") #t)
(test (string=? (format #f "~{ ~a ~}" "") "") #t)
(test (string=? (format #f "~{ ~a ~}" #()) "") #t)
(test (string=? (format #f "~{ ~a,~a ~}" '(a 1 b 2 c 3)) " a,1  b,2  c,3 ") #t)
(test (string=? (format #f "abc ~^ xyz") "abc ") #t)
(test (format (values #f "~A ~D" 1 2)) "1 2")
(test (format #f "~A~^" 1) "1") ; clisp agrees here
(test (format #f "~A~*~* ~A" (values 1 2 3 4)) "1 4")
(test (format #f "~^~A~^~*~*~^ ~^~A~^" (values 1 2 3 4)) "1 4")

(test (string=? (format #f "~B" 123) "1111011") #t)
(test (string=? (format #f "~B" 123/25) "1111011/11001") #t)
(test (string=? (format #f "~B" 123.25) "1111011.01") #t)
(test (string=? (format #f "~B" 123+i) "1111011.0+1.0i") #t)

(test (string=? (format #f "~D" 123) "123") #t)
(test (string=? (format #f "~D" 123/25) "123/25") #t)

(test (string=? (format #f "~O" 123) "173") #t)
(test (string=? (format #f "~O" 123/25) "173/31") #t)
(test (string=? (format #f "~O" 123.25) "173.2") #t)
(test (string=? (format #f "~O" 123+i) "173.0+1.0i") #t)

(test (string=? (format #f "~X" 123) "7b") #t)
(test (string=? (format #f "~X" 123/25) "7b/19") #t)
(test (string=? (format #f "~X" 123.25) "7b.4") #t)
(test (string=? (format #f "~X" 123+i) "7b.0+1.0i") #t)

(test (string=? (format #f "~A" "hi") (format #f "~S" "hi")) #f)
(test (string=? (format #f "~A" #\a) (format #f "~S" #\a)) #f)
(for-each
 (lambda (arg)
   (test (string=? (format #f "~A" arg) (format #f "~S" arg)) #t))
 (list 1 1.0 #(1 2 3) '(1 2 3) '(1 . 2) () #f #t abs #<eof> #<unspecified> 'hi '\a))
(test (length (format #f "~S" (string #\\))) 4)                  ; "\"\\\\\""
(test (length (format #f "~S" (string #\a))) 3)                  ; "\"a\""
(test (length (format #f "~S" (string #\null))) 6)               ; "\"\\x00\""
(test (length (format #f "~S" (string (integer->char #xf0)))) 3) ; "\"ð\""
(test (length (format #f "~S" (string #\"))) 4)                  ; "\""

(test (format #f "~F" 3.0) "3.000000")
(test (format #f "~G" 3.0) "3.0")
(test (format #f "~E" 3.0) (if (not with-windows) "3.000000e+00" "3.000000e+000"))
(test (format #f "~F" 3.14159) "3.141590")
(test (format #f "~G" 3.14159) "3.14159")
(test (format #f "~E" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~,2F" 3.14159) "3.14")
(test (format #f "~,2G" 3.14159) "3.1")
(test (format #f "~,2E" 3.14159) (if (not with-windows) "3.14e+00" "3.14e+000"))
(test (format #f "~12F" 3.14159) "    3.141590")
(test (format #f "~12G" 3.14159) "     3.14159")
(test (format #f "~12E" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~12,3F" 3.14159) "       3.142")
(test (format #f "~n,nF" 12 3 3.14159) "       3.142")
(test (format #f "~12,nF" 3 3.14159) "       3.142")
(test (format #f "~12,3G" 3.14159) "        3.14")
(test (format #f "~12,3E" 3.14159) (if (not with-windows) "   3.142e+00" "  3.142e+000"))
(test (format #f "~12,'xD" 1) "xxxxxxxxxxx1")
(test (format #f "~12,'xF" 3.14159) "xxxx3.141590")
(test (format #f "~12,'xG" 3.14159) "xxxxx3.14159")
(test (format #f "~12,'xE" 3.14159) (if (not with-windows) "3.141590e+00" "3.141590e+000"))
(test (format #f "~12,'\\F" 3.14159) "\\\\\\\\3.141590")
(test (format #f "~20,20G" 3.0) "                   3.0")
(test (format #f "~20,20F" 3.0) "3.00000000000000000000")
(test (format #f "~20,20E" 3.0) (if (not with-windows) "3.00000000000000000000e+00" "3.00000000000000000000e+000"))

(test (format #f "~,3B" 0.99999) "0.111")
(test (format #f "~,3O" 0.99999) "0.777")
(test (format #f "~,3F" 0.99999) "1.000")
(test (format #f "~,3X" 0.99999) "0.fff")

(test (format #f "~-2F" 0.0) 'error)
(test (format #f "~,-2F" 0.0) 'error)
(test (format #f "~2/3F" 0.0) 'error)
(test (format #f "~2.3F" 0.0) 'error)
(test (format #f "~2,1,3,4F" 0.0) 'error)
(test (format #f "~'xF" 0.0) 'error)
(test (format #f "~3,3" pi) 'error)
(test (format #f "~3," pi) 'error)
(test (format #f "~3" pi) 'error)
(test (format #f "~," pi) 'error)
(test (format #f "~'," pi) 'error)
(test (format #f "~'" pi) 'error)

(test (format #f "~*" 1.0) "")
(test (format #f "~D" 1.0) (if (not with-windows) "1.000000e+00" "1.000000e+000"))
(test (format #f "~O" 1.0) "1.0")
(test (format #f "~P" 1.0) "")
(test (format #f "~P" '(1 2 3)) 'error)
(test (format #f "~\x00T") 'error)
(test (format #f "~9,'(T") "((((((((")
(test (format #f "~0F" 1+1i) "1.000000+1.000000i")
(test (format #f "~9F" 1) "        1")
(test (format #f "~,0F" 3.14) "3.0")
(test (format #f "~,0F" 1+1i) "1+1i")
(test (format #f "~,0X" 1+1i) "1.0+1.0i")
(test (format #f "~,9g" 1+1i) "1+1i")
(test (format #f "~,1e" 3.14) (if (not with-windows) "3.1e+00" "3.1e+000"))
(test (format #f "~9,0F" 3.14) "        3.0")
(test (format #f "~9,1F" 3.14) "      3.1")
(test (format #f "~9,2F" 3.14) "     3.14")
(test (format #f "~9,3F" 3.14) "    3.140")
(test (format #f "~9,4F" 3.14) "   3.1400")
(test (format #f "~n,4F" 9 3.14) "   3.1400")
(test (format #f "~9,nF" 4 3.14) "   3.1400")
(test (format #f "~n,nF" 9 4 3.14) "   3.1400")
(test (format #f "~9,5F" 3.14) "  3.14000")
(test (format #f "~9,6F" 3.14) " 3.140000")
(test (format #f "~9,7F" 3.14) "3.1400000")
(test (format #f "~9,8F" 3.14) "3.14000000")
(test (format #f "~9,9F" 3.14) "3.140000000")
(test (format #f "~9,9G" 1+1i) "     1+1i")
(if (not with-windows)
    (begin
      (test (format #f "~9,0e" 1+1i) "1e+00+1e+00i")
      (test (format #f "~9,1e" 1+1i) "1.0e+00+1.0e+00i")
      (test (format #f "~9,2e" 1+1i) "1.00e+00+1.00e+00i")
      (test (format #f "~9,3e" 1+1i) "1.000e+00+1.000e+00i")
      (test (format #f "~9,4e" 1+1i) "1.0000e+00+1.0000e+00i")
      (test (format #f "~9,5e" 1+1i) "1.00000e+00+1.00000e+00i")
      (test (format #f "~9,6e" 1+1i) "1.000000e+00+1.000000e+00i")
      (test (format #f "~9,7e" 1+1i) "1.0000000e+00+1.0000000e+00i")
      (test (format #f "~9,8e" 1+1i) "1.00000000e+00+1.00000000e+00i")
      (test (format #f "~9,9e" 1+1i) "1.000000000e+00+1.000000000e+00i"))
    (begin
      (test (format #f "~9,0e" 1+1i) "1e+000+1e+000i")
      (test (format #f "~9,1e" 1+1i) "1.0e+000+1.0e+000i")
      (test (format #f "~9,2e" 1+1i) "1.00e+000+1.00e+000i")
      (test (format #f "~9,3e" 1+1i) "1.000e+000+1.000e+000i")
      (test (format #f "~9,4e" 1+1i) "1.0000e+000+1.0000e+000i")
      (test (format #f "~9,5e" 1+1i) "1.00000e+000+1.00000e+000i")
      (test (format #f "~9,6e" 1+1i) "1.000000e+000+1.000000e+000i")
      (test (format #f "~9,7e" 1+1i) "1.0000000e+000+1.0000000e+000i")
      (test (format #f "~9,8e" 1+1i) "1.00000000e+000+1.00000000e+000i")
      (test (format #f "~9,9e" 1+1i) "1.000000000e+000+1.000000000e+000i")))
(test (format #f "~9,0x" 3.14) "      3.0")
(test (format #f "~9,1x" 3.14) "      3.2")
(test (format #f "~9,2x" 3.14) "     3.23")
(test (format #f "~9,3x" 3.14) "    3.23d")
(test (format #f "~9,4x" 3.14) "   3.23d7")
(test (format #f "~9,5x" 3.14) "   3.23d7")
(test (format #f "~9,6x" 3.14) " 3.23d70a")
(test (format #f "~9,7x" 3.14) "3.23d70a3")
(test (format #f "~9,8x" 3.14) "3.23d70a3d")
(test (format #f "~9,9x" 3.14) "3.23d70a3d7")
(test (format #f "~9,0b" 3.14) "     11.0")
(test (format #f "~9,1b" 3.14) "     11.0")
(test (format #f "~9,2b" 3.14) "     11.0")
(test (format #f "~9,3b" 3.14) "   11.001")
(test (format #f "~9,4b" 3.14) "   11.001")
(test (format #f "~9,5b" 3.14) "   11.001")
(test (format #f "~9,6b" 3.14) "   11.001")
(test (format #f "~9,7b" 3.14) "11.0010001")
(test (format #f "~9,8b" 3.14) "11.00100011")
(test (format #f "~9,9b" 3.14) "11.001000111")
(test (format #f "~0,'xf" 1) "1")
(test (format #f "~1,'xf" 1) "1")
(test (format #f "~2,'xf" 1) "x1")
(test (format #f "~3,'xf" 1) "xx1")
(test (format #f "~4,'xf" 1) "xxx1")
(test (format #f "~5,'xf" 1) "xxxx1")
(test (format #f "~6,'xf" 1) "xxxxx1")
(test (format #f "~7,'xf" 1) "xxxxxx1")
(test (format #f "~8,'xf" 1) "xxxxxxx1")
(test (format #f "~9,'xf" 1) "xxxxxxxx1")
(test (format #f "~11,'xf" 3.14) "xxx3.140000")
(test (format #f "~12,'xf" 3.14) "xxxx3.140000")
(test (format #f "~13,'xf" 3.14) "xxxxx3.140000")
(test (format #f "~14,'xf" 3.14) "xxxxxx3.140000")
(test (format #f "~15,'xf" 3.14) "xxxxxxx3.140000")
(test (format #f "~16,'xf" 3.14) "xxxxxxxx3.140000")
(test (format #f "~17,'xf" 3.14) "xxxxxxxxx3.140000")
(test (format #f "~18,'xf" 3.14) "xxxxxxxxxx3.140000")
(test (format #f "~19,'xf" 3.14) "xxxxxxxxxxx3.140000")
(test (format #f "~,f" 1.0) "1.000000")
(test (format #f "~,,f" 1.0) 'error)
(test (format #f "~p" '(1 2 3)) 'error) ; these are not errors in CL
(test (format #f "~p" #(())) 'error)
(test (format #f "~p" 'hi) 'error)
(test (format #f "~p" abs) 'error)
(test (format #f "~p" 1+i) 'error)
(test (format #f "~@p" '(1 2 3)) 'error)
(test (format #f "~@p" #(())) 'error)
(test (format #f "~@p" 'hi) 'error)
(test (format #f "~@p" abs) 'error)
(let-temporarily (((*s7* 'print-length) 3))
  (test (format #f "~{~A~| ~}" '(1 2 3 4 5 6)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" #(1 2 3 4 5 6)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" #(1 2)) "1 2")
  (test (format #f "~{~A~| ~}" #(1 2 3)) "1 2 3")
  (test (format #f "~{~A~| ~}" #(1 2 3 4)) "1 2 3 ...")
  (test (format #f "~{~A~| ~}" (inlet 'a 1 'b 2 'c 3 'd 4 'e 5)) "(a . 1) (b . 2) (c . 3) ...")
  (test (format #f "~{~{~A~| ~}~}" '((1 2 3 4 5 6))) "1 2 3 ...")
  (test (format #f "~{~{~A~| ~}~|~}" '((1 2) (3 4 5 6 7 8) (15) (16) ())) "1 23 4 5 ...15 ...")
  (test (format #f "~{~|~|~|~A ~}" '(1 2 3 4 5)) "1 2 3  ...")
  (test (format #f "~{~C~| ~}" "1234567") "1 2 3 ...")
  (test (format #f "~{~{~A~|~} ~}" '((1 2) (3 4))) "12 34 ")
  (test (format #f "~C ~^" #\a) "a ")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" '(((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" '((#(1 2) #(3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" #(((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~{~A~| ~}~| ~}~}" #(#((1 2) (3 4)))) "1 2 3 4")
  (test (format #f "~{~{~C~| ~}~| ~}" (list "hiho" "xxx")) "h i h ... x x x"))
(test (format #f "~{~{~A~^~} ~}" '((hi 1))) "hi1 ")
(test (format #f "~{~{~A~^~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '((1 2) (3 4))) "12 34 ")
(test (format #f "~{~{~A~} ~}" '(())) " ")
(test (format #f "~{~{~A~} ~}" '((()))) "() ")
(test (format #f "~{~{~F~} ~}" '(())) " ")
(test (format #f "~{~{~C~} ~}" '(())) " ")
(test (format #f "~{~C ~}" ()) "")
(test (format #f "~C ~^" #\a) "a ") ; CL ignores pointless ~^
(test (format #f "~^~A" #f) "#f")
(test (format #f "~^~^~A" #f) "#f")
(test (format #f "~*~*~A~*" 1 2 3 4) "3")
(test (format #f "~{~*~A~}" '(1 2 3 4)) "24")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~A" lst)) "#1=(1 2 3 . #1#)")
(test (let ((lst (list 1 2 3))) (set! (cdr (cddr lst)) lst) (format #f "~{~A~}" lst)) 'error)
(test (format #f "~{~A~}" (cons 1 2)) 'error)
(test (format #f "~{~A~}" '(1 2 3 . 4)) 'error)
(test (format #f "~20,vF" 3.14) 'error)
(test (format #f "~{~C~^ ~}" "hiho") "h i h o")
(test (format #f "~{~{~C~^ ~}~}" (list "hiho")) "h i h o")
(test (format #f "~{~A ~}" #(1 2 3 4)) "1 2 3 4 ")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~A" v)) "#1=#(#1#)")
(test (let ((v (vector 1))) (set! (v 0) v) (format #f "~{~A~}" v)) "#1=#(#1#)")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" '((#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#((1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) (3 4)))) "1 2 3 4")
(test (format #f "~{~{~{~A~^ ~}~^ ~}~}" #(#(#(1 2) #(3 4)))) "1 2 3 4")
(test (format #f "~{~{~C~^ ~}~^ ~}" (list "hiho" "xxx")) "h i h o x x x")
(test (format #f "~{~{~A~}~}" '((1 . 2) (3 . 4))) 'error)
(test (format #f "~{~A~^ ~}" '((1 . 2) (3 . 4))) "(1 . 2) (3 . 4)") 
(test (format #f "~{~A ~}" (hash-table)) "")
(test (format #f "~{~^~S ~}" (make-iterator '(1 2 3))) "1 2 3 ")
(test (format #f "~{~^~S ~}" (make-iterator (let ((lst (list 1))) (set-cdr! lst lst)))) "1 ")
(test (format #f "~{~^~S ~}" (make-iterator "")) "")
(test (format #f "~{~^~S ~}" (make-iterator #(1 2 3))) "1 2 3 ")

(test (format #f "~10,'-T") "---------")
(test (format #f "~10,'\\T") "\\\\\\\\\\\\\\\\\\")
(test (format #f "~10,'\"T") "\"\"\"\"\"\"\"\"\"")
(test (format #f "~10,'-T12345~20,'-T") "---------12345-----")
(test (format #f "~10,')T") ")))))))))")

(test (format #f "~,0F" 1.4) "1.0")
(test (format #f "~,0F" 1.5) "2.0")
(test (format #f "~,0F" 1.6) "2.0")
(test (format #f "~,0F" 0.4) "0.0")
(test (format #f "~,0F" 0.5) (if (not with-windows) "0.0" "1.0")) ; !!
(test (format #f "~,0F" 0.6) "1.0")
(test (format #f "~,-0F" 1.4) 'error)
(test (format #f "~, 0F" 1.4) 'error)
(test (format #f "~*1~*" 1) 'error)
(test (format #f "~*1~A" 1) 'error)

(let* ((str1 #t) (str2 (with-output-to-string (lambda () (set! str1 (format () "~D" 1)))))) (test (and (not str1) (equal? str2 "1")) #t))

(test (format #f "~,'") 'error)
(if with-bignums
    (begin
      (test (format #f "~F" 1e300) "9.999999999999999999999999999999999999987E299")
      (test (format #f "~F" 1e308) "9.999999999999999999999999999999999999982E307")
      (test (format #f "~G" 1e308) "9.999999999999999999999999999999999999982E307")
      (test (format #f "~E" 1e308)  "9.999999999999999999999999999999999999982E307")
      (test (format #f "~E" 1e308+1e308i) "9.999999999999999999999999999999999999982E307+9.999999999999999999999999999999999999982E307i")
      (test  (format #f "~F" 1e308+1e308i) "9.999999999999999999999999999999999999982E307+9.999999999999999999999999999999999999982E307i")
      (test (format #f "~F" -1e308-1e308i) "-9.999999999999999999999999999999999999982E307-9.999999999999999999999999999999999999982E307i")
      (test (format #f "~,32f" (/ 1.0 most-positive-fixnum)) "1.084202172485504434125002235952170462235E-19")
      (test (format #f "~{~^~f ~}" (vector 1e308)) "9.999999999999999999999999999999999999982E307 ")
      (test (object->string (vector 1e308)) "#(9.999999999999999999999999999999999999982E307)"))
    (begin
      (test (format #f "~F" 1e300) "1000000000000000052504760255204420248704468581108159154915854115511802457988908195786371375080447864043704443832883878176942523235360430575644792184786706982848387200926575803737830233794788090059368953234970799945081119038967640880074652742780142494579258788820056842838115669472196386865459400540160.000000")
      (test (format #f "~F" 1e308) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000")
      (test (format #f "~G" 1e308) "1e+308")
      (test (format #f "~E" 1e308) "1.000000e+308")
      (test (format #f "~E" 1e308+1e308i) "1.000000e+308+1.000000e+308i")
      (test  (format #f "~F" 1e308+1e308i) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000+100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000i")
      (test (format #f "~F" -1e308-1e308i) "-100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000-100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000i")
      (test (format #f "~,32f" (/ 1.0 most-positive-fixnum)) "0.00000000000000000010842021724855")
      (test (format #f "~{~^~f ~}" (vector 1e308)) "100000000000000001097906362944045541740492309677311846336810682903157585404911491537163328978494688899061249669721172515611590283743140088328307009198146046031271664502933027185697489699588559043338384466165001178426897626212945177628091195786707458122783970171784415105291802893207873272974885715430223118336.000000 ")
      (test (object->string (vector 1e308)) "#(1e+308)")))


(when full-test
  (let ()
    (define ctrl-chars (vector ;#\A #\S #\C #\F #\E #\G #\O #\D #\B #\X #\W
			#\, #\{ #\} #\@ #\P #\* #\< #\>
			#\a #\s #\c #\f #\e #\g #\o #\d #\b #\x #\p #\n #\w
			#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9
			#\~ #\T #\& #\% #\^ #\|
			#\~ #\~ #\~ #\~ 
			#\, #\, #\, #\, #\" #\" #\\ #\'
			#\+ #\- #\@ #\. #\/ #\; #\:
			))
    (define ctrl-chars-len (length ctrl-chars))
    
    (define (test-chars)
      (do ((size 1 (+ size 1)))
	  ((= size 7))
	(let ((tries (* size size 10000)))
	  (format *stderr* "~D " size)
	  (let ((ctrl-str (make-string (+ size 1)))
		(x 12)
		(y '(1 2))
		(z #\a))
	    (string-set! ctrl-str 0 #\~)
	    (do ((i 0 (+ i 1)))
		((= i tries))
	      (do ((j 1 (+ j 1)))
		  ((> j size))
		(string-set! ctrl-str j (vector-ref ctrl-chars (random ctrl-chars-len))))
					;(format *stderr* "~S " ctrl-str)
					;(catch #t (lambda () (format *stderr* "~S: ~A~%" ctrl-str (format #f ctrl-str))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str x (format #f ctrl-str x))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str y (format #f ctrl-str y))) (lambda arg 'error))
					;(catch #t (lambda () (format *stderr* "~S ~A: ~A~%" ctrl-str z (format #f ctrl-str z))) (lambda arg 'error)))))
	      (catch #t (lambda () (format #f ctrl-str)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str y)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str z)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x x)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x y)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str y z)) (lambda arg 'error))
	      (catch #t (lambda () (format #f ctrl-str x y z)) (lambda arg 'error)))))
	))
    (test-chars)))


(test (reverse (format #f "~{~A~}" '((1 2) (3 4)))) ")4 3()2 1(")
(test (string->symbol (format #f "~A" '(1 2))) (symbol "(1 2)"))

(test (string->number (format #f "~A" -1)) -1)
(test (string->number (format #f "~S" -1)) -1)
(test (string->number (format #f "~F" -1)) -1)
(test (string->number (format #f "~D" -1)) -1)
(test (string->number (format #f "~G" -1)) -1)
(test (string->number (format #f "~E" -1)) -1)
(test (string->number (format #f "~B" -1)) -1)
(test (string->number (format #f "~X" -1)) -1)
(test (string->number (format #f "~O" -1)) -1)
(num-test (string->number (format #f "~A" 1.5)) 1.5)
(num-test (string->number (format #f "~S" 1.5)) 1.5)
(num-test (string->number (format #f "~F" 1.5)) 1.5)
(num-test (string->number (format #f "~D" 1.5)) 1.5)
(num-test (string->number (format #f "~G" 1.5)) 1.5)
(num-test (string->number (format #f "~E" 1.5)) 1.5)
(num-test (string->number (format #f "~B" 1.5)) 1.1)
(num-test (string->number (format #f "~X" 1.5)) 1.8)
(num-test (string->number (format #f "~O" 1.5)) 1.4)
(num-test (string->number (format #f "~A" 1+1i)) 1+1i)
(num-test (string->number (format #f "~S" 1+1i)) 1+1i)
(num-test (string->number (format #f "~F" 1+1i)) 1+1i)
(num-test (string->number (format #f "~D" 1+1i)) 1+1i)
(num-test (string->number (format #f "~G" 1+1i)) 1+1i)
(num-test (string->number (format #f "~E" 1+1i)) 1+1i)
(num-test (string->number (format #f "~B" 1+1i)) 1+1i)
(num-test (string->number (format #f "~X" 1+1i)) 1+1i)
(num-test (string->number (format #f "~O" 1+1i)) 1+1i)
(test (string->number (format #f "~A" 3/4)) 3/4)
(test (string->number (format #f "~S" 3/4)) 3/4)
(test (string->number (format #f "~F" 3/4)) 3/4)
(test (string->number (format #f "~D" 3/4)) 3/4)
(test (string->number (format #f "~G" 3/4)) 3/4)
(test (string->number (format #f "~E" 3/4)) 3/4)
(test (string->number (format #f "~B" 3/4)) 11/100)
(test (string->number (format #f "~X" 3/4)) 3/4)
(test (string->number (format #f "~O" 3/4)) 3/4)
(num-test (string->number (format #f "~A" 0+1i)) 0+1i)
(num-test (string->number (format #f "~S" 0+1i)) 0+1i)
(num-test (string->number (format #f "~F" 0+1i)) 0+1i)
(num-test (string->number (format #f "~D" 0+1i)) 0+1i)
(num-test (string->number (format #f "~G" 0+1i)) 0+1i)
(num-test (string->number (format #f "~E" 0+1i)) 0+1i)
(num-test (string->number (format #f "~B" 0+1i)) 0+1i)
(num-test (string->number (format #f "~X" 0+1i)) 0+1i)
(num-test (string->number (format #f "~O" 0+1i)) 0+1i)

(test (format "~G" 1e10) (if (not with-windows) "1e+10" "1e+010"))
(test (format "~F" 1e10) "10000000000.000000")
(test (format "~E" 1e10) (if (not with-windows) "1.000000e+10" "1.000000e+010"))
(test (format "~A" 1e10) "10000000000.0")
(test (format "~D" 1e10) (if (not with-windows) "1.000000e+10" "1.000000e+010"))

(test (format #f "~P{T}'" 1) "{T}'")
(test (format #f "~") 'error)
(test (format #f "~B&B~X" 1.5 1.5) "1.1&B1.8")
(test (format #f ",~~~A~*1" 1 1) ",~11")
(test (format #f "~D~20B" 0 0) "0                   0")
(test (format #f "~D~20B" 1 1) "1                   1")
(test (format #f "~10B" 1) "         1")
(test (format #f "~10B" 0) "         0")
(test (format #f "~100B" 1) "                                                                                                   1")
(test (length (format #f "~1000B" 1)) 1000)
(test (format #f "~D~20D" 3/4 3/4) "3/4                 3/4")
(test (length (format #f "~20D" 3/4)) 20)
(test (format #f "~20B" 3/4) "              11/100")
(test (length (format #f "~20B" 3/4)) 20)
(test (format #f "~D~20B" 3/4 3/4) "3/4              11/100")
(test (format #f "~X~20X" 21/33 21/33) "7/b                 7/b")
(test (format #f "~D~20,'.B" 3/4 3/4) "3/4..............11/100")
(test (format #f "~20g" 1+i) "                1+1i")
(test (length (format #f "~20g" 1+i)) 20)
(test (format #f "~20f" 1+i) "  1.000000+1.000000i")
(test (length (format #f "~20f" 1+i)) 20)
(test (format #f "~20x" 17+23i) "          11.0+17.0i")
(test (length (format #f "~20x" 17+23i)) 20)

(test (format #f "~{~{~A~^~} ~}" (hash-table '((a . 1) (b . 2)))) "(a . 1)(b . 2) ")
(test (format #f "~{~{~A~^~}~^~}" (hash-table '((a . 1) (b . 2)))) "(a . 1)(b . 2)")
(test (format #f "~{~{~A~^ ~}~^~}" (hash-table '((a . 1) (b . 2)))) "(a . 1) (b . 2)")
(test (format #f "~{~{~{~A~^~} ~}~}" #(())) "")
(test (format #f "~{~{~{~P~^~} ~}~}" '((()))) " ")
(test (format #f "~{~{~{~P~^~}~}~}" '(((2 3 4)))) "sss")
(test (apply format #f "~T~~{~{~{~*~~0~1~*~}~@~}" '(())) "~{")
(test (format #f "~{~S}%~}" '(a b c)) "a}%b}%c}%")
(test (format #f "~&~^%~F." 0) "%0.")
(test (format #f "1~^2") "1")
(test (apply format #f "~P~d~B~~" '(1 2 3)) "211~")
(test (format #f "~T1~~^~P" 0) "1~^s")
(test (format #f "~S~^~{~^" '(+ x 1)) "(+ x 1)")
(test (format #f "1~^~{2") "1")
(test (format #f "~A~{~0~g~@~B~}" () ()) "()")
(test (format #f "1~^~^~^2") "1")
(test (format #f "~{~{~~}~~,~}~*" '(()) '(())) "~,")
(test (format #f "~~S~S~T~~C~g~~" 0 0) "~S0~C0~")
(test (format #f "~{~~e~}~~{~*~~" "" "") "~{~")

(let ()
  (define* (clean-string e (precision 3))
    (format #f (format #f "(~~{~~,~DF~~^ ~~})" precision) e))
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 1) "(1.1 -2.3 3.1 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3)) "(1.123 -2.312 3.142 4/3)")
  (test (clean-string '(1.123123 -2.31231323 3.141592653589 4/3) 6) "(1.123123 -2.312313 3.141593 4/3)"))

(when with-bignums
  (test (format #f "~P" (bignum "1")) "")
  (test (format #f "~P" (bignum "1.0")) "")
  (test (format #f "~P" (bignum "2")) "s")
  (test (format #f "~P" (bignum "2.0")) "s")
  (test (format #f "~10,' D" (bignum "1")) "         1")
  (test (format #f "~10,' D" (bignum "3/4")) "       3/4")
  (test (format #f "~10,'.D" (bignum "3/4")) ".......3/4")
  (test (format #f "~10D" (bignum "3/4")) "       3/4")
  (test (length (format #f "~100D" (bignum "34"))) 100)
  (test (format #f "~50F" (bignum "12345678.7654321")) "                                1.23456787654321E7"))


(call-with-output-file tmp-output-file (lambda (p) (format p "this ~A ~C test ~D" "is" #\a 3)))
(let ((res (call-with-input-file tmp-output-file (lambda (p) (read-line p)))))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "call-with-input-file + format to \"tmp1.r5rs\" ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(let ((val (format #f "line 1~%line 2~%line 3")))
  (with-input-from-string val
    (lambda ()
      (let ((line1 (read-line)))
	(test (string=? line1 "line 1") #t))
      (let ((line2 (read-line)))
	(test (string=? line2 "line 2") #t))
      (let ((line3 (read-line)))
	(test (string=? line3 "line 3") #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t))
      (let ((eof (read-line)))
	(test (eof-object? eof) #t)))))

(test (display 3 #f) 3)
(test (write 3 #f) 3)

(let ((val (format #f "line 1~%line 2~%line 3")))
  (call-with-input-string val
			  (lambda (p)
			    (let ((line1 (read-line p #t)))
			      (test (string=? line1 (string-append "line 1" (string #\newline))) #t))
			    (let ((line2 (read-line p #t)))
			      (test (string=? line2 (string-append "line 2" (string #\newline))) #t))
			    (let ((line3 (read-line p #t)))
			      (test (string=? line3 "line 3") #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t))
			    (let ((eof (read-line p #t)))
			      (test (eof-object? eof) #t)))))

(let ((res #f)) 
  (let ((this-file (open-output-string))) 
    (format this-file "this ~A ~C test ~D" "is" #\a 3)
    (set! res (get-output-string this-file))
    (close-output-port this-file))
  (if (not (string=? res "this is a test 3"))
      (begin 
	(display "open-output-string + format ... expected \"this is a test 3\", but got \"")
	(display res) (display "\"?") (newline))))

(test (with-output-to-string (lambda () (display 123) (flush-output-port))) "123")
(test (with-output-to-string (lambda () (display 123) (flush-output-port) (display 124))) "123124")

(test (call-with-output-string
       (lambda (p)
	 (write 1 p)
	 (display 2  p)
	 (format p "~D" 3)
	 (write-byte (char->integer #\4) p)
	 (write-char #\5 p)
	 (write-string "6" p)
	 (write 1 #f)
	 (display 2 #f)
	 (format #f "~D" 3)
	 (write-byte (char->integer #\4) #f)
	 (write-char #\5 #f)
	 (write-string "6" #f)))
      "123456")

(test (write-byte most-positive-fixnum #f) 'error)
(test (write-byte -1 #f) 'error)
(test (write-byte 256 #f) 'error)

(let ((res #f)) 
  (let ((this-file (open-output-string))) 
    (format this-file "this is a test")
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a test"))
	(format-logged #t "open-output-string + format expected \"this is a test\", but got ~S~%" res))
    (flush-output-port this-file)
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a test"))
	(format-logged #t "flush-output-port of string port expected \"this is a test\", but got ~S~%" res))
    (format this-file "this is a test")
    (set! res (get-output-string this-file))
    (if (not (string=? res "this is a testthis is a test"))
	(format-logged #t "open-output-string after flush expected \"this is a testthis is a test\", but got ~S~%" res))
    (close-output-port this-file)
    (test (flush-output-port this-file) this-file)))

(test (flush-output-port "hiho") 'error)
(test (flush-output-port *stdin*) 'error)

(call-with-output-file tmp-output-file
  (lambda (p)
    (format p "123456~%")
    (format p "67890~%")
    (flush-output-port p)
    (test (call-with-input-file tmp-output-file
	    (lambda (p)
	      (read-line p)))
	  "123456")
    (close-output-port p)))
    
(let ((res1 #f)
      (res2 #f)
      (res3 #f))
  (let ((p1 (open-output-string)))
    (format p1 "~D" 0)
    (let ((p2 (open-output-string)))
      (format p2 "~D" 1)
      (let ((p3 (open-output-string)))
	(if (not (string=? (get-output-string p1) "0"))
	    (format-logged #t ";format to nested ports, p1: ~S~%" (get-output-string p1)))	
	(if (not (string=? (get-output-string p2) "1"))
	    (format-logged #t ";format to nested ports, p2: ~S~%" (get-output-string p2)))	
	(format p3 "~D" 2)
	(format p2 "~D" 3)
	(format p1 "~D" 4)
	(format p3 "~D" 5)
	(set! res3 (get-output-string p3))
	(close-output-port p3)
	(if (not (string=? (get-output-string p1) "04"))
	    (format-logged #t ";format to nested ports after close, p1: ~S~%" (get-output-string p1)))	
	(if (not (string=? (get-output-string p2) "13"))
	    (format-logged #t ";format to nested ports after close, p2: ~S~%" (get-output-string p2))))
      (format (or p1 p3) "~D" 6)
      (format (and p1 p2) "~D" 7)
      (set! res1 (get-output-string p1))
      (close-output-port p1)
      (if (not (string=? (get-output-string p2) "137"))
	  (format-logged #t ";format to nested ports after 2nd close, p2: ~S~%" (get-output-string p2)))
      (format p2 "~D" 8)
      (set! res2 (get-output-string p2))
      (test (get-output-string p1) 'error)
      (test (get-output-string p2 "hi") 'error)
      (close-output-port p2)))
  (if (not (string=? res1 "046"))
      (format-logged #t ";format to nested ports, res1: ~S~%" res1))
  (if (not (string=? res2 "1378"))
      (format-logged #t ";format to nested ports, res2: ~S~%" res2))
  (if (not (string=? res3 "25"))
      (format-logged #t ";format to nested ports, res3: ~S~%" res3)))

(test (call/cc (lambda (return) 
		 (let ((val (format #f "line 1~%line 2~%line 3")))
		   (call-with-input-string val
					   (lambda (p) (return "oops"))))))
      "oops")

(test (get-output-string #f 64) 'error)

;(format-logged #t "format #t: ~D" 1)
;(format (current-output-port) " output-port: ~D! (this is testing output ports)~%" 2)

(call-with-output-file tmp-output-file
  (lambda (p)
    (display 1 p)
    (write 2 p)
    (write-char #\3 p)
    (format p "~D" 4)
    (write-byte (char->integer #\5) p)
    (call-with-output-file "tmp2.r5rs"
      (lambda (p)
	(display 6 p)
	(write 7 p)
	(write-char #\8 p)
	(format p "~D" 9)
	(write-byte (char->integer #\0) p)
	(newline p)))
    (call-with-input-file "tmp2.r5rs"
      (lambda (pin)
	(display (read-line pin) p)))
    (newline p)))

(test (call-with-input-file tmp-output-file
	(lambda (p)
	  (read-line p)))
      "1234567890")

(call-with-output-file tmp-output-file
  (lambda (p)
    (format p "12345~%")
    (format p "67890~%")))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-char p) #\1)
    (test (read-byte p) (char->integer #\2))
    (test (peek-char p) #\3)
    (if (not pure-s7) (test (char-ready? p) #t))
    (test (read-line p) "345")
    (test (read-line p) "67890")))

(call-with-output-file tmp-output-file
  (lambda (p)
    (write-string "123" p)
    (write-string "" p)
    (write-string "456\n789" p)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "123456")
    (test (read-char p) #\7)
    (test (read-char p) #\8)
    (test (read-char p) #\9)
    (test (eof-object? (read-char p)) #t)))
    
(test (with-output-to-string
	(lambda ()
	  (write-string "123")
	  (write-string "")
	  (write-string "456")))
      "123456")

(test (with-output-to-string
	(lambda ()
	  (write-string "123" (current-output-port))
	  (write-string "" (current-output-port))
	  (write-string "456" (current-output-port))
	  (write-string "678" (current-output-port) 1)
	  (write-string "679" (current-output-port) 2 3)
	  (write-string "079" (current-output-port) 0 1)
	  (write-string "123" (current-output-port) 0 3)
	  (write-string "123" (current-output-port) 3 3)
	  (write-string "" (current-output-port) 0 0)
	  (write-string "1423" (current-output-port) 1 1) ; 1.3.3: end is exclusive, if start=end, empty result
	  (write-string "1423" (current-output-port) 1 4/2)
	  (write-string "5423" (current-output-port) -0 1)))
      "123456789012345")

(test (write-string "12345" -1) 'error)
(test (write-string "12345" 0 -1) 'error)
(test (write-string "12345" 0 18) 'error)
(test (write-string "12345" 18) 'error)
(test (write-string "12345" 2 1) 'error)
(test (write-string "12345" 5 5) 'error)
(test (write-string "12345" 0.0 2) 'error)
(test (write-string "12345" 0 2.0) 'error)
(test (write-string "12345" 0 1+i) 'error)
(test (write-string "12345" 0 2/3) 'error)
(test (write-string "12345" 0 #\a) 'error)
(test (write-string "12345" #\null) 'error)
(test (write-string "12345" most-negative-fixnum) 'error)
(test (write-string "12345" 0 most-positive-fixnum) 'error)
(test (write-string "12345" 0 4294967296) 'error)
(test (write-string "a" #f 1) "")
(test (write-string "abc" #f 3) "")
(test (write-string "ab" #f 1) "b")
(test (write-string "ab" #f 2) "")
(test (write-string "abc" #f 1 2) "b")
(test (write-string "abc" #f 1 3) "bc")

(test (with-input-from-string "12345" (lambda () (read-string 3))) "123")
(test (with-input-from-string "" (lambda () (read-string 3))) #<eof>)
(test (with-input-from-string "" (lambda () (read-string 0))) "")
(test (with-input-from-string "1" (lambda () (read-string 0))) "")
(test (with-input-from-string "1" (lambda () (read-string -1))) 'error)
(test (with-input-from-string "1" (lambda () (read-string #f))) 'error)
(test (with-input-from-string "123" (lambda () (read-string 10))) "123")
(test (call-with-input-string "123" (lambda (p) (read-string 2 p))) "12")
(test (call-with-input-string "123" (lambda (p) (read-string 2 #f))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 2 (current-output-port)))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 0 #<unspecified>))) 'error)
(test (call-with-input-string "123" (lambda (p) (read-string 0 123))) 'error)

(test (read-string most-positive-fixnum) 'error)
(test (read-string -1) 'error)
(test (read-string most-negative-fixnum) 'error)
;(test (read-string 0) "")
; (test (read-string 123) "")
; s7 considers this file (during load) to be the current-input-file, so the above read-string ruins the load
; the other choice is to hang (waiting for stdin)
; perhaps this choice should be documented since it is specifically contrary to r7rs

(test (write 1 (current-input-port)) 'error)
(test (write-char #\a (current-input-port)) 'error)
(test (write-byte 0 (current-input-port)) 'error)
(test (read (current-output-port)) 'error)
(test (read-char (current-output-port)) 'error)
(test (read-byte (current-output-port)) 'error)
(test (read-line (current-output-port)) 'error)
(test (display 3) 3)
(test (display 3 #f) 3)

(when (not pure-s7)
  (let ((op1 (set-current-output-port (open-output-file tmp-output-file))))
    (display 1)
    (write 2)
    (write-char #\3)
    (format-logged #t "~D" 4) ; #t -> output port
    (write-byte (char->integer #\5))
    (let ((op2 (set-current-output-port (open-output-file "tmp2.r5rs"))))
      (display 6)
      (write 7)
      (write-char #\8)
      (format-logged #t "~D" 9)
      (write-byte (char->integer #\0))
      (newline)
      (close-output-port (current-output-port))
      (set-current-output-port op2)
      (let ((ip1 (set-current-input-port (open-input-file "tmp2.r5rs"))))
	(display (read-line))
	(close-input-port (current-input-port))
	(set-current-input-port ip1))
      (newline)
      (close-output-port (current-output-port)))

    (set-current-output-port #f)
    (test (string? (format #t "~%")) #t)
    (write "write: should not appear" #f) (newline #f)
    (display "display: should not appear" #f) (newline #f)
    (format #f "format: should not appear") (newline #f)
    (write-string "write-string: should not appear" #f) (newline #f)
    (write-char #\! #f)
    (write-byte 123 #f)

    (write "write: should not appear" (current-output-port)) (newline (current-output-port))
    (display "display: should not appear" (current-output-port)) (newline (current-output-port))
    (format (current-output-port) "format: should not appear") (newline (current-output-port))
    (write-string "write-string: should not appear" (current-output-port)) (newline (current-output-port))
    (write-char #\! (current-output-port))
    (write-byte 123 (current-output-port))

    (write "write: should not appear") (newline)
    (display "display: should not appear") (newline)
    (format #t "format: should not appear") (newline)
    (write-string "write-string: should not appear") (newline)
    (write-char #\!)
    (write-byte 123)

    (set-current-output-port op1))

  (let ((op1 (open-output-file tmp-output-file)))
    (let-temporarily (((current-output-port) op1))
      (display 1)
      (write 2)
      (write-char #\3)
      (format-logged #t "~D" 4) ; #t -> output port
      (write-byte (char->integer #\5))
      (let ((op2 (open-output-file "tmp2.r5rs")))
	(let-temporarily (((current-output-port) op2))
	  (display 6)
	  (write 7)
	  (write-char #\8)
	  (format-logged #t "~D" 9)
	  (write-byte (char->integer #\0))
	  (newline)
	  (close-output-port (current-output-port)))
	(let ((ip1 (open-input-file "tmp2.r5rs")))
	  (let-temporarily (((current-input-port) ip1))
	    (display (read-line))
	    (close-input-port (current-input-port))))
	(newline)
	(close-output-port (current-output-port)))))

  (test (call-with-input-file tmp-output-file
	  (lambda (p)
	    (read-line p)))
	"1234567890"))

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg display) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file))

(for-each 
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #f #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (for-each 
    (lambda (arg)
      (test (op "hi" arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list write display write-byte newline write-char 
       read read-char read-byte peek-char char-ready? read-line))

(for-each 
 (lambda (arg)
   (test (write-char arg) 'error)
   (test (write-byte arg) 'error)
   (test (read-char arg) 'error)
   (test (read-byte arg) 'error)
   (test (peek-char arg) 'error)
   (test (write-char #\a arg) 'error)
   (test (write-byte 1 arg) 'error))
 (list "hi" 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (write-byte -1) 'error)
(test (write-byte most-positive-fixnum) 'error)
(test (write-byte 300) 'error)

(for-each 
 (lambda (arg)
   (test (write-string arg) 'error)
   (test (write-string "hi" arg) 'error))
 (list 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs #t :hi (if #f #f) (lambda (a) (+ a 1))))

(test (with-output-to-string (lambda () (newline #f))) "")
(test (with-output-to-string (lambda () (write-byte 95 #f))) "")
(test (with-output-to-string (lambda () (write-char #\a #f))) "")
(test (with-output-to-string (lambda () (write-string "a" #f))) "")
(test (with-output-to-string (lambda () (write "hiho" #f))) "")
(test (with-output-to-string (lambda () (display "hiho" #f))) "")
(test (with-output-to-string (lambda () (format #f "hiho"))) "")

(when (not pure-s7)
  (test (with-output-to-string
	  (lambda ()
	    (set! (current-output-port) #f)
	    (newline (current-output-port))
	    (write-byte 95 (current-output-port))
	    (write-char #\a (current-output-port))
	    (write-string "a" (current-output-port))
	    (write "hiho" (current-output-port))
	    (display "hiho" (current-output-port))
	    (format (current-output-port) "hiho")))
	"")
  (set! (current-output-port) *stdout*))

(for-each
 (lambda (op)
   (for-each
    (lambda (arg)
      (test (op arg) 'error))
    (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs :hi (if #f #f) (lambda (a) (+ a 1)))))
 (list set-current-input-port set-current-error-port set-current-output-port close-input-port close-output-port))

(let ((hi (open-output-string)))
  (test (get-output-string hi) "")
  (close-output-port hi)
  (test (get-output-string hi) 'error))

(test (open-output-string "hiho") 'error)
(test (with-output-to-string "hi") 'error)
(test (call-with-output-string "hi") 'error)

(test (get-output-string 1 2) 'error)
(test (get-output-string) 'error)
(for-each 
 (lambda (arg)
   (test (get-output-string arg) 'error))
 (list "hi" 1 1.0 1+i 2/3 'a-symbol (make-vector 3) '(1 2) (cons 1 2) abs :hi (if #f #f) (lambda (a) (+ a 1))))

(let ((p (open-output-string)))
  (write 123 p)
  (test (get-output-string p) "123")
  (write 124 p)
  (test (get-output-string p #t) "123124")
  (test (get-output-string p #t) "")
  (write 123 p)
  (test (get-output-string p) "123"))

;; since read of closed port will generate garbage, it needs to be an error,
;;   so I guess write of closed port should also be an error

(let ((hi (open-output-string)))
  (close-output-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 (if pure-s7 newline set-current-output-port)
	 (if pure-s7 newline set-current-input-port)
	 set-current-error-port
	 newline)))

(let ((hi (open-input-string "hiho")))
  (test (get-output-string hi) 'error)
  (close-input-port hi)
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'closed-port))
   (list read read-char read-byte peek-char read-line 
	 port-filename port-line-number
	 (if pure-s7 read-line char-ready?)
	 (if pure-s7 read-line set-current-output-port)
	 (if pure-s7 read-line set-current-input-port)
	 set-current-error-port
	 )))
  
(test (close-output-port (open-input-string "hiho")) 'error)
(test (close-input-port (open-output-string)) 'error)
(test (set! (port-filename) "hiho") 'error)
(test (set! (port-closed? (current-output-port)) "hiho") 'error)
(test (begin (close-output-port *stderr*) (port-closed? *stderr*)) #f)
(test (begin (close-output-port *stdout*) (port-closed? *stdout*)) #f)
(test (begin (close-input-port *stdin*) (port-closed? *stdin*)) #f)

(test (let ((str ""))
	(with-input-from-string "1234567890" (lambda ()
          (with-input-from-string "1234567890" (lambda ()
            (with-input-from-string "1234567890" (lambda ()
              (with-input-from-string "1234567890" (lambda ()
                (with-input-from-string "1234567890" (lambda ()
                  (with-input-from-string "1234567890" (lambda ()
                    (with-input-from-string "1234567890" (lambda ()
	              (set! str (string-append str (string (read-char))))))
	            (set! str (string-append str (string (read-char) (read-char))))))
	          (set! str (string-append str (string (read-char) (read-char) (read-char))))))
	        (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char))))))
              (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char))))))
            (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char) (read-char))))))
          (set! str (string-append str (string (read-char) (read-char) (read-char) (read-char) (read-char) (read-char) (read-char))))))
	  str)
      "1121231234123451234561234567")

(let* ((new-error-port (open-output-string))
       (old-error-port (set-current-error-port new-error-port)))
  (catch #t
    (lambda ()
      (format #f "~R" 123))
    (lambda args
      (format (current-error-port) "oops")))
  (let ((str (get-output-string new-error-port)))
    (set-current-error-port old-error-port)
    (test str "oops")))

(let ((hi (open-input-string "hiho")))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'input-port))
   (list (lambda (p) (display 1 p))
	 (lambda (p) (write 1 p))
	 (lambda (p) (write-char #\a p))
	 (lambda (p) (write-byte 0 p))
	 (lambda (p) (format p "hiho"))
	 newline))
  (close-input-port hi))

(let ((hi (open-output-file tmp-output-file)))
  (write-byte 1 hi)
  (close-output-port hi)
  (test (write-byte 1 hi) 'error))

(let ((hi (open-output-string)))
  (for-each
   (lambda (op)
     (test-e (op hi) (object->string op) 'output-port))
   (list read read-char read-byte peek-char char-ready? read-line))
  (close-output-port hi))

(test (output-port? (current-error-port)) #t)
(test (and (not (null? (current-error-port))) (input-port? (current-error-port))) #f)

(call-with-output-file tmp-output-file
  (lambda (p)
    (test (get-output-string p) 'error)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-byte i p))))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (get-output-string p) 'error)
    (call-with-exit
     (lambda (quit)
       (do ((i 0 (+ i 1)))
	   ((= i 256))
	 (let ((b (read-byte p)))
	   (if (or (not (number? b))
		   (not (= b i)))
	       (begin
		 (format-logged #t "read-byte got ~A, expected ~A~%" b i)
		 (quit)))))))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format-logged #t "read-byte at end: ~A~%" eof)))
    (let ((eof (read-byte p)))
      (if (not (eof-object? eof))
	  (format-logged #t "read-byte at end: ~A~%" eof)))))

(call-with-output-file tmp-output-file
  (lambda (p)
    (do ((i 0 (+ i 1)))
	((= i 256))
      (write-char (integer->char i) p))))

(define our-eof #f)

(call-with-input-file tmp-output-file
  (lambda (p)
    (call-with-exit
     (lambda (quit)
       (do ((i 0 (+ i 1)))
	   ((= i 256))
	 (let ((b (read-char p)))
	   (if (or (not (char? b))
		   (not (char=? b (integer->char i))))
	       (begin
		 (format-logged #t "read-char got ~A, expected ~A (~D: char? ~A)~%" b (integer->char i) i (char? (integer->char i)))
		 (quit)))))))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format-logged #t "read-char at end: ~A~%" eof))
      (set! our-eof eof))
    (let ((eof (read-char p)))
      (if (not (eof-object? eof))
	  (format-logged #t "read-char again at end: ~A~%" eof)))))

(test (eof-object? (integer->char 255)) #f)
(test (eof-object? our-eof) #t)
(test (char->integer our-eof) 'error)
(test (char? our-eof) #f)
(test (eof-object? ((lambda () our-eof))) #t)

(for-each
 (lambda (op)
   (test (op *stdout*) 'error)
   (test (op *stderr*) 'error)
   (test (op (current-output-port)) 'error)
   (test (op (current-error-port)) 'error)
   (test (op ()) 'error))
 (list read read-line read-char read-byte peek-char char-ready?))

(for-each
 (lambda (op)
   (test (op #\a *stdin*) 'error)
   (test (op #\a (current-input-port)) 'error)
   (test (op #\a ()) 'error))
 (list write display write-char))
	 
(test (write-byte 0 *stdin*) 'error)
(test (write-byte (char->integer #\space) *stdout*) (char->integer #\space))
(test (write-byte (char->integer #\space) *stderr*) (char->integer #\space))
(test (newline *stdin*) 'error)
(test (format *stdin* "hiho") 'error)

(test (port-filename *stdin*) "*stdin*")	 
(test (port-filename *stdout*) "*stdout*")	 
(test (port-filename *stderr*) "*stderr*")	

(test (input-port? *stdin*) #t) 
(test (output-port? *stdin*) #f) 
(test (port-closed? *stdin*) #f)
(test (input-port? *stdout*) #f) 
(test (output-port? *stdout*) #t) 
(test (port-closed? *stdout*) #f)
(test (input-port? *stderr*) #f) 
(test (output-port? *stderr*) #t) 
(test (port-closed? *stderr*) #f)

(test (port-line-number *stdin*) 0)
(test (port-line-number *stdout*) 'error)
(test (port-line-number *stderr*) 'error)

(test (open-input-file "[*not-a-file!*]-") 'error)
(test (call-with-input-file "[*not-a-file!*]-" (lambda (p) p)) 'error)
(test (with-input-from-file "[*not-a-file!*]-" (lambda () #f)) 'error)

(test (open-input-file "") 'error)
(test (call-with-input-file "" (lambda (p) p)) 'error)
(test (with-input-from-file "" (lambda () #f)) 'error)

;(test (open-output-file "/bad-dir/badness/[*not-a-file!*]-") 'error)
;(test (call-with-output-file "/bad-dir/badness/[*not-a-file!*]-" (lambda (p) p)) 'error)
;(test (with-output-to-file "/bad-dir/badness/[*not-a-file!*]-" (lambda () #f)) 'error)

(with-output-to-file "tmp.r5rs"
  (lambda ()
    (write-char #\a)
    (with-output-to-file tmp-output-file
      (lambda ()
	(format-logged #t "~C" #\b)
	(with-output-to-file "tmp2.r5rs"
	  (lambda ()
	    (display #\c)))
	(display (with-input-from-file "tmp2.r5rs"
		   (lambda ()
		     (read-char))))))
    (with-input-from-file tmp-output-file
      (lambda ()
	(write-byte (read-byte))
	(write-char (read-char))))))

(with-input-from-file "tmp.r5rs"
  (lambda ()
    (test (read-line) "abc")))

(with-input-from-file "tmp.r5rs" ; this assumes tmp.r5rs has "abc" as above
  (lambda ()
    (test (read-char) #\a)
    (test (eval-string "(+ 1 2)") 3)
    (test (read-char) #\b)
    (with-input-from-string "(+ 3 4)"
      (lambda ()
	(test (read) '(+ 3 4))))
    (test (read-char) #\c)))

(test (eval-string (object->string (with-input-from-string "(+ 1 2)" read))) 3)
(test (eval (eval-string "(with-input-from-string \"(+ 1 2)\" read)")) 3)
(test (eval-string "(eval (with-input-from-string \"(+ 1 2)\" read))") 3)
(test (eval-string (object->string (eval-string (format #f "(+ 1 2)")))) 3)


;;; -------- test that we can plow past errors --------

(if (and (defined? 'file-exists?) ; (ifdef name ...)?
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(call-with-output-file "tests.data"
  (lambda (p)
    (format p "start ")
    (catch #t 
      (lambda () 
	(format p "next ") (abs "hi") (format p "oops "))
      (lambda args
	'error))
    (format p "done\n")))

(let ((str (call-with-input-file "tests.data" 
             (lambda (p) 
	       (read-line p)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format-logged #t ";call-with-output-file + error -> ~S~%" str)))

(let ((str (call-with-input-file "tests.data" 
             (lambda (p) 
	       (catch #t
		      (lambda ()
			(read-char p)
			(abs "hi")
			(read-char p))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format-logged #t ";call-with-input-file + error -> ~S~%" str)))

(if (and (defined? 'file-exists?)
	 (file-exists? "tests.data"))
    (delete-file "tests.data"))

(with-output-to-file "tests.data"
  (lambda ()
    (format-logged #t "start ")
    (catch #t 
      (lambda () 
	(format-logged #t "next ") (abs "hi") (format-logged #t "oops "))
      (lambda args
	'error))
    (format-logged #t "done\n")))

(let ((str (with-input-from-file "tests.data" 
             (lambda () 
	       (read-line)))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format-logged #t ";with-output-to-file + error -> ~S~%" str)))

(let ((str (with-input-from-file "tests.data" 
             (lambda () 
	       (catch #t
		      (lambda ()
			(read-char)
			(abs "hi")
			(read-char))
		      (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format-logged #t ";with-input-from-file + error -> ~S~%" str)))

(test (call-with-output-string newline) (string #\newline))
(test (call-with-output-string append) "")

(let ((str (call-with-output-string
	    (lambda (p)
	      (format p "start ")
	      (catch #t 
		     (lambda () 
		       (format p "next ") (abs "hi") (format p "oops "))
		     (lambda args
		       'error))
	      (format p "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format-logged #t ";call-with-output-string + error -> ~S~%" str)))

(let ((str (with-output-to-string
	    (lambda ()
	      (format-logged #t "start ")
	      (catch #t 
		     (lambda () 
		       (format-logged #t "next ") (abs "hi") (format-logged #t "oops "))
		     (lambda args
		       'error))
	      (format-logged #t "done")))))
  (if (or (not (string? str))
	  (not (string=? str "start next done")))
      (format-logged #t ";with-output-to-string + error -> ~S~%" str)))

(test (with-output-to-string (lambda () (format (current-output-port) "a test ~D" 123))) "a test 123")
;(test (with-output-to-string (lambda () (format *stdout* "a test ~D" 1234))) "a test 1234")

(test (string=? (with-output-to-string (lambda () (write #\null))) "#\\null") #t)
(test (string=? (with-output-to-string (lambda () (write #\space))) "#\\space") #t)
(test (string=? (with-output-to-string (lambda () (write #\return))) "#\\return") #t)
(test (string=? (with-output-to-string (lambda () (write #\escape))) "#\\escape") #t)
(test (string=? (with-output-to-string (lambda () (write #\tab))) "#\\tab") #t)
(test (string=? (with-output-to-string (lambda () (write #\newline))) "#\\newline") #t)
(test (string=? (with-output-to-string (lambda () (write #\backspace))) "#\\backspace") #t)
(test (string=? (with-output-to-string (lambda () (write #\alarm))) "#\\alarm") #t)
(test (string=? (with-output-to-string (lambda () (write #\delete))) "#\\delete") #t)

(test (string=? (with-output-to-string (lambda () (write-char #\space))) " ") #t)  ; weird -- the name is backwards
(test (string=? (with-output-to-string (lambda () (display #\space))) " ") #t)

(let ((str (call-with-input-string "12345"
	    (lambda (p)
	      (catch #t
		     (lambda ()
		       (read-char p)
		       (abs "hi")
		       (read-char p))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format-logged #t ";call-with-input-string + error -> ~S~%" str)))

(let ((str (with-input-from-string "12345"
	    (lambda ()
	      (catch #t
		     (lambda ()
		       (read-char)
		       (abs "hi")
		       (read-char))
		     (lambda args "s"))))))
  (if (or (not (string? str))
	  (not (string=? str "s")))
      (format-logged #t ";with-input-from-string + error -> ~S~%" str)))

(for-each
 (lambda (arg)
   (test (port-line-number arg) 'error)
   (test (port-filename arg) 'error))
 (list "hi" -1 0 #\a 'a-symbol #(1 2 3) '(1 . 2) '(1 2 3) 3.14 3/4 1.0+1.0i #t abs #<eof> #<unspecified> (lambda () 1)))

(test (catch #t (lambda () (eval-string (port-filename))) (lambda args #f)) #f)
(test (symbol? (string->symbol (port-filename))) #t)

(for-each
 (lambda (arg)
   (test
    (with-input-from-string (format #f "~A" arg)
      (lambda ()
	(read)))
    arg))
 (list 1 3/4 '(1 2) #(1 2) :hi #f #t))

(num-test (with-input-from-string "3.14" read) 3.14)
(num-test (with-input-from-string "3.14+2i" read) 3.14+2i)
(num-test (with-input-from-string "#x2.1" read) 2.0625)
(test (with-input-from-string "'hi" read) ''hi)
(test (with-input-from-string "'(1 . 2)" read) ''(1 . 2))

(test
 (let ((cin #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-input-from-string "123"
	      (lambda ()
		(set! cin (current-input-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cin cerr))
 "<input-string-port (closed)> #t")

;;; old form:  "<port string input (closed)> #t")

(test
 (let ((cp (current-output-port))
       (cout #f)
       (cerr #f))
   (catch #t
	  (lambda ()
	    (with-output-to-string
	      (lambda ()
		(set! cout (current-output-port))
		(error 'testing "jump out"))))
	  (lambda args
	    (set! cerr #t)))
   (format #f "~A ~A" cout cerr))
 "<output-string-port (closed)> #t")
(if (not (eq? *stdout* old-stdout))
    (format *stderr* ";~D: stdout clobbered~%" (port-line-number)))

;;; old form:  "<port string output (closed)> #t")

(let ((port (open-input-file #u8(115 55 116 101 115 116 46 115 99 109 0) #u8(114 0 98)))) ; "s7test.scm" "r\x00b"
  (test (input-port? port) #t) ; ??
  (close-input-port port))

(call-with-output-file tmp-output-file
  (lambda (p)
    (display "1" p)
    (newline p)
    (newline p)
    (display "2345" p)
    (newline p)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (eof-object? (read-line p)) #t)))

(let ((p (open-output-file tmp-output-file "a")))
  (display "678" p)
  (newline p)
  (close-output-port p))

(if (not with-windows) ; "xyzzy" is legit in windows??
    (begin
      (test (let ((p (open-output-file tmp-output-file "xyzzy"))) (close-output-port p)) 'error)
      (test (let ((p (open-input-file tmp-output-file "xyzzy"))) (close-input-port p)) 'error)))

(call-with-input-file tmp-output-file
  (lambda (p)
    (test (read-line p) "1")
    (test (read-line p) "")
    (test (read-line p) "2345")
    (test (read-line p) "678")
    (test (eof-object? (read-line p)) #t)))

(test (let ((a 1))
	(define-macro (m1) `(set! a (read)))
	(with-input-from-string "123" m1)
	a)
      123)

(test (let ((a 1))
	(define-macro (m3 p) `(set! a (read ,p)))
	(call-with-input-string "123" m3)
	a)
      123)
  
(test (let ()
	(define-macro (m1) `(define a (read)))
	(with-input-from-string "123" m1)
	a)
      123)

(test (let ()
	(define-macro (m3 p) `(define a (read ,p)))
	(call-with-input-string "123" m3)
	a)
      123)
  
(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (port-filename arg) 'error))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (open-input-file "s7test.scm" arg) 'error)
   (test (open-output-file tmp-data-file arg) 'error))
 (list -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t () (list 1 2 3) '(1 . 2)))

(test (current-input-port ()) 'error)
(test (current-output-port ()) 'error)
(test (current-error-port ()) 'error)

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format-logged #t ";(~A) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format                     ; newline
       ;read read-char read-byte peek-char char-ready? read-line      ; these can default to current input
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))

(for-each
 (lambda (op)
   (let ((tag (catch #t (lambda () (op 1 2 3 4 5)) (lambda args 'error))))
     (if (not (eq? tag 'error))
	 (format-logged #t ";(~A 1 2 3 4 5) -> ~A (expected 'error)~%" op tag))))
 (list set-current-input-port set-current-error-port set-current-output-port 
       close-input-port close-output-port
       write display write-byte write-char format newline
       read read-char read-byte peek-char char-ready? read-line
       call-with-output-file call-with-input-file
       call-with-output-string call-with-input-string
       with-input-from-string with-input-from-file
       with-output-to-file
       open-output-file open-input-file 
       open-input-string))

;;; (string-set! (with-input-from-string "\"1234\"" read) 1 #\a)
(test (with-input-from-string "(+ 1 2)" read) '(+ 1 2))

(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\tab))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\newline))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\"))))) 512) #t)
(test (>= (length (with-output-to-string (lambda () (write (make-string 512 #\x65))))) 512) #t)

(if (and (defined? 'file-exists?)
	 (file-exists? "/home/bil/test"))
    (let-temporarily ((*load-path* (cons "/home/bil/test" *load-path*)))
      (with-output-to-file "/home/bil/test/load-path-test.scm"
	(lambda ()
	  (format-logged #t "(define (load-path-test) *load-path*)~%")))

      (load "load-path-test.scm")
      (if (or (not (defined? 'load-path-test))
	      (not (equal? *load-path* (load-path-test))))
	  (format-logged #t ";*load-path*: ~S, but ~S~%" *load-path* (load-path-test)))))

;;; function ports
(when with-block

  (let ((p (function-open-output)))
    (write-char #\a p)
    (let ((val (function-get-output p)))
      (function-close-output p)
      (if (not (string=? val "a"))
	  (format *stderr* ";function port write #\\a: ~S (~D, ~A)~%" val (length val) (string->vector val)))))
  
  (let ((p (function-open-output)))
    (display "123" p)
    (format p "4~D6" 5)
    (write-string "789" p)
    (write-byte (char->integer #\0) p)
    (newline p)
    (let ((val (function-get-output p)))
      (function-close-output p)
      (if (not (string=? val "1234567890\n"))
	  (format *stderr* ";function port outputs: ~S (~D, ~A)~%" val (length val) (string->vector val)))))
  
  (let ((str "0123"))
    (let ((p (function-open-input str)))
      (let ((val (read-char p)))
	(if (not (char=? val #\0))
	    (format *stderr* ";function port read #\\0: ~S~%" val)))))
  
  (let ((str "0123\n45678"))
    (let ((p (function-open-input str)))
      (let ((val (read-line p)))
	(if (not (string=? val "0123"))
	    (format *stderr* ";function port read-line: ~S~%" val))
	(set! val (read-byte p))
	(if (not (= val (char->integer #\4)))
	    (format *stderr* ";function port read-byte: ~S~%" val))
	(set! val (peek-char p))
	(if (not (char=? val #\5))
	    (format *stderr* ";function port peek-char: ~S~%" val))
	(set! val (read-string 2 p))
	(if (not (string=? val "56"))
	    (format *stderr* ";function port read-string: ~S~%" val))
	(if (and (not pure-s7)
		 (not (char-ready? p)))
	    (format *stderr* ";function port has no char ready?~%"))
	(close-input-port p)))))



;;; -------- poke at the reader --------

(test (cdr '(1 ."a")) "a")
(test (cadr '(1 .#d2)) '.#d2)
(test '(1 .(2 3)) '(1 2 3))
(test '(1 .(2 3)) '(1 . (2 3)))
(test (+ .(2 .(3))) 5)
(test (cadr '(1 '0,)) ''0,)
(test (equal? 3 ' 3) #t)
(test (equal? '   
	             3 3) #t)
(test (equal? '"hi" ' "hi") #t)
(test (equal? '#\a '    #\a) #t)
(test (let ((nam()e 1)) 1) 'error)
(test (let ((nam""e 1)) nam""e) 'error) ; this was 1 originally
(test (cadr '(1 ']x)) '']x)
(test `1 1)
(test (equal? '(1 .(1 .())) '(1 1)) #t)
(test (equal? '("hi"."ho") ' ("hi" . "ho")) #t)
(test (equal? '("hi""ho") '("hi" "ho")) #t)
(test '("""""") '("" "" ""))
(test '(#|;"();|#) ())
(test '(#||##\# #||##b1) '(#\# 1))
(test (#|s'!'|#*) 1)
(test (#|==|#) ())
(test -#|==|#1 'error) ; unbound variable
(test '((). '()) '(() quote ()))
(test '(1. . .2) '(1.0 . 0.2))
(test (equal? '(().()) '(())) #t)
(test (equal? '(()()) '(() ())) #t)
(test (equal? '(()..()) '(() .. ())) #t)
(test '((().()).()) '((())))
(test '(((().()).()).()) '(((()))))
(test '((().(().())).()) '((() ())))
(test '((()().(().()))) '((() () ())))
(test '(1 .;
	  2) '(1 . 2))
(test (vector .(1 .(2))) #(1 2))
(test (vector 0. .(.1)) #(0.0 0.1))
(test '(a #|foo||# b) '(a b)) ; from bug-guile
(test '(a #|foo|||# b) '(a b))
(test '(a #|foo||||# b) '(a b))
(test '(a #|foo|||||# b) '(a b))

(test (let () (define (f' x) (+ x x)) (f' 10)) 20) ; from /r/scheme
(test (let () (define (f'' a'b) (+ a'b a'b)) (f'' 10)) 20)
(test (symbol? 'a'b) #t)

(test (char? #\#) #t)
(test (eval-string "'#<float-vector>") 'error)
(test (eval-string "'(#<float-vector>)") 'error)
(test (car `(,.1e0)) .1)
(test (car `(,.1E0)) .1)
(test (let ((x "hi")) (set! x"asdf") x) "asdf")
(test (let* ((x "hi") (y x)) (set! x "asdf") y) "hi")
(test (let ((x 1)) (set! x(list 1 2)) x) '(1 2))
(num-test (let ((x 1)) (set!;"
			x;)
			12.;(
			);#|
	       x) 12.0)
(test (let ((\x00}< 1) (@:\t{ 2)) (+ \x00}< @:\t{)) 3)
(test (let ((| 1) (|| 2) (||| 3)) (+ || | |||)) 6)
(test (let ((|a#||#b| 1)) |a#||#b|) 1)
(test (let ((@,@'[1] 1) (\,| 2)) (+ @,@'[1] \,|)) 3)
(test (list"0"0()#()#\a"""1"'x(list)+(cons"""")#f) (list "0" 0 () #() #\a "" "1" 'x (list) + '("" . "") #f))
(test (let ((x, 1)) x,) 1)
(test (length (eval-string (string #\' #\( #\1 #\space #\. (integer->char 200) #\2 #\)))) 2) ; will be -1 if dot is for improper list, 3 if dot is a symbol
(test (eval-string "(list \\\x001)") 'error)
(test (eval-string "(list \\\x00 1)") 'error)
(test (+ `,0(angle ```,`11)) 0)
(test (map . (char->integer "123")) '(49 50 51))
(test (map .(values "0'1")) '(#\0 #\' #\1))
(test (map /""'(123)) ())
(num-test (+ 1 .()) 1)
(test (let () (define (x .()) (list .())) (x)) ())

;; how is ...#(... parsed?
(test (eval-string "'(# (1))") 'error)
(test (let ((lst (eval-string "'(#(1))"))) (and (= (length lst) 1) (vector? (car lst)))) #t)                     ; '(#(1))
(test (let ((lst (eval-string "'(-#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(-# (1))
(test (let ((lst (eval-string "'(1#(1))"))) (and (= (length lst) 2) (symbol? (car lst)) (pair? (cadr lst)))) #t) ; '(1# (1))
(test (let ((lst (eval-string "'('#(1))"))) (and (= (length lst) 1) (vector? (cadar lst)))) #t)                  ; '((quote #(1)))
(test (let ((lst (eval-string "'(()#())"))) (and (= (length lst) 2) (null? (car lst)) (vector? (cadr lst)))) #t) ; '(() #())
(test (let ((lst (eval-string "'(().())"))) (and (= (length lst) 1) (null? (car lst)))) #t)                      ; '(())
(test (let ((lst (eval-string "'(()-())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t)  ; '(() - ())
(test (let ((lst (eval-string "'(().#())"))) (and (= (length lst) 3) (null? (car lst)) (null? (caddr lst)))) #t) ; '(() .# ())
(test (let ((lst (eval-string "'((). #())"))) (and (= (length lst) -1) (null? (car lst)) (vector? (cdr lst)))) #t) ; '(() . #())
(test (let ((lst (eval-string "'(\"\"#())"))) (and (= (length lst) 2) (string? (car lst)) (vector? (cadr lst)))) #t) ; '("" #())
(test (length (car '("#\\("))) 3)
(test (length (car '("#\\\""))) 3)
(test (char=? ((car '("#\\\"")) 2) #\") #t)
(test (length '(()#\(())) 3)
(test (length (eval-string "'(()#\\(())")) 3)
(test (char=? ((eval-string "'(()#\\#())") 1) #\#) #t)
(test (length (list""#t())) 3)
(test (length (list""#())) 2)
(test (length (eval-string "'(#xA(1))")) 2)
(test (length '(#xA""#(1))) 3)
(test (length (eval-string "'(#xA\"\"#(1))")) 3)
(test (length (eval-string "'(1#f)")) 1)
(test (eval-string "'(#f#())") 'error)
(test (length '(#f())) 2)
(test (length '(#f"")) 2)
(test (eval-string "#F") 'error)
(test (eval-string "'(#<eof>#<eof>)") 'error)
(test (eval-string "'(#<eof>#())") 'error)
(test (equal? '('#()) '(#())) #f)
(test (equal? (list #()) '(#())) #t)
(test (equal? '(#()) '(#())) #t)
(test (equal? '('#()) '(`#())) #f) ;  [guile agrees]
(test (equal? '('()) '(`())) #f) ; ! quote != quasiquote [guile agrees]
(test (equal? '('(1)) '(`(1))) #t) ;  but lists are different? [guile says #f]
(test (equal? '('#(1)) '(`#(1))) #f) ; [guile agrees]
(test (equal? '('#()) '(#())) #f)
(test (equal? '(`#()) '(`#())) #t)
(test (equal? #() `#()) #t)
(test (equal? (list #()) (list `#())) #t)
(test (equal? (list #()) '(`#())) #t)
(test (equal? '(`#()) '(#())) #t)
(test (equal? `#() #()) #t) ; and also (1) () #(1) etc
(test (equal? `#() '#()) #t) ; "
(test (equal? '`#() ''#()) #f) ; it equals #() -- this is consistent -- see below
(test (equal? '`#() ``#()) #t)

(test (equal? '() '()) #t)
(test (equal? (quote ()) '()) #t)
(test (equal? '() (quote ())) #t)
(test (equal? (quote ()) (quote ())) #t)
(test (equal? `(1) '(1)) #t)
(test (equal? (quasiquote (1)) '(1)) #t)
(test (equal? `(1) (quote (1))) #t)
(test (equal? (quasiquote (1)) (quote (1))) #t)
(test (equal? ``''1 '``'1) #t)
(test (equal? (quasiquote `(quote (quote 1))) '``'1) #t)
(test (equal? ``''1 (quote ``(quote 1))) #t)
(test (equal? (quasiquote `(quote (quote 1))) (quote ``(quote 1))) #t)
(test (equal? '``'#f ```'#f) #t)
(test (equal? (quote ``(quote #f)) ```'#f) #t)
(test (equal? '``'#f (quasiquote ``(quote #f))) #t)
(test (equal? (quote ``(quote #f)) (quasiquote ``(quote #f))) #t)
;;; etc:

#|
(equal? (quote `1) (quote (quasiquote 1))) -> #f
the reader sees `1 and turns it into 1 in the first case, but does not collapse the 2nd case to 1
  (who knows, quasiquote might have been redefined in context... but ` can't be redefined):
:(define (` a) a)
;define: define a non-symbol? 'a
;    (define ('a) a)

this is different from guile which does not handle ` at read time except to expand it:

guile> (quote `1) 
(quasiquote 1)

:(quote `1)
1

so anything that quotes ` is not going to equal quote quasiquote

(define (check-strs str1 str2)
  (for-each
   (lambda (arg)
     (let ((expr (format #f "(equal? ~A~A ~A~A)" str1 arg str2 arg)))
       (let ((val (catch #t 
			 (lambda () (eval-string expr))
			 (lambda args 'error))))
	 (format #t "--------~%~S -> ~S" expr val)
	 (let* ((parens3 0)
		(parens4 0)
		(str3 (apply string-append (map (lambda (c)
						 (if (char=? c #\`)
						     (if (= parens3 0)
							 (begin
							   (set! parens3 (+ parens3 1))
							   "(quasiquote ")
							 "`")
						     (if (char=? c #\')
							 (begin
							   (set! parens3 (+ parens3 1))
							   "(quote ")
							 (string c))))
						str1)))
		(str4 (apply string-append (map (lambda (c)
						 (if (char=? c #\`)
						     (if (= parens4 0)
							 (begin
							   (set! parens4 (+ parens4 1))
							   "(quasiquote ")
							 "`")
						     (if (char=? c #\')
							 (begin
							   (set! parens4 (+ parens4 1))
							   "(quote ")
							 (string c))))
						str2))))
	   (let ((expr (format #f "(equal? ~A~A~A ~A~A)" str3 arg (make-string parens3 #\)) str2 arg)))
	     (let* ((val1 (catch #t 
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S" expr val1))))
	   (let ((expr (format #f "(equal? ~A~A ~A~A~A)" str1 arg str4 arg (make-string parens4 #\)))))
	     (let* ((val1 (catch #t 
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S" expr val1))))
	   (let ((expr (format #f "(equal? ~A~A~A ~A~A~A)" str3 arg (make-string parens3 #\)) str4 arg (make-string parens4 #\)))))
	     (let* ((val1 (catch #t 
			       (lambda () (eval-string expr))
			       (lambda args 'error)))
		    (trouble (and (not (eq? val1 'error))
				  (not (eq? val1 val)))))
	       (if trouble
		   (format #t "~%~8T~A~S -> ~S~A~%" bold-text expr val1 unbold-text)
		   (format #t "~%~8T~S -> ~S~%" expr val1))))
	   ))))
   (list "()" "(1)" "#()" "#(1)" "1" "#f")))
   ;; (list ",(+ 1 2)" "\"\"" "(())" "#\\1" "3/4" ",1")

(check-strs "'" "'")
(check-strs "`" "'")
(check-strs "'" "`")
(check-strs "`" "`")

(let ((strs ()))
  (do ((i 0 (+ i 1)))
      ((= i 4))
    (let ((c1 ((vector #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 4))
	(let ((d1 ((vector #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\`) k)))
	  (let ((str1 (string c1 c2))
		(str2 (string d1 d2)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))

(let ((strs ()))
  (do ((i 0 (+ i 1)))
      ((= i 8))
    (let ((c1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) i))
	  (c2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) i))
	  (c3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) i)))
      (do ((k 0 (+ k 1)))
	  ((= k 8))
	(let ((d1 ((vector #\' #\` #\' #\` #\' #\` #\' #\`) k))
	      (d2 ((vector #\' #\' #\` #\` #\' #\' #\` #\`) k))
	      (d3 ((vector #\' #\' #\' #\' #\` #\` #\` #\`) k)))
	  (let ((str1 (string c1 c2 c3))
		(str2 (string d1 d2 d3)))
	    (if (not (member (list str1 str2) strs))
		(begin
		  (check-strs str1 str2)
		  (set! strs (cons (list str1 str2) strs))
		  (set! strs (cons (list str2 str1) strs))))))))))


;;; --------------------------------

(do ((i 0 (+ i 1)))
    ((= i 256))
  (if (and (not (= i (char->integer #\))))
	   (not (= i (char->integer #\"))))
      (let ((str (string #\' #\( #\1 #\space #\. (integer->char i) #\2 #\))))
	(catch #t
	       (lambda ()
		 (let ((val (eval-string str)))
		   (format #t "[~D] ~A -> ~S (~S ~S)~%" i str val (car val) (cdr val))))
	       (lambda args
		 (format #t "[~D] ~A -> ~A~%" i str args))))))

(let ((chars (vector (integer->char 0) #\newline #\space #\tab #\. #\, #\@ #\= #\x #\b #\' #\` 
		     #\# #\] #\[ #\} #\{ #\( #\) #\1 #\i #\+ #\- #\e #\_ #\\ #\" #\: #\; #\> #\<)))
  (let ((nchars (vector-length chars)))
    (do ((len 2 (+ len 1)))
	((= len 3))
      (let ((str (make-string len))
	    (ctrs (make-vector len 0)))

	(do ((i 0 (+ i 1)))
	    ((= i (expt nchars len)))

	  (let ((carry #t))
	    (do ((k 0 (+ k 1)))
		((or (= k len)
		     (not carry)))
	      (vector-set! ctrs k (+ 1 (vector-ref ctrs k)))
	      (if (= (vector-ref ctrs k) nchars)
		  (vector-set! ctrs k 0)
		  (set! carry #f)))
	    (do ((k 0 (+ k 1)))
		((= k len))
	      (string-set! str k (vector-ref chars (vector-ref ctrs k)))))

	  (format #t "~A -> " str)
	  (catch #t
		 (lambda ()
		   (let ((val (eval-string str)))
		     (format #t " ~S -> ~S~%" str val)))
		 (lambda args
		   ;(format #t " ~A~%" args)
		   #f
		   )))))))
|#

(let ((äåæéîå define)
      (ìåîçôè length)
      (äï do)
      (ìåô* let*)
      (éæ if)
      (áâó abs)
      (ìïç log)
      (óåô! set!))

  (äåæéîå (óòã-äõòáôéïî å)
    (ìåô* ((ìåî (ìåîçôè å))
           (åø0 (å 0))
           (åø1 (å (- ìåî 2)))
           (áìì-ø (- åø1 åø0))
           (äõò 0.0))
      (äï ((é 0 (+ é 2)))
          ((>= é (- ìåî 2)) äõò)
        (ìåô* ((ø0 (å é))
               (ø1 (å (+ é 2)))
               (ù0 (å (+ é 1))) ; 1/ø ø ðïéîôó
               (ù1 (å (+ é 3)))
               (áòåá (éæ (< (áâó (- ù0 ù1)) .0001)
                         (/ (- ø1 ø0) (* ù0 áìì-ø))
                         (* (/ (- (ìïç ù1) (ìïç ù0)) 
                               (- ù1 ù0)) 
                            (/ (- ø1 ø0) áìì-ø)))))
         (óåô! äõò (+ äõò (áâó áòåá)))))))

  (num-test (óòã-äõòáôéïî (list 0 1 1 2)) 0.69314718055995)
  (num-test (óòã-äõòáôéïî (vector 0 1 1 2)) 0.69314718055995))

(test (let ((ÿa 1)) ÿa) 1)
(test (+ (let ((!a 1)) !a) (let (($a 1)) $a) (let ((%a 1)) %a) (let ((&a 1)) &a) (let ((*a 1)) *a) (let ((+a 1)) +a) (let ((-a 1)) -a) (let ((.a 1)) .a) (let ((/a 1)) /a) (let ((0a 1)) 0a) (let ((1a 1)) 1a) (let ((2a 1)) 2a) (let ((3a 1)) 3a) (let ((4a 1)) 4a) (let ((5a 1)) 5a) (let ((6a 1)) 6a) (let ((7a 1)) 7a) (let ((8a 1)) 8a) (let ((9a 1)) 9a) (let ((<a 1)) <a) (let ((=a 1)) =a) (let ((>a 1)) >a) (let ((?a 1)) ?a) (let ((@a 1)) @a) (let ((Aa 1)) Aa) (let ((Ba 1)) Ba) (let ((Ca 1)) Ca) (let ((Da 1)) Da) (let ((Ea 1)) Ea) (let ((Fa 1)) Fa) (let ((Ga 1)) Ga) (let ((Ha 1)) Ha) (let ((Ia 1)) Ia) (let ((Ja 1)) Ja) (let ((Ka 1)) Ka) (let ((La 1)) La) (let ((Ma 1)) Ma) (let ((Na 1)) Na) (let ((Oa 1)) Oa) (let ((Pa 1)) Pa) (let ((Qa 1)) Qa) (let ((Ra 1)) Ra) (let ((Sa 1)) Sa) (let ((Ta 1)) Ta) (let ((Ua 1)) Ua) (let ((Va 1)) Va) (let ((Wa 1)) Wa) (let ((Xa 1)) Xa) (let ((Ya 1)) Ya) (let ((Za 1)) Za) (let (([a 1)) [a) (let ((\a 1)) \a) (let ((]a 1)) ]a) (let ((^a 1)) ^a) (let ((_a 1)) _a) (let ((aa 1)) aa) (let ((ba 1)) ba) (let ((ca 1)) ca) (let ((da 1)) da) (let ((ea 1)) ea) (let ((fa 1)) fa) (let ((ga 1)) ga) (let ((ha 1)) ha) (let ((ia 1)) ia) (let ((ja 1)) ja) (let ((ka 1)) ka) (let ((la 1)) la) (let ((ma 1)) ma) (let ((na 1)) na) (let ((oa 1)) oa) (let ((pa 1)) pa) (let ((qa 1)) qa) (let ((ra 1)) ra) (let ((sa 1)) sa) (let ((ta 1)) ta) (let ((ua 1)) ua) (let ((va 1)) va) (let ((wa 1)) wa) (let ((xa 1)) xa) (let ((ya 1)) ya) (let ((za 1)) za) (let (({a 1)) {a) (let ((|a 1)) |a) (let ((}a 1)) }a) (let ((~a 1)) ~a) (let (( a 1))  a) (let ((¡a 1)) ¡a) (let ((¢a 1)) ¢a) (let ((£a 1)) £a) (let ((¤a 1)) ¤a) (let ((¥a 1)) ¥a) (let ((¦a 1)) ¦a) (let ((§a 1)) §a) (let ((¨a 1)) ¨a) (let ((©a 1)) ©a) (let ((ªa 1)) ªa) (let ((«a 1)) «a) (let ((¬a 1)) ¬a) (let ((­a 1)) ­a) (let ((®a 1)) ®a) (let ((¯a 1)) ¯a) (let ((°a 1)) °a) (let ((±a 1)) ±a) (let ((²a 1)) ²a) (let ((³a 1)) ³a) (let ((´a 1)) ´a) (let ((µa 1)) µa) (let ((¶a 1)) ¶a) (let ((·a 1)) ·a) (let ((¸a 1)) ¸a) (let ((¹a 1)) ¹a) (let ((ºa 1)) ºa) (let ((»a 1)) »a) (let ((¼a 1)) ¼a) (let ((½a 1)) ½a) (let ((¾a 1)) ¾a) (let ((¿a 1)) ¿a) (let ((Àa 1)) Àa) (let ((Áa 1)) Áa) (let ((Âa 1)) Âa) (let ((Ãa 1)) Ãa) (let ((Äa 1)) Äa) (let ((Åa 1)) Åa) (let ((Æa 1)) Æa) (let ((Ça 1)) Ça) (let ((Èa 1)) Èa) (let ((Éa 1)) Éa) (let ((Êa 1)) Êa) (let ((Ëa 1)) Ëa) (let ((Ìa 1)) Ìa) (let ((Ía 1)) Ía) (let ((Îa 1)) Îa) (let ((Ïa 1)) Ïa) (let ((Ða 1)) Ða) (let ((Ña 1)) Ña) (let ((Òa 1)) Òa) (let ((Óa 1)) Óa) (let ((Ôa 1)) Ôa) (let ((Õa 1)) Õa) (let ((Öa 1)) Öa) (let ((×a 1)) ×a) (let ((Øa 1)) Øa) (let ((Ùa 1)) Ùa) (let ((Úa 1)) Úa) (let ((Ûa 1)) Ûa) (let ((Üa 1)) Üa) (let ((Ýa 1)) Ýa) (let ((Þa 1)) Þa) (let ((ßa 1)) ßa) (let ((àa 1)) àa) (let ((áa 1)) áa) (let ((âa 1)) âa) (let ((ãa 1)) ãa) (let ((äa 1)) äa) (let ((åa 1)) åa) (let ((æa 1)) æa) (let ((ça 1)) ça) (let ((èa 1)) èa) (let ((éa 1)) éa) (let ((êa 1)) êa) (let ((ëa 1)) ëa) (let ((ìa 1)) ìa) (let ((ía 1)) ía) (let ((îa 1)) îa) (let ((ïa 1)) ïa) (let ((ða 1)) ða) (let ((ña 1)) ña) (let ((òa 1)) òa) (let ((óa 1)) óa) (let ((ôa 1)) ôa) (let ((õa 1)) õa) (let ((öa 1)) öa) (let ((÷a 1)) ÷a) (let ((øa 1)) øa) (let ((ùa 1)) ùa) (let ((úa 1)) úa) (let ((ûa 1)) ûa) (let ((üa 1)) üa) (let ((ýa 1)) ýa) (let ((þa 1)) þa) (let ((ÿa 1)) ÿa)) 181)

;;; there are about 50 non-printing chars, some of which would probably work as well


;; (eval-string "(eval-string ...)") is not what it appears to be -- the outer call
;;    still sees the full string when it evaluates, not the string that results from
;;    the inner call.


(let () ; from scheme bboard
  (define (maxlist list) 
    (define (maxlist' l max) 
      (if (null? l) max 
	  (if (> (car l) max) 
	      (maxlist' (cdr l) (car l)) 
	      (maxlist' (cdr l) max)))) 
    (if (null? list) 'undef 
	(maxlist' list (car list)))) 
  (test (maxlist '(1 2 3)) 3) ; quote is ok in s7 if not the initial char (sort of like a number)

  (let ((h'a 3))
    (test h'a 3))
  (let ((1'2 32))
    (test 1'2 32))
  (let ((1'`'2 32))
    (test 1'`'2 32))
  (let ((1'`,@2 32))
    (test 1'`,@2 32))

;  (test (define '3 32) 'error) ;define quote: syntactic keywords tend to behave badly if redefined
  )

(let ((|,``:,*|',## 1)
      (0,,&:@'>>.<# 2)
      (@.*0`#||\<,, 3)
      (*&:`&'>#,*<` 4)
      (*0,,`&|#*:`> 5)
      (>:|<*.<@:\|` 6)
      (*',>>:.'@,** 7)
      (0|.'@<<:,##< 8)
      (<>,\',\.>>#` 9)
      (@#.>|&#&,\0* 10)
      (0'.`&<','<<. 11)
      (&@@*<*\'&|., 12)
      (|0*&,':|0\** 13)
      (<:'*@<>*,<&` 14)
      (>@<@<|>,`&'. 15)
      (@#,00:<:@*.\ 16)
      (*&.`\>#&,&., 17)
      (0|0|`,,..<@, 18)
      (0@,'>\,,&.@# 19)
      (>@@>,000`\#< 20)
      (|>*'',<:&@., 21)
      (|>,0>0|,@'|. 22)
      (0,`'|'`,:`@` 23)
      (<>#'>,,\'.'& 24)
      (*..,|,.,&&@0 25))
  (test (+ |,``:,*|',## 0,,&:@'>>.<# @.*0`#||\<,, *&:`&'>#,*<` *0,,`&|#*:`> >:|<*.<@:\|` *',>>:.'@,**
	   0|.'@<<:,##< <>,\',\.>>#` @#.>|&#&,\0* 0'.`&<','<<.  &@@*<*\'&|., |0*&,':|0\** <:'*@<>*,<&`
           >@<@<|>,`&'. @#,00:<:@*.\ *&.`\>#&,&., 0|0|`,,..<@, 0@,'>\,,&.@# >@@>,000`\#<
           |>*'',<:&@., |>,0>0|,@'|. 0,`'|'`,:`@` <>#'>,,\'.'& *..,|,.,&&@0)
	325))

(when full-test
  (let ((first-chars (list #\. #\0 #\@ #\! #\& #\| #\* #\< #\>))
	(rest-chars (list #\. #\0 #\@ #\! #\| #\, #\# #\' #\\ #\` #\, #\: #\& #\* #\< #\>)))
    (let ((first-len (length first-chars))
	  (rest-len (length rest-chars)))
      (let ((n 100)
	    (size 12))
	(let ((str (make-string size #\space)))
	  (do ((i 0 (+ i 1)))
	      ((= i n))
	    (set! (str 0) (first-chars (random first-len)))
	    (do ((k 1 (+ 1 k)))
		((= k size))
	      (set! (str k) (rest-chars (random rest-len))))
	    (catch #t (lambda ()
			(let ((val (eval-string (format #f "(let () (define ~A 3) ~A)" str str))))
			  (format #f "~A -> ~A~%" str val)))
		   (lambda args
		     (format #f "~A error: ~A~%" str args)))))))))

(let ((List 1)
      (LIST 2)
      (lIsT 3)
      (-list 4)
      (_list 5)
      (+list 6))
  (test (apply + (list List LIST lIsT -list _list +list)) 21))

(let ()
  (define (\ arg) (+ arg 1))
  (test (+ 1 (\ 2)) 4)
  (define (@\ arg) (+ arg 1))
  (test (+ 1 (@\ 2)) 4)
  (define (@,\ arg) (+ arg 1))
  (test (+ 1 (@,\ 2)) 4)
  (define (\,@\ arg) (+ arg 1))
  (test (+ 1 (\,@\ 2)) 4)
  )

;;; these are from the r7rs discussions
(test (let ((a'b 3)) a'b) 3) ; allow variable names like "can't-go-on" or "don't-ask"
(test (let () (define (f x y) (+ x y)) (let ((a 3) (b 4)) (f a, b))) 'error) ; unbound variable a,
(test (let () (define (f x y) (+ x y)) (let ((a 3) (b 4)) (f a ,b))) 'error) ; unquote outside quasiquote

(test (vector? (owlet 0. 3/4 #(reader-cond ))) 'error)
(test (vector? #(reader-cond)) #t)
(test ((cond-expand (string (integer->char 255)) (hash-table '(a . 2) 0 (abs (append (string (integer->char 255)) (make-block 32) (inlet 'value 1 '+ (lambda args 1)) (int-vector 1 2 3))))) cdddr) 'error)



;;; -------- object->string
;;; object->string

(test (string=? (object->string 32) "32") #t)
(test (string=? (object->string 32.5) "32.5") #t)
(test (string=? (object->string 32/5) "32/5") #t)
(test (object->string 1+i) "1+1i")
(test (string=? (object->string "hiho") "\"hiho\"") #t)
(test (string=? (object->string 'symb) "symb") #t)
(test (string=? (object->string (list 1 2 3)) "(1 2 3)") #t)
(test (string=? (object->string (cons 1 2)) "(1 . 2)") #t)
(test (string=? (object->string #(1 2 3)) "#(1 2 3)") #t)
(test (string=? (object->string +) "+") #t)
(test (object->string (object->string (object->string "123"))) "\"\\\"\\\\\\\"123\\\\\\\"\\\"\"")
(test (object->string #<eof>) "#<eof>")
(test (object->string (if #f #f)) "#<unspecified>")
(test (object->string #<undefined>) "#<undefined>")
(test (object->string #f) "#f")
(test (object->string #t) "#t")
(test (object->string ()) "()")
(test (object->string #()) "#()")
(test (object->string "") "\"\"")
(test (object->string abs) "abs")
(test (object->string lambda) "lambda")
(test (object->string (lambda () a)) "#<lambda ()>")
(test (object->string (lambda a a)) "#<lambda a>")
(test (object->string (lambda (a) a)) "#<lambda (a)>")
(test (object->string (lambda (a . b) a)) "#<lambda (a . b)>")
(test (object->string (lambda (a b) a)) "#<lambda (a b)>")
(test (object->string (lambda (a b c) a)) "#<lambda (a b ...)>")
(test (object->string (lambda (a b . c) a)) "#<lambda (a b ...)>")
(test (object->string (lambda* (a :rest b) a)) "#<lambda* (a :rest b)>")
(test (object->string (lambda* (:rest a) a)) "#<lambda* (:rest a)>")
(test (object->string (lambda* (a b :rest c) a)) "#<lambda* (a b ...)>")
(let () (define-macro (mac a) a) (test (object->string mac) "mac"))
(let ((m (define-macro (mac a) a))) (test (object->string m) "#<macro (a)>"))
(let ((m (define-macro* (mac a) a))) (test (object->string m) "#<macro* (a)>"))
(let ((m (define-bacro (mac a) a))) (test (object->string m) "#<bacro (a)>"))
(let ((m (define-bacro* (mac a) a))) (test (object->string m) "#<bacro* (a)>"))
(let ((_?_m (define-expansion (_?_mac a) a))) (test (object->string _?_m) "#<expansion (a)>"))
(test (object->string +) "+")
(test (object->string +) "+")
(test (object->string '''2) "''2")
(test (object->string (lambda () #f)) "#<lambda ()>") ;"#<closure>"
(test (call-with-exit (lambda (return) (object->string return))) "#<goto>")
(test (call/cc (lambda (return) (object->string return))) "#<continuation>")
(test (let () (define-macro (hi a) `(+ 1 ,a)) (object->string hi)) "hi")
(test (let () (define (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (let () (define* (hi a) (+ 1 a)) (object->string hi)) "hi")
(test (object->string dynamic-wind) "dynamic-wind")
(test (object->string (dilambda (lambda () 1) (lambda (val) val))) "#<lambda ()>") ;"#<closure>"
(test (object->string object->string) "object->string")
(test (object->string 'if) "if")
(test (object->string begin) "begin")
(test (object->string let) "let")

(test (object->string #\n #f) "n")
(test (object->string #\n) "#\\n")
(test (object->string #\r) "#\\r")
(test (object->string #\r #f) "r")
(test (object->string #\t #f) "t")
(test (object->string #\t) "#\\t")

(test (object->string "a\x00b" #t) "\"a\\x00b\"")
(test (object->string "a\x00b" #f) "a\x00b")

#|
(do ((i 0 (+ i 1))) 
    ((= i 256)) 
  (let ((c (integer->char i))) 
    (let ((str (object->string c))) 
      (if (and (not (= (length str) 3))       ; "#\\a"
	       (or (not (char=? (str 2) #\x))
		   (not (= (length str) 5)))) ; "#\\xee"
	  (format-logged #t "(#t) ~C: ~S~%" c str))
      (set! str (object->string c #f))
      (if (not (= (length str) 1))
	  (format-logged #t "(#f) ~C: ~S~%" c str)))))
this prints:
(#t) : "#\\null"
(#f) : ""
(#t) : "#\\x1"
(#t) : "#\\x2"
(#t) : "#\\x3"
(#t) : "#\\x4"
(#t) : "#\\x5"
(#t) : "#\\x6"
(#t) : "#\\x7"
(#t): "#\\x8"
(#t) 	: "#\\tab"
(#t) 
: "#\\newline"
(#t) 
     : "#\\xb"
(#t) 
     : "#\\xc"
: "#\\return"
(#t) : "#\\xe"
(#t) : "#\\xf"
(#t)  : "#\\space"
|#

(test (object->string #\x30) "#\\0")
(test (object->string #\x91) "#\\x91")
(test (object->string #\x10) "#\\x10")
(test (object->string #\xff) "#\\xff")
(test (object->string #\x55) "#\\U")
(test (object->string #\x7e) "#\\~")
(test (object->string #\newline) "#\\newline")
(test (object->string #\return) "#\\return")
(test (object->string #\tab) "#\\tab")
(test (object->string #\null) "#\\null")
(test (object->string #\space) "#\\space")
(test (object->string (integer->char 8)) "#\\backspace")
(test (object->string ''#\a) "'#\\a")
(test (object->string (list 1 '.' 2)) "(1 .' 2)")
(test (object->string (quote (quote))) "(quote)")
(test (object->string (quote quote)) "quote")
(test (object->string (quote (quote (quote)))) "'(quote)")

(test (object->string) 'error)
(test (object->string 1 2) 'error)
(test (object->string 1 #f #t) 'error)
(test (object->string abs) "abs")
(test(let ((val 0)) (cond (else (set! val (object->string else)) 1)) val) "else")
(test (cond (else (object->string else))) "else")
(test (object->string (string->symbol (string #\; #\" #\)))) "(symbol \";\\\")\")")

(test (object->string "hi" #f) "hi")
(test (object->string "h\\i" #f) "h\\i")
(test (object->string -1.(list? -1e0)) "-1.0")

(test (object->string catch) "catch")
(test (object->string lambda) "lambda")
(test (object->string dynamic-wind) "dynamic-wind")
(test (object->string quasiquote) "quasiquote")
;(test (object->string else) "else") ; this depends on previous code
(test (object->string do) "do")

(for-each
 (lambda (arg)
   (test (object->string 1 arg) 'error)
   (test (object->string arg) (with-output-to-string (lambda () (write arg))))
   (test (object->string arg #t) (with-output-to-string (lambda () (write arg))))
   (test (object->string arg #f) (with-output-to-string (lambda () (display arg)))))
 (list "hi" -1 #\a 1 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i () (list 1 2 3) '(1 . 2)))

(test (string->symbol (object->string #(1 #\a (3)) #f)) (symbol "#(1 #\\a (3))"))
(test (string->list (object->string #(1 2) #f)) '(#\# #\( #\1 #\space #\2 #\)))
(test (string->list (object->string #(1 #\a (3)) #f)) '(#\# #\( #\1 #\space #\# #\\ #\a #\space #\( #\3 #\) #\)))
(test (reverse (object->string #2D((1 2) (3 4)) #f))  "))4 3( )2 1((D2#")

;; write readably (this affects ~W in format as well)
;; :readable special things
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (not (eq? n obj))
	   (format *stderr* "~A not eq? ~A (~S)~%" n obj str)))))
 (list #<eof> #<undefined> #<unspecified> #t #f #true #false else ()
       lambda lambda* begin case if do quote set! let let* letrec
       cond and or define define* define-constant define-macro
       define-macro* define-bacro define-bacro*
       with-baffle
       *stdin* *stdout* *stderr*))

;; :readable characters
(do ((i 0 (+ i 1)))
    ((= i 256))
  (let ((c (integer->char i)))
    (let ((str (object->string c :readable)))
      (let ((nc (with-input-from-string str
		  (lambda ()
		    (eval (read)))))) ; no need for eval here or in some other cases, but might as well be consistent
	(if (not (eq? c nc))
	    (format *stderr* "~C (~D) != ~C (~S)~%" c i nc str))))))

;; :readable integers
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((nn (with-input-from-string str
		 (lambda ()
		   (eval (read))))))
       (if (or (not (integer? n))
	       (not (integer? nn))
	       (not (= n nn)))
	   (format *stderr* "~D != ~D (~S)~%" n nn str)))))
 (list 0 1 3 most-positive-fixnum -0 -1 -3 most-negative-fixnum
       -9223372036854775808 9223372036854775807))
;; but unless gmp at read end we'll fail with most-positive-fixnum+1
;; -> check *features* at start of read

;; :readable ratios
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((nn (with-input-from-string str
		 (lambda ()
		   (eval (read))))))
       (if (or (not (rational? n))
	       (not (rational? nn))
	       (not (= n nn)))
	   (format *stderr* "~A != ~A (~S)~%" n nn str)))))
 (list 1/2 -1/2 123456789/2 -2/123456789 2147483647/2147483646 312689/99532
       -9223372036854775808/3 9223372036854775807/2  1/1428571428571429 1/1152921504606846976))

(when (not (provided? 'solaris))
  ;; :readable reals
  (for-each
   (lambda (n)
     (let ((str (object->string n :readable)))
       (let ((nn (with-input-from-string str
		   (lambda ()
		     (eval (read))))))
	 (if (or (not (real? n))
		 (not (real? nn))
		 (not (morally-equal? n nn)))
	     (format *stderr* "~A != ~A (~S)~%" n nn str)))))
   (list 1.0 0.0 -0.0 pi 0.1 -0.1 0.9999999995 9007199254740993.1 (sqrt 2) 1/100000000000
	 1.5e-16 1.5e16 3.141592653589793238462643383279502884197169399375105820 1e-300 8.673617379884e-19
	 1/0 (- 1/0) (real-part (log 0)) (- (real-part (log 0)))))
  
  ;; :readable complex
  (for-each
   (lambda (n)
     (let ((str (object->string n :readable)))
       (let ((nn (with-input-from-string str
		   (lambda ()
		     (eval (read))))))
	 (if (or (not (complex? n))
		 (not (complex? nn))
		 (not (morally-equal? n nn)))
	     (format *stderr* "~A != ~A (~S)~%" n nn str)))))
   (list 0+i 0-i 1+i 1.4+i 3.0+1.5i
	 (log 0) (- (log 0)) 
	 (complex 1/0 1.0) (complex 1/0 1/0) (complex 1.0 1/0) ; default: nan+1i nannani 1nani!
	 (complex 1/0 (real-part (log 0))) (complex (real-part (log 0)) 1/0) 
	 1e-14+1e14i 0+1e-16i (complex pi pi))))


;; :readable strings/byte-vectors
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (string? n))
	       (not (string? obj))
	       (not (string=? n obj))
	       (and (byte-vector? n)
		    (not (byte-vector? obj))))
	   (format *stderr* "~S not string=? ~S (~S)~%" n obj str)))))
 (list "" "abc" (string #\newline) "#<abc>" "a\"b\"c" "a\\b\nc" "aBc"
       (let ((s (make-string 4 #\space))) (set! (s 3) #\null) s) ; writes as "   \x00"
       "ab
c"
       (string #\a #\b #\null #\c #\escape #\newline)
       (string #\x (integer->char #xf0) #\x)
       (string #\null)
       #u8() #u8(0 1 2 3) 
       (let ((str (make-string 256 #\null)))
	 (do ((i 0 (+ i 1)))
	     ((= i 256) str)
	   (set! (str i) (integer->char i))))))

;; :readable symbols/keywords
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (symbol? n))
	       (not (symbol? obj))
	       (not (eq? n obj)))
	   (format *stderr* "~A not eq? ~A (~S)~%" n obj str)))))
 (list 'abc :abc abc:
       (symbol "a") (symbol "#<>")
       (gensym "|") (gensym "#<>") (gensym "}")
       :: ':abc
       (gensym "\\")))
  
;; :readable environments
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (let? n))
	       (not (let? obj))
	       (not (equal? n obj)))
	   (format *stderr* "~A not equal?~%~A~%    (~S)~%" n obj str)))))
 (list (inlet '(a . 1))
       (inlet)
       (rootlet)
       (inlet (cons 't12 "12") (cons (symbol "#<") 12))
       (inlet 'a 1 'a 2)))

;(test (object->string (list (owlet)) :readable) "(list (owlet))")

;; :readable hash-tables
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (hash-table? n))
	       (not (hash-table? obj))
	       (not (equal? n obj)))
	   (format *stderr* ";readable hash-tables, ~A not equal? ~A (~S)~%" n obj str)))))
 (list (hash-table '(a . 1))
       (hash-table '(a . 1) (cons 'b "hi"))
       (let ((ht (make-hash-table 31)))
	 (set! (ht 1) 321)
	 (set! (ht 2) 123)
	 ht)
       (let ((ht (make-hash-table)))
	 (set! (ht 'b) 1)
	 (set! (ht 'a) ht)
	 ht)
       ;(let ((ht (make-hash-table))) (set! (ht ht) 123) ht) 
       ;(let ((ht (make-hash-table))) (set! (ht ht) ht)  ht)
       (hash-table)))
  
;; :readable vectors
(let-temporarily (((*s7* 'print-length) 8))
  (for-each
   (lambda (p)
     (set! (*s7* 'print-length) p)
     (for-each
      (lambda (n)
	(let ((str (object->string n :readable)))
	  (let ((obj (with-input-from-string str
		       (lambda ()
			 (eval (read))))))
	    (if (or (not (vector? n))
		    (not (vector? obj))
		    (not (equal? n obj)))
		(format *stderr* ";readable vectors, ~A not equal? ~A (~S)~%" n obj str)))))
      (list #() #(1) #(1 #(2)) #2d((1 2) (3 4))
	    #3d(((1 2 3) (4 5 6) (7 8 9)) ((9 8 7) (6 5 4) (3 2 1)))
	    #2d()
	    #(1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0)
	    (let ((v (vector 1 2 3))) (set! (v 1) v) v)
	    (let ((v (vector 1 #(2) 3))) (set! ((v 1) 0) v) v)
	    (let ((v #2d((1 2 3) (4 5 6)))) (set! (v 1 1) v) v)
	    (make-int-vector 3 0)
	    (make-float-vector 3 0.0)
	    (make-int-vector '(2 3) 1))))
   (list 8 2 1)))

(test (object->string (vector 1 2 3) :readable) "(vector 1 2 3)")
  
;; :readable lists (circular, dotted)
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (pair? n))
	       (not (pair? obj))
	       (not (equal? n obj)))
	   (format *stderr* ";readable lists, ~A not equal? ~A (~S)~%" n obj str)))))
 (list '(1) '(1 . 2) '((1 ()) 3) '((1 2) (3 4))
       '(1 2 . 3) '(1 2 3 . 4) '(())
       (let ((lst (cons 1 2))) (set-cdr! lst lst) lst)
       (let ((lst (list 1 2 3))) (set-cdr! (cddr lst) lst) lst)
       (let ((lst (list 1 2 3))) (set-car! (cddr lst) lst) lst)
       ))

;; :readable macros
(let ()
  (define-macro (mac1) `(+ 1 2))
  (test ((eval-string (object->string mac1 :readable))) 3)
  (define-macro (mac2 a) `(+ ,a 2))
  (test ((eval-string (object->string mac2 :readable)) 1) 3)
  (define-macro* (mac3 (a 1)) `(+ ,a 2))
  (test ((eval-string (object->string mac3 :readable))) 3)
  (define-macro (mac4 . a) `(+ ,@a 2))
  (test ((eval-string (object->string mac4 :readable)) 1 3) 6)
  (define-macro (mac5 a b . c) `(+ ,a ,b ,@c 2))
  (test ((eval-string (object->string mac5 :readable)) 1 5 3 4) 15)
  (define-macro (mac7 a) (let ((b (+ a 1))) `(+ ,b ,a)))
  (test ((eval-string (object->string mac7 :readable)) 2) 5)
  )

;; :readable closures/functions/built-in (C) functions + the setters thereof
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (procedure? n))
	       (not (procedure? obj))
	       (not (equal? (procedure-source n) (procedure-source obj))))
	   (format *stderr* "'~A not equal? '~A (~S)~%" n obj str)))))
 (list abs
       (lambda () 1)
       (lambda (a) (+ a 1))
       (lambda args (display args) (cdr args))
       (lambda* (a b) (or a b))
       (let ((a 1)) (lambda (b) (+ a b)))
       (let ((b 2)) (let ((a 1)) (lambda* (c . d) (display (+ a b c) *stdout*) d)))
       (lambda* (:rest b) b)
       ))

(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (test ((eval-string str) 21) (n 21))))
 (list (lambda (a) (+ a 1))
       (lambda args (cdr args))
       (lambda* (a b) (or a b))
       (let ((a 1)) (lambda (b) (+ a b)))
       (let ((b 2)) (let ((a 1)) (lambda* (c . d) (+ a b c))))
       (lambda* (:rest b) b)
       ))

(let ()
  (define* (f1 a :allow-other-keys) (+ a 1))
  (let ((str (object->string f1 :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (test (f1 2 :b 3) 3)
       (test (obj 2) 3)
       (test (obj 2 :b 3) 3)))) ; too many args
    

;; :readable ports
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (input-port? n))
	       (not (input-port? obj))
	       (not (equal? (port-closed? n) (port-closed? obj))))
	   (format *stderr* "~A not equal? ~A (~S)~%" n obj str)
	   (if (and (not (port-closed? n))
		    (not (eq? n *stdin*))
		    (not (eq? n (current-input-port))))
	       (let ((c1 (read-char n))
		     (c2 (read-char obj)))
		 (if (not (equal? c1 c2))
		     (format *stderr* "read-char results ~A not equal? ~A (~S)~%" c1 c2 str)))))
       (if (and (not (eq? n *stdin*))
		(not (eq? n (current-input-port))))
	   (begin
	     (close-input-port n)
	     (close-input-port obj))))))
 (list *stdin*
       (open-input-string "a test")
       (call-with-input-string "a test" (lambda (p) p))
       (let ((p (open-input-string "a test"))) (read-char p) p)
       (call-with-input-file "s7test.scm" (lambda (p) p))
       (open-input-file "write.scm")
       (let ((p (open-input-file "write.scm"))) (read-char p) p)))

;; :readable environments
(for-each
 (lambda (n)
   (let ((str (object->string n :readable)))
     (let ((obj (with-input-from-string str
		  (lambda ()
		    (eval (read))))))
       (if (or (not (let? n))
	       (not (let? obj))
	       (not (equal? n obj)))
	   (format *stderr* "~A not equal? ~A (~S)~%" n obj str)))))
 (list (rootlet)
       (let ((a 1)) (curlet))
       (let ((a 1) (b 2)) (curlet))))

(when with-block
  (let ((b (block 1.0 2.0 3.0 4.0 5.0 6.0 7.0 8.0)))
    (let-temporarily (((*s7* 'print-length) 2))
      (test (format #f "~W" b) "(block 1.000 2.000 3.000 4.000 5.000 6.000 7.000 8.000)")
      (test (format #f "~A" b) "(block 1.000 2.000 ...)"))))

(test (object->string (define (ex1 a b) (+ a  b)) :readable) "(lambda (a b) (+ a b))")
(test (object->string (let ((c 3)) (define (ex1 a b) (+ a c b))) :readable) "(let ((c 3)) (lambda (a b) (+ a c b)))")
(test (object->string (let ((c 3)) (define (ex1) (+ c 1))) :readable) "(let ((c 3)) (lambda () (+ c 1)))")
(test (object->string (define* (ex1 a (b 0)) (+ a  b)) :readable) "(lambda* (a (b 0)) (+ a b))")
(test (object->string (define (ex1 a . b) (+ a  b)) :readable) "(lambda (a . b) (+ a b))")

(let-temporarily (((*s7* 'print-length) 4))
  (define (f1) (vector-ref #(0 1 2 3 4 5 6 7 8) 2))
  (test (object->string f1 :readable) "(lambda () (vector-ref #(0 1 2 3 4 5 6 7 8) 2))"))

(test (object->string (make-iterator #u8(12 41 2)) :readable) "(make-iterator #u8(12 41 2))")
(test (object->string (inlet) :readable) "(inlet)")
(test (object->string (inlet 'a 1) :readable) "(inlet 'a 1)")
(test (object->string (inlet 'a 1 'b 2) :readable) "(inlet 'a 1 'b 2)")
(test (object->string (inlet 'a #\1) :readable) "(inlet 'a #\\1)")
(test (object->string (inlet 'a #\newline) :readable) "(inlet 'a #\\newline)")
(test (object->string (inlet 'a #\null) :readable) "(inlet 'a #\\null)")
(test (object->string (inlet 'a 3.0) :readable) "(inlet 'a 3.0)")
(test (object->string (inlet 'a 1/2) :readable) "(inlet 'a 1/2)")
(test (object->string (inlet 'a 1+i) :readable) "(inlet 'a 1+1i)")
(test (object->string (inlet 'a (log 0)) :readable) (format #f "(inlet 'a (~S -inf.0 3.141592653589793))" 'complex))
(test (object->string (inlet 'a 1/0) :readable) "(inlet 'a nan.0)")
(test (object->string (inlet 'a "1") :readable) "(inlet 'a \"1\")")
(test (object->string (inlet 'a "") :readable) "(inlet 'a \"\")")
(test (object->string (inlet 'a #<undefined>) :readable) "(inlet 'a #<undefined>)")
(test (object->string (inlet 'a #<unspecified>) :readable) "(inlet 'a #<unspecified>)")
(test (object->string (inlet 'a #<eof>) :readable) "(inlet 'a (begin #<eof>))")
(test (object->string (inlet 'a lambda) :readable) "(inlet 'a lambda)")
(test (object->string (inlet 'a 'b) :readable) "(inlet 'a 'b)")
(test (object->string (inlet 'a (symbol "( a b c )")) :readable) "(inlet 'a (symbol \"( a b c )\"))")
(test (object->string (inlet 'a else) :readable) "(inlet 'a else)")
(test (object->string (inlet 'a (cons 1 2)) :readable) "(inlet 'a (cons 1 2))")
(test (object->string (inlet 'a (list 1 2)) :readable) "(inlet 'a (list 1 2))")
(test (object->string (inlet 'a (list "hi")) :readable) "(inlet 'a (list \"hi\"))")
(test (object->string (inlet 'a ()) :readable) "(inlet 'a ())")
(test (object->string (inlet 'a '(1 2 . 3)) :readable) "(inlet 'a (cons 1 (cons 2 3)))")
(test (object->string (inlet 'a #t) :readable) "(inlet 'a #t)")
(test (object->string (inlet 'a #f) :readable) "(inlet 'a #f)")
(test (object->string (inlet 'a :b) :readable) "(inlet 'a :b)")
(test (object->string (inlet 'a (hash-table)) :readable) "(inlet 'a (hash-table))")
(test (object->string (inlet 'a (hash-table* 'b 1))  :readable) "(inlet 'a (hash-table (cons 'b 1)))")
(test (object->string (inlet 'a (hash-table* 'b "hi")) :readable) "(inlet 'a (hash-table (cons 'b \"hi\")))")
(test (object->string (inlet 'a (hash-table* 'b "h\"i")) :readable) "(inlet 'a (hash-table (cons 'b \"h\\\"i\")))")
(test (object->string (inlet 'a #()) :readable) "(inlet 'a #())")
(test (object->string (inlet 'a #(1 2 3)) :readable) "(inlet 'a (vector 1 2 3))")
(test (object->string (inlet 'a (vector "hi" #\a 'b)) :readable) "(inlet 'a (vector \"hi\" #\\a 'b))")
(test (object->string (inlet 'a (float-vector 1 2 3)) :readable) "(inlet 'a (float-vector 1.0 2.0 3.0))")
(test (object->string (inlet 'a (int-vector 1 2 3)) :readable) "(inlet 'a (int-vector 1 2 3))")
(test (object->string (inlet 'a #2d((1 2 3) (4 5 6))) :readable) "(inlet 'a (make-shared-vector (vector 1 2 3 4 5 6) '(2 3)))")
(test (object->string (inlet 'a abs) :readable) "(inlet 'a abs)")
(test (object->string (inlet 'a (lambda (b) (+ b 1))) :readable) "(inlet 'a (lambda (b) (+ b 1)))")
(test (object->string (inlet 'a (lambda b (list b 1))) :readable) "(inlet 'a (lambda b (list b 1)))")
(test (object->string (inlet 'a (lambda (a . b) (list a b))) :readable) "(inlet 'a (lambda (a . b) (list a b)))")
(test (object->string (inlet 'a (define-macro (_m_ b) `(+ ,b 1))) :readable) "(inlet 'a (define-macro (_m_ b) (list-values '+ b 1)))")
(test (object->string (inlet 'a (define-bacro (_m_ b) `(+ ,b 1))) :readable) "(inlet 'a (define-bacro (_m_ b) (list-values '+ b 1)))")
(test (object->string (inlet 'a (lambda* ((b 1)) (+ b 1))) :readable) "(inlet 'a (lambda* ((b 1)) (+ b 1)))")
(test (object->string (inlet 'a (lambda* a (list a))) :readable) "(inlet 'a (lambda* a (list a)))")
(test (object->string (inlet 'a (lambda* (a (b 1) c) (list a b c))) :readable) "(inlet 'a (lambda* (a (b 1) c) (list a b c)))")
(test (object->string (inlet 'a (define-macro* (_m_ (b 1)) `(+ ,b 1))) :readable) "(inlet 'a (define-macro* (_m_ (b 1)) (list-values '+ b 1)))")
(test (object->string (inlet 'a (define-bacro* (_m_ (b 1)) `(+ ,b 1))) :readable) "(inlet 'a (define-bacro* (_m_ (b 1)) (list-values '+ b 1)))")
(when with-block
  (test (object->string (inlet 'a (block)) :readable) "(inlet 'a (block))")
  (test (object->string (inlet 'a blocks) :readable) "(inlet 'a blocks)")
  (test (object->string (inlet 'a (block 1 2 3)) :readable) "(inlet 'a (block 1.000 2.000 3.000))"))
(test (object->string (inlet 'a (c-pointer 0)) :readable) "(inlet 'a (c-pointer 0))")
(test (object->string (inlet 'a (c-pointer 1)) :readable) "(inlet 'a (c-pointer 1))")
(test (object->string (inlet 'a quasiquote) :readable) "(inlet 'a quasiquote)")
(test (object->string (inlet 'a (dilambda (lambda () 1) (lambda (x) x))) :readable) "(inlet 'a (dilambda (lambda () 1) (lambda (x) x)))")
(test (object->string (inlet 'a (inlet 'b 1)) :readable) "(inlet 'a (inlet 'b 1))")
(test (object->string (inlet 'a (let ((b 1)) (lambda () b))) :readable) "(inlet 'a (let ((b 1)) (lambda () b)))")
(test (object->string 
       (inlet 'a (let ((y 1)) (dilambda (lambda () y) (lambda (x) (set! y x))))) :readable) 
      "(inlet 'a (let ((y 1)) (dilambda (lambda () y) (lambda (x) (set! y x)))))")
(test (object->string (inlet 'a (open-input-string "123456")) :readable) "(inlet 'a (open-input-string \"123456\"))")
(test (object->string (inlet 'a (let ((p (open-input-string "123456"))) (read-char p) p)) :readable) "(inlet 'a (open-input-string \"23456\"))")
(test (object->string (inlet 'a (let ((p (open-input-string "1"))) (read-char p) p)) :readable) "(inlet 'a (open-input-string \"\"))")
(test (object->string (inlet 'a (let ((p (open-input-string "1"))) (read-char p) (read-char p) p)) :readable) "(inlet 'a (open-input-string \"\"))")
(test (object->string (inlet 'a (call-with-input-string "1" (lambda (p) p))) :readable) "(inlet 'a (call-with-input-string \"\" (lambda (p) p)))")
(test (object->string (inlet 'a (let ((p (open-input-string "1"))) (close-input-port p) p)) :readable) "(inlet 'a (call-with-input-string \"\" (lambda (p) p)))")
(test (object->string (inlet 'a *stdin*) :readable) "(inlet 'a *stdin*)")
(test (object->string (inlet 'a *stdout*) :readable) "(inlet 'a *stdout*)")
(test (object->string (inlet 'a *stderr*) :readable) "(inlet 'a *stderr*)")
(test (object->string
       (inlet 'a (let ((p (open-output-string))) (close-output-port p) p)) :readable) 
      "(inlet 'a (let ((p (open-output-string))) (close-output-port p) p))")
(test (object->string (inlet 'a (open-output-string)) :readable) "(inlet 'a (let ((p (open-output-string))) p))")
(test (object->string 
       (inlet 'a (let ((p (open-output-string))) (display 32 p) p)) :readable) 
      "(inlet 'a (let ((p (open-output-string))) (display \"32\" p) p))")
(test (object->string (inlet 'a (open-output-file "test.test")) :readable) "(inlet 'a (open-output-file \"test.test\" \"a\"))")
(test (object->string (inlet 'a (open-input-file "test.test")) :readable) "(inlet 'a (open-input-file \"test.test\"))")
(test (object->string (inlet 'a (make-iterator "123")) :readable) "(inlet 'a (make-iterator \"123\"))")
(test (object->string (inlet 'a (let ((iter (make-iterator "123"))) (iter) iter)) :readable) "(inlet 'a (make-iterator \"23\"))")
(test (object->string (inlet 'a (make-iterator #(1 2 3))) :readable) "(inlet 'a (make-iterator (vector 1 2 3)))")
(test (object->string (inlet 'a (make-iterator '(1 2 3))) :readable) "(inlet 'a (make-iterator (list 1 2 3)))")
(test (object->string
       (inlet 'a (let ((iter (make-iterator (float-vector 1 2 3)))) (iter) iter)) :readable) 
      "(inlet 'a (let ((iter (make-iterator (float-vector 1.0 2.0 3.0)))) (do ((i 0 (+ i 1))) ((= i 1) iter) (iterate iter))))")

(test (object->string (let () (define (f1) (+ a 1)) (curlet)) :readable) "(inlet 'f1 (lambda () (+ a 1)))")
(test (object->string (let () (define (f1) 1) (let () (define f2 f1) (curlet))) :readable) "(inlet 'f2 (lambda () 1))")
(test (object->string 
       (let ((a 1)) (define d (let ((b 1)) (lambda (c) (+ a b c)))) (curlet)) :readable) 
      "(inlet 'a 1 'd (let ((b 1) (a 1)) (lambda (c) (+ a b c))))")
(test (object->string 
       (let () (define a (let ((b 1) (c 2)) (lambda (d) (+ b c d)))) (curlet)) :readable) 
      "(inlet 'a (let ((c 2) (b 1)) (lambda (d) (+ b c d))))")
(test (object->string (let ((a 1)) (define d (let ((b 1)) (let ((c b)) (lambda (e) (+ a b c e))))) (curlet)) :readable)
      "(inlet 'a 1 'd (let ((c 1) (b 1) (a 1)) (lambda (e) (+ a b c e))))")
(test (object->string (inlet 'a (let ((b 1)) (lambda () (+ b c)))) :readable) "(inlet 'a (let ((b 1)) (lambda () (+ b c))))")
(test (object->string (inlet 'a (let ((b 1)) (lambda () (+ b pi)))) :readable) "(inlet 'a (let ((b 1)) (lambda () (+ b pi))))")
(test (object->string (let* ((a 1) (b a)) (curlet)) :readable) "(inlet 'b 1)")
(test (object->string (let ((a 1)) (define (b c) (+ c a)) (curlet)) :readable) "(inlet 'a 1 'b (let ((a 1)) (lambda (c) (+ c a))))")
;;; ideally we'd catch the documentation setting

(test (string? (object->string (let ((lst (list 1))) (set-cdr! lst lst) (make-iterator lst)) :readable)) #t)

;;; these are not readable:
(test (object->string (inlet 'a (call-with-exit (lambda (return) return))) :readable) "(inlet 'a goto)")
(test (object->string (inlet 'a (call/cc (lambda (return) return))) :readable) "(inlet 'a continuation)")

;;; these are incorrect:
;(test (object->string (let () (define-constant a 32) (curlet)) :readable) "(inlet 'a 32)")
;(test (object->string #('1)) "(vector '1)")
;(test (object->string (inlet 'a ''()) :readable) "(inlet 'a '())")
(test (object->string (c-pointer 1234) :readable) "(c-pointer 1234)")

(test (string? (object->string (*s7* 'gc-protected-objects) :readable)) #t)
(test (string? (object->string (*s7* 'c-objects) :readable)) #t)
(test (string? (object->string (*s7* 'file-names) :readable)) #t)
(test (string? (object->string (*s7* 'c-types) :readable)) #t)
(test (string? (object->string (*s7* 'cpu-time) :readable)) #t)
(test (string? (object->string (*s7* 'catches) :readable)) #t)
(test (string? (object->string (*s7* 'exits) :readable)) #t)
(test (string? (object->string (*s7* 'stack) :readable)) #t)

(let ((ht (hash-table* 'a 1))
      (lt (inlet :b 1))
      (lst (list 1)))
  (set! (ht 'a) lst)
  (set! (lst 0) lt)
  (set! (lt 'b) ht)
  (test (object->string ht) "#1=(hash-table '(a (inlet 'b #1#)))"))

(let ((ht (hash-table* 'a 1)))
  (fill! ht ht)
  (test (object->string ht) "#1=(hash-table '(a . #1#))"))

(let ((ht (hash-table* 'a 1)))
  (set! (ht 'a) ht)
  (fill! ht (list ht))
  (test (object->string ht) "#1=(hash-table '(a #1#))"))

(let ((ht (hash-table* 'a 1)))
  (let ((lt (curlet)))
    (set! (ht 'a) ht)
    (fill! ht (list lt))
    (test (object->string ht) "#1=(hash-table '(a (inlet 'ht #1#)))")))

(if (not with-bignums)
    (begin
      (test (object->string (random-state 123 321)) "#<rng 123 321>")
      (test (object->string (random-state 9223372036854775807 9223372036854775807)) "#<rng 9223372036854775807 9223372036854775807>")
      (test (object->string (random-state 123 321) :readable) "(random-state 123 321)") 
      (test (object->string (random-state 9223372036854775807 9223372036854775807) :readable) "(random-state 9223372036854775807 9223372036854775807)"))
    (begin
      (test (substring (object->string (random-state 9223372036854775807 9223372036854775807)) 0 6) "#<rng ")
      (test (object->string (random-state 9223372036854775807 9223372036854775807) :readable) "#<unprint-readable object>")))

(when full-test
  (let ()
    (define (testlet e)
      (let ((data (cons #f #f)))
	(let ((iter (make-iterator e data)))
	  (do ((val (iterate iter) (iterate iter)))
	      ((iterator-at-end? iter))
	    (let ((sym (car val))
		  (fnc (cdr val)))
	      (if (procedure? fnc)
		  (let ((sig (catch #t (lambda () (procedure-signature fnc)) (lambda args #f)))
			(doc (catch #t (lambda () (procedure-documentation fnc)) (lambda args #f)))
			(src (catch #t (lambda () (procedure-source fnc)) (lambda args #f)))
			(ari (catch #t (lambda () (arity fnc)) (lambda args #f))))
		    (let ((lst (list sym fnc sig doc src ari)))
		      (object->string lst)
		      (object->string lst :readable)))
		  (begin
		    (object->string val)
		    (object->string val :readable))))))))
    
    (testlet (rootlet))
    
    (require libc.scm)
    (testlet *libc*)
    
    (require libm.scm)
    (when (defined? '*libm*) (testlet *libm*))
    
    (when (provided? 'gtk)
      (testlet *gtk*))
    
    (require libgsl.scm)
    (when (defined? '*libgsl*) (testlet *libgsl*))
    
    (require libgdbm.scm)
    (when (defined? '*libgdbm*) (testlet *libgdbm*))
    
    (require libdl.scm)
    (when (defined? '*libdl*) (testlet *libdl*))
    
    (require libutf8proc.scm)
    (when (defined? '*libutf8proc*) (testlet *libutf8proc*))))
      


;;; --------------------------------------------------------------------------------
;;; CONTROL OPS
;;; --------------------------------------------------------------------------------

(define control-ops (list lambda define quote if begin set! let let* letrec cond case and or do
			  call/cc eval apply for-each map values call-with-values dynamic-wind))
(for-each
 (lambda (op)
   (if (not (eq? op op))
       (format-logged #t "~A not eq? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (eqv? op op))
       (format-logged #t "~A not eqv? to itself?~%" op)))
 control-ops)

(for-each
 (lambda (op)
   (if (not (equal? op op))
       (format-logged #t "~A not equal? to itself?~%" op)))
 control-ops)

(define question-ops (list boolean? eof-object? string?
		           number? integer? real? rational? complex? char?
			   list? vector? pair? null?))

(for-each
 (lambda (ques)
   (for-each
    (lambda (op)
      (if (ques op)
	  (format-logged #t ";(~A ~A) returned #t?~%" ques op)))
    control-ops))
 question-ops)

(let ((unspecified (if #f #f)))
  (for-each
   (lambda (op)
     (if (op unspecified)
	 (format-logged #t ";(~A #<unspecified>) returned #t?~%" op)))
   question-ops))

(for-each 
 (lambda (s)
   (if (not (symbol? s))
       (format-logged #t ";(symbol? ~A returned #f?~%" s)))
 '(+ - ... !.. $.+ %.- &.! *.: /:. <-. =. >. ?. ~. _. ^.))



;;; --------------------------------------------------------------------------------
;;; if
;;; --------------------------------------------------------------------------------

(test ((if #f + *) 3 4) 12)
(test (if (> 3 2) 'yes 'no) 'yes)
(test (if (> 2 3) 'yes 'no) 'no)
(test (if (> 3 2) (- 3 2) (+ 3 2)) 1)
(test (if (> 3 2) 1) 1)
(test (if '() 1 2) 1)
(test (if 't 1 2) 1)
(test (if #t 1 2) 1)
(test (if #() 1 2) 1)
(test (if 1 2 3) 2)
(test (if 0 2 3) 2)
(test (if (list) 2 3) 2)
(test (if "" 2 3) 2)
(test (eq? (if #f #f) (if #f #f)) #t) ; I assume there's only one #<unspecified>!
(test (if . (1 2)) 2)
(test (if (if #f #f) #f #t) #f)
(test (if 1/0 0 1) 0)

(test (let ((a #t) (b #f) (c #t) (d #f)) (if (if (if (if d d c) d b) d a) 'a 'd)) 'a)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if a (if b (if c (if d d c) c) 'b) 'a)) 'b)
(test (let ((a #t) (b #f) (c #t) (d #f)) (if b (if a (if d 'gad) 'gad) (if d 'gad 'a))) 'a)

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if a arg 'gad) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(let ((a #t))
  (for-each
   (lambda (arg)
     (test (if (not a) 'gad arg) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2))))

(test (let ((ctr 0) (a #t)) (if a (let ((b ctr)) (set! ctr (+ ctr 1)) (list b ctr)) (let ((c ctr)) (set! ctr (+ ctr 100)) (list c ctr)))) (list 0 1))

(test (if if if if) if)
(test (((if if if) if if) if if 'gad) if)
(test (if if (if if if) if) if)
(test ((car (list if)) #t 0 1) 0)
(test (symbol->string 'if) "if")
(test (if (and if (if if if)) if 'gad) if)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (= ctr 1)) 0 1)) 0)
(test (let ((ctr 0)) (if (let () (set! ctr (+ ctr 1)) (if (= ctr 1) (> 3 2) (< 3 2))) 0 1)) 0)
(test (        if (> 3 2) 1 2) 1)
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assoc if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assv if alist)) (list if 3))
(test (let ((alist (list (list map 1) (list car 2) (list if 3) (list do 4)))) (assq if alist)) (list if 3))
(test (let ((alist (list map car if do))) (member if alist)) (list if do))
(test (let ((alist (list map car if do))) (memv if alist)) (list if do))
(test (let ((alist (list map car if do))) (memq if alist)) (list if do))
(test ((vector-ref (vector if) 0) #t 1 2) 1)
(test ((vector-ref (make-vector 1 if) 0) #t 1 2) 1)
(test ((if #t + -) 3 4) 7)
(test (list (if 0 1 2)) (list 1))
(test ((car (list if map)) #f 1 2) 2)
(test (let ((ctr 0)) (if (= ctr 0) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 2 3)) (let () (set! ctr (+ ctr 1)) (if (= ctr 1) 4 5)))) 2)
(test (let ((x (cons 1 2))) (set-cdr! x x) (if x 1 2)) 1)
(test (let ((ctr 0)) (if (let ((ctr 123)) (set! ctr (+ ctr 1)) (= ctr 124)) (let () (set! ctr (+ ctr 100)) ctr) (let () (set! ctr (+ ctr 1000)) ctr)) ctr) 100)
(test (let () (if #t (define (hi a) a)) (hi 1)) 1)
(test (let () (if #f (define (hi a) (+ a 1)) (define (hi a) a)) (hi 1)) 1)
(test (let ((oddp (lambda (a) (not (even? a))))) (define (hi a) (if (a 123) (a 321))) (hi oddp)) #t)

(test (let ((ctr 0)) (call/cc (lambda (exit) (if (> 3 2) (let () (exit ctr) (set! ctr 100) ctr) #f)))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (< 3 2) #f (let () (exit ctr) (set! ctr 100) ctr))))) 0)
(test (let ((ctr 0)) (call/cc (lambda (exit) (if (let () (exit ctr) (set! ctr 100) ctr) 123 321)))) 0)
(test (let ((ctr 0)) (if (> 3 2) (call/cc (lambda (exit) (set! ctr (+ ctr 1)) (exit ctr))) #f) ctr) 1)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4)))
				      (set! ctr (- ctr 5)))
				  (set! ctr (- ctr 6)))
			      (set! ctr (- ctr 7)))
			  (set! ctr (- ctr 8)))
		      (set! ctr (- ctr 9)))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      934)

(test (let ((ctr 0))
	(do ((x 0 (+ x 1)))
	    ((= x 12))
	  (if (> x 0)
	      (if (> x 1)
		  (if (> x 2)
		      (if (> x 3)
			  (if (> x 4)
			      (if (> x 5)
				  (if (> x 6)
				      (if (> x 7)
					  (if (> x 8)
					      (if (> x 9)
						  (if (> x 10)
						      (set! ctr (+ ctr 1000))
						      (set! ctr (- ctr 1)))
						  (set! ctr (- ctr 2)))
					      (set! ctr (- ctr 3)))
					  (set! ctr (- ctr 4))))))))
		  (set! ctr (- ctr 10)))
	      (set! ctr (- ctr 11))))
	ctr)
      969)

(test (if #f) 'error)
(test (if (< 2 3)) 'error)
(test (if #f 1 2 3) 'error)
(test (if 1 2 3 4) 'error)
(test (if #f 1 else 2) 'error)
(test (if) 'error)
(test ('+ '1 '2) 'error)
(test (if 1 . 2) 'error)
(test (if 1 2 . 3) 'error)
(test (if . 1) 'error)
(test (if _no_var_ 1) 'error)
(test (if (values) (values) (values) 1) 'error)
(num-test (+ 1 (if #t (values 3 4) (values 5 6)) 2) 10)
(let ()
  (define (bad a) (if a 1 2 3))
  (test (bad #f) 'error)
  (test (bad #t) 'error))


;;; when
(test (when #f #f) #<unspecified>)
(test (when #t #f) #f)
(test (when) 'error)
(test (when #t) 'error)
(test (when . #t) 'error)
(test (when #t . 1) 'error)
(test (when when when) when)
(test (symbol->string 'when) "when")
(test (when #t 1 2 3) 3)
(test (when #t (define a 1) (+ a 1)) 2)
(test ((when #t +) 2 3) 5)
(test (when #t (when #f #f)) #<unspecified>)
(test (+ (when #t (values 2 3))) 5)
(test (when (when #t #t) 2) 2)
(test (apply when '(< 2 3) '((+ 2 1))) 3)

(let ((x 0))
  (define (t1 a) (when a (set! x (+ x 1)) x))
  (test (t1 #t) 1)
  (test (t1 #f) #<unspecified>)
  (test (t1 #t) 2))


;;; unless
(test (unless #t #f) #<unspecified>)
(test (unless #f #f) #f)
(test (unless) 'error)
(test (unless #f) 'error)
(test (unless . #t) 'error)
(test (unless #f . 1) 'error)
(test (unless (not unless) unless) unless)
(test (symbol->string 'unless) "unless")
(test (unless #f 1 2 3) 3)
(test (unless #f (define a 1) (+ a 1)) 2)
(test ((unless #f +) 2 3) 5)
(test (unless #f (unless #t #f)) #<unspecified>)
(test (+ (unless #f (values 2 3))) 5)
(test (unless (unless #f #f) 2) 2)
(test (apply unless '(= 2 3) '((+ 2 1))) 3)

(let ((x 0))
  (define (t1 a) (unless a (set! x (+ x 1)) x))
  (test (t1 #f) 1)
  (test (t1 #t) #<unspecified>)
  (test (t1 #f) 2))

(test (when (unless (= 2 3) #t) 1) 1)




;;; --------------------------------------------------------------------------------
;;; quote
;;; --------------------------------------------------------------------------------

(test (quote a) 'a)
(test 'a (quote a))
(test '1 1)
(test '1/4 1/4)
(test '(+ 2 3) '(+ 2 3))
(test '"hi" "hi")
(test '#\a #\a)
(test '#f #f)
(test '#t #t)
(test '#b1 1)
(when (not pure-s7) (test (= 1/2 '#e#b1e-1) #t))
(test '() '())
(test '(1 . 2) (cons 1 2))
(test #(1 2) #(1 2))
(test (+ '1 '2) 3)
(test (+ '1 '2) '3)
(test (+ ' 1 '   2) '    3)
(test (char? '#\a) #t)
(test (string? '"hi") #t)
(test (boolean? '#t) #t)
(test (if '#f 2 3) 3)
(test (if '#t 2 3) 2)
(test (vector? #()) #t)
(test (char? (quote #\a)) #t)
(test (string? (quote "hi")) #t)
(test (boolean? (quote #t)) #t)
(test (if (quote #f) 2 3) 3)
(test (if (quote #t) 2 3) 2)
(test (vector? (quote #())) #t)
(test (+ (quote 1) (quote 2)) (quote 3))
(test (list? (quote ())) #t)
(test (pair? (quote (1 . 2))) #t)
(test (+ '1.0 '2.0) 3.0)
(test (+ '1/2 '3/2) 2)
(test (+ '1.0+1.0i '-2.0) -1.0+1.0i)
(test (let ((hi 2)) (equal? hi 'hi)) #f)
(test ''1 (quote (quote 1)))
(test ''a (quote (quote a)))
(test (symbol? '#f) #f)
(test (symbol? '.') #t)
(test ''quote (quote (quote quote)))
(test (+ (cadr ''3) (cadadr '''4) (cadr (cadr (cadr ''''5)))) 12)
(test (+ (cadr ' '   3) (cadadr '  
  '    ' 4)) 7)
(test (+ '#| a comment |#2 3) 5)
(test (+ ' #| a comment |# 2 3) 5)
(test (eq? lambda 'lambda) #f)
(test (equal? + '+) #f)
(test (eq? '() ()) #t) ; s7 specific

(test (quote) 'error)
(test (quote . -1) 'error)
(test (quote 1 1) 'error)
(test (quote . 1) 'error)
(test (quote . (1 2)) 'error)
(test (quote 1 . 2) 'error)
(test (symbol? '1'1) #t) 
(test (apply '+ (list 1 2)) 'error)
(test ((quote . #\h) (2 . #\i)) 'error)
(test ((quote "hi") 1) #\i)

(test (equal? '(1 2 '(3 4)) '(1 2 (3 4))) #f)
(test (equal? '(1 2 '(3 4)) (quote (list 1 2 (quote (list 3 4))))) #f)
(test (equal? (list-ref '(1 2 '(3 4)) 2) '(3 4)) #f)
(test (equal? '(1 2 '(3 4)) (list 1 2 (list 'quote (list 3 4)))) #t)
(test (equal? '(1 2 ''(3 4)) (list 1 2 (list 'quote (list 'quote (list 3 4))))) #t)
(test (equal? '('3 4) (list (list 'quote 3) 4)) #t)
(test (equal? '('3 4) (list 3 4)) #f)
(test (equal? '('() 4) (list (list 'quote '()) 4)) #t)
(test (equal? '('('4)) (list (list quote (list (list quote 4))))) #f)
(test (equal? '('('4)) (list (list 'quote (list (list 'quote 4))))) #t) 
(test (equal? '('('4)) '((quote ((quote 4))))) #t)
(test (equal? '1 ''1) #f)
(test (equal? ''1 ''1) #t)
(test (equal? '(1 '(1 . 2)) (list 1 (cons 1 2))) #f)
(test (equal? #(1 #(2 3)) '#(1 '#(2 3))) #f)
(test (equal? #(1) #('1)) #f)
(test (equal? #(()) #('())) #f)
(test (equal? cons 'cons) #f)

(test (eqv? #\a (quote #\a)) #t)
(test (eqv? 1 (quote 1)) #t)
(test (eqv? 0 (quote 0)) #t)
(test (equal? #(1 2 3) (quote #(1 2 3))) #t)
(test (eqv? 3.14 (quote 3.14)) #t)
(test (eqv? 3/4 (quote 3/4)) #t)
(test (eqv? 1+1i (quote 1+1i)) #t)
(test (eq? #f (quote #f)) #t)
(test (eq? #t (quote #t)) #t)
(test (eq? '() (quote ())) #t)
(test (equal? '(1 2 3) (quote (1 2 3))) #t)
(test (equal? '(1 . 2) (quote (1 . 2))) #t)
(test ('abs -1) 'error)
(test ('"hi" 0) #\h)

(test (''begin 1) 'begin)
(test (''let ((x 1)) ('set! x 3) x) 'error)
(test ('and #f) 'error)
(test ('and 1 #f) 'error)
(test ('begin 1) 'error)
(test ('cond ('define '#f)) 'error)
(test ('let ((x 1)) ('set! x 3) x) 'error)
(test ('let* () ('define x 3) x) 'error)
(test ('or #f) 'error)
(test ('quote 3) 'error)
(test ((copy quote) 1) 1)
(test ((copy quote) quote) 'quote)
(test ((lambda (q) (let ((x 1)) (q x))) quote) 'x) ; these two are strange -- not sure about them, but Guile 1.8 is the same
(test ((lambda (s c) (s c)) quote #f) 'c)
;;; ((lambda (lambda) (lambda (else))) quote) -> '(else)
(test ((quote and) #f) 'error)
(test ((values quote) 1) 1)

;; see also quasiquote




;;; --------------------------------------------------------------------------------
;;; for-each
;;; --------------------------------------------------------------------------------

(test (let ((v (make-vector 5))) (for-each (lambda (i) (vector-set! v i (* i i))) '(0 1 2 3 4)) v) #(0 1 4 9 16))
(test (let ((ctr 0) (v (make-vector 5))) (for-each (lambda (i) (vector-set! v ctr (* i i)) (set! ctr (+ ctr 1))) '(0 1 2 3 4)) v) #(0 1 4 9 16))
(for-each (lambda (x) (display "for-each should not have called this")) ())
(test (let ((ctr 0)) (for-each (lambda (x y) (if (= x y) (set! ctr (+ ctr 1)))) '(1 2 3 4 5 6) '(2 3 3 4 7 6)) ctr) 3)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5)) ctr) 15)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) '(5)) ctr) 9)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) () () ()) ctr) 0)
(test (let () (for-each abs '(1 2)) 1) 1)
(test (let ((ctr 0)) (for-each (lambda (a) (for-each (lambda (b) (set! ctr (+ ctr 1))) '(0 1))) '(2 3 4)) ctr) 6)
(test (let ((sum 0)) (for-each (lambda args (set! sum (+ sum (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) sum) 72)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2) '(2 1 0)) sum) 6)
(test (let () (for-each + '(0 1 2) '(2 1 0)) 0) 0)
(test (let () () ()) ())
(test (for-each + ()) #<unspecified>)
(test (let ((sum 0)) (for-each (lambda a (set! sum (+ sum (apply + a)))) '(1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda* ((a 1)) (set! sum (+ sum a))) '(1 2 3)) sum) 6)
(test (let ((sum 0)) (for-each (lambda (a . b) (set! sum (+ sum a))) '(1 2 3)) sum) 6)
(test (let ((sum 0) (lst (list 1 2 3))) (for-each (lambda (a b c) (set! sum (+ sum a b c))) lst lst lst) sum) 18)
(test (let ((sum 0) (lst (vector 1 2 3))) (for-each (lambda (a b c) (set! sum (+ sum a b c))) lst lst lst) sum) 18)
(test (let ((v (vector 1 2 3))) (for-each vector-set! (list v v v) (list 0 1 2) (list 32 33 34)) v) #(32 33 34))
(test (let () (define (hi) (for-each (lambda (x) (+ x 1)) (list 1 2 3))) (hi) (hi)) #<unspecified>)

(test (let ((d 0))
	(for-each (let ((a 0))
		    (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		    (lambda (c) (set! d (+ d c a))))
		  (list 3 4 5))
	d)
      21)
(test (let ((d 0))
	(for-each (lambda (c)
		    (let ((a 0))
		      (for-each (lambda (b) (set! a (+ a b))) (list 1 2))
		      (set! d (+ d a c))))
		  (list 3 4 5))
	d)
      21)

(test (let ((ctr 0)) 
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (> a 3) (exit a)) 
				  (set! ctr (+ ctr 1))) 
				(list 0 1 2 3 4 5)))))) 
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (for-each 
	  (lambda (x) 
	    (if (negative? x) (exit x)))
	  '(54 0 37 -3 245 19))
	 #t))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst ()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (for-each (lambda (a) 
				  (if (and (not cont) (= a 2))
				      (exit a)) 
				  (if (and cont (= a 5)) 
				      (exit a))
				  (call/cc (lambda (c) (set! cont c)))
				  (set! lst (cons ctr lst))
				  (set! ctr (+ ctr 1)))
				(list 0 1 2 3 4 5)))))) 
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(test (let ((lst ())) 
	(for-each (lambda (a) (set! lst (cons a lst))) 
		  (let ((lst ())) 
		    (for-each (lambda (b) (set! lst (cons b lst))) 
			      (list 1 2 3)) 
		    lst)) 
	lst) 
      (list 1 2 3))

;;; this is an infinite loop?
					; (let ((cont #f)) (call/cc (lambda (x) (set! cont x))) (for-each cont (list 1 2 3)))
(test (call/cc (lambda (x) (for-each x (list 1 2 3)))) 1) ; map also gives 1 ... perhaps not actually legal?

(let ((args (list 0 1 2))
      (xx (list 4)))
  (define (it1)
    (for-each
     (lambda (x)
       (catch #t
	 (lambda ()
	   (set-car! xx x))
	 (lambda any 'error)))
     (cdr args))
    (car xx))
  (test (it1) 2))

(let ((args (list 0 1 2))
      (xx (list 4)))
  (define (it1)
    (for-each
     (lambda (x)
       (catch #t
	 (lambda ()
	   (set-car! xx x))
	 (lambda any 'error))
       (set-car! xx (+ (car xx) 32)))
     (cdr args))
    (car xx))
  (test (it1) 34))

(test (let ((ctr 0))
	(for-each 
	 (lambda (x)
	   (for-each
	    (lambda (x y)
	      (for-each 
	       (lambda (x y z)
		 (set! ctr (+ x y z)))
	       (list x (+ x 1))
	       (list y (+ y 2))
	       (list (+ x y) (- x y))))
	    (list (+ x 3) (+ x 4) (+ x 5))
	    (list (- x 3) (- x 4) (- x 5))))
	 (list 1 2 3 4 5))
	ctr)
      23)

(for-each
 (lambda (a)
   (if (not (string=? a "hi"))
       (format-logged #t "yow: ~S" a)))
 (list "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi" "hi"))


;; now some mixed cases
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (list 1 2) (vector 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m) (set! sum (+ sum n m))) (vector 1 2) (list 3 4)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m))) (vector 1 2) (list 3 4) (vector 5 6)) sum) 10)
(test (let ((sum 0)) (for-each (lambda (n m p) (if (char=? p #\x) (set! sum (+ sum n m)))) (vector 1 2 3) (list 3 4 5) "xax") sum) 12)

(test (let* ((x (list (list 1 2 3))) (y (apply for-each abs x))) x) '((1 2 3)))

(test (for-each (lambda (x) (display "for-each should not have called this"))) 'error)
(test (for-each (lambda () 1) ()) 'error) ; #<unspecified>
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(1) '(3) ()) ctr) 0)
(test (let ((ctr 0)) (for-each (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6)) ctr) 15)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) 'error) ; #<unspecified>
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (for-each (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) #<unspecified>)
(test (for-each (lambda (a b) (+ a b)) (cons 1 2) (list 1 2)) #<unspecified>)
(test (for-each (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (for-each (lambda (a) (+ a 1)) #\a) 'error)
(test (for-each (lambda (a) (+ a 1)) (cons 1 2)) #<unspecified>)
(test (for-each (lambda (x) x) (openlet (inlet 'make-iterator (lambda (v) v)))) 'error)
(test (for-each (lambda (x) x) (openlet (inlet 'make-iterator (let ((iterator? #t)) (lambda (v) v))))) 'error)
(test (let ((sum 0)) (for-each (lambda (a b . args) (set! sum (+ sum a b (apply + args)))) '(0 1 2)) sum) 'error)
(test (for-each (lambda (a) a) '(1 2 . 3)) #<unspecified>)
(test (for-each #(0 1 2) #(2 1 0)) #<unspecified>)
(for-each
 (lambda (arg)
   (test (for-each arg (list 1)) #<unspecified>))
 (list (list 1 2 3) #(1 2 3) "hi"))

(for-each
 (lambda (op)
   (test (for-each op ()) 'error)
   (test (for-each op "") 'error)
   (test (for-each op #(1 2 3) ()) 'error)
   (test (for-each op #() (list) (string)) 'error))
 (list 0 () #f #t 'a-symbol :hi #\a #<eof> #<unspecified> #<undefined> 0.0 1+i 1/2 1/0 0/0 *stdout* (current-input-port)))
(for-each
 (lambda (arg)
   (test (for-each arg (list 1)) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (for-each (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (for-each (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(test (for-each) 'error)
(test (for-each #t) 'error)
(test (for-each map #t) 'error)

(test (for-each abs () abs) 'error)
(test (for-each abs '(1) #(1)) 'error)
(test (let ((vals ())) (for-each for-each (list (lambda (a) (set! vals (cons (abs a) vals)))) (list (list -1 -2))) vals) '(2 1))
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "a") c) #\a)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) "") c) #f)
(test (let ((c #f)) (for-each (lambda (x) (set! c x)) (string #\null)) c) #\null)

(test (let ((L (list 1 2 3 4 5)) (sum 0)) (for-each (lambda (x) (set-cdr! (cddr L) 5) (set! sum (+ sum x))) L) sum) 6)
; map (below) has more tests along this line
(test (let ((f #f)) (for-each (lambda (a) (if (eq? a 'a) (set! f (lambda () a)))) '(a b c)) (f)) 'a)
(test (let ((i 0) (f (make-vector 3))) (for-each (lambda (b) (vector-set! f i b) (set! i (+ i 1))) '(a b c)) f) #(a b c))
(test (let ((i 0) (f (make-vector 3)) (lst '(a b c))) (define (hi) (for-each (lambda (b) (vector-set! f i b) (set! i (+ i 1))) lst)) (hi) f) #(a b c))
(test (let ((i 0) (f (make-vector 3)) (lst '(a b c))) (define (hi) (for-each (lambda (b) (let () (vector-set! f i b) (set! i (+ i 1)))) lst)) (hi) f) #(a b c))
(test (let ((i 0) (f (make-vector 3)) (lst (list 1 2 3))) (define (hi) (for-each (lambda (b) (vector-set! f i (let ((b (+ b 1))) b)) (set! i (+ i 1))) lst)) (hi) f) #(2 3 4))
(test (let ((i 0) (f (make-vector 3)) (lst (list 1 2 3))) (define (hi) (for-each (lambda (b) (let ((b (+ b 1))) (vector-set! f i (let ((b (+ b 1))) b)) (set! i (+ i 1)))) lst)) (hi) f) #(3 4 5))
(test (let ((f #f)) (define (hi) (for-each (lambda (a) (if (eq? a 'a) (set! f (lambda () (let () a))))) '(a b c))) (hi) (f)) 'a)
(test (let ((lst '((a b c) (1 2 3)))) (define (hi) (for-each (lambda (a) a) (apply values lst))) (hi)) 'error)
(test (let ((lst ())) (for-each (lambda args (set! lst (cons args lst))) (values (list 1 2 3) '(4 5 6) (list 7 8 9))) lst) '((3 6 9) (2 5 8) (1 4 7)))

(test (for-each ="") 'error) ; #<unspecified>
(test (for-each =""=) 'error)
(test (for-each = "" 123) 'error)
(test (for-each = () 123) 'error)
(test (for-each =()=) 'error)
(test (for-each abs "") #<unspecified>)
(test (for-each null? () #() "") 'error) ; #<unspecified>
(test (for-each null? () #() 0 "") 'error)
(test (for-each define '(a) '(3)) #<unspecified>)
(test (let () (for-each define '(a b c) '(1 2 3)) (list a b c)) '(1 2 3))
(test (let () (for-each define '(first second third fourth) '(car cadr caddr cadddr)) (third '(1 2 3 4 5))) 3)
(test (for-each '(()) #()) #<unspecified>)
(test (for-each '(1 2 . 3) '(1 . 2)) #<unspecified>)
(test (for-each '(()) ()) #<unspecified>)
(test (for-each #2D((1 2) (3 4)) '(1)) #<unspecified>)
(test (for-each "a\x00b" #(1 2)) #<unspecified>)
(test (for-each #(1 (3)) '(1)) #<unspecified>)
(test (for-each '((1 (2)) (((3) 4))) '(1)) #<unspecified>)
(test (for-each "hi" '(1)) #<unspecified>)
(test (for-each #() #()) 'error) ; #<unspecified>
(test (for-each '(1 . 2) #()) #<unspecified>)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (for-each ht ht)) #<unspecified>)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (let ((sum 0)) (for-each (lambda (c) (set! sum (+ sum (cdr c)))) ht) sum)) 3)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (for-each ht '(a b))) #<unspecified>)
(test (for-each ''2 '(1)) #<unspecified>)
(let ((os (*s7* 'safety)))
  (set! (*s7* 'safety) 1)
  (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (for-each lst lst) #<unspecified>)) ; 'error
  (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (for-each #() lst) 'error))
  (set! (*s7* 'safety) os))
(test (for-each 1 "hi" ()) 'error)
(test (for-each 0 #() ()) 'error)
(test (for-each #\a #(1 2) '(3 4) "") 'error)
(test (for-each '2 ()) 'error)
(test (let ((a 1) (b 2)) (for-each apply (list set! set!) '(a b) '((12) (32))) (list a b)) '(12 32))
(test (let ((a 1) (b 2) (c 3)) (for-each apply (make-list 3 set!) '(a b c) '((12) (32) (0))) (list a b c)) '(12 32 0))
(test (let ((a 1) (b 2) (c 3)) (for-each set! '(a b c) '(12 32 0)) (list a b c)) '(12 32 0))

(let ()
  (define (hi)
    (let ((lst '(1 2 3)))
      (for-each
       (lambda (x)
	 (catch #t
	   (lambda ()
	     (if (defined? 'local-x)
		 (format-logged #t ";for-each catch local env not cleared: ~A~%" local-x))
	     (define local-x x)
	     local-x)
	   (lambda args #f)))
       lst)))
  (hi)
  (hi))

;; this caught me -- Guile returns 6 3 also
(test (let ((fnc #f))
	(for-each
	 (let ((ctr 0))
	   (lambda (x)
	     (if (= ctr 3)
		 (set! fnc (lambda () ctr)))
	     (set! ctr (+ ctr 1))))
	 '(1 2 3 4 5 6))
	(fnc))
      6)

(test (let ((fnc #f))
	(for-each
	 (let ((ctr 0))
	   (lambda (x)
	     (if (= ctr 3)
		 (set! fnc (let ((local-ctr ctr))
			     (lambda () local-ctr))))
	     (set! ctr (+ ctr 1))))
	 '(1 2 3 4 5 6))
	(fnc))
      3)

(let ((x 0))
  (let ((p1 (dilambda (lambda (a) (set! x (+ x a))) (lambda (a b) (+ a b)))))
    (for-each p1 '(1 2 3))
    (test x 6))
  (set! x 0)
  (for-each (lambda args (set! x (+ x (car args)))) '(1 2 3))
  (test x 6)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a))) '(1 2 3))
  (test x 6)
  (set! x 0)
  (for-each (lambda args (set! x (+ x (car args) (cadr args)))) '(1 2 3) '(3 2 1))
  (test x 12)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a b))) '(1 2 3) '(3 2 1))
  (test x 12)
  (set! x 0)
  (for-each (lambda* (a (b 2)) (set! x (+ x a b))) '(1 2 3))
  (test x 12))

(test (let ((lst '(1 2 3)) (sum 0)) (define-macro (hi a) `(set! sum (+ sum (+ 1 ,a)))) (for-each hi lst) sum) 9)
(test (let ((lst '(1 2 3)) (sum 0)) (define-bacro (hi a) `(set! sum (+ sum (+ 1 ,a)))) (for-each hi lst) sum) 9)
  
(let ((sum 0))
  (define (and-for-each func . args)
    ;; apply func to first of each arg, stopping if func returns #f
    (call-with-exit
     (lambda (quit)
       (apply for-each 
	      (lambda arglist
		(if (not (apply func arglist))
		    (quit #<unspecified>)))
	      args))))
  
  (test (and-for-each (lambda (arg) 
			(and (not (null? arg))
			     (set! sum (+ sum arg))))
		      (list 1 2 () 3 4))
	#<unspecified>)
  (test sum 3)
  (set! sum 0)
  
  (and-for-each (lambda (arg) 
		  (and (not (null? arg))
		       (set! sum (+ sum arg))))
		(list 1 2 3 4))
  (test sum 10)
  (set! sum 0)
  
  (and-for-each (lambda (arg1 arg2) 
		  (and (not (null? arg1))
		       (not (null? arg2))
		       (set! sum (+ sum arg1 arg2))))
		(list 1 2 3 4)
		(list 5 6 () 7 8))
  (test sum 14))


(define (and-map func . args) ; see stuff.scm for a better version
  (call-with-exit
   (lambda (quit)
     (let ((result ()))
       (apply for-each 
	      (lambda arglist
		(let ((val (apply func arglist)))
		  (if (not val)
		      (quit (reverse result))
		      (set! result (cons val result)))))
	    args)
       (reverse result)))))

(test (and-map even? '(0 2 4 5 6)) '(#t #t #t))

(define (find-if f . args)
  (call-with-exit
   (lambda (return) 
     (apply for-each (lambda main-args 
		       (if (apply f main-args) 
			   (apply return main-args)))
	    args))))

(test (find-if even? #(1 3 5 2)) 2)
(test (* (find-if > #(1 3 5 2) '(2 2 2 3))) 6)

(define (position-if f . args)
  (let ((pos 0))
    (call-with-exit
     (lambda (return) 
       (apply for-each (lambda main-args 
			 (if (apply f main-args) 
			     (return pos))
			 (set! pos (+ pos 1)))
	    args)))))

(test (position-if even? #(1 3 5 2)) 3)
(test (position-if > #(1 3 5 2) '(2 2 2 3)) 1)

(let ((summer (lambda (v)
		(let ((sum 0))
		  (do ((i 0 (+ i 1)))
		      ((= i 10) sum)
		    (set! sum (+ sum ((v i)))))))))
		
  (test (let ((saved-args (make-vector 10))
	      (i 0))
	  (for-each
	   (lambda (arg)
	     (set! (saved-args i) arg)
	     (set! i (+ i 1)))
	   (list 0 1 2 3 4 5 6 7 8 9))
	  (set! (saved-args 0) 32)
	  saved-args)
	#(32 1 2 3 4 5 6 7 8 9))
  
  (test (let ((f #f))
	  (for-each
	   (lambda (i)
	     (let ()
	       (define (x) i)
	       (if (= i 1) (set! f x))))
	   (list 0 1 2 3))
	  (f))
	1)
  
  (test (let ((saved-args (make-vector 10))
	      (i 0))
	  (for-each
	   (lambda (arg)
	     (set! (saved-args i) (lambda () arg))
	     (set! i (+ i 1)))
	   (list 0 1 2 3 4 5 6 7 8 9))
	  (summer saved-args))
	45)
  
  (test (let ((saved-args (make-list 10))
	      (i 0))
	  (for-each
	   (lambda (arg)
	     (list-set! saved-args i (lambda () arg))
	     (set! i (+ i 1)))
	   (list 0 1 2 3 4 5 6 7 8 9))
	  (summer saved-args))
	45)
  
;;; these are the same but use map
  (test (let ((saved-args (make-vector 10))
	      (i 0))
	  (map
	   (lambda (arg)
	     (set! (saved-args i) arg)
	     (set! i (+ i 1)))
	   (list 0 1 2 3 4 5 6 7 8 9))
	  (set! (saved-args 0) 32)
	  saved-args)
	#(32 1 2 3 4 5 6 7 8 9))
  
  (test (let ((f #f))
	  (map
	   (lambda (i)
	     (let ()
	       (define (x) i)
	       (if (= i 1) (set! f x))))
	   (list 0 1 2 3))
	  (f))
	1)
  
  (test (let ((saved-args (make-vector 10))
	      (i 0))
	  (map
	   (lambda (arg)
	     (set! (saved-args i) (lambda () arg))
	     (set! i (+ i 1)))
	   (list 0 1 2 3 4 5 6 7 8 9))
	  (summer saved-args))
	45)
  
  ;; and again but with named let
  (test (let ((saved-args (make-vector 10)))
	  (let runner ((arg 0))
	    (set! (saved-args arg) arg)
	    (if (< arg 9)
		(runner (+ arg 1))))
	  (set! (saved-args 0) 32)
	  saved-args)
	#(32 1 2 3 4 5 6 7 8 9))
  
  (test (let ((f #f))
	  (let runner ((i 0))
	    (let ()
	      (define (x) i)
	      (if (= i 1) (set! f x))
	      (if (< i 3)
		  (runner (+ i 1)))))
	  (f))
	1)
  
  (test (let ((saved-args (make-vector 10)))
	  (let runner ((i 0))
	    (set! (saved-args i) (lambda () i))
	    (if (< i 9)
		(runner (+ i 1))))
	  (summer saved-args))
	45)
  
  
;;; and recursion
  (test (let ((saved-args (make-vector 10)))
	  (define (runner arg)
	    (set! (saved-args arg) arg)
	    (if (< arg 9)
		(runner (+ arg 1))))
	  (runner 0)
	  (set! (saved-args 0) 32)
	  saved-args)
	#(32 1 2 3 4 5 6 7 8 9))
  
  (test (let ((f #f))
	  (define (runner i)
	    (let ()
	      (define (x) i)
	      (if (= i 1) (set! f x))
	      (if (< i 3)
		  (runner (+ i 1)))))
	  (runner 0)
	  (f))
	1)
  
  (test (let ((saved-args (make-vector 10)))
	  (define (runner i)
	    (set! (saved-args i) (lambda () i))
	    (if (< i 9)
		(runner (+ i 1))))
	  (runner 0)
	  (summer saved-args))
	45)
  
  
;;; and member/assoc
  (test (let ((saved-args (make-vector 10)))
	  (member 'a '(0 1 2 3 4 5 6 7 8 9) 
		  (lambda (a b)
		    (set! (saved-args b) (lambda () b))
		    #f))
	  (summer saved-args))
	45)
  
  (test (let ((saved-args (make-vector 10)))
	  (assoc 'a '((0 b) (1 b) (2 b) (3 b) (4 b) (5 b) (6 b) (7 b) (8 b) (9 b))
		 (lambda (a b)
		   (set! (saved-args b) (lambda () b))
		   #f))
	  (summer saved-args))
	45)
  
  (test (let ((saved-args (make-vector 10 #f)))
	  (sort! '(3 2 1 4 6 5 9 8 7 0)
		 (lambda (a b)
		   (if (not (saved-args b))
		       (set! (saved-args b) (lambda () b)))
		   (< a b)))
	  (summer saved-args))
	45)
  
;;; and do which has never worked in this way
#|  
  (test (let ((saved-args (make-vector 10)))
	  (do ((i 0 (+ i 1)))
	      ((= i 10))
	    (set! (saved-args i) (lambda () i)))
	  (summer saved-args))
	45)
|#
  )

;;; originally for-each/map used old_frame_with_slot, but if the closure had
;;;   a local define, the arg symbol was not updated (let_id), leading to segfaults
;;;   if the optimizer thought checks were unneeded.  So...

(let ()
  (define (f1) (for-each (lambda (f3) (let ((x 0)) (if (> f3 x) (abs f3)))) (list 1 2))) (f1)
  (define (f2) (for-each (lambda (f3) (define x 0) (if (> f3 x) (abs f3))) (list 1 2))) (f2)
  (let ((f (let flet ((x 1)) (if (> x 0) (flet (- x 1))) flet))) (map f (list 1 2)))
  (define (f4) (map (let flet ((x 1)) (if (> x 0) (flet (- x 1))) flet) (list 1 2))) (f4)
  (define (f4a) (for-each (let flet ((x 1)) (if (> x 0) (flet (- x 1))) flet) (list 1 2))) (f4a)
  (define (f4b) (for-each (let flet ((x 1)) (define y 0) (if (> x y) (flet (- x 1))) flet) (list 1 2))) (f4b)
  (define (f5) (for-each (let () (define (f x) (if (> x 0) (f (- x 1)))) f) (list 1 2))) (f5)
  (define (f5a) (for-each (let () (define (f x) (define y 0) (if (> x y) (f (- x 1)))) f) (list 1 2))) (f5a)
  (define (f5b) (for-each (lambda (x) (define z 2) (define y 0) (if (> z x y) (display (- x 1)))) (list 1 2))) (f5b)
  (define (f6) (map (let () (define (f x) (if (> x 0) (f (- x 1)) x)) f) (list 1 2))) (test (f6) '(0 0))
  (define (f7) (map (let () (define (f x) (define y (+ x 1)) (if (> y 0) (f (- x 1)) x)) f) (list 1 2))) (test (f7) '(-1 -1))
  )



;;; --------------------------------------------------------------------------------
;;; map
;;; --------------------------------------------------------------------------------

(test (map cadr '((a b) (d e) (g h))) '(b e h))
(test (map (lambda (n) (expt n n)) '(1 2 3 4 5)) '(1 4 27 256 3125))
(test (map + '(1 2 3) '(4 5 6)) '(5 7 9))

(test (apply vector (map (lambda (i) (* i i)) '(0 1 2 3 4))) #(0 1 4 9 16))
(map (lambda (x) (display "map should not have called this")) ())
(test (let ((ctr 0)) (map (lambda (x y) (if (= x y) (set! ctr (+ ctr 1))) ctr) '(1 2 3 4 5 6) '(2 3 3 4 7 6))) (list 0 0 1 2 2 3))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(0 1) '(2 3) '(4 5))) (list 6 15))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) '(5))) (list 9))
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) () () ())) ())
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2)) (list 2 4))
(test (map abs '(1 -2)) (list 1 2))
(test (map + '(0 1 2) '(2 1 0) '(3 4 5) '(5 4 3) '(6 7 8) '(8 7 6)) (list 24 24 24))
(test (map (lambda (a) (cons a (map (lambda (b) (+ b 1)) (list 0 1 2)))) (list 3 4 5)) '((3 1 2 3) (4 1 2 3) (5 1 2 3)))
(test (map (lambda (a) (+ a 1)) (map (lambda (b) (+ b 1)) (map (lambda (c) (+ c 1)) (list 0 1 2)))) '(3 4 5))
(test (map (lambda args (apply + args)) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5) '(6 7 8) '(9 10 11) '(12 13 14)) '(30 35 40))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2) '(3 4 5)) '(3 5 7))
(test (map (lambda args args) '(1 2 3)) '((1) (2) (3)))
(test (map + () ()) ())
(test (map + (#(#() #()) 1)) ())
(test (map + #(1) #(1) #(1)) '(3))
(test (map list '(a b c)) '((a) (b) (c)))
(test (map (lambda (a b) (- a b)) (list 1 2) (vector 3 4)) '(-2 -2))
(test (map (lambda (a b c) (if (char=? a #\a) (+ b c) (- b c))) "axa" (list 1 2 3) (vector 4 5 6)) '(5 -3 9))
(test (map vector (memv 1 (list 1 2 3))) '(#(1) #(2) #(3)))
(test (map append #(1 2 3)) '(1 2 3))
(test (map eval '((+ 1 2) (* 3 4))) '(3 12))
(test (map (map + (list 1 2 3)) (list 0 1 2)) '(1 2 3))
(test (let ((a #t) (b #f) (c #t)) (map when '(a b c) '(12 32 0))) '(12 #<unspecified> 0))
(test (let ((a #t) (b #f)) (map if '(a b) '(1 2) '(3 4))) '(1 4))
(test (let ((a #t) (b #f)) (map unless '(a b) '(1 2))) '(#<unspecified> 2))
(test (let ((a #t) (b #f)) (list (map set! '(a b) '(1 2)) a b)) '((1 2) 1 2))
(test (let ((a #t) (b #f)) (map begin '(a b))) '(#t #f))
(test (let () (map apply (map lambda '(a b) '((car a) (car b))) '((2) (3)))) '(2 3))
(test (let () (map apply (map lambda* '(((a 1)) ((b 2))) '(a b)) '((3) ()))) '(3 2))
(test (map + '(1 2 3) '(4 5 6) '(7 8 9)) '(12 15 18))

(test (let* ((x (list (list 1 2 3))) (y (apply map abs x))) (list x y)) '(((1 2 3)) (1 2 3)))
(test (let* ((x (quote ((1 2) (3 4)))) (y (apply map ash x))) (list x y)) '(((1 2) (3 4)) (8 32)))
(test (let* ((x (quote ((1 2 3) (4 5 6) (7 8 9)))) (y (apply map + x))) (list x y)) '(((1 2 3) (4 5 6) (7 8 9)) (12 15 18)))
(test (map * (map + '(1 2 3) '(4 5 6)) '(1 2 3)) '(5 14 27))
(test (apply map * (apply map + '(1 2 3) '((4 5 6))) '((1 2 3))) '(5 14 27))
(test (let* ((x (lambda () '(1 2 3))) (y (apply map - (list (x))))) (x)) '(1 2 3))

;(test (map car (list (list 0) (list (values)) (list 2))) (map (lambda (x) (car x)) (list (list 0) (list (values)) (list 2))))
(test (apply append (map list '((a . 1) (b . 2) #<eof>))) '((a . 1) (b . 2) #<eof>))
(test (apply append (map list '(a b #<eof> d))) '(a b #<eof> d))
(test (map values (vector 1 2 #<eof> 3)) '(1 2 #<eof> 3))

(test (let ((d 0))
	(map (let ((a 0))
	       (map (lambda (b) (set! a (+ a b))) (list 1 2))
	       (lambda (c) (set! d (+ d c a)) d))
	     (list 3 4 5)))
      (list 6 13 21))
(test (let ((d 0))
	(map (lambda (c)
	       (let ((a 0))
		 (map (lambda (b) (set! a (+ a b))) (list 1 2))
		 (set! d (+ d a c))
		 d))
	     (list 3 4 5)))
      (list 6 13 21))

(test (let ((ctr 0))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (> a 3) (exit a)) 
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (list ctr val)))
      (list 4 4))

(test (call-with-current-continuation
       (lambda (exit)
	 (map 
	  (lambda (x) 
	    (if (negative? x) (exit x))
	    x)
	  '(54 0 37 -3 245 19))))
      -3)

(test (let ((ctr 0)
	    (cont #f)
	    (lst ()))
	(let ((val (call/cc 
		    (lambda (exit) 
		      (map (lambda (a) 
			     (if (and (not cont) (= a 2))
				 (exit a)) 
			     (if (and cont (= a 5)) 
				 (exit a))
			     (call/cc (lambda (c) (set! cont c)))
			     (set! lst (cons ctr lst))
			     (set! ctr (+ ctr 1))
			     ctr)
			   (list 0 1 2 3 4 5))))))
	  (if (< val 5)
	      (cont))
	  (list ctr val lst)))
      (list 5 5 (list 4 3 2 1 0)))

(let ()
  (define (tree-add x lst)
    (define (tree-add-1 lst-1)
      (map (lambda (a)
	     (if (pair? a) (tree-add-1 a) (+ a x)))
	   lst-1))
    (tree-add-1 lst))
  (test (tree-add 12 '((1 2) ((3)) 4 5)) '((13 14) ((15)) 16 17)))

(test (map (lambda (a) a) (map (lambda (b) b) (list 1 2 3))) (list 1 2 3))
(test (map cons '(a b c) '(() () ())) '((a) (b) (c)))
(test (map (lambda a (list a)) '(1 2 3)) '(((1)) ((2)) ((3))))
(test (map (lambda* a (list a)) '(1 2 3)) '(((1)) ((2)) ((3))))
(test (map (lambda* (a) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda* ((a 0)) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda* ((a 0) (b 1)) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda (a . b) (list a)) '(1 2 3)) '((1) (2) (3)))
(test (map list '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (apply list a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (apply values a)) '(1 2 3)) '(1 2 3))
(test (map (lambda a (values a)) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda a (append a)) '(1 2 3)) '((1) (2) (3)))
(test (map values '(1 2 3)) '(1 2 3))
;(test ((lambda* ('a) quote) 1) 1)
(test (procedure? (car (map lambda '(()) '((1))))) #t)
(test (procedure? (car (map lambda '((x)) '(((+ x 1)))))) #t)
(test (map #(0 1 2) #(2 1 0)) '(2 1 0))
;(test (map quasiquote '((quasiquote 1) (quasiquote 2))) '(1 2)) -- this has changed (12-May-14)
(test (map (lambda (a b) (a b)) (map lambda '((x) (y) (z)) '((+ x x) (* y y) (expt z z))) (list 1 2 3)) '(2 4 27))
(test (map apply (map lambda '((x) (y) (z)) '((+ x x) (* y y) (expt z z))) '((1) (2) (3))) '(2 4 27))
(test (let () (define (add-some x) (define (add-some x) (+ x 2)) (+ x 1)) (map add-some '(1 2 3 4))) '(2 3 4 5)) ; from some CL website -- kinda ridiculous
(test (map gcd #(1 2)) '(1 2))
(test (apply vector (map values #(1 2) #(3 4))) #(1 3 2 4))
(test (map values '(1 2 3) '(4 5 6) '(7 8 9)) '(1 4 7 2 5 8 3 6 9))
(test (map eval (list (+ 1 2) (+ 3 4))) '(3 7))
(test (map apply (list + - * /) (list 1 2 3 4) '((5) (6) (7) (8))) '(6 -4 21 1/2))

;;; (let ((val ())) (list (map (lambda a (set! val (cons a val)) a) '(1 2 3)) val)) -> ((#3=(1) #2=(2) #1=(3)) (#1# #2# #3#))
(test (map if '(#f #f #t) '(0 1 2) '(3 4 5)) '(3 4 2))
(test (map apply (map lambda '(() (a) (a b)) '(1 (+ a 1) (+ a b 1))) '(() (2) (3 4))) '(1 3 8))
(test (map values (list 1 2 3) (list 4 5 6)) '(1 4 2 5 3 6))
(test (map map (list values) '((1 2)) '((3 4))) '((1 3 2 4)))
(test (map values '((1 2)) '((3 4))) '((1 2) (3 4)))
(test (map map (list map) (list (list values)) '(((1 2))) '(((3 4)))) '(((1 3 2 4))))
(test (map apply (list values) '(((1 2))) '(((3 4)))) (apply map values '(((1 2))) '(((3 4))))) ; !
(test (let ((x '((1 2)))) (eval `(apply apply values x)) (object->string x)) "((1 2))") ; not "((values 1 2))" -- 24-Aug-12
(test (map (lambda (x) x) #u8(255)) (list 255))

(let ()
  (define (shuffle . args) 
    (apply map values args))
  (test (shuffle '(1 2 3) #(4 5 6) '(7 8 9)) '(1 4 7 2 5 8 3 6 9))
  (test (shuffle '(1 2 3)) '(1 2 3))
  (test (shuffle '(1 2 3) '(4)) '(1 4))
  (test (shuffle '(1 2 3) ()) ())
  )

(test (map list "hi") '((#\h) (#\i)))
(test (map string "hi") '("h" "i"))
(test (map vector "hi") '(#(#\h) #(#\i)))
(test (map char-upcase "hi") '(#\H #\I))
(test (map append #(#() #())) '(#() #()))

(test (map abs () abs) 'error)
(test (map (lambda (x) (display "map should not have called this"))) 'error)
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z)) ctr) '(1) '(3) ())) ())
(test (let ((ctr 0)) (map (lambda (x y z) (set! ctr (+ ctr x y z))) '(0 1) '(2 3) '(4 5 6))) '(6 15))

(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list)) ())
(test (map (lambda (a b) (+ a b)) (list 1) (list 2)) (list 3))
(test (map (lambda (a b) (+ a b)) (list 1)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 2) (list 3)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1) (list 1 2)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2 3)) '(2 4))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1)) '(2))
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list)) 'error) ; ()
(test (map (lambda (a b) (+ a b)) (list 1 2) (list 1 2) (list 1 2)) 'error)
(test (map (lambda (a b) (+ a b)) (list 1 2) (cons 1 2)) '(2))

(test (map (lambda* (x . args) args) '(1 2 3)) '(() () ()))
(test (map (lambda (x . args) args) '(1 2 3)) '(() () ()))
(test (map (lambda* (x . args) (list x args)) '(1 2 3)) '((1 ()) (2 ()) (3 ())))
(test (map (lambda (x . args) (list x args)) '(1 2 3)) '((1 ()) (2 ()) (3 ())))
(test (map (lambda args args) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda* args args) '(1 2 3)) '((1) (2) (3)))
(test (map (lambda (x y . args) args) '(1 2 3)) 'error)
(test (map (lambda* (x y . args) args) '(1 2 3)) '(() () ())) ; all args are optional in lambda*
(test (map (lambda (x y . args) args) '(1 2 3) '(4 5 6)) '(() () ()))
(test (map (lambda* (x y . args) args) '(1 2 3) '(4 5 6)) '(() () ()))
(test (map (lambda (x y . args) (list x y args)) '(1 2 3) '(4 5 6)) '((1 4 ()) (2 5 ()) (3 6 ())))
(test (map (lambda* (x y . args) (list x y args)) '(1 2 3) '(4 5 6)) '((1 4 ()) (2 5 ()) (3 6 ())))
(test (map (lambda (x y . args) (list x y args)) '(1 2 3) '(4 5 6) '(7 8 9)) '((1 4 (7)) (2 5 (8)) (3 6 (9))))
(test (map (lambda* (x y . args) (list x y args)) '(1 2 3) '(4 5 6) '(7 8 9)) '((1 4 (7)) (2 5 (8)) (3 6 (9))))
(test (map (lambda* (x y :rest args) (list x y args)) '(1 2 3) '(4 5 6) '(7 8 9)) '((1 4 (7)) (2 5 (8)) (3 6 (9))))

(test (map (lambda . (x y z 8)) '(1 2 3))  'error) ; (y unbound) but other schemes ignore unused args
(test (map (lambda . (x 8)) '(1 2)) '(8 8)) 

(test (map (lambda (a) (+ a 1)) (list 1) (list 2)) 'error)
(test (map (lambda (a) (+ a 1)) #\a) 'error)
(test (map (lambda (a) (+ a 1)) (cons 1 2)) '(2))
(test (map (lambda (a b . args) (+ a b (apply + args))) '(0 1 2)) 'error)
(test (map (lambda (a) a) '(1 2 . 3)) '(1 2))
(test (map) 'error)
(test (map #t) 'error)
(test (map set-cdr! '(1 2 3)) 'error)
(test (map (lambda (a b) (set-cdr! a b) b) '((1) (2) (3)) '(4 5 6)) '(4 5 6))
(test (let ((str "0123")) (set! (str 2) #\null) (map append str)) '(#\0 #\1 #\null #\3))

(test (map ((lambda () abs)) '(-1 -2 -3)) '(1 2 3))
(test (apply map ((lambda () abs)) (list (list -1 -2 -3))) '(1 2 3))
(test (apply apply map ((lambda () abs)) (list (list (list -1 -2 -3)))) '(1 2 3))
(test (apply apply apply map ((lambda () abs)) '((((-1 -2 -3))))) '(1 2 3))
(test (apply apply apply (list (list map abs (list (list -1 -2 -3))))) '(1 2 3))
(test (apply apply list 1 '((1 2) (3 4))) '(1 (1 2) 3 4))
(test (apply + (apply apply apply (list (list map abs (list (list -1 -2 -3)))))) 6)
(test (apply (apply apply lambda '(a) '(((+ a 1)))) '(14)) 15)
(test (let ((a 14)) (apply apply quasiquote '(((+ ,a 1))))) '(+ 14 1))
(test (apply map vector (values (list (vector 1 2)))) '(#(1) #(2)))
(test (apply map string (list "123")) '("1" "2" "3"))
(test (apply map string '("123" "456")) '("14" "25" "36"))
(test (apply map list '((1 2) (3 4))) '((1 3) (2 4))) ; matrix transpose

;;; Is <code>(apply apply func arglist)</code> the same as <code>(apply func (apply values arglist))</code>,
;;; or (leaving aside <code>'(()))</code>, <code>(func (apply values (apply values arglist)))</code>?
(test (apply apply + '((1 2 3))) (apply + (apply values '((1 2 3)))))
(test (apply apply + '((1 2 3))) (+ (apply values (apply values '((1 2 3))))))

(test (map string "123") '("1" "2" "3"))
(test (map "hi" '(0 1)) '(#\h #\i))
(test (map (list 2 3) '(0 1)) '(2 3))
(test (map #(2 3) '(1 0)) '(3 2))
(for-each
 (lambda (arg)
   (test (map arg (list 1)) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (map (lambda (n m) n) (list 1) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))
(for-each
 (lambda (arg)
   (test (map (lambda (a) a) arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #f #t))

(let ()
  (define (concatenate . args)
    (apply append (map (lambda (arg) (map values arg)) args)))
  (test (concatenate "hi" #(#\h #\o)) '(#\h #\i #\h #\o))
  (test (let ((lst (concatenate '(1 2) (let ((a 2) (b 3)) (curlet)) (hash-table* 'c 4))))
	  (or (equal? lst '(1 2 (b . 3) (a . 2) (c . 4)))
	      (equal? lst '(1 2 (a . 2) (b . 3) (c . 4)))))
	#t))

(test (map (lambda (a1 a2 a3 a4 a5 a6 a7 a8 a9 a10)
	     (max a1 a2 a3 a4 a5 a6 a7 a8 a9 a10))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))
  
(test (map (lambda (a1 a2 a3 a4 a5 a6 a7 a8 a9 . a10)
	     (apply max a1 a2 a3 a4 a5 a6 a7 a8 a9 a10))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))

(test (map (lambda* (a1 a2 a3 . a10)
	     (apply max a1 a2 a3 a10))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))

(test (map (lambda args
	     (apply max args))
	   (list 6 7 8 9 10)
	   (list 21 22 23 24 25)
	   (list 16 17 18 19 20)
	   (list 11 12 13 14 15)
	   (list 26 27 28 29 30)
	   (list 1 2 3 4 5)
	   (list 36 37 38 39 40)
	   (list 41 42 43 44 45)
	   (list 46 47 48 49 50)
	   (list 31 32 33 34 35))
      (list 46 47 48 49 50))
  
(test (map map (list abs) (list (list -1))) '((1)))
(test (map map (list map) (list (list abs)) (list (list (list -1)))) '(((1))))
(test (map map (list map) (list (list map)) (list (list (list abs))) (list (list (list (list -1 -3))))) '((((1 3)))))
(test (map map (list lcm) (vector #(1 2))) '((1 2)))
(test (map map (list integer?) (list (vector "hi" 1 2/3))) '((#f #t #f)))
(test (map map (list char-lower-case?) (list "hAba")) '((#t #f #t #t)))
(test (map map (list char-lower-case? char-upper-case?) (list "hAba" "HacDf")) '((#t #f #t #t) (#t #f #f #t #f)))
(test (map map (list + -) (list (list 1 2) (list 3 4))) '((1 2) (-3 -4)))
(test (map map (list map map) (list (list + -) (list - +)) '(((1 2) (3 4)) ((4 5) (6 7)))) '(((1 2) (-3 -4)) ((-4 -5) (6 7))))

(test (map member (list 1 2 3) (list (list 1 2 3) (list 1 3 4) (list 3 4 5))) '((1 2 3) #f (3 4 5)))
(test (map - (list 1 2 3) (list 1 2 3) (list 1 3 4) (list 3 4 5)) '(-4 -7 -9))
(test (map - (list 1 2 3) (list 1 2 3 'hi) (list 1 3 4 #\a "hi") (list 3 4 5)) '(-4 -7 -9))
(test (let () (define (mrec a b) (if (<= b 0) (list a) (map mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))
(test (map append '(3/4)) '(3/4))
(test (map list '(1.5)) '((1.5)))
(test (map vector '("hi")) '(#("hi")))
(test (map object->string '(:hi (1 2) (()))) '(":hi" "(1 2)" "(())"))
(test (map map (list for-each) (list (list abs)) (list (list (list 1 2 3)))) '((#<unspecified>)))
(test (map map (list vector) '((#(1 #\a (3))))) '((#(#(1 #\a (3))))))
(test (apply map map (list cdr) '((((1 2) (3 4 5))))) '(((2) (4 5))))
(test (apply map map (list char-upcase) '(("hi"))) '((#\H #\I)))
(test (apply map map (list *) '(((1 2)) ((3 4 5)))) '((3 8))) ; (* 1 3) (* 2 4)
(test (map apply (list map) (list map) (list (list *)) '((((1 2)) ((3 4 5))))) '(((3 8))))
(test (map map (list magnitude) '((1 . 2))) '((1))) ; magnitude is called once with arg 1
(test (map magnitude '(1 . 2)) '(1))
(test (map call/cc (list (lambda (r1) 1) (lambda (r2) (r2 2 3)) (lambda (r3) (values 4 5)))) '(1 2 3 4 5))
(test (map call/cc (list number? continuation?)) '(#f #t))

;; from scheme working group 
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) 5) x) L)) '(1 2 3))
(test (let ((L (list 1 2))) (map (lambda (x) (set! (cdr (cdr L)) L) x) L)) '(1)) ;'(1 2)
(test (let ((L (list 1 2))) (object->string (map (lambda (x) (set! (car (cdr L)) L) x) L))) "(1 #1=(1 #1#))")
;;;(test (let ((L (list 1 2))) (map (lambda (x) (set-cdr! L L) x) L)) '(1 2)) ;?? this depends on when we cdr? infinite loop in Guile
;;;(let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! L ()) x) L)) ; another similar case -- s7 doesn't notice what happened
;;;  does that mean a GC during this map would leave us accessing freed memory? 
;;;  I think not because the original list is held by map (eval) locals that are protected
;;;  we simply stepped on something after looking at it, similar to:
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-car! L 123) x) L)) '(1 2 3 4 5))
(test (let ((L (list 1 2 3 4 5))) (map (lambda (x) (set-cdr! (cddr L) (list 6 7 8)) x) L)) '(1 2 3 6 7 8))
;;; we could do something similar with strings:
(test (let ((S "12345")) (map (lambda (x) (set! (S 2) #\null) x) S)) '(#\1 #\2 #\null #\4 #\5))
;;; (length S) is still 5 even with the embedded null
(test (let ((L (list 1 2 3))) (map (lambda (x) (set! L (list 6 7 8)) x) L)) '(1 2 3))
(test (let ((L1 (list 1 2 3)) (L2 (list 4 5 6 7))) (map (lambda (x1 x2) (set-cdr! (cdr L1) ()) (cons x1 x2)) L1 L2)) '((1 . 4) (2 . 5)))
(test (let ((L (list 1 2 3))) (map (lambda (x) (set-car! (cddr L) 32) x) L)) '(1 2 32))
;;; should these notice the increased length?:
(test (let ((L1 (list 1 2)) (L2 (list 6 7 8 9))) (map (lambda (x y) (set-cdr! (cdr L1) (list 10 11 12 13 14)) (cons x y)) L1 L2)) '((1 . 6) (2 . 7) (10 . 8) (11 . 9)))
(test (let ((L1 (list 1)) (L2 (list 6 7 8))) (map (lambda (x y) (set-cdr! L1 (list 10 11 12)) (cons x y)) L1 L2)) '((1 . 6)))
(test (let ((L1 (list 1 2))) (map (lambda (x) (set-cdr! (cdr L1) (list 10 11 12)) x) L1)) '(1 2 10 11 12))
;;; a similar case could be made from hash-tables
(test (let ((H (hash-table '(a . 3) '(b . 4)))) (pair? (map (lambda (x) (set! (H 'c) 32) (cdr x)) H))) #t)
(test (let ((H (hash-table '(a . 3) '(b . 4)))) 
	(let ((L (map (lambda (x) (set! (H 'b) 32) (cdr x)) H)))
	  (or (equal? L '(3 32))
	      (equal? L '(4 3)))))
      #t)
;; in that first example, the set-cdr! is not the problem (map supposedly can treat its args in any order),
;;   any set! will do:
(test (let ((x 0)) (map (lambda (y) (set! x (+ x y)) x) '(1 2 3 4))) '(1 3 6 10))

(test (map begin '(1 2 3)) '(1 2 3))
(let ((funcs (map (lambda (lst) (eval `(lambda ,@lst))) '((() #f) ((arg) (+ arg 1))))))
  (test ((car funcs)) #f)
  (test ((cadr funcs) 2) 3))

(test (map = #() =) 'error)
(test (map ="") 'error)
(test (map abs ()) ())
(test (map abs "") ())
(test (map abs "123" "") 'error)
(test (map abs "123" "" #f) 'error)
(test (map null? () #() "") 'error)
(test (map null? () #() 0 "") 'error)
(test (map '(()) #()) ())
(test (map '(1 2 . 3) '(1 . 2)) '(2))
(test (map '(()) ()) ())
(test (map #2D((1 2) (3 4)) '(1)) '(#(3 4)))
(test (map "a\x00b" #(1 2)) '(#\null #\b))
(test (map #(1 (3)) '(1)) '((3)))
(test (map '((1 (2)) (((3) 4))) '(1)) '((((3) 4))))
(test (map "hi" '(1)) '(#\i))
(test (map #() #()) 'error)
(test (map '(1 . 2) #()) ())
(test (map ''2 '(1)) '(2))
(test (((map lambda '((x)) '(1 2 . 3)) 0) 0) 1)
(test (((map lambda '(()) #(1 2)) 0)) 1)
(test (((map lambda '((x)) '((+ x 1))) 0) 32) 33)
(test (map #() ()) 'error)
(test (map () ()) 'error)
(test (map "" "") 'error)
(test (map (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) lst) '(0)) '(1))
(let ((lst (list 1 2)) (os (*s7* 'safety))) (set! (*s7* 'safety) 1) (set! (cdr (cdr lst)) lst) (test (map lst lst) '(2)) (set! (*s7* 'safety) os))
(test (map 1 "hi" ()) 'error)
(test (map 0 #() ()) 'error)
(test (map #\a #(1 2) '(3 4) "") 'error)
(test (map or '(1 2 . 3)) '(1 2))
(test (map or "a\x00b") '(#\a #\null #\b))
(test (map cond '((1 2) (3 4))) '(2 4)) ; (cond (1 2)) -> 2
(test (map begin "hi") '(#\h #\i))
(test (map quote "hi") '(#\h #\i))
(test (map quote '(a b c)) '(a b c)) ; when are (map quote ...) and (map values ...) different?
(test (map (begin #(1 (3))) '(1)) '((3)))
(test (map (''2 0) ''2) 'error)
(test (map (apply lambda 'a '(-1)) '((1 2))) '(-1))
(test (map (apply lambda 'a '(-1)) '(1 2)) '(-1 -1))
(test (map do '(()) '((1 2))) '(2)) ; (list 2) because it's map, not just do
(test (map case '(1) '(((-1 1) 2) 3)) '(2))
(test (map let '(()) "a\x00b") '(#\a))
(test (map "hi" '(0 1) '(0 1)) 'error)
(test (map '((1 2) (3 4)) '(0 1) '(0 1)) '(1 4))
(test (map #2d((1 2) (3 4)) '(0 1) '(0 1)) '(1 4))
(test (map #2d((1 2) (3 4)) '(0 1)) '(#(1 2) #(3 4)))
(let ((os (*s7* 'safety)))
  (set! (*s7* 'safety) 1)
  (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (map (lambda (a) a) lst) '(1)))
  (let ((lst (list 1 2))) (set! (cdr (cdr lst)) lst) (test (map (lambda (a) a) lst lst) 'error))
  (set! (*s7* 'safety) os))
(test (map "hi" ('((1)) 0)) '(#\i))
(test (map "hi" ('((1 0)) 0)) '(#\i #\h))
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (map ht ht)) '(#f #f))
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (let ((lst (map (lambda (c) (cdr c)) ht))) (or (equal? lst '(1 2)) (equal? lst '(2 1))))) #t)
(test (let ((ht (hash-table '(a . 1) '(b . 2)))) (map ht '(a b))) '(1 2))

(test (map '((1 2) (3 4)) #(1) #(1)) '(4))
(test (map (quasiquote ((1 2) (3 4))) #(1) #(1 2)) '(4))

(let ((pws (dilambda (lambda (a) a) (lambda (a b) b))))
  (test (map append pws) 'error)
  (test (map pws '(1 2 3)) '(1 2 3)))

(test (map abs '(1 2 . 3)) '(1 2)) ;; ?? Guile says wrong type arg here
(test (map + '(1) '(1 2 . 3)) '(2))
(test (map abs '(1 . 2)) '(1))
;; problematic because last thing is completely ignored:
(test (map abs '(1 . "hi")) '(1))
(test (map floor '(1 . "hi")) '(1))

(for-each
 (lambda (op)
   (test (map op ()) 'error)
   (test (map op "") 'error)
   (test (map op #() (list) (string)) 'error))
 (list 0 () #f #t 'a-symbol :hi #\a #<eof> #<unspecified> #<undefined> 0.0 1+i 1/2 1/0 0/0 *stdout* (current-input-port)))

(test (map append (make-vector (list 2 0))) ())
(let ((p1 (dilambda (lambda (a) (+ a 1)) (lambda (a b) (+ a b)))))
  (test (map p1 '(1 2 3)) '(2 3 4)))
(test (map (lambda args (+ (car args) 1)) '(1 2 3)) '(2 3 4))
(test (map (lambda* (a (b 2)) (+ a 1)) '(1 2 3)) '(2 3 4))
(let ((p1 (dilambda (lambda (a b) (+ a b)) (lambda (a b c) (+ a b c)))))
  (test (map p1 '(1 2 3) '(3 2 1)) '(4 4 4)))
(test (map (lambda args (+ (car args) (cadr args))) '(1 2 3) '(3 2 1)) '(4 4 4))
(test (map (lambda* (a (b 2)) (+ a b)) '(1 2 3) '(3 2 1)) '(4 4 4))
(test (map (lambda* (a (b 2)) (+ a b)) '(1 2 3)) '(3 4 5))
(test (map (lambda* ((a 1) (b (map (lambda (c) (+ c 1)) (list 1 2)))) (+ a (apply + b))) (list 4 5 6)) '(9 10 11))
(test (let ((lst (list 0 1 2))) (map (lambda* ((a 1) (b (for-each (lambda (c) (set! (lst c) (+ (lst c) 1))) (list 0 1 2)))) a) lst)) '(0 2 4))

(test (let ((lst '(1 2 3))) (define-macro (hiho a) `(+ 1 ,a)) (map hiho lst)) '(2 3 4))
(test (let ((lst '(1 2 3))) (define-bacro (hiho a) `(+ 1 ,a)) (map hiho lst)) '(2 3 4))
(test (let ((lst '(1 2 3))) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst lst)) '(4 7 10))
(test (let ((lst '(1 2 3))) (define-macro (hi1 a) `(+ 1 ,a)) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst (map hi1 lst))) '(6 9 12))
(test (let ((lst '(1 2 3))) (define-macro (hiho a b) `(+ 1 ,a (* 2 ,b))) (map hiho lst (map (define-macro (hi1 a) `(+ 1 ,a)) lst))) '(6 9 12))
(test (let ((lst '(1 2 3))) (define-macro (hi a) `(+ 1 ,a)) (define-macro (ho b) `(+ 1 (hi ,b))) (map ho lst)) '(3 4 5))
(test (let ((lst '(1 2 3))) (define-macro* (hi a (b 2)) `(+ 1 ,a (* 2 ,b))) (map hi lst)) '(6 7 8))
(test (let ((lst '(1 2 3))) (define-macro* (hi a (b 2)) `(+ 1 ,a (* 2 ,b))) (map hi lst (map hi lst))) '(14 17 20))

(let ()
  (define (hi)
    (map (lambda (a) (a 0))
	 (list (vector 1 2 3)
	       (string #\a #\b #\c)
	       (list 'e 'f 'g))))
  (test (hi) '(1 #\a e)))

(let ((ctr -1)) 
  (apply begin (map (lambda (symbol) 
		      (set! ctr (+ ctr 1))
		      (list 'define symbol ctr))
		    '(_zero_ _one_ _two_)))
  (+ _zero_ _one_ _two_))

(let ()
  (define (map-with-exit func . args)
  ;; func takes escape thunk, then args
  (let* ((result ())
	 (escape-tag (gensym))
	 (escape (lambda () (throw escape-tag))))
    (catch escape-tag
      (lambda ()
	(let ((len (apply max (map length args))))
	  (do ((ctr 0 (+ ctr 1)))
	      ((= ctr len) (reverse result))      ; return the full result if no throw
	    (let ((val (apply func escape (map (lambda (x) (x ctr)) args))))
	      (set! result (cons val result))))))
      (lambda args
	(reverse result))))) ; if we catch escape-tag, return the partial result

  (define (truncate-if func lst)
    (map-with-exit (lambda (escape x) (if (func x) (escape) x)) lst))

  (test (truncate-if even? #(1 3 5 -1 4 6 7 8)) '(1 3 5 -1))
  (test (truncate-if negative? (truncate-if even? #(1 3 5 -1 4 6 7 8))) '(1 3 5))
  )

;;; this is testing the one-liner unsafe closure optimizations
(test (let ()
	(define (fib n)
	  (if (< n 2)
	      n
	      (+ (fib (- n 1))
		 (fib (- n 2)))))
	(let ((x 0)
	      (ctr -1))
	  (map
	   (lambda (f)
	     (let ((z 1))
	       (set! ctr (+ ctr 1))
	       (case ctr
		 ((0 1 2 3) (f z))
		 ((4 5) (f z z))
		 ((6 7) (values (f (+ 1)) (f (+ 2)))))))
	   (list
	    (lambda (i)
	      (set! x (list i)))
	    (lambda (i)
	      (set! x (list i))
	      (set! x (list (+ i 1))))
	    (vector 1 2 3)
	    (list 3 2 1)
	    (lambda (a b)
	      (+ a b))
	    (lambda (a b)
	      (+ a b)
	      (+ a b a))
	    (lambda (a)
	      (if (< a 2) a (+ (fib (- a 1)) (fib (- a 2)))))
	    (lambda (a)
	      (if (zero? a) a (list a))
	      (list (+ a 10)))
	    ))))
      '((1) (2) 2 2 2 3 1 1 (11) (12)))
;;; more along the same lines
(test (let () (define (f2 a) (+ a 1)) (define (f1 a) (f2 a)) (define (f2 a) (- a)) (f1 12)) -12)
(test (let () (define (f2 a) (+ a 1)) (define (f1 a) (f2 a 1)) (define (f2 a b) (- a b)) (f1 12)) 11)
(test (let () (define (f2 a) (+ a 1)) (define (f1 a) (f2 a)) (define (f2 a) (- a 1)) (f1 12)) 11)

(test (let () (define* (f2 a) (+ a 1)) (define (f1 a) (f2 a)) (define* (f2 a) (- a)) (f1 12)) -12)
(test (let () (define* (f2 a) (+ a 1)) (define (f1 a) (f2 a 1)) (define* (f2 a b) (- a b)) (f1 12)) 11)
(test (let () (define* (f2 a) (+ a 1)) (define (f1 a) (f2 a)) (define* (f2 a) (- a 1)) (f1 12)) 11)

(test (map symbol->value (let ((lst (list 'integer? 'boolean?))) (set-cdr! (cdr lst) lst) lst)) (list integer?))
;;; I think this depends on when the list iterator notices the cycle


#|
;;; this is from the r6rs comment site
(let ((resume #f)
       (results ()))
   (set! results
         (cons (map (lambda (x)
                      (call/cc (lambda (k)
                                 (if (not resume) (set! resume k))
                                 0)))
                    '(#f #f))
               results ))
   (display results) (newline)
   (if resume
       (let ((resume* resume))
         (set! resume #f)
         (resume* 1))))

With a careful implementation of MAP, a new list is returned every
time, so that the displayed results are

   ((0 0))
   ((1 0) (0 0))
   ((1 1) (1 0) (0 0))

in s7:
((0 0))
((1 0) (0 0))
((0 . #1=(1 1)) #1# (0 0))

|#

;; from Doug Hoyte, Let Over Lambda
(let ()
  (define (batcher n) 
    (let* ((network ())
	   (tee (ceiling (log n 2)))
	   (p (ash 1 (- tee 1))))
      (do ()
	  ((= p 0) (reverse network))
	(let ((q (ash 1 (- tee 1)))
	      (r 0)
	      (d p))
	  (do ()
	      ((= d 0))
	    (do ((i 0 (+ i 1)))
		((= i (- n d)))
	      (if (= (logand i p) r)
		  (set! network (cons (list i (+ i d)) network))))
	    (set! d (- q p))
	    (set! q (ash q -1))
	    (set! r p)))
	(set! p (ash p -1)))))
  
  (define-macro (sortf comparator . places)
    (let ((tmp (gensym))
	  (net (batcher (length places))))
      `(begin
	 ,@(map (lambda (a b)
		  `(if (,comparator ,a ,b) ; we're ignoring the fancy CL getf|setf business
		       (let ((,tmp ,a))    ;   I suppose if it's a list, get procedure-setter?
			 (set! ,a ,b)
			 (set! ,b ,tmp))))
		(map (lambda (ab) (places (car ab))) net)
		(map (lambda (ab) (places (cadr ab))) net)))))
  
  (test (let ((a 1) (b 3) (c 0))
	  (sortf > a b c)
	  (list a b c))
	'(0 1 3))
  (test (let ((a 1) (b 3) (c 0))
	  (sortf < a b c)
	  (list a b c))
	'(3 1 0))
  (test (let ((v #(1 3 2)))
	  (sortf > (v 0) (v 1) (v 2))
	  v)
	#(1 2 3)))

;;; fftf?

(let ()
  (define-macro (shiftf . places)
    (let ((tmp (gensym)))
      `(let ((,tmp ,(car places)))
	 ,@(map (lambda (a b)
		  `(set! ,a ,b))
		places
		(cdr places))
	 ,tmp)))

  (define-macro (rotatef . places)
    (let ((tmp (gensym))
	  (last (car (list-tail places (- (length places) 1)))))
      `(let ((,tmp ,(car places)))
	 ,@(map (lambda (a b)
		  `(set! ,a ,b))
		places
		(cdr places))
	 (set! ,last ,tmp))))
  
  (test (let ((a 1) (b 2) (c 3))
	  (rotatef a b c)
	  (list a b c))
	'(2 3 1))
  (test (let ((a 1) (b 2) (c 3))
	  (rotatef a b c)
	  (rotatef a b c)
	  (list a b c))
	'(3 1 2))
  (test (let ((v #(1 3 2)))
	  (rotatef (v 0) (v 1) (v 2))
	  v)
	#(3 2 1))
  
  (test (let ((a 1) (b 2) (c 3))
	  (let ((d (shiftf a b c (+ 3 2))))
	    (list a b c d)))
	'(2 3 5 1))
  
  (test (let ((a 1) (b 2) (c 3))
	  (let ((d (shiftf a b c (shiftf a b c))))
	    (list a b c d)))
	'(3 3 2 1))
  ;; this expands to:
  ;; (let ((a 1) (b 2) (c 3))
  ;;   (let (({gensym}-22 a)) 
  ;;     (set! a b) 
  ;;     (set! b c) 
  ;;     (set! c (let (({gensym}-23 a))
  ;; 	         (set! a b) 
  ;; 	         (set! b c) 
  ;; 	         (set! c (* 2 3))
  ;; 	         {gensym}-23))
  ;;     (list a b c {gensym}-22)))
  
  (test (let ((v #(1 3 2)))
	  (let ((d (shiftf (v 0) (v 1) (v 2) (* 4 3))))
	    (list d v)))
	'(1 #(3 2 12))))


;;; --------------------------------------------------------------------------------
;;; iterate
;;; make-iterator
;;; iterator?
;;; iterator-sequence
;;; iterator-at-end?
;;; --------------------------------------------------------------------------------

(test (iterate) 'error)
(test (iterator?) 'error)
(test (iterator-sequence) 'error)
(test (iterator-at-end?) 'error)
(test (make-iterator) 'error)
(test (make-iterator "hi" "ho") 'error)
(test (iterator? "hi" "ho") 'error)
(test (iterator-sequence "hi" "ho") 'error)
(test (iterator-at-end? "hi" "ho") 'error)
(test (iterate "hi" "ho") 'error)

(for-each
 (lambda (arg)
   (if (iterator? arg)
       (format-logged #t ";~A: (iterator? ~A) -> #t?~%" (port-line-number) arg)))
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))

(for-each
 (lambda (arg)
   (test (iterate arg) 'error)
   (test (iterator-sequence arg) 'error)
   (test (iterate-at-end? arg) 'error))
 (list "hi" '(1 2) (integer->char 65) 1 'a-symbol (make-vector 3) abs _ht_ _null_ _c_obj_ quasiquote macroexpand 1/0 (log 0) 
       3.14 3/4 1.0+1.0i #\f (lambda (a) (+ a 1)) :hi (if #f #f) #<eof> #<undefined>))

(for-each
 (lambda (arg)
   (test (make-iterator arg) 'error)
   (test (make-iterator #(1 2) arg) 'error))
 (list 1 'a-symbol quasiquote macroexpand 3.14 3/4 1.0+1.0i #\f :hi (if #f #f) #<eof> #<undefined>))

(let ((str "12345"))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator? str) #f)
    (test (make-iterator s1) 'error)
    (test (iterator-sequence s1) str)
    (test (iterator-at-end? s1) #f)
    (test (iterate s1) #\1)
    (test (iterate s1 s1) 'error)
    (test (object->string s1) "#<iterator: string>")
    (test (s1) #\2)
    (test (list (s1) (s1) (s1)) (list #\3 #\4 #\5))
    (test (s1) #<eof>)
    (test (iterator-at-end? s1) #t)
    (let ((s2 (copy s1)))
      (test (equal? s1 s2) #t)
      (test (morally-equal? s1 s2) #t)
      (test (eq? s1 s2) #f)
      (test (eqv? s1 s2) #f))))

(let ((str ""))
  (let ((s1 (make-iterator str)))
    (let ((s2 (copy s1)))
      (test (equal? s1 s2) #t)
      (test (iterator? s1) #t)
      (test (iterator-sequence s1) str)
      (test (s1) #<eof>)
      (test (iterator-at-end? s1) #t)
      (test (iterator? s1) #t))))

(let ((s1 (make-iterator "1234")))
  (test (iterator? s1) #t)
  (test (s1) #\1))

(let ((str (vector #\1 #\2 #\3 #\4 #\5)))
  (let ((s1 (make-iterator str)))
    (let ((s2 (copy s1)))
      (test (equal? s1 s2) #t)
      (test (iterator? s1) #t)
      (test (iterator? str) #f)
      (test (make-iterator s1) 'error)
      (test (iterator-sequence s1) str)
      (test (iterate s1) #\1)
      (test (object->string s1) "#<iterator: vector>")
      (test (equal? s1 s2) #f)
      (s2)
      (test (equal? s1 s2) #t)
      (test (iterate s1 s1) 'error)
      (test (s1) #\2)
      (test (list (s1) (s1) (s1)) (list #\3 #\4 #\5))
      (test (s1) #<eof>))))

(let ((str #()))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator-sequence s1) str)
    (test (s1) #<eof>)
    (test (iterator-at-end? s1) #t)
    (test (iterator? s1) #t)))

(let ((str #2d((1 2) (3 4))))
  (let ((s1 (make-iterator str)))
    (test (iterator-at-end? s1) #f)
    (test (s1) 1)
    (test (iterate s1) 2)
    (test (s1) 3)))


(let ((str (float-vector 1.0 2.0 3.0 4.0)))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator? str) #f)
    (test (iterator-sequence s1) str)
    (test (iterate s1) 1.0)
    (test (s1) 2.0)
    (test (list (s1) (s1)) (list 3.0 4.0))
    (test (s1) #<eof>)))

(let ((str (float-vector)))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator-sequence s1) str)
    (test (s1) #<eof>)
    (test (iterator-at-end? s1) #t)
    (test (iterator? s1) #t)))


(let ((str (make-int-vector 4 0)))
  (do ((i 1 (+ i 1))) ((= i 4)) (set! (str i) i))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator-sequence s1) str)
    (test (iterate s1) 0)
    (test (s1) 1)
    (test (list (s1) (s1) (s1)) (list 2 3 #<eof>))
    (test (s1) #<eof>)))


(let ((str (list 0 1 2 3)))
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator-sequence s1) str)
    (test (iterate s1) 0)
    (test (s1) 1)
    (test (s1 0) 'error)
    (test (iterate s1 0) 'error)
    (test (list (s1) (s1) (s1)) (list 2 3 #<eof>))
    (test (s1) #<eof>)))

(let ((str ()))
  (test (make-iterator str) 'error))

(let ((str '((1 2) (3 4))))
  (let ((s1 (make-iterator str)))
    (test (s1) '(1 2))
    (test (iterate s1) '(3 4))
    (test (s1) #<eof>)))

(let ((str (list 0 1)))
  (set! (cdr (cdr str)) str)
  (let ((s1 (make-iterator str)))
    (test (iterator? s1) #t)
    (test (iterator-sequence s1) str)
    (test (iterate s1) 0)
    (test (s1) 1)
    (test (s1) #<eof>)))

(let ((p (cons #f #f))
      (h (hash-table* 'a 1 'b 2)))
  (let ((iter (make-iterator h p)))
    (let ((v (iter)))
      (test (pair? v) #t)
      (test (eq? v p) #t)
      (test (pair? (memq (car v) '(a b))) #t)
      (set! v (iter))
      (test (pair? v) #t)
      (test (eq? v p) #t)
      (test (pair? (memq (car v) '(a b))) #t))))

;; hash-table and let dealt with elsewhere

(when with-block
  (let ((b (block 0.0 1.0 2.0)))
    (let ((b1 (make-iterator b)))
      (test (iterator? b1) #t)
      ;(test (iterator-sequence b1) b) ; this is now a function
      (test (b1) 0.0)
      (test (iterate b1) 1.0)
      (test (list (b1) (b1)) '(2.0 #<eof>)))))

(let ((c1 (let ((iterator? #t) (a 0)) (lambda () (set! a (+ a 1))))))
  (let ((iter (make-iterator c1)))
    (test (iterate iter) 1)
    (test (iterator? c1) #t)
    (test (iterator? iter) #t)
    (test (eq? (iterator-sequence iter) c1) #t)))

(let ()
  (define c1 #f)
  (let ((length (lambda (x) 3))
	(iterator? #t)
	(x 0))
    (set! c1 (openlet (lambda () (let ((res (* x 2))) (set! x (+ x 1)) res)))))
  (let ((c2 (make-iterator c1)))
    (test (iterator? c2) #t)
    (test (iterator-sequence c2) c1)
    (test (c2) 0)
    (test (c2) 2)
    (test (c2) 4)))

(let ((lst (list 1))) (set-cdr! lst lst) (let ((i (make-iterator lst))) (test (map values i) '(1))))
(let ((lst (list 1 2))) (set-cdr! (cdr lst) lst) (let ((i (make-iterator lst))) (test (map values i) '(1 2))))
(let ((lst (list 1 2 3))) (set-cdr! (cddr lst) lst) (let ((i (make-iterator lst))) (test (> (length (map values i)) 2) #t))) ; '(1 2 3) ideally
(let ((lst (list 1 2 3 4))) (set-cdr! (cdddr lst) lst) (let ((i (make-iterator lst))) (test (> (length (map values i)) 3) #t))) ; '(1 2 3 4)?

(test (format #f "~{~{[~A ~A]~}~}" 
	      (make-iterator 
	       (let ((lst '((a . 1) (b . 2) (c . 2)))
		     (iterator? #t))
		 (lambda () 
		   (if (pair? lst)
		       (let ((res (list (caar lst) (cdar lst))))
			 (set! lst (cdr lst))
			 res)
		       #<eof>)))))
      "[a 1][b 2][c 2]")

(let ()
  (define (make-diagonal-iterator matrix)
    (if (or (= (length (vector-dimensions matrix)) 1)
	    (< (length matrix) 2))
	(make-iterator matrix)
	(make-iterator (let ((inds (length (vector-dimensions matrix)))
			     (len (apply min (vector-dimensions matrix))))
			 (let ((length (lambda (obj) len))
			       (iterator? #t)
			       (pos 0))
			   (openlet 
			    (lambda ()
			      (if (>= pos len)
				  #<eof>
				  (let ((res (apply matrix (make-list inds pos))))
				    (set! pos (+ pos 1))
				    res)))))))))
  
  (define v #2d((1 2 3 4) (5 6 7 8)))

  (let ((iv (make-diagonal-iterator v)))
    (let ((vals (map values iv)))
      (test vals '(1 6)))))

(test (let ((iter (make-iterator "123"))) (map values iter)) '(#\1 #\2 #\3))
(test (let ((iter (make-iterator '(1 2 3))) (str "456") (vals ())) 
	(for-each (lambda (x y) (set! vals (cons (cons y x) vals))) iter str) vals)
      '((#\6 . 3) (#\5 . 2) (#\4 . 1)))

(let ()
  (define* (make-full-let-iterator lt (stop (rootlet)))
    (if (eq? stop lt)
	(make-iterator #())
	(let ((iter (make-iterator lt)))
	  (if (eq? stop (outlet lt))
	      iter
	      (letrec ((iterloop 
			(let ((iterator? #t))
			  (lambda ()
			    (let ((result (iter)))
			      (if (and (eof-object? result)
				       (iterator-at-end? iter))
				  (if (eq? stop (outlet (iterator-sequence iter)))
				      result
				      (begin 
					(set! iter (make-iterator (outlet (iterator-sequence iter))))
					(iterloop)))
				  (if (not (char=? ((symbol->string (car result)) 0) #\_))
				      result
				      (iterloop))))))))
		(make-iterator iterloop))))))
  (let ((stop #f)) 
    (set! stop (curlet)) 
    (test (let ((a 1)) (map values (make-full-let-iterator (curlet) stop))) '((a . 1))))
  (let ((stop #f)) 
    (set! stop (curlet)) 
    (test (let ((b 2)) (let ((a 1)) (map values (make-full-let-iterator (curlet) stop)))) '((a . 1) (b . 2))))
  (let ((stop #f)) 
    (set! stop (curlet))
    (test (let ((b 2) (c 3)) (let () (let ((a 1)) (map values (make-full-let-iterator (curlet) stop))))) '((a . 1) (c . 3) (b . 2))))
  )

(let ()
  (define (make-range lo hi)
    (make-iterator
     (let ((iterator? #t)
	   (now lo))
       (lambda ()
	 (if (> now hi)
	     #<eof>
	     (let ((result now))
	       (set! now (+ now 1))
	       result))))))
  (test (map values (make-range 4 8)) '(4 5 6 7 8)))

(let ()
  (define (make-input-iterator port)
    (make-iterator (let ((iterator? #t)) (lambda () (read-char port)))))

  (test (let ((p (open-input-string "12345")))
	  (let ((ip (make-input-iterator p)))
	    (let ((res (map values ip)))
	      (close-input-port p)
	      res)))
	'(#\1 #\2 #\3 #\4 #\5)))

(let ()
  (define (make-input-iterator port)
    (make-iterator (let ((iterator? #t)) (define-macro (_m_) `(read-char ,port)))))

  (test (let ((p (open-input-string "12345")))
	  (let ((ip (make-input-iterator p)))
	    (let ((res (map values ip)))
	      (close-input-port p)
	      res)))
	'(#\1 #\2 #\3 #\4 #\5)))

(let ((iter (make-iterator (let ((iterator? #t)
				 (pos 0))
			     (lambda ()          
			       (if (< pos 3) 
				   (let ((p pos))
				     (set! pos (+ pos 1))
				     (values p (* p 2)))         ; ?? maybe this is inconsistent? 
				   #<eof>))))))
  (test (map values iter) '(0 0 1 2 2 4)))

(let ()
  (define (make-row-iterator v)
    (make-iterator (let ((iterator? #t)
			 (col 0))
		     (lambda ()
		       (if (< col (car (vector-dimensions v)))
			   (let ((c col))
			     (set! col (+ col 1))
			     (make-shared-vector v (cadr (vector-dimensions v)) (* c (cadr (vector-dimensions v)))))
			   #<eof>)))))

  (let ((v #2d((0 1 2) (4 5 6))))
    (let ((iter (make-row-iterator v)))
      (test (map values iter) '(#(0 1 2) #(4 5 6))))))

(let ()
  (define (make-semi-complete-iterator obj)
    (make-iterator
     (let ((iters ())
	   (iter (make-iterator obj)))
       (define (iterloop)
	 (let ((result (iter)))
	   (if (length result) ; i.e. result is a sequence
	       (begin
		 (set! iters (cons iter iters))
		 (set! iter (make-iterator result))
		 result)       ; this returns the sequence before we descend into it
			       ;   we could also call iterloop here to skip that step
	       (if (eof-object? result)
		   (if (null? iters)
		       result  ; return #<eof>
		       (begin
			 (set! iter (car iters))
			 (set! iters (cdr iters))
			 (iterloop)))
		   result))))
       (let ((iterator? #t))
	 (lambda () 
	   (iterloop))))))
  
  (test (let ((v '(1 2 (4 5)))) (let ((i (make-semi-complete-iterator v))) (map values i))) '(1 2 (4 5) 4 5))
  (test (let ((v #(1 2 (4 5)))) (let ((i (make-semi-complete-iterator v))) (map values i))) '(1 2 (4 5) 4 5))
  (test (let ((v '((1 2 (4 5))))) (let ((i (make-semi-complete-iterator v))) (map values i))) '((1 2 (4 5)) 1 2 (4 5) 4 5))
  (test (let ((v '(1 2 #(4 5)))) (let ((i (make-semi-complete-iterator v))) (map values i))) '(1 2 #(4 5) 4 5))
  (test (let ((v '(1 2 #(4 5) ("67")))) (let ((i (make-semi-complete-iterator v))) (map values i))) '(1 2 #(4 5) 4 5 ("67") "67" #\6 #\7)))

(let ()
  (define (make-settable-iterator obj)
    (make-iterator (let ((iterator? #t)
			 (pos 0))
		     (dilambda (lambda ()
				 (let ((res (obj pos)))
				   (set! pos (+ pos 1))
				   res))
			       (lambda (val)
				 (set! (obj pos) val))))))
  (test (procedure? (procedure-setter (make-settable-iterator (vector 1 2 3)))) #t)
  (let ((v (vector 1 2 3)))
    (let ((iter (make-settable-iterator v)))
      (set! (iter) 32)
      (test v #(32 2 3)))))

(let ()
  (define (make-circular-iterator obj)
    (let ((iter (make-iterator obj)))
      (make-iterator 
       (let ((iterator? #t))
	 (lambda ()
	   (let ((result (iter)))
	     (if (eof-object? result)
		 ((set! iter (make-iterator obj)))
		 result)))))))
  
  (let ((iter (make-circular-iterator '(1 2 3)))
	(lst ()))
    (do ((i 0 (+ i 1)))
	((= i 10)
	 (test (reverse lst) '(1 2 3 1 2 3 1 2 3 1)))
      (set! lst (cons (iter) lst))))
  
  (let ((iter (make-circular-iterator (hash-table* :a 1 :b 2)))
	(lst ()))
    (do ((i 0 (+ i 1)))
	((= i 4)
	 (test (let ((r (reverse lst)))
		 (or (equal? r '((:a . 1) (:b . 2) (:a . 1) (:b . 2)))
		     (equal? r '((:b . 2) (:a . 1) (:b . 2) (:a . 1)))))
	       #t))
      (set! lst (cons (iter) lst)))))
  

(test (procedure-setter (make-iterator "123")) #f)
(test (procedure-setter (make-iterator #(1))) #f)
(test (procedure-setter (make-iterator '(1))) #f)
(test (procedure-setter (make-iterator (float-vector pi))) #f)

(test (copy (make-iterator '(1 2 3)) (vector 1)) 'error)

(let ()
  (define (make-file-iterator file)
    ;; reads a text file, returning one word at a time
    (let* ((port (open-input-file file))
	   (line (read-line port #t))
	   (pos -1)
	   (new-pos 0)
	   (eol (string #\space #\tab #\newline #\linefeed)))
      
      (define (next-word)
	(set! pos (char-position eol line (+ pos 1)))
	(if (not pos)
	    (begin
	      (set! pos -1)
              (set! new-pos 0)
	      (set! line (read-line port #t))
	      (if (eof-object? line)
		  (begin
		    (close-input-port port)
		    line)
		  (next-word)))
	    (if (= new-pos pos)
		(next-word)
		(let ((start (do ((k new-pos (+ k 1)))
				 ((or (= k pos)
				      (char-alphabetic? (string-ref line k))
				      (char-numeric? (string-ref line k)))
				  k))))
		  (if (< start pos)
		      (let ((end (do ((k (- pos 1) (- k 1)))
				     ((or (= k start)
					  (char-alphabetic? (string-ref line k))
					  (char-numeric? (string-ref line k)))
				      (+ k 1)))))
			(set! new-pos (+ pos 1))
			(if (> end start)
			    (substring line start end)
			    (next-word)))
		      (next-word))))))
      
      (make-iterator 
       (let ((iterator? #t))
	 (lambda* (p)
	   (if (eq? p :eof)
	       (begin
		 (close-input-port port)
		 #<eof>)
	       (next-word)))))))

  (define iter (make-file-iterator "s7.c"))
  (test (iter) "s7")
  (test (iter) "a")
  (test (iter) "Scheme")
  (test (iter) "interpreter")
  (test ((iterator-sequence iter) :eof) #<eof>)
  )




;;; --------------------------------------------------------------------------------
;;; do
;;; --------------------------------------------------------------------------------

(test (do () (#t 1)) 1)
(for-each
 (lambda (arg)
   (test (do () (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (do ((i arg)) (#t i)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (do ((i 0 (+ i 1))) ((= i 3) #f)) #f)
(test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(test (do ((i 1/2 (+ i 1/8))) ((= i 2) i)) 2)
(test (do ((i 1/2 (+ i 1/8))) ((> i 2) i)) 17/8)
(test (do ((vec (make-vector 5)) (i 0 (+ i 1))) ((= i 5) vec) (vector-set! vec i i)) #(0 1 2 3 4))
(test (let ((x '(1 3 5 7 9))) (do ((x x (cdr x)) (sum 0 (+ sum (car x))))  ((null? x) sum))) 25)
(test (do ((i 4 (- i 1)) (a 1 (* a i))) ((zero? i) a)) 24)
(test (do ((i 2 (+ i 1))) ((> i 0) 123)) 123)

(test (do () (() ()) ()) ())
(test (do () ('() ())) ())
(test (do () ('())) ())
(test (do () (())) ())
(test (do) 'error)

(test (let ((x 0) (y 0)) (set! y (do () (#t (set! x 32) 123))) (list x y)) (list 32 123))
(test (let ((i 32)) (do ((i 0 (+ i 1)) (j i (+ j 1))) ((> j 33) i))) 2)
(test (let ((i 0)) (do () ((> i 1)) (set! i (+ i 1))) i) 2)
(test (let ((i 0) (j 0)) (do ((k #\a)) (#t i) (set! i (char->integer k)) (set! j (+ j i)))) 0)
(test (let ((i 0) (j 0)) (do ((k #\a)) ((> i 1) j) (set! i (char->integer k)) (set! j (+ j i)))) (char->integer #\a))
(test (let ((x 0)) (do ((i 0 (+ i 2)) (j 1 (* j 2))) ((= i 4) x) (set! x (+ x i j)))) 5)
(test (let ((sum 0)) (do ((lst '(1 2 3 4) (cdr lst))) ((null? lst) sum) (set! sum (+ sum (car lst))))) 10)
(test (do ((i 0 (+ 1 i))) ((= i 4) (do ((i 0 (+ i 2))) ((= i 10) i)))) 10)
(test (let ((i 0)) (do ((i 1 (+ i 1))) ((= i 3) i))) 3)
(test (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 3) (+ i j)) (do ((j 0 (+ j i 1))) ((> j 3) j)))) 3)
(test (let ((add1 (lambda (a) (+ a 1)))) (do ((i 0 (add1 i))) ((= i 10) (add1 i)))) 11)
(test (do ((i 0 (do ((j 0 (+ j 1))) ((= j i) (+ i 1))))) ((= i 3) i)) 3)
(test (do ((i 0 (do ((i 0 (+ i 1))) ((= i 3) i)))) ((= i 3) i)) 3)
(test (let ((i 123)) (do ((i 0 (+ i 1)) (j i (+ j i))) ((> j 200) i))) 13)
(test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* i 10))) 11)
(test (do ((i 123) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 123 i) (j 0 (+ j i))) ((= j 246) i)) 123)
(test (do ((i 0 i)) (i i)) 0)
(test (do ((i 1 i)) (i i (+ i i) (+ i i i))) 3)
(test (do ((i 1)) (#t 1) 123) 1)
(test (do ((i 0 (+ i j)) (j 0 (+ j 1))) (#t 1)) 1)
(test (do ((i 0 j) (j 0 (+ j 1))) ((= j 3) i)) 2) ; uh, lessee... lexical scoping...
(test (do ((i 1 j) (j 0 k) (k 0 m) (m 0 (+ i j k))) ((> m 10) (list i j k m))) (list 4 5 8 11))
(test (let ((i 10) (j 11) (k 12)) (do ((i i j) (j j k) (k k m) (m (+ i j k) (+ i j k))) ((> m 100) (list i j k m)))) (list 33 56 78 122))
(test (let ((x 0) (i 3)) (do ((i i (+ i 1))) ((= i 6)) (do ((i i (+ i 1))) ((= i 7)) (set! x (+ x i)))) x) 44)
(test (let () (define (hi) (let ((x 0) (i 3)) (do ((i i (+ i 1))) ((= i 6)) (do ((i i (+ i 1))) ((= i 7)) (set! x (+ x i)))) x)) (hi)) 44)
(test (do ((i 0 (let () (set! j 3) (+ i 1))) (j 0 (+ j 1))) ((= i 3) j)) 4)
(test (let ((i 0)) (do () ((= i 3) (* i 2)) (set! i (+ i 1)))) 6)
(num-test (do ((i 0 (- i 1))) ((= i -3) i)) -3)
(num-test (do ((i 1/2 (+ i 1/2))) ((> i 2) i)) 5/2)
(num-test (do ((i 0.0 (+ i 0.1))) ((>= i 0.9999) i)) 1.0)
(num-test (do ((i 0 (- i 1/2))) ((< i -2) i)) -5/2)
(num-test (do ((i 0+i (+ i 0+i))) ((> (magnitude i) 2) i)) 0+3i)
(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((> i 4) x) 
	  (set! x (+ x i))
	  (set! i (+ i 0.5))))
      4.5)
(test (do ((i 0 1)) ((> i 0) i)) 1)
(test (do ((i 1.0+i 3/4)) ((= i 3/4) i)) 3/4)
(test (do ((i 0 "hi")) ((not (number? i)) i)) "hi")
(test (do ((i "hi" 1)) ((number? i) i)) 1)
(test (do ((i #\c "hi")) ((string? i) i)) "hi")
(test (do ((i #\c +)) ((not (char? i)) i)) +)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i))) 1)
(test (let ((j 1)) (do ((i 0 j)) ((= i j) i) (set! j 2))) 2)
(test (do ((j 1 2) (i 0 j)) ((= i j) i)) 2)
(test (let ((old+ +) (j 0)) (do ((i 0 (old+ i 1))) ((or (< i -3) (> i 3))) (set! old+ -) (set! j (+ j i))) j) -6)
(test (do ((i 0 (case i ((0) 1) ((1) "hi")))) ((string? i) i)) "hi")
(test (do ((i if +)) ((equal? i +) i)) +)
(test (let ((k 0)) (do ((j 0 (+ j 1)) (i 0 ((if (= i 0) + -) i 1))) ((= j 5)) (set! k (+ k i))) k) 2)
(test (let ((j -10) (k 0)) (do ((i 0 (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) 6)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((> i 4) k) (set! k (+ k i)))) -24)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) k) (set! k (+ k i)))) -30)
(test (let ((j -10) (k 0)) (do ((i j (+ i j)) (j 2)) ((= i j) j) (set! k (+ k i)))) 2)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i))) 3)
(test (let ((equal =)) (do ((i 0 (+ i 1))) ((equal i 3) i) (set! equal >))) 4)
(test (do ((equal =) (i 0 (+ i 1))) ((equal i 3) i)) 3)
(test (do ((equal = >) (i 0 (+ i 1))) ((equal i 3) i)) 4)
(test (do ((j 0) (plus + -) (i 0 (plus i 1))) ((= i -1) j) (set! j (+ j 1))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i 3) j) (set! j (+ j 1)))) 3)
(test (let ((expr `(+ i 1))) (do ((j 0) (i 0 (eval expr))) ((= i -3) j) (set! j (+ j 1)) (if (= j 3) (set! expr `(- i 1))))) 7)
(test (do ((i 0 (+ i 1))) ((or (= i 12) (not (number? i)) (> (expt 2 i) 32)) (expt 2 i))) 64)
(test (let ((k 0)) (do ((i 0 (+ i 1))) ((let () (set! k (+ k 1)) (set! i (+ i 1)) (> k 3)) i))) 7)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (* .9 i))) 3.439)
(test (let ((v #(0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (set! (v i) i))) #(0 1 2))
(test (let ((v (list 0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (set! (v i) i))) '(0 1 2))
(test (let ((sum 0)) ((do ((i 0 (+ i 1))) ((> i 64) (lambda () sum)) (set! sum (+ sum i))))) 2080)
(test (do ((lst () (cons i lst)) (i 0 (+ i 1))) ((> i 6) (reverse lst))) '(0 1 2 3 4 5 6))
(let ()
  (define (d1) (do ((i 0 (+ i 1))) ((= i 10) i)))
  (define (d2) (do ((i 0 (+ i 1))) ((= i 10) i) i))
  (test (d1) 10)
  (test (d1) (d2)))

(test (do () (1)) ())
(test (do () (1 2)) 2)
(test (do () '2) 2)
(test (do () (())) ())
(test (do ((i 0 (+ i 1))) ((= i 2) (+ i j)) (define j i) j) 3) ; bizarre (Guile sez "bad define placement" from r6|7rs I guess)
(test (do ((i 1 (+ i j))) ((>= i 2) (+ i j)) (define j i) j) 3)
(test (do ((i 1 (+ i j))) ((>= i 2) (+ i j)) (define j (+ i 10)) j) 23)
;(test (do ((i 1 (+ i j))) ((>= i 2) (+ i j)) (if (integer? i) (define j i)) j) 3)
;; this ^ will cause a segfault if optimized but how to catch it without a huge speed penalty?
(test (do ((i 0 (+ i 1))) ((>= i 2) i) (define i 10) i) 11)

(test (let ((x (do ((i 0 (+ i 1))) (#t)))) x) ()) ; guile: #<unspecified>

(test (let ((lst '(1 2 3))
	    (v (vector 0 0 0)))
	(do ((l lst (map (lambda (a) (+ a 1)) (cdr l))))
	    ((null? l))
	  (set! (v (- (length l) 1)) (apply + l)))
	v)
      #(5 7 6))

(test (do ((i 0 (+ i 1)) (j 1 2)) ((= i 4) j) (set! j 3)) 2)

(test (let ((lst '(1 2 3)))
	(map (lambda (a)
	       (let ((! 1))
		 (do ((i 0 (+ i 1))
		      (sum 0))
		     ((= i a) sum)
		   (set! sum (+ sum a)))))
	     lst))
      '(1 4 9))

(test (let ((sum 0)) (do ((i_0 0 (+ i_0 0))(i_1 1 (+ i_1 1))(i_2 2 (+ i_2 2))(i_3 3 (+ i_3 3))(i_4 4 (+ i_4 4))(i_5 5 (+ i_5 5))(i_6 6 (+ i_6 6))(i_7 7 (+ i_7 7))(i_8 8 (+ i_8 8))(i_9 9 (+ i_9 9))(i_10 10 (+ i_10 10))(i_11 11 (+ i_11 11))(i_12 12 (+ i_12 12))(i_13 13 (+ i_13 13))(i_14 14 (+ i_14 14))(i_15 15 (+ i_15 15))(i_16 16 (+ i_16 16))(i_17 17 (+ i_17 17))(i_18 18 (+ i_18 18))(i_19 19 (+ i_19 19))(i_20 20 (+ i_20 20))(i_21 21 (+ i_21 21))(i_22 22 (+ i_22 22))(i_23 23 (+ i_23 23))(i_24 24 (+ i_24 24))(i_25 25 (+ i_25 25))(i_26 26 (+ i_26 26))(i_27 27 (+ i_27 27))(i_28 28 (+ i_28 28))(i_29 29 (+ i_29 29))(i_30 30 (+ i_30 30))(i_31 31 (+ i_31 31))(i_32 32 (+ i_32 32))(i_33 33 (+ i_33 33))(i_34 34 (+ i_34 34))(i_35 35 (+ i_35 35))(i_36 36 (+ i_36 36))(i_37 37 (+ i_37 37))(i_38 38 (+ i_38 38))(i_39 39 (+ i_39 39)))
    ((= i_1 10) sum)
  (set! sum (+ sum i_0 i_1 i_2 i_3 i_4 i_5 i_6 i_7 i_8 i_9 i_10 i_11 i_12 i_13 i_14 i_15 i_16 i_17 i_18 i_19 i_20 i_21 i_22 i_23 i_24 i_25 i_26 i_27 i_28 i_29 i_30 i_31 i_32 i_33 i_34 i_35 i_36 i_37 i_38 i_39))))
      35100)

(let () (define (jtest) (let ((j 0)) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set! j i))))) (test (jtest) 3))
(let () (define (jtest1) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (j 0) i))))) (test (jtest1) 3))
(let () (define (jtest2) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (vector-set! j 0 i))))) (test (jtest2) 3))
(let () (define (jtest3) (let ((j (vector 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (vector-ref j 0) i))))) (test (jtest3) 3))
(let () (define (jtest4) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (j 0) i))))) (test (jtest4) 3))
(let () (define (jtest5) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! (car j) i))))) (test (jtest5) 3))
(let () (define (jtest6) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set-car! j i))))) (test (jtest6) 3))
(let () (define (jtest7) (let ((j (list 0))) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (list-set! j 0 i))))) (test (jtest7) 3))
(let () (define (jtest8) (let ((j #f)) (do ((i 0 (+ i 1))) ((= i 10) (car j)) (if (= i 3) (set! j (list i)))))) (test (jtest8) 3))
(let () (define (jtest9) (let ((j #f)) (do ((i 0 (+ i 1))) ((= i 10) (j 0)) (if (= i 3) (set! j (vector i)))))) (test (jtest9) 3))
(let () (define (jtest10) (let ((j (cons 1 2))) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set-car! j i))))) (test (jtest10) '(3 . 2)))
(let () (define (jtest10a) (let ((j (cons 1 2))) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (list-set! j 0 i))))) (test (jtest10a) '(3 . 2)))
(let () (define (jtest11) (let ((j (cons 1 2))) (do ((i 0 (+ i 1))) ((= i 10) j) (if (= i 3) (set! j (cons 0 i)))))) (test (jtest11) '(0 . 3)))
;; (let ((f #f)) (define (jtest12) (do ((i 0 (+ i 1))) ((= i 10) (f)) (if (= i 3) (set! f (lambda () i))))) (test (jtest12) 3))
;; this lambda business is a separate issue (s7 returns 10 here)

;; do_all_x:
(let () (define (f1) (let ((v (vector 0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (vector-set! v i (abs i))))) (test (f1) #(0 1 2)))
(let () (define (f1) (let ((v (vector 0 0 0)) (x #f)) (do ((i 0 (+ i 1))) ((= i 3) (set! x v) x) (vector-set! v i (abs i))))) (test (f1) #(0 1 2)))
(let () (define (f1) (let ((end 3) (v (vector 0 0 0))) (vector (do ((i 0 (+ i 1))) ((= i end)) (vector-set! v i (abs i))) v))) (test (f1) #(() #(0 1 2))))
(let () (define (f1) (let ((v (vector 0 0 0))) (do ((i 0 (+ i 1))) ((= i 3) v) (display i #f) (vector-set! v i (abs i))))) (test (f1) #(0 1 2)))

(let () (define (f1) (let ((v (vector 0 0 0))) (do ((i 0 (+ i 1))) ((= i 0) v) (vector-set! v i (abs i))))) (test (f1) #(0 0 0)))

(let () (define (safe-do-all-x)
	  (let ((v1 (vector 1 2 3 4 5 6))
		(v2 (vector 10 11 12 13 14 15)))
	    (do ((i 0 (+ i 1)))
		((= i 3))
	      (vector-set! v1 i (vector-ref v2 (+ i 1))))
	    v1))
     (test (safe-do-all-x) (vector 11 12 13 4 5 6)))


(test (let () (define (step-it a) (+ a 1)) (define (hi) (do ((i 0 (step-it i))) ((= i 3) i))) (hi) (hi)) 3)

(test (call-with-exit (lambda (return) (do () () (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#f) (if #t (return 123))))) 123)
(test (call-with-exit (lambda (return) (do ((i 0 (+ i 1))) () (if (= i 100) (return 123))))) 123)
(test (call-with-exit (lambda (return) (do () ((return 123))))) 123)
(test (call-with-exit (lambda (return) (do () (#t (return 123))))) 123)

(test (do () (/ 0)) 0)
(test (do () (+)) ())
(test (do () (+ +) *) +)

(when with-bignums
  (num-test (do ((i 24444516448431392447461 (+ i 1))
		 (j 0 (+ j 1)))
		((>= i 24444516448431392447471) j))
	    10)
  (num-test (do ((i 0 (+ i 24444516448431392447461))
		 (j 0 (+ j 1)))
		((>= i 244445164484313924474610) j))
	    10)
  (num-test (do ((i 4096 (* i 2))
		 (j 0 (+ j 1)))
		((= i 4722366482869645213696) j))
	    60))

(test (do ((i 9223372036854775805 (+ i 1))
	   (j 0 (+ j 1)))
	  ((>= i 9223372036854775807) j))
      2)
(test (do ((i -9223372036854775805 (- i 1))
	   (j 0 (+ j 1)))
	  ((<= i -9223372036854775808) j))
      3)

(num-test (do ((x (list 1 2 3) (cdr x)) (j -1)) ((null? x) j) (set! j (car x))) 3)

(test (let ((x 0)) 
	(do ((i 0 (+ i 1)))
	    ((= i (do ((j 0 (+ j 1))) ((= j 2) (+ j 1)))))
	  (set! x (+ x i)))
	x)
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) x)
	  (set! x (+ x i))))
      3)
(test (let ((x 0)) 
	(do ((i 0 (+ i (do ((j 0 (+ j 1))) ((= j 2) 1)))))
	    ((= i 3) (do ((j 0 (+ j 1))) ((= j 5) x) (set! x j)))
	  (set! x (+ x i))))
      4)

(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 100) i) (if (= i 2) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (if (= i 3) (exit 321) (+ i 1)))) ((= i 100) i)))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) (exit 321))))) 321)
(test (call-with-exit (lambda (exit) (do ((i 0 (+ i 1))) ((= i 10) i) (if (= i -2) (exit 321))))) 10)
(test (do ((x 0 (+ x 1)) (y 0 (call/cc (lambda (c) c)))) ((> x 5) x) #f) 6)
(test (let ((happy #f)) (do ((i 0 (+ i 1))) (happy happy) (if (> i 3) (set! happy i)))) 4)

(test (+ (do ((i 0 (+ i 1))) ((= i 3) i)) (do ((j 0 (+ j 1))) ((= j 4) j))) 7)
(test (do ((i (if #f #f))) (i i)) (if #f #f))
(test (do ((i (if #f #f)) (j #f i)) (j j)) (if #f #f))

(test (let ((cont #f)
	    (j 0)
	    (k 0))
	(call/cc (lambda (exit) 
		   (do ((i 0 (+ i 1))) 
		       ((= i 100) i) 
		     (set! j i)
		     (call/cc (lambda (r) (set! cont r)))
		     (if (= j 2) (exit))
		     (set! k i))))
	(if (= j 2)
	    (begin
	      (set! j 3)
	      (cont))
	    (list j k)))
      (list 99 99))

(test (call/cc (lambda (r) (do () (#f) (r 1)))) 1)
(test (let ((hi (lambda (x) (+ x 1)))) (do ((i 0 (hi i))) ((= i 3) i))) 3)
(test (do ((i 0 (+ i 1))) (list 1) ((= i 3) #t)) 1) ; a typo originally -- Guile and Gauche are happy with it
(test (do () (1 2) 3) 2)

;; from sacla tests
(test (let ((rev (lambda (list)
		   (do ((x list (cdr x))
			(reverse () (cons (car x) reverse)))
		       ((null? x) reverse)))))
	(and (null? (rev ()))
	     (equal? (rev '(0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (let ((nrev (lambda (list)
		    (do ((f1st (if (null? list) () (cdr list)) (if (null? f1st) () (cdr f1st)))
			 (s2nd list f1st)
			 (t3rd () s2nd))
			((null? s2nd) t3rd)
		      (set-cdr! s2nd t3rd)))))
	(and (null? (nrev ()))
	     (equal? (nrev (list 0 1 2 3 4)) '(4 3 2 1 0))))
      #t)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (- temp-two 1)))
	  ((> (- temp-one temp-two) 5) temp-one))
      4)

(test (do ((temp-one 1 (+ temp-one 1))
	   (temp-two 0 (+ temp-one 1)))     
	  ((= 3 temp-two) temp-one))
      3)

(let ((vec (vector 0 1 2 3 4 5 6 7 8 9)))
  (test (do ((i 0 (+ 1 i))
	     (n #f)
	     (j 9 (- j 1)))
	    ((>= i j) vec)
	  (set! n (vector-ref vec i))
	  (vector-set! vec i (vector-ref vec j))
	  (vector-set! vec j n))
	#(9 8 7 6 5 4 3 2 1 0)))

(test (do ((i 0 (+ i 1))) (#t i) (error 'syntax-error "do evaluated its body?")) 0)
(test (do '() (#t 1)) 'error)
(test (do . 1) 'error)
(test (do ((i i i)) (i i)) 'error)
(test (do ((i 0 i (+ i 1))) (i i)) 'error)
(test (do ((i)) (#t i)) 'error)
(test (do ((i 0 (+ i 1))) #t) 'error)
(test (do 123 (#t 1)) 'error)
(test (do ((i 1)) (#t . 1) 1) 'error)
(test (do ((i 1) . 1) (#t 1) 1) 'error)
(test (do ((i 1) ()) (= i 1)) 'error)
(test (do ((i 0 . 1)) ((= i 1)) i) 'error)
(test (do ((i 0 (+ i 1))) ((= i 3)) (set! i "hiho")) 'error)
(test (let ((do+ +)) (do ((i 0 (do+ i 1))) ((= i 3)) (set! do+ abs))) 'error)
(test (do () . 1) 'error)
(test (do ((i)) (1 2)) 'error)
(test (do (((i))) (1 2)) 'error)
(test (do ((i 1) ((j))) (1 2)) 'error)
(test (do (((1))) (1 2)) 'error)
(test (do ((pi 1 2)) (#t pi)) 'error)
(test (do ((1+i 2 3)) (#t #t)) 'error)
(test (do ((1.2 2 3)) (#t #t)) 'error)
(test (do (((1 . 2) "hi" (1 2))) (#t 1)) 'error)
(test (do ((() () ())) (#t #t)) 'error)
(test (do (("hi" "hi")) ("hi")) 'error)
(test (do ((:hi 1 2)) (#t :hi)) 'error)
(test (do ((i 0 (abs ()))) ((not (= i 0)) i)) 'error)
(test (do ((i j) (j i)) (i i)) 'error)
(test (do ((i 0 0) . ((j 0 j))) (#t j)) 0)
(test (do ((i 0 1 . 2)) (#t i)) 'error)
(test (do ((i 0 "hi")) ((string? i) . i)) 'error)
(test (do ((i 0 j)) (#t i)) 0) ; guile also -- (do ((i 0 (abs "hi"))) (#t i)) etc (do ((i 0 1)) (#t i) (abs "hi"))
(test (do ((i 0 1) . (j 0 0)) ((= i 1) i) i) 'error)
(test (do ((i 0 1) ((j 0 0)) ((= i 1) i)) i) 'error)
(test (do #f) 'error)
(test (do () #f) 'error)
(test (do () #()) 'error)
(test (do '((i 1)) ()) 'error)
(test (do #() ()) 'error)
(test (do ((#() 1)) ()) 'error)
(test (do ((1)) ()) 'error)
(test (do ((i 1) . #(a 1)) ()) 'error)
(test (do () ((3 4))) 'error)
(test (do ((i 1)) '()) ())
(test (do . (() (#t 1))) 1)
(test (do () . ((#t 1))) 1)
(test (do ((i 1 (+ i 1))) . ((() . ()))) ())
(test (do ((a . 1) (b . 2)) () a) 'error)
(let () (define (d1) (do ((i 0 (+ i 1))) ((= i 3) . i) (display i))) (test (d1) 'error))
(let () (define (d1) (do ((i 0 (+ i 1))) ((= i 3) i) . i)) (test (d1) 'error))

(test (define-constant) 'error)
(test (define-constant _asdf_ 2 3) 'error)
(test (define-constant pi 3) 'error) ; except in Utah
(test (define-constant pi . 3) 'error)
(define-constant __do_step_var_check__ 1)
(test (do ((__do_step_var_check__ 2 3)) (#t #t)) 'error)
(test (let ((__do_step_var_check__ 2)) 1) 'error)
(test (let () (set! __do_step_var_check__ 2)) 'error)
(test (let ((__a_var__ 123))
	(set! (symbol-access '__a_var__) (lambda (val sym) 0))
	(set! __a_var__ -1123))
      0)
(test (do ((hi #3d(((1 2) (3 4)) ((5 6) (7 8))) (hi 1))) ((equal? hi 8) hi)) 8)
(test (do ((i 0 ('((1 2) (3 4)) 0 1))) ((not (= i 0)) i)) 2)
(test (do () (#t (+ 1 2 3))) 6)
(test (do ((f + *) (j 1 (+ j 1))) ((= j 2) (apply f (list j j)))) 4)
(test (do ((f lambda) (j 1 (+ j 1))) ((= j 2) ((f (a) (+ a j)) 3))) 5)

(let ()
  (define-macro (add-1 x) `(+ ,x 1))
  (test (do ((i 0 (add-1 i))) ((= i 3) i)) 3)
  (test (do ((i 0 (add-1 i))) ((= i 3) (add-1 i))) 4))

(test (let ((j #f))
	(do ((i 0 (let ((x 0))
		    (dynamic-wind
			(lambda ()
			  (set! x i))
			(lambda ()
			  (+ x 1))
			(lambda ()
			  (if (> x 3)
			      (set! j #t)))))))
	    (j i)))
      5)
(test (let ((j 0)) (do ((i 0 (eval-string "(+ j 1)"))) ((= i 4) j) (set! j i))) 3)
(test (do ((i (do ((i (do ((i 0 (+ i 1)))
			  ((= i 3) (+ i 1)))
		      (do ((j 0 (+ j 1)))
			  ((= j 3)) (+ j i))))
		  ((> (do ((k 0 (+ k 1)))
			  ((= k 2) (* k 4)))
		      (do ((n 0 (+ n 1)))
			  ((= n 3) n)))
		   (do ((m 0 (+ m 1)))
		       ((= m 3) (+ m i)))))
	      i))
	  ((> i 6) i))
      7)

(test (let ((L (list 1 2))) 
	(do ((sum 0 (+ sum (car lst))) 
	     (i 0 (+ i 1)) 
	     (lst L (cdr lst))) 
	    ((or (null? lst) 
		 (> i 10)) 
	     sum) 
	  (set-cdr! (cdr L) L))) 
      16)

;;; optimizer checks
(num-test (let ((x 0)) (do ((i 1.0 (+ i 1))) ((> i 3)) (set! x (+ x i))) x) 6.0)
(num-test (let ((x 0)) (do ((i 1 4)) ((> i 3)) (set! x (+ x i))) x) 1)
(num-test (let ((x 0)) (do ((i 1 ((if #t + -) i 1))) ((> i 3)) (set! x (+ x i))) x) 6)
(num-test (let ((x 0)) (do ((i 1 (+))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 1 (+ 1))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 1 (+ 1 i 2))) ((> i 10)) (set! x (+ x i))) x) 22)
(num-test (let ((x 0)) (do ((i 1 (+ 1.0 i))) ((> i 3)) (set! x (+ x i))) x) 6.0)
(num-test (let ((x 0)) (do ((i 1 (+ 1 pi))) ((> i 2)) (set! x (+ x i))) x) 1)
(num-test (do ((i 0 (+ 1 pi))) ((> i 2) i)) (+ pi 1.0))
(num-test (let ((x 0)) (do ((i 0 (+ i 8796093022208))) ((> i 0)) (set! x (+ x i))) x) 0)
(num-test (let ((x 0)) (do ((i 0 (+ i 8796093022208))) ((> i 17592186044416)) (set! x (+ x i))) x) (+ (expt 2 44) (expt 2 43)))
(num-test (let ((x 0)) (do ((i 1 (* i 2))) ((> i 10)) (set! x (+ x i))) x) 15)
(num-test (do ((i 0 (+ i 1))) ((> i 2) i) (set! i (+ i 3.0))) 4.0)
(num-test (let ((x 0)) (let ((add +)) (do ((i 0 (add i 1))) ((< i -2)) (set! add -) (set! x (+ x i)))) x) -3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)))) 3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! i (* i 1.0)))) 3.0)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! equals >))) 6)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) x) (set! x (+ x i)) (set! equals =))) 3)
(num-test (let ((equals =) (x 0)) (do ((i 0 (+ i 1))) ((equals i 3) (set! x (+ x 1)) x) (set! x (+ x i)) (set! equals =))) 4)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i) (set! i (expt 2 60))) (+ 1 (expt 2 60)))
(num-test (let ((x 0) (n 3)) (do ((i 0 (+ i 1))) ((= i n) x) (set! x (+ x i)))) 3)
(num-test (let ((x 0) (n 3)) (do ((i 0 (+ i 1))) ((= 1 1) x) (set! x (+ x i)))) 0)
(num-test (let ((x 0) (n (expt 2 50))) (do ((i 0 (+ i n))) ((= i (expt 2 51)) x) (set! x (+ x i)))) (expt 2 50))
(num-test (let ((x 0) (n 31.0)) (do ((i 0 (+ i 1))) ((= i n) x) (set! x (+ x i)) (set! n 3))) 3)
(num-test (let ((x 0)) (do ((i 0 (+ i 1/2))) ((= i 3) x) (set! x (+ x i)))) 15/2)
(num-test (let ((x 0)) (do ((i 0 (+ i 1+i))) ((> (magnitude i) 3) x) (set! x (+ x i)))) 3+3i)
(num-test (call-with-exit (lambda (r) (do () () (r 1)))) 1)
(num-test (call-with-exit (lambda (r) (do () (#t 10 14) (r 1)))) 14)
(num-test (do ((i 0 (+ i 1))) (#t 10 12)) 12)
(num-test (do ((i 0 (+ i 1))) ((= i 3) i)) 3)
(num-test (do ((i 0 (+ i 1))) ((> i 3) i)) 4)
(num-test (do ((i 0 (+ i 1))) ((< i 3) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((<= i 3) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((>= i 3) i)) 3)
(num-test (do ((i 0 (+ i 1))) ((>= 3 i) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((> 3 i) i)) 0)
(num-test (do ((i 0 (+ i 1))) ((< 3 i) i)) 4)
(num-test (do ((i 0 (+ i 1))) ((<= 3 i) i)) 3)
(num-test (let ((n 3)) (do ((i 0 (+ i 1))) ((> i n) i))) 4)
(num-test (let ((n 3)) (do ((i 0 (+ i 1))) ((< n i) i))) 4)
(num-test (do ((i 10 (- i 1))) ((= i 0) i)) 0)
(num-test (do ((i 10 (- 1 i))) ((< i 0) i)) -9)
(num-test (do ((i 10 (- i 3))) ((< i 0) i)) -2)
(let () (define (hi) (do ((i 1 (+ 1 i))) ((= i 1) i))) (hi) (test (hi) 1))
(let () (define (hi) (do ((i 10 (+ i 1))) ((= i 10) i) (abs i))) (hi) (test (hi) 10))
(let ((sum 0)) (define (hi) (do ((i 10 (+ i 1))) ((= i 10) i) (set! sum (+ sum i)))) (hi) (test (hi) 10))
(let () (define (hi a) (do ((i a (+ i 1))) ((= i a) i) (+ a 1))) (hi 1) (test (hi 1) 1))
(let () 
  (define (fx) 
    (let ((iter (make-iterator #(1 2 3)))) 
      (do () ((or (string? (iterate iter)) (iterator-at-end? iter)) (not (iterator-at-end? iter))))))
  (test (fx) #f))
(let ()
  (define (fx1)
    (let ((iter (make-iterator #(1 2 3)))) 
      (do () ((or (= (iterate iter) 2) (iterator-at-end? iter)) (iterate iter)))))
  (test (fx1) 3))
(let ()
  (define (fx2)
    (let ((iter (make-iterator #(1 2 3)))) 
      (do () ((= (iterate iter) 2) (iterate iter)))))
  (test (fx2) 3))
(let ()
  (define (fdo1)
    (let ((abs (lambda (x) (+ x 1)))
	  (x '(1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 1))
	(if (not (equal? (map abs x) '(2 3 4)))
	    (display "fdo1 map case")))
      (do ((i 0 (+ i 1)))
	  ((= i 1))
	(if (not (equal? (for-each abs x) #<unspecified>))
	    (display "fdo1 for-each case")))))
  (define (fdo2)
    (let ((abs (lambda (x y) (= x y)))
	  (x '(1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 1))
	(if (not (member 2 x abs))
	    (display "fdo2 member case")))))
  (define (fdo3)
    (let ((abs (lambda (x y) (= x y)))
	  (x '((1 a) (2 b) (3 c))))
      (do ((i 0 (+ i 1)))
	  ((= i 1))
	(if (not (assoc 2 x abs))
	    (display "fdo3 assoc case")))))
  (define (fdo4)
    (let ((abs (lambda (x y) (> x y)))
	  (x (list 1 2 3)))
      (do ((i 0 (+ i 1)))
	  ((= i 1))
	(if (not (equal? (sort! x abs) '(3 2 1)))
	    (display "fdo4 sort! case")))))
  (fdo1)
  (fdo2)
  (fdo3)
  (fdo4))

(let () (define (fdo5) (do ((si () '())) ((null? si) 'mi))) (test (fdo5) 'mi))
(let () (define (fdo5) (do ((si '() '())) ((null? si) 'mi))) (test (fdo5) 'mi))
(let () (define (fdo5) (do ((si () ())) ((null? si) 'mi))) (test (fdo5) 'mi))
(let () (define (fdo5) (do ((si () ())) ((null? si) 'mi))) (test (fdo5) 'mi))


;;; check an optimizer bug
(define _do_call_cc_end_ 1)
(define (call-cc-do-test)
  (do ((i 0 (+ i 1)))
      ((= i _do_call_cc_end_))
    (let ((ctr 0)) 
      (call/cc (lambda (exit) 
		 (if (> 3 2) 
		     (let () 
		       (exit ctr) 
		       (set! ctr 100) ctr) 
		     #f)))))
  (do ((i 0 (+ 1 i)))
      ((= i _do_call_cc_end_))
    (let ((ctr 0)) 
      (call/cc (lambda (exit) 
		 (if (> 3 2) 
		     (let () 
		       (exit ctr) 
		       (set! ctr 100) ctr) 
		     #f))))))
(call-cc-do-test)

;;; and another
(let()
  (define (hi)
    (let ((checker (lambda (nlst v)
		     (let ((chr (car nlst)))
		       (if (not (char-alphabetic? chr))
			   (if (not (char=? v chr))
			       (format-logged #t ";(char-downcase #\\~A) -> ~A" chr v))
			   (if (and (not (char=? chr v))
				    (not (char=? chr (char-upcase v))))
			       (format-logged #t ";(char-downcase #\\~A) -> ~A~%" chr v))))))
	  (result 0))
      (let ((try 0))
	(do ((i 0 (+ i 1)))
	    ((> i 10))
	  (set! try i)
	  (checker '(#\a) #\a)
	  (checker '(#\a) #\a)))))
  (test (hi) ()))


(define (__a-func__ a)
  (format-logged #t ";oops called first a-func by mistake: ~A~%" a)
  (if (> a 0)
      (__a-func__ (- a 1))))

(define (__a-func__ a)
  (if (> a 0)
      (__a-func__ (- a 1))))

(__a-func__ 3)

(define (__c-func__ a)
  (format-logged #t ";oops called first __c-func__ by mistake: ~A~%" a)
  (if (> a 0)
      (__c-func__ (- a 1))))
  
(let ()
  (define (__c-func__ a)
    (if (> a 0)
	(__c-func__ (- a 1))))
  
  (__c-func__ 3))

;;; more optimizer checks
(let () 
  (define (do-test-1) (do ((i 0 (+ i 1))) ((= i 10)) (display i)))
  (test (with-output-to-string (lambda () (do-test-1))) "0123456789"))

(let () 
  (define (do-test-2) (do ((i 0 (+ 1 i))) ((= i 10)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-2))) "0123456789"))

(let ((start 0)) 
  (define (do-test-3) (do ((i start (+ i 1))) ((= i 10)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-3))) "0123456789"))

(let ((start 0) (end 10)) 
  (define (do-test-4) (do ((i start (+ i 1))) ((= i end)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-4))) "0123456789"))

(let ((start 0) (end 10)) 
  (define (do-test-5) (do ((i start (+ i 1))) ((= end i)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-5))) "0123456789"))

(let () 
  (define (do-test-6) (do ((i 0 (+ i 1))) ((= i 10)) (let ((k i)) (display k)))) 
  (test (with-output-to-string (lambda () (do-test-6))) "0123456789"))

(let () 
  (define (do-test-7) (do ((i 0 (+ i 2))) ((= i 20)) (display (/ i 2)))) 
  (test (with-output-to-string (lambda () (do-test-7))) "0123456789"))

(let () 
  (define (do-test-8) (do ((i 0 (+ i 1))) ((= i 10)) (let ((a (+ 1 2))) (display #\0)))) 
  (test (with-output-to-string (lambda () (do-test-8))) "0000000000"))

(let () 
  (define (do-test-9) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (set! j i) (display j)))) 
  (test (with-output-to-string (lambda () (do-test-9))) "0123456789"))

(let () 
  (define (do-test-10) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (display i)))) 
  (test (with-output-to-string (lambda () (do-test-10))) "0123456789"))

(let () 
  (define (do-test-11) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j 0)) (set! j 32) (display i)))) 
  (test (with-output-to-string (lambda () (do-test-11))) "0123456789"))

(let ()
  (define (do-test-12) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j i)) (display j)))) 
  (test (with-output-to-string (lambda () (do-test-12))) "0123456789"))

(let () 
  (define (do-test-13) (do ((i 0 (+ i 1))) ((= i 5)) (let ((j (+ i 1))) (let ((i j)) (display (- i 1)))))) 
  (test (with-output-to-string (lambda () (do-test-13))) "01234"))

(let () 
  (define (do-test-14) (do ((i 0 (+ i 1))) ((= i 10)) (set! i (+ i 1)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-14))) "13579"))

(let ((lst ()))
  (define (do-test-15) (set! lst ()) (do ((i 0 (+ i 1))) ((= i 10)) (set! lst (cons i lst))) lst) 
  (test (do-test-15) '(9 8 7 6 5 4 3 2 1 0)))

(let ((lst ()))
  (define (do-test-15a) (set! lst ()) (do ((i 0 (+ i 1))) ((= i 10)) (set! lst (append (list i) lst))) lst) 
  (test (do-test-15a) '(9 8 7 6 5 4 3 2 1 0)))

(let ((lst (list 9 8 7 6 5 4 3 2 1 0))) 
  (define (do-test-16) (do ((i 0 (+ i 1))) ((= i 10)) (list-set! lst i i)) lst) 
  (test (do-test-16) '(0 1 2 3 4 5 6 7 8 9)))

(let ((lst ())) 
  (define (do-test-17) (set! lst ()) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j i)) (set! lst (cons j lst)))) lst) 
  (test (do-test-17) '(9 8 7 6 5 4 3 2 1 0)))

(let ((lst ())) 
  (define (do-test-17a) (set! lst ()) (do ((i 0 (+ i 1))) ((= i 10)) (let ((j (min i 100))) (set! lst (cons j lst)))) lst) 
  (test (do-test-17a) '(9 8 7 6 5 4 3 2 1 0)))

(let () 
  (define (do-test-18) (do ((i 0 (+ i 1)) (j 0)) ((= i 10) j) (if (= i 3) (set! j i))))
  (test (do-test-18) 3))

(let ((end 10)) 
  (define (do-test-19) (do ((i 0 (+ i 1))) ((= i end)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-19))) "0123456789"))

(let ((end 10)) 
  (define (do-test-19A) (do ((i 0 (+ 1 i))) ((= end i)) (display i))) 
  (test (with-output-to-string (lambda () (do-test-19A))) "0123456789"))

(let ((end 10)) 
  (define (do-test-20) (do ((i 0 (+ i 1))) ((= i end)) (set! end 8) (display i))) 
  (test (with-output-to-string (lambda () (do-test-20))) "01234567"))

(let ((end 10)) 
  (define (do-test-20A) (do ((i 0 (+ 1 i))) ((= end i)) (set! end 8) (display i))) 
  (test (with-output-to-string (lambda () (do-test-20A))) "01234567"))

(let () 
  (define (do-test-21) (do ((i 0 (+ i 1))) ((= i 3)) (with-let (rootlet) (+ 1 2)))) 
  (do-test-21))

(let ((v (vector 0 0 0))) 
  (define (hi a) (do ((i 0 (+ i 1))) ((> i a)) (vector-set! v i 1))) (hi 2) 
  (test v (vector 1 1 1)))

(let () ; dotimes_c_c case can't involve set so we use write-char
  (define (hi a) 
    (do ((i 0 (+ i 1))) 
	((= i a)) 
      (write-char #\a))) 
  (with-output-to-file tmp-output-file 
    (lambda () 
      (hi 3)))
  (let ((str (with-input-from-file tmp-output-file 
	       (lambda () 
		 (read-line)))))
    (test str "aaa")))

(let ()
  (define (do-test-22) (do ((i 0 (+ i 1))) ((= i 10)) (display i)))
  (test (with-output-to-string (lambda () (do-test-22))) "0123456789"))

(let ((v (make-list 10)))
  (define (do-test-23) (do ((i 0 (+ i 1))) ((= i 10)) (list-set! v i i)))
  (do-test-23) 
  (test v '(0 1 2 3 4 5 6 7 8 9)))

;;; safe simple h_safe_c_s
(let ()
  (define (do-test-24) (do ((i 0 (+ i 1))) ((> i 10)) (display i)))
  (test (with-output-to-string (lambda () (do-test-24))) "012345678910"))

;;; safe simple h_safe_c_ss
(let ()
  (define (do-test-25 p) (do ((i 0 (+ i 1))) ((> i 10)) (display i p)))
  (test (call-with-output-string (lambda (p) (do-test-25 p))) "012345678910"))

;;; safe simple h_safe_c_c
(let ()
  (define (do-test-26) (do ((i 0 (+ i 1))) ((> i 10)) (display 0)))
  (test (with-output-to-string (lambda () (do-test-26))) "00000000000"))

;;; safe simple h_safe_c_opsq_s
(let ()
  (define (do-test-27 p) (do ((i 0 (+ i 1))) ((> i 10)) (display (- i) p)))
  (test (call-with-output-string (lambda (p) (do-test-27 p))) "0-1-2-3-4-5-6-7-8-9-10"))

(let ()
  (define (do-test-22 i o) (catch #t (lambda () (do () () (write-char (read-char i) o))) (lambda err (get-output-string o))))
  (test (call-with-output-string (lambda (out) (call-with-input-string "0123" (lambda (in) (do-test-22 in out))))) "0123"))





;;; --------------------------------------------------------------------------------
;;; set!
;;; --------------------------------------------------------------------------------

(test (let ((a 1)) (set! a 2) a) 2)
(for-each
 (lambda (arg)
   (test (let ((a 0)) (set! a arg) a) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #f #t (list 1 2 3) '(1 . 2)))

(test (let ((a 1)) (call/cc (lambda (r) (set! a (let () (if (= a 1) (r 123)) 321)))) a) 1)
(test (let ((a (lambda (b) (+ b 1)))) (set! a (lambda (b) (+ b 2))) (a 3)) 5)
(test (let ((a (lambda (x) (set! x 3) x))) (a 1)) 3)

(test (let ((x (vector 1 2 3))) (set! (x 1) 32) x) #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x)))
	(set! ((y) 1) 32)
	x)
      #(1 32 3))
(test (let* ((x (vector 1 2 3))
	     (y (lambda () x))
	     (z (lambda () y)))
	(set! (((z)) 1) 32)
	x)
      #(1 32 3))

(test (let ((a 1)) (set! a)) 'error)
(test (let ((a 1)) (set! a 2 3)) 'error)
(test (let ((a 1)) (set! a . 2)) 'error)
(test (let ((a 1)) (set! a 1 . 2)) 'error)
(test (let ((a 1)) (set! a a) a) 1)
(test (set! "hi" 1) 'error)
(test (set! 'a 1) 'error)
(test (set! 1 1) 'error)
(test (set! (list 1 2) 1) 'error)
(test (set! (let () 'a) 1) 'error)
(test (set!) 'error)
(test (set! #t #f) 'error)
(test (set! () #f) 'error)
(test (set! #(1 2 3) 1) 'error)
(test (set! (call/cc (lambda (a) a)) #f) 'error)
(test (set! 3 1) 'error)
(test (set! 3/4 1) 'error)
(test (set! 3.14 1) 'error)
(test (set! #\a 12) 'error)
(test (set! (1 2) #t) 'error)
(test (set! _not_a_var_ 1) 'error)
(test (set! (_not_a_pws_) 1) 'error)
(test (let ((x 1)) (set! ((lambda () 'x)) 3) x) 'error)
(test (let ((x '(1 2 3))) (set! (((lambda () 'x)) 0) 3) x) '(3 2 3))
(test (let ((x '(1 2 3))) (set! (((lambda () x)) 0) 3) x) '(3 2 3)) ; ?? 
(test (let ((x '(1 2 3))) (set! ('x 0) 3) x) '(3 2 3)) ; ???  I suppose that's similar to
(test (let ((x '((1 2 3)))) (set! ((car x) 0) 3) x) '((3 2 3)))
(test (let ((x '((1 2 3)))) (set! ('(1 2 3) 0) 32) x) '((1 2 3))) ; this still looks wrong... (expands to (list-set! '(1 2 3) 0 3) I guess)

(test (let ((a (lambda (x) (set! a 3) x))) (list (a 1) a)) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error)            
(test (let ((a (lambda () "hi"))) (set! (a) "ho")) 'error)
(test (let ((a (let ((b 1)) (set! a 3) b))) a) 'error) 

(test (set! . -1) 'error)
(test (set!) 'error)
(test (let ((x 1)) (set! x x x)) 'error)
(test (let ((x 1)) (set! x x) x) 1)
(test (set! set! 123) 'error)
(test (set! (cons 1 2) 3) 'error)
(test (let ((var 1) (val 2)) (set! var set!) (var val 3) val) 3)
(test (let ((var 1) (val 2)) (set! var +) (var val 3)) 5)
(test (let ((sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 1)
	    (sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456780 3))
	(set! sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789 2)
	sym0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789)
      2)

(test (let ((x '(1)) (y '(2))) (set! ((if #t x y) 0) 32) x) '(32))
(test (let ((hi 0)) (set! hi 32)) 32)
(test (let ((hi 0)) ((set! hi ('((1 2) (3 4)) 0)) 0)) 1)

(test (set! #<undefined> 1) 'error)
(test (set! #<eof> 1) 'error)
(test (set! #<unspecified> 1) 'error)
(test (let ((x 0)) (define-macro (hi) 'x) (set! (hi) 3) x) 'error)

(test (set! ("hi" . 1) #\a) 'error)
(test (set! (#(1 2) . 1) 0) 'error)
(test (set! ((1 . 2)) 3) 'error)
(test (let ((lst (list 1 2))) (set! (lst . 0) 3) lst) 'error)
(test (let ((lst (list 1 2))) (set! (list-ref lst . 1) 2)) 'error)
(test (let ((v #2d((1 2) (3 4)))) (set! (v 0 . 0) 2) v) 'error)
(test (set! ('(1 2) . 0) 1) 'error)
(test (set! ('(1 2) 0) 3) 3)
(test (set! (''(1 . 2)) 3) 'error)
(test (set! (''(1 2)) 3) 'error)
(test (set! ('(1 . 2)) 3) 'error)
(test (set! ('(1 2)) 3) 'error)
(test (set! (''(1 2) 0 0) 3) 'error)
(test (set! (#(1 2) 0 0) 3) 'error)
(test (let ((x 1)) (set! (quasiquote . x) 2) x) 'error)
(test (let ((x 1)) (set! (quasiquote x) 2) x) 'error)
(test (set! `,(1) 3) 'error)
(test (set! (1) 3) 'error)
(test (set! `,@(1) 3) 'error)
(test (let ((x 0)) (set! x 1 . 2)) 'error)
(test (let ((x 0)) (apply set! x '(3))) 'error) ; ;set!: can't alter immutable object: 0
(test (let ((x 0)) (apply set! 'x '(3)) x) 3)
(test (set! (#(a 0 (3)) 1) 0) 0)
(test (set! ('(a 0) 1) 0) 0)
(test (apply set! (apply list (list ''(1 2 3) 1)) '(32)) 32)
(test (set! (let ((x 1)) x) 3) 'error)
(test (set! (lambda () 1) 4) 'error)
(test (set! (with-baffle (display x)) 5) 'error)
(test (set! (define x 3) 6) 'error)

(let ()
  (define-macro (symbol-set! var val) `(apply set! ,var (list ,val))) ; but this evals twice
  (test (let ((x 32) (y 'x)) (symbol-set! y 123) (list x y)) '(123 x)))
(let ()
  (define-macro (symbol-set! var val) ; like CL's set
    `(apply set! ,var ',val ()))
  (test (let ((var '(+ 1 2)) (val 'var)) (symbol-set! val 3) (list var val)) '(3 var))
  (test (let ((var '(+ 1 2)) (val 'var)) (symbol-set! val '(+ 1 3)) (list var val)) '((+ 1 3) var)))

(test (set! ('(1 2) 1 . 2) 1) 'error)
(test (set! ('((1 2) 1) () . 1) 1) 'error)
(test (set! ('(1 1) () . 1) 1) 'error)

(test (let () (define (hi) (let ((x 1000)) (set! x (+ x 1)) x)) (hi) (hi)) 1001)
(test (let () (define (hi) (let ((x 1000.5)) (set! x (+ x 1)) x)) (hi) (hi)) 1001.5)
(test (let () (define (hi) (let ((x 3/2)) (set! x (+ x 1)) x)) (hi) (hi)) 5/2)
(test (let () (define (hi) (let ((x 3/2)) (set! x (- x 1)) x)) (hi) (hi)) 1/2)
(test (let () (define (hi) (let ((x 3/2)) (set! x (- x 2)) x)) (hi) (hi)) -1/2)
(test (let () (define (hi) (let ((x "asdf")) (set! x (+ x 1)) x)) (hi) (hi)) 'error)

(let ()
  ;; check an optimizer bug
  (define (bad-increment a b)
    (cons a b))

  (define (use-bad-increment b)
    (let ((x ()))
      (set! x (bad-increment x b))
      x))

  (use-bad-increment 1)
  (use-bad-increment 1)
  (use-bad-increment 1))




;;; --------------------------------------------------------------------------------
;;; or
;;; --------------------------------------------------------------------------------

(test (or (= 2 2) (> 2 1)) #t)
(test (or (= 2 2) (< 2 1)) #t)
(test (or #f #f #f) #f)
(test (or) #f)
(test (or (memq 'b '(a b c)) (+ 3 0)) '(b c))
(test (or 3 9) 3)
(test (or #f 3 asdf) 3) ; "evaluation stops immediately"
(test (or 3 (/ 1 0) (display "or is about to exit!") (exit)) 3)

(for-each
 (lambda (arg)
   (test (or arg) arg)
   (test (or #f arg) arg)
   (test (or arg (error 'test-error "oops or ")) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) #<eof> #<unspecified> '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (or (eof-object? val)
		  (> loc 1000) ; try to avoid the read-error stuff
		  (begin
		    (set! loc (+ loc 1))
		    (loop (read-char p)))))
	    (> loc 1000))))
      #t)

(test (or (and (or (> 3 2) (> 3 4)) (> 2 3)) 4) 4)
(test (or or) or)
(test (or (or (or))) #f)
(test (or (or (or) (and))) #t)
(test (let ((a 1)) (or (let () (set! a 2) #f) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) 3)
(test (or '#f ()) ())
(test (call/cc (lambda (r) (or #f (> 3 2) (r 123) 321))) #t)
(test (call/cc (lambda (r) (or #f (< 3 2) (r 123) 321))) 123)
(test (+ (or #f (not (null? ())) 3) (or (zero? 1) 2)) 5)
(test (or 0) 0)
(test (if (or) 1 2) 2)

(test (or . 1) 'error)
(test (or #f . 1) 'error)
(test (or . (1 2)) 1)
(test (or . ()) (or))
; (test (or 1 . 2) 1) ; this fluctuates

(test (let () (or (define (hi a) a)) (hi 1)) 1)
(test (let () (or #t (define (hi a) a)) (hi 1)) 'error)
(test (let () (and (define (hi a) a) (define (hi a) (+ a 1))) (hi 1)) 2) ; guile agrees with this
(test ((lambda (arg) (arg #f 123)) or) 123)
(test (let ((oar or)) (oar #f 43)) 43)
(test (let ((oar #f)) (set! oar or) (oar #f #f 123)) 123)



;;; --------------------------------------------------------------------------------
;;; and
;;; --------------------------------------------------------------------------------

(test (and (= 2 2) (> 2 1)) #t)
(test (and (= 2 2) (< 2 1)) #f)
(test (and 1 2 'c '(f g)) '(f g))
(test (and) #t)
(test (and . ()) (and))
(test (and 3) 3)
(test (and (memq 'b '(a b c)) (+ 3 0)) 3)
(test (and 3 9) 9)
(test (and #f 3 asdf) #f) ; "evaluation stops immediately"
(test (and 3 (zero? 1) (/ 1 0) (display "and is about to exit!") (exit)) #f)
(test (if (and) 1 2) 1)
(test (if (+) 1 2) 1)
(test (if (*) 1 2) 1)
(test (and (if #f #f)) (if #f #f))
(test (let ((x '(1))) (eq? (and x) x)) #t)

(for-each
 (lambda (arg)
   (test (and arg) arg)
   (test (and #t arg) arg)
   (test (and arg #t) #t))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-input-file "s7test.scm"
	(lambda (p)
	  (let ((loc 0))
	    (let loop ((val (read-char p)))
	      (and (not (eof-object? val))
		   (< loc 1000)
		   (begin
		     (set! loc (+ loc 1))
		     (loop (read-char p)))))
	    (>= loc 1000))))
      #t)

(test (and (or (and (> 3 2) (> 3 4)) (> 2 3)) 4) #f)
(test (and and) and)
(test (and (and (and))) #t)
(test (and (and (and (and (or))))) #f)
(test (let ((a 1)) (and (let () (set! a 2) #t) (= a 1) (let () (set! a 3) #f) (and (= a 3) a) (let () (set! a 4) #f) a)) #f)
(test (and '#t ()) ())
(test (call/cc (lambda (r) (and #t (> 3 2) (r 123) 321))) 123)
(test (call/cc (lambda (r) (and #t (< 3 2) (r 123) 321))) #f)
(test (+ (and (null? ()) 3) (and (zero? 0) 2)) 5)

(test (and . #t) 'error)
(test (and 1 . 2) 'error)
(test (and . (1 2)) 2)

(test (let () (and (define (hi a) a)) (hi 1)) 1)
(test (let () (and #f (define (hi a) a)) (hi 1)) 'error)
(test (+ 1 (and (define (hi a) a) (hi 2))) 3)

;;; from some net site 
(let ()
  (define (fold fn accum list)
    (if (null? list)
	accum
	(fold fn
	      (fn accum
		  (car list))
	      (cdr list))))
  (test (fold and #t '(#t #f #t #t)) #f))

(test (let ((and! and)) (and! #f (error 'test-error "oops"))) #f)
(test (let ((and! #f)) (set! and! and) (and! #f (error 'test-error "oops"))) #f)
(test (let () (define (try and!) (and! #f (error 'test-error "oops"))) (try and)) #f)

;;; here are some tests from S. Lewis in the r7rs mailing list
(let ()
  (define myand and)
  (test (myand #t (+ 1 2 3)) 6)
  (define (return-op) and)
  (define myop (return-op))
  (test (myop #t (+ 1 2 3)) 6)
  (test (and #t (+ 1 2 3)) 6)
  (test ((return-op) #t (+ 1 2 3)) 6)
  (test ((and and) #t (+ 1 2 3)) 6)
  (define ops `(,* ,and))
  (test ((car ops) 2 3) 6)
  (test ((cadr ops) #t #f) #f)
  (test (and #f never) #f)
  (test (and #f and) #f)
  (test ((and #t and) #t (+ 1 2 3)) 6))





;;; --------------------------------------------------------------------------------
;;; cond
;;; --------------------------------------------------------------------------------

(test (cond ('a)) 'a)
(test (cond (3)) 3)
(test (cond (#f 'a) ('b)) 'b)
(test (cond (#t 'a) (#t 'b)) 'a)
(test (cond ((> 3 2) 'greater) ((< 3 2) 'less)) 'greater)
(test (cond((> 3 2)'greater)((< 3 2)'less)) 'greater)
(test (cond ((> 3 3) 'greater) ((< 3 3) 'less)  (else 'equal)) 'equal)
(test (cond ((assv 'b '((a 1) (b 2))) => cadr)  (else #f)) 2)
(test (cond (#f 2) (else 5)) 5)
(test (cond (1 2) (else 5)) 2)
(test (cond (1 => (lambda (x) (+ x 2))) (else 8)) 3)
(test (cond ((+ 1 2))) 3)
(test (cond ((zero? 1) 123) ((= 1 1) 321)) 321)
(test (cond ('() 1)) 1)
(test (let ((x 1)) (cond ((= 1 2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let((x 1))(cond((= 1 2)3)(else(* x 2)(+ x 3)))) 4)
(test (let ((x 1)) (cond ((= x 1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (cond ((= x 1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (cond ((= x 2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (cond ((= x 2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (cond ((= 1 2) 3) (else 4) (else 5)) 4) ; this should probably be an error
(test (cond (1 2 3)) 3)
(test (cond (1 2) (3 4)) 2)
(test (cond ((= 1 2) 3) ((+ 3 4))) 7)
(test (cond ((= 1 1) (abs -1) (+ 2 3) (* 10 2)) (else 123)) 20)
(test (let ((a 1)) (cond ((= a 1) (set! a 2) (+ a 3)))) 5)
(test (let ((a 1)) (cond ((= a 2) (+ a 2)) (else (set! a 3) (+ a 3)))) 6)
(test (cond ((= 1 1))) #t)
(test (cond ((= 1 2) #f) (#t)) #t)
(test (cond ((+ 1 2))) 3)
(test (cond ((cons 1 2))) '(1 . 2))
(test (cond (#f #t) ((string-append "hi" "ho"))) "hiho")
(test (cond ('() 3) (#t 4)) 3)
(test (cond ((list) 3) (#t 4)) 3)
;;; (cond (1 1) (asdf 3)) -- should this be an error?
(test (cond (+ 0)) 0)
(test (cond (lambda ())) ())
(test (cond . ((1 2) ((3 4)))) 2)
(test (cond (define #f)) #f)
(test (let () (cond ((> 2 1) (define x 32) x) (#t 1)) x) 32) ; ? a bit strange
(test (let ((x 1)) (+ x (cond ((> x 0) (define x 32) x)) x)) 65)
(test (cond (("hi" 1))) #\i)
(test (cond (()())) ())
(test (let ((a 0)) (let ((b (lambda () (set! a 1) #f))) (cond ((> a 0) 3) ((b) 4) ((> a 0) 5) (#t 6)))) 5)
(test (let ((a #t)) (let ((b (lambda () (set! a (not a)) a))) (cond ((b) 1) ((b) 2) (t 3)))) 2)
(test (let ((otherwise else)) (cond ((= 1 2) 1) (otherwise 3))) 3)
(test (let ((otherwise #t)) (cond ((= 1 2) 1) (otherwise 3))) 3) ; or actually anything... 12 for example
(test (let ((else #f)) (cond ((= 1 2) 1) (else 3))) #<unspecified>) ; was () -- 31-Dec-15
(test (let ((else #f)) (cond ((= 1 2) 1) (#_else 3))) 3)
(test (let ((else 1)) (let ((otherwise else)) (case 0 (otherwise 1)))) 'error)
(test (let ((x 1)) (cond ((< x 0) 1))) #<unspecified>)

(for-each
 (lambda (arg)
   (test (cond ((or arg) => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((+ 1 2) => (lambda (x) (+ 1 x)))) 4)
(test (cond ((cons 1 2) => car)) 1)
(test (cond ((values 1 2) => +)) 3)
(test (cond (1 2 => +)) 'error)
(test (cond ((begin 1 2) => +)) 2)
(test (cond ((values -1) => abs)) 1)
(test (cond ((= 1 2) => +) (#t => not)) #f)
(test (cond ((* 2 3) => (let () -))) -6)
(test (cond ((* 2 3) => (cond ((+ 3 4) => (lambda (a) (lambda (b) (+ b a))))))) 13)
(test (let ((x 1)) ((cond ((let () (set! x 2) #f) => boolean?) (lambda => (lambda (a) (apply a '((b) (+ b 123)))))) x)) 125)
(test (cond ((values 1 2 3) => '(1 (2 3 (4 5 6 7 8))))) 7)
(test (cond ((values 1 2 3) => +)) 6)
(test (cond ((values #f #f) => equal?)) #t) ; (values #f #f) is not #f
(test (let () (cond (#t (define (hi a) a))) (hi 1)) 1)
(test (let () (cond (#f (define (hi a) a))) (hi 1)) 'error)
(test (let () (cond ((define (hi a) a) (hi 1)))) 1)
(test (cond ((assq 'x '((x . 1) (y . 0))) => abs (display 'oops)) (else #f)) 'error)

(test (cond (else 1)) 1)
(test (call/cc (lambda (r) (cond ((r 4) 3) (else 1)))) 4)
(test (cond ((cond (#t 1)))) 1)
(test (symbol? (cond (else else))) #f)
(test (equal? else (cond (else else))) #t)
(test (cond (#f 2) ((cond (else else)) 1)) 1)
(test (let ((x #f) (y #t)) (cond (x 1) (y 2))) 2)
(test (cond ((- 3 2)) ((< 2 3))) (or (- 3 2) (< 3 2))) ; (cond (e) ...)) is the same as (or e ...)

(for-each
 (lambda (arg)
   (test (cond (#t arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (#f 1) (else arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (cond (arg => (lambda (x) x))) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (cond ((let () 1) => (let ((x 2)) (lambda (n) (+ n x))))) 3)
(test (cond ((let () 1) => (let ((x 2)) (cond (3 => (let ((y 4)) (lambda (n) (lambda (m) (+ n m x y))))))))) 10)

(test (let ((=> 3)) (cond (=> =>))) 3)
(test (cond (cond 'cond)) 'cond)
(test (cond (3 => (lambda args (car args)))) 3)
(test (cond (3 => (lambda (a . b) a))) 3)
(test (cond ((list 3 4) => (lambda (a . b) b))) ())
(test (cond) 'error)
(test (let () (define-macro (mac x) `(+ ,x 1)) (cond (1 => mac))) 2)
					;(test (cond ((= 1 2) 3) (else 4) (4 5)) 'error) ; trailing ignored 
(test (cond ((+ 1 2) => (lambda (a b) (+ a b)))) 'error)
(test (equal? (cond (else)) else) #t)
(test (cond (#t => 'ok)) 'error)
(test (cond (else =>)) 'error)
(test (cond ((values -1) => => abs)) 'error)
(test (cond ((values -1) =>)) 'error)
(test (cond (cond (#t 1))) 'error)
(test (cond 1) 'error)
(test (cond) 'error)
(test (cond (1 . 2) (else 3)) 'error)
(test (cond (#f 2) (else . 4)) 'error)
(test (cond ((values 1 2) => (lambda (x y) #t))) #t)
(test (cond #t) 'error)
(test (cond 1 2) 'error)
(test (cond 1 2 3) 'error)
(test (cond 1 2 3 4) 'error)
(test (cond (1 => (lambda (x y) #t))) 'error)
(test (cond . 1) 'error)
(test (cond ((1 2)) . 3) 'error)
(test (cond (1 => + abs)) 'error)
(test (cond (1 =>)) 'error)
(test (cond ((values 1 2) => + abs)) 'error)
(test (cond (else => symbol?)) #f) ; (symbol? else) -> #f
(test (eq? (cond (else => or)) else) #t)
(test (cond ((values #f 1) => or)) 1)
(test (+ (cond ((values 1 2 3)))) 6)
(test (let ((else 3)) (cond ((= else 3) 32) (#t 1))) 32)
(test (let ((else #f)) (cond (else 32) (#t 1))) 1)
(test (cond #((1 2))) 'error)
(test (cond (/ 0)) 0) ; / is not #f, so return 0
(test (cond (string-ref 2)) 2) 

(test (let ((=> 3)) (cond (1 =>))) 3)
(test (let ((=> 3)) (cond (1 => abs))) abs)
(test (let ((=> 3) (else 4)) (cond (else => abs))) abs)
(test (let ((=> 3)) (cond (1 => "hi"))) "hi")

(test (let ((=> 3)) (cond (12 => abs) (else #f))) abs)              ; guile/chicken/chibi agree
(test (let ((else #f)) (cond ((< 2 1) 3) (else 4))) #<unspecified>) ;  same 
(test (let ((=> 3)) (case => ((3) => abs) (else #f))) abs)
(test (let ((else #f)) (case 3 ((2 1) 3) (else 4))) 4)              ; chibi says #f (its choice for the unspecified value), Guile 4, chicken error
(test (let ((else #f)) (cond ((< 2 1) 1) (else 3) (#t 4))) 4)       ; ! (same for chibi)
;; to run chibi repl, goto /home/bil/test/chibi-scheme-master, setenv LD_LIBRARY_PATH /home/bil/test/chibi-scheme-master, chibi-scheme
;; to run chicken, goto /home/bil/test/chicken-4.7.0.6/, csi

(test (let ((x 0))
	(cond ((let ((y x)) (set! x 1) (= y 1)) 0)
	      ((let ((y x)) (set! x 1) (= y 1)) 1)
	      (#t 2)))
      1)

(test (cond 
       ((cond 
          ((cond ((assoc 'x '((x 1 2 3))) => cdr)
                 (else #f))
            => apply-values) ; ouch!
          (else #f))
         => +)
       (else #f))
      6)

(let ((c1 #f)
      (x 1))
  (let ((y (cond ((let ()
		    (call/cc
		     (lambda (r)
		       (set! c1 r)
		       (r x))))
		  => (lambda (n) (+ n 3)))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 324))))

(let ((c1 #f)
      (x 1))
  (let ((y (cond (x => (lambda (n) 
			 (call/cc
			  (lambda (r)
			    (set! c1 r)
			    (r (+ 3 x))))))
		 (#t 123))))
    (if (= y 4) (begin (set! x 2) (c1 321)))
    (test (list x y) '(2 321))))




;;; -------- cond-expand --------
;;; cond-expand

(test (let ()
	(cond-expand (guile )
		     (s7 (define (hi a) a)))
	(hi 1))
      1)
(test (let ((x 0))
	(cond-expand (guile (format-logged #t ";oops~%"))
		     (else (set! x 32)))
	x)
      32)
(test (let ()
	(cond-expand
	 (guile 
	  (define (hi a) (+ a 1)))
	 ((or common-lisp s7)
	  (define (hi a) a)))
	(hi 1))
      1)
(test (let ()
	(cond-expand
	 ((not guile)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      1)

(test (let ()
	(cond-expand 
	 ((and s7 dfls-exponents)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      (if (provided? 'dfls-exponents) 1 2))
(test (let ()
	(cond-expand 
	 ((or s7 guile)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      1)
(test (let ()
	(cond-expand 
	 ((and s7 dfls-exponents unlikely-feature)
	  (define (hi a) a))
	 (else 
	  (define (hi a) (+ a 1))))
	(hi 1))
      2)
(test (let ()
	(cond-expand
	 ((and s7 (not s7)) 'oops)
	 (else 1)))
      1)
(test (let ()
	(cond-expand
	 ("not a pair" 1)
	 (2 2)
	 (#t 3)
	 ((1 . 2) 4)
	 (() 6)
	 (list 7)
	 (else 5)))
      'error)




;;; --------------------------------------------------------------------------------
;;; case
;;; --------------------------------------------------------------------------------

(test (case (* 2 3) ((2 3 5 7) 'prime) ((1 4 6 8 9) 'composite))  'composite)
(test (case (car '(c d)) ((a e i o u) 'vowel) ((w y) 'semivowel) (else 'consonant)) 'consonant)
(test (case 3.1 ((1.3 2.4) 1) ((4.1 3.1 5.4) 2) (else 3)) 2)
(test (case 3/2 ((3/4 1/2) 1) ((3/2) 2) (else 3)) 2)
(test (case 3 ((1) 1 2 3) ((2) 2 3 4) ((3) 3 4 5)) 5)
(test (case 1+i ((1) 1) ((1/2) 1/2) ((1.0) 1.0) ((1+i) 1+i)) 1+i)
(test (case 'abs ((car cdr) 1) ((+ cond) 2) ((abs) 3) (else 4)) 3)
(test (case #\a ((#\b) 1) ((#\a) 2) ((#\c) 3)) 2)
(test (case (boolean? 1) ((#t) 2) ((#f) 1) (else 0)) 1)
(test (case 1 ((1 2 3) (case 2 ((1 2 3) 3)))) 3)
(test (case 1 ((1 2) 1) ((3.14 2/3) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2)) 1)
(test (case 1 ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 1)
(test (case #f ((1 2) 1) ((#\a) 2) ((car cdr) 3) ((#f #t) 4)) 4)
(test (case 1 ((#t) 2) ((#f) 1) (else 0)) 0)
(test (let ((x 1)) (case x ((x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case x ((1) "hi") (else "ho"))) "hi")
(test (let ((x 1)) (case x (('x) "hi") (else "ho"))) "ho")
(test (let ((x 1)) (case 'x ((x) "hi") (else "ho"))) "hi")
(test (case () ((()) 1)) 1)
(test (case #() ((#()) 1) (else 2)) 2)
(test (let ((x '(1))) (eval `(case ',x ((,x) 1) (else 0)))) 1)    ; but we can overcome that! (also via apply)
(test (let ((x #())) (eval `(case ',x ((,x) 1) (else 0)))) 1)
(test (case ''2 (('2) 1) (else 0)) 0)
(test (let ((otherwise else)) (case 1 ((0) 123) (otherwise 321))) 321)
(test (case 1 ((0) 123) (#t 321)) 'error)

(test (case else ((#f) 2) ((#t) 3) ((else) 4) (else 5)) 5)          ; (eqv? 'else else) is #f (Guile says "unbound variable: else")
(test (case #t ((#f) 2) ((else) 4) (else 5)) 5)                     ; else is a symbol here         
(test (equal? (case 0 ((0) else)) else) #t)
(test (cond ((case 0 ((0) else)) 1)) 1)
;(test (let () (case (define b 3) ((b) b))) 3) ; changed define 25-Jul-14

(test (let ((x 1)) (case x ((2) 3) (else (* x 2) (+ x 3)))) 4)
(test (let ((x 1)) (case x ((1) (* x 2) (+ x 3)) (else 32))) 4)
(test (let ((x 1)) (case x ((1) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 5)
(test (let ((x 1)) (case x ((2) (let () (set! x (* x 2))) (+ x 3)) (else 32))) 32)
(test (let ((x 1)) (case x ((2) 3) (else (let () (set! x (* x 2))) (+ x 3)))) 5)
(test (let((x 1))(case x((2)3)(else(let()(set! x(* x 2)))(+ x 3)))) 5)
(test (let ((x 1)) (case x ((2) 3) (else 4) (else 5))) 'error)

(test (case () ((()) 2) (else 1)) 2)    ; car: (), value: (), eqv: 1, null: 1 1
(test (case () (('()) 2) (else 1)) 1)   ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () (('()) 2) (else 1)) 1)    ; car: (quote ()), value: (), eqv: 0, null: 0 1
(test (case () ((()) 2) (else 1)) 2)     ; car: (), value: (), eqv: 1, null: 1 1

;;; this is a difference between () and () ?
;;; (eqv? () ()) -> #t and (eqv? () ()) is #t so it's the lack of evaluation in the search case whereas the index is evaluated
;;; equivalent to:
 
(test (case 2 (('2) 3) (else 1)) 1)      ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 (('2) 3) (else 1)) 1)     ; car: (quote 2), value: 2, eqv: 0, null: 0 0
(test (case '2 ((2) 3) (else 1)) 3)      ; car: 2, value: 2, eqv: 1, null: 0 0
(test (case 2 ((2) 3) (else 1)) 3)       ; car: 2, value: 2, eqv: 1, null: 0 0

(test (case '(()) ((()) 1) (((())) 2) (('()) 3) (('(())) 4) ((((()))) 5) (('((()))) 6) (else 7)) 7) ; (eqv? '(()) '(())) is #f

(test (let ((x 1)) (case (+ 1 x) ((0 "hi" #f) 3/4) ((#\a 1+3i '(1 . 2)) "3") ((-1 'hi 2 2.0) #\f))) #\f)
(test (case (case 1 ((0 2) 3) (else 2)) ((0 1) 2) ((4 2) 3) (else 45)) 3)
(test (case 3/4 ((0 1.0 5/6) 1) (("hi" 'hi 3/4) 2) (else 3)) 2)
(test (case (case (+ 1 2) (else 3)) ((3) (case (+ 2 2) ((2 3) 32) ((4) 33) ((5) 0)))) 33)
(test (let ((x 1)) (case x ((0) (set! x 12)) ((2) (set! x 32))) x) 1)

(test (case 1 (else #f)) #f)
(test (let () (case 0 ((0) (define (hi a) a)) (else (define (hi a) (+ a 1)))) (hi 1)) 1)
(test (let () (case 1 ((0) (define (hi a) a)) (else (define (hi a) (+ a 1)))) (hi 1)) 2)
;(test (let () (case (define (hi a) a) ((hi) (hi 1)))) 1) ; 25-Jul-14

(for-each
 (lambda (arg)
   (test (case 1 ((0) 'gad) ((1 2 3) arg) (else 'gad)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (case arg ((0) 'gad) ((1 2 3) arg) (else 'gad)) 'gad))
 (list "hi" -1 #\a 0 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call/cc (lambda (r) (case 1 ((1) (r 123) #t) (else #f)))) 123)
(test (call/cc (lambda (r) (case 1 ((0) 0) (else (r 123) #f)))) 123)

(test (case () ((1) 1) ('() 2)) 2)
(test (case (list) ((1) 1) ('() 2)) 2)
(test (case () ((1) 1) ((()) 2)) 2)
(test (case (list) ((1) 1) ((()) 2)) 2)
(test (case #<eof> ((#<eof>) 1)) 1)
(test (case #\newline ((#\newline) 1)) 1)
(test (case 'c (else => (lambda (x) (symbol? x)))) #t)
(test (case 1.0 ((1e0) 3) ((1.0) 4) (else 5)) 3)
(test (case 1.0 ((#i1) 2) ((1e0) 3) ((1.0) 4) (else 5)) 2)
(test (case 1 ((#i1) 2) ((1e0) 3) ((1.0) 4) (else 5)) 5)
(test (let ((x :a)) (case x ((:b) 1) ((:a) 0) (else 3))) 0)

(test (eval `(case ,+ ((,-) 0) ((,+) 1) (else 2))) 1)
(test (case + ((#_-) 0) ((#_+) 2) (else 3)) 2)

(let ()
  (define (c1 x)
    (case x
      ((3001) 1)
      ((12345) 2)
      ((8589934592) 3)
      (else 4)))
  (test (c1 3001) 1)
  (test (c1 12345) 2)
  (test (c1 8589934592) 3)
  (test (c1 -1) 4)

  (define (c2 x)
    (case x
      ((0 1 -1) 3)
      ((9223372036854775807 -9223372036854775808) 4)
      ((1.5) 5)
      ((2/3 1+i) 6)))

  (test (c2 -1) 3)
  (test (c2 most-positive-fixnum) 4)
  (test (c2 1.5) 5)
  (test (c2 2/3) 6)
  (test (c2 1+i) 6))

; case uses eqv? -- why not case-equal?
(test (case "" (("") 1)) #<unspecified>)
(test (case abs ((abs) 1)) #<unspecified>)
(test (case (if #f #f) ((1) 1) ((#<unspecified>) 2) (else 3)) 2)
;;; if case falls through, it should return #<unspecified> (not #f for example):
;;;   (case x ((a) 1)) should be equivalent to (if (eq? x 'a) 1)
(let ((x 'b)) (test (case x ((a) 1)) (if (eq? x 'a) 1)))

(test (case) 'error)
(test (case 1) 'error)
(test (case 1 . "hi") 'error)
(test (case 1 ("hi")) 'error)
(test (case 1 ("a" "b")) 'error)
(test (case 1 (else #f) ((1) #t)) 'error)
(test (case "hi" (("hi" "ho") 123) ("ha" 321)) 'error)
(test (case) 'error)
(test (case . 1) 'error)
(test (case 1 . 1) 'error)
(test (case 1 (#t #f) ((1) #t)) 'error)
(test (case 1 (#t #f)) 'error)
(test (case -1 ((-1) => abs)) 1)
(test (case 1 (else =>)) 'error)
(test (case 1 (else => + - *)) 'error)
(test (case #t ((1 2) (3 4)) -1) 'error)
(test (case 1 1) 'error)
(test (case 1 ((2) 1) . 1) 'error)
(test (case 1 (2 1) (1 1)) 'error)
(test (case 1 (else)) 1)   ; case null 4-Jan-17 
(test (case () ((1 . 2) . 1) . 1) 'error)
(test (case 1 ((1))) 1)                 ; case null
(test (case 1 ((else))) #<unspecified>) ; case null -- unexpected but ...
(test (case 1 ((2) 3) ((1))) 1)         ; case null
(test (case 1 ((1)) 1 . 2) 'error)
(test (case () ((()))) ())              ; case null
(test (case 1 (else 3) . 1) 'error)
(test (case 1 ((1 2)) (else 3)) 1)      ; case null
(test (case 1 ('(1 2) 3) (else 4)) 4)
(test (case 1 (('1 2) 3) (else 4)) 4)
(test (case 1 ((1 . 2) 3) (else 4)) 'error) ; ?? in guile it's an error
(test (case 1 ((1 2 . 3) 3) (else 4)) 'error)
(test (case 1 (('1 . 2) 3) (else 4)) 'error)
(test (case 1 ((1 . (2)) 3) (else 4)) 3)
(test (case 1 ((1 2) . (3)) (else 4)) 3)
(test (case 1 ((2) 3) (else)) 1)        ; case null
(test (case 1 ((2) 3) ()) 'error)
(test (case 1 ((2) 3) (() 2)) 'error) ; ?? in Guile this is #<unspecified>; our error is confusing: ;case clause key list () is not a list or 'else'
(test (case () ('() 2)) 2)            ; ?? error??
(test (case () ((()) 2)) 2) 
(test (case 1 else) 'error)
(test (case 1 (((1) 1) 2) (else 3)) 2) ; the (1) can't be matched -- should it be an error?
(test (case 1 ((1) . (else 3))) 3)     ; ?? guile says "unbound variable: else"
(test (case . (1 ((2) 3) ((1) 2))) 2)
(test (case 1 (#(1 2) 3)) 'error)
(test (case 1 #((1 2) 3)) 'error)
(test (case 1 ((2) 3) () ((1) 2)) 'error)
(test (case 1 ((2) 3) (1 2) ((1) 2)) 'error)
(test (case 1 ((2) 3) (1 . 2) ((1) 2)) 'error)
(test (case 1 ((2) 3) (1) ((1) 2)) 'error)
(test (case 1 ((2) 3) ((1)) ((1) 2)) 1)   ; case null 
(test (case 1 ((1) 2) ((1) 3)) 2) ; should this be an errror?
(test (let () (define-macro (mac x) `(+ ,x 1)) (case 1 ((1) => mac))) 2)

(test (let ((x 1)) (case x (else))) 1) 
(test (let ((x 1)) (case x ((0) 2) (else))) 1) 
(test (let ((x 1)) (case x ((1) 2) (else))) 2) 
(test (let ((x 1)) (case x ((0) 2))) #<unspecified>)
(test (let ((x 1)) (case x ((1)) ((2) 0))) 1)
(test (let ((x 1)) (case x ((lambda lambda*) (display "x") (+ 2 3)) ((case when) 3))) #<unspecified>)
(test (let ((x 'lambda)) (case x ((lambda lambda*)) ((case when) (display "x") 3))) 'lambda)
(test (let ((s "01234") (i 1)) (case (string-ref s i) ((#\3 #\4) 32) ((#\1 #\2)))) #\1)
(test (let ((s "01234") (i 3)) (case (string-ref s i) ((#\3 #\4) 32) ((#\1 #\2)))) 32)
(test (let ((head 'and)) (case head ((and if cond when)) ((or if2) (list 'not arg1)))) 'and)
(test (let ((s "01234") (i 1)) (case (string-ref s i) ((#\1)) ((#\2) 32))) #\1)
(test (let ((s "01234") (i 2)) (case (string-ref s i) ((#\1)) ((#\2) 32))) 32)
(test (let ((x 'a)) (case x ((a)))) 'a)
(test (case + ((-) 0) ((+)) (else 3)) 3)
(test (case + ((-) 0) ((#_+)) (else 3)) +)

;; check optimizer
(let () (define (c1 s i) (case (string-ref s i) ((#\a) 1) ((#\i)))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) #\i))
(let () (define (c1 s i) (case s (else))) (define (c3 s i) (c1 s i)) (c3 "hiho" 1) (test (c3 "hiho" 1) "hiho"))
(let () (define (c1 s i) (case s (else))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 2 1) 2))
(let () (define (c1 s i) (case s ((0) 2) (else))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 1 1) 1))
(let () (define (c1 s i) (case s ((0) 2) (else))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 0 1) 2))
(let () (define (c1 s i) (case s ((1) 2) (else))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 1 1) 2))
(let () (define (c1 s i) (case s ((1) 2) (else))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 3 1) 3))
(let () (define (c1 s i) (case s ((0) 2))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 1 1) #<unspecified>))
(let () (define (c1 s i) (case s ((1)) ((2) 0))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 1 1) 1))
(let () (define (c1 s i) (case s ((lambda lambda*)) ((case when) (display "x") 3))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 'lambda 1) 'lambda))
(let () (define (c1 s i) (case (string-ref s i) ((#\3 #\4) 32) ((#\1 #\2)))) (define (c3 s i) (c1 s i)) (c3 "0123" 1) (test (c3 "01234" 1) #\1))
(let () (define (c1 s i) (case s ((and if cond when)) ((or if2) (list 'not arg1)))) (define (c3 s i) (c1 s i)) (c3 2 1) (test (c3 'and 1) 'and))
(let () (define (c1 s i) (case (string-ref s i) ((#\1)) ((#\2) 32))) (define (c3 s i) (c1 s i)) (c3 "0123" 1) (test (c3 "01234" 1) #\1))
(let () (define (c1 s i) (case s ((a)))) (define (c3 s i) (c1 s i)) (c3 "0123" 1) (test (c3 'a 1) 'a))
(let () (define (c1 s i) (case + ((-) 0) ((+)) ((#_+)) (else 3))) (define (c3 s i) (c1 s i)) (c3 "0123" 1) (test (c3 'a 1) +))

;; newly optimized case
(let ((lt (inlet 'a 1 'b 2))) 
  (define (c1 s) (case (let-ref lt s) ((#<undefined>) 3) (else))) 
  (define (c3 s) (c1 s)) (c3 'a) 
  (test (c3 'b) 2)
  (test (c3 'a) 1)
  (test (c3 'c) 3))
(let ((lt (inlet 'a 1 'b 2))) 
  (define (c1 s) (case (let-ref lt s) ((#<eof> #<undefined>) 3) (else))) 
  (define (c3 s) (c1 s)) (c3 'a)
  (test (c3 'b) 2)
  (test (c3 'a) 1)
  (test (c3 'c) 3))
(let ((lt (inlet 'a 1 'b 2)))
  (define (c1 s) (let ((x 1)) (case (let-ref lt s) ((#<undefined>) (set! x 2) (+ x 1)) (else))))
  (define (c3 s) (c1 s))  (c3 'a)
  (test (c3 'a) 1)
  (test (c3 'c) 3))

(let ()
  (define (hi x) (case x ((a) 'a) ((b) 'b) (else 'c)))
  (test (hi 'a) 'a)
  (test (hi 'd) 'c))

(test (case 'case ((case) 1) ((cond) 3)) 1)
(test (case 101 ((0 1 2) 200) ((3 4 5 6) 600) ((7) 700) ((8) 800) ((9 10 11 12 13) 1300) ((14 15 16) 1600) ((17 18 19 20) 2000) ((21 22 23 24 25) 2500) ((26 27 28 29) 2900) ((30 31 32) 3200) ((33 34 35) 3500) ((36 37 38 39) 3900) ((40) 4000) ((41 42) 4200) ((43) 4300) ((44 45 46) 4600) ((47 48 49 50 51) 5100) ((52 53 54) 5400) ((55) 5500) ((56 57) 5700) ((58 59 60) 6000) ((61 62) 6200) ((63 64 65) 6500) ((66 67 68 69) 6900) ((70 71 72 73) 7300) ((74 75 76 77) 7700) ((78 79 80) 8000) ((81) 8100) ((82 83) 8300) ((84 85 86 87) 8700) ((88 89 90 91 92) 9200) ((93 94 95) 9500) ((96 97 98) 9800) ((99) 9900) ((100 101 102) 10200) ((103 104 105 106 107) 10700) ((108 109) 10900) ((110 111) 11100) ((112 113 114 115) 11500) ((116) 11600) ((117) 11700) ((118) 11800) ((119 120) 12000) ((121 122 123 124 125) 12500) ((126 127) 12700) ((128) 12800) ((129 130) 13000) ((131 132) 13200) ((133 134 135 136) 13600) ((137 138) 13800)) 10200)
(test (case most-positive-fixnum ((-1231234) 0) ((9223372036854775807) 1) (else 2)) 1)
(test (case most-negative-fixnum ((123123123) 0) ((-9223372036854775808) 1) (else 2)) 1)
(test (case 0 ((3/4 "hi" #t) 0) ((#f #() -1) 2) ((#\a 0 #t) 3) (else 4)) 3)
(test (case 3/4 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 0)
(test (case 'hi ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case #f ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 2)
(test (case 3 ((3/4 "hi" #t) 0) ((#f #() hi) 2) ((#\a 0 #t) 3) (else 4)) 4)
(test (case 0 ((values 0 1) 2) (else 3)) 2)
(test (+ (case 0 ((0) (values 1 2 3))) 4) 10)
(test (+ (case 1 ((0) (values 1 2 3)) (else (values 1 2))) 4) 7)

(test (let ((else 3)) (case 0 ((1) 2) (else 3))) 3) ; changed my mind about this -- else is not evaluated here unless it's some other symbol
(test (let ((otherwise else)) (case 0 ((1) 2) (otherwise 3))) 3) ; maybe this isn't a great idea...
(test (let ((else 3)) (case else ((3) else))) 3)
(test (case 0 ((1) 2) (else (let ((else 3)) else))) 3)
(test (case 0 ((1) #t) ((2 else 3) #f) ((0) 0)) 0) ; should this be an error? (it isn't in Guile)
(test (case 0 ((1) #t) ((else) #f) ((0) 0)) 0)
(test (apply case 1 '(((0) -1) ((1) 2))) 2)
(test (let ((x #(1))) (apply case x (list (list (list #()) 1) (list (list #(1)) 2) (list (list x) 3) (list 'else 4)))) 3)


(test (let ((x 0)) (let ((y (case 1 ((2) (set! x (+ x 3))) ((1) (set! x (+ x 4)) (+ x 2))))) (list x y))) '(4 6))
(let ()
  (define (hi a)
    (case a
      ((0) 1)
      ((1) 2)
      (else 3)))
  (test (hi 1) 2)
  (test (hi 2) 3)
  (test (hi "hi") 3))

(when with-bignums
  (test (case 8819522415901031498123 ((1) 2) ((8819522415901031498123) 3) (else 4)) 3) 
  (test (case -9223372036854775809 ((1 9223372036854775807) 2) (else 3)) 3))

;;; one thing that will hang case I think: circular key list

;;; C-style case
(define-macro (switch selector . clauses)
  `(call-with-exit
    (lambda (break)
      (case ,selector
	,@(do ((clause clauses (cdr clause))
	       (new-clauses ()))
	      ((null? clause) (reverse new-clauses))
	    (set! new-clauses (cons `(,(caar clause) 
				      ,@(cdar clause)
				      ,@(map (lambda (nc)
					       (apply values (cdr nc)))
					     (if (pair? clause) (cdr clause) ())))
				    new-clauses)))))))

(test (switch 1 ((1) (break 1)) ((2) 3) (else 4)) 1)
(test (switch 2 ((1) (break 1)) ((2) 3) (else 4)) 4)
(test (switch 4 ((1) (break 1)) ((2) 3) (else 4)) 4)

(let ()
  (call-with-output-file "test.scm"
    (lambda (p)
      (format p "(define (big-cond x)~%")
      (format p "  (cond~%")
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "    ((= x ~D) x)~%" i))
      (format p "  ))~%~%")
      (format p "(define (big-case x)~%")
      (format p "  (case x~%")
      (do ((i 0 (+ i 1)))
	  ((= i 1000))
	(format p "    ((~D) x)~%" i))
      (format p "  ))~%~%")))
  (load "test.scm" (curlet))
  (test (big-cond 541) 541)
  (test (big-case 541) 541))



;;; --------------------------------------------------------------------------------
;;; lambda
;;; --------------------------------------------------------------------------------

(test (procedure? (lambda (x) x)) #t)
(test ((lambda (x) (+ x x)) 4) 8)
(test (let ((reverse-subtract (lambda (x y) (- y x)))) (reverse-subtract 7 10)) 3)
(test (let ((add4 (let ((x 4)) (lambda (y) (+ x y))))) (add4 6)) 10)
(test ((lambda x x) 3 4 5 6) (list 3 4 5 6))
(test ((lambda (x y . z) z) 3 4 5 6) (list 5 6))
(test ((lambda (a b c d e f) (+ a b c d e f)) 1 2 3 4 5 6) 21)
(test (let ((foo (lambda () 9))) (+ (foo) 1)) 10)
(test (let ((a 1)) (let ((f (lambda (x) (set! a x) a))) (let ((c (f 123))) (list c a)))) (list 123 123))
(test (let ((a 1) (b (lambda (a) a))) (b 3)) 3)
(test (let ((ctr 0)) (letrec ((f (lambda (x) (if (> x 0) (begin (set! ctr (+ ctr 1)) (f (- x 1))) 0)))) (f 10) ctr)) 10)
(test (let ((f (lambda (x) (car x)))) (f '(4 5 6))) 4)
(test ((lambda () ((lambda (x y) ((lambda (z) (* (car z) (cdr z))) (cons x y))) 3 4))) 12)
(test (let ((ctr 0)) (define (f) (set! ctr (+ ctr 1)) ctr) (let ((x (f))) (let ((y (f))) (list x y ctr)))) (list 1 2 2))

(test (let ((x 5)) (define foo (lambda (y) (bar x y))) (define bar (lambda (a b) (+ (* a b) a))) (foo (+ x 3))) 45)
(test (let ((x 5)) (letrec ((foo (lambda (y) (bar x y))) (bar (lambda (a b) (+ (* a b) a)))) (foo (+ x 3)))) 45)

(num-test (let () (define compose (lambda (f g) (lambda args (f (apply g args))))) ((compose sqrt *) 12 75))  30.0)
(let ()
  (define (compose . args) ; this just removes parens
    (if (procedure? (car args))
	(if (null? (cdr args))
	    ((car args))
	    ((car args) (apply compose (cdr args))))
	(apply values args)))
  (test (compose - + (lambda (a b c) (values a (* b c))) 2 3 4) -14)
  (test (- (+ ((lambda (a b c) (values a (* b c))) 2 3 4))) -14)) ; I prefer this

(test (let ((f (lambda () (lambda (x y) (+ x y))))) ((f) 1 2)) 3)
(test ((lambda (x) (define y 4) (+ x y)) 1) 5)
(test ((lambda(x)(define y 4)(+ x y))1) 5)
(test ((lambda () (define (y x) (+ x 1)) (y 1))) 2)
(test ((lambda (x) 123 (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) "documentation" (let ((a (+ x 1))) a)) 2) 3)
(test ((lambda (x) (x 1)) (lambda (y) (+ y 1))) 2)
(test (let ((a 1)) (let ((b (lambda (x) (define y 1) (define z 2) (define a 3) (+ x y z a)))) (b a))) 7)
(test ((lambda (f x) (f x x)) + 11) 22)
(test ((lambda () (+ 2 3))) 5)
(test (let ((x (let () (lambda () (+ 1 2))))) (x)) 3)
(test (cond (0 => (lambda (x) x))) 0)
(test ((lambda () "hiho")) "hiho")
(test ((lambda()()))())
(test (procedure-source (apply lambda (list) (list (list)))) '(lambda () ()))

(test (letrec ((f (lambda (x) (g x)))
	       (g (lambda (x) x)))
	(let ((top (f 1)))
	  (set! g (lambda (x) (- x)))
	  (+ top (f 1))))
      0)

(for-each
 (lambda (arg)
   (test ((lambda (x) x) arg) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(let ((list-length
       (lambda (obj)
	 (call-with-current-continuation
	  (lambda (return)
	    (letrec ((r (lambda (obj) (cond ((null? obj) 0)
					    ((pair? obj) (+ (r (cdr obj)) 1))
					    (else (return #f))))))
	      (r obj)))))))
  (test (list-length '(1 2 3 4)) 4)
  (test (list-length '(a b . c)) #f))

(test (let ((samples (vector 0 1 2 3 4 5 6 7 8 9 10)))
	(let ((make-scaler 
	       (lambda (start end)
		 (letrec ((ctr start)
			  (us (lambda (them)
				(vector-set! samples ctr (* 2 (vector-ref samples ctr)))
				(set! ctr (+ ctr 2))
				(if (<= ctr end)
				    (them us)))))
		   us))))
	  ((make-scaler 0 11)
	   (make-scaler 1 11))) 
	samples)
      (vector 0 2 4 6 8 10 12 14 16 18 20))

(test ((lambda (x . y) y) 1 2 '(3 . 4)) '(2 (3 . 4)))
(test ((lambda (x . y) y) 1) ())
(test ((lambda x x) ()) '(()))
(test ((lambda x x)) ())
(test ((lambda (x) x) ()) ())
(test ((lambda (x) (+ x ((lambda (x) (+ x 1)) 2))) 3) 6)
(test ((lambda (x) (define y 1) (+ x y)) 2) 3)
(test ((lambda (a) "this is a doc string" a) 1) 1)
;;; ideally ((lambda (a) "hiho" (define x 1) x) 1) -> 1 but I'm not sure it's r5rs-ish
(test (let ((g (lambda () '3))) (= (g) 3)) #t)
(test ((((lambda () lambda)) () 1)) 1)

(test (let () ; PLP Scott p168
	(define A
	  (lambda ()
	    (let* ((x 2)
		   (C (lambda (P)
			(let ((x 4))
			  (P))))
		   (D (lambda ()
			x))
		   (B (lambda ()
			(let ((x 3))
			  (C D)))))
	      (B))))
	(A))
      2)

#|
;;; here s7 "do" uses set!
(test (let ((funcs (make-vector 3 #f)))
	(do ((i 0 (+ i 1)))
	    ((= i 3))
	  (vector-set! funcs i (lambda () (+ i 1))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)
|#

;;; the equivalent named let version:
(test (let ((funcs (make-vector 3 #f)))
	(let loop ((i 0))
	  (if (< i 3)
	      (begin
		(vector-set! funcs i (lambda () (+ i 1)))
		(loop (+ i 1)))))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((i 1))
	(let ((func1 (lambda () i)))
	  (let ((i 2))
	    (let ((func2 (lambda () i)))
	      (+ (func1) (func2))))))
      3)

(test (let ((funcs (make-vector 3 #f)))
	(map
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(for-each
	 (lambda (i)
	   (vector-set! funcs i (lambda () (+ i 1))))
	 (list 0 1 2))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(sort! (list 0 1 2)
	 (lambda (i j)
	   (vector-set! funcs i (lambda () (+ i 1))) 
	   (> i j)))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(member 4 (list 0 1 2)
	 (lambda (j i)
	   (vector-set! funcs i (lambda () (+ i 1)))
	   #f))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((funcs (make-vector 3 #f)))
	(assoc 4 (list (cons 0 0) (cons 1 0) (cons 2 0))
	 (lambda (j i) 
	   (vector-set! funcs i (lambda () (+ i 1)))
	   #f))
	(+ ((vector-ref funcs 0))
	   ((vector-ref funcs 1))
	   ((vector-ref funcs 2))))
      6)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (+ x 1))))
	(func1 1)
	(+ (func)
	   (let ()
	     (func1 2)
	     (func))))
      5)

(test (((lambda (x) (lambda () (+ x 1))) 32)) 33)

(test (let ((func #f))
	(define (func1 x)
	  (set! func (lambda () (string-append x "-"))))
	(func1 "hi")
	(string-append (func)
		       (let ()
			 (func1 "ho")
			 (func))))
      "hi-ho-")

(test (let ((func1 #f)
	    (func2 #f))
	(let ((x 1))
	  (set! func1 (lambda () x))
	  (set! func2 (lambda (y) (set! x y) y)))
	(+ (func1)
	   (let ()
	     (func2 32)
	     (func1))))
      33)

(test (let ((funcs (make-vector 3)))
	(let ((hi (lambda (a) (vector-set! funcs (- a 1) (lambda () a)))))
	  (hi 1) (hi 2) (hi 3)
	  (+ ((vector-ref funcs 0))
	     ((vector-ref funcs 1))
	     ((vector-ref funcs 2)))))
      6)

(test (let ((hi (lambda (a) (+ a 1)))
	    (ho (lambda (a) (a 32))))
	(+ (hi (hi (hi 1)))
	   (ho hi)))
      37)

(test (let ((x 0)
	    (b 4)
	    (f1 #f)
	    (f2 #f))
	(let ((x 1))
	  (let ((x 2))
	    (set! f1 (lambda (a) (+ a b x)))))
	(let ((x 3))
	  (let ((b 5))
	    (set! f2 (lambda (a) (+ a b x)))))
	(+ (f1 10) (f2 100)))  ; (+ 10 4 2) (+ 100 5 3)
      124)

(test ((if (> 3 2) + -) 3 2) 5)
(test (let ((op +)) (op 3 2)) 5)
(test (((lambda () +)) 3 2) 5)
(test ((car (cons + -)) 3 2) 5)
(test ((do ((i 0 (+ i 1))) ((= i 3) +) ) 3 2) 5)
(test (((lambda (x) x) (lambda (x) x)) 3) 3)
(test ((((lambda (x) x) (lambda (x) x)) (lambda (x) x)) 3) 3)
(test (((lambda (x) (lambda (y) x)) 3) 4) 3)
(test (((lambda (x) (lambda (x) x)) 3) 4) 4)
(test (let ((x 32)) (((lambda (x) (lambda (y) x)) 3) x)) 3)
(test ((call/cc (lambda (return) (return +))) 3 2) 5)
(test ((call-with-values (lambda () (values +)) (lambda (x) x)) 3 2) 5)
(test ((case '+ ((+) +)) 3 2) 5)
(test ((case '+ ((-) -) (else +)) 3 2) 5)
(test ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 5)
(test (+ 1 ((call/cc (lambda (return) (dynamic-wind (lambda () #f) (lambda () (return +)) (lambda () #f)))) 3 2) 2) 8)
(test (let ((lst (list + -))) ((car lst) 1 2 3)) 6)
(test (let ((a +)) ((let ((b -)) (if (eq? a b) a *)) 2 3)) 6)
(test ((list-ref (list + - * /) 0) 2 3) 5)
(test (((if #t list-ref oops) (list + - * /) 0) 2 3) 5)
(test ((((car (list car cdr)) (list car cdr)) (list + -)) 2 3) 5)
(test (let ()
	(define function lambda)
	(define hiho (function (a) (+ a 1)))
	(hiho 2))
      3)
(test ((lambda (a b c d e f g h i j k l m n o p q r s t u v x y z)
	 (+ a b c d e f g h i j k l m n o p q r s t u v x y z))
       1 2 3 4 5 6 7 8 9 11 12 13 14 15 16 17 18 19 21 22 23 24 25 26 27)
      348)
(test ((lambda (x) "a useless string" x) 32) 32)
(test ((lambda (>< =0=? .arg.) (+ >< =0=? .arg.)) 1 2 3) 6)
(test ((apply ((lambda () lambda)) ((lambda () (list 'a))) ((lambda () '((+ a 1))))) 3) 4)

(define-constant (_?_3 a) #f)
(let () (define (hi x) (_?_3 x)) (hi 1) (test (let ((x 1)) (hi x)) #f))
(let () (define (_?_4 x y) x) (define (hi x) (_?_4 x (* x x (+ 1 x)))) (hi 1) (test (let ((x 1)) (hi x)) 1))
(define-constant (_?_5 x) (if (zero? x) x (+ x (_?_5 (- x 1)))))
(let () (define (hi x) (_?_5 x)) (hi 1) (test (let ((x 1)) (hi x)) 1))
(let () (define (hi x) (_?_5 (_?_5 1))) (hi 1) (test (hi 1) 1))
(let ((x 1)) (define (hi y) (set! x (* y y y))) (hi 1) (test (hi 1) 1))

(test
 (let ()
   (begin
     (define f1 #f)
     (define f2 #f)
     (let ((lv 32))
       (set! f1 (lambda (a) (+ a lv)))
       (set! f2 (lambda (a) (- a lv)))))
   (+ (f1 1) (f2 1)))
 2)

(test ((lambda () => abs)) 'error)
(test ((lambda () => => 3)) 'error)
;; actually, both Guile and Gauche accept
;; ((lambda () + 3)) and (begin + 3)
;; but surely => is an undefined variable in this context?

(test (lambda) 'error)
(test (lambda (a) ) 'error)
;; should this be an error: (lambda (a) (define x 1)) ?
(test (lambda . 1) 'error)
(test ((lambda . (x 1))) 1)
(test ((lambda . ((x . y) 2)) 1) 2)
(test ((lambda (x) . (x)) 1) 1)
(test ((lambda . ((x) . (x))) 1) 1)
(test ((lambda . (x . (x))) 1) '(1))
(test ((lambda . ((x . ()) x)) 1) 1)
(test (eval-string "((lambda . (x 1 . 3)) 1)") 'error)

(test (lambda 1) 'error)
(test (lambda (x 1) x) 'error)
(test (lambda "hi" 1) 'error)
(test (lambda (x x) x) 'error)
(test ((lambda (x x) x) 1 2) 'error) 
(test (lambda (x "a")) 'error)
(test ((lambda (x y) (+ x y a)) 1 2) 'error)
(test ((lambda ())) 'error)
(test (lambda (x (y)) x) 'error)
(test ((lambda (x) x . 5) 2) 'error)
(test (lambda (1) #f) 'error)
(test (eval-string "(lambda (x . y z) x)") 'error) 
(test ((lambda () 1) 1) 'error)
(test ((lambda (()) 1) 1) 'error)
(test ((lambda (x) x) 1 2) 'error)
(test ((lambda (x) x)) 'error)
(test ((lambda ("x") x)) 'error)
(test ((lambda "x" x)) 'error)
(test ((lambda (x . "hi") x)) 'error)
(test (lambda ((:hi . "hi") . "hi") 1) 'error)
(test ((lambda (x) (* quote ((x . 1) . 2))) 1) 'error)
(test ((lambda* (a (quote . -1)) a)) 'error)

(test (let ((hi (lambda (a 0.0) (b 0.0) (+ a b)))) (hi)) 'error)
(test (object->string
       ((lambda (arg)
	  (list arg
		(list (quote quote)
		      arg)))
	(quote (lambda (arg)
		 (list arg
		       (list (quote quote)
			     arg))))))
      "((lambda (arg) (list arg (list 'quote arg))) '(lambda (arg) (list arg (list 'quote arg))))")
      ;; was "(#1=(lambda (arg) (list arg (list 'quote arg))) '#1#)"
      
(test ((apply lambda '((a) (+ a 1))) 2) 3)
(test ((apply lambda '(() #f))) #f)
(test ((apply lambda '(arg arg)) 3) '(3))
(test ((apply lambda* '((a (b 1)) (+ a b))) 3 4) 7)
(test ((apply lambda* '((a (b 1)) (+ a b))) 3) 4)

(let ()
  (define-macro (progv vars vals . body)
    `(apply (apply lambda ,vars ',body) ,vals))
  (test (let ((s '(one two)) (v '(1 2))) (progv s v (+ one two))) 3)
  (test (progv '(one two) '(1 2) (+ one two)) 3))

(test (lambda #(a b) a) 'error)
(test (lambda* (#(a 1)) a) 'error)

(test ((lambda (a) a) #<eof>) #<eof>)
(test ((lambda () (let ((a #<undefined>)) a))) #<undefined>)

(test (let () (define* (foo (a 0) (b (+ a 4)) (c (+ a 7))) (list a b c)) (foo :b 2 :a 60)) '(60 2 67))
(test (let () (define* (f1 (a 0) (b (* 2 a))) (+ a b)) (f1 2)) 6) ; this used to be 2
;; one oddness:
(test (let () (define* (f1 (a (* b 2)) (b 3)) (list a b)) (f1 :b 1)) '(2 1))
;; (f1) however would be an error?  or should we preset args if we can?

(let ()
  (define* (f1 (a (+ b 1)) (b (+ a 1))) (list a b))
  (test (f1 1) '(1 2))
  (test (f1 :b 1) '(2 1))
  (test (f1 :b 0 :a 1) '(1 0))
  (test (f1 :a 1) '(1 2))
  (test (f1 2 3) '(2 3))
  (test (f1) 'error))

(let ()
  (define* (f1 (a (if (number? b) (+ b 1) 3)) (b (+ a 1))) (list a b))
  (test (f1) '(3 4)))

(let ()
  (define* (f1 (a 1) (b (+ a 1))) (+ a b))
  (define* (f2 (a (f1))) a)
  (test (f2) 3))

(let ()
  (define* (f1 (a 1) (b (+ a 1))) (+ a b))
  (define* (f2 (a (f1)) (b (f1 2))) (list a b))
  (test (f2) '(3 5)))



;;; --------------------------------------------------------------------------------
;;; begin
;;; --------------------------------------------------------------------------------

(test (begin) ()) ; I think Guile returns #<unspecified> here
(test (begin (begin)) ())
(test ((lambda () (begin))) ())
(test (let () (begin) #f) #f)
(test (let () (begin (begin (begin (begin)))) #f) #f)
(test (let () (begin (define x 2) (define y 1)) (+ x y)) 3)
(test (let () (begin (define x 0)) (begin (set! x 5) (+ x 1)))  6)
(test (let () (begin (define first car)) (first '(1 2))) 1)
(test (let () (begin (define x 3)) (begin (set! x 4) (+ x x))) 8)
(test (let () (begin (define x 0) (define y x) (set! x 3) y)) 0)         ; the let's block confusing global defines
(test (let () (begin (define x 0) (define y x) (begin (define x 3) y))) 0)
(test (let () (begin (define y x) (define x 3) y)) 'error)               ; guile says 3
(test (let ((x 12)) (begin (define y x) (define x 3) y)) 12)             ; guile says 3 which is letrec-style?
(test (begin (define (x) y) (define y 4) (x)) 4)
;; (let ((x 12)) (begin (define y x) y)) is 12
(test (let ((x 3)) (begin x)) 3)
(test (begin 3) 3)
(test (begin . (1 2)) 2)
(test (begin . ()) (begin))
(test (begin . 1) 'error)
(test (begin 1 . 2) 'error)
(test (begin ("hi" 1)) #\i)

(when (equal? (begin 1) 1)
  (test (let () (begin (define x 0)) (set! x (begin (begin 5))) (begin ((begin +) (begin x) (begin (begin 1))))) 6)      
  
  (test (let ((x 5))
	  (begin (begin (begin)
			(begin (begin (begin) (define foo (lambda (y) (bar x y)))
				      (begin)))
			(begin))
		 (begin)
		 (begin)
		 (begin (define bar (lambda (a b) (+ (* a b) a))))
		 (begin))
	  (begin)
	  (begin (foo (+ x 3))))
	45)
  
  (for-each
   (lambda (arg)
     (test (begin arg) arg))
   (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
  
  (test (if (= 1 1) (begin 2) (begin 3)) 2))

(test ((lambda (x) (begin (set! x 1) (let ((a x)) (+ a 1)))) 2) 2)
;;; apparently these can be considered errors or not (guile says error, stklos and gauche do not)
(test (begin (define x 0) (+ x 1)) 1)
(test ((lambda () (begin (define x 0) (+ x 1)))) 1)
(test (let ((f (lambda () (begin (define x 0) (+ x 1))))) (f)) 1)

(test ((lambda () (begin (define x 0)) (+ x 1))) 1)
(test (let ((f (lambda () (begin (define x 0)) (+ x 1)))) (f)) 1)
(test (let ((x 32)) (begin (define x 3)) x) 3)
(test ((lambda (x) (begin (define x 3)) x) 32) 3)
(test (let* ((x 32) (y x)) (define x 3) y) 32)

(test (let ((z 0)) (begin (define x 32)) (begin (define y x)) (set! z y) z) 32)
(test (let((z 0))(begin(define x 32))(begin(define y x))(set! z y)z) 32)
(test (let ((z 0)) (begin (define x 32) (define y x)) (set! z y) z) 32)        
(test (let () (begin (define b 1) (begin (define a b) (define b 3)) a)) 1)
(test (let () (begin (begin (define a1 1) (begin (define a1 b1) (define b1 3))) a1)) 'error)
(test (let () (begin (begin (define (a3) 1)) (begin (define (a3) b3) (define b3 3)) (a3))) 3) ; yow
(test (let () (begin (begin (define (a) 1)) (a))) 1)
(test (let ((a 1)) (begin (define a 2)) a) 2)
(test (+ 1 (begin (values 2 3)) 4) 10)
(test (+ 1 (begin (values 5 6) (values 2 3)) 4) 10)
(test (let ((hi 0)) (begin (values (define (hi b) (+ b 1))) (hi 2))) 3)




;;; --------------------------------------------------------------------------------
;;; apply
;;; --------------------------------------------------------------------------------

(test (apply (lambda (a b) (+ a b)) (list 3 4)) 7)
(test (apply + 10 (list 3 4)) 17)
(test (apply list ()) ())
(test (apply + '(1 2)) 3)
(test (apply - '(1 2)) -1)
(test (apply max 3 5 '(2 7 3)) 7)
(test (apply cons '((+ 2 3) 4)) '((+ 2 3) . 4))
(test (apply + ()) 0)
(test (apply + (list 3 4)) 7)
(test (apply + ()) 0)
(test (apply + 2 '(3)) 5)
(test (apply + 2 3 ()) 5)
(test (apply + '(2 3)) 5)
(test (apply list 1 '(2 3)) (list 1 2 3))
(test (apply apply (list list 1 2 '(3))) (list 1 2 3))
(test (vector? (apply make-vector '(1))) #t)
(test (apply make-vector '(1 1)) #(1))
(test (apply make-vector '((1) 1)) #(1))
(test (let ((f +)) (apply f '(1 2))) 3)
(test (apply min '(1 2 3 5 4 0 9)) 0)
(test (apply min 1 2 4 3 '(4 0 9)) 0)
(test (apply vector 1 2 '(3)) #(1 2 3))
(test (apply vector ()) #())
(test (apply (lambda (x . y) x) (list 1 2 3)) 1)
(test (apply * (list 2 (apply + 1 2 '(3)))) 12)
(test (apply (if (> 3 2) + -) '(3 2)) 5)
(test (let ((x (list 1 2))) (eq? x (append () x))) #t) ;; ?? guile says #t also
(test (apply (lambda* args args) 1 2 3 '(4 5 6 (7))) '(1 2 3 4 5 6 (7))) ; from lisp bboard
(test (apply (list 1 2) '(0)) 1)

(test (apply (cons 1 2) '(0)) 1) ; ! (apply (cons 1 2) '(1)) is an error
(test (procedure? apply) #t)
(test (help apply) "(apply func ...) applies func to the rest of the arguments")
(let ((lst (list 'values '(procedure? sequence?) #t))) ; values rather than #t since (+ (apply values '(1 2))) -> 3
  (set-cdr! (cddr lst) (cddr lst))
  (test (equal? lst (procedure-signature apply)) #t))

(for-each
 (lambda (arg)
   (test (apply (lambda (x) x) (list arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (apply cadadr (list '''4)) 4)
(test (apply string-ref "hi" '(0)) #\h)
(test (let ((x (string-copy "hi"))) (apply string-set! x 0 '(#\c)) x) "ci")
(test (apply apply (list + '(3  2))) 5)
(test (apply apply apply apply (list (list (list + '(3  2))))) 5)
(test (apply + 1 2 (list 3 4)) 10)
(test ((apply cdr '((1 2) (3 4)) ()) 0) '(3 4))
(test ((apply car '((1 2) (3 4)) ()) 1) 2)
(test ((apply cadr '((1 2) (3 4)) ()) 1) 4)
(test (apply append ()) ())
(test (apply apply append ()) ())
(test (apply apply apply append '(())) ())
(test (apply apply + ()) 0)
(test (apply apply * ()) 1)
(test (apply apply not not () ()) #f)
(test (apply apply apply eq? eq? eq? () () ()) #t)
(test (apply apply apply list list list () () ()) (list list list))
(test (apply apply vector cons (list '1 '2) ()) (vector cons 1 2))

(test (let ((x '(((1 2)) ((3 4))))) (catch #t (lambda () (apply apply apply apply x)) (lambda args 'error)) x) '(((1 2)) ((3 4))))
(test (let ((x '((1 2) (3 4)))) (catch #t (lambda () (apply apply apply apply x)) (lambda args 'error)) x) '((1 2) (3 4)))
(test (let ((x '((1 2) 3 4))) (catch #t (lambda () (apply apply apply x)) (lambda args 'error)) x) '((1 2) 3 4))
(test (let ((x '((1 2) (3 4)))) (catch #t (lambda () (apply apply apply not x)) (lambda args 'error)) x) '((1 2) (3 4)))

(test (eq? (apply apply apply values '(())) #<unspecified>) #t)
(test (eqv? (apply apply apply values '(())) #<unspecified>) #t)
(test (equal? (apply apply apply values '(())) #<unspecified>) #t)

(test (apply apply apply + '(((1)))) 1)
(test (apply apply map + '(((1)))) '(1))
(test (apply apply map quote '(((1)))) '(1))
(test (apply apply map values '(((1)) ((2)))) '((1) 2))
(test (apply apply map append '(((1)) ((2)))) '((1 . 2)))
(test (apply apply apply quote '(((1)))) 1)
(test (apply map cdr '(((1 2) (3 4)))) '((2) (4)))
(test (apply apply + '((1 2))) 3)
(test (apply apply cons '(((1 2) (3 4)))) '((1 2) 3 4))
(test (apply apply append '(((1 2) (3 4)))) '(1 2 3 4))
(test (apply map + '((1 2) (3 4))) '(4 6))
(test (apply map reverse '(((1 2) (3 4)))) '((2 1) (4 3)))
(test (apply apply map cons '(((1 2) (3 4)))) '((1 . 3) (2 . 4)))
(test (apply apply map list-tail '(((1 2) (3 4))) '(((1)))) '(((3 4))))
(test (apply apply map reverse '((1 2) (3 4)) '(())) '((2 1) (4 3)))
(test (apply apply map values '(((1)) ((2))) '(((1 2) (3 4)))) '(((1)) 1 3 ((2)) 2 4))
(test (apply apply map append '(((1 2) (3 4))) '(((1)) ((2)))) '(((1 2) (3 4) 1 . 2)))
(test (apply apply map append '(()) '(((1)) ((2)))) '((1 . 2)))
(test (apply apply map cdr '(((1 2) (3 4))) ()) '((2) (4)))
(test (apply apply apply list-tail '((1 2) (3 4)) '(((1)))) '((3 4)))
(test (apply apply apply reverse '(((1 2) (3 4))) '(())) '((3 4) (1 2)))
(test (apply apply apply values '(1) '(())) 1)
(test (apply apply apply values '(1) '((()))) '(1))
(test (apply apply apply values '((1)) ()) 1)
(test (apply apply apply values '((1)) '(())) '(1))
(test (apply apply reverse '(((1 2) (3 4))) ()) '((3 4) (1 2)))
(test (apply apply append () '(((1 2) (3 4)))) '(1 2 3 4))
(test (apply apply length '(()) ()) 0)
(test (apply apply let () '((1))) 1)
(test (apply apply apply apply + '((()))) 0)
(test (apply apply apply map reverse '((1 2) (3 4)) '((()))) '((2 1) (4 3)))
(test (apply apply apply map values '(((1 2) (3 4))) ()) '(1 3 2 4))
(test (apply apply apply apply + '(1) '((()))) 1)

(test (apply apply apply append (reverse '(((1)) ((2))))) '((2) . 1))
(test (apply apply map append (reverse '(((1)) ((2))))) '((2 . 1)))
(test (apply (apply apply lambda (quote '(1)))) 1)
(test (apply quote (map reverse (reverse '((1 2))))) '(2 1))
(test (map quote (apply map + '((1 2) (3 4)))) '(4 6))
(test (map car (apply map quote '(((1 2) (3 4))))) '(1 3))
(test (apply length (apply map append '(((1)) ((2))) '((1)))) -1)
(test (apply append (apply map list-tail '(((1 2) (3 4))) '((1)))) '((3 4)))
(test (apply append (apply map values '(((1)) ((2))) '(((1 2) (3 4))))) '((1) 1 2 (2) 3 4))
(test (apply append (apply map values '((1 2) (3 4)) '(((1 2) (3 4))))) '(1 2 1 2 3 4 3 4))
(test (apply append '((1) () (2 3 4) (5 6) ())) '(1 2 3 4 5 6))
(test (apply append '((1) () (2 3 4) (5 6) 7)) '(1 2 3 4 5 6 . 7))

(test (apply +) 0)
(test (apply + #f) 'error)
(test (apply #f '(2 3)) 'error)
(test (apply make-vector '(1 2 3)) 'error)
(test (apply + 1) 'error)
(test (apply) 'error)
(test (apply 1) 'error)
(test (apply . 1) 'error)
(test (apply car ''foo) 'error)
(test (apply + '(1 . 2)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply () ()) 'error)
(test (apply list '(1 . 2) ()) '((1 . 2)))
(test (apply (lambda (x) x) _ht_ _null_ _c_obj_) 'error)
(test (apply + #(1 2 3)) 'error)
(test (apply (lambda (a b) (+ a b)) '(1 . 2)) 'error)
(test (apply (lambda args (apply + args)) 1 2 3) 'error)
(test (apply (lambda args (apply + args)) 1 2 #f) 'error)
(test (apply (lambda args (apply list args)) 1 2 #f) 'error)
(test (apply (lambda args (apply + args)) 1 2 ()) 3)
(test (apply (lambda args (apply list args)) 1 2 ()) '(1 2))
(test (apply (lambda args (apply list args)) 1 '(2)) '(1 2))
(test (apply (lambda args (apply list args)) 1 '2) 'error)
(test (apply (lambda args (apply list args)) 1 'abs) 'error)
(test (apply (lambda args (apply list args)) 1 ''2) '(1 quote 2))
(test (apply (lambda args (apply list args)) () ()) '(()))
(test (apply (lambda args (apply list args)) () (cons 1 2)) 'error)
(test (apply (lambda args (apply list args)) (cons 1 2)) 'error)

(test (apply "hi" '(1 2)) 'error)
(test ("hi" 1 2) 'error)
(test (apply '(1 2) '(1 2)) 'error)
(test ((list 1 2 3) 1 2) 'error)

(test (apply "hi" '(1)) #\i)
(test ("hi" 1) #\i)
(test (apply '(1 2) '(1)) 2)
(test ((list 1 2 3) 1) 2)

(for-each
 (lambda (arg)
   (test (apply arg '(1)) 'error)
   (test (apply abs arg) 'error))
 (list -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t)) 

(test (apply "hi" '(1)) #\i)
(test (apply '(1 2 3) '(1)) 2)
(test (apply #(1 2 3) '(2)) 3)
(test (apply #2D((1 2) (3 4)) 0 0 ()) 1)
(test (apply '((1 2) (3 4)) 1 0 ()) 3)
(test (let ((ht (make-hash-table))) (set! (ht "hi") 32) (apply ht '("hi"))) 32)

(test (let ((x (list 1 2))) (set-cdr! x x) (apply + x)) 'error)
(test (apply + '(1 2 . 3)) 'error)
(test (apply + '(1 2) (list 3 4)) 'error)
(test (let () (define (mrec a b) (if (<= b 0) (list a) (apply mrec (list a) (list (- b 1))))) (mrec (list 1 2) 5)) '(((((((1 2))))))))

(let ((lst (list 1 2 3)))
   (set! (cdr (cddr lst)) lst)
   (test (apply + lst) 'error))

(test (let ((lst '(1 2 3))) (let ((lst1 (apply list lst))) (set! (car lst1) 21) lst)) '(1 2 3))
(test (let ((lst '(1 2))) (let ((lst1 (apply cons lst))) (set! (car lst1) 21) lst)) '(1 2))
(test (let* ((x '(1 2 3)) (y (apply list x))) (eq? x y)) #f) ; this was #t until 26-Sep-11

(test (apply values (values (cons 1 ()))) 1)
(test (+ (apply values (values (list 1 2)))) 3)
(test (port-filename) (apply port-filename (list)))
(num-test (apply atan (#(1 #\a (3)) (max (values 1 2)))) 1.2490457723983)
(test (apply #2D((1 2) (3 4)) (list (floor (acosh 1)))) #(1 2)) 
(test ((apply values (list + 1 2)) 3) 6)
(if with-complex (num-test (* 0-2i (acosh (asin 0.0))) pi))
(test (apply truncate (lognot (min 1)) (list)) -2)
(num-test (apply /(list 11 11)) 1)

(test (apply dynamic-wind (list (lambda () #f) (lambda () 1) (lambda () #f))) 1)
(test (apply call-with-exit (list (lambda (exit) 1))) 1)
(test (apply call-with-exit (list (lambda (exit) (exit 1) 32))) 1)
(test (apply catch (list #t (lambda () 1) (lambda args 'error))) 1)
(test (apply eval '((+ 1 2))) 3)
(test (apply eval ()) 'error) ; (eval) is an error -- should it be? (eval ()) is () so perhaps (following values), (eval) -> #<unspecified>?
(test (apply eval '(())) ())
(test (apply eval-string '("(+ 1 2)")) 3) 
(test (let () (apply begin '((define x 1) (define y x) (+ x y)))) 2)
(test (apply begin ()) (begin))
(test (apply if '(#f 1 2)) 2)
(test (apply if '(#f)) 'error)
(test (let ((x 1)) (apply set! '(x 3)) x) 3)
(test (let ((x 3)) (apply set! (list (values 'x 32))) x) 32)
(test (let ((x 1)) (apply cond '(((= x 2) 3) ((= x 1) 32)))) 32)
(test (apply and '((= 1 1) (> 2 3))) #f)
(test (apply and ()) (and))
(test (apply or '((= 1 1) (> 2 3))) #t)
(test (apply or ()) (or))
(test (let () (apply define '(x 32)) x) 32)
(test (let () (apply define* '((hi (a 1) (b 2)) (+ a b))) (hi 32)) 34)
(test ((apply lambda '((n) (+ n 1))) 2) 3)
(test ((apply lambda* '(((n 1)) (+ n 1)))) 2)
(test (apply let '(((x 1)) (+ x 2))) 3)
(test (apply let* '(((x 1) (y (* 2 x))) (+ x y))) 3)
(test (equal? (apply let* '((a 2) (b (+ a 3))) '(list + a b) ()) (list + 2 5)) #t)
(test (apply let 'func '((i 1) (j 2)) '((+ i j (if (> i 0) (func (- i 1) j) 0)))) 5)
(test (let () (apply define-macro `((hiho a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro `(hiho (a) `(+ ,a 1))) (hiho 2)) 3)
(test (let () (apply defmacro* `(hiho ((a 2)) `(+ ,a 1))) (hiho)) 3)
(test (let () (apply define-macro* `((hiho (a 2)) `(+ ,a 1))) (hiho)) 3)
(test (apply do '(((i 0 (+ i 1))) ((= i 3) i))) 3)
(test (apply case '(1 ((2 3) 4) ((1 5) 32))) 32)
(test (+ (apply values '(1 2 3))) 6)
(test (apply quote '(1)) 1)
(test (apply quote ()) 'error) ; (quote) is an error
(test (let () (apply letrec '(() (define x 9) x))) 9)
(test ((lambda (n) (apply n '(((x 1)) (+ x 2)))) let) 3)
(test ((apply lambda (list (apply let (list (list) (quote (list (apply case '(0 ((0 1) 'n))))))) (quasiquote (+ n 1)))) 2) 3)
(test (apply let '((x 1)) '((+ x 1))) 2)
(test ((apply dilambda (list (lambda (x) (+ x 1)) (lambda (x y) (+ x y)))) 23) 24)
(test (apply (apply dilambda (list (lambda (x) (+ x 1)) (lambda (x y) (+ x y)))) '(23)) 24)
(test (apply map list '((1 one) (2 two) (3 three))) '((1 2 3) (one two three))) ; from scheme bboard
;;; so (define (unzip l) (apply values (apply map list l)))

(test (apply 'begin) 'error)
(test (apply and) #t)
(test (apply begin) ())
(test (apply if '((> 1 2) 3 4)) 4)
(test (apply or) #f)
(test (apply quote '(1)) 1)

(let ()
  (define (min-max arg . args)
    (if (null? args)
	(apply max arg)
	(min (apply max arg) 
	     (apply min-max args))))

  (test (min-max '(1 2 3) '(0 -1 4)) 3)
  (test (min-max '(1 2 3) '(0 -1 4) '(1 2)) 2))





;;; --------------------------------------------------------------------------------
;;; define
;;; --------------------------------------------------------------------------------

;;; trying to avoid top-level definitions here

(let ()
  (define x 2)
  (test (+ x 1) 3)
  (set! x 4)
  (test (+ x 1) 5)
  (let ()
    (define (tprint x) #t)
    (test (tprint 56) #t)
    (let ()
      (define first car)
      (test (first '(1 2)) 1)
      (let ()
	(define foo (lambda () (define x 5) x))
	(test (foo) 5)
	(let ()
	  (define (foo x) ((lambda () (define x 5) x)) x)
	  (test (foo 88) 88))))))


(test (letrec ((foo (lambda (arg) (or arg (and (procedure? foo) (foo 99)))))) (define bar (foo #f)) (foo #f)) 99)
(test (letrec ((foo 77) (bar #f) (retfoo (lambda () foo))) (define baz (retfoo)) (retfoo)) 77)

(test (let () (define .. 1) ..) 1)

(test (let () (define (hi a) (+ a 1)) (hi 2)) 3)
(test (let () (define (hi a . b) (+ a (cadr b) 1)) (hi 2 3 4)) 7)
(test (let () (define (hi) 1) (hi)) 1)
(test (let () (define (hi . a) (apply + a)) (hi 1 2 3)) 6)

(for-each
 (lambda (arg)
   (test (let () (define x arg) x) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test ((lambda (x) (define (hi a) (+ a 1)) (hi x)) 1) 2)
(test (let ((x 2)) (define f (lambda (y) (+ y x))) (f 3)) 5)
(begin (define r5rstest-plus (lambda (x y) (+ x y))) (define r5rstest-x 32))
(test (r5rstest-plus r5rstest-x 3) 35)
(test (let ((x 2.0)) (define (hi a) (set! a 3.0)) (hi x) x) 2.0)

(test (let () (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let ((asdf 1)) (define (asdf a) (define (asdf a) (+ a 1)) (+ a (asdf a))) (asdf 4)) 9)
(test (let () (define (a1 a) (define (a2 a) (define (a3 a) (define (a4 a) (+ a 1)) (+ (a4 a) 1)) (+ (a3 a) 1)) (+ (a2 a) 1)) (a1 0)) 4)

(test (let () (define (hi1 a) (define (hi1 b) (+ b 1)) (hi1 a)) (hi1 1)) 2)
(test (let () (define (hi1 a) (begin (define (hi1 b) (+ b 1))) (hi1 a)) (hi1 1)) 2)
(test (let ((j 0) (k 0))
	(define (hi1 a)
	  (let ((hi1 (lambda (b) 
		       (set! k (+ k 1)) 
		       (hi1 (- b 1)))))
	    (if (<= a 0)
		(list j k)
		(begin
		  (set! j (+ j 1))
		  (hi1 (- a 1))))))
	(hi1 3))
      '(2 2))

(test (procedure? (let () (define (a) a) (a))) #t)
(let ((oddp (lambda (a) (not (even? a)))))
   (define (hi a) (and (a 123) (a 321))) 
   (test (hi oddp) #t))

(test (define) 'error)
(test (define*) 'error)
(test (define x) 'error)
(test (define . x) 'error)
(test (define x 1 2) 'error)
(test (define x x) 'error)
(test (define x x x) 'error)
(test (define x x . x) 'error)
(test (let ((x 0)) (define x (x . x))) 'error)
(test (define (x x) . x) 'error)
(test (eval-string "(define (x .) 1)") 'error) ; need eval-string else a read error that halts our tests
(test (eval-string "(define (x) (+ 1 .))") 'error)
(test (define (x x) x . x) 'error)
(test (let () (define (x x) x) (x 0)) 0)
(test (define (x 1)) 'error)
(test (define (x)) 'error)
(test (define 1 2) 'error)
(test (define "hi" 2) 'error)
(test (define :hi 2) 'error)
(test (define x 1 2) 'error)
(test (define x 1 . 2) 'error)
(test (define x . 1) 'error)
(test (define x (lambda ())) 'error)
(test (define #<eof> 3) 'error)
(test (define (#<undefined>) 4) 'error)
(test (define (:hi a) a) 'error)
(test (define (hi: a) a) 'error)
(test (define (#b1 a) a) 'error)
(test (define (hi #b1) #b1) 'error)
(test (define () 1) 'error)
(test (let() (define #(hi a) a)) 'error)
(test (let () (define hi (lambda args args)) (hi 1 . 2)) 'error)
(test (let () (define . 1) 1) 'error)
(test (let () (define func (do () (#t (lambda (y) 2)))) (func 1)) 2)
(test (let () (define* x 3)) 'error)
(test (let () (define (hi) 1 . 2)) 'error)
(test (let () (define (hi) (1) . "hi")) 'error)

(test (let () (define (f f) f) (f 0)) 0)
(test (let () (define (f . f) f) (f 1 2)) '(1 2))
(test (let () (define (f f) (define* (f1 (f f)) f) (f1)) (f 0)) 0)
(test (let () (define (f1 f) (define* (f (f f)) f) (f)) (procedure? (f1 0))) #t) ; ?? see comment in s7.c -- this might also return 0
(test (let () (define (f f) (define* (f (f f)) f) (f)) (procedure? (f 0))) #t)
(test (let ((f1 (define f2 32))) (+ f1 f2)) 64)
(test (let () (define x (+ (define y 3) 2)) (list x y)) '(5 3))
(test (let ((a 1) (b 2)) (define (f a b) (let ((a a) (b b)) (+ a b))) (f 4 3)) 7)
(test (let () (define (f x . y) (if (> x 0) (f (- x 1) y) y)) (f 4 1)) '(((((1))))))

;;; --------
;;; check envs
(test (let () (do ((i 0 (+ i 1))) ((= i 3) (define xyz 37) i)) xyz) 'error)
(test (let () (do ((i 0 (+ i 1))) ((= i 3)) (define xyz 37)) xyz) 'error)
(test (let () (do ((i (begin (define xyz 37) 0) (+ i 1))) ((= i 3))) xyz) 37)
(test (let () (do ((i 0 (begin (define xyz 37) (+ i 1)))) ((= i 3))) xyz) 'error)

(test (let () (let ((i (begin (define xyz 37) 0))) i) xyz) 37)
(test (let () (let ((i 0)) (define xyz 37) i) xyz) 'error)

(test (let () (let* ((i (begin (define xyz 37) 0))) i) xyz) 37)
(test (let () (let* ((i 0)) (define xyz 37) i) xyz) 'error)
(test (let () (let* ((k 0) (i (begin (define xyz 37) 0))) i) xyz) 'error)

(test (let () (letrec ((i (begin (define xyz 37) 0))) i) xyz) 'error)            ; ??? it's defined in the letrec
(test (let () (letrec ((i (begin (define xyz 37) 0))) xyz)) 37)                  ; !
(test (let () (letrec ((i 0)) (define xyz 37) i) xyz) 'error)

(test (let () (letrec* ((i (begin (define xyz 37) 0))) i) xyz) 'error)           ; ??? same as above
(test (let () (letrec* ((i (begin (define xyz 37) 0))) xyz)) 37)                 ; !
(test (let () (letrec* ((i 0)) (define xyz 37) i) xyz) 'error)
(test (let () (letrec* ((k 0) (i (begin (define xyz 37) 0))) i) xyz) 'error)

(test (let () (cond ((define xyz 37) #f)) xyz) 37)
(test (let () (cond ((> 2 1) (define xyz 37) #f)) xyz) 37)
(test (let () (cond ((< 2 1) 0) (else (define xyz 37) #f)) xyz) 37)

(test (let () (if (define xyz 37) 0 1) xyz) 37)
(test (let () (if (> 2 1) (define xyz 37) 1) xyz) 37)
(test (let () (if (< 2 1) 0 (define xyz 37)) xyz) 37)

(test (let () (when (define xyz 37) #f) xyz) 37)
(test (let () (when (> 2 1) (define xyz 37) #f) xyz) 37)
(test (let () (unless (define xyz 37) #f) xyz) 37)
(test (let () (unless (< 2 1) (define xyz 37) #f) xyz) 37)

(test (let () (quote (define xyz 37)) xyz) 'error)
(test (let () (begin (define xyz 37) 0) xyz) 37)
(test (let () (and (define xyz 37) 0) xyz) 37)
(test (let () (or (define xyz 37) 0) xyz) 37)
(test (let ((x 0)) (set! x (define xyz 37)) xyz) 37)
(test (let () (with-let (curlet) (define xyz 37)) xyz) 37)
(test (let () (with-let (inlet 'a 1) (define xyz 37)) xyz) 'error)
(test (let () (with-baffle (define xyz 37) 2) xyz) 'error)                      ; with-baffle introduces a new frame

(test (let () (case (define xyz 37) ((0) 1) ((37) #t)) xyz) 37)
(test (let () (case 1 ((1) (define xyz 37)) ((0) 1)) xyz) 37)
(test (let () (case 1 ((0) 1) (else (define xyz 37) 2)) xyz) 37)

(test (let () (lambda () (define xyz 37)) xyz) 'error)
(test (let () (define* (fxyz (a (define xyz 37))) a) (fxyz) xyz) 'error)        ; ??? it's defined in fxyz!
(test (let () (define* (fxyz (a (define xyz 37))) xyz) (fxyz)) 37)              ; !
;;; --------

(let ()
  (define a#b 3)
  (define a'b 4)
  (define a,b 5)
  (define a[b 6)
  (define a@b 7)
  (define a\b 8)
  (define a|b 9)
  (test (+ a#b a'b a,b a[b a@b a\b a|b) 42))

(let ()
  (define (make-func) (define (a-func a) (+ a 1)))
  (test (procedure? (make-func)) #t))

(let () (test (if (and (define x 3) (define y 4)) (+ x y)) 7))
(let () (test (if (not (and (define x 2) (define y 4))) (+ x y) (if (define x 3) x)) 3))
(let () (test (if (and (define x 2) (not (define y 4))) (+ x y) (- x y)) -2))
(test (let () (define (f a) (lambda () a)) (+ ((f 1)) ((f 2)))) 3)
(test (let () (define (hi) (let ((a 1)) (set! a 2) (define (ho) a) (set! a 3) (ho))) (hi)) 3)
;;; (define-macro (make-lambda args . body) `(apply lambda* ',args '(,@body))): (make-lambda (a b) (+ a b))

;; y combinator example from some CS website
(let ()
  (define Y
    (lambda (X)
      ((lambda (procedure)
         (X (lambda (arg) ((procedure procedure) arg))))
       (lambda (procedure)
         (X (lambda (arg) ((procedure procedure) arg)))))))

  (define M
    (lambda (func-arg)
      (lambda (l)
        (if (null? l)
            'no-list
            (if (null? (cdr l))
                (car l)
                (max (car l) (func-arg (cdr l))))))))

  (test ((Y M) '(4 5 6 3 4 8 6 2)) 8))

(test (((lambda (X)
	  ((lambda (procedure)
	     (X (lambda (arg) ((procedure procedure) arg))))
	   (lambda (procedure)
	     (X (lambda (arg) ((procedure procedure) arg))))))
	(lambda (func-arg)
	  (lambda (n)
	    (if (zero? n)
		1
		(* n (func-arg (- n 1)))))))
       5)
      120)

;;; from a paper by Mayer Goldberg
(let ()
  (define curry-fps
    (lambda fs
      (let ((xs
	     (map
	      (lambda (fi)
		(lambda xs
		  (apply fi
			 (map
			  (lambda (xi)
			    (lambda args
			      (apply (apply xi xs) args)))
			  xs))))
	      fs)))
	(map (lambda (xi)
	       (apply xi xs)) xs))))
  
  (define E
    (lambda (even? odd?)
      (lambda (n)
        (if (zero? n) #t ; return Boolean True
            (odd? (- n 1))))))
  
  (define O
    (lambda (even? odd?)
      (lambda (n)
        (if (zero? n) #f ; return Boolean False
            (even? (- n 1))))))
  
  (define list-even?-odd? (curry-fps E O))
  (define new-even? (car list-even?-odd?))
  (define new-odd? (cadr list-even?-odd?))
  
  (test (new-even? 6) #t)
  (test (new-odd? 6) #f))

(let ()
  (define (Cholesky:decomp P)
    ;; from Marco Maggi based on a Scheme bboard post
    ;; (Cholesky:decomp '((2 -2) (-2 5))) -> ((1.4142135623731 0) (-1.4142135623731 1.7320508075689))
    (define (Cholesky:make-square L)
      (define (zero-vector n)
	(if (zero? n)
	    ()
	    (cons 0 (zero-vector (- n 1)))))
      (map (lambda (v)
	     (append v (zero-vector (- (length L) (length v)))))
	   L))
    (define (Cholesky:add-element P L i j)
      (define (Cholesky:smaller P)
	(if (null? (cdr P))
	    ()
	    (reverse (cdr (reverse P)))))
      (define (Cholesky:last-row L)
	(car (reverse L)))
      (define (matrix:element A i j)
	(list-ref (list-ref A i) j))
      (define (Cholesky:make-element P L i j)
	(define (Cholesky:partial-sum L i j)
	  (let loop ((k j))
	    (case k
	      ((0) 0)
	      ((1) (* (matrix:element L i 0)
		      (matrix:element L j 0)))
	      (else
	       (+ (* (matrix:element L i k)
		     (matrix:element L j k))
		  (loop (- k 1)))))))
	(let ((x (- (matrix:element P i j)
		    (Cholesky:partial-sum L i j))))
	  (if (= i j)
	      (sqrt x)
	      (/ x (matrix:element L j j)))))
      (if (zero? j)
	  (append L `((,(Cholesky:make-element P L i j))))
	  (append (Cholesky:smaller L)
		  (list (append
			 (Cholesky:last-row L)
			 (list (Cholesky:make-element P L i j)))))))
    (Cholesky:make-square
     (let iter ((i 0) (j 0) (L ()))
       (if (>= i (length P))
	   L
	   (iter (if (= i j) (+ 1 i) i)
		 (if (= i j) 0 (+ 1 j))
		 (Cholesky:add-element P L i j))))))
  (let* ((lst (Cholesky:decomp '((2 -2) (-2 5))))
	 (lst0 (car lst))
	 (lst1 (cadr lst)))
    (if (or (> (abs (- (car lst0) (sqrt 2))) .0001)
	    (not (= (cadr lst0) 0))
	    (> (abs (+ (car lst1) (sqrt 2))) .0001)
	    (> (abs (- (cadr lst1) (sqrt 3))) .0001))
	(format-logged #t ";cholesky decomp: ~A~%" lst))))

(let () ; from Programming Praxis
  (define (A k x1 x2 x3 x4 x5)
    (define (B)
      (set! k (- k 1))
      (A k B x1 x2 x3 x4))
    (if (<= k 0) 
	(+ (x4) (x5))
	(B)))
  (test (A 10 (lambda () 1) (lambda () -1) (lambda () -1) (lambda () 1) (lambda () 0)) -67))

(let ()
  (define* (a1 (b (let ()
		    (define* (a1 (b 32)) b)
		    (a1))))
    b)
  (test (a1) 32)
  (test (a1 1) 1))

(test (let ((f1 (lambda (x) (+ x 1)))) 
	(define f1 (lambda (y) (if (zero? y) y (f1 (- y 1))))) 
	(f1 3))
      0)

(test (let ((x 1)) (cond (else (define x 2))) x) 2)
(test (let ((x 1)) (and (define x 2)) x) 2)
(test (let () (begin 1)) 1)
(test (let () (begin (define x 1) x)) 1)
(test (let () (let ((lst (define abc 1))) #f) abc) 1)                  ; ?? 
(test (let () (let ((lst (define abc 1))) abc)) 1)                     ; abcd is in the outer let
(test (let () (letrec ((lst (define abcd 1))) #f) abcd) 'error)        ; abcd: unbound variable
(test (let () (letrec ((lst (define abcd 1))) abcd)) 1)
(test (let? (let () (letrec ((lst (define abcd 1))) (curlet)))) #t)
(test (let () (letrec* ((lst (define abcd 1))) abcd)) 1)               ; unproblematic because no pending value
;(test (let () (define (f a) (if (symbol? a) b 0)) (f (define b 3))) 3) ; 25-Jul-14
(test (let () (+ (define b 1) (* b 2))) 3)
(test (let () (if (define b 3) b)) 3)
(test (let () (do ((i (define b 3)) (j 0 (+ j 1))) ((= j 0) b))) 3)
(test (let () (define* (f (a (define b 3))) a) (f) b) 'error)          ; ?? where is b?
(test (let () (define* (f (a (define b 3))) b) (f)) 3)                 ; inside the func apparently! 3 cases? let->outer letrec->cur, func->inner!
;(test (let () (define* (f (a (define a 3))) a) (f)) 'a)                ; yow -- no duplicate id check! 25-Jul-14
(test (let () (define* (f (a 1) (b (define a 3))) a) (f 2)) 3)
(test (let () (define-macro* (f (a 1) (b (define a 3))) a) (f 2)) 2)   ; also bacro -- b is '(define a 3)!
;(test (let () (letrec ((a (define a 3))) a)) 'a)                       ; letrec is the same (it checks normally) 25-Jul-14
(test (let () (letrec ((a 1) (b (define a 3))) a)) 1)
(test (let () (letrec* ((a 1) (b (define a 3))) a)) 3)                 ; same difference in let/let*
(test (let () (letrec ((a 1) (b (set! a 3))) a)) 1)
(test (let () (letrec* ((a 1) (b (set! a 3))) a)) 3)                   ; here the let case is an error, let* is 3
;(test (let () (list (with-let (sublet (curlet) (cons 'b (define b 3))) b) ((curlet) 'b))) '(b 3)) ; 2 b's with 1 define
;(test (let () (list (with-let (varlet (curlet) (cons 'b (define b 3))) b) ((curlet) 'b))) '(b b))

(let ()
  (define (f64 arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64) 
    (+ arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64))
  (test (f64 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64) 
	2080))

#|
(let ((n 12))
  (let ((nums (do ((lst () (cons i lst))
		   (i 0 (+ i 1)))
		  ((> i n) (reverse lst)))))
    (format-logged #t "(let ((f~D (lambda (~{arg~D~^ ~})~%    (+ ~{arg~D~^ ~}))))~%  (f~D ~{~D~^ ~}))~%" n nums nums n nums)))
|#

(test (let ((f128 (lambda (arg128 arg127 arg126 arg125 arg124 arg123 arg122 arg121 arg120 arg119 arg118 arg117 arg116 arg115 arg114 arg113 arg112 arg111 arg110 arg109 arg108 arg107 arg106 arg105 arg104 arg103 arg102 arg101 arg100 arg99 arg98 arg97 arg96 arg95 arg94 arg93 arg92 arg91 arg90 arg89 arg88 arg87 arg86 arg85 arg84 arg83 arg82 arg81 arg80 arg79 arg78 arg77 arg76 arg75 arg74 arg73 arg72 arg71 arg70 arg69 arg68 arg67 arg66 arg65 arg64 arg63 arg62 arg61 arg60 arg59 arg58 arg57 arg56 arg55 arg54 arg53 arg52 arg51 arg50 arg49 arg48 arg47 arg46 arg45 arg44 arg43 arg42 arg41 arg40 arg39 arg38 arg37 arg36 arg35 arg34 arg33 arg32 arg31 arg30 arg29 arg28 arg27 arg26 arg25 arg24 arg23 arg22 arg21 arg20 arg19 arg18 arg17 arg16 arg15 arg14 arg13 arg12 arg11 arg10 arg9 arg8 arg7 arg6 arg5 arg4 arg3 arg2 arg1 arg0)
		    (+ arg128 arg127 arg126 arg125 arg124 arg123 arg122 arg121 arg120 arg119 arg118 arg117 arg116 arg115 arg114 arg113 arg112 arg111 arg110 arg109 arg108 arg107 arg106 arg105 arg104 arg103 arg102 arg101 arg100 arg99 arg98 arg97 arg96 arg95 arg94 arg93 arg92 arg91 arg90 arg89 arg88 arg87 arg86 arg85 arg84 arg83 arg82 arg81 arg80 arg79 arg78 arg77 arg76 arg75 arg74 arg73 arg72 arg71 arg70 arg69 arg68 arg67 arg66 arg65 arg64 arg63 arg62 arg61 arg60 arg59 arg58 arg57 arg56 arg55 arg54 arg53 arg52 arg51 arg50 arg49 arg48 arg47 arg46 arg45 arg44 arg43 arg42 arg41 arg40 arg39 arg38 arg37 arg36 arg35 arg34 arg33 arg32 arg31 arg30 arg29 arg28 arg27 arg26 arg25 arg24 arg23 arg22 arg21 arg20 arg19 arg18 arg17 arg16 arg15 arg14 arg13 arg12 arg11 arg10 arg9 arg8 arg7 arg6 arg5 arg4 arg3 arg2 arg1 arg0))))
	(f128 128 127 126 125 124 123 122 121 120 119 118 117 116 115 114 113 112 111 110 109 108 107 106 105 104 103 102 101 100 99 98 97 96 95 94 93 92 91 90 89 88 87 86 85 84 83 82 81 80 79 78 77 76 75 74 73 72 71 70 69 68 67 66 65 64 63 62 61 60 59 58 57 56 55 54 53 52 51 50 49 48 47 46 45 44 43 42 41 40 39 38 37 36 35 34 33 32 31 30 29 28 27 26 25 24 23 22 21 20 19 18 17 16 15 14 13 12 11 10 9 8 7 6 5 4 3 2 1 0))
      8256)

(test (let ((f512 (lambda (arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64 arg65 arg66 arg67 arg68 arg69 arg70 arg71 arg72 arg73 arg74 arg75 arg76 arg77 arg78 arg79 arg80 arg81 arg82 arg83 arg84 arg85 arg86 arg87 arg88 arg89 arg90 arg91 arg92 arg93 arg94 arg95 arg96 arg97 arg98 arg99 arg100 arg101 arg102 arg103 arg104 arg105 arg106 arg107 arg108 arg109 arg110 arg111 arg112 arg113 arg114 arg115 arg116 arg117 arg118 arg119 arg120 arg121 arg122 arg123 arg124 arg125 arg126 arg127 arg128 arg129 arg130 arg131 arg132 arg133 arg134 arg135 arg136 arg137 arg138 arg139 arg140 arg141 arg142 arg143 arg144 arg145 arg146 arg147 arg148 arg149 arg150 arg151 arg152 arg153 arg154 arg155 arg156 arg157 arg158 arg159 arg160 arg161 arg162 arg163 arg164 arg165 arg166 arg167 arg168 arg169 arg170 arg171 arg172 arg173 arg174 arg175 arg176 arg177 arg178 arg179 arg180 arg181 arg182 arg183 arg184 arg185 arg186 arg187 arg188 arg189 arg190 arg191 arg192 arg193 arg194 arg195 arg196 arg197 arg198 arg199 arg200 arg201 arg202 arg203 arg204 arg205 arg206 arg207 arg208 arg209 arg210 arg211 arg212 arg213 arg214 arg215 arg216 arg217 arg218 arg219 arg220 arg221 arg222 arg223 arg224 arg225 arg226 arg227 arg228 arg229 arg230 arg231 arg232 arg233 arg234 arg235 arg236 arg237 arg238 arg239 arg240 arg241 arg242 arg243 arg244 arg245 arg246 arg247 arg248 arg249 arg250 arg251 arg252 arg253 arg254 arg255 arg256 arg257 arg258 arg259 arg260 arg261 arg262 arg263 arg264 arg265 arg266 arg267 arg268 arg269 arg270 arg271 arg272 arg273 arg274 arg275 arg276 arg277 arg278 arg279 arg280 arg281 arg282 arg283 arg284 arg285 arg286 arg287 arg288 arg289 arg290 arg291 arg292 arg293 arg294 arg295 arg296 arg297 arg298 arg299 arg300 arg301 arg302 arg303 arg304 arg305 arg306 arg307 arg308 arg309 arg310 arg311 arg312 arg313 arg314 arg315 arg316 arg317 arg318 arg319 arg320 arg321 arg322 arg323 arg324 arg325 arg326 arg327 arg328 arg329 arg330 arg331 arg332 arg333 arg334 arg335 arg336 arg337 arg338 arg339 arg340 arg341 arg342 arg343 arg344 arg345 arg346 arg347 arg348 arg349 arg350 arg351 arg352 arg353 arg354 arg355 arg356 arg357 arg358 arg359 arg360 arg361 arg362 arg363 arg364 arg365 arg366 arg367 arg368 arg369 arg370 arg371 arg372 arg373 arg374 arg375 arg376 arg377 arg378 arg379 arg380 arg381 arg382 arg383 arg384 arg385 arg386 arg387 arg388 arg389 arg390 arg391 arg392 arg393 arg394 arg395 arg396 arg397 arg398 arg399 arg400 arg401 arg402 arg403 arg404 arg405 arg406 arg407 arg408 arg409 arg410 arg411 arg412 arg413 arg414 arg415 arg416 arg417 arg418 arg419 arg420 arg421 arg422 arg423 arg424 arg425 arg426 arg427 arg428 arg429 arg430 arg431 arg432 arg433 arg434 arg435 arg436 arg437 arg438 arg439 arg440 arg441 arg442 arg443 arg444 arg445 arg446 arg447 arg448 arg449 arg450 arg451 arg452 arg453 arg454 arg455 arg456 arg457 arg458 arg459 arg460 arg461 arg462 arg463 arg464 arg465 arg466 arg467 arg468 arg469 arg470 arg471 arg472 arg473 arg474 arg475 arg476 arg477 arg478 arg479 arg480 arg481 arg482 arg483 arg484 arg485 arg486 arg487 arg488 arg489 arg490 arg491 arg492 arg493 arg494 arg495 arg496 arg497 arg498 arg499 arg500 arg501 arg502 arg503 arg504 arg505 arg506 arg507 arg508 arg509 arg510 arg511 arg512)
    (+ arg0 arg1 arg2 arg3 arg4 arg5 arg6 arg7 arg8 arg9 arg10 arg11 arg12 arg13 arg14 arg15 arg16 arg17 arg18 arg19 arg20 arg21 arg22 arg23 arg24 arg25 arg26 arg27 arg28 arg29 arg30 arg31 arg32 arg33 arg34 arg35 arg36 arg37 arg38 arg39 arg40 arg41 arg42 arg43 arg44 arg45 arg46 arg47 arg48 arg49 arg50 arg51 arg52 arg53 arg54 arg55 arg56 arg57 arg58 arg59 arg60 arg61 arg62 arg63 arg64 arg65 arg66 arg67 arg68 arg69 arg70 arg71 arg72 arg73 arg74 arg75 arg76 arg77 arg78 arg79 arg80 arg81 arg82 arg83 arg84 arg85 arg86 arg87 arg88 arg89 arg90 arg91 arg92 arg93 arg94 arg95 arg96 arg97 arg98 arg99 arg100 arg101 arg102 arg103 arg104 arg105 arg106 arg107 arg108 arg109 arg110 arg111 arg112 arg113 arg114 arg115 arg116 arg117 arg118 arg119 arg120 arg121 arg122 arg123 arg124 arg125 arg126 arg127 arg128 arg129 arg130 arg131 arg132 arg133 arg134 arg135 arg136 arg137 arg138 arg139 arg140 arg141 arg142 arg143 arg144 arg145 arg146 arg147 arg148 arg149 arg150 arg151 arg152 arg153 arg154 arg155 arg156 arg157 arg158 arg159 arg160 arg161 arg162 arg163 arg164 arg165 arg166 arg167 arg168 arg169 arg170 arg171 arg172 arg173 arg174 arg175 arg176 arg177 arg178 arg179 arg180 arg181 arg182 arg183 arg184 arg185 arg186 arg187 arg188 arg189 arg190 arg191 arg192 arg193 arg194 arg195 arg196 arg197 arg198 arg199 arg200 arg201 arg202 arg203 arg204 arg205 arg206 arg207 arg208 arg209 arg210 arg211 arg212 arg213 arg214 arg215 arg216 arg217 arg218 arg219 arg220 arg221 arg222 arg223 arg224 arg225 arg226 arg227 arg228 arg229 arg230 arg231 arg232 arg233 arg234 arg235 arg236 arg237 arg238 arg239 arg240 arg241 arg242 arg243 arg244 arg245 arg246 arg247 arg248 arg249 arg250 arg251 arg252 arg253 arg254 arg255 arg256 arg257 arg258 arg259 arg260 arg261 arg262 arg263 arg264 arg265 arg266 arg267 arg268 arg269 arg270 arg271 arg272 arg273 arg274 arg275 arg276 arg277 arg278 arg279 arg280 arg281 arg282 arg283 arg284 arg285 arg286 arg287 arg288 arg289 arg290 arg291 arg292 arg293 arg294 arg295 arg296 arg297 arg298 arg299 arg300 arg301 arg302 arg303 arg304 arg305 arg306 arg307 arg308 arg309 arg310 arg311 arg312 arg313 arg314 arg315 arg316 arg317 arg318 arg319 arg320 arg321 arg322 arg323 arg324 arg325 arg326 arg327 arg328 arg329 arg330 arg331 arg332 arg333 arg334 arg335 arg336 arg337 arg338 arg339 arg340 arg341 arg342 arg343 arg344 arg345 arg346 arg347 arg348 arg349 arg350 arg351 arg352 arg353 arg354 arg355 arg356 arg357 arg358 arg359 arg360 arg361 arg362 arg363 arg364 arg365 arg366 arg367 arg368 arg369 arg370 arg371 arg372 arg373 arg374 arg375 arg376 arg377 arg378 arg379 arg380 arg381 arg382 arg383 arg384 arg385 arg386 arg387 arg388 arg389 arg390 arg391 arg392 arg393 arg394 arg395 arg396 arg397 arg398 arg399 arg400 arg401 arg402 arg403 arg404 arg405 arg406 arg407 arg408 arg409 arg410 arg411 arg412 arg413 arg414 arg415 arg416 arg417 arg418 arg419 arg420 arg421 arg422 arg423 arg424 arg425 arg426 arg427 arg428 arg429 arg430 arg431 arg432 arg433 arg434 arg435 arg436 arg437 arg438 arg439 arg440 arg441 arg442 arg443 arg444 arg445 arg446 arg447 arg448 arg449 arg450 arg451 arg452 arg453 arg454 arg455 arg456 arg457 arg458 arg459 arg460 arg461 arg462 arg463 arg464 arg465 arg466 arg467 arg468 arg469 arg470 arg471 arg472 arg473 arg474 arg475 arg476 arg477 arg478 arg479 arg480 arg481 arg482 arg483 arg484 arg485 arg486 arg487 arg488 arg489 arg490 arg491 arg492 arg493 arg494 arg495 arg496 arg497 arg498 arg499 arg500 arg501 arg502 arg503 arg504 arg505 arg506 arg507 arg508 arg509 arg510 arg511 arg512))))
  (f512 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100 101 102 103 104 105 106 107 108 109 110 111 112 113 114 115 116 117 118 119 120 121 122 123 124 125 126 127 128 129 130 131 132 133 134 135 136 137 138 139 140 141 142 143 144 145 146 147 148 149 150 151 152 153 154 155 156 157 158 159 160 161 162 163 164 165 166 167 168 169 170 171 172 173 174 175 176 177 178 179 180 181 182 183 184 185 186 187 188 189 190 191 192 193 194 195 196 197 198 199 200 201 202 203 204 205 206 207 208 209 210 211 212 213 214 215 216 217 218 219 220 221 222 223 224 225 226 227 228 229 230 231 232 233 234 235 236 237 238 239 240 241 242 243 244 245 246 247 248 249 250 251 252 253 254 255 256 257 258 259 260 261 262 263 264 265 266 267 268 269 270 271 272 273 274 275 276 277 278 279 280 281 282 283 284 285 286 287 288 289 290 291 292 293 294 295 296 297 298 299 300 301 302 303 304 305 306 307 308 309 310 311 312 313 314 315 316 317 318 319 320 321 322 323 324 325 326 327 328 329 330 331 332 333 334 335 336 337 338 339 340 341 342 343 344 345 346 347 348 349 350 351 352 353 354 355 356 357 358 359 360 361 362 363 364 365 366 367 368 369 370 371 372 373 374 375 376 377 378 379 380 381 382 383 384 385 386 387 388 389 390 391 392 393 394 395 396 397 398 399 400 401 402 403 404 405 406 407 408 409 410 411 412 413 414 415 416 417 418 419 420 421 422 423 424 425 426 427 428 429 430 431 432 433 434 435 436 437 438 439 440 441 442 443 444 445 446 447 448 449 450 451 452 453 454 455 456 457 458 459 460 461 462 463 464 465 466 467 468 469 470 471 472 473 474 475 476 477 478 479 480 481 482 483 484 485 486 487 488 489 490 491 492 493 494 495 496 497 498 499 500 501 502 503 504 505 506 507 508 509 510 511 512))
      131328)


(let ((x 32))
  (define (f1) x)
  (define x 33)
  (test (f1) 33))

(let ()
  (define (c-2)
    (let ((v (vector 1 2 3)))
      (define (c-1 a b) (+ (vector-ref a 0) (* b 32)))
      (let ((c (c-1 v 1)))
	(test c 33)
	(set! c-1 vector-ref))
      (let ((d (c-1 v 1)))
	(test d 2))))
  (c-2))

(let ()
  (define (c-2)
    (let ((v (vector 1 2 3)))
      (let ()
	(define (c-1 a b) (+ (vector-ref a 0) (* b 32)))
	(let ((c (c-1 v 1)))
	  (set! c-1 vector-ref)))
      (test (c-1 v 1) 'error)))
  (c-2))
(let ()
  (define (f4 a b c d e) (list a b c d e))
  (test (f4 1 2 3 4 5) '(1 2 3 4 5)))

(define (redef-1 a) (+ a 1))
(define (use-redef-1 b) (+ (redef-1 b) 2))     ; [use-redef](+ [redef-1](+ b 1) 2)
(test (use-redef-1 3) 6)                       ; b=6
(define (redef-1 a) (+ a 4))
(test (use-redef-1 3) 9)                       ; [use-redef-1](+ [redef-1](+ a 4) 2), a=3
(let ()
  (define (use-redef-2 c) (+ (redef-1 c) 5))   ; [use-redef-2](+ [redef-1](+ a 4) 5)
  (test (use-redef-2 6) 15)                    ; a=6
  (define (redef-1 a) (+ a 7))                 ; so use-redef-1 is still [use-redef-1](+ [redef-1](+ a 4) 2)
  (test (use-redef-1 8) 14)                    ; a=8 -> 14
  (test (use-redef-2 8) 20))                   ; but use-redef-2 (same let as shadowing use-redef-1) is (+ [new redef-1](+ a 7) 5), a=8 -> 20

(test (let () (define (f1 x) (abs x)) (define (f2 x) (f1 x)) (f2 -1)) 1) ; just trying to hit a portion of the s7 code

(when with-block
  (let ()
    (define (f1) ((lambda (x) (cf11 x)) 3)) (f1) 
    (define (f2) ((lambda () (cf21 3 4)))) (f2) 
    (define (f3) ((lambda () (cf11 3)))) (f3) 
    (define (f4) ((lambda (x) (cf11 'x)) 4)) (f4) 
    
    (define (f5) ((lambda (x) (cf21 x 4)) 3)) (f5) 
    (define (f6) ((lambda (x) (cf21 3 x)) 4)) (f6) 
    (define (f7) ((lambda (x) (cf21 3 4)) 4)) (f7) 
    (define (f8) ((lambda (x y) (cf21 x y)) 3 4)) (f8) 
    
    (define (f9) ((lambda (x) (cf21 x 'y)) 'x)) (f9) 
    (define (f10) ((lambda (x) (cf21 'y x)) 'x)) (f10) 
    (define (f11) ((lambda (x) (cf21 'y 'x)) 'x)) (f11) 
    (define (f12) ((lambda (x) (cf21 1 'x)) 'x)) (f12) 
    (define (f13) ((lambda (x) (cf21 'x 1)) 'x)) (f13) 
    
    (define (f14) ((lambda (x y z) (cf31 x y z)) 1 2 3)) (f14)
    (define (f15) ((lambda (x y z) (cf31 x 2 z)) 1 2 3)) (f15)
    (define (f16) ((lambda (x y z) (cf31 x y 2)) 1 2 3)) (f16)
    (define (f17) ((lambda (x y z) (cf31 2 y z)) 1 2 3)) (f17)
    
    (define (f18) ((lambda (w x y z) (cf41 w x y z)) 1 2 3 4)) (f18)
    (define (f19) ((lambda (x y z) (cf31 x 'y z)) 1 2 3)) (f19)
    (define (f20) ((lambda (x y z) (cf41 'q x y z)) 1 2 3)) (f20)
    (define (f21) ((lambda (x y) (cf31 x y (+ x 6))) 1 2)) (f21)
    (define (f22) ((lambda (x y) (cf31 1 y (+ x 6))) 1 2)) (f22)
    (define (f23) ((lambda (x y) (cf31 x 1 (+ x 6))) 1 2)) (f23)
    (define (f24) ((lambda (x y) (cf31 1 (+ x 6) y)) 1 2)) (f24)
    (define (f25) ((lambda (x) (cf11 (cf11 'x))) 1)) (f25)
    
    (define (f26) ((lambda (w x y z) (cf51 'q w x y z)) 1 2 3 4)) (f26)
    (define (f27) ((lambda (w x y) (cf21 (cf21 (cf11 w) (cf11 x)) (cf11 y))) 1 2 3)) (f27)
    (define (f28) ((lambda (w x y) (cf31 (cf21 w x) 2 y)) 1 2 3)) (f28)
    
    (define (f29) ((lambda () (cf11 (cf11 0))))) (f29)
    (define (f30) ((lambda (x) (cf11 (cf11 x))) 0)) (f30)
    (define (f31) ((lambda (x) (cf11 (cf10 x))) 0)) (f31)
    (define (f32) ((lambda (x) (cf11 (cf10 'x))) 1)) (f32)
    (define (f33) ((lambda (w x y) (cf31 (cf20 w x) 2 y)) 1 2 3)) (f33)
    (define (f34) ((lambda (w x y) (cf22 (cf21 (cf11 w) (cf11 x)) (cf10 y))) 1 2 3)) (f34)
    (define (f35) ((lambda (w x y) (cf21 (cf20 (cf11 w) (cf11 x)) (cf11 y))) 1 2 3)) (f35)
    (define (f36) ((lambda (w x y) (cf21 (cf21 (cf10 w) (cf11 x)) (cf11 y))) 1 2 3)) (f36)
    
    (define (f37) ((lambda (x y) (cf33 x y (cf20 x 6))) 1 2)) (f37)
    (define (f38) ((lambda (x y) (cf33 1 y (cf20 x 6))) 1 2)) (f38)
    (define (f39) ((lambda (x y) (cf33 x 1 (cf20 x 6))) 1 2)) (f39)
    (define (f40) ((lambda (x y) (cf32 1 (cf20 x 6) y)) 1 2)) (f40)
    
    (define (f41) ((lambda (x y) (cf11 (cf21 x y))) 1 2)) (f41)
    (define (f42) ((lambda (x y) (cf11 (cf21 x 2))) 1 2)) (f42)
    (define (f43) ((lambda (x y) (cf11 (cf21 1 x))) 1 2)) (f43)
    (define (f44) ((lambda (x y) (cf21 x (cf11 y))) 1 2)) (f44)
    (define (f45) ((lambda (x y) (cf21 (cf11 x) y)) 1 2)) (f45)
    (define (f46) ((lambda (x y) (cf21 (cf11 x) 2)) 1 2)) (f46)
    (define (f47) ((lambda (x y) (cf21 (cf11 x) (cf11 y))) 1 2)) (f47)
    
    (define (f48) ((lambda (x y) (cf11 (cf20 x y))) 1 2)) (f48)
    (define (f49) ((lambda (x y) (cf11 (cf20 x 2))) 1 2)) (f49)
    (define (f50) ((lambda (x y) (cf11 (cf20 1 x))) 1 2)) (f50)
    (define (f51) ((lambda (x y) (cf22 x (cf10 y))) 1 2)) (f51)
    (define (f52) ((lambda (x y) (cf21 (cf10 x) y)) 1 2)) (f52)
    (define (f53) ((lambda (x y) (cf21 (cf10 x) 2)) 1 2)) (f53)
    (define (f54) ((lambda (x y) (cf21 (cf10 x) (cf11 y))) 1 2)) (f54)
    (define (f55) ((lambda (x y) (cf22 (cf11 x) (cf10 y))) 1 2)) (f55)
    
    (define (f56) ((lambda (x y) (cf21 1 (cf11 y))) 1 2)) (f56)
    (define (f57) ((lambda (x y) (cf22 1 (cf10 y))) 1 2)) (f57)
    (define (f58) ((lambda (x y z) (cf21 x (cf21 y z))) 1 2 3)) (f58)
    (define (f59) ((lambda (x y z) (cf22 x (cf20 y z))) 1 2 3)) (f59)
    (define (f60) ((lambda (x y z) (cf21 x (cf21 2 z))) 1 2 3)) (f60)
    (define (f61) ((lambda (x y z) (cf22 x (cf20 2 z))) 1 2 3)) (f61)
    (define (f62) ((lambda (x y z) (cf21 x (cf21 y 3))) 1 2 3)) (f62)
    (define (f63) ((lambda (x y z) (cf22 x (cf20 y 3))) 1 2 3)) (f63)
    (define (f64) ((lambda (x y z) (cf21 1 (cf21 2 z))) 1 2 3)) (f64)
    (define (f65) ((lambda (x y z) (cf22 1 (cf20 2 z))) 1 2 3)) (f65)
    (define (f66) ((lambda (x y z) (cf21 1 (cf21 y 3))) 1 2 3)) (f66)
    (define (f67) ((lambda (x y z) (cf22 1 (cf20 y 3))) 1 2 3)) (f67)
    
    (define (f68) ((lambda (x) (cf21 (cf21 2 x) 3)) 1)) (f68)
    (define (f69) ((lambda (x) (cf21 (cf20 2 x) 3)) 1)) (f69)
    (define (f70) ((lambda (x y) (cf21 1 (cf21 x y))) 2 3)) (f70)
    (define (f71) ((lambda (x y) (cf22 1 (cf20 x y))) 2 3)) (f71)
    (define (f72) ((lambda (x y) (cf21 (cf21 x y) 3)) 1 2)) (f72)
    (define (f73) ((lambda (x y) (cf21 (cf20 x y) 3)) 1 2)) (f73)
    
    (define (f74) ((lambda (x) (cf11 (cf21 x 'y))) 1)) (f74)
    (define (f75) ((lambda (x) (cf11 (cf20 x 'y))) 1)) (f75)
    (define (f76) ((lambda (x) (cf21 x (cf11 2))) 1)) (f76)
    (define (f77) ((lambda (x) (cf22 x (cf10 2))) 1)) (f77)
    (define (f78) ((lambda (x) (cf21 1 (cf11 2))) 1)) (f78)
    (define (f79) ((lambda (x) (cf22 1 (cf10 2))) 1)) (f79)
    (define (f80) ((lambda (x) (cf21 (cf11 1) x)) 2)) (f80)
    (define (f81) ((lambda (x) (cf21 (cf10 1) x)) 2)) (f81)
    (define (f82) ((lambda (x y z) (cf21 (cf21 x y) z)) 1 2 3)) (f82)
    (define (f83) ((lambda (x y z) (cf21 (cf20 x y) z)) 1 2 3)) (f83)
    (define (f84) ((lambda (x y z) (cf21 (cf21 x 2) z)) 1 2 3)) (f84)
    (define (f85) ((lambda (x y z) (cf21 (cf20 x 2) z)) 1 2 3)) (f85)
    (define (f86) ((lambda (x y z) (cf21 (cf21 1 y) z)) 1 2 3)) (f86)
    (define (f87) ((lambda (x y z) (cf21 (cf20 1 y) z)) 1 2 3)) (f87)
    (define (f88) ((lambda (x y z) (cf21 (cf21 x 2) 3)) 1 2 3)) (f88)
    (define (f89) ((lambda (x y z) (cf21 (cf20 x 2) 3)) 1 2 3)) (f89)
    (define (f90) ((lambda (x) (cf21 (cf11 1) 2)) 2)) (f90)
    (define (f91) ((lambda (x) (cf21 (cf10 1) 2)) 2)) (f91)
    
    (define (f92) ((lambda (w x y z) (cf21 (cf21 w x) (cf21 y z))) 1 2 3 4)) (f92)
    (define (f93) ((lambda (w x y z) (cf22 (cf21 w x) (cf20 y z))) 1 2 3 4)) (f93)
    (define (f94) ((lambda (w x y z) (cf21 (cf20 w x) (cf21 y z))) 1 2 3 4)) (f94)
    (define (f95) ((lambda (w x y z) (cf21 (cf21 w 2) (cf21 y 4))) 1 2 3 4)) (f95)
    (define (f96) ((lambda (w x y z) (cf22 (cf21 w 2) (cf20 y 4))) 1 2 3 4)) (f96)
    (define (f97) ((lambda (w x y z) (cf21 (cf20 w 2) (cf21 y 4))) 1 2 3 4)) (f97)
    (define (f98) ((lambda (x y z) (cf21 (cf11 x) (cf21 y z))) 1 2 3)) (f98)
    (define (f99) ((lambda (x y z) (cf22 (cf11 x) (cf20 y z))) 1 2 3)) (f99)
    (define (f100) ((lambda (x y z) (cf21 (cf10 x) (cf21 y z))) 1 2 3)) (f100)
    (define (f101) ((lambda (x y z) (cf21 (cf21 x y) (cf11 z))) 1 2 3)) (f101)
    (define (f102) ((lambda (x y z) (cf22 (cf21 x y) (cf10 z))) 1 2 3)) (f102)
    (define (f103) ((lambda (x y z) (cf21 (cf20 x y) (cf11 z))) 1 2 3)) (f103)
    (define (f104) ((lambda (x y z) (cf21 (cf21 x y) (cf11 3))) 1 2 3)) (f104)
    (define (f105) ((lambda (x y z) (cf22 (cf21 x y) (cf10 3))) 1 2 3)) (f105)
    (define (f106) ((lambda (x y z) (cf21 (cf20 x y) (cf11 3))) 1 2 3)) (f106)
    (define (f107) ((lambda (x y z) (cf21 (cf11 1) (cf21 y z))) 1 2 3)) (f107)
    (define (f108) ((lambda (x y z) (cf22 (cf11 1) (cf20 y z))) 1 2 3)) (f108)
    (define (f109) ((lambda (x y z) (cf21 (cf10 1) (cf21 y z))) 1 2 3)) (f109)
    (define (f110) ((lambda () (cf21 (cf11 1) (cf11 2))))) (f110)
    (define (f111) ((lambda () (cf22 (cf11 1) (cf10 2))))) (f111)
    (define (f112) ((lambda () (cf21 (cf10 1) (cf11 2))))) (f112)
    
    (define (f113) ((lambda (x) (cf11 (cf11 (cf11 x)))) 1)) (f113)
    (define (f114) ((lambda (x) (cf11 (cf11 (cf10 x)))) 1)) (f114)
    (define (f115) ((lambda (x) (cf11 (cf10 (cf11 x)))) 1)) (f115)
    (define (f116) ((lambda (w x y z) (cf22 w (cf21 (cf21 x y) z))) 1 2 3 4)) (f116)
    (define (f117) ((lambda (w x y z) (cf22 w (cf20 (cf21 x y) z))) 1 2 3 4)) (f117)
    (define (f118) ((lambda (w x y z) (cf22 w (cf21 (cf20 x y) z))) 1 2 3 4)) (f118)
    
    (define (f119) ((lambda (w x y z) (cf22 w (cf22 x (cf21 y z)))) 1 2 3 4)) (f119)
    (define (f120) ((lambda (w x y z) (cf22 w (cf20 x (cf21 y z)))) 1 2 3 4)) (f120)
    (define (f121) ((lambda (w x y z) (cf22 w (cf22 x (cf20 y z)))) 1 2 3 4)) (f121)
    
    (define (f122) ((lambda (x y) (cf11 (cf32 1 (cf21 x y) 4))) 2 3)) (f122) ; c_z
    (define (f123) ((lambda (x y) (cf11 (cf32 1 (cf20 x y) 4))) 2 3)) (f123) 
    (define (f124) ((lambda (x y) (cf11 (cf30 1 (cf21 x y) 4))) 2 3)) (f124)
    
    (define (f125) ((lambda (x y) (cf21 (cf11 (cf21 x y)) 3)) 1 2)) (f125)
    (define (f126) ((lambda (x y) (cf21 (cf10 (cf21 x y)) 3)) 1 2)) (f126)
    (define (f127) ((lambda (x y) (cf21 (cf11 (cf20 x y)) 3)) 1 2)) (f127)
    
    (define (f128) ((lambda (x) (cf22 1 (cf21 x (cf11 3)))) 2)) (f128)
    (define (f129) ((lambda (x) (cf22 1 (cf20 x (cf11 3)))) 2)) (f129)
    (define (f130) ((lambda (x) (cf22 1 (cf22 x (cf10 3)))) 2)) (f130)
    
    (define (f131) ((lambda (x y z) (cf22 x (cf21 (cf21 y z) (cf21 z y)))) 1 2 3)) (f131)
    (define (f132) ((lambda (x y z) (cf22 x (cf20 (cf21 y z) (cf21 z y)))) 1 2 3)) (f132)
    (define (f133) ((lambda (x y z) (cf22 x (cf21 (cf20 y z) (cf21 z y)))) 1 2 3)) (f133)
    (define (f134) ((lambda (x y z) (cf22 x (cf22 (cf21 y z) (cf20 z y)))) 1 2 3)) (f134)
    
    (define (f135) ((lambda (x y z) (cf21 (cf21 x y) (cf11 (cf21 y z)))) 1 2 3)) (f135)
    (define (f136) ((lambda (x y z) (cf21 (cf20 x y) (cf11 (cf21 y z)))) 1 2 3)) (f136)
    (define (f137) ((lambda (x y z) (cf22 (cf21 x y) (cf10 (cf21 y z)))) 1 2 3)) (f137)
    (define (f138) ((lambda (x y z) (cf22 (cf21 x y) (cf11 (cf20 y z)))) 1 2 3)) (f138)
    
    (define (f139) ((lambda (x y z) (cf21 (cf21 (cf21 x y) z) (cf21 y z))) 1 2 3)) (f139)
    (define (f140) ((lambda (x y z) (cf21 (cf20 (cf21 x y) z) (cf21 y z))) 1 2 3)) (f140)
    (define (f141) ((lambda (x y z) (cf21 (cf21 (cf20 x y) z) (cf21 y z))) 1 2 3)) (f141)
    (define (f142) ((lambda (x y z) (cf22 (cf21 (cf21 x y) z) (cf20 y z))) 1 2 3)) (f142)
    
    (define (f143) ((lambda (x y z) (cf22 x (cf21 (cf11 (cf21 x y)) (cf11 (cf21 y z))))) 1 2 3)) (f143)
    (define (f144) ((lambda (x y z) (cf22 x (cf20 (cf11 (cf21 x y)) (cf11 (cf21 y z))))) 1 2 3)) (f144)
    (define (f145) ((lambda (x y z) (cf22 x (cf21 (cf10 (cf21 x y)) (cf11 (cf21 y z))))) 1 2 3)) (f145)
    (define (f146) ((lambda (x y z) (cf22 x (cf21 (cf11 (cf20 x y)) (cf11 (cf21 y z))))) 1 2 3)) (f146)
    (define (f147) ((lambda (x y z) (cf22 x (cf22 (cf11 (cf21 x y)) (cf10 (cf21 y z))))) 1 2 3)) (f147)
    (define (f148) ((lambda (x y z) (cf22 x (cf22 (cf11 (cf21 x y)) (cf11 (cf20 y z))))) 1 2 3)) (f148)
    
    (define (f149) ((lambda (x y z) (cf22 x (cf31 (cf11 y) (cf11 z) (cf11 z)))) 1 2 3)) (f149)
    (define (f150) ((lambda (x y z) (cf22 x (cf30 (cf11 y) (cf11 z) (cf11 z)))) 1 2 3)) (f150)
    (define (f151) ((lambda (x y z) (cf22 x (cf31 (cf10 y) (cf11 z) (cf11 z)))) 1 2 3)) (f151)
    (define (f152) ((lambda (x y z) (cf22 x (cf32 (cf11 y) (cf10 z) (cf11 z)))) 1 2 3)) (f152)
    (define (f153) ((lambda (x y z) (cf22 x (cf33 (cf11 y) (cf11 z) (cf10 z)))) 1 2 3)) (f153)
    
    (define (f154) ((lambda (x y z) (cf21 (cf32 1 (cf21 x y) 4) (cf32 1 (cf21 y z) 4))) 1 2 3)) (f154)
    (define (f155) ((lambda (x y z) (cf21 (cf32 1 (cf20 x y) 4) (cf32 1 (cf21 y z) 4))) 1 2 3)) (f155)
    (define (f156) ((lambda (x y z) (cf22 (cf32 1 (cf21 x y) 4) (cf30 1 (cf21 y z) 4))) 1 2 3)) (f156)
    
    (define (f157) ((lambda (x y z) (cf22 x (cf32 1 (cf21 y z) 4))) 1 2 3)) (f157)
    (define (f158) ((lambda (x y z) (cf22 x (cf32 1 (cf20 y z) 4))) 1 2 3)) (f158)
    
    (define (f159) ((lambda (x y z) (cf21 (cf32 1 (cf21 y z) 4) x)) 1 2 3)) (f159)
    (define (f160) ((lambda (x y z) (cf21 (cf30 1 (cf20 y z) 4) x)) 1 2 3)) (f160)
    
    (define (f161) ((lambda (x y z) (cf22 1 (cf32 1 (cf21 y z) 4))) 1 2 3)) (f161)
    (define (f162) ((lambda (x y z) (cf22 1 (cf32 1 (cf20 y z) 4))) 1 2 3)) (f162)
    
    (define (f163) ((lambda (x y z) (cf21 (cf32 1 (cf21 y z) 4) x)) 1 2 3)) (f163)
    (define (f164) ((lambda (x y z) (cf21 (cf30 1 (cf20 y z) 4) x)) 1 2 3)) (f164)
    
    ;; --------
    (test (f1) 3) (test (f2) 3) (test (f3) 3) (test (f4) 'x) (test (f5) 3) (test (f6) 3) (test (f7) 3)
    (test (f8) 3) (test (f9) 'x) (test (f10) 'y) (test (f11) 'y) (test (f12) 1) (test (f13) 'x) (test (f14) 1)
    (test (f15) 1) (test (f16) 1) (test (f17) 2) (test (f18) 1) (test (f19) 1) (test (f20) 'q) (test (f21) 1)
    (test (f22) 1) (test (f23) 1) (test (f24) 1) (test (f25) 'x) (test (f26) 'q) (test (f27) 1) (test (f28) 1)
    (test (f29) 0) (test (f30) 0) (test (f31) 0) (test (f32) 'x) (test (f33) 1) (test (f34) 3) (test (f35) 1)
    (test (f36) 1) (test (f37) 1) (test (f38) 1) (test (f39) 1) (test (f40) 1) (test (f41) 1) (test (f42) 1)
    (test (f43) 1) (test (f44) 1) (test (f45) 1) (test (f46) 1) (test (f47) 1) (test (f48) 1) (test (f49) 1)
    (test (f50) 1) (test (f51) 2) (test (f52) 1) (test (f53) 1) (test (f54) 1) (test (f55) 2) (test (f56) 1)
    (test (f57) 2) (test (f58) 1) (test (f59) 2) (test (f60) 1) (test (f61) 2) (test (f62) 1) (test (f63) 2) 
    (test (f64) 1) (test (f65) 2) (test (f66) 1) (test (f67) 2) (test (f68) 2) (test (f69) 2) (test (f70) 1)
    (test (f71) 2) (test (f72) 1) (test (f73) 1) (test (f74) 1) (test (f75) 1) (test (f76) 1) (test (f77) 2)
    (test (f78) 1) (test (f79) 2) (test (f80) 1) (test (f81) 1) (test (f82) 1) (test (f83) 1) (test (f84) 1) 
    (test (f85) 1) (test (f86) 1) (test (f87) 1) (test (f88) 1) (test (f89) 1) (test (f90) 1) (test (f91) 1)
    (test (f92) 1) (test (f93) 3) (test (f94) 1) (test (f95) 1) (test (f96) 3) (test (f97) 1) (test (f98) 1)
    (test (f99) 2) (test (f100) 1) (test (f101) 1) (test (f102) 3) (test (f103) 1) (test (f104) 1) (test (f105) 3) 
    (test (f106) 1) (test (f107) 1) (test (f108) 2) (test (f109) 1) (test (f110) 1) (test (f111) 2) (test (f112) 1) 
    (test (f113) 1) (test (f114) 1) (test (f115) 1) (test (f116) 2) (test (f117) 2) (test (f118) 2) 
    (test (f119) 3) (test (f120) 2) (test (f121) 3) (test (f122) 2) (test (f123) 2) (test (f124) 1) 
    (test (f125) 1) (test (f126) 1) (test (f127) 1) (test (f128) 2) (test (f129) 2) (test (f130) 3) 
    (test (f131) 2) (test (f132) 2) (test (f133) 2) (test (f134) 3) 
    (test (f135) 1) (test (f136) 1) (test (f137) 2) (test (f138) 2) 
    (test (f139) 1) (test (f140) 1) (test (f141) 1) (test (f142) 2) 
    (test (f143) 1) (test (f144) 1) (test (f145) 1) (test (f146) 1) (test (f147) 2) (test (f148) 2) 
    (test (f149) 2) (test (f150) 2) (test (f151) 2) (test (f152) 3) (test (f153) 3) 
    (test (f154) 1) (test (f155) 1) (test (f156) 1) (test (f157) 2) (test (f158) 2) (test (f159) 2) (test (f160) 1) 
    (test (f161) 2) (test (f162) 2) (test (f163) 2) (test (f164) 1) 
    ))
  
(when with-block
  (let ()
    (define (thunk1) 3)
    (define (thunk2) 4)
    (define (f1) ((lambda (x) (thunk1)) 0)) (f1) 
    (test (f1) 3)
    (define thunk1 thunk2)
    (test (f1) 4)
    
    (define (thunk3) (*s7* 'max-stack-size) 5)
    (define (thunk4) (*s7* 'max-stack-size) 6)
    (define (f2) ((lambda (x) (thunk3)) 0)) (f2) 
    (test (f2) 5)
    (define thunk3 thunk4)
    (test (f2) 6)
    
    (define (close1 x) (*s7* 'max-stack-size) (+ x 1))
    (define (close2 x) (*s7* 'max-stack-size) (+ x 2))
    (define (qclose1 x) (*s7* 'max-stack-size) (eq? x 'q))
    (define (qclose2 x) (*s7* 'max-stack-size) (eq? x 'r))
    (define (sclose1 x) (+ x 1))
    (define (sclose2 x) (+ x 2))
    (define (qsclose1 x) (eq? x 'q))
    (define (qsclose2 x) (eq? x 'r))
    (define* (s*close1 (x 1)) (+ x 1))
    (define* (s*close2 (x 1)) (+ x 2))
    (define* (u*close1 (x 1)) (*s7* 'max-stack-size) (+ x 1))
    (define* (u*close2 (x 1)) (*s7* 'max-stack-size) (+ x 2))
    
    (define (close3 x y) (*s7* 'max-stack-size) (+ x y))
    (define (close4 x y) (*s7* 'max-stack-size) (+ x y 1))
    (define (sclose3 x y) (+ x y))
    (define (sclose4 x y) (+ x y 1))
    (define* (s*close3 (x 1) (y 0.0)) (+ x y))
    (define* (s*close4 (x 1) (y 0.0)) (+ x y 1))
    (define* (u*close3 (x 1) (y 0)) (*s7* 'max-stack-size) (+ x y))
    (define* (u*close4 (x 1) (y 0)) (*s7* 'max-stack-size) (+ x y 1))
    
    (define (f3) (close1 1)) (f3)
    (define (f4) (sclose1 1)) (f4)
    (define (f5 x) (close1 x)) (f5 0)
    (define (f6 x) (sclose1 x)) (f6 0)
    (define (f7 x) (close1 ((lambda () (cs11 x))))) (f7 0)
    (define (f8 x) (sclose1 ((lambda () (cs11 x))))) (f8 0)
    (define (f9 x) (close1 ((lambda () (cs11 1))))) (f9 0)
    (define (f10 x) (sclose1 ((lambda () (cs11 1))))) (f10 0)
    (define (f11 x) (s*close1 x)) (f11 0)
    (define (f12 x) (s*close1)) (f12 0)
    (define (f13 x) (u*close1 x)) (f13 0)
    (define (f14 x) (u*close1)) (f14 0)
    (define (f15) (qclose1 'q)) (f15)
    (define (f16) (qsclose1 'q)) (f16)
    (define (f17 x) (close1 (cdr x))) (f17 '(0 . 0))
    (define (f18) (close3 1 2)) (f18)
    (define (f19) (sclose3 1 2)) (f19)
    (define (f20) (s*close3 1 2)) (f20)
    (define (f21) (u*close3 1 2)) (f21)
    (define (f22 x) (close3 x 2)) (f22 0)
    (define (f23 x) (sclose3 x 2)) (f23 0)
    (define (f24 x) (s*close3 x 2)) (f24 0)
    (define (f25 x) (u*close3 x 2)) (f25 0)
    (define (f26 x) (close3 1 x)) (f26 0)
    (define (f27 x) (sclose3 1 x)) (f27 0)
    (define (f28 x) (s*close3 1 x)) (f28 0)
    (define (f29 x) (u*close3 1 x)) (f29 0)
    (define (f30 x y) (close3 x y)) (f30 0 0)
    (define (f31 x y) (sclose3 x y)) (f31 0 0)
    (define (f32 x y) (s*close3 x y)) (f32 0 0)
    (define (f33 x y) (u*close3 x y)) (f33 0 0)
    
    (test (f3) 2) (test (f4) 2) (test (f5 1) 2) (test (f6 1) 2) (test (f7 1) 2) (test (f8 1) 2) (test (f9 1) 2) 
    (test (f10 1) 2) (test (f11 1) 2) (test (f12 1) 2) (test (f13 1) 2) (test (f14 1) 2) (test (f15) #t)
    (test (f16) #t) (test (f17 '(1 . 1)) 2) (test (f18) 3) (test (f19) 3) (test (f20) 3) (test (f21) 3)
    (test (f22 1) 3) (test (f23 1) 3) (test (f24 1) 3) (test (f25 1) 3) (test (f26 2) 3) (test (f27 2) 3)
    (test (f28 2) 3)
    (test (f29 2) 3)
    (test (f30 1 2) 3)
    (test (f31 1 2) 3)
    (test (f32 1 2) 3)
    (test (f33 1 2) 3)
    
    (define cs11 rs11)
    ;;;(test (f7 1) 3) (test (f8 1) 3) (test (f9 1) 3) (test (f10 1) 3)

    (define close1 close2)
    (define close3 close4)
    (define qclose1 qclose2)
    (define sclose1 sclose2)
    (define sclose3 sclose4)
    (define qsclose1 qsclose2)
    (define s*close1 s*close2)
    (define s*close3 s*close4)
    (define u*close1 u*close2)
    (define u*close3 u*close4)

    (define (f31 x y) (sclose3 x y))
    
    (test (f3) 3) (test (f4) 3) (test (f5 1) 3) (test (f6 1) 3) 
    ;;; (test (f7 -1) 2) (test (f8 -1) 2) (test (f9 -1) 4) (test (f10 -1) 4) 
    (test (f11 1) 3) (test (f12 1) 3) (test (f13 1) 3) (test (f14 1) 3) (test (f15) #f)
    (test (f16) #f) (test (f17 '(1 . 1)) 3) (test (f18) 4) (test (f19) 4) (test (f20) 4) (test (f21) 4)
    (test (f22 1) 4) (test (f23 1) 4) (test (f24 1) 4) (test (f25 1) 4) (test (f26 2) 4) (test (f27 2) 4)
    (test (f28 2) 4) (test (f29 2) 4) (test (f30 1 2) 4) (test (f31 1 2) 4) 
    (test (f32 1 2) 4) (test (f33 1 2) 4)
    ))

;;; global name opts (or lack thereof)
;;; the funny names used here must be nonce words

;;; op_unknown (thunk):
(define *x1* #f) (define (test*x1*) (*x1*)) (define (set*x1* n) (set! *x1* (lambda () n))) (set*x1* 1) (test (test*x1*) 1) (set*x1* 2) (test (test*x1*) 2)
(define *x2* #f) (define (test*x2*) (*x2*)) (define (set*x2* n) (set! *x2* (define* (_) n))) (set*x2* 1) (test (test*x2*) 1) (set*x2* 2) (test (test*x2*) 2)

;;; op_unknown_q:
(define *x3* #f) (define (test*x3*) (*x3* 'a)) (define (set*x3* n) (set! *x3* (lambda (x) n))) (set*x3* 1) (test (test*x3*) 1) (set*x3* 2) (test (test*x3*) 2)

;;; op_unknown_s:
(define *x4* #f) (define (test*x4* a) (*x4* a)) (define (set*x4* n) (set! *x4* (lambda (x) n))) (set*x4* 1) (test (test*x4* 0) 1) (set*x4* 2) (test (test*x4* 0) 2)
(define *x5* #f) (define (test*x5* a) (*x5* a)) (define (set*x5* n) (set! *x5* (define* (_ __) n))) (set*x5* 1) (test (test*x5* 0) 1) (set*x5* 2) (test (test*x5* 0) 2)

;;; op_unknown_c:
(define *x6* #f) (define (test*x6*) (*x6* 0)) (define (set*x6* n) (set! *x6* (lambda (x) n))) (set*x6* 1) (test (test*x6*) 1) (set*x6* 2) (test (test*x6*) 2)

;;; op_unknown_ss:
(define *x7* #f) (define (test*x7* a b) (*x7* a b)) (define (set*x7* n) (set! *x7* (lambda (x y) n))) 
    (set*x7* 1) (test (test*x7* 0 0) 1) (set*x7* 2) (test (test*x7* 0 0) 2)
(define *x8* #f) (define (test*x8* a b) (*x8* a b)) (define (set*x8* n) (set! *x8* (define* (_ x y) n))) 
    (set*x8* 1) (test (test*x8* 0 0) 1) (set*x8* 2) (test (test*x8* 0 0) 2)

;;; op_unknown_sc:
(define *x9* #f) (define (test*x9* a b) (*x9* a 0)) (define (set*x9* n) (set! *x9* (lambda (x y) n))) 
    (set*x9* 1) (test (test*x9* 0 0) 1) (set*x9* 2) (test (test*x9* 0 0) 2)
(define *x10* #f) (define (test*x10* a b) (*x10* a 0)) (define (set*x10* n) (set! *x10* (define* (_ x y) n))) 
    (set*x10* 1) (test (test*x10* 0 0) 1) (set*x10* 2) (test (test*x10* 0 0) 2)

;;; op_unknown_cs:
(define *x11* #f) (define (test*x11* a b) (*x11* 0 b)) (define (set*x11* n) (set! *x11* (lambda (x y) n))) 
    (set*x11* 1) (test (test*x11* 0 0) 1) (set*x11* 2) (test (test*x11* 0 0) 2)
(define *x12* #f) (define (test*x12* a b) (*x12* 0 b)) (define (set*x12* n) (set! *x12* (define* (_ x y) n))) 
    (set*x12* 1) (test (test*x12* 0 0) 1) (set*x12* 2) (test (test*x12* 0 0) 2)

;;; globals 
(test (let () (define (call-func func arg1 arg2) (define (call) (func arg1 arg2)) (call)) (call-func + 1 2.5) (call-func - 5 2)) 3)
(test (let () (define (call-func arg1 arg2) (let ((func (if (= arg1 1) + -))) (define (call) (func arg1 arg2)) (call))) (call-func 1 2.5) (call-func 5 2)) 3)

;;; safe/unsafe troubles
(let ()
  (define (testit)
    (define (f1 x y z) (car z))
    (define (f2 lst) (cond ((null? lst) ()) (else (f1 lst (f2 (cdr lst)) lst))))
    (test (f2 '(abs -1)) 'abs))
  (testit))
(let ()
  (define (testit)
    (define (f1 x y z) (car z))
    (define (f2 lst) (cond ((null? lst) ()) (else (f1 (f2 (cdr lst)) (f2 (cdr lst)) lst))))
    (test (f2 '(abs -1)) 'abs))
  (testit))

;;; changing a function's environment
(let ()
  (let ((e (let ((a 1) (b 2)) (curlet))))
    (define (f1 c)
      (+ a b c))
    (set! (outlet (funclet f1)) e)
  (test (f1 4) 7))

  (let ((e (let ((a 1) (b 2)) (curlet))))
    (define f2 (let ((d 10))
		 (lambda (c)
		   (+ a b c d))))
   ;; (set! (outlet (funclet f2)) e) ; 'd unbound
    (set! (outlet (outlet (funclet f2))) e)
    (test (f2 4) 17))

  (let ((e (let ((a 1) (b 2)) (curlet))))
    (define f3
      (with-let e
	(lambda (c)
	  (+ a b c))))
    (test (f3 4) 7)))




;;; --------------------------------------------------------------------------------
;;; values
;;; call-with-values
;;; --------------------------------------------------------------------------------


(test (call-with-values (lambda () (values 1 2 3)) +) 6)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) b))  5)
(test (call-with-values (lambda () (values 4 5)) (lambda (a b) (+ a b))) 9)
(test (call-with-values * -) -1) ; yeah, right... (- (*))
(test (values 1) 1)
(test (call-with-values (lambda () (values 1 2 3 4)) list) (list 1 2 3 4))
(test (+ (values 1) (values 2)) 3)
(test (+ (values '1) (values '2)) 3)
(test (if (values #t) 1 2) 1)
(test (if (values '#t) 1 2) 1)
(test (if (values #f) 1 2) 2)
(test (if (values #f #f) 1 2) 1)
(test (equal? (values #t #t)) #t)
(test (call-with-values (lambda () 4) (lambda (x) x)) 4)
(test (let () (values 1 2 3) 4) 4)
(test (apply + (values ())) 0)
(test (+ (values 1 2 3)) 6)
(test (let ((f (lambda () (values 1 2 3)))) (+ (f))) 6)
(num-test (log (values 8 2)) 3)
(test (* (values 2 (values 3 4))) 24)
(test (* (values (+ (values 1 2)) (- (values 3 4)))) -3)
(test (list (values 1 2) (values 3) 4) '(1 2 3 4))
(test (let ((f1 (lambda (x) (values x (+ x 1)))) (f2 (lambda () (values 2)))) (+ (f1 3) (* 2 (f2)))) 11)
(test (+ (let () (values 1 2)) 3) 6)
(test (let () (values 1 2) 4) 4)
(test (let () + (values 1 2) 4) 4)
(test (string-ref (values "hiho" 2)) #\h)
(test (vector-ref (values (vector 1 2 3)) 1) 2)
(test (+ (values (+ 1 (values 2 3)) 4) 5 (values 6) (values 7 8 (+ (values 9 10) 11))) 66)
(test (+ (if (values) (values 1 2) (values 3 4)) (if (null? (values)) (values 5 6) (values 7 8))) 18) ; (values) is now #<unspecified> (sort of)
(test (+ (cond (#f (values 1 2)) (#t (values 3 4))) 5) 12)
(test (+ (cond (#t (values 1 2)) (#f (values 3 4))) 5) 8)
(test (apply + (list (values 1 2))) 3)
(test (apply + (list ((lambda (n) (values n (+ n 1))) 1))) 3)
(test (+ (do ((i 0 (+ i 1))) ((= i 3) (values i (+ i 1))))) 7)
(test (+ (with-input-from-string "(values 1 2 3)" (lambda () (eval (read)))) 2) 8)
(test (< (values 1 2 3)) #t)
(test (apply (values + 1 2) '(3)) 6)
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 1 2 3))) 7)
(test (+ 1 (eval-string "(values 2 3 4)")) 10)
(test (+ 1 (eval '(values 2 3 4))) 10)
(test (or (values #t) #f) #t)
(test (and (values #t) #f) #f)
(test (let ((x 1)) (set! x (values 32)) x) 32)
(test (let ((x #(32 33))) ((values x) 0)) 32)
(test (let ((x #(32 33))) (set! ((values x) 0) 123) x) #(123 33))
(test (list-ref '(1 (2 3)) (values 1 1)) 3)
(test (list-ref (values '(1 (2 3)) 1 1)) 3)
(test (list-ref ((lambda () (values '(1 (2 3)) 1 1)))) 3)
(test (set! (values) 1) 'error)
(test (+ (values (begin (values 1 2)) (let ((x 1)) (values x (+ x 1))))) 6)
(test (vector 1 (values 2 3) 4) #(1 2 3 4))
(test (vector (values 1 (values 2 3) (values (values 4)))) #(1 2 3 4))
(test(+ 1 (values (values (values 2) 3) (values (values (values 4)) 5) 6) 7) 28)
(test (map (values values #(1 2))) '(1 2))
(test ((values values) (values 0)) 0)
(test (((values values values) 0)) 0)
(test ((apply (values values values '((1 2))))) '(1 2))
(test (apply begin (values (list 1))) 1)
(test (apply begin (values '(values "hi"))) (apply (values begin '(values "hi"))))
(test ((object->string values) (abs 1)) #\a)
(test (list? (values 1 2 3)) 'error)
(test (list? (values 1)) #f)
(test (list? (values (list 1 2 3))) #t)

(test (let ((x 1)) (set! x (values)) x) #<unspecified>)
(test ((lambda () (let ((x 1)) (set! x (boolean? (values)))))) #f)
(test (let ((x 1)) (set! x (values 1 2 3)) x) 'error)
(test (let ((x 1)) (set! x (values 2)) x) 2)
(test (let ((x 1)) (set! (values x) 2) x) 'error) ; (no generalized set for values, so (values x) is not the same as x
(test (let ((x #(0 1))) (set! (values x 0 32)) x) 'error)
(test (let ((var (values 1 2 3))) var) 'error)
(test (let* ((var (values 1 2 3))) var) 'error)
(test (letrec ((var (values 1 2 3))) var) 'error)
(test (let ((x ((lambda () (values 1 2))))) x) 'error)
(test (+ 1 ((lambda () ((lambda () (values 2 3)))))) 6)
(test (let () (define (hi) (symbol? (values 1 2 3))) (hi)) 'error)
(test (let () (define (hi) (symbol? (values))) (hi)) #f) ; this is consistent with earlier such cases: (boolean? (values))
(test (let () (define (hi) (symbol? (values 'a))) (hi)) #t)
(test (let () (define (hi) (symbol? (values 1))) (hi)) #f)
(test (let () (define (hi a) (log (values 1 2) a)) (hi 2)) 'error)
(test (let () (define (arg2 a) (let ((b 1)) (set! b (+ a b)) (values b a))) (define (hi c) (expt (abs c) (arg2 2))) (hi 2)) 'error)
(test (let () (define (arg2 a) (let ((b 1)) (set! b (+ a b)) (values b))) (define (hi c) (expt (abs c) (arg2 2))) (hi 2)) 8)
(test (let () (define (hi) (+ (values 1 2) (values 3 4))) (hi)) 10)

(test (let ((str "hi")) (string-set! (values str 0 #\x)) str) "xi")
(test (values if) if)
(test (values quote) quote)

(test ((values '(1 (2 3)) 1 1)) 3)
(test (let ((x #(32 33))) ((values x 0))) 32)
(test (+ 1 (apply values '(2 3 4))) 10)
(test (eq? (values) (apply values ())) #t)
(test (+ 1 ((lambda args (apply values args)) 2 3 4)) 10)
(test (apply begin '(1 2 3)) 3)
(test (let ((x 1)) ((values set!) x 32) x) 32)
(let ((x 0)) (test (list (set! x 10)) (call-with-values (lambda () (set! x 10)) list))) ; from r7rs discussion

(let ()
  (define (curry function . args) 
    (lambda more-args 
      (function (apply values args) (apply values more-args)))) ; unfortunately this doesn't handle 0 args
  (test ((curry + 1 2) 3 4) 10))

(let ()
  (define (curry function . args)
    (if (null? args)
	function
	(lambda more-args
	  (if (null? more-args)
	      (apply function args)
	      (function (apply values args) (apply values more-args))))))
  (test ((curry + 1 2) 3 4) 10)
  (test ((curry + 2) 3 4) 9)
  (test ((curry +) 3 4) 7)
  (test ((curry +)) 0)
  (test ((curry + 1 2)) 3)
  (test ((curry + 1)) 1)
  (test ((curry +) 1) 1))

(test (or (values #t #f) #f) #t)
(test (or (values #f #f) #f) #f)
(test (or (values #f #t) #f) #t)
(test (or (values #f #f) #t) #t)
(test (or (values 1 2) #f) 1)
(test (+ 1 (or (values 2 3) 4)) 3)
(test (+ 1 (and 2 (values 3 4)) 5) 13)
(test (and (values) 1) 1)
(test (and (values 1 2 #f) 4) #f)
(test (and (values 1 2 3) 4) 4)
(test (length (values ())) 0)
(test (length (values #(1 2 3 4))) 4)
(test (vector? (values #())) #t)
(test (map + (values '(1 2 3) #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3)) (values #(1 2 3))) '(2 4 6))
(test (map + (values '(1 2 3) #(4 5 6)) (values '(7 8 9))) '(12 15 18))

(test (let ((x 1)) 
	(and (let () (set! x 2) #f) 
	     (let () (set! x 3) #f)) 
	x) 2)
(test (let ((x 1)) 
	(and (values (let () (set! x 2) #f) 
		     (let () (set! x 3) #f)))
	x) 3)

(test (+ (values 1 2) 3) 6)
(test (+ (values 1 (values 2))) 3)
(test (list (values 1 2)) '(1 2))
(test (+ 6 (values 1 (values 2 3) 4 ) 5) 21)
(test (+ ((lambda (x) (values (+ 1 x))) 2) 3) 6)
(test (list ((lambda (x) (values (+ 1 x))) 2)) '(3))
(test (+ (begin (values 1 2))) 3)
(test (+ 1 (let () (values 1 2))) 4)
(test (apply (values + 1 2) (list 3)) 6)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :a 3)) '(3 2))
(test (+ (values (values 1 2) (values 4 5))) 12)
(test (+ (begin 3 (values 1 2) 4)) 4)
(test (map (lambda (x) (if #f x (values))) (list 1 2)) ())
(test (map (lambda (x) (if #f x (begin (values)))) (list 1 2)) ())
(test (map (lambda (x) (if (odd? x) (values x (* x 20)) (values))) (list 1 2 3 4)) '(1 20 3 60))
(test (map (lambda (x) (if (odd? x) (values x (* x 20)) (if #f #f))) (list 1 2 3 4)) '(1 20 #<unspecified> 3 60 #<unspecified>))
(test (map (lambda (x) (if (odd? x) (apply values '(1 2 3)) (values))) (list 1 2 3 4)) '(1 2 3 1 2 3))
(test (object->string (map (lambda (x) (if (odd? x) (values x (* x 20)) (values))) (list 1 2 3 4))) "(1 20 3 60)") ; make sure no "values" floats through
(test (map (lambda (x) (if (odd? x) (values x (* x 20) (cons x (+ x 1))) (values))) (list 1 2 3 4 5 6)) '(1 20 (1 . 2) 3 60 (3 . 4) 5 100 (5 . 6)))
(test (* 2 (case 1 ((2) (values 3 4)) ((1) (values 5 6)))) 60)
(test (* 2 (case 1 ((2) (values 3 4)) (else (values 5 6)))) 60)
(test (* 2 (case 1 ((1) (values 3 4)) (else (values 5 6)))) 24)
(test (+ (values (* 3 2) (abs (values -1)))) 7)
(test (+ (let ((x 1)) (values x (+ x 1))) (if #f #f (values 2 3))) 8)

(test (let ((sum 0)) (for-each (lambda (n m p) (set! sum (+ sum n m p))) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) sum) 45)
(test (map (lambda (n m p) (+ n m p)) (values (list 1 2 3) (list 4 5 6) (list 7 8 9))) '(12 15 18))
(test (string-append (values "123" "4" "5") "6" (values "78" "90")) "1234567890")
(test (+ (dynamic-wind (lambda () #f) (lambda () (values 1 2 3)) (lambda () #f)) 4) 10)

(let ((x 'y)
      (y 32))
 (define (f1) (values #f))
 (test (if (values #f) x)               #<unspecified>)
 (test (if (values #t) x)               'y)
 (test (if (values #f) (values x))      #<unspecified>)
 (test (if (values #t) (values x))      'y)
 (test (if x (values x))                'y)
 (test (list (values x))                '(y))
 (test (list (values x y))              '(y 32))
 (test (cond ((values x) => list))      '(y))
 (test (cond ((values x y) => list))    '(y 32))
 (test (cond ((values #f) => list))     #<unspecified>)
 (test (cond ((values #t) => list))     '(#t))
 (test (cond ((f1) => list))            #<unspecified>)
 (test (list (cond ((values x))))       '(y))
 (test (list (cond ((values x y))))     '( y 32))
 (test (list (cond (#t (values x))))    '(y))
 (test (list (cond (#t (values x y))))  '(y 32))
 (test (cond ((values x) => (lambda args (apply list args))))    '(y))
 (test (cond ((values x y) => (lambda args (apply list args))))  '(y 32)))
 
(for-each
 (lambda (arg)
   (test (values arg) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(for-each
 (lambda (arg)
   (test (apply values arg) 'error)
   (test (apply values (list arg)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t '(1 . 2)))

(for-each
 (lambda (arg)
   (test (call-with-values (lambda () (values arg arg)) (lambda (a b) b)) arg))
 (list "hi" -1 #\a 1 'a-symbol #(1 2 3) 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))

(test (call-with-values (lambda () (values "hi" 1 3/2 'a)) (lambda (a b c d) (+ b c))) 5/2)
					;(test (call-with-values values (lambda arg arg)) ())
(test (string-ref (values "hi") 1) #\i)
(test ((lambda (a b) (+ a b)) ((lambda () (values 1 2)))) 3)

(test (list (letrec ((split (lambda (ls)
			      (if (or (null? ls) (null? (cdr ls)))
				  (values ls ())
				  (call-with-values
				      (lambda () (split (cddr ls)))
				    (lambda (odds evens)
				      (values (cons (car ls) odds)
					      (cons (cadr ls) evens))))))))
	      (split '(a b c d e f))))
      '((a c e) (b d f)))

(let ()
  (define (f1 . args)
    (apply values (+ (car args) (cadr args)) (cddr args)))
  (test (* (f1 2 3 4)) 20)
  (test (* (f1 2 3 4) (f1 1 2 3)) 180)
  (test (- (f1 2 3 4) (f1 1 2 3)) -5))

(test (call-with-values (lambda () (call/cc (lambda (k) (k 2 3)))) (lambda (x y) (list x y))) '(2 3))
(test (+ (call/cc (lambda (return) (return (values 1 2 3)))) 4) 10)

(test (let ((values 3)) (+ 2 values)) 5)
(test (let ((a (values 1))) a) 1)

(test (call-with-values (lambda () 2) (lambda (x) x)) 2)
(test (call-with-values (lambda () -1) abs) 1)
(test (call-with-values (lambda () (values -1)) abs) 1)
(test (call-with-values (lambda () (values -1)) (lambda (a) (abs a))) 1)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (- a b)))
      -18)

(test (call-with-values 
	  (lambda ()
	    (values
	     (call-with-values (lambda () (values 1 2 3)) +)
	     (call-with-values (lambda () (values 1 2 3 4)) *)))
	(lambda (a b)
	  (+ (* a (call-with-values (lambda () (values 1 2 3)) +))
	     (* b (call-with-values (lambda () (values 1 2 3 4)) *)))))
      612)

(test (call-with-values (lambda (x) (+ x 1)) (lambda (y) y)) 'error)
(test (+ (values . 1)) 'error)
(for-each
 (lambda (arg)
   (test (call-with-values arg arg) 'error))
 (list "hi" -1 #\a 1 'a-symbol 3.14 3/4 1.0+1.0i #t (list 1 2 3) '(1 . 2)))
(test (call-with-values (lambda () (values -1 2)) abs) 'error)

(test (multiple-value-bind (a b) (values 1 2) (+ a b)) 3)
(test (multiple-value-bind (a) 1 a) 1)
(test (multiple-value-bind (a . rest) (values 1 2 3) (+ a (apply + rest))) 6)
(test (multiple-value-bind a (values 1 2 3) a) '(1 2 3))
(test (multiple-value-bind (x y z) (values 1 2 3) (list z y x)) '(3 2 1))
(test (multiple-value-bind (x y z) (values 1 2 3) (let ((x 4)) (list x y z))) '(4 2 3))
(test (multiple-value-bind (x y z) (values 1 2 3 4 5 6) (list x y z)) 'error)                                    ; was '(1 2 3)) -- 25-Jan-16
(test (multiple-value-bind (x y z) (values 1 2) (list x y z)) 'error)                                            ; was '(1 2 #f))
(test (multiple-value-bind (x y z) (multiple-value-bind () (values 1 2) (values 'a 'b 'c)) (list x y z)) 'error) ;was '(a b c))

(test (let ((add (lambda (a b) (values (+ a 1) (+ b 1))))) (+ 1 (add 2 3))) 8)
(test (min (values 1 2) (values 3 0)) 0)
(test ((lambda* ((a 1) (b 2)) (list a b)) (values :b 231)) '(1 231))
(test (cons (values 1 2) (values 3 4)) 'error)

(test (cond ((values) 3) (#t 4)) 3)          ; an error in Guile "zero values returned"
(test (cond ((values (values)) 3) (#t 4)) 3) ; same
(test (+ (cond (#t (values 1 2)))) 3)        ; 1 in guile
(test (+ (cond ((values 3 4) => (lambda (a) a)))) 'error)
(test (+ (cond ((values 3 4) => (lambda (a b) (values a b))))) 7)
(test (+ 1 (cond ((values 2 3))) 4) 10)
(test (+ 1 (values)) 'error)

(test (case (values 1) ((1) 2) (else 3)) 2)
(test (case (values 1 2) ((1) 2) (else 3)) 3)
(test (case (values 1) (((values 1)) 2) (else 3)) 3)
(test (case (values 1 2) (((values 1 2)) 2) (else 3)) 3)

(test ((values) 0) 'error)
(test ((values "hi") 1) #\i)
(test (string-ref (values "hi") 0) #\h)
(test (string-ref (values "hi" "ho") 0) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (let ((str "hi")) (string-set! (values str) 0 #\x) str) "xi")
(test (let ((str "hi")) (set! (values str 0) #\x) str) 'error)
(test (let ((str "hi")) (string-set! (values str 0) #\x) str) "xi")

(test ((values 1 2 3) 0) 'error)
(test ((values "hi" "ho") 1) 'error)
(test ((values + 1 2 3)) 6)
(test ((values + 1 2) 3) 6)
(test ((values +) 1 2 3) 6)
(test ((values "hi" 0)) #\h)
(test ((values + 1) (values 2 3) 4) 10)
(test ((values - 10)) -10)
(test ((values - -10) 0) -10) ; looks odd but it's (- -10 0) that is (- a) != (- a 0)
(test ((values - 2 3) 0) -1)
(test ((values - 2 3) 1) -2)
(test ((values - 2 3) 2) -3)  ; it's actually (- 2 3 2) -> -3

(test (let ((str "hi")) (set! ((values str 0) 0) #\x) str) 'error)
(test (let ((str "hi")) (set! ((values str) 0) #\x) str) "xi")
(test (+ (let ((x 0)) (do ((i (values 0) (+ i 1))) (((values = i 10)) (values x 2 3)) (set! x (+ x i)))) 4) 54)

(test (map values (list (values 1 2) (values 3 4))) '(1 2 3 4))
(test (let () (define-macro (hi a) `(+ 1 ,a)) (hi (values 2 3 4))) 10)
(test (let () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let* () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let () (+ 4 (let* () (values 1 2 3)) 5)) 15)
(test (letrec () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let () (+ 4 (letrec () (values 1 2 3)) 5)) 15)
(test (letrec* () (+ 4 (let () (values 1 2 3)) 5)) 15)
(test (let* () (+ 4 (letrec* () (values 1 2 3)) 5)) 15)

(test (cons (values 1 2)) '(1 . 2))
(test (number->string (values 1 2)) "1")
(test (object->string (values)) "#<unspecified>")
(test (equal? (values) #<unspecified>) #t)
(test (equal? (begin) (begin (values))) #f) ; () but #<unspecified>
(test (map (lambda (x) (if #f x #<unspecified>)) (list 1 2)) '(#<unspecified> #<unspecified>))
(test (equal? (values) (if #f #f)) #t)
(test (substring (values "hi") (values 1 2)) "i")
(test (cond (call-with-exit (values "hi"))) "hi")
(test (values (begin (values "hi"))) "hi")
(test (< (values (values 1 2))) #t)

(test (let ((lst (list 0)))
	(set-cdr! lst lst)
	(format (values #f "~A" lst)))
      "#1=(0 . #1#)")

(let ()
  (define (mv n)
    (define (mv-1 a)
      (values a (+ a 1)))
    (define (mv-2 b)
      (values b (* b 2)))
    (values n (mv-1 n) (mv-2 n)))
  (test (lis