/***************************************************************************/
/* 		This code is part of WWW graber called pavuk		   */
/*		Copyright (c) 1997,1998,1999 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include "config.h"

#include <sys/socket.h>
#include <time.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "remind.h"

#include "bufio.h"
#include "url.h"
#include "mime.h"
#include "times.h"
#include "http.h"
#include "dllist.h"
#include "tools.h"
#include "recurse.h"

static dllist *remind_urls = NULL;

static int remind_check_if_newer(docp, modtime , nmdtm)
doc *docp;
time_t modtime;
time_t *nmdtm;
{
	int mres;
	int totallen;
	char *mime_hdr;
	char *p;
	int retv = 0;
	http_response *rsp;
	char *ustr;

	ustr = url_to_urlstr(docp->doc_url , FALSE);
	
	xprintf(1 , gettext("Checking: %s\n") , ustr);

	if (!(docp->doc_url->type == URLT_HTTP
#ifdef USE_SSL
		|| docp->doc_url->type == URLT_HTTPS
#endif
		))
	{
		xprintf(1 , "This URL type is not supported in reminder mode\n");
		return -1;
	}

	docp->datasock = http_get_data_socket(docp, 0, modtime, TRUE);

	if (!docp->datasock)
		return -1;

	mres = http_read_mime_header(docp, &mime_hdr , &totallen);

	if (mres <= 0)
	{
		xprintf(1 , "Bad response on HEAD request\n");
		return -1;
	}
	DEBUG_PROTOS(gettext("*********** HTTP Server response MIME header **********\n"));
	DEBUG_PROTOS("%s" , mime_hdr);
	DEBUG_PROTOS("*******************************************************\n");

	rsp = http_get_response_info(mime_hdr);

	if (rsp->ver_maj == 1 && rsp->ver_min == 1)
	{
		docp->is_http11 = TRUE;
		docp->is_persistent = TRUE;
	}

	p = get_mime_param_val_str("Connection:" , mime_hdr);
	if (p)
	{
		if (!strcasecmp(p, "close"))
			docp->is_persistent = FALSE;
	}

	if (!docp->is_persistent)
	{
		shutdown(bufio_getfd(docp->datasock) , 2);
		bufio_close(docp->datasock);
		docp->datasock = NULL;
	}

	if (rsp->ret_code == 304)
	{
		retv = 0;
	}
	else if ((p = get_mime_param_val_str("Last-Modified:" , mime_hdr)))
	{
		*nmdtm = scntime(p);

		if (modtime && (difftime(*nmdtm ,modtime) > 0))
		{
			retv = 1;
		}
	}
	else
	{
		retv = -1;
	}

	_free(rsp->text);
	_free(rsp);
	_free(mime_hdr);
	return retv;
}

static void remind_check_entry(docp, e)
doc *docp;
remind_entry *e;
{
	time_t t = 0L;
	int rv;

	rv = remind_check_if_newer(docp, e->mdtm , &t);

	if (!(cfg.stop || cfg.rbreak))
	{
		if (t) e->mdtm = t;

		if (rv == 1)
			e->status |= REMIND_MODIFIED;
		else if (rv == -1)
			e->status |= REMIND_ERROR;
	}

}

void remind_load_db()
{
	bufio *fd;
	char buf[PATH_MAX];

	sprintf(buf , "%s/.pavuk_remind_db" , cfg.path_to_home);

	while(remind_urls)
	{
		free_deep_url(((remind_entry *)remind_urls->data)->urlp);
		_free(remind_urls->data);
		remind_urls = dllist_remove_entry(remind_urls , remind_urls);
	}

	fd = bufio_copen(buf , O_BINARY | O_RDWR | O_CREAT , 0644);

	if (!fd)
	{
		xperror(buf);
		return;
	}

	_flock(bufio_getfd(fd) , buf , O_BINARY | O_RDWR | O_CREAT , TRUE);

	while(bufio_readln(fd , buf , sizeof(buf)) > 0)
	{
		time_t t;
		url *purl;
		char *eptr;
		remind_entry *entry;

		strip_nl(buf);
		t = strtol(buf , &eptr , 10);
		if (!t && errno == ERANGE)
		{
			xprintf(1 , gettext("Bad reminder db entry - %s\n") , buf);
			continue;
		}

		while(*eptr && isspace(*eptr)) eptr++;
		purl = url_parse(eptr);

		entry = _malloc(sizeof(remind_entry));
		entry->urlp = purl;
		entry->mdtm = t;
		entry->status = 0;
		remind_urls = dllist_append(remind_urls , entry);
	}

	_funlock(bufio_getfd(fd));
	bufio_close(fd);
}

void remind_save_db()
{
	bufio *fd;
	char buf[PATH_MAX];
	dllist *ptr;

	sprintf(buf , "%s/.pavuk_remind_db" , cfg.path_to_home);

	fd = bufio_copen(buf , O_BINARY | O_RDWR | O_CREAT , 0644);

	if (!fd)
	{
		xperror(buf);
		return;
	}

	_flock(bufio_getfd(fd) , buf , O_BINARY | O_RDWR | O_CREAT , TRUE);

	ftruncate(bufio_getfd(fd) , 0);

	for (ptr = remind_urls ; ptr ; ptr = ptr->next)
	{
		char *p;

		p = url_to_urlstr(((remind_entry*)ptr->data)->urlp , FALSE);
		sprintf(buf , "%ld %s\n" , ((remind_entry *)ptr->data)->mdtm , p);
		_free(p);

		bufio_write(fd , buf , strlen(buf));
	}

	_funlock(bufio_getfd(fd));
	bufio_close(fd);
}

static remind_entry *remind_find_entry(urlp)
url *urlp;
{
	dllist *ptr = remind_urls;
	remind_entry *rv = NULL;
	char *ustr;

	ustr = url_to_urlstr(urlp , FALSE);

	while(ptr && !rv)
	{
		remind_entry *e = ptr->data;
		char *p;

		p = url_to_urlstr(e->urlp , FALSE);

		if (!strcmp(ustr , p))
			rv = e;
		_free(p);
		ptr = ptr->next;
	}
	_free(ustr);
	return rv;
}

void remind_start_add()
{
	dllist *ptr = cfg.request;

	while (ptr)
	{
		url_info *ui = (url_info *)ptr->data;
		url *urlp = url_parse(ui->urlstr);

		if (!remind_find_entry(urlp))
		{
			remind_entry *e = _malloc(sizeof(remind_entry));

			e->urlp = urlp;
			e->status = 0;
			e->mdtm = 0L;

			remind_urls = dllist_append(remind_urls , e);
		}
		else
		{
			free_deep_url(urlp);
			_free(urlp);
		}

		ptr = ptr->next;
	}
}

void remind_do()
{
	dllist *ptr;
	remind_entry *e;
	doc docu;
	global_connection_info con_info;

	init_global_connection_data(&con_info);

	for(ptr = remind_urls; ptr; ptr = ptr->next)
	{
		e = ptr->data;

		if (e->status & REMIND_PROCESSED)
			continue;

		doc_init(&docu, e->urlp);
		restore_global_connection_data(&con_info, &docu);

		remind_check_entry(&docu, e);

		save_global_connection_data(&con_info, &docu);

		if (cfg.stop || cfg.rbreak)
			break;
		else 
			e->status |= REMIND_PROCESSED;
	}

	kill_global_connection_data(&con_info);
}

void remind_send_result()
{
	FILE *fp;
	char buf[PATH_MAX];
	char *p,*dp;
	int cnt;
	dllist *ptr;

	if (cfg.remind_cmd)
	{
		for (dp = buf , p = cfg.remind_cmd ; *p ; p++)
		{
			if (*p == '%')
			{
				p++;
				switch (*(p))
				{
					case '%':
						*dp = *p;
						dp++;
					break;
					case 'e':
						strcpy(dp , cfg.from);
						while(*dp)dp++;
					break;
					case 'd':
					{
						time_t t = time(NULL);
						char pom[100];

						LOCK_TIME
						strftime(pom , sizeof(pom) , "%a %H:%M %d.%m.%Y" , localtime(&t));
						UNLOCK_TIME
						strcpy(dp , pom);
						while(*dp)dp++;
					}
					break;
					default: 
						*dp = *p;
						dp++;
						p--;
				}
			}
			else
			{
				*dp = *p;
				dp++;
			}
		}
		*dp = '\0';
	}
	else
		strcpy(buf , "cat");
/*
		sprintf(buf , "mailx %s -s \"pavuk reminder result\"\n" , cfg.from);
*/


	fp = popen(buf , "w");

	if (!fp)
	{
		xperror("popen");
	}

	fprintf(fp , gettext("This is the result of running pavuk reminder mode\n\n"));

	cnt = 0;
	for (ptr = remind_urls ; ptr ; ptr = ptr->next)
	{
		remind_entry *e = ptr->data;

		if (e->status & REMIND_MODIFIED)
		{
			if (!cnt)
			{
				fprintf(fp , gettext("Changed URLs\n"));
				fprintf(fp , "-------------------------\n");
				cnt ++;
			}
			p = url_to_urlstr(e->urlp , FALSE);
			fprintf(fp , "%s %s" , p , ctime(&e->mdtm));
			_free(p);
		}
	}

	fprintf(fp , "\n\n");

	cnt = 0;
	for (ptr = remind_urls ; ptr ; ptr = ptr->next)
	{
		remind_entry *e = ptr->data;

		if (e->status & REMIND_ERROR)
		{
			if (!cnt)
			{
				fprintf(fp , gettext("URLs with some errors\n"));
				fprintf(fp , "-------------------------\n");
				cnt ++;
			}
			p = url_to_urlstr(e->urlp , FALSE);
			fprintf(fp , "%s\n" , p);
			_free(p);
		}
	}


	pclose (fp);
}

