/*
 * Call Table for OpenGate
 *
 * Copyright (c) Egoboo Ltd. 2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: CallTabl.cxx,v $
 * Revision 1.7  2001/02/18 16:37:36  aunitt
 * Fixed incorrect logic in selecting maximum bandwidth if nothing is specified
 * in the configuration.
 *
 * Revision 1.6  2000/05/12 14:10:17  aunitt
 * Now only does bogus includes when necessary.
 *
 * Revision 1.5  2000/05/12 10:23:40  aunitt
 * Changed to use PWaitAndSignal where appropriate.
 *
 * Revision 1.4  2000/05/05 11:38:50  aunitt
 * Added bandwidth management.
 *
 * Revision 1.3  2000/04/21 13:47:09  aunitt
 * Removed bogus warnings in Microsoft STL headers.
 * Changed insert to do back insert not front insert.
 *
 * Revision 1.2  2000/04/12 13:35:44  aunitt
 * Added more debugging.
 *
 * Revision 1.1  2000/04/10 19:21:42  aunitt
 * Call table for OpenGatekeeper.
 *
 *
 */

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/svcproc.h>
#if (_MSC_VER >= 1200)
#include <q931.h>
#include <h245.h>  
#pragma warning( push, 3 )
#endif
#include <algorithm>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif
#include "CallTabl.h"
#include "Environ.h"

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // remove performance warning about bool...
#endif 

CallDetails::CallDetails( const H225_CallReferenceValue &    CRV,
                          const H225_EndpointIdentifier &    AkaCaller,
                          const H225_EndpointIdentifier &    AkaCalled,
                          unsigned                           AkaBandwidth
                        ) : Ref(CRV), Caller(AkaCaller), Called(AkaCalled),
                            Bandwidth(AkaBandwidth), HasId(false)
{
}

CallDetails::CallDetails( const H225_CallReferenceValue &    CRV,
                          const H225_CallIdentifier &        AkaId,
                          const H225_EndpointIdentifier &    AkaCaller,
                          const H225_EndpointIdentifier &    AkaCalled,
                          unsigned                           AkaBandwidth
                        ) : Ref(CRV), Caller(AkaCaller), Called(AkaCalled),
                            Bandwidth(AkaBandwidth), HasId(true), Id(AkaId)
{
}

CallDetails::~CallDetails()
{
}

struct hasCallRef
{
	const H225_CallReferenceValue & CRV;
	hasCallRef(const H225_CallReferenceValue &Ref) : CRV(Ref) {}
	bool operator()( const CallDetails & Call ) const
	{
		return (Call.GetCRV() == CRV);
	}
};

struct hasCallId
{
	const H225_CallIdentifier & Id;
	hasCallId(const H225_CallIdentifier &AkaId) : Id(AkaId) {}
	bool operator()( const CallDetails & Call ) const
	{
	    if ( Call.HasIdentifier() )
		    return (Call.GetId() == Id);
		else
		    return false;
	}
};

CallTable::CallTable( const Environ & Env ) : MyEnviron(Env)
{
    TotalBandwidth = 0;
}

CallTable::~CallTable()
{
}

void CallTable::Insert( CallDetails & Call )
// Task: insert the given call into the table
{
    // No consistency checks at the moment, e.g. we could get two calls with the same
    // CRV
//  	PSYSTEMLOG( Info, "Inserting call " << Call.GetCRV() << " into table" );
//  	PSYSTEMLOG( Info, "Bandwidth " << Call.GetBandwidth() );

    // Check the available bandwidth
    if ( MyEnviron.MaxBandwidth() != 0 )
    {
        // This is a bit too simplistic. If we haven't got the bandwidth we should
        // try and get the other calls to reduce their bandwidth by sending them BRQs.
        // However for now just do it the simple minded way....
        if ( ( TotalBandwidth + Call.GetBandwidth() ) > MyEnviron.MaxBandwidth() )
            throw BandwidthUnavailable("Maximum bandwidth would be exceeded");
    }

	TableMutex.Wait();
	Table.push_back( Call );
	TotalBandwidth += Call.GetBandwidth();
	TableMutex.Signal();
}

void CallTable::RemoveEntry( Table_t::iterator i )
// Task: to remove the given entry from the table
{
    TotalBandwidth -= (*i).GetBandwidth();
    Table.erase(i);
}

void CallTable::Remove( const H225_CallIdentifier & Id )
// Task: to remove the given call from the table (by identifier)
{
//   	PSYSTEMLOG( Info, "Removing call " << Id << " from table" );
	TableMutex.Wait();
	Table_t::iterator i = find_if(Table.begin(),
	                              Table.end(),
	                              hasCallId(Id)
	                             );
    if ( i != Table.end() )	
        RemoveEntry(i);
 	TableMutex.Signal();
}

void CallTable::Remove( const H225_CallReferenceValue & CRV )
// Task: to remove the given call from the table
{
//   	PSYSTEMLOG( Info, "Removing call " << CRV << " from table" );
	TableMutex.Wait();
	Table_t::iterator i = find_if(Table.begin(),
	                              Table.end(),
	                              hasCallRef(CRV)
	                             );
    if ( i != Table.end() )	
	    RemoveEntry(i);
	TableMutex.Signal();
}

H225_BandWidth CallTable::GetAllowedBandwidth() const
// Task: to return the amount of available bandwidth
{
    H225_BandWidth  Result;
    unsigned        ResultVal;

    if ( MyEnviron.MaxBandwidth() == 0 )
        ResultVal = UINT_MAX;
    else
        ResultVal = MyEnviron.MaxBandwidth() - TotalBandwidth;

    Result.SetValue( ResultVal );
    return Result;
}

CallDetails CallTable::FindByRef( const H225_CallReferenceValue & CRV )
// Task: to return the call that has the given identifier
//	 will throw an exception if the call id doesn't exist
{
	TableMutex.Wait();
	Table_t::const_iterator i = find_if(Table.begin(),
	               		                Table.end(),
	                                    hasCallRef(CRV)
   	                                   );
	TableMutex.Signal();	

	if ( i == Table.end() )
		throw NotFoundError("Call reference not found");	

	return *i;
}

CallDetails CallTable::FindByRef( unsigned Reference )
// Task: to return the call that has the given reference
//	 will throw an exception if the call reference doesn't exist
{
    H225_CallReferenceValue CRV;
    CRV = Reference;
    return FindByRef( CRV );
}

CallDetails CallTable::FindById( const H225_CallIdentifier & Id )
// Task; to return the call that has the given call identifier
//	 will throw an exception if the call identifier doesn't exist
{
	TableMutex.Wait();
	Table_t::const_iterator i = find_if(Table.begin(),
	               		                Table.end(),
	                                    hasCallId(Id)
   	                                   );
	TableMutex.Signal();	

	if ( i == Table.end() )
		throw NotFoundError("Call id not found");	

	return *i;
}

bool CallTable::AlterBandwidth( const H225_CallReferenceValue & CRV,
                                const H225_EndpointIdentifier & Id,
                                      unsigned                  NewBandwidth
                           )
// Task: to handle a request to alter the bandwidth of the given call
//       to the given value. Will return true if the request has been honoured.
//       Will throw an exception if the call cannot be found or if it doesn't
//       "belong" to the given endpoint
{
    return AlterBandwidth( FindByRef( CRV ),
                           Id,
                           NewBandwidth,
                           "Call reference and endpoint id inconsistent"
                         );
}

bool CallTable::AlterBandwidth( const H225_CallIdentifier &     CallId,
                                const H225_EndpointIdentifier & Id,
                                      unsigned                  NewBandwidth
                              )
// Task: to handle a request to alter the bandwidth of the given call
//       to the given value. Will return true if the request has been honoured.
//       Will throw an exception if the call cannot be found or if it doesn't
//       "belong" to the given endpoint
//       Similar to the previous call except that it takes a call identifier
//       This is the preferable version to use if possible
{
    return AlterBandwidth( FindById( CallId ),
                           Id,
                           NewBandwidth,
                           "Call identifier and endpoint id inconsistent"
                         );
}

bool CallTable::AlterBandwidth(       CallDetails               Call,
                                const H225_EndpointIdentifier & Id,
                                      unsigned                  NewBandwidth,
                                const string &                  InconsistentError
                              )
{
    // Firstly check it's consistent
    if ( !(( Call.GetCaller() == Id ) || ( Call.GetCalled() == Id )) )
    {
        throw RequestInconsistent(InconsistentError);
    }

    PWaitAndSignal MutexWait(TableMutex);
    if ( MyEnviron.MaxBandwidth() != 0 )
    {
        if ( ( TotalBandwidth - Call.GetBandwidth() + NewBandwidth ) >
               MyEnviron.MaxBandwidth()
           )
        {
            // We would exceed our maximum bandwidth
            return false;
        }
   	}
   	
   	TotalBandwidth -= Call.GetBandwidth();
   	Call.SetBandwidth( NewBandwidth );
   	TotalBandwidth += NewBandwidth;
  	
   	return true;
}
