/*
 * Call routing thread for OpenGate
 *
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: CallThread.cxx,v $
 * Revision 1.11  2000/08/04 14:14:47  aunitt
 * Updated to new naming conventions in Q931 in 1.1beta3 of OpenH323.
 * Commented out unnecessary logging.
 *
 * Revision 1.10  2000/05/23 10:20:07  aunitt
 * Removed unused calling parameters.
 *
 * Revision 1.9  2000/05/19 12:10:39  aunitt
 * Now supports handling of tunneled H.245 messages.
 *
 * Revision 1.8  2000/05/15 19:07:07  aunitt
 * Now logs messages in both directions.
 * Added IP address to message logs.
 *
 * Revision 1.7  2000/05/12 14:09:30  aunitt
 * Updated to use new logging.
 * Fixed compiler warnings.
 *
 * Revision 1.5  2000/05/10 09:17:10  aunitt
 * Added H.245 thread to handle routing of H.245 messages.
 *
 * Revision 1.4  2000/05/08 17:46:45  aunitt
 * Added support for recording H245 address.
 *
 * Revision 1.3  2000/04/27 18:07:15  aunitt
 * Added support for thread naming.
 *
 * Revision 1.2  2000/04/12 13:34:48  aunitt
 * Now uses call identifier in preference to call reference.
 * Fixed translation of setup messages.
 * Changed to use new convenience functions in AddrUtils.
 *
 * Revision 1.1  2000/04/10 19:22:25  aunitt
 * Call handling thread for gatekeeper call routing for OpenGatekeeper.
 *
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#include <q931.h>
#include <h245.h>
#include "Log.h"
#include "CallTabl.h"
#include "EndpointTabl.h"
#include "AddrUtils.h"
#include "Defaults.h"
#include "CallThread.h"

CallThread::CallThread( const Environ & AkaEnviron )
    : PThread(1000, NoAutoDeleteThread, NormalPriority, "CallThread"),
      MyEnviron(AkaEnviron), Running(true)
{
    // Make sure you open the socket before calling Resume
    if ( !CallSocket.Listen( MyEnviron.LocalAddr ) )
        PSYSTEMLOG( Error, "Listen failed on call signalling socket" );
    Resume();
}

CallThread::~CallThread()
{
}

H225_TransportAddress CallThread::GetMyCallSignallingAddr()
{
    PIPSocket::Address Addr;
    WORD Port;

    // This may seem bizarre, but we need GetLocalAddress to get the port, however
    // as we often bind to "*" that won't give us our IP address. Therefore we call
    // GetHostAddress to get our real IP address!
    CallSocket.GetLocalAddress( Addr, Port );
    CallSocket.GetHostAddress( Addr );

    return AddrUtils::ConvertToH225TransportAddr( Addr, Port );
}

void CallThread::Main()
{
    while ( CallSocket.IsOpen() )
    {
        PTCPSocket * CallChannel = new PTCPSocket;
        if ( CallChannel->Accept( CallSocket ) )
        {
            // We have a new call, spawn a thread to handle it
            // It will handle the clean up of the socket when it finishes...
            SignallingThread * SigThread = new SignallingThread( this,
                                                                 MyEnviron,
                                                                 CallChannel
                                                               );

            // Fool the compiler into not complaining about SigThread not being used
            SigThread = SigThread;
        }
        else
        {
            // Whoops, error in accepting call channel
            PSYSTEMLOG( Info, "Error in accepting call" );
            delete CallChannel;
        }
    }

    // Give children time to die...
  	Sleep(1000);
}

void CallThread::Close()
{
    Running = false;
    CallSocket.Close();
}





SignallingThread::SignallingThread( CallThread *    AkaParent,
                                    const Environ & AkaEnviron,
   		                            PTCPSocket *    Socket
   		                          )
    : PThread(1000, AutoDeleteThread, NormalPriority, "SignallingThread"),
      Parent(AkaParent), MyEnviron(AkaEnviron), CallerSocket(Socket)
{
    CalledSocket = NULL;
    MyH245Thread = NULL;
    Resume();
}

SignallingThread::~SignallingThread()
{
}

static bool ReadMsg( PTCPSocket *   Socket,
                     Q931 &	        Mesg
                   )
// Task: to read a Q931 message from the given socket
{
	// Make sure it is a RFC1006 TPKT
	if ( Socket->ReadChar() != 3 )
	{
//	    PSYSTEMLOG( Info, "Invalid call signalling message, not a TKPT." );
        // I'm not happy that this is correct, if we don't get a '3' as the first
        // character we will lose synchronisation with the stream and Bad Things (tm)
        // will happen.
        // However I'm not sure what the correct solution is....
        return false;
	}

	BYTE Header[3];
	if ( Socket->ReadBlock( Header, sizeof(Header) ) )
	{
	    int BufferSize = ((Header[1] << 8)|Header[2]) - 4;
    	PBYTEArray  Buffer(BufferSize);
	    PPER_Stream ReadStream(Buffer);
	
	    if ( Socket->Read( ReadStream.GetPointer(), BufferSize ) )
	    {
	        return Mesg.Decode( ReadStream );
		}
    }
	
	PSYSTEMLOG( Info, "Failed to read" );
	// Hmm, is this too drastic? could we get transient errors?
	Socket->Close();
	return false;
}

static bool HasUUField( const Q931 &                Mesg,
                        H225_H323_UserInformation & UUField
                      )
// Task: to return true if the given message includes an H323 user user field.
//       If it does return it in the body parameter
{
    if ( Mesg.HasIE(Q931::UserUserIE) )
    {
        PPER_Stream Strm = Mesg.GetIE(Q931::UserUserIE);
        if ( UUField.Decode(Strm) )
            return true;
    }

    return false;
}

static bool HasUUField( const Q931 &                        Mesg,
                        H225_H323_UU_PDU_h323_message_body &Body
                      )
// Task: to return true if the given message includes an H323 user user field.
//       If it does return it in the body parameter
{
    H225_H323_UserInformation UUField;

    if ( HasUUField( Mesg, UUField ) )
    {
        Body = UUField.m_h323_uu_pdu.m_h323_message_body;
        return true;
    }

    return false;
}

static CallDetails FindCallDetails( const Q931 & Mesg,
                                    CallTable *  CTable
                                  )
// Task: to find the call referenced by this message
// Note: will throw CallTable::NotFoundError if we can't find the call
{
    // Try and find the call by the embedded call identifier, if it's there, because
    // because it is "more unique". If it's not there then use the call reference...

    H225_H323_UU_PDU_h323_message_body Body;

    if ( HasUUField( Mesg, Body ) )
    {
        switch( Body.GetTag() )
        {
            case H225_H323_UU_PDU_h323_message_body::e_setup            :
                {
                    H225_Setup_UUIE & Setup = Body;
                    if ( Setup.HasOptionalField( H225_Setup_UUIE::e_callIdentifier ) )
                        return CTable->FindById( Setup.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_callProceeding    :
                {
                    H225_CallProceeding_UUIE & CallProceeding = Body;
                    if ( CallProceeding.HasOptionalField( H225_CallProceeding_UUIE::e_callIdentifier ) )
                        return CTable->FindById( CallProceeding.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_connect           :
                {
                    H225_Connect_UUIE & Connect = Body;
                    if ( Connect.HasOptionalField( H225_Connect_UUIE::e_callIdentifier ) )
                        return CTable->FindById( Connect.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_alerting          :
                {
                    H225_Alerting_UUIE & Alerting = Body;
                    if ( Alerting.HasOptionalField( H225_Alerting_UUIE::e_callIdentifier ) )
                        return CTable->FindById( Alerting.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_information       :
                {
                    H225_Information_UUIE & Information = Body;
                    if ( Information.HasOptionalField( H225_Information_UUIE::e_callIdentifier ) )
                        return CTable->FindById( Information.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_releaseComplete   :
                {
                    H225_ReleaseComplete_UUIE & ReleaseComplete = Body;
                    if ( ReleaseComplete.HasOptionalField( H225_ReleaseComplete_UUIE::e_callIdentifier ) )
                        return CTable->FindById( ReleaseComplete.m_callIdentifier );
                }
                break;
            case H225_H323_UU_PDU_h323_message_body::e_facility          :
                {
                    H225_Facility_UUIE & Facility = Body;
                    if ( Facility.HasOptionalField( H225_Facility_UUIE::e_callIdentifier ) )
                        return CTable->FindById( Facility.m_callIdentifier );
                }
                break;
            default :
                break;
        }
    }

    // Fall back to call reference
    return CTable->FindByRef( Mesg.GetCallReference() );
}

static bool HasH245Address( H225_H323_UserInformation & UUField,
                            H225_TransportAddress &     Addr
                          )
// Task: given an Q931 message, returns true iff it includes an H245 transport address
//       and if it does, return it in the Addr parameter
{
    H225_H323_UU_PDU_h323_message_body & Body = UUField.m_h323_uu_pdu.m_h323_message_body;

    switch( Body.GetTag() )
    {
        case H225_H323_UU_PDU_h323_message_body::e_callProceeding    :
            {
                H225_CallProceeding_UUIE & CallProceeding = Body;
                if ( CallProceeding.HasOptionalField( H225_CallProceeding_UUIE::e_h245Address ) )
                {
                    Addr = CallProceeding.m_h245Address;
                    return true;
                }
            }
            break;
        case H225_H323_UU_PDU_h323_message_body::e_connect           :
            {
                H225_Connect_UUIE & Connect = Body;
                if ( Connect.HasOptionalField( H225_Connect_UUIE::e_h245Address ) )
                {
                    Addr = Connect.m_h245Address;
                    return true;
                }
            }
            break;
        case H225_H323_UU_PDU_h323_message_body::e_alerting          :
            {
                H225_Alerting_UUIE & Alerting = Body;
                if ( Alerting.HasOptionalField( H225_Alerting_UUIE::e_h245Address ) )
                {
                    Addr = Alerting.m_h245Address;
                    return true;
                }
            }
            break;
        default :
            return false;
    }

    return false;
}


static void SendMsg( PPER_Stream &  Stream,
                     PTCPSocket *   Destination
                   )
// Task: to send the given message to the given destination...
{
    BYTE tpkt[4];
    size_t PacketLength = sizeof(tpkt) + Stream.GetSize();
    tpkt[0] = 3;
    tpkt[1] = 0;
    tpkt[2] = (BYTE)(PacketLength >> 8);
    tpkt[3] = (BYTE)PacketLength;

	Destination->Write( tpkt, sizeof(tpkt) );
	Destination->Write( Stream.GetPointer(), Stream.GetSize() );
}


static void SendMesgToDest( const Q931 &    Mesg,
                            PTCPSocket *    Destination,
                            OpengateLog *   Log
                          )
// Task: to send the given message to the destination
{
	PPER_Stream WriteStream;
	
	if( !Mesg.Encode( WriteStream ) )
	    PSYSTEMLOG( Error, "Failed to encode message!" );
	
    PIPSocket::Address  PeerAddr;
    WORD                PeerPort;
    Destination->GetPeerAddress( PeerAddr, PeerPort );
    Log->LogQ931Msg( Mesg, OpengateLog::Sending, PeerAddr, PeerPort );
	
    SendMsg( WriteStream, Destination );
}

static void TranslateSetup(       H225_Setup_UUIE &         Setup,
                            const H225_TransportAddress &   SourceCallSigAddr,
                            const H225_TransportAddress &   DestCallSigAddr
                          )
// Task: to translate a setup user user information element
{
    if ( Setup.HasOptionalField( H225_Setup_UUIE::e_sourceCallSignalAddress ) )
    {
        Setup.m_sourceCallSignalAddress = SourceCallSigAddr;
    }

    if ( Setup.HasOptionalField( H225_Setup_UUIE::e_destCallSignalAddress ) )
    {
        Setup.m_destCallSignalAddress = DestCallSigAddr;
    }
}

static void EncodeH225IntoQ931( H225_H323_UserInformation & UUField,
                                Q931 &                      Mesg
                              )
// Task: to encode the given H225 field into the Q931 message
{
    PPER_Stream Strm;
    UUField.Encode(Strm);
    Strm.CompleteEncoding();
    Mesg.SetIE(Q931::UserUserIE, Strm);
}

static void SendProceeding( const Q931 &        SetupMesg,
                            PTCPSocket *        SourceSocket,
                            const Endpoint &    Destination,
                            const CallDetails & Call,
                            OpengateLog *       Log
                          )
// Task: to construct and send a call proceeding message back to the source after
//       processing a succesful setup
{
    Q931 Proceeding;
    Proceeding.BuildCallProceeding( SetupMesg.GetCallReference() );

    H225_H323_UserInformation UUField;
    H225_H323_UU_PDU_h323_message_body & Body = UUField.m_h323_uu_pdu.m_h323_message_body;
    Body.SetTag( H225_H323_UU_PDU_h323_message_body::e_callProceeding );

    H225_CallProceeding_UUIE & CallProceeding = Body;
    CallProceeding.m_protocolIdentifier.SetValue( GKDefaults::H225_ProtocolID );
    CallProceeding.m_destinationInfo    = Destination.GetType();
    if ( Call.HasIdentifier() )
    {
        CallProceeding.IncludeOptionalField( H225_CallProceeding_UUIE::e_callIdentifier );
        CallProceeding.m_callIdentifier = Call.GetId();
    }

    EncodeH225IntoQ931( UUField, Proceeding );

    SendMesgToDest( Proceeding, SourceSocket, Log );
}

void SignallingThread::ConnectToDestination( const Q931 & SetupMesg )
// Task: to connect the socket to the desired destination of the given setup message
{
    try
    {
        CallDetails MyCall = FindCallDetails( SetupMesg, MyEnviron.CTable );
        Endpoint Destination = MyEnviron.EPTable->FindByEndpointId( MyCall.GetCalled() );
        H225_TransportAddress_ipAddress IpAddr;
        if ( Destination.GetCallSigIPAddress( IpAddr ) )
        {
            PIPSocket::Address DestAddr( IpAddr.m_ip[0],
                                         IpAddr.m_ip[1],
                                         IpAddr.m_ip[2],
                                         IpAddr.m_ip[3]
                                       );
            CalledSocket = new PTCPSocket( static_cast<WORD>(IpAddr.m_port) );
            if ( CalledSocket->Connect( DestAddr ) )
            {
                SendProceeding( SetupMesg,
                                CallerSocket,
                                Destination,
                                MyCall,
                                MyEnviron.Log
                              );
            }
            else
            {
                delete CalledSocket;
                PSYSTEMLOG( Info, "Failed to connect to call signalling channel at " << IpAddr );
                return;
            }
        }
        else
        {
            PSYSTEMLOG( Info, "Failed to find IP address for Q931 destination" );
            return;
        }
    }
    catch ( CallTable::NotFoundError )
    {
        PSYSTEMLOG( Error, "Failed to find call reference " << SetupMesg.GetCallReference() );
        return;
    }
    catch ( EndpointTable::NotFoundError )
    {
        PSYSTEMLOG( Error, "Failed to find endpoint for Q931 message" );
        return;
    }
}

static void HandleTunneledH245( H225_H323_UserInformation & UUField,
 		                        H245Handler &               Handler
    	                      )
// Task: to handle tunneled H.245 messages in the given message (if any)
{
    H225_H323_UU_PDU & UU_PDU = UUField.m_h323_uu_pdu;

    if ( UU_PDU.HasOptionalField( H225_H323_UU_PDU::e_h245Control ) )
    {
        for( PINDEX i=0; i < UU_PDU.m_h245Control.GetSize(); ++i )
        {
            PPER_Stream Strm = UU_PDU.m_h245Control[i].GetValue();
       	    H245_MultimediaSystemControlMessage H245Mesg;
            H245Mesg.Decode( Strm );
            Handler.HandleMesg( H245Mesg );
            // We should re-encode it here if we change it....
        }
    }
}

bool SignallingThread::HandleSetup( Q931 &                      Mesg,
                                    bool                        HasUU,
		                            H225_H323_UserInformation & UUField
                                  )
// Task: to handle a setup message
// Retn: true iff message should be forwarded
{
    // We need to establish the connection to the destination
    ConnectToDestination( Mesg );

    if ( CalledSocket == NULL )
    {
        PSYSTEMLOG( Error, "Failed to connect to calling signalling channel of endpoint" );
        return false;
    }

    if ( HasUU )
    {
        // Handle any tunneled H.245
        HandleTunneledH245( UUField, MyH245Handler );

        // Translate setup message so that it references our call signalling
        // address

        PIPSocket::Address SourceAddr;
        WORD               SourcePort;
        PIPSocket::Address DestAddr;
        WORD               DestPort;

        CalledSocket->GetLocalAddress( SourceAddr, SourcePort );
        CalledSocket->GetPeerAddress( DestAddr, DestPort );
        TranslateSetup( UUField.m_h323_uu_pdu.m_h323_message_body,
                        AddrUtils::ConvertToH225TransportAddr( SourceAddr, SourcePort ),
                        AddrUtils::ConvertToH225TransportAddr( DestAddr, DestPort )
                      );

        EncodeH225IntoQ931( UUField, Mesg );
    }
    else
    {
        PSYSTEMLOG( Error, "Failed to decode Q931 User-User information" );
    }

    return true;
}

static void SetH245Address(       Q931 &                        Mesg,
		                          H225_H323_UserInformation &   UUField,
                            const H225_TransportAddress &       Addr
                          )
// Task: given an Q931 message sets the H245 transport address
{
    H225_H323_UU_PDU_h323_message_body & Body =
            UUField.m_h323_uu_pdu.m_h323_message_body;
    switch( Body.GetTag() )
    {
        case H225_H323_UU_PDU_h323_message_body::e_callProceeding    :
            {
                H225_CallProceeding_UUIE & CallProceeding = Body;
                CallProceeding.IncludeOptionalField( H225_CallProceeding_UUIE::e_h245Address );
                CallProceeding.m_h245Address = Addr;
            }
            break;
        case H225_H323_UU_PDU_h323_message_body::e_connect           :
            {
                H225_Connect_UUIE & Connect = Body;
                Connect.IncludeOptionalField( H225_Connect_UUIE::e_h245Address );
                Connect.m_h245Address = Addr;
            }
            break;
        case H225_H323_UU_PDU_h323_message_body::e_alerting          :
            {
                H225_Alerting_UUIE & Alerting = Body;
                Alerting.IncludeOptionalField( H225_Alerting_UUIE::e_h245Address );
                Alerting.m_h245Address = Addr;
            }
            break;
        default :
            // Do nothing
            break;
    }

    EncodeH225IntoQ931( UUField, Mesg );
}



void SignallingThread::HandleH245Setup( Q931 &                      Mesg,
   				                        H225_H323_UserInformation & UUField
   				                      )
// Task: to handle potential setting up of H.245 channel
{
    H225_TransportAddress Addr;

    if ( MyEnviron.RouteH245() && HasH245Address( UUField, Addr ) )
    {
        // Create the H.245 handler thread....
        MyH245Thread = new H245Thread( Parent, MyEnviron, Addr );

        // Substitute the H.245 address with ours
        SetH245Address( Mesg, UUField, MyH245Thread->GetAddress() );
    }
}

bool SignallingThread::HandleProceeding( Q931 &                         Mesg,
                                         bool                           HasUU,
		                                 H225_H323_UserInformation &    UUField
		                               )
// Task: to handle a call proceeding
// Retn: true iff message should be forwarded
{
    if ( HasUU )
    {
        HandleH245Setup( Mesg, UUField );
        HandleTunneledH245( UUField, MyH245Handler );
    }
    return false;   // We should have already sent a proceeding in response to the setup
}

bool SignallingThread::HandleAlerting( Q931 &                       Mesg,
                                       bool                         HasUU,
                                       H225_H323_UserInformation &  UUField
		                             )
// Task: to handle an alerting message
// Retn: true iff message should be forwarded
{
    if ( HasUU )
    {
        HandleH245Setup( Mesg, UUField );
        HandleTunneledH245( UUField, MyH245Handler );
    }
    return true;
}

bool SignallingThread::HandleConnect( Q931 &                        Mesg,
                                      bool                          HasUU,
		                              H225_H323_UserInformation &   UUField
		                            )
// Task: to handle a connect message
// Retn: true iff message should be forwarded
{
    if ( HasUU )
    {
        HandleH245Setup( Mesg, UUField );
        HandleTunneledH245( UUField, MyH245Handler );
    }
    return true;
}

#if (_MSC_VER >= 1200)
#pragma warning ( push )
#pragma warning ( disable : 4100 )  // Remove warning about unused parameters
#endif

bool SignallingThread::HandleRelease( Q931 &                        Mesg,
                                      bool                          HasUU,
		                              H225_H323_UserInformation &   UUField
		                            )
// Task: to handle a release complete message
// Retn: true iff message should be forwarded
{
    if ( HasUU )
        HandleTunneledH245( UUField, MyH245Handler );
    return true;
}

bool SignallingThread::HandleFacility( Q931 &                        Mesg,
                                       bool                          HasUU,
		                               H225_H323_UserInformation &   UUField
  		                             )
// Task: to handle a facility message
// Retn: true iff message should be forwarded
{
    if ( HasUU )
        HandleTunneledH245( UUField, MyH245Handler );
    return true;
}

#if (_MSC_VER >= 1200)
#pragma warning ( pop )
#endif

void SignallingThread::HandleMesg( Q931 & Mesg, bool FromCaller )
// Task: to handle the given Q931 message
{
    bool                        ForwardMesg;
    bool                        HasUU;
    H225_H323_UserInformation   UUField;

    HasUU = HasUUField( Mesg, UUField );

    switch ( Mesg.GetMessageType() )
    {
        case Q931::SetupMsg             :
            ForwardMesg = HandleSetup( Mesg, HasUU, UUField );
            break;
        case Q931::CallProceedingMsg    :
            ForwardMesg = HandleProceeding( Mesg, HasUU, UUField );
            break;
        case Q931::AlertingMsg          :
            ForwardMesg = HandleAlerting( Mesg, HasUU, UUField );
            break;
        case Q931::ConnectMsg           :
            ForwardMesg = HandleConnect( Mesg, HasUU, UUField );
            break;
        case Q931::ReleaseCompleteMsg   :
            ForwardMesg = HandleRelease( Mesg, HasUU, UUField );
            break;
        case Q931::FacilityMsg          :
            ForwardMesg = HandleFacility( Mesg, HasUU, UUField );
            break;
        default :   // Nothing to do for now...
            ForwardMesg = true;
            break;
    }

    if ( ( CalledSocket != NULL ) &&  // Will be created when we receive a setup mesg
         ( ForwardMesg )
       )
    {
        PTCPSocket * Socket = (FromCaller ? CalledSocket : CallerSocket);

        SendMesgToDest( Mesg, Socket, MyEnviron.Log );
    }

    if ( Mesg.GetMessageType() == Q931::ReleaseCompleteMsg )
    {
        // Call has finished...
        if ( FromCaller )
        {
            // PSYSTEMLOG( Info, "Closed caller" );
            CallerSocket->Close();
        }
        else
        {
            // PSYSTEMLOG( Info, "Closed called" );
            CalledSocket->Close();
        }
    }
}

void SignallingThread::Close()
{
    // Later on this should be improved to do a proper cleanup of the call
    // rather than just dropping it....
    CallerSocket->Close();
    if ( CalledSocket != NULL )
        CalledSocket->Close();
    if ( MyH245Thread != NULL )
        MyH245Thread->Close();
}

void SignallingThread::ReceiveMesg( bool FromCaller )
// Task: to receive a Q931 message
{
    Q931 Mesg;

    PTCPSocket * Socket = (FromCaller ? CallerSocket : CalledSocket);

    if ( ReadMsg( Socket, Mesg ) )
    {
        PIPSocket::Address  PeerAddr;
        WORD                PeerPort;
        Socket->GetPeerAddress( PeerAddr, PeerPort );

        MyEnviron.Log->LogQ931Msg( Mesg,
                                   OpengateLog::Receiving,
                                   PeerAddr,
                                   PeerPort
                                 );
        HandleMesg( Mesg, FromCaller );
    }
    else
    {
        if ( !Parent->IsRunning() )
            Close();
    }
}

void SignallingThread::Main()
{
    // It's enough to check that the caller socket is open as they will both get
    // closed when the call is torn down...
    while ( CallerSocket->IsOpen() )
    {
        if (   CallerSocket->IsOpen()   &&
             ( CalledSocket != NULL   ) &&
               CalledSocket->IsOpen()
           )
        {
            int Selection = PSocket::Select( *CallerSocket, *CalledSocket );

            switch( Selection )
            {
                case -1 :   // Data available on CallerSocket
                            ReceiveMesg(true);
                            break;
                case -2 :   // Data available on CalledSocket
                            ReceiveMesg(false);
                            break;
                case -3 :   // Data available on both
                            ReceiveMesg(true);
                            ReceiveMesg(false);
                            break;
                case 0  :   // Timeout
                            break;
                default :   // Error on select...
                            PSYSTEMLOG( Info, "Error " << Selection << " on select" );
                            break;
            }
        }
        else if ( ( CalledSocket != NULL ) && CalledSocket->IsOpen() )
        {
            ReceiveMesg(false);
        }
        else
        {
            // We only have the socket from the caller...
            ReceiveMesg(true);
        }
    }

    if ( MyH245Thread != NULL )
    {
        MyH245Thread->WaitForTermination();
        delete MyH245Thread;
    }
    if ( CalledSocket != NULL )
    {
        delete CalledSocket;
    }
    delete CallerSocket;
}

////////////////

H245Handler::H245Handler()
{
}

H245Handler::~H245Handler()
{
}

void H245Handler::HandleMesg( H245_MultimediaSystemControlMessage & Mesg )
// Task: to handle the given H.245 message
{
    switch( Mesg.GetTag() )
    {
        case H245_MultimediaSystemControlMessage::e_request     :
            HandleRequest( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_response    :
            HandleResponse( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_command     :
            HandleCommand( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_indication  :
            HandleIndication( Mesg );
            break;
        default :
            PSYSTEMLOG( Error, "Unknown H245 message type not handled : " << Mesg.GetTag() );
            break;
    }
}

static void HandleOpenLogicalChannel( H245_OpenLogicalChannel & Open )
// Task: to handle an open logical channel request
{
    H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters & MultiParams =
        Open.m_forwardLogicalChannelParameters.m_multiplexParameters;

    if ( MultiParams.GetTag() ==
            H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters::e_h2250LogicalChannelParameters
       )
    {
        H245_H2250LogicalChannelParameters & H225Params = MultiParams;
        if ( H225Params.HasOptionalField( H245_H2250LogicalChannelParameters::e_mediaControlChannel ) )
        {
            H245_TransportAddress & Addr = H225Params.m_mediaControlChannel;
            if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
            {
                H245_UnicastAddress & UnicastAddr = Addr;
                if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
                {
                    // Woooo!!!
                    H245_UnicastAddress_iPAddress & IP = UnicastAddr;
                    PIPSocket::Address IPAddr(IP.m_network[0],
                                              IP.m_network[1],
                                              IP.m_network[2],
                                              IP.m_network[3]
                                             );

                    /*
                    PSYSTEMLOG( Info,
                                "Control chan addr is " << IPAddr << " port " << IP.m_tsapIdentifier
                              );
                     */
                }
            }
        }
    }
}

void H245Handler::HandleRequest( H245_RequestMessage & Request )
// Task: to handle the given H.245 request
{
    switch( Request.GetTag() )
    {
        case H245_RequestMessage::e_openLogicalChannel  :
            HandleOpenLogicalChannel( Request );
            break;
        default : // Nothing to do for now....
            break;
    }
}

void H245Handler::HandleResponse( H245_ResponseMessage & Response )
// Task: to handle the given H.245 response
{
    switch( Response.GetTag() )
    {
        case H245_ResponseMessage::e_openLogicalChannelAck      :
            break;
        case H245_ResponseMessage::e_openLogicalChannelReject   :
            break;
        default : // Nothing to do for now....
            break;
    }
}

#if (_MSC_VER >= 1200)
#pragma warning ( push )
#pragma warning ( disable : 4100 )  // Remove warning about unused parameters
#endif

void H245Handler::HandleIndication( H245_IndicationMessage & Indication )
// Task: to handle the given H.245 indication
{
    // Nothing to do for now
}

void H245Handler::HandleCommand( H245_CommandMessage & Command )
// Task: to handle the given H.245 command
{
    // Nothing to do for now
}

#if (_MSC_VER >= 1200)
#pragma warning ( pop )
#endif

////////////



H245Thread::H245Thread( CallThread *                    AkaParent,
                        const Environ &                 AkaEnviron,
                        const H225_TransportAddress &   CalledAddress
                      )
    : PThread(1000, NoAutoDeleteThread, NormalPriority, "H245Thread"),
      Parent(AkaParent), MyEnviron(AkaEnviron), SessionEnded(false)
{
    PAssert( (CalledAddress.GetTag() == H225_TransportAddress::e_ipAddress),
             "H245 target address is not an IP address"
           );
    const H225_TransportAddress_ipAddress & CalledIP = CalledAddress;
    WORD Port;
    AddrUtils::ConvertToIPAddress( CalledIP, CalledAddr, Port );
    CalledSocket = new PTCPSocket( Port );
    CallerSocket = new PTCPSocket;

    // Make sure you open the socket before calling Resume
    if ( !Listener.Listen( MyEnviron.LocalAddr, 1 ) )
        PSYSTEMLOG( Error, "Listen failed on call signalling socket" );
    Resume();
}

H245Thread::~H245Thread()
{
    delete CallerSocket;
    delete CalledSocket;
}

H225_TransportAddress H245Thread::GetAddress()
// Task: to return the transport address to give to the caller..
{
    PIPSocket::Address Addr;
    WORD Port;

    Listener.GetLocalAddress( Addr, Port );
    Listener.GetHostAddress( Addr );

    return AddrUtils::ConvertToH225TransportAddr( Addr, Port );
}

static bool ConnectionEstablished( PTCPSocket &                 Listener,
                                   PTCPSocket *                 Caller,
                                   PTCPSocket *                 Called,
                                   const PIPSocket::Address &   CalledAddr
                                 )
// Task: to wait for a connect from the caller on the listener socket, when it happens
//       establish a connection to the called party...
//       Returns true if successful
{
    if ( Listener.IsOpen() )
    {
        if ( Caller->Accept( Listener ) )
        {
            // Connect to the caller
            if ( Called->Connect( CalledAddr ) )
                return true;
            else
                PSYSTEMLOG( Error, "Failed to connect to H245 target" );
        }
        else
            PSYSTEMLOG( Error, "H245 Accept failed" );
    }
    else
        PSYSTEMLOG( Error, "H425 listener isn't open" );

    return false;
}

static void SendH245Mesg( H245_MultimediaSystemControlMessage & Mesg,
                          PTCPSocket *                          Destination,
                          OpengateLog *                         Log
                        )
// Task: to send the given H.245 message to the given destination
{
    PIPSocket::Address  PeerAddr;
    WORD                PeerPort;
    Destination->GetPeerAddress( PeerAddr, PeerPort );
    Log->LogH245Msg( Mesg, OpengateLog::Sending, PeerAddr, PeerPort );

	PPER_Stream WriteStream;
	
	Mesg.Encode( WriteStream );
	WriteStream.CompleteEncoding();
    SendMsg( WriteStream, Destination );
}

void H245Thread::HandleRequest( H245_RequestMessage & Request )
// Task: to handle the given H.245 request
{
    H245Handler::HandleRequest( Request );
}

void H245Thread::HandleResponse( H245_ResponseMessage & Response )
// Task: to handle the given H.245 response
{
    H245Handler::HandleResponse( Response );
}

void H245Thread::HandleIndication( H245_IndicationMessage & Indication )
// Task: to handle the given H.245 indication
{
    H245Handler::HandleIndication( Indication );
}

void H245Thread::HandleCommand( H245_CommandMessage & Command )
// Task: to handle the given H.245 command
{
    H245Handler::HandleCommand( Command );
    if ( Command.GetTag() == H245_CommandMessage::e_endSessionCommand )
    {
        // Session has ended....
        SessionEnded = true;
    }
}

void H245Thread::HandleMesg( H245_MultimediaSystemControlMessage &  Mesg,
                             PTCPSocket *                           Target
                           )
// Task: to handle the given H.245 message
{
    switch( Mesg.GetTag() )
    {
        case H245_MultimediaSystemControlMessage::e_request     :
            HandleRequest( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_response    :
            HandleResponse( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_command     :
            HandleCommand( Mesg );
            break;
        case H245_MultimediaSystemControlMessage::e_indication  :
            HandleIndication( Mesg );
            break;
        default :
            PSYSTEMLOG( Error, "Unknown H245 message type received : " << Mesg.GetTag() );
            break;
    }

    SendH245Mesg( Mesg, Target, MyEnviron.Log );
}

void H245Thread::ReceiveMesg( PTCPSocket * From,
                              PTCPSocket * To
                            )
// Task: to receive an H.245 message and handle it as appropriate
{
    // Put in log here...
//  PSYSTEMLOG( Info, "Received H245 message" );

    BYTE tpkt[4];

    if (!From->ReadBlock( tpkt, sizeof(tpkt) ) )
    {
        // Assume that the connection has been closed....
        Close();
        return;
    }

    if ( tpkt[0] != 3 )
    {
        PSYSTEMLOG( Error, "Expecting TPKT in H.245" );
        return;
    }

	int BufferSize = ((tpkt[2] << 8)|tpkt[3]) - 4;
    PBYTEArray  Buffer(BufferSize);
    PPER_Stream ReadStream(Buffer);

	if ( From->Read( ReadStream.GetPointer(), BufferSize ) )
	{
        PIPSocket::Address  PeerAddr;
        WORD                PeerPort;
        From->GetPeerAddress( PeerAddr, PeerPort );
	    H245_MultimediaSystemControlMessage H245Mesg;
	    H245Mesg.Decode( ReadStream );

        MyEnviron.Log->LogH245Msg( H245Mesg,
                                   OpengateLog::Receiving,
                                   PeerAddr,
                                   PeerPort
                                 );
	
	    HandleMesg( H245Mesg, To );
	}
}

void H245Thread::Main()
{
    if ( ConnectionEstablished( Listener, CallerSocket, CalledSocket, CalledAddr ) )
    {
        while( !SessionEnded && CallerSocket->IsOpen() && CalledSocket->IsOpen() )
        {
            int Selection = PSocket::Select( *CallerSocket, *CalledSocket );

            switch( Selection )
            {
                case -1 :   // Data available on CallerSocket
                            ReceiveMesg( CallerSocket, CalledSocket );
                            break;
                case -2 :   // Data available on CalledSocket
                            ReceiveMesg( CalledSocket, CallerSocket );
                            break;
                case -3 :   // Data available on both
                            ReceiveMesg( CallerSocket, CalledSocket );
                            ReceiveMesg( CalledSocket, CallerSocket );
                            break;
                case 0  :   // Timeout
                            break;
                default :   // Error on select...
                            PSYSTEMLOG( Info, "Error " << Selection << " on H245 select" );
                            break;
            }
        }
    }

    Close();
}

void H245Thread::Close()
{
    Listener.Close();
    CallerSocket->Close();
    CalledSocket->Close();
}
