/*
 * Endpoint Table for OpenGate
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: EndpointTabl.h,v $
 * Revision 1.17  2000/05/23 10:16:02  aunitt
 * Added support for statically defined gateway prefixes.
 *
 * Revision 1.16  2000/05/15 19:05:18  aunitt
 * Changed GetRasIPAddress to return full address structure not just ipAddress
 * structure.
 *
 * Revision 1.15  2000/05/12 10:19:06  aunitt
 * Changed the way that searches for gateway prefixes are handled by having
 * a seperate list of gateways.
 *
 * Revision 1.14  2000/05/10 09:15:10  aunitt
 * Removed truncated id warning in Windows debug build.
 *
 * Revision 1.13  2000/05/05 11:21:50  aunitt
 * Added bandwidth management.
 *
 * Revision 1.12  2000/05/02 10:24:02  aunitt
 * Added support for setting endpoint TTL from RRQ.
 *
 * Revision 1.11  2000/04/27 18:01:31  aunitt
 * Finished support for registration timeouts.
 *
 * Revision 1.10  2000/04/26 17:10:50  aunitt
 * Added initial support for registration time to live.
 *
 * Revision 1.9  2000/04/21 13:44:22  aunitt
 * Removed bogus warnings in Microsoft STL headers.
 *
 * Revision 1.8  2000/04/20 18:51:33  aunitt
 * Added support for storing endpoint vendor.
 * Added support for storing details on endpoints that are not registered with us
 * (e.g. discovered by doing an LRQ to a neighbouring gatekeeper)
 *
 * Revision 1.7  2000/04/17 17:39:18  aunitt
 * Changed underlying EndpointTable structure from a list to a set.
 * Added a map for Aliases to speed up lookups.
 *
 * Revision 1.6  2000/04/13 18:45:26  aunitt
 * Added new access functions for RAS and call signal addresses.
 *
 * Revision 1.5  2000/04/10 19:19:05  aunitt
 * Made destructor virtual.
 *
 * Revision 1.4  2000/03/21 21:17:15  aunitt
 * Added function to check if an Endpoint has all the aliases in a given array.
 *
 * Revision 1.3  2000/02/27 20:48:06  aunitt
 * Added windows support.
 *
 * Revision 1.2  2000/02/15 20:46:10  aunitt
 * Added support for E164 prefixes for Gateways
 *
 * Revision 1.1.1.1  2000/02/01 22:25:33  aunitt
 * Initial revision
 *
 *
 */

#ifndef _ENDPOINTTABL_H
#define _ENDPOINTTABL_H

#include <ptlib.h>
#include <ptlib/timer.h>
#include <ptlib/timeint.h>
#include <h225.h>
#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // Disable about debug information being truncated
#pragma warning( push, 3 )
#endif
#include <set>
#include <map>
#include <stdexcept>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif


using namespace std; 

// Definition of information held about an Endpoint

class Endpoint
{
	public:
		Endpoint( const H225_ArrayOf_TransportAddress & RasAddress,
			      const H225_ArrayOf_TransportAddress & CallSigAddr,
			      const H225_EndpointType &		        Type,
			      const H225_ArrayOf_AliasAddress &     Alias,
		          const H225_EndpointIdentifier &	    Id,
		          const H225_VendorIdentifier &         Vendor,
		                bool                            IsLocal = true
			    );
		Endpoint( const H225_ArrayOf_TransportAddress & RasAddress,
			      const H225_ArrayOf_TransportAddress & CallSigAddr,
			      const H225_EndpointType &		        Type,
			      const H225_ArrayOf_AliasAddress &     Alias,
		          const H225_EndpointIdentifier &	    Id,
		                bool                            IsLocal = true
			    );
		Endpoint( const H225_ArrayOf_TransportAddress & RasAddress,
			      const H225_ArrayOf_TransportAddress & CallSigAddr,
			      const H225_ArrayOf_AliasAddress &     Alias,
		          const H225_EndpointIdentifier &	    Id,
		                bool                            IsLocal = true
			    );
		Endpoint( const H225_ArrayOf_TransportAddress & CallSigAddr,
		          const H225_ArrayOf_AliasAddress &     Alias,
		          const H225_EndpointIdentifier &	    Id
		        );      // Assumed to be not local....
    	Endpoint( const H225_ArrayOf_AliasAddress &     Alias,
		          const H225_EndpointIdentifier &	    Id
		        );      // Assumed to be not local....
	    Endpoint( const H225_EndpointIdentifier & Id );
		virtual ~Endpoint();

        enum TerminalTypes
        {
            TerminalOnly = 50,
            TerminalAndMC = 70,
            GatewayOnly = 60,
            GatewayAndMC = 80,
            GatewayAndMCWithDataMP = 90,
            GatewayAndMCWithAudioMP = 100,
            GatewayAndMCWithAVMP = 110,
            GatekeeperOnly = 120,
            GatekeeperWithDataMP = 130,
            GatekeeperWithAudioMP = 140,
            GatekeeperWithAVMP = 150,
            MCUOnly = 160,
            MCUWithDataMP = 170,
            MCUWithAudioMP = 180,
            MCUWithAVMP = 190
        };

		const H225_ArrayOf_TransportAddress & GetRasAddresses() const
			{ return RasAddress; };
        // Task: returns the RAS addresses of this endpoint
        			
		const H225_ArrayOf_TransportAddress & GetCallSigAddresses() const
			{ return CallSignalAddr; };	
	    // Task: returns the call signalling addresses of this endpoint
	
	    H225_ArrayOf_TransportAddress & GetWriteableRasAddresses()
	        { return RasAddress; };
	    // Task: return a writeable reference to the RAS addresses of this endpoint
	
	    H225_ArrayOf_TransportAddress & GetWriteableCallSigAddresses()
	        { return CallSignalAddr; };	
   	    // Task: return a writeable reference to the call signalling addresses
   	    //      of this endpoint	
   	
		const H225_ArrayOf_AliasAddress & GetAliases() const
			{ return Alias; };
	    // Task: return the aliases of this endpoint
	
		const H225_EndpointIdentifier & GetId() const { return Id; };
		// Task: returns the id of this endpoint
		
		bool IsTypeKnown() const { return TypeKnown; };
		
		const H225_EndpointType & GetType() const { return Type; };
		// Task: returns the type of this endpoint
		
		void SetType( const H225_EndpointType & AkaType ) { Type = AkaType; }
		
		bool IsVendorKnown() const { return VendorKnown; };
		// Task: return true iff the vendor of this endpoint is known
		
		const H225_VendorIdentifier & GetVendor() const { return Vendor; };
		// Task: returns the vendor of this endpoint

		bool HasRasAddr( const H225_TransportAddress & Addr ) const;
		// Task: returns true iff this Endpoint has this RAS address
		
		bool HasCallAddr( const H225_TransportAddress & Addr ) const;
		// Task: returns true iff this Endpoint has this call signalling address
		
		bool HasAlias( const H225_AliasAddress & Alias ) const;
		// Task: returns true iff this Endpoint has this alias
		
		bool Endpoint::HasAllAliases( const H225_ArrayOf_AliasAddress & Aliases ) const;
        // Task: to return true iff we have have the given aliases

		bool GetRasIPAddress( H225_TransportAddress & Addr ) const;
		// Task: to get the Ras IP address for this endpoint
		//       (the endpoint can have many different types of address, but
		//       but we normally use the IP address)
		//       returns true iff found

		bool GetCallSigIPAddress( H225_TransportAddress_ipAddress & Addr ) const;
		// Task: to get the call signalling IP address for this endpoint

		void UpdateId( const H225_EndpointIdentifier & NewId )
			{ Id = NewId; };
		// Task: updates the endpoint id
		
		void UpdateAlias( const H225_ArrayOf_AliasAddress & NewAlias )
			{ Alias = NewAlias; };
		// Task: updates the array of alias addresses
		
		bool IsInLocalZone() const { return IsLocal; };
		// Task: to return true iff the endpoint is in our local zone
		
		bool operator==( const Endpoint & EP ) const;
		bool operator!=( const Endpoint & EP ) const;
		bool operator< ( const Endpoint & EP ) const;

		bool IsGateway() const;
		// Task: to return true iff this endpoint is a Gateway
		
		bool IsSupportedPrefix( const H225_AliasAddress & Alias ) const;
		// Task: to return true iff this alias is a supported prefix on this endpoint.
		//       Assumes this endpoint is a gateway.
		
	protected:
		Endpoint() {};	// Stop clients using default constructor

		H225_ArrayOf_TransportAddress	RasAddress;
		H225_ArrayOf_TransportAddress	CallSignalAddr;
		H225_ArrayOf_AliasAddress	    Alias;
		H225_EndpointIdentifier		    Id;
		bool                            TypeKnown;
		bool                            VendorKnown;
		bool                            IsLocal;
		H225_EndpointType		        Type;
		H225_VendorIdentifier           Vendor;
};


class Environ;
class TimeToLiveEntry;
PSORTED_LIST(TTLList,TimeToLiveEntry);
class TableCleaner;

class RegistrationTimeoutHdlr
{
    // Definition of callback functions needed to handle registration timeouts
    public:
        virtual bool OnRegistrationTimeout( const H225_EndpointIdentifier & Id,
                                                  unsigned                  Chance,
                                                  PTimeInterval &           NextPeriod
                                          ) = 0;
        // Called by the endpoint table when a registration timeout occurs.
        // Chance is the number of times that this timeout has been called for this
        // endpoint without the timeout being reset.
        // If the endpoint should be given another chance this function should return
        // true, otherwise false.
        // If it should be given another chance then the time to give it should be
        // specified in NextPeriod

        // ResetTTL should be called if the endpoint is alive
};

class EndpointTable
{
	public:
		EndpointTable( const Environ & Environ );
		virtual ~EndpointTable();

		class NotFoundError : public runtime_error 
		{
		public:
			NotFoundError(const string& what_arg): runtime_error(what_arg) {}
		};

		class InconsistentError : public runtime_error
		{
		public:
			enum Type	// What was inconsistent
			{
				Alias,
				RASAddress,
				CallSignallingAddress
			};
			InconsistentError( Type          T,
			                   const string& what_arg
			                 ): runtime_error(what_arg), ErrType(T) {}
			Type ErrType;
		};

		class RegistrationError : public runtime_error
		{
		public:
			RegistrationError( H225_RegistrationRejectReason AkaReason, 
			                   const string&                 Descript
			                 ): runtime_error(Descript), Reason(AkaReason) {}
			H225_RegistrationRejectReason Reason;
		};

		void Insert( Endpoint &                     EP,
            		 const H225_TransportAddress &  RegAddr,
		             PTimeInterval                  TTL = PMaxTimeInterval
		           );
		// Task: insert the given endpoint into the table with the given time to live
		//       If no TTL is specified then the default read from the configuration
		//       will be used.
		//       Will throw an error if the insert fails because constraints
		//       are violated (e.g. alias is already registered with a different
		//       transport address)
		//       Will update the values of an endpoint if it already exists but
		//       the aliases have changed
		//       Will assign an alias to the endpoint if it hasn't got on
		
		void Remove( const Endpoint & EP );
		// Task: to remove the given endpoint from the table
		// Note: should NOT be called during processing a timeout, call
		//       deferred remove instead
		
		void RemoveByRasAddr( const H225_ArrayOf_TransportAddress & Addr );
        // Task: to remove the endpoint with the given RAS addresses

		void RemoveByCallSigAddr( const H225_ArrayOf_TransportAddress & Addr );
        // Task: to remove the endpoint with the given call signal addresses

        void DeferredRemove( const H225_EndpointIdentifier & Id ) const;
        // Task: to remove the endpoint with the specified identifier as soon as possible.
        //       Call this if you want to remove an endpoint during processing a timeout

		size_t Size() const { return Table.size(); };
		// Task: to return the number of endpoints in the table

		bool IsEmpty() const { return Table.empty(); };
		// Task: to return true iff the table is empty

		bool AliasExists( const H225_AliasAddress & Alias );
		// Task: to return true iff the given alias is already registered in
		//	 the table

		Endpoint FindByAlias( const H225_ArrayOf_AliasAddress & Alias );
		// Task: to return the endpoint that has the given alias, will throw
		//       an exception if the alias doesn't exist or is defined to multiple
		//       endpoints
    	
		bool RasAddrExists( const H225_TransportAddress & Addr );
		// Task: to return true iff the given RAS address is already registered in
		//	 the table

		Endpoint FindByRasAddr( const H225_ArrayOf_TransportAddress & Addr );
		// Task: to return the endpoint that has the given RAS address, will throw
		//       an exception if the address doesn't exist or is defined to multiple
		//       endpoints

		bool CallSigAddrExists( const H225_TransportAddress & Addr );
		// Task: to return true iff the given call signal address is already 
		//       registered in the table

		Endpoint FindByCallSigAddr( const H225_ArrayOf_TransportAddress & Addr );
		// Task: to return the endpoint that has the given call signal address, 
		//       will throw an exception if the addresss doesn't exist or is
		//       defined to multiple endpoints

		Endpoint FindByEndpointId( const H225_EndpointIdentifier & Id );
		// Task: to return the endpoint that has the given identifier
		//	     will throw an exception if the endpoint id doesn't exist

		Endpoint Pop();
		// Task: pop the first endpoint from the table
		
		H225_EndpointIdentifier GenerateEndpointId() const;
		// Task: to generate a new, unique Endpoint Id
		
		void ResetTTL( const H225_EndpointIdentifier & Id );
		// Task: to reset the time to live of the given endpoint, i.e. let it live
		//       longer
		
        H225_TimeToLive GetTimeToLive( const H225_EndpointIdentifier & Id );
        // Task: to return the time to live for the given endpoint

        void SetTimeoutHdlr( RegistrationTimeoutHdlr * Hdlr )
        // Task: to set the registration timeout handler
        { TimeoutHdlr = Hdlr; }

        typedef map<H225_AliasAddress,H225_EndpointIdentifier> AliasMap_t;
   	    PLIST(GatewayList_t,H225_EndpointIdentifier);

	protected:
	
        typedef set<Endpoint> Table_t;
        	
		EndpointTable();                        // Disallow calls to default constructor
		EndpointTable(const EndpointTable &T);  // and copy constructor
	 	
	    Endpoint FindByPrefix( const H225_ArrayOf_AliasAddress & Alias );
        // Task: to return the endpoint (gateway) that supports routing to a
        //       telephone number prefix of the given alias, e.g. gateway that
        //       supports calling "020" when the alias is "02073620359".

        void EndpointTable::DoRemove( Table_t::iterator i );
        // Task: to do the actual removal of the given entry in the table.
        //       Assumes mutual exclusion has been obtained.

  		
   	    const Environ &                 MyEnviron;
		H225_GatekeeperIdentifier       GatekeeperId;
        Table_t		                    Table;
		PMutex                          TableMutex;
		AliasMap_t                      AliasMap;
		GatewayList_t                   Gateways;
		TTLList                         TTL;
		TableCleaner *                  Cleaner;
		RegistrationTimeoutHdlr *       TimeoutHdlr;
};

// Useful operators...

bool operator==( const H225_AliasAddress & a,
                 const H225_AliasAddress & b
               );
// Equality operator for alias addresses

bool operator==( const H225_TransportAddress & a,
                 const H225_TransportAddress & b
               );
// Equality operator for transport addresses

bool operator< ( const H225_EndpointIdentifier & a,
                 const H225_EndpointIdentifier & b
               );
// Less than operator for endpoint ids....

bool operator==( const H225_EndpointIdentifier & a,
                 const H225_EndpointIdentifier & b
               );
// Equality operator for endpoint ids....

#endif // _ENDPOINTTABL_H

