/* Seahorse
 *
 * Copyright (C) 1999, 2000 Anthony Mulcahy
 *   
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_FORKPTY
#if defined(HAVE_PTY_H)
#    include <pty.h>
#elif defined(HAVE_UTIL_H) /* OpenBSD */
#    include <util.h>
#elif defined(HAVE_LIBUTIL_H) /* FreeBSD */
#    include <libutil.h>
#endif
#else
#include <stropts.h>
#endif

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <termios.h>
#include <unistd.h>
#include <signal.h>


#include "gpg-interface.h"
#include "seahorsesig.h"

/* #define _DEBUG */

extern char	*comment;
extern char	*default_cipher;
extern int	enable_comment;

static int
gpg_exec			       (char	*argv[],
					char	**output,
                                        char	**diagnostics);

static int
gpg_exec_with_passwd		       (char	*argv[],
					int	clipboard,
					int	passwd_fds[],
					char	*passphrase,
					char	*clipboard_string,
					char	**output,
                                        char	**diagnostics);


static int
gpg_edit_key			       (char	*argv[],
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox);


static int
gpg_deluid			       (char	*argv[],
  					char	*passphrase,
  					char	*uid,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox);


static int
gpg_edit_sig			       (char	*argv[],
  					char	*passphrase,
  					char	*sig,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox);


int
gpg_edit_expiry			       (char	*argv[],
					char	*expiry,
					int	passwd_fds[],
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox);


static int
gpg_edit_trust			       (char	*argv[],
  					int	trust,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox);


void
show_error (char *error_msg)
{
	fprintf(stderr,"seahorse: error: %s\n",error_msg);
	errorbox = gnome_error_dialog (error_msg);
	gtk_signal_connect (GTK_OBJECT (errorbox), "clicked",
			GTK_SIGNAL_FUNC (on_errorbox_clicked), NULL);
	gtk_signal_connect (GTK_OBJECT (errorbox), "close",
			GTK_SIGNAL_FUNC (on_errorbox_clicked), NULL);
	gtk_widget_show (errorbox);
	for (;;)
		gtk_main_iteration();
}

char *gpg_directory()
{
    char *gnupg_dir;
    char *home_dir;
    DIR *dir;

    gnupg_dir = getenv("GNUPGHOME");
    home_dir = getenv("HOME");
    if (!gnupg_dir || !*gnupg_dir) {
	gnupg_dir = malloc(strlen(home_dir) + sizeof(GPG_DIR));
	if (gnupg_dir == NULL) {
	    fprintf(stderr, "seahorse: error: can't allocate gnupg_dir\n");
	    gtk_exit(1);
	}
	strcpy(gnupg_dir, home_dir);
	strcat(gnupg_dir, GPG_DIR);
    } else {
        gnupg_dir=strdup(gnupg_dir);
	if (gnupg_dir == NULL) {
	    fprintf(stderr, "seahorse: error: can't allocate gnupg_dir\n");
	    gtk_exit(1);
	}
    }
    if ((dir=opendir(gnupg_dir)) == NULL) {
        free(gnupg_dir);
        return NULL;
    } else {
        closedir(dir);
        return gnupg_dir;
    }
}


static int gpg_get_timestamps(const char *dir,const char *file,time_t *mtime,time_t *ctime)
{
    char *fn;
    struct stat sbuf;

    *mtime=0;
    *ctime=0;
    fn=malloc(strlen(dir)+1+strlen(file)+1);
    if(fn==NULL)
        return -1;
    strcpy(fn,dir);
    strcat(fn,"/");
    strcat(fn,file);
    if(stat(fn,&sbuf)<0)
    {
        int err=errno;
        free(fn);
        return err==ENOENT; /* ENOENT ok, rest failure */
    }
    free(fn);
    *mtime=sbuf.st_mtime;
    *ctime=sbuf.st_ctime;
    return 1; /* ok */
}

struct gpg_key_cache_entry_t
{
    const char *filename;
    time_t mtime,ctime;
};

#define GPG_CACHE_SIZE	4
struct gpg_key_cache_t
{
    struct gpg_key_cache_entry_t entries[GPG_CACHE_SIZE];
    char *pubkeys_cache, *seckeys_cache;
};

static struct gpg_key_cache_t gpg_key_cache=
{
    {
        { "options",(time_t)0,(time_t)0 },
        { "pubring.gpg",(time_t)0,(time_t)0 },
        { "secring.gpg",(time_t)0,(time_t)0 },
        { "trustdb.gpg",(time_t)0,(time_t)0 }
    },
    NULL,NULL
};

/* throw away obsolete cache content */
static void gpg_cleanup_cache()
{
    char *dir;
    time_t ct, mt;
    int do_cleanup=0;
    int i;

    dir=gpg_directory();
    if( dir==NULL )
    {
        gpg_key_cache.entries[0].mtime=
            gpg_key_cache.entries[0].ctime=
            gpg_key_cache.entries[1].mtime=
            gpg_key_cache.entries[1].ctime=
            gpg_key_cache.entries[2].mtime=
            gpg_key_cache.entries[2].ctime=
            gpg_key_cache.entries[3].mtime=
            gpg_key_cache.entries[3].ctime=0;
        do_cleanup=1;
    } else {
        for(i=0;i<GPG_CACHE_SIZE;i++)
        {
            if( !gpg_get_timestamps(dir,
                                   gpg_key_cache.entries[i].filename,
                                   &mt,  &ct) )
                do_cleanup=1;
            else
            {
                if( (gpg_key_cache.entries[i].ctime!=ct)
                    || (gpg_key_cache.entries[i].mtime!=mt) )
                    do_cleanup=1;
                gpg_key_cache.entries[i].ctime=ct;
                gpg_key_cache.entries[i].mtime=mt;
            }
        }
    }
    
    if( do_cleanup )
    {
        if( gpg_key_cache.pubkeys_cache!=NULL )
        {
            free(gpg_key_cache.pubkeys_cache);
            gpg_key_cache.pubkeys_cache=NULL;
        }
        if( gpg_key_cache.seckeys_cache!=NULL )
        {
            free(gpg_key_cache.seckeys_cache);
            gpg_key_cache.seckeys_cache=NULL;
        }
    }
    return;
}


void cleanup_before_exec	        (int	fd) {
	int maxfd,i;

	if( (maxfd=sysconf(_SC_OPEN_MAX))<0 ) {
		show_error("sysconf(_SC_OPEN_MAX) failed");
		return;
	}
	for(i=0;i<maxfd;i++) { /* loop over all fds */
		if( (STDIN_FILENO!=i) && (STDOUT_FILENO!=i)
				&& (STDERR_FILENO!=i) && (fd!=i) )
			close(i);
	}
}


#ifndef HAVE_FORKPTY

static int
open_pty_pair			       (int *amaster,
					int *aslave,
					struct	termios *TERMP,
					struct	winsize *WINP)
{
	int master, slave;
	char *name;

	master = open("/dev/ptmx", O_RDWR | O_NOCTTY);
	if (master < 0)
		return 0;

	if (grantpt (master) < 0 || unlockpt (master) < 0)
		goto close_master;
	name = ptsname (master);
	if (name == NULL)
		goto close_master;

	slave = open (name, O_RDWR);
	if (slave == -1)
		goto close_master;

	if (isastream (slave)) {
		if (ioctl (slave, I_PUSH, "ptem") < 0
				|| ioctl (slave, I_PUSH, "ldterm") < 0)
			goto close_slave;
	} 
	*amaster = master;
	*aslave = slave;
	return 1; 

close_slave:
	if (TERMP)
		tcsetattr (slave, TCSAFLUSH, TERMP);

	if (WINP)
		ioctl (slave, TIOCSWINSZ, WINP);
	
	close (slave);

close_master:
	close (master);
	return 0;
}


int
forkpty				       (int	*AMASTER,
					char	*NAME,
					struct	termios *TERMP,
					struct	winsize *WINP)
{

	int amaster, aslave;
	pid_t pid;

	open_pty_pair (&amaster, &aslave, TERMP, WINP);

	pid = fork ();

	if (pid == -1)
		return -1;

	/* Child */
	if (pid == 0){
		pid_t slave_pid;

		close (amaster);
		slave_pid = getpid ();
	 
		/* Create the session */
		setsid ();

		#ifdef TIOCSCTTY
		if (ioctl (aslave, TIOCSCTTY, 0) == -1)
			return -1;
		#else /* !TIOCSTTY */
        	{
                	char *ctty;
                	int ct_fdes;

                	ctty = ttyname(aslave);
                	ct_fdes = open(ctty, O_RDWR);
                	close(ct_fdes);
        	}
		#endif /* !TIOCSTTY */

		#if defined (_POSIX_VERSION) || defined (__svr4__)
		tcsetpgrp (0, slave_pid);
		#elif defined (TIOCSPGRP)
		ioctl (0, TIOCSPGRP, &slave_pid);
		#endif
	
		dup2 (aslave, 0);
		dup2 (aslave, 1);
		dup2 (aslave, 2);
		if (aslave > 2)
			close (aslave);
	} else {
		*AMASTER = amaster;
		close (aslave);
	}
	
	return pid;
}
#endif

int cleanup_child(pid_t child)
{
    int status;
    pid_t wait_result;
    if( (wait_result=waitpid (child, &status, WNOHANG)) == 0) /* child hangs */
    {
        kill(child,SIGTERM); /* send friendly reminder */
        sleep(1);
        if( (wait_result=waitpid (child, &status, WNOHANG)) == 0) /* child still hangs */
        {
            kill(child,SIGKILL); /* use brute force */
            sleep(1);
            wait_result=waitpid (child, &status, WNOHANG);
        }
    }

    if( wait_result < 0)
        show_error ("error in waitpid");
    
    if (WIFEXITED (status)) {
        return WEXITSTATUS (status);
    }
    else return 1;
}

int
gpg_exec			       (char	*argv[],
					char	**output,
                                        char	**diagnostics)
{
	fd_set fdset;
	int op_fds[2];
	int diag_fds[2];
	int select_result, read_len;
	pid_t child;
	char *buf;
	size_t size, alloc_size;
        int eof_seen;
	char *diag_buf;
	size_t diag_size, diag_alloc_size;
        int diag_eof_seen;
	struct timeval timeout;

	if( ( pipe (op_fds)<0 ) || ( pipe (diag_fds)<0 ) )
            show_error ("failed to create pipe");
	
	if (!(child = fork ())) {
		/* in child */

                close (op_fds[0]);
                close (diag_fds[0]);

                if( ( dup2 (op_fds[1], STDOUT_FILENO)<0 )
                    || ( dup2 (diag_fds[1], STDERR_FILENO)<0 ) )
                    show_error ("dup2 failed");

		close (op_fds[1]);
		close (diag_fds[1]);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
	} else if (child < 0)
		show_error ("can't fork");

	close (op_fds[1]);
	close (diag_fds[1]);

	timeout.tv_sec = 120; /* timeout in seconds */
	timeout.tv_usec = 0;

	size = 0;
	alloc_size = 4096;
        eof_seen=0;

	buf = malloc (alloc_size);
	if (buf == NULL)
		show_error ("can't allocate buf");
	
	diag_size = 0;
	diag_alloc_size = 1024;
        diag_eof_seen=0;

	diag_buf = malloc (diag_alloc_size);
	if (diag_buf == NULL)
		show_error ("can't allocate diag_buf");
	
	for (;;) {
		FD_ZERO (&fdset);
                if( !eof_seen )
                    FD_SET (op_fds[0], &fdset);
                if( !diag_eof_seen )
                    FD_SET (diag_fds[0], &fdset);
		if ((select_result = select (FD_SETSIZE, &fdset, NULL, NULL, &timeout)) < 0) {
			if (errno == EINTR)
				continue;
			break /* otherwise stop */;
		}
                if (select_result == 0)
                    break /* timeout 120 seconds, no output for 1 minute */;
                if( FD_ISSET(op_fds[0], &fdset) )
                {
                    if (size + 4096 > alloc_size) {
			alloc_size += 4096;
			buf = realloc (buf , alloc_size);
			if (buf == NULL)
                            show_error ("can't allocate buf\n");
                    }
                    if ((read_len =  read (op_fds[0], &buf[size], alloc_size - size - 1)) < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
                    if (read_len == 0)  /* EOF */
                    {
                        eof_seen=1;
                        if( diag_eof_seen )
                            break /* EOF on both files */;
                    }
                    size += read_len;
                }
                if( FD_ISSET(diag_fds[0], &fdset) )
                {
                    if (diag_size + 1024 > diag_alloc_size) {
			diag_alloc_size += 1024;
			diag_buf = realloc (diag_buf , diag_alloc_size);
			if (diag_buf == NULL)
                            show_error ("can't allocate diag_buf\n");
                    }
                    if ((read_len =  read (diag_fds[0], &diag_buf[diag_size], diag_alloc_size - diag_size - 1)) < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
                    if (read_len == 0)  /* EOF */
                    {
                        diag_eof_seen=1;
                        if( eof_seen )
                            break /* EOF on both files */;
                    }
                    diag_size += read_len;
                }
	}

	buf[size] = 0;
	diag_buf[diag_size] = 0;
	close (op_fds[0]);
	close (diag_fds[0]);

	*output = buf;
	*diagnostics = diag_buf;
        return cleanup_child( child );
}


int
gpg_exec_with_passwd		       (char	*argv[],
					int	clipboard,
					int	passwd_fds[],
					char	*passphrase,
					char	*clipboard_string,
					char	**output,
                                        char	**diagnostics)
{
	fd_set fdset;
	fd_set write_fdset;
	int ip_fds[2];
	int op_fds[2];
	int diag_fds[2];
	int select_result, read_len, write_len;
        size_t tmp_len;
	pid_t child;
	char *buf;
	size_t size, alloc_size;
        int eof_seen;
	char *diag_buf;
	size_t diag_size, diag_alloc_size;
        int diag_eof_seen;
        char *passwd_next;
        size_t passwd_remaining;
        size_t passwd_incr;
        int passwd_eof_seen;
        char *clipboard_next;
        size_t clipboard_remaining;
        size_t clipboard_incr;
        int clipboard_eof_seen;
	struct timeval timeout;
        long tmp;

	if( ( pipe (ip_fds)<0 ) || ( pipe (op_fds)<0 ) || ( pipe (diag_fds)<0 ) )
            show_error ("failed to create pipe");
	
	if (!(child = fork ())) {
		/* in child */

		close (ip_fds[1]);
		close (op_fds[0]);
		close (passwd_fds[1]);

                if( ( dup2 (ip_fds[0], STDIN_FILENO)<0 )
                    || ( dup2 (op_fds[1], STDOUT_FILENO)<0 )
                    || ( dup2 (diag_fds[1], STDERR_FILENO)<0 ) )
                    show_error ("dup2 failed");

		close (ip_fds[0]);
		close (op_fds[1]);
		close (diag_fds[1]);

                cleanup_before_exec(passwd_fds[0]);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");

	} else if (child < 0)
		show_error ("can't fork");

	close (ip_fds[0]);
	close (op_fds[1]);
	close (diag_fds[1]);
	close (passwd_fds[0]);

	timeout.tv_sec = 120; /* timeout in seconds */
	timeout.tv_usec = 0;

	size = 0;
	alloc_size = 4096;
        eof_seen=0;

	buf = malloc (alloc_size);
	if (buf == NULL)
		show_error ("can't allocate buf");
	
	diag_size = 0;
	diag_alloc_size = 1024;
        diag_eof_seen=0;

	diag_buf = malloc (diag_alloc_size);
	if (diag_buf == NULL)
		show_error ("can't allocate diag_buf");

        passwd_next=passphrase;
        passwd_remaining=strlen (passphrase);
        tmp=fpathconf(passwd_fds[1],_PC_PIPE_BUF);
        if( tmp<=0 ) /* error -1 or impossible value 0 */
        {
            passwd_incr=1024; /* small default */
        } else {
            passwd_incr=(size_t)tmp;
        }
        passwd_eof_seen=0;

        
	if (clipboard) {
            clipboard_next=clipboard_string;
            clipboard_remaining=strlen (clipboard_string);
            tmp=fpathconf(ip_fds[1],_PC_PIPE_BUF);
            if( tmp<=0 ) /* error -1 or impossible value 0 */
            {
                clipboard_incr=1024; /* small default */
            } else {
                clipboard_incr=(size_t)tmp;
            }
            clipboard_eof_seen=0;
	} else {
            clipboard_next=NULL;
            clipboard_remaining=0;
            clipboard_incr=1024; /* small default */
            close (ip_fds[1]);
            clipboard_eof_seen=1;
        }

	for (;;) {
		FD_ZERO (&fdset);
                if( !eof_seen )
                    FD_SET (op_fds[0], &fdset);
                if( !diag_eof_seen )
                    FD_SET (diag_fds[0], &fdset);
		FD_ZERO (&write_fdset);
                if( !passwd_eof_seen )
                    FD_SET (passwd_fds[1], &write_fdset);
                if( !clipboard_eof_seen )
                    FD_SET (ip_fds[1], &write_fdset);
		if ((select_result = select (FD_SETSIZE, &fdset, &write_fdset, NULL, &timeout)) < 0) {
			if (errno == EINTR)
				continue;
			break /* otherwise stop */;
		}
		if (select_result == 0)
			break /* timeout 120 seconds, no output for 1 minute */;
                if( FD_ISSET(op_fds[0], &fdset) )
                {
                    if (size + 4096 > alloc_size) {
			alloc_size += 4096;
			buf = realloc (buf , alloc_size);
			if (buf == NULL)
                            show_error ("can't allocate buf\n");
                    }
                    if ((read_len =  read (op_fds[0], &buf[size], alloc_size - size - 1)) < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
#ifdef _DEBUG
                    fprintf(stderr,"stdout %d\n",read_len);
#endif
                    if (read_len == 0)  /* EOF */
                    {
                        eof_seen=1;
                        if( diag_eof_seen )
                            break /* EOF on both files */;
                    }
                    size += read_len;
                }
                if( FD_ISSET(diag_fds[0], &fdset) )
                {
                    if (diag_size + 1024 > diag_alloc_size) {
			diag_alloc_size += 1024;
			diag_buf = realloc (diag_buf , diag_alloc_size);
			if (diag_buf == NULL)
                            show_error ("can't allocate diag_buf\n");
                    }
                    if ((read_len =  read (diag_fds[0], &diag_buf[diag_size], diag_alloc_size - diag_size - 1)) < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
#ifdef _DEBUG
                    fprintf(stderr,"stderr %d\n",read_len);
#endif
                    if (read_len == 0)  /* EOF */
                    {
                        diag_eof_seen=1;
                        if( eof_seen )
                            break /* EOF on both files */;
                    }
                    diag_size += read_len;
                }
                if( FD_ISSET(passwd_fds[1], &write_fdset) )
                {
                    tmp_len=passwd_incr;
                    if( tmp_len>passwd_remaining )
                        tmp_len=passwd_remaining;
                    write_len=write(passwd_fds[1],passwd_next,tmp_len);
#ifdef _DEBUG
                    fprintf(stderr,"passwd %d\n",write_len);
#endif
                    if( write_len < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
                    passwd_next += write_len;
                    passwd_remaining -= write_len;
                    if( passwd_remaining == 0 ) /* EOF */
                    {
                        close(passwd_fds[1]);
                        passwd_eof_seen=1;
                    }
                }
                if( FD_ISSET(ip_fds[1], &write_fdset) )
                {
                    tmp_len=clipboard_incr;
                    if( tmp_len>clipboard_remaining )
                        tmp_len=clipboard_remaining;
                    write_len=write(ip_fds[1],clipboard_next,tmp_len);
#ifdef _DEBUG
                    fprintf(stderr,"clipboard %d\n",write_len);
#endif
                    if( write_len < 0) {
			if (errno == EINTR)
                            continue;
			break;
                    }
                    clipboard_next += write_len;
                    clipboard_remaining -= write_len;
                    if( clipboard_remaining == 0 ) /* EOF */
                    {
                        close(ip_fds[1]);
                        clipboard_eof_seen=1;
                    }
                }
	}

	buf[size] = 0;
	diag_buf[diag_size] = 0;
	close (op_fds[0]);
	close (diag_fds[0]);

	*output = buf;
	*diagnostics = diag_buf;
        return cleanup_child( child );
}


int
gpg_edit_key	       (char	*argv[],
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char prvchr = '\0';
	fd_set ready;
	int cntr = 1;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			
			if (read (master, buf, 1) > 0) {

				if (buf[0] == ' ') {
					if ((prvchr == '?') && (cntr >= 1)) {
						write (master, "y\n", 2);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 2)) {
						cntr++;
					} else if ((prvchr == ':') && (cntr == 3)) {
						write (master, passphrase,
						       strlen (passphrase));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr > 3)) {
						write (master, "\n", 1);
						cntr++;
					}
				}
				prvchr = buf[0];
				
				gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, buf, 1);
				while (gtk_events_pending())
				    gtk_main_iteration();

			} else done = 1;
		}

	} while (!done);

        return cleanup_child( child );
}


/* This function is still under development and is going to undergo
 * many changes in the next few releases. The final objective is to
 * make this a generic function for handling all key editing
 * operations. */
int
gpg_edit_sig			       (char	*argv[],
					char	*passphrase,
					char	*sig,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char *linebuf;
	char *prevlinebuf;
	fd_set ready;
	size_t alloc_size;
	int cntr = 1;
	int read_len;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	alloc_size = 1024;
	linebuf = malloc (alloc_size);
	prevlinebuf = malloc (alloc_size);
	read_len = 0;

	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			if (read (master, &linebuf[read_len], 1) > 0) {

				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == 1) {
						write (master, "uid 1\n", 6);
						cntr++;
					} else if (cntr == 2) {
						write (master, "delsig\n", 7);
						cntr++;
					} else if (cntr == 3) {
						write (master, "save\n", 5);
						cntr++;
					} else if (cntr == 4) {
						done = 1;
					}
				} else if ((linebuf[read_len] == ')') && (read_len > 0) && (linebuf[read_len - 1] == 'q')) {
					if (!strncmp(prevlinebuf + 11, sig, 8)) {
						write (master, "y\n", 2);
					}
					else write (master, "n\n", 2);
				} else if ((linebuf[read_len] == ')') && (read_len > 0) && (linebuf[read_len - 1] == 'N')) {
					write (master, "y\n", 2);
				}
				read_len++;
				if (read_len > alloc_size) {
					alloc_size += 1024;
					linebuf = realloc (linebuf , alloc_size);
					if (buf == NULL)
						show_error ("can't allocate buf\n");
				}
				if (linebuf[read_len - 1] == '\n') {
					linebuf[read_len] = '\0'; 
					gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, linebuf, -1);
					read_len = 0;
					if (prevlinebuf != NULL)
						free (prevlinebuf);
        				prevlinebuf = strdup (linebuf);
				}
				while (gtk_events_pending())
					gtk_main_iteration();
			} else done = 1;
		}
	} while (!done);

        return cleanup_child( child );
}


int
gpg_deluid			       (char	*argv[],
					char	*passphrase,
					char	*uid,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char *linebuf;
	char *prevlinebuf;
	fd_set ready;
	size_t alloc_size;
	int cntr = 1;
	int read_len;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	alloc_size = 1024;
	linebuf = malloc (alloc_size);
	prevlinebuf = malloc (alloc_size);
	read_len = 0;

	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			if (read (master, &linebuf[read_len], 1) > 0) {

				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == 1) {
						write (master, "uid ", 4);
						write (master, uid, strlen (uid));
						write (master, "\n", 1);
						cntr++;
					} else if (cntr == 2) {
						write (master, "deluid\n", 7);
						cntr++;
					} else if (cntr == 3) {
						write (master, "save\n", 5);
						cntr++;
					} else if (cntr == 4) {
						done = 1;
					}
				} else if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '?')) {
					write (master, "y\n", 2);
				}
				read_len++;
				if (read_len > alloc_size) {
					alloc_size += 1024;
					linebuf = realloc (linebuf , alloc_size);
					if (buf == NULL)
						show_error ("can't allocate buf\n");
				}
				if (linebuf[read_len - 1] == '\n') {
					linebuf[read_len] = '\0'; 
					gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, linebuf, -1);
					read_len = 0;
					if (prevlinebuf != NULL)
						free (prevlinebuf);
        				prevlinebuf = strdup (linebuf);
				}
				while (gtk_events_pending())
					gtk_main_iteration();
			} else done = 1;
		}
	} while (!done);

        return cleanup_child( child );
}


int
gpg_edit_expiry			       (char	*argv[],
					char	*expiry,
					int	passwd_fds[],
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char *linebuf;
	char *prevlinebuf;
	fd_set ready;
	size_t alloc_size;
	int cntr = 1;
	int passwdcntr = 4;
	int read_len;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	alloc_size = 1024;
	prevlinebuf = malloc (alloc_size);
	linebuf = malloc (alloc_size);
	read_len = 0;

	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			if (read (master, &linebuf[read_len], 1) > 0) {

				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == 1) {
						write (master, "expire\n", 7);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == passwdcntr) {
						write (master, "save\n", 5);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '?')) {
					if (cntr == 2) {
						write (master, expiry, strlen (expiry));
						write (master, "\n", 1);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == ':') && (linebuf[read_len - 2] == 'e')) {
					if ((cntr > 3) && (cntr < 6)) {
					/* As a temporary solution,
					 * repeat the passphrase 3
					 * times to prevent hangups.
					 */
						write (master, passphrase, strlen (passphrase));
						write (master, "\n", 1);
						passwdcntr++;
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '?')) {
					if (cntr == 3) {
						write (master, "y\n", 2);
						cntr++;
					}
				}
				read_len++;
				if (read_len > alloc_size) {
					alloc_size += 1024;
					linebuf = realloc (linebuf , alloc_size);
					if (buf == NULL)
						show_error ("can't allocate buf\n");
				}
				if (linebuf[read_len - 1] == '\n') {
					linebuf[read_len] = '\0'; 
					gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, linebuf, -1);
					read_len = 0;
					if (prevlinebuf != NULL)
						free (prevlinebuf);
        				prevlinebuf = strdup (linebuf);
				}
				while (gtk_events_pending())
					gtk_main_iteration();
			} else done = 1;
		}
	} while (!done);

        return cleanup_child( child );
}


int
gpg_edit_trust			       (char	*argv[],
					int	trust,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char *linebuf;
	char *prevlinebuf;
	fd_set ready;
	size_t alloc_size;
	int cntr = 1;
	int read_len;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	alloc_size = 1024;
	linebuf = malloc (alloc_size);
	prevlinebuf = malloc (alloc_size);
	read_len = 0;

	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			if (read (master, &linebuf[read_len], 1) > 0) {

				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == 1) {
						write (master, "trust\n", 6);
						cntr++;
					} else if (cntr == 3) {
						write (master, "q\n", 2);
						cntr++;
					} else if (cntr == 4) {
						done = 1;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '?')) {
					if (cntr == 2) {
						if (trust == 10)
							write (master, "4\n", 2);
						else if (trust == 5)
							write (master, "3\n", 2);
						else if (trust == 0)
							write (master, "2\n", 2);
						else write (master, "1\n", 2);
						cntr++;
					}
				}
				read_len++;
				if (read_len > alloc_size) {
					alloc_size += 1024;
					linebuf = realloc (linebuf , alloc_size);
					if (buf == NULL)
						show_error ("can't allocate buf\n");
				}
				if (linebuf[read_len - 1] == '\n') {
					linebuf[read_len] = '\0'; 
					gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, linebuf, -1);
					read_len = 0;
					if (prevlinebuf != NULL)
						free (prevlinebuf);
        				prevlinebuf = strdup (linebuf);
				}
				while (gtk_events_pending())
					gtk_main_iteration();
			} else done = 1;
		}
	} while (!done);

        return cleanup_child( child );
}


int
gpg_edit_uid			       (char	*argv[],
					char	*realname,
					char	*email,
					char	*comment,
					int	passwd_fds[],
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char buf[2];
	char *linebuf;
	char *prevlinebuf;
	fd_set ready;
	size_t alloc_size;
	int cntr = 1;
	int passwdcntr = 6;
	int read_len;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		putenv ("LANG=C");
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	alloc_size = 1024;
	prevlinebuf = malloc (alloc_size);
	linebuf = malloc (alloc_size);
	read_len = 0;

	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			if (read (master, &linebuf[read_len], 1) > 0) {

				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == 1) {
						write (master, "adduid\n", 7);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '>')) {
					if (cntr == passwdcntr) {
						write (master, "save\n", 5);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == ':')) {
					if (cntr == 2) {
						write (master, realname, strlen (realname));
						write (master, "\n", 1);
						cntr++;
					} else if (cntr == 3) {
						write (master, email, strlen (email));
						write (master, "\n", 1);
						cntr++;
					} else if (cntr == 4) {
						write (master, comment, strlen (comment));
						write (master, "\n", 1);
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == ':') && (linebuf[read_len - 2] == 'e')) {
					if ((cntr > 5) && (cntr < 9)) {
					/* As a temporary solution,
					 * repeat the passphrase 3
					 * times to prevent hangups.
					 */
						write (master, passphrase, strlen (passphrase));
						write (master, "\n", 1);
						passwdcntr++;
						cntr++;
					}
				}
				if ((linebuf[read_len] == ' ') && (read_len > 0) && (linebuf[read_len - 1] == '?')) {
					if (cntr == 5) {
						write (master, "o\n", 2);
						cntr++;
					}
				}
				read_len++;
				if (read_len > alloc_size) {
					alloc_size += 1024;
					linebuf = realloc (linebuf , alloc_size);
					if (buf == NULL)
						show_error ("can't allocate buf\n");
				}
				if (linebuf[read_len - 1] == '\n') {
					linebuf[read_len] = '\0'; 
					gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, linebuf, -1);
					read_len = 0;
					if (prevlinebuf != NULL)
						free (prevlinebuf);
        				prevlinebuf = strdup (linebuf);
				}
				while (gtk_events_pending())
					gtk_main_iteration();
			} else done = 1;
		}
	} while (!done);

        return cleanup_child( child );
}


int
gpg_sign_encrypt        (int	sign,
					int	clearsign,
					int	detachedsign,
					int	encrypt,
					int	ascii_arm,
					int	textmode,
					int	clipboard,
					int	output_to_file,
					char	*filename,
					char	*outfilename,
					char	*clipboard_string,
					char	*passphrase,
					char	*username,
					char	*default_key,
					char	**output,
                         char	**diagnostics)
{
	char *argv[23];
	char passwd_fd[8];
	int i = 0;
	int passwd_fds[2];

	if( pipe (passwd_fds)<0 )
            show_error ("failed to create pipe");

	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";

	if (enable_comment) {
		argv[i++] = "--comment";
		if (comment != NULL) {
			argv[i++] = comment;
		} else {
			argv[i++] = "Seahorse v" VERSION " http://seahorse.sourceforge.net";
		}
	}


	if (sign) {
		argv[i++] = "--sign";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	} else if (clearsign) {
		argv[i++] = "--clearsign";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	} else if (detachedsign) {
		argv[i++] = "--detach-sign";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	}

	if (encrypt) {
		argv[i++] = "--encrypt";
		argv[i++] = "--recipient";
		argv[i++] = username;
	}

	if (default_cipher != NULL) {
		argv[i++] = "--cipher-algo";
		argv[i++] = default_cipher;
	}

	if (default_key != NULL) {
		argv[i++] = "--default-key";
		argv[i++] = default_key;
	}

	if (ascii_arm)
		argv[i++] = "--armor";

	if (textmode)
		argv[i++] = "--textmode";

	argv[i++] = "--output";
	if (output_to_file) {
		argv[i++] = outfilename;
	} else {
		argv[i++] = "-";
	}
	
	if (!clipboard) {
		argv[i++] = filename;
	}
	
	argv[i++] = NULL;

	return gpg_exec_with_passwd (argv, clipboard, passwd_fds, passphrase, clipboard_string, output, diagnostics);
}


int
gpg_decrypt			       (int	decrypt,
					int	clipboard,
					int	output_to_file,
					char	*filename,
					char	*outfilename,
					char	*clipboard_string,
					char	*passphrase,
					char	**output,
                                        char	**diagnostics)
{

	char *argv[12];
	char passwd_fd[8];
	int i = 0;
	int passwd_fds[2];

	if( pipe (passwd_fds)<0 )
            show_error ("failed to create pipe");

	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";

	argv[i++] = "--output";
	if (output_to_file) {
		argv[i++] = outfilename;
	} else {
		argv[i++] = "-";
	}
			
	if (decrypt) {
		argv[i++] = "--decrypt";
		argv[i++] = "--passphrase-fd";
		argv[i++] = passwd_fd;
	}
	else argv[i++] = "--verify";

	if (!clipboard)
		argv[i++] = filename;
	argv[i++] = NULL;

	return gpg_exec_with_passwd (argv, clipboard, passwd_fds, passphrase, clipboard_string, output, diagnostics);
}


int
gpg_version			       (char	**output,
                                        char	**diagnostics)
{
	char *argv[9];
	int i = 0, res;

	argv[i++] = "gpg";
	argv[i++] = "--version";
	argv[i++] = NULL;

	res=gpg_exec (argv, output, diagnostics);
        if( *output==NULL )
            show_error ("can't allocate *output");
        return res;
}


int
gpg_list_pubkeys		       (char	**output,
                                        char	**diagnostics)
{
	char *argv[9];
	int i = 0, res;

        gpg_cleanup_cache();
        if( gpg_key_cache.pubkeys_cache!=NULL )
        {
            *output=strdup(gpg_key_cache.pubkeys_cache);
            if( *output==NULL )
                show_error ("can't allocate *output");
            *diagnostics=strdup("");
            return 0;
        }
        
	argv[i++] = "gpg";
	argv[i++] = "--verbose";
/*	argv[i++] = "--verbose"; */
	/*argv[i++] = "--with-key-data";*/
	argv[i++] = "--with-colons";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--list-keys";
	argv[i++] = "--fingerprint";
/*	argv[i++] = "--check-sigs";*/
	argv[i++] = NULL;

	res=gpg_exec (argv, output, diagnostics);
        gpg_key_cache.pubkeys_cache=strdup(*output);
        if( *output==NULL )
            show_error ("can't allocate *output");
        return res;
}


int
gpg_list_seckeys		       (char	**output,
                                        char	**diagnostics)
{
	char *argv[8];
	int i = 0, res;

        gpg_cleanup_cache();
        if( gpg_key_cache.seckeys_cache!=NULL )
        {
            *output=strdup(gpg_key_cache.seckeys_cache);
            if( *output==NULL )
                show_error ("can't allocate *output");
            *diagnostics=strdup("");
            return 0;
        }

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--with-colons";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--list-secret-keys";
	argv[i++] = NULL;

	res=gpg_exec (argv, output, diagnostics);
        gpg_key_cache.seckeys_cache=strdup(*output);
        if( *output==NULL )
            show_error ("can't allocate *output");
        return res;
}


int
gpg_revoke_key			       (char 	*keyid,
					char	*passphrase,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;
	char buf[2];
	char prvchr = '\0';
	fd_set ready;
	int cntr = 1;
	int done = 0;
	int master;
	pid_t child;

	argv[i++] = "gpg";
	argv[i++] = "--gen-revoke";
	argv[i++] = keyid;
	argv[i++] = NULL;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		putenv ("LANG=C");
		execvp ("gpg", argv);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	output = NULL;
	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			
			if (read (master, buf, 1) > 0) {

				if (buf[0] == ' ') {
					if ((prvchr == '?') && (cntr == 1)) {
						write (master, "y\n", 2);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 2)) {
						/* FIXME
						 * Need to support
						 * other revoke
						 * reasons.
						 */
						write (master, "1\n", 2);
						cntr++;
					} else if ((prvchr == '>') && (cntr == 3)) {
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 4)) {
						write (master, "y\n", 2);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 5)) {
						write (master, passphrase,
						       strlen (passphrase));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr > 5)) {
						write (master, "\n", 1);
						cntr++;
					}
				}
				prvchr = buf[0];
				
				gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, buf, 1);
				while (gtk_events_pending())
				    gtk_main_iteration();

			} else done = 1;
		}

	} while (!done);

        return cleanup_child( child );
}


int
gpg_lsign_key			       (char 	*keyid,
					char	*passphrase,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--lsign-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_key (argv, passphrase, output, diagnostics, 
			log_textbox);
}


int
gpg_delete_pubkey		       (char 	*keyid,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--delete-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_key (argv, NULL, output, diagnostics, 
			log_textbox);
}


int
gpg_delete_seckey		       (char 	*keyid,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--delete-secret-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_key (argv, NULL, output, diagnostics, 
			log_textbox);
}


int
gpg_change_expiry		       (char 	*keyid,
					char	*expiry,
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;
	char passwd_fd[8];
	int passwd_fds[2];

	if( pipe (passwd_fds)<0 )
            show_error ("failed to create pipe");

	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--edit-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_expiry (argv, expiry, passwd_fds, passphrase, 
			output, diagnostics, log_textbox);
}


int
gpg_change_trust		       (char 	*keyid,
					int	trust,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--edit-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_trust (argv, trust, output, diagnostics, 
			log_textbox);
}


int
gpg_add_uid			       (char	*keyid,
					char	*realname,
					char	*email,
					char	*comment,
					char	*passphrase,
					char	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;
	char passwd_fd[8];
	int passwd_fds[2];

	if( pipe (passwd_fds)<0 )
            show_error ("failed to create pipe");

	sprintf (passwd_fd, "%d", passwd_fds[0]);

	argv[i++] = "gpg";
	argv[i++] = "--edit-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_uid (argv, realname, email, comment, passwd_fds, passphrase, output, diagnostics, 
			log_textbox);
}


int
gpg_delete_sig			       (char 	*keyid,
					char	*sig,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--edit-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_sig (argv, NULL, sig, output, diagnostics, 
			log_textbox);
}


int
gpg_delete_uid			       (char 	*keyid,
					char	*uid,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--edit-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_deluid (argv, NULL, uid, output, diagnostics, 
			log_textbox);
}


int
gpg_sign_key			       (char 	*keyid,
					char	*passphrase,
					char 	**output,
                                        char	**diagnostics,
					GtkWidget *log_textbox)
{
	char *argv[12];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--sign-key";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_edit_key (argv, passphrase, output, diagnostics, 
			log_textbox);
}


int
gpg_recv_key			       (char *keyserver,
					char *keyid,
					int	use_proxy,
					char **output,
                                        char	**diagnostics)
{
	char *argv[11];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	if (use_proxy)
		argv[i++] = "--honor-http-proxy";
	argv[i++] = "--keyserver";
	argv[i++] = keyserver;
	argv[i++] = "--recv-keys";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output, diagnostics);
}


int
gpg_export_key			       (char	*keyfile,
					char	*keyid,
					int	ascii_arm,
					char	**output,
					char	**diagnostics)
{
	char *argv[11];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	if (ascii_arm)
		argv[i++] = "--armor";
	argv[i++] = "-o";
	argv[i++] = keyfile;
	argv[i++] = "--export";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output, diagnostics);
}


int
gpg_import_key			       (char *keyfile,
					char **output,
                                        char	**diagnostics)
{
	char *argv[8];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	argv[i++] = "--import";
	argv[i++] = keyfile;
	argv[i++] = NULL;

	return gpg_exec (argv, output, diagnostics);
}


int
gpg_send_key			       (char	*keyserver,
					char	*keyid,
					int	use_proxy,
					char	**output,
                                        char	**diagnostics)
{
	char *argv[11];
	int i = 0;

	argv[i++] = "gpg";
	argv[i++] = "--verbose";
	argv[i++] = "--yes";
	argv[i++] = "--batch";
	if (use_proxy)
		argv[i++] = "--honor-http-proxy";
	argv[i++] = "--keyserver";
	argv[i++] = keyserver;
	argv[i++] = "--send-keys";
	argv[i++] = keyid;
	argv[i++] = NULL;

	return gpg_exec (argv, output, diagnostics);
}


int
gpg_gen_key			       (char	*keytype,
					int	keysize,
					int	expiration,
					char	*realname,
					char	*email,
					char	*comment,
					char	*passphrase,
					GtkWidget *log_textbox)
{
	char buf[2];
	char prvchr = '\0';
	fd_set ready;
	int cntr = 1;
	int done = 0;
	int master;
	pid_t child;


	if ((child = forkpty (&master, NULL, NULL, NULL)) < 0) {
		show_error ("ptypair");
	} else if (child == 0) {
		close (master);

                cleanup_before_exec(-1);
		putenv ("LANG=C");
		execlp ("gpg", "gpg", "--gen-key", 0);
                show_error ("Cannot execute gpg");
		exit (1);
	}


	do {
		FD_ZERO (&ready);
		FD_SET (master, &ready);
		select (FD_SETSIZE, &ready, NULL, NULL, NULL);

		if (FD_ISSET (master, &ready)) {
			
			if (read (master, buf, 1) > 0) {
				/*write (STDOUT_FILENO, buf, 1);*/

				if (buf[0] == ' ') {
					if ((prvchr == '?') && (cntr == 1)) {
						write (master, keytype, 1);
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ')') && (cntr == 2)) {
						write (master, "1024\n", 5);
						cntr++;
					} else if ((prvchr == ')') && (cntr == 3)) {
						write (master, "0\n", 2);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 4)) {
						write (master, "y\n", 2);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 5)) {
						write (master, realname, strlen (realname));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 6)) {
						write (master, email, strlen (email));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == ':') && (cntr == 7)) {
						write (master, comment, strlen (comment));
						write (master, "\n", 1);
						cntr++;
					} else if ((prvchr == '?') && (cntr == 8)) {
						write (master, "o\n", 2);
						cntr++;
					/* Sometimes the passphrase isn't read*/
					/* correctly so 5 attempts are made to*/
					/* write it */
					} else if ((prvchr == ':') && (8 < cntr)
						    && (cntr < 13)) {
						write (master, passphrase,
						       strlen (passphrase));
						write (master, "\n", 1);
						cntr++;
					}
				}
				prvchr = buf[0];
				gtk_text_insert(GTK_TEXT (log_textbox), NULL, NULL, NULL, buf, 1);
				while (gtk_events_pending())
				    gtk_main_iteration();

			} else done = 1;
		}

	} while (!done);

        return cleanup_child( child );
}


