/*
 * virtual private network daemon (vpnd)
 *
 * cryptographic stuff (c) 1999 Andreas Steinmetz, astmail@yahoo.com
 * other code (c) 1999 D.O.M. Datenverarbeitung GmbH, author Andreas Steinmetz
 *
 * License:
 * This code is in the public domain (*) under the GNU public license.
 * The copyright holders will however retain their copyright.
 * There is no guarantee for the fitness and usability of this code
 * for any purpose. The author and the copyright holders take no
 * responsibility for any damages caused by the use of this code.
 * Distribution and use of this code is explicitly granted provided
 * that the above header is not modified and the above conditions
 * are met.
 * (*) 'public domain' is used here in the sense of the Wassenaar treaty.
 */

#include "vpnd.h"

/*============================================================================*/
/* whereto, my friend: routing                                                */
/*============================================================================*/

/*
 * route
 *
 * input:  how     - 1=add route, 0=delete route
 *	   dest    - the route destination in network byte order
 *	   mask    - the destination netmask in network byte order or -1 for
 *		     a host route
 *	   gateway - the gateway to the destination in network byte order or
 *		     0 if no gateway
 *	   device  - the device for the route
 *
 * return: 0 in case of success, 1 in case of error
 *
 * This procedure adds/deletes the given route to/from the system routing table.
 * The metric of the route is always 0 (rt_metric=real metric+1).
 */

int route(int how,WORD32 dest,WORD32 mask,WORD32 gateway,WORD08 *device)
{
#ifdef LINUX

	struct rtentry rt;	/* used to add routes		*/
	int s;			/* routing socket handle	*/


	/* debug message */

	ENTER("route");

	/* set up routing structure */

	memset(&rt,0,sizeof(rt));
	rt.rt_dev=device;
	if(mask==-1)
	{
		rt.rt_flags=RTF_UP|RTF_HOST|(gateway?RTF_GATEWAY:0);
		mask=htonl(0xffffffff);
	}
	else rt.rt_flags=RTF_UP|(gateway?RTF_GATEWAY:0);
	rt.rt_metric=1;
	((struct sockaddr_in *)(&(rt.rt_dst)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_dst)))->sin_addr.s_addr=dest;
	((struct sockaddr_in *)(&(rt.rt_genmask)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_genmask)))->sin_addr.s_addr=mask;
	((struct sockaddr_in *)(&(rt.rt_gateway)))->sin_family=AF_INET;
	((struct sockaddr_in *)(&(rt.rt_gateway)))->sin_addr.s_addr=gateway;

	/* add routing entry, handle errors */

	if((s=socket(AF_INET,SOCK_DGRAM,0))==-1)JUMP("socket",err1);
	if(ioctl(s,how?SIOCADDRT:SIOCDELRT,&rt)==-1)
		JUMP(how?"ioctl(SIOCADDRT)":"ioctl(SIOCDELRT)",err2);
	if(close(s))JUMP("close",err1);

	/* debug message */

	LEAVE("route");

	/* signal success */

	return 0;

	/* error handling */

err2:	close(s);
err1:	return 1;

#elif FreeBSD

	int s;				/* routing socket handle	*/
	union
	{
	    struct
	    {
		struct rt_msghdr h;	/* route header data		*/
		struct sockaddr_in d;	/* destination			*/
		struct sockaddr_in gm1;	/* gateway or netmask		*/
		struct sockaddr_in gm2;	/* netmask or empty		*/
	    } gr;
	    struct
	    {
		struct rt_msghdr h;	/* route header data		*/
		struct sockaddr_in d;	/* destination			*/
		struct sockaddr_dl i;	/* interface			*/
	    } ir;
	} rt;
	size_t len;			/* routing message length	*/

	/* debug message */

	ENTER("route");

	/* reset routing structure */

	memset(&rt,0,sizeof(rt));

	/* set up route data */

	if(gateway||!device)
	{
		len=rt.gr.h.rtm_msglen=sizeof(rt.gr)-
			(gateway?0:sizeof(struct sockaddr_in));
		rt.gr.h.rtm_type=how?RTM_ADD:RTM_DELETE;
		if(mask==-1)rt.gr.h.rtm_flags=RTF_UP|RTF_HOST|
			(gateway?RTF_GATEWAY:0);
		else rt.gr.h.rtm_flags=RTF_UP|(gateway?RTF_GATEWAY:0);
		rt.gr.h.rtm_version=RTM_VERSION;
		rt.gr.h.rtm_seq=1;
		rt.gr.h.rtm_addrs=RTA_DST|RTA_NETMASK|(gateway?RTA_GATEWAY:0);
		rt.gr.gm2.sin_len=rt.gr.gm1.sin_len=rt.gr.d.sin_len=
			sizeof(struct sockaddr_in);
		rt.gr.gm2.sin_family=rt.gr.gm1.sin_family=rt.gr.d.sin_family=
			AF_INET;
		rt.gr.d.sin_addr.s_addr=dest;
		if(gateway)
		{
			rt.gr.gm1.sin_addr.s_addr=gateway;
			rt.gr.gm2.sin_addr.s_addr=
				mask==-1?htonl(0xffffffff):mask;
		}
		else rt.gr.gm1.sin_addr.s_addr=mask==-1?htonl(0xffffffff):mask;
	}
	else
	{
		len=rt.ir.h.rtm_msglen=sizeof(rt.ir);
		rt.ir.h.rtm_type=how?RTM_ADD:RTM_DELETE;
		rt.ir.h.rtm_flags=RTF_UP|RTF_HOST;
		rt.ir.h.rtm_version=RTM_VERSION;
		rt.ir.h.rtm_seq=1;
		rt.ir.h.rtm_addrs=RTA_DST|RTA_GATEWAY;
		rt.ir.d.sin_len=sizeof(struct sockaddr_in);
		rt.ir.d.sin_family=AF_INET;
		rt.ir.d.sin_addr.s_addr=dest;
		rt.ir.i.sdl_len=sizeof(struct sockaddr_dl);
		rt.ir.i.sdl_family=AF_LINK;
		strcpy(rt.ir.i.sdl_data,device);
		rt.ir.i.sdl_nlen=strlen(device);
	}

	/* add routing entry, handle errors */

	if((s=socket(PF_ROUTE,SOCK_RAW,0))==-1)JUMP("socket",err1);
	if(write(s,&rt,len)!=len)
		JUMP(how?"RTM_ADD":"RTM_DELETE",err2);
	if(close(s))JUMP("close",err1);

	/* debug message */

	LEAVE("route");

	/* signal success */

	return 0;

	/* error handling */

err2:	close(s);
err1:	return 1;

#endif
}

/*
 * peerroute
 *
 * input:  anchor - pointer to vpnd global data
 *
 * return: 0  in case of failure, else > 0
 *
 * This procedure searches the routing tables for a routing entry
 * that points to the given peer. If suche an entry is found
 * parameters for a priority route to the peer are set up.
 */

int peerroute(VPN *anchor)
{
#ifdef LINUX

	FILE *fp;		/* routing table access	*/
	int i;			/* counter/index	*/
	char *item;		/* current item		*/
	char *device;		/* peer route device	*/
	WORD32 data[7];		/* routing table data	*/
	WORD32 mask;		/* best routing mask	*/
	char *bfr;		/* input buffer		*/


	/* debug message */

	ENTER("peerroute");

	/* set up buffer pointer */

	bfr=anchor->u2.bfr;

	/* open routing table, handle errors */

	if((fp=fopen(ROUTELIST,"r"))==NULL)RETURN("fopen",0);

	/* reset netmask and first line flag */

	mask=i=0;

	/* for all lines of the routing table */

	while(fgets(bfr,sizeof(anchor->u2.bfr),fp))
	{
		/* skip first line (only contains header) */

		if(!i)
		{
			i++;
			continue;
		}

		/* get pointer to routing table entry 1 */

		if((device=strtok(bfr," \t\r\n"))==NULL)
		{
			fclose(fp);
			RET("device",0);
		}

		/* convert routing entries 2 to 8 to numeric
		   (actually values 5 to 7 are never used) */

		for(i=0;i<7;i++)
		{
			if((item=strtok(NULL," \t\r\n"))==NULL)
			{
				fclose(fp);
				RET("item n",0);
			}

			data[i]=0;
			while(*item)
			{
				data[i]=(data[i]<<4)|(*item>='A'?
					*item-'A'+10:*item-'0');
				item++;
			}
		}

		/* if a route to the peer exists with a 
		   higher or equal precedence
		   than the previous one (if any) */

		if((anchor->peerip&data[6])==(data[0]&data[6]))
			if(ntohl(data[6])>=mask)
		{
			/* set route flag according to
			   route type */

			anchor->peerroute=data[2]&RTF_GATEWAY?2:1;

			/* memorize netmask of route */

			mask=ntohl(data[6]);

			/* memorize gateway (if route
			   through gateway */

			anchor->peergate=data[1];

			/* sanity copy of routing device name */

			strncpy(anchor->peerdev,device,16);
			anchor->peerdev[15]=0;
		}
	}

	/* close routing table */

	fclose(fp);

	/* debug message */

	LEAVE("peerroute");

	/* signal success or error */

	return anchor->peerroute;

#elif FreeBSD

	int mib[6];			/* sysctl request buffer	*/
	size_t len;			/* routing buffer length	*/
	size_t ifl;			/* interface buffer length	*/
	size_t iln;			/* ditto (current)		*/
	unsigned char *bfr;		/* routing buffer		*/
	unsigned char *ifb;		/* interface buffer		*/
	struct rt_msghdr *h;		/* routing message header	*/
	struct if_msghdr *i;		/* interface message header	*/
	struct sockaddr_dl *j;		/* interface data		*/
	struct in_addr dst;		/* destination address		*/
	struct in_addr gw;		/* gateway address		*/
	struct in_addr msk;		/* netmask			*/
	char *ifn;			/* interface name		*/
	struct sockaddr_in *data;	/* address/netmask access	*/
	int state;			/* retrieval bit field		*/
	int datalen;			/* current routing data length	*/
	int reallen;			/* padded routing data length	*/
	WORD32 mask;			/* best routing mask		*/


	/* debug message */ 

	ENTER("peerroute");

	/* reset netmask */

	mask=0;

	/* shutup compiler */

	ifn=NULL;

	/* get kernel interface list, handle errors */

	mib[0] = CTL_NET;
	mib[1] = PF_ROUTE;
	mib[2] = PF_INET;
	mib[3] = AF_INET;
	mib[4] = NET_RT_IFLIST;
	mib[5] = 0;
	if(sysctl(mib,6,NULL,&ifl,NULL,0)<0)RETURN("sysctl",0);
	if((ifb=malloc(ifl))==NULL)RETURN("malloc",0);
	if(sysctl(mib,6,ifb,&ifl,NULL,0)<0)
	{
		free(ifb);
		RET("sysctl",0);
	}

	/* get kernel ip routing table, handle errors */

	mib[0] = CTL_NET;
	mib[1] = PF_ROUTE;
	mib[2] = PF_INET;
	mib[3] = AF_INET;
	mib[4] = NET_RT_DUMP;
	mib[5] = 0;
	if(sysctl(mib,6,NULL,&len,NULL,0)<0)
	{
		free(ifb);
		RET("sysctl",0);
	}
	if((bfr=malloc(len))==NULL)
	{
		free(ifb);
		RET("malloc",0);
	}
	if(sysctl(mib,6,bfr,&len,NULL,0)<0)
	{
		free(bfr);
		free(ifb);
		RET("sysctl",0);
	}

	/* process routing table */

	h=(struct rt_msghdr *)(bfr);
	while(len)
	{
		/* reset retrieval state */

		state=0;

		/* get pointer to route data and data length */

		datalen=h->rtm_msglen-sizeof(struct rt_msghdr);
		data=(struct sockaddr_in *)(h+1);

		/* if destination is an inet route */

		if(data->sin_family==AF_INET)while(datalen)
		{
			/* store destination, adjust state */

			if(data->sin_family==AF_INET&&!(state&1))
			{
				dst=data->sin_addr;
				state|=1;
			}

			/* store gateway, adjust state */

			else if(data->sin_family==AF_INET&&!(state&2)&&
				(h->rtm_flags&RTF_GATEWAY))
			{
				gw=data->sin_addr;
				state|=2;
			}

			/* store netmask, adjust state */

			else if(data->sin_family==0xff&&!(state&4))
			{
				msk=data->sin_addr;
				state|=4;
			}

			/* get padded data size */

			reallen=data->sin_len+(data->sin_len%sizeof(long)?
				sizeof(long)-data->sin_len%sizeof(long):0);

			/* skip to next data */

			datalen-=reallen;
			data=(struct sockaddr_in *)
				((unsigned char *)(data)+reallen);
		}

		/* for all interfaces */

		i=(struct if_msghdr *)(ifb);
		iln=ifl;
		while(iln)
		{
			/* sanity check */

			if(i->ifm_type==RTM_IFINFO)
			{
				/* if interface index matches route interface */

				j=(struct sockaddr_dl *)(i+1);
				if(h->rtm_index==j->sdl_index)
				{
					/* get pointer to inteface name,
					   adjust state */

					ifn=(char *)(j->sdl_data);
					state|=8;
					break;
				}
			}

			/* next interface */

			iln-=i->ifm_msglen;
			i=(struct if_msghdr *)
				((unsigned char *)(i)+i->ifm_msglen);
		}

		/* if a destination but no netmask is given */

		if((state&1)&&!(state&4))
		{
			/* create artificial netmask, adjust state */

			if(h->rtm_flags&RTF_HOST)msk.s_addr=htonl(0xffffffff);
			else if(IN_CLASSA(ntohl(dst.s_addr)))
				msk.s_addr=htonl(IN_CLASSA_NET);
			else if(IN_CLASSB(ntohl(dst.s_addr)))
				msk.s_addr=htonl(IN_CLASSB_NET);
			else if(IN_CLASSC(ntohl(dst.s_addr)))
				msk.s_addr=htonl(IN_CLASSC_NET);
			else if(IN_CLASSD(ntohl(dst.s_addr)))
				msk.s_addr=htonl(IN_CLASSD_NET);
			state|=4;
		}

		/* if we now have destination, netmask and interface */

		if((state&13)==13)
		{
			/* if a route to the peer exists with a
			   higher or equal precedence
			   than the previous one (if any) */ 

			if((anchor->peerip&msk.s_addr)==(dst.s_addr&msk.s_addr))
				if(ntohl(msk.s_addr)>=mask)
			{
				/* set route flag according to
				   route type */

				anchor->peerroute=state&2?2:1;

				/* memorize netmask of route */

				mask=ntohl(msk.s_addr);

				/* memorize gateway (if route
				   through gateway */

				anchor->peergate=dst.s_addr;

				/* sanity copy of routing device name */

				strncpy(anchor->peerdev,ifn,16);
				anchor->peerdev[15]=0;
			}
		}

		/* next route */

		len-=h->rtm_msglen;
		h=(struct rt_msghdr *)((unsigned char *)(h)+h->rtm_msglen);
	}

	/* free table memory */

	free(bfr);
	free(ifb);

	/* debug message */

	LEAVE("peerroute");

	/* signal success or error */

	return anchor->peerroute;

#endif
}
