/*
 * prop6.c
 * Copyright (C) 1998-2001 A.J. van Os; Released under GPL
 *
 * Description:
 * Read the property information from a MS Word 6 or 7 file
 */

#include <stdlib.h>
#include <string.h>
#include "antiword.h"


/*
 * iGet6InfoLength - the length of the information for Word 6/7 files
 */
static int
iGet6InfoLength(int iByteNbr, const unsigned char *aucFpage)
{
	int	iTmp, iDel, iAdd;

	switch (ucGetByte(iByteNbr, aucFpage)) {
	case 0x02: case 0x10: case 0x11: case 0x13:
	case 0x15: case 0x16: case 0x1b: case 0x1c:
	case 0x26: case 0x27: case 0x28: case 0x29:
	case 0x2a: case 0x2b: case 0x2d: case 0x2e:
	case 0x2f: case 0x30: case 0x31: case 0x50:
	case 0x5d: case 0x60: case 0x61: case 0x63:
	case 0x65: case 0x69: case 0x6a: case 0x6b:
	case 0x90: case 0xa4: case 0xa5: case 0xb6:
	case 0xb8: case 0xbd: case 0xc3: case 0xc5:
	case 0xc6:
		return 3;
	case 0x03: case 0x0c: case 0x0f: case 0x51:
	case 0x67: case 0x6c: case 0xbc: case 0xbe:
	case 0xbf:
		return 2 + (int)ucGetByte(iByteNbr + 1, aucFpage);
	case 0x14: case 0x49: case 0x4a: case 0xc0:
	case 0xc2: case 0xc4: case 0xc8:
		return 5;
	case 0x17:
		iTmp = (int)ucGetByte(iByteNbr + 1, aucFpage);
		if (iTmp == 255) {
			iDel = (int)ucGetByte(iByteNbr + 2, aucFpage);
			iAdd = (int)ucGetByte(
					iByteNbr + 3 + iDel * 4, aucFpage);
			iTmp = 2 + iDel * 4 + iAdd * 3;
		}
		return 2 + iTmp;
	case 0x44: case 0xc1: case 0xc7:
		return 6;
	case 0x5f: case 0x88: case 0x89:
		return 4;
	case 0x78:
		return 14;
	case 0xbb:
		return 13;
	default:
		return 2;
	}
} /* end of iGet6InfoLength */

/*
 * Translate the rowinfo to a member of the row_info enumeration
 */
row_info_enum
eGet6RowInfo(int iFodo,
	const unsigned char *aucGrpprl, int iBytes, row_block_type *pRow)
{
	int	iFodoOff, iInfoLen;
	int	iIndex, iSize, iCol;
	int	iPosCurr, iPosPrev;
	BOOL	bFound24_0, bFound24_1, bFound25_0, bFound25_1, bFound190;

	fail(iFodo < 0 || aucGrpprl == NULL || pRow == NULL);

	iFodoOff = 0;
	bFound24_0 = FALSE;
	bFound24_1 = FALSE;
	bFound25_0 = FALSE;
	bFound25_1 = FALSE;
	bFound190 = FALSE;
	while (iBytes >= iFodoOff + 1) {
		iInfoLen = 0;
		switch (ucGetByte(iFodo + iFodoOff, aucGrpprl)) {
		case  24:	/* fIntable */
			if (odd(ucGetByte(iFodo + iFodoOff + 1, aucGrpprl))) {
				bFound24_1 = TRUE;
			} else {
				bFound24_0 = TRUE;
			}
			break;
		case  25:	/* fTtp */
			if (odd(ucGetByte(iFodo + iFodoOff + 1, aucGrpprl))) {
				bFound25_1 = TRUE;
			} else {
				bFound25_0 = TRUE;
			}
			break;
		case 190:	/* cDefTable */
			iSize = (int)ucGetByte(iFodo + iFodoOff + 1, aucGrpprl);
			if (iSize < 6 || iBytes < iFodoOff + 7) {
				DBG_DEC(iSize);
				DBG_DEC(iFodoOff);
				iInfoLen = 1;
				break;
			}
			iCol = (int)ucGetByte(iFodo + iFodoOff + 3, aucGrpprl);
			if (iCol < 1 ||
			    iBytes < iFodoOff + 3 + (iCol + 1) * 2) {
				DBG_DEC(iCol);
				DBG_DEC(iFodoOff);
				iInfoLen = 1;
				break;
			}
			if (iCol >= (int)elementsof(pRow->asColumnWidth)) {
				werr(1, "The number of columns is corrupt");
			}
			pRow->ucNumberOfColumns = (unsigned char)iCol;
			pRow->iColumnWidthSum = 0;
			iPosPrev = (int)(short)usGetWord(
					iFodo + iFodoOff + 4,
					aucGrpprl);
			for (iIndex = 0; iIndex < iCol; iIndex++) {
				iPosCurr = (int)(short)usGetWord(
					iFodo + iFodoOff + 6 + iIndex * 2,
					aucGrpprl);
				pRow->asColumnWidth[iIndex] =
						(short)(iPosCurr - iPosPrev);
				pRow->iColumnWidthSum +=
					pRow->asColumnWidth[iIndex];
				iPosPrev = iPosCurr;
			}
			bFound190 = TRUE;
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet6InfoLength(iFodo + iFodoOff, aucGrpprl);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
	if (bFound24_1 && bFound25_1 && bFound190) {
		return found_end_of_row;
	}
	if (bFound24_0 && bFound25_0 && !bFound190) {
		return found_not_end_of_row;
	}
	if (bFound24_1) {
		return found_a_cell;
	}
	if (bFound24_0) {
		return found_not_a_cell;
	}
	return found_nothing;
} /* end of eGet6RowInfo */

/*
 * Fill the style information block with information
 * from a Word 6/7 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet6StyleInfo(int iFodo, const unsigned char *aucFpage,
		style_block_type *pStyle)
{
	int	iBytes, iFodoOff, iInfoLen;
	int	iTmp, iDel, iAdd;
	short	sTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pStyle == NULL);

	iBytes = 2 * (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 3;
	if (iBytes < 1) {
		return FALSE;
	}
	(void)memset(pStyle, 0, sizeof(*pStyle));
	pStyle->ucStyle = ucGetByte(iFodo + 1, aucFpage);
	while (iBytes >= iFodoOff + 1) {
		iInfoLen = 0;
		switch (ucGetByte(iFodo + iFodoOff, aucFpage)) {
		case   2:	/* istd */
			sTmp = (short)ucGetByte(
					iFodo + iFodoOff + 1, aucFpage);
			NO_DBG_DEC(sTmp);
			NO_DBG_DEC(pStyle->ucStyle);
			break;
		case   5:	/* jc */
			pStyle->ucAlignment = ucGetByte(
					iFodo + iFodoOff + 1, aucFpage);
			break;
		case  12:	/* anld */
			iTmp = (int)ucGetByte(
					iFodo + iFodoOff + 1, aucFpage);
			if (iTmp < 52) {
				DBG_DEC(iTmp);
				break;
			}
			pStyle->ucListType = ucGetByte(
					iFodo + iFodoOff + 2, aucFpage);
			pStyle->bInList = TRUE;
			pStyle->ucListCharacter = ucGetByte(
					iFodo + iFodoOff + 22, aucFpage);
			break;
		case  13:	/* nLvlAnm */
			iTmp = (int)ucGetByte(
					iFodo + iFodoOff + 1, aucFpage);
			pStyle->bInList = TRUE;
			if (iTmp == 0x0c) {
				pStyle->bUnmarked = TRUE;
			}
			break;
		case  15:	/* ChgTabsPapx */
			iTmp = (int)ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			if (iTmp < 2) {
				iInfoLen = 1;
				break;
			}
			NO_DBG_DEC(iTmp);
			iDel = (int)ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			if (iTmp < 2 + 2 * iDel) {
				iInfoLen = 1;
				break;
			}
			NO_DBG_DEC(iDel);
			iAdd = (int)ucGetByte(
				iFodo + iFodoOff + 3 + 2 * iDel, aucFpage);
			if (iTmp < 2 + 2 * iDel + 2 * iAdd) {
				iInfoLen = 1;
				break;
			}
			NO_DBG_DEC(iAdd);
			break;
		case  16:	/* dxaRight */
			pStyle->sRightIndent = (short)usGetWord(
					iFodo + iFodoOff + 1, aucFpage);
			NO_DBG_DEC(pStyle->sRightIndent);
			break;
		case  17:	/* dxaLeft */
			pStyle->sLeftIndent = (short)usGetWord(
					iFodo + iFodoOff + 1, aucFpage);
			NO_DBG_DEC(pStyle->sLeftIndent);
			break;
		case  18:	/* Nest dxaLeft */
			sTmp = (short)usGetWord(iFodo + iFodoOff + 1, aucFpage);
			pStyle->sLeftIndent += sTmp;
			if (pStyle->sLeftIndent < 0) {
				pStyle->sLeftIndent = 0;
			}
			NO_DBG_DEC(sTmp);
			NO_DBG_DEC(pStyle->sLeftIndent);
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet6InfoLength(iFodo + iFodoOff, aucFpage);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
	return TRUE;
} /* end of bGet6StyleInfo */

/*
 * Build the lists with Paragraph Information for Word 6/7 files
 */
void
vGet6PapInfo(FILE *pFile, long lStartBlock,
	const long *alBBD, size_t tBBDLen,
	const unsigned char *aucHeader)
{
	row_block_type		tRow;
	style_block_type	tStyle;
	unsigned short	*ausParfPage;
	unsigned char	*aucBuffer;
	long	lParmOffset, lParmOffsetFirst, lParmOffsetLast, lBeginParfInfo;
	size_t	tParfInfoLen, tParfPageNum, tOffset, tSize, tLenOld, tLen;
	int	iIndex, iIndex2, iRun, iFodo, iLen;
	row_info_enum	eRowInfo;
	unsigned short	usParfFirstPage, usCount;
	unsigned char	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || aucHeader == NULL);
	fail(lStartBlock < 0);
	fail(alBBD == NULL);

	lBeginParfInfo = (long)ulGetLong(0xc0, aucHeader); /* fcPlcfbtePapx */
	NO_DBG_HEX(lBeginParfInfo);
	tParfInfoLen = (size_t)ulGetLong(0xc4, aucHeader); /* lcbPlcfbtePapx */
	NO_DBG_DEC(tParfInfoLen);
	if (tParfInfoLen < 4) {
		DBG_DEC(tParfInfoLen);
		return;
	}

	aucBuffer = xmalloc(tParfInfoLen);
	if (!bReadBuffer(pFile, lStartBlock,
			alBBD, tBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, lBeginParfInfo, tParfInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tParfInfoLen);

	tLen = (tParfInfoLen - 4) / 6;
	tSize = tLen * sizeof(unsigned short);
	ausParfPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 2) {
		 ausParfPage[iIndex] = usGetWord(tOffset, aucBuffer);
		 NO_DBG_DEC(ausParfPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	aucBuffer = NULL;
	tParfPageNum = (size_t)usGetWord(0x190, aucHeader);
	DBG_DEC(tParfPageNum);
	if (tLen < tParfPageNum) {
		/* Replace ParfPage by a longer version */
		tLenOld = tLen;
		usParfFirstPage = usGetWord(0x18c, aucHeader);
		DBG_DEC(usParfFirstPage);
		tLen += tParfPageNum - 1;
		tSize = tLen * sizeof(unsigned short);
		ausParfPage = xrealloc(ausParfPage, tSize);
		/* Add new values */
		usCount = usParfFirstPage + 1;
		for (iIndex = (int)tLenOld; iIndex < (int)tLen; iIndex++) {
			ausParfPage[iIndex] = usCount;
			NO_DBG_DEC(ausParfPage[iIndex]);
			usCount++;
		}
	}

	(void)memset(&tStyle, 0, sizeof(tStyle));
	(void)memset(&tRow, 0, sizeof(tRow));
	lParmOffsetFirst = -1;
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		if (!bReadBuffer(pFile, lStartBlock,
				alBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				(long)ausParfPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
			NO_DBG_HEX(ulGetLong(iIndex2 * 4, aucFpage));
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2 * 7, aucFpage);
			if (iFodo <= 0) {
				continue;
			}
			if (bGet6StyleInfo(iFodo, aucFpage, &tStyle)) {
				lParmOffset = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffset);
				tStyle.lFileOffset = lTextOffset2FileOffset(
							lParmOffset);
				vAdd2StyleInfoList(&tStyle);
				(void)memset(&tStyle, 0, sizeof(tStyle));
			}

			iLen = 2 * (int)ucGetByte(iFodo, aucFpage);
			eRowInfo = eGet6RowInfo(iFodo,
					aucFpage + 3, iLen - 3, &tRow);
			switch(eRowInfo) {
			case found_a_cell:
				if (lParmOffsetFirst >= 0) {
					break;
				}
				lParmOffsetFirst = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffsetFirst);
				tRow.lTextOffsetStart = lParmOffsetFirst;
				tRow.lFileOffsetStart = lTextOffset2FileOffset(
							lParmOffsetFirst);
				DBG_HEX_C(tRow.lFileOffsetStart < 0,
							lParmOffsetFirst);
				break;
			case found_end_of_row:
				lParmOffsetLast = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffsetLast);
				tRow.lTextOffsetEnd = lParmOffsetLast;
				tRow.lFileOffsetEnd = lTextOffset2FileOffset(
							lParmOffsetLast);
				DBG_HEX_C(tRow.lFileOffsetEnd < 0,
							lParmOffsetLast);
				vAdd2RowInfoList(&tRow);
				(void)memset(&tRow, 0, sizeof(tRow));
				lParmOffsetFirst = -1;
				break;
			case found_nothing:
				break;
			default:
				DBG_DEC(eRowInfo);
				break;
			}
		}
	}
	ausParfPage = xfree(ausParfPage);
} /* end of vGet6PapInfo */

/*
 * Fill the font information block with information
 * from a Word 6/7 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet6FontInfo(int iFodo, const unsigned char *aucFpage,
		font_block_type *pFont)
{
	int	iBytes, iFodoOff, iInfoLen;
	unsigned short	usTmp;
	short		sTmp;
	unsigned char	ucTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pFont == NULL);

	iBytes = (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 1;
	if (iBytes < 1) {
		return FALSE;
	}
	(void)memset(pFont, 0, sizeof(*pFont));
	pFont->sFontsize = DEFAULT_FONT_SIZE;
	while (iBytes >= iFodoOff + 1) {
		switch (ucGetByte(iFodo + iFodoOff, aucFpage)) {
		case  80:	/* cIstd */
			sTmp = (short)usGetWord(iFodo + iFodoOff + 1, aucFpage);
			DBG_DEC(sTmp);
                        break;
		case  82:	/* cDefault */
			pFont->ucFontstyle &= FONT_HIDDEN;
			pFont->ucFontcolor = FONT_COLOR_DEFAULT;
			break;
		case  85:	/* fBold */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_BOLD;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_BOLD;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_BOLD;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  86:	/* fItalic */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_ITALIC;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_ITALIC;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_ITALIC;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  87:	/* fStrike */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_STRIKE;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_STRIKE;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_STRIKE;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  90:	/* fSmallCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_SMALL_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_SMALL_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_SMALL_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  91:	/* fCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  92:	/* fVanish */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_HIDDEN;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_HIDDEN;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_HIDDEN;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case  93:	/* cFtc */
			usTmp = usGetWord(iFodo + iFodoOff + 1, aucFpage);
			if (usTmp <= (unsigned short)UCHAR_MAX) {
				pFont->ucFontnumber = (unsigned char)usTmp;
			}
			break;
		case  94:	/* cKul */
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			if (ucTmp != 0 && ucTmp != 5) {
				NO_DBG_MSG("Underline text");
				pFont->ucFontstyle |= FONT_UNDERLINE;
				if (ucTmp == 6) {
					DBG_MSG("Bold text");
					pFont->ucFontstyle |= FONT_BOLD;
				}
			}
			break;
		case  98:	/* cIco */
			pFont->ucFontcolor =
				ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			break;
		case  99:	/* cHps */
			usTmp = usGetWord(iFodo + iFodoOff + 1, aucFpage);
			if (usTmp > (unsigned short)SHRT_MAX) {
				pFont->sFontsize = SHRT_MAX;
			} else {
				pFont->sFontsize = (short)usTmp;
			}
			break;
		default:
			break;
		}
		iInfoLen = iGet6InfoLength(iFodo + iFodoOff, aucFpage);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
	return TRUE;
} /* end of bGet6FontInfo */

/*
 * Fill the picture information block with information
 * from a Word 6/7 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet6PicInfo(int iFodo, const unsigned char *aucFpage,
		picture_block_type *pPicture)
{
	int	iBytes, iFodoOff, iInfoLen;
	BOOL	bFound;
	unsigned char	ucTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pPicture == NULL);

	iBytes = (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 1;
	if (iBytes < 1) {
		return FALSE;
	}
	bFound = FALSE;
	(void)memset(pPicture, 0, sizeof(*pPicture));
	while (iBytes >= iFodoOff + 1) {
		switch (ucGetByte(iFodo + iFodoOff, aucFpage)) {
		case 0x44:
			pPicture->lPictureOffset = (long)ulGetLong(
					iFodo + iFodoOff + 2, aucFpage);
			bFound = TRUE;
			break;
		case 0x47:
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			if (ucTmp == 0x01) {
				/* Not a picture, but a form field */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		case 0x4b:
			ucTmp = ucGetByte(iFodo + iFodoOff + 1, aucFpage);
			if (ucTmp == 0x01) {
				/* Not a picture, but an OLE object */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		default:
			break;
		}
		iInfoLen = iGet6InfoLength(iFodo + iFodoOff, aucFpage);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
	return bFound;
} /* end of bGet6PicInfo */

/*
 * Build the lists with Character Information for Word 6/7 files
 */
void
vGet6ChrInfo(FILE *pFile, const pps_info_type *pPPS,
	const long *alBBD, size_t tBBDLen,
	const unsigned char *aucHeader)
{
	font_block_type		tFont;
	picture_block_type	tPicture;
	unsigned short	*ausCharPage;
	unsigned char	*aucBuffer;
	long	lFileOffset, lParmOffset, lBeginCharInfo;
	size_t	tCharInfoLen, tOffset, tSize, tLenOld, tLen, tCharPageNum;
	int	iIndex, iIndex2, iRun, iFodo;
	unsigned short	usCharFirstPage, usCount;
	unsigned char	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(alBBD == NULL);

	lBeginCharInfo = (long)ulGetLong(0xb8, aucHeader); /* fcPlcfbteChpx */
	NO_DBG_HEX(lBeginCharInfo);
	tCharInfoLen = (size_t)ulGetLong(0xbc, aucHeader); /* lcbPlcfbteChpx */
	NO_DBG_DEC(tCharInfoLen);
	if (tCharInfoLen < 4) {
		DBG_DEC(tCharInfoLen);
		return;
	}

	aucBuffer = xmalloc(tCharInfoLen);
	if (!bReadBuffer(pFile, pPPS->tWordDocument.lSb,
			alBBD, tBBDLen, BIG_BLOCK_SIZE,
			aucBuffer, lBeginCharInfo, tCharInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}

	tLen = (tCharInfoLen - 4) / 6;
	tSize = tLen * sizeof(unsigned short);
	ausCharPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 2) {
		 ausCharPage[iIndex] = usGetWord(tOffset, aucBuffer);
		 NO_DBG_DEC(ausCharPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	aucBuffer = NULL;
	tCharPageNum = (size_t)usGetWord(0x18e, aucHeader);
	DBG_DEC(tCharPageNum);
	if (tLen < tCharPageNum) {
		/* Replace CharPage by a longer version */
		tLenOld = tLen;
		usCharFirstPage = usGetWord(0x18a, aucHeader);
		DBG_DEC(usCharFirstPage);
		tLen += tCharPageNum - 1;
		tSize = tLen * sizeof(unsigned short);
		ausCharPage = xrealloc(ausCharPage, tSize);
		/* Add new values */
		usCount = usCharFirstPage + 1;
		for (iIndex = (int)tLenOld; iIndex < (int)tLen; iIndex++) {
			ausCharPage[iIndex] = usCount;
			NO_DBG_DEC(ausCharPage[iIndex]);
			usCount++;
		}
	}

	(void)memset(&tFont, 0, sizeof(tFont));
	(void)memset(&tPicture, 0, sizeof(tPicture));
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		if (!bReadBuffer(pFile, pPPS->tWordDocument.lSb,
				alBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				(long)ausCharPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
		  	lParmOffset = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
			lFileOffset = lTextOffset2FileOffset(lParmOffset);
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2, aucFpage);
			if (iFodo == 0) {
				NO_DBG_HEX(lParmOffset);
				vReset2FontInfoList(lFileOffset);
			} else {
				if (bGet6FontInfo(iFodo, aucFpage, &tFont)) {
					tFont.lFileOffset = lFileOffset;
					vAdd2FontInfoList(&tFont);
				}
			}
			(void)memset(&tFont, 0, sizeof(tFont));
			if (iFodo <= 0) {
				continue;
			}
			if (bGet6PicInfo(iFodo, aucFpage, &tPicture)) {
				tPicture.lFileOffset = lFileOffset;
				tPicture.lFileOffsetPicture =
					lDataOffset2FileOffset(tPicture.lPictureOffset);
				vAdd2PicInfoList(&tPicture);
			}
			(void)memset(&tPicture, 0, sizeof(tPicture));
		}
	}
	ausCharPage = xfree(ausCharPage);
} /* end of vGet6ChrInfo */
