/*
 * Copyright (C) 2008-2014 Free Software Foundation, Inc.
 *
 * Author: Simon Josefsson, Nikos Mavrogiannopoulos
 *
 * This file is part of GnuTLS.
 *
 * GnuTLS is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * GnuTLS is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GnuTLS; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

/* *INDENT-OFF* */

#define MAX_CHAIN 10

/* This is the same chain as modified1 but with no modification */
static const char *modified2[] = {
    "-----BEGIN CERTIFICATE-----\n"
    "MIIFXjCCBEagAwIBAgIQHYWDpKNVUzEFx4Pq8yjxbTANBgkqhkiG9w0BAQUFADCB\n"
    "tTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
    "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
    "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMm\n"
    "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwHhcNMTQwMjI3\n"
    "MDAwMDAwWhcNMTUwMjI4MjM1OTU5WjBnMQswCQYDVQQGEwJVUzETMBEGA1UECBMK\n"
    "V2FzaGluZ3RvbjEQMA4GA1UEBxQHU2VhdHRsZTEYMBYGA1UEChQPQW1hem9uLmNv\n"
    "bSBJbmMuMRcwFQYDVQQDFA53d3cuYW1hem9uLmNvbTCCASIwDQYJKoZIhvcNAQEB\n"
    "BQADggEPADCCAQoCggEBAJdfieOPrf4Arf1Iled/ii97407ZnjpaB5xxm49Q4Pz3\n"
    "+5xmD0LYre7Cjn1A7W3ZlHki5zFVZpW9Jb/3PfSEDY5slyjkLD2jdl2gVefSthSZ\n"
    "tYxb5eYv79tIEN0U9AZ8/VaGwUokl8n1MitcECxNLMe4LqoVmS29nXITTTzX5t3I\n"
    "4dUeMBDNI+xgVpJSpxwzA+/L+wxoj5Sb4YJ/Y+iUknCkjX6PpaZMRWBEE0dqvG02\n"
    "qlxXesAV0nmKYvjbtqAyoW6vgjP85h6gJEESIqTTZy1HOgFpO8XT05CpGDcjhP1s\n"
    "TvXF7Vx1aj+xDidGLLW188G35oFKIhEyHQV2V7vzRAUCAwEAAaOCAbUwggGxMFAG\n"
    "A1UdEQRJMEeCEXVlZGF0YS5hbWF6b24uY29tggphbWF6b24uY29tgghhbXpuLmNv\n"
    "bYIMd3d3LmFtem4uY29tgg53d3cuYW1hem9uLmNvbTAJBgNVHRMEAjAAMA4GA1Ud\n"
    "DwEB/wQEAwIFoDAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYBBQUHAwIwQwYDVR0g\n"
    "BDwwOjA4BgpghkgBhvhFAQc2MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZl\n"
    "cmlzaWduLmNvbS9jcHMwHwYDVR0jBBgwFoAUDURcFlNEwYJ+HSCrJfQBY9i+eaUw\n"
    "RQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL1NWUlNlY3VyZS1HMy1jcmwudmVyaXNp\n"
    "Z24uY29tL1NWUlNlY3VyZUczLmNybDB2BggrBgEFBQcBAQRqMGgwJAYIKwYBBQUH\n"
    "MAGGGGh0dHA6Ly9vY3NwLnZlcmlzaWduLmNvbTBABggrBgEFBQcwAoY0aHR0cDov\n"
    "L1NWUlNlY3VyZS1HMy1haWEudmVyaXNpZ24uY29tL1NWUlNlY3VyZUczLmNlcjAN\n"
    "BgkqhkiG9w0BAQUFAAOCAQEAOeZfjkI0yR/nutCMHp5/uB/evkB8qIYxh1KKbhPB\n"
    "TmpykmJxiLKrBBcBBu9kW5lMbNPSNclE4sCyN0dxCJHwPm7ubNUxsmDSTPTJOx0M\n"
    "zl0WZVaZ7eX3nw1kj8jSoK0f5n87RzKK85MwBFsEn73Z2pDvxTcd72BE0T1UJLcU\n"
    "2A5uHAJyvm2QpOWBIRKlJHIHCcu5xjj5yLnGO9id0cjOjEgj9f1Mo4hzawL5vQfk\n"
    "o/xFxAsA70bk2trv54kgLvhmAW+B6OpN3Z/xB4mWNOw3G8bg/u+pCLvd8KRO5V8K\n"
    "TlgO1NTaOgYC6OAF3USNMhuNZh6h5tWA3mA8rFr8ZsayhA==\n"
    "-----END CERTIFICATE-----\n",
    "-----BEGIN CERTIFICATE-----\n"
    "MIIF7DCCBNSgAwIBAgIQbsx6pacDIAm4zrz06VLUkTANBgkqhkiG9w0BAQUFADCB\n"
    "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
    "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
    "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
    "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
    "aG9yaXR5IC0gRzUwHhcNMTAwMjA4MDAwMDAwWhcNMjAwMjA3MjM1OTU5WjCBtTEL\n"
    "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
    "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
    "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMmVmVy\n"
    "aVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwggEiMA0GCSqGSIb3\n"
    "DQEBAQUAA4IBDwAwggEKAoIBAQCxh4QfwgxF9byrJZenraI+nLr2wTm4i8rCrFbG\n"
    "5btljkRPTc5v7QlK1K9OEJxoiy6Ve4mbE8riNDTB81vzSXtig0iBdNGIeGwCU/m8\n"
    "f0MmV1gzgzszChew0E6RJK2GfWQS3HRKNKEdCuqWHQsV/KNLO85jiND4LQyUhhDK\n"
    "tpo9yus3nABINYYpUHjoRWPNGUFP9ZXse5jUxHGzUL4os4+guVOc9cosI6n9FAbo\n"
    "GLSa6Dxugf3kzTU2s1HTaewSulZub5tXxYsU5w7HnO1KVGrJTcW/EbGuHGeBy0RV\n"
    "M5l/JJs/U0V/hhrzPPptf4H1uErT9YU3HLWm0AnkGHs4TvoPAgMBAAGjggHfMIIB\n"
    "2zA0BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZlcmlz\n"
    "aWduLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEAMHAGA1UdIARpMGcwZQYLYIZIAYb4\n"
    "RQEHFwMwVjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2Nw\n"
    "czAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQG\n"
    "A1UdHwQtMCswKaAnoCWGI2h0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMtZzUu\n"
    "Y3JsMA4GA1UdDwEB/wQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglp\n"
    "bWFnZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNo\n"
    "dHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjAoBgNVHREEITAfpB0w\n"
    "GzEZMBcGA1UEAxMQVmVyaVNpZ25NUEtJLTItNjAdBgNVHQ4EFgQUDURcFlNEwYJ+\n"
    "HSCrJfQBY9i+eaUwHwYDVR0jBBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJ\n"
    "KoZIhvcNAQEFBQADggEBAAyDJO/dwwzZWJz+NrbrioBL0aP3nfPMU++CnqOh5pfB\n"
    "WJ11bOAdG0z60cEtBcDqbrIicFXZIDNAMwfCZYP6j0M3m+oOmmxw7vacgDvZN/R6\n"
    "bezQGH1JSsqZxxkoor7YdyT3hSaGbYcFQEFn0Sc67dxIHSLNCwuLvPSxe/20majp\n"
    "dirhGi2HbnTTiN0eIsbfFrYrghQKlFzyUOyvzv9iNw2tZdMGQVPtAhTItVgooazg\n"
    "W+yzf5VK+wPIrSbb5mZ4EkrZn0L74ZjmQoObj49nJOhhGbXdzbULJgWOw27EyHW4\n"
    "Rs/iGAZeqa6ogZpHFt4MKGwlJ7net4RYxh84HqTEy2Y=\n"
    "-----END CERTIFICATE-----\n",
    NULL,
    "-----BEGIN CERTIFICATE-----\n"
    "MIIExjCCBC+gAwIBAgIQNZcxh/OHOgcyfs5YDJt+2jANBgkqhkiG9w0BAQUFADBf\n"
    "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
    "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
    "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
    "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
    "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
    "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
    "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
    "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
    "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
    "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
    "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
    "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
    "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
    "AAGjggGRMIIBjTAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
    "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjA9\n"
    "BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVy\n"
    "aXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwNAYD\n"
    "VR0lBC0wKwYJYIZIAYb4QgQBBgpghkgBhvhFAQgBBggrBgEFBQcDAQYIKwYBBQUH\n"
    "AwIwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAHBgUr\n"
    "DgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVyaXNp\n"
    "Z24uY29tL3ZzbG9nby5naWYwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhho\n"
    "dHRwOi8vb2NzcC52ZXJpc2lnbi5jb20wDQYJKoZIhvcNAQEFBQADgYEADyWuSO0b\n"
    "M4VMDLXC1/5N1oMoTEFlYAALd0hxgv5/21oOIMzS6ke8ZEJhRDR0MIGBJopK90Rd\n"
    "fjSAqLiD4gnXbSPdie0oCL1jWhFXCMSe2uJoKK/dUDzsgiHYAMJVRFBwQa2DF3m6\n"
    "CPMr3u00HUSe0gST9MsFFy0JLS1j7/YmC3s=\n"
    "-----END CERTIFICATE-----\n"
};

static const char *nc_bad1[] = {
/* DNSname: localhost
   DNSname: www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDSzCCAjOgAwIBAgIMU/xqxDpxZ3J5cUcrMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTEwIhgPMjAxNDA4MjYxMTA4NTJaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDP3GV/JSLCCmx09hJINJZC8fSUBE1IVbZsY/q00rZOw5KwPioLiMOIup7W\n"
"na8YJ2ama0GJjU86PlJDhBH6soaY24ZCW5kKhvfnSw2TkpW6umL7psvuBslRoMxA\n"
"t12MpapZDFZixZjV44Bstuyt9sI1ze3au+5C7E4+z40o/3uvbIiN2iz4bPgwPIMu\n"
"5V/bVTei6uAcu4fNHh/AGnAUJa201QsUhM4+VRFc0XmanjSulySD3obwERDneqab\n"
"77gnIP6zwuFXxHgucbmzU8DIgVhes2k4v6AB1nPxlpUL5+E+W4XDg4ckGGkfxgcn\n"
"dGYvuv3pwIyHvb7Z0A8D6bE435cnAgMBAAGjgZ4wgZswDAYDVR0TAQH/BAIwADAl\n"
"BgNVHREEHjAcgglsb2NhbGhvc3SCD3d3dy5leGFtcGxlLmNvbTATBgNVHSUEDDAK\n"
"BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBS5dbTqiPbhe7Fg\n"
"e/Bs92qsNgWSzjAfBgNVHSMEGDAWgBSgAJcc9Q5KDpAhkrMORPJSboq3vzANBgkq\n"
"hkiG9w0BAQsFAAOCAQEAdkVCQQ5mCuV5WTqCeH7AtXxYM7IV7q6O7uqirKwDUzGD\n"
"wB5shurAPMn9IG46O68P+BPLMfwszFDgszNrjuMsKb7hLT5+pVsd6XRQeimPJ5rN\n"
"/szMfALLAcw8Yxt+1YbxlgUgybFsiK2zdBpAshU+FzMRvFfq9rnqW/VXM0J6ghz3\n"
"VqLbviOY5KpCLzfG0yM+CTcKXVFau9QZK962AfXzUwaCymw1cRHzQlpdMQtTtcIp\n"
"nci6MKXViEdeHbPLcZe9+vzSSpFh5u/l47w+2B1oz7mndFFpxkw37zDaVH5yAFxK\n"
"+5VijiKxH6nmniLUX8Zsv82YBaO0liNb2fOZopxQGQ==\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical): (empty) */
"-----BEGIN CERTIFICATE-----\n"
"MIIDFTCCAf2gAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEwODUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC+iPUnEs+qmj2U\n"
"Rz8plNAE/CpeUxUfNNVonluu4DzulsxAJMN78g+Oqx+ggdkECZxHLISkzErMgiuv\n"
"bG+nr9yxjyHH2YoOAgzgknar5JkOBkKp1bIvyA950ZSygMFEHX1qoaM+F/1/DKjG\n"
"NmMCNUpR0c4m+K22s72LnrpMLMmCZU0fnqngb1+F+iZE6emhcX5Z5D0QTJTAeiYK\n"
"ArnO0rpVEvU0o3nwe3dDrT0YyoCYrzCsCOKUa2wFtkOzLZKJbMBRMflL+fBmtj/Q\n"
"7xUe7ox62ZEqSD7W+Po48/mIuSOhx7u+yToBZ60wKGz9OkQ/JwykkK5ZgI+nPWGT\n"
"1au1K4V7AgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8wEgYDVR0eAQH/BAgwBqEE\n"
"MAKCADAPBgNVHQ8BAf8EBQMDBwQAMB0GA1UdDgQWBBSgAJcc9Q5KDpAhkrMORPJS\n"
"boq3vzAfBgNVHSMEGDAWgBQ/lKQpHoyEFz7J+Wn6eT5qxgYQpjANBgkqhkiG9w0B\n"
"AQsFAAOCAQEAoMeZ0cnHes8bWRHLvrGc6wpwVnxYx2CBF9Xd3k4YMNunwBF9oM+T\n"
"ZYSMo4k7C1XZ154avBIyiCne3eU7/oHG1nkqY9ndN5LMyL8KFOniETBY3BdKtlGA\n"
"N+pDiQsrWG6mtqQ+kHFJICnGEDDByGB2eH+oAS+8gNtSfamLuTWYMI6ANjA9OWan\n"
"rkIA7ta97UiH2flvKRctqvZ0n6Vp3n3aUc53FkAbTnxOCBNCBx/veCgD/r74WbcY\n"
"jiwh2RE//3D3Oo7zhUlwQEWQSa/7poG5e6bl7oj4JYjpwSmESCYokT83Iqeb9lwO\n"
"D+dr9zs1tCudW9xz3sUg6IBXhZ4UvegTNg==\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEwODUyWhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC2cd2vlg/9nuzi\n"
"6S6/qyJRnaUoFutajTwGqfQKZpbqXI7TcFZwKidzAZlZiU+sAvkY8d/9cadlbrde\n"
"S9HGv31QmexWjgWAMGNpeyiPlXIN8xGzIbZRM3FCih0bnIyibdwgAuU14dUrChGD\n"
"sQ4SAmRUpThkB8anvC10PIsxfnifBwJI6dGQZb1KOxVOIWg7Gb5tNFkZILBGv8wk\n"
"cbycIBYC3lRX8svUj9mMiro53f+4ZGbi4DcSLIdw4ebAczfBd+uHM2jkHFZUNuAY\n"
"7rGZAAuqEh5IE0QHS9CV6mg6Pf9+sLGMBZUbix2sxRntAEyz8+kO7W2zgmKPla4+\n"
"y54cIUmBAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUP5SkKR6MhBc+yflp+nk+asYGEKYwDQYJKoZIhvcNAQELBQAD\n"
"ggEBAC4N4dUoGDTGf93DtjEcZzqGoAd6TsCYy6k5zeYMZFwogfArg0IMurcFDLeN\n"
"PXe3xl9RiUjbiZMaHWju81kIO34z0NLd705XR9QFEc+xiuZOMmm4SxciAF5xo+Hh\n"
"Fhc9cVa8Icm2ju86Q4yhJziYrElH8VwHTBE0k+RE1cK65F5PQFGGBlpGm9EMcYTv\n"
"EQQATPLuWwKRAFNJBx2t3DAeMseo/Iq6Snd/UfdqgLkV61YtbzqL8bu+a8rgMAYz\n"
"ovgORsI48TlbU4H7YI+vzPO33tRV2m4dOxppMHzv8Ie2LIIfqYn0HRd87c06djEA\n"
"EpXfXGqxjX5vAtNPO5fGGzghol4=\n"
"-----END CERTIFICATE-----",
NULL
};

static const char *nc_bad2[] = {
/* DNSname: www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDQDCCAiigAwIBAgIMU/x5DBI1pGSO2eYZMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDA4MjYxMjA5NDhaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDL/hbj+RVDQ5sA4VR2FZ+P+/jju7jkUFUovwhCrWpGXFs0WDokcOkcXc3f\n"
"0yturFWazVEKaaZECiDpGf6iXmNoJA6fPT+G0gPlIL6wh4wKQG+vwVYsX5ZkXOMR\n"
"sl7BqCpeCChkth54mClEwOYW3WohdXqiJfxoFVdgnisbqkxYtz9aXzYE71cZIFAx\n"
"nL7V/gY+G/m0iZCdfh7YEDlT+qtLkGyHsyyTxwUH4yyqcsFl4WWG6wAdKF5U69yw\n"
"uo61J5wpE+yDyS0u4Cjw67d29OIHsT7GAq+fP69vMoEHPvPUM/aA68AycybV2OYt\n"
"8OJAyZqf/6zvnlrbLuk08kWf1TD3AgMBAAGjgZMwgZAwDAYDVR0TAQH/BAIwADAa\n"
"BgNVHREEEzARgg93d3cuZXhhbXBsZS5jb20wEwYDVR0lBAwwCgYIKwYBBQUHAwEw\n"
"DwYDVR0PAQH/BAUDAwegADAdBgNVHQ4EFgQUz+XUeM8bwvK3BD7NVRRYSeZSlA0w\n"
"HwYDVR0jBBgwFoAU/n2NeCCnwHQwFpKwi42A3H+w9W8wDQYJKoZIhvcNAQELBQAD\n"
"ggEBAPJZO19PhmxQ1ZRDLQWZWxuQgD2Wwy8sS+wnlUo/TZM7+pT5ICjLdEKgETPd\n"
"HOqgAMQFaUpp5vx1jUBmmKdPOmwEnV/2zbw3GrYwAQjxunXD66iHjYbodl9zBumM\n"
"NXDGsHnKYNu9sPdQSMLC7OEOrKvEhH2afOvYDORQbSGXh7+3js7Mzggy0NoYtxnK\n"
"4wqt6g73SFkV82mTQpUBK218ROjuWVBUmWxq2JU+qvsAKbhz+Tjr9+kmFcNBRgmA\n"
"Zga26uoQhd6YP9DKbCvf3sK4bi6A5NROeLf9BzJHWkGani4F9wOjRmLVnLlB3BPi\n"
"tHZaLDU9fUnf6I6p3nu6LPTH3JU=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwODI2MTIwOTQ4WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQD6kCE6KmDCkagX\n"
"fvq0rx73h7zn23phJYBmugfp5fPQYIJ1463dGmlFK4Lfkz2V4StgM2mbFFEkcyVC\n"
"pz+PyeSDgyWJJ/RwG690lLfu5JfNLvwxj/rFNK6rS7EpADew6RgURCpEMt6z1uEk\n"
"+IQsxKoXQmAcdtc/ubPFWInotg7Avoid0sG69s/+hq/nlGE9A8JMFnsLh/n01d+F\n"
"9dWsjrNiZ+mfTE8w0MVTq4+8mvmPmnjKsiu0rgqaVTmYpZW5chz9gGrZCr/Wr5CL\n"
"zPsAYaWie+wo8cR5qMEoX+JPHqM8eP9K1v+uYc03aD3u1/QYdxY73OLn31+jYAqq\n"
"tRJjgSERAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oA8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU/n2N\n"
"eCCnwHQwFpKwi42A3H+w9W8wHwYDVR0jBBgwFoAUJ2ngSf1CqeYVG3eUNcH7Lt6/\n"
"ECUwDQYJKoZIhvcNAQELBQADggEBAIQruSBUyQ268Js61XT3nlr1Y9HxmKA2DIuM\n"
"WcAicE1XIpuxDpZ/VmKH0/o2JVR1A7uwSMEnHdShHixMbpYrHRDnZITxs2lsJijE\n"
"r7YdqadH7EbjDIXv1DJcPnNaeqFPbyXEWqLYoQf6UPBLVRWeKISPN0hMaIZv4Y/X\n"
"OcBceajAr0XGxASRFDky26M01AVPZoYjgT7vLp835yk9BY5+q0GxlSJl6HbQ5ESA\n"
"IoC3Limt72niobmvEryQDq7qUUoR7hB1SMKfyX/qktxT3UCBLKXHsp80ECJ2A7Sd\n"
"YrHjFE6LnWHwGJFYZ1eYKiOjglVRGv3+bNX07bQBWKzRbLWYM+0=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwODI2MTIwOTQ4WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDUbGpRRMdnTzbG\n"
"R7J9qYJoHOhgRp3lEa9sBP7/pNNIsEhXmvzBu5J0buJpAfRPmWcoZauKsVrKnE69\n"
"CFlTDVCIahQ5gtJkGdjrDrQDFFCMnKC04Lhq+EmbASBTn5GRQwJqEUi9xRpj0yOL\n"
"0XGlMp4JS44eAL4giywzPtOAZaJlr4kdOnSPK2SHFVwQGfQiNmzD5ajmsjM3k4o3\n"
"R2gXAsudyasQzRAjFyeo2ry7klPUPS5RHJ6B2n87e9kLGrYb8+O9I9FNc/w4J49W\n"
"AovVr5vcs9Km25jLUn43KDprDhpXddEraz6WyZJRMTZVRRUizET3gmojZFFD4zOQ\n"
"mneVYerpAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oQ8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQUJ2ng\n"
"Sf1CqeYVG3eUNcH7Lt6/ECUwHwYDVR0jBBgwFoAUyFGHFFLCWYOQTLmh8jJpGyxS\n"
"bSUwDQYJKoZIhvcNAQELBQADggEBALGFCZXC1KPBBPMtLJNfhNBtBBC4i5q+1Qeo\n"
"aJL7dKVuBn79WAuND8rvJvrPKpGTmyxkcnqRXSBVH3c+Xi+v5ykLrtHJ2x4TOrmg\n"
"RBAaBqDuecQ9Ec0dCc5ODKwjdI/wEOGAS4sfrMXzQCv+UJqi2lE0fo/xDmS/azCc\n"
"WUjFSQOuWnCJIIAIyWlF2bPtdtiaydHKkTcG7c/zwrxRaWE2Q2G+dm+itpJ7sCtx\n"
"ZFfGMLUl7mDadhiYrxq1SnwrObMwbngPNZyUBi2G7jnXlyFc9X/w6fVIULLxN+bn\n"
"IzHWcRrBZ/ShdvCStmgbTlKNtvg0LWAk7QWzy2ibaXS5jp2r+Fc=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.net
   Excluded: DNSname: example.org */
"-----BEGIN CERTIFICATE-----\n"
"MIIDMTCCAhmgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTIwOTQ3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDsnSNH0g4bvAXx\n"
"zJAfs+XFWpNIpJtD/1H3Ei0ezfH5Ug7GNpHKlYKLCIShqCFj9WSwzSyKNGCHafdt\n"
"PPhKo9uAo2bSaBZjmzxfqSOGDEXZ+4LlRgSPX2Arc0i97ZsPf0nkWLfrxlTOAQIm\n"
"DxDNrWlGrCl1dfPiB+EyMzo+++MCdTGEsdEdRFm85QNjOOTiaTeUpUh5G27+hbuk\n"
"PaRlZ1GHJYlrLHK/2qw9/Mw+gNnfn/Efw+lNeYuQ3tco8IAMN0jB8x1hDfOxTx93\n"
"mrFzAdGTfsYZc31YapATk2re8IJGeKSCY4XP2HvYZEE1fYdw8ZcqZ/Gv1RdXyxvc\n"
"6oT5r/PNAgMBAAGjgZMwgZAwDwYDVR0TAQH/BAUwAwEB/zAsBgNVHR4BAf8EIjAg\n"
"oR4wDYILZXhhbXBsZS5uZXQwDYILZXhhbXBsZS5vcmcwDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUyFGHFFLCWYOQTLmh8jJpGyxSbSUwHwYDVR0jBBgwFoAUTBVq\n"
"WCSLkLZte75Q9bgKeM851qowDQYJKoZIhvcNAQELBQADggEBAGKVEQfAzrWj8wmQ\n"
"l4sm+i/pgK0I07jNMvgUDsvAmjlkndWxoX1ROe0Nd2I3d5te0+G9MR6CTOByr8VE\n"
"NLyXEsrk++BsfLk/0UNFHwq97QLTzzyEXQYQnDza8R1jdlr7XpGZOoWczi08yMAk\n"
"UiJyq2xaqerTlIYp01T9a3Nb5tWFyUVekJeyJQakj2VLaKkl4hCfK3h/HFBNJ3yf\n"
"AvBu77wQeh6n8osNDCpW9e1KRAGisDCFrTMUlyxQIK/OXhjLzu7qDKShdNnfNRmc\n"
"H4W9ODLIm8AX1S0udg9OebPhNWfM2bDFzI/dIX+yHp6q0oepbT11rKG8G+5M25uU\n"
"AUhTFC4=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTIwOTQ3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCZSS6SaBALkN4W\n"
"Tao6pqaDmi1edtBskNHdO/lYomElQtIkw6wQHlJjIdjF3vnhEWU9+HOIZrWKth4X\n"
"u+naDZquOb4GWPq/X6/KBoQ4hq/XZJaFEDPeciNcVeylWVlHi1OeGm8uHZxAK/6d\n"
"wpGoe/0K+QaLFdbm/srw1LGvCwbLwNDKePX9TgOfVKdZtGZUdDDo6TXUmhNG+QeP\n"
"7Fv1n2PjQFkXiRwVLgJj06DvR+ft81x2gjEVS+vxWg0+cbJvBI2ItpNGnIWvbwl7\n"
"BTyNRjvsi7ljFn+SfaRBLXE4aygQFQ9UCHNNYtkBO73BXv/SgcFXzSDDN5ZMfpg9\n"
"SSWkEApFAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUTBVqWCSLkLZte75Q9bgKeM851qowDQYJKoZIhvcNAQELBQAD\n"
"ggEBAH3bezfaVpnyqZRJsZ8sHAIpJWa03mHl/mqRpT0qg45Agzwo7yb5dubiHUBJ\n"
"BK/kAMhICjHAH+6E5XubGVSFvbzBX1FeKQQgzghN4niosOCLZPUtl8gJGZlsOoCy\n"
"6HldkxXa26GBZR0NvJb/p83VA7w5Zlp5j7Rp2VkWwRniaPex39dogDX3IwnoZKzL\n"
"ogyeNQPG2qLDBdZRAVng0eJK1Ml5PHxoEkcFwFsxd4B1cJV0VCMk7X7oEc9qBtUB\n"
"Ye/bst72puWDK1lBhT6EFhDDbY9xKm7pvUkGx80gWm9JZ0xGCaoM4tyEAaCd9tYZ\n"
"JFvnIEGJGeGjlRLJZGS4mZ/Q5mI=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *nc_bad3[] = {
/* CN=www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDPDCCAiSgAwIBAgIMU/xvqR+qZTQTaWIIMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTEwIhgPMjAxNDA4MjYxMTI5NDVaGA85OTk5MTIzMTIzNTk1OVow\n"
"LTEYMBYGA1UEAxMPd3d3LmV4YW1wbGUuY29tMREwDwYDVQQKEwhzZXJ2ZXItMjCC\n"
"ASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJ46qo4PFrBfYlQXSgtbk/rO\n"
"0aO8/Gh/bIh4c/JX0RfqC55bnfbkO9SwwR+eU76INyVb1exmd7qsl4R2EgS8V3Gf\n"
"3K5k4tNecMfxT98MWiuSSp8Q8+affUF5t9TSLujL1dckMlPfH9hdxCYhJGH51mkf\n"
"wr3oEmwNXsA9OQ8oxq2i8WxQTJGUXkwx/k2L2NRF3L8vjRnXRfKSISkkDXeKYMvo\n"
"V5ElQwlKo0sonttUIGOVav8Cf4GnFQzSJW+RfANTniGIq16jE+flKz1kQYRLLoeA\n"
"fgH/1vI1v5xqMURNW/BQlawAE0HGj4MAyfebhsWmhqmcNqGBf1OfHMNdB1vamGkC\n"
"AwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDATAPBgNV\n"
"HQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBRiCra2BJERmr2/+Klot32criLTbTAfBgNV\n"
"HSMEGDAWgBToCEW507CZ42L/fn7H+DLcx+zW/DANBgkqhkiG9w0BAQsFAAOCAQEA\n"
"fsG45/VuJzw5DRbrE6o67T3EgFfPCzr+xc0JmTJSCHvWIx+2O1VspmJiArNTwQ5O\n"
"l8Hq2Sag9Wi0cyRC8lVKPbC7Im2fZ4m4endOhiEmaOHBCru5bIFRwDvtG3u+yEYI\n"
"rzRU+6PdwmLYwc+ks8qEqACw772nElJxOWXmYEMtFpYh8eujfzjmUGIJyTotrm72\n"
"WX8phKA/xogZaSLD21t8u77PE/JEcJ2LXAa9dq6pGYru1vyuRqq8ZeWiVAAqD6hZ\n"
"cglKk8dLi6esywQMGEGqhRx9y1A0mPZO+M599GOgWTbShUB3pUyaLLLLnD9Dciwq\n"
"4E4iP9rdfgStOfz12BsKOw==\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Excluded: DNSname: example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDIjCCAgqgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEyOTQ1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDW2Z64Ax+MNj1j\n"
"2QM9hjP4ybs+mz51vfDAlDVVdh2bzJOHyjTIlzI71QloH9aPuv5d92tTxe7/7afY\n"
"fWC61AF0WKj7x8h570IW0Zye4ITEnDLlqk5Bn46IP9fWDq5xYVEGMaYT4l409Fyw\n"
"JUZjfXqMefAXhj55wZoz+WMM0AB5LABlojLkV+iPMVJgfYWhcVijd92Yebp8R2/+\n"
"z1nF0vQtV01tatWTEiJajPRHZCwVe71rXEf02nYiqCw5RwLZrsug5LZ+K8LoBbeE\n"
"ezcJT5y8uf4mpTmTj2Po7Kby22yl1wkVV925a2Of7ufDL3d56SIM1foNXAAmlFar\n"
"M5Y9hIZLAgMBAAGjgYQwgYEwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHR4BAf8EEzAR\n"
"oQ8wDYILZXhhbXBsZS5jb20wDwYDVR0PAQH/BAUDAwcEADAdBgNVHQ4EFgQU6AhF\n"
"udOwmeNi/35+x/gy3Mfs1vwwHwYDVR0jBBgwFoAUWU1t/YrnYJGhEfuoDj42A4ui\n"
"j0gwDQYJKoZIhvcNAQELBQADggEBAMj2jjHmYLWxGMkLOIQ/MGtvchZ+v6vmEl7m\n"
"GaoHkz0sxFeJqs7mwcybvwG4tlHB/PhaLTH4HfN6PLNbRA4oamr3gFsEtd/JRihw\n"
"X/5CvdJdu/d7uN36yrD5ZTJmt5v1sAXqzkVYXHUSQLOLTIVfwQfUv8IrxTWgbhNI\n"
"mIi55bjCyOWYzZsZ5kTDNFcBkoYiMks2fVuUdP8xrxoweedVswUdkwg1TyWLikG3\n"
"47VuQP3eA7+zEkFUeywG89DTOpDURAlvBzaVTjKn++3RgH/A4Wa+MX6HTHXjxBIU\n"
"1uGcMjhPjc99F81RaYdIlFsQiQ74b5RwdSvGo0e67ssgar0XKgw=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTEyOTQ1WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDYTAYDM0lZ+RMg\n"
"M1M3q4n6H/VebQ1CG1ztkinGzj7eH+fNyi9Wq5EqB/0S7jVPEuD0o5jBrwI6XFoS\n"
"MQiWtqWHGh47qijX8y/oc75Sn/2b1gGF3zDWM9LygZZW2+QOIrvK5TcU+rAmXKsA\n"
"765z0nTIbL3vNr9n0yEM3E13tk3Qjqx+OLhJ/ZyLKW+w+BuhLp79LcVtjNnlVfvC\n"
"nVgLvo69YGdJxhPUjjVqKwTlvptyzELQSSQMenPmvhz2kRXjQ/6jog4tb1qkzfpP\n"
"eYB0MVgSLeWBgNF3VLTSH06RHvXEQcdP2e3AR67sJxd6UJ4vOo1widQs0yWTZpCB\n"
"ZJawOPqDAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUWU1t/YrnYJGhEfuoDj42A4uij0gwDQYJKoZIhvcNAQELBQAD\n"
"ggEBACUbydVQKZi5ulzn/lQK5x/DZySJflrevZ1plV7BVBhZhlvBO0VARGNn+NW5\n"
"G9RqY+itmyBbW/Fl43gWiHQynYneK9tYBub7WeJqr9iTX4zvI7V8fk/vbyfVRODX\n"
"cJ8JzeLYqi6Hm1PK7Q9dz0rgyulXeuCyDeQ4jzoGIm2l7atUoGZB0f9YCJyeV2ew\n"
"t8jMZr2sSVMgvT87S/EHMe5q5YAJQzDBAadH64icaxW3e03UeH6JYblohsZVQTIE\n"
"wl60jozIStml73oyocfytsErDdKArrSSHxHaygAqoVu+9O5U90vwK6VDuGF0YzZj\n"
"ZKOAu2HuFHpCMbYzUYi3FMOUU5k=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *nc_good1[] = {
/* DNSname: www.example.com */
"-----BEGIN CERTIFICATE-----\n"
"MIIDQDCCAiigAwIBAgIMU/xyoxPcYVSaqH7/MA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
"BgNVBAMTBENBLTMwIhgPMjAxNDA4MjYxMTQyMjdaGA85OTk5MTIzMTIzNTk1OVow\n"
"EzERMA8GA1UEChMIc2VydmVyLTQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
"AoIBAQDkemVOFdbhBX1qwjxQHr3LmPktNEVBmXjrIvyp++dN7gCYzubnpiLcBE+B\n"
"S2b+ppxBYm9ynKijhGrO+lZPCQRXWmqUg4YDfvnEqM4n04dCE98jN4IhwvWZyP3p\n"
"+U8Ra9mVIBAY2MReo1dcJQHNmo560xzxioHsGNQHAfYgVRHiE5hIXchYbWCkBrKt\n"
"XOoSSTmfgCF3L22p6S1q143VoKUr/C9zqinZo6feGAiTprj6YH0tHswjGBbxTFLb\n"
"q3ThbGDR5FNYL5q0FvQRNbjoF4oFitZ3P1Qkrzq7VIJd9k8J1C3g/16U2dDTKqRX\n"
"ejX7maFZ6oRZJASsRSowEs4wTfRpAgMBAAGjgZMwgZAwDAYDVR0TAQH/BAIwADAa\n"
"BgNVHREEEzARgg93d3cuZXhhbXBsZS5jb20wEwYDVR0lBAwwCgYIKwYBBQUHAwEw\n"
"DwYDVR0PAQH/BAUDAwegADAdBgNVHQ4EFgQUAEYPmcA7S/KChiet+Z6+RRmogiww\n"
"HwYDVR0jBBgwFoAUjxZogHO3y4VdOLuibQHsQYdsGgwwDQYJKoZIhvcNAQELBQAD\n"
"ggEBABlA3npOWwl3eBycaLVOsmdPS+fUwhLnF8hxoyKpHe/33k1nIxd7iiqNZ3iw\n"
"6pAjnuRUCjajU+mlx6ekrmga8mpmeD6JH0I3lq+mrPeCeFXm8gc1yJpcFJ/C2l4o\n"
"+3HNY7RJKcfoQxIbiKOtZ6x9E0aYuk3s1Um3Pf8GLwENoou7Stg5qHsLbkN/GBuP\n"
"n3p/4iqik2k7VblldDe3oCob5vMp0qrAEhlNl2Fn65rcB4+bp1EiC1Z+y6X8DpRb\n"
"NomKUsOiGcbFjQ4ptT6fePmPHX1mgDCx+5/22cyBUYElefYP7Xzr+C8tqqO3JFKe\n"
"hqEmQRsll9bkqpu2dh83c3i9u4g=\n"
"-----END CERTIFICATE-----\n",
/* - */
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBAzANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0y\n"
"MCIYDzIwMTQwODI2MTE0MjI3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTMwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC/4ofaL+ilmmM+\n"
"bGaFRy5GYQXtkD8sA3+/GWsunR928fQS68Zh6iWU+gPm52i7Gfbh7piKWA5Tb63w\n"
"unbS6dPsfPSvgRMZGKJpzxqVcBQAnTS4MuDPlXNg3K3HMyVtbxekII8jFeGEJuCL\n"
"mBMT4dI48IZRzj+2mir38w2cQPfomaKtjg2jMokG8Z9/4+SU9VJCcY1/yZk8fCbS\n"
"dBbwhnDq10yvhPCHgX6KMYmoJr28CYgH29Q9sDP1XN3VvAx5X+PtW/6pyF0U5E2e\n"
"gRzVv7Hr3FJKvytbNxRMCoy2YOyvsTP0fIhiXdtkulTKXyiq4cxA+aYByOu1FjU4\n"
"NicWbiZ/AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQUjxZogHO3y4VdOLuibQHsQYdsGgwwHwYDVR0jBBgwFoAUwAx0\n"
"aL2SrsoSZcZUuFlq0O17BSgwDQYJKoZIhvcNAQELBQADggEBAGQvj8SquT31w8JK\n"
"tHDL4hWOU0EwVwWl4aYsvP17WspiFIIHKApPFfQOD0/Wg9zB48ble5ZSwKA3Vc3B\n"
"DJgd77HgVAd/Nu1TS5TFDKhpuvFPJVpJ3cqt3pTsVGMzf6GRz5kG3Ly/pBgkqiMG\n"
"gv6vTlEvzNe4FcnhNBEaRKpK5Hc5+GnxtfVoki3tjG5u+oa9/OwzAT+7IOyiIKHw\n"
"7F4Cm56QAWMJgVNm329AjZrJLeNuKoQWGueNew4dOe/zlYEaVMG4So74twXQwIAB\n"
"Zko7+wk6eI4CkI4Zair36s1jLkCF8xnL8FExTT3sg6B6KBHaNUuwc67WPILVuFuc\n"
"VfVBOd8=\n"
"-----END CERTIFICATE-----\n",
/* Name Constraints (critical):
   Permitted: DNSname: example.com
   Excluded:  DNSname: example.org
 */
"-----BEGIN CERTIFICATE-----\n"
"MIIDMzCCAhugAwIBAgIBAjANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0x\n"
"MCIYDzIwMTQwODI2MTE0MjI3WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDIf3as4EONSgWu\n"
"Mbm9w3DbKd/su1UWlrYrcpVqmU3MKD5jXBxyoThSBWxmq1+wcNDmE1on6pHY1aad\n"
"k3188JKMC83wEcyQXaiH3DlTYFXXkkI+JJNUGlfAMSoXG248SpoCIOhCETUG03iP\n"
"Z3AZludaHYsv4akAh1Kl6qn66+bKM53l/YhoQDxhoGaYvO8ZSwKnx5DEiq447jpW\n"
"M+sUFe38RPaMjHpyc1GRctvQDzJGm+8ZRujYDH+fGNzVDDlRyRnsVanFGNdyfhmy\n"
"BN2D2+2VEvzAWlaGg2wQN8gF3+luavIVEgETXODZPa5FF7ulmQmhqGrZcw6WtDmY\n"
"hUbNmbL7AgMBAAGjgZUwgZIwDwYDVR0TAQH/BAUwAwEB/zAuBgNVHR4BAf8EJDAi\n"
"oA8wDYILZXhhbXBsZS5jb22hDzANggtleGFtcGxlLm9yZzAPBgNVHQ8BAf8EBQMD\n"
"BwQAMB0GA1UdDgQWBBTADHRovZKuyhJlxlS4WWrQ7XsFKDAfBgNVHSMEGDAWgBTg\n"
"+khaP8UOjcwSKVxgT+zhh0aWPDANBgkqhkiG9w0BAQsFAAOCAQEASq5yBiib8FPk\n"
"oRONZ4COgGqjXvigeOBRgbHf9AfagpoYDbOKDQS8Iwt9VHZfJxdcJ1OuM1aQqXlN\n"
"dUyf+JdR/24Nv1yrhL+dEfRGka6Db96YuPsbetVhNIiMm2teXDIPgGzAKuTm4xPA\n"
"6zyNVy5AwfDQ5hIZ+EUsfOoerIElNyAbh66law4MWuiv4oyX4u49m5lxLuL6mFpR\n"
"CIZYWjZMa0MJvWMKGm/AhpfEOkbT58Fg5YmxhnKMk6ps1eR6mh3NgH1IbUqvEYNC\n"
"eS42X3kAMxEDseBOMths0gxeLL+IHdQpXnAjZppW8zEIcN3yfknul35r8R6Qt9aK\n"
"q5+/m1ADBw==\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIDATCCAemgAwIBAgIBATANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTE0MjI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDIe0eOnLaV750K\n"
"4+mVaAftRrJp8t68KJivcRFpkl0ucQs6gwNf9EsVkHineOR3RXypjJ7Hsv+4PIKp\n"
"BhEOTprYUKcBaxHK/NIezV6NrO1AwuD6MtJDQF9jGpSy0F3eRUoBCjVYhTl+JxcZ\n"
"hGHPJd8WMeanQWY4xG4gTwtpjF3tPU5+JGQwLk5SbcLicM2QMG3CapZinOGK3/XC\n"
"Fjsvf5ZhxnixayhfiX/n9BmeP1yxz7YORNYPlL8z1CcLZqJsyjZnNkVwNvl4ft9I\n"
"FOKBLoOTSGocHFIFXh5b50GG6QHgvN+TiAwdpfRTUskWVg8VVIh7ymgDoI2jQhk4\n"
"EeMaZHd/AgMBAAGjZDBiMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQU4PpIWj/FDo3MEilcYE/s4YdGljwwHwYDVR0jBBgwFoAU6XJK\n"
"EOUYTuioWHG+1YBuz0yPFmowDQYJKoZIhvcNAQELBQADggEBAJOCrGvbeRqPj+uL\n"
"2FIfbkYZAx2nGl3RVv5ZK2YeDpU1udxLihc6Sr67OZbiA4QMKxwgI7pupuwXmyql\n"
"vs9dWnNpjzgfc0OqqzVdOFlfw8ew2DQb2sUXCcIkwqXb/pBQ9BvcgdDASu+rm74j\n"
"JWDZlhcqeVhZROKfpsjsl+lHgZ7kANwHtUJg/WvK8J971hgElqeBO1O97cGkw/in\n"
"e8ooK9Lxk3Td+WdI8C7juCYiwsGqFEKuj7b6937uzvpFmm1fYDdOHhTMcHTHIVTr\n"
"uxSSurQ4XSDF6Iuel3+IdpLL79UYJ7Cf4IhBWj0EloF6xWTA6nUYl3gzKpx1Tg1U\n"
"x2+26YY=\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIIC4DCCAcigAwIBAgIBADANBgkqhkiG9w0BAQsFADAPMQ0wCwYDVQQDEwRDQS0w\n"
"MCIYDzIwMTQwODI2MTE0MjI2WhgPOTk5OTEyMzEyMzU5NTlaMA8xDTALBgNVBAMT\n"
"BENBLTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCqLuVrTyiqz+Zs\n"
"9Qw5V2Z1y1YSWU6aRDMs+34rP2gwT41C69HBh2LLRS04iJUVQydwnEJukwKlTNRn\n"
"1lEpvWgtYmySWA2SyI4xkVzCXgwv0k7WyLwa39hfNY1rXAqhDTL8VO0nXxi8hCMW\n"
"ohaXcvsieglhN5uwu6voEdY3Gwtx4V8ysDJ2P9EBo49ZHdpBOv+3YLDxbWZuL/tI\n"
"nYkBUHHfWGhUHsRsu0EGob3SFnfiooCbE/vtmn9rUuBEQDqOjOg3el/aTPJzcMi/\n"
"RTz+8ho17ZrQRKHZGKWq9Skank+2X9FZoYKFCUlBm6RVud1R54QYZEIj7W9ujQLN\n"
"LJrcIwBDAgMBAAGjQzBBMA8GA1UdEwEB/wQFMAMBAf8wDwYDVR0PAQH/BAUDAwcE\n"
"ADAdBgNVHQ4EFgQU6XJKEOUYTuioWHG+1YBuz0yPFmowDQYJKoZIhvcNAQELBQAD\n"
"ggEBAEeXYGhZ8fWDpCGfSGEDX8FTqLwfDXxw18ZJjQJwus7bsJ9K/hAXnasXrn0f\n"
"TJ+uJi8muqzP1V376mSUzlwXIzLZCtbwRdDhJJYRrLvf5zfHxHeDgvDALn+1AduF\n"
"G/GzCVIFsYNSMdKGwNRp6Ucgl43BPZs6Swn2DXrxxW7Gng+8dvUS2XGLLdH6q1O3\n"
"U1EgJilng+VXx9Rg3yCs5xDiehASySsM6MN/+v+Ouf9lkoQCEgrtlW5Lb/neOBlA\n"
"aS8PPQuKkIEggNd8hW88YWQOJXMiCAgFppVp5B1Vbghn9IDJQISx/AXAoDXQvQfE\n"
"bdOzcKFyDuklHl2IQPnYTFxm/G8=\n"
"-----END CERTIFICATE-----\n",
NULL
};

static const char *v1_intermed_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC3jCCAcagAwIBAgIBATANBgkqhkiG9w0BAQsFADAOMQwwCgYDVQQDEwNDQTIw\n"
  "IhgPMjAxNDAyMTIxNDU0NTJaGA85OTk5MTIzMTIzNTk1OVowDjEMMAoGA1UEAxMD\n"
  "Q0EyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAwZPztwmz136OBSTT\n"
  "t4pXys9dTfaOBajrO2s4JcPa0C+7D2wfWD941q1q01TV5+QgLDTF7OO5VSXt2W3p\n"
  "cKlXS/Ll2N8sxTaULcVCodFrHOdfHV1V65VlWhJOnPdpboBtM2V8Iory+d2mNXZT\n"
  "wkcNJ/Z8YBUZCTeR1zaLjq0GAITyJoMiI4+x9Djc+iBDGJarRW7A/JyDN4EFjDzw\n"
  "svdWpHg710I+qtKnlMO/whEmw9r3L486JTSlrrrruUSVGY9UWJpv62az1jbu63d8\n"
  "6/PBp0xbBpiv1xA0qSSquN/THurTZ0Y0MS0vbpnAYkws8YxnFAV1TU4B7AZ0IQId\n"
  "Zjo6HQIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/MA8GA1UdDwEB/wQFAwMHBAAw\n"
  "HQYDVR0OBBYEFNVrkTf8WoQKImozc6dBqx8J8tvhMA0GCSqGSIb3DQEBCwUAA4IB\n"
  "AQBpX+j6Fd5aLnMs55qqbFBTWU9bH4/fGh6OVcJztZyvRTOQMLoha9Lsa2C1c1u0\n"
  "kjj3coRWIq8YH8FbOhu0x3pij5dcnn1FQCKcwEmjdDf6ltxplkZXpR86yW2ZyR2W\n"
  "WmIPUrMPJNFkBbgVKFyYoj+9QUyoWHAWNSLJhqBI5v5CRNYIIat1Nt5SuTDm3ggw\n"
  "GUfMH/snytxVq23tj+02pBCdahTqN1w83W1yFX39URChPpl9RZ6HcIg3DFrXhXte\n"
  "lA+/t8l+o7w7POJ4xMyRtbTuGpGHQac+VJBWKFkduY3sbXN2GdQPL6/VvKH115Tr\n"
  "Bos85afmGYPR/gUP0hVSlFzj\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *v1_root_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDAjCCAeqgAwIBAgIMVDP8wwGyCHAlXREsMA0GCSqGSIb3DQEBCwUAMA8xDTAL\n"
  "BgNVBAMTBENBLTAwIhgPMjAxNDEwMDcxNDQ2MjdaGA85OTk5MTIzMTIzNTk1OVow\n"
  "EzERMA8GA1UEAxMIc2VydmVyLTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
  "AoIBAQCp7I46Ga7QxXCq6SegwcKCbnMeIG2P5IaJXknjBy4rq7P3dqEdEmm/UdwS\n"
  "Wp2jH+k+OFvzFe2FR1fY7UBSFdpLTaMz8YIQ1ESPu2afAxWCE1drJnphVCZyMskp\n"
  "d9P9p+TXE4Y7ppxPRTvp2D/chfIcByIKPKwsjU37hrgFehb7Jolq3Er0pOPitSPj\n"
  "KFVKNAktu8Z411S1hQdO7+jjr8pbFoROm2VDbYRpowCHw9ZhlC51SFKeqPTslUdv\n"
  "53pmq1p02d3WMvKWuFRAIMs1UPba5prN9UI7jZztR0o7xnHm3KtWk+o4+YSDCM5R\n"
  "EoiGvyhKgDoM3B/KQG+1rin2/uMdAgMBAAGjVjBUMAwGA1UdEwEB/wQCMAAwFAYD\n"
  "VR0RBA0wC4IJbG9jYWxob3N0MA8GA1UdDwEB/wQFAwMHoAAwHQYDVR0OBBYEFDoL\n"
  "zL5Pd48Psh5oe9VGrkmXclumMA0GCSqGSIb3DQEBCwUAA4IBAQA21lKw/Vr7P4hl\n"
  "VBY27GOGIh7Tw0dFN8HiYX0jFRlyn7zgoRYfJMuLxtC+jJl02s+iljl9gdTlcAgt\n"
  "dWs74QdKUIMHfoT12WOcIwAIMZLBspbUjn6+eoVPE6zCOfrChRCv4dM4BCz/kg6w\n"
  "MqxM/UE+OS+AgO8hHN1boMbBWMcMR/ylpJE2P3nHGgTg6xsZrEn9aH8y+uqUuScn\n"
  "P13H74zV1f1tTi2QT7Y/lpNKuB8vvqJgcUdxGaVQhz3q+YooSwmkDaWTz9HIAqpr\n"
  "77BBxvr5eok6o528h4qJ5nA3NgLw8nIgyBP2vN/N4CsmRAZ/vHVbGEOeWRkH0rGE\n"
  "SemQfe/B\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICljCCAX4CAQAwDQYJKoZIhvcNAQELBQAwDzENMAsGA1UEAxMEQ0EtMDAiGA8y\n"
  "MDE0MTAwNzE0NDYyNloYDzk5OTkxMjMxMjM1OTU5WjAPMQ0wCwYDVQQDEwRDQS0w\n"
  "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAnTMg33+4Q3ULz+//A9Wn\n"
  "m0DCq9X1ne0q8TfCriOo4Zc2VuFsR3Bm90btX9KqVEfByD8Ba4fb/oF0F6+sb4Ej\n"
  "imPI6PSwHXEX0BPSHcmv6lb+iXHRwpqsx+r9GIuPS+0vuTu07sj1yjszlx7aNXOx\n"
  "hsDAFaedzO8/9nCjbrQ79cLcGusPETjGFAD9vDfBTUNtebPiW9CCNsmRUVWOm6/s\n"
  "5kfy1AfcH6FZCoqvmoELz55JWzYHrWSQhgIopJ0DLfYoF8fg3XeIMUNLk+Lrcoe6\n"
  "ZDCxDxGwXReNfIWnf1l1OnZAdMAt/egr9jeQR+l9y4jBhcssW7Wb8M6+wvinsMOG\n"
  "MQIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQBosiz3l31++ZErMs09bnGe+DEQfL+t\n"
  "iWRmhzLC7kF64+O4Eu+IkWvEUv+LEhHm2GLrqaKu7FAlWUMWRX3GHHxAOtNeLR7U\n"
  "qXBZtq56wHA+fMv8+lqLQuT6eKLNbbuxVWZ/E1qaMax7rlQrtEpAC9ruuafcPlC5\n"
  "U4YbJ4VOhfBnJzQ6KlFtbqOkGr7v/l2d9NRxorAWawPVhIteZv1Ahiu++5g+dn/R\n"
  "z8ehN9SEm+c6C5mWrqHiQka3yi060gO8kBcumM/cE6BxffiOUxy2gsPC2ZrI8xkB\n"
  "ghrpQ87AmWXsvVk03U1l0vHpiE3kXb5FIAbWW7In1mfULqLKgeqllMhD\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *pathlen_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDGzCCAgOgAwIBAgIIUvuL4ymDgpEwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0ExMCIYDzIwMTQwMjEyMTQ1NzQwWhgPOTk5OTEyMzEyMzU5NTlaMBExDzAN\n"
  "BgNVBAMTBnNlcnZlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBALkh\n"
  "OTKvbV/OQcP9fn02UGzWNLGwS06248rOo+fHqCngf3nl/IefzktuI4Al5Qq9pq42\n"
  "X2oLn/zr8kaO3L0rDcc54nVRuipfsw5nxAmwmjpfwnDgyla4Y88n57dhia/tmFlL\n"
  "rpspg4YJ8Jt4/tGaNVS0OZ57LEoW7/OrqoGM7U3Xxa2QbzaNYMGcSt2ePvccCg13\n"
  "+CJcXxOQcr/cUxyuk9neATJoulFtO8ycpmkLFUdi0WoThBjNCCJ8s7ZuvnGpF4vD\n"
  "3fuvyM2ftiS08B2c5cv6FH9+4I7Elrb++TdVf43F0Awc2pLhm8L1fAuTtMjtbd9w\n"
  "pxs0yaWR8IvQYbWM/XUCAwEAAaN2MHQwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAK\n"
  "BggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBTlqmq9/mceVMMP\n"
  "Z0HtoWZvHDc/tTAfBgNVHSMEGDAWgBRq1Eg1exmWS45j+lFklVwTQe5NsjANBgkq\n"
  "hkiG9w0BAQsFAAOCAQEAAWX2f1XE/bR+DYCMaNAFpTjOFZ54eCBdazUqfVamPRoP\n"
  "/8qyEYpMaA+IpHkJ5tXsx/rdKLgg1kNv/6bXyCwVgVcNBxpt05WUxqFG9xxLLz1K\n"
  "UkbOZUA0/P9GqjRt9HeodP0Hqog1c1d4jgU3Ng7FIn5JXmLNVfl5qXfXCJ2S9WKu\n"
  "pHw0M9TLOKUD2DD2T5K/iZAU0AXGRVUH39e+xsFIoawPWflfN5eNz8uikMQc+kxt\n"
  "DdEMfPZidecToAcMolle53F7zZvqQswfla/3esb/bnndFAqIsnXRpi6Oj06ajzwE\n"
  "TKP745KouHnNPZ3Hz1mPeusn4EJkLfTb5aBT3nJ+Kw==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIICmzCCAYMCCFL7i7wg78R3MA0GCSqGSIb3DQEBCwUAMA4xDDAKBgNVBAMTA0NB\n"
  "MjAiGA8yMDE0MDIxMjE0NTcwMloYDzk5OTkxMjMxMjM1OTU5WjAOMQwwCgYDVQQD\n"
  "EwNDQTEwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDS0OoKnGy/0Ju\n"
  "U/Q+T3b5RR53oe24OoISkKE4Sia50aHeQZbYitIsQUEvMq06kdjaSA2p8qT29p9n\n"
  "feDYsrhgZkWm4stp2EMkbbkB6k0hBobo4YQiQRa4ttYIsOWLMk/xR0z5ST+UoUv6\n"
  "L/5IOxxJzdpUEbMJpM0Zp8AUZRKnXTc88a/zpPbYiO+LicdhlIKiUvIlnVTlvHOz\n"
  "yN9y5o0Dry9i3IlDSTK8Ls54Nm6v7Z+1F1UwIXkYJCX0WxJ6w/4jHmbiRSitbH9s\n"
  "UqSUm9sHCUakBJA3Y9/9K2SVWNJrG/G4LmZ+Zwr8NdZN3RrxQnWnudL4bTtM0WgY\n"
  "QynV12XDAgMBAAEwDQYJKoZIhvcNAQELBQADggEBAGJbRVVxERtx5Li15n1bdAzW\n"
  "HaDNKt/TkUcZvnfdtggvZLkKhPiNWksQ+9jk1RS71dSZHT9Kb9bIVhaYzaHdI+hG\n"
  "7coftkY66wjD9xLv2DyqnwvuR0S8Uhj9jas5xf/P9S79ZDk61Afg7yX8aLBJpJIH\n"
  "citi1c65C8rYwB8zsF1Zbbh2/6Enty+eFhS1JOuEgUFP1oO2Nj2vh4IqR3yEGdGt\n"
  "Tr57CD/C97fcaeRE4LlHJIMQ9toeZ5Fc9avnOzNIxJd7BPqWWvOnu3TWufj7uaq+\n"
  "CcHTlq9h0NKf9dI1GsxbscJbO3+I+hzOwYfFcNrQ+8BFGbcwx9ZcS2xO3Rx9dbc=\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC6DCCAdCgAwIBAgIIUvuTdCOiZ3IwDQYJKoZIhvcNAQELBQAwDjEMMAoGA1UE\n"
  "AxMDQ0EyMCIYDzIwMTQwMjEyMTUyOTU3WhgPOTk5OTEyMzEyMzU5NTlaMA4xDDAK\n"
  "BgNVBAMTA0NBMjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMGT87cJ\n"
  "s9d+jgUk07eKV8rPXU32jgWo6ztrOCXD2tAvuw9sH1g/eNatatNU1efkICw0xezj\n"
  "uVUl7dlt6XCpV0vy5djfLMU2lC3FQqHRaxznXx1dVeuVZVoSTpz3aW6AbTNlfCKK\n"
  "8vndpjV2U8JHDSf2fGAVGQk3kdc2i46tBgCE8iaDIiOPsfQ43PogQxiWq0VuwPyc\n"
  "gzeBBYw88LL3VqR4O9dCPqrSp5TDv8IRJsPa9y+POiU0pa6667lElRmPVFiab+tm\n"
  "s9Y27ut3fOvzwadMWwaYr9cQNKkkqrjf0x7q02dGNDEtL26ZwGJMLPGMZxQFdU1O\n"
  "AewGdCECHWY6Oh0CAwEAAaNGMEQwEgYDVR0TAQH/BAgwBgEB/wIBADAPBgNVHQ8B\n"
  "Af8EBQMDBwQAMB0GA1UdDgQWBBTVa5E3/FqECiJqM3OnQasfCfLb4TANBgkqhkiG\n"
  "9w0BAQsFAAOCAQEAtQudk32tQ30ldwLy5QyNzwpxTq1izycXGMkh3LvNUQrxmwzl\n"
  "8EPi1d4bxdAi3ghwppImJPZ1aWOrSl9cxl7kH4clq/QdG6bKhhr/40ImZctV35fA\n"
  "Kd1/aDlUUNQIv7cD/T8fb8rMmZ7RPoLsgLcVfodKwafY+X/y4ZacA2uF2L2dX37T\n"
  "etQprA+hjeKu6rej9eb+ERZqYChDvp7FNbJ5fOnIZ9iG1Z714fUeuRDzvosJl6n8\n"
  "aVIRHXdZbhCgKdJTR4bvFPGVFL86xLMkV7jhCImNBN9rmd59wD6g79nTUUoPDM3r\n"
  "rpNkoLGmlBhUorRWbx0YAz9UojNdd4GWMefwZw==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

static const char *cve_2014_0092_check[] = {
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDtDCCAmygAwIBAgIETeC0yjANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5H\n"
  "bnVUTFMgVGVzdCBDQTAeFw0xMTA1MjgwODM5MzlaFw0zODEwMTIwODM5NDBaMC8x\n"
  "LTArBgNVBAMTJEdudUFBQSBUZXN0IFNlcnZlciAoUlNBIGNlcnRpZmljYXRlKTCC\n"
  "AVIwDQYJKoZIhvcNAQEBBQADggE/ADCCAToCggExALRrJ5glr8H/HsqwfvTYvO1D\n"
  "hmdUXdq0HsKQX4M8AhH8E3KFsoikZUELdl8jvoqf/nlLczsux0s8vxbJl1U1F/Oh\n"
  "ckswwuAnlBLzVgDmzoJLEV2kHpv6+rkbKk0Ytbql5gzHqKihbaqIhNyWDrJsHDWq\n"
  "58eUPfnVx8KiDUuzbnr3CF/FCc0Vkxr3mN8qTGaJJO0f0BZjgWWlWDuhzSVim5mB\n"
  "VAgXGOx8LwiiOyhXMp0XRwqG+2KxQZnm+96o6iB+8xvuuuqaIWQpkvKtc+UZBZ03\n"
  "U+IRnxhfIrriiw0AjJ4vp4c9QL5KoqWSCAwuYcBYfJqZ4dasgzklzz4b7eujbZ3L\n"
  "xTjewcdumzQUvjA+gpAeuUqaduTvMwxGojFy9sNhC/iqZ4n0peV2N6Epn4B5qnUC\n"
  "AwEAAaOBjTCBijAMBgNVHRMBAf8EAjAAMBQGA1UdEQQNMAuCCWxvY2FsaG9zdDAT\n"
  "BgNVHSUEDDAKBggrBgEFBQcDATAPBgNVHQ8BAf8EBQMDB6AAMB0GA1UdDgQWBBR2\n"
  "B1hM6rUp9S2ABoyDSoINCeyT3jAfBgNVHSMEGDAWgBRNVrdqAFjxZ5L0pnVVG45T\n"
  "AQPvzzANBgkqhkiG9w0BAQsFAAOCATEBdNWmTsh5uIfngyhOWwm7pK2+vgUMY8nH\n"
  "gMoMFHt0yuxuImcUMXu3LRS1dZSoCJACBpTFGi/Dg2U0qvOHQcEmc3OwNqHB90R3\n"
  "LG5jUSCtq/bYW7h/6Gd9KeWCgZczaHbQ9IPTjLH1dLswVPt+fXKB6Eh0ggSrGATE\n"
  "/wRZT/XgDCW8t4C+2+TmJ8ZEzvU87KAPQ9rUBS1+p3EUAR/FfMApApsEig1IZ+ZD\n"
  "5joaGBW7zh1H0B9mEKidRvD7yuRJyzAcvD25nT15NLW0QR3dEeXosLc720xxJl1h\n"
  "h8NJ7YOvn323mOjR9er4i4D6iJlXmJ8tvN9vakCankWvBzb7plFn2sfMQqICFpRc\n"
  "w075D8hdQxfpGffL2tEeKSgjyNHXS7x3dFhUpN3IQjUi2x4f2e/ZXg==\n"
  "-----END CERTIFICATE-----\n",
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDUDCCAgigAwIBAgIBADANBgkqhkiG9w0BAQsFADAZMRcwFQYDVQQDEw5HbnVU\n"
  "TFMgVGVzdCBDQTAeFw0xMTA1MjgwODM2MzBaFw0zODEwMTIwODM2MzNaMBkxFzAV\n"
  "BgNVBAMTDkdudVRMUyBUZXN0IENBMIIBUjANBgkqhkiG9w0BAQEFAAOCAT8AMIIB\n"
  "OgKCATEAnORCsX1unl//fy2d1054XduIg/3CqVBaT3Hca65SEoDwh0KiPtQoOgZL\n"
  "dKY2cobGs/ojYtOjcs0KnlPYdmtjEh6WEhuJU95v4TQdC4OLMiE56eIGq252hZAb\n"
  "HoTL84Q14DxQWGuzQK830iml7fbw2WcIcRQ8vFGs8SzfXw63+MI6Fq6iMAQIqP08\n"
  "WzGmRRzL5wvCiPhCVkrPmwbXoABub6AAsYwWPJB91M9/lx5gFH5k9/iPfi3s2Kg3\n"
  "F8MOcppqFYjxDSnsfiz6eMh1+bYVIAo367vGVYHigXMEZC2FezlwIHaZzpEoFlY3\n"
  "a7LFJ00yrjQ910r8UE+CEMTYzE40D0olCMo7FA9RCjeO3bUIoYaIdVTUGWEGHWSe\n"
  "oxGei9Gkm6u+ASj8f+i0jxdD2qXsewIDAQABo0MwQTAPBgNVHRMBAf8EBTADAQH/\n"
  "MA8GA1UdDwEB/wQFAwMHBgAwHQYDVR0OBBYEFE1Wt2oAWPFnkvSmdVUbjlMBA+/P\n"
  "MA0GCSqGSIb3DQEBCwUAA4IBMQAesOgjGFi1zOYpA/N3gkUVRcBHDxmN7g2yOcqH\n"
  "VfhFc+e4zhOehR11WCt2RgzNlnYVmV5zBmQBdTAt8Po/MVhLCDW1BULHlLvL0DFc\n"
  "4sB1RlcGeQcCKQa4b+Q9VWf4f6TfuEWZQC5j5stiXjVgOqrOMrzKZ2eKWA4JsL9s\n"
  "V+7ANSZE+hOt1X1mA8moyqe95U2Ecih+nFJSWSBd1WFiEzVnXv4FVWPXbH9HERDK\n"
  "VbasjofWWmQO1YlQPishLgm1IbwqOkOk4sDgoLuUZ4GgP0DDeN6EmRDOzByrv+9u\n"
  "f45Bl9IQf4IJNPLU9lEqjyMOydqT6kBi7fjV5ICuQZ4EeVJsOGuX7PqNyoDzJHLv\n"
  "ferRfNLr6eQSHSxBhS0cVyDjb5gCawK6u7xTU+b7xikEie9k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Triggers incorrect verification success on older versions */
static const char *cve_2008_4989_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIB6zCCAVQCCQCgwnB/k0WZrDANBgkqhkiG9w0BAQUFADA9MQswCQYDVQQGEwJE\n"
  "RTEXMBUGA1UEChMOR05VIFRMUyBBdHRhY2sxFTATBgNVBAMTDGludGVybWVkaWF0\n"
  "ZTAeFw0wODExMDMxMjA1MDRaFw0wODEyMDMxMjA1MDRaMDcxCzAJBgNVBAYTAkRF\n"
  "MRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazEPMA0GA1UEAxMGc2VydmVyMIGfMA0G\n"
  "CSqGSIb3DQEBAQUAA4GNADCBiQKBgQDKdL9g5ErMLOLRCjiomZlNLhy0moWGaKIW\n"
  "aX6vyUIfh8d6FcArHoKoqhmX7ckvod50sOYPojQesDpl7gVaQNA6Ntr1VCcuNPef\n"
  "UKWtEwL0Qu9JbPnUoIYd7mAaqVQgFp6W6yzV/dp63LH4XSdzBMhpZ/EU6vZoE8Sv\n"
  "VLdqj5r6jwIDAQABMA0GCSqGSIb3DQEBBQUAA4GBAH4QRR7sZEbjW00tXYk/3O/Z\n"
  "96AxJNg0F78W5B68gaJrLJ7DTE2RTglscuEq1+2Jyb4AIziwXpYqxgwcP91QpH97\n"
  "XfwdXIcyjYvVLHiKmkQj2zJTY7MeyiEQQ2it8VstZG2fYmi2EiMZIEnyJ2JJ7bA7\n"
  "bF7pG7Cg3oEHUM0H5KUU\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (not signed by next cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICADCCAWmgAwIBAgIJAIZ4nkHQAqTFMA0GCSqGSIb3DQEBBQUAMDUxCzAJBgNV\n"
  "BAYTAkRFMRcwFQYDVQQKEw5HTlUgVExTIEF0dGFjazENMAsGA1UEAxMEcm9vdDAe\n"
  "Fw0wODExMDMxMjA0NDVaFw0wODEyMDMxMjA0NDVaMD0xCzAJBgNVBAYTAkRFMRcw\n"
  "FQYDVQQKEw5HTlUgVExTIEF0dGFjazEVMBMGA1UEAxMMaW50ZXJtZWRpYXRlMIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDvBpW8sAhIuUmNvcBE6wv/q7MtM1Z9\n"
  "2I1SDL8eJ8I2nPg6BlCX+OIqNruynj8J7uPEQ04ZLwLxNXoyZa8057YFyrKLOvoj\n"
  "5IfBtidsLWYv6PO3qqHJXVvwGdS7PKMuUlsjucCRyXVgQ07ODF7piqoVFi9KD99w\n"
  "AU5+9plGrZNP/wIDAQABoxAwDjAMBgNVHRMEBTADAQH/MA0GCSqGSIb3DQEBBQUA\n"
  "A4GBAGPg+M+8MsB6zHN2o+jAtyqovrTTwmzVWEgfEH/aHC9+imGZRQ5lFNc2vdny\n"
  "AgaJ9/izO5S6Ibb5zUowN2WhoUJOVipuQa2m9AviOgheoU7tmANC9ylm/pRkKy/0\n"
  "n5UVzlKxDhRp/xBb7MWOw3KEQjiAf2Z3wCLcCPUqcJUdJC4v\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (trusted CA cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEIDCCAwigAwIBAgIQNE7VVyDV7exJ9C/ON9srbTANBgkqhkiG9w0BAQUF\n"
  "ADCBqTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYG\n"
  "A1UECxMfQ2VydGlmaWNhdGlvbiBTZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UE\n"
  "CxMvKGMpIDIwMDYgdGhhd3RlLCBJbmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNl\n"
  "IG9ubHkxHzAdBgNVBAMTFnRoYXd0ZSBQcmltYXJ5IFJvb3QgQ0EwHhcNMDYx\n"
  "MTE3MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCBqTELMAkGA1UEBhMCVVMxFTAT\n"
  "BgNVBAoTDHRoYXd0ZSwgSW5jLjEoMCYGA1UECxMfQ2VydGlmaWNhdGlvbiBT\n"
  "ZXJ2aWNlcyBEaXZpc2lvbjE4MDYGA1UECxMvKGMpIDIwMDYgdGhhd3RlLCBJ\n"
  "bmMuIC0gRm9yIGF1dGhvcml6ZWQgdXNlIG9ubHkxHzAdBgNVBAMTFnRoYXd0\n"
  "ZSBQcmltYXJ5IFJvb3QgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK\n"
  "AoIBAQCsoPD7gFnUnMekz52hWXMJEEUMDSxuaPFsW0hoSVk3/AszGcJ3f8wQ\n"
  "LZU0HObrTQmnHNK4yZc2AreJ1CRfBsDMRJSUjQJib+ta3RGNKJpchJAQeg29\n"
  "dGYvajig4tVUROsdB58Hum/u6f1OCyn1PoSgAfGcq/gcfomk6KHYcWUNo1F7\n"
  "7rzSImANuVud37r8UVsLr5iy6S7pBOhih94ryNdOwUxkHt3Ph1i6Sk/KaAcd\n"
  "HJ1KxtUvkcx8cXIcxcBn6zL9yZJclNqFwJu/U30rCfSMnZEfl2pSy94JNqR3\n"
  "2HuHUETVPm4pafs5SSYeCaWAe0At6+gnhcn+Yf1+5nyXHdWdAgMBAAGjQjBA\n"
  "MA8GA1UdEwEB/wQFMAMBAf8wDgYDVR0PAQH/BAQDAgEGMB0GA1UdDgQWBBR7\n"
  "W0XPr87Lev0xkhpqtvNG61dIUDANBgkqhkiG9w0BAQUFAAOCAQEAeRHAS7OR\n"
  "tvzw6WfUDW5FvlXok9LOAz/t2iWwHVfLHjp2oEzsUHboZHIMpKnxuIvW1oeE\n"
  "uzLlQRHAd9mzYJ3rG9XRbkREqaYB7FViHXe4XI5ISXycO1cRrK1zN44veFyQ\n"
  "aEfZYGDm/Ac9IiAXxPcW6cTYcvnIc3zfFi8VqT79aie2oetaupgf1eNNZAqd\n"
  "E8hhuvU5HIe6uL17In/2/qxAeeWsEG89jxt5dovEN7MhGITlNgDrYyCZuen+\n"
  "MwS7QcjBAvlEYyCegc5C09Y/LHbTY5xZ3Y+m4Q6gLkH3LpVHz7z9M/P2C2F+\n"
  "fpErgUfCJzDupxBdN49cOSvkBPB7jVaMaA==\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 chain */
static const char *verisign_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCBPCgAwIBAgIQakrDGzEQ5utI8PxRo5oXHzANBgkqhkiG9w0BAQUFADCB\n"
  "vjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMv\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0Ew\n"
  "HhcNMDcwNTA5MDAwMDAwWhcNMDkwNTA4MjM1OTU5WjCCAUAxEDAOBgNVBAUTBzI0\n"
  "OTc4ODYxEzARBgsrBgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMIRGVs\n"
  "YXdhcmUxCzAJBgNVBAYTAlVTMQ4wDAYDVQQRFAU5NDA0MzETMBEGA1UECBMKQ2Fs\n"
  "aWZvcm5pYTEWMBQGA1UEBxQNTW91bnRhaW4gVmlldzEiMCAGA1UECRQZNDg3IEVh\n"
  "c3QgTWlkZGxlZmllbGQgUm9hZDEXMBUGA1UEChQOVmVyaVNpZ24sIEluYy4xJTAj\n"
  "BgNVBAsUHFByb2R1Y3Rpb24gU2VjdXJpdHkgU2VydmljZXMxMzAxBgNVBAsUKlRl\n"
  "cm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjEZMBcGA1UE\n"
  "AxQQd3d3LnZlcmlzaWduLmNvbTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA\n"
  "xxA35ev879drgQCpENGRQ3ARaCPz/WneT9dtMe3qGNvzXQJs6cjm1Bx8XegyW1gB\n"
  "jJX5Zl4WWbr9wpAWZ1YyJ0bEyShIGmkU8fPfbcXYwSyWoWwvE5NRaUB2ztmfAVdv\n"
  "OaGMUKxny2Dnj3tAdaQ+FOeRDJJYg6K1hzczq/otOfsCAwEAAaOCAf8wggH7MAkG\n"
  "A1UdEwQCMAAwHQYDVR0OBBYEFPFaiZNVR0u6UfVO4MsWVfTXzDhnMAsGA1UdDwQE\n"
  "AwIFoDA+BgNVHR8ENzA1MDOgMaAvhi1odHRwOi8vRVZJbnRsLWNybC52ZXJpc2ln\n"
  "bi5jb20vRVZJbnRsMjAwNi5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXBjAq\n"
  "MCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQGA1Ud\n"
  "JQQtMCsGCCsGAQUFBwMBBggrBgEFBQcDAgYJYIZIAYb4QgQBBgorBgEEAYI3CgMD\n"
  "MB8GA1UdIwQYMBaAFE5DyB127zdTek/yWG+U8zji1b3fMHYGCCsGAQUFBwEBBGow\n"
  "aDArBggrBgEFBQcwAYYfaHR0cDovL0VWSW50bC1vY3NwLnZlcmlzaWduLmNvbTA5\n"
  "BggrBgEFBQcwAoYtaHR0cDovL0VWSW50bC1haWEudmVyaXNpZ24uY29tL0VWSW50\n"
  "bDIwMDYuY2VyMG0GCCsGAQUFBwEMBGEwX6FdoFswWTBXMFUWCWltYWdlL2dpZjAh\n"
  "MB8wBwYFKw4DAhoEFI/l0xqGrI2Oa8PPgGrUSBgsexkuMCUWI2h0dHA6Ly9sb2dv\n"
  "LnZlcmlzaWduLmNvbS92c2xvZ28uZ2lmMA0GCSqGSIb3DQEBBQUAA4IBAQBEueAg\n"
  "xZJrjGPKAZk1NT8VtTn0yi87i9XUnSOnkFkAuI3THDd+cWbNSUzc5uFJg42GhMK7\n"
  "S1Rojm8FHxESovLvimH/w111BKF9wNU2XSOb9KohfYq3GRiQG8O7v9JwIjjLepkc\n"
  "iyITx7sYiJ+kwZlrNBwN6TwVHrONg6NzyzSnxCg+XgKRbJu2PqEQb6uQVkYhb+Oq\n"
  "Vi9d4by9YqpnuXImSffQ0OZ/6s3Rl6vY08zIPqa6OVfjGs/H45ETblzezcUKpX0L\n"
  "cqnOwUB9dVuPhtlX3X/hgz/ROxz96NBwwzha58HUgfEfkVtm+piI6TTI7XxS/7Av\n"
  "nKMfhbyFQYPQ6J9g\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCjCCBPKgAwIBAgIQESoAbTflEG/WynzD77rMGDANBgkqhkiG9w0BAQUFADCB\n"
  "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
  "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
  "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
  "aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMTYxMTA3MjM1OTU5WjCBvjEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
  "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNjE4MDYGA1UEAxMvVmVy\n"
  "aVNpZ24gQ2xhc3MgMyBFeHRlbmRlZCBWYWxpZGF0aW9uIFNTTCBTR0MgQ0EwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC9Voi6iDRkZM/NyrDu5xlzxXLZ\n"
  "u0W8taj/g74cA9vtibcuEBolvFXKQaGfC88ZXnC5XjlLnjEcX4euKqqoK6IbOxAj\n"
  "XxOx3QiMThTag4HjtYzjaO0kZ85Wtqybc5ZE24qMs9bwcZOO23FUSutzWWqPcFEs\n"
  "A5+X0cwRerxiDZUqyRx1V+n1x+q6hDXLx4VafuRN4RGXfQ4gNEXb8aIJ6+s9nriW\n"
  "Q140SwglHkMaotm3igE0PcP45a9PjP/NZfAjTsWXs1zakByChQ0GDcEitnsopAPD\n"
  "TFPRWLxyvAg5/KB2qKjpS26IPeOzMSWMcylIDjJ5Bu09Q/T25On8fb6OCNUfAgMB\n"
  "AAGjggH0MIIB8DAdBgNVHQ4EFgQUTkPIHXbvN1N6T/JYb5TzOOLVvd8wEgYDVR0T\n"
  "AQH/BAgwBgEB/wIBADA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczA9BgNVHR8ENjA0MDKgMKAuhixo\n"
  "dHRwOi8vRVZTZWN1cmUtY3JsLnZlcmlzaWduLmNvbS9wY2EzLWc1LmNybDAgBgNV\n"
  "HSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFn\n"
  "ZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRw\n"
  "Oi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjApBgNVHREEIjAgpB4wHDEa\n"
  "MBgGA1UEAxMRQ2xhc3MzQ0EyMDQ4LTEtNDgwPQYIKwYBBQUHAQEEMTAvMC0GCCsG\n"
  "AQUFBzABhiFodHRwOi8vRVZTZWN1cmUtb2NzcC52ZXJpc2lnbi5jb20wHwYDVR0j\n"
  "BBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQEFBQADggEBAFqi\n"
  "sb/rjdQ4qIBywtw4Lqyncfkro7tHu21pbxA2mIzHVi67vKtKm3rW8oKT4BT+is6D\n"
  "t4Pbk4errGV5Sf1XqbHOCR+6EBXECQ5i4/kKJdVkmPDyqA92Mn6R5hjuvOfa0E6N\n"
  "eLvincBZK8DOlQ0kDHLKNF5wIokrSrDxaIfz7kSNKEB3OW5IckUxXWs5DoYC6maZ\n"
  "kzEP32fepp+MnUzOcW86Ifa5ND/5btia9z7a84Ffelxtj3z2mXS3/+QXXe1hXqtI\n"
  "u5aNZkU5tBIK9nDpnHYiS2DpKhs0Sfei1GfAsSatE7rZhAHBq+GObXAWO3eskZq7\n"
  "Gh/aWKfkT8Fhrryi/ks=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCBGigAwIBAgIQY5Jrio9Agv2swDvTeCmmwDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
  "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
  "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
  "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
  "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
  "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
  "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
  "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
  "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
  "AAGjggHKMIIBxjAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
  "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjBt\n"
  "BggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglpbWFnZS9naWYwITAfMAcGBSsOAwIa\n"
  "BBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNodHRwOi8vbG9nby52ZXJpc2lnbi5j\n"
  "b20vdnNsb2dvLmdpZjA9BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYc\n"
  "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7Lvw\n"
  "MAnzQzn6Aq8zMTMwgYAGA1UdIwR5MHehY6RhMF8xCzAJBgNVBAYTAlVTMRcwFQYD\n"
  "VQQKEw5WZXJpU2lnbiwgSW5jLjE3MDUGA1UECxMuQ2xhc3MgMyBQdWJsaWMgUHJp\n"
  "bWFyeSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eYIQcLrkHRDZKTS2OMp7A8y6vzAg\n"
  "BgNVHSUEGTAXBglghkgBhvhCBAEGCmCGSAGG+EUBCAEwDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUNfnArcMK6xK11/59ADJdeNqKOck4skH3qw6WCAYQxfrcn4eobTInOn5G3Gu\n"
  "39g6DapSHmBex2UtZSxvKnJVlWYQgE4P4wGoXdzV69YdCNssXNVVc59DYhDH05dZ\n"
  "P4sJH99fucYDkJjUgRUYw35ww0OFwKgUp3CxiizbXxCqEQc=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[3] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2 ends with trusted v1 RSA-MD2 cert */
static const char *citibank_com_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIENDCCA52gAwIBAgIQauOJMlH5Ob2tFZ6rJMBdjjANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA4MjkwMDAwMDBaFw0xMDA4MjkyMzU5NTlaMHgxCzAJBgNVBAYTAlVTMRMwEQYD\n"
  "VQQIEwpOZXcgSmVyc2V5MRIwEAYDVQQHFAlXZWVoYXdrZW4xEjAQBgNVBAoUCUNp\n"
  "dGlncm91cDERMA8GA1UECxQId2hnLW9hazYxGTAXBgNVBAMUEHd3dy5jaXRpYmFu\n"
  "ay5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBALQJbSYtbndsIlslFveP\n"
  "IlVNE38HnUD56BHcwfvcb8rQItXeHzYmgOf/RgHPTKG3LEZOxKqM0QpcZtEJ6xwV\n"
  "cTG7Wjw/FrMisN8aO4JWaxe8dFGajstlEMxz43G5zlprb9jzjnbIvvcnz0ILikOQ\n"
  "qmcThopBTs1+d4j7w/yEJo1zAgMBAAGjggF6MIIBdjAJBgNVHRMEAjAAMAsGA1Ud\n"
  "DwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8vY3JsLnZlcmlzaWduLmNv\n"
  "bS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBEBgNVHSAEPTA7MDkGC2CG\n"
  "SAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZlcmlzaWduLmNv\n"
  "bS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEFBQcDAQYIKwYBBQUHAwIw\n"
  "NAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC52ZXJpc2ln\n"
  "bi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJaW1hZ2UvZ2lmMCEwHzAH\n"
  "BgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYkaHR0cDovL2xvZ28udmVy\n"
  "aXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEBBQUAA4GBAFDXKsxtWkoo\n"
  "HBkNjcCvcnjNAo3Pe+eOtLHb39e5qhkNQLPGA/1/7AofY9KmEtSV2LVGeuuJI4Pi\n"
  "Lg7fPl9Q0OE/oHJpj5JkObBP9Wo1vbrDR2nGWUlCRWm20rH81dTn7OcDxarwGWsR\n"
  "ewTCNmpKYaMx8Q1dyMYunHJApu+fbrHu\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Self-signed certificate */
static const char *pem_self_cert[] = {
  "-----BEGIN CERTIFICATE-----\n"
    "MIIDgjCCAmygAwIBAgIBADALBgkqhkiG9w0BAQUwSzELMAkGA1UEBhMCQlIxFDAS\n"
    "BgNVBAoTC01pbmFzIExpdnJlMSYwJAYDVQQDEx1UaGFkZXUgTGltYSBkZSBTb3V6\n"
    "YSBDYXNjYXJkbzAeFw0wODA1MzAxOTUzNDNaFw0wODExMjYxOTUzNDNaMEsxCzAJ\n"
    "BgNVBAYTAkJSMRQwEgYDVQQKEwtNaW5hcyBMaXZyZTEmMCQGA1UEAxMdVGhhZGV1\n"
    "IExpbWEgZGUgU291emEgQ2FzY2FyZG8wggEfMAsGCSqGSIb3DQEBAQOCAQ4AMIIB\n"
    "CQKCAQC4D934O6wrXJbMyu1w8gu6nN0aNUDGqrX9UgaB/4xVuYhPlhjH0z9Dqic9\n"
    "0pEZmyNCjQmzDSg/hnlY3fBG0i9Iel2oYn1UB4SdcJ2qGkLS87y2ZbMTS1oyMR7/\n"
    "y9l3WGEWqwgjIvOjGstcZo0rCIF8Qr21QGX22KWg2HXlMaZyA9bGtJ+L+x6f2hoo\n"
    "yIPCA30VMvIgHjOSPQJF3iJFE4Uxq1PQ65W91NyI6/bRKFOmFdCUJW8tqqvntYP8\n"
    "hEE08wGlKimFNv7CqZuRI8QuOnhZ7pBXkyvQpW8yHrORlOHxSjkNQKjddt92TCJb\n"
    "1q6eKv2CtCuDLgCuIy0Onr4U9n+hAgMBAAGjeDB2MA8GA1UdEwEB/wQFMAMBAf8w\n"
    "HgYDVR0RBBcwFYITbWFpbC5taW5hc2xpdnJlLm9yZzATBgNVHSUEDDAKBggrBgEF\n"
    "BQcDATAPBgNVHQ8BAf8EBQMDB6QAMB0GA1UdDgQWBBQ/5v42y0jBHUKEfqpPmr5a\n"
    "WsjCGjALBgkqhkiG9w0BAQUDggEBAC/WfO2yK3vM9bG0qFEj8sd0cWiapMhf5PtH\n"
    "jigcPb/OKqSFQVXpAdNiUclPRP79Ih3CuWiXfZ/CW0+k2Z8tyy6AnEQItWvoVh/b\n"
    "8lS7Ph/f9JUYHp2DtgsQWcNQbrUZOPFBu8J4MD6cDWG5Uxwl3YASg30ZdmMDNT8B\n"
    "HshYz0HUOAhYwVSI3J/f7LFhD5OpjSroHgE7wA9UJrerAp9f7e3e9D7kNQ8DlvLP\n"
    "kz6Jh+5M/xD3JO1yl+evaCp3LA+z4M2xiNvtzkAEgj3t6RaJ81Sh5XGiooDYZ14R\n"
    "DgEBYLTUfBYBPzoaahPEdG/f0kUjUBJ34fkBUSjJKURPTHJfDfA=\n"
    "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 2, CA constraint FALSE in v3 CA cert)*/
static const char *thea_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIC7DCCAlWgAwIBAgIBATANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDUyMDdaFw0xODA5MDkxMDUyMDdaMIGTMQswCQYDVQQGEwJERTEMMAoGA1UE\n"
  "CBMDUkxQMRcwFQYDVQQHEw5LYWlzZXJzbGF1dGVybjEgMB4GA1UEChMXVGVjaG5p\n"
  "c2NoZSBVbml2ZXJzaXRhZXQxGzAZBgNVBAsTEkZhY2hiZXJlaWNoIFBoeXNpazEe\n"
  "MBwGA1UEAxMVdGhlYS5waHlzaWsudW5pLWtsLmRlMIGfMA0GCSqGSIb3DQEBAQUA\n"
  "A4GNADCBiQKBgQC/gTUrXSeNvuRH+ibdR7zvlCGs+66C6tDaq14SpEDiY/FEw/S4\n"
  "mkhsHohiQkmqpcPJ0FONok7bvJryKZwwhGFHeESvvWjFVNIdxFgf6Jx2McKsRzBD\n"
  "nbgVNeK6bywh2L5WgOeckRm0vUxCwX+jWtETorNHSYnZI9smmBtJ1FIPkQIDAQAB\n"
  "o3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVyYXRl\n"
  "ZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQUS0IiRshnnlH2bneYeCn6OkY9nZAwHwYD\n"
  "VR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQADgYEA\n"
  "g0f6XFxpUL2hncpQcnKorNYdOkZkZHiKqu2SINtla+IbLZFO4nVVO+LKt+RCo2o7\n"
  "tZIMLEU3aCeH5dgSEKQeyL5MPMg3MbA6ezjOBTkT/YgngzM4CMLOKcvAMLncfH/z\n"
  "GYBW1DXijIy1r/SxO0k9zy8OEtKeOOUO0GqQTWuTOOg=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICvzCCAiigAwIBAgIBADANBgkqhkiG9w0BAQUFADBnMQswCQYDVQQGEwJERTEM\n"
  "MAoGA1UECBMDUkxQMSAwHgYDVQQKExdUZWNobmlzY2hlIFVuaXZlcnNpdGFldDEb\n"
  "MBkGA1UECxMSRmFjaGJlcmVpY2ggUGh5c2lrMQswCQYDVQQDEwJDQTAeFw0wODA5\n"
  "MTExMDQ3NDRaFw0xODA5MDkxMDQ3NDRaMGcxCzAJBgNVBAYTAkRFMQwwCgYDVQQI\n"
  "EwNSTFAxIDAeBgNVBAoTF1RlY2huaXNjaGUgVW5pdmVyc2l0YWV0MRswGQYDVQQL\n"
  "ExJGYWNoYmVyZWljaCBQaHlzaWsxCzAJBgNVBAMTAkNBMIGfMA0GCSqGSIb3DQEB\n"
  "AQUAA4GNADCBiQKBgQC76RbqsB5J+VvU1KbBCrkIL3lgY8BxgFvYF3HiHgxtCdqq\n"
  "BmRpAaDBcVAuEb1ihhP68181sYQ1UPMY+zwBwXVNSVvjeBba1JjGmagwPnJXOCay\n"
  "7Cw5orY8KB7U33neEOGrlz1EKQGVaPsr993wGD/7AmntuVuxrRVpzoDP5s0PIwID\n"
  "AQABo3sweTAJBgNVHRMEAjAAMCwGCWCGSAGG+EIBDQQfFh1PcGVuU1NMIEdlbmVy\n"
  "YXRlZCBDZXJ0aWZpY2F0ZTAdBgNVHQ4EFgQU+rCwSUUzK53X9W5otZG4okyY/rsw\n"
  "HwYDVR0jBBgwFoAU+rCwSUUzK53X9W5otZG4okyY/rswDQYJKoZIhvcNAQEFBQAD\n"
  "gYEAUT+LmosiDHGuLAZmY40obam0eexJzn/g++mDy3FMh3WmMBKSsfwFsFsQ4k7N\n"
  "lv1SCfTYeh2hpw/DQzkiYZUkcQI4mBR4hG5Zv56AfYQLGeLtN4VOOCMxguftvzv0\n"
  "kziQa2QW+VzVJqV1gpRCRT30Jaa9s4u6ipO9DT5N03F4CcI=\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Chain length 3 ends with trusted v1 RSA-MD2 cert, similar to
   verisign_com_chain above */
static const char *hbci_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEczCCA9ygAwIBAgIQeODCPg2RbK2r7/1KoWjWZzANBgkqhkiG9w0BAQUFADCB\n"
  "ujEfMB0GA1UEChMWVmVyaVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVy\n"
  "aVNpZ24sIEluYy4xMzAxBgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2Vy\n"
  "dmVyIENBIC0gQ2xhc3MgMzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMg\n"
  "SW5jb3JwLmJ5IFJlZi4gTElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjAeFw0w\n"
  "ODA2MTAwMDAwMDBaFw0wOTA3MzAyMzU5NTlaMIG2MQswCQYDVQQGEwJERTEPMA0G\n"
  "A1UECBMGSGVzc2VuMRowGAYDVQQHFBFGcmFua2Z1cnQgYW0gTWFpbjEsMCoGA1UE\n"
  "ChQjU3Bhcmthc3NlbiBJbmZvcm1hdGlrIEdtYkggJiBDby4gS0cxKTAnBgNVBAsU\n"
  "IFRlcm1zIG9mIHVzZSBhdCB3d3cudmVyaXNpZ24uY29tMSEwHwYDVQQDFBhoYmNp\n"
  "LXBpbnRhbi1ycC5zLWhiY2kuZGUwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AK1CdQ9lqmChZWaRAInimuK7I36VImTuAVU0N6BIS4a2BbblkiekbVf15GVHGb6e\n"
  "QV06ANN6Nd8XIdfoxi3LoAs8sa+Ku7eoEsRFi/XIU96GgtFlxf3EsVA9RbGdtfer\n"
  "9iJGIBae2mJTlk+5LVg2EQr50PJlBuTgiYFc41xs9O2RAgMBAAGjggF6MIIBdjAJ\n"
  "BgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBGBgNVHR8EPzA9MDugOaA3hjVodHRwOi8v\n"
  "Y3JsLnZlcmlzaWduLmNvbS9DbGFzczNJbnRlcm5hdGlvbmFsU2VydmVyLmNybDBE\n"
  "BgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0dHBzOi8v\n"
  "d3d3LnZlcmlzaWduLmNvbS9ycGEwKAYDVR0lBCEwHwYJYIZIAYb4QgQBBggrBgEF\n"
  "BQcDAQYIKwYBBQUHAwIwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRw\n"
  "Oi8vb2NzcC52ZXJpc2lnbi5jb20wbgYIKwYBBQUHAQwEYjBgoV6gXDBaMFgwVhYJ\n"
  "aW1hZ2UvZ2lmMCEwHzAHBgUrDgMCGgQUS2u5KJYGDLvQUjibKaxLB4shBRgwJhYk\n"
  "aHR0cDovL2xvZ28udmVyaXNpZ24uY29tL3ZzbG9nbzEuZ2lmMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAJ03R0YAjYzlWm54gMSn6MqJi0mHdLCO2lk3CARwjbg7TEYAZvDsKqTd\n"
  "cRuhNk079BqrQ3QapffeN55SAVrc3mzHO54Nla4n5y6x3XIQXVvRjbJGwmWXsdvr\n"
  "W899F/pBEN30Tgdbmn7JR/iZlGhIJpY9Us1i7rwQhKYir9ZQBdj3\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDgzCCAuygAwIBAgIQJUuKhThCzONY+MXdriJupDANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNOTcwNDE3MDAwMDAwWhcNMTExMDI0MjM1OTU5WjCBujEfMB0GA1UEChMWVmVy\n"
  "aVNpZ24gVHJ1c3QgTmV0d29yazEXMBUGA1UECxMOVmVyaVNpZ24sIEluYy4xMzAx\n"
  "BgNVBAsTKlZlcmlTaWduIEludGVybmF0aW9uYWwgU2VydmVyIENBIC0gQ2xhc3Mg\n"
  "MzFJMEcGA1UECxNAd3d3LnZlcmlzaWduLmNvbS9DUFMgSW5jb3JwLmJ5IFJlZi4g\n"
  "TElBQklMSVRZIExURC4oYyk5NyBWZXJpU2lnbjCBnzANBgkqhkiG9w0BAQEFAAOB\n"
  "jQAwgYkCgYEA2IKA6NYZAn0fhRg5JaJlK+G/1AXTvOY2O6rwTGxbtueqPHNFVbLx\n"
  "veqXQu2aNAoV1Klc9UAl3dkHwTKydWzEyruj/lYncUOqY/UwPpMo5frxCTvzt01O\n"
  "OfdcSVq4wR3Tsor+cDCVQsv+K1GLWjw6+SJPkLICp1OcTzTnqwSye28CAwEAAaOB\n"
  "4zCB4DAPBgNVHRMECDAGAQH/AgEAMEQGA1UdIAQ9MDswOQYLYIZIAYb4RQEHAQEw\n"
  "KjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL0NQUzA0BgNV\n"
  "HSUELTArBggrBgEFBQcDAQYIKwYBBQUHAwIGCWCGSAGG+EIEAQYKYIZIAYb4RQEI\n"
  "ATALBgNVHQ8EBAMCAQYwEQYJYIZIAYb4QgEBBAQDAgEGMDEGA1UdHwQqMCgwJqAk\n"
  "oCKGIGh0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA0GCSqGSIb3DQEB\n"
  "BQUAA4GBAAgB7ORolANC8XPxI6I63unx2sZUxCM+hurPajozq+qcBBQHNgYL+Yhv\n"
  "1RPuKSvD5HKNRO3RrCAJLeH24RkFOLA9D59/+J4C3IYChmFOJl9en5IeDCSk9dBw\n"
  "E88mw0M9SR2egi5SX7w+xmYpAY5Okiy8RnUDgqxz6dl+C2fvVFIa\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzELMAkG\n"
  "A1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTk2\n"
  "MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UEBhMCVVMxFzAVBgNV\n"
  "BAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFzcyAzIFB1YmxpYyBQcmlt\n"
  "YXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGfMA0GCSqGSIb3DQEBAQUAA4GN\n"
  "ADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69qRUCPhAwL0TPZ2RHP7gJYHyX3KqhE\n"
  "BarsAx94f56TuZoAqiN91qyFomNFx3InzPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/is\n"
  "I19wKTakyYbnsZogy1Olhec9vn2a/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0G\n"
  "CSqGSIb3DQEBAgUAA4GBALtMEivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Do\n"
  "lbwdj2wsqFHMc9ikwFPwTtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNyc\n"
  "AA9WjQKZ7aKQRUzkuxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* End-entity cert signed using RSA-MD5. */
static const char *mayfirst_chain[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIDVTCCAr6gAwIBAgIDCHp1MA0GCSqGSIb3DQEBBAUAMFoxCzAJBgNVBAYTAlVT\n"
  "MRwwGgYDVQQKExNFcXVpZmF4IFNlY3VyZSBJbmMuMS0wKwYDVQQDEyRFcXVpZmF4\n"
  "IFNlY3VyZSBHbG9iYWwgZUJ1c2luZXNzIENBLTEwHhcNMDgwNTE5MDUyOTE5WhcN\n"
  "MDkxMDE5MDUyOTE5WjCBxDELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFHN1cHBvcnQu\n"
  "bWF5Zmlyc3Qub3JnMRMwEQYDVQQLEwpHVDY5MDc5ODgwMTEwLwYDVQQLEyhTZWUg\n"
  "d3d3LnJhcGlkc3NsLmNvbS9yZXNvdXJjZXMvY3BzIChjKTA3MS8wLQYDVQQLEyZE\n"
  "b21haW4gQ29udHJvbCBWYWxpZGF0ZWQgLSBSYXBpZFNTTChSKTEdMBsGA1UEAxMU\n"
  "c3VwcG9ydC5tYXlmaXJzdC5vcmcwgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGB\n"
  "AN0TWIZwJ/hIfMHc08/bBMlzZ5WucJqEvxU/ZnxPo/H6V/m4v1iLpM2hip2c5cg0\n"
  "BcEMc/TBHQ1UEV8sb0Lh91kWfiMB1Sp+L2Fpz/wnhsivXC5j6jq9IcPqmOZOXBYX\n"
  "k04W1B6FKTvk9KrZJ0at2J44hp4SsAfWQI0eCKuas+R1AgMBAAGjgb0wgbowDgYD\n"
  "VR0PAQH/BAQDAgTwMB0GA1UdDgQWBBS0D4iuCxp35TLADTkINq2AhgTYVTA7BgNV\n"
  "HR8ENDAyMDCgLqAshipodHRwOi8vY3JsLmdlb3RydXN0LmNvbS9jcmxzL2dsb2Jh\n"
  "bGNhMS5jcmwwHwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0l\n"
  "BBYwFAYIKwYBBQUHAwEGCCsGAQUFBwMCMAwGA1UdEwEB/wQCMAAwDQYJKoZIhvcN\n"
  "AQEEBQADgYEAXNWYnrO1mZgBSCwPlWhVa2aOKGCFmehLIcAPEBN+8xhXuOeigYBm\n"
  "ic5ShCO583ttgHNCV3Y5dW9sNhv1US4vSb6soKjgUlG11fJKUqU8mwFKvbs7TUSq\n"
  "j6h+1uvlfFI34WzODjJloY4QSM7FmbnW+HCiFKYyvra3iUqjcl9AeR4=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICkDCCAfmgAwIBAgIBATANBgkqhkiG9w0BAQQFADBaMQswCQYDVQQGEwJV\n"
  "UzEcMBoGA1UEChMTRXF1aWZheCBTZWN1cmUgSW5jLjEtMCsGA1UEAxMkRXF1\n"
  "aWZheCBTZWN1cmUgR2xvYmFsIGVCdXNpbmVzcyBDQS0xMB4XDTk5MDYyMTA0\n"
  "MDAwMFoXDTIwMDYyMTA0MDAwMFowWjELMAkGA1UEBhMCVVMxHDAaBgNVBAoT\n"
  "E0VxdWlmYXggU2VjdXJlIEluYy4xLTArBgNVBAMTJEVxdWlmYXggU2VjdXJl\n"
  "IEdsb2JhbCBlQnVzaW5lc3MgQ0EtMTCBnzANBgkqhkiG9w0BAQEFAAOBjQAw\n"
  "gYkCgYEAuucXkAJlsTRVPEnCUdXfp9E3j9HngXNBUmCbnaEXJnitx7HoJpQy\n"
  "td4zjTov2/KaelpzmKNc6fuKcxtc58O/gGzNqfTWK8D3+ZmqY6KxRwIP1ORR\n"
  "OhI8bIpaVIRw28HFkM9yRcuoWcDNM50/o5brhTMhHD4ePmBudpxnhcXIw2EC\n"
  "AwEAAaNmMGQwEQYJYIZIAYb4QgEBBAQDAgAHMA8GA1UdEwEB/wQFMAMBAf8w\n"
  "HwYDVR0jBBgwFoAUvqigdHJQa0S3ySPY+6j/s1draGwwHQYDVR0OBBYEFL6o\n"
  "oHRyUGtEt8kj2Puo/7NXa2hsMA0GCSqGSIb3DQEBBAUAA4GBADDiAVGqx+pf\n"
  "2rnQZQ8w1j7aDRRJbpGTJxQx78T3LUX47Me/okENI7SS+RkAZ70Br83gcfxa\n"
  "z2TE4JaY0KNA4gGK7ycH8WUBikQtBmV1UsCGECAhX2xrD2yuCRyv8qIYNMR1\n"
  "pHMc8Y3c7635s3a0kr/clRAevsvIO1qEYBlWlKlV\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test V1 CA without basicConstraint. */
static const char *v1ca[] = {
  /* chain[0] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE/zCCA+egAwIBAgIQBSsgZODO6vk6ayagofBQJDANBgkqhkiG9w0BAQUFADCB\n"
  "sDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
  "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
  "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMh\n"
  "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBMB4XDTA4MDUwNTAwMDAw\n"
  "MFoXDTA5MDUyMjIzNTk1OVowczELMAkGA1UEBhMCVVMxETAPBgNVBAgTCElsbGlu\n"
  "b2lzMRAwDgYDVQQHFAdEdSBQYWdlMSQwIgYDVQQKFBtBcmdvbm5lIE5hdGlvbmFs\n"
  "IExhYm9yYXRvcnkxGTAXBgNVBAMUEGF1dGgyLml0LmFubC5nb3YwgZ8wDQYJKoZI\n"
  "hvcNAQEBBQADgY0AMIGJAoGBAMg6YPOXsPQedkLUug3RoMjv/OB+SfuDgGXxtef5\n"
  "iE0SjCcsKT5v+bfxt2+ccs7IN7kWn1luJ5NTb0ZrdE6LQoYp9oLsaX/ukOnxKUMY\n"
  "YhJJyHgutPtwyPvfZTZPpATWycJnZGIehY1S6thwxeofUyE3ykec2lalULzwXgel\n"
  "iC97AgMBAAGjggHTMIIBzzAJBgNVHRMEAjAAMAsGA1UdDwQEAwIFoDBEBgNVHR8E\n"
  "PTA7MDmgN6A1hjNodHRwOi8vU1ZSU2VjdXJlLWNybC52ZXJpc2lnbi5jb20vU1ZS\n"
  "U2VjdXJlMjAwNS5jcmwwRAYDVR0gBD0wOzA5BgtghkgBhvhFAQcXAzAqMCgGCCsG\n"
  "AQUFBwIBFhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMB0GA1UdJQQWMBQG\n"
  "CCsGAQUFBwMBBggrBgEFBQcDAjAfBgNVHSMEGDAWgBRv7K+g3Yqk7/UqEGctP1WC\n"
  "vNfvJTB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZl\n"
  "cmlzaWduLmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL1NWUlNlY3VyZS1haWEudmVy\n"
  "aXNpZ24uY29tL1NWUlNlY3VyZTIwMDUtYWlhLmNlcjBuBggrBgEFBQcBDARiMGCh\n"
  "XqBcMFowWDBWFglpbWFnZS9naWYwITAfMAcGBSsOAwIaBBRLa7kolgYMu9BSOJsp\n"
  "rEsHiyEFGDAmFiRodHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvMS5naWYw\n"
  "DQYJKoZIhvcNAQEFBQADggEBAEAKzE4gXeyjRDBSgAgWIaCozbWK+b1ct4aZhWZt\n"
  "ihAyASxsNgDjDJzkInINjXoO5FWuJHDuoyyWHBQBb7t96+KgFu/4ye90VUDNTuin\n"
  "mmqdOKeLSHVnlhfvGLCdrhWSWg/jZmAjYrXYRwkvxehl9IcHmOgNrHV3INdrSTdZ\n"
  "ZCVLL74tuMqhMMm/NJ0tdEmWgpJe+/0dky2F2gAB+mFXlyzFvCLoyS2Vl0PW/BxM\n"
  "Ly5t+scmAbgni9gzmFTNhbKHd0s2UE395z4ra6fUdZ0BClFgMDvUnb6kJ/uyKRSa\n"
  "h7uQbWFJbA8aNgGLvfTf6o9n+GwbZkcgtBgIVENt8wzqg2I=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIEnDCCBAWgAwIBAgIQdTN9mrDhIzuuLX3kRpFi1DANBgkqhkiG9w0BAQUFADBf\n"
  "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
  "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
  "HhcNMDUwMTE5MDAwMDAwWhcNMTUwMTE4MjM1OTU5WjCBsDELMAkGA1UEBhMCVVMx\n"
  "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
  "dCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQgaHR0cHM6Ly93d3cu\n"
  "dmVyaXNpZ24uY29tL3JwYSAoYykwNTEqMCgGA1UEAxMhVmVyaVNpZ24gQ2xhc3Mg\n"
  "MyBTZWN1cmUgU2VydmVyIENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC\n"
  "AQEAlcMhEo5AxQ0BX3ZeZpTZcyxYGSK4yfx6OZAqd3J8HT732FXjr0LLhzAC3Fus\n"
  "cOa4RLQrNeuT0hcFfstG1lxToDJRnXRkWPkMmgDqXkRJZHL0zRDihQr5NO6ziGap\n"
  "paRa0A6Yf1gNK1K7hql+LvqySHyN2y1fAXWijQY7i7RhB8m+Ipn4G9G1V2YETTX0\n"
  "kXGWtZkIJZuXyDrzILHdnpgMSmO3ps6wAc74k2rzDG6fsemEe4GYQeaB3D0s57Rr\n"
  "4578CBbXs9W5ZhKZfG1xyE2+xw/j+zet1XWHIWuG0EQUWlR5OZZpVsm5Mc2JYVjh\n"
  "2XYFBa33uQKvp/1HkaIiNFox0QIDAQABo4IBgTCCAX0wEgYDVR0TAQH/BAgwBgEB\n"
  "/wIBADBEBgNVHSAEPTA7MDkGC2CGSAGG+EUBBxcDMCowKAYIKwYBBQUHAgEWHGh0\n"
  "dHBzOi8vd3d3LnZlcmlzaWduLmNvbS9ycGEwMQYDVR0fBCowKDAmoCSgIoYgaHR0\n"
  "cDovL2NybC52ZXJpc2lnbi5jb20vcGNhMy5jcmwwDgYDVR0PAQH/BAQDAgEGMBEG\n"
  "CWCGSAGG+EIBAQQEAwIBBjApBgNVHREEIjAgpB4wHDEaMBgGA1UEAxMRQ2xhc3Mz\n"
  "Q0EyMDQ4LTEtNDUwHQYDVR0OBBYEFG/sr6DdiqTv9SoQZy0/VYK81+8lMIGABgNV\n"
  "HSMEeTB3oWOkYTBfMQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIElu\n"
  "Yy4xNzA1BgNVBAsTLkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlv\n"
  "biBBdXRob3JpdHmCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQEFBQADgYEA\n"
  "w34IRl2RNs9n3Nenr6+4IsOLBHTTsWC85v63RBKBWzFzFGNWxnIu0RoDQ1w4ClBK\n"
  "Tc3athmo9JkNr+P32PF1KGX2av6b9L1S2T/L2hbLpZ4ujmZSeD0m+v6UNohKlV4q\n"
  "TBnvbvqCPy0D79YoszcYz0KyNCFkR9MgazpM3OYDkAw=\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIICPDCCAaUCEHC65B0Q2Sk0tjjKewPMur8wDQYJKoZIhvcNAQECBQAwXzEL\n"
  "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQL\n"
  "Ey5DbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\n"
  "aXR5MB4XDTk2MDEyOTAwMDAwMFoXDTI4MDgwMTIzNTk1OVowXzELMAkGA1UE\n"
  "BhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMTcwNQYDVQQLEy5DbGFz\n"
  "cyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MIGf\n"
  "MA0GCSqGSIb3DQEBAQUAA4GNADCBiQKBgQDJXFme8huKARS0EN8EQNvjV69q\n"
  "RUCPhAwL0TPZ2RHP7gJYHyX3KqhEBarsAx94f56TuZoAqiN91qyFomNFx3In\n"
  "zPRMxnVx0jnvT0Lwdd8KkMaOIG+YD/isI19wKTakyYbnsZogy1Olhec9vn2a\n"
  "/iRFM9x2Fe0PonFkTGUugWhFpwIDAQABMA0GCSqGSIb3DQEBAgUAA4GBALtM\n"
  "EivPLCYATxQT3ab7/AoRhIzzKBxnki98tsX63/Dolbwdj2wsqFHMc9ikwFPw\n"
  "TtYmwHYBV4GSXiHx0bH/59AhWM1pF+NEHJwZRDmJXNycAA9WjQKZ7aKQRUzk\n"
  "uxCkPfAyAw7xzvjoyVGM5mKf5p/AfbdynMk2OmufTqj/ZA1k\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test CACert chain with RSA-MD5 signature. */
static const char *cacertrsamd5[] = {
  /* chain[0] (EE cert) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIE3zCCAsegAwIBAgICbmgwDQYJKoZIhvcNAQEFBQAwVDEUMBIGA1UEChMLQ0Fj\n"
  "ZXJ0IEluYy4xHjAcBgNVBAsTFWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZzEcMBoGA1UE\n"
  "AxMTQ0FjZXJ0IENsYXNzIDMgUm9vdDAeFw0wOTAxMTYyMjI5NDdaFw0xMTAxMTYy\n"
  "MjI5NDdaMBsxGTAXBgNVBAMTEGZyeS5zZXJ2ZXJhbWEuZGUwgZ8wDQYJKoZIhvcN\n"
  "AQEBBQADgY0AMIGJAoGBAMh3WSR8Dxw/zDAZpHMjA1To0HJIaoyR9TqzQfLgn7Yr\n"
  "sQFrRMefVMWYHiEFAVJTRcO5GuUtkw3IxALLlyNMl7xJbZESzRKw3Tz3NtM3DopB\n"
  "8L7rI8ANy7Hh6P5QRMWJ9OJyiLhSpAi0TuJeGr+kKovHRj64V2/NtoPgDsytHMt9\n"
  "AgMBAAGjggF2MIIBcjAMBgNVHRMBAf8EAjAAMDQGA1UdJQQtMCsGCCsGAQUFBwMC\n"
  "BggrBgEFBQcDAQYJYIZIAYb4QgQBBgorBgEEAYI3CgMDMAsGA1UdDwQEAwIFoDAz\n"
  "BggrBgEFBQcBAQQnMCUwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLmNhY2VydC5v\n"
  "cmcvMIHpBgNVHREEgeEwgd6CEGZyeS5zZXJ2ZXJhbWEuZGWgHgYIKwYBBQUHCAWg\n"
  "EgwQZnJ5LnNlcnZlcmFtYS5kZYIQZnJ5LnNlcnZlcmFtYS5kZaAeBggrBgEFBQcI\n"
  "BaASDBBmcnkuc2VydmVyYW1hLmRlghIqLmZyeS5zZXJ2ZXJhbWEuZGWgIAYIKwYB\n"
  "BQUHCAWgFAwSKi5mcnkuc2VydmVyYW1hLmRlggd6cHViLmRloBUGCCsGAQUFBwgF\n"
  "oAkMB3pwdWIuZGWCCSouenB1Yi5kZaAXBggrBgEFBQcIBaALDAkqLnpwdWIuZGUw\n"
  "DQYJKoZIhvcNAQEFBQADggIBAEWSsOlLbjdRjijMmOnDc2RcLQ5PQC9pjUW+bzGR\n"
  "KTJbf8Hf/wSdmHAam+UsIM6HzdQVi058dGyb8/NJQJD+9Dgv1m57x1prLerkt6xq\n"
  "UQCYmOpMxCJOykLqzEUnou9WtL5FaD+wBlOuqWFy0Cy2O3LHXkSkaMR+gdxC4pkI\n"
  "wSkI2SDdC0juvnoVI7iBaaIhYI/1FwV56hc6lxsAslf0NbtiiwhneVbHm5XRK1d4\n"
  "tabVKwOHnEuDyAnZd1yM1EqXKz+NwBlhoKWhC0fVUByID5A2WGEejBJcW/lVrYft\n"
  "4+sJpnwS+/VDS5yrDXMqMdYGE8TVMy7RsaoUdaeFQYv4Go48BBGDJB5uEkBJiSq8\n"
  "ViZA4iEKujBa5zKJ+CZXy3D/eHLBKUL+ayc9dLeeFTPZU0jYb83kE1wtlnWwF4J1\n"
  "8lUQI10nLFg+ALoZoAmFZej19XgbyG6im+ZRFuwrpV6F3HJRP+AMNInsLoQTuD9I\n"
  "l2gftVaIU1MqUmVMBcUeeNXG1BZ9vRonKzAC4Otfk1B6aW4Lz0E+sZ+HfCMicD3j\n"
  "N01KAeNZ64j8emgnLffurb7qUWbanTpMEzxrelBRufxJkXcn6BcFcxPBVgFnsMgF\n"
  "tP7e7N/cm55pI8Et+Gjp+ORJetSio118yu9bf7etSAJWOS6tQ2Ac7JeKP+a8jsvq\n"
  "Uyx7\n"
  "-----END CERTIFICATE-----\n",
  /* chain[1] (Class 3 CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIGCDCCA/CgAwIBAgIBATANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wNTEwMTQwNzM2NTVaFw0zMzAzMjgwNzM2NTVaMFQxFDAS\n"
  "BgNVBAoTC0NBY2VydCBJbmMuMR4wHAYDVQQLExVodHRwOi8vd3d3LkNBY2VydC5v\n"
  "cmcxHDAaBgNVBAMTE0NBY2VydCBDbGFzcyAzIFJvb3QwggIiMA0GCSqGSIb3DQEB\n"
  "AQUAA4ICDwAwggIKAoICAQCrSTURSHzSJn5TlM9Dqd0o10Iqi/OHeBlYfA+e2ol9\n"
  "4fvrcpANdKGWZKufoCSZc9riVXbHF3v1BKxGuMO+f2SNEGwk82GcwPKQ+lHm9WkB\n"
  "Y8MPVuJKQs/iRIwlKKjFeQl9RrmK8+nzNCkIReQcn8uUBByBqBSzmGXEQ+xOgo0J\n"
  "0b2qW42S0OzekMV/CsLj6+YxWl50PpczWejDAz1gM7/30W9HxM3uYoNSbi4ImqTZ\n"
  "FRiRpoWSR7CuSOtttyHshRpocjWr//AQXcD0lKdq1TuSfkyQBX6TwSyLpI5idBVx\n"
  "bgtxA+qvFTia1NIFcm+M+SvrWnIl+TlG43IbPgTDZCciECqKT1inA62+tC4T7V2q\n"
  "SNfVfdQqe1z6RgRQ5MwOQluM7dvyz/yWk+DbETZUYjQ4jwxgmzuXVjit89Jbi6Bb\n"
  "6k6WuHzX1aCGcEDTkSm3ojyt9Yy7zxqSiuQ0e8DYbF/pCsLDpyCaWt8sXVJcukfV\n"
  "m+8kKHA4IC/VfynAskEDaJLM4JzMl0tF7zoQCqtwOpiVcK01seqFK6QcgCExqa5g\n"
  "eoAmSAC4AcCTY1UikTxW56/bOiXzjzFU6iaLgVn5odFTEcV7nQP2dBHgbbEsPyyG\n"
  "kZlxmqZ3izRg0RS0LKydr4wQ05/EavhvE/xzWfdmQnQeiuP43NJvmJzLR5iVQAX7\n"
  "6QIDAQABo4G/MIG8MA8GA1UdEwEB/wQFMAMBAf8wXQYIKwYBBQUHAQEEUTBPMCMG\n"
  "CCsGAQUFBzABhhdodHRwOi8vb2NzcC5DQWNlcnQub3JnLzAoBggrBgEFBQcwAoYc\n"
  "aHR0cDovL3d3dy5DQWNlcnQub3JnL2NhLmNydDBKBgNVHSAEQzBBMD8GCCsGAQQB\n"
  "gZBKMDMwMQYIKwYBBQUHAgEWJWh0dHA6Ly93d3cuQ0FjZXJ0Lm9yZy9pbmRleC5w\n"
  "aHA/aWQ9MTAwDQYJKoZIhvcNAQEEBQADggIBAH8IiKHaGlBJ2on7oQhy84r3HsQ6\n"
  "tHlbIDCxRd7CXdNlafHCXVRUPIVfuXtCkcKZ/RtRm6tGpaEQU55tiKxzbiwzpvD0\n"
  "nuB1wT6IRanhZkP+VlrRekF490DaSjrxC1uluxYG5sLnk7mFTZdPsR44Q4Dvmw2M\n"
  "77inYACHV30eRBzLI++bPJmdr7UpHEV5FpZNJ23xHGzDwlVks7wU4vOkHx4y/CcV\n"
  "Bc/dLq4+gmF78CEQGPZE6lM5+dzQmiDgxrvgu1pPxJnIB721vaLbLmINQjRBvP+L\n"
  "ivVRIqqIMADisNS8vmW61QNXeZvo3MhN+FDtkaVSKKKs+zZYPumUK5FQhxvWXtaM\n"
  "zPcPEAxSTtAWYeXlCmy/F8dyRlecmPVsYGN6b165Ti/Iubm7aoW8mA3t+T6XhDSU\n"
  "rgCvoeXnkm5OvfPi2RSLXNLrAWygF6UtEOucekq9ve7O/e0iQKtwOIj1CodqwqsF\n"
  "YMlIBdpTwd5Ed2qz8zw87YC8pjhKKSRf/lk7myV6VmMAZLldpGJ9VzZPrYPvH5JT\n"
  "oI53V93lYRE9IwCQTDz6o2CTBKOvNfYOao9PSmCnhQVsRqGP9Md246FZV/dxssRu\n"
  "FFxtbUFm3xuTsdQAw+7Lzzw9IYCpX2Nl/N3gX6T0K/CFcUHUZyX7GrGXrtaZghNB\n"
  "0m6lG5kngOcLqagA\n"
  "-----END CERTIFICATE-----\n",
  /* chain[2] (Root CA) */
  "-----BEGIN CERTIFICATE-----\n"
  "MIIHPTCCBSWgAwIBAgIBADANBgkqhkiG9w0BAQQFADB5MRAwDgYDVQQKEwdSb290\n"
  "IENBMR4wHAYDVQQLExVodHRwOi8vd3d3LmNhY2VydC5vcmcxIjAgBgNVBAMTGUNB\n"
  "IENlcnQgU2lnbmluZyBBdXRob3JpdHkxITAfBgkqhkiG9w0BCQEWEnN1cHBvcnRA\n"
  "Y2FjZXJ0Lm9yZzAeFw0wMzAzMzAxMjI5NDlaFw0zMzAzMjkxMjI5NDlaMHkxEDAO\n"
  "BgNVBAoTB1Jvb3QgQ0ExHjAcBgNVBAsTFWh0dHA6Ly93d3cuY2FjZXJ0Lm9yZzEi\n"
  "MCAGA1UEAxMZQ0EgQ2VydCBTaWduaW5nIEF1dGhvcml0eTEhMB8GCSqGSIb3DQEJ\n"
  "ARYSc3VwcG9ydEBjYWNlcnQub3JnMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC\n"
  "CgKCAgEAziLA4kZ97DYoB1CW8qAzQIxL8TtmPzHlawI229Z89vGIj053NgVBlfkJ\n"
  "8BLPRoZzYLdufujAWGSuzbCtRRcMY/pnCujW0r8+55jE8Ez64AO7NV1sId6eINm6\n"
  "zWYyN3L69wj1x81YyY7nDl7qPv4coRQKFWyGhFtkZip6qUtTefWIonvuLwphK42y\n"
  "fk1WpRPs6tqSnqxEQR5YYGUFZvjARL3LlPdCfgv3ZWiYUQXw8wWRBB0bF4LsyFe7\n"
  "w2t6iPGwcswlWyCR7BYCEo8y6RcYSNDHBS4CMEK4JZwFaz+qOqfrU0j36NK2B5jc\n"
  "G8Y0f3/JHIJ6BVgrCFvzOKKrF11myZjXnhCLotLddJr3cQxyYN/Nb5gznZY0dj4k\n"
  "epKwDpUeb+agRThHqtdB7Uq3EvbXG4OKDy7YCbZZ16oE/9KTfWgu3YtLq1i6L43q\n"
  "laegw1SJpfvbi1EinbLDvhG+LJGGi5Z4rSDTii8aP8bQUWWHIbEZAWV/RRyH9XzQ\n"
  "QUxPKZgh/TMfdQwEUfoZd9vUFBzugcMd9Zi3aQaRIt0AUMyBMawSB3s42mhb5ivU\n"
  "fslfrejrckzzAeVLIL+aplfKkQABi6F1ITe1Yw1nPkZPcCBnzsXWWdsC4PDSy826\n"
  "YreQQejdIOQpvGQpQsgi3Hia/0PsmBsJUUtaWsJx8cTLc6nloQsCAwEAAaOCAc4w\n"
  "ggHKMB0GA1UdDgQWBBQWtTIb1Mfz4OaO873SsDrusjkY0TCBowYDVR0jBIGbMIGY\n"
  "gBQWtTIb1Mfz4OaO873SsDrusjkY0aF9pHsweTEQMA4GA1UEChMHUm9vdCBDQTEe\n"
  "MBwGA1UECxMVaHR0cDovL3d3dy5jYWNlcnQub3JnMSIwIAYDVQQDExlDQSBDZXJ0\n"
  "IFNpZ25pbmcgQXV0aG9yaXR5MSEwHwYJKoZIhvcNAQkBFhJzdXBwb3J0QGNhY2Vy\n"
  "dC5vcmeCAQAwDwYDVR0TAQH/BAUwAwEB/zAyBgNVHR8EKzApMCegJaAjhiFodHRw\n"
  "czovL3d3dy5jYWNlcnQub3JnL3Jldm9rZS5jcmwwMAYJYIZIAYb4QgEEBCMWIWh0\n"
  "dHBzOi8vd3d3LmNhY2VydC5vcmcvcmV2b2tlLmNybDA0BglghkgBhvhCAQgEJxYl\n"
  "aHR0cDovL3d3dy5jYWNlcnQub3JnL2luZGV4LnBocD9pZD0xMDBWBglghkgBhvhC\n"
  "AQ0ESRZHVG8gZ2V0IHlvdXIgb3duIGNlcnRpZmljYXRlIGZvciBGUkVFIGhlYWQg\n"
  "b3ZlciB0byBodHRwOi8vd3d3LmNhY2VydC5vcmcwDQYJKoZIhvcNAQEEBQADggIB\n"
  "ACjH7pyCArpcgBLKNQodgW+JapnM8mgPf6fhjViVPr3yBsOQWqy1YPaZQwGjiHCc\n"
  "nWKdpIevZ1gNMDY75q1I08t0AoZxPuIrA2jxNGJARjtT6ij0rPtmlVOKTV39O9lg\n"
  "18p5aTuxZZKmxoGCXJzN600BiqXfEVWqFcofN8CCmHBh22p8lqOOLlQ+TyGpkO/c\n"
  "gr/c6EWtTZBzCDyUZbAEmXZ/4rzCahWqlwQ3JNgelE5tDlG+1sSPypZt90Pf6DBl\n"
  "Jzt7u0NDY8RD97LsaMzhGY4i+5jhe1o+ATc7iwiwovOVThrLm82asduycPAtStvY\n"
  "sONvRUgzEv/+PDIqVPfE94rwiCPCR/5kenHA0R6mY7AHfqQv0wGP3J8rtsYIqQ+T\n"
  "SCX8Ev2fQtzzxD72V7DX3WnRBnc0CkvSyqD/HMaMyRa+xMwyN2hzXwj7UfdJUzYF\n"
  "CpUCTPJ5GhD22Dp1nPMd8aINcGeGG7MW9S/lpOt5hvk9C8JzC6WZrG/8Z7jlLwum\n"
  "GCSNe9FINSkYQKyTYOGWhlC0elnYjyELn8+CkcY7v2vcB5G5l1YjqrZslMZIBjzk\n"
  "zk6q5PYvCdxTby78dOs6Y5nCpqyJvKeyRKANihDjbPIky/qbn3BHLt4Ui9SyIAmW\n"
  "omTxJBzcoTWcFbLUvFUufQb1nA5V9FrWk9p2rSVzTMVD\n"
  "-----END CERTIFICATE-----\n",
  NULL
};

/* Test Certicom cert with ECC-SHA256 signature. */
static const char *ecc_cert[] = {
  /* chain[0] (ECC cert) */
"-----BEGIN CERTIFICATE-----\n"
"MIICbzCCAhSgAwIBAgIIZLkW6EZO5PQwCgYIKoZIzj0EAwIwgZsxFDASBgNVBAsT\n"
"C1NBTVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMH\n"
"VG9yb250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3Jn\n"
"IEVDQyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJD\n"
"QTAeFw0wOTA1MDcwMDAwMDBaFw0xNTA1MDEwMDAwMDBaMIGYMRQwEgYDVQQLEwtT\n"
"QU1QTEUgT05MWTEXMBUGA1UEChMOQ2VydGljb20gQ29ycC4xEDAOBgNVBAcTB1Rv\n"
"cm9udG8xEDAOBgNVBAgTB09udGFyaW8xNjA0BgNVBAMTLXRscy5zZWNnLm9yZyBF\n"
"Q0Mgc2VjcDI1NnIxIFNlcnZlciBDZXJ0aWZpY2F0ZTELMAkGA1UEBhMCQ0EwWTAT\n"
"BgcqhkjOPQIBBggqhkjOPQMBBwNCAATf63kPhr3D6a2scrHWVr0oOXQMnBDT6Jv/\n"
"ifqzt4/xTbXsZNEyD96nyh82sk0tM+FVfBlsIwGc7vqBfyq0mC/Io0MwQTAOBgNV\n"
"HQ8BAf8EBAMCA4gwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwEwFwYDVR0RBBAwDoIM\n"
"dGxzLnNlY2cub3JnMAoGCCqGSM49BAMCA0kAMEYCIQDfacZHsdsj6SXQ2hyJS4Do\n"
"SMclqGLo2Sop7hfAeEJA+wIhAOMo7eLya44SIcuzrLBpg29g5ZzYOeuEzRcg9mch\n"
"AB1w\n"
"-----END CERTIFICATE-----\n",
"-----BEGIN CERTIFICATE-----\n"
"MIICTjCCAfagAwIBAgIICvq6Bj3Av6EwCQYHKoZIzj0EATCBmzEUMBIGA1UECxML\n"
"U0FNUExFIE9OTFkxFzAVBgNVBAoTDkNlcnRpY29tIENvcnAuMRAwDgYDVQQHEwdU\n"
"b3JvbnRvMRAwDgYDVQQEEwdPbnRhcmlvMTkwNwYDVQQDEzB0bHMuc2VjZy5vcmcg\n"
"RUNDIHNlY3AyNTZyMSBDZXJ0aWZpY2F0ZSBBdXRob3JpdHkxCzAJBgNVBAYTAkNB\n"
"MB4XDTA2MDUwMTAwMDAwMFoXDTE1MDUwMTAwMDAwMFowgZsxFDASBgNVBAsTC1NB\n"
"TVBMRSBPTkxZMRcwFQYDVQQKEw5DZXJ0aWNvbSBDb3JwLjEQMA4GA1UEBxMHVG9y\n"
"b250bzEQMA4GA1UEBBMHT250YXJpbzE5MDcGA1UEAxMwdGxzLnNlY2cub3JnIEVD\n"
"QyBzZWNwMjU2cjEgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MQswCQYDVQQGEwJDQTBZ\n"
"MBMGByqGSM49AgEGCCqGSM49AwEHA0IABB2oofFVa6akTK6hpaJLs+6skdhn0sQp\n"
"uJwVwG99T0VZY8v7q6NMIWrpYQFmOxQyVVNlxWWyr2cLYJTyqx/zuDejIzAhMA4G\n"
"A1UdDwEB/wQEAwIBhjAPBgNVHRMBAf8EBTADAQH/MAkGByqGSM49BAEDRwAwRAIg\n"
"W2KAhfAKWFoh47A7muk8K10cGqOKRtb9lCgdOltj19oCIG+ZJQv5m+RnL4X2Ti0y\n"
"ZJzOOuzBQVGiUFwZdn1dLv4X\n"
"-----END CERTIFICATE-----\n",
  NULL
};

static const char *verisign_com_chain_g5[] = {
  /* chain[0] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIFXjCCBEagAwIBAgIQHYWDpKNVUzEFx4Pq8yjxbTANBgkqhkiG9w0BAQUFADCB\n"
 "tTELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2Ug\n"
 "YXQgaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMm\n"
 "VmVyaVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwHhcNMTQwMjI3\n"
 "MDAwMDAwWhcNMTUwMjI4MjM1OTU5WjBnMQswCQYDVQQGEwJVUzETMBEGA1UECBMK\n"
 "V2FzaGluZ3RvbjEQMA4GA1UEBxQHU2VhdHRsZTEYMBYGA1UEChQPQW1hem9uLmNv\n"
 "bSBJbmMuMRcwFQYDVQQDFA53d3cuYW1hem9uLmNvbTCCASIwDQYJKoZIhvcNAQEB\n"
 "BQADggEPADCCAQoCggEBAJdfieOPrf4Arf1Iled/ii97407ZnjpaB5xxm49Q4Pz3\n"
 "+5xmD0LYre7Cjn1A7W3ZlHki5zFVZpW9Jb/3PfSEDY5slyjkLD2jdl2gVefSthSZ\n"
 "tYxb5eYv79tIEN0U9AZ8/VaGwUokl8n1MitcECxNLMe4LqoVmS29nXITTTzX5t3I\n"
 "4dUeMBDNI+xgVpJSpxwzA+/L+wxoj5Sb4YJ/Y+iUknCkjX6PpaZMRWBEE0dqvG02\n"
 "qlxXesAV0nmKYvjbtqAyoW6vgjP85h6gJEESIqTTZy1HOgFpO8XT05CpGDcjhP1s\n"
 "TvXF7Vx1aj+xDidGLLW188G35oFKIhEyHQV2V7vzRAUCAwEAAaOCAbUwggGxMFAG\n"
 "A1UdEQRJMEeCEXVlZGF0YS5hbWF6b24uY29tggphbWF6b24uY29tgghhbXpuLmNv\n"
 "bYIMd3d3LmFtem4uY29tgg53d3cuYW1hem9uLmNvbTAJBgNVHRMEAjAAMA4GA1Ud\n"
 "DwEB/wQEAwIFoDAdBgNVHSUEFjAUBggrBgEFBQcDAQYIKwYBBQUHAwIwQwYDVR0g\n"
 "BDwwOjA4BgpghkgBhvhFAQc2MCowKAYIKwYBBQUHAgEWHGh0dHBzOi8vd3d3LnZl\n"
 "cmlzaWduLmNvbS9jcHMwHwYDVR0jBBgwFoAUDURcFlNEwYJ+HSCrJfQBY9i+eaUw\n"
 "RQYDVR0fBD4wPDA6oDigNoY0aHR0cDovL1NWUlNlY3VyZS1HMy1jcmwudmVyaXNp\n"
 "Z24uY29tL1NWUlNlY3VyZUczLmNybDB2BggrBgEFBQcBAQRqMGgwJAYIKwYBBQUH\n"
 "MAGGGGh0dHA6Ly9vY3NwLnZlcmlzaWduLmNvbTBABggrBgEFBQcwAoY0aHR0cDov\n"
 "L1NWUlNlY3VyZS1HMy1haWEudmVyaXNpZ24uY29tL1NWUlNlY3VyZUczLmNlcjAN\n"
 "BgkqhkiG9w0BAQUFAAOCAQEAOeZfjkI0yR/nutCMHp5/uB/evkB8qIYxh1KKbhPB\n"
 "TmpykmJxiLKrBBcBBu9kW5lMbNPSNclE4sCyN0dxCJHwPm7ubNUxsmDSTPTJOx0M\n"
 "zl0WZVaZ7eX3nw1kj8jSoK0f5n87RzKK85MwBFsEn73Z2pDvxTcd72BE0T1UJLcU\n"
 "2A5uHAJyvm2QpOWBIRKlJHIHCcu5xjj5yLnGO9id0cjOjEgj9f1Mo4hzawL5vQfk\n"
 "o/xFxAsA70bk2trv54kgLvhmAW+B6OpN3Z/xB4mWNOw3G8bg/u+pCLvd8KRO5V8K\n"
 "TlgO1NTaOgYC6OAF3USNMhuNZh6h5tWA3mA8rFr8ZsayhA==\n"
 "-----END CERTIFICATE-----\n",
  /* chain[1] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIF7DCCBNSgAwIBAgIQbsx6pacDIAm4zrz06VLUkTANBgkqhkiG9w0BAQUFADCB\n"
 "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
 "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
 "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
 "aG9yaXR5IC0gRzUwHhcNMTAwMjA4MDAwMDAwWhcNMjAwMjA3MjM1OTU5WjCBtTEL\n"
 "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
 "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTswOQYDVQQLEzJUZXJtcyBvZiB1c2UgYXQg\n"
 "aHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL3JwYSAoYykxMDEvMC0GA1UEAxMmVmVy\n"
 "aVNpZ24gQ2xhc3MgMyBTZWN1cmUgU2VydmVyIENBIC0gRzMwggEiMA0GCSqGSIb3\n"
 "DQEBAQUAA4IBDwAwggEKAoIBAQCxh4QfwgxF9byrJZenraI+nLr2wTm4i8rCrFbG\n"
 "5btljkRPTc5v7QlK1K9OEJxoiy6Ve4mbE8riNDTB81vzSXtig0iBdNGIeGwCU/m8\n"
 "f0MmV1gzgzszChew0E6RJK2GfWQS3HRKNKEdCuqWHQsV/KNLO85jiND4LQyUhhDK\n"
 "tpo9yus3nABINYYpUHjoRWPNGUFP9ZXse5jUxHGzUL4os4+guVOc9cosI6n9FAbo\n"
 "GLSa6Dxugf3kzTU2s1HTaewSulZub5tXxYsU5w7HnO1KVGrJTcW/EbGuHGeBy0RV\n"
 "M5l/JJs/U0V/hhrzPPptf4H1uErT9YU3HLWm0AnkGHs4TvoPAgMBAAGjggHfMIIB\n"
 "2zA0BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLnZlcmlz\n"
 "aWduLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEAMHAGA1UdIARpMGcwZQYLYIZIAYb4\n"
 "RQEHFwMwVjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVyaXNpZ24uY29tL2Nw\n"
 "czAqBggrBgEFBQcCAjAeGhxodHRwczovL3d3dy52ZXJpc2lnbi5jb20vcnBhMDQG\n"
 "A1UdHwQtMCswKaAnoCWGI2h0dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMtZzUu\n"
 "Y3JsMA4GA1UdDwEB/wQEAwIBBjBtBggrBgEFBQcBDARhMF+hXaBbMFkwVzBVFglp\n"
 "bWFnZS9naWYwITAfMAcGBSsOAwIaBBSP5dMahqyNjmvDz4Bq1EgYLHsZLjAlFiNo\n"
 "dHRwOi8vbG9nby52ZXJpc2lnbi5jb20vdnNsb2dvLmdpZjAoBgNVHREEITAfpB0w\n"
 "GzEZMBcGA1UEAxMQVmVyaVNpZ25NUEtJLTItNjAdBgNVHQ4EFgQUDURcFlNEwYJ+\n"
 "HSCrJfQBY9i+eaUwHwYDVR0jBBgwFoAUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJ\n"
 "KoZIhvcNAQEFBQADggEBAAyDJO/dwwzZWJz+NrbrioBL0aP3nfPMU++CnqOh5pfB\n"
 "WJ11bOAdG0z60cEtBcDqbrIicFXZIDNAMwfCZYP6j0M3m+oOmmxw7vacgDvZN/R6\n"
 "bezQGH1JSsqZxxkoor7YdyT3hSaGbYcFQEFn0Sc67dxIHSLNCwuLvPSxe/20majp\n"
 "dirhGi2HbnTTiN0eIsbfFrYrghQKlFzyUOyvzv9iNw2tZdMGQVPtAhTItVgooazg\n"
 "W+yzf5VK+wPIrSbb5mZ4EkrZn0L74ZjmQoObj49nJOhhGbXdzbULJgWOw27EyHW4\n"
 "Rs/iGAZeqa6ogZpHFt4MKGwlJ7net4RYxh84HqTEy2Y=\n"
 "-----END CERTIFICATE-----\n",
  /* chain[2] */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIExjCCBC+gAwIBAgIQNZcxh/OHOgcyfs5YDJt+2jANBgkqhkiG9w0BAQUFADBf\n"
 "MQswCQYDVQQGEwJVUzEXMBUGA1UEChMOVmVyaVNpZ24sIEluYy4xNzA1BgNVBAsT\n"
 "LkNsYXNzIDMgUHVibGljIFByaW1hcnkgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkw\n"
 "HhcNMDYxMTA4MDAwMDAwWhcNMjExMTA3MjM1OTU5WjCByjELMAkGA1UEBhMCVVMx\n"
 "FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVz\n"
 "dCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZv\n"
 "ciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAz\n"
 "IFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwggEi\n"
 "MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1nmAMqudLO07cfLw8\n"
 "RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbext0uz/o9+B1fs70Pb\n"
 "ZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIzSdhDY2pSS9KP6HBR\n"
 "TdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQGBO+QueQA5N06tRn/\n"
 "Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+rCpSx4/VBEnkjWNH\n"
 "iDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/NIeWiu5T6CUVAgMB\n"
 "AAGjggGRMIIBjTAPBgNVHRMBAf8EBTADAQH/MDEGA1UdHwQqMCgwJqAkoCKGIGh0\n"
 "dHA6Ly9jcmwudmVyaXNpZ24uY29tL3BjYTMuY3JsMA4GA1UdDwEB/wQEAwIBBjA9\n"
 "BgNVHSAENjA0MDIGBFUdIAAwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cudmVy\n"
 "aXNpZ24uY29tL2NwczAdBgNVHQ4EFgQUf9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwNAYD\n"
 "VR0lBC0wKwYJYIZIAYb4QgQBBgpghkgBhvhFAQgBBggrBgEFBQcDAQYIKwYBBQUH\n"
 "AwIwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAHBgUr\n"
 "DgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVyaXNp\n"
 "Z24uY29tL3ZzbG9nby5naWYwNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhho\n"
 "dHRwOi8vb2NzcC52ZXJpc2lnbi5jb20wDQYJKoZIhvcNAQEFBQADgYEADyWuSO0b\n"
 "M4VMDLXC1/5N1oMoTEFlYAALd0hxgv5/21oOIMzS6ke8ZEJhRDR0MIGBJopK90Rd\n"
 "fjSAqLiD4gnXbSPdie0oCL1jWhFXCMSe2uJoKK/dUDzsgiHYAMJVRFBwQa2DF3m6\n"
 "CPMr3u00HUSe0gST9MsFFy0JLS1j7/YmC3s=\n"
 "-----END CERTIFICATE-----\n",
 NULL,
  /* chain[4] - alt CA */
 "-----BEGIN CERTIFICATE-----\n"
 "MIIE0zCCA7ugAwIBAgIQGNrRniZ96LtKIVjNzGs7SjANBgkqhkiG9w0BAQUFADCB\n"
 "yjELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL\n"
 "ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJp\n"
 "U2lnbiwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxW\n"
 "ZXJpU2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0\n"
 "aG9yaXR5IC0gRzUwHhcNMDYxMTA4MDAwMDAwWhcNMzYwNzE2MjM1OTU5WjCByjEL\n"
 "MAkGA1UEBhMCVVMxFzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZW\n"
 "ZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2ln\n"
 "biwgSW5jLiAtIEZvciBhdXRob3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJp\n"
 "U2lnbiBDbGFzcyAzIFB1YmxpYyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9y\n"
 "aXR5IC0gRzUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCvJAgIKXo1\n"
 "nmAMqudLO07cfLw8RRy7K+D+KQL5VwijZIUVJ/XxrcgxiV0i6CqqpkKzj/i5Vbex\n"
 "t0uz/o9+B1fs70PbZmIVYc9gDaTY3vjgw2IIPVQT60nKWVSFJuUrjxuf6/WhkcIz\n"
 "SdhDY2pSS9KP6HBRTdGJaXvHcPaz3BJ023tdS1bTlr8Vd6Gw9KIl8q8ckmcY5fQG\n"
 "BO+QueQA5N06tRn/Arr0PO7gi+s3i+z016zy9vA9r911kTMZHRxAy3QkGSGT2RT+\n"
 "rCpSx4/VBEnkjWNHiDxpg8v+R70rfk/Fla4OndTRQ8Bnc+MUCH7lP59zuDMKz10/\n"
 "NIeWiu5T6CUVAgMBAAGjgbIwga8wDwYDVR0TAQH/BAUwAwEB/zAOBgNVHQ8BAf8E\n"
 "BAMCAQYwbQYIKwYBBQUHAQwEYTBfoV2gWzBZMFcwVRYJaW1hZ2UvZ2lmMCEwHzAH\n"
 "BgUrDgMCGgQUj+XTGoasjY5rw8+AatRIGCx7GS4wJRYjaHR0cDovL2xvZ28udmVy\n"
 "aXNpZ24uY29tL3ZzbG9nby5naWYwHQYDVR0OBBYEFH/TZafC3ey78DAJ80M5+gKv\n"
 "MzEzMA0GCSqGSIb3DQEBBQUAA4IBAQCTJEowX2LP2BqYLz3q3JktvXf2pXkiOOzE\n"
 "p6B4Eq1iDkVwZMXnl2YtmAl+X6/WzChl8gGqCBpH3vn5fJJaCGkgDdk+bW48DW7Y\n"
 "5gaRQBi5+MHt39tBquCWIMnNZBU4gcmU7qKEKQsTb47bDN0lAtukixlE0kF6BWlK\n"
 "WE9gyn6CagsCqiUXObXbf+eEZSqVir2G3l6BFoMtEMze/aiCKm0oHw0LxOXnGiYZ\n"
 "4fQRbxC1lfznQgUy286dUV4otp6F01vvpX1FQHKOtw5rDgb7MzVIcbidJ4vEZV8N\n"
 "hnacRHr2lVz2XTIIM6RUthg/aFzyQkqFOFSDX9HoLPKsEdao7WNq\n"
 "-----END CERTIFICATE-----\n",
 NULL
};

static struct
{
  const char *name;
  const char **chain;
  const char **ca;
  unsigned int verify_flags;
  unsigned int expected_verify_result;
  time_t expected_time;
} chains[] =
{
  { "CVE-2014-0092", cve_2014_0092_check, &cve_2014_0092_check[1],
    0, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID, 1412850586},
  { "CVE-2008-4989", cve_2008_4989_chain, &cve_2008_4989_chain[2],
    0, GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_INVALID },
  { "amazon.com ok", verisign_com_chain_g5, &verisign_com_chain_g5[4],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LOW),
    0 },
  { "verisign.com v1 fail", verisign_com_chain, &verisign_com_chain[3],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "verisign.com v1 ok", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LOW),
    0 },
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_LEGACY),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "verisign.com v1 not ok due to profile", verisign_com_chain, &verisign_com_chain[3],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH),
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "citibank.com v1 fail", citibank_com_chain, &citibank_com_chain[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "expired self signed", pem_self_cert, &pem_self_cert[0],
    0, GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "self signed", pem_self_cert, &pem_self_cert[0],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },
  { "ca=false", thea_chain, &thea_chain[1],
    0,
    GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "ca=false2", thea_chain, &thea_chain[1],
    0, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "hbci v1 fail", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID},
  { "hbci v1 ok expired", hbci_chain, &hbci_chain[2],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "hbci v1 ok", hbci_chain, &hbci_chain[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0 },
  { "rsa-md5 fail", mayfirst_chain, &mayfirst_chain[1],
    0, GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "rsa-md5 not ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD2,
    GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "rsa-md5 not ok2", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "rsa-md5 ok", mayfirst_chain, &mayfirst_chain[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5, 0 },
  { "v1ca fail", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },

  { "pathlen fail", pathlen_check, &pathlen_check[2],
    GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE},

  /* Test whether a V1 root certificate is correctly accepted */
  { "v1root fail", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DO_NOT_ALLOW_X509_V1_CA_CRT | GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "v1root ok", v1_root_check, &v1_root_check[1],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },

  /* test whether a v1 intermediate certificate is rejected */
  { "v1invalid fail", v1_intermed_check, &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, GNUTLS_CERT_SIGNER_NOT_CA | GNUTLS_CERT_INVALID },
  { "v1 leaf ok", &v1_intermed_check[1], &v1_intermed_check[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS, 0 },

  { "v1ca expired", v1ca, &v1ca[2],
    0,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID  },
  { "v1ca ok", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS,
    0 },
  { "v1ca2 expired", v1ca, &v1ca[2],
    GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    GNUTLS_CERT_EXPIRED | GNUTLS_CERT_INVALID },
  { "v1ca2 ok", v1ca, &v1ca[2],
    GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_ALLOW_ANY_X509_V1_CA_CRT,
    0 },
  { "cacertrsamd5 fail", cacertrsamd5, &cacertrsamd5[2],
    0, GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "cacertrsamd5 ok", cacertrsamd5, &cacertrsamd5[2],
    GNUTLS_VERIFY_ALLOW_SIGN_RSA_MD5, 0 },
  { "cacertrsamd5 short-cut not ok", cacertrsamd5, &cacertrsamd5[0],
    GNUTLS_VERIFY_DO_NOT_ALLOW_SAME,
    GNUTLS_CERT_SIGNER_NOT_FOUND | GNUTLS_CERT_INVALID },
  { "cacertrsamd5 short-cut ok", cacertrsamd5, &cacertrsamd5[1],
    0, 0 },
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_HIGH), 0 },
  { "ecc cert ok", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB128), 0 },
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_ULTRA), 
  	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "ecc cert not ok (due to profile)", ecc_cert, &ecc_cert[1], GNUTLS_PROFILE_TO_VFLAGS(GNUTLS_PROFILE_SUITEB192), 
  	GNUTLS_CERT_INSECURE_ALGORITHM | GNUTLS_CERT_INVALID },
  { "name constraints chain ok1", nc_good1, &nc_good1[4], 0, 0, 1412850586 },
  { "name constraints chain bad1", nc_bad1, &nc_bad1[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 1412850586},
  { "name constraints chain bad2", nc_bad2, &nc_bad2[4], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 1412850586},
  { "name constraints chain bad3", nc_bad3, &nc_bad3[2], 0, GNUTLS_CERT_INVALID | GNUTLS_CERT_SIGNER_CONSTRAINTS_FAILURE, 1412850586},
  { "not-modified", modified2, &modified2[3], 0, 0, 1412850586},
  { NULL, NULL, NULL, 0, 0}
};
/* *INDENT-ON* */
