/*
 *  quartzgen.c
 *  graphviz
 *
 *  Created by Glen Low on Tue Nov 25 2003.
 *  Copyright (c) 2003, Pixelglow Software. All rights reserved.
 *  http://www.pixelglow.com/graphviz/
 *  graphviz@pixelglow.com
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted
 *  provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions
 *    and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions
 *    and the following disclaimer in the documentation and/or other materials provided with the distribution.
 *  * Neither the name of Pixelglow Software nor the names of its contributors may be used to endorse or
 *    promote products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ApplicationServices/ApplicationServices.h>
#include <QuickTime/QuickTime.h>
#include <Quicktime/QuicktimeComponents.k.h>

#include "render.h"

#define MAX_QD 32767

#define PDF_MAGIC "%PDF-"
#define JPEG_MAGIC "\xFF\xD8\xFF\xE0"
#define PNG_MAGIC "\x89PNG\x0D\x0A\x1A\x0A"
#define PS_MAGIC "%!PS-Adobe-"
#define EPS_MAGIC "\xC5\xD0\xD3\xC6"

typedef struct graphics_context {
    ATSURGBAlphaColor pencolor;
    ATSURGBAlphaColor fillcolor;
    char* fontname;
    double fontsize;
    int visible;
} graphics_context;

static TECObjectRef converter = NULL;				/* Unicode text converter */
static CFMutableDictionaryRef text_cache = NULL;	/* cache of ATSUI layouts, for measuring & drawing text. */
static CFMutableDictionaryRef image_cache = NULL;	/* cache of shape file images */
static Component cfile = NULL;						/* data handler component for FILE* */

static int onetime = TRUE;						/* whether more than one graph is being drawn */

static CGContextRef graphics = NULL;			/* graphics context to draw into */
static CFDictionaryRef auxiliary_info = NULL;    /* title, user, creator for PDF */
static CGRect media_box;						/* media box for PDF */
// static float scale_factor = 1.0;				/* the scale, used to derive dashes */
static node_t* current_node = NULL;				/* the current node */

static double dpi;
static double dev_scale;
static double zoom;
static double comp_scale;
static pointf graph_focus;
static point view_port;

#define STACKSIZE 32
static graphics_context context [STACKSIZE];
static graphics_context* current_context = context;

/* error handling */

static void check_status (char* context, OSStatus status)
{
	if (status)
		agerr (AGWARN, "%s status = %d.\n", context, status);
}

static void check_null (char* context, const void* ptr)
{
	if (!ptr)
		agerr (AGERR, "%s = null.\n", context);
}

static CFHashCode djb2_hash (char *str)
{
	CFHashCode hash = 5381;
	while (*str++)
		hash = ((hash << 5) + hash) + *str;
		
    return hash;
}

/* text cache */

typedef struct text_key {
    char* str;
    char* fontname;
    double fontsize;
} text_key;

typedef struct text_value {
    ConstUniCharArrayPtr buffer;
    ATSUStyle style;
    ATSUTextLayout layout;
} text_value;

static const void* text_key_retain (CFAllocatorRef allocator, const void *value)
{
    /* allocate fresh memory and copy from stack version */
    text_key* copy;
	check_null ("CFAllocatorAllocate", copy = CFAllocatorAllocate(allocator,sizeof(text_key),0));
	text_key* original = (text_key*) value;
	copy->str = strdup (original->str);
	copy->fontname = strdup (original->fontname);
	copy->fontsize = original->fontsize;
    return copy;
}

static void text_key_release (CFAllocatorRef allocator, const void *value)
{
	text_key* key = (text_key*) value;
	free (key->str);
	free (key->fontname);
    CFAllocatorDeallocate(allocator,(void*)value);
}

static Boolean text_key_equals (const void *value1, const void *value2)
{
    const text_key* key1 = (text_key*)value1;
    const text_key* key2 = (text_key*)value2;
    return streq(key1->str,key2->str) && streq(key1->fontname,key2->fontname) && key1->fontsize == key2->fontsize;
}

static CFHashCode text_key_hash (const void *value)
{
	/* since fontname, fontsize are likely to remain constant, we hash only on str using djb2 */
    return djb2_hash (((text_key*)value)->str);
}

static const void* text_value_retain (CFAllocatorRef allocator, const void *value)
{
    /* already allocated in get_text */
    return value;
}

static void text_value_release (CFAllocatorRef allocator, const void *value)
{
    text_value* val = (text_value*)value;
    ATSUDisposeTextLayout(val->layout);
    ATSUDisposeStyle(val->style);
	CFAllocatorDeallocate(kCFAllocatorDefault,(void*)val->buffer);   /* was allocated in get_text */
    CFAllocatorDeallocate(kCFAllocatorDefault,val);		/* was allocated in get_text */
}

static CFDictionaryKeyCallBacks text_key_callbacks = {
    NULL,
    text_key_retain,
    text_key_release,
    NULL,  /* copyDescription */
    text_key_equals,
    text_key_hash
};

static CFDictionaryValueCallBacks text_value_callbacks = {
    NULL,
    text_value_retain,
    text_value_release,
    NULL,  /* copyDescription */
    NULL /* equals */
};

static const void* image_key_retain (CFAllocatorRef allocator, const void *value)
{
	return strdup ((char*)value);
}

static void image_key_release (CFAllocatorRef allocator, const void *value)
{
	free ((void*)value);
}

static Boolean image_key_equals (const void *value1, const void *value2)
{
    return streq ((char*)value1, (char*)value2);
}

static CFHashCode image_key_hash (const void *value)
{
	return djb2_hash ((char*)value);
}

static CFDictionaryKeyCallBacks image_key_callbacks = {
    NULL,
    image_key_retain,
    image_key_release,
    NULL,  /* copyDescription */
    image_key_equals,
    image_key_hash
};

static void
release_data (void *info, const void *data, size_t size)
{
	CFRelease (info);
}

/* file_consumer */

static size_t
file_consumer_put_bytes(void* info, const void* buffer, size_t count)
{
    return fwrite(buffer, 1, count, (FILE*) info);
}

static CGDataConsumerCallbacks file_consumer_callbacks = {
    file_consumer_put_bytes,
    NULL
};

/* convert_consumer */

static size_t
convert_consumer_put_bytes(void* info, const void* buffer, size_t count)
{
	CFDataAppendBytes((CFMutableDataRef)info, (const UInt8*)buffer, count);
    return count;
}

static CGDataConsumerCallbacks convert_consumer_callbacks = {
    convert_consumer_put_bytes,
    NULL
};

static CGPSConverterCallbacks converter_callbacks = {
	0,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

/* cfile_handler */

#define CFILE_TYPE 'CFIL'

ComponentDescription cfile_description = {
	DataHandlerType,
	CFILE_TYPE,
	CFILE_TYPE,
	0,
	0
};

static int cfile_select(short selector, ProcPtr *proc, ProcInfoType *proc_info);

static ComponentResult cfile_open(Handle storage, DataHandler self)   
{
	Handle new_storage;
	check_null ("NewHandle", new_storage = NewHandle (sizeof (FILE*)));
	SetComponentInstanceStorage (self, new_storage);
	return noErr;
}

static ComponentResult cfile_close(Handle storage, DataHandler self)   
{
	DisposeHandle (storage);
	return noErr;
}

static ComponentResult cfile_cando(Handle storage, short selector)   
{
	ProcPtr proc;
	ProcInfoType proc_info;
	return cfile_select(selector, &proc, &proc_info);
}

static ComponentResult cfile_putdata(Handle storage, Handle data, long hoffset, long *offset, long size)
{
	FILE* stream = *(FILE**)*storage;
	*offset = ftell(stream);
	HLock(data);
    fwrite(*data + hoffset, 1, size, stream);	
	HUnlock(data);
	return noErr;
}

static ComponentResult cfile_canusedataref (Handle storage, Handle dataref, long *use_flags)
{
	*use_flags = kDataHCanWrite;
	return noErr;
}

static ComponentResult cfile_setdataref (Handle storage, Handle dataref)
{
	*(FILE**)*storage = *(FILE**)*dataref;
	return noErr;
}

static ComponentResult cfile_getdataref (Handle storage, Handle *dataref)
{
	*dataref = NewHandle(sizeof(FILE*));
	*(FILE**)*dataref = *(FILE**)*storage;
	return noErr;
}

static ComponentResult cfile_comparedataref (Handle storage, Handle dataref, Boolean *equal)
{
	*equal = *(FILE**)*dataref == *(FILE**)*storage;
	return noErr;
}

static ComponentResult cfile_write(Handle storage, Ptr data, long offset, long size, DataHCompletionUPP completion, long refcon)
{
	FILE* stream = *(FILE**)*storage;
	fseek(stream,offset,SEEK_SET);
    fwrite(data, 1, size, stream);
	
	return noErr;
}

static int cfile_select(short selector, ProcPtr *proc, ProcInfoType *proc_info)
{
	switch (selector) {
		case kComponentOpenSelect:
			*proc = (ProcPtr) &cfile_open;
			*proc_info = uppCallComponentOpenProcInfo;
			return 1;
		case kComponentCloseSelect:
			*proc = (ProcPtr) &cfile_close;
			*proc_info = uppCallComponentCloseProcInfo;
			return 1;
		case kComponentCanDoSelect:
			*proc = (ProcPtr) &cfile_cando;
			*proc_info = uppCallComponentCanDoProcInfo;
			return 1;
		case kDataHPutDataSelect:
			*proc = (ProcPtr) &cfile_putdata;
			*proc_info = uppDataHPutDataProcInfo;
			return 1;
		case kDataHCanUseDataRefSelect:
			*proc = (ProcPtr) &cfile_canusedataref;
			*proc_info = uppDataHCanUseDataRefProcInfo;
			return 1;
		case kDataHSetDataRefSelect:
			*proc = (ProcPtr) &cfile_setdataref;
			*proc_info = uppDataHSetDataRefProcInfo;
			return 1;
		case kDataHGetDataRefSelect:
			*proc = (ProcPtr) &cfile_getdataref;
			*proc_info = uppDataHGetDataRefProcInfo;
			return 1;
		case kDataHCompareDataRefSelect:
			*proc = (ProcPtr) &cfile_comparedataref;
			*proc_info = uppDataHCompareDataRefProcInfo;
			return 1;
		case kDataHWriteSelect:
			*proc = (ProcPtr) &cfile_write;
			*proc_info = uppDataHWriteProcInfo;
			return 1;

		case kDataHOpenForWriteSelect:
		case kDataHSetFileSizeSelect:
		case kDataHSetMacOSFileTypeSelect:
			*proc = NULL;
			*proc_info = NULL;
			return 1;
			
		default:
			return 0;
	}
}

static ComponentResult cfile_entry(ComponentParameters *params, Handle storage)
{
	ProcPtr proc = NULL;
	ProcInfoType proc_info = NULL;
	if (cfile_select(params->what, &proc, &proc_info))
		return proc ?
			CallComponentFunctionWithStorageProcInfo (storage, params, proc, proc_info) : noErr;
	else
		return badComponentSelector;
}

static text_value* fetch_text (char* str, char* fontname, double fontsize)
{
	text_key key = {str,fontname,fontsize};
    
	if (!text_cache)
		check_null ("CFDictionaryCreateMutable",
			text_cache = CFDictionaryCreateMutable(kCFAllocatorDefault,0,&text_key_callbacks,&text_value_callbacks));
	
	text_value* value;
	if (!CFDictionaryGetValueIfPresent(text_cache,&key,(const void**)&value)) {
		check_null ("CFAllocatorAllocate", value = CFAllocatorAllocate(kCFAllocatorDefault,sizeof(text_value),0));
		
		/* set style to matched font and given size */
		
		ATSUFontID font;
		check_status ("ATSUFindFontFromName",
			ATSUFindFontFromName(fontname,
			strlen(fontname),
			kFontPostscriptName,
			kFontNoPlatformCode,
			kFontNoScriptCode,
			kFontNoLanguageCode,
			&font));
		Fixed size = X2Fix(fontsize);
		
		ATSUAttributeTag style_tags[] = {kATSUFontTag,kATSUSizeTag};
		ByteCount style_sizes[] = {sizeof(ATSUFontID),sizeof(Fixed)};
		ATSUAttributeValuePtr style_values[] = {&font,&size};

		check_status ("ATSUCreateStyle", ATSUCreateStyle(&value->style));
		check_status ("ATSUSetAttributes", ATSUSetAttributes(value->style,2,style_tags,style_sizes,style_values));
		
		/* convert str into Unicode */
		
		ByteCount ail, aol;
		ByteCount len = strlen(str);
		if (!converter)
			check_status ("TECCreateConverter",
				TECCreateConverter(&converter,
				kCFStringEncodingUTF8,
				kCFStringEncodingUnicode));
		check_null ("CFAllocatorAllocate", value->buffer = CFAllocatorAllocate(kCFAllocatorDefault, len ? 2*len : 1, 0));
		if (len)
			check_status ("TECConvertText", TECConvertText(converter,str,len,&ail, (TextPtr)value->buffer, 2*len, &aol));
		
		/* create layout with Unicode text and style run for all text */
		UniCharCount run = kATSUToTextEnd;
		check_status ("ATSUCreateTextLayoutWithTextPtr",
			ATSUCreateTextLayoutWithTextPtr(value->buffer,
			0,
			aol / 2,
			aol / 2,
			1,
			&run,
			&value->style,
			&value->layout));

		CFDictionaryAddValue(text_cache,&key,value);
    }
		
    return value;
}

static CFTypeRef fetch_image (char* shapeimagefile)
{
	if (!image_cache)
		check_null ("CFDictionaryCreateMutable",
			image_cache = CFDictionaryCreateMutable (kCFAllocatorDefault, 0, &image_key_callbacks, &kCFTypeDictionaryValueCallBacks));
		
	CFTypeRef image;
	if (!CFDictionaryGetValueIfPresent(image_cache, shapeimagefile, &image)) {
		/* determine extension */
		char* suffix = strrchr (shapeimagefile,'.');
		if (!suffix) suffix = shapeimagefile; else suffix++;
		
		/* get the URL to the shape: if it has a scheme, it's a real URL; if not, we take as a file system path */
		CFURLRef url = CFURLCreateWithBytes (
			kCFAllocatorDefault,
			shapeimagefile,
			strlen (shapeimagefile),
			kCFStringEncodingUTF8,
			NULL); // no base URL
		CFStringRef scheme = CFURLCopyScheme (url);
		if (scheme)
			CFRelease (scheme);
		else {
			CFRelease (url);
			url = CFURLCreateFromFileSystemRepresentation (
				kCFAllocatorDefault,
				shapeimagefile,
				strlen (shapeimagefile),
				NULL); // not a directory
		}
		
		CFDataRef data;
		CFDictionaryRef properties;
		SInt32 error;
		
		/* load entire url */
		if (CFURLCreateDataAndPropertiesFromResource (
			kCFAllocatorDefault,
			url,
			&data,
			&properties,
			NULL, // no desired properties
			&error)) {
			
			CFStringRef content_type = (CFStringRef) CFDictionaryGetValue (properties, CFSTR ("Content-Type"));
			const UInt8* bytes = CFDataGetBytePtr (data);
			CFIndex length = CFDataGetLength (data);

			/* it's a PDF */
			if (streq (suffix, "pdf")
				|| (content_type && CFEqual (content_type, CFSTR ("application/pdf")))
				|| (length >= strlen (PDF_MAGIC) && !memcmp (bytes, PDF_MAGIC, strlen (PDF_MAGIC)))) {
				CFRetain (data);	// balanced by release_data callback
				CGDataProviderRef provider;
				check_null ("CGDataProviderCreateWithData", provider = CGDataProviderCreateWithData (
					(void*)data,
					bytes,
					length,
					&release_data));
				check_null ("CGPDFDocumentCreateWithProvider", image = CGPDFDocumentCreateWithProvider (provider));
				CGDataProviderRelease (provider);
			}
			/* it's a JPEG */
			else if (streq (suffix, "jpg") || streq (suffix, "jpeg") || streq (suffix, "jpe")
				|| (content_type && CFEqual (content_type, CFSTR ("image/jpeg")))
				|| (length >= strlen (JPEG_MAGIC) && !memcmp (bytes, JPEG_MAGIC, strlen (JPEG_MAGIC)))) {
				CGDataProviderRef provider;
				check_null ("CGDataProviderCreateWithData", provider = CGDataProviderCreateWithData (
					(void*)data,
					bytes,
					length,
					&release_data));
				if (provider) CFRetain (data);	// balanced by release_data callback
				check_null ("CGImageCreateWithJPEGDataProvider", image = CGImageCreateWithJPEGDataProvider (
					provider,
					NULL, // no decode
					1, // should interpolate
					kCGRenderingIntentDefault));
				CGDataProviderRelease (provider);
			}
			/* it's a PNG */
			else if (streq (suffix, "png")
				|| (content_type && CFEqual (content_type, CFSTR ("image/png")))
				|| (length >= strlen (PNG_MAGIC) && !memcmp (bytes, PNG_MAGIC, strlen (PNG_MAGIC)))) {
				CGDataProviderRef provider;
				check_null ("CGDataProviderCreateWithData", provider = CGDataProviderCreateWithData (
					(void*)data,
					bytes,
					length,
					&release_data));
				if (provider) CFRetain (data);	// balanced by release_data callback
				check_null ("CGImageCreateWithPNGDataProvider", image = CGImageCreateWithPNGDataProvider (
					provider,
					NULL, // no decode
					1, // should interpolate
					kCGRenderingIntentDefault));
				CGDataProviderRelease (provider);
			}
			/* we have the PS to PDF converter, so try that */
			else if (CGPSConverterCreate && CGPSConverterConvert &&
				(streq (suffix, "ps") || streq (suffix, "eps") || streq (suffix, "epi") || streq (suffix, "epsf") || streq (suffix, "epsi")
				|| (content_type && (CFEqual (content_type, CFSTR ("application/postscript")) || CFEqual (content_type, CFSTR ("application/eps"))))
				|| (length >= strlen (PS_MAGIC) && !memcmp (bytes, PS_MAGIC, strlen (PS_MAGIC)))
				|| (length >= strlen (EPS_MAGIC) && !memcmp (bytes, EPS_MAGIC, strlen (EPS_MAGIC)))))
				{
					CGDataProviderRef provider;
					check_null ("CGDataProviderCreateWithData", provider = CGDataProviderCreateWithData (
						(void*)data,
						bytes,
						length,
						&release_data));
					if (provider) CFRetain (data);
					
					CFMutableDataRef convert;
					check_null ("CFDataCreateMutable", convert = CFDataCreateMutable (kCFAllocatorDefault, 0));
					
					CGDataConsumerRef convert_consumer;
					check_null ("CGDataConsumerCreate", convert_consumer = CGDataConsumerCreate (convert, &convert_consumer_callbacks));
					
					CGPSConverterRef converter;
					check_null ("CGPSConverterCreate", converter = CGPSConverterCreate (NULL, &converter_callbacks, NULL));
					CGPSConverterConvert (converter, provider, convert_consumer, NULL);
					CFRelease (converter);
					CFRelease (convert_consumer);
					
					CGDataProviderRef convert_provider;
					check_null ("CGDataProviderCreateWithData", convert_provider = CGDataProviderCreateWithData (
						(void*)convert,
						CFDataGetBytePtr(convert),
						CFDataGetLength(convert),
						&release_data));
					if (convert_provider) CFRetain (data);
					check_null ("CGPDFDocumentCreateWithProvider", image = CGPDFDocumentCreateWithProvider (convert_provider));
					CFRelease (convert_provider);
						
					CFRelease (converter);
					CGDataProviderRelease (provider);
				}
			/* try Quicktime, if available */
			else if (GraphicsImportCreateCGImage) {
				Handle data_ref;
				check_null ("NewHandle", data_ref = NewHandle (sizeof (PointerDataRefRecord)));
				(*(PointerDataRef)data_ref)->data = (void*)bytes;
				(*(PointerDataRef)data_ref)->dataLength = length;
			
				DataHandler handler;
				Handle extension;
			
				check_status ("OpenADataHandler", OpenADataHandler (
					data_ref,
					PointerDataHandlerSubType,
					NULL, 
					0,
					NULL,
					kDataHCanRead,
					&handler));
							
				/* set MIME type, if any; we assume any mime type <= 255 chars (Pascal strings are limited to 255 chars) */
				CFIndex type_length;
				if (content_type && (type_length = CFStringGetLength (content_type)) <= 255) {
					extension = NewHandle (type_length + 1);
					CFStringGetPascalString (content_type, *extension, type_length + 1, kCFStringEncodingASCII);
					check_status ("DataHSetDataRefExtension", DataHSetDataRefExtension (
						handler,
						extension, 
						kDataRefExtensionFileName));
					DisposeHandle (extension);
				}
							
				/* set file name using up to 255 chars of the last path component (Pascal strings are limited to 255 chars) */
				char* lastpath = strrchr (shapeimagefile, '/');
				if (lastpath)
					++lastpath;
				else
					lastpath = shapeimagefile;
				if (strlen (lastpath) > 255)
					lastpath += strlen (lastpath) - 255;					
				extension = NewHandle (strlen (lastpath) + 1);
				CopyCStringToPascal (lastpath, *extension);
				check_status ("DataHSetDataRefExtension", DataHSetDataRefExtension (
					handler,
					extension, 
					kDataRefExtensionFileName));
				DisposeHandle (extension);
							
				CloseComponent (handler);
				
				/* open the graphics importer for the data, and grab the image thereof */
				GraphicsImportComponent importer;
				check_status ("GetGraphicsImporterForDataRef", GetGraphicsImporterForDataRef (
					data_ref,
					PointerDataHandlerSubType, 
					&importer));
				check_status ("GraphicsImportCreateCGImage", GraphicsImportCreateCGImage (
					importer,
					(CGImageRef*) &image, 
					kGraphicsImportCreateCGImageUsingCurrentSettings));
								
				CloseComponent (importer);
			}
			
			/* did create an image from the shape, so associate it */
			CFDictionaryAddValue (image_cache, shapeimagefile, image);

			CFRelease (data);
			CFRelease (properties);
		}
		else
			check_status ("CFURLCreateDataAndPropertiesFromResource", error);
	}

	return image;
}

static CGPathDrawingMode
get_mode(int filled)
{
 	if (filled)
		return current_context->pencolor.red == current_context->fillcolor.red
			&& current_context->pencolor.green == current_context->fillcolor.green
			&& current_context->pencolor.blue == current_context->fillcolor.blue
			&& current_context->pencolor.alpha == current_context->fillcolor.alpha
			? kCGPathFill : kCGPathFillStroke;
    else
		return kCGPathStroke;
}

static void
quartz_reset(void)
{
	onetime = TRUE;
}
   
static void
quartz_begin_job_for_pdf(FILE* ofp, graph_t* g, char** lib, char* user, char* info[], point pages)
{
	CFStringRef auxiliary_keys [] = {
		CFSTR ("Title"),
		CFSTR ("Author"),
		CFSTR ("Creator")
	};
    
    CFStringRef auxiliary_values [] = {
		CFStringCreateWithCString(kCFAllocatorDefault,g->name,kCFStringEncodingISOLatin1), // title
		CFStringCreateWithCString(kCFAllocatorDefault,user,kCFStringEncodingISOLatin1), // user
		CFStringCreateWithFormat(kCFAllocatorDefault,0,CFSTR("%s version %s (%s)"),info[0],info[1],info[2])
	};
	
    check_null ("CFDictionaryCreate", auxiliary_info = CFDictionaryCreate(kCFAllocatorDefault,
        (void *)auxiliary_keys,(void *)auxiliary_values,3,
        &kCFTypeDictionaryKeyCallBacks,&kCFTypeDictionaryValueCallBacks));
	
    CFRelease(auxiliary_values[0]);
    CFRelease(auxiliary_values[1]);
    CFRelease(auxiliary_values[2]);
}

static void
quartz_end_job_for_pdf(void)
{
    if (graphics)
		CGContextRelease(graphics);
    if (auxiliary_info)
		CFRelease(auxiliary_info);
}


static void
quartz_end_job_for_bitmap(void)
{
	if (graphics) {
		CGContextFlush(graphics);
		void* backing;
		check_null ("CGBitmapContextGetData", backing = CGBitmapContextGetData(graphics));

		codegen_info_t* pcg;
		for (pcg = first_codegen(); pcg->name; pcg = next_codegen(pcg))
			if (pcg->id == Output_lang) {
		
				/* get the Quicktime component that exports this file format */
				ComponentInstance component;
				check_null ("OpenComponent", component = OpenComponent(pcg->info));

				/* set input to a GWorld that points to same backing store as bitmap */
				GWorldPtr gworld = NULL;
				if (GraphicsExportSetInputCGBitmapContext)
					/* we have Quicktime 6.4 or better (Mac OS X 10.3 or better), so use the one stop shop */
					check_status ("GraphicsExportSetInputCGBitmapContext", GraphicsExportSetInputCGBitmapContext(component, graphics));
				else
					{
					/* we have Quicktime 6.3 (Mac OS X 10.2), so we go through a GWorld instead */
					Rect qd_bounds;
					qd_bounds.left = qd_bounds.top = 0;
					
					if (view_port.y > MAX_QD) {
						agerr (AGWARN, "height too large for Quicktime 6.3, cropping\n");
						qd_bounds.bottom = MAX_QD;
					}
					else
						qd_bounds.bottom = view_port.y;
						
					if (view_port.x > MAX_QD) {
						agerr (AGWARN, "width too large for Quicktime 6.3, cropping\n");
						qd_bounds.right = MAX_QD;
					}
					else
						qd_bounds.right = view_port.x;
						
					check_status ("NewGWorldFromPtr",
						NewGWorldFromPtr(&gworld,
						k32ARGBPixelFormat,
						&qd_bounds,
						NULL,						/* cTable */
						NULL,						/* aGDevice */
						0,						/* GWorldFlags */
						backing,
						qd_bounds.right * 4));  /* rowBytes */
					check_status ("GraphicsExportSetInputGWorld", GraphicsExportSetInputGWorld(component, gworld));
				}
				/* set resolution in metadata to 96 dpi */
				check_status ("GraphicsExportSetResolution", GraphicsExportSetResolution(component, X2Fix(dpi), X2Fix(dpi)));
				
				/* register our FILE*-handling component if not done so already */
				if (!cfile)
					check_null ("RegisterComponent", cfile = RegisterComponent (&cfile_description,
						NewComponentRoutineUPP(&cfile_entry),
						0,
						NULL,
						NULL,
						NULL));
				
				/* set exporter to output to the FILE* */
				Handle cfile_ref;
				check_null ("NewHandle", cfile_ref = NewHandle(sizeof(FILE*)));
				*(FILE**)*cfile_ref = Output_file;
				check_status ("GraphicsExportSetOutputDataReference", GraphicsExportSetOutputDataReference(component,cfile_ref,CFILE_TYPE));
				
				/* do the export */
				unsigned long actual_bytes = 0;
				check_status ("GraphicsExportDoExport", GraphicsExportDoExport(component,&actual_bytes));
						
				/* clean up */
				DisposeHandle(cfile_ref);
				if (gworld)
					DisposeGWorld(gworld);
				
				break;
			}
			
		CGContextRelease(graphics);
		free(backing);
	}
}

static void
setup_view_port(GVC_t* gvc, graph_t* g, box bb)
{
	dpi = gvc->dpi < 1.0 ? DEFAULT_DPI : gvc->dpi;
	dev_scale = dpi / POINTS_PER_INCH;
	
	view_port = gvc->size;
	if (view_port.x) {
		zoom = gvc->zoom;
		graph_focus = gvc->focus;
	}
	else {
		view_port.x = (bb.UR.x - bb.LL.x + 2*GD_drawing(g)->margin.x) * dev_scale + 2;
		view_port.y = (bb.UR.y - bb.LL.y + 2*GD_drawing(g)->margin.y) * dev_scale + 2;
		graph_focus.x = (GD_bb(g).UR.x - GD_bb(g).LL.x) / 2.0;
		graph_focus.y = (GD_bb(g).UR.y - GD_bb(g).LL.y) / 2.0;
		
		zoom = 1.0;
	}

}
static void
begin_graph_for_pdf()
{
	if (onetime) {
		CGDataConsumerRef file_consumer;
		check_null ("CGDataConsumerCreate", file_consumer = CGDataConsumerCreate(Output_file,&file_consumer_callbacks));
		check_null ("CGPDFContextCreate", graphics = CGPDFContextCreate(file_consumer,&media_box,auxiliary_info));
		CGDataConsumerRelease(file_consumer);
    }
}

static void
quartz_begin_graph_for_paged_pdf(GVC_t *gvc, graph_t* g, box bb, point pb)
{
	media_box = CGRectMake(0,0,pb.x,pb.y);
	begin_graph_for_pdf();
}

static void
quartz_begin_graph_for_embedded_pdf(GVC_t *gvc, graph_t* g, box bb, point pb)
{
	setup_view_port(gvc,g,bb);
	media_box = CGRectMake(0,0,view_port.x,view_port.y);
	begin_graph_for_pdf();
}

static void
quartz_begin_graph_for_bitmap(GVC_t *gvc, graph_t* g, box bb, point pb)
{
    if (onetime) {
		setup_view_port(gvc,g,bb);
		CGColorSpaceRef color_space;
		check_null ("CGColorSpaceCreateDeviceRGB", color_space = CGColorSpaceCreateDeviceRGB());
		
		/* allocate backing store for bitmap */
		size_t pixel_count = view_port.x*view_port.y;
		long *backing = malloc(pixel_count*sizeof(long));
		
		/* if there's a background that has transparency, initialize store to clear (transparent black) in order to receive it */
		/* if there's no background, initialize store to opaque white */
		long backfill = 0xFFFFFFFF;
		char *bgcolor = agget(g, "bgcolor");
		if (bgcolor && bgcolor [0]) {
			color_t coloring;
			colorxlate(bgcolor,&coloring,RGBA_WORD);
			if (coloring.u.rrggbbaa[3] < 65535)
				backfill = 0x00000000;
		}
		long *backing_ptr;
		for (backing_ptr = backing; backing_ptr < backing + pixel_count; ++backing_ptr)
			*backing_ptr = backfill;
			
		check_null ("CGBitmapContextCreate", graphics = CGBitmapContextCreate(backing,
			view_port.x,
			view_port.y,
			8,							/* bits per pixel component */
			view_port.x * 4,		/* bytes per row */
			color_space,
			kCGImageAlphaPremultipliedFirst));
		CGColorSpaceRelease(color_space);
    }
}

static void
quartz_end_graph(void)
{
    onetime = FALSE;
}

static void
begin_page()
{
    /* initialize the context stack */
    current_context = context;
    memset(&current_context->pencolor,0,sizeof(ATSURGBAlphaColor));
    memset(&current_context->fillcolor,0,sizeof(ATSURGBAlphaColor));
    current_context->fontname = NULL;
    current_context->fontsize = 0.0;
    current_context->visible = TRUE;
}

static void
begin_page_for_embedded(graph_t* g, double scale, int rot, point offset)
{
	CGContextSaveGState(graphics);
	
	
	comp_scale = zoom * scale * dev_scale;
	
	CGContextTranslateCTM(graphics,
		view_port.x / 2.0,
		view_port.y / 2.0);
	CGContextScaleCTM(graphics, comp_scale, comp_scale);
	if (rot) CGContextRotateCTM(graphics,RADIANS(rot));
	CGContextTranslateCTM(graphics,
		-graph_focus.x,
		-graph_focus.y);

	begin_page();
}

static void
quartz_begin_page_for_paged_pdf(graph_t* g, point page, double scale, int rot, point offset)
{
	CGContextBeginPage(graphics,&media_box);
    CGContextSaveGState(graphics);

	comp_scale = scale;
	CGContextTranslateCTM(graphics,
		GD_drawing(g)->margin.x+offset.x*scale,
		GD_drawing(g)->margin.y+offset.y*scale);
    if (scale != 1.0) CGContextScaleCTM(graphics,scale,scale);
	if (rot) CGContextRotateCTM(graphics,RADIANS(rot));
	
	begin_page();
}

static void
quartz_begin_page_for_embedded_pdf(graph_t* g, point page, double scale, int rot, point offset)
{
	CGContextBeginPage(graphics,&media_box);
	begin_page_for_embedded(g, scale, rot, offset);
}

static void
quartz_begin_page_for_bitmap(graph_t* g, point page, double scale, int rot, point offset)
{
	begin_page_for_embedded(g, scale, rot, offset);
}

static void
quartz_end_page_for_pdf(void)
{
    CGContextRestoreGState(graphics);
    CGContextEndPage(graphics);
}

static void
quartz_end_page_for_bitmap(void)
{
    CGContextRestoreGState(graphics);
}

static void
quartz_begin_node(node_t* n)
{
	current_node = n;
}

static void
quartz_end_node()
{
	current_node = NULL;
}

static void
quartz_begin_context(void)
{
    CGContextSaveGState(graphics);
    
    ++current_context;
    *current_context = *(current_context-1);
}

static void
quartz_end_context(void)
{
    CGContextRestoreGState(graphics);
    
    --current_context;
}

static void
quartz_set_font(char* fontname, double fontsize)
{
    current_context->fontname = fontname;
    current_context->fontsize = fontsize;
}

static void
get_color(ATSURGBAlphaColor* color, char* name)
{
	color_t coloring;
	colorxlate(name,&coloring,RGBA_WORD);
	color->red = coloring.u.rrggbbaa[0]/65535.0;
	color->green = coloring.u.rrggbbaa[1]/65535.0;
	color->blue = coloring.u.rrggbbaa[2]/65535.0;
	color->alpha =	coloring.u.rrggbbaa[3]/65535.0;
}

static void
quartz_textline(point p, textline_t* str)
{
    if (current_context->visible) {
	double adj;
	
	switch (str->just) {
	    case 'l':
			adj = 0.0;
			break;
	    case 'r':
			adj = -1.0;
			break;
	    default:
		case 'n':
			adj = -0.5;
		break;
	}

	/* fetch any cached text for this string */
	text_value* text = fetch_text(str->str,current_context->fontname,current_context->fontsize);
	if (text) {
		/* associate graphics context with layout so that we can draw it */
		ATSUAttributeTag lay_tags[] = {kATSUCGContextTag};
		ByteCount lay_sizes[] = {sizeof(CGContextRef)};
		ATSUAttributeValuePtr lay_values[] = {&graphics};
		check_status ("ATSUSetLayoutControls", ATSUSetLayoutControls(text->layout,1,lay_tags,lay_sizes,lay_values));

		/* set color of text */
		ATSUAttributeTag style_tags[] = {kATSURGBAlphaColorTag};
		ByteCount style_sizes[] = {sizeof(ATSURGBAlphaColor)};
		ATSUAttributeValuePtr style_values[] = {&current_context->pencolor};
		check_status ("ATSUSetAttributes", ATSUSetAttributes(text->style,1,style_tags,style_sizes,style_values));
		
		/* draw it */
		CGContextSaveGState (graphics);
		CGContextTranslateCTM (graphics, p.x+adj*str->width, p.y);
		check_status ("ATSUDrawText", ATSUDrawText(text->layout,
			kATSUFromTextBeginning,
			kATSUToTextEnd,
			0,
			0));
		CGContextRestoreGState (graphics);
	}
    }
}

static void
quartz_set_pencolor(char *name)
{
	get_color(&current_context->pencolor,name);
	CGContextSetRGBStrokeColor(graphics,
		current_context->pencolor.red,
		current_context->pencolor.green,
		current_context->pencolor.blue,
		current_context->pencolor.alpha);
}

static void
quartz_set_fillcolor(char* name)
{
	get_color(&current_context->fillcolor,name);
	CGContextSetRGBFillColor(graphics,
		current_context->fillcolor.red,
		current_context->fillcolor.green,
		current_context->fillcolor.blue,
		current_context->fillcolor.alpha);
}

static void
quartz_set_style(char** s)
{
    const char* line;
    
    while ((line = *s++))
		if (streq(line,"solid"))
			CGContextSetLineDash(graphics,0.0,0,0);
		else if (streq(line,"dashed")) {
			float dash[2];
			dash[0] = dash[1] = 9/comp_scale;
			CGContextSetLineDash(graphics,0.0,dash,2);
			}
		else if (streq(line,"dotted")) {
			float dash[2];
			dash[0] = comp_scale;
			dash[1] = 6/comp_scale;
			CGContextSetLineDash(graphics,0.0,dash,2);
			}
		else if (streq(line,"invis"))
			current_context->visible = FALSE;
		else if (streq(line,"bold"))
			CGContextSetLineWidth(graphics,2.0);
		else if (streq(line, "setlinewidth")) {
			const char *p = line;
			while (*p) p++;
			p++;
			CGContextSetLineWidth(graphics,atof(p));
		}
}

static void
quartz_ellipse(point p, int rx, int ry, int filled)
{
    if (current_context->visible) {
		CGContextSaveGState(graphics);
		CGContextTranslateCTM(graphics,p.x,p.y);
		CGContextScaleCTM(graphics,rx,ry);
		CGContextBeginPath(graphics);
		CGContextAddArc(graphics,0,0,1,0,2*PI,1);
		CGContextClosePath(graphics);
		CGContextRestoreGState(graphics);
		
		CGContextDrawPath(graphics,get_mode(filled));
    }
}

static void
quartz_polygon(point* A, int n, int filled)
{
    if (current_context->visible) {
		int j;
		
		CGContextBeginPath(graphics);
		CGContextMoveToPoint(graphics,A[0].x,A[0].y);
		for (j = 1; j < n; ++j)
			CGContextAddLineToPoint(graphics,A[j].x,A[j].y);
		CGContextClosePath(graphics);
		CGContextDrawPath(graphics,get_mode(filled));
    }
}

static void
quartz_beziercurve(point* A, int n, int arrow_at_start, int arrow_at_end)
{
    if (current_context->visible) {
		int j;
		
		CGContextBeginPath(graphics);
		CGContextMoveToPoint(graphics,A[0].x,A[0].y);
		for (j = 1; j < n; j += 3)
			CGContextAddCurveToPoint(graphics,A[j].x,A[j].y,A[j+1].x,A[j+1].y,A[j+2].x,A[j+2].y);
		CGContextDrawPath(graphics,kCGPathStroke);
    }
}

static void
quartz_polyline(point* A,int n)
{
    if (current_context->visible) {
		int j;
		
		CGContextBeginPath(graphics);
		CGContextMoveToPoint(graphics,A[0].x,A[0].y);
		for (j = 1; j < n; ++j)
			CGContextAddLineToPoint(graphics,A[j].x,A[j].y);
		CGContextDrawPath(graphics,kCGPathStroke);
    }
}

static  void
quartz_user_shape(char *name, point *A, int n, int filled)
{
	CFTypeRef image;

	if (streq(name,"custom"))
		image = fetch_image (agget(current_node,"shapefile"));
	else
		image = fetch_image (name);
	if (image) {
		pointf	min, max;		/* upper left, lower right */
		int i;
		/* compute dest origin and size */
		min.x = max.x = A[0].x; min.y = max.y = A[0].y;
		for (i = 1; i < n; i++) {
			if (min.x > A[i].x) min.x = A[i].x;
			if (min.y > A[i].y) min.y = A[i].y;
			if (max.x < A[i].x) max.x = A[i].x;
			if (max.y < A[i].y) max.y = A[i].y;
		}
		CGRect bounds;
		bounds.origin.x = min.x;
		bounds.origin.y = min.y;
		bounds.size.width = max.x - min.x;
		bounds.size.height = max.y - min.y;
		
		CFTypeID id = CFGetTypeID (image);
		if (id == CGImageGetTypeID ())
			CGContextDrawImage (graphics, bounds, (CGImageRef)image);
		else if (id == CGPDFDocumentGetTypeID ())
			CGContextDrawPDFDocument (graphics, bounds, (CGPDFDocumentRef)image, 1);
	}
}


char*
gd_textsize(char* str, char* fontname, double fontsz, pointf* textsize)
{
	text_value* text = fetch_text(str,fontname,fontsz);
	if (text) {
		ATSUTextMeasurement before,after,ascent,descent;
		check_status ("ATSUGetUnjustifiedBounds",
			ATSUGetUnjustifiedBounds(text->layout,
			kATSUFromTextBeginning,kATSUToTextEnd,&before,&after,&ascent,&descent));
		
		textsize->x = Fix2X(after-before);
		textsize->y = Fix2X(ascent+descent);
	} else
		textsize->x = textsize->y = 0.0;
		
    return NULL;
}

point quartz_image_size(graph_t *g, char *shapeimagefile)
{
	point	rv = {-1, -1};
	
	CFTypeRef image = fetch_image (shapeimagefile);
	if (image) {
		CFTypeID id = CFGetTypeID (image);
		if (id == CGImageGetTypeID ()) {
			rv.x = CGImageGetWidth ((CGImageRef)image);
			rv.y = CGImageGetHeight ((CGImageRef)image);
		}
		else if (id == CGPDFDocumentGetTypeID ()) {
			CGRect bounds = CGPDFDocumentGetMediaBox ((CGPDFDocumentRef)image, 1);
			rv.x = bounds.size.width;
			rv.y = bounds.size.height;
		}
	}
	
	return rv;
}

point quartz_user_shape_size(node_t *n, char *shapeimagefile)
{
	return (quartz_image_size (n->graph, shapeimagefile));
}

codegen_t QPDF_CodeGen = {
    quartz_reset,
    quartz_begin_job_for_pdf,
    quartz_end_job_for_pdf,
    quartz_begin_graph_for_paged_pdf,
    quartz_end_graph,
    quartz_begin_page_for_paged_pdf,
    quartz_end_page_for_pdf,
    NULL, /* begin_layer */
    NULL, /* end_layer */
    NULL, /* begin_cluster */
    NULL, /* end_cluster */
    NULL, /* begin_nodes */
    NULL, /* end_nodes */ 
    NULL, /* begin_edges */
    NULL, /* end_edges */ 
    quartz_begin_node,
    quartz_end_node,
    NULL, /* begin_edge */
    NULL, /* end_edge */
    quartz_begin_context,
    quartz_end_context,
	NULL, /* begin_anchor */
	NULL, /* end_anchor */
    quartz_set_font,
    quartz_textline,
    quartz_set_pencolor,
    quartz_set_fillcolor,
    quartz_set_style,
    quartz_ellipse,
    quartz_polygon,
    quartz_beziercurve,
    quartz_polyline,
    NULL, /* bezier_has_arrows */
    NULL, /* comment */
    NULL, // quartz_textsize,
    quartz_user_shape,
    NULL, /* usershapesize */
};

codegen_t QEPDF_CodeGen = {
    quartz_reset,
    quartz_begin_job_for_pdf,
    quartz_end_job_for_pdf,
    quartz_begin_graph_for_embedded_pdf,
    quartz_end_graph,
    quartz_begin_page_for_embedded_pdf,
    quartz_end_page_for_pdf,
    NULL, /* begin_layer */
    NULL, /* end_layer */
    NULL, /* begin_cluster */
    NULL, /* end_cluster */
    NULL, /* begin_nodes */
    NULL, /* end_nodes */ 
    NULL, /* begin_edges */
    NULL, /* end_edges */ 
    quartz_begin_node,
    quartz_end_node,
    NULL, /* begin_edge */
    NULL, /* end_edge */
    quartz_begin_context,
    quartz_end_context,
	NULL, /* begin_anchor */
	NULL, /* end_anchor */
    quartz_set_font,
    quartz_textline,
    quartz_set_pencolor,
    quartz_set_fillcolor,
    quartz_set_style,
    quartz_ellipse,
    quartz_polygon,
    quartz_beziercurve,
    quartz_polyline,
    NULL, /* bezier_has_arrows */
    NULL, /* comment */
    NULL, // quartz_textsize,
    quartz_user_shape,
    NULL, /* usershapesize */
};

codegen_t QBM_CodeGen = {
    quartz_reset,
    NULL, /* begin_job */
    quartz_end_job_for_bitmap,
    quartz_begin_graph_for_bitmap,
    quartz_end_graph,
    quartz_begin_page_for_bitmap,
    quartz_end_page_for_bitmap,
    NULL, /* begin_layer */
    NULL, /* end_layer */
    NULL, /* begin_cluster */
    NULL, /* end_cluster */
    NULL, /* begin_nodes */
    NULL, /* end_nodes */ 
    NULL, /* begin_edges */
    NULL, /* end_edges */ 
    quartz_begin_node,
    quartz_end_node,
    NULL, /* begin_edge */
    NULL, /* end_edge */
    quartz_begin_context,
    quartz_end_context,
	NULL, /* begin_anchor */
	NULL, /* end_anchor */
    quartz_set_font,
    quartz_textline,
    quartz_set_pencolor,
    quartz_set_fillcolor,
    quartz_set_style,
    quartz_ellipse,
    quartz_polygon,
    quartz_beziercurve,
    quartz_polyline,
    NULL, /* bezier_has_arrows */
    NULL, /* comment */
    NULL, // quartz_textsize,
    quartz_user_shape,
    NULL, /* usershapesize */
};


