# Check Protocol-A.texi.
# Copyright (C) 2001-2002  Lysator Academic Computer Association.
#
# This file is part of the LysKOM server.
# 
# LysKOM is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by 
# the Free Software Foundation; either version 1, or (at your option) 
# any later version.
# 
# LysKOM is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
# FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
# 
# You should have received a copy of the GNU General Public License
# along with LysKOM; see the file COPYING.  If not, write to
# Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
# or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
# MA 02139, USA.
#
# Please mail bug reports to bug-lyskom@lysator.liu.se. 
#

# Check @aarg{} and @rarg{} usage in Protocol-A.texi.
# ...and a lot of other stuff...

import sys
import types

# A mapping from type names (as strings) to prot_a_* objects.
defined_types = {}

# All fields that have been seen so far.  Initialized with some
# built-in fields.
defined_fields = {
    'error-code': None,
    'error-status': None,
    'ref-no': None,
    'reply-data': None,
}
# Fields seen in @field{} but not yet defined.
undefined_fields = {}

def number_suffixed(s, base):
    if not s.startswith(base):
        return 0
    s = s[len(base):]
    if len(s) < 1:
        return 0
    try:
        int(s)
        return 1
    except:
        return 0

class reader_eof(Exception):
    pass


class prot_a_type:
    __usage_count = 0

    def __init__(self, line):
        self.__line = line

    def line(self):
        return self.__line

    def use(self):
        self.__usage_count += 1
        if self.__usage_count == 1:
            self.use_recurse()

    def usage(self):
        return self.__usage_count

class prot_a_builtin(prot_a_type):
    def use_recurse(self):
        pass

class prot_a_simple(prot_a_type):
    def __init__(self, line, typ, is_array):
        self.__type = typ
        self.__array = is_array
        prot_a_type.__init__(self, line)

    def use_recurse(self):
        defined_types[self.__type].use()

class prot_a_alternate(prot_a_type):
    def __init__(self, line, type_a, type_b):
        self.__type_a = type_a
        self.__type_b = type_b
        prot_a_type.__init__(self, line)

    def use_recurse(self):
        defined_types[self.__type_a].use()
        defined_types[self.__type_b].use()

class prot_a_struct(prot_a_type):
    def __init__(self, line):
        prot_a_type.__init__(self, line)
        self.__fields = []
        self.__used_names = {}

    def add_field(self, field_name, type_name, is_array):
        if self.__used_names.has_key(field_name):
            return "field name ``%s'' used twice" % field_name

        o = (field_name, type_name, is_array)
        self.__used_names[field_name] = o
        self.__fields.append(o)
        defined_fields[field_name] = None
        if undefined_fields.has_key(field_name):
            del undefined_fields[field_name]
        return None

    def use_recurse(self):
        for (fn, tn, ar) in self.__fields:
            defined_types[tn].use()

class prot_a_bitstring(prot_a_type):
    def __init__(self, line):
        prot_a_type.__init__(self, line)
        self.__bits = {}

    def add_field(self, field_name):
        if self.__bits.has_key(field_name):
            return "bit ``%s'' used twice" % field_name
        self.__bits[field_name] = None
        defined_fields[field_name] = None
        if undefined_fields.has_key(field_name):
            del undefined_fields[field_name]

    def check_implemented(self, name, implemented_fields):
        res = []
        i = {}
        for f in implemented_fields:
            if not self.__bits.has_key(f):
                res.append("%s: bit ``%s'' not documented" % (name, f))
            i[f] = None
        for f in self.__bits.keys():
            if not i.has_key(f):
                res.append("%s: bit ``%s'' not implemented" % (name, f))
        return res

    def use_recurse(self):
        pass

class prot_a_selection(prot_a_type):
    def __init__(self, line):
        prot_a_type.__init__(self, line)
        self.__fields = []
        self.__used_nums = {}
        self.__used_names = {}
        self.__used_tailnames = {}

    def add_variant(self, number, name, tailname, tail_type, is_array):
        if self.__used_nums.has_key(number):
            return "selection number ``%s'' used twice" % number
        if self.__used_names.has_key(name):
            return "selection name ``%s'' used twice" % name
        if self.__used_tailnames.has_key(tailname):
            return "selection tailname ``%s'' used twice" % tailname
        o = (number, name, tailname, tail_type, is_array)
        self.__used_nums[number] = o
        self.__used_names[name] = o
        self.__used_tailnames[tailname] = o
        self.__fields.append(o)

    def check_implemented(self, imp_nr, imp_name):
        if not self.__used_names.has_key(imp_name):
            return "``%s'' implemented but not documented" % imp_name
        if self.__used_names[imp_name][0] != imp_nr:
            return "``%s'' implemented as %s but documented as %s" % (
                imp_name, repr(imp_nr), repr(self.__used_names[imp_name][0]))
        return None

    def all_names(self):
        return self.__used_names.keys()

    def use_recurse(self):
        for (number, name, tailname, tail_type, is_array) in self.__fields:
            defined_types[tail_type].use()

class prot_a_enumration_of(prot_a_type):
    def __init__(self, line, base):
        prot_a_type.__init__(self, line)
        self.__base = base

    def use_recurse(self):
        defined_types[self.__base].use()

class reader:
    parens = {'(': ')',
              '[': ']',
              '{': '}'}
    rev_parens = {}
    for k, v in parens.items():
        rev_parens[v] = k
    del v
    del k

    def __init__(self, file):
        self.__filename = file
        self.__file = open(file, "r")
        self.__line_no = 0
        self.__line = []
        self.__eof = 0
        self.__parenstack = []
        self.__errfound = 0
        self.__quoted = 0

    def filename(self):
        return self.__filename

    def error(self, line_no, errmsg):
        sys.stderr.write("%s:%s:%s\n" % (self.filename(), line_no, errmsg))
        self.__errfound = 1

    def errfound(self):
        return self.__errfound

    def ungetc(self, c):
        if self.rev_parens.has_key(c):
            self.__parenstack.append((self.rev_parens[c], 0))
        if self.parens.has_key(c):
            assert len(self.__parenstack) > 0
            assert self.__parenstack[-1][0] == c
            del self.__parenstack[-1]
        if c == '@':
            self.__quoted = not self.__quoted
        self.__line.insert(0, c)

    def getc_eofok(self):
        if self.__line == []:
            if self.__eof:
                return None
            line = self.__file.readline()
            if line == '':
                self.__eof = 1
                return None
            self.__line = list(line)
            self.__line_no += 1
        ret = self.__line[0]
        del self.__line[0]
        if not self.__quoted:
            if ret in '([{':
                self.__parenstack.append((ret, self.line_no()))
            if ret in '}])':
                if len(self.__parenstack) == 0:
                    self.error(self.line_no(),
                               "unmatched paren ``%s''" % ret)
                else:
                    if self.__parenstack[-1][0] != self.rev_parens[ret]:
                        self.error(self.line_no(),
                                   "badly matched parens ``%s'' ``%s''" %
                                   (self.__parenstack[-1][0], ret))
                    del self.__parenstack[-1]
        if ret == '@':
            self.__quoted = not self.__quoted
        else:
            self.__quoted = 0
        return ret

    def getc(self):
        c = self.getc_eofok()
        if c == None:
            raise reader_eof
        return c

    def line_no(self):
        return self.__line_no

    def check_paren_null(self):
        for par, line in self.__parenstack:
            self.error(line, "unclosed ``%s''" % par)
        self.__parenstack = []

class lexer:
    def __init__(self, file):
        self.__reader = reader(file)
        self.__findex = None
        self.__amindex = None
        self.__linkhere = None
        self.__builtin_types = ["BOOL", "INT8", "INT16", "INT32",
                                "HOLLERITH"]
        self.__builtin_aggregates = ["BITSTRING", "ENUMERATION",
                                     "ENUMERATION-OF",
                                     "ARRAY", "SELECTION", "RPC"]
        self.__builtin_typelike = (self.__builtin_types
                                   + self.__builtin_aggregates)
        self.__tindex_seen = {}
        for t in self.__builtin_types:
            defined_types[t] = prot_a_builtin('*builtin*')
        self.__defined_requests = {}

        self.__defined_asyncs = {}

        self.__implemented_reqs = {}
        f = open("requests-numbered.tmp", "r")
        for line in f.readlines():
            [nr, name] = line.split(" ")
            self.__implemented_reqs[name.strip()] = nr
        f.close()

        self.__implemented_asyncs = {}
        n = {}
        f = open("asyncs-numbered.tmp", "r")
        for line in f.readlines():
            [nr, name] = line.split(" ")
            name = name.strip()
            if n.has_key(nr):
                sys.stderr.write("../src/server/async.h:1:enum async:"
                                 " number %s used for both %s and %s\n" % (
                    nr, n[nr], name))
                sys.exit(1)
            n[nr] = name
            self.__implemented_asyncs[name] = nr
        f.close()

        self.__implemented_miscs = {}
        n = {}
        f = open("miscs-numbered.tmp", "r")
        for line in f.readlines():
            [nr, name] = line.split(" ")
            nr = int(nr)
            name = name.strip()
            if n.has_key(nr):
                sys.stderr.write("../src/include/kom-types.h:1:enum info_type:"
                                 " number %s used for both %s and %s\n" % (
                    nr, n[nr], name))
                sys.exit(1)
            n[nr] = name
            self.__implemented_miscs[name] = nr
        f.close()

        self.__implemented_conftypes = {}
        f = open("conftypes.tmp", "r")
        for line in f.readlines():
            name = line.strip()
            self.__implemented_conftypes[name] = nr
        f.close()

        self.__implemented_privbits = {}
        f = open("privbits.tmp", "r")
        for line in f.readlines():
            name = line.strip()
            self.__implemented_privbits[name] = nr
        f.close()

    def run(self):
        while 1:
            c = self.__reader.getc()
            if c == '@':
                self.__toplevel_at()

    def __toplevel_at(self):
        line_no = self.__reader.line_no()
        c = self.__reader.getc()
        if c in '{}@-"*':
            return
        cmd = ""
        while 1:
            if c.isalpha() or c in '_"':
                cmd = cmd + c
                c = self.__reader.getc()
            elif c in ' \t\n{@=-':
                assert cmd != ''
                if c == '{':
                    arg = self.__read_arg()
                elif c == ' ' and cmd not in ['tab']:
                    arg = self.__read_line()
                else:
                    arg = None
                if c == '@':
                    self.__reader.ungetc(c)

                if hasattr(self, 'toplevel_' + cmd):
                    getattr(self, 'toplevel_' + cmd)(arg, line_no)
                else:
                    self.error(line_no, "unknown command @%s{}" % cmd)
                return
            else:
                self.error(line_no, "bad command ``@%s%s''" % (cmd, c))
                return

    def __read_line(self):
        line = ""
        while 1:
            c = self.__reader.getc()
            if c == '\n':
                return line
            line = line + c

    def ignore(self, arg, line_no):
        pass

    toplevel_setfilename = ignore
    toplevel_settitle = ignore
    toplevel_setchapternewpage = ignore
    toplevel_set = ignore
    toplevel_macro = ignore
    toplevel_end = ignore
    toplevel_ifinfo = ignore
    toplevel_value = ignore
    toplevel_copyright = ignore
    toplevel_iftex = ignore
    toplevel_parindent = ignore
    toplevel_ifinfo = ignore
    toplevel_begin = ignore
    toplevel_titlepage = ignore
    toplevel_title = ignore
    toplevel_subtitle = ignore
    toplevel_author = ignore
    toplevel_font = ignore
    toplevel_ignore = ignore
    toplevel_tensltt = ignore
    toplevel_page = ignore
    toplevel_vskip = ignore
    toplevel_ifnothtml = ignore
    toplevel_contents = ignore
    toplevel_dircategory = ignore
    toplevel_direntry = ignore
    toplevel_ifhtml = ignore
    toplevel_html = ignore
    toplevel_ifnottex = ignore
    toplevel_top = ignore
    toplevel_menu = ignore
    toplevel_chapter = ignore
    toplevel_penalty = ignore
    toplevel_section = ignore
    toplevel_table = ignore
    toplevel_item = ignore
    toplevel_req = ignore
    toplevel_reqlink = ignore
    toplevel_asynclink = ignore
    toplevel_subsection = ignore
    toplevel_itemize = ignore
    toplevel_bullet = ignore
    toplevel_multitable = ignore
    toplevel_tab = ignore
    toplevel_aux = ignore
    toplevel_errorcode = ignore
    toplevel_misc = ignore
    toplevel_type = ignore
    toplevel_dfn = ignore
    toplevel_conftype = ignore
    toplevel_command = ignore
    toplevel_ref = ignore
    toplevel_subsubsection = ignore
    toplevel_priv = ignore
    toplevel_enumerate = ignore
    toplevel_itemx = ignore
    toplevel_ae = ignore
    toplevel_i = ignore
    toplevel_subheading = ignore
    toplevel_c = ignore
    toplevel_t = ignore
    toplevel_aa = ignore
    toplevel_async = ignore
    toplevel_unnumbered = ignore
    toplevel_printindex = ignore
    toplevel_example = ignore
    toplevel_unmacro = ignore
    toplevel_dots = ignore
    toplevel_tex = ignore
    toplevel_defcodeindex = ignore
    toplevel_syncodeindex = ignore
    toplevel_need = ignore
    toplevel_reqexample = ignore
    toplevel_anchor = ignore
    toplevel_appendix = ignore
    toplevel_display = ignore
    toplevel_daemon = ignore
    toplevel_TeX = ignore

    def pushback(self, arg, line_no):
        lst = list(arg)
        lst.reverse()
        for char in lst:
            self.__reader.ungetc(char)

    toplevel_w = pushback
    toplevel_code = pushback
    toplevel_uref = pushback
    toplevel_footnote = pushback
    toplevel_email = pushback
    toplevel_asis = pushback
    toplevel_samp = pushback
    toplevel_pxref = pushback
    toplevel_var = pushback
    toplevel_emph = pushback
    toplevel_xref = pushback
    toplevel_badspell = pushback
    toplevel_holl = pushback
    toplevel_file = pushback
    toplevel_cindex = pushback

    def toplevel_node(self, arg, line_no):
        if self.__findex != None:
            self.__findex = None
            for (argname, [lineno, usage]) in self.__args.items():
                if usage == 0:
                    self.error(lineno,
                               "Undocumented argument ``%s''" % (argname, ))
        if self.__amindex != None:
            self.__amindex = None
            for (argname, [lineno, usage]) in self.__args.items():
                if usage == 0:
                    self.error(lineno,
                               "Undocumented argument ``%s''" % (argname, ))
        self.__node_name = arg
        self.__node_start = line_no
        self.__assert_no_linkhere()
        self.__reader.check_paren_null()

    def toplevel_findex(self, arg, line_no):
        if self.__node_name != arg:
            self.error(line_no, "@node/@findex mismatch: %s..." % arg)
            self.error(line_no, "...inside node %s" % self.__node_name)
            return
        if self.__amindex != None:
            self.error(line_no, "@findex and @amindex used in the same node")
            return
        if self.__findex != None:
            self.error(line_no, "multiple @findex in single @node")
            return
        self.__findex = arg
        self.__args = {}
        if self.__defined_requests.has_key(arg):
            self.error(line_no, "request ``%s'' redefined" % arg)
            self.error(self.__defined_requests[arg], "previous definition")
        self.__defined_requests[arg] = line_no
        if self.__get_token() != '@example':
            self.error(self.__reader.line_no(), "missing @example")
            return
        self.__parse_request()

    def toplevel_rarg(self, arg, line_no):
        if self.__findex == None:
            self.error(line_no, "@rarg outside @findex node")
            return
        if not self.__args.has_key(arg):
            self.error(line_no, "undefined argument ``%s''" % (arg, ))
            return
        self.__args[arg][1] += 1

    def toplevel_amindex(self, arg, line_no):
        if self.__node_name != arg:
            self.error(line_no, "@node/@amindex mismatch: %s..." % arg)
            self.error(line_no, "...inside node %s" % self.__node_name)
            return
        if self.__findex != None:
            self.error(line_no, "@findex and @amindex used in the same node")
            return
        if self.__amindex != None:
            self.error(line_no, "multiple @amindex in single @node")
            return
        self.__amindex = arg
        self.__args = {}
        if self.__defined_asyncs.has_key(arg):
            self.error(line_no, "async message ``%s'' redefined" % arg)
            self.error(self.__defined_asyncs[arg], "previous definition")
        self.__defined_asyncs[arg] = line_no
        token = self.__get_token()
        obsolete = 0
        if token == '@c':
            token = self.__get_token()
            if token == 'obsolete':
                obsolete = 1
            else:
                self.error(self.__reader.line_no(),
                           "broken comment within @amindex block")
                return
            token = self.__get_token()
        if token != '@example':
            self.error(self.__reader.line_no(), "missing @example")
            return
        self.__parse_async(obsolete)

    def __parse_async(self, obsolete):
        self.__tokens = []
        async = self.__get_token()
        if async != self.__amindex:
            self.error(self.__reader.line_no(),
                       "wrong async name ``%s''" % async)
            return

        if self.__get_token() != '[':
            self.error(self.__reader.line_no(), "missing ``[''")
            return
            
        nr = self.__get_token()
        if type(nr) != types.IntType:
            self.error(self.__reader.line_no(), "bad async number")

        if self.__implemented_asyncs.has_key(async):
            if self.__implemented_asyncs[async] != str(nr):
                self.error(self.__reader.line_no(),
                           "``%s'' is implemented as asynchronous message %s, "
                           "not %s" % (async, self.__implemented_asyncs[async],
                                       nr))
        elif not obsolete:
            self.error(self.__reader.line_no(),
                       "asynchronous message ``%s'' not implemented" % async)
        
        if self.__get_token() != ']':
            self.error(self.__reader.line_no(), "missing ``]''")
            return

        paren = self.__get_token()
        if paren == '(':
            next = self.__get_token()
            if next != ')':
                self.__unget_token(next)
                self.__parse_request_arg()
                next = self.__get_token()
            if next != ')':
                self.error(self.__reader.line_no(),
                           "missing close parenthesis after arguments")
                return
        elif paren == '((':
            self.__parse_request_arg()
            next = self.__get_token()
            while next == ';':
                self.__parse_request_arg()
                next = self.__get_token()
            if next != '))':
                self.error(self.__reader.line_no(),
                           "missing double close parenthesis after arguments")
                return
        else:
            self.error(self.__reader.line_no(),
                       "missing argument list")
            return

        if self.__get_token() != ';':
            self.error(self.__reader.line_no(), "missing final ``;''")
            return

        if self.__get_token() != '@end':
            self.error(self.__reader.line_no(), "extra garbage found")
            return

        return

        

    def toplevel_aarg(self, arg, line_no):
        if self.__amindex == None:
            self.error(line_no, "@aarg outside @amindex node")
            return
        if not self.__args.has_key(arg):
            self.error(line_no, "undefined argument ``%s''" % (arg, ))
            return
        self.__args[arg][1] += 1

    def toplevel_bye(self, arg, line_no):
        if self.__findex != None:
            self.error(self.__reader.line_no(), "unterminated @findex node")
        if self.__amindex != None:
            self.error(self.__reader.line_no(), "unterminated @amindex node")

        # Check types.
        for (n, o) in defined_types.items():
            if o.usage() < 1:
                self.error(o.line(), "unused type ``%s''" % n)

        # Check requests.
        for req in self.__implemented_reqs.keys():
            if not self.__defined_requests.has_key(req):
                self.error(self.__reader.line_no(),
                           "request ``%s'' not documented" % req)

        # Check async messages.
        for req in self.__implemented_asyncs.keys():
            if not self.__defined_asyncs.has_key(req):
                self.error(self.__reader.line_no(),
                           "asynchronous message ``%s'' not documented" % req)

        # Check misc-info stuff.
        for name, nr in self.__implemented_miscs.items():
            err = defined_types['Misc-Info'].check_implemented(nr, name)
            if err:
                self.error(self.__reader.line_no(), "Misc-info " + err)
        for name in defined_types['Misc-Info'].all_names():
            if not self.__implemented_miscs.has_key(name):
                self.error(self.__reader.line_no(),
                           "Misc-info ``%s'' not implemented." % name)

        # Check bitfields.
        for emsg in defined_types['Extended-Conf-Type'].check_implemented(
            "conference type", self.__implemented_conftypes):

            self.error(self.__reader.line_no(), emsg)

        for emsg in defined_types['Priv-Bits'].check_implemented(
            "conference type", self.__implemented_privbits):

            self.error(self.__reader.line_no(), emsg)

        # Check fields.
        for (field, lines) in undefined_fields.items():
            for lin in lines:
                self.error(lin, "undefined field ``%s'' referenced" % field)

        self.__reader.check_paren_null()
        
        # This is the normal exit of the program.  It isn't nice to hide
        # it deep in the code like this, but neither is having an EOF
        # command before the real EOF of the file.
        sys.exit(self.__reader.errfound())



    def toplevel_reqdlink(self, arg, line_no):
        self.__assert_no_linkhere()
        self.__linkhere = ("@reqdlink{%s}" % arg, line_no)

    def toplevel_asyncdlink(self, arg, line_no):
        self.__assert_no_linkhere()
        self.__linkhere = ("@asyncdlink{%s}" % arg, line_no)

    def __assert_no_linkhere(self):
        if self.__linkhere != None:
            self.error(self.__linkhere[1],
                       "@linkhere{} after %s missing" % (self.__linkhere[0]))

    def toplevel_linkhere(self, arg, line_no):
        if self.__linkhere == None:
            self.error(line_no, "spurious @linkhere{}")
        self.__linkhere = None

    def __seen_type(self, arg, line_no):
        if self.__tindex_seen.has_key(arg):
            self.error(line_no, "multiple @tindex entries for %s" % arg)
            self.error(self.__tindex_seen[arg], "previous location")
        else:
            self.__tindex_seen[arg] = line_no

    def toplevel_tindex(self, arg, line_no):
        self.__seen_type(arg, line_no)
        if arg in self.__builtin_typelike:
            return

        self.__bad_type(arg)

        tindexed_types = [arg]
        self.__tokens = []
        t = self.__get_token()
        while t == '@tindex':
            t = self.__get_token()
            self.__seen_type(t, self.__reader.line_no())
            self.__bad_type(t)
            tindexed_types.append(t)
            t = self.__get_token()
        if t != '@example':
            self.error(self.__reader.line_no(),
                       "missing @example after @tindex for defined type")
            return
        self.__parse_userdefined_types(tindexed_types)

    def __parse_userdefined_types(self, tindexed_types):
        """Parse type definitions for TINDEXED_TYPES.

        TINDEXED_TYPES is a list of type names, as strings, that
        originates from the @tindex statements above this @example
        section.  An error message is given if TINDEXED_TYPES doesn't
        correspond to the types actually defined in this @example
        section.
        """
        seen_types = []
        while 1:
            newtype = self.__get_token()
            line = self.__reader.line_no()
            if newtype == '@end':
                token = self.__get_token()
                if token != 'example':
                    self.error(self.__reader.line_no(),
                               "``@end example'' expected")
                break
            if newtype == '@need':
                self.__get_token() # Skip the argument to @need.
                newtype = self.__get_token()
            if newtype == '@anchor':
                if self.__get_token() != '{':
                    self.error(self.__reader.line_no(),
                               "@anchor must be followed by ``{''")
                newtype = self.__get_token()
                if self.__get_token() != '}':
                    self.error(self.__reader.line_no(),
                               "@anchor missing ``}''")
                token = self.__get_token()
                if token != newtype:
                    self.error(self.__reader.line_no(),
                               "@anchor mismatch -- ``%s'' or ``%s''?" % (
                        newtype, token))
            token = self.__get_token()
            if token != '::=':
                self.error(self.__reader.line_no(), "``::='' expected")
                return
            token = self.__get_token()
            if token == '(':
                typedef = self.__parse_userdefined_struct(line)
            elif token == 'SELECTION':
                typedef = self.__parse_userdefined_selection(line)
            elif token == 'BITSTRING':
                typedef = self.__parse_userdefined_bitstring(line)
            elif token == 'ENUMERATION-OF':
                typedef = self.__parse_userdefined_enumeration_of(line)
            else:
                self.__unget_token(token)
                token = self.__get_lt_token()
                if token == 'ARRAY':
                    name = self.__get_lt_token()
                    array = 1
                else:
                    array = 0
                    name = token
                token = self.__get_token()
                if token == '|':
                    token = self.__get_lt_token()
                    if array or token == 'ARRAY':
                        self.error(self.__reader.line_no(),
                                   "cannot mix ARRAY and ``|''")
                        return
                    typedef = prot_a_alternate(line, name, token)
                    token = self.__get_token()
                else:
                    typedef = prot_a_simple(line, name, array)

                if token != ';':
                    # This is not a fatal error; we try to recover
                    # from it.
                    self.error(self.__reader.line_no(), "missing ;")

            if typedef != None:
                if newtype not in tindexed_types:
                    self.error(self.__reader.line_no(),
                               "missing @tindex entry for %s" % newtype)
                seen_types.append(newtype)
                if defined_types.has_key(newtype):
                    self.error(self.__reader.line_no(),
                               "redefinition of ``%s''" % newtype)
                else:
                    defined_types[newtype] = typedef

        # Check that all types in the @tindex were defined.
        for typename in tindexed_types:
            if typename not in seen_types:
                self.error(self.__reader.line_no(),
                           "type %s not defined but @tindex entry exists" %
                           typename)

    def __parse_userdefined_struct(self, line):
        res = prot_a_struct(line)
        while 1:
            token = self.__get_token()
            if token == ')':
                return res
            name = token
            token = self.__get_token()
            if token != ':':
                self.error(self.__reader.line_no(),
                           "missing ``:'' near %s" % token)
            token = self.__get_lt_token()
            if token == 'ARRAY':
                array = 1
                token = self.__get_lt_token()
            else:
                array = 0
            if not self.__bad_type(token) and not self.__bad_arg(name):
                ret = res.add_field(name, token, array)
                if ret:
                    self.error(self.__reader.line_no(), ret)
            token = self.__get_token()
            if token != ';':
                self.error(self.__reader.line_no(),
                           "missing ``;''")

    def __parse_userdefined_bitstring(self, line):
        res = prot_a_bitstring(line)
        token = self.__get_token()
        if token != '(':
            self.error(self.__reader.line_no(),
                       "expected ``('' after BITSTRING, not ``%s''" % token)
            return None
        while 1:
            token = self.__get_token()
            if token == ')':
                return res
            name = token
            if not self.__bad_arg(name):
                ret = res.add_field(name)
                if ret:
                    self.error(self.__reader.line_no(), ret)

            token = self.__get_token()
            if token != ';':
                self.error(self.__reader.line_no(),
                           "missing ``;''")

    def __parse_userdefined_selection(self, line):
        res = prot_a_selection(line)
        token = self.__get_token()
        if token != '(':
            self.error(self.__reader.line_no(),
                       "expected ``('' after SELECTION, not ``%s''" % token)
            return None
        while 1:
            token = self.__get_token()
            if token == ')':
                return res
            number = token
            try:
                int(number)
            except:
                self.error(self.__reader.line_no(),
                           "bad number ``%s''" % number)
                return None
            token = self.__get_token()
            if token != '=':
                self.error(self.__reader.line_no(),
                           "missing ``:'' near %s" % token)
            name = self.__get_token()
            tail_name = self.__get_token()
            token = self.__get_token()
            if token != ':':
                self.error(self.__reader.line_no(),
                           "missing ``:'' near %s" % token)
            token = self.__get_lt_token()
            if token == 'ARRAY':
                array = 1
                token = self.__get_lt_token()
            else:
                array = 0
            if (not self.__bad_type(token) and not self.__bad_arg(name)
                and not self.__bad_arg(tail_name)):

                ret = res.add_variant(number, name, tail_name, token, array)
                if ret:
                    self.error(self.__reader.line_no(), ret)
            token = self.__get_token()
            if token != ';':
                self.error(self.__reader.line_no(),
                           "missing ``;''")


    def __parse_userdefined_enumeration_of(self, line):
        token = self.__get_token()
        if token != '(':
            self.error(self.__reader.line_no(),
                       "expected ``('' after ENUMERATION-OF, not ``%s''" % 
                       token)
            return None
        name = self.__get_lt_token()
        token = self.__get_token()
        if token != ')':
            self.error(self.__reader.line_no(),
                       "missing close ``)'', got ``%s''" % 
                       token)
            return None
        if self.__bad_type(name):
            return None
        return prot_a_enumration_of(line, name)

    def __parse_request(self):
        self.__tokens = []
        req = self.__get_token()
        if req != self.__findex:
            self.error(self.__reader.line_no(),
                       "wrong request name ``%s''" % req)
            return

        if self.__get_token() != '[':
            self.error(self.__reader.line_no(), "missing ``[''")
            return
            
        nr = self.__get_token()
        if type(nr) != types.IntType:
            self.error(self.__reader.line_no(), "bad request number")

        if self.__implemented_reqs.has_key(req):
            if self.__implemented_reqs[req] != str(nr):
                self.error(self.__reader.line_no(),
                           "``%s'' is implemented as request number %s, not %s"
                           % (req, self.__implemented_reqs[req], nr))
        else:
            self.error(self.__reader.line_no(),
                       "request ``%s'' not implemented" % req)
        
        if self.__get_token() != ']':
            self.error(self.__reader.line_no(), "missing ``]''")
            return

        paren = self.__get_token()
        if paren == '(':
            next = self.__get_token()
            if next != ')':
                self.__unget_token(next)
                self.__parse_request_arg()
                next = self.__get_token()
            if next != ')':
                self.error(self.__reader.line_no(),
                           "missing close parenthesis after arguments")
                return
        elif paren == '((':
            self.__parse_request_arg()
            next = self.__get_token()
            while next == ';':
                self.__parse_request_arg()
                next = self.__get_token()
            if next != '))':
                self.error(self.__reader.line_no(),
                           "missing double close parenthesis after arguments")
                return
        else:
            self.error(self.__reader.line_no(),
                       "missing argument list")
            return

        if self.__get_token() != '->':
            self.error(self.__reader.line_no(), "missing ``->''")
            return

        if self.__get_token() != '(':
            self.error(self.__reader.line_no(), "missing ``('' for result")
            return

        next = self.__get_token()
        if next != ')':
            self.__unget_token(next)
            (ret_type, ret_array) = self.__parse_type()
            next = self.__get_token()
        if next != ')':
            self.error(self.__reader.line_no(), "missing ``)'' for result")
            return

        if self.__get_token() != ';':
            self.error(self.__reader.line_no(), "missing final ``;''")
            return

        if self.__get_token() != '@end':
            self.error(self.__reader.line_no(), "extra garbage found")
            return

        return

    def __parse_type(self):
        token = self.__get_lt_token()
        if token == 'ARRAY':
            token = self.__get_lt_token()
            array = 1
        else:
            array = 0
        if self.__bad_type(token):
            return (None, 0)
        if not defined_types.has_key(token):
            self.error(self.__reader.line_no(),
                       "undefined type ``%s''" % token)
        defined_types[token].use()
        return (token, array)

    def __get_lt_token(self):
        token = self.__get_token()
        if token != '@lt':
            self.error(self.__reader.line_no(),
                       'expected @lt{}-enclosed token')
            return token
        token = self.__get_token()
        if token != '{':
            self.error(self.__reader.line_no(),
                       "expected ``{'', not ``%s''" % token)
            return token
        result = self.__get_token()
        token = self.__get_token()
        if token != '}':
            self.error(self.__reader.line_no(),
                       "expected ``}'', not ``%s''" % token)
        return result
        

    def __bad_type(self, tp):
        if tp in self.__builtin_types:
            return 0

        # This is an ugly special case, that is a good type name.
        # More or less.
        if tp == "UConference":
            return 0

        ok = 1
        if len(tp) < 0:
            ok = 0
        if ok and not tp[0].isupper():
            ok = 0
        upper = 0
        for c in tp[1:]:
            if upper:
                if c == '-' or not c.isupper():
                    ok = 0
                upper = 0
            else:
                if c == '-':
                    upper = 1
                elif not c.islower():
                    ok = 0
        if ok and tp[-1] == '-':
            ok = 0
        if not ok:
            self.error(self.__reader.line_no(), "bad type name ``%s''" % (tp,))
        return not ok

    def __bad_arg(self, arg):
        if number_suffixed(arg, 'reserved'):
            return 0
        if number_suffixed(arg, 'flg'):
            return 0
        ok = 1
        if len(arg) < 0:
            ok = 0
        if ok and not arg[0].islower():
            ok = 0
        for c in arg[1:]:
            if not c.islower() and c != '-':
                ok = 0
        if ok and arg[-1] == '-':
            ok = 0
        if not ok:
            if arg == '))':
                self.__unget_token(arg)
                self.error(self.__reader.line_no(),
                           "extra semicolon after last arg")
            else:
                self.error(self.__reader.line_no(),
                           "bad argument ``%s''" % (arg,))
        return not ok

    def __parse_request_arg(self):
        argname = self.__get_token()
        if self.__bad_arg(argname):
            return
        if self.__get_token() != ':':
            self.error(self.__reader.line_no(), "missing ``:'' after argument")
            return
        (tp, array) = self.__parse_type()
        if tp == None:
            return
        if self.__args.has_key(argname):
            self.error(self.__reader.line_no(),
                       "argument name ``%s'' used twice" % (argname, ))
            return
        self.__args[argname] = [self.__reader.line_no(), 0]

    def toplevel_field(self, arg, line_no):
        if not defined_fields.has_key(arg):
            if not undefined_fields.has_key(arg):
                undefined_fields[arg] = []
            undefined_fields[arg].append(line_no)

    def __unget_token(self, token):
        self.__tokens.insert(0, token)

    def __get_token(self):
        if len(self.__tokens) > 0:
            res = self.__tokens[0]
            del self.__tokens[0]
            return res
        c = self.__reader.getc()
        while c.isspace():
            c = self.__reader.getc()
        if c.isalpha():
            res = c
            while 1:
                c = self.__reader.getc()
                if not c.isalpha() and c not in "-" and not c.isdigit():
                    if c.isdigit():
                        self.error(self.__reader.line_no(),
                                   "bad token ``%s''" % (res + c))
                    self.__reader.ungetc(c)
                    return res
                res = res + c
        elif c == ':':
            d = self.__reader.getc()
            if d != ':':
                self.__reader.ungetc(d)
                return c
            d = self.__reader.getc()
            if d != '=':
                self.__reader.ungetc(d)
                self.__reader.ungetc(':')
                return c
            return '::='
        elif c in '[];|={}':
            return c
        elif c in '()':
            d = self.__reader.getc()
            if c != d:
                self.__reader.ungetc(d)
                return c
            else:
                return c + d
        elif c.isdigit():
            res = c
            while 1:
                c = self.__reader.getc()
                if not c.isdigit():
                    if c.isalpha():
                        self.error(self.__reader.line_no(),
                                   "bad token ``%s''" % (res + c))
                    self.__reader.ungetc(c)
                    if res[0] == '0' and len(res) > 1:
                        self.error(self.__reader.line_no(),
                                   "bad number ``%s''" % (res,))
                    return int(res)
                res = res + c
        elif c == '-':
            d = self.__reader.getc()
            if d == '>':
                return '->'
            else:
                self.error(self.__reader.line_no(),
                           "bad token ``%s%s''" % (c, d))
                self.__reader.ungetc(d)
                return c
        elif c == '@':
            res = c
            while 1:
                c = self.__reader.getc()
                if not c.isalpha():
                    self.__reader.ungetc(c)
                    return res
                res = res + c
        elif c == '!':
            # This is a comment.  Skip it.  We *should* parse it like
            # toplevel Texinfo code until the next end of line, but
            # doing so is too hard -- there is only a single comment
            # in the entire document.
            while 1:
                c = self.__reader.getc()
                if c == '\n':
                    return self.__get_token()
        else:
            self.error(self.__reader.line_no(),
                       "bad character ``%s''" % (c,))
            return c
        
                
    def __read_arg(self):
        arg = ""
        level = 0
        while 1:
            c = self.__reader.getc()

            if c == '@':
                # Handle '@{', '@}', and '@@' (and everything else
                # that starts with `@').
                arg = arg + c
                c = self.__reader.getc()
            elif c == '{':
                level = level + 1
            elif c == '}':
                if level == 0:
                    return arg
                level = level - 1

            arg = arg + c

    def error(self, line_no, errmsg):
        self.__reader.error(line_no, errmsg)

if __name__ == '__main__':
    l = lexer(sys.argv[1])
    sys.exit(l.run())
