/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: remove.c,v 1.2 94/06/19 11:22:12 nau Exp $";

/* functions used to remove vias, pins ...
 */

#include <string.h>

#include "global.h"

#include "data.h"
#include "draw.h"
#include "error.h"
#include "memory.h"
#include "remove.h"
#include "search.h"

/* ---------------------------------------------------------------------------
 * remove PCB
 */
void RemovePCB(PCBTypePtr Ptr)
{
	FreePCBMemory(Ptr);
	SaveFree(Ptr);
}

/* ---------------------------------------------------------------------------
 * removes a via
 */
Boolean RemoveVia(PinTypePtr Via)
{
	if (!Via)
		return(False);

		/* erase from screen and memory */
	ErasePinOrVia(Via);
	SaveFree(Via->Name);
	*Via = PCB->Via[--PCB->ViaN];
	memset(&PCB->Via[PCB->ViaN], 0, sizeof(PinType));

		/* the calling routine is responsible for a status update */
	return(PCB->Changed = True);
}

/* ---------------------------------------------------------------------------
 * removes a line from a layer 
 */
Boolean RemoveLine(LayerTypePtr Layer, LineTypePtr Line)
{
	if (!Layer || !Line)
		return(False);

		/* erase from screen */
	EraseLine(Line);
	*Line = Layer->Line[--Layer->LineN];
	memset(&Layer->Line[Layer->LineN], 0, sizeof(LineType));

		/* the calling routine is responsible for a status update */
	return(PCB->Changed = True);
}

/* ---------------------------------------------------------------------------
 * removes a text from a layer
 */
Boolean RemoveText(LayerTypePtr Layer, TextTypePtr Text)
{
	if (!Layer || !Text)
		return(False);

		/* erase from screen */
	EraseText(Text);
	SaveFree(Text->TextString);
	*Text = Layer->Text[--Layer->TextN];
	memset(&Layer->Text[Layer->TextN], 0, sizeof(TextType));

		/* the calling routine is responsible for a status update */
	return(PCB->Changed = True);
}

/* ---------------------------------------------------------------------------
 * removes a rectangle from a layer
 */
Boolean RemoveRect(LayerTypePtr Layer, RectTypePtr Rect)
{
	if (!Layer || !Rect)
		return(False);

		/* erase from screen */
	EraseRect(Rect);
	*Rect = Layer->Rect[--Layer->RectN];
	memset(&Layer->Rect[Layer->RectN], 0, sizeof(RectType));

		/* the calling routine is responsible for a status update */
	return(PCB->Changed = True);
}
/* ---------------------------------------------------------------------------
 * removes a element
 */
Boolean RemoveElement(ElementTypePtr Element)
{
	if (!Element)
		return(False);

		/* erase from screen */
	EraseElement(Element);
	FreeElementMemory(Element);
	*Element = PCB->Element[--PCB->ElementN];
	memset(&PCB->Element[PCB->ElementN], 0, sizeof(ElementType));

		/* the calling routine is responsible for a status update */
	return(PCB->Changed = True);
}

/* ---------------------------------------------------------------------------
 * removes vias, text, rectangles and lines located within
 * the specified block on allvisible layers
 */
Boolean RemoveAllInRectangle(Position X1, Position Y1, Position X2, Position Y2)
{
	Position		x1, y1,		/* passed arguments are not necessary top-left */
					x2, y2;		/* and lower-right corner */
	PinTypePtr		via;
	LineTypePtr		line;
	RectTypePtr		rect;
	LayerTypePtr	layer;
	TextTypePtr		text;
	ElementTypePtr	element;
	Cardinal		n,
					i;
	Boolean			changed = False;

	x1 = MIN(X1, X2);
	x2 = MAX(X1, X2);
	y1 = MIN(Y1, Y2);
	y2 = MAX(Y1, Y2);

		/* start with vias */
	if (PCB->ViaOn)
		for (n = PCB->ViaN, via = PCB->Via+n-1; n; n--, via--)
			if (VIA_OR_PIN_IN_RECTANGLE(via, x1, y1, x2, y2))
				changed |= RemoveVia(via);

		/* now check all layers */
	for (i = 0; i < MAX_LAYER; i++)
		if (PCB->Layer[i].On)
		{
			layer = &PCB->Layer[i];

				/* check lines */
			for (n = layer->LineN, line = layer->Line+n-1; n; n--, line--)
				if (LINE_IN_RECTANGLE(line, x1, y1, x2, y2))
					changed |= RemoveLine(layer, line);

				/* check rectangles */
			for (n = layer->RectN, rect = layer->Rect+n-1; n; n--, rect--)
				if (RECT_IN_RECTANGLE(rect, x1, y1, x2, y2))
					changed |= RemoveRect(layer, rect);

				/* check text */
			for (n = layer->TextN, text = layer->Text+n-1; n; n--, text--)
				if (TEXT_IN_RECTANGLE(text, x1, y1, x2, y2))
					changed |= RemoveText(layer, text);
		}

		/* elements */
	for (n = PCB->ElementN, element = PCB->Element+n-1; n; n--, element--)
		if (RECT_IN_RECTANGLE(&element->Rect, x1, y1, x2, y2))
			changed |= RemoveElement(element);

	PCB->Changed |= changed;
	return(changed ? True : False);
}
