/*
 * Author:      William Chia-Wei Cheng (william@cs.ucla.edu)
 *
 * Copyright (C) 1990-1995, William Cheng.
 *
 * Permission limited to the use, copy, display, distribute without
 * charging for a fee, and produce derivative works of "tgif" and
 * its documentation for not-for-profit purpose is hereby granted by
 * the Author, provided that the above copyright notice appears in
 * all copies made of "tgif" and that both the copyright notice
 * and this permission notice appear in supporting documentation,
 * and that the name of the Author not be used in advertising or
 * publicity pertaining to distribution of the software without
 * specific, written prior permission.  The Author makes no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied
 * warranty.  All other rights (including, but not limited to, the
 * right to sell "tgif", the right to sell derivative works of
 * "tgif", and the right to distribute "tgif" for a fee) are
 * reserved by the Author.
 *
 * THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#ifndef lint
static char RCSid[] =
      "@(#)$Header: /u/multimedia/william/X11/TGIF2/RCS/navigate.c,v 2.36 1995/05/15 04:14:10 william Exp $";
#endif

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include "const.h"
#include "types.h"

#include "auxtext.e"
#include "button.e"
#include "choice.e"
#include "cmd.e"
#include "color.e"
#include "cursor.e"
#include "dialog.e"
#include "drawing.e"
#include "file.e"
#include "font.e"
#include "mainloop.e"
#include "mainmenu.e"
#include "menu.e"
#include "msg.e"
#include "names.e"
#ifndef _NO_EXTERN
#include "navigate.e"
#endif
#include "obj.e"
#include "page.e"
#include "rect.e"
#include "remote.e"
#include "scroll.e"
#include "select.e"
#include "setup.e"
#include "stk.e"
#include "util.e"

#ifndef XK_KP_Left
#define XK_KP_Left	0xFF96
#define XK_KP_Up	0xFF97
#define XK_KP_Right	0xFF98
#define XK_KP_Down	0xFF99
#endif /* ~XK_KP_LEFT */

extern int atoi ARGS_DECL((char*));
extern char *getenv ARGS_DECL((char*));

int navigatingBackAndForth=FALSE;
int inHyperSpace=FALSE;
int autoHyperSpaceOnRemote=TRUE;
int allowLaunchInHyperSpace=FALSE;

char * navigateMenuStr[] = {
      "GoBack             ",
      "GoForward          ",
      "RefreshCurrent     ",
      "HotList            ",
      "AddCurrentToHotList",
      "SessionHistory     ",
      "GoHyperSpace       ",
      NULL
};
char * navigateMenuStrInHyperSpace[] = {
      "GoBack             ",
      "GoForward          ",
      "RefreshCurrent     ",
      "HotList            ",
      "AddCurrentToHotList",
      "SessionHistory     ",
      "LeaveHyperSpace    ",
      NULL
};
static char * navigateMenuDescription[] = {
      "Go back one file",
      "Go forward one file",
      "Reload the current file",
      "Navigate using the hot-list",
      "Add the current file to the hot-list",
      "Go to a file visited during this session",
      "Go into HyperSpace -- cursor will indicate hot spots",
      NULL
};
static char * navigateMenuDescriptionInHyperSpace[] = {
      "Go back one file",
      "Go forward one file",
      "Reload the current file",
      "Navigate using the hot-list",
      "Add the current file to the hot-list",
      "Go to a file visited during this session",
      "Get out of HyperSpace",
      NULL
};

static int validHotListFileName=FALSE;
static char *hotListFileName=NULL;

struct NavigateRec {
   struct StkRec *stk;
   struct NavigateRec *next, *prev;
   char *full_fname;
   char *doc_name;
   int cur_page_num, orig_x, orig_y, zoom_scale, zoomed_in;
};

static struct NavigateRec *firstNavigate=NULL, *lastNavigate=NULL;
static struct NavigateRec *curNavigate=NULL;

static struct NavigateRec *firstSessionHistory=NULL, *lastSessionHistory=NULL;

static struct URLCacheRec *firstURLCache=NULL, *lastURLCache=NULL;
static int maxURLCache=(-1), curURLCache=0;

static
void InsertNavigate(pnr_prev, pnr_next, pnr)
   struct NavigateRec *pnr_prev, *pnr_next, *pnr;
{
   pnr->prev = pnr_prev;
   pnr->next = pnr_next;

   if (pnr_prev == NULL) {
      firstNavigate = pnr;
   } else {
      pnr_prev->next = pnr;
   }

   if (pnr_next == NULL) {
      lastNavigate = pnr;
   } else {
      pnr_next->prev = pnr;
   }
}

static
void DeleteNavigate(pnr)
   struct NavigateRec *pnr;
{
   if (pnr->full_fname != NULL) cfree(pnr->full_fname);
   if (pnr->doc_name != NULL) cfree(pnr->doc_name);
   cfree(pnr);
}

static
void ClearNavigateRecords(pnr)
   struct NavigateRec *pnr;
{
   struct NavigateRec *pnr_next;

   for (; pnr != NULL; pnr=pnr_next) {
      pnr_next = pnr->next;
      DeleteNavigate(pnr);
   }
   lastNavigate = curNavigate;
   if (lastNavigate == NULL) firstNavigate = NULL;
}

static
void ClearSessionHistory()
{
   struct NavigateRec *pnr, *pnr_prev;

   for (pnr=lastSessionHistory; pnr != NULL; pnr=pnr_prev) {
      pnr_prev = pnr->prev;
      if (pnr->full_fname != NULL) cfree(pnr->full_fname);
      if (pnr->doc_name != NULL) cfree(pnr->doc_name);
      cfree(pnr);
   }
   lastSessionHistory = firstSessionHistory = NULL;
}

static
void FreeAnURLCache(url_cache)
   struct URLCacheRec *url_cache;
{
   if (url_cache == NULL) return;
   if (url_cache->remote_buf != NULL) cfree(url_cache->remote_buf);
   if (url_cache->content_type != NULL) cfree(url_cache->content_type);
   if (url_cache->simple_url_name != NULL) cfree(url_cache->simple_url_name);
   cfree(url_cache);
}

static
void CleanUpURLCache()
{
   struct URLCacheRec *next_cache;

   for ( ; firstURLCache != NULL; firstURLCache=next_cache) {
      next_cache = firstURLCache->next;
      FreeAnURLCache(firstURLCache);
   }
   curURLCache = 0;
   firstURLCache = lastURLCache = NULL;
}

void CleanUpNavigate()
{
   curNavigate = NULL;
   ClearNavigateRecords(firstNavigate);
   if (hotListFileName != NULL) cfree(hotListFileName);
   hotListFileName = NULL;
   validHotListFileName = FALSE;
   ClearSessionHistory();

   CleanUpURLCache();
}

static
void InitURLCache()
{
   if (maxURLCache == (-1)) {
      char *c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"MaxNavigateCacheBuffers");

      maxURLCache = 40;
      if (c_ptr != NULL) {
         maxURLCache = atoi(c_ptr);
         if (maxURLCache < 0) {
            fprintf(stderr, "Invalid %s*%s: '%s', 40 is used.\n",
                  TOOL_NAME, "MaxNavigateCacheBuffers", c_ptr);
            maxURLCache = 40;
         }
      }
      curURLCache = 0;
      firstURLCache = lastURLCache = NULL;
   }
}

static
char *SimpleURLName(psz_url)
   char *psz_url;
{
   char *c_ptr=strchr(psz_url, '#');

   if (c_ptr != NULL) {
      char *return_buf;

      *c_ptr = '\0';
      return_buf = UtilStrDup(psz_url);
      *c_ptr = '#';
      return return_buf;
   }
   return UtilStrDup(psz_url);
}

static
void UnlinkURLCache(url_cache)
   struct URLCacheRec *url_cache;
{
   if (url_cache == NULL) return;
   if (url_cache->prev == NULL) {
      firstURLCache = url_cache->next;
   } else {
      url_cache->prev->next = url_cache->next;
   }
   if (url_cache->next == NULL) {
      lastURLCache = url_cache->prev;
   } else {
      url_cache->next->prev = url_cache->prev;
   }
   url_cache->prev = url_cache->next = NULL;
   curURLCache--;
}

static
void InsertURLCache(prev_url, next_url, url_cache)
   struct URLCacheRec *prev_url, *next_url, *url_cache;
{
   url_cache->prev = prev_url;
   url_cache->next = next_url;
   if (prev_url == NULL) {
      firstURLCache = url_cache;
   } else {
      prev_url->next = url_cache;
   }
   if (next_url == NULL) {
      lastURLCache = url_cache;
   } else {
      next_url->prev = url_cache;
   }
   curURLCache++;
}

void UpdateLRU(url_cache)
   struct URLCacheRec *url_cache;
{
   UnlinkURLCache(url_cache);
   InsertURLCache(lastURLCache, NULL, url_cache);
}

struct URLCacheRec *FindURLCache(psz_url, update_lru)
   char *psz_url;
   int update_lru;
{
   char *simple_url_name=SimpleURLName(psz_url);
   struct URLCacheRec *url_cache;

   InitURLCache();
   if (simple_url_name == NULL) return NULL;
   for (url_cache=lastURLCache; url_cache != NULL; url_cache=url_cache->prev) {
      if (strcmp(simple_url_name, url_cache->simple_url_name) == 0) {
         cfree(simple_url_name);
         if (update_lru) UpdateLRU(url_cache);
         return url_cache;
      }
   }
   cfree(simple_url_name);
   return NULL;
}

void UpdateURLCache(psz_url, psz_remote_buf, psz_content_type, remote_buf_sz,
      is_html)
   char *psz_url, *psz_remote_buf, *psz_content_type;
   int remote_buf_sz, is_html;
{
   char *simple_url_name=SimpleURLName(psz_url);
   struct URLCacheRec *url_cache;

   InitURLCache();
   if (simple_url_name == NULL) return;
   for (url_cache=lastURLCache; url_cache != NULL; url_cache=url_cache->prev) {
      if (strcmp(simple_url_name, url_cache->simple_url_name) == 0) {
         break;
      }
   }
   if (url_cache != NULL) {
      UnlinkURLCache(url_cache);
      FreeAnURLCache(url_cache);
   } else {
      if (curURLCache >= maxURLCache) {
         url_cache = firstURLCache;
         UnlinkURLCache(url_cache);
         FreeAnURLCache(url_cache);
      }
   }
   url_cache = (struct URLCacheRec*)calloc(1, sizeof(struct URLCacheRec));
   if (url_cache == NULL) {
      FailAllocMessage();
      cfree(simple_url_name);
      return;
   }
   memset(url_cache, 0, sizeof(struct URLCacheRec));
   url_cache->remote_buf_sz = remote_buf_sz;
   url_cache->is_html = is_html;
   url_cache->remote_buf = UtilStrDup(psz_remote_buf);
   url_cache->content_type = UtilStrDup(psz_content_type);
   url_cache->simple_url_name = simple_url_name;
   InsertURLCache(lastURLCache, NULL, url_cache);
}

static
int InitHotListFileName()
{
   if (!validHotListFileName)
   {
      char *c_ptr;

      hotListFileName = NULL;
      if ((c_ptr=XGetDefault(mainDisplay,TOOL_NAME,"HotListFileName")) !=
            NULL) {
         hotListFileName = UtilStrDup(c_ptr);
      } else if ((c_ptr=getenv("HOME")) != NULL) {
         int len=strlen(c_ptr)+30;

         hotListFileName = (char*)calloc(len+1, sizeof(char));
         if (hotListFileName == NULL) fprintf(stderr, "Can not calloc().\n");
         sprintf(hotListFileName, "%s/.%s_hotlist", c_ptr, TOOL_NAME);
      } else {
         return FALSE;
      }
   }
   validHotListFileName = TRUE;
   return TRUE;
}

void BeforeNavigate()
{
   if (curFileDefined && curNavigate != NULL) {
      curNavigate->cur_page_num = curPageNum;
      curNavigate->orig_x = drawOrigX;
      curNavigate->orig_y = drawOrigY;
      curNavigate->zoom_scale = zoomScale;
      curNavigate->zoomed_in = zoomedIn;
   }
}

static
void AddToSessionHistory(cur_pnr)
   struct NavigateRec *cur_pnr;
{
   struct NavigateRec *pnr=(struct NavigateRec *)calloc(1,
         sizeof(struct NavigateRec));

   if (pnr == NULL) fprintf(stderr, "Can not calloc().\n");

   pnr->prev = lastSessionHistory;
   pnr->next = NULL;
   pnr->stk = NULL;
   pnr->full_fname = (cur_pnr->full_fname==NULL ? NULL :
         UtilStrDup(cur_pnr->full_fname));
   pnr->doc_name = (cur_pnr->doc_name==NULL ? NULL :
         UtilStrDup(cur_pnr->doc_name));

   if (lastSessionHistory == NULL) {
      firstSessionHistory = pnr;
   } else {
      lastSessionHistory->next = pnr;
   }
   lastSessionHistory = pnr;
}

void CommitNavigate()
   /* add a navigation record at the end */
{
   int len;

   if (curNavigate == NULL) {
      ClearNavigateRecords(firstNavigate);
   } else if (curNavigate != lastNavigate) {
      ClearNavigateRecords(curNavigate->next);
   }

   if (!curFileDefined) return;

   curNavigate = (struct NavigateRec *)calloc(1, sizeof(struct NavigateRec));
   if (curNavigate == NULL) fprintf(stderr, "Can not calloc().\n");
   curNavigate->stk = NULL;
   curNavigate->next = curNavigate->prev = NULL;
   curNavigate->full_fname = curNavigate->doc_name = NULL;
   curNavigate->cur_page_num = 1;
   curNavigate->orig_x = curNavigate->orig_y = 0;
   curNavigate->zoom_scale = 0;
   curNavigate->zoomed_in = FALSE;

   len = strlen(curDir)+1+strlen(curFileName);

   curNavigate->full_fname = (char*)calloc(len+1, sizeof(char));
   if (curNavigate->full_fname == NULL) {
      fprintf(stderr, "Can not calloc().\n");
   }
   sprintf(curNavigate->full_fname, "%s/%s", curDir, curFileName);
   if (firstPage != NULL && firstPage->name != NULL) {
      curNavigate->doc_name = UtilStrDup(firstPage->name);
   } else {
      curNavigate->doc_name = NULL;
   }
   curNavigate->stk = topStk;

   InsertNavigate(lastNavigate, NULL, curNavigate);
   curNavigate = lastNavigate;
   AddToSessionHistory(curNavigate);
}

static
void PostNavigate(pnr)
   struct NavigateRec *pnr;
{
   struct AttrRec *exec_attr=FindFileAttrWithName("auto_exec=");

   if (exec_attr != NULL) {
      DoExecLoop(NULL, exec_attr);
   } else if (pnr != NULL) {
      ScrollToSpecifiedOrigin(pnr->cur_page_num, pnr->orig_x, pnr->orig_y,
            pnr->zoom_scale, pnr->zoomed_in);
   }
}

static
void NavigateTo(full_fname, do_not_save, force_load)
   char *full_fname;
   int do_not_save, force_load;
{
   if (FileIsRemote(full_fname)) {
      char *buf=NULL, *content_type=NULL;
      int rc, buf_sz=0, is_html=FALSE;

      SetWatchCursor(drawWindow);
      SetWatchCursor(mainWindow);
      SaveStatusStrings();
      rc = LoadRemoteFileInMem(full_fname, &buf, &content_type, &buf_sz,
            &is_html, force_load);
      RestoreStatusStrings();
      SetDefaultCursor(mainWindow);
      ShowCursor();
      if (rc && buf != NULL) {
         navigatingBackAndForth = TRUE;
         LoadRemoteFileFromMem(full_fname, buf, content_type, buf_sz, is_html);
         navigatingBackAndForth = FALSE;
      } else if (do_not_save) {
         SetFileModified(TRUE);
      }
      if (content_type != NULL) FreeRemoteBuf(content_type);
      if (buf != NULL) FreeRemoteBuf(buf);
   } else {
      navigatingBackAndForth = TRUE;
      if (!LoadFile(full_fname, TRUE)) {
         if (do_not_save) {
            SetFileModified(TRUE);
         }
      }
      navigatingBackAndForth = FALSE;
   }
}

void NavigateBack()
{
   int do_not_save=FALSE;
   struct NavigateRec nr;

   if (curNavigate != NULL && curNavigate->stk != NULL &&
         curNavigate->stk == topStk && (curNavigate->prev == NULL ||
         (curNavigate->prev != NULL && curNavigate->prev->stk != topStk))) {
      /* curNavigate->stk = NULL; */
      /* if (curFileDefined) curNavigate = curNavigate->prev; */
      PopIcon();
      return;
   }
   while (!DirIsRemote(curDir) && fileModified) {
      switch (MsgBox(
            "File modified, save file before going back? [ync](y)",
            TOOL_NAME, YNC_MB)) {
      case MB_ID_YES: SaveFile(); break;
      case MB_ID_NO: do_not_save=TRUE; SetFileModified(FALSE); break;
      case MB_ID_CANCEL: return;
      }
   }
   MakeQuiescent();
   if (curNavigate == NULL || (curFileDefined && curNavigate->prev == NULL)) {
      MsgBox("Nothing to go back to!", TOOL_NAME, INFO_MB);
      if (do_not_save) SetFileModified(TRUE);
      return;
   }
   BeforeNavigate();
   if (curFileDefined) curNavigate = curNavigate->prev;
   memcpy(&nr, curNavigate, sizeof(struct NavigateRec));
   NavigateTo(curNavigate->full_fname, do_not_save, FALSE);
   PostNavigate(&nr);
}

void NavigateForward()
{
   int do_not_save=FALSE;
   struct NavigateRec nr;

   while (!DirIsRemote(curDir) && fileModified) {
      switch (MsgBox(
            "File modified, save file before going forward? [ync](y)",
            TOOL_NAME, YNC_MB)) {
      case MB_ID_YES: SaveFile(); break;
      case MB_ID_NO: do_not_save=TRUE; SetFileModified(FALSE); break;
      case MB_ID_CANCEL: return;
      }
   }
   MakeQuiescent();
   if (curNavigate == lastNavigate) {
      MsgBox("Nothing to go forward to!", TOOL_NAME, INFO_MB);
      if (do_not_save) SetFileModified(TRUE);
      return;
   }
   BeforeNavigate();
   if (curFileDefined) curNavigate = curNavigate->next;
   memcpy(&nr, curNavigate, sizeof(struct NavigateRec));
   NavigateTo(curNavigate->full_fname, do_not_save, FALSE);
   PostNavigate(&nr);
}

void AdjustNavigate()
{
   while (curNavigate != NULL && curNavigate->stk != NULL &&
         curNavigate->stk == topStk) {
      curNavigate->stk = NULL;
      if (curNavigate->prev == NULL || (curNavigate->prev != NULL &&
            curNavigate->prev->stk != topStk)) {
         if (curFileDefined && curNavigate->prev != NULL) {
            curNavigate = curNavigate->prev;
         }
         return;
      } else {
         curNavigate = curNavigate->prev;
      }
   }
   Msg("Possible fatal error.  Please read the terminal error output.");
   fprintf(stderr, "Warning:  fail to find record in AdjustNavigate().\n");
   fprintf(stderr, "\nNavigating may crash %s.\n", TOOL_NAME);
   fprintf(stderr, "Safest thing to do is to save and exit.\n");
   fprintf(stderr, "Please try to reproduce this error and\n");
   fprintf(stderr, "\tsend bug report to william@cs.ucla.edu.\n");
   MsgBox("Possible fatal error.  Please read the terminal error output.",
         TOOL_NAME, STOP_MB);
   XFlush(mainDisplay);
   XSync(mainDisplay, False);
}

void NavigateRefresh()
{
   if (!curFileDefined) {
      MsgBox("Can not refresh an undefined file.", TOOL_NAME, INFO_MB);
      return;
   }
   while (!DirIsRemote(curDir) && fileModified) {
      switch (MsgBox("File modified, save file before reloading? [ync](y)",
         TOOL_NAME, YNC_MB)) {
      case MB_ID_YES: SaveFile(); break;
      case MB_ID_NO: SetFileModified(FALSE); break;
      case MB_ID_CANCEL: return;
      }
   }
   NavigateTo(curNavigate->full_fname, TRUE, TRUE);
   PostNavigate(NULL);
}

static
char **ReadHotListFile(pn_count)
   int *pn_count;
{
   FILE *fp;
   char **ppsz_buf=NULL, *buf;
   int num_lines=0;

   if (pn_count != NULL) *pn_count = 0;
   if (!InitHotListFileName() || hotListFileName==NULL) {
      sprintf(gszMsgBox, "%s.  %s %s*%s.",
            "Hot list file undefined",
            "Please specify it in", TOOL_NAME, "HotListFileName");
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return NULL;
   }
   if ((ppsz_buf=(char**)calloc(num_lines+1, sizeof(char*))) == NULL) {
      fprintf(stderr, "Can not calloc().\n");
      return NULL;
   }
   if ((fp=fopen(hotListFileName, "r")) == NULL) {
      ppsz_buf[num_lines] = NULL;
      return ppsz_buf;
   }
   while ((buf=UtilGetALine(fp)) != NULL) {
      if ((ppsz_buf=(char**)realloc(ppsz_buf,
            ((++num_lines)+1)*sizeof(char*))) == NULL) {
         fprintf(stderr, "Can not calloc().\n");
         return NULL;
      }
      ppsz_buf[num_lines-1] = buf;
   }
   ppsz_buf[num_lines] = NULL;
   if ((num_lines & 0x1) != 0) {
      ppsz_buf[num_lines-1] = NULL;
      sprintf(gszMsgBox, "Malformed hot list file '%s'.", hotListFileName);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
   }
   fclose(fp);
   if (pn_count != NULL) *pn_count = (num_lines>>1);
   return ppsz_buf;
}

static
DspList *HotListListing(ppsz_buf, pn_count)
   char **ppsz_buf;
   int *pn_count;
{
   int i;
   char **s_ptr;
   DspList *pdl, *dsp_ptr;

   *pn_count = 0;
   for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr=(&s_ptr[2])) {
      (*pn_count)++;
   }
   pdl = (DspList*)calloc(*pn_count, sizeof(DspList));
   if (pdl == NULL) fprintf(stderr, "Can not calloc().\n");
   for (i=(*pn_count)-1, dsp_ptr=pdl, s_ptr=ppsz_buf; *s_ptr != NULL;
         i--, dsp_ptr++, s_ptr=(&s_ptr[2])) {
      if (**s_ptr == '\0') {
         strcpy(dsp_ptr->itemstr, s_ptr[1]);
         dsp_ptr->directory = FALSE; /* use file name as title */
      } else {
         strcpy(dsp_ptr->itemstr, *s_ptr);
         dsp_ptr->directory = TRUE; /* file has a title */
      }
      strcpy(dsp_ptr->pathstr, s_ptr[1]);
      dsp_ptr->next = (i==0 ? NULL : &dsp_ptr[1]);
   }
   nameEntries = (*pn_count);
   return pdl;
}

static
DspList *HistoryListing(pn_count)
   int *pn_count;
{
   int i;
   struct NavigateRec *pnr;
   DspList *pdl, *dsp_ptr;

   *pn_count = 0;
   for (pnr=lastSessionHistory; pnr != NULL; pnr=pnr->prev) {
      (*pn_count)++;
   }
   if (*pn_count == 0) return NULL;
   pdl = (DspList*)calloc(*pn_count, sizeof(DspList));
   if (pdl == NULL) fprintf(stderr, "Can not calloc().\n");
   for (i=0, dsp_ptr=pdl, pnr=firstSessionHistory; pnr != NULL;
         i++, dsp_ptr++, pnr=pnr->next) {
      if (pnr->doc_name == NULL) {
         strcpy(dsp_ptr->itemstr, pnr->full_fname);
         dsp_ptr->directory = FALSE; /* use file name as title */
      } else {
         strcpy(dsp_ptr->itemstr, pnr->doc_name);
         dsp_ptr->directory = TRUE; /* file has a title */
      }
      strcpy(dsp_ptr->pathstr, pnr->full_fname);
      dsp_ptr->next = (i==(*pn_count)-1 ? NULL : &dsp_ptr[1]);
   }
   nameEntries = (*pn_count);
   return pdl;
}

static XComposeStatus c_stat;

static
int SelectForNavigate(TopStr, Which)
   char *TopStr;
   int Which; /* either NAVIGATE_HOTLIST or NAVIGATE_HISTORY */
{
   int button_widths, str_width, graph_width;
   int str_start, button_start, graph_start;
   int dsp_w, dsp_h, w, h, win_x, win_y, exposed=FALSE;
   int goto_something=FALSE, something_deleted=FALSE;
   int button_selected=BUTTON_CANCEL;
   char buf[80];
   XWMHints wmhints;
   XSizeHints sizehints;
   XSetWindowAttributes win_attrs;

   dsp_w = DisplayWidth(mainDisplay, mainScreen);
   dsp_h = DisplayHeight(mainDisplay, mainScreen);

   switch (Which) {
   case NAVIGATE_HOTLIST:
      button_widths = ButtonWidth("GOTO", 8) + ButtonWidth("DELETE", 8) +
            ButtonWidth("CLOSE", 8) + (defaultFontWidth<<1);
      numButtons = 3;
      strcpy(buttonStr[0], "GOTO");
      strcpy(buttonStr[1], "DELETE");
      strcpy(buttonStr[2], "CLOSE");
      break;
   case NAVIGATE_HISTORY:
      button_widths = ButtonWidth("GOTO", 8) + ButtonWidth("CANCEL", 8) +
            defaultFontWidth;
      numButtons = 2;
      strcpy(buttonStr[0], "GOTO");
      strcpy(buttonStr[1], "CANCEL");
      break;
   default: return BUTTON_CANCEL;
   }

   str_width = defaultFontWidth * strlen(TopStr);
   graph_width = nameDspWinW + scrollBarW + (brdrW<<1);

   if (str_width > graph_width) {
      w = str_width + 4 * defaultFontWidth;
      str_start = 2 * defaultFontWidth;
      graph_start = (w - graph_width) / 2;
   } else {
      w = graph_width + 4 * defaultFontWidth;
      str_start = (w - str_width) / 2;
      graph_start = 2 * defaultFontWidth;
   }
   button_start = (w - button_widths) / 2;
   h = (8 + ITEM_DSPED) * ROW_HEIGHT;

   win_x = (w > dsp_w) ? 0 : (dsp_w - w)/2;
   win_y = (h > dsp_h) ? 0 : (dsp_h - h)/3;

   if ((nameBaseWin=XCreateSimpleWindow(mainDisplay, rootWindow,
         win_x, win_y, w, h, brdrW, myBorderPixel, myBgPixel)) == 0) {
      Error("SelectForNavigate()",
            "Can not XCreateSimpleWindow() for nameBaseWin");
   }
   XDefineCursor(mainDisplay, nameBaseWin, defaultCursor);

   if ((nameDspWin=XCreateSimpleWindow(mainDisplay, nameBaseWin, graph_start,
         5*ROW_HEIGHT, nameDspW, nameDspH, brdrW, myBorderPixel,
         myBgPixel)) == 0) {
      Error("SelectForNavigate()",
            "Can not XCreateSimpleWindow() for nameDspWin");
   }
   if ((nameScrollWin=XCreateSimpleWindow(mainDisplay, nameBaseWin,
         graph_start+nameDspWinW, 5*ROW_HEIGHT, scrollBarW, nameDspH,
         brdrW, myBorderPixel, myBgPixel)) == 0) {
      Error("SelectForNavigate()",
            "Can not XCreateSimpleWindow() for nameScrollWin");
   }
   win_attrs.save_under = True;
   win_attrs.colormap = mainColormap;
   XChangeWindowAttributes (mainDisplay, nameBaseWin, CWSaveUnder | CWColormap,
         &win_attrs);

   wmhints.flags = InputHint | StateHint;
   wmhints.input = True;
   wmhints.initial_state = NormalState;
   XSetWMHints(mainDisplay, nameBaseWin, &wmhints);

   sizehints.flags = PPosition | PSize | USPosition | PMinSize | PMaxSize;
   sizehints.x = win_x;
   sizehints.y = win_y;
   sizehints.width = sizehints.min_width = sizehints.max_width = w;
   sizehints.height = sizehints.min_height = sizehints.max_height = h;
#ifdef NOTR4MODE
   XSetNormalHints(mainDisplay, nameBaseWin, &sizehints);
#else
   XSetWMNormalHints(mainDisplay, nameBaseWin, &sizehints);
#endif
   switch (Which) {
   case NAVIGATE_HOTLIST: sprintf(buf, "%s - Hot List", TOOL_NAME); break;
   case NAVIGATE_HISTORY: sprintf(buf, "%s - History", TOOL_NAME); break;
   }
   XStoreName(mainDisplay, nameBaseWin, buf);

   XSetTransientForHint(mainDisplay, nameBaseWin, mainWindow);

#ifdef MAPBEFORESELECT
   XMapWindow(mainDisplay, nameBaseWin);
   XSelectInput(mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow(mainDisplay, nameDspWin);
   XSelectInput(mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow(mainDisplay, nameScrollWin);
   XSelectInput(mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
#else
   XSelectInput(mainDisplay, nameBaseWin,
         KeyPressMask | ButtonPressMask | ExposureMask | StructureNotifyMask);
   XMapWindow(mainDisplay, nameBaseWin);
   XSelectInput(mainDisplay, nameDspWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow(mainDisplay, nameDspWin);
   XSelectInput(mainDisplay, nameScrollWin,
         KeyPressMask | ButtonPressMask | ExposureMask);
   XMapWindow(mainDisplay, nameScrollWin);
#endif

   if (warpToWinCenter) {
      XWarpPointer(mainDisplay, None, nameBaseWin, 0, 0, 0, 0, (w>>1), (h>>1));
   }

   XSync(mainDisplay, False);

   justClicked = FALSE;

   Msg("");

   do {
      int i, entries=0, rc, modified=FALSE;
      int changing=TRUE, name_index, ok=TRUE, navigated_to=FALSE;
      char name[MAXPATHLENGTH+1], **s_ptr=NULL, **ppsz_buf=NULL;
      DspList *dsp_ptr;

      button_selected = BUTTON_CANCEL;
      switch (Which) {
      case NAVIGATE_HOTLIST:
         if ((ppsz_buf=ReadHotListFile(NULL)) == NULL) {
            ok = FALSE;
         } else if (*ppsz_buf == NULL) {
            MsgBox("Hot list file empty!", TOOL_NAME, INFO_MB);
            cfree(ppsz_buf);
            ok = FALSE;
         } else {
            dsp_ptr = HotListListing(ppsz_buf, &entries);

            ignoreDirectoryFlag = TRUE;
            nameDspPtr = MakeNameDspItemArray(nameEntries, dsp_ptr);
            ignoreDirectoryFlag = FALSE;
            nameFirst = 0;
            if (something_deleted) {
               nameMarked--;
            } else if (!goto_something) {
               nameMarked = nameEntries-1;
            }
            if (nameMarked < 0) nameMarked = INVALID;
         }
         break;

      case NAVIGATE_HISTORY:
         dsp_ptr = HistoryListing(&entries);
         if (dsp_ptr == NULL) {
            MsgBox("Sesion history has not been established yet!",
                  TOOL_NAME, INFO_MB);
            ok = FALSE;
         } else {
            ignoreDirectoryFlag = TRUE;
            nameDspPtr = MakeNameDspItemArray(nameEntries, dsp_ptr);
            ignoreDirectoryFlag = FALSE;
            nameFirst = 0;
            if (!goto_something) {
               nameMarked = nameEntries-1;
            }
         }
         break;
      }
      if (!ok) break;

      SaveStatusStrings ();

      if (nameMarked == INVALID) {
         name[0] = '\0';
         name_index = 0;
      } else {
         if (nameMarked >= ITEM_DSPED) {
            if (nameMarked < nameEntries-ITEM_DSPED) {
               nameFirst = nameMarked;
            } else {
               nameFirst = nameEntries-ITEM_DSPED;
            }
         }
         strcpy(name, nameDspPtr[nameMarked]);
         name_index = strlen(name);
      }
      if (exposed) {
         RedrawNameScrollWin();
         RedrawDspWindow();
      }
      SetStringStatus(nameMarked==INVALID ? "" : dsp_ptr[nameMarked].pathstr);
      XSync(mainDisplay, False);
      while (changing) {
         KeySym key_sym;
         XEvent input, ev;
         XKeyEvent *key_ev;
         XButtonEvent *button_ev;

         XNextEvent(mainDisplay, &input);

         if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
               (input.type==Expose && (input.xany.window==nameBaseWin ||
               input.xany.window==nameScrollWin ||
               input.xany.window==nameDspWin)) ||
               (!exposed &&
               (XCheckWindowEvent(mainDisplay,nameBaseWin,ExposureMask,&ev) ||
               XCheckWindowEvent(mainDisplay,nameScrollWin,ExposureMask,&ev) ||
               XCheckWindowEvent(mainDisplay,nameDspWin,ExposureMask,&ev) ||
               XCheckWindowEvent(mainDisplay,nameBaseWin,StructureNotifyMask,
               &ev)))) {
            while (XCheckWindowEvent(mainDisplay, nameBaseWin, ExposureMask,
                  &ev))
               ;
            while (XCheckWindowEvent(mainDisplay, nameScrollWin, ExposureMask,
                  &ev))
               ;
            while (XCheckWindowEvent(mainDisplay, nameDspWin, ExposureMask,
                  &ev))
               ;
            while (XCheckWindowEvent(mainDisplay, nameBaseWin,
                  StructureNotifyMask, &ev))
               ;

            RedrawNameBaseWindow(TopStr, name, str_start, graph_start,
                  button_start, w, h);
            RedrawNameScrollWin();
            RedrawDspWindow();

            exposed = TRUE;
            XSync(mainDisplay, False);

            if ((input.type==MapNotify && input.xany.window==nameBaseWin) ||
                  (input.type==Expose && (input.xany.window==nameBaseWin ||
                  input.xany.window==nameScrollWin ||
                  input.xany.window==nameDspWin))) {
               continue;
            }
         }

         if (input.type==Expose) {
            ExposeEventHandler(&input, FALSE);
         } else if (input.type==VisibilityNotify &&
               input.xany.window==mainWindow &&
               input.xvisibility.state==VisibilityUnobscured) {
            while (XCheckWindowEvent(mainDisplay, mainWindow,
                  VisibilityChangeMask, &ev)) ;
            if (pinnedMainMenu) XMapRaised(mainDisplay, mainMenuWindow);
            for (i = 0; i < numExtraWins; i++) {
               if (extraWinInfo[i].mapped && extraWinInfo[i].raise &&
                     extraWinInfo[i].window != None) {
                  XMapRaised(mainDisplay, extraWinInfo[i].window);
               }
            }
            XMapRaised(mainDisplay, nameBaseWin);
         } else if (input.type == KeyPress) {
            key_ev = &(input.xkey);
            XLookupString(key_ev, buf, 80-1, &key_sym, &c_stat);
            TranslateKeys(buf, &key_sym);

            if ((buf[0]=='\r' && (key_sym & 0xff)=='\r') ||
                (buf[0]=='\n' && (key_sym & 0xff)=='\n')) {
               changing = FALSE;
               button_selected = BUTTON_OK;
            } else if (buf[0]=='\033' && (key_sym & 0xff)=='\033') {
               changing = FALSE;
               button_selected = BUTTON_CANCEL;
            } else if ((buf[0] == '\b'  && (key_sym & 0xff)=='\b') ||
                  (buf[0] == '\b' && (key_sym & 0xff)=='h') ||
                  (buf[0] == '\177' && (key_sym & 0x7f)=='\177') ||
                  key_sym==XK_Left || key_sym==XK_KP_Left) {
               switch (Which) {
               case NAVIGATE_HOTLIST:
                  if (nameMarked != INVALID && name_index != 0) {
                     name[--name_index] = '\0';
                     if (nameMarked != INVALID) {
                        strcpy(nameDspPtr[nameMarked], name);
                     }
                     if (exposed) {
                        RedrawNamePath(name, graph_start,
                              3*ROW_HEIGHT+defaultFontAsc+2);
                        RedrawDspWindow();
                     }
                  }
                  break;
               }
            }
            else if (nameEntries != 0 && ((key_sym>'\040' && key_sym<='\177' &&
                  (key_ev->state & ControlMask)) || key_sym==XK_Up ||
                  key_sym==XK_Down || key_sym==XK_KP_Up ||
                  key_sym==XK_KP_Down)) {
               if ((i = ControlChar(key_ev, key_sym)) != BAD) {
                  if (i == INVALID) {
                     name[0] = '\0';
                     name_index = 0;
                     nameFirst = 0;
                     nameMarked = INVALID;
                  } else {
                     strcpy(name, nameDspPtr[i]);
                     name_index = strlen(name);

                     if (i < nameFirst) {
                        nameFirst = i;
                     } else if (i >= nameFirst+ITEM_DSPED) {
                        if (i < nameEntries-ITEM_DSPED)
                           nameFirst = i;
                        else
                           nameFirst = nameEntries-ITEM_DSPED;
                     }
                     nameMarked = i;
                  }
                  if (exposed) {
                     RedrawNamePath(name, graph_start,
                           3*ROW_HEIGHT+defaultFontAsc+2);
                     RedrawNameScrollWin();
                     RedrawDspWindow();
                  }
                  SetStringStatus(nameMarked==INVALID ? "" :
                        dsp_ptr[nameMarked].pathstr);
                  XSync(mainDisplay, False);
               }
            } else if (nameMarked != INVALID &&
                  key_sym>='\040' && key_sym<='\177' && nameEntries != 0) {
               switch (Which) {
               case NAVIGATE_HOTLIST:
                  name[name_index++] = buf[0];
                  name[name_index] = '\0';
                  strcpy(nameDspPtr[nameMarked], name);

                  if (exposed) {
                     RedrawNamePath(name, graph_start,
                           3*ROW_HEIGHT+defaultFontAsc+2);
                     RedrawNameScrollWin();
                     RedrawDspWindow();
                  }
                  break;
               }
            }
         } else if (input.type == ButtonPress) {
            button_ev = &(input.xbutton);
            if (button_ev->window == nameBaseWin) {
               if (PointInBBox(button_ev->x, button_ev->y, buttonBBox[0])) {
                  changing = FALSE;
                  button_selected = BUTTON_OK;
               } else if (PointInBBox(button_ev->x, button_ev->y,
                     buttonBBox[1])) {
                  changing = FALSE;
                  switch (Which) {
                  case NAVIGATE_HOTLIST: button_selected = BUTTON_SETDIR; break;
                  case NAVIGATE_HISTORY: button_selected = BUTTON_CANCEL; break;
                  }
               } else if (Which == NAVIGATE_HOTLIST &&
                     PointInBBox(button_ev->x, button_ev->y, buttonBBox[2])) {
                  changing = FALSE;
                  button_selected = BUTTON_CANCEL;
               }
            } else if (button_ev->window == nameScrollWin) {
               NameScrollHandler(button_ev);
            } else if (button_ev->window == nameDspWin) {
               int saved_name_marked=nameMarked;
               int tmp_rc=NameDspHandler(button_ev);

               if (nameMarked != INVALID) {
                  if (tmp_rc != INVALID) {
                     changing = FALSE;
                     button_selected = BUTTON_OK;
                  } else {
                     strcpy(name, nameDspPtr[nameMarked]);
                     name_index = strlen(name);
                     RedrawNamePath(name, graph_start,
                           3*ROW_HEIGHT+defaultFontAsc+2);
                  }
               }
               if (nameMarked != saved_name_marked) {
                  SetStringStatus(nameMarked==INVALID ? "" :
                        dsp_ptr[nameMarked].pathstr);
                  XSync(mainDisplay, False);
               }
            }
         }
      }
      if (exposed) {
         DisplayButton(nameBaseWin, buttonStr[button_selected], 8,
               &buttonBBox[button_selected], BUTTON_INVERT);
         XSync(mainDisplay, False);
      }
      rc = button_selected;
      RestoreStatusStrings ();

      switch (Which) {
      case NAVIGATE_HOTLIST:
         switch (rc) {
         case BUTTON_OK: goto_something=TRUE; break;
         case BUTTON_SETDIR: something_deleted=TRUE; break;
         default: something_deleted=goto_something=FALSE; break;
         }

         for (s_ptr=ppsz_buf, i=0; *s_ptr != NULL; s_ptr=(&s_ptr[2]), i++) {
            int delete_this=FALSE;

            if (rc != BUTTON_CANCEL && i == nameMarked && something_deleted) {
               sprintf(gszMsgBox, "Delete '%s' from the hot list? [ync](y)",
                     (**s_ptr == '\0' ? s_ptr[1] : *s_ptr));
               if (MsgBox(gszMsgBox, TOOL_NAME, YNC_MB)==MB_ID_YES) {
                  **s_ptr = *s_ptr[1] = '\0';
                  modified = TRUE;
                  delete_this = TRUE;
               }
            }
            if (delete_this) continue;
            if (**s_ptr == '\0') {
               if (strcmp(nameDspPtr[i], s_ptr[1]) != 0) {
                  cfree(*s_ptr);
                  if (((*s_ptr)=UtilStrDup(nameDspPtr[i])) == NULL) {
                     fprintf(stderr, "Can not calloc().\n");
                  }
                  modified = TRUE;
               }
            } else {
               if (strcmp(nameDspPtr[i], *s_ptr) != 0) {
                  cfree(*s_ptr);
                  if (((*s_ptr)=UtilStrDup(nameDspPtr[i])) == NULL) {
                     fprintf(stderr, "Can not calloc().\n");
                  }
                  modified = TRUE;
               }
            }
         }
         if (modified || something_deleted) {
            FILE *fp;

            if ((fp=fopen(hotListFileName, "w")) == NULL) {
               sprintf(gszMsgBox, "Can not open '%s' for write.",
                     hotListFileName);
               MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
            } else {
               for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr=(&s_ptr[2])) {
                  if (!(**s_ptr == '\0' && *s_ptr[1] == '\0')) {
                     fprintf(fp, "%s\n%s\n", *s_ptr, s_ptr[1]);
                  }
               }
               fclose(fp);
            }
         }
         cfree(dsp_ptr);
         cfree(*nameDspPtr);
         cfree(nameDspPtr);
         nameDspPtr = NULL;
         if (goto_something) {
            int do_not_save=FALSE, canceled=FALSE;

            while (!canceled && !DirIsRemote(curDir) && fileModified)
            {
               sprintf(gszMsgBox, "%s, %s? [ync](y)",
                     "File modified", "save file before opening another file");
               switch (MsgBox(gszMsgBox, TOOL_NAME, YNC_MB)) {
               case MB_ID_YES: SaveFile(); break;
               case MB_ID_NO: do_not_save=TRUE; SetFileModified(FALSE); break;
               case MB_ID_CANCEL: canceled=TRUE; break;
               }
            }
            if (!canceled) {
               for (i=0, s_ptr=ppsz_buf; *s_ptr != NULL;
                     s_ptr=(&s_ptr[2]), i++) {
                  if (i == nameMarked) {
                     NavigateTo(s_ptr[1], do_not_save, FALSE);
                     CommitNavigate();
                     navigated_to = TRUE;
                     break;
                  }
               }
            }
         }
         for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr++) {
            cfree(*s_ptr);
         }
         cfree(ppsz_buf);
         break;

      case NAVIGATE_HISTORY:
         goto_something = (rc==BUTTON_OK);

         cfree(dsp_ptr);
         cfree(*nameDspPtr);
         cfree(nameDspPtr);
         nameDspPtr = NULL;

         if (goto_something) {
            int do_not_save=FALSE, i, canceled=FALSE;
            struct NavigateRec *pnr;

            while (!canceled && !DirIsRemote(curDir) && fileModified) {
               sprintf(gszMsgBox, "%s, %s? [ync](y)",
                     "File modified", "save file before opening another file");
               switch (MsgBox(gszMsgBox, TOOL_NAME, YNC_MB)) {
               case MB_ID_YES: SaveFile(); break;
               case MB_ID_NO: do_not_save=TRUE; SetFileModified(FALSE); break;
               case MB_ID_CANCEL: canceled=TRUE; break;
               }
            }
            if (!canceled) {
               pnr = firstSessionHistory;
               for (i=0; pnr != NULL; pnr=pnr->next, i++) {
                  if (i == nameMarked) {
                     NavigateTo(pnr->full_fname, do_not_save, FALSE);
                     CommitNavigate();
                     navigated_to = TRUE;
                     break;
                  }
               }
            }
         }
         break;
      }
      if (exposed) {
         DisplayButton(nameBaseWin, buttonStr[button_selected], 8,
               &buttonBBox[button_selected], BUTTON_INVERT);
         XSync(mainDisplay, False);
      }
      if (navigated_to) {
         PostNavigate(NULL);
      }
   } while ((Which == NAVIGATE_HOTLIST &&
         (goto_something || something_deleted)) ||
         (Which == NAVIGATE_HISTORY && goto_something));

   XDestroyWindow(mainDisplay, nameBaseWin);
   if (warpToWinCenter) {
      XWarpPointer(mainDisplay, None, drawWindow, 0, 0, 0, 0,
            (int)(ZOOMED_SIZE(drawWinW)>>1), (int)(ZOOMED_SIZE(drawWinH)>>1));
   }
   return button_selected;
}

void NavigateHotList()
{
   MakeQuiescent ();
   if (!InitHotListFileName() || hotListFileName==NULL) {
      sprintf(gszMsgBox, "%s.  %s %s*%s.",
            "Hot list file undefined",
            "Please specify it in", TOOL_NAME, "HotListFileName");
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      return;
   }
   SelectForNavigate("Hot list...", NAVIGATE_HOTLIST);
}

void NavigateAddToHotList()
{
   int len=strlen(curDir)+1+strlen(curFileName), num_entries=0;
   char **s_ptr, **ppsz_buf, *full_fname;
   FILE *fp;

   if (!curFileDefined) {
      MsgBox("Can not add an undefined file to hot list.", TOOL_NAME, INFO_MB);
      return;
   }
   if ((full_fname=(char*)calloc(len+1, sizeof(char))) == NULL) {
      fprintf(stderr, "Can not calloc().\n");
      return;
   }
   sprintf(full_fname, "%s/%s", curDir, curFileName);
   if ((ppsz_buf=ReadHotListFile(&num_entries)) == NULL) return;
   if ((fp=fopen(hotListFileName, "w")) == NULL) {
      sprintf(gszMsgBox, "Can not open '%s' for write.", hotListFileName);
      MsgBox(gszMsgBox, TOOL_NAME, INFO_MB);
      cfree(full_fname);
      return;
   }
   for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr=(&s_ptr[2])) {
      if (strcmp(s_ptr[1], full_fname) == 0) {
         char **s_ptr1;

         if (s_ptr[2] != NULL) {
            for (s_ptr1=ppsz_buf; s_ptr1 != s_ptr; s_ptr1++) {
               fprintf(fp, "%s\n", *s_ptr1);
            }
            for (s_ptr1=(&s_ptr[2]); *s_ptr1 != NULL; s_ptr1++) {
               fprintf(fp, "%s\n", *s_ptr1);
            }
         } else {
            for (s_ptr1=ppsz_buf; s_ptr1 != s_ptr; s_ptr1++) {
               fprintf(fp, "%s\n", *s_ptr1);
            }
         }
         fprintf(fp, "%s\n", (firstPage->name==NULL ? "" : firstPage->name));
         fprintf(fp, "%s\n", full_fname);
         break;
      }
   }
   if (*s_ptr == NULL) {
      for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr++) {
         fprintf(fp, "%s\n", *s_ptr);
      }
      fprintf(fp, "%s\n", (firstPage->name==NULL ? "" : firstPage->name));
      fprintf(fp, "%s\n", full_fname);
      num_entries++;
   }
   for (s_ptr=ppsz_buf; *s_ptr != NULL; s_ptr++) {
      cfree(*s_ptr);
   }
   cfree(ppsz_buf);
   fclose(fp);
   cfree(full_fname);
   sprintf(gszMsgBox, "Hot list file updated (%1d entrie%s).", num_entries,
         (num_entries>1 ? "s" : ""));
   Msg(gszMsgBox);
}

void NavigateSessionHistory()
{
   MakeQuiescent ();
   SelectForNavigate("Session history...", NAVIGATE_HISTORY);
}

void ToggleHyperSpace(KeepSelected)
   int KeepSelected;
{
   inHyperSpace = !inHyperSpace;
   if (inHyperSpace) {
      if (!KeepSelected) MakeQuiescent();
      Msg("Entering hyperspace...");
   } else {
      ShowCursor();
      Msg("Leaving hyperspace...");
   }
   UpdateSubMenu(MENU_NAVIGATE);
   RedrawDummyWindow1();
}

void NavigateSubMenu(index)
   int index;
{
   switch (index) {
   case NAVIGATE_BACK: NavigateBack(); break;
   case NAVIGATE_FORWARD: NavigateForward(); break;
   case NAVIGATE_REFRESH: NavigateRefresh(); break;
   case NAVIGATE_HOTLIST: NavigateHotList(); break;
   case NAVIGATE_ADD: NavigateAddToHotList(); break;
   case NAVIGATE_HISTORY: NavigateSessionHistory(); break;
   case NAVIGATE_HYPERSPACE: ToggleHyperSpace(FALSE); break;
   }
}

int NavigateMenu(X, Y, TrackMenubar)
   int X, Y, TrackMenubar;
{
   int index;
   int *fore_colors, *valid, *init_rv;

   DefaultColorArrays(MAXNAVIGATEMENUS, &fore_colors, &valid, &init_rv, NULL);
   activeMenu = MENU_NAVIGATE;
   if (inHyperSpace) {
      index = TextMenuLoop(X, Y, navigateMenuStrInHyperSpace, MAXNAVIGATEMENUS,
            fore_colors, valid, init_rv, navigateMenuDescriptionInHyperSpace,
            SINGLECOLOR, TrackMenubar);
   } else {
      index = TextMenuLoop(X, Y, navigateMenuStr, MAXNAVIGATEMENUS, fore_colors,
            valid, init_rv, navigateMenuDescription, SINGLECOLOR, TrackMenubar);
   }

   if (index >= 0) NavigateSubMenu(index);
   return index;
}
