/*
 * options.c - Option parsing code.
 *
 * Copyright (c) 1994, 1995 Eric Schenk.
 * All rights reserved. Please see the file LICENSE which should be
 * distributed with this software for terms of use.
 */

#include "diald.h"

#define MAXLINELEN 1024
#define MAXARGS 10


/* Configuration variables */
char **devices = 0;
int device_count = 0;
int inspeed = DEFAULT_SPEED;
int mtu = DEFAULT_MTU;
char *connector = 0;
char *disconnector = 0;
char *local_ip = 0;
unsigned long local_addr = 0;
char *orig_local_ip = 0;
char *netmask = 0;
char *remote_ip = 0;
char *orig_remote_ip = 0;
char *addroute = 0;
char *delroute = 0;
char *ip_up = 0;
char *ip_down = 0;
char *acctlog = 0;
char *pidlog = "diald.pid";
char *fifoname = 0;
FILE *acctfp = 0;
int mode = MODE_SLIP;
int debug = 0;
int modem = 0;
int crtscts = 0;
int daemon = 1;
int slip_encap = 0;
int lock_dev = 0;
int default_route = 0;
int dynamic_addrs = 0;
int dynamic_mode = DMODE_REMOTE_LOCAL;
int rotate_devices = 0;

int connect_timeout = 60;
int disconnect_timeout = 60;
int redial_timeout = DEFAULT_DIAL_DELAY;
int stop_dial_timeout = 60;
int kill_timeout = 60;
int start_pppd_timeout = 60;
int stop_pppd_timeout = 60;
int first_packet_timeout = DEFAULT_FIRST_PACKET_TIMEOUT;
int retry_count = 0;
int died_retry_count = 1;


struct {
    char *str;
    char *uargs;
    char args;
    void *var;
    void (*parser)();
} commands[] = {
    {"-f","<file>",1,0,read_config_file},
    {"-file","<file>",1,0,read_config_file},
/* Mode options */
    {"-m","[ppp|slip|cslip|slip6|cslip6|aslip]",1,0,set_mode},
    {"mode","[ppp|slip|cslip|slip6|cslip6|aslip]",1,0,set_mode},
/* general options */
    {"accounting-log","<f>",1,&acctlog,set_str},
    {"pidfile","<f>",1,&pidlog,set_str},
    {"fifo","<f>",1,&fifoname,set_str},
/* Networking addressing and control options */
    {"mtu","<m>",1,&mtu,set_int},
    {"local","<ip-address>",1,&local_ip,set_str},
    {"remote","<ip-address>",1,&remote_ip,set_str},
    {"netmask","<ip-address>",1,&netmask,set_str},
    {"dynamic","",0,&dynamic_addrs,set_flag},
    {"dslip-mode","<mode>",1,0,set_dslip_mode},
    {"defaultroute","",0,&default_route,set_flag},
    {"addroute","<script>",1,&addroute,set_str},
    {"delroute","<script>",1,&delroute,set_str},
    {"ip-up","<script>",1,&ip_up,set_str},
    {"ip-down","<script>",1,&ip_down,set_str},
/* Modem options */
    {"connect","<script>",1,&connector,set_str},
    {"disconnect","<script>",1,&disconnector,set_str},
    {"lock","",0,&lock_dev,set_flag},
    {"speed","<baudrate>",1,&inspeed,set_int},
    {"modem","",0,&modem,set_flag},
    {"crtscts","",0,&crtscts,set_flag},
    {"rotate-devices","",0,&rotate_devices,set_flag},
/* Debugging options */
    {"debug","<debugmask>",1,&debug,set_int},
    {"-daemon","",0,&daemon,clear_flag},
/* Timeout control options */
    {"connect-timeout","<timeout>",1,&connect_timeout,set_int},
    {"disconnect-timeout","<timeout>",1,&disconnect_timeout,set_int},
    {"redial-timeout","<timeout>",1,&redial_timeout,set_int},
    {"stop-dial-timeout","<timeout>",1,&stop_dial_timeout,set_int},
    {"kill-timeout","<timeout>",1,&kill_timeout,set_int},
    {"start-pppd-timeout","<timeout>",1,&start_pppd_timeout,set_int},
    {"stop-pppd-timeout","<timeout>",1,&stop_pppd_timeout,set_int},
    {"first-packet-timeout","<timeout>",1,&first_packet_timeout,set_int},
    {"retry-count","<count>",1,&retry_count,set_int},
    {"died-retry-count","<count>",1,&died_retry_count,set_int},
    {"prule","<name> <protocol> <spec>",3,0,&parse_prule},
    {"var","<name> <spec>",2,0,&parse_var},
    {"restrict","<start-time> <end-time> <weekday> <day> <month>",5,0,&parse_restrict},
    {"or-restrict","<start-time> <end-time> <weekday> <day> <month>",5,0,&parse_or_restrict},
    {"bringup","<protocol-rule> <timeout> <packet-rule>",3,0,&parse_bringup},
    {"keepup","<protocol-rule> <timeout> <packet-rule>",3,0,&parse_keepup},
    {"accept","<protocol-rule> <timeout> <packet-rule>",3,0,&parse_accept},
    {"ignore","<protocol-rule> <packet-rule>",2,0,&parse_ignore},
    {"impulse","[<duration>,<fuzz>|<duration1>,<duration2>,<fuzz>]",1,0,&parse_impulse},
    {"up","",0,0,&parse_up},
    {"down","",0,0,&parse_down},
    {"flushfilters","",0,0,&flush_filters},
    {"flushprules","",0,0,&flush_prules},
    {"flushvars","",0,0,&flush_vars},
    {0,0,0,0}
};

void set_int(int *var, char **argv)
{
    *var = atoi(*argv);
}

void set_str(char **var, char **argv)
{
    *var = strdup(*argv);
}

void set_mode(char **var, char **argv)
{
    if (strcmp(argv[0],"ppp") == 0)
	mode = MODE_PPP;
    else if (strcmp(argv[0],"slip") == 0)
	mode = MODE_SLIP, slip_encap = 0;
    else if (strcmp(argv[0],"cslip") == 0)
	mode = MODE_SLIP, slip_encap = 1;
    else if (strcmp(argv[0],"slip6") == 0)
	mode = MODE_SLIP, slip_encap = 2;
    else if (strcmp(argv[0],"cslip6") == 0)
	mode = MODE_SLIP, slip_encap = 3;
    else if (strcmp(argv[0],"aslip") == 0)
	mode = MODE_SLIP, slip_encap = 8;
    else
	syslog(LOG_ERR,"Unknown mode %s.\nValid modes are: ppp, slip, cslip, slip6, cslip6, or aslip.",argv[0]);
}

void set_dslip_mode(char **var, char **argv)
{
    if (strcmp(argv[0],"remote") == 0)
	dynamic_mode = DMODE_REMOTE;
    else if (strcmp(argv[0],"local") == 0)
	dynamic_mode = DMODE_LOCAL;
    else if (strcmp(argv[0],"remote-local") == 0)
	dynamic_mode = DMODE_REMOTE_LOCAL;
    else if (strcmp(argv[0],"local-remote") == 0)
	dynamic_mode = DMODE_LOCAL_REMOTE;
    else if (strcmp(argv[0],"bootp") == 0)
	dynamic_mode = DMODE_BOOTP;
    else
	syslog(LOG_ERR,"Unknown dynamic slip mode %s.\nValid modes are: remote, local, remote-local, local-remote or bootp.",argv[0]);
}

void set_flag(int *var, char **argv)
{
    *var = 1;
}

void clear_flag(int *var, char **argv)
{
    *var = 0;
}

void read_config_file(int *var, char **argv)
{
    parse_options_file(argv[0]);
}

void usage(void)
{
    int i;
    syslog(LOG_ERR,"usage: diald modem-device1 [modem-device2 ...] [options...] [-- [pppd options...]]\n");
    syslog(LOG_ERR,"where valid options are:");
    for (i = 0; commands[i].str; i++)
        syslog(LOG_ERR,"    %s %s",commands[i].str,commands[i].uargs);
    exit(1);
}

void parse_args(int argc, char *argv[])
{
    int i;
    struct stat st;

    /* The first thing MUST be a device specification */
    if (argc == 0) {
	syslog(LOG_ERR,
	    "No device specified. You must have at least one device!");
	usage();
    }

    /* Get the first device */
    devices = argv++;
    argc--;
    if (stat(devices[0],&st) < 0 || !S_ISCHR(st.st_mode)) {
	syslog(LOG_ERR,
	    "Specified device '%s' not a character device.",devices[0]);
	usage();
    }

    /* Get any extra devices on the list */
    for (i = 1; argc > 0 && stat(*argv,&st) == 0 && S_ISCHR(st.st_mode); i++)
	argc--, argv++;
    device_count = i;

    /*
     * Ok, parse the options list now,
     * command line options override/augment defaults
     */
    while (argc-- > 0) {
	if (strcmp(*argv,"--") == 0) {
	    argv++;
	    argc--;
	    break;
	}
	for (i = 0; commands[i].str; i++)
	    if (strcmp(commands[i].str,*argv) == 0)
		break;
	if (commands[i].parser) {
	    argc -= commands[i].args;
	    if (argc < 0) {
		syslog(LOG_ERR,"Insufficient arguments to option '%s'",*argv);
		usage();
	    }
	    (commands[i].parser)(commands[i].var,++argv);
	    argv += commands[i].args;
	} else {
	    syslog(LOG_ERR,"Unknown option '%s'",*argv);
	    usage();
	}
    }
    /* Any remaining options are food for pppd */
    pppd_argv = argv;
    pppd_argc = argc+1;
}

/* Parse the given options file */
void parse_options_file(char *file)
{
    char line[MAXLINELEN];
    char *argv[MAXARGS];
    char *s;
    char *t1,*t2;

    int argc, i;
    FILE *fp = fopen(file,"r");

    if (fp == NULL) return;	/* no options file */
    while (fscanf(fp,"%1024[^\n]\n",line) != -1) {
	argc = 0;
	for (s = line, argc = 0; *s, argc < MAXARGS; s++) {
	    if (*s == ' ' || *s == '\t')
		*s = 0;
	    else if (*s == '#' && argc == 0) /* the line is a comment */
		goto comment;
	    else if (*s == '\"') {            /* start of a quoted argument */
		s++;
		argv[argc] = s;
		while (*s) {
		    if (*s == '\"') { *s++ = 0; break; }
		    if (*s == '\\' && s[1] == '\"') s++;
		    s++;
		}
		s--;
		/* go back and fix up "quoted" characters */
		t1 = t2 = argv[argc++];
		while (*t1) {
		   if (*t1 == '\\') t1++;
		   *t2++ = *t1++;
		}
	    } else { /* just a normal word */
		argv[argc++] = s;
		while (*s) {
		    if (*s == ' ' || *s == '\t') break;
		    s++;
		}
		s--;
	    }
	}
	*s = 0;

	for (i = 0; commands[i].str; i++)
	    if (strcmp(commands[i].str,argv[0]) == 0)
		break;
	if (commands[i].parser) {
	    argc -= commands[i].args;
	    if (argc < 0) {
		syslog(LOG_ERR,"Insufficient arguments to option '%s'",argv[0]);
	    } else {
	        (commands[i].parser)(commands[i].var,&argv[1]);
	    }
	} else {
	    syslog(LOG_ERR,"Unknown option '%s'",*argv);
	}
comment:;
    }
    fclose(fp);
}

void check_setup()
{
    int flag = 0;

    if (!connector)
	flag = 1,
	syslog(LOG_ERR,
	    "You must define a connector script (option 'connect').");
    if (!remote_ip)
	flag = 1, syslog(LOG_ERR,"You must define the remote ip address.");
    else if (inet_addr(remote_ip) == -1)
	flag = 1, syslog(LOG_ERR,"Bad remote ip address specification.");
    if (!local_ip)
	flag = 1, syslog(LOG_ERR,"You must define the local ip address.");
    else if (inet_addr(local_ip) == -1)
	flag = 1, syslog(LOG_ERR,"Bad local ip address specification.");
    else
    	local_addr = inet_addr(local_ip);

    if (acctlog && (acctfp = fopen(acctlog,"a")) == NULL)
	syslog(LOG_ERR,"Can't open accounting log file %s: %m",acctlog);
    else
	fclose(acctfp);
    
    if (flag) exit(1);
}
