/*
 * slip.c - Proxy interface specific code in diald.
 *	    The proxy interface is used to monitor packets
 *	    when the physical link is down.
 *
 * Copyright (c) 1994, 1995 Eric Schenk.
 * All rights reserved. Please see the file LICENSE which should be
 * distributed with this software for terms of use.
 */

#include "diald.h"

/*
 * SLIP PACKET READING CODE FROM RFC 1055 by J. Romkey.
 *
 * RECV_PACKET: receives a packet into the buffer located at "p".
 *      If more than len bytes are received, the packet will
 *      be truncated.
 *      Returns the number of bytes stored in the buffer.
 */

/* SLIP special character codes */
#define END             0300    /* indicates end of packet */
#define ESC             0333    /* indicates byte stuffing */
#define ESC_END         0334    /* ESC ESC_END means END data byte */
#define ESC_ESC         0335    /* ESC ESC_ESC means ESC data byte */

int recv_packet(unsigned char *p, int len)
{
    int c;
    int received = 0;


    /* sit in a loop reading bytes until we put together
    * a whole packet.
    * Make sure not to copy them into the packet if we
    * run out of room.
    */
    while(1) {
       /* get a character to process
	*/
       c = getc(proxy_mfp);

       /* handle bytestuffing if necessary
	*/
       switch(c) {

       /* if it's an END character then we're done with
	* the packet
	*/
       case END:
	       /* a minor optimization: if there is no
		* data in the packet, ignore it. This is
		* meant to avoid bothering IP with all
		* the empty packets generated by the
		* duplicate END characters which are in
		* turn sent to try to detect line noise.
		*/
	       if(received)
		       return received;
	       else
		       break;

       /* if it's the same code as an ESC character, wait
	* and get another character and then figure out
	* what to store in the packet based on that.
	*/
       case ESC:
	       c = getc(proxy_mfp);

	       /* if "c" is not one of these two, then we
		* have a protocol violation.  The best bet
		* seems to be to leave the byte alone and
		* just stuff it into the packet
		*/
	       switch(c) {
	       case ESC_END:
		       c = END;
		       break;
	       case ESC_ESC:
		       c = ESC;
		       break;
		       }

       /* here we fall into the default handler and let
	* it store the character for us
	*/
       default:
	       if(received < len)
		       p[received++] = c;
	       }
     }
}

/*
 * Set the pty to an 8 bit clean mode and change it to the
 * desired SLIP line disciple, and run ifconfig to configure the
 * device as up.
 * The SLIP configuration is essentially what slattach
 * does, but we do it here so we know what interface (sl*)
 * gets opened as a result. (slattach doesn't return this)
 */

void proxy_up(void)
{
    int disc, sencap = 0;

    /* change proxy_sfd to 8 bit clean line, 38400 speed */
    set_up_tty(proxy_sfd,1, 38400);

    if (ioctl(proxy_sfd, TIOCGETD, &orig_disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline on proxy device: %m"), die(1);

    /* change line disciple to SLIP and set the SLIP encapsulation */
    disc = N_SLIP;
    if ((proxy_iface = ioctl(proxy_sfd, TIOCSETD, &disc)) < 0) {
	if (errno == ENFILE) {
	   syslog(LOG_ERR,"No free slip device available for proxy."), die(1);
	} else if (errno == EEXIST) {
	    syslog(LOG_ERR,"Proxy device already in slip mode!?");
	} else if (errno == EINVAL) {
	    syslog(LOG_ERR,"SLIP not supported by kernel, can't build proxy.");
	    die(1);
	} else
	   syslog(LOG_ERR,"Can't set line discipline: %m"), die(1);
    }

    if (ioctl(proxy_sfd, SIOCSIFENCAP, &sencap) < 0)
	syslog(LOG_ERR,"Can't set encapsulation: %m"), die(1);

    /* verify that it worked */
    if (ioctl(proxy_sfd, TIOCGETD, &disc) < 0)
	syslog(LOG_ERR,"Can't get line discipline: %m"), die(1);
    if (ioctl(proxy_sfd, SIOCGIFENCAP, &sencap) < 0)
	syslog(LOG_ERR,"Can't get encapsulation: %m"), die(1);

    if (disc != N_SLIP || sencap != 0)
        syslog(LOG_ERR,"Couldn't set up the proxy link correctly!"), die(1);

    if (debug&DEBUG_VERBOSE)
        syslog(LOG_INFO,"Proxy device established on interface sl%d",
	    proxy_iface);

    /* Mark the interface as up */
    proxy_config(orig_local_ip,orig_remote_ip);
    /* set the routing to the interface */
    set_ptp("sl",proxy_iface,orig_remote_ip);
    add_routes("sl",proxy_iface,orig_local_ip,orig_remote_ip);
}

/*
 * Configure the proxy SLIP lines address.
 * This may change in a dynamic setting.
 */
void proxy_config(char *lip, char *rip)
{
    char buf[128];
    /* mark the interface as up */
    if (netmask) {
        sprintf(buf,"%s sl%d %s pointopoint %s netmask %s mtu %d up",
	    PATH_IFCONFIG,proxy_iface,lip,rip,netmask,mtu);
    } else {
        sprintf(buf,"%s sl%d %s pointopoint %s mtu %d up",
	    PATH_IFCONFIG,proxy_iface,lip,rip,mtu);
    }
    system(buf);
}

/*
 * Call the delroute script and take the pty out of slip mode.
 */
void proxy_down()
{
    if (debug&DEBUG_VERBOSE)
	syslog(LOG_INFO,"taking proxy device down");
    del_routes("sl",proxy_iface,orig_local_ip,orig_remote_ip);
    /* clear the line discipline */
    if ((proxy_iface = ioctl(proxy_sfd, TIOCSETD, &orig_disc)) < 0)
	syslog(LOG_ERR,"Can't set line discipline: %m"), die(1);
}

void run_ip_up()
{
    if (ip_up) {
    	if (debug&DEBUG_VERBOSE)
	    syslog(LOG_INFO,"running ip-up script '%s'",ip_up);
        if (system(ip_up) < 0) {
	    syslog(LOG_ERR,"ip_up script '%s' failed",ip_up);
        }
    }
}

void run_ip_down()
{
    if (ip_down) {
    	if (debug&DEBUG_VERBOSE)
	    syslog(LOG_INFO,"running ip-down script '%s'",ip_down);
        if (system(ip_down) < 0) {
	    syslog(LOG_ERR,"ip_down script '%s' failed",ip_down);
        }
    }
}
