/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

/* WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!
 *
 * Legal for distribution in the U.S./Canada ONLY!  Exporting this file
 * outside of the U.S./Canada may be in violation of ITAR regulations!
 */

/*
 * This file contains all of the PGP routines necessary to sign, encrypt,
 * verify and decrypt PGP messages in either the new PGP/MIME format, or
 * in the older Application/Pgp format.  It also contains some code to
 * cache the user's passphrase for repeat use when decrypting or signing
 * a message.
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "send.h"
#include "pgp.h"

#include <sys/wait.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <ctype.h>

#ifdef _PGPPATH

char PgpPass[STRING];
static time_t PgpExptime = 0; /* when does the cached passphrase expire? */

void pgp_void_passphrase (void)
{
  memset (PgpPass, 0, sizeof (PgpPass));
  PgpExptime = 0;
}

int pgp_valid_passphrase (void)
{
  time_t now = time (NULL);

  if (now < PgpExptime) return 1; /* just use the cached copy. */
  pgp_void_passphrase ();

  if (mutt_get_password ("Enter PGP passphrase:", PgpPass, sizeof (PgpPass)) == 0)
  {
    PgpExptime = time (NULL) + PgpTimeout;
    return (1);
  }
  else
  {
    PgpExptime = 0;
    return (0);
  }
  /* not reached */
}

/* ----------------------------------------------------------------------------
 * Routines for handing PGP input.
 */

/* print the current time to avoid spoofing of the signature output */
static void pgp_current_time (STATE *s)
{
  time_t t;
  char p[STRING];

  state_puts ("[-- PGP output follows (current time: ", s);

  t = time (NULL);
  strfcpy (p, asctime (localtime (&t)), sizeof (p));
  p[strlen (p) - 1] = 0; /* kill the newline */
  state_puts (p, s);

  state_puts (") --]\n", s);
}

/* Support for the Application/PGP Content Type. */
void application_pgp_handler (BODY *m, STATE *s)
{
  int needpass = -1, pgp_keyblock = 0;
  long bytes, last_pos, offset;
  char buf[HUGE_STRING];
  char cmd[_POSIX_PATH_MAX+SHORT_STRING];
  char outfile[_POSIX_PATH_MAX];
  FILE *pgpout, *pgpin, *pgperr;
  pid_t thepid;
  

  fseek (s->fpin, m->offset, 0);
  last_pos = m->offset;
  for (bytes = m->length; bytes > 0;)
  {
    if (fgets (buf, sizeof (buf) - 1, s->fpin) == NULL)
      break;
    offset = ftell (s->fpin);
    bytes -= (offset - last_pos); /* don't rely on strlen(buf) */
    last_pos = offset;

    if (strncmp ("-----BEGIN PGP ", buf, 15) == 0)
    {
      if (strcmp ("MESSAGE-----\n", buf + 15) == 0)
        needpass = 1;
      else if (strcmp ("SIGNED MESSAGE-----\n", buf + 15) == 0)
        needpass = 0;
      else if (strcmp ("PUBLIC KEY BLOCK-----\n", buf + 15) == 0) 
      {
        needpass = 0;
        pgp_keyblock =1;
      } 
      else
      {
	if (s->prefix)
	  state_puts (s->prefix, s);
	state_puts (buf, s);
	continue;
      }

      mutt_mktemp (outfile);
      if ((pgpout = safe_fopen (outfile, "w+")) == NULL)
      {
	mutt_perror (outfile);
	return;
      }

      snprintf (cmd, sizeof (cmd), "%s%s +verbose=0 +batchmode -f",
		needpass ? "PGPPASSFD=0; export PGPPASSFD; " : "", Pgp);

      if ((thepid = mutt_create_filter_fd (cmd, &pgpin, NULL, &pgperr, -1, 
					   fileno (pgpout), -1)) == -1)
      {
	fclose (pgpout);
	unlink (outfile);
	state_puts ("[-- Error: unable to create PGP subprocess --]\n", s);
	state_puts (buf, s);
	continue;
      }

      if (needpass)
      {
	if (!pgp_valid_passphrase ())
	  pgp_void_passphrase ();
	fputs (PgpPass, pgpin);
	fputc ('\n', pgpin);
      }

      fputs (buf, pgpin);
      while (bytes > 0 && fgets (buf, sizeof (buf) - 1, s->fpin) != NULL)
      {
	offset = ftell (s->fpin);
	bytes -= (offset - last_pos); /* don't rely on strlen(buf) */
	last_pos = offset;

	fputs (buf, pgpin);
	if ((needpass && strcmp ("-----END PGP MESSAGE-----\n", buf) == 0) ||
	    (!needpass 
             && (strcmp ("-----END PGP SIGNATURE-----\n", buf) == 0
                 || strcmp ("-----END PGP PUBLIC KEY BLOCK-----\n",buf) == 0)))
	  break;
      }
      fclose (pgpin);

      if (s->displaying)
	pgp_current_time (s);

      mutt_wait_filter (thepid);
      
      if (s->displaying)
      {
	while (fgets (buf, sizeof (buf) - 1, pgperr) != NULL)
	  state_puts (buf, s);
      }
      fclose (pgperr);

      if (s->displaying)
      {
	state_puts ("[-- End of PGP output --]\n\n", s);

	if (needpass)
	  state_puts ("[-- BEGIN PGP MESSAGE --]\n\n", s);
	else if (pgp_keyblock)
	  state_puts ("[-- BEGIN PGP PUBLIC KEY BLOCK --]\n", s);
	else
	  state_puts ("[-- BEGIN PGP SIGNED MESSAGE --]\n\n", s);
      }

      fflush (pgpout);
      rewind (pgpout);
      while (fgets (buf, sizeof (buf) - 1, pgpout) != NULL)
      {
	if (s->prefix)
	  state_puts (s->prefix, s);
	state_puts (buf, s);
      }

      fclose (pgpout);
      unlink (outfile);

      if (s->displaying)
      {
	if (needpass)
	  state_puts ("\n[-- END PGP MESSAGE --]\n", s);
	else if (pgp_keyblock)
	  state_puts ("[-- END PGP PUBLIC KEY BLOCK --]\n", s);
	else
	  state_puts ("\n[-- END PGP SIGNED MESSAGE --]\n", s);
      }
    }
    else
    {
      if (s->prefix)
	state_puts (s->prefix, s);
      state_puts (buf, s);
    }
  }

  if (needpass == -1)
  {
    state_puts ("[-- Error!  Could not find beginning of PGP message! --]\n\n", s);
    return;
  }

}

int pgp_query (BODY *m)
{
  char *p;
  int t = 0;

  /* Check for old-style APPLICATION/PGP messages */
  if (m->type == TYPEAPPLICATION)
  {
    if (!strcasecmp (m->subtype, "pgp") || !strcasecmp (m->subtype, "x-pgp-message"))
    {
      if ((p = mutt_get_parameter ("x-action", m->parameter))
	  && (!strcasecmp (p, "sign") || !strcasecmp (p, "signclear")))
	t |= PGPSIGN;
      else if((p = mutt_get_parameter ("format", m->parameter))
	      && !strcasecmp(p, "keys-only"))
	t |= PGPKEY;

      if ((p = mutt_get_parameter ("format", m->parameter)) && 
	  !strcasecmp (p, "keys-only"))
	t |= PGPKEY;

      if(!t) t |= PGPENCRYPT;  /* not necessarily correct, but... */
    }

    if (!strcasecmp (m->subtype, "pgp-signed"))
      t |= PGPSIGN;

    if (!strcasecmp (m->subtype, "pgp-keys"))
      t |= PGPKEY;
  }

  /* Check for PGP/MIME messages. */
  if (m->type == TYPEMULTIPART)
  {
    if (strcasecmp (m->subtype, "signed") == 0 &&
	(p = mutt_get_parameter("protocol", m->parameter)) &&
	strcasecmp (p, "application/pgp-signature") == 0)
      t |= PGPSIGN;
    else if ((strcasecmp (m->subtype, "encrypted") == 0) && 
	     (p = mutt_get_parameter ("protocol", m->parameter)) &&
	     strcasecmp (p, "application/pgp-encrypted") == 0)
      t |= PGPENCRYPT;
  }

  if(m->type == TYPEMULTIPART || m->type == TYPEMESSAGE)
  {
    BODY *p;
 
    for(p = m->parts; p; p = p->next)
      t |= pgp_query(p);
  }
  
  return t;
}

/*
 * This routine verfies a PGP/MIME signed body.
 */
void pgp_signed_handler (BODY *a, STATE *s)
{
  FILE *fp, *pgpout;
  char tempfile[_POSIX_PATH_MAX], sigfile[STRING], buffer[LONG_STRING];
  int bytes, len;
  pid_t thepid;

  a = a->parts;

  /* First do some error checking to make sure that this looks like a valid
   * multipart/signed body.
   */
  if (a && a->next && a->next->type == TYPEAPPLICATION && a->next->subtype &&
      strcasecmp (a->next->subtype, "pgp-signature") == 0)
  {
    if (s->displaying)
    {
      mutt_mktemp (tempfile);
      fp = safe_fopen (tempfile, "w");

      fseek (s->fpin, a->hdr_offset, 0);
      bytes = a->length + a->offset - a->hdr_offset;
      while (bytes > 0)
      {
	fgets (buffer, sizeof(buffer)-1, s->fpin);
	len = strlen (buffer);
	/* Convert LF=>CRLF when required (probably always) */
	if (buffer[len-2] != '\r' && buffer[len-1] == '\n')
	{
	  buffer[len-1] = '\r';
	  buffer[len] = '\n';
	  buffer[len+1] = 0;
	}
	fputs (buffer, fp);
	bytes -= len;
      }
      fclose (fp);

      /* Now grab the signature.  Since signature data is required to be 7bit,
       * we don't have to worry about doing CTE decoding...
       */
      snprintf (sigfile, sizeof (sigfile), "%s.asc", tempfile);
      fp = safe_fopen (sigfile, "w");
      fseek (s->fpin, a->next->offset, 0);
      mutt_copy_bytes (s->fpin, fp, a->next->length);
      fclose (fp);

      snprintf (buffer, sizeof (buffer), "%s +batchmode +verbose=0 %s %s", Pgp, sigfile, tempfile);

      pgp_current_time (s);

      thepid = mutt_create_filter (buffer, NULL, &pgpout, NULL);
      while (fgets (buffer, sizeof (buffer) - 1, pgpout) != NULL)
	state_puts (buffer, s);
      fclose (pgpout);
      mutt_wait_filter (thepid);
      state_puts ("[-- End of PGP output --]\n\n", s);

      mutt_unlink (tempfile);
      mutt_unlink (sigfile);

      /* Now display the signed body */
      state_puts ("[-- The following data is PGP/MIME signed --]\n\n", s);
    }

    mutt_body_handler (a, s);

    if (s->displaying)
      state_puts ("\n[-- End of PGP/MIME signed data --]\n", s);
  }
  else
  {
    dprint (1,(debugfile, "pgp_signed_handler: invalid format!\n"));
    state_puts ("[-- Error!  This message does not comply with the PGP/MIME specification! --]\n\n", s);
    mutt_decode_attachment (a, s); /* just treat the data as text/plain... */
  }
}

BODY *pgp_decrypt_part (BODY *a, STATE *s, FILE *fpout)
{
  char buf[LONG_STRING];
  FILE *pgpin, *pgpout, *pgperr;
  struct stat info;
  BODY *tattach;
  int len;
  char pgperrfile[_POSIX_PATH_MAX];
  pid_t thepid;

  mutt_mktemp (pgperrfile);
  if ((pgperr = safe_fopen (pgperrfile, "w+")) == NULL)
  {
    mutt_perror (pgperrfile);
    return NULL;
  }
  unlink (pgperrfile);

  snprintf (buf, sizeof (buf), "PGPPASSFD=0; export PGPPASSFD; %s +verbose=0 +batchmode -f", Pgp);

  if ((thepid = mutt_create_filter_fd (buf, &pgpin, &pgpout, NULL, -1, -1,
				       fileno (pgperr))) == -1)
  {
    fclose (pgperr);
    if (s->displaying)
      state_puts ("[-- Error: could not create a PGP subprocess! --]\n\n", s);
    return (NULL);
  }

  /* send the PGP passphrase to the subprocess */
  fputs (PgpPass, pgpin);
  fputc ('\n', pgpin);

  /* Position the stream at the beginning of the body, and send the data to
   * the PGP subprocess.
   */
  fseek (s->fpin, a->offset, 0);
  mutt_copy_bytes (s->fpin, pgpin, a->length);
  fclose (pgpin);

  /* Read the output from PGP, and make sure to change CRLF to LF, otherwise
   * read_mime_header has a hard time parsing the message.
   */
  while (fgets (buf, sizeof (buf) - 1, pgpout) != NULL)
  {
    len = strlen (buf);
    if (len > 1 && buf[len - 2] == '\r')
      strcpy (buf + len - 2, "\n");
    fputs (buf, fpout);
  }

  fclose (pgpout);
  mutt_wait_filter (thepid);
  
  if (s->displaying)
  {
    fflush (pgperr);
    rewind (pgperr);
    mutt_copy_stream (pgperr, s->fpout);
    state_puts ("[-- End of PGP output --]\n\n", s);
  }
  fclose (pgperr);

  fflush (fpout);
  rewind (fpout);
  if ((tattach = mutt_read_mime_header (fpout, 0)) != NULL)
  {
    /*
     * Need to set the length of this body part.
     */
    fstat (fileno (fpout), &info);
    tattach->length = info.st_size - tattach->offset;

    /* See if we need to recurse on this MIME part.  */

    if (tattach->type == TYPEMULTIPART)
    {
      fseek (fpout, tattach->offset, 0);
      tattach->parts = mutt_parse_multipart (fpout, mutt_get_parameter ("boundary", tattach->parameter), tattach->offset + tattach->length, strcasecmp ("digest", tattach->subtype) == 0);
    }
    else if (tattach->type == TYPEMESSAGE)
    {
      fseek (fpout, tattach->offset, 0);
      tattach->parts = mutt_parse_messageRFC822 (fpout, tattach);
    }
  }

  return (tattach);
}

void pgp_encrypted_handler (BODY *a, STATE *s)
{
  char tempfile[_POSIX_PATH_MAX];
  FILE *fpout, *fpin;
  BODY *tattach;

  a = a->parts;
  if (!a || a->type != TYPEAPPLICATION || !a->subtype || 
      strcasecmp ("pgp-encrypted", a->subtype) != 0 ||
      !a->next || a->next->type != TYPEAPPLICATION || !a->next->subtype ||
      strcasecmp ("octet-stream", a->next->subtype) != 0)
  {
    if (s->displaying)
      state_puts ("[-- Error: malformed PGP/MIME message --]\n\n", s);
    return;
  }

  /*
   * Move forward to the application/pgp-encrypted body.
   */
  a = a->next;

  mutt_mktemp (tempfile);
  if ((fpout = safe_fopen (tempfile, "w+")) == NULL)
  {
    if (s->displaying)
      state_puts ("[-- Error: could not create temporary file --]\n", s);
    return;
  }
  unlink (tempfile);

  if (s->displaying) pgp_current_time (s);

  if ((tattach = pgp_decrypt_part (a, s, fpout)) != NULL)
  {
    if (s->displaying)
      state_puts ("[-- The following data is PGP/MIME encrypted --]\n\n", s);

    fpin = s->fpin;
    s->fpin = fpout;
    mutt_body_handler (tattach, s);
    s->fpin = fpin;

    if (s->displaying)
      state_puts ("\n[-- End of PGP/MIME encrypted data --]\n", s);

    mutt_free_body (&tattach);
  }

  fclose (fpout);
}

/* ----------------------------------------------------------------------------
 * Routines for sending PGP/MIME messages.
 */

static void convert_to_7bit (BODY *a)
{
  while (a)
  {
    if (a->type == TYPEMULTIPART)
    {
      if (a->encoding != ENC7BIT)
      {
        a->encoding = ENC7BIT;
        convert_to_7bit (a->parts);
      }
    } 
    else if (a->type == TYPEMESSAGE
	     && strcasecmp(a->subtype, "delivery-status"))
    {
      if(a->encoding != ENC7BIT)
	mutt_message_to_7bit(a, NULL);
    }
    else if (a->encoding == ENC8BIT)
      a->encoding = ENCQUOTEDPRINTABLE;
    else if (a->encoding == ENCBINARY)
      a->encoding = ENCBASE64;
    else if (a->content && (a->content->from || (a->content->space && option (OPTPGPSTRICTENC))))
      a->encoding = ENCQUOTEDPRINTABLE;
    a = a->next;
  }
}

BODY *pgp_sign_message (BODY *a)
{
  PARAMETER *p;
  BODY *t;
  char cmd[STRING], sigfile[_POSIX_PATH_MAX], buffer[LONG_STRING];
  FILE *pgpin, *pgpout, *pgperr, *fp;
  int err = 0;
  int empty = 1;
  pid_t thepid;

  convert_to_7bit (a); /* Signed data _must_ be in 7-bit format. */

  mutt_mktemp (sigfile);
  if ((fp = safe_fopen (sigfile, "w")) == NULL)
  {
    return (NULL);
  }

  snprintf (cmd, sizeof (cmd), "%s%s +verbose=0 +batchmode -abfst %s %s",
	    PgpPass[0] ? "PGPPASSFD=0; export PGPPASSFD; " : "",
	    Pgp, PgpSignAs[0] ? "-u" : "", PgpSignAs[0] ? PgpSignAs : "");

  if ((thepid = mutt_create_filter (cmd, &pgpin, &pgpout, &pgperr)) == -1)
  {
    safe_free ((void **)&sigfile);
    fclose (fp);
    return (NULL);
  }

  fputs (PgpPass, pgpin);
  fputc ('\n', pgpin);

  mutt_write_mime_header (a, pgpin);
  fputc ('\n', pgpin);
  mutt_write_mime_body (a, pgpin);

  if (fclose (pgpin) != 0)
  {
    dprint (1,(debugfile, "pgp_sign_message: fclose() returned -1; errno=%d.\n",
	       errno));
    mutt_wait_filter (thepid);
    fclose (pgpout);
    fclose (pgperr);
    fclose (fp);
    unlink (sigfile);
    return (NULL);
  }

  /*
   * Read back the PGP signature.  Also, change MESSAGE=>SIGNATURE as
   * recommended for future releases of PGP.
   */
  while (fgets (buffer, sizeof (buffer) - 1, pgpout) != NULL)
  {
    if (strcmp ("-----BEGIN PGP MESSAGE-----\n", buffer) == 0)
      fputs ("-----BEGIN PGP SIGNATURE-----\n", fp);
    else if (strcmp("-----END PGP MESSAGE-----\n", buffer) == 0)
      fputs ("-----END PGP SIGNATURE-----\n", fp);
    else
      fputs (buffer, fp);
    empty = 0; /* got some output, so we're ok */
  }

  /* check for errors from PGP */
  err = 0;
  while (fgets (buffer, sizeof (buffer) - 1, pgperr) != NULL)
  {
    err = 1;
    fputs (buffer, stdout);
  }

  mutt_wait_filter (thepid);
  fclose (pgperr);
  fclose (pgpout);

  if (fclose (fp) != 0)
  {
    mutt_perror ("fclose");
    unlink (sigfile);
    return (NULL);
  }

  if (err) ci_any_key_to_continue (NULL);
  if (empty)
  {
    unlink (sigfile);
    return (NULL); /* fatal error while signing */
  }

  t = mutt_new_body ();
  t->type = TYPEMULTIPART;
  t->subtype = safe_strdup ("signed");
  t->use_disp = 0;
  t->encoding = ENC7BIT;

  t->parameter = p = mutt_new_parameter ();
  p->attribute = safe_strdup ("protocol");
  p->value = safe_strdup ("application/pgp-signature");

  p->next = mutt_new_parameter ();
  p = p->next;
  p->attribute = safe_strdup ("micalg");
  p->value = safe_strdup ("pgp-md5");

  p->next = mutt_new_parameter ();
  p = p->next;
  p->attribute = safe_strdup ("boundary");
  p->value = mutt_generate_boundary ();

  t->parts = a;
  a = t;

  t->parts->next = mutt_new_body ();
  t = t->parts->next;
  t->type = TYPEAPPLICATION;
  t->subtype = safe_strdup ("pgp-signature");
  t->filename = safe_strdup (sigfile);
  t->use_disp = 0;
  t->encoding = ENC7BIT;
  t->unlink = 1; /* ok to remove this file after sending. */

  return (a);
}

/* This routine attempts to find the keyids of the recipients of a message.
 * It returns NULL if any of the keys can not be found.
 */
char *pgp_findKeys (ADDRESS *to, ADDRESS *cc, ADDRESS *bcc)
{
  KEYINFO *db = ki_getdb (0);
  char *key, keylist[1024];
  ADDRESS *p = to;
  ADDRESS *tmp;
  int count = 0;

  *keylist = 0;
  for (count = 0; count < 3; count++)
  {
    switch (count)
    {
      case 0:
	p = to;
	break;

      case 1:
	p = cc;
	break;

      case 2:
	p = bcc;
	break;
    }
    while (p)
    {
      if ((key = ki_getkeybyaddr (p, db)) == NULL)
      {
	char buf[LONG_STRING];

	strfcpy (buf, "Enter keyID for ", sizeof (buf));
	tmp = p->next;
	rfc822_address (buf + strlen (buf), sizeof (buf) - strlen (buf), p);
	strcat (buf, ": ");
	p->next = tmp;

	if ((key = pgp_ask_for_key (0, db, buf)) == NULL)
	{
	  ki_closedb (db);
	  return NULL;
	}
      }
      sprintf (keylist + strlen (keylist), " 0x%s", key);
      p = p->next;
    }
  }
  ki_closedb (db);
  return (safe_strdup (keylist));
}

BODY *pgp_encrypt_message (BODY *a, char *keylist, int sign)
{
  char buf[LONG_STRING], tempfile[_POSIX_PATH_MAX];
  char pgperrfile[_POSIX_PATH_MAX];
  char signopts[SHORT_STRING];
  FILE *pgpin, *pgperr, *fpout;
  BODY *t;
  PARAMETER *p;
  int err = 0;
  int empty;
  pid_t thepid;

  mutt_mktemp (tempfile);
  if ((fpout = safe_fopen (tempfile, "w+")) == NULL)
  {
    mutt_perror (tempfile);
    return (NULL);
  }

  mutt_mktemp (pgperrfile);
  if ((pgperr = safe_fopen (pgperrfile, "w+")) == NULL)
  {
    mutt_perror (pgperrfile);
    return NULL;
  }
  unlink (pgperrfile);

  if (sign)
  {
    convert_to_7bit (a);

    if (PgpSignAs[0])
      snprintf (signopts, sizeof (signopts), "s -u \"%s\"", PgpSignAs);
    else
      strfcpy (signopts, "s", sizeof (signopts));
  }

  snprintf (buf, sizeof (buf),
	    "%s%s +verbose=0 +batchmode %s -aeft%s %s",
	    sign ? "PGPPASSFD=0; export PGPPASSFD; " : "",
	    Pgp,
	    option (OPTPGPENCRYPTSELF) ? "+encrypttoself" : "",
	    sign ? signopts : "",
	    keylist);

  if ((thepid = mutt_create_filter_fd (buf, &pgpin, NULL, NULL, -1, 
				       fileno (fpout), fileno (pgperr))) == -1)
  {
    fclose (pgperr);
    return (NULL);
  }

  if (sign)
  {
    fputs (PgpPass, pgpin);
    fputc ('\n', pgpin);
  }

  mutt_write_mime_header (a, pgpin);
  fputc ('\n', pgpin);
  mutt_write_mime_body (a, pgpin);

  if (fclose (pgpin) != 0)
  {
    mutt_wait_filter (thepid);
    fclose (fpout);
    fclose (pgperr);
    unlink (tempfile);
    return (NULL);
  }

  mutt_wait_filter (thepid);

  fflush (fpout);
  rewind (fpout);
  empty = (fgetc (fpout) == EOF);
  fclose (fpout);

  fflush (pgperr);
  rewind (pgperr);
  while (fgets (buf, sizeof (buf) - 1, pgperr) != NULL)
  {
    err = 1;
    fputs (buf, stdout);
  }
  fclose (pgperr);

  /* pause if there is any error output from PGP */
  if (err)
    ci_any_key_to_continue (NULL);

  if (empty)
  {
    /* fatal error while trying to encrypt message */
    unlink (tempfile);
    return (NULL);
  }

  t = mutt_new_body ();
  t->type = TYPEMULTIPART;
  t->subtype = safe_strdup ("encrypted");
  t->encoding = ENC7BIT;
  t->use_disp = 0;

  t->parameter = p = mutt_new_parameter ();
  p->attribute = safe_strdup ("protocol");
  p->value = safe_strdup ("application/pgp-encrypted");

  p->next = mutt_new_parameter ();
  p = p->next;
  p->attribute = safe_strdup ("boundary");
  p->value = mutt_generate_boundary ();

  t->parts = mutt_new_body ();
  t->parts->type = TYPEAPPLICATION;
  t->parts->subtype = safe_strdup ("pgp-encrypted");
  t->parts->encoding = ENC7BIT;
  t->parts->use_disp = 0;

  t->parts->next = mutt_new_body ();
  t->parts->next->type = TYPEAPPLICATION;
  t->parts->next->subtype = safe_strdup ("octet-stream");
  t->parts->next->encoding = ENC7BIT;
  t->parts->next->filename = safe_strdup (tempfile);
  t->parts->next->use_disp = 0;
  t->parts->next->unlink = 1; /* delete after sending the message */

  mutt_free_body (&a); /* no longer needed! */

  return (t);
}

int pgp_protect (HEADER *msg)
{
  char *pgpkeylist = NULL;
  BODY *pbody = NULL;

  /* Do a quick check to make sure that we can find all of the encryption
   * keys if the user has requested this service.
   */
  if (msg->pgp & PGPENCRYPT)
  {
    if ((pgpkeylist = pgp_findKeys (msg->env->to, msg->env->cc, msg->env->bcc)) == NULL)
      return (-1);
  }

  if ((msg->pgp & PGPSIGN) && !pgp_valid_passphrase ())
    return (-1);

  endwin ();
  if (msg->pgp & PGPENCRYPT)
  {
    pbody = pgp_encrypt_message (msg->content, pgpkeylist, msg->pgp & PGPSIGN);
    safe_free ((void **) &pgpkeylist);
    if (!pbody)
      return (-1);
  }
  else if (msg->pgp == PGPSIGN)
  {
    if ((pbody = pgp_sign_message (msg->content)) == NULL)
      return (-1);
  }
  msg->content = pbody;
  return 0;
}

#endif /* _PGPPATH */
