/*
 * Copyright (C) 1996,1997 Michael R. Elkins <me@cs.hmc.edu>
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 

/* WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!  WARNING!!!
 *
 * Legal for distribution in the U.S./Canada ONLY!  Exporting this file
 * outside of the U.S./Canada may be in violation of ITAR regulations!
 */

#include "mutt.h"
#include "mutt_curses.h"
#include "mutt_menu.h"
#include "pgp.h"

#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>

#ifdef _PGPPATH

/* ---------------------------------------------------------------------------
 * Menu used to select a key when multiple keys match an address/name
 */

typedef struct
{
  KEYINFO *k;
  char *a;
} pgp_key_t;

static void pgp_entry (char *s, size_t l, MUTTMENU *menu, int num)
{
  pgp_key_t *KeyTable = (pgp_key_t *) menu->data;

  snprintf (s, l, "%2d   %4d/0x%8s   %s",
	    num + 1, KeyTable[num].k->keylen, KeyTable[num].k->keyid,
	    KeyTable[num].a);
}

static int pgp_search (MUTTMENU *m, regex_t *re, int n)
{
  char buf[LONG_STRING];
  
  pgp_entry (buf, sizeof (buf), m, n);
  return (regexec (re, buf, 0, NULL, 0));
}

static int pgp_compare (const void *a, const void *b)
{
  pgp_key_t *s = (pgp_key_t *) a;
  pgp_key_t *t = (pgp_key_t *) b;

  return (strcasecmp (s->a, t->a));
}

static char *pgp_select_key (LIST *keys, ADDRESS *p, const char *s)
{
  int keymax;
  pgp_key_t *KeyTable;
  MUTTMENU *menu;
  ADDRESS *a;
  int i;  
  int done = 0;
  LIST *l;
  char *id = NULL, helpstr[SHORT_STRING], buf[LONG_STRING];

  for (i = 0, l = keys; l; l = l->next)
    for (a = ((KEYINFO *) l->data)->address; a; i++, a = a->next)
      ;

  if (i == 0) return NULL;

  keymax = i;
  KeyTable = safe_malloc (sizeof (pgp_key_t) * i);

  for (i = 0, l = keys; l; l = l->next)
    for (a = ((KEYINFO *) l->data)->address; a ; i++, a = a->next)
    {
      KeyTable[i].k = (KEYINFO *) l->data;
      buf[0] = 0;
      rfc822_address (buf, sizeof (buf), a);
      if (a->personal)
	sprintf (buf + strlen (buf), " (%s)", a->personal);
      KeyTable[i].a = safe_strdup (buf);
    }

  qsort (KeyTable, i, sizeof (pgp_key_t), pgp_compare);

  helpstr[0] = 0;
  mutt_make_help (buf, sizeof (buf), "Exit  ", MENU_PGP, OP_EXIT);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Select  ", MENU_PGP, 
		  OP_GENERIC_SELECT_ENTRY);
  strcat (helpstr, buf);
  mutt_make_help (buf, sizeof (buf), "Help", MENU_PGP, OP_HELP);
  strcat (helpstr, buf);

  menu = mutt_new_menu ();
  menu->max = keymax;
  menu->make_entry = pgp_entry;
  menu->search = pgp_search;
  menu->menu = MENU_PGP;
  menu->help = helpstr;
  menu->data = KeyTable;

  strfcpy (buf, "PGP keys matching ", sizeof (buf));
  if (p)
    rfc822_address (buf, sizeof (buf) - strlen (buf), p);
  else
    strcat (buf, s);
  menu->title = buf;

  while (!done)
  {
    switch (mutt_menuLoop (menu))
    {
      case OP_GENERIC_SELECT_ENTRY:

	id = KeyTable[menu->current].k->keyid;
	done = 1;
	break;

      case OP_EXIT:

	id = NULL;
	done = 1;
	break;
    }
  }

  mutt_menuDestroy (&menu);
  for (i = 0; i < keymax; i++)
    safe_free ((void **) &KeyTable[i].a);
  safe_free ((void **) &KeyTable);

  return (id);
}

char *pgp_ask_for_key (int secring, KEYINFO *udb, char *tag)
{
  KEYINFO *db;
  char *key;
  char resp[SHORT_STRING];

  db = udb ? udb : ki_getdb (secring);

  resp[0] = 0;

  FOREVER
  {
    if (ci_get_field (tag, resp, sizeof (resp), 0) != 0)
    {
      if (!udb) ki_closedb (db);
      return NULL;
    }
    if ((key = ki_getkeybystr (resp, db)))
    {
      key = safe_strdup (key);
      if (!udb) ki_closedb (db);
      return (key);
    }
    BEEP ();
  }
  /* not reached */
}

/* ----------------------------------------------------------------------------
 *
 * The following routines are used to parse the output of "pgp -kv" to build
 * a list of the public keys in the user's database.
 */
static KEYINFO *ki_parse_line (char *s)
{
  char *keyid, *address, *q;
  KEYINFO *p;

  s += 3; /* skip the "pub" */
  SKIPWS (s);
  if (!*s) return 0;

  /* extract the key length */
  q = s;
  while (*s && *s != '/') s++;
  if (*s != '/') return 0;
  *s++ = 0;

  keyid = s;
  while (*s && !ISSPACE (*s)) s++;
  if (!*s) return 0;
  *s++ = 0;

  SKIPWS (s);
  if ((s = strchr (s, ' ')) == 0) return 0;
  SKIPWS (s);
  address = s;
  while (*s && *s != '\n') s++;
  *s = 0;

  p = (KEYINFO *) malloc (sizeof (KEYINFO));
  p->keyid = safe_strdup (keyid);
  p->next = 0;
  p->revoked = 0; /* FOO - need to look for the REVOKED keys */
  p->keylen = atoi (q);
  p->address = 0;
  rfc822_parse_adrlist (&p->address, address, "@");
  return (p);
}

KEYINFO *ki_getdb (int secring)
{
  FILE *in;
  KEYINFO *db = NULL, *end = 0;
  char buf[LONG_STRING], *c;
  char ring[_POSIX_PATH_MAX];

  if ((c = getenv ("PGPPATH")) == NULL)
  {
    snprintf (buf, sizeof (buf), "%s/.pgp", Homedir);
    c = buf;
  }
  snprintf (ring, sizeof (ring), "%s/%s.pgp", c, secring ? "secring" : "pubring");

  snprintf (buf, sizeof (buf), "%s +verbose=0 -kv +language=en %s", Pgp, ring);

  if (mutt_create_filter (buf, NULL, &in, NULL) == -1) return NULL;

  while (fgets (buf, STRING, in) != NULL)
  {
    if (strncmp (buf, "pub", 3) == 0 || strncmp (buf, "sec", 3) == 0)
    {
      if (db)
      {
        end->next = ki_parse_line (buf);
        end = end->next;
      }
      else
        db = end = ki_parse_line (buf);
    }
    else if (buf[0] == ' ') /* alias */
    {
      c = buf;
      while (*c && (*c == ' ' || *c == '\t')) c++;
      rfc822_parse_adrlist (&end->address, c, "@");
    }
  }
  fclose (in);
  wait (NULL);
  return (db);
}

void ki_closedb (KEYINFO *k)
{
  KEYINFO *tmp;

  while (k)
  {
    if (k->keyid) free (k->keyid);
    mutt_free_address (&k->address);
    tmp = k;
    k = k->next;
    free (tmp);
  }
}

/* case insensitive strstr() */
static char *mutt_stristr (char *haystack, char *needle)
{
  char *p, *q;

  if (!haystack)
    return NULL;
  if (!needle)
    return (haystack);

  while (*(p = haystack))
  {
    for (q = needle ; *p && *q && tolower (*p) == tolower (*q) ; p++, q++)
      ;
    if (!*q)
      return (haystack);
    haystack++;
  }
  return NULL;
}

char *ki_getkeybyaddr (ADDRESS *a, KEYINFO *k)
{
  ADDRESS *p;
  LIST *l = NULL, *t = NULL;
  char *id = NULL;

  for ( ; k ; k = k->next)
  {
    if (k->revoked)
      continue;

    for (p = k->address ; p ; p = p->next)
    {
      if ((p->mailbox && p->host && a->mailbox && a->host &&
	   strcasecmp (p->mailbox, a->mailbox) == 0 && 
	   strcasecmp (p->host, a->host) == 0) ||
	  (a->personal && p->personal &&
	   mutt_stristr (p->personal, a->personal) != NULL))
      {
	t = mutt_new_list ();
	t->data = (void *) k;
	t->next = l;
	l = t;
	break;
      }
    }
  }

  if (l)
  {
    if (l->next)
    {
      /* query for which key the user wants */
      id = pgp_select_key (l, a, NULL);
    }
    else
      id = ((KEYINFO *)l->data)->keyid;

    /* mutt_free_list() frees the .data member, so clear the pointers */
    t = l;
    while (t)
    {
      t->data = NULL;
      t = t->next;
    }

    mutt_free_list (&l);
  }

  return (id);
}

char *ki_getkeybystr (char *p, KEYINFO *k)
{
  LIST *t = NULL, *l = NULL;
  char *id = NULL;

  for ( ; k ; k = k->next)
  {
    if (k->revoked)
      continue;

    if (strcasecmp (p, k->keyid) == 0 ||
	(k->address->personal && mutt_stristr (k->address->personal, p) != NULL) ||
	mutt_stristr (k->address->mailbox, p) != NULL)
    {
      t = mutt_new_list ();
      t->data = (void *) k;
      t->next = l;
      l = t;
    }
  }

  if (l)
  {
    if (l->next)
      id = pgp_select_key (l, NULL, p);
    else
      id = ((KEYINFO *) l->data)->keyid;

    t = l;
    while (t)
    {
      t->data = NULL;
      t = t->next;
    }

    mutt_free_list (&l);
  }

  return (id);
}
#endif /* _PGPPATH */
