/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Core FTPD module
 */

#include "conf.h"

#include <ctype.h>

#define CONF_ERROR(x,s) return pstrcat((x)->tmp_pool, \
                        (x)->argv[0],": ",(s),NULL)

#define CHECK_ARGS(x,n)	if((x)->argc-1 < n) \
                        CONF_ERROR(x,"missing arguments")

#define CHECK_CONF(x,p)  if(!check_conf((x),(p))) \
                         CONF_ERROR((x), \
                         pstrcat((x)->tmp_pool,"directive not allowed in ", \
                                 get_section_name((x)), \
                                 " section.",NULL))

/* This is declared static to this module because it's not needed,
 * except for the HELP command.
 */

static struct {
  char *cmd;
  char *syntax;
  int implemented;
} _help[] = {
  { C_USER, "<sp> username",			TRUE },
  { C_PASS, "<sp> password",			TRUE },
  { C_ACCT, "is not implemented",		FALSE },
  { C_CWD,  "<sp> pathname",			TRUE },
  { C_XCWD, "<sp> pathname",			TRUE },
  { C_CDUP, "(up one directory)",		TRUE },
  { C_XCUP, "(up one directory)",		TRUE },
  { C_SMNT, "is not implemented",		FALSE },
  { C_QUIT, "(close control connection)",	TRUE },
  { C_REIN, "is not implemented",		FALSE },
  { C_PORT, "<sp> h1,h2,h3,h4,p1,p2",		TRUE },
  { C_PASV, "(returns address/port)",		TRUE },
  { C_TYPE, "<sp> type-code (A or I)",		TRUE },
  { C_STRU, "is not implemented",		FALSE },
  { C_MODE, "is not implemented (always S)",	FALSE },
  { C_RETR, "<sp> pathname",			TRUE },
  { C_STOR, "<sp> pathname",			TRUE },
  { C_STOU, "is not implemented",		FALSE },
  { C_APPE, "is not implemented",		FALSE },
  { C_ALLO, "is not implemented",		FALSE },
  { C_REST, "<sp> byte-count",			TRUE },
  { C_RNFR, "<sp> pathname",			TRUE },
  { C_RNTO, "<sp> pathname",			TRUE },
  { C_ABOR, "(abort current operation)",	TRUE },
  { C_DELE, "<sp> pathname",			TRUE },
  { C_MDTM, "<sp> pathname",			TRUE },
  { C_RMD,  "<sp> pathname",			TRUE },
  { C_XRMD, "<sp> pathname",			TRUE },
  { C_MKD,  "<sp> pathname",			TRUE },
  { C_XMKD, "<sp> pathname",			TRUE },
  { C_PWD,  "(returns current working directory)", TRUE },
  { C_XPWD, "(returns current working directory)", TRUE },
  { C_SIZE, "<sp> pathname",			TRUE },
  { C_LIST, "[<sp> pathname]",			TRUE },
  { C_NLST, "[<sp> (pathname)]",		TRUE },
  { C_SITE, "is not implemented",		TRUE },
  { C_SYST, "(returns system type)",		TRUE },
  { C_STAT, "[<sp> pathname]",			TRUE },
  { C_HELP, "[<sp> command]",			TRUE },
  { C_NOOP, "(no operation)",			TRUE },
  { NULL,   NULL,          			FALSE }
};

extern module site_module;
extern xaset_t *servers;

/* from mod_site */
extern char *site_dispatch(cmd_rec*);
/* from mod_xfer */
extern void xfer_abort(IOREQ*,int);

int check_conf(cmd_rec *cmd,int allowed)
{
  int x;

  x = (cmd->config && cmd->config->config_type != CONF_PARAM ? 
       cmd->config->config_type : CONF_ROOT);

  return (x & allowed);
}

char *get_section_name(cmd_rec *cmd)
{
  if(!cmd->config || cmd->config->config_type == CONF_PARAM)
    return "top level";

  switch(cmd->config->config_type) {
  case CONF_DIR: return "<Directory>";
  case CONF_ANON: return "<Anonymous>";
  case CONF_LIMIT: return "<Limit>";
  };

  return "(null)";
}

int _get_boolean(cmd_rec *cmd,int av)
{
  char *cp = cmd->argv[av];

  /* Boolean string can be "on","off","yes","no",
   * "true","false","1" or "0".
   */

  if(!strcasecmp(cp,"on"))
    return 1;
  if(!strcasecmp(cp,"off"))
    return 0;
  if(!strcasecmp(cp,"yes"))
    return 1;
  if(!strcasecmp(cp,"no"))
    return 0;
  if(!strcasecmp(cp,"true"))
    return 1;
  if(!strcasecmp(cp,"false"))
    return 0;
  if(!strcasecmp(cp,"1"))
    return 1;
  if(!strcasecmp(cp,"0"))
    return 0;

  return -1;
}

static char *set_servername(cmd_rec *cmd)
{
  server_rec *s = cmd->server;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  s->ServerName = pstrdup(s->pool,cmd->argv[1]);
  return NULL;
}

static char *set_servertype(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT);

  if(!strcasecmp(cmd->argv[1],"inetd"))
    ServerType = SERVER_INETD;
  else if(!strcasecmp(cmd->argv[1],"standalone"))
    ServerType = SERVER_STANDALONE;
  else
    CONF_ERROR(cmd,"type must be either 'inetd' or 'standalone'.");
  return NULL;
}

static char *set_serveradmin(cmd_rec *cmd)
{
  server_rec *s = cmd->server;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  s->ServerAdmin = pstrdup(s->pool,cmd->argv[1]);

  return NULL;
}

static char *set_serverport(cmd_rec *cmd)
{
  server_rec *s = cmd->server;
  int port;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  port = atoi(cmd->argv[1]);
  if(port < 1 || port > 65535)
    CONF_ERROR(cmd,"value must be between 1 and 65535");

  s->ServerPort = port;
  return NULL;
}

static char *set_deferwelcome(cmd_rec *cmd)
{
  int b;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  if((b = _get_boolean(cmd,1)) == -1)
    CONF_ERROR(cmd,"expected boolean argument.");

  add_config_param("DeferWelcome",1,(void*)b);

  return NULL;
}

static char *set_defaultserver(cmd_rec *cmd)
{
  int b;
  server_rec *s;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  if((b = _get_boolean(cmd,1)) == -1)
    CONF_ERROR(cmd,"expected boolean argument.");

  if(!b)
    return NULL;

  /* DefaultServer is not allowed if already set somewhere */
  for(s = (server_rec*)servers->xas_list; s; s=s->next)
    if(find_config(s->conf,CONF_PARAM,"DefaultServer",FALSE)) {
      CONF_ERROR(cmd,"DefaultServer has already been set.");
    }

  add_config_param("DefaultServer",1,(void*)b);
  return NULL;
}

static char *_set_timeout(int *v, cmd_rec *cmd)
{
  int timeout;
  char *endp;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT);

  timeout = (int)strtol(cmd->argv[1],&endp,10);

  if((endp && *endp) || timeout < 0 || timeout > 65535)
    CONF_ERROR(cmd,"timeout values must be between 0 and 65535");

  *v = timeout;
  return NULL;
}

static char *set_maxclients(cmd_rec *cmd)
{
  int max;
  char *endp;
  config_rec *c;

  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  if(cmd->argc < 2 || cmd->argc > 3)
    CONF_ERROR(cmd,"invalid number of arguments");

  if(!strcasecmp(cmd->argv[1],"none"))
    max = -1;
  else {
    max = (int)strtol(cmd->argv[1],&endp,10);

    if((endp && *endp) || max < 1)
      CONF_ERROR(cmd,"argument must be 'none' or a number greater than 0.");
  }

  if(cmd->argc == 3) {
    c = add_config_param("MaxClients",2,(void*)max,NULL);
    c->argv[1] = pstrdup(c->pool,cmd->argv[2]);   
  } else
    add_config_param("MaxClients",1,(void*)max);

  return NULL;
}

static char *set_maxloginattempts(cmd_rec *cmd)
{
  int max;
  char *endp;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  if(!strcasecmp(cmd->argv[1],"none"))
    max = 0;
  else {
    max = (int)strtol(cmd->argv[1],&endp,10);

    if((endp && *endp) || max < 1)
      CONF_ERROR(cmd,"argument must be 'none' or a number greater than 0.");
  }

  add_config_param("MaxLoginAttempts",1,(void*)max);
  return NULL;
}

static char *set_useftpusers(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  c = add_config_param("UseFtpUsers",1,(void*)_get_boolean(cmd,1));

  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_timeoutlogin(cmd_rec *cmd)
{
  return _set_timeout(&TimeoutLogin,cmd);
}

static char *set_timeoutidle(cmd_rec *cmd)
{
  return _set_timeout(&TimeoutIdle,cmd);
}

static char *set_timeoutnoxfer(cmd_rec *cmd)
{
  return _set_timeout(&TimeoutNoXfer,cmd);
}

static char *set_socketbindtight(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT);

  SocketBindTight = _get_boolean(cmd,1);
  return NULL;  
}  

static char *set_tcpbacklog(cmd_rec *cmd)
{
  int backlog;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT);

  backlog = atoi(cmd->argv[1]);

  if(backlog < 1 || backlog > 255)
    CONF_ERROR(cmd,"parameter must be a number between 1 and 255.");

  tcpBackLog = backlog;
  return NULL;
}

static char *set_tcpreceivewindow(cmd_rec *cmd)
{
  int rwin;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  rwin = atoi(cmd->argv[1]);

  if(rwin < 1024)
    CONF_ERROR(cmd,"parameter must be number equal to or greater than 1024.");

  cmd->server->tcp_rwin = rwin;
  return NULL;
}

static char *set_tcpsendwindow(cmd_rec *cmd)
{
  int swin;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  swin = atoi(cmd->argv[1]);

  if(swin < 1024)
    CONF_ERROR(cmd,"parameter must be number equal to or greater than 1024.");

  cmd->server->tcp_swin = swin;
  return NULL;
}

static char *set_user(cmd_rec *cmd)
{
  struct passwd *pw;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  if((pw = getpwnam(cmd->argv[1])) == NULL) {
    endpwent();
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"Unknown user '",
                           cmd->argv[1],"'.",NULL));
  }


  /* The extra cast is required to avoid compiler warning */

  add_config_param("User",1,(void*)((int)pw->pw_uid));
  add_config_param_str("UserName",1,(void*)cmd->argv[1]);

  /* We don't need extra fds sitting around open */
  endpwent();
  return NULL;
}

static char *set_group(cmd_rec *cmd)
{
  struct group *grp;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  if((grp = getgrnam(cmd->argv[1])) == NULL) {
    endgrent();
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"Unknown group '",
                           cmd->argv[1],"'.",NULL));
  }

  /* The extra cast is needed to avoid compiler warning */

  add_config_param("Group",1,(void*)((int)grp->gr_gid));
  add_config_param_str("GroupName",1,(void*)cmd->argv[1]);

  endgrent();
  return NULL;
}

static char *add_userpassword(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,2);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  add_config_param_str("UserPassword",2,cmd->argv[1],cmd->argv[2]);
  return NULL;
}

static char *add_grouppassword(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,2);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  add_config_param_str("GroupPassword",2,cmd->argv[1],cmd->argv[2]);
  return NULL;
}

static char *set_accessgrantmsg(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  add_config_param_str("AccessGrantMsg",1,cmd->argv[1]);
  return NULL;
}

static char *set_umask(cmd_rec *cmd)
{
  config_rec *c;
  char *endp;
  int _umask;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_DIR|CONF_ANON);

  _umask = strtol(cmd->argv[1],&endp,0);

  if(endp && *endp)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"'",cmd->argv[1],"' is not "
                           "a valid umask.",NULL));

  c = add_config_param("Umask",1,(void*)_umask);
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_requirevalidshell(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  c = add_config_param("RequireValidShell",1,_get_boolean(cmd,1));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_showsymlinks(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  c = add_config_param("ShowSymlinks",1,_get_boolean(cmd,1));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}


static char *add_directory(cmd_rec *cmd)
{
  config_rec *c;
  char *dir,*rootdir = NULL;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  dir = cmd->argv[1];

  if(*dir != '/' && *dir != '~' && 
     (!cmd->config || cmd->config->config_type != CONF_ANON))
    CONF_ERROR(cmd,"relative pathname not allowed in non-anonymous blocks.");

  /* If in anonymous mode, and path is relative, just cat anon root
   * and relative path
   *
   * NOTE [Flood,9/97]: This is no longer necessary, because we don't
   * interpolate anonymous dirs at run-time.
   *
   */
  if(cmd->config && cmd->config->config_type == CONF_ANON &&
     *dir != '/' && *dir != '~') {
    if(strcmp(dir,"*") != 0)
      dir = pdircat(cmd->tmp_pool,"/",dir,NULL);
    rootdir = cmd->config->name;
  }
  else {
    dir = dir_abs_path(cmd->tmp_pool,dir);

    if(!dir)
      CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,cmd->argv[1],": ",
                     strerror(errno),NULL));
  }

  c = start_sub_config(dir);
  c->argc = 2;
  c->argv = pcalloc(c->pool,3*sizeof(void*));
  if(rootdir)
    c->argv[1] = pstrdup(c->pool,rootdir);

  c->config_type = CONF_DIR;
  return NULL;
}

static char *set_allowretrieverestart(cmd_rec *cmd)
{
  config_rec *c;
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_DIR|CONF_ANON);

  c = add_config_param("AllowRetrieveRestart",1,
                       (void*)_get_boolean(cmd,1));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_allowstorerestart(cmd_rec *cmd)
{
  config_rec *c;
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_DIR|CONF_ANON);

  c = add_config_param("AllowStoreRestart",1,
                       (void*)_get_boolean(cmd,1));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *add_hidenoaccess(cmd_rec *cmd)
{
  config_rec *c;
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_DIR|CONF_ANON);

  c = add_config_param("HideNoAccess",1,(void*)1);
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *add_hideuser(cmd_rec *cmd)
{
  config_rec *c;
  struct passwd *pw;
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_DIR|CONF_ANON);

  pw = getpwnam(cmd->argv[1]);
  if(!pw)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"'",cmd->argv[1],"' is not "
                   "a valid user.",NULL));

  c = add_config_param("HideUser",1,(void*)((int)pw->pw_uid));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *add_hidegroup(cmd_rec *cmd)
{
  config_rec *c;
  struct group *gr;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_DIR|CONF_ANON);

  gr = getgrnam(cmd->argv[1]);
  if(!gr)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"'",cmd->argv[1],"' is not "
                   "a valid group.",NULL));

  c = add_config_param("HideGroup",1,(void*)((int)gr->gr_gid));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *add_groupowner(cmd_rec *cmd)
{
  config_rec *c;
  struct group *gr;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ANON|CONF_DIR);

  gr = getgrnam(cmd->argv[1]);
  if(!gr)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"'",cmd->argv[1],"' is not "
                           "a valid group.",NULL));

  c = add_config_param("GroupOwner",1,(void*)((int)gr->gr_gid));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_allowoverwrite(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON|CONF_DIR);

  c = add_config_param("AllowOverwrite",1,(void*)_get_boolean(cmd,1));
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *end_directory(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_DIR);

  end_sub_config();
  return NULL;
}

static char *add_anonymous(cmd_rec *cmd)
{
  config_rec *c;
  char *dir;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  dir = cmd->argv[1];

  if(*dir != '/' && *dir != '~')
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"(",dir,") absolute pathname "
               "required.",NULL));

  if(strchr(dir,'*'))
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"(",dir,") wildcards not allowed "
               "in pathname."));

  if(!strcmp(dir,"/"))
    CONF_ERROR(cmd,"'/' not permitted for anonymous root directory.");

  if(*(dir+strlen(dir)-1) != '/')
    dir = pstrcat(cmd->tmp_pool,dir,"/",NULL);

  if(!dir)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,cmd->argv[1],": ",
               strerror(errno),NULL));

  c = start_sub_config(dir);

  c->config_type = CONF_ANON;
  return NULL;
}

static char *set_anonrequirepassword(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ANON);

  add_config_param("AnonRequirePassword",1,
                   (void*)_get_boolean(cmd,1));
  return NULL;
}

static char *end_anonymous(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_ANON);

  end_sub_config();
  return NULL;
}

static char *add_limit(cmd_rec *cmd)
{
  config_rec *c;
  int cargc;
  char **argv,**cargv;

  if(cmd->argc < 2)
    CONF_ERROR(cmd,"directive requires one or more FTP commands.");
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_DIR|CONF_ANON|CONF_DYNDIR);

  c = start_sub_config("Limit");
  c->config_type = CONF_LIMIT;
  cargc = cmd->argc-1;
  cargv = cmd->argv+1;

  c->argc = cmd->argc-1;
  argv = (char**)c->argv = pcalloc(c->pool,cmd->argc*sizeof(void*));
    
  while(cargc--)
    *argv++ = pstrdup(c->pool,*cargv++);

  *argv = NULL;

  return NULL;
}

static char *add_order(cmd_rec *cmd)
{
  int order = -1,argc = cmd->argc;
  char *arg = "",**argv = cmd->argv+1;

  CHECK_CONF(cmd,CONF_LIMIT);

  while(--argc && *argv)
    arg = pstrcat(cmd->tmp_pool,arg,*argv++,NULL);

  if(!strcasecmp(arg,"allow,deny"))
    order = ORDER_ALLOWDENY;
  else if(!strcasecmp(arg,"deny,allow"))
    order = ORDER_DENYALLOW;
  else
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"'",arg,"': invalid argument",NULL));

  add_config_param("Order",1,(void*)order);
  return NULL;
}

static char *add_defaultroot(cmd_rec *cmd)
{
  config_rec *c;
  char *s,*ent,*dir,**argv;
  int argc;
  array_header *acl = NULL;

  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL);

  if(cmd->argc < 2)
    CONF_ERROR(cmd,"syntax: DefaultRoot <directory> [<group-expression>]");

  argv = cmd->argv;
  argc = cmd->argc-2;

  dir = *++argv;

  /* dir must be / or ~
   */

  if(*dir != '/' && *dir != '~')
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"(",dir,") absolute pathname "
               "required.",NULL));

  if(strchr(dir,'*'))
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"(",dir,") wildcards not allowed "
               "in pathname."));

  if(*(dir+strlen(dir)-1) != '/')
    dir = pstrcat(cmd->tmp_pool,dir,"/",NULL);

  if(!dir)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,cmd->argv[1],": ",
               strerror(errno),NULL));

  if(argc) {
    acl = make_array(cmd->tmp_pool,argc,sizeof(char*));

    while(argc-- && *(++argv)) {
      s = pstrdup(cmd->tmp_pool,*argv);

      /* parse the string into coma-delimited entries */
      while((ent = get_token(&s,",")) != NULL)
        if(*ent)
          *((char**)push_array(acl)) = ent;
    }

    argc = acl->nelts;
  }

  c = add_config_param("DefaultRoot",0);

  c->argc = argc+1;
  c->argv = pcalloc(c->pool,(argc+2) * sizeof(char*));
  argv = (char**)c->argv;
  *argv++ = pstrdup(c->pool,dir);

  if(argc && acl)
    while(argc--) {
      *argv++ = pstrdup(c->pool,*((char**)acl->elts));
      ((char**)acl->elts)++;
    }

  *argv = NULL;
  return NULL;
}

static char *_add_allow_deny(cmd_rec *cmd, char *name)
{
  int argc;
  char *s,*ent,**argv;
  array_header *acl;
  config_rec *c;

  CHECK_CONF(cmd,CONF_LIMIT);

  /* Syntax: allow [from] [all|none]|host|network[,...] */
  acl = make_array(cmd->tmp_pool,cmd->argc,sizeof(char*));
  argc = cmd->argc-1; argv = cmd->argv;

  /* Skip optional "from" keyword */
  while(argc && *(argv+1)) {
    if(!strcasecmp("from",*(argv+1))) {
      argv++; argc--; continue;
    } else if(!strcasecmp("all",*(argv+1))) {
      *((char**)push_array(acl)) = "ALL";
      argc = 0;
    } else if(!strcasecmp("none",*(argv+1))) {
      *((char**)push_array(acl)) = "NONE";
      argc = 0;
    }
    break;
  }

  while(argc-- && *(++argv)) {
    s = pstrdup(cmd->tmp_pool,*argv);

    /* parse the string into coma-delimited entries */
    while((ent = get_token(&s,",")) != NULL)
      if(*ent) {
        if(!strcasecmp(ent,"all") || !strcasecmp(ent,"none")) {
          acl->nelts = 0; argc = 0; break;
        }

        *((char**)push_array(acl)) = ent;
      }
  }

  if(!acl->nelts)
    CONF_ERROR(cmd,pstrcat(cmd->tmp_pool,"syntax: ",name,
                   " [from] [all|none]|host|network[,...]",NULL));

  c = add_config_param(name,0);

  c->argc = acl->nelts;
  c->argv = pcalloc(c->pool,(c->argc+1) * sizeof(char*));
  argv = (char**)c->argv;
  while(acl->nelts--) {
    *argv++ = pstrdup(c->pool,*((char**)acl->elts));
    ((char**)acl->elts)++;
  }

  *argv = NULL;
  return NULL;
}

static char *add_allow(cmd_rec *cmd)
{
  return _add_allow_deny(cmd,"Allow");
}

static char *add_deny(cmd_rec *cmd)
{
  return _add_allow_deny(cmd,"Deny");
}

static char *set_denyall(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_LIMIT|CONF_ANON);

  add_config_param("DenyAll",1,(void*)1);
  return NULL;
}

static char *set_allowall(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_LIMIT|CONF_ANON);

  add_config_param("AllowAll",1,(void*)1);
  return NULL;
}

static char *end_limit(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);
  CHECK_CONF(cmd,CONF_LIMIT);

  end_sub_config();
  return NULL;
}

static char *set_ignorehidden(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_LIMIT);

  c = add_config_param("IgnoreHidden",1,(void*)_get_boolean(cmd,1));
  return NULL;
}

static char *add_useralias(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,2);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  add_config_param_str("UserAlias",2,(void*)cmd->argv[1],(void*)cmd->argv[2]);

  return NULL;
}

static char *set_displaylogin(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON);

  c = add_config_param_str("DisplayLogin",1,(void*)cmd->argv[1]);
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *set_displayfirstchdir(cmd_rec *cmd)
{
  config_rec *c;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT|CONF_VIRTUAL|CONF_ANON|CONF_DIR);

  c = add_config_param_str("DisplayFirstChdir",1,(void*)cmd->argv[1]);
  c->flags |= CF_MERGEDOWN;
  return NULL;
}

static char *add_virtualhost(cmd_rec *cmd)
{
  server_rec *s;

  CHECK_ARGS(cmd,1);
  CHECK_CONF(cmd,CONF_ROOT);

  if(cmd->server != main_server)
    CONF_ERROR(cmd,"directive cannot be nested.");

  s = start_new_server(cmd->argv[1]);
  if(!s)
    CONF_ERROR(cmd,"unable to create virtual server configuration.");

  s->ServerPort = main_server->ServerPort;
  return NULL;
}

static char *end_virtualhost(cmd_rec *cmd)
{
  CHECK_ARGS(cmd,0);

  if(cmd->server == main_server)
    CONF_ERROR(cmd,"must be matched with <VirtualServer> directive.");

  end_new_server();
  return NULL;
}

static char *_get_full_cmd(cmd_rec *cmd)
{
  pool *p = cmd->tmp_pool;
  char *res = "";
  int i;

  if(cmd->arg)
    res = pstrcat(p,cmd->argv[0]," ",cmd->arg,NULL);
  else {
    for(i = 0; i < cmd->argc; i++)
      res = pstrcat(p,res,cmd->argv[i]," ",NULL);

    while(res[strlen(res)-1] == ' ')
      res[strlen(res)-1] = '\0';
  }

  return res;
}

static void _passive_cleanup(void *cv)
{
  conn_t *c = (conn_t*)cv;

  c->inf = c->outf = NULL;
  inet_close(c->pool,c);
}

/* io.c calls here when someone connects to our passive connection
 */

static int _passive_accept(IOREQ *req,char *buf,int bufsize)
{
  conn_t *c;
  /* Yes, this will block... Someone should recode it. :-) */
  
  inet_setblock(session.pool,session.d);
  c = inet_accept(session.pool,session.d,-1,-1,TRUE);

  if(c) {
    /* Make sure that the incoming connection's address matches
     * what we are expecting from our client
     */
    if(c->remote_ipaddr->s_addr == session.c->remote_ipaddr->s_addr) {
      /* Kill the old listener */
      
      /* We have to use a cleanup handler here, because if we
       * just inet_close()d the connection, destroy_pool would
       * cause all the subpools (including the request!) to 
       * get destroyed... BAD!
       */

      register_cleanup(req->file->pool,(void*)session.d,_passive_cleanup,
                       _passive_cleanup);

      io_close(req->file,IOR_REMOVE_FILE);
      inet_setnonblock(session.pool,c);
      session.d = c;
      io_setbuf(c->inf,0);
      io_setbuf(c->outf,0);

      session.d_req = NULL;
      return 0;
    }

    /* Otherwise something went wrong, and because we're in async mode
     * we can't inform the user, so just close the connection and
     * repost our data listener
     */
    inet_close(session.pool,c);
  }

  inet_resetlisten(session.pool,session.d);
  inet_setnonblock(session.pool,session.d);
  io_repost_req(req);
  return 0;
}
    
static void _passive_error(IOREQ *req, int err)
{
  inet_resetlisten(session.pool,session.d);
  inet_setnonblock(session.pool,session.d);
  io_repost_req(req);
  log_pri(LOG_WARNING,"Warning: error on listening data connection: %s",
          strerror(err));
}

static void _passive_close(IOREQ *req)
{
  register_cleanup(req->file->pool,(void*)session.d,
                   _passive_cleanup,_passive_cleanup);
  io_close(req->file,IOR_REMOVE_FILE);
  session.d = NULL;
  session.flags &= (SF_ALL^SF_PASSIVE);
  log_pri(LOG_WARNING,"Warning: passive listening data connection "
          "closed (shouldn't happen)");
}

/* Display a file via a given response numeric.  File is displayed
 * in psuedo "multiline" protocol, with a "-" after the numeric code,
 * followed by a text line.  0 is returned if the file was displayed
 * succesfully, otherwise -1 on error.
 */

int core_display_file(const char *numeric, const char *fn)
{
  FILE *fp;
  char buf[1024];
  int len,max;
  unsigned long fs_size = 0;
  pool *p;
  xaset_t *s;
  char *outs,*mg_time,mg_size[12],mg_max[12] = "unlimited";
  char mg_cur[12];

#if defined(HAVE_SYS_STATVFS_H) || defined(HAVE_SYS_VFS_H)
  fs_size = get_fs_size((char*)fn);
#endif

  if((fp = fopen(fn,"r")) == NULL)
    return -1;

  p = make_sub_pool(permanent_pool);

  s = (session.anon_config ? session.anon_config->subset : main_server->conf);

  mg_time = fmt_time(time(NULL));
  sprintf(mg_size,"%lu",fs_size);
  max = get_param_int(s,"MaxClients",FALSE);
  sprintf(mg_cur,"%d",(int)get_param_int(main_server->conf,
          "CURRENT-CLIENTS",FALSE)+1);

  if(max != -1)
    sprintf(mg_max,"%d",max);

  while(fgets(buf,sizeof(buf),fp) != NULL) {
    buf[1023] = '\0';

    len = strlen(buf);

    while(len && (buf[len-1] == '\r' || buf[len-1] == '\n')) {
      buf[len-1] = '\0';
      len--;
    }

    outs = sreplace(p,buf,
             "%T",mg_time,
             "%F",mg_size,
	     "%C",(session.cwd[0] ? session.cwd : "(none)"),
	     "%R",(session.c && session.c->remote_name ?
		   session.c->remote_name : "(unknown)"),
	     "%L",main_server->ServerAddress,
             "%u",session.ident_user,
	     "%U",(char*)get_param_ptr(main_server->conf,"USER",FALSE),
	     "%M",mg_max,
             "%N",mg_cur,
	     "%E",main_server->ServerAdmin,
             NULL);

    send_response_raw("%s-%s",numeric,outs);
  }

  fclose(fp);
  return 0;
}

static char *cmd_quit(cmd_rec *cmd)
{
  if(session.flags & SF_XFER)
    xfer_abort(NULL,0);

  send_response(R_221,"Goodbye.");
  log_pri(LOG_NOTICE,"FTP session closed.");
  end_login(0);
  return NULL;				/* Avoid compiler warning */
}

static char *cmd_pwd(cmd_rec *cmd)
{
  if(cmd->argc != 1) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  send_response(R_257,"\"%s\" is current directory.",session.cwd);
  return NULL;
}

static char *cmd_pasv(cmd_rec *cmd)
{
  union {
    in_addr_t addr;
    unsigned char u[4];
  } addr;

  union {
    short port;
    unsigned char u[2];
  } port;

  if(cmd->argc != 1) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  /* If we already have a passive listen data connection open,
   * kill it.
   */

  if(session.d) {
    inet_close(session.d->pool,session.d);
    session.d = NULL;
  }

  session.d = inet_create_connection(session.pool,NULL,-1,
                                     session.c->local_ipaddr,INPORT_ANY);

  if(!session.d)
     return R_425 " Unable to build data connection: Internal error.";

  inet_setnonblock(session.pool,session.d);
  inet_listen(session.pool,session.d,1);

  /* Even though the connection isn't open, we open an IOFILE for it
   * and post a read request so we can be informed when it connects.
   */

  session.d->inf = io_open(session.pool,session.d->listen_fd,IO_READ);
  io_setbuf(session.d->inf,0);

  session.d_req = io_post_req(session.d->inf,IO_READ,
                              _passive_accept,
                              _passive_error,
                              _passive_close);


  /* Now tell the client our address/port */
  session.data_port = session.d->local_port;
  session.flags |= SF_PASSIVE;

  addr.addr = *session.d->local_ipaddr;
  port.port = htons(session.data_port);


  log_debug(DEBUG1,"Entering passive mode (%u,%u,%u,%u,%u,%u)",
		(int)addr.u[0],(int)addr.u[1],(int)addr.u[2],
		(int)addr.u[3],(int)port.u[0],(int)port.u[1]);

  send_response(R_227, "Entering passive mode (%u,%u,%u,%u,%u,%u)",
                (int)addr.u[0],(int)addr.u[1],(int)addr.u[2],
                (int)addr.u[3],(int)port.u[0],(int)port.u[1]);

  return NULL;
}

static char *cmd_port(cmd_rec *cmd)
{
  union {
    in_addr_t addr;
    unsigned char u[4];
  } addr;

  union {
    short port;
    unsigned char u[2];
  } port;

  char *a,*endp,*arg;
  int i,cnt = 0;

  if(cmd->argc != 2) {
    send_response(R_500,"'%s' not understood.",_get_full_cmd(cmd));
    return NULL;
  }

  /* Format is h1,h2,h3,h4,p1,p2 (ASCII in network order) */
  a = pstrdup(cmd->tmp_pool,cmd->argv[1]);

  while(a && *a && cnt < 6) {
    arg = strsep(&a,",");

    if(!arg && a && *a) {
      arg = a;
      a = NULL;
    } else if(!arg)
      break;

    i = strtol(arg,&endp,10);
    if(*endp || i < 0 || i > 255)
      break;

    if(cnt < 4)
      addr.u[cnt++] = (unsigned char)i;
    else
      port.u[cnt++ - 4] = (unsigned char)i;
  }

  if(cnt != 6 || (a && *a))
    return R_501 " Illegal PORT command.";

  /* Make sure that the address specified matches the address
   * that the control connection is coming from.
   */

  if(addr.addr.s_addr != session.c->remote_ipaddr->s_addr ||
     !port.port)
    return R_501 " Illegal PORT command.";

  session.data_port = ntohs(port.port);
  session.flags &= (SF_ALL^SF_PASSIVE);

  return R_200 " PORT command successful.";
}

static char *cmd_help(cmd_rec *cmd)
{
  int i,c = 0;
  char buf[9];

  if(cmd->argc == 1) {
    /* Print help for all commands */
    char *outa[8];
    char *outs = "";

    bzero(outa,sizeof(outa));

    send_response_ml_start(R_214,
      "The following commands are recognized (* =>'s unimplemented).");
    for(i = 0; _help[i].cmd; i++) {

      if(_help[i].implemented)
        outa[c++] = _help[i].cmd;
      else
        outa[c++] = pstrcat(cmd->tmp_pool,_help[i].cmd,"*",NULL);

      /* 8 rows */
      if(((i+1) % 8 == 0) || !_help[i+1].cmd) {
        int j;

        for(j = 0; j < 8; j++) {
          if(outa[j]) {
            sprintf(buf,"%-8s",outa[j]);
            outs = pstrcat(cmd->tmp_pool,outs,buf,NULL);
          } else
            break;
        }

        if(*outs)
          send_response_ml(outs);
        outs = "";
        c = 0;
        bzero(outa,sizeof(outa));
      }
    }

    send_response_ml_end("Direct comments to %s.",
                         (cmd->server->ServerAdmin ? cmd->server->ServerAdmin :
                          "ftp-admin"));
  } else {
    char *cp;

    for(cp = cmd->argv[1]; *cp; cp++)
      *cp = toupper(*cp);

    if(!strcmp(cmd->argv[1],"SITE"))
      return (char*)call_module(&site_module,site_dispatch,cmd);

    for(i = 0; _help[i].cmd; i++)
      if(!strcasecmp(cmd->argv[1],_help[i].cmd)) {
        send_response(R_214,"Syntax: %s %s",cmd->argv[1],_help[i].syntax);
        return NULL;
      }

    send_response(R_502,"Unknown command '%s'.",cmd->argv[1]);
  }

  return NULL;
}

static char *cmd_syst(cmd_rec *cmd)
{
  return R_215 " UNIX Type: L8";
}

int core_chmod(cmd_rec *cmd, char *dir, int mode)
{
  if(!dir_check(cmd->tmp_pool,"SITE_CHMOD","WRITE",dir))
    return -1;

  return chmod(dir,(mode_t)mode);
}

static void _chdir(cmd_rec *cmd,char *ndir)
{
  char *display = NULL;
  char *dir;

  if(*ndir && *(ndir + strlen(ndir) - 1) != '/')
    dir = pstrcat(cmd->tmp_pool,ndir,"/",NULL);
  else
    dir = ndir;
 
  dir = dir_realpath(cmd->tmp_pool,dir);

  if(!dir || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,dir) ||
      chdir(dir) == -1) {
    send_response(R_550,"%s: %s",ndir,strerror(errno));
    return;
  }

  strncpy(session.cwd,dir,sizeof(session.cwd));
  dir_setcwd(session.cwd);

  if(session.dir_config)
    display = (char*)get_param_ptr(session.dir_config->subset,
                                   "DisplayFirstChdir",FALSE);
  if(!display && session.anon_config)
    display = (char*)get_param_ptr(session.anon_config->subset,
                                   "DisplayFirstChdir",FALSE);
  if(!display)
    display = (char*)get_param_ptr(cmd->server->conf,
                                   "DisplayFirstChdir",FALSE);

  if(display) {
    config_rec *c;
    time_t last;
    struct stat sbuf;

    c = find_config(cmd->server->conf,CONF_USERDATA,session.cwd,FALSE);

    if(!c) {
      time(&last);
      c = add_config_set(&cmd->server->conf,session.cwd);
      c->config_type = CONF_USERDATA;
      c->argc = 1;
      c->argv = pcalloc(c->pool,sizeof(void**) * 2);
      c->argv[0] = (void*)last;
      last = (time_t)0L;
    } else {
      last = (time_t)c->argv[0];
      c->argv[0] = (void*)time(NULL);
    }

    if(stat(display,&sbuf) != -1 && !S_ISDIR(sbuf.st_mode) &&
       sbuf.st_mtime > last)
      core_display_file(R_250,display);
  }

  send_response(R_250,"CWD command successful.");
}

static char *cmd_rmd(cmd_rec *cmd)
{
  char *dir;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  dir = dir_realpath(cmd->tmp_pool,cmd->arg);

  if(!dir || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,dir) ||
     rmdir(dir) == -1)
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
  else
    send_response(R_200,"%s command successful.",cmd->argv[0]);

  return NULL;
}

static char *cmd_mkd(cmd_rec *cmd)
{
  char *dir;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  if(strchr(cmd->arg,'*')) {
    send_response(R_550,"%s: Invalid directory name", cmd->argv[1]);
    return NULL;
  }

  dir = dir_canonical_path(cmd->tmp_pool,cmd->arg);


  if(!dir || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,dir) ||
     mkdir(dir,0777) == -1)
    send_response(R_550,"%s: %s",cmd->argv[1],strerror(errno));
  else {
    if(session.fsgid) {
      struct stat sbuf;

      stat(dir,&sbuf);
      if(chown(dir,(uid_t)-1,(gid_t)session.fsgid) == -1)
        log_pri(LOG_WARNING,"chown() failed: %s",strerror(errno));
      else
        chmod(dir,sbuf.st_mode);
    }
    send_response(R_200,"%s command successful.",cmd->argv[0]);
  }
    
  return NULL;
}

static char *cmd_cwd(cmd_rec *cmd)
{
  
  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  _chdir(cmd,cmd->arg);
  return NULL;
}

static char *cmd_cdup(cmd_rec *cmd)
{
  if(cmd->argc != 1) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  _chdir(cmd,"..");
  return NULL;
}

/* Returns the modification time of a file.  This is not in RFC959,
 * but supposedly will be in the future.  Command/response:
 * - MDTM <sp> path-name <crlf>
 * - 213 <sp> YYYYMMDDHHMMSS <crlf>
 *
 * We return the time as GMT, not localtime.  WU-ftpd returns localtime,
 * which seems like a Bad Thing<tm> to me.  However, my reasoning might
 * not be correct.
 */

static char *cmd_mdtm(cmd_rec *cmd)
{
  char *path;
  char buf[16];
  struct tm *tm;
  struct stat sbuf;
  
  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  path = dir_realpath(cmd->tmp_pool,cmd->arg);
  if(!path || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,path) ||
     stat(path,&sbuf) == -1)
    send_response(R_550,"%s: %s",cmd->argv[1],strerror(errno));
  else {
    if(!S_ISREG(sbuf.st_mode))
      send_response(R_550,"%s: not a plain file.",cmd->argv[1]);
    else {
      tm = gmtime(&sbuf.st_mtime);
      if(tm)
        sprintf(buf,"%04d%02d%02d%02d%02d%02d",
                tm->tm_year+1900,tm->tm_mon+1,tm->tm_mday,
                tm->tm_hour,tm->tm_min,tm->tm_sec);
      else
        sprintf(buf,"00000000000000");        
      send_response(R_213,buf);
    }
  }

  return NULL;
}

static char *cmd_size(cmd_rec *cmd)
{
  char *path;
  struct stat sbuf;
  unsigned long st_size;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  path = dir_realpath(cmd->tmp_pool,cmd->arg);

  if(!path || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,path) ||
      stat(path,&sbuf) == -1)
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
  else {
    if(!S_ISREG(sbuf.st_mode))
      send_response(R_550,"%s: not a regular file.",cmd->arg);
    else {
      st_size = sbuf.st_size;
      if(session.flags & (SF_ASCII|SF_ASCII_OVERRIDE)) {
        int fd,cnt;
	char buf[4096];

        if((fd = open(path,O_RDONLY)) != -1) {
          st_size = 0;
          while((cnt = read(fd,buf,sizeof(buf))) > 0) {
            st_size += cnt;
            while(cnt--)
              if(buf[cnt] == '\n') st_size++;
          }
          
          close(fd);
        }
      }

      send_response(R_213,"%lu",st_size);
    }
  }

  return NULL;
}

static char *cmd_dele(cmd_rec *cmd)
{
  char *path;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  path = dir_realpath(cmd->tmp_pool,cmd->arg);
  if(!path || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,path) ||
     unlink(path) == -1)
    send_response(R_550,"%s: %s",cmd->arg,strerror(errno));
  else
    send_response(R_200,"%s command successful.",cmd->argv[0]);
  return NULL;
}

static char *cmd_rnto(cmd_rec *cmd)
{
  char *path;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  if(!session.xfer.path) {
    if(session.xfer.p) {
      destroy_pool(session.xfer.p);
      bzero(&session.xfer,sizeof(session.xfer));
    }

    send_response(R_503,"Bad sequence of commands.");
    return NULL;
  }

  path = dir_canonical_path(cmd->tmp_pool,cmd->arg);

  if(!path || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,path) 
     || rename(session.xfer.path,path) == -1)
    send_response(R_550,"rename: %s",strerror(errno));
  else
    send_response(R_200,"rename successful.");

  destroy_pool(session.xfer.p);
  bzero(&session.xfer,sizeof(session.xfer));
  return NULL;
}

static char *cmd_rnfr(cmd_rec *cmd)
{
  char *path;

  if(cmd->argc < 2) {
    send_response(R_500,"'%s' is an unknown command.",_get_full_cmd(cmd));
    return NULL;
  }

  path = dir_realpath(cmd->tmp_pool,cmd->arg);

  if(!path || !dir_check(cmd->tmp_pool,cmd->argv[0],cmd->group,path) ||
     !exists(path))
    send_response(R_550,"%s: %s",cmd->argv[1],strerror(errno));
  else {
    /* We store the path in session.xfer.path */
    if(session.xfer.p) {
      destroy_pool(session.xfer.p);
      bzero(&session.xfer,sizeof(session.xfer));
    }

    session.xfer.p = make_sub_pool(session.pool);
    session.xfer.path = pstrdup(session.xfer.p,path);
    send_response(R_350,"File or directory exists, ready for destination name.");
  }

  return NULL;
}

static char *cmd_site(cmd_rec *cmd)
{
  char *cp;

  cmd->argc--;
  cmd->argv++;

  if(cmd->argc)
    for(cp = cmd->argv[0]; *cp; cp++)
      *cp = toupper(*cp);

  return (char*)call_module(&site_module,site_dispatch,cmd);
}

static char *cmd_noop(cmd_rec *cmd)
{
  return "200 NOOP command successful.";
}

/* Configuration directive table */

conftable core_conftable[] = {
  { "ServerName",		set_servername, 		NULL },
  { "ServerType",		set_servertype,			NULL },
  { "ServerAdmin",		set_serveradmin,		NULL },
  { "Port",			set_serverport, 		NULL },
  { "SocketBindTight",		set_socketbindtight,		NULL },
  { "tcpBackLog",		set_tcpbacklog,			NULL },
  { "tcpReceiveWindow",		set_tcpreceivewindow,		NULL },
  { "tcpSendWindow",		set_tcpsendwindow,		NULL },
  { "DeferWelcome",		set_deferwelcome,		NULL },
  { "DefaultServer",		set_defaultserver,		NULL },
  { "DefaultRoot",		add_defaultroot,		NULL },
  { "User",			set_user,			NULL },
  { "Group",			set_group, 			NULL },
  { "UserPassword",		add_userpassword,		NULL },
  { "GroupPassword",		add_grouppassword,		NULL },
  { "Umask",			set_umask,			NULL },
  { "MaxLoginAttempts",		set_maxloginattempts,		NULL },
  { "MaxClients",		set_maxclients,			NULL },
  { "RequireValidShell",	set_requirevalidshell,		NULL },
  { "ShowSymlinks",		set_showsymlinks,		NULL },
  { "TimeoutLogin",		set_timeoutlogin,		NULL },
  { "TimeoutIdle",		set_timeoutidle,		NULL },
  { "TimeoutNoTransfer",	set_timeoutnoxfer,		NULL },
  { "UseFtpUsers",		set_useftpusers,		NULL },
  { "AccessGrantMsg",		set_accessgrantmsg,		NULL },
  { "<VirtualHost>",		add_virtualhost,		NULL },
  { "</VirtualHost>",		end_virtualhost,		NULL },
  { "<Directory>",		add_directory,			NULL },
  { "HideNoAccess",		add_hidenoaccess,		NULL },
  { "HideUser",			add_hideuser,			NULL },
  { "HideGroup",		add_hidegroup,			NULL },
  { "GroupOwner",		add_groupowner,			NULL },
  { "AllowOverwrite",		set_allowoverwrite,		NULL },
  { "DisplayFirstChdir",	set_displayfirstchdir,		NULL },
  { "AllowRetrieveRestart",	set_allowretrieverestart,	NULL },
  { "AllowStoreRestart",	set_allowstorerestart,		NULL },
  { "</Directory>",		end_directory,			NULL },
  { "<Limit>",			add_limit,			NULL },
  { "IgnoreHidden",		set_ignorehidden,		NULL },
  { "Order",			add_order,			NULL },
  { "Allow",			add_allow,			NULL },
  { "Deny",			add_deny,			NULL },
  { "AllowAll",			set_allowall,			NULL },
  { "DenyAll",			set_denyall,			NULL },
  { "</Limit>", 		end_limit, 			NULL },
  { "DisplayLogin",		set_displaylogin,		NULL },
  { "<Anonymous>",		add_anonymous,			NULL },
  { "UserAlias",		add_useralias, 			NULL },
  { "AnonRequirePassword",	set_anonrequirepassword,	NULL },
  { "</Anonymous>",		end_anonymous,			NULL },
  { NULL, NULL, NULL }
};

cmdtable core_commands[] = {
  { C_HELP,	G_NONE,		cmd_help,	FALSE,	FALSE, 	NULL },
  { C_PORT,	G_NONE,		cmd_port,	TRUE,	FALSE,	NULL },
  { C_PASV,	G_NONE,		cmd_pasv,	TRUE,	FALSE,	NULL },
  { C_SYST,	G_NONE,		cmd_syst,	TRUE,	FALSE,	NULL },
  { C_PWD,	G_NONE,		cmd_pwd,	TRUE,	FALSE,	NULL },
  { C_XPWD,	G_NONE,		cmd_pwd,	TRUE,	FALSE,	NULL },
  { C_CWD,	G_DIRS,		cmd_cwd,	TRUE,	FALSE,	NULL },
  { C_XCWD,	G_DIRS,		cmd_cwd,	TRUE,	FALSE,	NULL },
  { C_MKD,	G_WRITE,	cmd_mkd,	TRUE,	FALSE,	NULL },
  { C_XMKD,	G_WRITE,	cmd_mkd,	TRUE,	FALSE,	NULL },
  { C_RMD,	G_WRITE,	cmd_rmd,	TRUE,	FALSE,	NULL },
  { C_XRMD,	G_WRITE,	cmd_rmd,	TRUE,	FALSE,	NULL },
  { C_CDUP,	G_NONE,		cmd_cdup,	TRUE,	FALSE,	NULL },
  { C_XCUP,	G_NONE,		cmd_cdup,	TRUE,	FALSE,	NULL },
  { C_SITE,	G_NONE,		cmd_site,	TRUE,	FALSE,	NULL },
  { C_DELE,	G_WRITE,	cmd_dele,	TRUE,	FALSE,	NULL },
  { C_MDTM,	G_DIRS,		cmd_mdtm,	TRUE,	FALSE,	NULL },
  { C_RNFR,	G_DIRS,		cmd_rnfr,	TRUE,	FALSE,	NULL },
  { C_RNTO,	G_WRITE,	cmd_rnto,	TRUE,	FALSE,	NULL },
  { C_SIZE,	G_READ,		cmd_size,	TRUE,	FALSE,  NULL },
  { C_QUIT,	G_NONE,		cmd_quit,	FALSE,	TRUE,	NULL },
  { C_NOOP,	G_NONE,		cmd_noop,	FALSE,	TRUE,	NULL },
  { NULL,	NULL,		NULL,		FALSE,	FALSE,	NULL }
};

/* Module interface */

module core_module = {
  NULL,NULL,			/* always NULL */
  0x10,				/* API Version 1.0 */
  "core",
  core_conftable,
  core_commands,
  NULL,NULL			/* No initialization needed */
};
