/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "conf.h"

#include <signal.h>

static xaset_t *reqs;
static int _inio = 0;			/* Prevent re-entrancy problems */

static void _iofile_cleanup(void *fv)
{
  IOFILE *f = (IOFILE*)fv;

  if(f->req && f->req->file == f)
    f->req->file = NULL;
}

static void _ioreq_cleanup(void *rv)
{
  IOREQ *r = (IOREQ*)rv;

  if(r->next || r->prev || (IOREQ*)reqs->xas_list == r)
    xaset_remove(reqs,(xasetmember_t*)r);
  if(r->file && r->file->req == r)
    r->file->req = NULL;
}

void init_io()
{
  reqs = xaset_create(permanent_pool,NULL);
  /* SIGPIPE signal is ignored, the _ioreq_service function handles
   * broken pipes.
   */
  signal(SIGPIPE,SIG_IGN);
}

IOFILE *io_create(pool *p)
{
  IOFILE *res;
  pool *newp;
 
  newp = make_sub_pool(p);
  res = (IOFILE*)pcalloc(newp,sizeof(IOFILE));
  res->pool = newp;
  res->fd = -1;
  res->bufsize = 4096;			/* Default */

  register_cleanup(newp,res,_iofile_cleanup,_iofile_cleanup);
  return res;
}

IOFILE *io_open(pool *p, int fd, int mode)
{
  IOFILE *res;

  res = io_create(p);

  res->fd = fd;
  res->mode = mode;
  return res;
}

IOFILE *io_reopen(IOFILE *f, int fd, int mode)
{
  if(f->fd != -1)
    close(f->fd);

  f->fd = fd;
  f->mode = mode;

  return f;
}

int io_close(IOFILE *f, int opt_flags)
{
  int res;

  res = close(f->fd);
  f->fd = -1;

  if(f->req) {
    if(_inio)
      f->req->req_flags |= (IOR_REMOVE|opt_flags);
    else {
      if((f->req->req_flags & IOR_REMOVE_FILE) ||
         (opt_flags & IOR_REMOVE_FILE))
        destroy_pool(f->pool);
      else
        destroy_pool(f->req->pool);
    }
  } else if(opt_flags & IOR_REMOVE_FILE)
    destroy_pool(f->pool);

  return res;
}

int io_setbuf(IOFILE *f, int bufsize)
{
  int old_bufsize = f->bufsize;
  f->bufsize = bufsize;

  return old_bufsize;
}

void io_kill_req(IOREQ *r)
{
  /* If in the service routine, set the IOR_REMOVE flag to avoid
   * re-entrancy problems.
   */

  if(_inio)
    r->req_flags |= IOR_REMOVE;
  else
    /* Cleanup handles removal from list, if necessary */
    destroy_pool(r->pool);

  return;
}

IOREQ *io_repost_req(IOREQ *r)
{
  if(r->bp == r->bufnext)
    r->bp = r->bufnext = r->buf;

  /* If in the service routine, simply clear req_flags and
   * _io_service will repost for us
   */

  if(_inio)
    r->req_flags = 0;
  else {
    if(!r->next && !r->prev && reqs->xas_list != (xasetmember_t*)r)
      xaset_insert(reqs,(xasetmember_t*)r);
  }

  return r;
}

IOREQ *io_post_req(IOFILE *f,int type,
                   int (*cl_io)(IOREQ*,char*,int),
                   void (*cl_err)(IOREQ*,int),
                   void (*cl_close)(IOREQ*))
{
  IOREQ *res;
  pool *p;

  p = make_sub_pool(f->pool);
  res = (IOREQ*)pcalloc(p,sizeof(IOREQ));

  res->pool = p;
  res->file = f;
  res->req_type = type;
  f->req = res;
  res->bufsize = f->bufsize;
  if(res->bufsize) {
    res->buf = (char*)palloc(p,res->bufsize);
    res->bp = res->buf;
    res->bufnext = res->buf;
  }

  res->cl_io = cl_io;
  res->cl_err = cl_err;
  res->cl_close = cl_close;

  register_cleanup(p,res,_ioreq_cleanup,_ioreq_cleanup);
  xaset_insert(reqs,(xasetmember_t*)res);
  return res;
}

void io_set_req_errno(IOREQ *req, int xerrno)
{
  req->req_flags |= IOR_ERROR;
  if(req->file)
    req->file->xerrno = xerrno;
}

/* timeout == -1 if no timeout
 * Returns NULL on error, and
 * empty set for timeout
 */

static xaset_t *_ioreq_block(pool *p, int timeout)
{
  fd_set rs,ws;
  struct timeval tv;
  IOREQ *r;
  xaset_t *res;
  int err;

 
  while(1) {
    FD_ZERO(&rs); FD_ZERO(&ws);

    /* _inio will take care of most cases, however an alarm might
     * trigger removal/addition of requests during a poll, so
     * this is critical code.
     */

    block_alarms();
    for(r = (IOREQ*)reqs->xas_list; r; r=r->next)
      if(!r->req_flags && r->file->fd != -1) {
        if(r->req_type == IO_READ)
          FD_SET(r->file->fd,&rs);
        else if(r->req_type == IO_WRITE)
          FD_SET(r->file->fd,&ws);
      }

    unblock_alarms();

    if(timeout != -1) {
      tv.tv_sec = timeout;
      tv.tv_usec = 0;

      err = select(NFDBITS,&rs,&ws,NULL,&tv);
    } else {
      if(schedulep()) {
        tv.tv_sec = 1;
        tv.tv_usec = 0;

        err = select(NFDBITS,&rs,&ws,NULL,&tv);
      } else
        err = select(NFDBITS,&rs,&ws,NULL,NULL);
    }

    if(err == -1) {
      if(errno == EINTR)
        continue;

      return NULL;
    }
      
    res = xaset_create(p,NULL);
    if(err > 0) {
      IOREQ *rnext;

      block_alarms();

      for(r = (IOREQ*)reqs->xas_list; r; r=rnext)
        if(!r->req_flags && r->file->fd != -1 &&
           ((r->req_type == IO_READ && FD_ISSET(r->file->fd,&rs)) ||
            (r->req_type == IO_WRITE && FD_ISSET(r->file->fd,&ws)))) {
              rnext = r->next;
              xaset_remove(reqs,(xasetmember_t*)r);
              xaset_insert(res,(xasetmember_t*)r);
            } else 
              rnext = r->next;

      unblock_alarms();

      /* If there is still nothing in the result set, the request
       * containing the fd we were selecting on has been removed else-
       * where so just continue with the main loop, and everything will
       * get re-built properly.
       */

      if(!res->xas_list)
        continue;
    }

    return res;
  }

  /* Avoid compiler warning */
  return NULL;
}

/*
 * Service all IOREQs, if block is non-zero, _ioreq_block will be
 * called with a timeout of -1, otherwise with 0 (return immediately).
 * NULL is returned if trying to reenter, or block is 0 and nothing
 * happened.
 */

static IOREQ *_ioreq_service(IOREQ *waiting_on, int block, int timeout)
{
  IOREQ *res = NULL,*r;
  xaset_t *s;
  pool *p;

  if(_inio)
    return NULL;

  _inio++;
  
  p = make_sub_pool(permanent_pool);
  while(!res) {
    s = _ioreq_block(p,(block ? timeout : 0));

    if(!s) {
      log_pri(LOG_ERR,"Internal error: _ioreq_block() failed: %s",
              strerror(errno));
      end_login(1);
    }

    if(!s->xas_list) {		/* Timeout */
      /* Run any schedules */
      run_schedule();

      if(!block)		/* Nothing happened during poll */
        break;
      else
        continue;
    }

    while((r = (IOREQ*)s->xas_list) != NULL) {
      if(r == waiting_on)
        res = r;

      xaset_remove(s,(xasetmember_t*)r);

      switch(r->req_type) {
      case IO_READ: {
        int bread = 0;
        int clread = 0;

        if(r->buf) {
          bread = read(r->file->fd,r->bufnext,r->bufsize - (r->bufnext-r->bp));

          if(bread > 0)
            r->bufnext += bread;

          if(bread > 0 && r->cl_io)
            clread = r->cl_io(r,r->bp,(r->bufnext-r->bp));
        } else
          clread = r->cl_io(r,NULL,0);

        if(clread > 0 && r->bp)
          r->bp += clread;
        else if(clread < 0)
          r->req_flags |= IOR_REMOVE;
  
        if(bread == 0 && r->buf) {
          /* EOF */
          r->req_flags |= IOR_CLOSED;
	  r->file->xerrno = 0;
	} else if(bread == -1) {
          switch(errno) {
          case EAGAIN:
            break;
          default:
            r->req_flags |= IOR_ERROR;
            r->file->xerrno = errno;
          }
        }

        break;
      }
      case IO_WRITE: {
        int bwrite;

        if(r->bp == r->bufnext) {
          r->bp = r->bufnext = r->buf;
          if(r->cl_io && (bwrite = r->cl_io(r,r->buf,r->bufsize)) > 0)
            r->bufnext += bwrite;
        }

        if(r->bp != r->bufnext) {
          bwrite = write(r->file->fd,r->bp,r->bufnext - r->bp);
          if(bwrite == -1) {
            switch(errno) {
            case EAGAIN:
              break;
            case EPIPE:
              r->req_flags |= IOR_CLOSED;
              r->file->xerrno = 0;
              break;
            default:
              r->req_flags |= IOR_ERROR;
              r->file->xerrno = errno;
            }
            bwrite = 0;
          }

          r->bp += bwrite;
        } else
          r->req_flags |= IOR_REMOVE;

        break;
      }
      }

      if((r->req_flags & IOR_CLOSED) && r->cl_close)
        r->cl_close(r);
      else if(r->req_flags & IOR_CLOSED)
        r->req_flags |= IOR_REMOVE;

      if((r->req_flags & IOR_ERROR) && r->cl_err)
        r->cl_err(r,r->file->xerrno);
      else if(r->req_flags & IOR_ERROR)
        r->req_flags |= IOR_REMOVE;

      if((r->req_flags & IOR_REMOVE) && res != r) {
        if(r->req_flags & IOR_REMOVE_FILE)
          destroy_pool(r->file->pool);
        else
          destroy_pool(r->pool);

      /* If not returning immediately to caller and no errors,
       * repost request for async read/writes
       */

      } else if(!r->req_flags && res != r) {
        if(r->bp == r->bufnext)
          r->bp = r->bufnext = r->buf;

        xaset_insert(reqs,(xasetmember_t*)r);
      }
    }
    if(!block)
      break;

    destroy_pool(p);
    p = make_sub_pool(permanent_pool);
  }

  destroy_pool(p);
  _inio--;
  return res;
}

static int _async_getdata(IOREQ *r, char *buf, int size)
{
  if(size > r->cl_bytes)
    size = r->cl_bytes;

  memcpy(buf,r->cl_bp,size);
  r->cl_bp += size;
  r->cl_bytes -= size;

  if(r->cl_bytes)
    io_repost_req(r);

  return size;
}

static int _async_putdata(IOREQ *r, char *buf, int size)
{
  if(size > r->cl_bytes)
    size = r->cl_bytes;

  memcpy(r->cl_bp,buf,size);
  r->cl_bp += size;
  r->cl_bytes -= size;

  if(r->cl_bytes)
    io_repost_req(r);
 
  return size;
}

static void _async_err(IOREQ *r, int err)
{
  r->file->xerrno = err;
  close(r->file->fd);
  r->file->fd = -1;
  io_kill_req(r);
}

static void _async_close(IOREQ *r)
{
  _async_err(r,0);
}

/* Low-level client functions.  DO NOT MIX ASYNC AND SYNC CALLS ON
 * A GIVEN IOFILE.
 */

IOREQ *io_read_async(IOFILE *f, char *buf, int size,
                   int (*cl_io)(IOREQ*,char*,int),
                   void (*cl_err)(IOREQ*,int),
                   void (*cl_close)(IOREQ*))
{
  IOREQ *req = NULL;

  if(f->fd == -1) {
    errno = (f->xerrno ? f->xerrno : EBADF);
    return NULL;
  }

  if(buf) {
    req = io_post_req(f,IO_READ,_async_putdata,_async_err,_async_close);

    req->cl_bp = req->cl_buf = buf;
    req->cl_bytes = size;
  } else
    req = io_post_req(f,IO_READ,cl_io,cl_err,cl_close);

  _ioreq_service(NULL,FALSE,-1);
  return req;
}

IOREQ *io_write_async(IOFILE *f, char *buf, int size,
                   int (*cl_io)(IOREQ*,char*,int),
                   void (*cl_err)(IOREQ*,int),
                   void (*cl_close)(IOREQ*))
{
  IOREQ *req = NULL;
  /* Two modes of operation, either user buffer is supplied,
   * or client functions
   */

  if(f->fd == -1) {
    errno = (f->xerrno ? f->xerrno : EBADF);
    return NULL;			/* Sanity check */
  }

  if(buf) {
    req = io_post_req(f,IO_WRITE,_async_getdata,_async_err,
                                     _async_close);
    req->cl_bp = req->cl_buf = palloc(req->pool,size);
    memcpy(req->cl_buf,buf,size);
    req->cl_bytes = size;
  } else
    req = io_post_req(f,IO_WRITE,cl_io,cl_err,cl_close);

  _ioreq_service(NULL,FALSE,-1);
  return req;  
}

int io_write(IOFILE *f, char *buf, int size)
{
  IOREQ *req,*r;
  int avail,trans = 0;

  if(f->fd == -1) {
    errno = (f->xerrno ? f->xerrno : EBADF);
    return 0;
  }

  if(f->req) {
    _ioreq_service(f->req,TRUE,-1);
    io_kill_req(f->req);
  }

  req = io_post_req(f,IO_WRITE,NULL,NULL,NULL);

  while(size) {
    avail = req->bufsize - (req->bufnext - req->buf);
    if(avail > size)
      avail = size;

    memcpy(req->buf,buf,avail);
    buf += avail;
    size -= avail;
    req->bufnext += avail;

    r = _ioreq_service(req,TRUE,-1);
    
    if(!r) {
      io_kill_req(r);
      f->xerrno = 0;
      errno = EAGAIN;
      return -1;
    }

    if(req->req_flags & IOR_CLOSED) {
      close(f->fd);
      f->fd = -1;
      errno = f->xerrno;
      io_kill_req(r);
      return trans;
    }

    if(req->req_flags & IOR_ERROR) {
      errno = f->xerrno;
      return -1;
    }

    trans += avail;
    
    if(size)
      io_repost_req(r);
    else
      io_kill_req(r);
  }

  errno = 0;
  return trans;
}

/* size - buffer size
 * min  - min. octets to read before returning (1 or >)
 */

int io_read(IOFILE *f, char *buf, int size, int min)
{
  IOREQ *req,*r;
  int avail,rcv = 0;

  if(f->fd == -1) {
    errno = (f->xerrno ? f->xerrno : EBADF);
    return 0;
  }

  if(min < 1)
    min = 1;
  if(min > size)
    min = size;

  if(f->req) {
    /* There is already a request present */
    req = f->req;
    rcv = req->bufnext - req->bp;

    if(rcv > size) rcv = size;
    memcpy(buf,req->bp,rcv);
    req->bp += rcv;
    buf += rcv;
    size -= rcv; min -= rcv;

    if(min > 0)
      io_repost_req(req);

  } else
    req = io_post_req(f,IO_READ,NULL,NULL,NULL);
  
  while(min > 0) {
 
    r = _ioreq_service(req,TRUE,-1);

    if(!r) {			/* Can't re-enter */
      f->xerrno = 0;
      errno = EAGAIN;
      return -1;
    }

    if(req->req_flags & IOR_CLOSED) {
      close(f->fd);
      f->fd = -1;
      errno = f->xerrno;
      return rcv;
    }

    if(req->req_flags & IOR_ERROR) {
      errno = f->xerrno;
      return -1;
    }

    avail = r->bufnext - r->bp;
    if(avail > size)
      avail = size;

    memcpy(buf,r->bp,avail);
    size -= avail; min -= avail; 
    r->bp += avail;
    rcv += avail;

    if(min > 0)
      io_repost_req(r);
  }

  errno = 0;
  return rcv;
}

/* io_yield is for yielding the process when waiting for I/O.  No need
 * to use this if using sync I/O (or a mixture), as the sync calls
 * will yield (block).
 */

void io_yield(int timeout)
{
  _ioreq_service(NULL,TRUE,timeout);
}


/* High-level functions, all of these are sync and potentially block
 * (except for the _raw function(s).  They should not be intermixed
 * with lower level async calls.
 */

int io_printf(IOFILE *f,char *fmt,...)
{
  va_list msg;
  char buf[1025];

  va_start(msg,fmt);
  vsnprintf(buf,sizeof(buf),fmt,msg);
  va_end(msg);

  buf[1024] = '\0';

  return io_write(f,buf,strlen(buf));
}

/* The raw printf bypasses the buffering and writes _directly_
 * to the underlying socket.  Don't use this except in emergency or
 * shutdown situations, as delivery is not guaraunteed in the slightes.
 */

int io_printf_raw(IOFILE *f, char *fmt, ...)
{
  va_list msg;
  char buf[1025];

  va_start(msg,fmt);
  vsnprintf(buf,sizeof(buf),fmt,msg);
  va_end(msg);

  buf[1024] = '\0';

  return write(f->fd,buf,strlen(buf));
}

char *io_gets(char *buf, int size, IOFILE *f)
{
  char *bp = buf;
  int toread;
  IOREQ *r;

  size--;

  /* Use the cl_buf/cl_bytes */
  if(!f->req)
    r = io_post_req(f,IO_READ,NULL,NULL,NULL);
  else
    r = f->req;

  if(!r->cl_buf) {
    r->cl_buf = palloc(f->req->pool,1024);
    r->cl_bytes = 1024;
  }
  
  while(size) {
    
    if(!r->cl_bp || r->cl_bytes == 1024) {		/* Empty buffer */
      toread = io_read(f,r->cl_buf,(size < 1024 ? size : 1024),1);
      if(toread <= 0)
        if(bp != buf) {
          *bp = '\0';
          return buf;
        } else
          return NULL;
      r->cl_bytes = 1024 - toread;
      r->cl_bp = r->cl_buf;
    } else
      toread = 1024 - r->cl_bytes;

    while(size && *r->cl_bp != '\n' && toread--) {
      if(*r->cl_bp & 0x80)		/* 7-bit ASCII only please */
        r->cl_bp++;
      else {
        *bp++ = *r->cl_bp++;
        size--;
      }
      r->cl_bytes++;
    }

    if(size && toread && *r->cl_bp == '\n') {
      size--; toread--;
      *bp++ = *r->cl_bp++;
      r->cl_bytes++;
      break;
    }

    if(!toread)
      r->cl_bp = NULL;
  }

  *bp = '\0';
  return buf;
}
