/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Libc seems to close /etc/passwd and /etc/group occasionally, even
 * when no endpwent()/endgrent() has been called.  This can't happen
 * during anon. logins, or uid/gid mapping ceases to function.
 * #define NEED_PERSISTANT_PASSWD in options.h to enable the use of
 * these functions.
 */

#include "conf.h"

#ifdef NEED_PERSISTANT_PASSWD

static FILE *pwdf = NULL;
static FILE *grpf = NULL;

#define PASSWD		"/etc/passwd"
#define GROUP		"/etc/group"

void p_setpwent()
{
  if(pwdf)
    rewind(pwdf);
  else
    pwdf = fopen(PASSWD,"r");
}

void p_endpwent()
{
  if(pwdf) {
    fclose(pwdf);
    pwdf = NULL;
  }
}

void p_setgrent()
{
  if(grpf)
    rewind(grpf);
  else
    grpf = fopen(GROUP,"r");
}

void p_endgrent()
{
  if(grpf) {
    fclose(grpf);
    grpf = NULL;
  }
}

struct passwd *p_getpwent()
{
  if(!pwdf)
    p_setpwent();

  return fgetpwent(pwdf);
}

struct group *p_getgrent()
{
  struct group *gr;

  if(!grpf)
    p_setgrent();

  gr = fgetgrent(grpf);

  return gr;
}

struct passwd *p_getpwnam(const char *name)
{
  struct passwd *pw;

  p_setpwent();
  while((pw = p_getpwent()) != NULL)
    if(!strcmp(name,pw->pw_name))
      break;

  return pw;
}

struct passwd *p_getpwuid(uid_t uid)
{
  struct passwd *pw;

  p_setpwent();
  while((pw = p_getpwent()) != NULL)
    if(pw->pw_uid == uid)
      break;

  return pw;
}

struct group *p_getgrnam(const char *name)
{
  struct group *gr;

  p_setgrent();
  while((gr = p_getgrent()) != NULL)
    if(!strcmp(name,gr->gr_name))
      break;

  return gr;
}

struct group *p_getgrgid(gid_t gid)
{
  struct group *gr;

  p_setgrent();
  while((gr = p_getgrent()) != NULL)
    if(gr->gr_gid == gid)
      break;

  return gr;
}

#endif /* NEED_PERSISTANT_PASSWD */
