/* -*- Mode: C; tab-width: 4 -*- */
/* life --- Conway's game of Life */

#if !defined( lint ) && !defined( SABER )
static const char sccsid[] = "@(#)life.c	4.06 97/10/27 xlockmore";

#endif

/*-
 * Copyright (c) 1991 by Patrick J. Naughton.
 * Copyright (c) 1997 by David Bagley.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Revision History:
 * 27-Oct-97: xpm and ras capability added.
 * 04-Jun-97: Removed old algorithm, now use wator's.  I could not
 *            understand it and had trouble adding more features.
 *            New algorithm is more efficient iff there lots of blank areas
 * 10-May-97: Compatible with xscreensaver
 * 07-May-97: life neighbor option.  Still have to fix -neighbor 3
 * 07-Jan-95: life now has a random soup pattern.
 * 07-Dec-94: life now has new organisms.  They are now better centered.
 *            Some of the nonperiodic forms were removed. New life forms
 *            were taken from xlife (an AMAZING collection of life forms).
 *            life's gliders now come from the edge of the screen except
 *            when generated by a life form.
 * 23-Nov-94: Bug fix for different iconified window sizes
 * 21-Jul-94: Took out bzero & bcopy since memset & memcpy is more portable
 * 10-Jun-94: Changed name of function 'kill', which is a libc function on
 *            many systems from Victor Langeveld <vic@mbfys.kun.nl>
 * Changes in original xlock
 * 24-May-91: Added wraparound code from johnson@bugs.comm.mot.com.
 *        Made old cells stay blue.
 *        Made batchcount control the number of generations until restart.
 * 29-Jul-90: support for multiple screens.
 * 07-Feb-90: remove bogus semi-colon after #include line.
 * 15-Dec-89: Fix for proper skipping of {White,Black}Pixel() in colors.
 * 08-Oct-89: Moved seconds() to an extern.
 * 20-Sep-89: Written, life algorithm courtesy of Jim Graham <flar@sun.com>
 */

/*-
  Grid     Number of Neigbors
  ----     ------------------
  Square   4 or 8
  Hexagon  6
  Triangle 3, 9, or 12

  Conway's Life: -neighbors 8 -rule 2333
  Other things to try:
    -neighbors 4 -rule 2422
    -neighbors 6 -rule 2333 -size 16
    -neighbors 3 -rule 1223 -size 16
*/

#ifdef STANDALONE
#define PROGCLASS "Life"
#define HACK_INIT init_life
#define HACK_DRAW draw_life
#define life_opts xlockmore_opts
#define DEFAULTS "*delay: 750000 \n" \
 "*count: 40 \n" \
 "*cycles: 140 \n" \
 "*size: 0 \n" \
 "*ncolors: 200 \n"
#define DEF_BITMAP ""
#define SPREAD_COLORS
#include "xlockmore.h"		/* in xscreensaver distribution */
#else /* STANDALONE */
#include "xlock.h"		/* in xlockmore distribution */

#endif /* STANDALONE */

#define DEF_RULE  "2333"

static int  rule;

static XrmOptionDescRec opts[] =
{
	{"-rule", ".life.rule", XrmoptionSepArg, (caddr_t) NULL},
};
static argtype vars[] =
{
	{(caddr_t *) & rule, "rule", "Rule", DEF_RULE, t_Int}
};
static OptionStruct desc[] =
{
   {"-rule num", "<living_min><living_max><birth_min><birth_max> parameters"}
};

ModeSpecOpt life_opts =
{1, opts, 1, vars, desc};

/* aliases for vars defined in the bitmap file */
#define CELL_WIDTH   image_width
#define CELL_HEIGHT    image_height
#define CELL_BITS    image_bits

#include "life.xbm"

#if defined( USE_XPM ) || defined( USE_XPMINC )
static char *image_name[] =
{""};

#define CELL_NAME image_name
#define DEFAULT_XPM 0
#endif

extern int  neighbors;
static int  local_neighbors = 0;

#define REDRAWSTEP 2000		/* How many cells to draw per cycle */
#define MINGRIDSIZE 20
#define MINSIZE 4
#define NEIGHBORKINDS 6
#define DEAD 0
#define LIVE 1
#define STATES 2

typedef struct {
	long        position;
	unsigned short age;
	unsigned char state;
	unsigned char toggle;
} cellstruct;

/* Singly linked list */
typedef struct _CellList {
	cellstruct  info;
	struct _CellList *previous, *next;
} CellList;

typedef struct {
	int         pattern;
	int         pixelmode;
	int         generation;
	int         xs, ys, xb, yb;	/* cell size, grid border */
	int         nrows, ncols, npositions;
	int         width, height;
	int         state;
	int         redrawing, redrawpos;
	int         ncells[STATES];
	CellList   *last[STATES], *first[STATES];
	CellList  **arr;
	XPoint      hexagonList[6];
	XPoint      triangleList[2][3];
	XImage     *logo;
	Colormap    cmap;
	unsigned long black, white;
	int         graphics_format;
	GC          backGC;
} lifestruct;

static int  initVal[NEIGHBORKINDS] =
{
	3, 4, 6, 8, 9, 12	/* Neighborhoods */
};

static unsigned char living_min, living_max, birth_min, birth_max;

static lifestruct *lifes = NULL;

#define NUMPTS  63

/*-
 * Patterns have < NUMPTS pts (and should have a size of <= 32x32,
 * the Glider Gun is an exception)
 */
static char patterns_2333[][2 * NUMPTS + 1] =
{
	{			/* GLIDER GUN */
		6, -4,
		5, -3, 6, -3,
		-6, -2, -5, -2, 8, -2, 9, -2, 16, -2,
		-7, -1, 8, -1, 9, -1, 10, -1, 16, -1, 17, -1,
		-18, 0, -17, 0, -8, 0, 8, 0, 9, 1,
		-17, 1, -8, 1, 5, 1, 6, 1,
		-8, 2, 6, 2,
		-7, 3,
		-6, 4, -5, 4,
		127
	},
	{			/* SHOWER TUB (PART OF OSCILATORS) */
		-3, -6, -2, -6, 2, -6, 3, -6,
		-4, -5, -2, -5, 2, -5, 4, -5,
		-4, -4, 4, -4,
		-7, -3, -6, -3, -4, -3, -3, -3, 3, -3, 4, -3, 6, -3, 7, -3,
		-7, -2, -6, -2, -4, -2, 0, -2, 4, -2, 6, -2, 7, -2,
		-4, -1, -2, -1, 2, -1, 4, -1,
		-4, 0, -2, 0, 2, 0, 4, 0,
		-5, 1, -4, 1, -2, 1, 2, 1, 4, 1, 5, 1,
		-4, 2, -1, 2, 0, 2, 1, 2, 4, 2,
		-4, 3, 4, 3, 6, 3,
		-3, 4, -2, 4, -1, 4, 5, 4, 6, 4,
		-1, 5,
		127
	},
	{			/* P4 LARGE TOASTER */
		-5, -3, -4, -3, -2, -3, 2, -3, 4, -3, 5, -3,
		-5, -2, 5, -2,
		-4, -1, -3, -1, 3, -1, 4, -1,
       -7, 0, -6, 0, -5, 0, -2, 0, -1, 0, 0, 0, 1, 0, 2, 0, 5, 0, 6, 0, 7, 0,
		-7, 1, -4, 1, 4, 1, 7, 1,
		-6, 2, -5, 2, 5, 2, 6, 2,
		127
	},
	{			/* STARGATE REPEATER P3 */
		0, -6,
		-1, -5, 1, -5,
		-2, -4, 0, -4, 2, -4,
		-2, -3, 2, -3,
		-4, -2, -3, -2, 0, -2, 3, -2, 4, -2,
		-5, -1, 0, -1, 5, -1,
		-6, 0, -4, 0, -2, 0, -1, 0, 1, 0, 2, 0, 4, 0, 6, 0,
		-5, 1, 0, 1, 5, 1,
		-4, 2, -3, 2, 0, 2, 3, 2, 4, 2,
		-2, 3, 2, 3,
		-2, 4, 0, 4, 2, 4,
		-1, 5, 1, 5,
		0, 6,
		127
	},
	{			/* OSCILLATOR 7 (P8) */
		-4, -2, -3, -2, -2, -2, -1, -2, 4, -2, 5, -2, 6, -2, 7, -2,
		-9, -1, -8, -1, 0, -1, 8, -1,
		-9, 0, -8, 0, -5, 0, -4, 0, 0, 0, 3, 0, 4, 0, 8, 0,
		-5, 1, -4, 1, -1, 1, 3, 1, 4, 1, 7, 1,
		127
	},
	{			/* P144 */
		-14, -9, -13, -9, 12, -9, 13, -9,
		-14, -8, -13, -8, 12, -8, 13, -8,
		4, -7, 5, -7,
		3, -6, 6, -6,
		4, -5, 5, -5,
		-1, -3, 0, -3, 1, -3,
		-1, -2, 1, -2,
		-1, -1, 0, -1, 1, -1,
		-1, 0, 0, 0,
		-2, 1, -1, 1, 0, 1,
		-2, 2, 0, 2,
		-2, 3, -1, 3, 0, 3,
		-6, 5, -5, 5,
		-7, 6, -4, 6,
		-6, 7, -5, 7,
		-14, 8, -13, 8, 12, 8, 13, 8,
		-14, 9, -13, 9, 12, 9, 13, 9,
		127
	},
	{			/* FIGURE EIGHT */
		-3, -3, -2, -3, -1, -3,
		-3, -2, -2, -2, -1, -2,
		-3, -1, -2, -1, -1, -1,
		0, 0, 1, 0, 2, 0,
		0, 1, 1, 1, 2, 1,
		0, 2, 1, 2, 2, 2,
		127
	},
	{			/* PULSAR 18-22-20 */
		0, -4, 1, -4, 2, -4,
		-1, -2, 4, -2,
		-2, -1, 0, -1, 4, -1,
		-4, 0, -3, 0, -2, 0, 1, 0, 4, 0,
		2, 1,
		0, 2, 1, 2,
		0, 3,
		0, 4,
		127
	},
	{			/* PULSAR 48-56-72 */
		-2, -1, -1, -1, 0, -1, 1, -1, 2, -1,
		-2, 0, 2, 0,
		127
	},
	{			/* BARBER POLE P2 */
		-6, -6, -5, -6,
		-6, -5, -4, -5,
		-4, -3, -2, -3,
		-2, -1, 0, -1,
		0, 1, 2, 1,
		2, 3, 4, 3,
		5, 4,
		4, 5, 5, 5,
		127
	},
	{			/* ACHIM P5 */
		-6, -6, -5, -6,
		-6, -5,
		-4, -4,
		-4, -3, -2, -3,
		-2, -1, 0, -1,
		0, 1, 2, 1,
		2, 3, 3, 3,
		5, 4,
		4, 5, 5, 5,
		127
	},
	{			/* HERTZ P4 */
		-2, -5, -1, -5,
		-2, -4, -1, -4,
		-7, -2, -6, -2, -2, -2, -1, -2, 0, -2, 1, -2, 5, -2, 6, -2,
		-7, -1, -5, -1, -3, -1, 2, -1, 4, -1, 6, -1,
		-5, 0, -3, 0, -2, 0, 2, 0, 4, 0,
		-6, 1, -5, 1, -3, 1, 2, 1, 4, 1, 5, 1,
		-2, 2, -1, 2, 0, 2, 1, 2,
		-2, 4, -1, 4,
		-2, 5, -1, 5,
		127
	},
	{			/* PUMP (TUMBLER, P14) */
		-2, -3, -1, -3, 1, -3, 2, -3,
		-2, -2, -1, -2, 1, -2, 2, -2,
		-1, -1, 1, -1,
		-3, 0, -1, 0, 1, 0, 3, 0,
		-3, 1, -1, 1, 1, 1, 3, 1,
		-3, 2, -2, 2, 2, 2, 3, 2,
		127
	},
	{			/* SMILEY (P8) */
		-3, -3, -2, -3, -1, -3, 1, -3, 2, -3, 3, -3,
		-2, -2, 0, -2, 2, -2,
		-2, 0, 2, 0,
		-3, 2, -1, 2, 1, 2, 3, 2,
		-1, 3, 1, 3,
		127
	},
	{			/* PULSE1 P4 */
		0, -3, 1, -3,
		-2, -2, 0, -2,
		-3, -1, 3, -1,
		-2, 0, 2, 0, 3, 0,
		0, 2, 2, 2,
		1, 3,
		127
	},
	{			/* SHINING FLOWER P5 */
		-1, -4, 0, -4,
		-2, -3, 1, -3,
		-3, -2, 2, -2,
		-4, -1, 3, -1,
		-4, 0, 3, 0,
		-3, 1, 2, 1,
		-2, 2, 1, 2,
		-1, 3, 0, 3,
		127
	},
	{			/* PULSE2 P6 */
		0, -4, 1, -4,
		-4, -3, -3, -3, -1, -3,
		-4, -2, -3, -2, 0, -2, 3, -2,
		1, -1, 3, -1,
		2, 0,
		1, 2, 2, 2,
		1, 3, 2, 3,
		127
	},
	{			/* PINWHEEL P4 */
		-2, -6, -1, -6,
		-2, -5, -1, -5,
		-2, -3, -1, -3, 0, -3, 1, -3,
		-3, -2, -1, -2, 2, -2, 4, -2, 5, -2,
		-3, -1, 1, -1, 2, -1, 4, -1, 5, -1,
		-6, 0, -5, 0, -3, 0, 0, 0, 2, 0,
		-6, 1, -5, 1, -3, 1, 2, 1,
		-2, 2, -1, 2, 0, 2, 1, 2,
		0, 4, 1, 4,
		0, 5, 1, 5,
		127
	},
	{			/* CLOCK P4 */
		-2, -6, -1, -6,
		-2, -5, -1, -5,
		-2, -3, -1, -3, 0, -3, 1, -3,
		-6, -2, -5, -2, -3, -2, 0, -2, 2, -2,
		-6, -1, -5, -1, -3, -1, -1, -1, 2, -1,
		-3, 0, -1, 0, 2, 0, 4, 0, 5, 0,
		-3, 1, 2, 1, 4, 1, 5, 1,
		-2, 2, -1, 2, 0, 2, 1, 2,
		0, 4, 1, 4,
		0, 5, 1, 5,
		127
	},
	{			/* CROSS P3 */
		-2, -4, -1, -4, 0, -4, 1, -4,
		-2, -3, 1, -3,
		-4, -2, -3, -2, -2, -2, 1, -2, 2, -2, 3, -2,
		-4, -1, 3, -1,
		-4, 0, 3, 0,
		-4, 1, -3, 1, -2, 1, 1, 1, 2, 1, 3, 1,
		-2, 2, 1, 2,
		-2, 3, -1, 3, 0, 3, 1, 3,
		127
	},
	{			/* BIG CROSS P3 */
		0, -5,
		-1, -4, 0, -4, 1, -4,
		-3, -3, -2, -3, -1, -3, 1, -3, 2, -3, 3, -3,
		-3, -2, 3, -2,
		-4, -1, -3, -1, 3, -1, 4, -1,
		-5, 0, -4, 0, 4, 0, 5, 0,
		-4, 1, -3, 1, 3, 1, 4, 1,
		-3, 2, 3, 2,
		-3, 3, -2, 3, -1, 3, 1, 3, 2, 3, 3, 3,
		-1, 4, 0, 4, 1, 4,
		0, 5,
		127
	},
	{			/* P4 DIAG SYM */
		-2, -4, 0, -4,
		-2, -3, 0, -3, 2, -3, 3, -3,
		-4, -2, -3, -2, 2, -2,
		0, -1, 1, -1, 2, -1,
		-4, 0, -3, 0, -1, 0, 2, 0,
		-1, 1, 2, 1,
		-3, 2, -2, 2, -1, 2, 0, 2, 1, 2,
		-3, 3,
		127
	},
	{			/* P4 ASYM */
		-4, -4, -2, -4,
		-4, -3, -1, -3,
		-1, -2,
		-2, -1, -1, -1, 0, -1, 3, -1, 4, -1, 5, -1,
		-5, 0, -4, 0, -3, 0, 0, 0, 1, 0, 2, 0,
		1, 1,
		1, 2, 4, 2,
		2, 3, 4, 3,
		127
	},
	{			/* P4 ASYM 2 */
		-3, -3, -1, -3, 2, -3, 4, -3, 5, -3, 6, -3,
		-4, -2, -1, -2, 1, -2, 3, -2, 5, -2,
		-4, -1,
		3, 0,
		-6, 1, -4, 1, -2, 1, 0, 1, 3, 1,
		-7, 2, -6, 2, -5, 2, -3, 2, 0, 2, 2, 2,
		127
	},
	{			/* P8 ASYM */
		-3, -4, -2, -4,
		-4, -3,
		-3, -2, 1, -2,
		-3, -1, 1, -1, 2, -1,
		-1, 0, 1, 0,
		-2, 1, -1, 1, 3, 1,
		-1, 2, 3, 2,
		4, 3,
		2, 4, 3, 4,
		127
	},
	{			/* P4 SYM */
		-6, -2, -5, -2, 4, -2, 5, -2,
		-6, -1, -5, -1, -3, -1, -2, -1, 1, -1, 2, -1, 4, -1, 5, -1,
		-5, 0, -2, 0, 1, 0, 4, 0,
		-5, 1, -4, 1, -2, 1, -1, 1, 0, 1, 1, 1, 3, 1, 4, 1,
		127
	},
	{			/* QUESTION P3 NOSYM */
		-2, -4, -1, -4, 0, -4,
		2, -3,
		-3, -2, 2, -2,
		1, -1,
		-2, 0, -1, 0,
		-2, 1,
		-2, 2,
		-2, 3,
		127
	},
	{			/* WHIRLY THING P12 */
		-5, -6,
		-5, -5, -4, -5, -3, -5, 5, -5, 6, -5,
		-2, -4, 5, -4,
		-3, -3, -2, -3, 3, -3, 5, -3,
		3, -2, 4, -2,
		0, -1, 1, -1,
		0, 0, 1, 0,
		0, 1, 1, 1,
		-4, 2, -3, 2,
		-5, 3, -3, 3, 2, 3, 3, 3,
		-5, 4, 2, 4,
		-6, 5, -5, 5, 3, 5, 4, 5, 5, 5,
		5, 6,
		127
	},
	{			/* PENTADECATHOLON P15 */
	     -5, 0, -4, 0, -3, 0, -2, 0, -1, 0, 0, 0, 1, 0, 2, 0, 3, 0, 4, 0,
		127
	},
	{			/* BALLOON P5 */
		-1, -3, 0, -3,
		-3, -2, 2, -2,
		-3, -1, 2, -1,
		-3, 0, 2, 0,
		-2, 1, 1, 1,
		-4, 2, -2, 2, 1, 2, 3, 2,
		-4, 3, -3, 3, 2, 3, 3, 3,
		127
	},
	{			/* FENCEPOST P12 */
		-11, -3, -9, -3, -7, -3,
	 -11, -2, -9, -2, -7, -2, 5, -2, 6, -2, 7, -2, 9, -2, 10, -2, 11, -2,
		-11, -1, -7, -1, -3, -1, 1, -1, 8, -1,
	  -10, 0, -8, 0, -3, 0, -2, 0, -1, 0, 1, 0, 5, 0, 6, 0, 10, 0, 11, 0,
		-11, 1, -7, 1, -3, 1, 1, 1, 8, 1,
		-11, 2, -9, 2, -7, 2, 5, 2, 6, 2, 7, 2, 9, 2, 10, 2, 11, 2,
		-11, 3, -9, 3, -7, 3,
		127
	},
	{			/* PISTON (SHUTTLE) P30 */
		1, -3, 2, -3,
		0, -2,
		-10, -1, -1, -1,
		-11, 0, -10, 0, -1, 0, 9, 0, 10, 0,
		-1, 1, 9, 1,
		0, 2,
		1, 3, 2, 3,
		127
	},
	{			/* P30 */
		-8, -5, 7, -5,
		-9, -4, -7, -4, 1, -4, 2, -4, 6, -4, 8, -4,
		-8, -3, 0, -3, 1, -3, 2, -3, 7, -3,
		1, -2, 2, -2,
		1, 2, 2, 2,
		-8, 3, 0, 3, 1, 3, 2, 3, 7, 3,
		-9, 4, -7, 4, 1, 4, 2, 4, 6, 4, 8, 4,
		-8, 5, 7, 5,
		127
	},
	{			/* PISTON2 P46 */
		-3, -5,
		-14, -4, -13, -4, -4, -4, -3, -4, 13, -4, 14, -4,
		-14, -3, -13, -3, -5, -3, -4, -3, 13, -3, 14, -3,
		-4, -2, -3, -2, 0, -2, 1, -2,
		-4, 2, -3, 2, 0, 2, 1, 2,
		-14, 3, -13, 3, -5, 3, -4, 3, 13, 3, 14, 3,
		-14, 4, -13, 4, -4, 4, -3, 4, 13, 4, 14, 4,
		-3, 5,
		127
	},
	{			/* GEARS (gear, flywheel, blinker) P2 */
		-1, -4,
		-1, -3, 1, -3,
		-3, -2,
		2, -1, 3, -1,
		-4, 0, -3, 0,
		2, 1,
		-2, 2, 0, 2,
		0, 3,

		5, 3,
		3, 4, 4, 4,
		5, 5, 6, 5,
		4, 6,

		8, 0,
		8, 1,
		8, 2,
		127
	},
	{			/* TURBINE8, KOK'S GALAXY */
		-4, -4, -3, -4, -2, -4, -1, -4, 0, -4, 1, -4, 3, -4, 4, -4,
		-4, -3, -3, -3, -2, -3, -1, -3, 0, -3, 1, -3, 3, -3, 4, -3,
		3, -2, 4, -2,
		-4, -1, -3, -1, 3, -1, 4, -1,
		-4, 0, -3, 0, 3, 0, 4, 0,
		-4, 1, -3, 1, 3, 1, 4, 1,
		-4, 2, -3, 2,
		-4, 3, -3, 3, -1, 3, 0, 3, 1, 3, 2, 3, 3, 3, 4, 3,
		-4, 4, -3, 4, -1, 4, 0, 4, 1, 4, 2, 4, 3, 4, 4, 4,
		127
	},
	{			/* P16 */
		-3, -6, 1, -6, 2, -6,
		-3, -5, 0, -5, 3, -5,
		3, -4,
		-5, -3, -4, -3, 1, -3, 2, -3, 5, -3, 6, -3,
		-6, -2, -3, -2,
		-6, -1, -3, -1,
		-5, 0, 5, 0,
		3, 1, 6, 1,
		3, 2, 6, 2,
		-6, 3, -5, 3, -2, 3, -1, 3, 4, 3, 5, 3,
		-3, 4,
		-3, 5, 0, 5, 3, 5,
		-2, 6, -1, 6, 3, 6,
		127
	},
	{			/* P28 (FLUTTER) */
		-9, -7, -7, -7, 7, -7, 9, -7,
		-6, -6, 6, -6,
		-10, -5, -7, -5, 7, -5, 10, -5,
		-11, -4, -9, -4, -7, -4, 7, -4, 9, -4, 11, -4,
		-11, -3, -8, -3, 8, -3, 11, -3,
		-10, -2, -9, -2, -4, -2, -3, -2, -2, -2,
		2, -2, 3, -2, 4, -2, 9, -2, 10, -2,
		-3, -1, 3, -1,
		-10, 1, -9, 1, 9, 1, 10, 1,
		-11, 2, -8, 2, 8, 2, 11, 2,
		-11, 3, -9, 3, -6, 3, 6, 3, 9, 3, 11, 3,
		-10, 4, 10, 4,
		-9, 5, -8, 5, -6, 5, 6, 5, 8, 5, 9, 5,
		-7, 6, 7, 6,
		127
	},
	{			/* P54 (PISTON3) */
		-14, -8, -13, -8, 13, -8, 14, -8,
		-13, -7, 13, -7,
		-13, -6, -11, -6, 11, -6, 13, -6,
		-12, -5, -11, -5, -1, -5, 11, -5, 12, -5,
		0, -4,
		-6, -3, -5, -3, 1, -3,
		-6, -2, -5, -2, -2, -2, 0, -2,
		-1, -1,
		-1, 1,
		-6, 2, -5, 2, -2, 2, 0, 2,
		-6, 3, -5, 3, 1, 3,
		0, 4,
		-12, 5, -11, 5, -1, 5, 11, 5, 12, 5,
		-13, 6, -11, 6, 11, 6, 13, 6,
		-13, 7, 13, 7,
		-14, 8, -13, 8, 13, 8, 14, 8,
		127
	},
	{			/* SWITCH ENGINE */
		-12, -3, -10, -3,
		-13, -2,
		-12, -1, -9, -1,
		-10, 0, -9, 0, -8, 0,
		13, 2, 14, 2,
		13, 3,
		127
	},
	{			/* PUFFER TRAIN */
		1, -9,
		2, -8,
		-2, -7, 2, -7,
		-1, -6, 0, -6, 1, -6, 2, -6,
		-2, -2,
		-1, -1, 0, -1,
		0, 0,
		0, 1,
		-1, 2,
		1, 5,
		2, 6,
		-2, 7, 2, 7,
		-1, 8, 0, 8, 1, 8, 2, 8,
		127
	},
	{			/* SCHOOL OF FISH (ESCORT) */
		3, -8,
		4, -7,
		-2, -6, 4, -6,
		-1, -5, 0, -5, 1, -5, 2, -5, 3, -5, 4, -5,
		-5, -1, -4, -1, -3, -1, -2, -1, -1, -1, 0, -1,
		1, -1, 2, -1, 3, -1, 4, -1, 5, -1, 6, -1,
		-6, 0, 6, 0,
		6, 1,
		5, 2,
		3, 4,
		4, 5,
		-2, 6, 4, 6,
		-1, 7, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7,

		127
	},
	{			/* DART SPEED 1/3 */
		3, -7,
		2, -6, 4, -6,
		1, -5, 2, -5,
		4, -4,
		0, -3, 4, -3,
		-3, -2, 0, -2,
		-4, -1, -2, -1, 1, -1, 2, -1, 3, -1, 4, -1,
		-5, 0, -2, 0,
		-4, 1, -2, 1, 1, 1, 2, 1, 3, 1, 4, 1,
		-3, 2, 0, 2,
		0, 3, 4, 3,
		4, 4,
		1, 5, 2, 5,
		2, 6, 4, 6,
		3, 7,
		127
	},
	{			/* PERIOD 4 SPEED 1/2 */
		-3, -5,
		-4, -4, -3, -4, -2, -4, -1, -4, 0, -4,
		-5, -3, -4, -3, 0, -3, 1, -3, 3, -3,
		-4, -2, 4, -2,
		-3, -1, -2, -1, 1, -1, 3, -1,
		-3, 1, -2, 1, 1, 1, 3, 1,
		-4, 2, 4, 2,
		-5, 3, -4, 3, 0, 3, 1, 3, 3, 3,
		-4, 4, -3, 4, -2, 4, -1, 4, 0, 4,
		-3, 5,
		127
	},
	{			/* ANOTHER PERIOD 4 SPEED 1/2 */
		-4, -7, -3, -7, -1, -7, 0, -7, 1, -7, 2, -7, 3, -7, 4, -7,
		-5, -6, -4, -6, -3, -6, -2, -6, 5, -6,
		-6, -5, -5, -5,
		-5, -4, 5, -4,
		-4, -3, -3, -3, -2, -3, 0, -3,
		-2, -2,
		-2, -1,
		-1, 0,
		-2, 1,
		-2, 2,
		-4, 3, -3, 3, -2, 3, 0, 3,
		-5, 4, 5, 4,
		-6, 5, -5, 5,
		-5, 6, -4, 6, -3, 6, -2, 6, 5, 6,
		-4, 7, -3, 7, -1, 7, 0, 7, 1, 7, 2, 7, 3, 7, 4, 7,
		127
	},
	{			/* SMALLEST KNOWN PERIOD 3 SPACESHIP SPEED 1/3 */
		0, -8,
		-1, -7, 1, -7,
		-1, -6, 1, -6,
		-1, -5,
		-2, -3, -1, -3,
		-1, -2, 1, -2,
		-2, -1, 0, -1,
		-2, 0, -1, 0, 0, 0,
		-1, 2, 1, 2,
		-1, 3, 0, 3,
		0, 4,
		0, 5, 2, 5,
		0, 6, 2, 6,
		1, 7,
		127
	},
	{			/* TURTLE SPEED 1/3 */
		-4, -5, -3, -5, -2, -5, 6, -5,
		-4, -4, -3, -4, 0, -4, 2, -4, 3, -4, 5, -4, 6, -4,
		-2, -3, -1, -3, 0, -3, 5, -3,
		-4, -2, -1, -2, 1, -2, 5, -2,
		-5, -1, 0, -1, 5, -1,
		-5, 0, 0, 0, 5, 0,
		-4, 1, -1, 1, 1, 1, 5, 1,
		-2, 2, -1, 2, 0, 2, 5, 2,
		-4, 3, -3, 3, 0, 3, 2, 3, 3, 3, 5, 3, 6, 3,
		-4, 4, -3, 4, -2, 4, 6, 4,
		127
	},
	{			/* SMALLEST KNOWN PERIOD 5 SPEED 2/5 */
		1, -7, 3, -7,
		-2, -6, 3, -6,
		-3, -5, -2, -5, -1, -5, 4, -5,
		-4, -4, -2, -4,
		-5, -3, -4, -3, -1, -3, 0, -3, 5, -3,
		-4, -2, -3, -2, 0, -2, 1, -2, 2, -2, 3, -2, 4, -2,
		-4, 2, -3, 2, 0, 2, 1, 2, 2, 2, 3, 2, 4, 2,
		-5, 3, -4, 3, -1, 3, 0, 3, 5, 3,
		-4, 4, -2, 4,
		-3, 5, -2, 5, -1, 5, 4, 5,
		-2, 6, 3, 6,
		1, 7, 3, 7,
		127
	},
	{			/* SYM PUFFER */
		1, -4, 2, -4, 3, -4, 4, -4,
		0, -3, 4, -3,
		4, -2,
		-4, -1, -3, -1, 0, -1, 3, -1,
		-4, 0, -3, 0, -2, 0,
		-4, 1, -3, 1, 0, 1, 3, 1,
		4, 2,
		0, 3, 4, 3,
		1, 4, 2, 4, 3, 4, 4, 4,
		127
	},
	{			/* RAKE P20 BACKWARDS */
		0, -10, 1, -10, 10, -10,
		-1, -9, 0, -9, 1, -9, 2, -9, 11, -9,
		-1, -8, 0, -8, 2, -8, 3, -8, 7, -8, 11, -8,
		1, -7, 2, -7, 8, -7, 9, -7, 10, -7, 11, -7,
		6, -3, 7, -3,
		5, -2, 6, -2, 8, -2, 9, -2,
		6, -1, 9, -1,
		6, 0, 9, 0,
		7, 1, 8, 1,
		10, 4,
		11, 5,
		-8, 6, 7, 6, 11, 6,
		-7, 7, 8, 7, 9, 7, 10, 7, 11, 7,
		-11, 8, -7, 8,
		-10, 9, -9, 9, -8, 9, -7, 9,
		127
	},
	{			/* RAKE P20 FORWARDS */
		0, -10, 1, -10, 10, -10,
		-1, -9, 0, -9, 1, -9, 2, -9, 11, -9,
		-1, -8, 0, -8, 2, -8, 3, -8, 7, -8, 11, -8,
		1, -7, 2, -7, 8, -7, 9, -7, 10, -7, 11, -7,
		6, -3, 7, -3,
		5, -2, 6, -2, 8, -2, 9, -2,
		6, -1, 9, -1,
		6, 0, 9, 0,
		7, 1, 8, 1,
		10, 4,
		11, 5,
		7, 6, 11, 6,
		-9, 7, -8, 7, -7, 7, -6, 7, 8, 7, 9, 7, 10, 7, 11, 7,
		-10, 8, -6, 8,
		-6, 9,
		-7, 10,
		127
	},
	{			/* RAKE P24 BACKWARDS */
		-5, -10,
		-4, -9,
		-10, -8, -4, -8,
		-9, -7, -8, -7, -7, -7, -6, -7, -5, -7, -4, -7,
		6, -6, 7, -6,
		5, -5, 6, -5, 7, -5, 8, -5,
		5, -4, 6, -4, 8, -4, 9, -4,
		7, -3, 8, -3,
		0, -2, 2, -2,
		-1, -1, 2, -1, 3, -1,
		0, 0, 2, 0,
		7, 1, 8, 1,
		5, 2, 6, 2, 8, 2, 9, 2,
		5, 3, 6, 3, 7, 3, 8, 3,
		6, 4, 7, 4,
		-5, 6, -4, 6,
		-9, 7, -8, 7, -7, 7, -6, 7, -4, 7, -3, 7,
		-9, 8, -8, 8, -7, 8, -6, 8, -5, 8, -4, 8,
		-8, 9, -7, 9, -6, 9, -5, 9,
		127
	},
	{			/* BIG GLIDER 1 */
		-4, -7, -3, -7,
		-4, -6, -2, -6,
		-4, -5,
		-7, -4, -6, -4, -4, -4,
		-7, -3, -2, -3,
		-7, -2, -5, -2, -4, -2, 3, -2, 4, -2, 5, -2,
		-2, -1, -1, -1, 0, -1, 5, -1, 6, -1,
		-1, 0, 0, 0, 1, 0, 3, 0, 5, 0,
		6, 1,
		-1, 2, 1, 2,
		-2, 3, -1, 3, 1, 3,
		-1, 4,
		-3, 5, -2, 5, 0, 5,
		0, 6,
		-2, 7, -1, 7,
		127
	},
	{			/* BIG GLIDER 2 */
		0, -9, 1, -9,
		0, -8, 2, -8,
		0, -7,
		1, -6, 2, -6, 3, -6,
		1, -5, 3, -5, 4, -5, 5, -5,
		1, -4, 4, -4, 5, -4,
		3, -3, 5, -3, 7, -3, 8, -3,
		2, -2, 3, -2, 5, -2, 7, -2,
		1, -1, 2, -1, 7, -1,
		-9, 0, -8, 0, -7, 0,
		-9, 1, -6, 1, -5, 1, -4, 1, -1, 1, 3, 1, 4, 1,
		-8, 2, -6, 2, -2, 2, -1, 2,
		-6, 3, -5, 3, -3, 3, -2, 3, 1, 3,
		-5, 4, -4, 4, 1, 4,
		-5, 5, -4, 5, -3, 5, -2, 5,
		-3, 7, -2, 7, -1, 7,
		-3, 8,
		127
	},
	{			/* BIG GLIDER 3 */
		-1, -8,
		-2, -7, -1, -7,
		-2, -6, 0, -6,
		0, -4, 2, -4,
		-2, -3, 0, -3, 2, -3,
		-7, -2, -3, -2, -2, -2, 0, -2, 2, -2, 3, -2,
		-8, -1, -7, -1, -6, -1, -5, -1, -2, -1, 4, -1,
		-8, 0, -5, 0, -4, 0, -2, 0, -1, 0, 3, 0, 4, 0,
		-6, 1, -5, 1, 0, 1,
		-4, 2, -3, 2, 1, 2,
		-3, 3, -2, 3, -1, 3, 2, 3, 4, 3, 5, 3, 6, 3, 7, 3,
		-3, 4, 0, 4, 3, 4, 4, 4, 5, 4,
		-1, 5, 0, 5, 1, 5,
		1, 6, 2, 6, 3, 6,
		2, 7, 3, 7,
		127
	},
	{			/* PI HEPTOMINO (], NEAR SHIP) */
		-2, -1, -1, -1, 0, -1,
		1, 0,
		-2, 1, -1, 1, 0, 1,
		127
	},
	{			/* R PENTOMINO */
		0, -1, 1, -1,
		-1, 0, 0, 0,
		0, 1,
		127
	},
	{			/* BUNNIES */
		-4, -2, 2, -2,
		-2, -1, 2, -1,
		-2, 0, 1, 0, 3, 0,
		-3, 1, -1, 1,
		127
	}
};

#define NPATS_2333	(sizeof patterns_2333 / sizeof patterns_2333[0])

static int
position_of_neighbor(lifestruct * lp, int n, int col, int row)
{
	int         dir = n * 360 / local_neighbors;

	if (local_neighbors == 6) {
		switch (dir) {
			case 0:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				break;
			case 60:
				if (!(row & 1))
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 120:
				if (row & 1)
					col = (!col) ? lp->ncols - 1 : col - 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 180:
				col = (!col) ? lp->ncols - 1 : col - 1;
				break;
			case 240:
				if (row & 1)
					col = (!col) ? lp->ncols - 1 : col - 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 300:
				if (!(row & 1))
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n", dir);
		}
	} else if (local_neighbors == 4 || local_neighbors == 8) {
		switch (dir) {
			case 0:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				break;
			case 45:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 90:
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 135:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 180:
				col = (!col) ? lp->ncols - 1 : col - 1;
				break;
			case 225:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 270:
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 315:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n", dir);
		}
	} else {		/* TRI */
		if ((col + row) % 2) {	/* right */
			switch (dir) {
				case 0:
					col = (!col) ? lp->ncols - 1 : col - 1;
					break;
				case 30:
				case 40:
					col = (!col) ? lp->ncols - 1 : col - 1;
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				case 60:
					col = (!col) ? lp->ncols - 1 : col - 1;
					if (row + 1 == lp->nrows)
						row = 1;
					else if (row + 2 == lp->nrows)
						row = 0;
					else
						row = row + 2;
					break;
				case 80:
				case 90:
					if (row + 1 == lp->nrows)
						row = 1;
					else if (row + 2 == lp->nrows)
						row = 0;
					else
						row = row + 2;
					break;
				case 120:
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				case 150:
				case 160:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				case 180:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					break;
				case 200:
				case 210:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				case 240:
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				case 270:
				case 280:
					if (!row)
						row = lp->nrows - 2;
					else if (!(row - 1))
						row = lp->nrows - 1;
					else
						row = row - 2;
					break;
				case 300:
					col = (!col) ? lp->ncols - 1 : col - 1;
					if (!row)
						row = lp->nrows - 2;
					else if (!(row - 1))
						row = lp->nrows - 1;
					else
						row = row - 2;
					break;
				case 320:
				case 330:
					col = (!col) ? lp->ncols - 1 : col - 1;
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				default:
					(void) fprintf(stderr, "wrong direction %d\n", dir);
			}
		} else {	/* left */
			switch (dir) {
				case 0:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					break;
				case 30:
				case 40:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				case 60:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					if (!row)
						row = lp->nrows - 2;
					else if (row == 1)
						row = lp->nrows - 1;
					else
						row = row - 2;
					break;
				case 80:
				case 90:
					if (!row)
						row = lp->nrows - 2;
					else if (row == 1)
						row = lp->nrows - 1;
					else
						row = row - 2;
					break;
				case 120:
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				case 150:
				case 160:
					col = (!col) ? lp->ncols - 1 : col - 1;
					row = (!row) ? lp->nrows - 1 : row - 1;
					break;
				case 180:
					col = (!col) ? lp->ncols - 1 : col - 1;
					break;
				case 200:
				case 210:
					col = (!col) ? lp->ncols - 1 : col - 1;
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				case 240:
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				case 270:
				case 280:
					if (row + 1 == lp->nrows)
						row = 1;
					else if (row + 2 == lp->nrows)
						row = 0;
					else
						row = row + 2;
					break;
				case 300:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					if (row + 1 == lp->nrows)
						row = 1;
					else if (row + 2 == lp->nrows)
						row = 0;
					else
						row = row + 2;
					break;
				case 320:
				case 330:
					col = (col + 1 == lp->ncols) ? 0 : col + 1;
					row = (row + 1 == lp->nrows) ? 0 : row + 1;
					break;
				default:
					(void) fprintf(stderr, "wrong direction %d\n", dir);
			}
		}
	}

	return (row * lp->ncols + col);
}

static void
init_list(lifestruct * lp, int state)
{
	/* Waste some space at the beginning and end of list
	   so we do not have to complicated checks against falling off the ends. */
	lp->last[state] = (CellList *) malloc(sizeof (CellList));
	lp->first[state] = (CellList *) malloc(sizeof (CellList));
	lp->first[state]->previous = lp->last[state]->next = NULL;
	lp->first[state]->next = lp->last[state]->previous = NULL;
	lp->first[state]->next = lp->last[state];
	lp->last[state]->previous = lp->first[state];
}

static void
addto_list(lifestruct * lp, int state, cellstruct info)
{
	CellList   *curr;

	curr = (CellList *) malloc(sizeof (CellList));
	lp->last[state]->previous->next = curr;
	curr->previous = lp->last[state]->previous;
	curr->next = lp->last[state];
	lp->last[state]->previous = curr;
	curr->info = info;
	if (info.position >= 0) {
		lp->arr[info.position] = curr;
		lp->ncells[state]++;
	}
}

static void
removefrom_list(lifestruct * lp, int state, CellList * curr)
{
	curr->previous->next = curr->next;
	curr->next->previous = curr->previous;
	if (curr->info.position >= 0) {
		lp->arr[curr->info.position] = NULL;
		lp->ncells[state]--;
	}
	(void) free((void *) curr);
}

#ifdef DEBUG
static void
print_state(ModeInfo * mi, int state)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	CellList   *curr;
	int         i = 0;

	curr = lp->first[state]->next;
	(void) printf("state %d\n", state);
	while (curr != lp->last[state]) {
		(void) printf("%d: position %ld,	age %d, state %d, toggle %d\n",
			      i, curr->info.position, curr->info.age,
			      curr->info.state, curr->info.toggle);
		curr = curr->next;
		i++;
	}
}

#endif

static void
flush_list(lifestruct * lp, int state)
{
	while (lp->last[state]->previous != lp->first[state]) {
		CellList   *curr = lp->last[state]->previous;

		curr->previous->next = lp->last[state];
		lp->last[state]->previous = curr->previous;
		(void) free((void *) curr);
	}
	lp->ncells[state] = 0;
}


static void
draw_cell(ModeInfo * mi, cellstruct info)
{
	Display    *display = MI_DISPLAY(mi);
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	GC          gc = lp->backGC;
	int         col, row;

	col = info.position % lp->ncols;
	row = info.position / lp->ncols;
	if (info.state == LIVE) {
		if (MI_NPIXELS(mi) > 2)
			XSetForeground(display, gc, MI_PIXEL(mi, info.age));
		else
			XSetForeground(display, gc, MI_WIN_WHITE_PIXEL(mi));
	} else
		XSetForeground(display, gc, lp->black);

	if (local_neighbors == 6) {
		int         ccol = 2 * col + !(row & 1), crow = 2 * row;

		lp->hexagonList[0].x = lp->xb + ccol * lp->xs;
		lp->hexagonList[0].y = lp->yb + crow * lp->ys;
		if (lp->xs == 1 && lp->ys == 1)
			XFillRectangle(MI_DISPLAY(mi), MI_WINDOW(mi),
				       lp->backGC, lp->hexagonList[0].x, lp->hexagonList[0].y, 1, 1);
		else
			XFillPolygon(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
			      lp->hexagonList, 6, Convex, CoordModePrevious);
	} else if (local_neighbors == 4 || local_neighbors == 8) {
		if (lp->pixelmode || info.state == DEAD)
			XFillRectangle(display, MI_WINDOW(mi), gc,
				       lp->xb + lp->xs * col, lp->yb + lp->ys * row, lp->xs, lp->ys);
		else
/*-
 * PURIFY 4.0.1 on SunOS4 and on Solaris 2 reports a 132 byte memory leak on
 * the next line */
			XPutImage(display, MI_WINDOW(mi), gc, lp->logo,
			  0, 0, lp->xb + lp->xs * col, lp->yb + lp->ys * row,
				  lp->logo->width, lp->logo->height);
	} else {		/* TRI */
		int         orient = (col + row) % 2;	/* O left 1 right */

		lp->triangleList[orient][0].x = lp->xb + col * lp->xs;
		lp->triangleList[orient][0].y = lp->yb + row * lp->ys;
		if (lp->xs <= 3 || lp->ys <= 3)
			XFillRectangle(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
			 ((orient) ? -1 : 1) + lp->triangleList[orient][0].x,
				       lp->triangleList[orient][0].y, 1, 1);
		else {
			if (orient)
				lp->triangleList[orient][0].x += (lp->xs / 2 - 1);
			else
				lp->triangleList[orient][0].x -= (lp->xs / 2 - 1);
			XFillPolygon(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
				     lp->triangleList[orient], 3, Convex, CoordModePrevious);

		}
	}
}

static void
setcelltoggles(ModeInfo * mi, int col, int row)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         position;
	CellList   *curr;

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	if (!curr) {
		(void) fprintf(stderr, "state toggling but not on list\n");
		return;
	}
	curr->info.toggle = True;
}

static void
setcellfromtoggle(ModeInfo * mi, int col, int row)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         neighbor, n, position;
	cellstruct  info;
	CellList   *curr, *currn;

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	if ((curr && curr->info.state == DEAD && curr->info.toggle) ||
	    (curr && curr->info.state == LIVE && !curr->info.toggle)) {
		for (n = 0; n < local_neighbors; n++) {
			neighbor = position_of_neighbor(lp, n, col, row);
			currn = lp->arr[neighbor];
			if (!currn) {
				info.position = neighbor;
				info.age = 0;
				info.state = DEAD;
				info.toggle = False;
				addto_list(lp, DEAD, info);
			}
		}
	}
	if (curr && curr->info.state == DEAD && curr->info.toggle) {
		removefrom_list(lp, DEAD, curr);
		info.age = 0;
		info.position = position;
		info.toggle = False;
		info.state = LIVE;
		addto_list(lp, LIVE, info);
		draw_cell(mi, info);
	} else if (curr && curr->info.state == LIVE && !curr->info.toggle) {
		info = curr->info;
		/* if we aren't up to blue yet, then keep aging the cell. */
		if ((MI_NPIXELS(mi) > 2) &&
		    (info.age < (unsigned short) (MI_NPIXELS(mi) * 0.7))) {
			++(info.age);
			curr->info.age = info.age;
			draw_cell(mi, info);
		}
	}
}

static void
setcell(ModeInfo * mi, int col, int row, int state)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         neighbor, n, position;
	cellstruct  info;
	CellList   *curr, *currn;

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	if (state == LIVE) {
		if (curr && curr->info.state == DEAD) {
			removefrom_list(lp, DEAD, curr);
			curr = NULL;
		}
		if (!curr) {
			for (n = 0; n < local_neighbors; n++) {
				neighbor = position_of_neighbor(lp, n, col, row);
				currn = lp->arr[neighbor];
				if (!currn) {
					info.age = 0;
					info.position = neighbor;
					info.state = DEAD;
					info.toggle = False;
					addto_list(lp, DEAD, info);
				}
			}
			info.age = 0;
			info.position = position;
			info.state = LIVE;
			info.toggle = False;
			addto_list(lp, LIVE, info);
			draw_cell(mi, info);
		} else {
			info = curr->info;
			info.age = 0;
			draw_cell(mi, info);
		}
	} else if (curr && curr->info.state == LIVE) {
		info.age = 0;
		info.position = position;
		info.state = DEAD;
		info.toggle = False;
		removefrom_list(lp, LIVE, curr);
		addto_list(lp, DEAD, info);	/* Just in case... */
		draw_cell(mi, info);
	}
}

static void
alloc_cells(lifestruct * lp)
{
	lp->arr = (CellList **) calloc(lp->npositions, sizeof (CellList *));
}

static void
free_cells(lifestruct * lp)
{
	if (lp->arr != NULL)
		(void) free((void *) lp->arr);
	lp->arr = NULL;
}

static int
n_neighbors(lifestruct * lp, CellList * curr)
{
	int         col, row, n, p, count = 0;;

	col = curr->info.position % lp->ncols;
	row = curr->info.position / lp->ncols;
	for (n = 0; n < local_neighbors; n++) {
		p = position_of_neighbor(lp, n, col, row);
		if (lp->arr[p] && lp->arr[p]->info.state == LIVE)
			count++;
	}
	return count;
}

static void
RandomSoup(ModeInfo * mi, int n, int v)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         row, col;

	v /= 2;
	if (v < 1)
		v = 1;
	for (row = lp->nrows / 2 - v; row < lp->nrows / 2 + v; ++row)
		for (col = lp->ncols / 2 - v; col < lp->ncols / 2 + v; ++col)
			if (NRAND(100) < n && col > 1 && row > 1 &&
			    col < lp->ncols && row < lp->nrows)
				setcell(mi, col, row, LIVE);
	if (MI_WIN_IS_VERBOSE(mi))
		(void) fprintf(stdout, "random pattern\n");
}

static void
GetPattern(ModeInfo * mi, int i)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         row, col;
	char       *patptr;

	patptr = &patterns_2333[i][0];
	while ((col = *patptr++) != 127) {
		row = *patptr++;
		col += lp->ncols / 2;
		row += lp->nrows / 2;
		if (col >= 0 && row >= 0 && col < lp->ncols && row < lp->nrows)
			setcell(mi, col, row, LIVE);
	}
	if (MI_WIN_IS_VERBOSE(mi))
		(void) fprintf(stdout, "table number %d\n", i);
}

static void
shooter(ModeInfo * mi)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         hsp, vsp, hoff = 1, voff = 1;

	/* Generate the glider at the edge of the screen */
	if (LRAND() & 1) {
		hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
		vsp = NRAND(lp->nrows);
	} else {
		vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
		hsp = NRAND(lp->ncols);
	}
	if (vsp > lp->nrows / 2)
		voff = -1;
	if (hsp > lp->ncols / 2)
		hoff = -1;
	if (local_neighbors == 8 && rule == 2333) {
		setcell(mi, hsp + 2 * hoff, vsp + 0 * voff, LIVE);
		setcell(mi, hsp + 2 * hoff, vsp + 1 * voff, LIVE);
		setcell(mi, hsp + 2 * hoff, vsp + 2 * voff, LIVE);
		setcell(mi, hsp + 1 * hoff, vsp + 2 * voff, LIVE);
		setcell(mi, hsp + 0 * hoff, vsp + 1 * voff, LIVE);
	}
}

static void
init_stuff(ModeInfo * mi)
{
	Display    *display = MI_DISPLAY(mi);
	Window      window = MI_WINDOW(mi);
	lifestruct *lp = &lifes[MI_SCREEN(mi)];

	if (!lp->logo)
		getImage(mi, &lp->logo, CELL_WIDTH, CELL_HEIGHT, CELL_BITS,
#if defined( USE_XPM ) || defined( USE_XPMINC )
			 DEFAULT_XPM, CELL_NAME,
#endif
			 &lp->graphics_format, &lp->cmap,
			 &lp->black, &lp->white);
	if (lp->cmap != None) {
		setColormap(display, window, lp->cmap, MI_WIN_IS_INWINDOW(mi));
		if (lp->backGC == None) {
			XGCValues   xgcv;

			xgcv.background = lp->black;
			lp->backGC = XCreateGC(display, window, GCBackground, &xgcv);
		}
	} else {
		lp->black = MI_WIN_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
	}
}

static void
free_stuff(Display * display, lifestruct * lp)
{
	if (lp->cmap != None) {
		XFreeColormap(display, lp->cmap);
		if (lp->backGC != None) {
			XFreeGC(display, lp->backGC);
			lp->backGC = None;
		}
		lp->cmap = None;
	} else
		lp->backGC = None;
	destroyImage(&lp->logo, &lp->graphics_format);
}

void
init_life(ModeInfo * mi)
{
	lifestruct *lp;
	int         size = MI_SIZE(mi), npats, i;

	if (lifes == NULL) {
		if ((lifes = (lifestruct *) calloc(MI_NUM_SCREENS(mi),
					       sizeof (lifestruct))) == NULL)
			return;
	}
	lp = &lifes[MI_SCREEN(mi)];

	lp->generation = 0;
	lp->redrawing = 0;

	if (!local_neighbors) {
		for (i = 0; i < NEIGHBORKINDS; i++) {
			if (neighbors == initVal[i]) {
				local_neighbors = neighbors;
				break;
			}
			if (i == NEIGHBORKINDS - 1) {
#if 0
				local_neighbors = initVal[NRAND(NEIGHBORKINDS)];
				local_neighbors = (LRAND() & 1) ? 4 : 8;
#else
				local_neighbors = 8;
#endif
				break;
			}
		}
		/*
		 * Rules should not have to be contiguous ...  that is, it should be
		 * possible to have rules where cells will live if there are 2 or 4
		 * (not 3) cells around them.   These rules are not general enough
		 * to allow for that.
		 */
		if (rule < 0)
			rule = -rule;
		living_min = rule / 1000;
		living_max = (rule / 100) % 10;
		birth_min = (rule / 10) % 10;
		birth_max = rule % 10;
		if (MI_WIN_IS_VERBOSE(mi))
			(void) fprintf(stdout, "neighbors %d, rule %d\n", local_neighbors, rule);
	}
	lp->width = MI_WIN_WIDTH(mi);
	lp->height = MI_WIN_HEIGHT(mi);

	if (lp->first[0])
		for (i = 0; i < STATES; i++)
			flush_list(lp, i);
	else
		for (i = 0; i < STATES; i++)
			init_list(lp, i);
	free_cells(lp);

	if (local_neighbors == 6) {
		int         nccols, ncrows, i;

		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 4)
			lp->height = 4;
		if (size < -MINSIZE)
			lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				      MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size)
				lp->ys = MAX(MINSIZE, MIN(lp->width, lp->height) / MINGRIDSIZE);
			else
				lp->ys = MINSIZE;
		} else
			lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
					       MINGRIDSIZE));
		lp->xs = lp->ys;
		nccols = MAX(lp->width / lp->xs - 2, 2);
		ncrows = MAX(lp->height / lp->ys - 1, 2);
		lp->ncols = nccols / 2;
		lp->nrows = 2 * (ncrows / 4);
		lp->xb = (lp->width - lp->xs * nccols) / 2 + lp->xs / 2;
		lp->yb = (lp->height - lp->ys * (ncrows / 2) * 2) / 2 + lp->ys;
		for (i = 0; i < 6; i++) {
			lp->hexagonList[i].x = (lp->xs - 1) * hexagonUnit[i].x;
			lp->hexagonList[i].y = ((lp->ys - 1) * hexagonUnit[i].y / 2) * 4 / 3;
		}
		lp->black = MI_WIN_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
	} else if (local_neighbors == 4 || local_neighbors == 8) {
		init_stuff(mi);
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 2)
			lp->height = 2;
		if (size == 0 ||
		    MINGRIDSIZE * size > lp->width || MINGRIDSIZE * size > lp->height) {
			if (lp->width > MINGRIDSIZE * lp->logo->width &&
			    lp->height > MINGRIDSIZE * lp->logo->height) {
				lp->pixelmode = False;
				lp->xs = lp->logo->width;
				lp->ys = lp->logo->height;
			} else {
				lp->pixelmode = True;
				lp->xs = lp->ys = MAX(MINSIZE, MIN(lp->width, lp->height) /
						      MINGRIDSIZE);
			}
		} else {
			lp->pixelmode = True;
			if (size < -MINSIZE)
				lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				      MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
			else if (size < MINSIZE)
				lp->ys = MINSIZE;
			else
				lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
						       MINGRIDSIZE));
			lp->xs = lp->ys;
		}
		lp->ncols = MAX(lp->width / lp->xs, 4);
		lp->nrows = MAX(lp->height / lp->ys, 4);
		lp->xb = (lp->width - lp->xs * lp->ncols) / 2;
		lp->yb = (lp->height - lp->ys * lp->nrows) / 2;
	} else {		/* TRI */
		int         orient, i;

		lp->black = MI_WIN_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 2)
			lp->height = 2;
		if (size < -MINSIZE)
			lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				      MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		else if (size < MINSIZE) {
			if (!size)
				lp->ys = MAX(MINSIZE, MIN(lp->width, lp->height) / MINGRIDSIZE);
			else
				lp->ys = MINSIZE;
		} else
			lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
					       MINGRIDSIZE));
		lp->xs = (int) (1.52 * lp->ys);
		lp->ncols = (MAX(lp->width / lp->xs - 1, 2) / 2) * 2;
		lp->nrows = (MAX(lp->height / lp->ys - 1, 2) / 2) * 2;
		lp->xb = (lp->width - lp->xs * lp->ncols) / 2 + lp->xs / 2;
		lp->yb = (lp->height - lp->ys * lp->nrows) / 2 + lp->ys / 2;
		for (orient = 0; orient < 2; orient++) {
			for (i = 0; i < 3; i++) {
				lp->triangleList[orient][i].x =
					(lp->xs - 2) * triangleUnit[orient][i].x;
				lp->triangleList[orient][i].y =
					(lp->ys - 2) * triangleUnit[orient][i].y;
			}
		}
	}
	lp->npositions = lp->nrows * lp->ncols;

	XSetForeground(MI_DISPLAY(mi), lp->backGC, lp->black);
	XFillRectangle(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
		       0, 0, lp->width, lp->height);

	alloc_cells(lp);

	if (local_neighbors == 8 && rule == 2333)
		npats = NPATS_2333;
	else
		npats = 0;
	lp->pattern = NRAND(npats + 2);
	if (lp->pattern >= npats)
		RandomSoup(mi, 30, MAX(2 * MIN(lp->nrows, lp->ncols) / 3, 15));
	else
		GetPattern(mi, lp->pattern);
}

void
draw_life(ModeInfo * mi)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	CellList   *middle[STATES];	/* To distinguish between old and new stuff */
	CellList   *curr;
	cellstruct  info;
	int         i, count;

/*-
 * LIVE list are the on cells
 * DEAD list are the cells that may go on in the next iteration.
 * Init plan:
     Create live list and dead list which border all live cells
       (no good for rules like 0000 :) )
 * Big loop plan:
     Setup toggles, toggle state next iteration?
     Remove all from dead list except toggled and remove all from live list
       that are dead (but in this case draw background square)
     Toggle toggled states, age existing ones, create a new dead list, draw
 */

	/* Go through dead list to see if anything spawns (generate new lists),
	   then delete the used dead list */

	/* Setup toggles */
	curr = lp->first[DEAD]->next;
	while (curr != lp->last[DEAD]) {
		count = n_neighbors(lp, curr);
		if (count >= birth_min && count <= birth_max) {
			setcelltoggles(mi, curr->info.position % lp->ncols,
				       curr->info.position / lp->ncols);
		}
		curr = curr->next;
	}
	curr = lp->first[LIVE]->next;
	while (curr != lp->last[LIVE]) {
		count = n_neighbors(lp, curr);
		if (!(count >= living_min && count <= living_max)) {
			setcelltoggles(mi, curr->info.position % lp->ncols,
				       curr->info.position / lp->ncols);
		}
		curr = curr->next;
	}

	/* Bring out your dead! */
	curr = lp->first[DEAD]->next;
	while (curr != lp->last[DEAD]) {
		curr = curr->next;
		if (!curr->previous->info.toggle)
			removefrom_list(lp, DEAD, curr->previous);
	}
	curr = lp->first[LIVE]->next;
	while (curr != lp->last[LIVE]) {
		if (curr->info.toggle) {
			curr->info.state = DEAD;
			draw_cell(mi, curr->info);
			curr = curr->next;
			removefrom_list(lp, LIVE, curr->previous);
		} else
			curr = curr->next;
	}

	/* Fence off the babies */
	info.position = -1;	/* dummy value */
	info.age = 0;		/* dummy value */
	addto_list(lp, DEAD, info);
	addto_list(lp, LIVE, info);
	middle[DEAD] = lp->last[DEAD]->previous;
	middle[LIVE] = lp->last[LIVE]->previous;

	/* Toggle toggled states, age existing ones, create a new dead list */
	while (lp->first[DEAD]->next != middle[DEAD]) {
		curr = lp->first[DEAD]->next;
		setcellfromtoggle(mi, curr->info.position % lp->ncols,
				  curr->info.position / lp->ncols);
	}
	curr = lp->first[LIVE]->next;
	while (curr != middle[LIVE]) {
		setcellfromtoggle(mi, curr->info.position % lp->ncols,
				  curr->info.position / lp->ncols);
		curr = curr->next;
	}
	removefrom_list(lp, DEAD, middle[DEAD]);
	removefrom_list(lp, LIVE, middle[LIVE]);

	if (lp->redrawing) {
		for (i = 0; i < REDRAWSTEP; i++) {
			CellList   *curr = lp->arr[lp->redrawpos];

			/* TODO: More efficient to use list rather than array. */
			if (curr && curr->info.state == LIVE) {
				draw_cell(mi, curr->info);
			}
			if (++(lp->redrawpos) >= lp->npositions) {
				lp->redrawing = 0;
				break;
			}
		}
	}
	if (++lp->generation > MI_CYCLES(mi))
		init_life(mi);

	/*
	 * generate a randomized shooter aimed roughly toward the center of the
	 * screen after batchcount.
	 */

	if (lp->generation && lp->generation %
	    ((MI_BATCHCOUNT(mi) < 0) ? 1 : MI_BATCHCOUNT(mi)) == 0)
		shooter(mi);

}

void
release_life(ModeInfo * mi)
{
	if (lifes != NULL) {
		int         screen;

		for (screen = 0; screen < MI_NUM_SCREENS(mi); screen++) {
			lifestruct *lp = &lifes[screen];
			int         state;

			for (state = 0; state < STATES; state++) {
				flush_list(lp, state);
				(void) free((void *) lp->last[state]);
				(void) free((void *) lp->first[state]);
			}
			free_cells(lp);
			free_stuff(MI_DISPLAY(mi), lp);
		}
		(void) free((void *) lifes);
		lifes = NULL;
	}
}

void
refresh_life(ModeInfo * mi)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];

	lp->redrawing = 1;
	lp->redrawpos = 0;
}

void
change_life(ModeInfo * mi)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int         npats, i;

	lp->generation = 0;
	if (lp->first[0])
		for (i = 0; i < STATES; i++)
			flush_list(lp, i);
	else
		for (i = 0; i < STATES; i++)
			init_list(lp, i);
	free_cells(lp);
	alloc_cells(lp);

	XSetForeground(MI_DISPLAY(mi), lp->backGC, lp->black);
	XFillRectangle(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
		       0, 0, lp->width, lp->height);


	lp->pattern = lp->pattern + 1;
	if (local_neighbors == 8 && rule == 2333)
		npats = NPATS_2333;
	else
		npats = 0;
	if (lp->pattern >= npats + 2)
		lp->pattern = 0;
	if (lp->pattern >= npats)
		RandomSoup(mi, 30, MAX(2 * MIN(lp->nrows, lp->ncols) / 3, 15));
	else
		GetPattern(mi, lp->pattern);
}
