;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  kernel.asm
;* Purpose: Setup the stack and data areas, and then call the higher
;*          level routines.
;* Entries: Called by loader at offset 0.
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Load the boot rom segment layout and assembler macros.
;
INITSEG		equ	1		; create end-of-segment labels

include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include ..\..\headers\asm\version.inc


;
;**************************************************************************
;
; Equates for this module:
;
STACKSIZE	equ	4096		; Size of stack


;
;**************************************************************************
;
; The kernel starts here. At the beginning there is some
; miscellaneous data.
;
text_start

	public	start

	org	0			; the kernel always starts at offset 0

start:	jmp	short krnstart		; the loader jumps to this address

	db	(_text:DATAOFS - $) dup (0)	; Fill up

; The following value gets filled in by the binary patch program which
; concatenates the kernel with the packet driver programs.

prgadr	dw	0			; Address of program pointer list

; This value can be used by the binary patch program to strip off the
; BSS segment from the final binary. The BSS segment just contains zeroes
; which don't have to be saved in the tight ROM.

	dw	offset cgroup:_end_of_data

; Put the version number into the final kernel image. This allows the
; binary patch program to verify that it's using the correct kernel
; image.

	db	MAJOR_VER		; major version number
	db	MINOR_VER		; minor version number


;
;**************************************************************************
;
; Now start with the actual kernel code, which sets up the stack, copies
; the data segment into it's place, and then calls the higher level code.
;
krnstart	proc	near

	assume	ds:nothing
	assume	es:nothing
	assume	ss:nothing

	extrn	_kernelmain:near
	extrn	_fatal:near
	extrn	prnstr:near

	cli
	cld
	mov	bx,NEWDATA
	mov	dx,offset dgroup:_end_of_bss + STACKSIZE
	mov	ss,bx
	mov	sp,dx		; set new stack pointer
	assume	ss:dgroup

	xor	ax,ax
	mov	ds,ax
	mov	ax,OLDINTS
	mov	es,ax		; save old interrupt vectors
	xor	si,si
	xor	di,di
	mov	cx,VECTSIZE / 2
	rep	movsw

	mov	ax,cs
	mov	ds,ax		; move data segment into lower ram
	mov	es,bx
	mov	si,offset cgroup:_end_of_code + 1
	and	si,0FFFEh
	mov	di,si		; start the data segment at the right offset
	and	di,000Fh
	mov	cx,offset dgroup:_end_of_data
	mov	ax,cx
	rep	movsb

	mov	cx,offset dgroup:_end_of_bss
	sub	cx,ax
	xor	al,al
	rep	stosb		; clear BSS segment

; The boot rom code is completely within the OS loading area. In order
; to avoid overwriting it with the loaded OS we have to copy it into
; a secure memory area right at the end of the physical ram. However,
; don't copy the packet driver which gets loaded into the DOS ram
; area lateron.
; Note that BX and DX should not be changed during all this copying.

	mov	si,offset cgroup:_end_of_code
	mov	di,si
	shift	shr,si,10	; compute the number of kB required
	add	si,(HIGHMEM shr 6) + 1
	int	12h		; determine number of kB in the system
	cmp	ax,si
	ja	short memok	; check if enough memory available
	sti
	mov	bx,offset cgroup:memerr
	call	prnstr
	jmp	_fatal		; print error message and terminate

memok:	mov	ax,cs
	mov	ds,ax		; copy the bootrom code into the secure
	mov	ax,HIGHMEM	; high memory area
	mov	es,ax
	mov	cx,di
	xor	si,si
	xor	di,di
	rep	movsb

	mov	es,bx		; set new data segment
	assume	es:dgroup
ifdef IS186
	push	ax
	push	offset cgroup:domain
else
	mov	cx,offset cgroup:domain
	push	ax
	push	cx
endif
	mov	ax,cs		; save old code segment
	retf			; call the new boot rom copy

; The main routine of the kernel gets three farpointers onto the
; stack: a pointer to the end of the data area and the two pointers
; at the beginning of the rom code. But before calling the main routine
; set the reboot flag to indicate warm boot. This is necessary so that
; a Ctrl-Alt-Del keypress reboots the system correctly.

domain:	mov	cx,0040h
	mov	ds,cx
	mov	ds:[0072h],1234h	; indicates warm boot
	mov	ds,bx			; set new data segment
	assume	ds:dgroup

	sti
	push	ds
	add	dx,2		; push the initial stack pointer
	push	dx		; the packet driver is still in the decompressed
	push	ax		; copy of the bootrom in the OS loading area
	push	cs:prgadr	; push the program pointer list address
	call	_kernelmain	; now start the game
	add	sp,8

; If the main program returns restore the interrupt vectors and call the
; bootstrap vector. Also deinitialize any resident program.

	mov	ax,0CF45h
	int	2Fh			; deinitialize resident programs

	cli
	cld
	xor	ax,ax
	mov	es,ax
	mov	ax,OLDINTS
	mov	ds,ax			; restore old interrupt vectors
	xor	si,si
	xor	di,di
	mov	cx,VECTSIZE / 2
	rep	movsw
	sti
	int	BOOTSTRAP		; call the bootstrap vector
dolop:	jmp	short dolop

; Messages:

memerr	db	'Base mem err',0

krnstart	endp


;
;**************************************************************************
;
text_end			; end text segment

	end
