;**************************************************************************
;*
;* Boot-ROM-Code to load an operating system across a TCP/IP network.
;*
;* Module:  packet.asm
;* Purpose: Interface to the packet driver
;* Entries: _reg_type, _write_packet, _pktpoll, _init_packet
;*
;**************************************************************************
;*
;* Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
;*
;*  This program is free software; you can redistribute it and/or modify
;*  it under the terms of the GNU General Public License as published by
;*  the Free Software Foundation; either version 2 of the License, or
;*  any later version.
;*
;*  This program is distributed in the hope that it will be useful,
;*  but WITHOUT ANY WARRANTY; without even the implied warranty of
;*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;*  GNU General Public License for more details.
;*
;*  You should have received a copy of the GNU General Public License
;*  along with this program; if not, write to the Free Software
;*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;*


;
;**************************************************************************
;
; Include assembler macros:
;
include ..\..\headers\asm\macros.inc
include ..\..\headers\asm\layout.inc
include ..\..\headers\asm\memory.inc
include .\pktpriv.inc


;
;**************************************************************************
;
; Miscelleneous equates:
;
MIN_INT		equ	60h		; lowest usable packet driver interrupt
MAX_INT		equ	80h		; highest usable packet driver interrupt
DRIVER_ID	equ	<'PKT DRVR'>	; ID at beginning of packet driver
DRIVER_LEN	equ	8		; length of ID string
DRIVER_OFS	equ	3		; offset of ID string


;
;**************************************************************************
;
; Define entry for packet type table:
;
type_entry	struc

type_id		dw	?		; ID for packet type
type_driver	dw	?		; pointer to type event routine
type_handle	dw	?		; packet driver handle for type

type_entry	ends


;
;**************************************************************************
;
; BSS segment. We can safely include static read and write buffers here
; as the BSS segment will not be stored in the final bootrom and therefore
; doesn't occupy any space.
;
bss_start

		extrn	_myhwaddr:byte

pktmtu		dw	?		; packet driver MTU
pktoldint	dw	?		; packet driver interrupt no.
readtype	dw	?		; type of packet in buffer
readsize	dw	?		; actual size of data in buffer
readbuf		dw	?		; pointer to read buffer
writebuf	dw	?		; pointer to write buffer

typetab		type_entry	TYPENUM dup (<>)

bss_end


;
;**************************************************************************
;
; Start code segment.
;
text_start

	public	_reg_type		; define entry points
	public	_write_packet
	public	_pktpoll
	public	_init_packet

; General library functions:

	extrn	prnstr:near
	extrn	prnbyte:near
	extrn	_malloc:near
	extrn	_fatal:near


;
;**************************************************************************
;
; Register a new packet type.
; Input:  1. arg  -  type identifier
;         2. arg  -  near pointer to poll callback routine
; Output: error code
;
_reg_type	proc	near

	penter				; setup standard stack frame
	push	es
	push	di
	push	si
	mov	si,offset dgroup:typetab
regt1:	mov	cx,type_id[si]
	jcxz	short regt3		; is the slot available?
	add	si,size type_entry
	jmp	short regt1

regt3:	getarg	ax,0
	mov	type_id[si],ax		; save type id
	getarg	ax,1			; get pointer to handler routine
	mov	type_driver[si],ax
	push	si
	mov	di,offset cgroup:receiver
	mov	ax,cs
	mov	es,ax			; ES:DI  -  pointer to receive handler
	lea	si,type_id[si]		; DS:SI  -  pointer to type ID
	mov	cx,size type_id		; CX     -  size of type ID
	mov	al,CL_DEFAULT		; AL     -  default class
	mov	bx,TYPE_DEFAULT		; BX     -  default type
	xor	dl,dl			; DL     -  default interface number
	mov	ah,ACCESS_TYPE
	int	PKTINT			; setup access type
	pop	si
	jnc	short regt4
regt5:
ifdef IS386
	movzx	ax,dh			; prepare return value
else
	mov	al,dh			; in case of error prepare return
	xor	ah,ah			; value, e.g. error code
endif
	neg	ax
	jmp	short regt9

regt4:	mov	type_handle[si],ax	; save access type handle
	cmp	si,offset dgroup:typetab; check if hardware address already
	jnz	short regt7		; initialized
	mov	bx,ax
	mov	ax,ds
	mov	es,ax
	mov	di,offset dgroup:_myhwaddr
	mov	cx,ETH_ALEN		; get hardware address for current
	mov	ah,GET_ADDRESS		; handle from packet driver
	int	PKTINT
	jc	short regt5
regt7:	xor	ax,ax			; return without error
regt9:	pop	si
	pop	di
	pop	es
	pleave
	ret

_reg_type	endp


;
;**************************************************************************
;
; Send a packet using the packet driver
; Input:  1. arg  -  near pointer to data buffer
;         2. arg  -  number of bytes in buffer
;         3. arg  -  packet type value
;         4. arg  -  near pointer to destination hardware address
; Output: error code
;
_write_packet	proc	near

	penter				; setup standard stack frame
	cld
	push	es
	push	si
	push	di
	mov	ax,ds
	mov	es,ax
	getarg	ax,2			; get packet type
	mov	bx,offset dgroup:typetab
	mov	cx,length typetab
wrtp1:	cmp	ax,type_id[bx]		; find packet type in type table
	je	short wrtp2
	add	bx,size type_entry
	loop	short wrtp1
	mov	ax,-ERR_BAD_TYPE	; type not found
	jmp	short wrtp9

wrtp2:	mov	di,writebuf
	mov	cx,ETH_ALEN		; copy destination address into
	getarg	si,3			; send buffer
	rep	movsb
	mov	cx,ETH_ALEN			; copy source address into send
	mov	si,offset dgroup:_myhwaddr	; buffer
	rep	movsb
	stosw				; copy packet type
	getarg	cx,1			; get number of characters to write
	getarg	si,0			; get data pointer
	cmp	cx,pktmtu
	jbe	short wrtp4		; check if maximum number of data
	mov	cx,pktmtu		; bytes exceeded
wrtp4:	mov	ax,cx
	rep	movsb			; copy payload into output buffer
	cmp	ax,ETH_DATA_MIN
	jae	short wrtp5		; check if minimum number of data
	mov	cx,ETH_DATA_MIN		; bytes copied
	sub	cx,ax
	xor	al,al			; pad with zeroes if not
	rep	stosb
	mov	ax,ETH_DATA_MIN

wrtp5:	mov	si,writebuf		; set pointer to write buffer
	mov	bx,type_handle[bx]	; set packet driver handle
	mov	cx,ax
	add	cx,ETH_HLEN		; set packet buffer size
	mov	ah,SEND_PKT		; now send the packet
	int	PKTINT
	jnc	short wrtp8
ifdef IS386
	movzx	ax,dh			; return with error
else
	mov	al,dh
	xor	ah,ah			; return with error
endif
	neg	ax
	jmp	short wrtp9

wrtp8:	xor	ax,ax			; no error
wrtp9:	pop	di
	pop	si
	pop	es
	pleave
	ret

_write_packet	endp


;
;**************************************************************************
;
; This routine gets called by the packet driver when a new packet arrives.
; Input:  AX  -  function number: 0 = return buffer pointer, 1 = buffer filled
;         BX  -  handle
;         CX  -  length of data received (only when AX = 1)
;         DS:SI  -  pointer to receive buffer (only when AX = 1)
; Output: CX     -  length of buffer (only when AX = 0)
;         ES:DI  -  pointer to receive buffer (only when AX = 0)
; Registers changed: AX, BX, CX, DX, ES, SI, DI
;
receiver	proc	far

	assume	ds:dgroup
	assume	ss:nothing

	push	ds
	mov	dx,NEWDATA
	mov	ds,dx
	or	ax,ax			; check function code
	jnz	short rec3

; Return address of receive buffer in ES:DI. Discard the packet if the
; receive buffer is not empty.

	mov	es,ax
	mov	di,ax			; discard packet if no free buffer
	mov	ax,readsize		; buffer free?
	or	ax,ax
	jnz	short rec9
	mov	es,dx
	mov	di,readbuf		; return pointer to read buffer
	jmp	short rec9

; The buffer has been read. Now mark it as busy, so that the polling routine
; can give it up one level.

rec3:	mov	si,offset dgroup:typetab
	mov	dx,length typetab
rec4:	cmp	bx,type_handle[si]	; find handle in type table
	je	short rec5
	add	si,size type_entry
	dec	dx
	jnz	short rec4		; discard packet if handle not found
	jmp	short rec9

rec5:	mov	dx,type_id[si]
	mov	readsize,cx		; save buffer size and mark buffer
	mov	readtype,dx		; busy
rec9:	pop	ds
	ret

	assume	ds:dgroup
	assume	ss:dgroup

receiver	endp


;
;**************************************************************************
;
; Get a received packet out of the read buffer, and call the upper layer
; handler, which gets the following parameters on it's stack:
;	char *buf    -  pointer to receive buffer
;	int bufsize  -  number of data bytes in receive buffer
;	char *addr   -  pointer to sender's address
;
; Input:  none
; Output: none
;
_pktpoll	proc	near

	mov	cx,readsize		; immediately terminate if no
	jcxz	short pktp9		; packet received

	mov	bx,readbuf
	lea	ax,ETH_ALEN[bx]		; push pointer to source hardware
	push	ax			; address
	sub	cx,ETH_HLEN
	push	cx			; push buffer size (without header)
	add	bx,ETH_HLEN
	push	bx			; push buffer pointer (without header)

	mov	bx,offset dgroup:typetab
	mov	cx,length typetab
	mov	dx,readtype
pktp1:	cmp	dx,type_id[bx]
	je	short pktp2
	add	bx,size type_entry	; search type in type list
	loop	short pktp1
	jmp	short pktp3

pktp2:	mov	ax,1
	mov	bx,type_driver[bx]	; call type handler
	or	bx,bx
	jz	short pktp3
	call	bx
pktp3:	add	sp,6			; cleanup stack
	mov	readsize,0		; discard packet
pktp9:	ret

_pktpoll	endp


;
;**************************************************************************
;
; Initialize packet driver interface.
; Input:  None
; Output: None
;
_init_packet	proc	near

	push	es
	push	di

; First determine the driver's interrupt, and redirect it to our own
; interrupt. Using an interrupt which is known at assembly time allows
; the use of the int instruction.

	call	pktfind			; try to find packet driver int
	or	ax,ax
	jnz	short init1
	mov	bx,offset cgroup:intmsg	; print interrupt error message
initF:	jmp	pktfatal

init1:	mov	pktoldint,ax		; save packet driver interrupt
	cli
	xor	ax,ax
	mov	es,ax			; save packet driver interrupt vector
ifdef IS386
	mov	eax,es:[bx]
	mov	es:[PKTINT * 4],eax
else
	mov	ax,es:[bx+0]
	mov	word ptr es:[PKTINT * 4 + 0],ax
	mov	ax,es:[bx+2]
	mov	word ptr es:[PKTINT * 4 + 2],ax
endif
	sti

; Setup the packet driver interface. This involves checking the maximum
; transfer data size, the network class and the address size. Then allocate
; memory for the send and receive buffers.

	mov	ah,GET_PARAMETERS
	int	PKTINT			; get parameters from packet
	mov	ax,ETH_FRAME_MAX	; driver. if driver is too old,
	jc	short init4		; assume default value
	mov	ax,es:[di + 4]		; get MTU from driver
	cmp	ax,ETH_FRAME_MAX
	jae	short init3
init2:	mov	bx,offset cgroup:pktmsg	; invalid packet driver
	jmp	short initF

init3:	mov	dl,es:[di + 3]		; get size of MAC address
	cmp	dl,ETH_ALEN		; check if ethernet address length
	jne	short init2		; is correct
init4:	mov	dx,ax
	sub	dx,ETH_HLEN		; MTU is buffer size minus header
	mov	pktmtu,dx		; length

	add	ax,ETH_FLEN
	push	ax
	add	ax,ETH_BUF_SIZE + 10
	push	ax
	call	_malloc
	pop	dx
	pop	dx
	or	ax,ax
	jnz	short init5
	mov	bx,offset cgroup:memmsg	; not enough memory
	jmp	short initF

init5:	mov	readbuf,ax
	add	ax,dx			; set buffer pointers
	mov	writebuf,ax

	push	ds
	push	si
	mov	ax,(DRIVER_INFO shl 8) + 0FFh
	xor	bx,bx			; dummy handle
	int	PKTINT			; get packet driver info
	pop	si
	pop	ds
	jc	short init2		; packet driver is too old
	cmp	ch,CL_DEFAULT		; packet driver is not for ethernet
	jne	short init2

; Tell the user that we found something.

	mov	bx,offset cgroup:fndmsg
	call	prnstr
	mov	ax,pktoldint
	call	prnbyte
	mov	bx,offset cgroup:crlf
	call	prnstr
	pop	di
	pop	es
	ret


; Error messages:

intmsg	db	'No int',0
memmsg	db	'Not enough memory',0
pktmsg	db	'Invalid PKTDRVR',0


; Message to tell user about packet driver interrupt:

fndmsg	db	0Dh,0Ah
	db	'Found packet driver at int ',0
crlf	db	0Dh,0Ah,0

_init_packet	endp


;
;**************************************************************************
;
; Find packet driver interrupt
; Input:  none
; Output: AX  -  Interrupt number, or 0 if not found
;         BX  -  pointer to interrupt vector in segment 0
; Registers changed: AX, BX, CX, DI, ES
;
pktfind		proc	near

	cld
	push	ds
	push	si
	mov	ax,cs
	mov	ds,ax
	mov	bx,MIN_INT * 4		; start looking at interrupt 60h
pktf1:	xor	ax,ax
	mov	es,ax
	les	di,es:[bx]		; get interrupt vector
	mov	ax,es
	or	ax,di			; ignore null pointers
	jz	short pktf2
	add	di,DRIVER_OFS		; compute address of ID string
	mov	si,offset cgroup:pktdrv_id
	mov	cx,DRIVER_LEN
	repe	cmpsb			; compare both strings
	jz	short pktf3		; found it
pktf2:	add	bx,4			; advance to next interrupt vector
	cmp	bx,MAX_INT * 4		; at the end of the list?
	jb	short pktf1
	xor	ax,ax
	jmp	short pktf4

pktf3:	mov	ax,bx
ifdef IS186
	shr	ax,2			; return interrupt number
else
	shr	ax,1
	shr	ax,1			; return interrupt number
endif
pktf4:	pop	si
	pop	ds
	ret

; ID string at the beginning of the packet driver code:

pktdrv_id	db	DRIVER_ID

pktfind		endp


;
;**************************************************************************
;
; Print fatal error message and exit
; Input:  BX  -  pointer to error message
; Output: routine does not return
;
pktfatal	proc	near

	push	bx
	mov	bx,offset cgroup:fatmsg
	call	prnstr
	pop	bx
	call	prnstr
	jmp	_fatal

; Preceding error message:

fatmsg	db	0Dh,0Ah
	db	'PKTDRV ERROR: '
	db	0

pktfatal	endp


;
;**************************************************************************
;
text_end

	end

