/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** lcllib.c
**
** routines for loading and creating library files
**
** this is a brute-force implementation, a more efficient
** representation should be designed.
**
*/

# include "lclintMacros.nf"
# include "llbasic.h"
# include "osd.h"
# include "gram.h"
# include "herald.h"
# include "lclscan.h"
# include "lcllib.h"
# include "llglobals.h"
# include "llmain.h"

/*@-incondefs@*/ /*@-redecl@*/
extern /*@dependent@*/ FILE *yyin;
/*@=incondefs@*/ /*@=redecl@*/

/*@constant int NUMLIBS; @*/
# define NUMLIBS 19

/* removed:  "sys/time", "sys/signal" */

static ob_mstring stdlibs[NUMLIBS] =
{
  "assert", 
  "errno",
  "limits",
  "locale",
  "math", 
  "setjmp",
  "signal",
  "stddef",
  "time",
  "ctype",
  "float", 
  "locale",
  "stdarg",
  "stdlib",
  "stdio",
  "strings", 
  "string",
  "unistd",
  "varargs"  /* this must be last! */
} ;

static void loadStateFile (FILE * f);

bool
lcllib_isAnsiHeader (char *name)
{
  int i;
  char *libname;

  if (name == NULL)
    {
      llcontbuglit ("lcllib_isAnsiHeader: undefined name!");
      return FALSE;
    }

  name = removeExtension (name, ".h");

  libname = strrchr (name, '/');

  if (libname == NULL) 
    {
      libname = name;
    }
  else
    {
      libname++;
      /*@-branchstate@*/
    }
  /*@=branchstate@*/
  
  for (i = 0; i < NUMLIBS; i++)
    {
      if (mstring_equal (libname, stdlibs[i]))
	{
	  if (i == NUMLIBS - 1) /* used <varargs> */
	    {
	      fileloc tmp = fileloc_makePreprocPrevious (currentloc);

	      voptgenerror 
		(FLG_USEVARARGS,
		 message ("Include file <%s> is inconsistent with "
			  "ANSI library (should use <stdarg.h>)",
			  libname),
		 tmp);

	      fileloc_free (tmp);
	    }

	  sfree (name);
	  return TRUE;
	}
    }

  sfree (name);
  return FALSE;
}

static void printDot (void)
{
  if (context_getFlag (FLG_SHOWSCAN)) 
    {
      fprintf (stderr, "."); 
    }
}

void
dumpState (cstring cfname)
{
  FILE *f;
  char *fname = cstring_toCharsSafe (cfname);
  
  fname = addExtension (fname, DUMP_SUFFIX);
  
  f = fopen (fname, "w");

  if (context_getFlag (FLG_SHOWSCAN))
    {
      fprintf (stderr, "< Dumping to %s ", fname);
    }
  
  if (f == NULL)
    {
      llgloberror (message ("Cannot open dump file for writing: %s", cfname));
    }
  else
    {
      /*
      ** sequence is convulted --- must call usymtab_prepareDump before
      **    dumping ctype table to convert type uid's
      */

      printDot ();

      usymtab_prepareDump ();

      fprintf (f, ";;LCLint Dump: %s from %s\n", fname, LCL_VERSION);
      fprintf (f, ";;ctTable\n");

      printDot ();
      DPRINTF (("dump ctypes:"));
      ctype_dumpTable (f);
      printDot ();

      fprintf (f, ";;tistable\n");
      typeIdSet_dumpTable (f);
      printDot ();

      DPRINTF (("dump usymtab"));
      fprintf (f, ";;symTable\n");
      usymtab_dump (f);
      printDot ();

      fprintf (f, ";; Modules access\n");
      context_dumpModuleAccess (f);
      fprintf (f, ";;End\n");
      check (fclose (f) == 0);
    }

  if (context_getFlag (FLG_SHOWSCAN))
    {
      fprintf (stderr, " >\n");
    }

  sfree (fname);
}

bool
loadStandardState ()
{
  char *fpath;
  FILE *stdlib;
  bool result;
  char *libname = 
    addExtension (context_getFlag (FLG_STRICTLIB) 
		  ? LLSTRICTLIBS_NAME 
		  : (context_getFlag (FLG_UNIXLIB)
		     ? LLUNIXLIBS_NAME
		     : LLSTDLIBS_NAME),
		  DUMP_SUFFIX);
    
  DPRINTF (("loadStandardState"));

  if (osd_getLarchPath (libname, &fpath) != OSD_FILEFOUND)
    {
      llmsg (message ("Cannot find %sstandard library: %s", 
		      cstring_makeLiteralTemp 
		      (context_getFlag (FLG_STRICTLIB) ? "strict " 
		       : (context_getFlag (FLG_UNIXLIB) ? "unix " : "")),
		      cstring_makeLiteralTemp (libname)));
      llmsglit ("     Check LARCH_PATH environment variable.");
      result = FALSE;
    }
  else
    {
      stdlib = fopen (fpath, "r");

      if (stdlib == NULL)
	{
	  llmsg (message ("Cannot read standard library: %s",
			  cstring_fromChars (fpath)));
	  llmsglit ("     Check LARCH_PATH environment variable.");
	  result = FALSE;
	}
      else
	{
	  if (context_getFlag (FLG_WHICHLIB))
	    {
	      char *t = mstring_create (MAX_NAME_LENGTH);
	      char *ot = t;

	      if (fgets (t, MAX_NAME_LENGTH, stdlib) == NULL)
		{
		  llfatalerror (cstring_makeLiteral ("Standard library format invalid"));
		}

	      t = strstr (t, "from");

	      if (t == NULL)
		{
		  llmsg (message ("Standard library: %s <cannot read creation information>", 
				  cstring_fromChars (fpath)));
		}
	      else
		{
		  char *tt;

		  t += 5;
		  tt = strrchr (t, '\n');
		  if (tt != NULL)
		    *tt = '\0';

		  llmsg (message ("Standard library: %s", cstring_fromChars (fpath)));
		  llmsg (message ("   (created using %s)", cstring_fromChars (t)));
		}

	      sfree (ot);
	    }

	  fileloc_reallyFree (currentloc);
	  currentloc = fileloc_createLib (cstring_makeLiteralTemp (libname));

	  if (context_getDebug (FLG_SHOWSCAN))
	    {
	      context_hideShowscan ();
	      loadStateFile (stdlib);
	      context_unhideShowscan ();
	    }
	  else
	    {
	      loadStateFile (stdlib);
	    }

	  result = TRUE;
	}
    }

  sfree (libname);
  return result;
}

static void
loadStateFile (FILE *f)
{
  DPRINTF (("loadStateFile"));

  ctype_loadTable (f);
  DPRINTF (("done type table"));
  printDot ();

  DPRINTF (("load typeIdSet table"));
  typeIdSet_loadTable (f);
  printDot ();

  DPRINTF (("usymtab..."));
  usymtab_load (f);
  DPRINTF (("done usymtab"));
  printDot ();

  DPRINTF (("module access"));
  context_loadModuleAccess (f);
  DPRINTF (("done module access"));
  printDot ();

  check (fclose (f) == 0);
}

/*
** load state from file created by dumpState
*/

void
loadState (cstring cfname)
{
  FILE *f;
  char *fname = cstring_toCharsSafe (cfname);
  cstring ofname = cstring_copy (cfname);

  fname = addExtension (fname, DUMP_SUFFIX);

  f = fopen (fname, "r");

  if (f == NULL)
    {
      if (context_getDebug (FLG_SHOWSCAN))
	fprintf (stderr, " >\n");

      llfatalerror (message ("Cannot open dump file for loading: %s", cfname));
    }
  else
    {
      fileloc_reallyFree (currentloc);
      currentloc = fileloc_createLib (ofname);
      loadStateFile (f);
    }

  cstring_free (ofname);
  sfree (fname);
}

