/*
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/
/*
** macrocache.c
**
** rep Invariant:
**     no two fileloc's may be equal
**
*/

# include "lclintMacros.nf"
# include "basic.h"
# include "llglobals.h"
# include "llmain.h"

/*@constant int MCEBASESIZE;@*/
# define MCEBASESIZE 8

/*@constant int DNE;@*/
# define DNE -1

/*
** mcDisable is set to TRUE when a macro is being processed, so
** its contents are not added to the macrocache again, creating
** a nasty infinite loop.
*/

static bool mcDisable = TRUE;


static void macrocache_grow (macrocache s);
static int macrocache_exists (macrocache s, fileloc fl);
static void macrocache_processMacro (macrocache m, int i);

static /*@only@*/ mce
  mce_create (/*@only@*/ fileloc fl, /*@only@*/ cstring def, bool comment)
{
  mce m = (mce) dmalloc (sizeof (*m));
  m->fl = fl;
  m->def = def; /*< had a copy here! check this carefully */
  m->defined = FALSE;
  m->scomment = comment;
  return m;
}

static void mce_free (/*@only@*/ mce m)
{
  fileloc_free (m->fl);
  cstring_free (m->def);
  sfree (m);
}

/*@only@*/ macrocache
macrocache_create (void)
{
  macrocache s = (macrocache) dmalloc (sizeof (*s));

  s->entries = 0;
  s->nspace = MCEBASESIZE;
  s->contents = (mce *) dmalloc (sizeof (*s->contents) * MCEBASESIZE);

  mcDisable = FALSE;

  return (s);
}

void
macrocache_free (macrocache s)
{
  int i;

  for (i = 0; i < s->entries; i++)
    {
      mce_free (s->contents[i]);
    }

  sfree (s->contents);
  sfree (s);
}

static void
macrocache_grow (macrocache s)
{
  int i;
  o_mce *oldcontents = s->contents;

  s->nspace = MCEBASESIZE;
  s->contents = (mce *) dmalloc 
    (sizeof (*s->contents) * (s->entries + MCEBASESIZE + 1));

  for (i = 0; i < s->entries; i++)
    {
      s->contents[i] = oldcontents[i];
    }

  sfree (oldcontents);
}

static void
macrocache_addGenEntry (macrocache s, /*@only@*/ fileloc fl,
			/*@only@*/ cstring def, bool sup)
{
  int i;

  if (mcDisable)
    {
      DPRINTF (("disable: %s / %s", fileloc_unparse (fl), def));
      fileloc_free (fl);
      cstring_free (def);
      return;
    }

  if ((i = macrocache_exists (s, fl)) != DNE)
    {
      if (cstring_equal (def, s->contents[i]->def))
	{
	  fileloc_free (fl);
	  cstring_free (def);

	  return;
	}
      else
	{
	  /*
	  ** macro definition contained macro that is expanded
	  ** replace with def
	  **
	  ** how do we know which is better??
	  */
	  
	  cstring_free (s->contents[i]->def);
	  s->contents[i]->def = def;

	  fileloc_free (fl);
	  return;
	}
    }

  if (s->nspace <= 0)
    macrocache_grow (s);

  DPRINTF (("add: %s", def));

  s->nspace--;
  s->contents[s->entries] = mce_create (fl, def, sup);
  s->entries++;
}

void
macrocache_addEntry (macrocache s, /*@only@*/ fileloc fl, /*@only@*/ cstring def)
{
  macrocache_addGenEntry (s, fl, def, FALSE);
}


void
macrocache_addComment (macrocache s, /*@only@*/ fileloc fl, /*@only@*/ cstring def)
{
  macrocache_addGenEntry (s, fl, def, TRUE);
}

static int
macrocache_exists (macrocache s, fileloc fl)
{
  int i;

  for (i = 0; i < s->entries; i++)
    {
      if (fileloc_equal (s->contents[i]->fl, fl))
	return (i);
    }

  return (DNE);
}

/*@only@*/ cstring
macrocache_unparse (macrocache m)
{
  cstring s = cstring_undefined;
  int i;

  for (i = 0; i < m->entries; i++)
    {
      fileloc fl = m->contents[i]->fl;
      cstring def = m->contents[i]->def;
      bool defined = m->contents[i]->defined;
      
      s = message ("%q%q: %s [%s]\n", s, fileloc_unparse (fl), def, 
		   bool_unparse (defined));
    }
  
  return (s);
}

/*
** needs to call lex by hand...yuk!
**
** modifies gc fileloc!
*/

/*
** there's gotta be a better way of doing this!
*/

static void pushString (/*@only@*/ cstring s)
{
  static fileId mtid = fileId_invalid;
  static FILE *tmp = NULL;
  long floc;

  if (tmp == NULL)
    {
      cstring fname;
      mtid = fileTable_addMacrosFile (context_fileTable ());

      fname = fileName (mtid);
      tmp = fopen (cstring_toCharsSafe (fname), "w+");
      
      if (tmp == NULL)
	{
	  llcontbug (message ("Cannot open tmp file %s needed to process macro: %s", 
			      fname, s));
	  cstring_free (s);
	  return;
    	}
    }

  llassert (tmp != NULL);

  /* SunOS, others? don't define SEEK_CUR and SEEK_SET */
# ifndef SEEK_CUR 
# define SEEK_CUR 1
# endif
  check (fseek (tmp, 0, SEEK_CUR) == 0);

  floc = ftell (tmp);

  check (fputs (cstring_toCharsSafe (s), tmp) != EOF);
  check (fputc ('\n', tmp) == (int) '\n');

# ifndef SEEK_SET 
# define SEEK_SET 0
# endif
  check (fseek (tmp, floc, SEEK_SET) == 0);

  yyin = tmp;
  (void) yyrestart (yyin);
  cstring_free (s);
}

static void
macrocache_processMacro (macrocache m, int i)
{
  fileloc fl = m->contents[i]->fl;
 
  DPRINTF (("process macro: %d", i));

  m->contents[i]->defined = TRUE;

  if (!fileId_equal (currentFile (), fileloc_fileId (fl)))
    {
      currentloc = fileloc_update (currentloc, fl);
      context_enterMacroFile ();
    }
  else
    {
      setLine (fileloc_lineno (fl));
    }

  beginLine ();

  if (m->contents[i]->scomment)
    {
      DPRINTF (("macro: %s", m->contents[i]->def));
      pushString (message ("@%s@", m->contents[i]->def));
      (void) yyparse ();
      DPRINTF (("okay"));
    }
  else
    {
      bool insup = context_inSuppressRegion ();

      pushString (message ("%s %s", 
			   cstring_makeLiteralTemp (PPMRCODE),
			   m->contents[i]->def));
      (void) yyparse ();

      if (context_inSuppressRegion () && !insup)
	{
	  llerrorlit (FLG_SYNTAX, "Macro ends in ignore region");
	}
    }
  
  incLine ();  
  context_exitMacroCache ();
}

extern void macrocache_processUndefinedElements (macrocache m)
{
  fileloc lastfl = fileloc_undefined;
  int i;
 
  mcDisable = TRUE;

  if (!context_getFlag (FLG_PARTIAL))
    {
      for (i = 0; i < m->entries; i++) 
	{
	  if (m->contents[i]->defined)
	    {
	      ;
	    }
	  else 
	    { 
	      fileloc fl = m->contents[i]->fl; 
	      
	      if (fileloc_isDefined (lastfl) && fileloc_sameFile (fl, lastfl)) 
		{
		  ;
		}
	      else
		{
		  if (context_getFlag (FLG_SHOWSCAN))
		    {
		      if (!fileloc_isLib (fl))
			{
			  llmsg (message ("< checking macros %s >", fileloc_filename (fl)));
			}
		    }
		  
		  lastfl = fl;
		  cleanupMessages ();
		}
	      
	      macrocache_processMacro (m, i);	
	    }
	}
    }

  mcDisable = FALSE;
}

extern /*@observer@*/ fileloc macrocache_processFileElements (macrocache m, cstring base)
{
  fileloc lastfl = fileloc_undefined;
  int i;
 
  mcDisable = TRUE;

  for (i = 0; i < m->entries; i++) 
    {
      if (m->contents[i]->defined)
	{
	  ;
	}
      else 
        { 
	  fileloc fl = m->contents[i]->fl;  /* should be dependent! */
	  cstring fb = fileloc_getBase (fl);

	  if (cstring_equal (fb, base))
	    {
	      lastfl = fl;
	      macrocache_processMacro (m, i);	
	    }
	}
    }

  mcDisable = FALSE;
  return lastfl;
}


