/*****************************************************************
**
** MathSpad 0.60
**
** Copyright 1996, Eindhoven University of Technology (EUT)
** 
** Permission to use, copy, modify and distribute this software
** and its documentation for any purpose is hereby granted
** without fee, provided that the above copyright notice appear
** in all copies and that both that copyright notice and this
** permission notice appear in supporting documentation, and
** that the name of EUT not be used in advertising or publicity
** pertaining to distribution of the software without specific,
** written prior permission.  EUT makes no representations about
** the suitability of this software for any purpose. It is provided
** "as is" without express or implied warranty.
** 
** EUT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
** SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL EUT
** BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
** DAMAGES OR ANY DAMAGE WHATSOEVER RESULTING FROM
** LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
** CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING
** OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
** OF THIS SOFTWARE.
** 
** 
** Roland Backhouse & Richard Verhoeven.
** Department of Mathematics and Computing Science.
** Eindhoven University of Technology.
**
********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include "sun_kludge.h"
#include "mathpad.h"
#include "system.h"
#include "funcs.h"
#include "latex.h"
#include "intstack.h"


/* these are defined in various header-files (output, sources) */

extern void out_string(char* str);
extern void open_tabbing(void);
extern void close_tabbing(void);
extern char *font_opentexttex(int fontnr, int size);
extern char *font_openmathtex(int fontnr, int size);
extern char *font_closetex(int fontnr, int size);
extern char *char_latex(Char c, Bool math);

static Bool def_thinspace;
static int output_mode;
static int display_delta;

/*  */

#define PLACENAMESIZE 40
static char placename[PLACENAMESIZE+1];
static int placepos=0;
static int nameset=0;
static int inplacename=0;

#define NAMEDHEAD    "\\mpN{"
#define NAMEDTAIL    "}"

#define SYMBOL_TAB(c)      (modetab[Num2Tab(c)][0])
#define LATEX_TAB(c)       (modetab[Num2Tab(c)][1])
#define ASCII_TAB(c)       (modetab[Num2Tab(c)][2])
#define SYMBOL_PH(c)       (modeph[Ph2Num(c)][0])
#define LATEX_PH(c)        (modeph[Ph2Num(c)][1])
#define ASCII_PH(c)        (modeph[Ph2Num(c)][2])
#define SYMBOL_PHNUM(c)    (modephnum[Num(c)][0])
#define LATEX_PHNUM(c)     (modephnum[Num(c)][1])
#define ASCII_PHNUM(c)     (modephnum[Num(c)][2])


static
char *modetab[41][3] =
{  /* Screen       LaTeX               ASCII      ..... */
    { "",          "\\\\\n\t",           "\n"  },
    { "[set]",     "\\=",                ""    },
    { "[tab]",     "\t\\>",              "\t"  },
    { "[back]",    "\\<",                ""    },
    { "[plus]",    "\\+",                ""    },
    { "[minus]",   "\\-",                ""    },
    { "[push]",    "\\push",             ""    },
    { "[pop]",     "\\pop",              ""    },
    { "[NwLn]",    "\n\t",               "\n"  },
    { "[TEXT]",    "",                   ""    },
    { "[text]",    "",                   ""    },
    { "[math]",    "",                   ""    },
    { "[disp]",    "",                   ""    },
    { "]",         "}",                  ""    },
    { "[Hide:",    "",                   ""    },
    { "]",         "}",                  ""    },
    { "[Top]",     "{",                  ""    },
    { "[top]",     "}",                  ""    },
    { "[Bot]",     "{",                  ""    },
    { "[bot]",     "}",                  ""    },
    { "[Gap]",     "{",                  ""    },
    { "[gap]",     "}",                  ""    },
    { "[Fill]",    "\\hfill",            ""    },
    { "[Line]",    "\\hrulefill",        ""    },
    { "[Dots]",    "\\dotfill",          ""    },
    { "[StackB:",  "\\stackrel{",        ""    },
    { "[StackC:",  "\\stackrel{",        ""    },
    { ":",         "}{\\stackrel{",      ""    },
    { ":",         "}{",                 ""    },
    { "]",         "}}",                 ""    },
    { "]",         "",                   ""    },
    { "[Tabbing:", "\\begin{tabbing}",   ""    },
    { "]",         "\\end{tabbing}",     ""    },
    { "[Display:", "\\begin{mpdisplay}", ""    },
    { "]",         "\\end{mpdisplay}",   ""    },
    { "[Bar]",     "|",                  ""    },
    { "[Space]",   "\\ms{%i}",           ""    },
    { "[MATH]",    "",                   ""    },
    { "[BOTH]",    "",                   ""    },
    { "[Name:",    "",                   ""    },
    { "]",         "",                   ""    }
};

static
char *modeph[7][3] =
{
    { "E",         "\\mpE",              "Expr" },
    { "O",         "\\mpO",              "Op"   },
    { "I",         "\\mpI",              "Id"   },
    { "V",         "\\mpV",              "Var"  },
    { "T",         "\\mpT",              "Text" },
    { "L",         "\\mpL",              "List" },
    { "D",         "\\mpD",              "Disp" }
};

static
char *modephnum[16][3] =
{
    { "",          "",                   ""   },
    { "1",         "_1",                 "1"  },
    { "2",         "_2",                 "2"  },
    { "3",         "_3",                 "3"  },
    { "4",         "_4",                 "4"  },
    { "5",         "_5",                 "5"  },
    { "6",         "_6",                 "6"  },
    { "7",         "_7",                 "7"  },
    { "8",         "_8",                 "8"  },
    { "9",         "_9",                 "9"  },
    { "10",        "_{10}",              "10" },
    { "11",        "_{11}",              "11" },
    { "12",        "_{12}",              "12" },
    { "13",        "_{13}",              "13" },
    { "14",        "_{14}",              "14" },
    { "15",        "_{15}",              "15" }
};

static
char *modesize[10][3] =
{
    { "",          "{\\tiny ",           "" },
    { "",          "{\\scriptsize ",     "" },
    { "",          "{\\footnotesize ",   "" },
    { "",          "{\\small ",          "" },
    { "",          "{\\normalsize ",     "" },
    { "",          "{\\large ",          "" },
    { "",          "{\\Large ",          "" },
    { "",          "{\\LARGE ",          "" },
    { "",          "{\\huge ",           "" },
    { "",          "{\\Huge ",           "" }
};

#define SIZELATEX(A) ((A)>128? ((A)>197? modesize[9][1]: \
                                ((A)<188? modesize[0][1]: \
                                 modesize[(A)-188][1]))\
                      :  ((A)>69? modesize[9][1]: \
                         ((A)<60? modesize[0][1]: modesize[(A)-60][1])))

static Bool in_math, in_tabbing, tex_plhl, after_slash, after_macro,
            secure, setms;
static FILE *texf = NULL;
static char **texstring = NULL;       /* LaTeX -> string */
static int stringlen=0, stringmax=0;  /* length of string, size allocated */
static INTSTACK *prefmode=NULL;
static INTSTACK *ownmode=NULL;
static int nrdisp = 0;
static int texmode = 0;
static int texoutput=0; /* 0=FILE 1=SCHERM 2=STRING */


static void add_string(char *str)
{
    int i,nm;
    char *c;

    if (!str || !(i = strlen(str))) return;
    if (stringlen+i>=stringmax) {
	nm = (((stringlen+i)>>11)+1)<<11;
	if (*texstring)
	    c = (char*) realloc(*texstring, nm);
	else
	    c = (char*) malloc(nm);
	if (!c) return;
        *texstring = c;
	stringmax=nm;
    }
    strcpy(*texstring+stringlen, str);
    stringlen+=i;
}

static void printlatex(Bool bld, char *str)
{
    switch (texoutput) {
    case 0:
        if (str) fputs(str,texf);
        break;
    case 1:
        switch (bld) {
        case False:
            out_string(str);
            break;
        default:
            out_bold(str);
            break;
        }
        break;
    case 2:
        add_string(str);
        break;
    default:
        break;
    }
}

static void close_slash(void)
{
    if (after_slash) {
        printlatex(False," ");
        after_slash = after_macro = False;
    }
}

static void close_macro_space(void)
{
    if (after_macro) {
	printlatex(True, "{}");
	after_slash=after_macro=False;
    }
}

static void close_macro(void)
{
    if (after_macro) {
        printlatex(False," ");
        after_slash=after_macro=False;
    }
}

static void close_math(void)
{
    close_slash();
    if (in_math)
	if (head_int(ownmode))
	    printlatex(True, "\\mbox{");
	else
	    printlatex(True,"$");
    after_macro = after_macro && !in_math;
    in_math = False;
}

static void open_math(void)
{
    close_slash();
    after_macro = after_macro && in_math;
    if (!in_math)
	if (head_int(ownmode))
	    printlatex(True, "}");
	else
	    printlatex(True, "$");
    in_math=True;
}

void push_math_pref(Bool premath)
{
    if (in_math != head_int(ownmode))
	if (in_math) close_math(); else open_math();
    if (premath != head_int(ownmode))
	if (!premath) close_math(); else open_math();
    push_int(&ownmode, premath);
    push_int(&prefmode, premath);
}

static void soft_math_pref(Bool premath)
{
    if (ownmode) ownmode->nr=premath;
} 

void pop_math_pref(void)
{
    if (in_math != head_int(ownmode))
	if (in_math) close_math(); else open_math();
    pop_int(&ownmode);
    if (head_int(ownmode) != head_int(prefmode))
	if (head_int(prefmode)) close_math(); else open_math();
    pop_int(&prefmode);
}

static void clear_state(void)
{
    in_tabbing = 
        in_math =
        tex_plhl =
        after_slash =
        after_macro =
        secure = False;
    def_thinspace=0;
    texmode = output_mode;
    setms = (output_mode!=MPTEX);
    nrdisp = 0;
}

void tex_set_file(FILE *f)
{
    texf=f;
    clear_state();
    texoutput = (!f);
}

void tex_set_string(char **str)
{
    texstring = str;
    stringlen = stringmax = 0;
    if (*str) free(*str);
    *str = NULL;
    clear_state();
    texoutput = 2;
    setms=True;
}

void tex_mode(int mode)
{
    texmode = mode;
    setms = setms || mode!=MPTEX;
}

void tex_open_proof(void)
{
    int d = (display_delta+latex_side<0 ? 0 : display_delta+latex_side); 
    if (!in_tabbing) {
        char buf[500];
        close_math();
        if (texoutput!=1) {
            sprintf(buf, "\\begin{mpdisplay}{%s}{%s}{%s}{%i}\n\t",
                    latex_space_unit, latex_tab_unit,
                    latex_line_unit, d);
            printlatex(True, buf);
        } else
            open_tabbing();
        in_tabbing = True;
        after_slash = False;
        after_macro = False;
        secure = False;
        in_math = False;
        nrdisp = 1;
    } else nrdisp++;
    display_delta = 0;
}

void tex_close_proof(void)
{
    if (in_tabbing && nrdisp==1) {
        close_math();
        if (texoutput!=1)
            printlatex(True,"\n\\end{mpdisplay}");
        else
            close_tabbing();
        in_tabbing = 
            after_slash =
            after_macro =
            secure =
            in_math = False;
        nrdisp = 0;
    } else if (nrdisp) nrdisp--;
}

void tex_unset(void)
{
    close_math();
    if (in_tabbing) 
        if (texoutput!=1) 
            printlatex(True,"\n\\end{mpdisplay}");
        else
            close_tabbing();
    printlatex(False,"\n");
    clear_state();
    texf = NULL;
    texstring=0;
    texoutput=0;
    stringlen=0;
    stringmax=0;    
}

void tex_placeholders(Bool texthem)
{
    tex_plhl = texthem;
}

static void update_after(char *str)
{
    while (*str) {
        if (*str=='\\')
            after_slash= after_macro=!after_slash;
        else {
            after_macro=after_macro && isalpha(*str);
            after_slash = False;
        }
        str++;
    }
}

void out_latex_char(Char c)
{
    int newmode;
    char *lc=NULL;

    if (!c) return;
    if (inplacename) {
	if (placepos<PLACENAMESIZE && !(c&0xff00))
            placename[placepos++]=c&0xff;
        else if (c==PlNameEnd) inplacename=0;
        return;
    }
    if (IsTab(c)) {
	if (c==PlName) { placepos=0; inplacename=1; } else
        if (output_mode==ASCII) {
            printlatex(True,ASCII_TAB(c));
        } else  if (in_tabbing && (c>=SoftNewline)) {
            if (c!=SoftNewline) close_math();
            printlatex(True,LATEX_TAB(c));
        } else {
            close_slash();
            if (!head_int(ownmode)) close_math();
            switch (c) {
            case Rtab:
                printlatex(False,"\t");
                break;
            case Newline:
            case SoftNewline:
                printlatex(False,"\n");
                break;
            case AskText:
                soft_math_pref(False);
                if (texoutput==1) printlatex(True, SYMBOL_TAB(c));
                break;
            case InText:
                in_math = False;
                soft_math_pref(False);
                if (texoutput==1) printlatex(True, SYMBOL_TAB(c));
                break;
            case InMath:
	    case InDisp:
                in_math = True;
                soft_math_pref(True);
                if (texoutput==1) printlatex(True, SYMBOL_TAB(InMath));
                break;
            case ThinSpace:
                { char oss[10];
                  sprintf(oss, LATEX_TAB(ThinSpace), def_thinspace);
                  close_slash();
                  printlatex(False,oss); }
                break;
            case PopSize:
            case CloseStack:
            case OpenTop:
            case CloseTop:
            case OpenBottom:
            case CloseBottom:
            case OpenGap:
            case CloseGap:
            case StackClose:
            case StackB:
            case StackC:
            case TopGap:
            case GapBottom:
                if (head_int(ownmode)) open_math();
                printlatex(True,LATEX_TAB(c));
                break;
            case TabOpen:
                if (!in_tabbing) {
                    in_tabbing = True;
                    nrdisp=1;
                    close_math();
                    printlatex(True, LATEX_TAB(TabOpen));
                } else if (texoutput==1) {
                    nrdisp++;
                    close_math();
                    printlatex(True, LATEX_TAB(TabOpen));
                } else nrdisp++;
                break;
            case TabClose:
                nrdisp--;
                if (!nrdisp) {
                    in_tabbing = False;
                    close_math();
                    printlatex(True, LATEX_TAB(TabClose));
                } else if (texoutput==1) {
                    close_math();
                    printlatex(True, LATEX_TAB(TabClose));
                }
                break;
            case DisplayOpen:
                tex_open_proof();
                if (texoutput==1)
                    printlatex(True, LATEX_TAB(DisplayOpen));
                break;
            case DisplayClose:
                tex_close_proof();
                if (texoutput==1)
                    printlatex(True, LATEX_TAB(DisplayClose));
                break;
            default:
                printlatex(True,LATEX_TAB(c));
                break;
            }
        }
    } else if (IsPh(c)) {
        if (output_mode==ASCII) {
            if (texoutput==1) {
                printlatex(True, SYMBOL_PH(c));
		printlatex(True, SYMBOL_PHNUM(c));
	    } else if (placepos || nameset) {
		if (!nameset) placename[placepos]='\0';
		printlatex(True, placename);
		placepos=nameset=0;
	    } else
                printlatex(True,ASCII_PH(c));
	} else {
	    if (tex_plhl) {
                if (Ph(c)!=Text) open_math();
		if (placepos || nameset) {
                    if (!nameset) placename[placepos]='\0';
                    printlatex(True,NAMEDHEAD);
                    printlatex(True,placename);
                    printlatex(True,NAMEDTAIL);
                    placepos=nameset=0;
                } else printlatex(True,LATEX_PH(c));
		if (Ph(c)!=Text) printlatex(True,LATEX_PHNUM(c));
            }
            if (texoutput==1) {
                if (Ph(c) == Text)
                    close_math();
                else
                    open_math();
                printlatex(True, SYMBOL_PH(c));
		printlatex(True, SYMBOL_PHNUM(c));
            }
        }
    } else if (IsOpspace(c)) {
        if (output_mode!=ASCII) {
            char oss[10];
            int i = Char2ASCII(c);

            if (i>0) {
                switch (texmode) {
                case PROOFTEX:
                    close_slash();
                    while (i) {
                        printlatex(False, "\\,");
                        i--;
                    }
                    break;
                case PLAINTEX:
                    break;
                case MPTEX:
                default:
                    sprintf(oss, "\\ms{%i}", i);
                    close_slash();
                    printlatex(False,oss);
                    break;
                }
            }
        }
    } else if (Char2Font(c)==StackFont) {
        if (output_mode!=ASCII) {
            open_math();
            close_slash();
            printlatex(False,LATEX_TAB(StackC));
        }
    } else if (Char2Font(c)==FontFont) {
        if (output_mode!=ASCII) {
	    char *h=NULL;
	    int npm=head_int(ownmode);
	    if (npm) {
		h = font_openmathtex(Char2ASCII(c),0);
		if (!h && (h= font_opentexttex(Char2ASCII(c),0)))
                    npm=0;
            } else {
                h = font_opentexttex(Char2ASCII(c),0);
                if (!h && (h= font_openmathtex(Char2ASCII(c),0)))
                    npm=1;
            }
	    push_math_pref(npm);
            if (h) {
                if (!head_int(ownmode))
                    close_math();
                else
                    open_math();
                secure=True;
                printlatex(True,h);
                update_after(h);
            }
        }
    } else if (Char2Font(c)==PopFont) {
        if (output_mode!=ASCII) {
            char *h=font_closetex(Char2ASCII(c),0);
            if (h) {
                if (!head_int(ownmode))
                    close_math();
                else
                    open_math();
                secure=True;
                printlatex(True,h);
                update_after(h);
            }
	    pop_math_pref();
        }
    } else if (Char2Font(c)==SizeFont) {
        if (output_mode!=ASCII) {
            if (!head_int(ownmode))
                close_math();
            else
                open_math();
            printlatex(True,SIZELATEX(Char2ASCII(c)));
        }
    } else if (c<BUTTONFONT*256) {
        if (!(lc = char_latex(c, head_int(ownmode))))
            if (!(lc = char_latex(c, !head_int(ownmode))))
                newmode = head_int(ownmode);
            else
                newmode = !head_int(ownmode);
        else
            newmode = head_int(ownmode);
        if (output_mode!=ASCII && newmode && !in_math) open_math();
        if (output_mode!=ASCII && !newmode && in_math) close_math();
        if (!lc) {
            close_slash();
            printlatex(False,"~");
            after_macro = False;
        } else {
            if (output_mode!=ASCII) {
                if (strlen(lc)>1) {
                    if (isalpha(*lc))
                        close_macro();
                    else
                        close_slash();
                    secure=True;
                    printlatex(True,lc);
                } else {
                    if (secure) {
                        if (isalpha(*lc))
                            close_macro();
                        else if (isspace(*lc))
			    close_macro_space();
			else
                            close_slash();
                        secure=False;
                    }
                    printlatex(False,lc);
                }
                update_after(lc);
            } else
                if (strlen(lc)>1)
                    printlatex(True,lc);
                else
                    printlatex(False,lc);
        }
    }
}

void set_display_delta(int d)
{
    display_delta = d;
}

void tex_to_mode(int tm)
{
    switch (tm) {
    case LTEXTMODE:
        if (head_int(ownmode)) {
            pop_math_pref();
            push_math_pref(False);
        }
        break;
    case LMATHMODE:
        if (!head_int(ownmode)) {
            pop_math_pref();
            push_math_pref(True);
        }
        break;
    case LBOTHMODE:
        pop_math_pref();
        push_math_pref(head_int(ownmode));
        break;
    default: break;
    }
}

void tex_code(TexCode c)
{
    if (output_mode!=ASCII) {
        secure = True;
        switch (c) {
        case ExprOpen:
            if (!setms && texf) {
                close_slash();
                fprintf(texf, "\\setms{%s}",latex_space_unit);
		after_macro=False;
                setms=True;
            }
            push_math_pref(True);
            break;
        case ExprClose:
            pop_math_pref();
            break;
        case SOpOpen:
            switch (texmode) {
            case PLAINTEX:
                break;
            case PROOFTEX:
            case MPTEX:
            default:
                open_math();
                printlatex(False,"{");
                after_macro = False;
                break;
            }
            push_math_pref(True);
            break;
        case SOpClose:
            switch (texmode) {
            case PLAINTEX:
                break;
            case PROOFTEX:
            case MPTEX:
            default:
                open_math();
                printlatex(False,"}");
                after_macro = False;
                break;
            }
            pop_math_pref();
            break;
        case LOpOpen:
            push_math_pref(True);
            break;
        case LOpClose:
            pop_math_pref();
            break;
        case SIdOpen:
            /* open_math();*/
            push_math_pref(True);
            break;
        case SIdClose:
            pop_math_pref();
            break;
        case LIdOpen:
            push_math_pref(True);
            /*open_math();*/
            printlatex(False,"{\\it ");
            after_macro = False;
            break;
        case LIdClose:
            if (head_int(ownmode) != in_math) {
                if (in_math)
                    close_math();
                else
                    open_math();
            } else close_slash();
            printlatex(False,"\\/}");
            pop_math_pref();
            break;
        case VarOpen:
            push_math_pref(True);
            break;
        case VarClose:
            pop_math_pref();
            break;
        case TextOpen:
            secure = False;
            push_math_pref(False);
            break;
        case TextClose:
            secure = False;
            pop_math_pref();
            break;
        case DispOpen:
            if (!setms && texf) {
                close_slash();
                fprintf(texf, "\\setms{%s}",latex_space_unit);
                setms=True;
            }
            tex_open_proof();
            push_math_pref(True);
            display_delta = 0;
            break;
        case DispClose:
            pop_math_pref();
            tex_close_proof();
            break;
        default:
            break;
        }
    }
}

