/*
WINIO.C
Stdio (e.g. printf) functionality for Windows - implementation
Dave Maxey - 1991

(Quickly) Ported and Augmented for WIN32 / WIN16 by Dan Larner - 1995
*/

#include <windows.h>
#include <stdlib.h>
#include <stdarg.h>
#include <malloc.h>
#include <string.h>
#include <direct.h>

#include "wmhandlr.h"
#include "winio.h"

#ifdef WIN16
#include <commdlg.h>
#endif

/* RESTRICTIONS: - Dan Larner
   If stdout, stdin, or stderr have been redirected, behaviour will STILL
   send/receive to/from the window.

*/


/* PROTOTYPES in alphabetic order */

void            addchars(char *, unsigned);
void            adjust_caret(void);
void            append2buffer(char *, unsigned);
int             chInput(void);
void            compute_repaint(void);
int             initialize_buffers(unsigned);
int             initialize_class(HANDLE);
void            initialize_state(void);
int             initialize_window(HANDLE, HANDLE, int);
void            make_avail(unsigned);
char *      	nextline(char *);
char *      	prevline(char *);
void            set_font(void);
long            winio_wmpaint(HWND, unsigned, WORD, LONG);
long            winio_wmsize(HWND, unsigned, WORD, LONG);
long            winio_wmdestroy(HWND, unsigned, WORD, LONG);
long            winio_wmchar(HWND, unsigned, WORD, LONG);
long            winio_wmkeydown(HWND, unsigned, WORD, LONG);
long            winio_wmhscroll(HWND, unsigned, WORD, LONG);
long            winio_wmvscroll(HWND, unsigned, WORD, LONG);
long            winio_wmsetfocus(HWND, unsigned, WORD, LONG);
long            winio_wmkillfocus(HWND, unsigned, WORD, LONG);

// this doesn't get declared in stdio.h if _WINDOWS is defined
// although it is in the Windows libraries!
//int             vsprintf(char *, const char *, va_list);

#define         winio_caret_visible() \
                ((yCurrLine <= (yTopOfWin + yWinHeight)) && \
                (xCurrPos <= (xLeftOfWin + xWinWidth)) && \
                (xCurrPos >= xLeftOfWin))

#define         CHECK_INIT() if (! tWinioVisible) return FALSE
                 
#define         MAX_X                   127
#define         TABSIZE                 8
#define         TYPE_AHEAD              256
#define         WINIO_DEFAULT_BUFFER    8192
#define         MIN_DISCARD             256
#define         CARET_WIDTH             2

// For scrolling procedures
#define         USE_PARAM               10000
#define         DO_NOTHING              10001

char            winio_class[15] = "winio_class";
char            winio_icon[15] = "winio_icon";
char            winio_title[128] = "Console I/O";
unsigned long   bufsize = WINIO_DEFAULT_BUFFER;
unsigned long   kbsize = TYPE_AHEAD;
unsigned        bufused, bufSOI;
unsigned        curr_font = SYSTEM_FIXED_FONT;
int             tWinioVisible = FALSE;
int             tCaret = FALSE, tFirstTime = TRUE;
int             cxChar, cyChar, cxScroll, cyScroll, cxWidth, cyHeight;
int             xWinWidth, yWinHeight, xCurrPos;
int             xLeftOfWin, yTopOfWin, yCurrLine;
unsigned        pchKbIn, pchKbOut;
static char *fpBuffer, *fpTopOfWin, *fpCurrLine; 
static char *fpKeyboard;
static HANDLE   hBuffer, hKeyboard;
static HWND     hwnd;
BOOL            tTerminate = TRUE,
                tPaint = TRUE;
DESTROY_FUNC    destroy_func;

typedef struct {
    int hSB, vSB;
    } recVKtoSB;
                
/* This table defines, by scroll message, what increment to try */
/* and scroll horizontally. PGUP and PGDN entries are updated   */
/* in the winio_wmsize function.                                */
int             cScrollLR[SB_ENDSCROLL + 1] =
//UP  DOWN PGUP     PGDN    POS        TRACK      TOP     BOT    ENDSCROLL
{ -1, +1,  -1,      +1,     USE_PARAM, USE_PARAM, -MAX_X, MAX_X, DO_NOTHING};
                
/* This table defines, by scroll message, what increment to try */
/* and scroll horizontally. PGUP and PGDN entries are updated   */
/* in the winio_wmsize function, and the TOP & BOTTOM entries   */
/* are updated by addchar function.                             */
int             cScrollUD[SB_ENDSCROLL + 1] =
//UP  DOWN PGUP     PGDN    POS        TRACK      TOP     BOT    ENDSCROLL
{ -1, +1,  -1,      +1,     USE_PARAM, USE_PARAM, -1,     +1,    DO_NOTHING};
                
/* This table associates horizontal and vertical scroll         */
/* messages that should be generated by the arrow and page keys */
recVKtoSB       VKtoSB[VK_DOWN - VK_PRIOR + 1] =
//                  VK_PRIOR                    VK_NEXT
                {   { DO_NOTHING, SB_PAGEUP },  { DO_NOTHING, SB_PAGEDOWN },
//                  VK_END                      VK_HOME
                    { SB_TOP, SB_BOTTOM },      { SB_TOP, SB_TOP },
//                  VK_LEFT                     VK_UP
                    { SB_LINEUP, DO_NOTHING },  { DO_NOTHING, SB_LINEUP },
//                  VK_RIGHT                    VK_DOWN
                    { SB_LINEDOWN, DO_NOTHING },{ DO_NOTHING, SB_LINEDOWN } };
                
/* ===================================================================  */
/* the interface functions themselves.....                              */
/* ===================================================================  */

char *gets(char *pchTmp)
    {
    char *pch = pchTmp;
    int c;

    CHECK_INIT();
    bufSOI = bufused; /* mark beginning of line to limit backspace */
    do {
        switch (c = fgetchar())
            {
            case '\b' :     if (pch > pchTmp) pch--; break;
            case 0x1b :     pch = pchTmp; break;
            case EOF :      bufSOI = -1; return NULL;
            default :       *pch = (char) c; pch++;
            }
        } while (c);
    bufSOI = -1;
    return pchTmp;
    }

int printf(const char *fmt, ...)
    {
    va_list marker;
    va_start(marker, fmt);
    return vprintf(fmt, marker);
    }

int vprintf(const char *fmt, va_list marker)
    {
    static char s[4096];
    int len;

    CHECK_INIT();
    len = vsprintf(s, fmt, marker);
    addchars(s,len);
    return len;
    }

/* vfprintf and fprintf added by Dan Larner */

/* we want the real vfprintf (called by vfprintf_winio) in here */
#undef vfprintf

int vfprintf_winio(FILE* stream, const char *fmt, va_list marker) {
    static char s[4096];
    int len;

	if ((stream == stdout) || (stream == stderr)) {
		/* output should go to window */
    	CHECK_INIT();
    	len = vsprintf(s, fmt, marker);
    	addchars(s,len);
    	return len;
    }

	/* must be writing to a file call the real vfprintf */
	return vfprintf(stream, fmt, marker);
}

int fprintf(FILE* stream, const char *fmt, ...) {
    va_list marker;
    va_start(marker, fmt);
    return vfprintf_winio(stream, fmt, marker);
}

int fgetchar(void)
    {
    CHECK_INIT();
    return fputchar(chInput());
    }

int _kbhit(void)
    {
    CHECK_INIT();
    return (pchKbIn == pchKbOut);
    }

int fputchar(int c)
    {
    CHECK_INIT();
    addchars(&((char) c), 1);
    return c;
    }

int puts(const char *s)
    {
    char c = '\n';
    CHECK_INIT();
    addchars((char *) s, strlen(s));
    addchars(&c, 1);
    return 0;
    }

/* ---------------------------------------------------------------  */
/* USED INTERNALLY - pops up an error window and returns FALSE      */
/* ---------------------------------------------------------------  */
int fail(char *s)
    {
    MessageBox(NULL,s,"ERROR",MB_OK);
    return FALSE;
    }

/* ---------------------------------------------------------------  */
/* pops up a message window                                         */
/* ---------------------------------------------------------------  */
BOOL winio_warn(BOOL confirm, const char *fmt, ...)
    {
    char s[256];
    va_list marker;

    va_start(marker, fmt);
    vsprintf(s, fmt, marker);
    va_end(marker);
    
    return (MessageBox(NULL, s, winio_title, 
        confirm? MB_OKCANCEL : MB_OK) == IDOK);
    }

/* ---------------------------------------------------------------  */
/* The application must call this function before using any of the  */
/* covered stdio type calls. We need the parameter info in order    */
/* to create the window. The function allocates the buffer and      */
/* creates the window. It returns TRUE or FALSE.                    */
/* ---------------------------------------------------------------  */
int winio_init(HANDLE hInstance, HANDLE hPrevInstance,
            int nCmdShow, unsigned wBufSize)
    {
    if (tWinioVisible)
        return FALSE;
    
    if (! initialize_buffers(wBufSize))
        return FALSE;

    initialize_state();
    
    if (! initialize_window(hInstance, hPrevInstance, nCmdShow))
        return FALSE;
    
    tWinioVisible = TRUE;
    
    atexit(winio_end);  /* hook into exit chain */

    winio_yield();
    return TRUE;
    }

/* ---------------------------------------------------------------  */
/* Clear the contents of the buffer.                                */
/* ---------------------------------------------------------------  */
void winio_clear(void)
    {
    memset(fpBuffer,0,(int) bufsize - 1);
    fpCurrLine = fpTopOfWin = fpBuffer;
    *fpBuffer = '\0';
    xCurrPos = 0;
    yCurrLine = 0;
    yTopOfWin = 0;
    xLeftOfWin = 0;
    bufused = 0;

    if (tWinioVisible)
        {
        SetScrollRange(hwnd, SB_VERT, 1, yCurrLine + 1, FALSE);
        SetScrollPos(hwnd, SB_VERT, yTopOfWin + 1, TRUE);
        }
    }

/* ---------------------------------------------------------------  */
/* Return the window handle of the underlying Windows object.       */
/* Can be used by an application to customize the WINIO window      */
/* ---------------------------------------------------------------  */
HWND winio_hwnd(void)
    {
    return hwnd;
    }

/* ---------------------------------------------------------------  */
/* This function is called by winio_init(). It initializes a number */
/* of global variables, including the WM_ handler table.            */
/* ---------------------------------------------------------------  */
void initialize_state()
    {
    winio_clear();
    destroy_func = 0;
    
    /* set up our message handlers */
    wmhandler_init();
    wmhandler_set(WM_PAINT,       winio_wmpaint);
    wmhandler_set(WM_SIZE,        winio_wmsize);
    wmhandler_set(WM_DESTROY,     winio_wmdestroy);
    wmhandler_set(WM_CHAR,        winio_wmchar);
    wmhandler_set(WM_HSCROLL,     winio_wmhscroll);
    wmhandler_set(WM_VSCROLL,     winio_wmvscroll);
    wmhandler_set(WM_SETFOCUS,    winio_wmsetfocus);
    wmhandler_set(WM_KILLFOCUS,   winio_wmkillfocus);
    wmhandler_set(WM_KEYDOWN,     winio_wmkeydown);
    }

/* ---------------------------------------------------------------  */
/* This function is called by winio_init(). It initializes our      */
/* Windows class, and some global variables                         */
/* ---------------------------------------------------------------  */
int initialize_window(HANDLE hInst, HANDLE hPrev, int nCmdShow)
    {
    static RECT start;
    int cx, cy, inc;

    cx = GetSystemMetrics(SM_CXSCREEN);
    cy = GetSystemMetrics(SM_CYSCREEN);
    inc = GetSystemMetrics(SM_CYCAPTION);
    cxScroll = GetSystemMetrics(SM_CXVSCROLL);
    cyScroll = GetSystemMetrics(SM_CYHSCROLL);

    if (! initialize_class(hInst)) 
            return fail("Could not create class");

    start.left = cx >> 3;
    start.right = 3 * (cx >> 2);
    start.top = cy >> 3;
    start.bottom = 3 * (cy >> 2);
        
    hwnd = CreateWindow(winio_class, winio_title,
        WS_OVERLAPPEDWINDOW | WS_HSCROLL | WS_VSCROLL,
        start.left, start.top, start.right, start.bottom,
        NULL, NULL, hInst, NULL);
    if (! hwnd)
        return fail("Could not create window");

    set_font();
    
    ShowWindow(hwnd, nCmdShow);
    UpdateWindow(hwnd);

    return TRUE;
    }

/* -----------------------------------------------------------------------  */
/* Initializes Window Class                                                 */
/* -----------------------------------------------------------------------  */
int initialize_class(HANDLE hInst)
    {

    WNDCLASS  wc;

    wc.style = CS_HREDRAW | CS_VREDRAW | CS_BYTEALIGNCLIENT;
    wc.lpfnWndProc = (WNDPROC) winio_WndProc;
    wc.cbClsExtra = 0;
    wc.cbWndExtra = 0;
    wc.hInstance = hInst;
    wc.hIcon = LoadIcon(hInst, winio_icon);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH); 
    wc.lpszMenuName = NULL;
    wc.lpszClassName = winio_class;

    return RegisterClass(&wc);
    }
    
/* -----------------------------------------------------------------------  */
/* Uses GlobalAlloc() to allocate the display and keyboard buffers          */
/* -----------------------------------------------------------------------  */
int initialize_buffers(unsigned wBufSize)
    {
    if (wBufSize)
        bufsize = max(wBufSize, 1024);

    if (! (hBuffer = GlobalAlloc(GMEM_MOVEABLE, bufsize)))
        return fail("Could not allocate\nconsole I/O buffer");
    
    fpBuffer = GlobalLock(hBuffer); // keep locked; assume protected mode
    
    if (! (hKeyboard = GlobalAlloc(GMEM_MOVEABLE, kbsize)))
        return fail("Could not allocate\ntype ahead buffer");
        
    fpKeyboard = GlobalLock(hKeyboard);

    *fpBuffer = '\0';
    fpBuffer++;

    return TRUE;
    }

/* -----------------------------------------------------------------------  */
/* Undoes the work of the above. Allows an application to close the window  */
/* Terminates the prog.                                                     */
/* -----------------------------------------------------------------------  */
void winio_end()
    {
    while (tWinioVisible)
        winio_yield();
    }

/* -------------------------------------------------------------------  */
/* Closes the window by sending it a WM_DESTROY message. Note that it   */
/* does not disable the _onclose defined function. So the user program  */
/* handler will be triggered. Does NOT cause the app. to terminate.     */
/* -------------------------------------------------------------------  */
void winio_close()
    {
    tTerminate = FALSE;
    DestroyWindow(hwnd);
    tTerminate = TRUE;
    }
    
/* -------------------------------------------------------------------  */
/* processes any outstanding events waiting. These may be characters    */
/* typed at the keyboard, WM_PAINT messages, etc. It is called          */
/* internally but should also be used liberally by the application      */
/* within loops.                                                        */
/* -------------------------------------------------------------------  */
void winio_yield()
    {
    MSG msg;
    /* while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))	 
	   Only look for messages for the winio window */
    while (PeekMessage(&msg, hwnd, 0, 0, PM_REMOVE))
        {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
        }
    }  

/* -------------------------------------------------------------------  */
/* Let the application install an exit routine, called back from        */
/* winio_wmdestroy(). Deinstall by winio_onclose(NULL)                  */
/* -------------------------------------------------------------------  */
void winio_onclose(DESTROY_FUNC exitfunc)
    {
    destroy_func = exitfunc;
    }

/* -------------------------------------------------------------------  */
/* This function allows the font of the window to be modified, and may  */
/* be used BEFORE winio_init. Currently, only SYSTEM_, ANSI_, and       */
/* OEM_FIXED_FONTs are supported.                                       */
/* -------------------------------------------------------------------  */
BOOL winio_setfont(WORD wFont)
    {
    if ((wFont != SYSTEM_FIXED_FONT) &&
        (wFont != ANSI_FIXED_FONT) &&
        (wFont != OEM_FIXED_FONT) )
        return FALSE;
    curr_font = wFont;
    if (tWinioVisible)
        {
        set_font();
        if (tPaint)
            InvalidateRect(hwnd, NULL, TRUE);
        }
    return TRUE;
    }

/* -------------------------------------------------------------------  */
/* This function allows the title of the window to be modified, and may */
/* be used BEFORE winio_init.                                           */
/* -------------------------------------------------------------------  */
void winio_settitle(char *pchTitle)
    {
    strncpy(winio_title, pchTitle, 127);
    winio_title[127] = '\0';
    if (tWinioVisible)
        SetWindowText(hwnd, winio_title);
    }

/* -------------------------------------------------------------------  */
/* This function allows the caller to specifiy immediate or deferred    */
/* screen updates. The call may not be issued before winio_init().      */
/* -------------------------------------------------------------------  */
BOOL winio_setpaint(BOOL on)
    {
    BOOL ret = tPaint;
    
    CHECK_INIT();
    if (tPaint = on)
        InvalidateRect(hwnd, NULL, TRUE);
    return ret;
    }

/* ---------------------------------------------------------------  */
/* Our WM_PAINT handler. It sends the currrent 'in view' piece of   */
/* the buffer to the window. Note that an embedded NULL character   */
/* signifies an end of line, not '\n'.                              */
/* ---------------------------------------------------------------  */
long winio_wmpaint(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    HDC hdc;
    PAINTSTRUCT ps;
    char *pchSOL = fpTopOfWin;
    char *pchEOL;
    int i, j, xStart;
    int xLeft, xRight, yTop, yBottom;

    hdc = BeginPaint(hwnd, &ps);

    xLeft = (ps.rcPaint.left / cxChar) + xLeftOfWin;
    xRight = (ps.rcPaint.right / cxChar) + xLeftOfWin;
    yTop = ps.rcPaint.top / cyChar;
    yBottom = ps.rcPaint.bottom / cyChar;
    SelectObject(hdc, GetStockObject(curr_font));

    for (i = 0; i < yTop; i++)      // lines above repaint region
        {
        while (*pchSOL)
            pchSOL++;
        pchSOL++;
        }

    if (i <= yCurrLine) // something needs repainting..
        {
        for (i = yTop; i <= yBottom; i++)   // lines in repaint region
            {
            for (j = 0; (j < xLeft) && (*pchSOL); j++, pchSOL++)
                ; // Scroll right
            pchEOL = pchSOL;
            xStart = j - xLeftOfWin;
            for (j = 0; (*pchEOL) ; j++, pchEOL++) ; // end of line
            TextOut(hdc, cxChar * xStart, cyChar * i, pchSOL,
                    min(j, xRight - xLeft + 2));
            if ((unsigned)(pchEOL - fpBuffer) >= bufused)
                break;
            pchSOL = ++pchEOL;  
            }
        }
    
    EndPaint(hwnd, &ps);
    adjust_caret();
    return 0;
    }

/* ---------------------------------------------------------------  */
/* Our WM_SIZE handler. It updates the internal record of our       */
/* window size, minus the scroll bars, and recalcs the scroll bar   */
/* ranges.                                                          */
/* ---------------------------------------------------------------  */
long winio_wmsize(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    cxWidth = LOWORD(lParam);
    cyHeight = HIWORD(lParam);

    xWinWidth   = (cxWidth - cxScroll ) / cxChar;
    yWinHeight  = (cyHeight - cyScroll ) / cyChar;

    cScrollLR[SB_PAGEUP]    = -xWinWidth / 2;
    cScrollLR[SB_PAGEDOWN]  = +xWinWidth / 2;
    cScrollUD[SB_PAGEUP]    = -yWinHeight + 1;
    cScrollUD[SB_PAGEDOWN]  = +yWinHeight - 1;
    
    SetScrollRange(hwnd, SB_HORZ, 1, MAX_X, FALSE);
    SetScrollPos(hwnd, SB_HORZ, xLeftOfWin + 1, TRUE);

    SetScrollRange(hwnd, SB_VERT, 1, yCurrLine + 1, FALSE);
    SetScrollPos(hwnd, SB_VERT, yTopOfWin + 1, TRUE);
    
    return 0;
    }

/* ---------------------------------------------------------------  */
/* Our WM_DESTROY handler. It frees up storage associated with the  */
/* window, and resets its state to uninitialized.                   */
/* ---------------------------------------------------------------  */
long winio_wmdestroy(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    if (destroy_func)
        (*destroy_func)();
    GlobalUnlock(hBuffer);
    GlobalUnlock(hKeyboard);
    GlobalFree(hBuffer);
    GlobalFree(hKeyboard);
    tWinioVisible = FALSE;
    if (tTerminate) exit(0);
    return 0;
    }

/* --------------------------------------------------------------- */
/* Our WM_char handler. It adds the char to the internal kb buffer */
/* if there is room otherwise it queues a BEEP                     */
/* --------------------------------------------------------------- */
long winio_wmchar(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    char *lpchKeybd = fpKeyboard;
    unsigned pchSave = pchKbIn;
    
    pchKbIn++;
    if (pchKbIn == TYPE_AHEAD)
        pchKbIn = 0;
    if (pchKbIn == pchKbOut)
        {
        MessageBeep(0);
        pchKbIn = pchSave;
        }
    else
        *(lpchKeybd + pchSave) = LOBYTE(wParam);

    return 0;
    }

/* ---------------------------------------------------------------  */
/* Our WM_KEYDOWN handler. This handles what would be called        */
/* function keys in the DOS world. In this case the function keys   */
/* operate as scroll bar controls, so we generate messages to the   */
/* scroll message handlers below.                                   */
/* ---------------------------------------------------------------  */
long winio_wmkeydown(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    int hSB, vSB;
    
    if ((wParam < VK_PRIOR) || (wParam > VK_DOWN))
        return 0;
    
    hSB = VKtoSB[wParam - VK_PRIOR].hSB;
    vSB = VKtoSB[wParam - VK_PRIOR].vSB;
    if (hSB != DO_NOTHING)
        SendMessage(hwnd, WM_HSCROLL, hSB, 0L);
    if (vSB != DO_NOTHING)
        SendMessage(hwnd, WM_VSCROLL, vSB, 0L);
    return 0;
    }

/* --------------------------------------------------------------- */
/* Our WM_HSCROLL handler. It adjusts what part of the buffer      */
/* is visible. It operates as left/right arrow keys.               */
/* --------------------------------------------------------------- */
long winio_wmhscroll(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    int cxSave = xLeftOfWin,
        xInc = cScrollLR[wParam];
    
    if (xInc == DO_NOTHING)
        return 0;
    else if (xInc == USE_PARAM)
        xLeftOfWin = LOWORD(lParam) - 1;
    else
        xLeftOfWin += xInc;
    
    if ((xLeftOfWin = max(0, min(MAX_X - 1, xLeftOfWin))) == cxSave)
        return 0;

    ScrollWindow(hwnd, (cxSave - xLeftOfWin) * cxChar, 0, NULL, NULL);
    SetScrollPos(hwnd, SB_HORZ, xLeftOfWin + 1, TRUE);
    UpdateWindow(hwnd);

    return 0;
    }

/* --------------------------------------------------------------- */
/* Our WM_VSCROLL handler. It adjusts what part of the buffer      */
/* is visible. It operates as page and line up/down keys.          */
/* --------------------------------------------------------------- */
long winio_wmvscroll(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    int cySave = yTopOfWin,
        yInc = cScrollUD[wParam],
        i;
    
    if (yInc == DO_NOTHING)
        return 0;
    else if (yInc == USE_PARAM)
        yTopOfWin = LOWORD(lParam) - 1;
    else
        yTopOfWin += yInc;

    if ((yTopOfWin = max(0, min(yCurrLine, yTopOfWin))) == cySave)
        return 0;

    if (yTopOfWin > cySave)
        for (i = cySave; i < yTopOfWin; i++)
            fpTopOfWin = nextline(fpTopOfWin);
    else
        for (i = cySave; i > yTopOfWin; i--)
            fpTopOfWin = prevline(fpTopOfWin);
        
    ScrollWindow(hwnd, 0, (cySave - yTopOfWin) * cyChar, NULL, NULL);
    SetScrollPos(hwnd, SB_VERT, yTopOfWin + 1, TRUE);
    UpdateWindow(hwnd);

    return 0;
    }

/* ---------------------------------------------------------------  */
/* Our WM_SETFOCUS handler. It sets up the text caret.              */
/* ---------------------------------------------------------------  */
long winio_wmsetfocus(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    CreateCaret(hwnd, NULL, CARET_WIDTH, cyChar);
    
    if ((tCaret = winio_caret_visible()))
        {
        SetCaretPos((xCurrPos - xLeftOfWin) * cxChar,
                    (yCurrLine - yTopOfWin) * cyChar);
        ShowCaret(hwnd);
        }

    return 0;
    }

/* ---------------------------------------------------------------  */
/* Our WM_KILLFOCUS handler. It destroys the text caret.            */
/* ---------------------------------------------------------------  */
long winio_wmkillfocus(HWND hwnd, unsigned message, WORD wParam, LONG lParam)
    {
    if (tCaret)
        {
        HideCaret(hwnd);
        tCaret = FALSE;
        }
    DestroyCaret();
    return 0;
    }

void set_font(void)
    {
    HDC hdc;
    TEXTMETRIC tm;
        
    hdc = GetDC(hwnd);
    SelectObject(hdc, GetStockObject(curr_font));
    GetTextMetrics(hdc,&tm);
    ReleaseDC(hwnd,hdc);
    cxChar = tm.tmAveCharWidth;
    cyChar = tm.tmHeight+tm.tmExternalLeading;
    xWinWidth   = (cxWidth - cxScroll ) / cxChar;
    yWinHeight  = (cyHeight - cyScroll ) / cyChar;
    }

/* ---------------------------------------------------------------  */
/* Adjusts the position of the caret, and shows or hides it, as     */
/* appropriate.                                                     */
/* ---------------------------------------------------------------  */
void adjust_caret()
    {
    int t = winio_caret_visible();

    if (t)
        SetCaretPos((xCurrPos - xLeftOfWin) * cxChar,
                    (yCurrLine - yTopOfWin) * cyChar);
    if (t && (! tCaret))
        ShowCaret(hwnd);
    if ((! t) && tCaret)
        HideCaret(hwnd);
    tCaret = t;
    }

/* ---------------------------------------------------------------  */
/* Computes, on the basis of what has just been updated, what area  */
/* of the window needs to be repainted.                             */
/* ---------------------------------------------------------------  */
void compute_repaint(void)
    {
    RECT rc;
    static int xCP = 0, yCL = 0;
    int tWholeWin = FALSE;
    
    if (yCurrLine > (yTopOfWin + yWinHeight))
        {
        yTopOfWin = 0;
        fpTopOfWin = fpBuffer;
        while (yTopOfWin < (yCurrLine - ((yWinHeight + 1) / 2)))
            {
            fpTopOfWin = nextline(fpTopOfWin);
            yTopOfWin++;
            }
        tWholeWin = TRUE;
        }

    if ((xCurrPos < xLeftOfWin) || (xCurrPos > (xLeftOfWin + xWinWidth)))
        {
        xLeftOfWin = 0;
        while (xLeftOfWin < (xCurrPos - ((xWinWidth + 1) / 2)))
            xLeftOfWin++;
        tWholeWin = TRUE;
        }

    if (tWholeWin)
        InvalidateRect(hwnd, NULL, TRUE);
    else
        {
        rc.left = ((yCL == yCurrLine) ?
            (min(xCP, xCurrPos) - xLeftOfWin) * cxChar : 0);
        rc.top = (yCL - yTopOfWin) * cyChar;
        rc.right = (xWinWidth + 1) * cxChar;
        rc.bottom = (yCurrLine - yTopOfWin + 1) * cyChar;
        InvalidateRect(hwnd, &rc, TRUE);
        }
    
    yCL = yCurrLine;
    xCP = xCurrPos;
    }

/* ---------------------------------------------------------------  */
/* Adds the supplied cch-long string to the display buffer, and     */
/* ensures any changed part of the window is repainted.             */
/* ---------------------------------------------------------------  */
void addchars(char *pch, unsigned cch)
    {
    int ycSave = yCurrLine;
    int ytSave = yTopOfWin;
    int xSave = xLeftOfWin;

    make_avail(cch);

    append2buffer(pch, cch);

    if (ycSave != yCurrLine)
        SetScrollRange(hwnd, SB_VERT, 1, yCurrLine + 1, FALSE);

    if (! tPaint)
        return;
    
    compute_repaint();

    cScrollUD[SB_TOP]       = -yCurrLine;
    cScrollUD[SB_BOTTOM]    = yCurrLine;
    if (ytSave != yTopOfWin)
        SetScrollPos(hwnd, SB_VERT, yTopOfWin + 1, TRUE);       

    if (xSave != xLeftOfWin)
        SetScrollPos(hwnd, SB_HORZ, xLeftOfWin + 1, TRUE);

    winio_yield();
    }

/* ---------------------------------------------------------------  */
/* Add chars onto the display buffer, wrapping at end of line,      */
/* expanding tabs, etc.                                             */
/* ---------------------------------------------------------------  */
void append2buffer(char *pch, unsigned cch)
    {
    unsigned i;
    
    for (i = 0; i < cch; i++, pch++)
        {
        switch (*pch)
            {
            case '\n' :
                *pch = '\0';
                *(fpBuffer + bufused) = '\0';
                bufused++;
                fpCurrLine = fpBuffer + bufused;
                yCurrLine++;
                xCurrPos = 0;
                bufSOI = bufused;
                break;
            case '\t' :
                do  {
                    *(fpBuffer + bufused) = ' ';
                    bufused++;
                    xCurrPos++;
                    } while ((xCurrPos % TABSIZE) != 0);
                break;
            case EOF :
                break;
            case '\b' :
                if (bufused > bufSOI)
                    {
                    bufused--;
                    xCurrPos--;
                    }
                break;
            case 0x1b :
                while (bufused > bufSOI)
                    {
                    bufused--;
                    xCurrPos--;
                    }
                break;
            case 0x07 :
                MessageBeep(0);
                break;
            default :
                if (*pch > 0x1a)
                    {
                    if (xCurrPos >= MAX_X)
                        {
                        *(fpBuffer + bufused) = '\0';
                        bufused++;
                        xCurrPos = 0;
                        yCurrLine++;
                        fpCurrLine = fpBuffer + bufused;
                        }
                    xCurrPos++;
                    *(fpBuffer + bufused) = *pch;
                    bufused++;
                    }
            }
        }
    
    *(fpBuffer + bufused) = '\0'; // '\0' terminator after end of buffer
    }

/* ---------------------------------------------------------------  */
/* If we have run out of room in the display buffer, drop whole     */
/* lines, and move the remaining buffer up.                         */
/* ---------------------------------------------------------------  */
void make_avail(unsigned cch)
    {
    unsigned cDiscard = 0;
    char *fpTmp;
    unsigned i;

    if ((unsigned long)(bufused + cch + TABSIZE) < bufsize)
        return;

    fpTmp = fpBuffer;
    cDiscard = ((max(MIN_DISCARD, cch + 1) + MIN_DISCARD - 1)
        / MIN_DISCARD)      // this gives a whole number of
        * MIN_DISCARD;      // our allocation units.
    fpTmp += (LONG) cDiscard;
    fpTmp = nextline(fpTmp);
    cDiscard = fpTmp - fpBuffer; 
    memcpy(fpBuffer, fpTmp, bufused - cDiscard + 1);
    bufused -= cDiscard;
    if ((int) bufSOI != -1) bufSOI -= cDiscard;
    fpTmp = fpBuffer + (LONG) bufused;
    for (i = 0; i < cDiscard; i++) *fpTmp++ = '\0';
    fpCurrLine = fpBuffer;
    xCurrPos = yCurrLine = 0;
    for (i = 0; i < bufused; i++)
        {
        if (*fpCurrLine)
            xCurrPos++;
        else
            {
            xCurrPos = 0;
            yCurrLine++;
            }
        fpCurrLine++;
        }
    xLeftOfWin = yTopOfWin = -9999;
    
    InvalidateRect(hwnd, NULL, TRUE);
    }


/* -------------------------------------------------------------------  */
/* These two routines find the beginning of the next, and previous      */
/* lines relative to their input pointer                                */
/* -------------------------------------------------------------------  */

char *nextline(char *p) { while (*p) p++; return ++p; }
char *prevline(char *p) { p--; do p--; while (*p); return ++p; }

/* -------------------------------------------------------------------  */
/* Waits for a character to appear in the keyboard buffer, yielding     */
/* while nothing is available. Then inserts it into the buffer.         */
/* -------------------------------------------------------------------  */
int chInput(void)
    {
    char *lpchKeyBd;
    char c;
    
    CHECK_INIT();
    while (pchKbIn == pchKbOut)
        winio_yield();
        
    lpchKeyBd = fpKeyboard;
    c = *(lpchKeyBd + pchKbOut);

    pchKbOut++;
    if (pchKbOut == TYPE_AHEAD)
        pchKbOut = 0;
    
    // Do CR/LF and EOF translation
    return (c == 0x1a) ? EOF : (c == '\r') ? '\n' : c;
    }

/*
Addition to WINIO that prompts for working directory
command line args and then calls main.

Dan Larner, 1995
*/


// *****************************************************************
// Globals

// filters for common file dialogs (ugly initializer, but works)
/*
	"All Files (*.*)", "*.*",
	"" };
*/
const char g_szCDFilters[] = { 
 'A', 'l', 'l', ' ', 'F', 'i', 'l', 'e', 's', '(', '*', '.', '*', ')', '\0',
 '*', '.', '*', '\0',
 '\0'
};

/* points to where to arguments can be read into */
char* g_szArguments = NULL; 

// -----------------------------------------------------------------
// Setupofname - set up default filename structure for use
//               common file dialogs

void Setupofname (OPENFILENAME* ofname, HWND hwnd, char* p_szCDPathName)
{

p_szCDPathName[0] = '\0';

ofname->lStructSize = sizeof(OPENFILENAME);    // structure length
ofname->hwndOwner = hwnd;                      // owner window
ofname->hInstance = NULL;                      // data block handle
ofname->lpstrFilter = g_szCDFilters;	       // filter list
ofname->lpstrCustomFilter = NULL;              // custom filter
ofname->nMaxCustFilter = 0;                    // custom filter length
ofname->nFilterIndex = 1;                      // *.ISL filter
ofname->lpstrFile = p_szCDPathName;            // where to put pathname
ofname->nMaxFile = _MAX_PATH;                  // size of lpstrFile buffer
ofname->lpstrFileTitle = NULL;      		   // just filename buffer
ofname->nMaxFileTitle = 0;     				   // size of nMaxFileTitle buffer
ofname->lpstrInitialDir = NULL;                // initial directory
ofname->lpstrTitle = "Set or Select Any File to Set Default Drive & Directory";  // dialog box title
ofname->Flags = OFN_PATHMUSTEXIST |            // dialog init flags
				OFN_HIDEREADONLY;
ofname->lpstrDefExt = NULL;                    // default extension to add
ofname->lpTemplateName = NULL;                 // custom dialog template file

return;

}

/* sets up our working drive and directory - returns 1 on success, else 0.  
   On success, g_szCDPathName contains the directory, and g_szCDFileTitle 
   contains the file name (which we don't care about) */
int set_working_drive_directory (HWND h_wnd, char* p_szCDPathName) {

	OPENFILENAME ofname;       /* common dialog box structure   */

	/* set up the structure */
	Setupofname (&ofname, h_wnd, p_szCDPathName);

	/* Display the Open dialog box. */
	if (GetOpenFileName(&ofname)) {
	  p_szCDPathName[ofname.nFileOffset] = '\0';
	return 1;
	}

	return 0;
}


/* sets up the argc and argv used to call a main()
	p_argc is the address of argc
	argv is the argument array  (should declared in caller as
		char* argv [<maximum number of args + 2>]; 
		(+ 2 since argv[0] is the program name and the last argv will be NULLed)
	i_arvdimension is the dimension of argv, i. e. <maximum number of args + 2>
	pc_title is the title to give the window
	pc_progname is the program name (i.e. what argv[0] will be set to)
	pc_arg_space is where the actual argument strings can be stored and
		should be declared in the caller as
		char szArguments[<large enough to hold all the argument strings>]; 
 	Other args are what the apps winmain receives

	Note that if the command line is not null and the first char of it is
	not null, then is is assumed that the working directory is already established
	and that arguments should be taken from the command line rather than prompted for.
	Note that pc_arg_space should still be provided as this routine preseves the
	command line for (potentially) other uses by the application.

	See example at end of this file
*/
int winio_setmain(HANDLE hInstance, HANDLE hPrevInstance, 
    LPSTR lpCmdLine, int nCmdShow,  
    int* p_argc, char * argv[], int i_arvdimension,
    char* pc_title, char* pc_progname, char* pc_arg_space) {
    
   	HWND h_winio_wnd;
	char sz_delims[] = {' ', '\t', '\0'};
	char* pc_argptr;
	int i_index;
	MSG msg;
	int i_args_on_command_line = 0;

	/* for getting pathnames from common dialogs */
	char szCDPathName [_MAX_PATH + 1];

	/* See if args were given on the command line. If so, we assume
	   our working directory is already established, and we'll take our
	   args from the command line */
	if ((lpCmdLine != NULL) && (lpCmdLine[0] != '\0'))
		i_args_on_command_line = 1;	

	/* set up our window title */
    winio_settitle(pc_title);

	/* init winio */
    if (! winio_init(hInstance, hPrevInstance, nCmdShow, 0))
        return -1;

	/* get the handle to winio's window */
   	h_winio_wnd =  winio_hwnd();

	if (i_args_on_command_line == 0) { 
	    /* if our directory is not already established */
		if (set_working_drive_directory(h_winio_wnd, szCDPathName) != 1) { 
    		winio_close(); /* couldn't get file or cancelled - cleanup */
			return -1;
		}
	}

	/* Note the drive and directory are set as a byproduct of the file 
	dialog box  */
 
	/* show where we're working */
	printf("Working Drive = %c\n", ('A' + _getdrive() - 1));
	_getcwd(szCDPathName, _MAX_PATH);
	printf("Working Directory = %s\n", szCDPathName);

	if (i_args_on_command_line == 0) { 
	    /* if our arguments are not already supplied */ 
		/* get any arguments */
		printf("Enter args:\n");
		while (g_szArguments != NULL)
		/* prevent problems with shared dll data in win32s
		/* NOTE in Win3.x, the apps share the dll's data - thus if more than 
	  		one application is calling setupmain at once, there is a potential problem
	  	with this global, so we loop till it's NULL */
		PeekMessage(&msg, NULL, 0, 0, PM_NOREMOVE);

		g_szArguments = pc_arg_space;

		if (gets(g_szArguments) == NULL) {
			winio_close(); /* cleanup */
			return -1;
		}
	}
	else {
		strcpy(pc_arg_space, lpCmdLine); /* don't disturb command line */ 
		g_szArguments = pc_arg_space;
	}


	printf("Arguments = %s\n", g_szArguments);

	/* build up the argv array */
	argv[0] = pc_progname;  /* set program name */
	(*p_argc) = 1;
	if ((pc_argptr = strtok(g_szArguments, sz_delims)) != NULL) {
		argv[(*p_argc)] = pc_argptr;
		(*p_argc)++;
		while (((pc_argptr = strtok(NULL, sz_delims)) != NULL) &&
			   ((*p_argc) < (i_arvdimension - 1))) {
		 argv[(*p_argc)] = pc_argptr;
		 (*p_argc)++;
		}
	}
	/* Null the last entry */
	argv[(*p_argc)] = NULL;
   	g_szArguments = NULL;

	/* show the argv array */
	for (i_index = 0; i_index < (*p_argc); i_index++)
		printf("argv[%i] = %s\n", i_index, argv[i_index]);

	/* return success */
	return 1;
}

void _winio_console_close(void) {
	/* set tWinioVisible FALSE so the winio_end function (added to the 
   		atexit chain in winio_init) won't loop */
	tWinioVisible = FALSE;
	winio_close();
}

/* creates an auxillary window for a Windows application,
   that receives all stdio output. 
   First 4 arguments are as in winio_init, 
   pc_title is as in winio_settitle */
 
int winio_console(HANDLE hInstance, HANDLE hPrevInstance,
            int nCmdShow, unsigned wBufSize, char* pc_title) {

	int i_initresult;

	winio_settitle(pc_title);

	i_initresult = winio_init(hInstance, hPrevInstance, nCmdShow, 0);

	atexit(_winio_console_close);

	return (i_initresult);
}


/* ********************** winio_setmain Example **********************
#include <windows.h>
#include <winio.h>
int main ( int argc, char *argv[ ]) {
	int i_index;
	printf("In main \n");
	for (i_index = 0; i_index < argc; i_index++)
		printf("argv[%i] = %s\n", i_index, argv[i_index]);
 	return 0;
}
int CALLBACK WinMain(HANDLE hInstance, HANDLE hPrevInstance, 
    LPSTR lpCmdLine, int nCmdShow) {
   	int i_argc;
	// we expect 8 args maximum 
	char* argv [10];
	// the chars in the args (including whitespace) will not exceed 512 
	char szArguments[512]; 
	int i_result;
	if (winio_setmain(hInstance, hPrevInstance, lpCmdLine, nCmdShow,  
    	&i_argc, argv, 10, "My Application", "myapp.exe", szArguments) != 1) 
    	return -1;
   	i_result = main(i_argc, argv); // call main
	winio_end(); 	// let user look at output till user closes 
    return i_result;
}
*/
