
/* This module registers a high score with the official Maelstrom
   score server
*/
#include <sys/types.h>
#include <signal.h>
#include <errno.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/socket.h>

#include "Maelstrom_Globals.h"
#include "netscore.h"

static int Goto_ScoreServer(char *server, int port);

/* This function actually registers the high scores */
void RegisterHighScore(Scores high)
{
	int sockfd;
	char scorebuf[1024];

	if ( (sockfd=Goto_ScoreServer(SCORE_HOST, SCORE_PORT)) < 0 )
		return;

	/* Send CRC validation check */
	sprintf(scorebuf, "CRC = 0x%x\n", Goto_ScoreServer);
	write(sockfd, scorebuf, strlen(scorebuf));

	/* Send the scores */
	sprintf(scorebuf, SCOREFMT, high.name, high.score, high.wave);
	write(sockfd, scorebuf, strlen(scorebuf));
	close(sockfd);
}

/* Connect to the score server and show the high scores */
void PrintHighScores(void)
{
	int sockfd;
	int len;
	char netbuf[1024];
	FILE *stream;

	if ( (sockfd=Goto_ScoreServer(SCORE_HOST, SCORE_PORT)) < 0 ) {
		fprintf(stderr,
			"Couldn't connect to Maelstrom Score Server.\n");
		return;
	}
	
	/* Read the welcome banner */
	stream = fdopen(sockfd, "r");
	fgets(netbuf, 1024-1, stream);

	/* Send our request */
	sprintf(netbuf, "SHOWSCORES\n");
	write(sockfd, netbuf, strlen(netbuf));

	/* Read the response */
	while ( fgets(netbuf, 1024-1, stream) != NULL ) {
		write(1, netbuf, strlen(netbuf));
	}
	fclose(stream);		// Does this close sockfd?
}

static int timed_out;
static void timeout(int sig)
{
	timed_out = 1;
}
static int Goto_ScoreServer(char *server, int port)
{
	struct sockaddr_in serv_addr;
	struct hostent    *hp;
	int                sockfd;
	unsigned char      netbuf[1024];

	/*
	 * Fill in the structure "serv_addr" with the address of the
	 * server that we want to connect with.
	 */
	memset(&serv_addr, 0, sizeof(serv_addr));
	if ( (serv_addr.sin_addr.s_addr=inet_addr(server)) == -1 ) {
		/* It's not a dotted-decimal address */
		if ( (hp=gethostbyname(server)) == NULL ) {
			/*fprintf(stderr, "%s: host name error.\n", server);*/
			return(-1);
		}
		else 
			memcpy(&serv_addr.sin_addr, hp->h_addr, hp->h_length);
	}
	serv_addr.sin_family      = AF_INET;
	serv_addr.sin_port        = htons(port);

	/*
	 * Open a TCP socket (an Internet stream socket).
	 */
	if ( (sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
		return(-1);

	/* Set up 30 second timeout just in case */
	signal(SIGALRM, timeout);
	alarm(30);

	timed_out=0;
	if ( connect(sockfd, (struct sockaddr *)&serv_addr, sizeof(serv_addr))
									< 0 ) {
		if ( errno == EINTR )
			errno = EAGAIN;
		return(-1);
	}
	if ( timed_out ) {
		errno = EAGAIN;
		return(-1);
	}
	return(sockfd);
}
