/************************************************************************
 * rpncalc.c								*
 *									*
 * A little RPN (Reverse Polish Notation) calculator,                   *
 * rudimentary emulating a HP 28S. 					*
 * 								        *
 * rpncalc is (c) David Frey, 1993, 1994, 1995				*
 *								        * 
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *									* 
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *									* 
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *									* 
 ************************************************************************/

/* $Id: rpncalc.c,v 1.0 1995/12/31 18:14:13 david Rel $
 * $Log: rpncalc.c,v $
 * Revision 1.0  1995/12/31 18:14:13  david
 * Initial revision
 * */ 

#include <stdio.h>
#include <stdlib.h> 
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <signal.h> 

#ifdef HAVE_READLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

#include "rpncalc.h"
#include "cmds.h"
#include "utils.h"

extern int pushtostack;

char *progname;

static struct option const long_options[] =
{
  {"help", no_argument, 0, 'h'},
  {"help", no_argument, 0, '?'},
  {"version", no_argument, 0, 'v'},
  {0, 0, 0, 0}
};

double help(void)
{
  printf("The following operations will be emulated:\n");
  printf("push  pop  pick  swap  over  roll  dup  dupn  drop  dropn  depth"\
	 "\n\n");
  printf("chs   +    -     *     /     ^     inv  sqrt  sqr\n");
  printf("sin   cos  tan   asin  acos  atan  atan2\n");
  printf("sinh  cosh tanh  asinh acosh atanh\n");
  printf("ln    log  ld    exp   alog  shl\n");
  printf("j0    j1   jn    y0    y1    yn\n");
  printf("erf   erfc lgamma\n");
  printf("abs   ceil fact  mod  div    gcd   sum  prod\n\n");
  printf("hex   dec  oct   and  &      or    |    not   !     xor\n");
  printf("prec\n");
  printf("The following constants will be recognised: pi, e.\n");
  printf("Delimiters are ',', ';', space, tab and newline.\n");
  printf("Case is significant. Use prefix 0x for hexadecimal "\
	 "and 0 for octal constants.\n");

  return 0.0; /* dummy value */
}

double warranty(void)
{
  printf("This program is free software; you can redistribute it and/or "\
	 "modify\n");
  printf("it under the terms of the GNU General Public License as published "\
	 "by\n");
  printf("the Free Software Foundation; either version 2 of the License, "\
	 "or\n");
  printf("(at your option) any later version.\n");
  printf("\n");
  printf("This program is distributed in the hope that it will be useful,\n");
  printf("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
  printf("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
  printf("GNU General Public License for more details.\n");
  printf("\n");
  printf("You should have received a copy of the GNU General Public License"\
	 "\n");
  printf("along with this program; if not, write to the Free Software\n");
  printf("Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n");

  return 0.0; /* dummy value */
}

static int cmp(const void *f1, const void *f2)
/* (String-)Compare the two function names, cf. iX 11/95 p 212 */
{
  return strcmp(((const struct cmd *)f1)->fname,
		((const struct cmd *)f2)->fname);
}

/* Evaluate the command line. The work is done in this procedure ! */
int eval(char line[])
{
  short int quit;		/* flags */
  static char delimiters[] = " ,;\t\n\0";
  char *cmnd, *rest;		 /* one command line */
  double op1, op2, res;		 /* operands */

  cmnd = strtok(line, delimiters); res = 0; pushtostack=TRUE;
  do
  {
    quit = ((strncmp(cmnd, "quit", 4) == 0) || (strncmp(cmnd,"q",1) == 0) || 
	    (strncmp(cmnd, "exit", 4) == 0));

    if ((!quit) && (cmnd != NULL))
    {
      /* Is it a number ? It yes, push it onto the stack. */      

      /* First assume it's a double */
      op1 = strtod(cmnd, &rest);
      if (strcmp(rest, "") == 0) 
      {
	push(op1); /* It is a float */
      }
      else
      {
	/* converting number according to the rules of the C programming
	 * language, e.g 0=octal, 0x=hexadecimal
	 */
	op1 = strtol(cmnd, &rest, 0);
	if (strcmp(rest, "") == 0) 
	{
	  push(op1); /* it is an integer */
	}
	else
	{
	  struct cmd dummykey;
	  struct cmd *f; 

	  /* the following bsearch-trick is from iX 11/95 p 212 */
	  dummykey.fname=cmnd;
   	  f=(struct cmd *)bsearch(&dummykey, cmdtab, NCMDTAB,
				   sizeof cmdtab[0], cmp);
	  if (f == NULL)
	  {
	    fprintf(stderr,"%s: unknown command `%s'.\n",progname, cmnd);
	  }
	  else
	  {
	    errno=0;
	    switch (f->argno)
	    {
	      case 0: res=(*f->fnct)(); break;
	      case 1: op1=pop(); res=(*f->fnct)(op1); break;
	      case 2: op2=pop(); op1=pop(); res=(*f->fnct)(op1,op2); break;
	    }
	    if (errno != 0) { perror(NULL); }
	    else
	    {
	      if (f->pushrestostack && pushtostack) push(res);
	    }
	  }
	}
      }
      cmnd = strtok(NULL, delimiters);
    }
  }
  while ((!quit) && (cmnd != NULL));
    
  return quit;
}

unsigned char *getline(int interactive)
/* Read a line of text without imposing an length limit */
{
  unsigned char *line=NULL;
  int c;
  int i;

#ifdef HAVE_READLINE
  if (interactive) line=(unsigned char*)readline(""); 
  else
  {
#endif
    i=0; line=xmalloc(LINEFRAGMENT+1);
    while (((c = getc(stdin)) != EOF) && (c != '\n') && (c != '\0'))
    {
     if ((i > LINEFRAGMENT) &&  (i % LINEFRAGMENT) == 0) 
       line=xrealloc(line,i+LINEFRAGMENT+1);
     line[i]=c; i++;
    }
    if ((c != EOF) && (c != '\0')) line[i]='\0';
    else
    {
      free(line); line=NULL; /* this is readline compatible; 
				return NULL when the input line is empty */
    }
#ifdef HAVE_READLINE
  }
#endif
  return line;
}
/* print a short usage statement. Indicate that the argument must be quoted. */
void usage(void)
{
  fprintf(stderr, "usage: %s [-h][-v] [\"expression\"]\n", progname);
}

int main(int argc, char *argv[])
{
  int c, i;
  char *line=NULL;		/* entire line  */
  short int quit;		/* should we quit ? */

  progname=strrchr(argv[0],'/');
  if (progname==NULL) progname=argv[0];
  else progname++;

  while ((c = getopt_long(argc, argv, "h?v",
			  long_options, (int *) 0)) != EOF)
  {
    switch (c)
    {
      case 0  : break;
      case 'h':
      case '?': usage(); return 0; break;
      case 'v': fprintf(stderr,"%s version 1.0\n", progname); 
	        return 1; break;
      default : break;
    }
  }

  fprintf(stderr,"%s version 1.0. Copyright (c) 1993, 1995 David Frey.\n",
	  progname);
  fprintf(stderr,"This is free software with ABSOLUTELY NO WARRANTY.\n");

  signal(SIGFPE,SIG_IGN); /* ignore floating point exceptions. */

  /* Sort the command table, from iX 11/95 p 212 */
  qsort(cmdtab, NCMDTAB, sizeof cmdtab[0], cmp);

  if (argc > optind)
  {
    short int l;

    l=0; for(i=optind;i<argc;i++) l += (strlen(argv[optind])+1);
    line=(unsigned char *)xmalloc(l+1);
    strcpy(line, argv[optind]);
    for (i = optind+1; i < argc; i++)
    {
      strcat(line, " "); strcat(line, argv[i]);
    }

    eval(line);    
    showstack();
    free(line); line=NULL;
  }
  else
  {
    int interactive;

    interactive=isatty(0); line=NULL;
    if (interactive)
    {
      fprintf(stderr,"For details, type `warranty'.\n");
      fprintf(stderr,"Type `quit' to quit and `?' to get a summary.\n");

#ifdef HAVE_READLINE    
      using_history();
#endif
    }

    do
    {
      line=getline(interactive);
      quit=(line == NULL);
      if (!quit)
      {
	/* Skip white spaces */
	while (*line != '\0' && isspace(*line)) line++;
	if (*line != '\0')
	{
#ifdef HAVE_READLINE          
	  if (interactive) add_history(line);
#endif
	  quit = eval(line);
	  if (!quit) showstack();
	}
      }
      free(line); line=NULL; /* line was malloc'd by readline/getline. */
    }
    while (!quit);
#ifdef HAVE_READLINE 
    if (interactive) remove_history();
#endif    
  }  
  return 0;
}						 /* main */
