# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <time.h>

# include "hfs.h"
# include "hcwd.h"
# include "hfsutil.h"
# include "version.h"

# include "hattrib.h"
# include "hcd.h"
# include "hcopy.h"
# include "hdel.h"
# include "hdir.h"
# include "hformat.h"
# include "hmkdir.h"
# include "hmount.h"
# include "hpwd.h"
# include "hrename.h"
# include "hrmdir.h"
# include "humount.h"
# include "hvol.h"

/*
 * NAME:	main()
 * DESCRIPTION:	program entry dispatch
 */
int main(int argc, char *argv[])
{
  int i;
  char *argv0;

  struct {
    char *name;
    int (*func)(int, char *[]);
  } list[] = {
    { "hattrib", hattrib_main },
    { "hcd",     hcd_main     },
    { "hcopy",   hcopy_main   },
    { "hdel",    hdel_main    },
    { "hdir",    hdir_main    },
    { "hformat", hformat_main },
    { "hmkdir",  hmkdir_main  },
    { "hmount",  hmount_main  },
    { "hpwd",    hpwd_main    },
    { "hrename", hrename_main },
    { "hrmdir",  hrmdir_main  },
    { "humount", humount_main },
    { "hvol",    hvol_main    },
    { 0,        0             }
  };

  if (argc > 1)
    {
      if (strcmp(argv[1], "--version") == 0)
	{
	  printf("%s - %s\n", VERSION, COPYRIGHT);
	  printf("`%s --license' for licensing information.\n", argv[0]);
	  return 0;
	}
      else if (strcmp(argv[1], "--license") == 0)
	{
	  printf("\n%s", LICENSE);
	  return 0;
	}
    }

  argv0 = strrchr(argv[0], '/');
  if (argv0 == 0)
    argv0 = argv[0];
  else
    ++argv0;

  for (i = 0; list[i].name; ++i)
    {
      if (strcmp(argv0, list[i].name) == 0)
	{
	  int ret;

	  if (hcwd_init() < 0)
	    {
	      perror("Failed to initialize HFS working directories");
	      return 1;
	    }

	  ret = list[i].func(argc, argv);

	  if (hcwd_finish() < 0)
	    {
	      perror("Failed to save working directory state");
	      return 1;
	    }

	  return ret;
	}
    }

  fprintf(stderr, "Unknown operation `%s'\n", argv0);
  return 1;
}

/*
 * NAME:	hfs->perror()
 * DESCRIPTION:	output an HFS error
 */
void hfs_perror(char *msg)
{
  fprintf(stderr, "%s: %s\n", msg, hfs_error);
}

/*
 * NAME:	hfs->remount()
 * DESCRIPTION:	mount a volume as though it were still mounted
 */
hfsvol *hfs_remount(char *progname, mountent *ment)
{
  hfsvol *vol;
  hfsvolent vent;

  if (ment == 0)
    {
      fprintf(stderr, "%s: No volume is current; use `hmount' or `hvol'\n",
	      progname);
      return 0;
    }

  vol = hfs_mount(ment->path, ment->partno);
  if (vol == 0)
    {
      hfs_perror(ment->path);
      return 0;
    }

  hfs_vstat(vol, &vent);

  if (strcmp(vent.name, ment->vname) != 0)
    {
      hfs_umount(vol);
      fprintf(stderr, "%s: Expected volume \"%s\" not found\n",
	      progname, ment->vname);
      fprintf(stderr, "%s: Replace media on %s or use `hmount'\n",
	      progname, ment->path);
      return 0;
    }

  if (hfs_chdir(vol, ment->cwd) < 0)
    {
      hfs_umount(vol);
      hfs_perror("Can't find current HFS directory");
      return 0;
    }

  return vol;
}

/*
 * NAME:	hfs->pinfo()
 * DESCRIPTION:	print information about a volume
 */
void hfs_pinfo(hfsvolent *ent)
{
  printf("Volume name is \"%s\"%s\n", ent->name,
	 (ent->flags & HFS_ISLOCKED) ? " (locked)" : "");
  printf("Volume was created on %s", ctime(&ent->crdate));
  printf("Volume was last modified on %s", ctime(&ent->mddate));
  printf("Volume has %lu bytes free\n", ent->freebytes);
}

/*
 * NAME:	hfs->getcwd()
 * DESCRIPTION:	return full path to current directory
 */
char *hfs_cwd(hfsvol *vol)
{
  char *path, name[33];
  long cwd;
  int pathlen;

  path    = malloc(1);
  path[0] = 0;
  pathlen = 0;
  cwd     = hfs_getcwd(vol);

  while (cwd != HFS_CNID_ROOTPAR)
    {
      char *new;
      int namelen, i;

      if (hfs_dirinfo(vol, &cwd, name) < 0)
	return 0;

      if (pathlen)
	strcat(name, ":");

      namelen = strlen(name);

      new = realloc(path, namelen + pathlen + 1);
      if (new == 0)
	{
	  free(path);
	  hfs_error = "out of memory";
	  return 0;
	}

      if (pathlen == 0)
	new[0] = 0;

      path = new;

      /* push string down to make room for path prefix (memmove()-ish) */

      i = pathlen + 1;
      for (new = path + namelen + pathlen; i--; new--)
	*new = *(new - namelen);

      memcpy(path, name, namelen);

      pathlen += namelen;
    }

  return path;
}
