/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <sys/stat.h>
#include "file.h"

char *expand_path(char *file_name)
{
    char	*path;
    long	len, pos;
    char	ch;

    if (*file_name == '~') {
	file_name++;
	pos = strlen(global.home);
	len = pos + 240;
	path = XtMalloc(len);
	strcpy(path, global.home);
    } else {
	pos = 0;
	len = 240;
	path = XtMalloc(len);
	path[0] = '\0';
    }

    for (ch = *file_name++ ; ch != '\0' ; ch = *file_name++) {
	if (pos + 8 > len) {
	    len = pos + 256;
	    path = XtRealloc(path, len);
	}

	if (ch != '%')
	    path[pos++] = ch;
	else {
	    char	*p, *c = NULL;
	    int		cap    = False;
	    int 	slash  = False;
	    int		clen   = 0;

	    ch = *file_name++;
	    switch (ch) {
	    case '%':
		path[pos++] = '%';
		continue; /* don't fall through */
	    case 'a':
	    case 'A':
		if (global.mode != NewsModeGroup &&
		    global.mode != NewsModeThread) {
		    fprintf(stderr, "knews: Not in a newsgroup!\n");
		    XtFree(path);
		    return NULL;
		}
		if (!global.curr_art) {
		    fprintf(stderr, "knews: No selected article!\n");
		    XtFree(path);
		    return NULL;
		}
		sprintf(path + pos, "%ld", global.curr_art->no);
		pos += strlen(path + pos);
		continue;
	    case 'g':
		slash  = True;
		break;
	    case 'G':
		cap    = True;
		slash  = True;
		break;
	    case 'n':
		break;
	    case 'N':
		cap    = True;
		break;
	    case 'p':
	    case 'P':
		c = global.nntp_server;
		if (c)
		    c = strchr(c, ':');
		if (!c)
		    continue;
		clen = strlen(c);
		break;
	    case 's':
	    case 'S':
		c = global.nntp_server;
		if (!c) {
		    fprintf(stderr, "knews: nntp_server is NULL!\n");
		    XtFree(path);
		    return NULL;
		}
		p = strchr(c, ':');
		if (p)
		    clen = p - c;
		else
		    clen = strlen(c);
		break;
	    default:
		fprintf(stderr,
			"knews: %%%c: Unknown format specifier.\n", ch);
		XtFree(path);
		return NULL;
	    }

	    if (!c)
		if ((global.mode == NewsModeGroup ||
		     global.mode == NewsModeThread) &&
		    global.curr_group && global.curr_group->name) {
		    c = global.curr_group->name;
		    clen = strlen(c);
		} else {
		    fprintf(stderr, "knews: Not in a newsgroup.\n");
		    XtFree(path);
		    return NULL;
		}

	    if (clen == 0)
		continue;
	    if (pos + clen + 8 > len) {
		len = pos + clen + 256;
		path = XtRealloc(path, len);
	    }

	    ch = *c++;
	    clen--;

	    if (cap && islower((unsigned char)ch))
		ch = toupper((unsigned char)ch);

	    path[pos++] = ch;
	    while (clen-- > 0) {
		ch = *c++;

		if (ch == '.' && slash)
		    ch ='/';
		path[pos++] = ch;
	    }
	    path[pos] = '\0';
	}
    }
    path[pos] = '\0';

    return path;
}

int open_mkdir(char *path, int flags, mode_t mode, int report)
{
    char	*c;
    int		fd;

    if (flags & O_CREAT)
	fd = open(path, flags, mode);
    else
	fd = open(path, flags);

    if (fd >= 0)
	return fd;

#ifndef O_ACCMODE
#define O_ACCMODE 3
#endif
    if (errno == ENOENT && (flags & O_ACCMODE) != O_RDONLY) {
	if (path[0] == '/')
	    c = strchr(path + 1, '/');
	else
	    c = strchr(path, '/');

	while (c) {
	    struct stat	st_buf;

	    *c = '\0';
	    if (stat(path, &st_buf) < 0 &&
		(errno != ENOENT || mkdir(path, 0755) < 0)) {
		*c++ = '/';
		return -1;
	    }

	    *c++ = '/';
	    c = strchr(c, '/');
	}
    }

    if (flags & O_CREAT)
	fd = open(path, flags, mode);
    else
	fd = open(path, flags);

    if (fd < 0 && (report || errno != ENOENT)) {
	int oerrno = errno;

	perror(path);
	errno = oerrno;
    }

    return fd;
}

int open_expand(char *file_name, int flags, mode_t mode, int report)
{
    char	*path;
    int		fd;

    path = expand_path(file_name);
    if (!path)
	return -1;

    fd = open_mkdir(path, flags, mode, report);
    XtFree(path);

    return fd;
}

FILE *fopen_mkdir(char *path, char *mode, int report)
{
    int		fd, flags;
    FILE	*ret;

    switch (*mode) {
    case 'a':
	flags = O_WRONLY|O_APPEND|O_CREAT;
	break;
    case 'r':
	flags = O_RDONLY;
	break;
    case 'w':
	flags = O_WRONLY|O_TRUNC|O_CREAT;
	break;
    default:
	fprintf(stderr, "knews: invalid mode to fopen_mkdir.\n");
	return NULL;
    }

    fd = open_mkdir(path, flags, 0644, report);
    if (fd < 0)
	return NULL;

    ret = fdopen(fd, mode);
    if (!ret) {
	perror("fdopen");
	close(fd);
    }

    return ret;
}

FILE *fopen_expand(char *file_name, char *mode, int report)
{
    char	*path;
    FILE	*fp;

    path = expand_path(file_name);
    if (!path)
	return NULL;

    fp = fopen_mkdir(path, mode, report);
    XtFree(path);

    return fp;
}

int unlink_expand(char *file_name)
{
    char	*path;
    int		ret;

    path = expand_path(file_name);
    if (!path)
	return -1;

    ret = unlink(path);
    XtFree(path);

    return ret;
}

int create_temp_fd(char **name)
{
    int	fd;

    *name = tmpnam(NULL);
    if (!*name)
	fd = -1;
    else {
	unlink(*name);
	fd = open(*name, O_RDWR|O_CREAT|O_TRUNC, S_IRUSR|S_IWUSR);
	if (fd < 0)
	    *name = NULL;
    }

    return fd;
}

FILE *create_temp_file(char **name)
{
    int	fd;

    fd = create_temp_fd(name);
    if (fd < 0)
	return NULL;

    return fdopen(fd, "w+");
}

char *snarf_file(int fd, long *lenp)
{
    struct stat	st_buf;
    char	*buffer;
    long	len, pos, n;

    if (fstat(fd, &st_buf) < 0) {
	perror("fstat");
	return NULL;
    }

    if (!S_ISREG(st_buf.st_mode)) {
	fprintf(stderr, "snarf_file: not a regular file!\n");
	errno = EINVAL;
	return NULL;
    }

    pos = 0;
    len = st_buf.st_size + 256;
    buffer = malloc(len + 1);
    if (!buffer) {
	perror("malloc");
	return NULL;
    }

    while ((n = read(fd, buffer + pos, len - pos)) != 0)
	if (n < 0)
	    if (errno == EINTR)
		continue;
	    else {
		perror("read");
		free(buffer);
		return NULL;
	    }
	else {
	    pos += n;
	    if (pos == len) {
		char	*tmp;

		len *= 2;
		tmp = realloc(buffer, len + 1);
		if (!tmp) {
		    free(buffer);
		    return NULL;
		}
		buffer = tmp;
	    }
	}

    buffer[pos] = '\0';
    if (lenp)
	*lenp = pos;

    return buffer;
}
