/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <pwd.h>
#include "file.h"
#include "newsrc.h"
#include "pcheck.h"
#include "psetup.h"
#include "thread.h"
#include "util.h"

static void str_trunc(char *c, int max)
{
    int		len = strlen(c);
    char	*p = strchr(c, '\n');

    if (!p && len < max + 6)
	return;

    if (!p || p > c + max)
	p = c + max;

    strcpy(p, " ...");
}

static int max_len(char *c, char sep)
{
    int		max = 0, len;
    char	*p;

    do {
	p = strchr(c, sep);
	if (!p)
	    len = strlen(c);
	else {
	    len = p - c;
	    c = p + 1;
	}
	if (len > max)
	    max = len;
    } while (p);

    return max;
}

static int an_admin(void)
{
    static int	inited = False;
    static int	result = False;

    if (!inited) {
	uid_t	uid;

	inited = True;
	if ((uid = getuid()) == 0)
	    result = True;
	else {
	    struct passwd	*pwd;

	    pwd = getpwnam("news");
	    if (pwd && uid == pwd->pw_uid)
		result = True;
	}
    }

    return result;
}

static char *check_cancel(char *field)
{
    while (*field != '\0') {
	ARTICLE	*art;
	char	*c;

	if (*field++ != '<')
	    return "Syntax error in control message.";

	c = field;
	while (*c != '\0' && *c != '\n' && *c != '>' && *c != '<')
	    c++;
	if (*c != '>')
	    return "Syntax error in control message.";

	if ((global.mode != NewsModeGroup && global.mode != NewsModeThread) ||
	    !(art = find_article(field, c - field)))
	    return
		"Error! Failed to authenticate cancel message:\n"
		"Couldn't find referenced article.";

	/* possible security hole */
	if (!strstr(art->from, global.mail_name) ||
	    !strstr(art->from, global.domain_name))
	    return "Error! You cannot cancel somebody else's article.";

	field = c + 1;
	while (IS_SPACE(*field) || *field == '\n')
	    field++;
    }

    return NULL; /* ok */
}

static int check_article(PostContext *context, char *art, char *message,
			 char **newsgroups, char **subject,
			 char **to, char **cc, int *line_len)
{
    int	has_from     = False;

    *newsgroups = NULL;
    *subject    = NULL;
    *to         = NULL;
    *cc         = NULL;
    *line_len   = 0;

#define RETURN_ERROR(err)     \
    do {                      \
	strcpy(message, err); \
	return False;         \
    } while (0)

    if (*art == '\0')
	RETURN_ERROR("Error!  Empty article.");
    if (*art == '\n')
	RETURN_ERROR("Error!  No headers in article.");

    /*
     * Check headers.
     */
    context->line = 1;
    while (*art != '\n') {
	char		*field = strchr(art, ':');
	char		*end;
	int		lines, len;
	unsigned char	ch;

	if (!field) {
	    if (strlen(art) > 128)
		art[128] = '\0';
	    sprintf(message,
		    "Syntax error in article header near\n\n"
		    "'%s'\n\n"
		    "Remember to put an EMPTY line between\n"
		    "the headers and the body (the article/mail text)!",
		    art);
	    return False;
	}

	len = field - art;
	*field++ = '\0';
	if (strchr(art, ' ') || strchr(art, '\t') || strchr(art, '\n')) {
	    if (strlen(art) > 128)
		art[128] = '\0';
	    sprintf(message,
		    "Syntax error in '%s' header.\n"
		    "Remember to put an EMPTY line between\n"
		    "the headers and the body (the article/mail text)!",
		    art);
	    return False;
	}

	if (*field++ != ' ') {
	    sprintf(message,
		    "Syntax error in '%s' header.\n"
		    "Must have a space after the colon!",
		    art);
	    return False;
	}

	while (IS_SPACE(*field))
	    field++;
	lines = 1;
	for (end = strchr(field, '\n') ;
	     end && IS_SPACE(end[1]) ;
	     end = strchr(end + 1, '\n'))
	    lines++;

	if (!end)
	    RETURN_ERROR("Error: article is all headers!");

	*end++ = '\0';

	ch = *art;
	if (isupper(ch))
	    ch = tolower(ch);

	switch (ch) {
	case 'a':
	    if (case_lstrcmp(art, "also-control") == 0 && !an_admin())
		if (case_lstrncmp(field, "cancel ", 7) != 0)
		    RETURN_ERROR("Error! No permission "
				 "to send control message.");
		else {
		    char	*tmp = check_cancel(field + 7);

		    if (tmp)
			RETURN_ERROR(tmp);
		}
	    break;
	case 'c':
	    if (case_lstrcmp(art, "cc") == 0) {
		*cc = field;
		str_trunc(*cc, 48);
	    } else if (case_lstrcmp(art, "control") == 0 && !an_admin())
		if (case_lstrncmp(field, "cancel ", 7) != 0)
		    RETURN_ERROR("Error! No permission "
				 "to send control message.");
		else {
		    char	*tmp = check_cancel(field + 7);

		    if (tmp)
			RETURN_ERROR(tmp);
		}
	    break;
	case 'f':
	    if (case_lstrcmp(art, "from") == 0) {
		if (has_from)
		    RETURN_ERROR("Error! Duplicate 'From:' header.");
		has_from = True;
		if (!strstr(field, global.domain_name) ||
		    !strstr(field, global.user_id))
		    context->flags |= NEEDS_SENDER;
	    } else if (case_lstrcmp(art, "followup-to")) {
		if (lines > 1)
		    RETURN_ERROR("Error! The 'Followup-To:' header cannot\n"
				 "be continued on several lines");
		if (*field == '\n' || *field == '\0')
		    RETURN_ERROR("Error! Empty 'Followup-To:' header");
		if (strchr(field, ' ') || strchr(field, '\t'))
		    RETURN_ERROR("Error! There can be no spaces between the\n"
				 "commas in the 'Followup-To:' header");
		break;
	    }
	case 'n':
	    if (case_lstrcmp(art, "newsgroups") == 0) {
		if (*newsgroups)
		    RETURN_ERROR("Error! Duplicate 'Newsgroups:' header.");
		if (lines > 1)
		    RETURN_ERROR("Error! The 'Newsgroups:' header cannot\n"
				 "be continued on several lines.");
		*newsgroups = field;
		if (*field == '\n' || *field == '\0')
		    RETURN_ERROR("Error! Empty 'Newsgroups:' header");
		if (strchr(field, ' ') || strchr(field, '\t'))
		    RETURN_ERROR("Error! There can be no spaces between the\n"
				 "commas in the 'Newsgroups:' header");
	    }
	    break;
	case 's':
	    if (case_lstrcmp(art, "sender") == 0)
		RETURN_ERROR("Error! You cannot specify a 'Sender:' header.");
	    else if (case_lstrcmp(art, "subject") == 0) {
		if (*subject)
		    RETURN_ERROR("Error! Duplicate 'Subject:' header.");
		if (*field == '\n' || *field == '\0')
		    RETURN_ERROR("Error! Empty 'Subject:' header.");
		*subject = field;
		str_trunc(*subject, 48);
	    } else if (case_lstrcmp(art, "supersedes") == 0 && !an_admin()) {
		char	*tmp = check_cancel(field);
		
		if (tmp)
		    RETURN_ERROR(tmp);
	    }
	    break;
	case 't':
	    if (case_lstrcmp(art, "to") == 0) {
		*to = field;
		str_trunc(*to, 48);
	    }
	    break;
	}

	context->line += lines;
	art = end;
    }

    art++;

    if (!*subject)
	RETURN_ERROR("Error! Article/mail has no 'Subject:' header.");
    if (!has_from)
	RETURN_ERROR("Error! Article/mail has no 'From:' header.");
    if (!*newsgroups && !*to && !*cc)
	RETURN_ERROR("Error! Article/mail has no "
		     "'Newsgroups:', 'To:' or 'Cc:' header.");

    context->line += 2;
    *line_len = max_len(art, '\n');

    if (*line_len > 1024)
	RETURN_ERROR("Error! The article/mail contains lines\n"
		     "longer than 1024 characters.\n"
		     "Posting will not be allowed.");

    return True;
}

static void format_ok_message(char *message, int len,
			      char *subject, char *newsgroups,
			      char *to, char *cc, int line_len)
{
#define SLACK   512
    int	pos = 0;

    if (!to)
	to = cc;

    len -= 8;
    sprintf(message, "Ready to %s.\n\nSubject: %s\n\n",
	    newsgroups ? (to ? "post and mail" : "post") : "mail", subject);
    pos = strlen(message);

    if (to) {
	sprintf(message + pos, "Mailing to: %s\n\n", to);
	pos += strlen(message + pos);
    }

    if (newsgroups) {
	char	*p = NULL;
	int	maxlen;

	maxlen = max_len(newsgroups, ',');

	strcpy(message + pos, "Posting to: ");
	pos += strlen(message + pos);
	if (global.mode == NewsModeDisconnected) {
	    strcpy(message + pos, "[can't check them: not connected]");
	    pos += strlen(message + pos);
	}
	message[pos++] = '\n';
	message[pos++] = '\n';

	do {
	    GROUP	*group;
	    int         glen;
	    char	*status;

	    if (maxlen + pos + SLACK > len)
		break;

	    p = strchr(newsgroups, ',');
	    if (!p)
		glen = strlen(newsgroups);
	    else {
		glen = p - newsgroups;
		*p++ = '\0';
	    }

	    glen = maxlen - glen;
	    while (glen-- > 0)
		message[pos++] = ' ';

	    strcpy(message + pos, newsgroups);
	    pos += strlen(message + pos);

	    if (global.mode == NewsModeDisconnected)
		status = "\n";
	    else if (!(group = find_group(newsgroups)))
		status = "  --  [couldn't find it]\n";
	    else if (group->moderated)
		status = "  --  [moderated]       \n";
	    else
		status = "  --  [ok]              \n";
	    strcpy(message + pos, status);
	    pos += strlen(message + pos);

	    if (p)
		newsgroups = p;
	} while (p);

	if (p) {
	    strcpy(message + pos,
		   "\n"
		   "And a bunch of groups that won't fit.\n"
		   "[Spam, maybe? Naah, couldn't be.]");
	    pos += strlen(message + pos);
	}
    }

    message[pos] = '\0';

    if (line_len > 75)
	sprintf(message + pos,
		"\n\n"
		"Note: The %s contains lines longer than 75 characters.\n"
		"You should consider editing it.",
		newsgroups ? "article" : "mail");
}

int check_article_to_post(PostContext *context, char *message, int len)
{
    char	*art, *newsgroups, *subject, *to, *cc;
    int		tmp, line_len;

    context->flags &= ~OK_TO_POST;
    message[0] = '\0';

    tmp = open(context->file_name, O_RDONLY);
    if (tmp < 0) {
	perror("open");
	strcpy(message, "Error: Failed to open temp file!");
	return False;
    }

    art = snarf_file(tmp, NULL);
    close(tmp);
    if (!art) {
	strcpy(message, "Error: Failed to read temp file!");
	return False;
    }

    if (!check_article(context, art, message,
		       &newsgroups, &subject, &to, &cc, &line_len)) {
	free(art);
	return False;
    }

    format_ok_message(message, len, subject, newsgroups, to, cc, line_len);

    if (to || cc)
	context->flags |= MAIL;
    else
	context->flags &= ~MAIL;
    if (newsgroups)
	context->flags |= POST;
    else
	context->flags &= ~POST;
    context->flags |= OK_TO_POST;
    free(art);

    return True;
}
