/* 
   PXKCell.m

   NSCell for GNUstep GUI X/DPS Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   A completely rewritten version of the original source by Pascal Forget.
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>
#include <Foundation/NSException.h>
#include <AppKit/NSGraphics.h>
#include <AppKit/NSControl.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSImage.h>
#include <gnustep/xdps/PXKCell.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKWindow.h>

/* TODO: Needs a lot of work to handle all the cell attributes */

@implementation PXKCell

/* Display the image centered in frame. */
- (void)_displayImage:(NSImage*)image inFrame:(NSRect)frame
{
  NSSize size;
  BOOL isBordered = [self isBordered];
  float _xDist = isBordered ? xDist : 0;
  float _yDist = isBordered ? yDist : 0;

  if (!image)
    return;

  PSgsave();
  PSrectclip (frame.origin.x + _xDist, frame.origin.y + _yDist,
	      frame.size.width - 2 * _xDist, frame.size.height - 2 * _yDist);
  size = [image size];
  frame.origin.x += (frame.size.width - size.width) / 2;
  frame.origin.y += (frame.size.height - size.height) / 2;
  [image compositeToPoint:frame.origin operation:NSCompositeCopy];
  PSgrestore();
}

/* Displays a text centered vertically. The horizontal alignment is determined
   by the cell alignment. */
- (void)_displayTitle:(NSString*)title
  inFrame:(NSRect)frame
  titleGray:(float)titleGray
{
  NSFont* titleFont;
  NSPoint point;
  float titleHeight, titleWidth;

  if (!title)
    return;

  PSgsave();
  PSsetgray (titleGray);
  titleFont = [self font];
  if (!titleFont)
    [NSException raise:NSInvalidArgumentException
		format:@"Request to draw a text cell but no font specified!"];

  [titleFont set];
  titleHeight = [titleFont pointSize];
  titleWidth = [titleFont widthOfString:title];

  point.y = frame.origin.y
	    + (frame.size.height - titleHeight - [titleFont descender]) / 2;

  PSrectclip (frame.origin.x + xDist, frame.origin.y + yDist,
	      frame.size.width - 2 * xDist, frame.size.height - 2 * yDist);

  /* Determine the x position of text */
  switch ([self alignment]) {
    case NSLeftTextAlignment:
	/* ignore the justified and natural alignments */
    case NSJustifiedTextAlignment:
    case NSNaturalTextAlignment:
      point.x = frame.origin.x + xDist;
      break;
    case NSRightTextAlignment:
      if (titleWidth < frame.size.width)
	point.x = frame.origin.x + frame.size.width - (xDist + titleWidth);
      else
	point.x = frame.origin.x + xDist;
      break;
    case NSCenterTextAlignment:
      if (titleWidth < frame.size.width)
	point.x = frame.origin.x + (frame.size.width - titleWidth) / 2;
      else
	point.x = frame.origin.x + xDist;
      break;
  }

  PSmoveto (point.x, point.y);
  PSshow ([title cString]);
  PSgrestore();
}

- (void)drawWithFrame:(NSRect)cellFrame
	       inView:(NSView*)controlView
{
  NSDebugLog (@"NSCell drawWithFrame:inView:");

  [super drawWithFrame: cellFrame inView: controlView];

  PSgsave();

  /* Clear the cell frame */
  if ([self isOpaque])
    NSEraseRect (cellFrame);

  /* Now draw the border if needed. */
  if ([self isBordered]) {
    if ([self isBezeled]) {
      NSDrawWhiteBezel (cellFrame, cellFrame);
      cellFrame.origin.x += 2;
      cellFrame.origin.y += 2;
      cellFrame.size.width -= 4;
      cellFrame.size.height -= 4;
    }
    else {
      NSFrameRect (cellFrame);
      cellFrame.origin.x += 1;
      cellFrame.origin.y += 1;
      cellFrame.size.width -= 2;
      cellFrame.size.height -= 2;
    }
  }

  PSrectclip (cellFrame.origin.x, cellFrame.origin.y,
	      cellFrame.size.width, cellFrame.size.height);
  PStranslate (cellFrame.origin.x, cellFrame.origin.y);
  cellFrame.origin = NSZeroPoint;
  [self drawInteriorWithFrame:cellFrame inView:controlView];
  PSgrestore();
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame inView:(NSView*)controlView
{
  switch ([self type]) {
    case NSTextCellType: {
      float titleGray = [self isEnabled] ? NSBlack : NSDarkGray;

      [self _displayTitle:[self stringValue]
	    inFrame:cellFrame
	    titleGray:titleGray];
      break;
    }
    case NSImageCellType:
      [self _displayImage:[self image] inFrame:cellFrame];
      break;
    case NSNullCellType:
      break;
  }
}

@end

//
// Methods the backend should implement
//
@implementation PXKCell (GNUstepBackend)

// Returns the size of a border
+ (NSSize)sizeForBorderType:(NSBorderType)aType
{
  switch (aType)
    {
    case NSLineBorder:
      return NSMakeSize(1, 1);
    case NSGrooveBorder:
    case NSBezelBorder:
      return NSMakeSize(2, 2);
    case NSNoBorder:
    default:
      return NSZeroSize;
    }
}

@end
