/*
  Copyright (C) 1997  Dimitrios P. Bouras

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   For author contact information, look in the README file.
*/

#include <stdio.h>
#include <stdlib.h>
#include <varargs.h>
#include <pwd.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <sys/stat.h>
#include <errno.h>
#include "common.h"
#include "version.h"
#include "pcode.h"

#define MAXNUM_VERSIONS 10
#define LASTBIN_VERSION 8

typedef struct {
	char descr[MAXLEN_DESCR];
	char account[8+1];
	char passwd[8+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
} xisprc_t12;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[8+1];
	char passwd[8+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
} xisprc_t13;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[8+1];
	char passwd[8+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemInit[MAXLEN_MODEM+1];
	unsigned char modemOpts;
} xisprc_t14;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[8+1];
	char passwd[8+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemInit[MAXLEN_MODEM+1];
	unsigned char operOpts;
	unsigned char compLevel;
} xisprc_t16;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[MAXLEN_ACCOUNT+1];
	char passwd[MAXLEN_PASSWD+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemInit[MAXLEN_MODEM+1];
	unsigned char operOpts;
	unsigned char compLevel;
	char asyncmap[MAXDIG_ASYNCMAP+1];
	char escape[MAXLEN_ESCAPE+1];
	unsigned char localIP[4];
	unsigned char remoteIP[4];
	unsigned char netmask[4];
} xisprc_t17;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[MAXLEN_ACCOUNT+1];
	char passwd[MAXLEN_PASSWD+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemInit[MAXLEN_MODEM+1];
	unsigned char operOpts;
	unsigned char compLevel;
	char asyncmap[MAXDIG_ASYNCMAP+1];
	char escape[MAXLEN_ESCAPE+1];
	unsigned char localIP[4];
	unsigned char remoteIP[4];
	unsigned char netmask[4];
	unsigned int mtu;
	unsigned int mru;
} xisprc_t18;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	unsigned char autoRedial;
	char account[MAXLEN_ACCOUNT+1];
	char passwd[MAXLEN_PASSWD+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][16+1];
	unsigned char numSlines;
	char sline[2*MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemInit[MAXLEN_MODEM+1];
	unsigned char operOpts;
	unsigned char compLevel;
	char asyncmap[MAXDIG_ASYNCMAP+1];
	char escape[MAXLEN_ESCAPE+1];
	unsigned char localIP[4];
	unsigned char remoteIP[4];
	unsigned char netmask[4];
	unsigned int mtu;
	unsigned int mru;
} xisprc_t19;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	char account[MAXLEN_ACCOUNT+1];
	char passwd[MAXLEN_PASSWD+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][MAXLEN_PHONE+1];
	unsigned char numSlines;
	char sline[2*MAXNUM_SLINES][MAXLEN_SLINE+1];
	unsigned char numCBSlns;
	char CBsln[2*MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemInit[MAXLEN_MODEM+1];
	unsigned short operOpts;
	unsigned char compLevel;
	char asyncmap[MAXDIG_ASYNCMAP+1];
	char escape[MAXLEN_ESCAPE+1];
	unsigned char localIP[4];
	unsigned char remoteIP[4];
	unsigned char netmask[4];
	unsigned int mtu;
	unsigned int mru;
} xisprc_t20;

typedef struct {
	char descr[MAXLEN_DESCR+1];
	char account[MAXLEN_ACCOUNT+1];
	char passwd[MAXLEN_PASSWD+1];
	char name[MAXLEN_UNR+1];
	char rname[MAXLEN_UNR+1];
	unsigned char maxAttempts;
	unsigned char sleepDelay;
	unsigned char connectWait;
	unsigned char numPhones;
	char phone[MAXNUM_TELS][MAXLEN_PHONE+1];
	unsigned char numSlines;
	char sline[2*MAXNUM_SLINES][MAXLEN_SLINE+1];
	unsigned char CBDelay;
	unsigned char numCBSlns;
	char CBsln[2*MAXNUM_SLINES][MAXLEN_SLINE+1];
	char modemDevice[MAXLEN_DEVICE+1];
	unsigned char modemSpeed;
	char modemReset[MAXLEN_MODEM+1];
	char modemInit[MAXLEN_MODEM+1];
	unsigned short operOpts;
	unsigned char compLevel;
	char asyncmap[MAXDIG_ASYNCMAP+1];
	char escape[MAXLEN_ESCAPE+1];
	unsigned char localIP[4];
	unsigned char remoteIP[4];
	unsigned char netmask[4];
	unsigned char dns1[4];
	unsigned char dns2[4];
	unsigned int mtu;
	unsigned int mru;
} xisprc_t21;

typedef union {
	xisprc_t12 opts12[MAXNUM_ISP];
	xisprc_t13 opts13[MAXNUM_ISP];
	xisprc_t14 opts14[MAXNUM_ISP];
	xisprc_t16 opts16[MAXNUM_ISP];
	xisprc_t17 opts17[MAXNUM_ISP];
	xisprc_t18 opts18[MAXNUM_ISP];
	xisprc_t19 opts19[MAXNUM_ISP];
	xisprc_t20 opts20[MAXNUM_ISP];
	xisprc_t21 opts21[MAXNUM_ISP];
} xisprc_ut;

xisprc_ut orc, nrc;

int rcsize[MAXNUM_VERSIONS] = {sizeof(xisprc_t12), sizeof(xisprc_t13),
							   sizeof(xisprc_t14), sizeof(xisprc_t14),
							   sizeof(xisprc_t16), sizeof(xisprc_t17),
							   sizeof(xisprc_t18), sizeof(xisprc_t19),
							   sizeof(xisprc_t20), sizeof(xisprc_t21)};

char *rcname[MAXNUM_VERSIONS] = {"1.2","1.3","1.4","1.5",
								 "1.6","1.7","1.8","1.9",
								 "2.0",Version};
#define MAXLEN_VERSION 4

char *orcfname, *nrcfname;

#define MSGLEN_ERR 128

void doErr(char *msg)
{
	char emsg[MSGLEN_ERR+1];

	if (errno < sys_nerr)
		sprintf(emsg, "xisprccv: %s: %s\n", msg, sys_errlist[errno]);
	else
		sprintf(emsg, "xisprccv: %s: error #%d\n", msg, errno);
	fputs(emsg, stderr);
	exit(1);
}

void initXisprc13(int rec)
{
	int i;
	xisprc_t12 *p12 = &(orc.opts12[rec]);
	xisprc_t13 *p13 = &(nrc.opts13[rec]);

	strcpy(p13->descr, p12->descr);
	p13->autoRedial = 0;
	strcpy(p13->account, p12->account);
	strcpy(p13->passwd, p12->passwd);
	p13->maxAttempts = p12->maxAttempts;
	p13->sleepDelay = p12->sleepDelay;
	p13->numPhones = p12->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p13->phone[i], p12->phone[i]);
	p13->numSlines = p12->numSlines;
	for (i=0; i<MAXNUM_SLINES; i++)
		strcpy(p13->sline[i], p12->sline[i]);
}

void initXisprc14(int rec)
{
	int i;
	xisprc_t13 *p13 = &(orc.opts13[rec]);
	xisprc_t14 *p14 = &(nrc.opts14[rec]);

	strcpy(p14->descr, p13->descr);
	p14->autoRedial = p13->autoRedial;
	strcpy(p14->account, p13->account);
	strcpy(p14->passwd, p13->passwd);
	p14->maxAttempts = p13->maxAttempts;
	p14->sleepDelay = p13->sleepDelay;
	p14->numPhones = p13->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p14->phone[i], p13->phone[i]);
	p14->numSlines = p13->numSlines;
	for (i=0; i<MAXNUM_SLINES; i++)
		strcpy(p14->sline[i], p13->sline[i]);
	strcpy(p14->modemInit, MODEM_INIT);
	p14->modemOpts = MODEM_TONEDIAL;
}

void initXisprc15(int rec)
{
	nrc.opts14[rec] = orc.opts14[rec];
}

void initXisprc16(int rec)
{
	int i;
	xisprc_t14 *p14 = &(orc.opts14[rec]);
	xisprc_t16 *p16 = &(nrc.opts16[rec]);

	strcpy(p16->descr, p14->descr);
	p16->autoRedial = p14->autoRedial;
	strcpy(p16->account, p14->account);
	strcpy(p16->passwd, p14->passwd);
	p16->maxAttempts = p14->maxAttempts;
	p16->sleepDelay = p14->sleepDelay;
	p16->numPhones = p14->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p16->phone[i], p14->phone[i]);
	p16->numSlines = p14->numSlines;
	for (i=0; i<MAXNUM_SLINES; i++)
		strcpy(p16->sline[i], p14->sline[i]);
	strcpy(p16->modemDevice, MODEM_DEVICE);
	p16->modemSpeed = MODEM_SPEED;
	strcpy(p16->modemInit, p14->modemInit);
	p16->operOpts = p14->modemOpts | PPPD_COMPRESS;
	p16->compLevel = COMP_LEVEL;
}

void initXisprc17(int rec)
{
	int i;
	xisprc_t16 *p16 = &(orc.opts16[rec]);
	xisprc_t17 *p17 = &(nrc.opts17[rec]);
	unsigned char ep[MAXLEN_PASSWD+1] = {0},
				  pp[MAXLEN_PASSWD+1] = {0};

	strcpy(p17->descr, p16->descr);
	p17->autoRedial = p16->autoRedial;
	strcpy(p17->account, p16->account);

	if (p16->passwd[0]) {
		pencode(ep, pp);
		memcpy(ep, p16->passwd, 8);
	}
	memcpy(p17->passwd, ep, MAXLEN_PASSWD);

	p17->maxAttempts = p16->maxAttempts;
	p17->sleepDelay = p16->sleepDelay;
	p17->numPhones = p16->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p17->phone[i], p16->phone[i]);
	p17->numSlines = p16->numSlines;
	for (i=0; i<MAXNUM_SLINES; i++)
		strcpy(p17->sline[i], p16->sline[i]);
	strcpy(p17->modemDevice, p16->modemDevice);
	p17->modemSpeed = p16->modemSpeed;
	strcpy(p17->modemInit, p16->modemInit);
	p17->operOpts = p16->operOpts |
				   (HW_FLOWCTRL|ACCEPT_LOCALIP|ACCEPT_REMOTEIP|DEFAULT_ROUTE);
	p17->compLevel = p16->compLevel;
	strcpy(p17->asyncmap, PPPD_SASYNCMAP);
	strcpy(p17->escape, PPPD_ESCAPE);
	memcpy(p17->localIP, LOCAL_IP, 4);
	memcpy(p17->remoteIP, REMOTE_IP, 4);
	memcpy(p17->netmask, NETMASK, 4);
}

void initXisprc18(int rec)
{
	int i;
	xisprc_t17 *p17 = &(orc.opts17[rec]);
	xisprc_t18 *p18 = &(nrc.opts18[rec]);

	strcpy(p18->descr, p17->descr);
	p18->autoRedial = p17->autoRedial;
	strcpy(p18->account, p17->account);
	memcpy(p18->passwd, p17->passwd, MAXLEN_PASSWD);
	p18->maxAttempts = p17->maxAttempts;
	p18->sleepDelay = p17->sleepDelay;
	p18->numPhones = p17->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p18->phone[i], p17->phone[i]);
	p18->numSlines = p17->numSlines;
	for (i=0; i<MAXNUM_SLINES; i++)
		strcpy(p18->sline[i], p17->sline[i]);
	strcpy(p18->modemDevice, p17->modemDevice);
	p18->modemSpeed = p17->modemSpeed;
	strcpy(p18->modemInit, p17->modemInit);
	p18->operOpts = p17->operOpts;
	p18->compLevel = p17->compLevel;
	strcpy(p18->asyncmap, PPPD_SASYNCMAP);
	strcpy(p18->escape, PPPD_ESCAPE);
	memcpy(p18->localIP, LOCAL_IP, 4);
	memcpy(p18->remoteIP, REMOTE_IP, 4);
	memcpy(p18->netmask, NETMASK, 4);
	p18->mtu = MTU;
	p18->mru = MRU;
}

void initXisprc19(int rec)
{
	int i;
	xisprc_t18 *p18 = &(orc.opts18[rec]);
	xisprc_t19 *p19 = &(nrc.opts19[rec]);
	char *p1, *p2, *pq;

	strcpy(p19->descr, p18->descr);
	p19->autoRedial = p18->autoRedial;
	strcpy(p19->account, p18->account);
	memcpy(p19->passwd, p18->passwd, MAXLEN_PASSWD);
	p19->maxAttempts = p18->maxAttempts;
	p19->sleepDelay = p18->sleepDelay;
	p19->numPhones = p18->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p19->phone[i], p18->phone[i]);
	p19->numSlines = p18->numSlines;
	for (i=0; i<p18->numSlines; i++) {
		p1 = p18->sline[i];
		p2 = strchr(p1, ' ');
		if (p2 < (pq = strchr(p1+1, '\'')))
			p2 = strchr(pq, ' ');
		*p2 = 0; ++p2;
		if (*p1 == '\'')
			++p1;
		if (*(p1+strlen(p1)-1) == '\'')
			*(p1+strlen(p1)-1) = 0;
		if (*p2 == '\'')
			++p2;
		if (*(p2+strlen(p2)-1) == '\'')
			*(p2+strlen(p2)-1) = 0;
		strcpy(p19->sline[i], p1);
		strcpy(p19->sline[MAXNUM_SLINES+i], p2);
	}
	strcpy(p19->modemDevice, p18->modemDevice);
	p19->modemSpeed = p18->modemSpeed;
	strcpy(p19->modemInit, p18->modemInit);
	p19->operOpts = p18->operOpts;
	p19->compLevel = p18->compLevel;
	strcpy(p19->asyncmap, p18->asyncmap);
	strcpy(p19->escape, p18->escape);
	memcpy(p19->localIP, p18->localIP, 4);
	memcpy(p19->remoteIP, p18->remoteIP, 4);
	memcpy(p19->netmask, p18->netmask, 4);
	p19->mtu = p18->mtu;
	p19->mru = p18->mru;
}

void initXisprc20(int rec)
{
	int i;
	xisprc_t19 *p19 = &(orc.opts19[rec]);
	xisprc_t20 *p20 = &(nrc.opts20[rec]);

	strcpy(p20->descr, p19->descr);
	strcpy(p20->account, p19->account);
	memcpy(p20->passwd, p19->passwd, MAXLEN_PASSWD);
	p20->maxAttempts = p19->maxAttempts;
	p20->sleepDelay = p19->sleepDelay;
	p20->numPhones = p19->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p20->phone[i], p19->phone[i]);
	p20->numSlines = p19->numSlines;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		strcpy(p20->sline[i], p19->sline[i]);
	strcpy(p20->modemDevice, p19->modemDevice);
	p20->modemSpeed = p19->modemSpeed;
	strcpy(p20->modemInit, p19->modemInit);
	p20->operOpts = p19->operOpts;
	if (p19->autoRedial)
		p20->operOpts |= AUTO_REDIAL;
	p20->compLevel = p19->compLevel;
	if ( (p20->operOpts & HW_FLOWCTRL) &&
		 ! strcmp(p19->asyncmap, PPPD_SASYNCMAP))
		strcpy(p20->asyncmap, PPPD_HASYNCMAP);
	else
		strcpy(p20->asyncmap, p19->asyncmap);
	strcpy(p20->escape, p19->escape);
	memcpy(p20->localIP, p19->localIP, 4);
	memcpy(p20->remoteIP, p19->remoteIP, 4);
	memcpy(p20->netmask, p19->netmask, 4);
	p20->mtu = p19->mtu;
	p20->mru = p19->mru;
}

void initXisprc21(int rec)
{
	int i;
	xisprc_t20 *p20 = &(orc.opts20[rec]);
	xisprc_t21 *p21 = &(nrc.opts21[rec]);

	strcpy(p21->descr, p20->descr);
	strcpy(p21->account, p20->account);
	memcpy(p21->passwd, p20->passwd, MAXLEN_PASSWD);
	p21->name[0] = 0;
	p21->rname[0] = 0;
	p21->maxAttempts = p20->maxAttempts;
	p21->sleepDelay = p20->sleepDelay;
	p21->connectWait = MAXSEC_CNWAIT;
	p21->numPhones = p20->numPhones;
	for (i=0; i<MAXNUM_TELS; i++)
		strcpy(p21->phone[i], p20->phone[i]);
	p21->numSlines = p20->numSlines;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		strcpy(p21->sline[i], p20->sline[i]);
	p21->CBDelay = MAXSEC_CBDELAY;
	p21->numCBSlns = p20->numCBSlns;
	for (i=0; i<(2*MAXNUM_SLINES); i++)
		strcpy(p21->CBsln[i], p20->CBsln[i]);
	strcpy(p21->modemDevice, p20->modemDevice);
	p21->modemSpeed = p20->modemSpeed;
	strcpy(p21->modemReset, MODEM_RESET);
	strcpy(p21->modemInit, p20->modemInit);
	p21->operOpts = p20->operOpts;
	p21->compLevel = p20->compLevel;
	strcpy(p21->asyncmap, p20->asyncmap);
	strcpy(p21->escape, p20->escape);
	memcpy(p21->localIP, p20->localIP, 4);
	memcpy(p21->remoteIP, p20->remoteIP, 4);
	memcpy(p21->netmask, p20->netmask, 4);
	memcpy(p21->dns1, DNS, 4);
	memcpy(p21->dns2, DNS, 4);
	p21->mtu = p20->mtu;
	p21->mru = p20->mru;
}

void (* initf[MAXNUM_VERSIONS])(int) = {initXisprc13, initXisprc14,
										initXisprc15, initXisprc16,
										initXisprc17, initXisprc18,
										initXisprc19, initXisprc20,
										initXisprc21, NULL};

char *passwd2ascii(unsigned char *pwd)
{
	static unsigned char apwd[2*MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i;

	if (*pwd) {
		for (i=0, p=apwd; i<MAXLEN_PASSWD; i++, pwd++) {
			msh = (*pwd & 0xF0) >> 4;
			lsh = *pwd & 0x0F;
			*p++ = lsh + 'A';
			*p++ = msh + 'A';
		}
		*p = 0;
	}
	else *apwd = 0;
	return apwd;
}

char *IP2str(unsigned char *ip)
{
	static char IPstr[16];

	sprintf(IPstr, "%u.%u.%u.%u",
			ip[0],ip[1],ip[2],ip[3]);
	return IPstr;
}

char *speed2str(unsigned char speed)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (speed & snum[i])
			return sstr[i];
	return NULL;
}

char *ascii2passwd(char *buf)
{
	static unsigned char pwd[MAXLEN_PASSWD+1];
	unsigned char msh, lsh, *p;
	int i = 0;

	if (*buf) {
		for (p=pwd; i<MAXLEN_PASSWD; i++) {
			lsh = (*buf++ - 'A') & 0x0F;
			msh = (*buf++ - 'A') & 0x0F;
			pwd[i] = (msh << 4) | lsh;
		}
	}
	pwd[i] = 0;
	return pwd;
}

unsigned int str2speed(char *buf)
{
	static char sstr[8][8] = {"1200","2400","4800","9600",
							  "19200","38400","57600","115200"};
	static int snum[8] = {0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80};
	int i;

	for (i=0; i<8; i++)
		if (strcmp(buf, sstr[i]) == 0)
			return snum[i];
	return 0;
}

unsigned char *str2IP(char *buf)
{
	static unsigned char ip[4];
	unsigned int iip[4];
	int i;

	i = sscanf(buf, "%3u.%3u.%3u.%3u", &iip[0],
			   &iip[1], &iip[2], &iip[3]);
	if (i < 4)
		return NULL;
	for (i=0; i<4; i++)
		ip[i] = iip[i];
	return ip;
}

void rcISP(FILE *fp, int nISP)
{
	int n;

	if (fscanf(fp, "-%3d- ", &n) < 1) {
		fprintf(stderr, "xisprccv: rcISP: error reading ISP number in %s\n",
				nrcfname);
		exit(1);
	}
	if (n != nISP) {
		fprintf(stderr, "xisprccv: rcISP: ISP sequence number wrong in %s\n",
				nrcfname);
		fprintf(stderr, "xisprccv: rcISP: expected %d, got %d\n", nISP, n);
		exit(1);
	}
}

void rcLineError(char *function, char *line)
{
	fprintf(stderr, "xisprccv: %s: error reading %s\n", function, nrcfname);
	fprintf(stderr, "xisprccv: %s: offending line: [%s]\n", function, line);
	exit(1);
}

#define MAXLEN_LINE 256
#define MAXLEN_PARAM 128

char *readParam(FILE *fp, char *name, char type,
			   int llimit, int ulimit, void *data)
{
	static char line[MAXLEN_LINE];
	char sparam[MAXLEN_PARAM] = {0}, *p, *endp;
	int iparam, hex = 0, len;

	if (fgets(line, MAXLEN_LINE, fp) == NULL) {
		fprintf(stderr, "xisprccv: readParam: %s: premature EOF\n", nrcfname);
		exit(1);
	}
	line[strlen(line)-1] = 0;
	if ((p=strchr(line,':')) == NULL)
		rcLineError("readParam", line);
	if (strncmp(name, line, (int)(p-line))) {
		fprintf(stderr, "xisprccv: readParam: expected [%s], got [%s]\n",
				name, line);
		exit(1);
	}
	for (++p; *p==' '; p++);
	strncpy(sparam, p, sizeof(sparam)-1);
	switch (type) {
		case 'S':
			len = strlen(sparam);
			if (ulimit<len || len<llimit)
				rcLineError("readParam", line);
			strcpy((char *)data, sparam);
			break;

		case 'X':
			hex = 1;
		case 'I':
			iparam = strtol(sparam, &endp, (hex)? 16:10);
			if ((!hex && (ulimit<iparam || iparam<llimit)) ||
				 endp == sparam || *endp )
				rcLineError("readParam", line);
			*((int *)data) = iparam;
			break;

		default: break;
	}
	return(line);
}

void inputAllXisprc20(xisprc_ut *up, FILE *rcfp)
{
	int i, j, n;
	char buf[2*MAXLEN_PASSWD+1], pname[32] = {0}, *line;
	unsigned char *ipp;
	xisprc_t20 *p = &(up->opts20[0]);

	fgets(buf, 2*MAXLEN_PASSWD-1, rcfp);
	for (i=0; i<MAXNUM_ISP; i++, p++) {
		rcISP(rcfp, i);
		readParam(rcfp, "DESCR", 'S', 0, MAXLEN_DESCR, p->descr);
		readParam(rcfp, "ACCOUNT", 'S', 0, MAXLEN_ACCOUNT, p->account);
		if (! *(p->account))
			n = 0;
		else
			n = 2*MAXLEN_PASSWD;
		readParam(rcfp, "PASSWD", 'S', n, n, buf);
		strcpy(p->passwd, ascii2passwd(buf));
		readParam(rcfp, "MAXATTEMPTS", 'I', 0, 255, &(p->maxAttempts));
		readParam(rcfp, "SLEEPDELAY", 'I', 0, 255, &(p->sleepDelay));
		readParam(rcfp, "NUMPHONES", 'I', 0, MAXNUM_TELS, &(p->numPhones));
		for (j=0; j<p->numPhones; j++) {
			sprintf(pname, " PHONE%d", j);
			readParam(rcfp, pname, 'S', 1, MAXLEN_PHONE, p->phone[j]);
		}
		readParam(rcfp, "NUMSLINES", 'I', 0, MAXNUM_SLINES, &(p->numSlines));
		for (j=0; j<p->numSlines; j++) {
			sprintf(pname, " SLINE%dE", j);
			readParam(rcfp, pname, 'S', 1, MAXLEN_SLINE, p->sline[j]);
			sprintf(pname, " SLINE%dS", j);
			readParam(rcfp, pname, 'S', 1, MAXLEN_SLINE,
					  p->sline[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "NUMCBSLINES", 'I', 0, MAXNUM_SLINES, &(p->numCBSlns));
		for (j=0; j<p->numCBSlns; j++) {
			sprintf(pname, " CBSLINE%dE", j);
			readParam(rcfp, pname, 'S', 1, MAXLEN_SLINE, p->CBsln[j]);
			sprintf(pname, " CBSLINE%dS", j);
			readParam(rcfp, pname, 'S', 1, MAXLEN_SLINE,
					  p->CBsln[MAXNUM_SLINES+j]);
		}
		readParam(rcfp, "MODEMDEVICE", 'S', 1, MAXLEN_DEVICE, p->modemDevice);
		line = readParam(rcfp, "MODEMSPEED", 'S', 4, 6, buf);
		if (! (n=str2speed(buf)))
			rcLineError("inputAllXisprc", line);
		p->modemSpeed = n;
		readParam(rcfp, "MODEMINIT", 'S', 0, MAXLEN_MODEM, p->modemInit);
		readParam(rcfp, "OPEROPTS", 'X', 0, 0, &(p->operOpts));
		readParam(rcfp, "COMPLEVEL", 'I', 9, 15, &(p->compLevel));
		readParam(rcfp, "ASYNCMAP", 'S', 2, MAXDIG_ASYNCMAP, p->asyncmap);
		readParam(rcfp, "ESCAPE", 'S', 0, MAXLEN_ESCAPE, p->escape);
		line = readParam(rcfp, "LOCALIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->localIP, ipp, 4);
		line = readParam(rcfp, "REMOTEIP", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->remoteIP, ipp, 4);
		line = readParam(rcfp, "NETMASK", 'S', 7, MAXLEN_IP, buf);
		if ((ipp=str2IP(buf)) == NULL)
			rcLineError("inputAllXisprc", line);
		memcpy(p->netmask, ipp, 4);
		readParam(rcfp, "MTU", 'I', 48, 1500, &(p->mtu));
		readParam(rcfp, "MRU", 'I', 48, 1500, &(p->mru));
	}
}

void (* inputAllXisprc[MAXNUM_VERSIONS])(xisprc_ut*, FILE*) =
	{NULL, NULL, NULL, NULL,
	 NULL, NULL, NULL, NULL,
	 inputAllXisprc20};

int rcVersion()
{
	struct stat st;
	FILE *rcfp;
	char buf[64];
	int ov;

	stat(nrcfname, &st);
	for (ov=0; ov<LASTBIN_VERSION; ov++)
		if (st.st_size == MAXNUM_ISP*rcsize[ov]) break;

	if (ov == LASTBIN_VERSION) {
		rcfp = fopen(nrcfname, "r");
		if (rcfp == NULL) {
			sprintf(buf, "rcVersion: fopen(%s)", nrcfname);
			doErr(buf);
		}
		if (fscanf(rcfp, "xISP-%7[.0-9] ", buf) != 1) {
			fputs("xisprccv: rcVersion: error reading file version!\n",
				  stderr);
			exit(1);
		}
		for (ov=LASTBIN_VERSION; ov<MAXNUM_VERSIONS; ov++)
			if (!strcmp(buf, rcname[ov])) break;
		if (ov == MAXNUM_VERSIONS) {
			fprintf(stderr, "xisprccv: rcVersion: %s: unknown version!\n",
					nrcfname);
			fprintf(stderr, "xisprccv: rcVersion: %s: not converted.\n",
					nrcfname);
			return 0;
		}
		fclose(rcfp);
	}
	return ov;
}

void readAllXisprc(int ver)
{
	FILE *rcfp;
	int rw;
	char msg[64];

	rcfp = fopen(nrcfname, "r");
	if (rcfp == NULL) {
		sprintf(msg, "readAll: fopen(%s)", nrcfname);
		doErr(msg);
	}
	if (ver < LASTBIN_VERSION) {
		rw = fread(&orc, rcsize[ver], MAXNUM_ISP, rcfp);
		if (rw != MAXNUM_ISP) {
			sprintf(msg, "readAll: fread(%s)", nrcfname);
			doErr(msg);
		}
	}
	else
		(*inputAllXisprc[ver])(&orc, rcfp);
	fclose(rcfp);
}

int rcPrint(va_alist) va_dcl
{
	int bw;
	va_list ap;
	char *fmt;
	FILE *fp;

	va_start(ap);
	fp = va_arg(ap, FILE*);
	fmt = va_arg(ap, char*);
	bw = vfprintf(fp, fmt, ap);
	va_end(ap);
	if (bw <= 0)
		doErr("outputAllXisprc");
	return bw;
}

void outputAllXisprc(xisprc_ut *up, FILE *rcfp)
{
	int i, n;
	xisprc_t21 *p = &(up->opts21[0]);

	rcPrint(rcfp, "xISP-%s\n", Version);
	for (i=0; i<MAXNUM_ISP; i++, p++) {
		rcPrint(rcfp, "-%d-\n", i);
		rcPrint(rcfp, "DESCR: %s\n", p->descr);
		rcPrint(rcfp, "ACCOUNT: %s\n", p->account);
		if (*(p->account))
			rcPrint(rcfp, "PASSWD: %s\n", passwd2ascii(p->passwd));
		else
			rcPrint(rcfp, "PASSWD: \n");
		rcPrint(rcfp, "NAME: %s\n", p->name);
		rcPrint(rcfp, "REMOTENAME: %s\n", p->rname);
		rcPrint(rcfp, "MAXATTEMPTS: %d\n", p->maxAttempts);
		rcPrint(rcfp, "SLEEPDELAY: %d\n", p->sleepDelay);
		rcPrint(rcfp, "CONNECTWAIT: %d\n", p->connectWait);
		rcPrint(rcfp, "NUMPHONES: %d\n", p->numPhones);
		for (n=0; n<p->numPhones; n++)
			rcPrint(rcfp, " PHONE%d: %s\n", n, p->phone[n]);
		rcPrint(rcfp, "NUMSLINES: %d\n", p->numSlines);
		for (n=0; n<p->numSlines; n++) {
			rcPrint(rcfp, " SLINE%dE: %s\n", n, p->sline[n]);
			rcPrint(rcfp, " SLINE%dS: %s\n", n, p->sline[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "CBDELAY: %d\n", p->CBDelay);
		rcPrint(rcfp, "NUMCBSLINES: %d\n", p->numCBSlns);
		for (n=0; n<p->numCBSlns; n++) {
			rcPrint(rcfp, " CBSLINE%dE: %s\n", n, p->CBsln[n]);
			rcPrint(rcfp, " CBSLINE%dS: %s\n", n, p->CBsln[MAXNUM_SLINES+n]);
		}
		rcPrint(rcfp, "MODEMDEVICE: %s\n", p->modemDevice);
		rcPrint(rcfp, "MODEMSPEED: %s\n", speed2str(p->modemSpeed));
		rcPrint(rcfp, "MODEMRESET: %s\n", p->modemReset);
		rcPrint(rcfp, "MODEMINIT: %s\n", p->modemInit);
		rcPrint(rcfp, "OPEROPTS: %X\n", p->operOpts);
		rcPrint(rcfp, "COMPLEVEL: %d\n", p->compLevel);
		rcPrint(rcfp, "ASYNCMAP: %s\n", p->asyncmap);
		rcPrint(rcfp, "ESCAPE: %s\n", p->escape);
		rcPrint(rcfp, "LOCALIP: %s\n", IP2str(p->localIP));
		rcPrint(rcfp, "REMOTEIP: %s\n", IP2str(p->remoteIP));
		rcPrint(rcfp, "NETMASK: %s\n", IP2str(p->netmask));
		rcPrint(rcfp, "DNS1: %s\n", IP2str(p->dns1));
		rcPrint(rcfp, "DNS2: %s\n", IP2str(p->dns2));
		rcPrint(rcfp, "MTU: %u\n", p->mtu);
		rcPrint(rcfp, "MRU: %u\n", p->mru);
	}
}

void writeAllXisprc(int ver)
{
	FILE *rcfp;
	int rw;
	char msg[64];

	rcfp = fopen(nrcfname, "w");
	if (rcfp == NULL) {
		sprintf(msg, "writeAllXisprc: fopen(%s)", nrcfname);
		doErr(msg);
	}
	if (ver < LASTBIN_VERSION) {
		rw = fwrite(&nrc, rcsize[ver], MAXNUM_ISP, rcfp);
		if (rw != MAXNUM_ISP) {
			sprintf(msg, "writeAllXisprc: fwrite(%s)", nrcfname);
			doErr(msg);
		}
	}
	else
		outputAllXisprc(&orc, rcfp);
	fclose(rcfp);
}

void outofMem(void)
{
	fprintf(stderr, "xisprccv: out of memory!\n");
	exit(1);
}

void initFnames(void )
{
	struct passwd *user;

	user = getpwuid(getuid());
	nrcfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(RCFNAME)+1);
	orcfname = (char *)malloc(strlen(user->pw_dir)+1+strlen(RCFNAME)+
							  1+MAXLEN_VERSION+1);
	if (orcfname != NULL && nrcfname != NULL) {
		strcpy(nrcfname, user->pw_dir); strcat(nrcfname, "/");
		strcat(nrcfname, RCFNAME);
		strcpy(orcfname, nrcfname);
		strcat(orcfname, "-");
	}
	else outofMem();
}

int main()
{
	int ov = 0, nv = MAXNUM_VERSIONS-1, i, j;
	char *rcsp;

	rcsp = RCSid;
	umask(077);
	pcode_init();
	initFnames();

	ov = rcVersion();
	if (ov == nv) {
		fprintf(stderr, "xisprccv: %s: compatible with v%s.\n",
				nrcfname, rcname[ov]);
		fprintf(stderr, "xisprccv: %s: not converted.\n", nrcfname);
		return 0;
	}
	strcpy(strchr(orcfname, '-')+1, rcname[ov]);

	readAllXisprc(ov);
	rename(nrcfname, orcfname);
	for (i=ov; i<nv; i++) {
		memset(&nrc, 0, MAXNUM_ISP*rcsize[i+1]);
		for (j=0; j<MAXNUM_ISP; j++)
			(*initf[i])(j);
		memcpy(&orc, &nrc, MAXNUM_ISP*rcsize[i+1]);
	}
	writeAllXisprc(nv);

	fprintf(stderr, "xisprccv: %s: converted from v%s to v%s.\n",
			nrcfname, rcname[ov], rcname[nv]);
	return 0;
}

