/*
 * petmem.c - PET memory handling.
 *
 * Written by
 *  Ettore Perazzoli (ettore@comm2000.it)
 *  Andre' Fachat (fachat@physik.tu-chemnitz.de)
 *
 * This file is part of VICE, the Versatile Commodore Emulator.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "vice.h"

#include <stdio.h>

#include "petmem.h"
#include "memutils.h"
#include "pia.h"
#include "via.h"
#include "crtc.h"
#include "kbd.h"
#include "resources.h"

/* ------------------------------------------------------------------------- */

BYTE ram[0x10000];   /* 64K just to make things easier.  Real size is 32K. */
BYTE rom[ROM_SIZE];

read_func_ptr_t _mem_read_tab[0x101];
store_func_ptr_t _mem_write_tab[0x101];

BYTE *_mem_read_base_tab[0x101];

int rom_loaded = 0;		/* FIXME! */

static BYTE crtc_ptr = 0;	/* CRTC register pointer. */

/* ------------------------------------------------------------------------- */

BYTE REGPARM1 read_zero(ADDRESS addr)
{
    return ram[addr & 0xff];
}

void REGPARM2 store_zero(ADDRESS addr, BYTE value)
{
    ram[addr & 0xff] = value;
}

BYTE REGPARM1 read_ram(ADDRESS addr)
{
    return ram[addr];
}

void REGPARM2 store_ram(ADDRESS addr, BYTE value)
{
    ram[addr] = value;
}

BYTE REGPARM1 read_rom(ADDRESS addr)
{
    return rom[addr & 0x7fff];
}

void REGPARM2 store_rom(ADDRESS addr, BYTE value)
{
    rom[addr & 0x7fff] = value;
}

void REGPARM2 store_io(ADDRESS addr, BYTE value)
{
    switch (addr & 0xf0) {
      case 0x10:		/* PIA1 */
	store_pia1(addr, value);
	break;
      case 0x20:		/* PIA2 */
	store_pia2(addr, value);
	break;
      case 0x40:
      case 0x50:
      case 0x60:
      case 0x70:
      case 0xc0:
      case 0xd0:
      case 0xe0:
      case 0xf0:
	store_via(addr, value);	/* VIA */
	break;
      case 0x80:
	if (addr & 1)
	    store_crtc(crtc_ptr, value);
	else
	    crtc_ptr = value;
	break;
    }
}

BYTE REGPARM1 read_io(ADDRESS addr)
{
    switch (addr & 0xf0) {
      case 0x10:		/* PIA1 */
	return read_pia1(addr);
      case 0x20:		/* PIA2 */
	return read_pia2(addr);
      case 0x40:
      case 0x50:
      case 0x60:
      case 0x70:
      case 0xc0:
      case 0xd0:
      case 0xe0:
      case 0xf0:
	return read_via(addr);	/* VIA */
      case 0x80:		/* CRTC */
	if (addr & 1)
	    return read_crtc(crtc_ptr);
	else
	    return 0x9f;	/* Status. */
      default:
	return addr >> 8;
    }
}

#if 0				/* (unused) */
static BYTE REGPARM1 read_dummy(ADDRESS addr)
{
    return 0xff;
}
#endif

static void REGPARM2 store_dummy(ADDRESS addr, BYTE value)
{
    return;
}

/* ------------------------------------------------------------------------- */

void initialize_memory(void)
{
    int i;

    /* Setup RAM at $0000 - $8FFF. */
    for (i = 0x00; i <= 0x8f; i++) {
	_mem_read_tab[i] = read_ram;
	_mem_write_tab[i] = store_ram;
	_mem_read_base_tab[i] = ram + (i << 8);
    }
    
    /* Setup ROM at $9000 - $E7FF. */
    for (i = 0x90; i <= 0xe7; i++) {
	_mem_read_tab[i] = read_rom;
	_mem_write_tab[i] = store_dummy;
	_mem_read_base_tab[i] = rom + ((i & 0x7f) << 8);
    }

    /* Setup I/O at $E800 - $EFFF. */
    for (i = 0xe8; i <= 0xef; i++) {
	_mem_read_tab[i] = read_io;
	_mem_write_tab[i] = store_io;
	_mem_read_base_tab[i] = NULL;
    }

    /* Setup ROM at $F000 - $FFFF. */
    for (i = 0xf0; i <= 0xff; i++) {
	_mem_read_tab[i] = read_rom;
	_mem_write_tab[i] = store_dummy;
	_mem_read_base_tab[i] = rom + ((i & 0x7f) << 8);
    }

    _mem_read_tab[0x100] = _mem_read_tab[0];
    _mem_write_tab[0x100] = _mem_write_tab[0];
    _mem_read_base_tab[0x100] = _mem_read_base_tab[0];
}

/* ------------------------------------------------------------------------- */

void mem_powerup(void)
{
    int i;

#ifndef __MSDOS__
    printf("Initializing RAM for power-up...\n");
#endif
    
    for (i = 0; i < RAM_SIZE; i += 0x80) {
	memset(ram + i, 0, 0x40);
	memset(ram + i + 0x40, 0xff, 0x40);
    }
}

/* Load memory image files.  This also selects the PET model. */
int mem_load(void)
{
    WORD sum;			/* ROM checksum */
    int screen_width;
    int i;
    
    /* Try to load a RAM image if available. */
    
    if (mem_load_sys_file(app_resources.directory, app_resources.ramName,
			  ram, RAM_SIZE) < 0) {
	mem_powerup();
    }

    /* Load Kernal ROM. */
    
    if (mem_load_sys_file(app_resources.directory, app_resources.kernalName,
			  rom, ROM_SIZE) < 0) {
	fprintf(stderr,"Couldn't load ROM.\n\n");
	return -1;
    }

    /* Checksum over top 4 kByte PET kernal. */
    for (i = 0x7000, sum = 0; i < 0x8000; i++)
	sum += rom[i]; 

    /* 4032 and 8032 have the same kernals, so we have to test more, here
       $E000 - $E800. */
    for (i = 0x6000; i < 0x6800; i++)
	sum += rom[i]; 

    printf("Loaded ROM, checksum is %d ($%04X).\n", sum, sum);

    if (sum == PET8032_CHECKSUM_A || sum == PET8032_CHECKSUM_B) {
	printf("Identified PET 8032 ROM by checksum.\n");
	screen_width = 80;
    } else if (sum == PET3032_CHECKSUM) {
	printf("Identified PET 3032 ROM by checksum.\n");
	screen_width = 40;
    } else if (sum == PET4032_CHECKSUM) { 
	printf("Identified PET 4032 ROM by checksum.\n");
	screen_width = 40;
    } else {
	screen_width = PET_COLS;
	printf("Unknown PET ROM.\n");
    }

    printf("Setting screen width to %d columns.\n", screen_width);
    crtc_set_screen_width(screen_width);
    
    /* Setup the corresponding PET keyboard. */
    if (screen_width == 80) {
	set80key();
    } else {
	set40key();
    }

    rom_loaded = 1;

    return 0;
}

/* ------------------------------------------------------------------------- */

/* FIXME: this is plain wrong. */

void mem_get_basic_text(ADDRESS *start, ADDRESS *end)
{
    if (start != NULL)
	*start = ram[0x2b] | (ram[0x2c] << 8);
    if (end != NULL)
	*end = ram[0x2d] | (ram[0x2e] << 8);
}

void mem_set_basic_text(ADDRESS start, ADDRESS end)
{
    ram[0x2b] = ram[0xac] = start & 0xff;
    ram[0x2c] = ram[0xad] = start >> 8;
    ram[0x2d] = ram[0x2f] = ram[0x31] = ram[0xae] = end & 0xff;
    ram[0x2e] = ram[0x30] = ram[0x32] = ram[0xaf] = end >> 8;
}
