// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _SgmlRenderer_h_
#define _SgmlRenderer_h_

#include "Renderer.h"
#include "SgmlParser.h"

class Style;

//: The SgmlRenderer class is the base class for any renderer that takes input from an SgmlParser.
//. This class processes start and end tag signals from the SgmlParser, and
//. maintains a tag path, tag stack, style sheet, and current style.  The tag
//. path is the concatenation of all open tags for the current document instance.
//. the tag stack keeps track of each open tags associated attributes.  The
//. style sheet is associated with the type of document being parsed.  The
//. style is the current style for the tag path, taking into account inheritance
//. of styles from parent elements in the document instance.
class SgmlRenderer : public Renderer {
    Q_OBJECT
    SgmlParser*  _parser;
public:
    //. Create a new SgmlRender.  An SgmlRenderer takes two additional parameters,
    //. tagPath and styleSheet.  The tagPath is the concatenation of all open
    //. tags at the time the SgmlRenderer was created.  The styleSheet is a
    //. pointer to the list of styles associated with the current document type.
    //. The style sheet determines how the content is rendered.
    SgmlRenderer( Canvas*     canvas,
                  SgmlParser* parser,
                  int         clipWidth=0,
                  QObject*    parent=0,
                  const char* name=0 );

    //. This is a virtual destructor that does nothing.
    virtual ~SgmlRenderer();

    SgmlParser* parser() { return _parser; }

    Style* style() { return _parser->styleStack().getLast(); }

    STag* tag() { return _parser->tagStack().getLast(); }

    const QList<STag>& tagStack() { return _parser->tagStack(); }

    const QList<Style>& styleStack() { return _parser->styleStack(); }

    virtual bool findAnchor( const QString& name, int& x, int& y );
public slots:
    //. Process a content signal from the SgmlParser.  All subclasses must
    //. reimplement this function.
    virtual void content( QString text ) = 0;

    //. Process the endOfData signal from the SgmlParser.  All subclasses
    //. must reimplement this.
    virtual void endOfData() {}

    virtual void endTag() {}

    virtual void startTag() {}
};

#endif
