// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
extern "C"
{
#include <stdio.h>
}
#include "SgmlParser.h"
#include "Style.h"
#include "StyleSheetProto.h"

Style::Style( const QString& name )
        : _name( name.copy() ),
          _display( styleSheetProto->findDisplay( "block" ) )
{
//    _sgmlAttrs.setAutoDelete( TRUE );
//    _attrs.setAutoDelete( TRUE );
}

Style::Style( const Style& src )
        : _name( src._name.copy() ),
          _display( src._display )
{
//    _sgmlAttrs.setAutoDelete( TRUE );
//    _attrs.setAutoDelete( TRUE );

    QListIterator<Attr> i( src._sgmlAttrs );
    for ( i.toFirst(); i.current(); ++i ) {
        _sgmlAttrs.append( new Attr( i.current()->name, i.current()->value ) );
    }

    StyleAttr* attr;
    QListIterator<StyleAttr> j( src._attrs );
    for ( j.toFirst(); j.current(); ++j ) {
        switch ( j.current()->type() ) {
            case StyleAttrProto::String:
                attr = new StringStyleAttr( *((StringStyleAttr*)j.current()) );
                break;

            case StyleAttrProto::Number:
                attr = new NumberStyleAttr( *((NumberStyleAttr*)j.current()) );
                break;

            case StyleAttrProto::Enumerated:
                attr = new EnumeratedStyleAttr( *((EnumeratedStyleAttr*)j.current()) );
                break;

            case StyleAttrProto::List:
                attr = new ListStyleAttr( *((ListStyleAttr*)j.current()) );
                break;

            default:
                attr = 0;
                break;
        }
        if ( attr ) {
            _attrs.append( attr );
        }
    }
}

Style::~Style()
{
    while ( _attrs.first() ) {
	delete _attrs.first();
	_attrs.remove();
    }
    while ( _sgmlAttrs.first() ) {
	delete _sgmlAttrs.first();
	_sgmlAttrs.remove();
    }
}

int Style::matchFactor( const STag* stag )
{
    int factor = 0;

    if ( _name != stag->name ) {
        return 0;
    }
    factor++;

    const QString* val;
    for ( _sgmlAttrs.first(); _sgmlAttrs.current(); _sgmlAttrs.next() ) {
        if ( !( val = stag->find( _sgmlAttrs.current()->name ) ) ) {
            return 0;
        }
        factor++;

        if ( _sgmlAttrs.current()->value != "" ) {
            if ( val->upper() != _sgmlAttrs.current()->value ) {
                return 0;
            }
            factor++;
        }
    }

    return factor;
}

bool Style::enumValue( const char* token, int& value )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( ( _attrs.current()->type() == StyleAttrProto::Enumerated ) && ( _attrs.current()->proto()->token() == token ) ) {
            value = ((EnumeratedStyleAttr*)_attrs.current())->value();
            return TRUE;
        }
    }

    return FALSE;
}

bool Style::listValue( const char* token, QString& value )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( ( _attrs.current()->type() == StyleAttrProto::List ) && ( _attrs.current()->proto()->token() == token ) ) {
            value = ((ListStyleAttr*)_attrs.current())->value();
            return TRUE;
        }
    }

    return FALSE;
}

bool Style::numberValue( const char* token, int& value )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( ( _attrs.current()->type() == StyleAttrProto::Number ) && ( _attrs.current()->proto()->token() == token ) ) {
            value = ((NumberStyleAttr*)_attrs.current())->value();
            return TRUE;
        }
    }

    return FALSE;
}

bool Style::stringValue( const char* token, QString& value )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( ( _attrs.current()->type() == StyleAttrProto::String ) && ( _attrs.current()->proto()->token() == token ) ) {
            value = ((StringStyleAttr*)_attrs.current())->value();
            return TRUE;
        }
    }

    return FALSE;
}

bool Style::flagValue( const char* token )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( ( _attrs.current()->type() == StyleAttrProto::Flag ) && ( _attrs.current()->proto()->token() == token ) ) {
            return TRUE;
        }
    }

    return FALSE;
}

const StyleAttr* Style::find( const char* token )
{
    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        if ( _attrs.current()->proto()->token() == token ) {
            return _attrs.current();
        }
    }

    return 0;
}

int Style::displayType()
{
    return _display->display;
}

void Style::print()
{
    printf( "Style::print() -- name = '%s'\n", _name.data() );

    printf( "Style::print() -- display = '%s'\n", _display->token.data() );

    for ( _sgmlAttrs.first(); _sgmlAttrs.current(); _sgmlAttrs.next() ) {
        printf( "Style::print() -- sgmlAttr = '%s, value = '%s'\n", _sgmlAttrs.current()->name.data(), _sgmlAttrs.current()->value.data() );
    }

    for ( _attrs.first(); _attrs.current(); _attrs.next() ) {
        switch ( _attrs.current()->type() ) {
            case StyleAttrProto::String:
                printf( "Style::print() -- attr = '%s', value = '%s'\n", _attrs.current()->name().data(), ((StringStyleAttr*)_attrs.current())->value().data() );
                break;

            case StyleAttrProto::Number:
                printf( "Style::print() -- attr = '%s', value = %d\n", _attrs.current()->name().data(), ((NumberStyleAttr*)_attrs.current())->value() );
                break;

            case StyleAttrProto::Enumerated:
                printf( "Style::print() -- attr = '%s', value = %d\n", _attrs.current()->name().data(), ((EnumeratedStyleAttr*)_attrs.current())->value() );
                break;

            case StyleAttrProto::List:
                printf( "Style::print() -- attr = '%s', value = '%s'\n", _attrs.current()->name().data(), ((ListStyleAttr*)_attrs.current())->value().data() );
                break;
        }
    }
}

void Style::setDisplay( StyleSheetProto::DisplayProto* display )
{
    _display = display;
}

QString Style::selectorString()
{
    QString selector;
    selector = _name.copy();
    selector += "(";
    QListIterator<Style::Attr> j( _sgmlAttrs );
    for ( j.toFirst(); j.current(); ++j ) {
        if ( !j.atFirst() ) {
            selector += ", ";
        }
        selector += j.current()->name;
        if ( j.current()->value.length() ) {
            selector += "=";
            selector += j.current()->value;
        }
    }
    selector += ")";

    return selector;
}
