/*
   Siag, Scheme In A Grid
   Copyright (C) 1996  Ulric Eriksson <ulric@edu.stockholm.se>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
   Module name:    window.c

   This module handles windows: creating, destroying, printing on windows.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "../common/cmalloc.h"
#include <curses.h>

#include "../siag/types.h"
#include "../siag/calc.h"
#include "tsiag.h"
#include "../common/fonts.h"
#include <unistd.h>
#include <ctype.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>

#ifdef GUILE
#include <guile/gh.h>
#endif

#ifdef TCL
#include <tcl.h>
#endif

#include "../siod/siod.h"

static int loopflag;

#ifndef ABS
#define ABS(a) ((a)>0?(a):-(a))
#endif

#define ESC (27)
#define TAB ('\t')
#define RET ('\r')
#define LFD ('\n')
#define DEL (0x7f)

#define C_BIT (31)

#define KB_MAX 10000
#define IQ_MAX 10000

int lastc;

int macro_flag;

textbuf kbd_macro = {0, 0, 0};

static struct {
  char *buf;
  int index, size;
} input_queue;

window *w_list;

int cursor_visible = FALSE;

int pr_scr_flag;
int grid_lines = 1;

extern int recalc;	/* number of iterations */

/*X
Print prompt on the bottom line and let the user enter a line into buffer.
The following commands exist:
a .. ~		Insert a character
TAB		Call comp(buffer)
C-t		Interchange the characters around point
C-b		Move one character backward
C-f		Move one character forward
C-a		Move to the beginning of the line
C-e		Move to the end of the line
C-d		Delete the character after point
DEL		Delete the character before point
C-k		Delete the rest of the line
ESC		Delete the whole line
C-g		Print "Quit" and return FALSE
LFD, RET	Return TRUE
X*/
/* 95-06-29: changed "buffer" to "buffr" to avoid confusing gcc */
int ask_for_str_comp(char *prompt, char *buffr, int (*comp)(char *))
{
  register int l, c, i;
  int p = strlen(prompt);

  l = strlen(buffr);
  for (;;) {
    standend();
    move(LINES-1, 0);
    clrtoeol();
    addstr(prompt);
    addstr(buffr);
    move(LINES-1, p+l);
    refresh();
    switch (c = get_char_from_input_queue()) {
    case C_BIT & 'a':
      l = 0;
      break;
    case C_BIT & 'b':
      if (l > 0) l--;
      break;
    case C_BIT & 'd':
      if (buffr[l]) memmove(buffr+l, buffr+l+1, strlen(buffr+l+1)+1);
			/* strcpy(buffr+l, buffr+l+1); */
      break;
    case C_BIT & 'e':
      l = strlen(buffr);
      break;
    case C_BIT & 'f':
      if (buffr[l]) l++;
      break;
    case C_BIT & 'g':
      llpr("Quit");
      return FALSE;
    case TAB:
      if ((*comp)(buffr)) return TRUE;
      l = strlen(buffr);
      break;
    case C_BIT & 'k':
      buffr[l] = '\0';
      break;
    case C_BIT & 't':
      if (strlen(buffr) >= 2) {
	if (buffr[l]) l++;
        if (l == 1) {
	  c = buffr[0];
	  buffr[0] = buffr[1];
	  buffr[1] = c;
	}
	else {
	  c = buffr[l-1];
	  buffr[l-1] = buffr[l-2];
	  buffr[l-2] = c;
	}
      }
      break;
    case ESC:
      buffr[l = 0] = '\0';
      break;
    case LFD:
    case RET:
      return TRUE;
    case DEL:
      if (l > 0) {
        l--;
	memmove(buffr+l, buffr+l+1, strlen(buffr+l+1)+1);
	/* strcpy(buffr+l, buffr+l+1); */
      }
      break;
    default:
      if (c >= ' ' && c < DEL) {
	for (i = strlen(buffr)+1; i > l; i--)
	  buffr[i] = buffr[i-1];
	buffr[l++] = c;
      }
      break;
    }
  }
  /*NOTREACHED*/
}

/* This particular completion function doesn't complete at all, it just
returns TRUE, making TAB equivalent to RET and LFD. */
static int nocomp(char *b)
{
  return TRUE;
}

/* Calls ask_for_str_comp with nocomp as completion function. */
/* 95-06-29: changed "buffer" to "buffr" to please gcc */
int ask_for_str(char *prompt, char *buffr)
{
  return ask_for_str_comp(prompt, buffr, nocomp);
}

/* Adds c to the keyboard macro.
Macros longer than KB_MAX charactesr are not allowed. */
static void record_macro(int c)
{
  if (kbd_macro.size >= KB_MAX) {
    llpr("Macro too long; definition terminated");
    kbd_macro.size = 0;
    macro_flag = FALSE;
    return;
  }
  kbd_macro.text[kbd_macro.size++] = c;
}

/*X
Writes the buffer to the input queue.  The size of the buffer is limited
to IQ_MAX characters to prevent recursive macros from crashing the program
(the macros crash, of course, but that is less of a disaster).
TRUE is returned if the string could be added, FALSE otherwise.
X*/
int add_str_to_input_queue(textbuf buf)
{
  int i = 0, j = input_queue.index+input_queue.size;
  
  if (input_queue.size+buf.size >= IQ_MAX) return FALSE;
  while (i < buf.size) {
    /* this isn't very fast, but it should be fast enough for our needs */
    j %= IQ_MAX;	/* wrap to beginning of text */
    input_queue.buf[j++] = buf.text[i++];
    input_queue.size++;
  }
  return TRUE;
}

/*X
Reads and returns one input character.
If there are characters in the input queue, these are used first.
If macro_flag is TRUE, the character is recorded.
X*/
int get_char_from_input_queue()
{
  int c;
  
  if (input_queue.size > 0) {
    c = input_queue.buf[input_queue.index++];
    input_queue.index %= IQ_MAX;
    input_queue.size--;
  }
  else {
    c = getch();
  }
  if (macro_flag) record_macro(c);
  return c;
}

enum {ABORT=0, DONE, WAITING};

int input_warp_pointer = 1;	/* move pointer to input field */

void mainloop()
{
	loopflag = TRUE;
	atexit(exit_windows);
	while (loopflag) {
		show_cur(w_list);
		lastc = get_char_from_input_queue();
		llpr("");
		do_cmd(lastc);
	}
	llpr("goodbye");
	exit(0);
}

static int columns;    /* number of cells per line */
static int width;      /* default cell width */

static int left_margin = 8;
/*static int right_margin = 5;*/

/* Prints a single line (inverted) with the name of the buffer.
If the current buffer has been changed since it was last saved, it is
marked with "**". */
void infopr(window *w)
{
  register int i;
  int row = w->begin_y+w->max_y-1;

TRACEME((f, "infopr(%s)", w->buf->name));

  move(row, 0);
  standout();
  for (i = 0; i < COLS; i++) addch('-');
  move(row, 0);
  printw("--%s- Siag: %s ", w->buf->change ? "**" : "--", w->buf->name);
  standend();
}

/* Prints the string p on the bottom line of the screen.  If p is empty and
the last string printed was also empty, the string isn't printed. */
void llpr(char *p)
{
  static int isclear = FALSE;

TRACEME((f, "llpr(%s)", p));
  
  if (isclear && p[0] == '\0') return;
  isclear = (p[0] == '\0');

  standend();
  move(LINES-1, 0);
  clrtoeol();
  addstr(p);
  refresh();
}

/* Remove w from the window list and free the memory it used.
If w was the active window, w_list is moved forward to the next window.
If w was the last window, w_list is set to NULL. */
void free_window(window *w)
{
  window *pw;
  
  /* unlink from window list */
  for (pw = w_list;
       pw->next != w && pw->next != pw;
       pw = pw->next);
  pw->next = w->next;
  
  /* make sure w_lst does not point to a deleted window */
  if (w_list == w) w_list = w_list->next;
  
  /* no windows in the list => w_list = NULL */
  if (w_list == w) w_list = NULL;
  cfree(w);
}

/*X
window *new_window(int begin_y, int max_y, buffer *b, window *prev)
Creates a new window of height max_y, starting at line begin_y.
The new window points to buffer b.  Point and top positions are set
to the first position in b.
The new window is inserted in the window list after prev, or after itself
if prev is NULL.
The new window is returned, or NULL, if it couldn't be created.
X*/
window *new_window(buffer *b, window *prev)
{
/* FIXME */
  int max_y = (LINES-1);
  int begin_y = 0;

  window *w;
  
  w = (window *)cmalloc(sizeof(window));
  
  if (max_y < 4) {
    llpr("This window is too small to create");
    cfree(w);
    return NULL;
  }
  
  w->begin_y = begin_y;
  w->max_y = max_y;

  w->buf = b;
  w->point_pos.row = 1;
  w->point_pos.col = 1;
  w->blku.row = w->blku.col = 2;
  w->blkl.row = w->blkl.col = 1;
  w->prot = w->top = w->point_pos;
  
  if (prev == NULL) prev = w;
  else w->next = prev->next;
  prev->next = w;
  
  return w;
}

/*X
Removes w from the window list and frees the memory.  The window below
w (or above if w was the lowest window on the screen) is expanded to
make use of the space that was previously occupied by w.
The last window on the screen can't be removed with remove_window.
Returns TRUE for success, FALSE for failure.
X*/
int remove_window(window *w)
{
  window *w2;
  buffer *b;
  position newtop, newpoint;
  int max_y, begin_y;
  
  if (w == w->next) return FALSE;
  
  if (w->begin_y < w->next->begin_y)
    w2 = w->next;       /* w2 is after w */
  else for (w2 = w; w2->next != w; w2 = w2->next);      /* w2 is before w */
  b = w2->buf;
  
  if ((begin_y = w->begin_y) > w2->begin_y)
    begin_y = w2->begin_y;
  max_y = w->max_y + w2->max_y;
  
  newtop = w2->top;
  newpoint = w2->point_pos;
  
  free_window(w2);
  w2 = new_window(b, w);
  w2->point_pos = newpoint;
  w2->top = newtop;
  free_window(w);
  return TRUE;
}

/*X
Splits w in two halves of the same size, if w is big enough.
Both windows point to the buffer that was previously in w.
Return TRUE for success, FALSE for failure.
X*/
int split_window(window *w)
{
  window *w1, *w2;
  int lines1, lines2, begin_y1, begin_y2;
  
  begin_y1 = w->begin_y;
  lines1 = (w->max_y) / 2;
  begin_y2 = begin_y1+lines1;
  lines2 = w->max_y-lines1;
  if (lines1+lines2 < 8) return FALSE;
  w1 = new_window(w->buf, w);
  w2 = new_window(w->buf, w1);
  
  w1->point_pos = w2->point_pos = w->point_pos;
  w1->top = w2->top = w->top;
  free_window(w);       /* takes care of the case w == w_list, don't worry */
  return TRUE;
}

void init_windows1(int *foo, char **bar)
{
  kbd_macro.maxsize = KB_MAX;
  kbd_macro.text = (char *)ccalloc(KB_MAX, sizeof(char));
  
  input_queue.buf = (char *)ccalloc(IQ_MAX, sizeof(char));
  input_queue.index = 0;
  input_queue.size = 0;
}

static void screxit()
{
TRACEME((f,"screxit()"));

  standend();
  move(LINES-1, 0);
  clrtoeol();
  refresh();

  nl();
  endwin();
}

/* Sets up the whole initial window structure and initializes scrupd.
The window list w_list is set to a list with a single window with the
buffer b. */
void init_windows(buffer *b, int argc, char **argv)
{
  initscr();    /* this curses function must be called first */
  raw();        /* Make input behave */
  noecho();
  nonl();

  keypad(stdscr, TRUE);
  standend();      /* turn off reverse */

  clear();
  refresh();

  width = 8;
  columns = (COLS-left_margin)/width;

  if (columns < 3 || LINES < 5) {
    fprintf(stderr, "The screen is too small\n");
    exit(EXIT_FAILURE);
  }
  
  w_list = new_window(b, (window *)0);
  init_calc_cmds();
}

/* Cleans up after Calc before exit.  All buffers and windows are freed. */
void exit_windows()
{
  /* free all buffers */
  while (b_list != NULL) free_buffer(b_list);
  
  /* free all windows */
  while (w_list != NULL) free_window(w_list);

  screxit();
}

/* Prints line i from the top, starting at 0. */
static void pr_line(window *w, int i)
{
  register int j;
  int row = w->begin_y+i;

  move(1+row, 0);

  standout();
  printw("%-*d", left_margin-1, w->top.row+i);
  for (j = 0; j < columns; j++)       /* mark block if needed */
    if (inblock(w, w->top.row+i, w->top.col+j)) {
      move(1+row, left_margin+width*j);
      addstr("        ");
    }
  standend();
  for (j = 0; j < columns; j++) {
    	char s[1024];
    	move(1+row, left_margin+width*j);
    	if (inblock(w, w->top.row+i, w->top.col+j)) standout();
    	else standend();
	s[0] = '\0';
	ret_pvalue(s, w->buf, w->top.row+i, w->top.col+j, -1);
	printw("%s", s);
  }
} /* pr_line */

/* Print and refresh all the windows. Sets pr_scr_flag to FALSE. */
static void pr_scr()
{
  register int i;
  window *w;
  buffer *b;
	b = b_list;
TRACEME((f,"pr_scr(): b->recalc=%d, recalc=%d",b->recalc,recalc));
	do {
		if (b->recalc) {
			int i;
			b->recalc = 0;
			for (i = 0; i < recalc; i++) {
TRACEME((f,"\trecalculating"));
				calc_matrix(b);
			}
		}
		b = b->next;
	} while (b != b_list);

  standend();
  erase();
  
TRACEME((f,"pr_scr()"));

  w = w_list;
  do {
    standout();
    for (i = 0; i < columns; i++) {
      move(w->begin_y, left_margin+width*i);
      printw("%-*d", width-1, w->top.col+i);
    }
    for (i = 0; i < w->max_y-2; i++) {
      pr_line(w, i);
    }

    infopr(w);
    standend();
    w = w->next;
  } while (w != w_list);

  pr_scr_flag = FALSE;

} /* pr_scr */

static long max_lines = 100000;
static long max_columns = 100000;

/* Move the cursor to reflect the position of point in w.
If point is not visible, the window is moved so that point is in
the middle of the screen. */
void show_cur(window *w)
{
  register int oldr, oldk;

TRACEME((f,"show_cur(): enter"));

  oldr = w->top.row;
  oldk = w->top.col;

TRACEME((f,"\tshow_cur(): point=(%d,%d) top=(%d,%d)",
	w->point_pos.row, w->point_pos.col, w->top.row, w->top.col));

  if (w->point_pos.row < w->top.row ||
      w->point_pos.row > w->top.row+(w->max_y-3))
    w->top.row = w->point_pos.row-((w->max_y-4)/2);

  if (w->top.row < 1) w->top.row = 1;
  else if (w->top.row > max_lines-(w->max_y-4))
    w->top.row = max_lines-(w->max_y-4);

  if (w->point_pos.col < w->top.col ||
      w->point_pos.col > w->top.col+(columns-1))
    w->top.col = w->point_pos.col-(columns/2);

  if (w->top.col < 1) w->top.col = 1;
  else if (w->top.col > max_columns-(columns-1))
    w->top.col = max_columns-(columns-1);

TRACEME((f,"\tshow_cur(): point=(%d,%d) top=(%d,%d)",
	w->point_pos.row, w->point_pos.col, w->top.row, w->top.col));

  if ((oldr != w->top.row) || (oldk != w->top.col))
    pr_scr_flag = TRUE;

TRACEME((f,"\tshow_cur(): pr_scr_flag=%d",pr_scr_flag));
  if (pr_scr_flag) {
    pr_scr();
    refresh();
  }

  move(1 + w->begin_y + w->point_pos.row - w->top.row,
	  left_margin + width*(w->point_pos.col - w->top.col));
  refresh();

TRACEME((f,"show_cur(): exit"));

} /* show_cur */

void activate_window(window *w)
{
	char b[256];

	w_list = w;
	strcpy(b, "Siag: ");
	strncat(b, w->buf->name, 200);
}

void draw_input(char *text)
{
	int r = w_list->point_pos.row, c = w_list->point_pos.col;
	int type = ret_type(w_list->buf, r, c);
	int intp = ret_interpreter(w_list->buf, r, c);
	char b[1024];
	if (type == EXPRESSION || type == STRING)
		sprintf(b, "[%d,%d] [%s] %s",
			r, c, interpreter2name(intp), text);
	else
		sprintf(b, "[%d,%d] %s",
			r, c, text);
}

/*
How to make Siag always draw the top row and/or the left column.

This is desirable to keep headers on the screen when the sheet
is scrolled. To make this possible, I need to add a field in
the window structure to specify the first unprotected cell.
Everything above and to the left of this cell (called w->prot)
should always be drawed, regardless of what top is.

While this sounds easy, it profoundly changes the mapping from
cell coordinates to screen coordinates and vice versa. For instance,
answering the question "is the cursor on the screen" is no longer
as easy.

Make top be the first unprotected column. Point can never leave
the unprotected area. Leave mark and other positions for now.
*/

/* 970427: the following four functions facilitate navigating in
	the cell area with protection implemented */

static int cell_next_row(window *w, int row)
{
	if (row+1 == w->prot.row) return w->top.row;
	return row+1;
}

static int cell_next_col(window *w, int col)
{
	if (col+1 == w->prot.col) return w->top.col;
	return col+1;
}

static int cell_prev_row(window *w, int row)
{
	if (row == w->top.row) return w->prot.row-1;
	return row-1;
}

static int cell_prev_col(window *w, int col)
{
	if (col == w->top.col) return w->prot.col-1;
	return col-1;
}

/* From (row, col), calculate (x, y) coordinates. This is a little tricker
	now. Rather than starting the calculation in (0, 0) we must
	take the protected cells into account. */
void get_cell_coords(window *w, int top_row, int top_col,
		     int cell_row, int cell_col,
		     int *cell_x, int *cell_y)
{
	int i;

	*cell_y = 0;
	for (i = 1; i < w->prot.row; i++)
		*cell_y += cell_height(w->buf, i);

	while (cell_row < top_row) {
		cell_row = cell_next_row(w, cell_row);
		*cell_y -= cell_height(w->buf, cell_row);
	}
	while (cell_row > top_row) {
		cell_row = cell_prev_row(w, cell_row);
		*cell_y += cell_height(w->buf, cell_row);
	}
	*cell_x = 0;
	for (i = 1; i < w->prot.col; i++)
		*cell_x += cell_width(w->buf, i);

	while (cell_col < top_col) {
		cell_col = cell_next_col(w, cell_col);
		*cell_x -= cell_width(w->buf, cell_col);
	}
	while (cell_col > top_col) {
		cell_col = cell_prev_col(w, cell_col);
		*cell_x += cell_width(w->buf, cell_col);
	}
}

void get_coords_cell(window *w, int top_row, int top_col,
		     int *cur_row, int *cur_col,
		     int cur_x, int cur_y)
{
	int prot_x = 0, prot_y = 0, i;

	for (i = 1; i < w->prot.col; i++)
		cur_x -= cell_width(w->buf, i);
	for (i = 1; i < w->prot.row; i++)
		cur_y -= cell_height(w->buf, i);

	*cur_row = top_row;
	*cur_col = top_col;
	while (cur_y < prot_y && *cur_row > 1) {
		cur_y += cell_height(w->buf, *cur_row);
		(*cur_row) = cell_prev_row(w, *cur_row);
	}
	while (cur_y > cell_height(w->buf, *cur_row) && *cur_row < BUFFER_ROWS) {
		cur_y -= cell_height(w->buf, *cur_row);
		(*cur_row) = cell_next_row(w, *cur_row);
	}
	while (cur_x < prot_x && *cur_col > 1) {
		cur_x += cell_width(w->buf, *cur_col);
		(*cur_col) = cell_prev_col(w, *cur_col);
	}
	while (cur_x > cell_width(w->buf, *cur_col) && *cur_col < BUFFER_COLS) {
		cur_x -= cell_width(w->buf, *cur_col);
		(*cur_col) = cell_next_col(w, *cur_col);
	}
}

void hide_cur(window *w) { ; }

